#!/usr/bin/rebol -qs
REBOL [;{{{ } } }
	Title:   "Conversion of a file containing GeolPDA orientations measurements"
	Date:    == 19-Oct-2013/10:52:07+2:00
	Version: 0.0.0 
	Purpose: {
		Conversion from a .csv file generated by GeolPDA,
		containing structural measurements expressed as 
		3x3 rotation matrices of the telephone orientation, 
		to a file containing "traditional" structural geological
		measurements, and fields necessary to be able to map
		structural symbols by using a GIS package.
	}
	History: [
	]
	License: {
This file is part of GeolLLibre software suite: FLOSS dedicated to Earth Sciences.
###########################################################################
##          ____  ___/_ ____  __   __   __   _()____   ____  _____       ##
##         / ___\/ ___// _  |/ /  / /  / /  /  _/ _ \ / __ \/ ___/       ##
##        / /___/ /_  / / | / /  / /  / /   / // /_/_/ /_/ / /_          ##
##       / /_/ / /___|  \/ / /__/ /__/ /___/ // /_/ / _, _/ /___         ##
##       \____/_____/ \___/_____/___/_____/__/_____/_/ |_/_____/         ##
##                                                                       ##
###########################################################################
  Copyright (C) 2013 Pierre Chevalier <pierrechevaliergeol@free.fr>
 
    GeolLLibre is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>
    or write to the Free Software Foundation, Inc., 51 Franklin Street, 
    Fifth Floor, Boston, MA 02110-1301, USA.
    See LICENSE file.
}
];}}}

; initialisation: ;{{{ } } }
if error? try [						; Récupération des routines (et des préférences) et connexion à la base
if error? try [						; Récupération des routines (et des préférences) et connexion à la base
do load to-file system/options/home/bin/gll_routines.r	; soit depuis ~/bin
] [
do load to-file %gll_routines.r				; ou sinon là où se trouve le script présent
]
] [
do load to-file system/options/home/geolllibre/gll_routines.r		; ou sinon dans ~/geolllibre
]

;}}}

; Open a geolpda .csv files:
file_in: to-file request-file/title/filter "GeolPDA orientations file" "Open" [geolpda*.csv *csv *.CSV]

;file_in: %geolllibre/qqtest_geolpda_orientations.csv	; DEBUG ###

; Read its contents into a block! named orientations:
orientations: copy []
foreach line at read/lines file_in 2 [
	r: parse/all line ","		; r stands for row: parse the csv contents
	;print r
	; add fields to orientations:       _id poi_id orientationtype        rot1    rot2    rot3    rot4    rot5    rot6     rot7     rot8     rot9   v1 v2 v3
	;                                     1      2               3           4       5       6       7       8       9       10       11       12   13 14 15 16
	append/only orientations reduce   [ r/1    r/2             r/3 rejoin [r/4 " " r/5 " " r/6 " " r/7 " " r/8 " " r/9 " " r/10 " " r/11 " " r/12 ]]
]
; exemple:
;print mold orientations/1	;DEBUG
;>> print mold orientations/1;DEBUG
;["867" "372" "P" {0.113424 0.0507597 -0.992249 -0.0104283 0.9987 0.0498977 0.993492 0.00468783 0.113806}]


; Iterate through orientations, and make another 
; block! with the computed structural measurements:
struct_measures: copy []
foreach i orientations [
	o: orientation/new to-block to-string (at i 4)
	append struct_measures o
	;print o/print_plane_line
]

length? struct_measures
;== 867

;=> too dispendious to make so many object!s with the code repeated every time.


;DEBUG
;i: orientations/1
;o: orientation/new to-block to-string (at i 4)
;o/trace_structural_symbol diagram

structural_symbol: make diagram []

l: layout [
    t: text-list 400x200 data [] [
		; when the contents of the text-list widget change:
		o: orientation/new to-block value					; make an orientation object named o
		print o/print_plane_line							; outputs the plane and line of the object
			structural_symbol: make diagram []				; make an empty diagram
		o/trace_structural_symbol structural_symbol			; call trace_structural_symbol method from o object, with the diagram as argument
		;append 
		zone_diagram/effect/draw: structural_symbol/plot	; set the diagram on the zone_diagram box defined below
		show zone_diagram									; refresh the zone_diagram box
		]
	zone_diagram: box ivory 220x220 effect [
		;grid 10x10
		draw []
		] [	; Click: reset plot
			zone_diagram/effect/draw: copy [] 
			show zone_diagram]
		[	; Right click: reset structural_symbol
			structural_symbol: make diagram []
			zone_diagram/effect/draw: copy [] 
			show zone_diagram]
	btn #"q" "quit"[unview]
]
foreach i orientations [
	append t/data i/4
]
view l

;*******************
;*******************
; Les expérimentations qu'il y avait dans gll_geolpda_compute_rotation_matrix_test.r:/*{{{*/
comment [

; L'emploi de l'objet orientation: {{{ } } }
; USAGE:
; on définit une orientation (la foliation du gneiss basque):/*{{{*/
;o: orientation/new [0.375471 -0.866153 -0.32985 0.669867 0.499563 -0.549286 0.640547 -0.0147148 0.767778]
o: orientation/new [0.13364 0.856407 0.498706 -0.794842 0.393185 -0.462203 -0.591918 -0.334624 0.733253]
print o/print_matrix
print o/print_plane
print o/print_line
print o/print_plane_line
;/*}}}*/
; on trace son Té sur diagram:/*{{{*/
structural_symbol: diagram/plot
o/trace_structural_symbol diagram
;== [pen black pen gray line 110x165 110x54 line 54x110 165x110 pen black line 152x135 67x84 line 110x110 102x122 line 110x110 153x1...
; le Té est contenu dans diagram/plot:
print mold diagram/plot
print mold structural_symbol
;>> print mold structural_symbol/plot
;[pen black pen gray line 110x165 110x54 line 54x110 165x110 pen black line 152x135 67x84 line 110x110 102x122 line 110x110 153x134]
;/*}}}*/

; autre exemple:/*{{{*/ } } }
; ~~~Nm30/40/E/30/N/I
;"868","387","PLM","0.13364","0.856407","0.498706","-0.794842","0.393185","-0.462203","-0.591918","-0.334624","0.733253","0","0","0"
o: orientation/new [0.13364 0.856407 0.498706 -0.794842 0.393185 -0.462203 -0.591918 -0.334624 0.733253]
print o/print_matrix
print o/print_plane
print o/print_line
print o/print_plane_line
;Matrix:     0.13364 0.856407 0.498706 -0.794842 0.393185 -0.462203 -0.591918 -0.334624 0.733253
;Plane:  Nm42/42/E
;Line:   Nm65/-19
;=> semble auquai/*}}}*/
; on trace son Té sur diagram:/*{{{*/
structural_symbol: diagram/plot
o/trace_structural_symbol diagram
; le Té est contenu dans diagram/plot:
print mold diagram/plot
print mold structural_symbol
;/*}}}*/


; on dessine ça dans un layout:
l: layout [
	list_orientations: list 240x300 "A" "b" "c"
;[0.13364 0.856407 0.498706 -0.794842 0.393185 -0.462203 -0.591918 -0.334624 0.733253] [0.13364 0.856407 0.498706 -0.794842 0.393185 -0.462203 -0.591918 -0.334624 0.733253]]
	zone_diagram: box ivory 220x220 effect	[
		;grid 10x10 gray
		;draw [circle 100 100 10]
		draw []
	]
	btn #"q" "quit" [unview]
]
append zone_diagram/effect/draw structural_symbol
view l

; a graphical user interface, more simple:
a: [ ;{{{ } } }
ui: layout [ ; Interface to debug rotation_matrix depatouillating:
	h3 "Matrice de rotation:"	;guide return
	field_matrix: field 250 "0.375471 -0.866153 -0.32985 0.669867 0.499563 -0.549286 0.640547 -0.0147148 0.767778" [
		update_matrix
	]
	structural_measurement: area 250x100
	return
	zone_diagram: box ivory 220x220 effect	[
		;grid 10x10
		draw structural_symbol				]	; strange: if I put diagram/plot (which is the same as structural_symbol, it does not work,
												; although diagram/plot points to structural_symbol => ?
	across
	btn #"t" "trace"	[ trace_diagram	]
	btn #"c" "calcule"	[ print "calcule"	]
	btn #"d" "DEBUG"	[?? o
						?? structural_symbol
						]
	btn #"q" "quitte"	[	;quit 
							halt			]
]
trace_diagram: does						[
	print "trace"
	o: orientation/new to-block field_matrix/text
	o/trace_structural_symbol diagram
	show zone_diagram					]
update_matrix: does [
	print "TODO changement matrice"
	o: orientation/new to-block field_matrix/text
	structural_measurement/text: reduce [o/print_plane_line]
	;append zone_diagram/effect/draw structural_symbol
	show zone_diagram
	show structural_measurement
]
field_matrix/text: "0.375471 -0.866153 -0.32985 0.669867 0.499563 -0.549286 0.640547 -0.0147148 0.767778"
show field_matrix
;append zone_diagram structural_symbol
view ui
do-events
] ;}}}
do a
do-events


;-------------------------------------------------------------------##
; Des essais:{{{ } } }
;[0.375471 -0.866153 -0.32985 0.669867 0.499563 -0.549286 0.640547 -0.0147148 0.767778]
; Une mesure réelle du géolpda:
; Exemple réel de plan (sans ligne), en Euskadi:/*{{{*/
;_id	poiname	poitime	elevation	poilat	poilon	photourl	audiourl	note
;358	PCh2013_0639	1377339260698	89.9	43.3359	-1.39078	1377339392897.jpg;1377339414464.jpg		Bel affleurement de gneiss, foliation Nm140/40/W
;_id	poi_id	orientationtype	rot1	rot2	rot3	rot4	rot5	rot6	rot7	rot8	rot9	v1	v2	v3
;851	358	P	0.375471	-0.866153	-0.32985	0.669867	0.499563	-0.549286	0.640547	-0.0147148	0.767778	0	0	0
;
;matrix:           [0.375471 -0.866153 -0.32985 0.669867 0.499563 -0.549286 0.640547 -0.0147148 0.767778]
o: orientation/new [0.375471 -0.866153 -0.32985 0.669867 0.499563 -0.549286 0.640547 -0.0147148 0.767778]
print o/print_plane_line
;=>
;>> o: orientation/new [0.375471 -0.866153 -0.32985 0.669867 0.499563 -0.549286 0.640547 -0.0147148 0.767778]
;>> o/print_plane_line
;Matrix:     0.375471 -0.866153 -0.32985 0.669867 0.499563 -0.549286 0.640547 -0.0147148 0.767778
;Plane:  Nm120/39/S
;Line:   Nm299/0

probe o
print mold o/matrix
print o/a
o/plane_normal_vector
o/axis_vector
o/plane_downdip_azimuth
o/plane_direction
o/plane_dip
o/plane_quadrant_dip
o/line_azimuth
o/line_plunge
o/print_plane_line
;/*}}}*/

; L'exemple de matrice pour une pseudomesure de faille Nm30/60/E/55/S/N:
o: orientation/new [-0.7   0     0.7 0.6   0.6  -0.5 0.4   0.8   0.4]
print o/print_plane_line
;=>
;>> o/print_plane_line
;Matrix:     -0.7 0 0.7 0.6 0.6 -0.5 0.4 0.8 0.4
;Plane:  Nm35/66/E
;Line:   Nm0/53

;# mesure de faille Nm30/60/E/55/S/N:
;rotation_matrix:	 [ -0.7   0     0.7
;					    0.6   0.6  -0.5
;					    0.4   0.8   0.4]

f: orientation/new [ -0.7   0     0.7 0.6   0.6  -0.5 0.4   0.8   0.4]
print f/print_plane_line
;>> f/print_plane_line
;Matrix:     -0.7 0 0.7 0.6 0.6 -0.5 0.4 0.8 0.4
;Plane:  Nm35/66/E
;Line:   Nm0/53

;}}}
;-------------------------------------------------------------------##
;}}}

;###### Other solution, a function which uses orientation object!, but in a temporary manner, so that an array of measurements would be waste way less memory: {{{ } } }
compute_geolpda_orientation_matrix: func ["Takes a rotation matrix as provided by GeolPDA, and outputs a structural measurement, for measured plane and/or line" geolpda_matrix [string! block!]] [ ;{{{ } } }
	; (function was previously named matrix2struct_measure)
	; DEBUG ######
		; L'exemple de matrice pour une pseudomesure de faille Nm30/60/E/55/S/N:
		matrix: {-0.7   0     0.7 0.6   0.6  -0.5 0.4   0.8   0.4}
	; ################
	; If matrix is given as a string!, convert it to a block!:
	if (type? matrix) = string! [matrix: to-block matrix]
	; Check if matrix is well a 9 elements block!:
	if (length? matrix) != 9 [
		print rejoin ["Error, rotation matrix given as argument has " length? matrix " elements instead of 9: matrix must be a 3x3 matrix written as 9 elements"] 
		return none]
	; Check if all 9 elements are numeric:
	bad: false
	foreach v matrix [
		if (all [((type? v) != number!) ((type? v) != decimal!) ((type? v) != integer!)]) [bad: true]
		]
	if bad [ print rejoin ["Error, rotation matrix given as argument contains elements which are not numeric"] return none]
	; If we got that far, no check error found.
	; Build an orientation object!:
	o: orientation/new matrix
	; Get its contents, put it into a *light* object! (without methods):
	res: make object! [
		plane_direction:        o/plane_direction
		plane_dip:              o/plane_dip
		plane_quadrant_dip:     o/plane_quadrant_dip
		plane_downdip_azimuth:  o/plane_downdip_azimuth
		line_azimuth:           o/line_azimuth
		line_plunge:            o/line_plunge
	]
	return res
] ;}}}

; test:
;>> probe compute_geolpda_orientation_matrix {-0.7   0     0.7 0.6   0.6  -0.5 0.4   0.8   0.4}
;make object! [
;    plane_direction: 35.5376777919744
;    plane_dip: 66.4218215217982
;    plane_quadrant_dip: "E"
;    plane_downdip_azimuth: 125.537677791974
;    line_azimuth: 0
;    line_plunge: 53.130102354156
;]

] ;/*}}}*/

;}}}

