#include "colormap.h"
#include "settings.h"
#include <QDebug>

namespace impl {

template <typename T>
inline constexpr T clamp_range(T x, T lower = 0, T upper = 1) noexcept {
  return (x < lower) ? lower : (x > upper) ? upper : x;
}

inline float linear(float a, float b, float t) noexcept {
  return a * (1 - t) + b * t;
}

inline QColor linearColorMap(double x, const std::vector<QColor> &data) {
  const double a = clamp_range(x) * (data.size() - 1);
  const double i = std::floor(a);
  const double t = a - i;
  const QColor &colorA = data[static_cast<std::size_t>(i)];
  const QColor &colorB = data[static_cast<std::size_t>(std::ceil(a))];

  float h1, s1, v1;
  float h2, s2, v2;
  colorA.getRgbF(&h1, &s1, &v1);
  colorB.getRgbF(&h2, &s2, &v2);

  return QColor::fromRgbF(linear(h1, h2, t), linear(s1, s2, t),
                          linear(v1, v2, t));
}

inline double quantize(double x, uint_fast8_t levels) {
  levels = std::max(uint_fast8_t{1}, std::min(levels, uint_fast8_t{255}));

  const double interval = 255.0 / levels;

  constexpr double eps = 0.0005;
  const unsigned int index =
      static_cast<uint_fast8_t>((x * 255.0 - eps) / interval);

  const unsigned int upper =
      static_cast<uint_fast8_t>(index * interval + interval);
  const unsigned int lower = static_cast<uint_fast8_t>(upper - interval);

  const double middle = static_cast<double>(upper + lower) * 0.5 / 255.0;
  return middle;
}

namespace colors {
// TODO move this to JSON, indexing by QString
inline const std::vector<QColor> Parula{
    QColor::fromRgbF(0.2081, 0.1663, 0.5292),
    QColor::fromRgbF(0.2091, 0.1721, 0.5411),
    QColor::fromRgbF(0.2101, 0.1779, 0.553),
    QColor::fromRgbF(0.2109, 0.1837, 0.565),
    QColor::fromRgbF(0.2116, 0.1895, 0.5771),
    QColor::fromRgbF(0.2121, 0.1954, 0.5892),
    QColor::fromRgbF(0.2124, 0.2013, 0.6013),
    QColor::fromRgbF(0.2125, 0.2072, 0.6135),
    QColor::fromRgbF(0.2123, 0.2132, 0.6258),
    QColor::fromRgbF(0.2118, 0.2192, 0.6381),
    QColor::fromRgbF(0.2111, 0.2253, 0.6505),
    QColor::fromRgbF(0.2099, 0.2315, 0.6629),
    QColor::fromRgbF(0.2084, 0.2377, 0.6753),
    QColor::fromRgbF(0.2063, 0.244, 0.6878),
    QColor::fromRgbF(0.2038, 0.2503, 0.7003),
    QColor::fromRgbF(0.2006, 0.2568, 0.7129),
    QColor::fromRgbF(0.1968, 0.2632, 0.7255),
    QColor::fromRgbF(0.1921, 0.2698, 0.7381),
    QColor::fromRgbF(0.1867, 0.2764, 0.7507),
    QColor::fromRgbF(0.1802, 0.2832, 0.7634),
    QColor::fromRgbF(0.1728, 0.2902, 0.7762),
    QColor::fromRgbF(0.1641, 0.2975, 0.789),
    QColor::fromRgbF(0.1541, 0.3052, 0.8017),
    QColor::fromRgbF(0.1427, 0.3132, 0.8145),
    QColor::fromRgbF(0.1295, 0.3217, 0.8269),
    QColor::fromRgbF(0.1147, 0.3306, 0.8387),
    QColor::fromRgbF(0.0986, 0.3397, 0.8495),
    QColor::fromRgbF(0.0816, 0.3486, 0.8588),
    QColor::fromRgbF(0.0646, 0.3572, 0.8664),
    QColor::fromRgbF(0.0482, 0.3651, 0.8722),
    QColor::fromRgbF(0.0329, 0.3724, 0.8765),
    QColor::fromRgbF(0.0213, 0.3792, 0.8796),
    QColor::fromRgbF(0.0136, 0.3853, 0.8815),
    QColor::fromRgbF(0.0086, 0.3911, 0.8827),
    QColor::fromRgbF(0.006, 0.3965, 0.8833),
    QColor::fromRgbF(0.0051, 0.4017, 0.8834),
    QColor::fromRgbF(0.0054, 0.4066, 0.8831),
    QColor::fromRgbF(0.0067, 0.4113, 0.8825),
    QColor::fromRgbF(0.0089, 0.4159, 0.8816),
    QColor::fromRgbF(0.0116, 0.4203, 0.8805),
    QColor::fromRgbF(0.0148, 0.4246, 0.8793),
    QColor::fromRgbF(0.0184, 0.4288, 0.8779),
    QColor::fromRgbF(0.0223, 0.4329, 0.8763),
    QColor::fromRgbF(0.0264, 0.437, 0.8747),
    QColor::fromRgbF(0.0306, 0.441, 0.8729),
    QColor::fromRgbF(0.0349, 0.4449, 0.8711),
    QColor::fromRgbF(0.0394, 0.4488, 0.8692),
    QColor::fromRgbF(0.0437, 0.4526, 0.8672),
    QColor::fromRgbF(0.0477, 0.4564, 0.8652),
    QColor::fromRgbF(0.0514, 0.4602, 0.8632),
    QColor::fromRgbF(0.0549, 0.464, 0.8611),
    QColor::fromRgbF(0.0582, 0.4677, 0.8589),
    QColor::fromRgbF(0.0612, 0.4714, 0.8568),
    QColor::fromRgbF(0.064, 0.4751, 0.8546),
    QColor::fromRgbF(0.0666, 0.4788, 0.8525),
    QColor::fromRgbF(0.0689, 0.4825, 0.8503),
    QColor::fromRgbF(0.071, 0.4862, 0.8481),
    QColor::fromRgbF(0.0729, 0.4899, 0.846),
    QColor::fromRgbF(0.0746, 0.4937, 0.8439),
    QColor::fromRgbF(0.0761, 0.4974, 0.8418),
    QColor::fromRgbF(0.0773, 0.5012, 0.8398),
    QColor::fromRgbF(0.0782, 0.5051, 0.8378),
    QColor::fromRgbF(0.0789, 0.5089, 0.8359),
    QColor::fromRgbF(0.0794, 0.5129, 0.8341),
    QColor::fromRgbF(0.0795, 0.5169, 0.8324),
    QColor::fromRgbF(0.0793, 0.521, 0.8308),
    QColor::fromRgbF(0.0788, 0.5251, 0.8293),
    QColor::fromRgbF(0.0778, 0.5295, 0.828),
    QColor::fromRgbF(0.0764, 0.5339, 0.827),
    QColor::fromRgbF(0.0746, 0.5384, 0.8261),
    QColor::fromRgbF(0.0724, 0.5431, 0.8253),
    QColor::fromRgbF(0.0698, 0.5479, 0.8247),
    QColor::fromRgbF(0.0668, 0.5527, 0.8243),
    QColor::fromRgbF(0.0636, 0.5577, 0.8239),
    QColor::fromRgbF(0.06, 0.5627, 0.8237),
    QColor::fromRgbF(0.0562, 0.5677, 0.8234),
    QColor::fromRgbF(0.0523, 0.5727, 0.8231),
    QColor::fromRgbF(0.0484, 0.5777, 0.8228),
    QColor::fromRgbF(0.0445, 0.5826, 0.8223),
    QColor::fromRgbF(0.0408, 0.5874, 0.8217),
    QColor::fromRgbF(0.0372, 0.5922, 0.8209),
    QColor::fromRgbF(0.0342, 0.5968, 0.8198),
    QColor::fromRgbF(0.0317, 0.6012, 0.8186),
    QColor::fromRgbF(0.0296, 0.6055, 0.8171),
    QColor::fromRgbF(0.0279, 0.6097, 0.8154),
    QColor::fromRgbF(0.0265, 0.6137, 0.8135),
    QColor::fromRgbF(0.0255, 0.6176, 0.8114),
    QColor::fromRgbF(0.0248, 0.6214, 0.8091),
    QColor::fromRgbF(0.0243, 0.625, 0.8066),
    QColor::fromRgbF(0.0239, 0.6285, 0.8039),
    QColor::fromRgbF(0.0237, 0.6319, 0.801),
    QColor::fromRgbF(0.0235, 0.6352, 0.798),
    QColor::fromRgbF(0.0233, 0.6384, 0.7948),
    QColor::fromRgbF(0.0231, 0.6415, 0.7916),
    QColor::fromRgbF(0.023, 0.6445, 0.7881),
    QColor::fromRgbF(0.0229, 0.6474, 0.7846),
    QColor::fromRgbF(0.0227, 0.6503, 0.781),
    QColor::fromRgbF(0.0227, 0.6531, 0.7773),
    QColor::fromRgbF(0.0232, 0.6558, 0.7735),
    QColor::fromRgbF(0.0238, 0.6585, 0.7696),
    QColor::fromRgbF(0.0246, 0.6611, 0.7656),
    QColor::fromRgbF(0.0263, 0.6637, 0.7615),
    QColor::fromRgbF(0.0282, 0.6663, 0.7574),
    QColor::fromRgbF(0.0306, 0.6688, 0.7532),
    QColor::fromRgbF(0.0338, 0.6712, 0.749),
    QColor::fromRgbF(0.0373, 0.6737, 0.7446),
    QColor::fromRgbF(0.0418, 0.6761, 0.7402),
    QColor::fromRgbF(0.0467, 0.6784, 0.7358),
    QColor::fromRgbF(0.0516, 0.6808, 0.7313),
    QColor::fromRgbF(0.0574, 0.6831, 0.7267),
    QColor::fromRgbF(0.0629, 0.6854, 0.7221),
    QColor::fromRgbF(0.0692, 0.6877, 0.7173),
    QColor::fromRgbF(0.0755, 0.6899, 0.7126),
    QColor::fromRgbF(0.082, 0.6921, 0.7078),
    QColor::fromRgbF(0.0889, 0.6943, 0.7029),
    QColor::fromRgbF(0.0956, 0.6965, 0.6979),
    QColor::fromRgbF(0.1031, 0.6986, 0.6929),
    QColor::fromRgbF(0.1104, 0.7007, 0.6878),
    QColor::fromRgbF(0.118, 0.7028, 0.6827),
    QColor::fromRgbF(0.1258, 0.7049, 0.6775),
    QColor::fromRgbF(0.1335, 0.7069, 0.6723),
    QColor::fromRgbF(0.1418, 0.7089, 0.6669),
    QColor::fromRgbF(0.1499, 0.7109, 0.6616),
    QColor::fromRgbF(0.1585, 0.7129, 0.6561),
    QColor::fromRgbF(0.1671, 0.7148, 0.6507),
    QColor::fromRgbF(0.1758, 0.7168, 0.6451),
    QColor::fromRgbF(0.1849, 0.7186, 0.6395),
    QColor::fromRgbF(0.1938, 0.7205, 0.6338),
    QColor::fromRgbF(0.2033, 0.7223, 0.6281),
    QColor::fromRgbF(0.2128, 0.7241, 0.6223),
    QColor::fromRgbF(0.2224, 0.7259, 0.6165),
    QColor::fromRgbF(0.2324, 0.7275, 0.6107),
    QColor::fromRgbF(0.2423, 0.7292, 0.6048),
    QColor::fromRgbF(0.2527, 0.7308, 0.5988),
    QColor::fromRgbF(0.2631, 0.7324, 0.5929),
    QColor::fromRgbF(0.2735, 0.7339, 0.5869),
    QColor::fromRgbF(0.2845, 0.7354, 0.5809),
    QColor::fromRgbF(0.2953, 0.7368, 0.5749),
    QColor::fromRgbF(0.3064, 0.7381, 0.5689),
    QColor::fromRgbF(0.3177, 0.7394, 0.563),
    QColor::fromRgbF(0.3289, 0.7406, 0.557),
    QColor::fromRgbF(0.3405, 0.7417, 0.5512),
    QColor::fromRgbF(0.352, 0.7428, 0.5453),
    QColor::fromRgbF(0.3635, 0.7438, 0.5396),
    QColor::fromRgbF(0.3753, 0.7446, 0.5339),
    QColor::fromRgbF(0.3869, 0.7454, 0.5283),
    QColor::fromRgbF(0.3986, 0.7461, 0.5229),
    QColor::fromRgbF(0.4103, 0.7467, 0.5175),
    QColor::fromRgbF(0.4218, 0.7473, 0.5123),
    QColor::fromRgbF(0.4334, 0.7477, 0.5072),
    QColor::fromRgbF(0.4447, 0.7482, 0.5021),
    QColor::fromRgbF(0.4561, 0.7485, 0.4972),
    QColor::fromRgbF(0.4672, 0.7487, 0.4924),
    QColor::fromRgbF(0.4783, 0.7489, 0.4877),
    QColor::fromRgbF(0.4892, 0.7491, 0.4831),
    QColor::fromRgbF(0.5, 0.7491, 0.4786),
    QColor::fromRgbF(0.5106, 0.7492, 0.4741),
    QColor::fromRgbF(0.5212, 0.7492, 0.4698),
    QColor::fromRgbF(0.5315, 0.7491, 0.4655),
    QColor::fromRgbF(0.5418, 0.749, 0.4613),
    QColor::fromRgbF(0.5519, 0.7489, 0.4571),
    QColor::fromRgbF(0.5619, 0.7487, 0.4531),
    QColor::fromRgbF(0.5718, 0.7485, 0.449),
    QColor::fromRgbF(0.5816, 0.7482, 0.4451),
    QColor::fromRgbF(0.5913, 0.7479, 0.4412),
    QColor::fromRgbF(0.6009, 0.7476, 0.4374),
    QColor::fromRgbF(0.6103, 0.7473, 0.4335),
    QColor::fromRgbF(0.6197, 0.7469, 0.4298),
    QColor::fromRgbF(0.629, 0.7465, 0.4261),
    QColor::fromRgbF(0.6382, 0.746, 0.4224),
    QColor::fromRgbF(0.6473, 0.7456, 0.4188),
    QColor::fromRgbF(0.6564, 0.7451, 0.4152),
    QColor::fromRgbF(0.6653, 0.7446, 0.4116),
    QColor::fromRgbF(0.6742, 0.7441, 0.4081),
    QColor::fromRgbF(0.683, 0.7435, 0.4046),
    QColor::fromRgbF(0.6918, 0.743, 0.4011),
    QColor::fromRgbF(0.7004, 0.7424, 0.3976),
    QColor::fromRgbF(0.7091, 0.7418, 0.3942),
    QColor::fromRgbF(0.7176, 0.7412, 0.3908),
    QColor::fromRgbF(0.7261, 0.7405, 0.3874),
    QColor::fromRgbF(0.7346, 0.7399, 0.384),
    QColor::fromRgbF(0.743, 0.7392, 0.3806),
    QColor::fromRgbF(0.7513, 0.7385, 0.3773),
    QColor::fromRgbF(0.7596, 0.7378, 0.3739),
    QColor::fromRgbF(0.7679, 0.7372, 0.3706),
    QColor::fromRgbF(0.7761, 0.7364, 0.3673),
    QColor::fromRgbF(0.7843, 0.7357, 0.3639),
    QColor::fromRgbF(0.7924, 0.735, 0.3606),
    QColor::fromRgbF(0.8005, 0.7343, 0.3573),
    QColor::fromRgbF(0.8085, 0.7336, 0.3539),
    QColor::fromRgbF(0.8166, 0.7329, 0.3506),
    QColor::fromRgbF(0.8246, 0.7322, 0.3472),
    QColor::fromRgbF(0.8325, 0.7315, 0.3438),
    QColor::fromRgbF(0.8405, 0.7308, 0.3404),
    QColor::fromRgbF(0.8484, 0.7301, 0.337),
    QColor::fromRgbF(0.8563, 0.7294, 0.3336),
    QColor::fromRgbF(0.8642, 0.7288, 0.33),
    QColor::fromRgbF(0.872, 0.7282, 0.3265),
    QColor::fromRgbF(0.8798, 0.7276, 0.3229),
    QColor::fromRgbF(0.8877, 0.7271, 0.3193),
    QColor::fromRgbF(0.8954, 0.7266, 0.3156),
    QColor::fromRgbF(0.9032, 0.7262, 0.3117),
    QColor::fromRgbF(0.911, 0.7259, 0.3078),
    QColor::fromRgbF(0.9187, 0.7256, 0.3038),
    QColor::fromRgbF(0.9264, 0.7256, 0.2996),
    QColor::fromRgbF(0.9341, 0.7256, 0.2953),
    QColor::fromRgbF(0.9417, 0.7259, 0.2907),
    QColor::fromRgbF(0.9493, 0.7264, 0.2859),
    QColor::fromRgbF(0.9567, 0.7273, 0.2808),
    QColor::fromRgbF(0.9639, 0.7285, 0.2754),
    QColor::fromRgbF(0.9708, 0.7303, 0.2696),
    QColor::fromRgbF(0.9773, 0.7326, 0.2634),
    QColor::fromRgbF(0.9831, 0.7355, 0.257),
    QColor::fromRgbF(0.9882, 0.739, 0.2504),
    QColor::fromRgbF(0.9922, 0.7431, 0.2437),
    QColor::fromRgbF(0.9952, 0.7476, 0.2373),
    QColor::fromRgbF(0.9973, 0.7524, 0.231),
    QColor::fromRgbF(0.9986, 0.7573, 0.2251),
    QColor::fromRgbF(0.9991, 0.7624, 0.2195),
    QColor::fromRgbF(0.999, 0.7675, 0.2141),
    QColor::fromRgbF(0.9985, 0.7726, 0.209),
    QColor::fromRgbF(0.9976, 0.7778, 0.2042),
    QColor::fromRgbF(0.9964, 0.7829, 0.1995),
    QColor::fromRgbF(0.995, 0.788, 0.1949),
    QColor::fromRgbF(0.9933, 0.7931, 0.1905),
    QColor::fromRgbF(0.9914, 0.7981, 0.1863),
    QColor::fromRgbF(0.9894, 0.8032, 0.1821),
    QColor::fromRgbF(0.9873, 0.8083, 0.178),
    QColor::fromRgbF(0.9851, 0.8133, 0.174),
    QColor::fromRgbF(0.9828, 0.8184, 0.17),
    QColor::fromRgbF(0.9805, 0.8235, 0.1661),
    QColor::fromRgbF(0.9782, 0.8286, 0.1622),
    QColor::fromRgbF(0.9759, 0.8337, 0.1583),
    QColor::fromRgbF(0.9736, 0.8389, 0.1544),
    QColor::fromRgbF(0.9713, 0.8441, 0.1505),
    QColor::fromRgbF(0.9692, 0.8494, 0.1465),
    QColor::fromRgbF(0.9672, 0.8548, 0.1425),
    QColor::fromRgbF(0.9654, 0.8603, 0.1385),
    QColor::fromRgbF(0.9638, 0.8659, 0.1343),
    QColor::fromRgbF(0.9623, 0.8716, 0.1301),
    QColor::fromRgbF(0.9611, 0.8774, 0.1258),
    QColor::fromRgbF(0.96, 0.8834, 0.1215),
    QColor::fromRgbF(0.9593, 0.8895, 0.1171),
    QColor::fromRgbF(0.9588, 0.8958, 0.1126),
    QColor::fromRgbF(0.9586, 0.9022, 0.1082),
    QColor::fromRgbF(0.9587, 0.9088, 0.1036),
    QColor::fromRgbF(0.9591, 0.9155, 0.099),
    QColor::fromRgbF(0.9599, 0.9225, 0.0944),
    QColor::fromRgbF(0.961, 0.9296, 0.0897),
    QColor::fromRgbF(0.9624, 0.9368, 0.085),
    QColor::fromRgbF(0.9641, 0.9443, 0.0802),
    QColor::fromRgbF(0.9662, 0.9518, 0.0753),
    QColor::fromRgbF(0.9685, 0.9595, 0.0703),
    QColor::fromRgbF(0.971, 0.9673, 0.0651),
    QColor::fromRgbF(0.9736, 0.9752, 0.0597),
    QColor::fromRgbF(0.9763, 0.9831, 0.0538)};

inline const std::vector<QColor> Heat{
    QColor::fromRgbF(0.0, 0.0, 1.0), QColor::fromRgbF(0.0, 1.0, 1.0),
    QColor::fromRgbF(0.0, 1.0, 0.0), QColor::fromRgbF(1.0, 1.0, 0.0),
    QColor::fromRgbF(1.0, 0.0, 0.0)};

inline const std::vector<QColor> Jet{
    QColor::fromRgbF(0.0, 0.0, 0.5), QColor::fromRgbF(0.0, 0.0, 1.0),
    QColor::fromRgbF(0.0, 0.5, 1.0), QColor::fromRgbF(0.0, 1.0, 1.0),
    QColor::fromRgbF(0.5, 1.0, 0.5), QColor::fromRgbF(1.0, 1.0, 0.0),
    QColor::fromRgbF(1.0, 0.5, 0.0), QColor::fromRgbF(1.0, 0.0, 0.0),
    QColor::fromRgbF(0.5, 0.0, 0.0)};

inline const std::vector<QColor> Turbo{
    QColor::fromRgbF(0.18995, 0.07176, 0.23217),
    QColor::fromRgbF(0.19483, 0.08339, 0.26149),
    QColor::fromRgbF(0.19956, 0.09498, 0.29024),
    QColor::fromRgbF(0.20415, 0.10652, 0.31844),
    QColor::fromRgbF(0.20860, 0.11802, 0.34607),
    QColor::fromRgbF(0.21291, 0.12947, 0.37314),
    QColor::fromRgbF(0.21708, 0.14087, 0.39964),
    QColor::fromRgbF(0.22111, 0.15223, 0.42558),
    QColor::fromRgbF(0.22500, 0.16354, 0.45096),
    QColor::fromRgbF(0.22875, 0.17481, 0.47578),
    QColor::fromRgbF(0.23236, 0.18603, 0.50004),
    QColor::fromRgbF(0.23582, 0.19720, 0.52373),
    QColor::fromRgbF(0.23915, 0.20833, 0.54686),
    QColor::fromRgbF(0.24234, 0.21941, 0.56942),
    QColor::fromRgbF(0.24539, 0.23044, 0.59142),
    QColor::fromRgbF(0.24830, 0.24143, 0.61286),
    QColor::fromRgbF(0.25107, 0.25237, 0.63374),
    QColor::fromRgbF(0.25369, 0.26327, 0.65406),
    QColor::fromRgbF(0.25618, 0.27412, 0.67381),
    QColor::fromRgbF(0.25853, 0.28492, 0.69300),
    QColor::fromRgbF(0.26074, 0.29568, 0.71162),
    QColor::fromRgbF(0.26280, 0.30639, 0.72968),
    QColor::fromRgbF(0.26473, 0.31706, 0.74718),
    QColor::fromRgbF(0.26652, 0.32768, 0.76412),
    QColor::fromRgbF(0.26816, 0.33825, 0.78050),
    QColor::fromRgbF(0.26967, 0.34878, 0.79631),
    QColor::fromRgbF(0.27103, 0.35926, 0.81156),
    QColor::fromRgbF(0.27226, 0.36970, 0.82624),
    QColor::fromRgbF(0.27334, 0.38008, 0.84037),
    QColor::fromRgbF(0.27429, 0.39043, 0.85393),
    QColor::fromRgbF(0.27509, 0.40072, 0.86692),
    QColor::fromRgbF(0.27576, 0.41097, 0.87936),
    QColor::fromRgbF(0.27628, 0.42118, 0.89123),
    QColor::fromRgbF(0.27667, 0.43134, 0.90254),
    QColor::fromRgbF(0.27691, 0.44145, 0.91328),
    QColor::fromRgbF(0.27701, 0.45152, 0.92347),
    QColor::fromRgbF(0.27698, 0.46153, 0.93309),
    QColor::fromRgbF(0.27680, 0.47151, 0.94214),
    QColor::fromRgbF(0.27648, 0.48144, 0.95064),
    QColor::fromRgbF(0.27603, 0.49132, 0.95857),
    QColor::fromRgbF(0.27543, 0.50115, 0.96594),
    QColor::fromRgbF(0.27469, 0.51094, 0.97275),
    QColor::fromRgbF(0.27381, 0.52069, 0.97899),
    QColor::fromRgbF(0.27273, 0.53040, 0.98461),
    QColor::fromRgbF(0.27106, 0.54015, 0.98930),
    QColor::fromRgbF(0.26878, 0.54995, 0.99303),
    QColor::fromRgbF(0.26592, 0.55979, 0.99583),
    QColor::fromRgbF(0.26252, 0.56967, 0.99773),
    QColor::fromRgbF(0.25862, 0.57958, 0.99876),
    QColor::fromRgbF(0.25425, 0.58950, 0.99896),
    QColor::fromRgbF(0.24946, 0.59943, 0.99835),
    QColor::fromRgbF(0.24427, 0.60937, 0.99697),
    QColor::fromRgbF(0.23874, 0.61931, 0.99485),
    QColor::fromRgbF(0.23288, 0.62923, 0.99202),
    QColor::fromRgbF(0.22676, 0.63913, 0.98851),
    QColor::fromRgbF(0.22039, 0.64901, 0.98436),
    QColor::fromRgbF(0.21382, 0.65886, 0.97959),
    QColor::fromRgbF(0.20708, 0.66866, 0.97423),
    QColor::fromRgbF(0.20021, 0.67842, 0.96833),
    QColor::fromRgbF(0.19326, 0.68812, 0.96190),
    QColor::fromRgbF(0.18625, 0.69775, 0.95498),
    QColor::fromRgbF(0.17923, 0.70732, 0.94761),
    QColor::fromRgbF(0.17223, 0.71680, 0.93981),
    QColor::fromRgbF(0.16529, 0.72620, 0.93161),
    QColor::fromRgbF(0.15844, 0.73551, 0.92305),
    QColor::fromRgbF(0.15173, 0.74472, 0.91416),
    QColor::fromRgbF(0.14519, 0.75381, 0.90496),
    QColor::fromRgbF(0.13886, 0.76279, 0.89550),
    QColor::fromRgbF(0.13278, 0.77165, 0.88580),
    QColor::fromRgbF(0.12698, 0.78037, 0.87590),
    QColor::fromRgbF(0.12151, 0.78896, 0.86581),
    QColor::fromRgbF(0.11639, 0.79740, 0.85559),
    QColor::fromRgbF(0.11167, 0.80569, 0.84525),
    QColor::fromRgbF(0.10738, 0.81381, 0.83484),
    QColor::fromRgbF(0.10357, 0.82177, 0.82437),
    QColor::fromRgbF(0.10026, 0.82955, 0.81389),
    QColor::fromRgbF(0.09750, 0.83714, 0.80342),
    QColor::fromRgbF(0.09532, 0.84455, 0.79299),
    QColor::fromRgbF(0.09377, 0.85175, 0.78264),
    QColor::fromRgbF(0.09287, 0.85875, 0.77240),
    QColor::fromRgbF(0.09267, 0.86554, 0.76230),
    QColor::fromRgbF(0.09320, 0.87211, 0.75237),
    QColor::fromRgbF(0.09451, 0.87844, 0.74265),
    QColor::fromRgbF(0.09662, 0.88454, 0.73316),
    QColor::fromRgbF(0.09958, 0.89040, 0.72393),
    QColor::fromRgbF(0.10342, 0.89600, 0.71500),
    QColor::fromRgbF(0.10815, 0.90142, 0.70599),
    QColor::fromRgbF(0.11374, 0.90673, 0.69651),
    QColor::fromRgbF(0.12014, 0.91193, 0.68660),
    QColor::fromRgbF(0.12733, 0.91701, 0.67627),
    QColor::fromRgbF(0.13526, 0.92197, 0.66556),
    QColor::fromRgbF(0.14391, 0.92680, 0.65448),
    QColor::fromRgbF(0.15323, 0.93151, 0.64308),
    QColor::fromRgbF(0.16319, 0.93609, 0.63137),
    QColor::fromRgbF(0.17377, 0.94053, 0.61938),
    QColor::fromRgbF(0.18491, 0.94484, 0.60713),
    QColor::fromRgbF(0.19659, 0.94901, 0.59466),
    QColor::fromRgbF(0.20877, 0.95304, 0.58199),
    QColor::fromRgbF(0.22142, 0.95692, 0.56914),
    QColor::fromRgbF(0.23449, 0.96065, 0.55614),
    QColor::fromRgbF(0.24797, 0.96423, 0.54303),
    QColor::fromRgbF(0.26180, 0.96765, 0.52981),
    QColor::fromRgbF(0.27597, 0.97092, 0.51653),
    QColor::fromRgbF(0.29042, 0.97403, 0.50321),
    QColor::fromRgbF(0.30513, 0.97697, 0.48987),
    QColor::fromRgbF(0.32006, 0.97974, 0.47654),
    QColor::fromRgbF(0.33517, 0.98234, 0.46325),
    QColor::fromRgbF(0.35043, 0.98477, 0.45002),
    QColor::fromRgbF(0.36581, 0.98702, 0.43688),
    QColor::fromRgbF(0.38127, 0.98909, 0.42386),
    QColor::fromRgbF(0.39678, 0.99098, 0.41098),
    QColor::fromRgbF(0.41229, 0.99268, 0.39826),
    QColor::fromRgbF(0.42778, 0.99419, 0.38575),
    QColor::fromRgbF(0.44321, 0.99551, 0.37345),
    QColor::fromRgbF(0.45854, 0.99663, 0.36140),
    QColor::fromRgbF(0.47375, 0.99755, 0.34963),
    QColor::fromRgbF(0.48879, 0.99828, 0.33816),
    QColor::fromRgbF(0.50362, 0.99879, 0.32701),
    QColor::fromRgbF(0.51822, 0.99910, 0.31622),
    QColor::fromRgbF(0.53255, 0.99919, 0.30581),
    QColor::fromRgbF(0.54658, 0.99907, 0.29581),
    QColor::fromRgbF(0.56026, 0.99873, 0.28623),
    QColor::fromRgbF(0.57357, 0.99817, 0.27712),
    QColor::fromRgbF(0.58646, 0.99739, 0.26849),
    QColor::fromRgbF(0.59891, 0.99638, 0.26038),
    QColor::fromRgbF(0.61088, 0.99514, 0.25280),
    QColor::fromRgbF(0.62233, 0.99366, 0.24579),
    QColor::fromRgbF(0.63323, 0.99195, 0.23937),
    QColor::fromRgbF(0.64362, 0.98999, 0.23356),
    QColor::fromRgbF(0.65394, 0.98775, 0.22835),
    QColor::fromRgbF(0.66428, 0.98524, 0.22370),
    QColor::fromRgbF(0.67462, 0.98246, 0.21960),
    QColor::fromRgbF(0.68494, 0.97941, 0.21602),
    QColor::fromRgbF(0.69525, 0.97610, 0.21294),
    QColor::fromRgbF(0.70553, 0.97255, 0.21032),
    QColor::fromRgbF(0.71577, 0.96875, 0.20815),
    QColor::fromRgbF(0.72596, 0.96470, 0.20640),
    QColor::fromRgbF(0.73610, 0.96043, 0.20504),
    QColor::fromRgbF(0.74617, 0.95593, 0.20406),
    QColor::fromRgbF(0.75617, 0.95121, 0.20343),
    QColor::fromRgbF(0.76608, 0.94627, 0.20311),
    QColor::fromRgbF(0.77591, 0.94113, 0.20310),
    QColor::fromRgbF(0.78563, 0.93579, 0.20336),
    QColor::fromRgbF(0.79524, 0.93025, 0.20386),
    QColor::fromRgbF(0.80473, 0.92452, 0.20459),
    QColor::fromRgbF(0.81410, 0.91861, 0.20552),
    QColor::fromRgbF(0.82333, 0.91253, 0.20663),
    QColor::fromRgbF(0.83241, 0.90627, 0.20788),
    QColor::fromRgbF(0.84133, 0.89986, 0.20926),
    QColor::fromRgbF(0.85010, 0.89328, 0.21074),
    QColor::fromRgbF(0.85868, 0.88655, 0.21230),
    QColor::fromRgbF(0.86709, 0.87968, 0.21391),
    QColor::fromRgbF(0.87530, 0.87267, 0.21555),
    QColor::fromRgbF(0.88331, 0.86553, 0.21719),
    QColor::fromRgbF(0.89112, 0.85826, 0.21880),
    QColor::fromRgbF(0.89870, 0.85087, 0.22038),
    QColor::fromRgbF(0.90605, 0.84337, 0.22188),
    QColor::fromRgbF(0.91317, 0.83576, 0.22328),
    QColor::fromRgbF(0.92004, 0.82806, 0.22456),
    QColor::fromRgbF(0.92666, 0.82025, 0.22570),
    QColor::fromRgbF(0.93301, 0.81236, 0.22667),
    QColor::fromRgbF(0.93909, 0.80439, 0.22744),
    QColor::fromRgbF(0.94489, 0.79634, 0.22800),
    QColor::fromRgbF(0.95039, 0.78823, 0.22831),
    QColor::fromRgbF(0.95560, 0.78005, 0.22836),
    QColor::fromRgbF(0.96049, 0.77181, 0.22811),
    QColor::fromRgbF(0.96507, 0.76352, 0.22754),
    QColor::fromRgbF(0.96931, 0.75519, 0.22663),
    QColor::fromRgbF(0.97323, 0.74682, 0.22536),
    QColor::fromRgbF(0.97679, 0.73842, 0.22369),
    QColor::fromRgbF(0.98000, 0.73000, 0.22161),
    QColor::fromRgbF(0.98289, 0.72140, 0.21918),
    QColor::fromRgbF(0.98549, 0.71250, 0.21650),
    QColor::fromRgbF(0.98781, 0.70330, 0.21358),
    QColor::fromRgbF(0.98986, 0.69382, 0.21043),
    QColor::fromRgbF(0.99163, 0.68408, 0.20706),
    QColor::fromRgbF(0.99314, 0.67408, 0.20348),
    QColor::fromRgbF(0.99438, 0.66386, 0.19971),
    QColor::fromRgbF(0.99535, 0.65341, 0.19577),
    QColor::fromRgbF(0.99607, 0.64277, 0.19165),
    QColor::fromRgbF(0.99654, 0.63193, 0.18738),
    QColor::fromRgbF(0.99675, 0.62093, 0.18297),
    QColor::fromRgbF(0.99672, 0.60977, 0.17842),
    QColor::fromRgbF(0.99644, 0.59846, 0.17376),
    QColor::fromRgbF(0.99593, 0.58703, 0.16899),
    QColor::fromRgbF(0.99517, 0.57549, 0.16412),
    QColor::fromRgbF(0.99419, 0.56386, 0.15918),
    QColor::fromRgbF(0.99297, 0.55214, 0.15417),
    QColor::fromRgbF(0.99153, 0.54036, 0.14910),
    QColor::fromRgbF(0.98987, 0.52854, 0.14398),
    QColor::fromRgbF(0.98799, 0.51667, 0.13883),
    QColor::fromRgbF(0.98590, 0.50479, 0.13367),
    QColor::fromRgbF(0.98360, 0.49291, 0.12849),
    QColor::fromRgbF(0.98108, 0.48104, 0.12332),
    QColor::fromRgbF(0.97837, 0.46920, 0.11817),
    QColor::fromRgbF(0.97545, 0.45740, 0.11305),
    QColor::fromRgbF(0.97234, 0.44565, 0.10797),
    QColor::fromRgbF(0.96904, 0.43399, 0.10294),
    QColor::fromRgbF(0.96555, 0.42241, 0.09798),
    QColor::fromRgbF(0.96187, 0.41093, 0.09310),
    QColor::fromRgbF(0.95801, 0.39958, 0.08831),
    QColor::fromRgbF(0.95398, 0.38836, 0.08362),
    QColor::fromRgbF(0.94977, 0.37729, 0.07905),
    QColor::fromRgbF(0.94538, 0.36638, 0.07461),
    QColor::fromRgbF(0.94084, 0.35566, 0.07031),
    QColor::fromRgbF(0.93612, 0.34513, 0.06616),
    QColor::fromRgbF(0.93125, 0.33482, 0.06218),
    QColor::fromRgbF(0.92623, 0.32473, 0.05837),
    QColor::fromRgbF(0.92105, 0.31489, 0.05475),
    QColor::fromRgbF(0.91572, 0.30530, 0.05134),
    QColor::fromRgbF(0.91024, 0.29599, 0.04814),
    QColor::fromRgbF(0.90463, 0.28696, 0.04516),
    QColor::fromRgbF(0.89888, 0.27824, 0.04243),
    QColor::fromRgbF(0.89298, 0.26981, 0.03993),
    QColor::fromRgbF(0.88691, 0.26152, 0.03753),
    QColor::fromRgbF(0.88066, 0.25334, 0.03521),
    QColor::fromRgbF(0.87422, 0.24526, 0.03297),
    QColor::fromRgbF(0.86760, 0.23730, 0.03082),
    QColor::fromRgbF(0.86079, 0.22945, 0.02875),
    QColor::fromRgbF(0.85380, 0.22170, 0.02677),
    QColor::fromRgbF(0.84662, 0.21407, 0.02487),
    QColor::fromRgbF(0.83926, 0.20654, 0.02305),
    QColor::fromRgbF(0.83172, 0.19912, 0.02131),
    QColor::fromRgbF(0.82399, 0.19182, 0.01966),
    QColor::fromRgbF(0.81608, 0.18462, 0.01809),
    QColor::fromRgbF(0.80799, 0.17753, 0.01660),
    QColor::fromRgbF(0.79971, 0.17055, 0.01520),
    QColor::fromRgbF(0.79125, 0.16368, 0.01387),
    QColor::fromRgbF(0.78260, 0.15693, 0.01264),
    QColor::fromRgbF(0.77377, 0.15028, 0.01148),
    QColor::fromRgbF(0.76476, 0.14374, 0.01041),
    QColor::fromRgbF(0.75556, 0.13731, 0.00942),
    QColor::fromRgbF(0.74617, 0.13098, 0.00851),
    QColor::fromRgbF(0.73661, 0.12477, 0.00769),
    QColor::fromRgbF(0.72686, 0.11867, 0.00695),
    QColor::fromRgbF(0.71692, 0.11268, 0.00629),
    QColor::fromRgbF(0.70680, 0.10680, 0.00571),
    QColor::fromRgbF(0.69650, 0.10102, 0.00522),
    QColor::fromRgbF(0.68602, 0.09536, 0.00481),
    QColor::fromRgbF(0.67535, 0.08980, 0.00449),
    QColor::fromRgbF(0.66449, 0.08436, 0.00424),
    QColor::fromRgbF(0.65345, 0.07902, 0.00408),
    QColor::fromRgbF(0.64223, 0.07380, 0.00401),
    QColor::fromRgbF(0.63082, 0.06868, 0.00401),
    QColor::fromRgbF(0.61923, 0.06367, 0.00410),
    QColor::fromRgbF(0.60746, 0.05878, 0.00427),
    QColor::fromRgbF(0.59550, 0.05399, 0.00453),
    QColor::fromRgbF(0.58336, 0.04931, 0.00486),
    QColor::fromRgbF(0.57103, 0.04474, 0.00529),
    QColor::fromRgbF(0.55852, 0.04028, 0.00579),
    QColor::fromRgbF(0.54583, 0.03593, 0.00638),
    QColor::fromRgbF(0.53295, 0.03169, 0.00705),
    QColor::fromRgbF(0.51989, 0.02756, 0.00780),
    QColor::fromRgbF(0.50664, 0.02354, 0.00863),
    QColor::fromRgbF(0.49321, 0.01963, 0.00955),
    QColor::fromRgbF(0.47960, 0.01583, 0.01055)};

inline const std::vector<QColor> Magma{
    QColor::fromRgbF(0.001462, 0.000466, 0.013866),
    QColor::fromRgbF(0.002258, 0.001295, 0.018331),
    QColor::fromRgbF(0.003279, 0.002305, 0.023708),
    QColor::fromRgbF(0.004512, 0.003490, 0.029965),
    QColor::fromRgbF(0.005950, 0.004843, 0.037130),
    QColor::fromRgbF(0.007588, 0.006356, 0.044973),
    QColor::fromRgbF(0.009426, 0.008022, 0.052844),
    QColor::fromRgbF(0.011465, 0.009828, 0.060750),
    QColor::fromRgbF(0.013708, 0.011771, 0.068667),
    QColor::fromRgbF(0.016156, 0.013840, 0.076603),
    QColor::fromRgbF(0.018815, 0.016026, 0.084584),
    QColor::fromRgbF(0.021692, 0.018320, 0.092610),
    QColor::fromRgbF(0.024792, 0.020715, 0.100676),
    QColor::fromRgbF(0.028123, 0.023201, 0.108787),
    QColor::fromRgbF(0.031696, 0.025765, 0.116965),
    QColor::fromRgbF(0.035520, 0.028397, 0.125209),
    QColor::fromRgbF(0.039608, 0.031090, 0.133515),
    QColor::fromRgbF(0.043830, 0.033830, 0.141886),
    QColor::fromRgbF(0.048062, 0.036607, 0.150327),
    QColor::fromRgbF(0.052320, 0.039407, 0.158841),
    QColor::fromRgbF(0.056615, 0.042160, 0.167446),
    QColor::fromRgbF(0.060949, 0.044794, 0.176129),
    QColor::fromRgbF(0.065330, 0.047318, 0.184892),
    QColor::fromRgbF(0.069764, 0.049726, 0.193735),
    QColor::fromRgbF(0.074257, 0.052017, 0.202660),
    QColor::fromRgbF(0.078815, 0.054184, 0.211667),
    QColor::fromRgbF(0.083446, 0.056225, 0.220755),
    QColor::fromRgbF(0.088155, 0.058133, 0.229922),
    QColor::fromRgbF(0.092949, 0.059904, 0.239164),
    QColor::fromRgbF(0.097833, 0.061531, 0.248477),
    QColor::fromRgbF(0.102815, 0.063010, 0.257854),
    QColor::fromRgbF(0.107899, 0.064335, 0.267289),
    QColor::fromRgbF(0.113094, 0.065492, 0.276784),
    QColor::fromRgbF(0.118405, 0.066479, 0.286321),
    QColor::fromRgbF(0.123833, 0.067295, 0.295879),
    QColor::fromRgbF(0.129380, 0.067935, 0.305443),
    QColor::fromRgbF(0.135053, 0.068391, 0.315000),
    QColor::fromRgbF(0.140858, 0.068654, 0.324538),
    QColor::fromRgbF(0.146785, 0.068738, 0.334011),
    QColor::fromRgbF(0.152839, 0.068637, 0.343404),
    QColor::fromRgbF(0.159018, 0.068354, 0.352688),
    QColor::fromRgbF(0.165308, 0.067911, 0.361816),
    QColor::fromRgbF(0.171713, 0.067305, 0.370771),
    QColor::fromRgbF(0.178212, 0.066576, 0.379497),
    QColor::fromRgbF(0.184801, 0.065732, 0.387973),
    QColor::fromRgbF(0.191460, 0.064818, 0.396152),
    QColor::fromRgbF(0.198177, 0.063862, 0.404009),
    QColor::fromRgbF(0.204935, 0.062907, 0.411514),
    QColor::fromRgbF(0.211718, 0.061992, 0.418647),
    QColor::fromRgbF(0.218512, 0.061158, 0.425392),
    QColor::fromRgbF(0.225302, 0.060445, 0.431742),
    QColor::fromRgbF(0.232077, 0.059889, 0.437695),
    QColor::fromRgbF(0.238826, 0.059517, 0.443256),
    QColor::fromRgbF(0.245543, 0.059352, 0.448436),
    QColor::fromRgbF(0.252220, 0.059415, 0.453248),
    QColor::fromRgbF(0.258857, 0.059706, 0.457710),
    QColor::fromRgbF(0.265447, 0.060237, 0.461840),
    QColor::fromRgbF(0.271994, 0.060994, 0.465660),
    QColor::fromRgbF(0.278493, 0.061978, 0.469190),
    QColor::fromRgbF(0.284951, 0.063168, 0.472451),
    QColor::fromRgbF(0.291366, 0.064553, 0.475462),
    QColor::fromRgbF(0.297740, 0.066117, 0.478243),
    QColor::fromRgbF(0.304081, 0.067835, 0.480812),
    QColor::fromRgbF(0.310382, 0.069702, 0.483186),
    QColor::fromRgbF(0.316654, 0.071690, 0.485380),
    QColor::fromRgbF(0.322899, 0.073782, 0.487408),
    QColor::fromRgbF(0.329114, 0.075972, 0.489287),
    QColor::fromRgbF(0.335308, 0.078236, 0.491024),
    QColor::fromRgbF(0.341482, 0.080564, 0.492631),
    QColor::fromRgbF(0.347636, 0.082946, 0.494121),
    QColor::fromRgbF(0.353773, 0.085373, 0.495501),
    QColor::fromRgbF(0.359898, 0.087831, 0.496778),
    QColor::fromRgbF(0.366012, 0.090314, 0.497960),
    QColor::fromRgbF(0.372116, 0.092816, 0.499053),
    QColor::fromRgbF(0.378211, 0.095332, 0.500067),
    QColor::fromRgbF(0.384299, 0.097855, 0.501002),
    QColor::fromRgbF(0.390384, 0.100379, 0.501864),
    QColor::fromRgbF(0.396467, 0.102902, 0.502658),
    QColor::fromRgbF(0.402548, 0.105420, 0.503386),
    QColor::fromRgbF(0.408629, 0.107930, 0.504052),
    QColor::fromRgbF(0.414709, 0.110431, 0.504662),
    QColor::fromRgbF(0.420791, 0.112920, 0.505215),
    QColor::fromRgbF(0.426877, 0.115395, 0.505714),
    QColor::fromRgbF(0.432967, 0.117855, 0.506160),
    QColor::fromRgbF(0.439062, 0.120298, 0.506555),
    QColor::fromRgbF(0.445163, 0.122724, 0.506901),
    QColor::fromRgbF(0.451271, 0.125132, 0.507198),
    QColor::fromRgbF(0.457386, 0.127522, 0.507448),
    QColor::fromRgbF(0.463508, 0.129893, 0.507652),
    QColor::fromRgbF(0.469640, 0.132245, 0.507809),
    QColor::fromRgbF(0.475780, 0.134577, 0.507921),
    QColor::fromRgbF(0.481929, 0.136891, 0.507989),
    QColor::fromRgbF(0.488088, 0.139186, 0.508011),
    QColor::fromRgbF(0.494258, 0.141462, 0.507988),
    QColor::fromRgbF(0.500438, 0.143719, 0.507920),
    QColor::fromRgbF(0.506629, 0.145958, 0.507806),
    QColor::fromRgbF(0.512831, 0.148179, 0.507648),
    QColor::fromRgbF(0.519045, 0.150383, 0.507443),
    QColor::fromRgbF(0.525270, 0.152569, 0.507192),
    QColor::fromRgbF(0.531507, 0.154739, 0.506895),
    QColor::fromRgbF(0.537755, 0.156894, 0.506551),
    QColor::fromRgbF(0.544015, 0.159033, 0.506159),
    QColor::fromRgbF(0.550287, 0.161158, 0.505719),
    QColor::fromRgbF(0.556571, 0.163269, 0.505230),
    QColor::fromRgbF(0.562866, 0.165368, 0.504692),
    QColor::fromRgbF(0.569172, 0.167454, 0.504105),
    QColor::fromRgbF(0.575490, 0.169530, 0.503466),
    QColor::fromRgbF(0.581819, 0.171596, 0.502777),
    QColor::fromRgbF(0.588158, 0.173652, 0.502035),
    QColor::fromRgbF(0.594508, 0.175701, 0.501241),
    QColor::fromRgbF(0.600868, 0.177743, 0.500394),
    QColor::fromRgbF(0.607238, 0.179779, 0.499492),
    QColor::fromRgbF(0.613617, 0.181811, 0.498536),
    QColor::fromRgbF(0.620005, 0.183840, 0.497524),
    QColor::fromRgbF(0.626401, 0.185867, 0.496456),
    QColor::fromRgbF(0.632805, 0.187893, 0.495332),
    QColor::fromRgbF(0.639216, 0.189921, 0.494150),
    QColor::fromRgbF(0.645633, 0.191952, 0.492910),
    QColor::fromRgbF(0.652056, 0.193986, 0.491611),
    QColor::fromRgbF(0.658483, 0.196027, 0.490253),
    QColor::fromRgbF(0.664915, 0.198075, 0.488836),
    QColor::fromRgbF(0.671349, 0.200133, 0.487358),
    QColor::fromRgbF(0.677786, 0.202203, 0.485819),
    QColor::fromRgbF(0.684224, 0.204286, 0.484219),
    QColor::fromRgbF(0.690661, 0.206384, 0.482558),
    QColor::fromRgbF(0.697098, 0.208501, 0.480835),
    QColor::fromRgbF(0.703532, 0.210638, 0.479049),
    QColor::fromRgbF(0.709962, 0.212797, 0.477201),
    QColor::fromRgbF(0.716387, 0.214982, 0.475290),
    QColor::fromRgbF(0.722805, 0.217194, 0.473316),
    QColor::fromRgbF(0.729216, 0.219437, 0.471279),
    QColor::fromRgbF(0.735616, 0.221713, 0.469180),
    QColor::fromRgbF(0.742004, 0.224025, 0.467018),
    QColor::fromRgbF(0.748378, 0.226377, 0.464794),
    QColor::fromRgbF(0.754737, 0.228772, 0.462509),
    QColor::fromRgbF(0.761077, 0.231214, 0.460162),
    QColor::fromRgbF(0.767398, 0.233705, 0.457755),
    QColor::fromRgbF(0.773695, 0.236249, 0.455289),
    QColor::fromRgbF(0.779968, 0.238851, 0.452765),
    QColor::fromRgbF(0.786212, 0.241514, 0.450184),
    QColor::fromRgbF(0.792427, 0.244242, 0.447543),
    QColor::fromRgbF(0.798608, 0.247040, 0.444848),
    QColor::fromRgbF(0.804752, 0.249911, 0.442102),
    QColor::fromRgbF(0.810855, 0.252861, 0.439305),
    QColor::fromRgbF(0.816914, 0.255895, 0.436461),
    QColor::fromRgbF(0.822926, 0.259016, 0.433573),
    QColor::fromRgbF(0.828886, 0.262229, 0.430644),
    QColor::fromRgbF(0.834791, 0.265540, 0.427671),
    QColor::fromRgbF(0.840636, 0.268953, 0.424666),
    QColor::fromRgbF(0.846416, 0.272473, 0.421631),
    QColor::fromRgbF(0.852126, 0.276106, 0.418573),
    QColor::fromRgbF(0.857763, 0.279857, 0.415496),
    QColor::fromRgbF(0.863320, 0.283729, 0.412403),
    QColor::fromRgbF(0.868793, 0.287728, 0.409303),
    QColor::fromRgbF(0.874176, 0.291859, 0.406205),
    QColor::fromRgbF(0.879464, 0.296125, 0.403118),
    QColor::fromRgbF(0.884651, 0.300530, 0.400047),
    QColor::fromRgbF(0.889731, 0.305079, 0.397002),
    QColor::fromRgbF(0.894700, 0.309773, 0.393995),
    QColor::fromRgbF(0.899552, 0.314616, 0.391037),
    QColor::fromRgbF(0.904281, 0.319610, 0.388137),
    QColor::fromRgbF(0.908884, 0.324755, 0.385308),
    QColor::fromRgbF(0.913354, 0.330052, 0.382563),
    QColor::fromRgbF(0.917689, 0.335500, 0.379915),
    QColor::fromRgbF(0.921884, 0.341098, 0.377376),
    QColor::fromRgbF(0.925937, 0.346844, 0.374959),
    QColor::fromRgbF(0.929845, 0.352734, 0.372677),
    QColor::fromRgbF(0.933606, 0.358764, 0.370541),
    QColor::fromRgbF(0.937221, 0.364929, 0.368567),
    QColor::fromRgbF(0.940687, 0.371224, 0.366762),
    QColor::fromRgbF(0.944006, 0.377643, 0.365136),
    QColor::fromRgbF(0.947180, 0.384178, 0.363701),
    QColor::fromRgbF(0.950210, 0.390820, 0.362468),
    QColor::fromRgbF(0.953099, 0.397563, 0.361438),
    QColor::fromRgbF(0.955849, 0.404400, 0.360619),
    QColor::fromRgbF(0.958464, 0.411324, 0.360014),
    QColor::fromRgbF(0.960949, 0.418323, 0.359630),
    QColor::fromRgbF(0.963310, 0.425390, 0.359469),
    QColor::fromRgbF(0.965549, 0.432519, 0.359529),
    QColor::fromRgbF(0.967671, 0.439703, 0.359810),
    QColor::fromRgbF(0.969680, 0.446936, 0.360311),
    QColor::fromRgbF(0.971582, 0.454210, 0.361030),
    QColor::fromRgbF(0.973381, 0.461520, 0.361965),
    QColor::fromRgbF(0.975082, 0.468861, 0.363111),
    QColor::fromRgbF(0.976690, 0.476226, 0.364466),
    QColor::fromRgbF(0.978210, 0.483612, 0.366025),
    QColor::fromRgbF(0.979645, 0.491014, 0.367783),
    QColor::fromRgbF(0.981000, 0.498428, 0.369734),
    QColor::fromRgbF(0.982279, 0.505851, 0.371874),
    QColor::fromRgbF(0.983485, 0.513280, 0.374198),
    QColor::fromRgbF(0.984622, 0.520713, 0.376698),
    QColor::fromRgbF(0.985693, 0.528148, 0.379371),
    QColor::fromRgbF(0.986700, 0.535582, 0.382210),
    QColor::fromRgbF(0.987646, 0.543015, 0.385210),
    QColor::fromRgbF(0.988533, 0.550446, 0.388365),
    QColor::fromRgbF(0.989363, 0.557873, 0.391671),
    QColor::fromRgbF(0.990138, 0.565296, 0.395122),
    QColor::fromRgbF(0.990871, 0.572706, 0.398714),
    QColor::fromRgbF(0.991558, 0.580107, 0.402441),
    QColor::fromRgbF(0.992196, 0.587502, 0.406299),
    QColor::fromRgbF(0.992785, 0.594891, 0.410283),
    QColor::fromRgbF(0.993326, 0.602275, 0.414390),
    QColor::fromRgbF(0.993834, 0.609644, 0.418613),
    QColor::fromRgbF(0.994309, 0.616999, 0.422950),
    QColor::fromRgbF(0.994738, 0.624350, 0.427397),
    QColor::fromRgbF(0.995122, 0.631696, 0.431951),
    QColor::fromRgbF(0.995480, 0.639027, 0.436607),
    QColor::fromRgbF(0.995810, 0.646344, 0.441361),
    QColor::fromRgbF(0.996096, 0.653659, 0.446213),
    QColor::fromRgbF(0.996341, 0.660969, 0.451160),
    QColor::fromRgbF(0.996580, 0.668256, 0.456192),
    QColor::fromRgbF(0.996775, 0.675541, 0.461314),
    QColor::fromRgbF(0.996925, 0.682828, 0.466526),
    QColor::fromRgbF(0.997077, 0.690088, 0.471811),
    QColor::fromRgbF(0.997186, 0.697349, 0.477182),
    QColor::fromRgbF(0.997254, 0.704611, 0.482635),
    QColor::fromRgbF(0.997325, 0.711848, 0.488154),
    QColor::fromRgbF(0.997351, 0.719089, 0.493755),
    QColor::fromRgbF(0.997351, 0.726324, 0.499428),
    QColor::fromRgbF(0.997341, 0.733545, 0.505167),
    QColor::fromRgbF(0.997285, 0.740772, 0.510983),
    QColor::fromRgbF(0.997228, 0.747981, 0.516859),
    QColor::fromRgbF(0.997138, 0.755190, 0.522806),
    QColor::fromRgbF(0.997019, 0.762398, 0.528821),
    QColor::fromRgbF(0.996898, 0.769591, 0.534892),
    QColor::fromRgbF(0.996727, 0.776795, 0.541039),
    QColor::fromRgbF(0.996571, 0.783977, 0.547233),
    QColor::fromRgbF(0.996369, 0.791167, 0.553499),
    QColor::fromRgbF(0.996162, 0.798348, 0.559820),
    QColor::fromRgbF(0.995932, 0.805527, 0.566202),
    QColor::fromRgbF(0.995680, 0.812706, 0.572645),
    QColor::fromRgbF(0.995424, 0.819875, 0.579140),
    QColor::fromRgbF(0.995131, 0.827052, 0.585701),
    QColor::fromRgbF(0.994851, 0.834213, 0.592307),
    QColor::fromRgbF(0.994524, 0.841387, 0.598983),
    QColor::fromRgbF(0.994222, 0.848540, 0.605696),
    QColor::fromRgbF(0.993866, 0.855711, 0.612482),
    QColor::fromRgbF(0.993545, 0.862859, 0.619299),
    QColor::fromRgbF(0.993170, 0.870024, 0.626189),
    QColor::fromRgbF(0.992831, 0.877168, 0.633109),
    QColor::fromRgbF(0.992440, 0.884330, 0.640099),
    QColor::fromRgbF(0.992089, 0.891470, 0.647116),
    QColor::fromRgbF(0.991688, 0.898627, 0.654202),
    QColor::fromRgbF(0.991332, 0.905763, 0.661309),
    QColor::fromRgbF(0.990930, 0.912915, 0.668481),
    QColor::fromRgbF(0.990570, 0.920049, 0.675675),
    QColor::fromRgbF(0.990175, 0.927196, 0.682926),
    QColor::fromRgbF(0.989815, 0.934329, 0.690198),
    QColor::fromRgbF(0.989434, 0.941470, 0.697519),
    QColor::fromRgbF(0.989077, 0.948604, 0.704863),
    QColor::fromRgbF(0.988717, 0.955742, 0.712242),
    QColor::fromRgbF(0.988367, 0.962878, 0.719649),
    QColor::fromRgbF(0.988033, 0.970012, 0.727077),
    QColor::fromRgbF(0.987691, 0.977154, 0.734536),
    QColor::fromRgbF(0.987387, 0.984288, 0.742002),
    QColor::fromRgbF(0.987053, 0.991438, 0.749504)};

inline const std::vector<QColor> Inferno{
    QColor::fromRgbF(0.001462, 0.000466, 0.013866),
    QColor::fromRgbF(0.002267, 0.001270, 0.018570),
    QColor::fromRgbF(0.003299, 0.002249, 0.024239),
    QColor::fromRgbF(0.004547, 0.003392, 0.030909),
    QColor::fromRgbF(0.006006, 0.004692, 0.038558),
    QColor::fromRgbF(0.007676, 0.006136, 0.046836),
    QColor::fromRgbF(0.009561, 0.007713, 0.055143),
    QColor::fromRgbF(0.011663, 0.009417, 0.063460),
    QColor::fromRgbF(0.013995, 0.011225, 0.071862),
    QColor::fromRgbF(0.016561, 0.013136, 0.080282),
    QColor::fromRgbF(0.019373, 0.015133, 0.088767),
    QColor::fromRgbF(0.022447, 0.017199, 0.097327),
    QColor::fromRgbF(0.025793, 0.019331, 0.105930),
    QColor::fromRgbF(0.029432, 0.021503, 0.114621),
    QColor::fromRgbF(0.033385, 0.023702, 0.123397),
    QColor::fromRgbF(0.037668, 0.025921, 0.132232),
    QColor::fromRgbF(0.042253, 0.028139, 0.141141),
    QColor::fromRgbF(0.046915, 0.030324, 0.150164),
    QColor::fromRgbF(0.051644, 0.032474, 0.159254),
    QColor::fromRgbF(0.056449, 0.034569, 0.168414),
    QColor::fromRgbF(0.061340, 0.036590, 0.177642),
    QColor::fromRgbF(0.066331, 0.038504, 0.186962),
    QColor::fromRgbF(0.071429, 0.040294, 0.196354),
    QColor::fromRgbF(0.076637, 0.041905, 0.205799),
    QColor::fromRgbF(0.081962, 0.043328, 0.215289),
    QColor::fromRgbF(0.087411, 0.044556, 0.224813),
    QColor::fromRgbF(0.092990, 0.045583, 0.234358),
    QColor::fromRgbF(0.098702, 0.046402, 0.243904),
    QColor::fromRgbF(0.104551, 0.047008, 0.253430),
    QColor::fromRgbF(0.110536, 0.047399, 0.262912),
    QColor::fromRgbF(0.116656, 0.047574, 0.272321),
    QColor::fromRgbF(0.122908, 0.047536, 0.281624),
    QColor::fromRgbF(0.129285, 0.047293, 0.290788),
    QColor::fromRgbF(0.135778, 0.046856, 0.299776),
    QColor::fromRgbF(0.142378, 0.046242, 0.308553),
    QColor::fromRgbF(0.149073, 0.045468, 0.317085),
    QColor::fromRgbF(0.155850, 0.044559, 0.325338),
    QColor::fromRgbF(0.162689, 0.043554, 0.333277),
    QColor::fromRgbF(0.169575, 0.042489, 0.340874),
    QColor::fromRgbF(0.176493, 0.041402, 0.348111),
    QColor::fromRgbF(0.183429, 0.040329, 0.354971),
    QColor::fromRgbF(0.190367, 0.039309, 0.361447),
    QColor::fromRgbF(0.197297, 0.038400, 0.367535),
    QColor::fromRgbF(0.204209, 0.037632, 0.373238),
    QColor::fromRgbF(0.211095, 0.037030, 0.378563),
    QColor::fromRgbF(0.217949, 0.036615, 0.383522),
    QColor::fromRgbF(0.224763, 0.036405, 0.388129),
    QColor::fromRgbF(0.231538, 0.036405, 0.392400),
    QColor::fromRgbF(0.238273, 0.036621, 0.396353),
    QColor::fromRgbF(0.244967, 0.037055, 0.400007),
    QColor::fromRgbF(0.251620, 0.037705, 0.403378),
    QColor::fromRgbF(0.258234, 0.038571, 0.406485),
    QColor::fromRgbF(0.264810, 0.039647, 0.409345),
    QColor::fromRgbF(0.271347, 0.040922, 0.411976),
    QColor::fromRgbF(0.277850, 0.042353, 0.414392),
    QColor::fromRgbF(0.284321, 0.043933, 0.416608),
    QColor::fromRgbF(0.290763, 0.045644, 0.418637),
    QColor::fromRgbF(0.297178, 0.047470, 0.420491),
    QColor::fromRgbF(0.303568, 0.049396, 0.422182),
    QColor::fromRgbF(0.309935, 0.051407, 0.423721),
    QColor::fromRgbF(0.316282, 0.053490, 0.425116),
    QColor::fromRgbF(0.322610, 0.055634, 0.426377),
    QColor::fromRgbF(0.328921, 0.057827, 0.427511),
    QColor::fromRgbF(0.335217, 0.060060, 0.428524),
    QColor::fromRgbF(0.341500, 0.062325, 0.429425),
    QColor::fromRgbF(0.347771, 0.064616, 0.430217),
    QColor::fromRgbF(0.354032, 0.066925, 0.430906),
    QColor::fromRgbF(0.360284, 0.069247, 0.431497),
    QColor::fromRgbF(0.366529, 0.071579, 0.431994),
    QColor::fromRgbF(0.372768, 0.073915, 0.432400),
    QColor::fromRgbF(0.379001, 0.076253, 0.432719),
    QColor::fromRgbF(0.385228, 0.078591, 0.432955),
    QColor::fromRgbF(0.391453, 0.080927, 0.433109),
    QColor::fromRgbF(0.397674, 0.083257, 0.433183),
    QColor::fromRgbF(0.403894, 0.085580, 0.433179),
    QColor::fromRgbF(0.410113, 0.087896, 0.433098),
    QColor::fromRgbF(0.416331, 0.090203, 0.432943),
    QColor::fromRgbF(0.422549, 0.092501, 0.432714),
    QColor::fromRgbF(0.428768, 0.094790, 0.432412),
    QColor::fromRgbF(0.434987, 0.097069, 0.432039),
    QColor::fromRgbF(0.441207, 0.099338, 0.431594),
    QColor::fromRgbF(0.447428, 0.101597, 0.431080),
    QColor::fromRgbF(0.453651, 0.103848, 0.430498),
    QColor::fromRgbF(0.459875, 0.106089, 0.429846),
    QColor::fromRgbF(0.466100, 0.108322, 0.429125),
    QColor::fromRgbF(0.472328, 0.110547, 0.428334),
    QColor::fromRgbF(0.478558, 0.112764, 0.427475),
    QColor::fromRgbF(0.484789, 0.114974, 0.426548),
    QColor::fromRgbF(0.491022, 0.117179, 0.425552),
    QColor::fromRgbF(0.497257, 0.119379, 0.424488),
    QColor::fromRgbF(0.503493, 0.121575, 0.423356),
    QColor::fromRgbF(0.509730, 0.123769, 0.422156),
    QColor::fromRgbF(0.515967, 0.125960, 0.420887),
    QColor::fromRgbF(0.522206, 0.128150, 0.419549),
    QColor::fromRgbF(0.528444, 0.130341, 0.418142),
    QColor::fromRgbF(0.534683, 0.132534, 0.416667),
    QColor::fromRgbF(0.540920, 0.134729, 0.415123),
    QColor::fromRgbF(0.547157, 0.136929, 0.413511),
    QColor::fromRgbF(0.553392, 0.139134, 0.411829),
    QColor::fromRgbF(0.559624, 0.141346, 0.410078),
    QColor::fromRgbF(0.565854, 0.143567, 0.408258),
    QColor::fromRgbF(0.572081, 0.145797, 0.406369),
    QColor::fromRgbF(0.578304, 0.148039, 0.404411),
    QColor::fromRgbF(0.584521, 0.150294, 0.402385),
    QColor::fromRgbF(0.590734, 0.152563, 0.400290),
    QColor::fromRgbF(0.596940, 0.154848, 0.398125),
    QColor::fromRgbF(0.603139, 0.157151, 0.395891),
    QColor::fromRgbF(0.609330, 0.159474, 0.393589),
    QColor::fromRgbF(0.615513, 0.161817, 0.391219),
    QColor::fromRgbF(0.621685, 0.164184, 0.388781),
    QColor::fromRgbF(0.627847, 0.166575, 0.386276),
    QColor::fromRgbF(0.633998, 0.168992, 0.383704),
    QColor::fromRgbF(0.640135, 0.171438, 0.381065),
    QColor::fromRgbF(0.646260, 0.173914, 0.378359),
    QColor::fromRgbF(0.652369, 0.176421, 0.375586),
    QColor::fromRgbF(0.658463, 0.178962, 0.372748),
    QColor::fromRgbF(0.664540, 0.181539, 0.369846),
    QColor::fromRgbF(0.670599, 0.184153, 0.366879),
    QColor::fromRgbF(0.676638, 0.186807, 0.363849),
    QColor::fromRgbF(0.682656, 0.189501, 0.360757),
    QColor::fromRgbF(0.688653, 0.192239, 0.357603),
    QColor::fromRgbF(0.694627, 0.195021, 0.354388),
    QColor::fromRgbF(0.700576, 0.197851, 0.351113),
    QColor::fromRgbF(0.706500, 0.200728, 0.347777),
    QColor::fromRgbF(0.712396, 0.203656, 0.344383),
    QColor::fromRgbF(0.718264, 0.206636, 0.340931),
    QColor::fromRgbF(0.724103, 0.209670, 0.337424),
    QColor::fromRgbF(0.729909, 0.212759, 0.333861),
    QColor::fromRgbF(0.735683, 0.215906, 0.330245),
    QColor::fromRgbF(0.741423, 0.219112, 0.326576),
    QColor::fromRgbF(0.747127, 0.222378, 0.322856),
    QColor::fromRgbF(0.752794, 0.225706, 0.319085),
    QColor::fromRgbF(0.758422, 0.229097, 0.315266),
    QColor::fromRgbF(0.764010, 0.232554, 0.311399),
    QColor::fromRgbF(0.769556, 0.236077, 0.307485),
    QColor::fromRgbF(0.775059, 0.239667, 0.303526),
    QColor::fromRgbF(0.780517, 0.243327, 0.299523),
    QColor::fromRgbF(0.785929, 0.247056, 0.295477),
    QColor::fromRgbF(0.791293, 0.250856, 0.291390),
    QColor::fromRgbF(0.796607, 0.254728, 0.287264),
    QColor::fromRgbF(0.801871, 0.258674, 0.283099),
    QColor::fromRgbF(0.807082, 0.262692, 0.278898),
    QColor::fromRgbF(0.812239, 0.266786, 0.274661),
    QColor::fromRgbF(0.817341, 0.270954, 0.270390),
    QColor::fromRgbF(0.822386, 0.275197, 0.266085),
    QColor::fromRgbF(0.827372, 0.279517, 0.261750),
    QColor::fromRgbF(0.832299, 0.283913, 0.257383),
    QColor::fromRgbF(0.837165, 0.288385, 0.252988),
    QColor::fromRgbF(0.841969, 0.292933, 0.248564),
    QColor::fromRgbF(0.846709, 0.297559, 0.244113),
    QColor::fromRgbF(0.851384, 0.302260, 0.239636),
    QColor::fromRgbF(0.855992, 0.307038, 0.235133),
    QColor::fromRgbF(0.860533, 0.311892, 0.230606),
    QColor::fromRgbF(0.865006, 0.316822, 0.226055),
    QColor::fromRgbF(0.869409, 0.321827, 0.221482),
    QColor::fromRgbF(0.873741, 0.326906, 0.216886),
    QColor::fromRgbF(0.878001, 0.332060, 0.212268),
    QColor::fromRgbF(0.882188, 0.337287, 0.207628),
    QColor::fromRgbF(0.886302, 0.342586, 0.202968),
    QColor::fromRgbF(0.890341, 0.347957, 0.198286),
    QColor::fromRgbF(0.894305, 0.353399, 0.193584),
    QColor::fromRgbF(0.898192, 0.358911, 0.188860),
    QColor::fromRgbF(0.902003, 0.364492, 0.184116),
    QColor::fromRgbF(0.905735, 0.370140, 0.179350),
    QColor::fromRgbF(0.909390, 0.375856, 0.174563),
    QColor::fromRgbF(0.912966, 0.381636, 0.169755),
    QColor::fromRgbF(0.916462, 0.387481, 0.164924),
    QColor::fromRgbF(0.919879, 0.393389, 0.160070),
    QColor::fromRgbF(0.923215, 0.399359, 0.155193),
    QColor::fromRgbF(0.926470, 0.405389, 0.150292),
    QColor::fromRgbF(0.929644, 0.411479, 0.145367),
    QColor::fromRgbF(0.932737, 0.417627, 0.140417),
    QColor::fromRgbF(0.935747, 0.423831, 0.135440),
    QColor::fromRgbF(0.938675, 0.430091, 0.130438),
    QColor::fromRgbF(0.941521, 0.436405, 0.125409),
    QColor::fromRgbF(0.944285, 0.442772, 0.120354),
    QColor::fromRgbF(0.946965, 0.449191, 0.115272),
    QColor::fromRgbF(0.949562, 0.455660, 0.110164),
    QColor::fromRgbF(0.952075, 0.462178, 0.105031),
    QColor::fromRgbF(0.954506, 0.468744, 0.099874),
    QColor::fromRgbF(0.956852, 0.475356, 0.094695),
    QColor::fromRgbF(0.959114, 0.482014, 0.089499),
    QColor::fromRgbF(0.961293, 0.488716, 0.084289),
    QColor::fromRgbF(0.963387, 0.495462, 0.079073),
    QColor::fromRgbF(0.965397, 0.502249, 0.073859),
    QColor::fromRgbF(0.967322, 0.509078, 0.068659),
    QColor::fromRgbF(0.969163, 0.515946, 0.063488),
    QColor::fromRgbF(0.970919, 0.522853, 0.058367),
    QColor::fromRgbF(0.972590, 0.529798, 0.053324),
    QColor::fromRgbF(0.974176, 0.536780, 0.048392),
    QColor::fromRgbF(0.975677, 0.543798, 0.043618),
    QColor::fromRgbF(0.977092, 0.550850, 0.039050),
    QColor::fromRgbF(0.978422, 0.557937, 0.034931),
    QColor::fromRgbF(0.979666, 0.565057, 0.031409),
    QColor::fromRgbF(0.980824, 0.572209, 0.028508),
    QColor::fromRgbF(0.981895, 0.579392, 0.026250),
    QColor::fromRgbF(0.982881, 0.586606, 0.024661),
    QColor::fromRgbF(0.983779, 0.593849, 0.023770),
    QColor::fromRgbF(0.984591, 0.601122, 0.023606),
    QColor::fromRgbF(0.985315, 0.608422, 0.024202),
    QColor::fromRgbF(0.985952, 0.615750, 0.025592),
    QColor::fromRgbF(0.986502, 0.623105, 0.027814),
    QColor::fromRgbF(0.986964, 0.630485, 0.030908),
    QColor::fromRgbF(0.987337, 0.637890, 0.034916),
    QColor::fromRgbF(0.987622, 0.645320, 0.039886),
    QColor::fromRgbF(0.987819, 0.652773, 0.045581),
    QColor::fromRgbF(0.987926, 0.660250, 0.051750),
    QColor::fromRgbF(0.987945, 0.667748, 0.058329),
    QColor::fromRgbF(0.987874, 0.675267, 0.065257),
    QColor::fromRgbF(0.987714, 0.682807, 0.072489),
    QColor::fromRgbF(0.987464, 0.690366, 0.079990),
    QColor::fromRgbF(0.987124, 0.697944, 0.087731),
    QColor::fromRgbF(0.986694, 0.705540, 0.095694),
    QColor::fromRgbF(0.986175, 0.713153, 0.103863),
    QColor::fromRgbF(0.985566, 0.720782, 0.112229),
    QColor::fromRgbF(0.984865, 0.728427, 0.120785),
    QColor::fromRgbF(0.984075, 0.736087, 0.129527),
    QColor::fromRgbF(0.983196, 0.743758, 0.138453),
    QColor::fromRgbF(0.982228, 0.751442, 0.147565),
    QColor::fromRgbF(0.981173, 0.759135, 0.156863),
    QColor::fromRgbF(0.980032, 0.766837, 0.166353),
    QColor::fromRgbF(0.978806, 0.774545, 0.176037),
    QColor::fromRgbF(0.977497, 0.782258, 0.185923),
    QColor::fromRgbF(0.976108, 0.789974, 0.196018),
    QColor::fromRgbF(0.974638, 0.797692, 0.206332),
    QColor::fromRgbF(0.973088, 0.805409, 0.216877),
    QColor::fromRgbF(0.971468, 0.813122, 0.227658),
    QColor::fromRgbF(0.969783, 0.820825, 0.238686),
    QColor::fromRgbF(0.968041, 0.828515, 0.249972),
    QColor::fromRgbF(0.966243, 0.836191, 0.261534),
    QColor::fromRgbF(0.964394, 0.843848, 0.273391),
    QColor::fromRgbF(0.962517, 0.851476, 0.285546),
    QColor::fromRgbF(0.960626, 0.859069, 0.298010),
    QColor::fromRgbF(0.958720, 0.866624, 0.310820),
    QColor::fromRgbF(0.956834, 0.874129, 0.323974),
    QColor::fromRgbF(0.954997, 0.881569, 0.337475),
    QColor::fromRgbF(0.953215, 0.888942, 0.351369),
    QColor::fromRgbF(0.951546, 0.896226, 0.365627),
    QColor::fromRgbF(0.950018, 0.903409, 0.380271),
    QColor::fromRgbF(0.948683, 0.910473, 0.395289),
    QColor::fromRgbF(0.947594, 0.917399, 0.410665),
    QColor::fromRgbF(0.946809, 0.924168, 0.426373),
    QColor::fromRgbF(0.946392, 0.930761, 0.442367),
    QColor::fromRgbF(0.946403, 0.937159, 0.458592),
    QColor::fromRgbF(0.946903, 0.943348, 0.474970),
    QColor::fromRgbF(0.947937, 0.949318, 0.491426),
    QColor::fromRgbF(0.949545, 0.955063, 0.507860),
    QColor::fromRgbF(0.951740, 0.960587, 0.524203),
    QColor::fromRgbF(0.954529, 0.965896, 0.540361),
    QColor::fromRgbF(0.957896, 0.971003, 0.556275),
    QColor::fromRgbF(0.961812, 0.975924, 0.571925),
    QColor::fromRgbF(0.966249, 0.980678, 0.587206),
    QColor::fromRgbF(0.971162, 0.985282, 0.602154),
    QColor::fromRgbF(0.976511, 0.989753, 0.616760),
    QColor::fromRgbF(0.982257, 0.994109, 0.631017),
    QColor::fromRgbF(0.988362, 0.998364, 0.644924)};

inline const std::vector<QColor> Plasma{
    QColor::fromRgbF(0.050383, 0.029803, 0.527975),
    QColor::fromRgbF(0.063536, 0.028426, 0.533124),
    QColor::fromRgbF(0.075353, 0.027206, 0.538007),
    QColor::fromRgbF(0.086222, 0.026125, 0.542658),
    QColor::fromRgbF(0.096379, 0.025165, 0.547103),
    QColor::fromRgbF(0.105980, 0.024309, 0.551368),
    QColor::fromRgbF(0.115124, 0.023556, 0.555468),
    QColor::fromRgbF(0.123903, 0.022878, 0.559423),
    QColor::fromRgbF(0.132381, 0.022258, 0.563250),
    QColor::fromRgbF(0.140603, 0.021687, 0.566959),
    QColor::fromRgbF(0.148607, 0.021154, 0.570562),
    QColor::fromRgbF(0.156421, 0.020651, 0.574065),
    QColor::fromRgbF(0.164070, 0.020171, 0.577478),
    QColor::fromRgbF(0.171574, 0.019706, 0.580806),
    QColor::fromRgbF(0.178950, 0.019252, 0.584054),
    QColor::fromRgbF(0.186213, 0.018803, 0.587228),
    QColor::fromRgbF(0.193374, 0.018354, 0.590330),
    QColor::fromRgbF(0.200445, 0.017902, 0.593364),
    QColor::fromRgbF(0.207435, 0.017442, 0.596333),
    QColor::fromRgbF(0.214350, 0.016973, 0.599239),
    QColor::fromRgbF(0.221197, 0.016497, 0.602083),
    QColor::fromRgbF(0.227983, 0.016007, 0.604867),
    QColor::fromRgbF(0.234715, 0.015502, 0.607592),
    QColor::fromRgbF(0.241396, 0.014979, 0.610259),
    QColor::fromRgbF(0.248032, 0.014439, 0.612868),
    QColor::fromRgbF(0.254627, 0.013882, 0.615419),
    QColor::fromRgbF(0.261183, 0.013308, 0.617911),
    QColor::fromRgbF(0.267703, 0.012716, 0.620346),
    QColor::fromRgbF(0.274191, 0.012109, 0.622722),
    QColor::fromRgbF(0.280648, 0.011488, 0.625038),
    QColor::fromRgbF(0.287076, 0.010855, 0.627295),
    QColor::fromRgbF(0.293478, 0.010213, 0.629490),
    QColor::fromRgbF(0.299855, 0.009561, 0.631624),
    QColor::fromRgbF(0.306210, 0.008902, 0.633694),
    QColor::fromRgbF(0.312543, 0.008239, 0.635700),
    QColor::fromRgbF(0.318856, 0.007576, 0.637640),
    QColor::fromRgbF(0.325150, 0.006915, 0.639512),
    QColor::fromRgbF(0.331426, 0.006261, 0.641316),
    QColor::fromRgbF(0.337683, 0.005618, 0.643049),
    QColor::fromRgbF(0.343925, 0.004991, 0.644710),
    QColor::fromRgbF(0.350150, 0.004382, 0.646298),
    QColor::fromRgbF(0.356359, 0.003798, 0.647810),
    QColor::fromRgbF(0.362553, 0.003243, 0.649245),
    QColor::fromRgbF(0.368733, 0.002724, 0.650601),
    QColor::fromRgbF(0.374897, 0.002245, 0.651876),
    QColor::fromRgbF(0.381047, 0.001814, 0.653068),
    QColor::fromRgbF(0.387183, 0.001434, 0.654177),
    QColor::fromRgbF(0.393304, 0.001114, 0.655199),
    QColor::fromRgbF(0.399411, 0.000859, 0.656133),
    QColor::fromRgbF(0.405503, 0.000678, 0.656977),
    QColor::fromRgbF(0.411580, 0.000577, 0.657730),
    QColor::fromRgbF(0.417642, 0.000564, 0.658390),
    QColor::fromRgbF(0.423689, 0.000646, 0.658956),
    QColor::fromRgbF(0.429719, 0.000831, 0.659425),
    QColor::fromRgbF(0.435734, 0.001127, 0.659797),
    QColor::fromRgbF(0.441732, 0.001540, 0.660069),
    QColor::fromRgbF(0.447714, 0.002080, 0.660240),
    QColor::fromRgbF(0.453677, 0.002755, 0.660310),
    QColor::fromRgbF(0.459623, 0.003574, 0.660277),
    QColor::fromRgbF(0.465550, 0.004545, 0.660139),
    QColor::fromRgbF(0.471457, 0.005678, 0.659897),
    QColor::fromRgbF(0.477344, 0.006980, 0.659549),
    QColor::fromRgbF(0.483210, 0.008460, 0.659095),
    QColor::fromRgbF(0.489055, 0.010127, 0.658534),
    QColor::fromRgbF(0.494877, 0.011990, 0.657865),
    QColor::fromRgbF(0.500678, 0.014055, 0.657088),
    QColor::fromRgbF(0.506454, 0.016333, 0.656202),
    QColor::fromRgbF(0.512206, 0.018833, 0.655209),
    QColor::fromRgbF(0.517933, 0.021563, 0.654109),
    QColor::fromRgbF(0.523633, 0.024532, 0.652901),
    QColor::fromRgbF(0.529306, 0.027747, 0.651586),
    QColor::fromRgbF(0.534952, 0.031217, 0.650165),
    QColor::fromRgbF(0.540570, 0.034950, 0.648640),
    QColor::fromRgbF(0.546157, 0.038954, 0.647010),
    QColor::fromRgbF(0.551715, 0.043136, 0.645277),
    QColor::fromRgbF(0.557243, 0.047331, 0.643443),
    QColor::fromRgbF(0.562738, 0.051545, 0.641509),
    QColor::fromRgbF(0.568201, 0.055778, 0.639477),
    QColor::fromRgbF(0.573632, 0.060028, 0.637349),
    QColor::fromRgbF(0.579029, 0.064296, 0.635126),
    QColor::fromRgbF(0.584391, 0.068579, 0.632812),
    QColor::fromRgbF(0.589719, 0.072878, 0.630408),
    QColor::fromRgbF(0.595011, 0.077190, 0.627917),
    QColor::fromRgbF(0.600266, 0.081516, 0.625342),
    QColor::fromRgbF(0.605485, 0.085854, 0.622686),
    QColor::fromRgbF(0.610667, 0.090204, 0.619951),
    QColor::fromRgbF(0.615812, 0.094564, 0.617140),
    QColor::fromRgbF(0.620919, 0.098934, 0.614257),
    QColor::fromRgbF(0.625987, 0.103312, 0.611305),
    QColor::fromRgbF(0.631017, 0.107699, 0.608287),
    QColor::fromRgbF(0.636008, 0.112092, 0.605205),
    QColor::fromRgbF(0.640959, 0.116492, 0.602065),
    QColor::fromRgbF(0.645872, 0.120898, 0.598867),
    QColor::fromRgbF(0.650746, 0.125309, 0.595617),
    QColor::fromRgbF(0.655580, 0.129725, 0.592317),
    QColor::fromRgbF(0.660374, 0.134144, 0.588971),
    QColor::fromRgbF(0.665129, 0.138566, 0.585582),
    QColor::fromRgbF(0.669845, 0.142992, 0.582154),
    QColor::fromRgbF(0.674522, 0.147419, 0.578688),
    QColor::fromRgbF(0.679160, 0.151848, 0.575189),
    QColor::fromRgbF(0.683758, 0.156278, 0.571660),
    QColor::fromRgbF(0.688318, 0.160709, 0.568103),
    QColor::fromRgbF(0.692840, 0.165141, 0.564522),
    QColor::fromRgbF(0.697324, 0.169573, 0.560919),
    QColor::fromRgbF(0.701769, 0.174005, 0.557296),
    QColor::fromRgbF(0.706178, 0.178437, 0.553657),
    QColor::fromRgbF(0.710549, 0.182868, 0.550004),
    QColor::fromRgbF(0.714883, 0.187299, 0.546338),
    QColor::fromRgbF(0.719181, 0.191729, 0.542663),
    QColor::fromRgbF(0.723444, 0.196158, 0.538981),
    QColor::fromRgbF(0.727670, 0.200586, 0.535293),
    QColor::fromRgbF(0.731862, 0.205013, 0.531601),
    QColor::fromRgbF(0.736019, 0.209439, 0.527908),
    QColor::fromRgbF(0.740143, 0.213864, 0.524216),
    QColor::fromRgbF(0.744232, 0.218288, 0.520524),
    QColor::fromRgbF(0.748289, 0.222711, 0.516834),
    QColor::fromRgbF(0.752312, 0.227133, 0.513149),
    QColor::fromRgbF(0.756304, 0.231555, 0.509468),
    QColor::fromRgbF(0.760264, 0.235976, 0.505794),
    QColor::fromRgbF(0.764193, 0.240396, 0.502126),
    QColor::fromRgbF(0.768090, 0.244817, 0.498465),
    QColor::fromRgbF(0.771958, 0.249237, 0.494813),
    QColor::fromRgbF(0.775796, 0.253658, 0.491171),
    QColor::fromRgbF(0.779604, 0.258078, 0.487539),
    QColor::fromRgbF(0.783383, 0.262500, 0.483918),
    QColor::fromRgbF(0.787133, 0.266922, 0.480307),
    QColor::fromRgbF(0.790855, 0.271345, 0.476706),
    QColor::fromRgbF(0.794549, 0.275770, 0.473117),
    QColor::fromRgbF(0.798216, 0.280197, 0.469538),
    QColor::fromRgbF(0.801855, 0.284626, 0.465971),
    QColor::fromRgbF(0.805467, 0.289057, 0.462415),
    QColor::fromRgbF(0.809052, 0.293491, 0.458870),
    QColor::fromRgbF(0.812612, 0.297928, 0.455338),
    QColor::fromRgbF(0.816144, 0.302368, 0.451816),
    QColor::fromRgbF(0.819651, 0.306812, 0.448306),
    QColor::fromRgbF(0.823132, 0.311261, 0.444806),
    QColor::fromRgbF(0.826588, 0.315714, 0.441316),
    QColor::fromRgbF(0.830018, 0.320172, 0.437836),
    QColor::fromRgbF(0.833422, 0.324635, 0.434366),
    QColor::fromRgbF(0.836801, 0.329105, 0.430905),
    QColor::fromRgbF(0.840155, 0.333580, 0.427455),
    QColor::fromRgbF(0.843484, 0.338062, 0.424013),
    QColor::fromRgbF(0.846788, 0.342551, 0.420579),
    QColor::fromRgbF(0.850066, 0.347048, 0.417153),
    QColor::fromRgbF(0.853319, 0.351553, 0.413734),
    QColor::fromRgbF(0.856547, 0.356066, 0.410322),
    QColor::fromRgbF(0.859750, 0.360588, 0.406917),
    QColor::fromRgbF(0.862927, 0.365119, 0.403519),
    QColor::fromRgbF(0.866078, 0.369660, 0.400126),
    QColor::fromRgbF(0.869203, 0.374212, 0.396738),
    QColor::fromRgbF(0.872303, 0.378774, 0.393355),
    QColor::fromRgbF(0.875376, 0.383347, 0.389976),
    QColor::fromRgbF(0.878423, 0.387932, 0.386600),
    QColor::fromRgbF(0.881443, 0.392529, 0.383229),
    QColor::fromRgbF(0.884436, 0.397139, 0.379860),
    QColor::fromRgbF(0.887402, 0.401762, 0.376494),
    QColor::fromRgbF(0.890340, 0.406398, 0.373130),
    QColor::fromRgbF(0.893250, 0.411048, 0.369768),
    QColor::fromRgbF(0.896131, 0.415712, 0.366407),
    QColor::fromRgbF(0.898984, 0.420392, 0.363047),
    QColor::fromRgbF(0.901807, 0.425087, 0.359688),
    QColor::fromRgbF(0.904601, 0.429797, 0.356329),
    QColor::fromRgbF(0.907365, 0.434524, 0.352970),
    QColor::fromRgbF(0.910098, 0.439268, 0.349610),
    QColor::fromRgbF(0.912800, 0.444029, 0.346251),
    QColor::fromRgbF(0.915471, 0.448807, 0.342890),
    QColor::fromRgbF(0.918109, 0.453603, 0.339529),
    QColor::fromRgbF(0.920714, 0.458417, 0.336166),
    QColor::fromRgbF(0.923287, 0.463251, 0.332801),
    QColor::fromRgbF(0.925825, 0.468103, 0.329435),
    QColor::fromRgbF(0.928329, 0.472975, 0.326067),
    QColor::fromRgbF(0.930798, 0.477867, 0.322697),
    QColor::fromRgbF(0.933232, 0.482780, 0.319325),
    QColor::fromRgbF(0.935630, 0.487712, 0.315952),
    QColor::fromRgbF(0.937990, 0.492667, 0.312575),
    QColor::fromRgbF(0.940313, 0.497642, 0.309197),
    QColor::fromRgbF(0.942598, 0.502639, 0.305816),
    QColor::fromRgbF(0.944844, 0.507658, 0.302433),
    QColor::fromRgbF(0.947051, 0.512699, 0.299049),
    QColor::fromRgbF(0.949217, 0.517763, 0.295662),
    QColor::fromRgbF(0.951344, 0.522850, 0.292275),
    QColor::fromRgbF(0.953428, 0.527960, 0.288883),
    QColor::fromRgbF(0.955470, 0.533093, 0.285490),
    QColor::fromRgbF(0.957469, 0.538250, 0.282096),
    QColor::fromRgbF(0.959424, 0.543431, 0.278701),
    QColor::fromRgbF(0.961336, 0.548636, 0.275305),
    QColor::fromRgbF(0.963203, 0.553865, 0.271909),
    QColor::fromRgbF(0.965024, 0.559118, 0.268513),
    QColor::fromRgbF(0.966798, 0.564396, 0.265118),
    QColor::fromRgbF(0.968526, 0.569700, 0.261721),
    QColor::fromRgbF(0.970205, 0.575028, 0.258325),
    QColor::fromRgbF(0.971835, 0.580382, 0.254931),
    QColor::fromRgbF(0.973416, 0.585761, 0.251540),
    QColor::fromRgbF(0.974947, 0.591165, 0.248151),
    QColor::fromRgbF(0.976428, 0.596595, 0.244767),
    QColor::fromRgbF(0.977856, 0.602051, 0.241387),
    QColor::fromRgbF(0.979233, 0.607532, 0.238013),
    QColor::fromRgbF(0.980556, 0.613039, 0.234646),
    QColor::fromRgbF(0.981826, 0.618572, 0.231287),
    QColor::fromRgbF(0.983041, 0.624131, 0.227937),
    QColor::fromRgbF(0.984199, 0.629718, 0.224595),
    QColor::fromRgbF(0.985301, 0.635330, 0.221265),
    QColor::fromRgbF(0.986345, 0.640969, 0.217948),
    QColor::fromRgbF(0.987332, 0.646633, 0.214648),
    QColor::fromRgbF(0.988260, 0.652325, 0.211364),
    QColor::fromRgbF(0.989128, 0.658043, 0.208100),
    QColor::fromRgbF(0.989935, 0.663787, 0.204859),
    QColor::fromRgbF(0.990681, 0.669558, 0.201642),
    QColor::fromRgbF(0.991365, 0.675355, 0.198453),
    QColor::fromRgbF(0.991985, 0.681179, 0.195295),
    QColor::fromRgbF(0.992541, 0.687030, 0.192170),
    QColor::fromRgbF(0.993032, 0.692907, 0.189084),
    QColor::fromRgbF(0.993456, 0.698810, 0.186041),
    QColor::fromRgbF(0.993814, 0.704741, 0.183043),
    QColor::fromRgbF(0.994103, 0.710698, 0.180097),
    QColor::fromRgbF(0.994324, 0.716681, 0.177208),
    QColor::fromRgbF(0.994474, 0.722691, 0.174381),
    QColor::fromRgbF(0.994553, 0.728728, 0.171622),
    QColor::fromRgbF(0.994561, 0.734791, 0.168938),
    QColor::fromRgbF(0.994495, 0.740880, 0.166335),
    QColor::fromRgbF(0.994355, 0.746995, 0.163821),
    QColor::fromRgbF(0.994141, 0.753137, 0.161404),
    QColor::fromRgbF(0.993851, 0.759304, 0.159092),
    QColor::fromRgbF(0.993482, 0.765499, 0.156891),
    QColor::fromRgbF(0.993033, 0.771720, 0.154808),
    QColor::fromRgbF(0.992505, 0.777967, 0.152855),
    QColor::fromRgbF(0.991897, 0.784239, 0.151042),
    QColor::fromRgbF(0.991209, 0.790537, 0.149377),
    QColor::fromRgbF(0.990439, 0.796859, 0.147870),
    QColor::fromRgbF(0.989587, 0.803205, 0.146529),
    QColor::fromRgbF(0.988648, 0.809579, 0.145357),
    QColor::fromRgbF(0.987621, 0.815978, 0.144363),
    QColor::fromRgbF(0.986509, 0.822401, 0.143557),
    QColor::fromRgbF(0.985314, 0.828846, 0.142945),
    QColor::fromRgbF(0.984031, 0.835315, 0.142528),
    QColor::fromRgbF(0.982653, 0.841812, 0.142303),
    QColor::fromRgbF(0.981190, 0.848329, 0.142279),
    QColor::fromRgbF(0.979644, 0.854866, 0.142453),
    QColor::fromRgbF(0.977995, 0.861432, 0.142808),
    QColor::fromRgbF(0.976265, 0.868016, 0.143351),
    QColor::fromRgbF(0.974443, 0.874622, 0.144061),
    QColor::fromRgbF(0.972530, 0.881250, 0.144923),
    QColor::fromRgbF(0.970533, 0.887896, 0.145919),
    QColor::fromRgbF(0.968443, 0.894564, 0.147014),
    QColor::fromRgbF(0.966271, 0.901249, 0.148180),
    QColor::fromRgbF(0.964021, 0.907950, 0.149370),
    QColor::fromRgbF(0.961681, 0.914672, 0.150520),
    QColor::fromRgbF(0.959276, 0.921407, 0.151566),
    QColor::fromRgbF(0.956808, 0.928152, 0.152409),
    QColor::fromRgbF(0.954287, 0.934908, 0.152921),
    QColor::fromRgbF(0.951726, 0.941671, 0.152925),
    QColor::fromRgbF(0.949151, 0.948435, 0.152178),
    QColor::fromRgbF(0.946602, 0.955190, 0.150328),
    QColor::fromRgbF(0.944152, 0.961916, 0.146861),
    QColor::fromRgbF(0.941896, 0.968590, 0.140956),
    QColor::fromRgbF(0.940015, 0.975158, 0.131326)};

inline const std::vector<QColor> Viridis{
    QColor::fromRgbF(0.267004, 0.004874, 0.329415),
    QColor::fromRgbF(0.268510, 0.009605, 0.335427),
    QColor::fromRgbF(0.269944, 0.014625, 0.341379),
    QColor::fromRgbF(0.271305, 0.019942, 0.347269),
    QColor::fromRgbF(0.272594, 0.025563, 0.353093),
    QColor::fromRgbF(0.273809, 0.031497, 0.358853),
    QColor::fromRgbF(0.274952, 0.037752, 0.364543),
    QColor::fromRgbF(0.276022, 0.044167, 0.370164),
    QColor::fromRgbF(0.277018, 0.050344, 0.375715),
    QColor::fromRgbF(0.277941, 0.056324, 0.381191),
    QColor::fromRgbF(0.278791, 0.062145, 0.386592),
    QColor::fromRgbF(0.279566, 0.067836, 0.391917),
    QColor::fromRgbF(0.280267, 0.073417, 0.397163),
    QColor::fromRgbF(0.280894, 0.078907, 0.402329),
    QColor::fromRgbF(0.281446, 0.084320, 0.407414),
    QColor::fromRgbF(0.281924, 0.089666, 0.412415),
    QColor::fromRgbF(0.282327, 0.094955, 0.417331),
    QColor::fromRgbF(0.282656, 0.100196, 0.422160),
    QColor::fromRgbF(0.282910, 0.105393, 0.426902),
    QColor::fromRgbF(0.283091, 0.110553, 0.431554),
    QColor::fromRgbF(0.283197, 0.115680, 0.436115),
    QColor::fromRgbF(0.283229, 0.120777, 0.440584),
    QColor::fromRgbF(0.283187, 0.125848, 0.444960),
    QColor::fromRgbF(0.283072, 0.130895, 0.449241),
    QColor::fromRgbF(0.282884, 0.135920, 0.453427),
    QColor::fromRgbF(0.282623, 0.140926, 0.457517),
    QColor::fromRgbF(0.282290, 0.145912, 0.461510),
    QColor::fromRgbF(0.281887, 0.150881, 0.465405),
    QColor::fromRgbF(0.281412, 0.155834, 0.469201),
    QColor::fromRgbF(0.280868, 0.160771, 0.472899),
    QColor::fromRgbF(0.280255, 0.165693, 0.476498),
    QColor::fromRgbF(0.279574, 0.170599, 0.479997),
    QColor::fromRgbF(0.278826, 0.175490, 0.483397),
    QColor::fromRgbF(0.278012, 0.180367, 0.486697),
    QColor::fromRgbF(0.277134, 0.185228, 0.489898),
    QColor::fromRgbF(0.276194, 0.190074, 0.493001),
    QColor::fromRgbF(0.275191, 0.194905, 0.496005),
    QColor::fromRgbF(0.274128, 0.199721, 0.498911),
    QColor::fromRgbF(0.273006, 0.204520, 0.501721),
    QColor::fromRgbF(0.271828, 0.209303, 0.504434),
    QColor::fromRgbF(0.270595, 0.214069, 0.507052),
    QColor::fromRgbF(0.269308, 0.218818, 0.509577),
    QColor::fromRgbF(0.267968, 0.223549, 0.512008),
    QColor::fromRgbF(0.266580, 0.228262, 0.514349),
    QColor::fromRgbF(0.265145, 0.232956, 0.516599),
    QColor::fromRgbF(0.263663, 0.237631, 0.518762),
    QColor::fromRgbF(0.262138, 0.242286, 0.520837),
    QColor::fromRgbF(0.260571, 0.246922, 0.522828),
    QColor::fromRgbF(0.258965, 0.251537, 0.524736),
    QColor::fromRgbF(0.257322, 0.256130, 0.526563),
    QColor::fromRgbF(0.255645, 0.260703, 0.528312),
    QColor::fromRgbF(0.253935, 0.265254, 0.529983),
    QColor::fromRgbF(0.252194, 0.269783, 0.531579),
    QColor::fromRgbF(0.250425, 0.274290, 0.533103),
    QColor::fromRgbF(0.248629, 0.278775, 0.534556),
    QColor::fromRgbF(0.246811, 0.283237, 0.535941),
    QColor::fromRgbF(0.244972, 0.287675, 0.537260),
    QColor::fromRgbF(0.243113, 0.292092, 0.538516),
    QColor::fromRgbF(0.241237, 0.296485, 0.539709),
    QColor::fromRgbF(0.239346, 0.300855, 0.540844),
    QColor::fromRgbF(0.237441, 0.305202, 0.541921),
    QColor::fromRgbF(0.235526, 0.309527, 0.542944),
    QColor::fromRgbF(0.233603, 0.313828, 0.543914),
    QColor::fromRgbF(0.231674, 0.318106, 0.544834),
    QColor::fromRgbF(0.229739, 0.322361, 0.545706),
    QColor::fromRgbF(0.227802, 0.326594, 0.546532),
    QColor::fromRgbF(0.225863, 0.330805, 0.547314),
    QColor::fromRgbF(0.223925, 0.334994, 0.548053),
    QColor::fromRgbF(0.221989, 0.339161, 0.548752),
    QColor::fromRgbF(0.220057, 0.343307, 0.549413),
    QColor::fromRgbF(0.218130, 0.347432, 0.550038),
    QColor::fromRgbF(0.216210, 0.351535, 0.550627),
    QColor::fromRgbF(0.214298, 0.355619, 0.551184),
    QColor::fromRgbF(0.212395, 0.359683, 0.551710),
    QColor::fromRgbF(0.210503, 0.363727, 0.552206),
    QColor::fromRgbF(0.208623, 0.367752, 0.552675),
    QColor::fromRgbF(0.206756, 0.371758, 0.553117),
    QColor::fromRgbF(0.204903, 0.375746, 0.553533),
    QColor::fromRgbF(0.203063, 0.379716, 0.553925),
    QColor::fromRgbF(0.201239, 0.383670, 0.554294),
    QColor::fromRgbF(0.199430, 0.387607, 0.554642),
    QColor::fromRgbF(0.197636, 0.391528, 0.554969),
    QColor::fromRgbF(0.195860, 0.395433, 0.555276),
    QColor::fromRgbF(0.194100, 0.399323, 0.555565),
    QColor::fromRgbF(0.192357, 0.403199, 0.555836),
    QColor::fromRgbF(0.190631, 0.407061, 0.556089),
    QColor::fromRgbF(0.188923, 0.410910, 0.556326),
    QColor::fromRgbF(0.187231, 0.414746, 0.556547),
    QColor::fromRgbF(0.185556, 0.418570, 0.556753),
    QColor::fromRgbF(0.183898, 0.422383, 0.556944),
    QColor::fromRgbF(0.182256, 0.426184, 0.557120),
    QColor::fromRgbF(0.180629, 0.429975, 0.557282),
    QColor::fromRgbF(0.179019, 0.433756, 0.557430),
    QColor::fromRgbF(0.177423, 0.437527, 0.557565),
    QColor::fromRgbF(0.175841, 0.441290, 0.557685),
    QColor::fromRgbF(0.174274, 0.445044, 0.557792),
    QColor::fromRgbF(0.172719, 0.448791, 0.557885),
    QColor::fromRgbF(0.171176, 0.452530, 0.557965),
    QColor::fromRgbF(0.169646, 0.456262, 0.558030),
    QColor::fromRgbF(0.168126, 0.459988, 0.558082),
    QColor::fromRgbF(0.166617, 0.463708, 0.558119),
    QColor::fromRgbF(0.165117, 0.467423, 0.558141),
    QColor::fromRgbF(0.163625, 0.471133, 0.558148),
    QColor::fromRgbF(0.162142, 0.474838, 0.558140),
    QColor::fromRgbF(0.160665, 0.478540, 0.558115),
    QColor::fromRgbF(0.159194, 0.482237, 0.558073),
    QColor::fromRgbF(0.157729, 0.485932, 0.558013),
    QColor::fromRgbF(0.156270, 0.489624, 0.557936),
    QColor::fromRgbF(0.154815, 0.493313, 0.557840),
    QColor::fromRgbF(0.153364, 0.497000, 0.557724),
    QColor::fromRgbF(0.151918, 0.500685, 0.557587),
    QColor::fromRgbF(0.150476, 0.504369, 0.557430),
    QColor::fromRgbF(0.149039, 0.508051, 0.557250),
    QColor::fromRgbF(0.147607, 0.511733, 0.557049),
    QColor::fromRgbF(0.146180, 0.515413, 0.556823),
    QColor::fromRgbF(0.144759, 0.519093, 0.556572),
    QColor::fromRgbF(0.143343, 0.522773, 0.556295),
    QColor::fromRgbF(0.141935, 0.526453, 0.555991),
    QColor::fromRgbF(0.140536, 0.530132, 0.555659),
    QColor::fromRgbF(0.139147, 0.533812, 0.555298),
    QColor::fromRgbF(0.137770, 0.537492, 0.554906),
    QColor::fromRgbF(0.136408, 0.541173, 0.554483),
    QColor::fromRgbF(0.135066, 0.544853, 0.554029),
    QColor::fromRgbF(0.133743, 0.548535, 0.553541),
    QColor::fromRgbF(0.132444, 0.552216, 0.553018),
    QColor::fromRgbF(0.131172, 0.555899, 0.552459),
    QColor::fromRgbF(0.129933, 0.559582, 0.551864),
    QColor::fromRgbF(0.128729, 0.563265, 0.551229),
    QColor::fromRgbF(0.127568, 0.566949, 0.550556),
    QColor::fromRgbF(0.126453, 0.570633, 0.549841),
    QColor::fromRgbF(0.125394, 0.574318, 0.549086),
    QColor::fromRgbF(0.124395, 0.578002, 0.548287),
    QColor::fromRgbF(0.123463, 0.581687, 0.547445),
    QColor::fromRgbF(0.122606, 0.585371, 0.546557),
    QColor::fromRgbF(0.121831, 0.589055, 0.545623),
    QColor::fromRgbF(0.121148, 0.592739, 0.544641),
    QColor::fromRgbF(0.120565, 0.596422, 0.543611),
    QColor::fromRgbF(0.120092, 0.600104, 0.542530),
    QColor::fromRgbF(0.119738, 0.603785, 0.541400),
    QColor::fromRgbF(0.119512, 0.607464, 0.540218),
    QColor::fromRgbF(0.119423, 0.611141, 0.538982),
    QColor::fromRgbF(0.119483, 0.614817, 0.537692),
    QColor::fromRgbF(0.119699, 0.618490, 0.536347),
    QColor::fromRgbF(0.120081, 0.622161, 0.534946),
    QColor::fromRgbF(0.120638, 0.625828, 0.533488),
    QColor::fromRgbF(0.121380, 0.629492, 0.531973),
    QColor::fromRgbF(0.122312, 0.633153, 0.530398),
    QColor::fromRgbF(0.123444, 0.636809, 0.528763),
    QColor::fromRgbF(0.124780, 0.640461, 0.527068),
    QColor::fromRgbF(0.126326, 0.644107, 0.525311),
    QColor::fromRgbF(0.128087, 0.647749, 0.523491),
    QColor::fromRgbF(0.130067, 0.651384, 0.521608),
    QColor::fromRgbF(0.132268, 0.655014, 0.519661),
    QColor::fromRgbF(0.134692, 0.658636, 0.517649),
    QColor::fromRgbF(0.137339, 0.662252, 0.515571),
    QColor::fromRgbF(0.140210, 0.665859, 0.513427),
    QColor::fromRgbF(0.143303, 0.669459, 0.511215),
    QColor::fromRgbF(0.146616, 0.673050, 0.508936),
    QColor::fromRgbF(0.150148, 0.676631, 0.506589),
    QColor::fromRgbF(0.153894, 0.680203, 0.504172),
    QColor::fromRgbF(0.157851, 0.683765, 0.501686),
    QColor::fromRgbF(0.162016, 0.687316, 0.499129),
    QColor::fromRgbF(0.166383, 0.690856, 0.496502),
    QColor::fromRgbF(0.170948, 0.694384, 0.493803),
    QColor::fromRgbF(0.175707, 0.697900, 0.491033),
    QColor::fromRgbF(0.180653, 0.701402, 0.488189),
    QColor::fromRgbF(0.185783, 0.704891, 0.485273),
    QColor::fromRgbF(0.191090, 0.708366, 0.482284),
    QColor::fromRgbF(0.196571, 0.711827, 0.479221),
    QColor::fromRgbF(0.202219, 0.715272, 0.476084),
    QColor::fromRgbF(0.208030, 0.718701, 0.472873),
    QColor::fromRgbF(0.214000, 0.722114, 0.469588),
    QColor::fromRgbF(0.220124, 0.725509, 0.466226),
    QColor::fromRgbF(0.226397, 0.728888, 0.462789),
    QColor::fromRgbF(0.232815, 0.732247, 0.459277),
    QColor::fromRgbF(0.239374, 0.735588, 0.455688),
    QColor::fromRgbF(0.246070, 0.738910, 0.452024),
    QColor::fromRgbF(0.252899, 0.742211, 0.448284),
    QColor::fromRgbF(0.259857, 0.745492, 0.444467),
    QColor::fromRgbF(0.266941, 0.748751, 0.440573),
    QColor::fromRgbF(0.274149, 0.751988, 0.436601),
    QColor::fromRgbF(0.281477, 0.755203, 0.432552),
    QColor::fromRgbF(0.288921, 0.758394, 0.428426),
    QColor::fromRgbF(0.296479, 0.761561, 0.424223),
    QColor::fromRgbF(0.304148, 0.764704, 0.419943),
    QColor::fromRgbF(0.311925, 0.767822, 0.415586),
    QColor::fromRgbF(0.319809, 0.770914, 0.411152),
    QColor::fromRgbF(0.327796, 0.773980, 0.406640),
    QColor::fromRgbF(0.335885, 0.777018, 0.402049),
    QColor::fromRgbF(0.344074, 0.780029, 0.397381),
    QColor::fromRgbF(0.352360, 0.783011, 0.392636),
    QColor::fromRgbF(0.360741, 0.785964, 0.387814),
    QColor::fromRgbF(0.369214, 0.788888, 0.382914),
    QColor::fromRgbF(0.377779, 0.791781, 0.377939),
    QColor::fromRgbF(0.386433, 0.794644, 0.372886),
    QColor::fromRgbF(0.395174, 0.797475, 0.367757),
    QColor::fromRgbF(0.404001, 0.800275, 0.362552),
    QColor::fromRgbF(0.412913, 0.803041, 0.357269),
    QColor::fromRgbF(0.421908, 0.805774, 0.351910),
    QColor::fromRgbF(0.430983, 0.808473, 0.346476),
    QColor::fromRgbF(0.440137, 0.811138, 0.340967),
    QColor::fromRgbF(0.449368, 0.813768, 0.335384),
    QColor::fromRgbF(0.458674, 0.816363, 0.329727),
    QColor::fromRgbF(0.468053, 0.818921, 0.323998),
    QColor::fromRgbF(0.477504, 0.821444, 0.318195),
    QColor::fromRgbF(0.487026, 0.823929, 0.312321),
    QColor::fromRgbF(0.496615, 0.826376, 0.306377),
    QColor::fromRgbF(0.506271, 0.828786, 0.300362),
    QColor::fromRgbF(0.515992, 0.831158, 0.294279),
    QColor::fromRgbF(0.525776, 0.833491, 0.288127),
    QColor::fromRgbF(0.535621, 0.835785, 0.281908),
    QColor::fromRgbF(0.545524, 0.838039, 0.275626),
    QColor::fromRgbF(0.555484, 0.840254, 0.269281),
    QColor::fromRgbF(0.565498, 0.842430, 0.262877),
    QColor::fromRgbF(0.575563, 0.844566, 0.256415),
    QColor::fromRgbF(0.585678, 0.846661, 0.249897),
    QColor::fromRgbF(0.595839, 0.848717, 0.243329),
    QColor::fromRgbF(0.606045, 0.850733, 0.236712),
    QColor::fromRgbF(0.616293, 0.852709, 0.230052),
    QColor::fromRgbF(0.626579, 0.854645, 0.223353),
    QColor::fromRgbF(0.636902, 0.856542, 0.216620),
    QColor::fromRgbF(0.647257, 0.858400, 0.209861),
    QColor::fromRgbF(0.657642, 0.860219, 0.203082),
    QColor::fromRgbF(0.668054, 0.861999, 0.196293),
    QColor::fromRgbF(0.678489, 0.863742, 0.189503),
    QColor::fromRgbF(0.688944, 0.865448, 0.182725),
    QColor::fromRgbF(0.699415, 0.867117, 0.175971),
    QColor::fromRgbF(0.709898, 0.868751, 0.169257),
    QColor::fromRgbF(0.720391, 0.870350, 0.162603),
    QColor::fromRgbF(0.730889, 0.871916, 0.156029),
    QColor::fromRgbF(0.741388, 0.873449, 0.149561),
    QColor::fromRgbF(0.751884, 0.874951, 0.143228),
    QColor::fromRgbF(0.762373, 0.876424, 0.137064),
    QColor::fromRgbF(0.772852, 0.877868, 0.131109),
    QColor::fromRgbF(0.783315, 0.879285, 0.125405),
    QColor::fromRgbF(0.793760, 0.880678, 0.120005),
    QColor::fromRgbF(0.804182, 0.882046, 0.114965),
    QColor::fromRgbF(0.814576, 0.883393, 0.110347),
    QColor::fromRgbF(0.824940, 0.884720, 0.106217),
    QColor::fromRgbF(0.835270, 0.886029, 0.102646),
    QColor::fromRgbF(0.845561, 0.887322, 0.099702),
    QColor::fromRgbF(0.855810, 0.888601, 0.097452),
    QColor::fromRgbF(0.866013, 0.889868, 0.095953),
    QColor::fromRgbF(0.876168, 0.891125, 0.095250),
    QColor::fromRgbF(0.886271, 0.892374, 0.095374),
    QColor::fromRgbF(0.896320, 0.893616, 0.096335),
    QColor::fromRgbF(0.906311, 0.894855, 0.098125),
    QColor::fromRgbF(0.916242, 0.896091, 0.100717),
    QColor::fromRgbF(0.926106, 0.897330, 0.104071),
    QColor::fromRgbF(0.935904, 0.898570, 0.108131),
    QColor::fromRgbF(0.945636, 0.899815, 0.112838),
    QColor::fromRgbF(0.955300, 0.901065, 0.118128),
    QColor::fromRgbF(0.964894, 0.902323, 0.123941),
    QColor::fromRgbF(0.974417, 0.903590, 0.130215),
    QColor::fromRgbF(0.983868, 0.904867, 0.136897),
    QColor::fromRgbF(0.993248, 0.906157, 0.143936)};

inline const std::vector<QColor> Cividis{
    QColor::fromRgbF(0.0000, 0.1262, 0.3015),
    QColor::fromRgbF(0.0000, 0.1292, 0.3077),
    QColor::fromRgbF(0.0000, 0.1321, 0.3142),
    QColor::fromRgbF(0.0000, 0.1350, 0.3205),
    QColor::fromRgbF(0.0000, 0.1379, 0.3269),
    QColor::fromRgbF(0.0000, 0.1408, 0.3334),
    QColor::fromRgbF(0.0000, 0.1437, 0.3400),
    QColor::fromRgbF(0.0000, 0.1465, 0.3467),
    QColor::fromRgbF(0.0000, 0.1492, 0.3537),
    QColor::fromRgbF(0.0000, 0.1519, 0.3606),
    QColor::fromRgbF(0.0000, 0.1546, 0.3676),
    QColor::fromRgbF(0.0000, 0.1574, 0.3746),
    QColor::fromRgbF(0.0000, 0.1601, 0.3817),
    QColor::fromRgbF(0.0000, 0.1629, 0.3888),
    QColor::fromRgbF(0.0000, 0.1657, 0.3960),
    QColor::fromRgbF(0.0000, 0.1685, 0.4031),
    QColor::fromRgbF(0.0000, 0.1714, 0.4102),
    QColor::fromRgbF(0.0000, 0.1743, 0.4172),
    QColor::fromRgbF(0.0000, 0.1773, 0.4241),
    QColor::fromRgbF(0.0000, 0.1798, 0.4307),
    QColor::fromRgbF(0.0000, 0.1817, 0.4347),
    QColor::fromRgbF(0.0000, 0.1834, 0.4363),
    QColor::fromRgbF(0.0000, 0.1852, 0.4368),
    QColor::fromRgbF(0.0000, 0.1872, 0.4368),
    QColor::fromRgbF(0.0000, 0.1901, 0.4365),
    QColor::fromRgbF(0.0000, 0.1930, 0.4361),
    QColor::fromRgbF(0.0000, 0.1958, 0.4356),
    QColor::fromRgbF(0.0000, 0.1987, 0.4349),
    QColor::fromRgbF(0.0000, 0.2015, 0.4343),
    QColor::fromRgbF(0.0000, 0.2044, 0.4336),
    QColor::fromRgbF(0.0000, 0.2073, 0.4329),
    QColor::fromRgbF(0.0055, 0.2101, 0.4322),
    QColor::fromRgbF(0.0236, 0.2130, 0.4314),
    QColor::fromRgbF(0.0416, 0.2158, 0.4308),
    QColor::fromRgbF(0.0576, 0.2187, 0.4301),
    QColor::fromRgbF(0.0710, 0.2215, 0.4293),
    QColor::fromRgbF(0.0827, 0.2244, 0.4287),
    QColor::fromRgbF(0.0932, 0.2272, 0.4280),
    QColor::fromRgbF(0.1030, 0.2300, 0.4274),
    QColor::fromRgbF(0.1120, 0.2329, 0.4268),
    QColor::fromRgbF(0.1204, 0.2357, 0.4262),
    QColor::fromRgbF(0.1283, 0.2385, 0.4256),
    QColor::fromRgbF(0.1359, 0.2414, 0.4251),
    QColor::fromRgbF(0.1431, 0.2442, 0.4245),
    QColor::fromRgbF(0.1500, 0.2470, 0.4241),
    QColor::fromRgbF(0.1566, 0.2498, 0.4236),
    QColor::fromRgbF(0.1630, 0.2526, 0.4232),
    QColor::fromRgbF(0.1692, 0.2555, 0.4228),
    QColor::fromRgbF(0.1752, 0.2583, 0.4224),
    QColor::fromRgbF(0.1811, 0.2611, 0.4220),
    QColor::fromRgbF(0.1868, 0.2639, 0.4217),
    QColor::fromRgbF(0.1923, 0.2667, 0.4214),
    QColor::fromRgbF(0.1977, 0.2695, 0.4212),
    QColor::fromRgbF(0.2030, 0.2723, 0.4209),
    QColor::fromRgbF(0.2082, 0.2751, 0.4207),
    QColor::fromRgbF(0.2133, 0.2780, 0.4205),
    QColor::fromRgbF(0.2183, 0.2808, 0.4204),
    QColor::fromRgbF(0.2232, 0.2836, 0.4203),
    QColor::fromRgbF(0.2281, 0.2864, 0.4202),
    QColor::fromRgbF(0.2328, 0.2892, 0.4201),
    QColor::fromRgbF(0.2375, 0.2920, 0.4200),
    QColor::fromRgbF(0.2421, 0.2948, 0.4200),
    QColor::fromRgbF(0.2466, 0.2976, 0.4200),
    QColor::fromRgbF(0.2511, 0.3004, 0.4201),
    QColor::fromRgbF(0.2556, 0.3032, 0.4201),
    QColor::fromRgbF(0.2599, 0.3060, 0.4202),
    QColor::fromRgbF(0.2643, 0.3088, 0.4203),
    QColor::fromRgbF(0.2686, 0.3116, 0.4205),
    QColor::fromRgbF(0.2728, 0.3144, 0.4206),
    QColor::fromRgbF(0.2770, 0.3172, 0.4208),
    QColor::fromRgbF(0.2811, 0.3200, 0.4210),
    QColor::fromRgbF(0.2853, 0.3228, 0.4212),
    QColor::fromRgbF(0.2894, 0.3256, 0.4215),
    QColor::fromRgbF(0.2934, 0.3284, 0.4218),
    QColor::fromRgbF(0.2974, 0.3312, 0.4221),
    QColor::fromRgbF(0.3014, 0.3340, 0.4224),
    QColor::fromRgbF(0.3054, 0.3368, 0.4227),
    QColor::fromRgbF(0.3093, 0.3396, 0.4231),
    QColor::fromRgbF(0.3132, 0.3424, 0.4236),
    QColor::fromRgbF(0.3170, 0.3453, 0.4240),
    QColor::fromRgbF(0.3209, 0.3481, 0.4244),
    QColor::fromRgbF(0.3247, 0.3509, 0.4249),
    QColor::fromRgbF(0.3285, 0.3537, 0.4254),
    QColor::fromRgbF(0.3323, 0.3565, 0.4259),
    QColor::fromRgbF(0.3361, 0.3593, 0.4264),
    QColor::fromRgbF(0.3398, 0.3622, 0.4270),
    QColor::fromRgbF(0.3435, 0.3650, 0.4276),
    QColor::fromRgbF(0.3472, 0.3678, 0.4282),
    QColor::fromRgbF(0.3509, 0.3706, 0.4288),
    QColor::fromRgbF(0.3546, 0.3734, 0.4294),
    QColor::fromRgbF(0.3582, 0.3763, 0.4302),
    QColor::fromRgbF(0.3619, 0.3791, 0.4308),
    QColor::fromRgbF(0.3655, 0.3819, 0.4316),
    QColor::fromRgbF(0.3691, 0.3848, 0.4322),
    QColor::fromRgbF(0.3727, 0.3876, 0.4331),
    QColor::fromRgbF(0.3763, 0.3904, 0.4338),
    QColor::fromRgbF(0.3798, 0.3933, 0.4346),
    QColor::fromRgbF(0.3834, 0.3961, 0.4355),
    QColor::fromRgbF(0.3869, 0.3990, 0.4364),
    QColor::fromRgbF(0.3905, 0.4018, 0.4372),
    QColor::fromRgbF(0.3940, 0.4047, 0.4381),
    QColor::fromRgbF(0.3975, 0.4075, 0.4390),
    QColor::fromRgbF(0.4010, 0.4104, 0.4400),
    QColor::fromRgbF(0.4045, 0.4132, 0.4409),
    QColor::fromRgbF(0.4080, 0.4161, 0.4419),
    QColor::fromRgbF(0.4114, 0.4189, 0.4430),
    QColor::fromRgbF(0.4149, 0.4218, 0.4440),
    QColor::fromRgbF(0.4183, 0.4247, 0.4450),
    QColor::fromRgbF(0.4218, 0.4275, 0.4462),
    QColor::fromRgbF(0.4252, 0.4304, 0.4473),
    QColor::fromRgbF(0.4286, 0.4333, 0.4485),
    QColor::fromRgbF(0.4320, 0.4362, 0.4496),
    QColor::fromRgbF(0.4354, 0.4390, 0.4508),
    QColor::fromRgbF(0.4388, 0.4419, 0.4521),
    QColor::fromRgbF(0.4422, 0.4448, 0.4534),
    QColor::fromRgbF(0.4456, 0.4477, 0.4547),
    QColor::fromRgbF(0.4489, 0.4506, 0.4561),
    QColor::fromRgbF(0.4523, 0.4535, 0.4575),
    QColor::fromRgbF(0.4556, 0.4564, 0.4589),
    QColor::fromRgbF(0.4589, 0.4593, 0.4604),
    QColor::fromRgbF(0.4622, 0.4622, 0.4620),
    QColor::fromRgbF(0.4656, 0.4651, 0.4635),
    QColor::fromRgbF(0.4689, 0.4680, 0.4650),
    QColor::fromRgbF(0.4722, 0.4709, 0.4665),
    QColor::fromRgbF(0.4756, 0.4738, 0.4679),
    QColor::fromRgbF(0.4790, 0.4767, 0.4691),
    QColor::fromRgbF(0.4825, 0.4797, 0.4701),
    QColor::fromRgbF(0.4861, 0.4826, 0.4707),
    QColor::fromRgbF(0.4897, 0.4856, 0.4714),
    QColor::fromRgbF(0.4934, 0.4886, 0.4719),
    QColor::fromRgbF(0.4971, 0.4915, 0.4723),
    QColor::fromRgbF(0.5008, 0.4945, 0.4727),
    QColor::fromRgbF(0.5045, 0.4975, 0.4730),
    QColor::fromRgbF(0.5083, 0.5005, 0.4732),
    QColor::fromRgbF(0.5121, 0.5035, 0.4734),
    QColor::fromRgbF(0.5158, 0.5065, 0.4736),
    QColor::fromRgbF(0.5196, 0.5095, 0.4737),
    QColor::fromRgbF(0.5234, 0.5125, 0.4738),
    QColor::fromRgbF(0.5272, 0.5155, 0.4739),
    QColor::fromRgbF(0.5310, 0.5186, 0.4739),
    QColor::fromRgbF(0.5349, 0.5216, 0.4738),
    QColor::fromRgbF(0.5387, 0.5246, 0.4739),
    QColor::fromRgbF(0.5425, 0.5277, 0.4738),
    QColor::fromRgbF(0.5464, 0.5307, 0.4736),
    QColor::fromRgbF(0.5502, 0.5338, 0.4735),
    QColor::fromRgbF(0.5541, 0.5368, 0.4733),
    QColor::fromRgbF(0.5579, 0.5399, 0.4732),
    QColor::fromRgbF(0.5618, 0.5430, 0.4729),
    QColor::fromRgbF(0.5657, 0.5461, 0.4727),
    QColor::fromRgbF(0.5696, 0.5491, 0.4723),
    QColor::fromRgbF(0.5735, 0.5522, 0.4720),
    QColor::fromRgbF(0.5774, 0.5553, 0.4717),
    QColor::fromRgbF(0.5813, 0.5584, 0.4714),
    QColor::fromRgbF(0.5852, 0.5615, 0.4709),
    QColor::fromRgbF(0.5892, 0.5646, 0.4705),
    QColor::fromRgbF(0.5931, 0.5678, 0.4701),
    QColor::fromRgbF(0.5970, 0.5709, 0.4696),
    QColor::fromRgbF(0.6010, 0.5740, 0.4691),
    QColor::fromRgbF(0.6050, 0.5772, 0.4685),
    QColor::fromRgbF(0.6089, 0.5803, 0.4680),
    QColor::fromRgbF(0.6129, 0.5835, 0.4673),
    QColor::fromRgbF(0.6168, 0.5866, 0.4668),
    QColor::fromRgbF(0.6208, 0.5898, 0.4662),
    QColor::fromRgbF(0.6248, 0.5929, 0.4655),
    QColor::fromRgbF(0.6288, 0.5961, 0.4649),
    QColor::fromRgbF(0.6328, 0.5993, 0.4641),
    QColor::fromRgbF(0.6368, 0.6025, 0.4632),
    QColor::fromRgbF(0.6408, 0.6057, 0.4625),
    QColor::fromRgbF(0.6449, 0.6089, 0.4617),
    QColor::fromRgbF(0.6489, 0.6121, 0.4609),
    QColor::fromRgbF(0.6529, 0.6153, 0.4600),
    QColor::fromRgbF(0.6570, 0.6185, 0.4591),
    QColor::fromRgbF(0.6610, 0.6217, 0.4583),
    QColor::fromRgbF(0.6651, 0.6250, 0.4573),
    QColor::fromRgbF(0.6691, 0.6282, 0.4562),
    QColor::fromRgbF(0.6732, 0.6315, 0.4553),
    QColor::fromRgbF(0.6773, 0.6347, 0.4543),
    QColor::fromRgbF(0.6813, 0.6380, 0.4532),
    QColor::fromRgbF(0.6854, 0.6412, 0.4521),
    QColor::fromRgbF(0.6895, 0.6445, 0.4511),
    QColor::fromRgbF(0.6936, 0.6478, 0.4499),
    QColor::fromRgbF(0.6977, 0.6511, 0.4487),
    QColor::fromRgbF(0.7018, 0.6544, 0.4475),
    QColor::fromRgbF(0.7060, 0.6577, 0.4463),
    QColor::fromRgbF(0.7101, 0.6610, 0.4450),
    QColor::fromRgbF(0.7142, 0.6643, 0.4437),
    QColor::fromRgbF(0.7184, 0.6676, 0.4424),
    QColor::fromRgbF(0.7225, 0.6710, 0.4409),
    QColor::fromRgbF(0.7267, 0.6743, 0.4396),
    QColor::fromRgbF(0.7308, 0.6776, 0.4382),
    QColor::fromRgbF(0.7350, 0.6810, 0.4368),
    QColor::fromRgbF(0.7392, 0.6844, 0.4352),
    QColor::fromRgbF(0.7434, 0.6877, 0.4338),
    QColor::fromRgbF(0.7476, 0.6911, 0.4322),
    QColor::fromRgbF(0.7518, 0.6945, 0.4307),
    QColor::fromRgbF(0.7560, 0.6979, 0.4290),
    QColor::fromRgbF(0.7602, 0.7013, 0.4273),
    QColor::fromRgbF(0.7644, 0.7047, 0.4258),
    QColor::fromRgbF(0.7686, 0.7081, 0.4241),
    QColor::fromRgbF(0.7729, 0.7115, 0.4223),
    QColor::fromRgbF(0.7771, 0.7150, 0.4205),
    QColor::fromRgbF(0.7814, 0.7184, 0.4188),
    QColor::fromRgbF(0.7856, 0.7218, 0.4168),
    QColor::fromRgbF(0.7899, 0.7253, 0.4150),
    QColor::fromRgbF(0.7942, 0.7288, 0.4129),
    QColor::fromRgbF(0.7985, 0.7322, 0.4111),
    QColor::fromRgbF(0.8027, 0.7357, 0.4090),
    QColor::fromRgbF(0.8070, 0.7392, 0.4070),
    QColor::fromRgbF(0.8114, 0.7427, 0.4049),
    QColor::fromRgbF(0.8157, 0.7462, 0.4028),
    QColor::fromRgbF(0.8200, 0.7497, 0.4007),
    QColor::fromRgbF(0.8243, 0.7532, 0.3984),
    QColor::fromRgbF(0.8287, 0.7568, 0.3961),
    QColor::fromRgbF(0.8330, 0.7603, 0.3938),
    QColor::fromRgbF(0.8374, 0.7639, 0.3915),
    QColor::fromRgbF(0.8417, 0.7674, 0.3892),
    QColor::fromRgbF(0.8461, 0.7710, 0.3869),
    QColor::fromRgbF(0.8505, 0.7745, 0.3843),
    QColor::fromRgbF(0.8548, 0.7781, 0.3818),
    QColor::fromRgbF(0.8592, 0.7817, 0.3793),
    QColor::fromRgbF(0.8636, 0.7853, 0.3766),
    QColor::fromRgbF(0.8681, 0.7889, 0.3739),
    QColor::fromRgbF(0.8725, 0.7926, 0.3712),
    QColor::fromRgbF(0.8769, 0.7962, 0.3684),
    QColor::fromRgbF(0.8813, 0.7998, 0.3657),
    QColor::fromRgbF(0.8858, 0.8035, 0.3627),
    QColor::fromRgbF(0.8902, 0.8071, 0.3599),
    QColor::fromRgbF(0.8947, 0.8108, 0.3569),
    QColor::fromRgbF(0.8992, 0.8145, 0.3538),
    QColor::fromRgbF(0.9037, 0.8182, 0.3507),
    QColor::fromRgbF(0.9082, 0.8219, 0.3474),
    QColor::fromRgbF(0.9127, 0.8256, 0.3442),
    QColor::fromRgbF(0.9172, 0.8293, 0.3409),
    QColor::fromRgbF(0.9217, 0.8330, 0.3374),
    QColor::fromRgbF(0.9262, 0.8367, 0.3340),
    QColor::fromRgbF(0.9308, 0.8405, 0.3306),
    QColor::fromRgbF(0.9353, 0.8442, 0.3268),
    QColor::fromRgbF(0.9399, 0.8480, 0.3232),
    QColor::fromRgbF(0.9444, 0.8518, 0.3195),
    QColor::fromRgbF(0.9490, 0.8556, 0.3155),
    QColor::fromRgbF(0.9536, 0.8593, 0.3116),
    QColor::fromRgbF(0.9582, 0.8632, 0.3076),
    QColor::fromRgbF(0.9628, 0.8670, 0.3034),
    QColor::fromRgbF(0.9674, 0.8708, 0.2990),
    QColor::fromRgbF(0.9721, 0.8746, 0.2947),
    QColor::fromRgbF(0.9767, 0.8785, 0.2901),
    QColor::fromRgbF(0.9814, 0.8823, 0.2856),
    QColor::fromRgbF(0.9860, 0.8862, 0.2807),
    QColor::fromRgbF(0.9907, 0.8901, 0.2759),
    QColor::fromRgbF(0.9954, 0.8940, 0.2708),
    QColor::fromRgbF(1.0000, 0.8979, 0.2655),
    QColor::fromRgbF(1.0000, 0.9018, 0.2600),
    QColor::fromRgbF(1.0000, 0.9057, 0.2593),
    QColor::fromRgbF(1.0000, 0.9094, 0.2634),
    QColor::fromRgbF(1.0000, 0.9131, 0.2680),
    QColor::fromRgbF(1.0000, 0.9169, 0.2731)};

inline const std::vector<QColor> Github{
    QColor::fromRgbF(0.933333, 0.933333, 0.933333),
    QColor::fromRgbF(0.776470, 0.894117, 0.545098),
    QColor::fromRgbF(0.482352, 0.788235, 0.435294),
    QColor::fromRgbF(0.137254, 0.603921, 0.231372),
    QColor::fromRgbF(0.098039, 0.380392, 0.152941)};

inline const std::vector<QColor> CubeHelix{
    QColor::fromRgbF(0.000000, 0.000000, 0.000000),
    QColor::fromRgbF(0.006716, 0.002119, 0.005970),
    QColor::fromRgbF(0.013252, 0.004287, 0.012162),
    QColor::fromRgbF(0.019599, 0.006514, 0.018563),
    QColor::fromRgbF(0.025748, 0.008803, 0.025162),
    QColor::fromRgbF(0.031691, 0.011164, 0.031946),
    QColor::fromRgbF(0.037421, 0.013600, 0.038902),
    QColor::fromRgbF(0.042932, 0.016118, 0.046016),
    QColor::fromRgbF(0.048217, 0.018724, 0.053275),
    QColor::fromRgbF(0.053271, 0.021423, 0.060666),
    QColor::fromRgbF(0.058090, 0.024220, 0.068173),
    QColor::fromRgbF(0.062670, 0.027119, 0.075781),
    QColor::fromRgbF(0.067008, 0.030126, 0.083478),
    QColor::fromRgbF(0.071101, 0.033243, 0.091246),
    QColor::fromRgbF(0.074947, 0.036475, 0.099072),
    QColor::fromRgbF(0.078546, 0.039824, 0.106939),
    QColor::fromRgbF(0.081898, 0.043295, 0.114834),
    QColor::fromRgbF(0.085002, 0.046889, 0.122740),
    QColor::fromRgbF(0.087860, 0.050609, 0.130643),
    QColor::fromRgbF(0.090474, 0.054457, 0.138527),
    QColor::fromRgbF(0.092845, 0.058434, 0.146378),
    QColor::fromRgbF(0.094978, 0.062542, 0.154180),
    QColor::fromRgbF(0.096875, 0.066781, 0.161918),
    QColor::fromRgbF(0.098542, 0.071152, 0.169579),
    QColor::fromRgbF(0.099984, 0.075655, 0.177147),
    QColor::fromRgbF(0.101205, 0.080290, 0.184609),
    QColor::fromRgbF(0.102212, 0.085055, 0.191951),
    QColor::fromRgbF(0.103013, 0.089951, 0.199159),
    QColor::fromRgbF(0.103615, 0.094975, 0.206221),
    QColor::fromRgbF(0.104025, 0.100126, 0.213124),
    QColor::fromRgbF(0.104252, 0.105403, 0.219855),
    QColor::fromRgbF(0.104305, 0.110801, 0.226402),
    QColor::fromRgbF(0.104194, 0.116320, 0.232755),
    QColor::fromRgbF(0.103929, 0.121956, 0.238903),
    QColor::fromRgbF(0.103519, 0.127705, 0.244834),
    QColor::fromRgbF(0.102976, 0.133564, 0.250541),
    QColor::fromRgbF(0.102310, 0.139529, 0.256012),
    QColor::fromRgbF(0.101534, 0.145596, 0.261240),
    QColor::fromRgbF(0.100659, 0.151759, 0.266217),
    QColor::fromRgbF(0.099697, 0.158016, 0.270935),
    QColor::fromRgbF(0.098661, 0.164359, 0.275388),
    QColor::fromRgbF(0.097563, 0.170785, 0.279569),
    QColor::fromRgbF(0.096415, 0.177287, 0.283474),
    QColor::fromRgbF(0.095232, 0.183860, 0.287097),
    QColor::fromRgbF(0.094026, 0.190498, 0.290434),
    QColor::fromRgbF(0.092810, 0.197194, 0.293483),
    QColor::fromRgbF(0.091597, 0.203943, 0.296240),
    QColor::fromRgbF(0.090402, 0.210739, 0.298703),
    QColor::fromRgbF(0.089237, 0.217573, 0.300873),
    QColor::fromRgbF(0.088115, 0.224441, 0.302747),
    QColor::fromRgbF(0.087051, 0.231334, 0.304327),
    QColor::fromRgbF(0.086056, 0.238247, 0.305612),
    QColor::fromRgbF(0.085146, 0.245171, 0.306606),
    QColor::fromRgbF(0.084331, 0.252101, 0.307310),
    QColor::fromRgbF(0.083626, 0.259028, 0.307728),
    QColor::fromRgbF(0.083043, 0.265946, 0.307863),
    QColor::fromRgbF(0.082594, 0.272848, 0.307720),
    QColor::fromRgbF(0.082291, 0.279726, 0.307304),
    QColor::fromRgbF(0.082148, 0.286573, 0.306621),
    QColor::fromRgbF(0.082174, 0.293383, 0.305677),
    QColor::fromRgbF(0.082381, 0.300147, 0.304480),
    QColor::fromRgbF(0.082780, 0.306860, 0.303037),
    QColor::fromRgbF(0.083383, 0.313514, 0.301356),
    QColor::fromRgbF(0.084198, 0.320102, 0.299448),
    QColor::fromRgbF(0.085235, 0.326618, 0.297320),
    QColor::fromRgbF(0.086504, 0.333055, 0.294984),
    QColor::fromRgbF(0.088014, 0.339406, 0.292449),
    QColor::fromRgbF(0.089772, 0.345666, 0.289728),
    QColor::fromRgbF(0.091787, 0.351829, 0.286831),
    QColor::fromRgbF(0.094066, 0.357887, 0.283771),
    QColor::fromRgbF(0.096615, 0.363836, 0.280560),
    QColor::fromRgbF(0.099441, 0.369671, 0.277211),
    QColor::fromRgbF(0.102549, 0.375385, 0.273736),
    QColor::fromRgbF(0.105944, 0.380974, 0.270151),
    QColor::fromRgbF(0.109630, 0.386433, 0.266468),
    QColor::fromRgbF(0.113611, 0.391757, 0.262703),
    QColor::fromRgbF(0.117891, 0.396943, 0.258868),
    QColor::fromRgbF(0.122472, 0.401985, 0.254979),
    QColor::fromRgbF(0.127356, 0.406881, 0.251051),
    QColor::fromRgbF(0.132543, 0.411627, 0.247099),
    QColor::fromRgbF(0.138035, 0.416220, 0.243137),
    QColor::fromRgbF(0.143832, 0.420656, 0.239182),
    QColor::fromRgbF(0.149933, 0.424934, 0.235247),
    QColor::fromRgbF(0.156336, 0.429052, 0.231350),
    QColor::fromRgbF(0.163040, 0.433007, 0.227504),
    QColor::fromRgbF(0.170042, 0.436798, 0.223726),
    QColor::fromRgbF(0.177339, 0.440423, 0.220029),
    QColor::fromRgbF(0.184927, 0.443882, 0.216431),
    QColor::fromRgbF(0.192802, 0.447175, 0.212944),
    QColor::fromRgbF(0.200958, 0.450301, 0.209585),
    QColor::fromRgbF(0.209391, 0.453260, 0.206367),
    QColor::fromRgbF(0.218092, 0.456053, 0.203306),
    QColor::fromRgbF(0.227057, 0.458680, 0.200415),
    QColor::fromRgbF(0.236277, 0.461144, 0.197707),
    QColor::fromRgbF(0.245744, 0.463444, 0.195197),
    QColor::fromRgbF(0.255451, 0.465584, 0.192898),
    QColor::fromRgbF(0.265388, 0.467565, 0.190822),
    QColor::fromRgbF(0.275545, 0.469391, 0.188982),
    QColor::fromRgbF(0.285913, 0.471062, 0.187389),
    QColor::fromRgbF(0.296481, 0.472584, 0.186055),
    QColor::fromRgbF(0.307239, 0.473959, 0.184992),
    QColor::fromRgbF(0.318175, 0.475191, 0.184208),
    QColor::fromRgbF(0.329277, 0.476285, 0.183716),
    QColor::fromRgbF(0.340534, 0.477243, 0.183523),
    QColor::fromRgbF(0.351934, 0.478072, 0.183638),
    QColor::fromRgbF(0.363463, 0.478776, 0.184071),
    QColor::fromRgbF(0.375109, 0.479360, 0.184829),
    QColor::fromRgbF(0.386858, 0.479829, 0.185918),
    QColor::fromRgbF(0.398697, 0.480190, 0.187345),
    QColor::fromRgbF(0.410613, 0.480448, 0.189115),
    QColor::fromRgbF(0.422591, 0.480609, 0.191235),
    QColor::fromRgbF(0.434618, 0.480679, 0.193708),
    QColor::fromRgbF(0.446680, 0.480665, 0.196538),
    QColor::fromRgbF(0.458762, 0.480574, 0.199728),
    QColor::fromRgbF(0.470850, 0.480412, 0.203280),
    QColor::fromRgbF(0.482930, 0.480186, 0.207197),
    QColor::fromRgbF(0.494987, 0.479903, 0.211478),
    QColor::fromRgbF(0.507008, 0.479572, 0.216124),
    QColor::fromRgbF(0.518978, 0.479198, 0.221136),
    QColor::fromRgbF(0.530883, 0.478789, 0.226510),
    QColor::fromRgbF(0.542708, 0.478353, 0.232247),
    QColor::fromRgbF(0.554441, 0.477898, 0.238342),
    QColor::fromRgbF(0.566067, 0.477430, 0.244794),
    QColor::fromRgbF(0.577573, 0.476958, 0.251597),
    QColor::fromRgbF(0.588945, 0.476490, 0.258747),
    QColor::fromRgbF(0.600171, 0.476032, 0.266239),
    QColor::fromRgbF(0.611237, 0.475592, 0.274067),
    QColor::fromRgbF(0.622132, 0.475178, 0.282223),
    QColor::fromRgbF(0.632842, 0.474798, 0.290702),
    QColor::fromRgbF(0.643357, 0.474459, 0.299495),
    QColor::fromRgbF(0.653665, 0.474168, 0.308593),
    QColor::fromRgbF(0.663755, 0.473933, 0.317987),
    QColor::fromRgbF(0.673616, 0.473761, 0.327668),
    QColor::fromRgbF(0.683239, 0.473658, 0.337626),
    QColor::fromRgbF(0.692613, 0.473632, 0.347849),
    QColor::fromRgbF(0.701729, 0.473690, 0.358327),
    QColor::fromRgbF(0.710579, 0.473838, 0.369047),
    QColor::fromRgbF(0.719155, 0.474083, 0.379998),
    QColor::fromRgbF(0.727448, 0.474430, 0.391167),
    QColor::fromRgbF(0.735453, 0.474886, 0.402541),
    QColor::fromRgbF(0.743162, 0.475457, 0.414106),
    QColor::fromRgbF(0.750569, 0.476148, 0.425849),
    QColor::fromRgbF(0.757669, 0.476964, 0.437755),
    QColor::fromRgbF(0.764458, 0.477911, 0.449811),
    QColor::fromRgbF(0.770932, 0.478994, 0.462001),
    QColor::fromRgbF(0.777086, 0.480216, 0.474310),
    QColor::fromRgbF(0.782918, 0.481583, 0.486725),
    QColor::fromRgbF(0.788426, 0.483098, 0.499228),
    QColor::fromRgbF(0.793609, 0.484765, 0.511805),
    QColor::fromRgbF(0.798465, 0.486587, 0.524441),
    QColor::fromRgbF(0.802993, 0.488567, 0.537119),
    QColor::fromRgbF(0.807196, 0.490708, 0.549824),
    QColor::fromRgbF(0.811072, 0.493013, 0.562540),
    QColor::fromRgbF(0.814625, 0.495483, 0.575253),
    QColor::fromRgbF(0.817855, 0.498121, 0.587945),
    QColor::fromRgbF(0.820767, 0.500927, 0.600602),
    QColor::fromRgbF(0.823364, 0.503903, 0.613208),
    QColor::fromRgbF(0.825649, 0.507050, 0.625748),
    QColor::fromRgbF(0.827628, 0.510368, 0.638207),
    QColor::fromRgbF(0.829305, 0.513857, 0.650570),
    QColor::fromRgbF(0.830688, 0.517516, 0.662822),
    QColor::fromRgbF(0.831781, 0.521346, 0.674949),
    QColor::fromRgbF(0.832593, 0.525345, 0.686938),
    QColor::fromRgbF(0.833130, 0.529511, 0.698773),
    QColor::fromRgbF(0.833402, 0.533844, 0.710443),
    QColor::fromRgbF(0.833416, 0.538342, 0.721933),
    QColor::fromRgbF(0.833181, 0.543001, 0.733232),
    QColor::fromRgbF(0.832708, 0.547820, 0.744327),
    QColor::fromRgbF(0.832006, 0.552795, 0.755206),
    QColor::fromRgbF(0.831086, 0.557924, 0.765859),
    QColor::fromRgbF(0.829958, 0.563202, 0.776274),
    QColor::fromRgbF(0.828633, 0.568627, 0.786443),
    QColor::fromRgbF(0.827124, 0.574193, 0.796354),
    QColor::fromRgbF(0.825442, 0.579897, 0.805999),
    QColor::fromRgbF(0.823599, 0.585733, 0.815370),
    QColor::fromRgbF(0.821608, 0.591698, 0.824459),
    QColor::fromRgbF(0.819482, 0.597785, 0.833258),
    QColor::fromRgbF(0.817233, 0.603990, 0.841761),
    QColor::fromRgbF(0.814875, 0.610307, 0.849963),
    QColor::fromRgbF(0.812421, 0.616730, 0.857858),
    QColor::fromRgbF(0.809884, 0.623252, 0.865441),
    QColor::fromRgbF(0.807278, 0.629869, 0.872709),
    QColor::fromRgbF(0.804617, 0.636573, 0.879658),
    QColor::fromRgbF(0.801914, 0.643359, 0.886286),
    QColor::fromRgbF(0.799183, 0.650218, 0.892592),
    QColor::fromRgbF(0.796438, 0.657146, 0.898574),
    QColor::fromRgbF(0.793692, 0.664134, 0.904231),
    QColor::fromRgbF(0.790959, 0.671176, 0.909565),
    QColor::fromRgbF(0.788253, 0.678264, 0.914576),
    QColor::fromRgbF(0.785586, 0.685392, 0.919267),
    QColor::fromRgbF(0.782973, 0.692553, 0.923639),
    QColor::fromRgbF(0.780425, 0.699738, 0.927695),
    QColor::fromRgbF(0.777957, 0.706942, 0.931441),
    QColor::fromRgbF(0.775579, 0.714157, 0.934879),
    QColor::fromRgbF(0.773305, 0.721375, 0.938016),
    QColor::fromRgbF(0.771147, 0.728589, 0.940857),
    QColor::fromRgbF(0.769116, 0.735793, 0.943409),
    QColor::fromRgbF(0.767224, 0.742979, 0.945678),
    QColor::fromRgbF(0.765481, 0.750140, 0.947673),
    QColor::fromRgbF(0.763898, 0.757269, 0.949402),
    QColor::fromRgbF(0.762485, 0.764360, 0.950874),
    QColor::fromRgbF(0.761251, 0.771405, 0.952098),
    QColor::fromRgbF(0.760207, 0.778399, 0.953084),
    QColor::fromRgbF(0.759360, 0.785335, 0.953843),
    QColor::fromRgbF(0.758718, 0.792207, 0.954386),
    QColor::fromRgbF(0.758290, 0.799008, 0.954724),
    QColor::fromRgbF(0.758082, 0.805734, 0.954869),
    QColor::fromRgbF(0.758101, 0.812378, 0.954833),
    QColor::fromRgbF(0.758353, 0.818934, 0.954629),
    QColor::fromRgbF(0.758842, 0.825399, 0.954270),
    QColor::fromRgbF(0.759575, 0.831767, 0.953769),
    QColor::fromRgbF(0.760554, 0.838033, 0.953140),
    QColor::fromRgbF(0.761784, 0.844192, 0.952397),
    QColor::fromRgbF(0.763267, 0.850242, 0.951554),
    QColor::fromRgbF(0.765006, 0.856178, 0.950625),
    QColor::fromRgbF(0.767001, 0.861997, 0.949624),
    QColor::fromRgbF(0.769255, 0.867695, 0.948567),
    QColor::fromRgbF(0.771766, 0.873270, 0.947467),
    QColor::fromRgbF(0.774535, 0.878718, 0.946340),
    QColor::fromRgbF(0.777561, 0.884039, 0.945201),
    QColor::fromRgbF(0.780841, 0.889230, 0.944063),
    QColor::fromRgbF(0.784374, 0.894289, 0.942942),
    QColor::fromRgbF(0.788156, 0.899216, 0.941853),
    QColor::fromRgbF(0.792184, 0.904010, 0.940809),
    QColor::fromRgbF(0.796453, 0.908669, 0.939825),
    QColor::fromRgbF(0.800958, 0.913194, 0.938916),
    QColor::fromRgbF(0.805694, 0.917586, 0.938095),
    QColor::fromRgbF(0.810654, 0.921845, 0.937376),
    QColor::fromRgbF(0.815832, 0.925971, 0.936772),
    QColor::fromRgbF(0.821221, 0.929967, 0.936297),
    QColor::fromRgbF(0.826811, 0.933833, 0.935962),
    QColor::fromRgbF(0.832595, 0.937572, 0.935781),
    QColor::fromRgbF(0.838565, 0.941187, 0.935766),
    QColor::fromRgbF(0.844709, 0.944679, 0.935927),
    QColor::fromRgbF(0.851018, 0.948053, 0.936275),
    QColor::fromRgbF(0.857482, 0.951311, 0.936822),
    QColor::fromRgbF(0.864090, 0.954457, 0.937578),
    QColor::fromRgbF(0.870830, 0.957495, 0.938550),
    QColor::fromRgbF(0.877690, 0.960430, 0.939749),
    QColor::fromRgbF(0.884659, 0.963266, 0.941183),
    QColor::fromRgbF(0.891723, 0.966009, 0.942858),
    QColor::fromRgbF(0.898871, 0.968662, 0.944783),
    QColor::fromRgbF(0.906088, 0.971233, 0.946962),
    QColor::fromRgbF(0.913362, 0.973726, 0.949402),
    QColor::fromRgbF(0.920679, 0.976147, 0.952108),
    QColor::fromRgbF(0.928026, 0.978504, 0.955083),
    QColor::fromRgbF(0.935387, 0.980802, 0.958331),
    QColor::fromRgbF(0.942750, 0.983048, 0.961854),
    QColor::fromRgbF(0.950101, 0.985249, 0.965654),
    QColor::fromRgbF(0.957424, 0.987412, 0.969733),
    QColor::fromRgbF(0.964706, 0.989543, 0.974090),
    QColor::fromRgbF(0.971932, 0.991652, 0.978724),
    QColor::fromRgbF(0.979088, 0.993744, 0.983635),
    QColor::fromRgbF(0.986161, 0.995828, 0.988820),
    QColor::fromRgbF(0.993136, 0.997910, 0.994276),
    QColor::fromRgbF(1.000000, 1.000000, 1.000000)};

inline const std::vector<QColor> HSV{
    QColor::fromRgbF(1.0000, 0.0000, 0.0000),
    QColor::fromRgbF(1.0000, 0.0234, 0.0000),
    QColor::fromRgbF(1.0000, 0.0469, 0.0000),
    QColor::fromRgbF(1.0000, 0.0703, 0.0000),
    QColor::fromRgbF(1.0000, 0.0938, 0.0000),
    QColor::fromRgbF(1.0000, 0.1172, 0.0000),
    QColor::fromRgbF(1.0000, 0.1406, 0.0000),
    QColor::fromRgbF(1.0000, 0.1641, 0.0000),
    QColor::fromRgbF(1.0000, 0.1875, 0.0000),
    QColor::fromRgbF(1.0000, 0.2109, 0.0000),
    QColor::fromRgbF(1.0000, 0.2344, 0.0000),
    QColor::fromRgbF(1.0000, 0.2578, 0.0000),
    QColor::fromRgbF(1.0000, 0.2812, 0.0000),
    QColor::fromRgbF(1.0000, 0.3047, 0.0000),
    QColor::fromRgbF(1.0000, 0.3281, 0.0000),
    QColor::fromRgbF(1.0000, 0.3516, 0.0000),
    QColor::fromRgbF(1.0000, 0.3750, 0.0000),
    QColor::fromRgbF(1.0000, 0.3984, 0.0000),
    QColor::fromRgbF(1.0000, 0.4219, 0.0000),
    QColor::fromRgbF(1.0000, 0.4453, 0.0000),
    QColor::fromRgbF(1.0000, 0.4688, 0.0000),
    QColor::fromRgbF(1.0000, 0.4922, 0.0000),
    QColor::fromRgbF(1.0000, 0.5156, 0.0000),
    QColor::fromRgbF(1.0000, 0.5391, 0.0000),
    QColor::fromRgbF(1.0000, 0.5625, 0.0000),
    QColor::fromRgbF(1.0000, 0.5859, 0.0000),
    QColor::fromRgbF(1.0000, 0.6094, 0.0000),
    QColor::fromRgbF(1.0000, 0.6328, 0.0000),
    QColor::fromRgbF(1.0000, 0.6562, 0.0000),
    QColor::fromRgbF(1.0000, 0.6797, 0.0000),
    QColor::fromRgbF(1.0000, 0.7031, 0.0000),
    QColor::fromRgbF(1.0000, 0.7266, 0.0000),
    QColor::fromRgbF(1.0000, 0.7500, 0.0000),
    QColor::fromRgbF(1.0000, 0.7734, 0.0000),
    QColor::fromRgbF(1.0000, 0.7969, 0.0000),
    QColor::fromRgbF(1.0000, 0.8203, 0.0000),
    QColor::fromRgbF(1.0000, 0.8438, 0.0000),
    QColor::fromRgbF(1.0000, 0.8672, 0.0000),
    QColor::fromRgbF(1.0000, 0.8906, 0.0000),
    QColor::fromRgbF(1.0000, 0.9141, 0.0000),
    QColor::fromRgbF(1.0000, 0.9375, 0.0000),
    QColor::fromRgbF(1.0000, 0.9609, 0.0000),
    QColor::fromRgbF(1.0000, 0.9844, 0.0000),
    QColor::fromRgbF(0.9922, 1.0000, 0.0000),
    QColor::fromRgbF(0.9688, 1.0000, 0.0000),
    QColor::fromRgbF(0.9453, 1.0000, 0.0000),
    QColor::fromRgbF(0.9219, 1.0000, 0.0000),
    QColor::fromRgbF(0.8984, 1.0000, 0.0000),
    QColor::fromRgbF(0.8750, 1.0000, 0.0000),
    QColor::fromRgbF(0.8516, 1.0000, 0.0000),
    QColor::fromRgbF(0.8281, 1.0000, 0.0000),
    QColor::fromRgbF(0.8047, 1.0000, 0.0000),
    QColor::fromRgbF(0.7812, 1.0000, 0.0000),
    QColor::fromRgbF(0.7578, 1.0000, 0.0000),
    QColor::fromRgbF(0.7344, 1.0000, 0.0000),
    QColor::fromRgbF(0.7109, 1.0000, 0.0000),
    QColor::fromRgbF(0.6875, 1.0000, 0.0000),
    QColor::fromRgbF(0.6641, 1.0000, 0.0000),
    QColor::fromRgbF(0.6406, 1.0000, 0.0000),
    QColor::fromRgbF(0.6172, 1.0000, 0.0000),
    QColor::fromRgbF(0.5938, 1.0000, 0.0000),
    QColor::fromRgbF(0.5703, 1.0000, 0.0000),
    QColor::fromRgbF(0.5469, 1.0000, 0.0000),
    QColor::fromRgbF(0.5234, 1.0000, 0.0000),
    QColor::fromRgbF(0.5000, 1.0000, 0.0000),
    QColor::fromRgbF(0.4766, 1.0000, 0.0000),
    QColor::fromRgbF(0.4531, 1.0000, 0.0000),
    QColor::fromRgbF(0.4297, 1.0000, 0.0000),
    QColor::fromRgbF(0.4062, 1.0000, 0.0000),
    QColor::fromRgbF(0.3828, 1.0000, 0.0000),
    QColor::fromRgbF(0.3594, 1.0000, 0.0000),
    QColor::fromRgbF(0.3359, 1.0000, 0.0000),
    QColor::fromRgbF(0.3125, 1.0000, 0.0000),
    QColor::fromRgbF(0.2891, 1.0000, 0.0000),
    QColor::fromRgbF(0.2656, 1.0000, 0.0000),
    QColor::fromRgbF(0.2422, 1.0000, 0.0000),
    QColor::fromRgbF(0.2188, 1.0000, 0.0000),
    QColor::fromRgbF(0.1953, 1.0000, 0.0000),
    QColor::fromRgbF(0.1719, 1.0000, 0.0000),
    QColor::fromRgbF(0.1484, 1.0000, 0.0000),
    QColor::fromRgbF(0.1250, 1.0000, 0.0000),
    QColor::fromRgbF(0.1016, 1.0000, 0.0000),
    QColor::fromRgbF(0.0781, 1.0000, 0.0000),
    QColor::fromRgbF(0.0547, 1.0000, 0.0000),
    QColor::fromRgbF(0.0312, 1.0000, 0.0000),
    QColor::fromRgbF(0.0078, 1.0000, 0.0000),
    QColor::fromRgbF(0.0000, 1.0000, 0.0156),
    QColor::fromRgbF(0.0000, 1.0000, 0.0391),
    QColor::fromRgbF(0.0000, 1.0000, 0.0625),
    QColor::fromRgbF(0.0000, 1.0000, 0.0859),
    QColor::fromRgbF(0.0000, 1.0000, 0.1094),
    QColor::fromRgbF(0.0000, 1.0000, 0.1328),
    QColor::fromRgbF(0.0000, 1.0000, 0.1562),
    QColor::fromRgbF(0.0000, 1.0000, 0.1797),
    QColor::fromRgbF(0.0000, 1.0000, 0.2031),
    QColor::fromRgbF(0.0000, 1.0000, 0.2266),
    QColor::fromRgbF(0.0000, 1.0000, 0.2500),
    QColor::fromRgbF(0.0000, 1.0000, 0.2734),
    QColor::fromRgbF(0.0000, 1.0000, 0.2969),
    QColor::fromRgbF(0.0000, 1.0000, 0.3203),
    QColor::fromRgbF(0.0000, 1.0000, 0.3438),
    QColor::fromRgbF(0.0000, 1.0000, 0.3672),
    QColor::fromRgbF(0.0000, 1.0000, 0.3906),
    QColor::fromRgbF(0.0000, 1.0000, 0.4141),
    QColor::fromRgbF(0.0000, 1.0000, 0.4375),
    QColor::fromRgbF(0.0000, 1.0000, 0.4609),
    QColor::fromRgbF(0.0000, 1.0000, 0.4844),
    QColor::fromRgbF(0.0000, 1.0000, 0.5078),
    QColor::fromRgbF(0.0000, 1.0000, 0.5312),
    QColor::fromRgbF(0.0000, 1.0000, 0.5547),
    QColor::fromRgbF(0.0000, 1.0000, 0.5781),
    QColor::fromRgbF(0.0000, 1.0000, 0.6016),
    QColor::fromRgbF(0.0000, 1.0000, 0.6250),
    QColor::fromRgbF(0.0000, 1.0000, 0.6484),
    QColor::fromRgbF(0.0000, 1.0000, 0.6719),
    QColor::fromRgbF(0.0000, 1.0000, 0.6953),
    QColor::fromRgbF(0.0000, 1.0000, 0.7188),
    QColor::fromRgbF(0.0000, 1.0000, 0.7422),
    QColor::fromRgbF(0.0000, 1.0000, 0.7656),
    QColor::fromRgbF(0.0000, 1.0000, 0.7891),
    QColor::fromRgbF(0.0000, 1.0000, 0.8125),
    QColor::fromRgbF(0.0000, 1.0000, 0.8359),
    QColor::fromRgbF(0.0000, 1.0000, 0.8594),
    QColor::fromRgbF(0.0000, 1.0000, 0.8828),
    QColor::fromRgbF(0.0000, 1.0000, 0.9062),
    QColor::fromRgbF(0.0000, 1.0000, 0.9297),
    QColor::fromRgbF(0.0000, 1.0000, 0.9531),
    QColor::fromRgbF(0.0000, 1.0000, 0.9766),
    QColor::fromRgbF(0.0000, 1.0000, 1.0000),
    QColor::fromRgbF(0.0000, 0.9766, 1.0000),
    QColor::fromRgbF(0.0000, 0.9531, 1.0000),
    QColor::fromRgbF(0.0000, 0.9297, 1.0000),
    QColor::fromRgbF(0.0000, 0.9062, 1.0000),
    QColor::fromRgbF(0.0000, 0.8828, 1.0000),
    QColor::fromRgbF(0.0000, 0.8594, 1.0000),
    QColor::fromRgbF(0.0000, 0.8359, 1.0000),
    QColor::fromRgbF(0.0000, 0.8125, 1.0000),
    QColor::fromRgbF(0.0000, 0.7891, 1.0000),
    QColor::fromRgbF(0.0000, 0.7656, 1.0000),
    QColor::fromRgbF(0.0000, 0.7422, 1.0000),
    QColor::fromRgbF(0.0000, 0.7188, 1.0000),
    QColor::fromRgbF(0.0000, 0.6953, 1.0000),
    QColor::fromRgbF(0.0000, 0.6719, 1.0000),
    QColor::fromRgbF(0.0000, 0.6484, 1.0000),
    QColor::fromRgbF(0.0000, 0.6250, 1.0000),
    QColor::fromRgbF(0.0000, 0.6016, 1.0000),
    QColor::fromRgbF(0.0000, 0.5781, 1.0000),
    QColor::fromRgbF(0.0000, 0.5547, 1.0000),
    QColor::fromRgbF(0.0000, 0.5312, 1.0000),
    QColor::fromRgbF(0.0000, 0.5078, 1.0000),
    QColor::fromRgbF(0.0000, 0.4844, 1.0000),
    QColor::fromRgbF(0.0000, 0.4609, 1.0000),
    QColor::fromRgbF(0.0000, 0.4375, 1.0000),
    QColor::fromRgbF(0.0000, 0.4141, 1.0000),
    QColor::fromRgbF(0.0000, 0.3906, 1.0000),
    QColor::fromRgbF(0.0000, 0.3672, 1.0000),
    QColor::fromRgbF(0.0000, 0.3438, 1.0000),
    QColor::fromRgbF(0.0000, 0.3203, 1.0000),
    QColor::fromRgbF(0.0000, 0.2969, 1.0000),
    QColor::fromRgbF(0.0000, 0.2734, 1.0000),
    QColor::fromRgbF(0.0000, 0.2500, 1.0000),
    QColor::fromRgbF(0.0000, 0.2266, 1.0000),
    QColor::fromRgbF(0.0000, 0.2031, 1.0000),
    QColor::fromRgbF(0.0000, 0.1797, 1.0000),
    QColor::fromRgbF(0.0000, 0.1562, 1.0000),
    QColor::fromRgbF(0.0000, 0.1328, 1.0000),
    QColor::fromRgbF(0.0000, 0.1094, 1.0000),
    QColor::fromRgbF(0.0000, 0.0859, 1.0000),
    QColor::fromRgbF(0.0000, 0.0625, 1.0000),
    QColor::fromRgbF(0.0000, 0.0391, 1.0000),
    QColor::fromRgbF(0.0000, 0.0156, 1.0000),
    QColor::fromRgbF(0.0078, 0.0000, 1.0000),
    QColor::fromRgbF(0.0312, 0.0000, 1.0000),
    QColor::fromRgbF(0.0547, 0.0000, 1.0000),
    QColor::fromRgbF(0.0781, 0.0000, 1.0000),
    QColor::fromRgbF(0.1016, 0.0000, 1.0000),
    QColor::fromRgbF(0.1250, 0.0000, 1.0000),
    QColor::fromRgbF(0.1484, 0.0000, 1.0000),
    QColor::fromRgbF(0.1719, 0.0000, 1.0000),
    QColor::fromRgbF(0.1953, 0.0000, 1.0000),
    QColor::fromRgbF(0.2188, 0.0000, 1.0000),
    QColor::fromRgbF(0.2422, 0.0000, 1.0000),
    QColor::fromRgbF(0.2656, 0.0000, 1.0000),
    QColor::fromRgbF(0.2891, 0.0000, 1.0000),
    QColor::fromRgbF(0.3125, 0.0000, 1.0000),
    QColor::fromRgbF(0.3359, 0.0000, 1.0000),
    QColor::fromRgbF(0.3594, 0.0000, 1.0000),
    QColor::fromRgbF(0.3828, 0.0000, 1.0000),
    QColor::fromRgbF(0.4062, 0.0000, 1.0000),
    QColor::fromRgbF(0.4297, 0.0000, 1.0000),
    QColor::fromRgbF(0.4531, 0.0000, 1.0000),
    QColor::fromRgbF(0.4766, 0.0000, 1.0000),
    QColor::fromRgbF(0.5000, 0.0000, 1.0000),
    QColor::fromRgbF(0.5234, 0.0000, 1.0000),
    QColor::fromRgbF(0.5469, 0.0000, 1.0000),
    QColor::fromRgbF(0.5703, 0.0000, 1.0000),
    QColor::fromRgbF(0.5938, 0.0000, 1.0000),
    QColor::fromRgbF(0.6172, 0.0000, 1.0000),
    QColor::fromRgbF(0.6406, 0.0000, 1.0000),
    QColor::fromRgbF(0.6641, 0.0000, 1.0000),
    QColor::fromRgbF(0.6875, 0.0000, 1.0000),
    QColor::fromRgbF(0.7109, 0.0000, 1.0000),
    QColor::fromRgbF(0.7344, 0.0000, 1.0000),
    QColor::fromRgbF(0.7578, 0.0000, 1.0000),
    QColor::fromRgbF(0.7812, 0.0000, 1.0000),
    QColor::fromRgbF(0.8047, 0.0000, 1.0000),
    QColor::fromRgbF(0.8281, 0.0000, 1.0000),
    QColor::fromRgbF(0.8516, 0.0000, 1.0000),
    QColor::fromRgbF(0.8750, 0.0000, 1.0000),
    QColor::fromRgbF(0.8984, 0.0000, 1.0000),
    QColor::fromRgbF(0.9219, 0.0000, 1.0000),
    QColor::fromRgbF(0.9453, 0.0000, 1.0000),
    QColor::fromRgbF(0.9688, 0.0000, 1.0000),
    QColor::fromRgbF(0.9922, 0.0000, 1.0000),
    QColor::fromRgbF(1.0000, 0.0000, 0.9844),
    QColor::fromRgbF(1.0000, 0.0000, 0.9609),
    QColor::fromRgbF(1.0000, 0.0000, 0.9375),
    QColor::fromRgbF(1.0000, 0.0000, 0.9141),
    QColor::fromRgbF(1.0000, 0.0000, 0.8906),
    QColor::fromRgbF(1.0000, 0.0000, 0.8672),
    QColor::fromRgbF(1.0000, 0.0000, 0.8438),
    QColor::fromRgbF(1.0000, 0.0000, 0.8203),
    QColor::fromRgbF(1.0000, 0.0000, 0.7969),
    QColor::fromRgbF(1.0000, 0.0000, 0.7734),
    QColor::fromRgbF(1.0000, 0.0000, 0.7500),
    QColor::fromRgbF(1.0000, 0.0000, 0.7266),
    QColor::fromRgbF(1.0000, 0.0000, 0.7031),
    QColor::fromRgbF(1.0000, 0.0000, 0.6797),
    QColor::fromRgbF(1.0000, 0.0000, 0.6562),
    QColor::fromRgbF(1.0000, 0.0000, 0.6328),
    QColor::fromRgbF(1.0000, 0.0000, 0.6094),
    QColor::fromRgbF(1.0000, 0.0000, 0.5859),
    QColor::fromRgbF(1.0000, 0.0000, 0.5625),
    QColor::fromRgbF(1.0000, 0.0000, 0.5391),
    QColor::fromRgbF(1.0000, 0.0000, 0.5156),
    QColor::fromRgbF(1.0000, 0.0000, 0.4922),
    QColor::fromRgbF(1.0000, 0.0000, 0.4688),
    QColor::fromRgbF(1.0000, 0.0000, 0.4453),
    QColor::fromRgbF(1.0000, 0.0000, 0.4219),
    QColor::fromRgbF(1.0000, 0.0000, 0.3984),
    QColor::fromRgbF(1.0000, 0.0000, 0.3750),
    QColor::fromRgbF(1.0000, 0.0000, 0.3516),
    QColor::fromRgbF(1.0000, 0.0000, 0.3281),
    QColor::fromRgbF(1.0000, 0.0000, 0.3047),
    QColor::fromRgbF(1.0000, 0.0000, 0.2812),
    QColor::fromRgbF(1.0000, 0.0000, 0.2578),
    QColor::fromRgbF(1.0000, 0.0000, 0.2344),
    QColor::fromRgbF(1.0000, 0.0000, 0.2109),
    QColor::fromRgbF(1.0000, 0.0000, 0.1875),
    QColor::fromRgbF(1.0000, 0.0000, 0.1641),
    QColor::fromRgbF(1.0000, 0.0000, 0.1406),
    QColor::fromRgbF(1.0000, 0.0000, 0.1172),
    QColor::fromRgbF(1.0000, 0.0000, 0.0938),
    QColor::fromRgbF(1.0000, 0.0000, 0.0703),
    QColor::fromRgbF(1.0000, 0.0000, 0.0469),
    QColor::fromRgbF(1.0000, 0.0000, 0.0234),
};
inline const std::vector<QColor> Archambault{
    QColor("#381a61"), QColor("#f9d14a"), QColor("#ab3329"), QColor("#88a0dc"),
    QColor("#e78429"), QColor("#ed968c"), QColor("#7c4b73"),
};
inline const std::vector<QColor> Austria{
    QColor("#a40000"), QColor("#16317d"), QColor("#007e2f"), QColor("#ffcd12"),
    QColor("#721b3e"), QColor("#b86092"), QColor("#00b7a7"),
};
inline const std::vector<QColor> Benedictus{
    QColor("#c5daf6"), QColor("#f9b4c9"), QColor("#6996e3"), QColor("#9a133d"),
    QColor("#ffffff"), QColor("#d8527c"), QColor("#1a318b"), QColor("#f28aaa"),
    QColor("#eaf3ff"), QColor("#b93961"), QColor("#4060c8"), QColor("#f9e0e8"),
    QColor("#a1c2ed"),
};
inline const std::vector<QColor> Cassatt1{
    QColor("#e3aba7"), QColor("#9d9dc7"), QColor("#b1615c"), QColor("#5a5a83"),
    QColor("#edd7d9"), QColor("#c9c9dd"), QColor("#d88782"), QColor("#8282aa"),
};
inline const std::vector<QColor> Cassatt2{
    QColor("#7fa074"), QColor("#90719f"), QColor("#2c4b27"), QColor("#2d223c"),
    QColor("#dec5da"), QColor("#c1d1aa"), QColor("#574571"), QColor("#0e2810"),
    QColor("#b695bc"), QColor("#466c4b"),
};
inline const std::vector<QColor> Cross{
    QColor("#eb7926"), QColor("#62929a"), QColor("#c969a1"),
    QColor("#004f63"), QColor("#ce4441"), QColor("#859b6c"),
    QColor("#ee8577"), QColor("#ffbb44"), QColor("#122451"),
};
inline const std::vector<QColor> Degas{
    QColor("#418979"), QColor("#96410e"), QColor("#591d06"), QColor("#e5a335"),
    QColor("#556219"), QColor("#053c29"), QColor("#2b614e"),
};
inline const std::vector<QColor> Demuth{
    QColor("#41485f"), QColor("#f7c267"), QColor("#591c19"), QColor("#8b8b99"),
    QColor("#b64f32"), QColor("#d39a2d"), QColor("#5d6174"), QColor("#9b332b"),
    QColor("#b9b9b8"), QColor("#262d42"),
};
inline const std::vector<QColor> Derain{
    QColor("#aab5d5"), QColor("#97c684"), QColor("#808fe1"), QColor("#454a74"),
    QColor("#efc86e"), QColor("#6f9969"), QColor("#5c66a8"),
};
inline const std::vector<QColor> Egypt{
    QColor("#dd5129"),
    QColor("#0f7ba2"),
    QColor("#43b284"),
    QColor("#fab255"),
};
inline const std::vector<QColor> Gauguin{
    QColor("#811e18"), QColor("#4c6216"), QColor("#c88a2c"),
    QColor("#9e4013"), QColor("#b04948"), QColor("#1a472a"),
};
inline const std::vector<QColor> Greek{
    QColor("#8d1c06"), QColor("#e67424"), QColor("#f5c34d"),
    QColor("#3c0d03"), QColor("#ed9b49"),
};
inline const std::vector<QColor> Hiroshige{
    QColor("#aadce0"), QColor("#ef8a47"), QColor("#376795"), QColor("#f7aa58"),
    QColor("#72bcd5"), QColor("#ffe6b7"), QColor("#e76254"), QColor("#1e466e"),
    QColor("#ffd06f"), QColor("#528fad"),
};
inline const std::vector<QColor> Hokusai1{
    QColor("#b75347"), QColor("#224b5e"), QColor("#e09351"), QColor("#94b594"),
    QColor("#edc775"), QColor("#6d2f20"), QColor("#df7e66"),
};
inline const std::vector<QColor> Hokusai2{
    QColor("#134b73"), QColor("#72aeb6"), QColor("#2f70a1"),
    QColor("#abc9c8"), QColor("#0a3351"), QColor("#4692b0"),
};
inline const std::vector<QColor> Hokusai3{
    QColor("#5a97c1"), QColor("#95c36e"), QColor("#295384"),
    QColor("#74c8c3"), QColor("#d8d97a"), QColor("#0a2e57"),
};
inline const std::vector<QColor> Homer1{
    QColor("#c3f4f6"), QColor("#df7700"), QColor("#a62f00"), QColor("#6ad5e8"),
    QColor("#f5b642"), QColor("#32b2da"), QColor("#fff179"), QColor("#551f00"),
};
inline const std::vector<QColor> Homer2{
    QColor("#e9851d"), QColor("#165d43"), QColor("#bf3626"), QColor("#f9c53b"),
    QColor("#788f33"), QColor("#e9724c"), QColor("#aeac4c"),
};
inline const std::vector<QColor> Ingres{
    QColor("#2e77ab"), QColor("#d1b252"), QColor("#18527e"), QColor("#a97f2f"),
    QColor("#06314e"), QColor("#7e5522"), QColor("#041d2c"), QColor("#472c0b"),
};
inline const std::vector<QColor> Isfahan1{
    QColor("#178f92"), QColor("#845d29"), QColor("#4fb6ca"), QColor("#175f5d"),
    QColor("#4e3910"), QColor("#1d1f54"), QColor("#d8c29d"),
};
inline const std::vector<QColor> Isfahan2{
    QColor("#34b6c6"), QColor("#ddc000"), QColor("#79ad41"),
    QColor("#4063a3"), QColor("#d7aca1"),
};
inline const std::vector<QColor> Java{
    QColor("#663171"), QColor("#e2998a"), QColor("#cf3a36"),
    QColor("#0c7156"), QColor("#ea7428"),
};
inline const std::vector<QColor> Johnson{
    QColor("#f6c200"), QColor("#a00e00"), QColor("#0086a8"),
    QColor("#d04e00"), QColor("#132b69"),
};
inline const std::vector<QColor> Juarez{
    QColor("#a82203"), QColor("#208cc0"), QColor("#f1af3a"),
    QColor("#cf5e4e"), QColor("#637b31"), QColor("#003967"),
};
inline const std::vector<QColor> Kandinsky{
    QColor("#3b7c70"),
    QColor("#ce9642"),
    QColor("#898e9f"),
    QColor("#3b3a3e"),
};
inline const std::vector<QColor> Klimt{
    QColor("#3c4b99"), QColor("#c93f55"), QColor("#eacc62"),
    QColor("#469d76"), QColor("#924099"), QColor("#df9ed4"),
};
inline const std::vector<QColor> Lakota{
    QColor("#04a3bd"), QColor("#f0be3d"), QColor("#931e18"),
    QColor("#da7901"), QColor("#247d3f"), QColor("#20235b"),
};
inline const std::vector<QColor> Manet{
    QColor("#225e92"), QColor("#d29c44"), QColor("#43429b"), QColor("#ebc174"),
    QColor("#4585b7"), QColor("#183571"), QColor("#5e65be"), QColor("#80521c"),
    QColor("#7ec5f4"), QColor("#3b2319"), QColor("#ede2cc"),
};
inline const std::vector<QColor> Monet{
    QColor("#749e89"), QColor("#c399a2"), QColor("#7d87b2"),
    QColor("#abccbe"), QColor("#e3cacf"), QColor("#c2cae3"),
    QColor("#4e6d58"), QColor("#9f6e71"), QColor("#41507b"),
};
inline const std::vector<QColor> Moreau{
    QColor("#792504"), QColor("#527baa"), QColor("#bc7524"), QColor("#8dadca"),
    QColor("#082844"), QColor("#421600"), QColor("#104839"),
};
inline const std::vector<QColor> Morgenstern{
    QColor("#a56457"), QColor("#ffb178"), QColor("#ffc680"), QColor("#db8872"),
    QColor("#dfbbc8"), QColor("#b08ba5"), QColor("#7c668c"),
};
inline const std::vector<QColor> Nattier{
    QColor("#52271c"), QColor("#184948"), QColor("#c08e39"), QColor("#7f793c"),
    QColor("#022a2a"), QColor("#944839"), QColor("#565c33"),
};
inline const std::vector<QColor> Navajo{
    QColor("#660d20"), QColor("#e59a52"), QColor("#edce79"),
    QColor("#094568"), QColor("#e1c59a"),
};
inline const std::vector<QColor> NewKingdom{
    QColor("#9eb4e0"), QColor("#e1846c"), QColor("#e6bb9e"),
    QColor("#9c6849"), QColor("#735852"),
};
inline const std::vector<QColor> Nizami{
    QColor("#8cc8bc"), QColor("#b83326"), QColor("#7da7ea"), QColor("#1d4497"),
    QColor("#c8570d"), QColor("#5773c0"), QColor("#edb144"), QColor("#dd7867"),
};
inline const std::vector<QColor> OKeeffe1{
    QColor("#7db0ea"), QColor("#f6f2ee"), QColor("#6b200c"), QColor("#ee956a"),
    QColor("#225bb2"), QColor("#da6c42"), QColor("#133e7e"), QColor("#fbc2a9"),
    QColor("#973d21"), QColor("#bad6f9"), QColor("#447fdd"),
};
inline const std::vector<QColor> OKeeffe2{
    QColor("#92351e"), QColor("#fbe3c2"), QColor("#b9563f"), QColor("#e69c6b"),
    QColor("#f2c88f"), QColor("#d37750"), QColor("#ecb27d"),
};
inline const std::vector<QColor> Paquin{
    QColor("#47632a"), QColor("#f7dea3"), QColor("#831818"), QColor("#98ab76"),
    QColor("#f78462"), QColor("#f05b43"), QColor("#feac81"), QColor("#748f46"),
    QColor("#c62320"), QColor("#ced1af"), QColor("#275024"),
};
inline const std::vector<QColor> Peru1{
    QColor("#1c9d7c"), QColor("#b5361c"), QColor("#369cc9"),
    QColor("#e35e28"), QColor("#31c7ba"), QColor("#3a507f"),
};
inline const std::vector<QColor> Peru2{
    QColor("#b36c06"), QColor("#65150b"), QColor("#c0431f"), QColor("#f19425"),
    QColor("#961f1f"), QColor("#533d14"), QColor("#c59349"),
};
inline const std::vector<QColor> Pillement{
    QColor("#44636f"), QColor("#738e8e"), QColor("#697852"),
    QColor("#2b4655"), QColor("#a9845b"), QColor("#0f252f"),
};
inline const std::vector<QColor> Pissaro{
    QColor("#1a5270"), QColor("#4c825d"), QColor("#8dc7dc"), QColor("#134130"),
    QColor("#0e2a4d"), QColor("#508ca7"), QColor("#8cae9e"),
};
inline const std::vector<QColor> Redon{
    QColor("#5b859e"), QColor("#1e395f"), QColor("#75884b"), QColor("#1e5a46"),
    QColor("#df8d71"), QColor("#af4f2f"), QColor("#d48f90"), QColor("#732f30"),
    QColor("#ab84a5"), QColor("#59385c"), QColor("#d8b847"), QColor("#b38711"),
};
inline const std::vector<QColor> Renoir{
    QColor("#2f357c"), QColor("#b0799a"), QColor("#e69b00"), QColor("#355828"),
    QColor("#6c5d9e"), QColor("#bf3729"), QColor("#e48171"), QColor("#f5bb50"),
    QColor("#9d9cd5"), QColor("#17154f"), QColor("#f6b3b0"), QColor("#ada43b"),
};
inline const std::vector<QColor> Robert{
    QColor("#375624"), QColor("#18505f"), QColor("#6ca4a0"),
    QColor("#11341a"), QColor("#062e3d"), QColor("#487a7c"),
};
inline const std::vector<QColor> Signac{
    QColor("#2b9b81"), QColor("#fe9b00"), QColor("#f4c40f"), QColor("#fbe183"),
    QColor("#633372"), QColor("#9b3441"), QColor("#e6a2a6"), QColor("#92c051"),
    QColor("#1f6e9c"), QColor("#9f5691"), QColor("#e87b89"), QColor("#d8443c"),
    QColor("#de597c"), QColor("#aa7aa1"),
};
inline const std::vector<QColor> Stevens{
    QColor("#ba5c3f"), QColor("#307d7f"), QColor("#598c4c"),
    QColor("#a13213"), QColor("#042e4e"), QColor("#470c00"),
};
inline const std::vector<QColor> Tam{
    QColor("#ef8737"), QColor("#341648"), QColor("#ffd353"), QColor("#9f2d55"),
    QColor("#ffb242"), QColor("#62205f"), QColor("#de4f33"), QColor("#bb292c"),
};
inline const std::vector<QColor> Tara{
    QColor("#eab1c6"), QColor("#e18a1f"), QColor("#d35e17"),
    QColor("#829d44"), QColor("#e9b109"),
};
inline const std::vector<QColor> Thomas{
    QColor("#4457a5"), QColor("#c44d76"), QColor("#7caf5c"), QColor("#59386c"),
    QColor("#b24422"), QColor("#13315f"), QColor("#447861"), QColor("#b1a1cc"),
};
inline const std::vector<QColor> Tiepolo{
    QColor("#802417"), QColor("#c06636"), QColor("#17486f"), QColor("#e8b960"),
    QColor("#ce9344"), QColor("#646e3b"), QColor("#508ea2"), QColor("#2b5851"),
};
inline const std::vector<QColor> Troy{
    QColor("#6c1d0e"), QColor("#235070"), QColor("#c27668"), QColor("#7ba0b4"),
    QColor("#421401"), QColor("#0a2d46"), QColor("#8b3a2b"), QColor("#44728c"),
};
inline const std::vector<QColor> Tsimshian{
    QColor("#2673a3"), QColor("#582310"), QColor("#473d7d"), QColor("#318f49"),
    QColor("#582310"), QColor("#0cb4bb"), QColor("#82c45f"),
};
inline const std::vector<QColor> VanGogh1{
    QColor("#6b6ca3"), QColor("#87bcbd"), QColor("#6f9954"), QColor("#969bc7"),
    QColor("#89ab7c"), QColor("#434475"), QColor("#2c2d54"),
};
inline const std::vector<QColor> VanGogh2{
    QColor("#bd3106"), QColor("#5b7314"), QColor("#454b87"), QColor("#d9700e"),
    QColor("#89a6bb"), QColor("#eebe04"), QColor("#c3d6ce"), QColor("#e9a00e"),
};
inline const std::vector<QColor> VanGogh3{
    QColor("#1e3d14"), QColor("#447243"), QColor("#e7e5cc"), QColor("#669d62"),
    QColor("#192813"), QColor("#c2d6a4"), QColor("#9cc184"), QColor("#1f5b25"),
};
inline const std::vector<QColor> Veronese{
    QColor("#2c6b67"), QColor("#67322e"), QColor("#122c43"), QColor("#99610a"),
    QColor("#c38f16"), QColor("#175449"), QColor("#6e948c"),
};
inline const std::vector<QColor> Wissing{
    QColor("#7c291e"), QColor("#ba7233"), QColor("#2d5380"),
    QColor("#3a4421"), QColor("#4b1d0d"),
};
} // namespace colors

} // namespace impl

QColor triColorMap(double x, double minValue, double maxValue,
                   const QColor &startColor, const QColor &midColor,
                   const QColor &endColor) {

  // Because the midColor is tied to 0 when using a "color mapped from color
  // range"
  // we can't allow minValue to become positive nor the maxValue to become
  // negative
  constexpr double LIMIT = 0.0001;
  minValue = (minValue > 0.0) ? -LIMIT : minValue;
  maxValue = (maxValue < 0.0) ? LIMIT : maxValue;

  QColor color = (x < 0.0) ? startColor : endColor;
  double denom = (x < 0.0) ? minValue : maxValue;
  double factor = 1.0 - x / denom;

  if (factor > 0.0) {
    return QColor::fromRgbF(
        (color.redF() + (midColor.redF() - color.redF()) * factor),
        (color.greenF() + (midColor.greenF() - color.greenF()) * factor),
        (color.blueF() + (midColor.blueF() - color.blueF()) * factor));
  }
  return color;
}

QColor colorMappedFromHueRange(double value, double minValue, double maxValue,
                               bool reverse, double minHue, double maxHue) {

  double newValue = std::max(std::min(value, maxValue), value);
  double range = maxValue - minValue;
  double rangeRatio = (range > 1e-6) ? (maxHue - minHue) / range : 0.0;

  double h;
  if (reverse) {
    h = std::max(minHue, (maxHue - rangeRatio * (newValue - minValue)));
    h = std::min(h, maxHue);
  } else {
    h = std::min(maxHue, (minHue + rangeRatio * (newValue - minValue)));
    h = std::max(h, minHue);
  }
  return QColor::fromHsvF(h, 1.0, 1.0).toRgb();
}

// Use the macro in the function
inline const std::vector<QColor> &colorsFromName(ColorMapName name) {
  switch (name) {
  case ColorMapName::Parula:
    return impl::colors::Parula;
  case ColorMapName::Heat:
    return impl::colors::Heat;
  case ColorMapName::Jet:
    return impl::colors::Jet;
  case ColorMapName::Turbo:
    return impl::colors::Turbo;
  case ColorMapName::Magma:
    return impl::colors::Magma;
  case ColorMapName::Inferno:
    return impl::colors::Inferno;
  case ColorMapName::Plasma:
    return impl::colors::Plasma;
  case ColorMapName::Viridis:
    return impl::colors::Viridis;
  case ColorMapName::Cividis:
    return impl::colors::Cividis;
  case ColorMapName::Github:
    return impl::colors::Github;
  case ColorMapName::Archambault:
    return impl::colors::Archambault;
  case ColorMapName::Austria:
    return impl::colors::Austria;
  case ColorMapName::Benedictus:
    return impl::colors::Benedictus;
  case ColorMapName::Cassatt1:
    return impl::colors::Cassatt1;
  case ColorMapName::Cassatt2:
    return impl::colors::Cassatt2;
  case ColorMapName::Cross:
    return impl::colors::Cross;
  case ColorMapName::Degas:
    return impl::colors::Degas;
  case ColorMapName::Demuth:
    return impl::colors::Demuth;
  case ColorMapName::Derain:
    return impl::colors::Derain;
  case ColorMapName::Egypt:
    return impl::colors::Egypt;
  case ColorMapName::Gauguin:
    return impl::colors::Gauguin;
  case ColorMapName::Greek:
    return impl::colors::Greek;
  case ColorMapName::Hiroshige:
    return impl::colors::Hiroshige;
  case ColorMapName::Hokusai1:
    return impl::colors::Hokusai1;
  case ColorMapName::Hokusai2:
    return impl::colors::Hokusai2;
  case ColorMapName::Hokusai3:
    return impl::colors::Hokusai3;
  case ColorMapName::Homer1:
    return impl::colors::Homer1;
  case ColorMapName::Homer2:
    return impl::colors::Homer2;
  case ColorMapName::Ingres:
    return impl::colors::Ingres;
  case ColorMapName::Isfahan1:
    return impl::colors::Isfahan1;
  case ColorMapName::Isfahan2:
    return impl::colors::Isfahan2;
  case ColorMapName::Java:
    return impl::colors::Java;
  case ColorMapName::Johnson:
    return impl::colors::Johnson;
  case ColorMapName::Juarez:
    return impl::colors::Juarez;
  case ColorMapName::Kandinsky:
    return impl::colors::Kandinsky;
  case ColorMapName::Klimt:
    return impl::colors::Klimt;
  case ColorMapName::Lakota:
    return impl::colors::Lakota;
  case ColorMapName::Manet:
    return impl::colors::Manet;
  case ColorMapName::Monet:
    return impl::colors::Monet;
  case ColorMapName::Moreau:
    return impl::colors::Moreau;
  case ColorMapName::Morgenstern:
    return impl::colors::Morgenstern;
  case ColorMapName::Nattier:
    return impl::colors::Nattier;
  case ColorMapName::Navajo:
    return impl::colors::Navajo;
  case ColorMapName::NewKingdom:
    return impl::colors::NewKingdom;
  case ColorMapName::Nizami:
    return impl::colors::Nizami;
  case ColorMapName::OKeeffe1:
    return impl::colors::OKeeffe1;
  case ColorMapName::OKeeffe2:
    return impl::colors::OKeeffe2;
  case ColorMapName::Paquin:
    return impl::colors::Paquin;
  case ColorMapName::Peru1:
    return impl::colors::Peru1;
  case ColorMapName::Peru2:
    return impl::colors::Peru2;
  case ColorMapName::Pillement:
    return impl::colors::Pillement;
  case ColorMapName::Pissaro:
    return impl::colors::Pissaro;
  case ColorMapName::Redon:
    return impl::colors::Redon;
  case ColorMapName::Renoir:
    return impl::colors::Renoir;
  case ColorMapName::Robert:
    return impl::colors::Robert;
  case ColorMapName::Signac:
    return impl::colors::Signac;
  case ColorMapName::Stevens:
    return impl::colors::Stevens;
  case ColorMapName::Tam:
    return impl::colors::Tam;
  case ColorMapName::Tara:
    return impl::colors::Tara;
  case ColorMapName::Thomas:
    return impl::colors::Thomas;
  case ColorMapName::Tiepolo:
    return impl::colors::Tiepolo;
  case ColorMapName::Troy:
    return impl::colors::Troy;
  case ColorMapName::Tsimshian:
    return impl::colors::Tsimshian;
  case ColorMapName::VanGogh1:
    return impl::colors::VanGogh1;
  case ColorMapName::VanGogh2:
    return impl::colors::VanGogh2;
  case ColorMapName::VanGogh3:
    return impl::colors::VanGogh3;
  case ColorMapName::Veronese:
    return impl::colors::Veronese;
  case ColorMapName::Wissing:
    return impl::colors::Wissing;
  // Add more cases as needed
  default:
    return impl::colors::Viridis;
  }
}

QColor linearColorMap(double x, ColorMapName name) {
  return impl::linearColorMap(x, colorsFromName(name));
}

QColor quantizedLinearColorMap(double x, unsigned int num_levels,
                               ColorMapName name) {
  return linearColorMap(impl::quantize(x, num_levels), name);
}

ColorMapFunc::ColorMapFunc(ColorMapName n) : name(n) {
  noneColor = QColor(
      settings::readSetting(settings::keys::NONE_PROPERTY_COLOR).toString());
}
ColorMapFunc::ColorMapFunc(ColorMapName n, double minValue, double maxValue)
    : name(n), lower(minValue), upper(maxValue) {
  noneColor = QColor(
      settings::readSetting(settings::keys::NONE_PROPERTY_COLOR).toString());
}

QColor ColorMapFunc::operator()(double x) const {
  switch (name) {
  case ColorMapName::CE_None:
    return noneColor;
    break;
  case ColorMapName::CE_bwr:
    return triColorMap(x, lower, upper, Qt::red, Qt::white, Qt::blue);
    break;
  case ColorMapName::CE_rgb: {
    const double minHue = 0.0;
    const double maxHue = 2.0 / 3.0;
    return colorMappedFromHueRange(x, lower, upper, reverse, minHue, maxHue);
    break;
  }
  default:
    return linearColorMap((x - lower) / (upper - lower), name);
    break;
  }
}

std::vector<ColorMapName> availableColorMaps() {
  return std::vector<ColorMapName>{
      ColorMapName::Parula,     ColorMapName::Heat,
      ColorMapName::Jet,        ColorMapName::Turbo,
      ColorMapName::Hot,        ColorMapName::Gray,
      ColorMapName::Magma,      ColorMapName::Inferno,
      ColorMapName::Plasma,     ColorMapName::Viridis,
      ColorMapName::Cividis,    ColorMapName::Github,
      ColorMapName::Cubehelix,  ColorMapName::HSV,
      ColorMapName::CE_bwr,     ColorMapName::CE_rgb,
      ColorMapName::CE_None,    ColorMapName::Archambault,
      ColorMapName::Austria,    ColorMapName::Benedictus,
      ColorMapName::Cassatt1,   ColorMapName::Cassatt2,
      ColorMapName::Cross,      ColorMapName::Degas,
      ColorMapName::Demuth,     ColorMapName::Derain,
      ColorMapName::Egypt,      ColorMapName::Gauguin,
      ColorMapName::Greek,      ColorMapName::Hiroshige,
      ColorMapName::Hokusai1,   ColorMapName::Hokusai2,
      ColorMapName::Hokusai3,   ColorMapName::Homer1,
      ColorMapName::Homer2,     ColorMapName::Ingres,
      ColorMapName::Isfahan1,   ColorMapName::Isfahan2,
      ColorMapName::Java,       ColorMapName::Johnson,
      ColorMapName::Juarez,     ColorMapName::Kandinsky,
      ColorMapName::Klimt,      ColorMapName::Lakota,
      ColorMapName::Manet,      ColorMapName::Monet,
      ColorMapName::Moreau,     ColorMapName::Morgenstern,
      ColorMapName::Nattier,    ColorMapName::Navajo,
      ColorMapName::NewKingdom, ColorMapName::Nizami,
      ColorMapName::OKeeffe1,   ColorMapName::OKeeffe2,
      ColorMapName::Paquin,     ColorMapName::Peru1,
      ColorMapName::Peru2,      ColorMapName::Pillement,
      ColorMapName::Pissaro,    ColorMapName::Redon,
      ColorMapName::Renoir,     ColorMapName::Robert,
      ColorMapName::Signac,     ColorMapName::Stevens,
      ColorMapName::Tam,        ColorMapName::Tara,
      ColorMapName::Thomas,     ColorMapName::Tiepolo,
      ColorMapName::Troy,       ColorMapName::Tsimshian,
      ColorMapName::VanGogh1,   ColorMapName::VanGogh2,
      ColorMapName::VanGogh3,   ColorMapName::Veronese,
      ColorMapName::Wissing};
}

ColorMapName colorMapFromString(const QString &s) {
  const auto &maps = availableColorMaps();
  QString input = s.toLower();
  for (int i = 0; i < maps.size(); i++) {
    if (input == QString(colorMapToString(maps[i])).toLower())
      return maps[i];
  }
  return ColorMapName::CE_None;
}

const char *colorMapToString(ColorMapName name) {
  switch (name) {
  case ColorMapName::Parula:
    return "Parula";
  case ColorMapName::Heat:
    return "Heat";
  case ColorMapName::Jet:
    return "Jet";
  case ColorMapName::Turbo:
    return "Turbo";
  case ColorMapName::Hot:
    return "Hot";
  case ColorMapName::Gray:
    return "Gray";
  case ColorMapName::Magma:
    return "Magma";
  case ColorMapName::Inferno:
    return "Inferno";
  case ColorMapName::Plasma:
    return "Plasma";
  case ColorMapName::Viridis:
    return "Viridis";
  case ColorMapName::Cividis:
    return "Cividis";
  case ColorMapName::Github:
    return "Github";
  case ColorMapName::Cubehelix:
    return "Cubehelix";
  case ColorMapName::HSV:
    return "HSV";
  case ColorMapName::CE_bwr:
    return "RedWhiteBlue";
  case ColorMapName::CE_rgb:
    return "RedGreenBlue";
  case ColorMapName::Archambault:
    return "Archambault";
  case ColorMapName::Austria:
    return "Austria";
  case ColorMapName::Benedictus:
    return "Benedictus";
  case ColorMapName::Cassatt1:
    return "Cassatt1";
  case ColorMapName::Cassatt2:
    return "Cassatt2";
  case ColorMapName::Cross:
    return "Cross";
  case ColorMapName::Degas:
    return "Degas";
  case ColorMapName::Demuth:
    return "Demuth";
  case ColorMapName::Derain:
    return "Derain";
  case ColorMapName::Egypt:
    return "Egypt";
  case ColorMapName::Gauguin:
    return "Gauguin";
  case ColorMapName::Greek:
    return "Greek";
  case ColorMapName::Hiroshige:
    return "Hiroshige";
  case ColorMapName::Hokusai1:
    return "Hokusai1";
  case ColorMapName::Hokusai2:
    return "Hokusai2";
  case ColorMapName::Hokusai3:
    return "Hokusai3";
  case ColorMapName::Homer1:
    return "Homer1";
  case ColorMapName::Homer2:
    return "Homer2";
  case ColorMapName::Ingres:
    return "Ingres";
  case ColorMapName::Isfahan1:
    return "Isfahan1";
  case ColorMapName::Isfahan2:
    return "Isfahan2";
  case ColorMapName::Java:
    return "Java";
  case ColorMapName::Johnson:
    return "Johnson";
  case ColorMapName::Juarez:
    return "Juarez";
  case ColorMapName::Kandinsky:
    return "Kandinsky";
  case ColorMapName::Klimt:
    return "Klimt";
  case ColorMapName::Lakota:
    return "Lakota";
  case ColorMapName::Manet:
    return "Manet";
  case ColorMapName::Monet:
    return "Monet";
  case ColorMapName::Moreau:
    return "Moreau";
  case ColorMapName::Morgenstern:
    return "Morgenstern";
  case ColorMapName::Nattier:
    return "Nattier";
  case ColorMapName::Navajo:
    return "Navajo";
  case ColorMapName::NewKingdom:
    return "NewKingdom";
  case ColorMapName::Nizami:
    return "Nizami";
  case ColorMapName::OKeeffe1:
    return "OKeeffe1";
  case ColorMapName::OKeeffe2:
    return "OKeeffe2";
  case ColorMapName::Paquin:
    return "Paquin";
  case ColorMapName::Peru1:
    return "Peru1";
  case ColorMapName::Peru2:
    return "Peru2";
  case ColorMapName::Pillement:
    return "Pillement";
  case ColorMapName::Pissaro:
    return "Pissaro";
  case ColorMapName::Redon:
    return "Redon";
  case ColorMapName::Renoir:
    return "Renoir";
  case ColorMapName::Robert:
    return "Robert";
  case ColorMapName::Signac:
    return "Signac";
  case ColorMapName::Stevens:
    return "Stevens";
  case ColorMapName::Tam:
    return "Tam";
  case ColorMapName::Tara:
    return "Tara";
  case ColorMapName::Thomas:
    return "Thomas";
  case ColorMapName::Tiepolo:
    return "Tiepolo";
  case ColorMapName::Troy:
    return "Troy";
  case ColorMapName::Tsimshian:
    return "Tsimshian";
  case ColorMapName::VanGogh1:
    return "VanGogh1";
  case ColorMapName::VanGogh2:
    return "VanGogh2";
  case ColorMapName::VanGogh3:
    return "VanGogh3";
  case ColorMapName::Veronese:
    return "Veronese";
  case ColorMapName::Wissing:
    return "Wissing";
  default:
    return "None";
  }
}
