package services

import (
	"aim-oscar/aimerror"
	"aim-oscar/models"
	"aim-oscar/oscar"
	"aim-oscar/util"
	"context"
	"fmt"
	"time"

	"github.com/pkg/errors"
	"github.com/uptrace/bun"
)

type ServiceVersion struct {
	Family  uint16
	Version uint16
}

var ServiceVersions []ServiceVersion

func init() {
	ServiceVersions = []ServiceVersion{
		{0x01, 3},
		{0x02, 1},
		{0x03, 1},
		{0x04, 1},
		{0x0f, 1},
		{0x13, 1},
		{0x17, 1},
		{0x18, 1},
	}
}

type GenericServiceControls struct {
	OnlineCh       chan *models.User
	ServerHostname string
}

func (g *GenericServiceControls) HandleSNAC(ctx context.Context, db *bun.DB, snac *oscar.SNAC) (context.Context, error) {
	session, _ := oscar.SessionFromContext(ctx)
	logger := session.Logger.With("service", "generic service controls")

	switch snac.Header.Subtype {

	// Client is ONLINE and READY
	case 0x02:
		user := models.UserFromContext(ctx)
		if user != nil {
			user.Status = models.UserStatusOnline
			if err := user.Update(ctx, db, "status"); err != nil {
				return ctx, errors.Wrap(err, "could not set user as active")
			}

			g.OnlineCh <- user

			return models.NewContextWithUser(ctx, user), nil
		}

		return ctx, nil

	case 0x04:
		family, err := snac.Data.ReadUint16()
		if err != nil {
			return ctx, errors.Wrap(err, "could not read family")
		}
		logger.Warn(fmt.Sprintf("client wants service for family 0x%02x", family))
		return ctx, nil

	// Client wants to know the rate limits for all services
	case 0x06:
		rateSnac := oscar.NewSNAC(1, 7)
		rateSnac.Data.WriteUint16(1) // one rate class

		// Define a Rate Class
		rc := oscar.Buffer{}
		rc.WriteUint16(1)    // ID
		rc.WriteUint32(80)   // Window Size
		rc.WriteUint32(2500) // Clear level
		rc.WriteUint32(2000) // Alert level
		rc.WriteUint32(1500) // Limit level
		rc.WriteUint32(800)  // Disconnect level
		rc.WriteUint32(3400) // Current level (fake)
		rc.WriteUint32(6000) // Max level
		rc.WriteUint32(0)    // Last time ?
		rc.WriteUint8(0)     // Current state ?
		rateSnac.Data.Write(rc.Bytes())

		// Define a Rate Group
		rg := oscar.Buffer{}
		rg.WriteUint16(1) // ID

		// TODO: make actual rate groups instead of this hack. I can't tell which subtypes are supported so
		// make it set rate limits for every family and all subtypes under 0x21.
		rg.WriteUint16(uint16(len(ServiceVersions)) * 0x21) // Number of rate groups
		for _, service := range ServiceVersions {
			for subtype := 0; subtype < 0x21; subtype++ {
				rg.WriteUint16(service.Family)
				rg.WriteUint16(uint16(subtype))
			}
		}
		rateSnac.Data.Write(rg.Bytes())

		rateFlap := oscar.NewFLAP(2)
		rateFlap.Data.WriteBinary(rateSnac)
		return ctx, session.Send(rateFlap)

	// Client notifying server it accepted rate limits
	case 0x08:
		return ctx, nil

	// Client wants their own online information
	case 0x0e:
		user := models.UserFromContext(ctx)
		if user == nil {
			return ctx, aimerror.NoUserInSession
		}

		onlineSnac := oscar.NewSNAC(0x1, 0xf)
		onlineSnac.Data.WriteUint8(uint8(len(user.ScreenName)))
		onlineSnac.Data.WriteString(user.ScreenName)
		onlineSnac.Data.WriteUint16(0) // warning level

		user.Status = models.UserStatusOnline
		if err := user.Update(ctx, db, "status"); err != nil {
			return ctx, errors.Wrap(err, "could not set user as active")
		}

		tlvs := []*oscar.TLV{
			oscar.NewTLV(0x01, util.Dword(0x0100)),                                            // User Class
			oscar.NewTLV(0x06, util.Dword(uint32(user.Status))),                               // user status
			oscar.NewTLV(0x0a, util.Dword(0)),                                                 // External IP of the client?
			oscar.NewTLV(0x0f, util.Dword(uint32(time.Since(user.LastActivityAt).Seconds()))), // Idle Time
			oscar.NewTLV(0x03, util.Dword(uint32(time.Now().Unix()))),                         // Client Signon Time
			oscar.NewTLV(0x1e, util.Dword(0x0)),                                               // Unknown value
			oscar.NewTLV(0x05, util.Dword(uint32(user.CreatedAt.Unix()))),                     // Member since
		}

		onlineSnac.AppendTLVs(tlvs)

		onlineFlap := oscar.NewFLAP(2)
		onlineFlap.Data.WriteBinary(onlineSnac)
		return models.NewContextWithUser(ctx, user), session.Send(onlineFlap)

	// Client tells us the idle time
	case 0x11:
		// TODO: keep track of idle time
		return ctx, nil

	case 0x16:
		// NOP, client keepalive
		return ctx, nil

	// Client wants to know the ServiceVersions of all of the services offered
	case 0x17:
		versionsSnac := oscar.NewSNAC(0x1, 0x18)
		for _, service := range ServiceVersions {
			versionsSnac.Data.WriteUint16(service.Family)
			versionsSnac.Data.WriteUint16(service.Version)
		}
		versionsFlap := oscar.NewFLAP(2)
		versionsFlap.Data.WriteBinary(versionsSnac)
		return ctx, session.Send(versionsFlap)
	}

	return ctx, nil
}
