import { KVStore } from "@keplr-wallet/common";
import {
  ObservableChainQuery,
  ObservableChainQueryMap,
} from "../../chain-query";
import { ChainGetter } from "../../../common";
import { DenomTraceResponse } from "./types";
import { autorun, computed } from "mobx";

export class ObservableChainQueryDenomTrace extends ObservableChainQuery<DenomTraceResponse> {
  protected disposer?: () => void;

  constructor(
    kvStore: KVStore,
    chainId: string,
    chainGetter: ChainGetter,
    protected readonly hash: string
  ) {
    super(
      kvStore,
      chainId,
      chainGetter,
      `/ibc/applications/transfer/v1beta1/denom_traces/${hash}`
    );
  }

  protected onStart() {
    super.onStart();

    return new Promise<void>((resolve) => {
      this.disposer = autorun(() => {
        const chainInfo = this.chainGetter.getChain(this.chainId);
        if (chainInfo.features && chainInfo.features.includes("ibc-go")) {
          this.setUrl(`/ibc/apps/transfer/v1/denom_traces/${this.hash}`);
        }
        resolve();
      });
    });
  }

  protected onStop() {
    if (this.disposer) {
      this.disposer();
      this.disposer = undefined;
    }
    super.onStop();
  }

  @computed
  get paths(): {
    portId: string;
    channelId: string;
  }[] {
    if (!this.response) {
      return [];
    }

    const rawPaths = this.response.data.denom_trace.path.split("/");

    if (rawPaths.length % 2 !== 0) {
      console.error("Failed to parse paths", rawPaths);
      return [];
    }

    const rawPathChunks: string[][] = [];
    for (let i = 0; i < rawPaths.length; i += 2) {
      rawPathChunks.push(rawPaths.slice(i, i + 2));
    }

    return rawPathChunks.map((chunk) => {
      return {
        portId: chunk[0],
        channelId: chunk[1],
      };
    });
  }

  get denom(): string | undefined {
    if (!this.response) {
      return undefined;
    }

    return this.response.data.denom_trace.base_denom;
  }

  @computed
  get denomTrace():
    | {
        denom: string;
        paths: {
          portId: string;
          channelId: string;
        }[];
      }
    | undefined {
    if (!this.response || !this.denom) {
      return undefined;
    }

    return {
      denom: this.denom,
      paths: this.paths,
    };
  }
}

export class ObservableQueryDenomTrace extends ObservableChainQueryMap<DenomTraceResponse> {
  constructor(
    protected readonly kvStore: KVStore,
    protected readonly chainId: string,
    protected readonly chainGetter: ChainGetter
  ) {
    super(kvStore, chainId, chainGetter, (hash: string) => {
      return new ObservableChainQueryDenomTrace(
        this.kvStore,
        this.chainId,
        this.chainGetter,
        hash
      );
    });
  }

  getDenomTrace(hash: string): ObservableChainQueryDenomTrace {
    return this.get(hash) as ObservableChainQueryDenomTrace;
  }
}
