package app.api

import app.common.accounting.ChartSpec
import app.common.money.Currency
import app.models.accounting.BalanceCheck
import app.models.accounting.Transaction
import app.models.accounting.TransactionGroup
import app.models.accounting.config._
import app.models.money.ExchangeRateMeasurement
import app.models.user.User
import boopickle.Default._
import hydro.api.StandardPicklers
import hydro.models.Entity
import hydro.models.UpdatableEntity.LastUpdateTime
import scala.collection.JavaConverters._

import java.time.Month
import scala.collection.immutable.Seq
import scala.collection.immutable.Set

object Picklers extends StandardPicklers {

  // Pickler that does the same as an autogenerated User pickler, except that it redacts the user's password
  implicit object UserPickler extends Pickler[User] {
    override def pickle(user: User)(implicit state: PickleState): Unit = logExceptions {
      state.pickle(user.loginName)
      // Password redacted
      state.pickle(user.name)
      state.pickle(user.isAdmin)
      state.pickle(user.expandCashFlowTablesByDefault)
      state.pickle(user.expandLiquidationTablesByDefault)
      state.pickle(user.idOption)
      state.pickle(user.lastUpdateTime)
    }
    override def unpickle(implicit state: UnpickleState): User = logExceptions {
      User(
        loginName = state.unpickle[String],
        passwordHash = "<redacted>",
        name = state.unpickle[String],
        isAdmin = state.unpickle[Boolean],
        expandCashFlowTablesByDefault = state.unpickle[Boolean],
        expandLiquidationTablesByDefault = state.unpickle[Boolean],
        idOption = state.unpickle[Option[Long]],
        lastUpdateTime = state.unpickle[LastUpdateTime],
      )
    }
  }

  // This pickler should normally be autogenerated by boopickle, but there seems to be a bug
  // causing this not to work. Fixed by manually picking templates.
  implicit object TemplatePickler extends Pickler[Template] {
    override def pickle(template: Template)(implicit state: PickleState): Unit = logExceptions {
      state.pickle(template.code)
      state.pickle(template.name)
      state.pickle(template.placement.map(_.toString))
      state.pickle(template.onlyShowForUserLoginNames)
      state.pickle(template.zeroSum)
      state.pickle(template.iconClass)
      state.pickle(template.transactions)
    }
    override def unpickle(implicit state: UnpickleState): Template = logExceptions {
      Template(
        code = state.unpickle[String],
        name = state.unpickle[String],
        placement = state.unpickle[Set[String]].map(Template.Placement.fromString),
        onlyShowForUserLoginNames = state.unpickle[Option[Set[String]]],
        zeroSum = state.unpickle[Boolean],
        iconClass = state.unpickle[String],
        transactions = state.unpickle[Seq[Template.Transaction]],
      )
    }
  }

  implicit object CurrencyPickler extends Pickler[Currency] {
    override def pickle(currency: Currency)(implicit state: PickleState): Unit = logExceptions {
      state.pickle(currency.code)
    }
    override def unpickle(implicit state: UnpickleState): Currency = logExceptions {
      Currency.of(state.unpickle[String])
    }
  }

  implicit object ChartSpecPickler extends Pickler[ChartSpec] {
    override def pickle(value: ChartSpec)(implicit state: PickleState): Unit = logExceptions {
      state.pickle(value.lines)
      state.pickle(value.correctForInflation)
      state.pickle(value.aggregationPeriod)
    }
    override def unpickle(implicit state: UnpickleState): ChartSpec = logExceptions {
      ChartSpec(
        state.unpickle[Seq[ChartSpec.Line]],
        state.unpickle[Boolean],
        state.unpickle[ChartSpec.AggregationPeriod],
      )
    }
  }
  implicit val chartSpecAggregationPeriodPickler: Pickler[ChartSpec.AggregationPeriod] =
    enumPicklerWithoutName(ChartSpec.AggregationPeriod.all)

  override implicit val entityPickler: Pickler[Entity] = compositePickler[Entity]
    .addConcreteType[User]
    .addConcreteType[Transaction]
    .addConcreteType[TransactionGroup]
    .addConcreteType[BalanceCheck]
    .addConcreteType[ExchangeRateMeasurement]

  implicit val MonthPickler: Pickler[Month] =
    enumPickler(stableNameMapper = _.name, values = Month.values().toVector)
}
