---
--- MIT License
--- Copyright (c) Evgeni Chasnovski
--- Copyright (c) Andreas Schneider
---
--- ==============================================================================
local Helpers = {}

-- Add extra expectations
Helpers.expect = vim.deepcopy(MiniTest.expect)

Helpers.expect.match = MiniTest.new_expectation(
    'string matching',
    function(str, pattern)
        return str:find(pattern) ~= nil
    end,
    function(str, pattern)
        return string.format(
            'Pattern: %s\nObserved string: %s',
            vim.inspect(pattern),
            str
        )
    end
)

Helpers.expect.no_match = MiniTest.new_expectation(
    'no string matching',
    function(str, pattern)
        return str:find(pattern) == nil
    end,
    function(str, pattern)
        return string.format(
            'Pattern: %s\nObserved string: %s',
            vim.inspect(pattern),
            str
        )
    end
)

-- Monkey-patch `MiniTest.new_child_neovim` with helpful wrappers
Helpers.new_child_neovim = function()
    local child = MiniTest.new_child_neovim()

    local prevent_hanging = function(method)
    -- stylua: ignore
    if not child.is_blocked() then return end

        local msg = string.format(
            'Can not use `child.%s` because child process is blocked.',
            method
        )
        error(msg)
    end

    child.setup = function()
        child.restart({ '-u', 'tests/minimal_init.lua' })

        -- Change initial buffer to be readonly. This not only increases execution
        -- speed, but more closely resembles manually opened Neovim.
        child.bo.readonly = false
    end

    child.set_lines = function(arr, start, finish)
        prevent_hanging('set_lines')

        if type(arr) == 'string' then
            arr = vim.split(arr, '\n')
        end

        child.api.nvim_buf_set_lines(0, start or 0, finish or -1, false, arr)
    end

    child.get_lines = function(start, finish)
        prevent_hanging('get_lines')

        return child.api.nvim_buf_get_lines(0, start or 0, finish or -1, false)
    end

    child.set_cursor = function(line, column, win_id)
        prevent_hanging('set_cursor')

        child.api.nvim_win_set_cursor(win_id or 0, { line, column })
    end

    child.get_cursor = function(win_id)
        prevent_hanging('get_cursor')

        return child.api.nvim_win_get_cursor(win_id or 0)
    end

    child.set_size = function(lines, columns)
        prevent_hanging('set_size')

        if type(lines) == 'number' then
            child.o.lines = lines
        end

        if type(columns) == 'number' then
            child.o.columns = columns
        end
    end

    child.get_size = function()
        prevent_hanging('get_size')

        return { child.o.lines, child.o.columns }
    end

    child.get_layout_windows = function(layout)
        local res = {}
        local traverse
        traverse = function(l)
            if l[1] == 'leaf' then
                table.insert(res, l[2])
                return
            end
            for _, sub_l in ipairs(l[2]) do
                traverse(sub_l)
            end
        end
        traverse(layout)

        return res
    end

    child.get_resize_state = function()
        local layout = child.fn.winlayout()

        local windows = child.get_layout_windows(layout)
        local win_ids, sizes, buffer = {}, {}, {}
        for _, win_id in ipairs(windows) do
            sizes[win_id] = {
                height = child.api.nvim_win_get_height(win_id),
                width = child.api.nvim_win_get_width(win_id),
            }
            buffer[win_id] = child.api.nvim_win_get_buf(win_id)
            table.insert(win_ids, win_id)
        end

        return {
            windows = win_ids,
            layout = layout,
            sizes = sizes,
            buffer = buffer,
        }
    end

    --- Assert visual marks
    ---
    --- Useful to validate visual selection
    ---
    ---@param first number|table Table with start position or number to check linewise.
    ---@param last number|table Table with finish position or number to check linewise.
    ---@private
    child.expect_visual_marks = function(first, last)
        child.ensure_normal_mode()

        first = type(first) == 'number' and { first, 0 } or first
        last = type(last) == 'number' and { last, 2147483647 } or last

        MiniTest.expect.equality(child.api.nvim_buf_get_mark(0, '<'), first)
        MiniTest.expect.equality(child.api.nvim_buf_get_mark(0, '>'), last)
    end

    child.expect_screenshot = function(opts, path, screenshot_opts)
        if child.fn.has('nvim-0.8') == 0 then
            MiniTest.skip(
                'Screenshots are tested for Neovim>=0.8 (for simplicity).'
            )
        end

        MiniTest.expect.reference_screenshot(
            child.get_screenshot(screenshot_opts),
            path,
            opts
        )
    end

    child.load_module = function(config)
        local lua_cmd = [[require('focus').setup(...)]]

        child.lua(lua_cmd, { config })
    end

    child.unload_module = function()
        -- Unload Lua module
        child.lua([[package.loaded['focus'] = nil]])

        -- Remove global table
        child.lua('_G[Focus] = nil')

        -- Remove autocmd group
        if child.fn.exists('#Focus') == 1 then
            vim.api.nvim_create_augroup('Focus', { clear = true })
        end
    end

    return child
end

-- Mark test failure as "flaky"
Helpers.mark_flaky = function()
    MiniTest.finally(function()
        if #MiniTest.current.case.exec.fails > 0 then
            MiniTest.add_note('This test is flaky.')
        end
    end)
end

return Helpers
