/*jshint esversion: 9 */
// models
const BotCommand = require('@models/BotCommand'),
    Common = require('@helpers/common');

/**
 * Command: permit
 * The bot master can permit authority to other users
 *
 * usage: !permit @username1 @username2
 *
 * @param   {[MessageDetails]}  msg     [message releated helper functions]
 *
 * @return  {[undefined]}
 */
function permit(msg) {
    const server = msg.server;

    if (!msg.ownerIsMaster()) {
        msg.il8nResponse('permit.nope');
        return;
    }

    let roles = msg.message.mentions.roles.concat(msg.getNonSnowflakeRoles());
    if (msg.message.mentions.members.size == 0 && roles.size == 0) {
        msg.il8nResponse('permit.none');
        return;
    }

    msg.message.mentions.members.forEach((member) => server.permit(member.id));
    roles.forEach((role) => server.permit(role.id));

    var nicks = Common.makeNiceCsv(
        msg.message.mentions.members.concat(roles),
        (e) => e.displayName || e.name,
    );
    msg.il8nResponse('permit.okay', { name: nicks });
}

/**
 * Command: unpermit
 * The bot master can permit authority to other users
 * A person permited may unpermit themselves
 *
 * usage: !unpermit @username1 @username2 @role1
 *
 * @param   {[MessageDetails]}  msg     [message releated helper functions]
 *
 * @return  {[undefined]}
 */
function unpermit(msg) {
    var server = msg.server;

    if (!msg.ownerIsPermitted()) {
        msg.il8nResponse('unpermit.deny');
        return;
    }

    let roles = msg.message.mentions.roles.concat(msg.getNonSnowflakeRoles());
    roles.forEach((role) => server.unpermit(role.id));

    if (msg.message.mentions.members.size == 0 && roles.size == 0) {
        server.unpermit(msg.message.member.id);
    }

    msg.message.mentions.members.forEach((member) => {
        if (member.id != msg.message.member.id && !msg.ownerIsMaster()) {
            msg.il8nResponse('unpermit.deny');
            return;
        }

        server.unpermit(member.id);
    });

    msg.message.mentions.roles.forEach((role) => {
        if (!msg.ownerIsMaster()) {
            msg.il8nResponse('unpermit.deny');
            return;
        }

        server.unpermit(role.id);
    });

    var nicks = Common.makeNiceCsv(
        msg.message.mentions.members.concat(roles),
        (e) => e.displayName || e.name,
    );
    msg.il8nResponse('unpermit.okay', { name: nicks });
}

var command_permit = new BotCommand({
    command_name: 'permit',
    execute: permit,
    short_help: 'permit.shorthelp',
    long_help: 'permit.longhelp',
    group: 'control',
    // parameters: "[<user>]",
    order: 5,
});

var command_unpermit = new BotCommand({
    command_name: 'unpermit',
    execute: unpermit,
    short_help: 'unpermit.shorthelp',
    long_help: 'unpermit.longhelp',
    group: 'control',
    // parameters: "[<user>]",
    order: 6,
});

exports.register = function (commands) {
    commands.add(command_permit);
    commands.add(command_unpermit);
};

exports.unRegister = function (commands) {
    commands.remove(command_permit);
    commands.remove(command_unpermit);
};
