# Extending Nx with Plugins

Nx's core functionality focuses on task running and understanding your project and task graph. Nx plugins leverage that functionality to enforce best practices, seamlessly integrate tooling and allow developers to get up and running quickly.

As your repository grows, you'll discover more reasons to create your own plugin

- You can help encourage your coworkers to consistently follow best practices by creating [code generators](/features/generate-code) that are custom built for your repository.
- You can remove duplicate configuration and ensure accurate caching settings by writing your own [inferred tasks](/concepts/inferred-tasks).
- For organizations with multiple monorepos, you can encourage consistency across repositories by providing a repository [preset](/extending-nx/recipes/create-preset) and writing [migrations](/extending-nx/recipes/migration-generators) that will help keep every project in sync.
- You can write a plugin that integrates a tool or framework into Nx and then [share your plugin](/extending-nx/recipes/publish-plugin) with the broader community.

## Create Your Own Plugin

Get started developing your own plugin with a few terminal commands:

{% side-by-side %}

```shell {% title="Create a plugin in a new workspace" %}
npx create-nx-plugin my-plugin
```

```shell {% title="Add a plugin to an existing workspace" %}
npx nx add @nx/plugin
npx nx g plugin tools/my-plugin
```

{% /side-by-side %}

## Learn by Doing

You can follow along with one of the step by step tutorials below that is focused on a particular use-case. These tutorials expect you to already have the following skills:

- [Run tasks](/features/run-tasks) with Nx and configure Nx to [infers tasks for you](/concepts/inferred-tasks)
- [Use code generators](/features/generate-code)
- Understand the [project graph](/features/explore-graph)
- Write [TypeScript](https://www.typescriptlang.org/) code

{% cards cols="2" %}

{% link-card title="Enforce Best Practices in Your Repository" type="tutorial" url="/extending-nx/tutorials/organization-specific-plugin" icon="BuildingOfficeIcon" /%}
{% link-card title="Integrate a Tool Into an Nx Repository" type="tutorial" url="/extending-nx/tutorials/tooling-plugin" icon="WrenchScrewdriverIcon" /%}

{% /cards %}

## Create Your First Code Generator

Wire up a new generator with this terminal command:

```shell
npx nx g generator my-plugin/src/generators/library-with-readme
```

### Understand the Generator Functionality

This command will register the generator in the plugin's `generators.json` file and create some default generator code in the `library-with-readme` folder. The `libraryWithReadmeGenerator` function in the `generator.ts` file is where the generator functionality is defined.

```typescript {% fileName="my-plugin/src/generators/library-with-readme/generator.ts" %}
export async function libraryWithReadmeGenerator(
  tree: Tree,
  options: LibraryWithReadmeGeneratorSchema
) {
  const projectRoot = `libs/${options.name}`;
  addProjectConfiguration(tree, options.name, {
    root: projectRoot,
    projectType: 'library',
    sourceRoot: `${projectRoot}/src`,
    targets: {},
  });
  generateFiles(tree, path.join(__dirname, 'files'), projectRoot, options);
  await formatFiles(tree);
}
```

This generator calls the following functions:

- `addProjectConfiguration` - Create a new project configured for TypeScript code.
- `generateFiles` - Create files in the new project based on the template files in the `files` folder.
- `formatFiles` - Format the newly created files with Prettier.

You can find more helper functions in the [Nx Devkit reference documentation](/nx-api/devkit/documents/nx_devkit).

### Create a README Template File

We can remove the generated `index.ts.template` file and add our own `README.md.template` file in the `files` folder.

```typescript {% fileName="my-plugin/src/generators/library-with-readme/files/README.md.template" %}
# <%= name %>

This was generated by the `library-with-readme` generator!
```

The template files that are used in the `generateFiles` function can inject variables and functionality using the EJS syntax. Our README template will replace `<%= name %>` with the name specified in the generator. Read more about the EJS syntax in the [creating files with a generator recipe](/extending-nx/recipes/creating-files).

### Run Your Generator

You can test your generator in dry-run mode with the following command:

```shell
npx nx g my-plugin:library-with-readme mylib --dry-run
```

If you're happy with the files that are generated, you can actually run the generator by removing the `--dry-run` flag.

## Next Steps

- [Browse the plugin registry](/plugin-registry) to find one that suits your needs.
- [Sign up for Nx Enterprise](/enterprise) to get dedicated support from Nx team members.
- [Collaborate on the Nx Discord](https://go.nx.dev/community) to work with other plugin authors.
