# Domain model.
class Domain < ActiveRecord::Base
  include Permissionable

  has_many :aliases, dependent: :destroy
  has_many :mailboxes, dependent: :destroy

  attr_accessible :active, :backupmx, :catch_all_address, :description, :name,
    :quick_access

  default_scope order('name asc')

  scope :quick_access, where(quick_access: true)

  validates :name,
    format: { with: /\A[a-z0-9]+([\-\.]{1}[a-z0-9]+)*\.[a-z]{2,5}(:[0-9]{1,5})?(\/.*)?\z/ix },
    presence: true

  before_save -> { name.downcase! }

  has_paper_trail ignore: :mx_set

  searchkick word_middle: [:name, :description]
  # Search fields options includable in search on model.
  SEARCH_FIELDS = [
    { name: :word_middle },
    { description: :word_middle }
  ]

  # Aliases count for simple_form.
  def aliases_count
    aliases.count
  end

  # Forwarding of all mails to non-existing addresses.
  def catch_all_alias
    aliases.where(username: nil).first
  end

  # Target(s) of catch all Alias.
  def catch_all_address
    catch_all_alias.try :goto
  end

  # Setter for target(s) of catch all Alias.
  def catch_all_address=(goto)
    return catch_all_alias.try(:destroy) if goto.blank?

    a = catch_all_alias || aliases.build
    a.goto = goto
    a.save! validate: false
  end

  # Returns URL array for editing a model instance.
  def edit_url_array
    [:edit, :admin, self]
  end

  # Mailboxes count for simple_form.
  def mailboxes_count
    mailboxes.count
  end

  # Mailboxes for select input.
  def mailboxes_for_select
    mailboxes.map { |m| [m.email, m.id] }
  end

  # Check if MX of Domain is set to default MX for Møil instance.
  def update_mx_set!
    self.mx_set = (Resolve.address(Resolve.mx(self)) == Resolve.address(Settings.mx_check_domain))
    save!
  end

  # String representation.
  def to_s
    name
  end

  # Default Domain. (First one, if not configured in config/settings.yml).
  def self.default
    domain = where(name: Settings.default_domain).limit(1).first
    domain = Domain.first if domain.nil?

    domain
  end

  # Managable Domains for given mailbox.
  def self.managable(mailbox)
    if mailbox.admin?
      Domain
    else
      Domain.joins(:permissions).where('
        (permissions.role = ? or permissions.role = ?)
        and subject_id = ?',
        'editor',
        'owner',
        mailbox.id
      )
    end
  end
end
