import React, { useRef, useState, useEffect, useCallback } from 'react';
import PropTypes from 'prop-types';
import { InternalOverlay, RenderIf } from 'react-rainbow-components';
import { useWindowResize, useUniqueIdentifier } from '@rainbow-modules/hooks';
import { StyledTooltip, StyledTitle, StyledText, StyledButton, StyledSample } from './styled';
import { ESCAPE_KEY } from './helpers/constants';

const RainbowIconItem = (props) => {
    const { rainbowIcon, iconBoxId } = props;

    const triggerRef = useRef();
    const helpTextId = useUniqueIdentifier('help-text');
    const [isFocused, setIsFocused] = useState(false);
    const isClickTooltip = useRef(false);
    const [isOpen, setIsOpen] = useState(false);
    const closeOverlay = useCallback(() => setIsOpen(false), []);
    const openOverlay = useCallback(() => setIsOpen(true), []);

    useEffect(() => {
        if (isFocused) {
            openOverlay();
        } else {
            closeOverlay();
        }
    }, [closeOverlay, isFocused, openOverlay]);

    useWindowResize(() => closeOverlay(), isOpen);

    const handleBlur = () => {
        if (!isClickTooltip.current) {
            setIsFocused(false);
        }
    };

    const handleTooltipMouseDown = () => {
        isClickTooltip.current = true;
    };

    const handleTooltipMouseUp = () => {
        isClickTooltip.current = false;
        triggerRef.current.focus();
    };

    const handleKeyPressed = (event) => {
        if (event.keyCode === ESCAPE_KEY) {
            event.preventDefault();
            closeOverlay();
        }
    };

    const Icon = rainbowIcon.icon;
    const iconName = rainbowIcon.name.charAt(0).toUpperCase() + rainbowIcon.name.slice(1);
    const importLine = `import { ${iconName} } from "@rainbow-modules/icons"`;
    const iconTag = `<${iconName} />`;

    return (
        <>
            <StyledButton
                id={iconBoxId}
                ref={triggerRef}
                onFocus={() => setIsFocused(true)}
                onBlur={handleBlur}
                onKeyDown={handleKeyPressed}
                type="button"
                ariaLabelledby={helpTextId}
            >
                <Icon />
                <div>{rainbowIcon.name}</div>
            </StyledButton>
            <RenderIf isTrue={!!rainbowIcon}>
                <InternalOverlay
                    isVisible={isOpen}
                    render={() => {
                        return (
                            <StyledTooltip
                                id={helpTextId}
                                role="tooltip"
                                onMouseDown={handleTooltipMouseDown}
                                onMouseUp={handleTooltipMouseUp}
                            >
                                <StyledTitle>Using rainbow/icons</StyledTitle>
                                <StyledSample>Example</StyledSample>
                                <StyledText>
                                    <div>{importLine}</div>
                                    <div>{iconTag}</div>
                                </StyledText>
                            </StyledTooltip>
                        );
                    }}
                    triggerElementRef={triggerRef}
                />
            </RenderIf>
        </>
    );
};

RainbowIconItem.propTypes = {
    /** Icons list with all existing icons in Rainbow Icons Collection. */
    rainbowIcon: PropTypes.object,
    /** id number for the box. */
    iconBoxId: PropTypes.number,
};

RainbowIconItem.defaultProps = {
    rainbowIcon: null,
    iconBoxId: null,
};

export default RainbowIconItem;
