import { InjectedMessages } from 'messages-modules'
import { useRouter } from 'next/router'
import { extname, parse as parsePath, resolve } from 'node:path'
import { useMemo } from 'react'
import { highlight, highlightFilePath, log, normalizeLocale } from '../'
import {
  getLastPathSegment,
  getPathWithoutLastSegment,
  normalizePagesFilePath,
} from '../helpers/paths-utils'
import { Messages } from './messages'

// Make message classes available without adding their files to the package exports.
export { Message } from './message'
export { Messages } from './messages'
export type { KeyValueObject } from './properties'

/** This is the regular expression to validate message key segments. */
export const keySegmentRegExp = /^[\da-z]{1,50}$/i
/** This is the regular expression description to keep logs consistent. */
export const keySegmentRegExpDescription = 'must be between 1 and 50 alphanumeric characters'

/**
 * The message key identifier used for slugs.
 */
export const SLUG_KEY_ID = 'slug'

/**
 * The message key identifier used for titles.
 */
export const TITLE_KEY_ID = 'title'

/**
 * Get a page's title from the locale scope messages.
 *
 * A page's `slug` (human readable short description) can meet most use cases for title but
 * sometimes you might want to customize it. This helper API will check if the `title` message
 * is available first, and if not try to fallback on the `slug`.
 *
 * @param messages - The object containing localized messages of a local scope.
 * @param values - The values of the title's placeholders (e.g., `{name: 'Joe'}`), if any.
 *
 * @returns The message message as a string.
 */
export const getTitle = (messages: Messages, values?: PlaceholderValues): string => {
  const titleMessage = messages.get(TITLE_KEY_ID)
  const slugMessage = messages.get(SLUG_KEY_ID)

  const applicableTitle = titleMessage === undefined ? slugMessage : titleMessage

  if (applicableTitle === undefined) {
    log.warn(
      `unable to use ${highlight('getTitle')} in ${highlightFilePath(
        messages.sourceFilePath
      )} because keys with identifiers ${highlight(TITLE_KEY_ID)} and ${highlight(
        SLUG_KEY_ID
      )} were not found in messages file ${highlightFilePath(messages.messagesFilePath)}`
    )
    return ''
  }

  return applicableTitle.format(values)
}

/**
 * Converts a localized message into its "slug format" representation.
 *
 * This is used by `next-multilingual` to build localized URLs and can be re-used for other similar
 * purposes such as anchor links.
 *
 * The `locale` must always be specified when available since some languages use ASCII characters for one of
 * their cases but not the other. For example, Turkish capital `I` is `ı` in lowercase and will only be
 * mapped correctly using `.toLocaleLowerCase('tr-TR')`.
 *
 * @param message - A localized message to "slugify".
 * @param locale - The locale of the message (used for locale specific case mapping).
 *
 * @returns The "slugified" version of a localized message.
 */
export const slugify = (message: string, locale: string): string =>
  /**
   * To stay ES5 compatible, and support all Unicode letters and numbers, we are using this very long regular
   * expression that has been generated using the `regexpu-core` npm package which is also used by
   * `@babel/plugin-transform-unicode-regex`. Since Next.js is moving to SWC instead of Babel, this equivalent
   * plugin did not seem available on SWC as per https://swc.rs/docs/migrating-from-babel.
   *
   * This regular expression can be updated by using the following code:
   *
   * import rewritePattern from 'regexpu-core';
   *
   * console.log(
   *   rewritePattern('[^\\p{L}\\p{N}]+', 'u', {
   *     unicodeFlag: 'transform',
   *     unicodePropertyEscapes: 'transform',
   *   })
   * );
   *
   */
  message
    .replaceAll(
      /(?:(?![\dA-Za-z\u00AA\u00B2\u00B3\u00B5\u00B9\u00BA\u00BC-\u00BE\u00C0-\u00D6\u00D8-\u00F6\u00F8-\u02C1\u02C6-\u02D1\u02E0-\u02E4\u02EC\u02EE\u0370-\u0374\u0376\u0377\u037A-\u037D\u037F\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u048A-\u052F\u0531-\u0556\u0559\u0560-\u0588\u05D0-\u05EA\u05EF-\u05F2\u0620-\u064A\u0660-\u0669\u066E\u066F\u0671-\u06D3\u06D5\u06E5\u06E6\u06EE-\u06FC\u06FF\u0710\u0712-\u072F\u074D-\u07A5\u07B1\u07C0-\u07EA\u07F4\u07F5\u07FA\u0800-\u0815\u081A\u0824\u0828\u0840-\u0858\u0860-\u086A\u0870-\u0887\u0889-\u088E\u08A0-\u08C9\u0904-\u0939\u093D\u0950\u0958-\u0961\u0966-\u096F\u0971-\u0980\u0985-\u098C\u098F\u0990\u0993-\u09A8\u09AA-\u09B0\u09B2\u09B6-\u09B9\u09BD\u09CE\u09DC\u09DD\u09DF-\u09E1\u09E6-\u09F1\u09F4-\u09F9\u09FC\u0A05-\u0A0A\u0A0F\u0A10\u0A13-\u0A28\u0A2A-\u0A30\u0A32\u0A33\u0A35\u0A36\u0A38\u0A39\u0A59-\u0A5C\u0A5E\u0A66-\u0A6F\u0A72-\u0A74\u0A85-\u0A8D\u0A8F-\u0A91\u0A93-\u0AA8\u0AAA-\u0AB0\u0AB2\u0AB3\u0AB5-\u0AB9\u0ABD\u0AD0\u0AE0\u0AE1\u0AE6-\u0AEF\u0AF9\u0B05-\u0B0C\u0B0F\u0B10\u0B13-\u0B28\u0B2A-\u0B30\u0B32\u0B33\u0B35-\u0B39\u0B3D\u0B5C\u0B5D\u0B5F-\u0B61\u0B66-\u0B6F\u0B71-\u0B77\u0B83\u0B85-\u0B8A\u0B8E-\u0B90\u0B92-\u0B95\u0B99\u0B9A\u0B9C\u0B9E\u0B9F\u0BA3\u0BA4\u0BA8-\u0BAA\u0BAE-\u0BB9\u0BD0\u0BE6-\u0BF2\u0C05-\u0C0C\u0C0E-\u0C10\u0C12-\u0C28\u0C2A-\u0C39\u0C3D\u0C58-\u0C5A\u0C5D\u0C60\u0C61\u0C66-\u0C6F\u0C78-\u0C7E\u0C80\u0C85-\u0C8C\u0C8E-\u0C90\u0C92-\u0CA8\u0CAA-\u0CB3\u0CB5-\u0CB9\u0CBD\u0CDD\u0CDE\u0CE0\u0CE1\u0CE6-\u0CEF\u0CF1\u0CF2\u0D04-\u0D0C\u0D0E-\u0D10\u0D12-\u0D3A\u0D3D\u0D4E\u0D54-\u0D56\u0D58-\u0D61\u0D66-\u0D78\u0D7A-\u0D7F\u0D85-\u0D96\u0D9A-\u0DB1\u0DB3-\u0DBB\u0DBD\u0DC0-\u0DC6\u0DE6-\u0DEF\u0E01-\u0E30\u0E32\u0E33\u0E40-\u0E46\u0E50-\u0E59\u0E81\u0E82\u0E84\u0E86-\u0E8A\u0E8C-\u0EA3\u0EA5\u0EA7-\u0EB0\u0EB2\u0EB3\u0EBD\u0EC0-\u0EC4\u0EC6\u0ED0-\u0ED9\u0EDC-\u0EDF\u0F00\u0F20-\u0F33\u0F40-\u0F47\u0F49-\u0F6C\u0F88-\u0F8C\u1000-\u102A\u103F-\u1049\u1050-\u1055\u105A-\u105D\u1061\u1065\u1066\u106E-\u1070\u1075-\u1081\u108E\u1090-\u1099\u10A0-\u10C5\u10C7\u10CD\u10D0-\u10FA\u10FC-\u1248\u124A-\u124D\u1250-\u1256\u1258\u125A-\u125D\u1260-\u1288\u128A-\u128D\u1290-\u12B0\u12B2-\u12B5\u12B8-\u12BE\u12C0\u12C2-\u12C5\u12C8-\u12D6\u12D8-\u1310\u1312-\u1315\u1318-\u135A\u1369-\u137C\u1380-\u138F\u13A0-\u13F5\u13F8-\u13FD\u1401-\u166C\u166F-\u167F\u1681-\u169A\u16A0-\u16EA\u16EE-\u16F8\u1700-\u1711\u171F-\u1731\u1740-\u1751\u1760-\u176C\u176E-\u1770\u1780-\u17B3\u17D7\u17DC\u17E0-\u17E9\u17F0-\u17F9\u1810-\u1819\u1820-\u1878\u1880-\u1884\u1887-\u18A8\u18AA\u18B0-\u18F5\u1900-\u191E\u1946-\u196D\u1970-\u1974\u1980-\u19AB\u19B0-\u19C9\u19D0-\u19DA\u1A00-\u1A16\u1A20-\u1A54\u1A80-\u1A89\u1A90-\u1A99\u1AA7\u1B05-\u1B33\u1B45-\u1B4C\u1B50-\u1B59\u1B83-\u1BA0\u1BAE-\u1BE5\u1C00-\u1C23\u1C40-\u1C49\u1C4D-\u1C7D\u1C80-\u1C88\u1C90-\u1CBA\u1CBD-\u1CBF\u1CE9-\u1CEC\u1CEE-\u1CF3\u1CF5\u1CF6\u1CFA\u1D00-\u1DBF\u1E00-\u1F15\u1F18-\u1F1D\u1F20-\u1F45\u1F48-\u1F4D\u1F50-\u1F57\u1F59\u1F5B\u1F5D\u1F5F-\u1F7D\u1F80-\u1FB4\u1FB6-\u1FBC\u1FBE\u1FC2-\u1FC4\u1FC6-\u1FCC\u1FD0-\u1FD3\u1FD6-\u1FDB\u1FE0-\u1FEC\u1FF2-\u1FF4\u1FF6-\u1FFC\u2070\u2071\u2074-\u2079\u207F-\u2089\u2090-\u209C\u2102\u2107\u210A-\u2113\u2115\u2119-\u211D\u2124\u2126\u2128\u212A-\u212D\u212F-\u2139\u213C-\u213F\u2145-\u2149\u214E\u2150-\u2189\u2460-\u249B\u24EA-\u24FF\u2776-\u2793\u2C00-\u2CE4\u2CEB-\u2CEE\u2CF2\u2CF3\u2CFD\u2D00-\u2D25\u2D27\u2D2D\u2D30-\u2D67\u2D6F\u2D80-\u2D96\u2DA0-\u2DA6\u2DA8-\u2DAE\u2DB0-\u2DB6\u2DB8-\u2DBE\u2DC0-\u2DC6\u2DC8-\u2DCE\u2DD0-\u2DD6\u2DD8-\u2DDE\u2E2F\u3005-\u3007\u3021-\u3029\u3031-\u3035\u3038-\u303C\u3041-\u3096\u309D-\u309F\u30A1-\u30FA\u30FC-\u30FF\u3105-\u312F\u3131-\u318E\u3192-\u3195\u31A0-\u31BF\u31F0-\u31FF\u3220-\u3229\u3248-\u324F\u3251-\u325F\u3280-\u3289\u32B1-\u32BF\u3400-\u4DBF\u4E00-\uA48C\uA4D0-\uA4FD\uA500-\uA60C\uA610-\uA62B\uA640-\uA66E\uA67F-\uA69D\uA6A0-\uA6EF\uA717-\uA71F\uA722-\uA788\uA78B-\uA7CA\uA7D0\uA7D1\uA7D3\uA7D5-\uA7D9\uA7F2-\uA801\uA803-\uA805\uA807-\uA80A\uA80C-\uA822\uA830-\uA835\uA840-\uA873\uA882-\uA8B3\uA8D0-\uA8D9\uA8F2-\uA8F7\uA8FB\uA8FD\uA8FE\uA900-\uA925\uA930-\uA946\uA960-\uA97C\uA984-\uA9B2\uA9CF-\uA9D9\uA9E0-\uA9E4\uA9E6-\uA9FE\uAA00-\uAA28\uAA40-\uAA42\uAA44-\uAA4B\uAA50-\uAA59\uAA60-\uAA76\uAA7A\uAA7E-\uAAAF\uAAB1\uAAB5\uAAB6\uAAB9-\uAABD\uAAC0\uAAC2\uAADB-\uAADD\uAAE0-\uAAEA\uAAF2-\uAAF4\uAB01-\uAB06\uAB09-\uAB0E\uAB11-\uAB16\uAB20-\uAB26\uAB28-\uAB2E\uAB30-\uAB5A\uAB5C-\uAB69\uAB70-\uABE2\uABF0-\uABF9\uAC00-\uD7A3\uD7B0-\uD7C6\uD7CB-\uD7FB\uF900-\uFA6D\uFA70-\uFAD9\uFB00-\uFB06\uFB13-\uFB17\uFB1D\uFB1F-\uFB28\uFB2A-\uFB36\uFB38-\uFB3C\uFB3E\uFB40\uFB41\uFB43\uFB44\uFB46-\uFBB1\uFBD3-\uFD3D\uFD50-\uFD8F\uFD92-\uFDC7\uFDF0-\uFDFB\uFE70-\uFE74\uFE76-\uFEFC\uFF10-\uFF19\uFF21-\uFF3A\uFF41-\uFF5A\uFF66-\uFFBE\uFFC2-\uFFC7\uFFCA-\uFFCF\uFFD2-\uFFD7\uFFDA-\uFFDC]|\uD800[\uDC00-\uDC0B\uDC0D-\uDC26\uDC28-\uDC3A\uDC3C\uDC3D\uDC3F-\uDC4D\uDC50-\uDC5D\uDC80-\uDCFA\uDD07-\uDD33\uDD40-\uDD78\uDD8A\uDD8B\uDE80-\uDE9C\uDEA0-\uDED0\uDEE1-\uDEFB\uDF00-\uDF23\uDF2D-\uDF4A\uDF50-\uDF75\uDF80-\uDF9D\uDFA0-\uDFC3\uDFC8-\uDFCF\uDFD1-\uDFD5]|\uD801[\uDC00-\uDC9D\uDCA0-\uDCA9\uDCB0-\uDCD3\uDCD8-\uDCFB\uDD00-\uDD27\uDD30-\uDD63\uDD70-\uDD7A\uDD7C-\uDD8A\uDD8C-\uDD92\uDD94\uDD95\uDD97-\uDDA1\uDDA3-\uDDB1\uDDB3-\uDDB9\uDDBB\uDDBC\uDE00-\uDF36\uDF40-\uDF55\uDF60-\uDF67\uDF80-\uDF85\uDF87-\uDFB0\uDFB2-\uDFBA]|\uD802[\uDC00-\uDC05\uDC08\uDC0A-\uDC35\uDC37\uDC38\uDC3C\uDC3F-\uDC55\uDC58-\uDC76\uDC79-\uDC9E\uDCA7-\uDCAF\uDCE0-\uDCF2\uDCF4\uDCF5\uDCFB-\uDD1B\uDD20-\uDD39\uDD80-\uDDB7\uDDBC-\uDDCF\uDDD2-\uDE00\uDE10-\uDE13\uDE15-\uDE17\uDE19-\uDE35\uDE40-\uDE48\uDE60-\uDE7E\uDE80-\uDE9F\uDEC0-\uDEC7\uDEC9-\uDEE4\uDEEB-\uDEEF\uDF00-\uDF35\uDF40-\uDF55\uDF58-\uDF72\uDF78-\uDF91\uDFA9-\uDFAF]|\uD803[\uDC00-\uDC48\uDC80-\uDCB2\uDCC0-\uDCF2\uDCFA-\uDD23\uDD30-\uDD39\uDE60-\uDE7E\uDE80-\uDEA9\uDEB0\uDEB1\uDF00-\uDF27\uDF30-\uDF45\uDF51-\uDF54\uDF70-\uDF81\uDFB0-\uDFCB\uDFE0-\uDFF6]|\uD804[\uDC03-\uDC37\uDC52-\uDC6F\uDC71\uDC72\uDC75\uDC83-\uDCAF\uDCD0-\uDCE8\uDCF0-\uDCF9\uDD03-\uDD26\uDD36-\uDD3F\uDD44\uDD47\uDD50-\uDD72\uDD76\uDD83-\uDDB2\uDDC1-\uDDC4\uDDD0-\uDDDA\uDDDC\uDDE1-\uDDF4\uDE00-\uDE11\uDE13-\uDE2B\uDE80-\uDE86\uDE88\uDE8A-\uDE8D\uDE8F-\uDE9D\uDE9F-\uDEA8\uDEB0-\uDEDE\uDEF0-\uDEF9\uDF05-\uDF0C\uDF0F\uDF10\uDF13-\uDF28\uDF2A-\uDF30\uDF32\uDF33\uDF35-\uDF39\uDF3D\uDF50\uDF5D-\uDF61]|\uD805[\uDC00-\uDC34\uDC47-\uDC4A\uDC50-\uDC59\uDC5F-\uDC61\uDC80-\uDCAF\uDCC4\uDCC5\uDCC7\uDCD0-\uDCD9\uDD80-\uDDAE\uDDD8-\uDDDB\uDE00-\uDE2F\uDE44\uDE50-\uDE59\uDE80-\uDEAA\uDEB8\uDEC0-\uDEC9\uDF00-\uDF1A\uDF30-\uDF3B\uDF40-\uDF46]|\uD806[\uDC00-\uDC2B\uDCA0-\uDCF2\uDCFF-\uDD06\uDD09\uDD0C-\uDD13\uDD15\uDD16\uDD18-\uDD2F\uDD3F\uDD41\uDD50-\uDD59\uDDA0-\uDDA7\uDDAA-\uDDD0\uDDE1\uDDE3\uDE00\uDE0B-\uDE32\uDE3A\uDE50\uDE5C-\uDE89\uDE9D\uDEB0-\uDEF8]|\uD807[\uDC00-\uDC08\uDC0A-\uDC2E\uDC40\uDC50-\uDC6C\uDC72-\uDC8F\uDD00-\uDD06\uDD08\uDD09\uDD0B-\uDD30\uDD46\uDD50-\uDD59\uDD60-\uDD65\uDD67\uDD68\uDD6A-\uDD89\uDD98\uDDA0-\uDDA9\uDEE0-\uDEF2\uDFB0\uDFC0-\uDFD4]|\uD808[\uDC00-\uDF99]|\uD809[\uDC00-\uDC6E\uDC80-\uDD43]|\uD80B[\uDF90-\uDFF0]|[\uD80C\uD81C-\uD820\uD822\uD840-\uD868\uD86A-\uD86C\uD86F-\uD872\uD874-\uD879\uD880-\uD883][\uDC00-\uDFFF]|\uD80D[\uDC00-\uDC2E]|\uD811[\uDC00-\uDE46]|\uD81A[\uDC00-\uDE38\uDE40-\uDE5E\uDE60-\uDE69\uDE70-\uDEBE\uDEC0-\uDEC9\uDED0-\uDEED\uDF00-\uDF2F\uDF40-\uDF43\uDF50-\uDF59\uDF5B-\uDF61\uDF63-\uDF77\uDF7D-\uDF8F]|\uD81B[\uDE40-\uDE96\uDF00-\uDF4A\uDF50\uDF93-\uDF9F\uDFE0\uDFE1\uDFE3]|\uD821[\uDC00-\uDFF7]|\uD823[\uDC00-\uDCD5\uDD00-\uDD08]|\uD82B[\uDFF0-\uDFF3\uDFF5-\uDFFB\uDFFD\uDFFE]|\uD82C[\uDC00-\uDD22\uDD50-\uDD52\uDD64-\uDD67\uDD70-\uDEFB]|\uD82F[\uDC00-\uDC6A\uDC70-\uDC7C\uDC80-\uDC88\uDC90-\uDC99]|\uD834[\uDEE0-\uDEF3\uDF60-\uDF78]|\uD835[\uDC00-\uDC54\uDC56-\uDC9C\uDC9E\uDC9F\uDCA2\uDCA5\uDCA6\uDCA9-\uDCAC\uDCAE-\uDCB9\uDCBB\uDCBD-\uDCC3\uDCC5-\uDD05\uDD07-\uDD0A\uDD0D-\uDD14\uDD16-\uDD1C\uDD1E-\uDD39\uDD3B-\uDD3E\uDD40-\uDD44\uDD46\uDD4A-\uDD50\uDD52-\uDEA5\uDEA8-\uDEC0\uDEC2-\uDEDA\uDEDC-\uDEFA\uDEFC-\uDF14\uDF16-\uDF34\uDF36-\uDF4E\uDF50-\uDF6E\uDF70-\uDF88\uDF8A-\uDFA8\uDFAA-\uDFC2\uDFC4-\uDFCB\uDFCE-\uDFFF]|\uD837[\uDF00-\uDF1E]|\uD838[\uDD00-\uDD2C\uDD37-\uDD3D\uDD40-\uDD49\uDD4E\uDE90-\uDEAD\uDEC0-\uDEEB\uDEF0-\uDEF9]|\uD839[\uDFE0-\uDFE6\uDFE8-\uDFEB\uDFED\uDFEE\uDFF0-\uDFFE]|\uD83A[\uDC00-\uDCC4\uDCC7-\uDCCF\uDD00-\uDD43\uDD4B\uDD50-\uDD59]|\uD83B[\uDC71-\uDCAB\uDCAD-\uDCAF\uDCB1-\uDCB4\uDD01-\uDD2D\uDD2F-\uDD3D\uDE00-\uDE03\uDE05-\uDE1F\uDE21\uDE22\uDE24\uDE27\uDE29-\uDE32\uDE34-\uDE37\uDE39\uDE3B\uDE42\uDE47\uDE49\uDE4B\uDE4D-\uDE4F\uDE51\uDE52\uDE54\uDE57\uDE59\uDE5B\uDE5D\uDE5F\uDE61\uDE62\uDE64\uDE67-\uDE6A\uDE6C-\uDE72\uDE74-\uDE77\uDE79-\uDE7C\uDE7E\uDE80-\uDE89\uDE8B-\uDE9B\uDEA1-\uDEA3\uDEA5-\uDEA9\uDEAB-\uDEBB]|\uD83C[\uDD00-\uDD0C]|\uD83E[\uDFF0-\uDFF9]|\uD869[\uDC00-\uDEDF\uDF00-\uDFFF]|\uD86D[\uDC00-\uDF38\uDF40-\uDFFF]|\uD86E[\uDC00-\uDC1D\uDC20-\uDFFF]|\uD873[\uDC00-\uDEA1\uDEB0-\uDFFF]|\uD87A[\uDC00-\uDFE0]|\uD87E[\uDC00-\uDE1D]|\uD884[\uDC00-\uDF4A])[\S\s])+/g,
      '-'
    )
    .replace(/^-/, '') // Remove leading dash, if present.
    .replace(/-$/, '') // Remove trailing dash, if present.
    .toLocaleLowerCase(locale)

/**
 * Get a localized messages file path associated with a Next.js page.
 *
 * @param filesystemPath - The filesystem path (file or directory).
 * @param locale - The locale of the message file.
 *
 * @returns A localized messages file path.
 */
export const getMessagesFilePath = (filesystemPath: string, locale: string): string => {
  const pageFileExtension = extname(filesystemPath)

  if (pageFileExtension) {
    // Filesystem path is a file.
    return `${filesystemPath.replace(pageFileExtension, '')}.${normalizeLocale(locale)}.properties`
  }

  // Filesystem path is a directory.
  return `${filesystemPath}/index.${normalizeLocale(locale)}.properties`
}

/**
 * Get the path of the source file that is calling `useMessages()`.
 *
 * @param messageFilePath - The file path of one of the messages files (any locale).
 * @param sourceFileExtension  - The extension of the source file.
 *
 * @returns The path of the source file that is calling `useMessages()`.
 */
export const getSourceFilePath = (messageFilePath: string, sourceFileExtension: string): string => {
  const messagesFile = parsePath(messageFilePath)
  const sourceFilename = messagesFile.name.split('.').slice(0, -1).join('.')
  return resolve(messagesFile.dir, `${sourceFilename}${sourceFileExtension}`)
}

/**
 * The value of a message's placeholder (e.g., `{name}`).
 */
export type PlaceholderValue = string | number

/**
 * The values of a message's placeholders (e.g., `{name: 'Joe'}`).
 */
export type PlaceholderValues = {
  [key: string]: string | number
}

/**
 * The value of a message's JSX element (e.g., `<b></b>`).
 */
export type JsxValue = JSX.Element

/**
 * The values of a message's JSX elements (e.g., `{b: <b></b>}`).
 */
export type JsxValues = {
  [key: string]: JsxValue
}

/**
 * Any (mixed) message value (placeholders and/or JSX).
 */
export type MixedValue = PlaceholderValue | JsxValue

/**
 * The (mixed) values of a message (placeholder and/or JSX).
 */
export type MixedValues = {
  [key: string]: MixedValue
}

/**
 * Message values by types.
 */
export type MessageValuesByType = {
  placeholderValues: PlaceholderValues
  jsxValues: JsxValues
}

/**
 * An index to optimize `get` access on messages.
 */
export type MessagesIndex = {
  [key: string]: number
}

/**
 * Type guard to check if a message value is a JSX element.
 *
 * @param value - The value of a message (placeholder and/or JSX).
 *
 * @returns True if the value is a JSX element, otherwise false.
 */
export const isJsxValue = (value: MixedValue): value is JsxValue => !isPlaceholderValue(value)

/**
 * Type guard to check if values of messages contain some JSX.
 *
 * @param values - The values of the message's placeholders and/or JSX elements.
 *
 * @returns True if the values contain JSX, otherwise false.
 */
export const isJsxValues = (values: MixedValues): values is PlaceholderValues =>
  Object.values(values).some((value) => isJsxValue(value))

/**
 * Type guard to check if a message value is a placeholder.
 *
 * @param value - The value of a message (placeholder and/or JSX).
 *
 * @returns True if the value is a placeholder, otherwise false.
 */
export const isPlaceholderValue = (value: MixedValue): value is PlaceholderValue =>
  ['string', 'number'].includes(typeof value)

/**
 * Type guard to check if values of messages are all of for placeholders.
 *
 * @param values - The values of the message's placeholders and/or JSX elements.
 *
 * @returns True if the values are all for placeholders (no JSX is present), otherwise false.
 */
export const isPlaceholderValues = (values: MixedValues): values is PlaceholderValues =>
  !isJsxValues(values)

/**
 * React hook to get the localized messages specific to a Next.js context.
 *
 * @returns An object containing the messages of the local scope.
 */
export function useMessages(): Messages {
  const { locale } = useRouter()
  // @ts-expect-error: `this` is injected using `bind` and will trigger a false compilation error.
  // eslint-disable-next-line @typescript-eslint/no-unsafe-argument
  return useMemo(() => handleMessages(this, 'useMessages', locale), [locale])
}

/**
 * Get the localized messages specific to a Next.js context.
 *
 * @param locale - The locale of the message file.
 *
 * @returns An object containing the messages of the local scope.
 */
export function getMessages(locale: string): Messages {
  // @ts-expect-error: `this` is injected using `bind` and will trigger a false compilation error.
  // eslint-disable-next-line @typescript-eslint/no-unsafe-argument
  return handleMessages(this, 'getMessages', locale.toLowerCase())
}

/**
 * The type of a `getMessage` function.
 */
export type GetMessagesFunction = typeof getMessages

/**
 * Handles messages coming from both `useMessages` and `getMessages`.
 *
 * @param injectedMessages - The injected messages object.
 * @param caller - The function calling the message handler.
 * @param locale - The locale of the message file.
 *
 * @returns An object containing the messages of the local scope.
 */
export const handleMessages = (
  injectedMessages: InjectedMessages,
  caller: string,
  locale?: string
): Messages => {
  if (locale === undefined) {
    throw new Error(`${caller}() requires the locales to be configured in Next.js`)
  }

  if (!injectedMessages || !injectedMessages.isInjected) {
    throw new Error(
      `${caller}() requires the 'next-multilingual/messages/babel-plugin' Babel plugin`
    )
  }

  const sourceFilePath = injectedMessages.sourceFilePath
  const sourceLastUrlPathSegment = getLastPathSegment(sourceFilePath)
  const sourceFileDirectoryPath = getPathWithoutLastSegment(normalizePagesFilePath(sourceFilePath))
  const messagesFilename = `${sourceLastUrlPathSegment}.${normalizeLocale(locale)}.properties`
  const messagesFilePath =
    sourceFileDirectoryPath.length > 0
      ? `${sourceFileDirectoryPath}/${messagesFilename}`
      : messagesFilename

  if (!injectedMessages.keyValueObjectCollection[locale]) {
    log.warn(
      `unable to use ${highlight(caller)}() in ${highlightFilePath(
        injectedMessages.sourceFilePath
      )} because no message file could be found at ${highlightFilePath(messagesFilePath)}`
    )
  }

  return new Messages(
    injectedMessages.keyValueObjectCollection[locale],
    locale.toLowerCase(),
    sourceFilePath,
    messagesFilePath
  )
}
