<?php

namespace N98\Magento\Command\Database;

use N98\Magento\Command\TestCase;
use SplFileInfo;

/**
 * @see \N98\Magento\Command\Database\DumpCommand
 */
class DumpCommandTest extends TestCase
{
    public function testExecute()
    {
        $input = [
            'command'        => 'db:dump',
            '--add-time'     => true,
            '--only-command' => true,
            '--force'        => true,
            '--compression'  => 'gz',
        ];

        $this->assertDisplayContains($input, 'mysqldump');
        $this->assertDisplayContains($input, '.sql');
        $this->assertDisplayContains($input, ".sql.gz");
    }

    /**
     * @see filenamePatterns
     */
    public function provideFilenamePatternsAndOptions()
    {
        return [
            # testAddTimeAutogenerated
            ['/^.*[0-9]{4}-[0-9]{2}-[0-9]{2}_[0-9]{6}\.sql$/', []],
            # testAddTimePrefixAutogenerated
            ['/^[0-9]{4}-[0-9]{2}-[0-9]{2}_[0-9]{6}.*\.sql$/', ['--add-time' => 'prefix']],
            # testAddTimeFilenameSpecified
            ['/^.*[0-9]{4}-[0-9]{2}-[0-9]{2}_[0-9]{6}\.sql.gz$/', ['--compression' => 'gzip']],
            # testAddTimeFilenameSpecified
            ['/^foo_[0-9]{4}-[0-9]{2}-[0-9]{2}_[0-9]{6}\.sql$/', ['filename' => 'foo.sql']],
            # testAddTimePrefixFilenameSpecified
            ['/^[0-9]{4}-[0-9]{2}-[0-9]{2}_[0-9]{6}_foo\.sql$/', ['filename' => 'foo.sql', '--add-time' => 'prefix']],
            # testAddTimeOffFilenameSpecified
            ['/^foo.sql$/', ['filename' => 'foo.sql', '--add-time' => 'no']],
            # testAddTimeFilenameSpecifiedRelative
            ['/^..\/foo_[0-9]{4}-[0-9]{2}-[0-9]{2}_[0-9]{6}\.sql$/', ['filename' => '../foo.sql']],
        ];
    }

    /**
     * @test
     * @dataProvider provideFilenamePatternsAndOptions
     *
     * @param string $regex
     * @param array $options
     * @return void
     */
    public function filenamePatterns($regex, array $options)
    {
        $mandatory = [
            'command'               => 'db:dump',
            '--force'               => true,
            '--print-only-filename' => true,
            '--dry-run'             => null,
        ];

        $defaults = [
            '--add-time' => true,
        ];

        $this->assertDisplayRegExp($mandatory + $options + $defaults, $regex);
    }

    public function testWithStripOption()
    {
        $input = [
            'command'        => 'db:dump',
            '--add-time'     => true,
            '--only-command' => true,
            '--force'        => true,
            '--strip'        => '@development not_existing_table_1',
            '--compression'  => 'gzip',
        ];

        $dbConfig = $this->getDatabaseConnection()->getConfig();
        $db = $dbConfig['dbname'];

        $this->assertDisplayRegExp($input, "/--ignore-table=$db.customer_entity/");
        $this->assertDisplayRegExp($input, "/--ignore-table=$db.customer_address_entity/");
        $this->assertDisplayRegExp($input, "/--ignore-table=$db.sales_order/");
        $this->assertDisplayRegExp($input, "/--ignore-table=$db.sales_order_item/");
        $this->assertDisplayRegExp($input, "/--ignore-table=$db.sales_order_item/");
        $this->assertDisplayNotContains($input, "not_existing_table_1");
        $this->assertDisplayContains($input, ".sql.gz");

        /**
         * Uncompressed
         */
        $this->assertDisplayNotContains(
            [
                'command'        => 'db:dump',
                '--add-time'     => true,
                '--only-command' => true,
                '--force'        => true,
                '--strip'        => '@development',
            ],
            ".sql.gz"
        );
    }

    public function testWithExcludeOption()
    {
        $input = [
            'command'        => 'db:dump',
            '--add-time'     => true,
            '--only-command' => true,
            '--force'        => true,
            '--exclude'      => 'core_config_data',
            '--compression'  => 'gzip',
        ];

        $dbConfig = $this->getDatabaseConnection()->getConfig();
        $db = $dbConfig['dbname'];

        $this->assertDisplayRegExp($input, "/--ignore-table=$db.core_config_data/");
        $this->assertDisplayNotContains($input, "not_existing_table_1");
        $this->assertDisplayContains($input, ".sql.gz");

        /**
         * Uncompressed
         */
        $this->assertDisplayNotContains(
            [
                'command'        => 'db:dump',
                '--add-time'     => true,
                '--only-command' => true,
                '--force'        => true,
                '--exclude'      => 'core_config_data',
            ],
            ".sql.gz"
        );
    }

    public function testWithIncludeOptions()
    {
        $input = [
            'command'        => 'db:dump',
            '--add-time'     => true,
            '--only-command' => true,
            '--force'        => true,
            '--include'      => 'core_config_data',
            '--compression'  => 'gzip',
        ];

        $dbConfig = $this->getDatabaseConnection()->getConfig();
        $db = $dbConfig['dbname'];

        $this->assertDisplayNotContains($input, "--ignore-table=$db.core_config_data");
        $this->assertDisplayContains($input, "--ignore-table=$db.catalog_product_entity");
        $this->assertDisplayContains($input, ".sql.gz");
    }

    public function testWithIncludeExcludeOptions()
    {
        $input = [
            'command'        => 'db:dump',
            '--add-time'     => true,
            '--only-command' => true,
            '--force'        => true,
            '--include'      => '@development',
        ];

        $dbConfig = $this->getDatabaseConnection()->getConfig();
        $db = $dbConfig['dbname'];

        $this->assertDisplayNotContains($input, "--ignore-table=$db.customer_entity");
        $this->assertDisplayNotContains($input, "--ignore-table=$db.customer_address_entity");
        $this->assertDisplayNotContains($input, "--ignore-table=$db.admin_user");
        $this->assertDisplayContains($input, "--ignore-table=$db.catalog_product_entity");
        $this->assertDisplayNotContains($input, ".sql.gz");

        $input['--exclude'] = '@admin';
        $this->assertDisplayNotContains($input, "--ignore-table=$db.customer_entity");
        $this->assertDisplayNotContains($input, "--ignore-table=$db.customer_address_entity");
        $this->assertDisplayContains($input, "--ignore-table=$db.admin_user");
        $this->assertDisplayContains($input, "--ignore-table=$db.catalog_product_entity");
    }

    /**
     * @test
     * @link https://github.com/netz98/n98-magerun2/issues/200
     */
    public function realDump()
    {
        $dumpFile = new SplFileInfo($this->getTestMagentoRoot() . '/test-dump.sql');
        if ($dumpFile->isReadable()) {
            $this->assertTrue(unlink($dumpFile), 'Precondition to unlink that the file does not exists');
        }
        $this->assertFalse(is_readable($dumpFile), 'Precondition that the file does not exists');

        $this->assertExecute(
            [
                'command'  => 'db:dump',
                '--strip'  => '@stripped',
                'filename' => $dumpFile,
            ]
        );

        $this->assertTrue($dumpFile->isReadable(), 'File was created');
        // dump should be larger than quarter a megabyte
        $this->assertGreaterThan(250000, $dumpFile->getSize());
    }
}
