<?php

namespace N98\Magento\Command\Database;

use InvalidArgumentException;
use N98\Magento\Command\TestCase;
use SplFileInfo;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Tester\CommandTester;

/**
 * @see \N98\Magento\Command\Database\DumpCommand
 */
class DumpCommandTest extends TestCase
{
    /**
     * @return Command
     */
    protected function getCommand()
    {
        $dumpCommand = new DumpCommand();
        if (!$dumpCommand->isEnabled()) {
            self::markTestSkipped('DumpCommand is not enabled.');
        }

        $application = $this->getApplication();
        $application->add($dumpCommand);
        $command = $this->getApplication()->find('db:dump');

        return $command;
    }

    public function testExecute()
    {
        $command = $this->getCommand();

        $commandTester = new CommandTester($command);
        $commandTester->execute(
            ['command'        => $command->getName(), '--add-time'     => true, '--only-command' => true, '--force'        => true, '--compression'  => 'gz']
        );

        self::assertMatchesRegularExpression('/mysqldump/', $commandTester->getDisplay());
        self::assertMatchesRegularExpression('/\.sql/', $commandTester->getDisplay());
        self::assertStringContainsString('.sql.gz', $commandTester->getDisplay());
    }

    /**
     * @see filenamePatterns
     */
    public function provideFilenamePatternsAndOptions()
    {
        return [
            # testAddTimeAutogenerated
            ['/^.*[0-9]{4}-[0-9]{2}-[0-9]{2}_[0-9]{6}\.sql$/', []],
            # testAddTimePrefixAutogenerated
            ['/^[0-9]{4}-[0-9]{2}-[0-9]{2}_[0-9]{6}.*\.sql$/', ['--add-time' => 'prefix']],
            # testAddTimeFilenameSpecified
            ['/^.*[0-9]{4}-[0-9]{2}-[0-9]{2}_[0-9]{6}\.sql.gz$/', ['--compression' => 'gzip']],
            # testAddTimeFilenameSpecified
            ['/^foo_[0-9]{4}-[0-9]{2}-[0-9]{2}_[0-9]{6}\.sql$/', ['filename' => 'foo.sql']],
            # testAddTimePrefixFilenameSpecified
            ['/^[0-9]{4}-[0-9]{2}-[0-9]{2}_[0-9]{6}_foo\.sql$/', ['filename' => 'foo.sql', '--add-time' => 'prefix']],
            # testAddTimeOffFilenameSpecified
            ['/^foo.sql$/', ['filename' => 'foo.sql', '--add-time' => 'no']],
            # testAddTimeFilenameSpecifiedRelative
            ['/^..\/foo_[0-9]{4}-[0-9]{2}-[0-9]{2}_[0-9]{6}\.sql$/', ['filename' => '../foo.sql']],
        ];
    }

    /**
     * @test
     * @dataProvider provideFilenamePatternsAndOptions
     *
     * @param string $regex
     * @param array $options
     * @return void
     */
    public function filenamePatterns($regex, array $options)
    {
        $command = $this->getCommand();

        $mandatory = ['command'               => $command->getName(), '--force'               => true, '--print-only-filename' => true, '--dry-run'             => null];

        $defaults = ['--add-time' => true];

        $this->assertDisplayRegExp($mandatory + $options + $defaults, $regex);
    }

    public function testWithStripOption()
    {
        $command = $this->getCommand();

        $this->getApplication()->initMagento();

        $commandTester = new CommandTester($command);
        $commandTester->execute(
            ['command'        => $command->getName(), '--add-time'     => true, '--only-command' => true, '--force'        => true, '--strip'        => '@development not_existing_table_1', '--compression'  => 'gzip']
        );

        $dbConfig = $this->getDatabaseConnection()->getConfig();
        $db = $dbConfig['dbname'];

        self::assertMatchesRegularExpression("/--ignore-table=$db.customer_entity/", $commandTester->getDisplay());
        self::assertMatchesRegularExpression("/--ignore-table=$db.customer_address_entity/", $commandTester->getDisplay());
        self::assertMatchesRegularExpression("/--ignore-table=$db.sales_flat_order/", $commandTester->getDisplay());
        self::assertMatchesRegularExpression("/--ignore-table=$db.sales_flat_order_item/", $commandTester->getDisplay());
        self::assertMatchesRegularExpression("/--ignore-table=$db.sales_flat_order_item/", $commandTester->getDisplay());
        self::assertStringNotContainsString('not_existing_table_1', $commandTester->getDisplay());
        self::assertStringContainsString('.sql.gz', $commandTester->getDisplay());

        /**
         * Uncompressed
         */
        $commandTester = new CommandTester($command);
        $commandTester->execute(
            ['command'        => $command->getName(), '--add-time'     => true, '--only-command' => true, '--force'        => true, '--strip'        => '@development']
        );
        self::assertStringNotContainsString('.sql.gz', $commandTester->getDisplay());
    }

    public function testWithIncludeExcludeOptions()
    {
        $command = $this->getCommand();
        $this->getApplication()->initMagento();
        $dbConfig = $this->getDatabaseConnection()->getConfig();
        $db = $dbConfig['dbname'];

        /**
         * Exclude
         */
        $commandTester = new CommandTester($command);
        $commandTester->execute(
            ['command'        => $command->getName(), '--add-time'     => true, '--only-command' => true, '--force'        => true, '--exclude'      => 'core_config_data', '--compression'  => 'gzip']
        );
        self::assertMatchesRegularExpression("/--ignore-table=$db\.core_config_data/", $commandTester->getDisplay());

        /**
         * Include
         */
        $commandTester = new CommandTester($command);
        $commandTester->execute(
            ['command'        => $command->getName(), '--add-time'     => true, '--only-command' => true, '--force'        => true, '--include'      => 'core_config_data', '--compression'  => 'gzip']
        );
        self::assertDoesNotMatchRegularExpression("/--ignore-table=$db\.core_config_data/", $commandTester->getDisplay());
        self::assertMatchesRegularExpression("/--ignore-table=$db\.catalog_product_entity/", $commandTester->getDisplay());
    }

    public function testIncludeExcludeMutualExclusivity()
    {
        /**
         * Both include and exclude.
         */
        $command = $this->getCommand();
        $this->getApplication()->initMagento();
        $this->expectException(InvalidArgumentException::class);

        $commandTester = new CommandTester($command);
        $commandTester->execute(
            ['command'        => $command->getName(), '--add-time'     => true, '--only-command' => true, '--force'        => true, '--include'      => 'core_config_data', '--exclude'      => 'catalog_product_entity', '--compression'  => 'gzip']
        );
    }

    /**
     * @test
     * @link https://github.com/netz98/n98-magerun2/issues/200
     */
    public function realDump()
    {
        $dumpFile = new SplFileInfo($this->getTestMagentoRoot() . '/test-dump.sql');
        if ($dumpFile->isReadable()) {
            self::assertTrue(unlink($dumpFile), 'Precondition to unlink that the file does not exists');
        }
        self::assertIsNotReadable((string)$dumpFile, 'Precondition that the file does not exists');

        $command = $this->getCommand();
        $commandTester = new CommandTester($command);
        $commandTester->execute(
            ['command'  => $command->getName(), '--strip'  => '@stripped', 'filename' => $dumpFile]
        );

        self::assertTrue($dumpFile->isReadable(), 'File was created');
        // dump should be larger than quarter a megabyte
        self::assertGreaterThan(250000, $dumpFile->getSize());
    }
}
