/*
 * Copyright (c) 2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.networknt.schema;

import static org.junit.jupiter.api.Assertions.*;

import org.junit.jupiter.api.Test;

import com.networknt.schema.regex.ECMAScriptRegularExpressionFactory;
import com.networknt.schema.regex.JDKRegularExpressionFactory;

/**
 * Test for SchemaValidatorsConfig.
 */
@SuppressWarnings("deprecation")
class SchemaValidatorsConfigTest {
    @Test
    void defaultEcma262Validator() {
        SchemaValidatorsConfig config = new SchemaValidatorsConfig();
        assertSame(JDKRegularExpressionFactory.getInstance(), config.getRegularExpressionFactory());
        assertFalse(config.isEcma262Validator());
    }

    @Test
    void setEcma262Validator() {
        SchemaValidatorsConfig config = new SchemaValidatorsConfig();

        config.setEcma262Validator(true);
        assertSame(ECMAScriptRegularExpressionFactory.getInstance(), config.getRegularExpressionFactory());
        assertTrue(config.isEcma262Validator());

        config.setEcma262Validator(false);
        assertSame(JDKRegularExpressionFactory.getInstance(), config.getRegularExpressionFactory());
        assertFalse(config.isEcma262Validator());
    }

    @Test
    void constructorPathType() {
        SchemaValidatorsConfig config = new SchemaValidatorsConfig();
        assertEquals(PathType.LEGACY, config.getPathType());
    }

    @Test
    void builderPathType() {
        SchemaValidatorsConfig config = SchemaValidatorsConfig.builder().build();
        assertEquals(PathType.JSON_POINTER, config.getPathType());
    }

    @Test
    void constructorCustomMessageSupported() {
        SchemaValidatorsConfig config = new SchemaValidatorsConfig();
        assertEquals(true, config.isCustomMessageSupported());
    }

    @Test
    void builderCustomMessageSupported() {
        SchemaValidatorsConfig config = SchemaValidatorsConfig.builder().build();
        assertEquals(false, config.isCustomMessageSupported());
    }

    @Test
    void constructorHandleNullableField() {
        SchemaValidatorsConfig config = new SchemaValidatorsConfig();
        assertEquals(true, config.isHandleNullableField());
    }

    @Test
    void builderHandleNullableField() {
        SchemaValidatorsConfig config = SchemaValidatorsConfig.builder().build();
        assertEquals(false, config.isHandleNullableField());
    }
    
    @Test
    void constructorMutable() {
        SchemaValidatorsConfig config = new SchemaValidatorsConfig();
        assertDoesNotThrow(() -> config.setFailFast(true));
    }

    @Test
    void builderImmutable() {
        SchemaValidatorsConfig config = SchemaValidatorsConfig.builder().build();
        assertThrows(UnsupportedOperationException.class, () -> config.setFailFast(true));
    }
}
