<?php
namespace Neos\Flow\Mvc\Routing;

/*
 * This file is part of the Neos.Flow package.
 *
 * (c) Contributors of the Neos Project - www.neos.io
 *
 * This package is Open Source Software. For the full copyright and license
 * information, please view the LICENSE file which was distributed with this
 * source code.
 */

use Neos\Flow\Mvc\Routing\Dto\MatchResult;
use Neos\Flow\Mvc\Routing\Dto\ResolveResult;

/**
 * Contract for all Route Parts.
 *
 * !!! Warning: If you write your own RoutePart handler which does some queries to the
 * persistence layer, be aware that *permission checks* are not yet done, i.e. you
 * get back *all* objects, not just the objects visible to the current role.
 */
interface RoutePartInterface
{
    /**
     * Sets name of the Route Part.
     *
     * @param string $partName
     * @return void
     */
    public function setName($partName);

    /**
     * Returns name of the Route Part.
     *
     * @return string
     */
    public function getName();

    /**
     * Returns true if a value is set for this Route Part, otherwise false.
     *
     * @return boolean
     */
    public function hasValue();

    /**
     * Returns value of the Route Part. Before match() is called this returns NULL.
     *
     * @return mixed
     */
    public function getValue();

    /**
     * Returns true if a default value is set for this Route Part, otherwise false.
     *
     * @return boolean
     */
    public function hasDefaultValue();

    /**
     * Sets default value of the Route Part.
     *
     * @param mixed $defaultValue
     * @return void
     */
    public function setDefaultValue($defaultValue);

    /**
     * Gets default value of the Route Part.
     *
     * @return mixed $defaultValue
     */
    public function getDefaultValue();

    /**
     * Specifies whether this Route part is optional.
     *
     * @param boolean $isOptional true: this Route part is optional. false: this Route part is required.
     * @return void
     */
    public function setOptional($isOptional);

    /**
     * @return boolean true if this Route part is optional, otherwise false.
     * @see setOptional()
     */
    public function isOptional();

    /**
     * Specifies whether this Route part should be converted to lower case when resolved.
     *
     * @param boolean $lowerCase true: this Route part is converted to lower case. false: this Route part is not altered.
     * @return void
     */
    public function setLowerCase($lowerCase);

    /**
     * Getter for $this->lowerCase.
     *
     * @return boolean true if this Route part will be converted to lower case, otherwise false.
     * @see setLowerCase()
     */
    public function isLowerCase();

    /**
     * Defines options for this Route Part.
     * Options can be used to enrich a route part with parameters or settings like case sensivitity.
     *
     * @param array $options
     * @return void
     */
    public function setOptions(array $options);

    /**
     * @return array options of this Route Part.
     */
    public function getOptions();

    /**
     * Checks whether this Route Part corresponds to the given $routePath.
     * This method does not only check if the Route Part matches. It can also
     * shorten the $routePath by the matching substring when matching is successful.
     * This is why $routePath has to be passed by reference.
     *
     * @param string &$routePath The request path to be matched - without query parameters, host and fragment.
     * @return bool|MatchResult true or an instance of MatchResult if Route Part matched $routePath, otherwise false.
     */
    public function match(&$routePath);

    /**
     * Checks whether this Route Part corresponds to the given $routeValues.
     * This method does not only check if the Route Part matches. It also
     * removes resolved elements from $routeValues-Array.
     * This is why $routeValues has to be passed by reference.
     *
     * @param array &$routeValues An array with key/value pairs to be resolved by Dynamic Route Parts.
     * @return bool|ResolveResult true or an instance of ResolveResult if Route Part can resolve one or more $routeValues elements, otherwise false.
     */
    public function resolve(array &$routeValues);
}
