/* Copyright © 2017-2020 ABBYY Production LLC

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

	http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
--------------------------------------------------------------------------------------------------------------*/

#pragma hdrstop

#include <common.h>
#pragma hdrstop

#include <NeoML/Random.h>

namespace NeoML {

static const double MaxRandomValue = 4294967296.;
static const int randCount = 12;
static const double sigmaCoeff = sqrt( 12. / randCount ) / MaxRandomValue;

CRandom::CRandom( unsigned int seed ) :
	carry( initialCarry ),
	lagPosition( 0 )
{
	Reset( seed );
}

void CRandom::Reset( unsigned int seed )
{
	carry = initialCarry;
	lagPosition = 0;
	// Calculate the seed hash to ensure better spread with small differences in seed
	unsigned int seedHash = (seed + 536870911) * 2147483647;

	// Initialize lag
	for( int i = 0; i < lagSize; ++i ) {
		lag[i] = stdLag[i] ^ seedHash;
	}
}

unsigned int CRandom::Next()
{
	++lagPosition;
	lagPosition &= lagSize - 1;

	unsigned __int64 step = static_cast<unsigned __int64>( multiplier ) * lag[lagPosition] + carry;
	carry = static_cast<unsigned int>( step >> 32 );

	lag[lagPosition] = 0xFFFFFFFF - ( unsigned int )( step );

	return lag[lagPosition];
}

double CRandom::Uniform( double min, double max )
{
	return min + ( max - min ) * Next() / MaxRandomValue;
}

int CRandom::UniformInt( int min, int max )
{
	unsigned __int64 distance = max - min;
	++distance;

	return ( int )( min + ( unsigned int )( ( static_cast<unsigned __int64>( distance ) * Next() ) >> 32 ) );
}

double CRandom::Normal( double mean, double sigma )
{
	double sum = -MaxRandomValue * randCount / 2;
	for( int i = 0; i < randCount; ++i ) {
		sum += Next();
	}

	sum *= sigma * sigmaCoeff;
	sum += mean;

	return sum;
}

const unsigned int CRandom::stdLag[] = {
	0x46979343, 0x8962D2FA, 0x273A24A4, 0x6118A180, 0x35475ABB, 0xE4613C7C, 0xEF32F4DB, 0x727BF199,
	0xC64DD8DC, 0x15C1FED6, 0x36DF5FF8, 0x02B905CA, 0x8C549443, 0x67187F67, 0x8EA04D0F, 0x0A77FFB1,
	0xB9CBB19A, 0x1DB42D6A, 0x556B750E, 0x322A0154, 0x8BEAE115, 0x0CF8423D, 0x8C95A1A0, 0xCCDD98B3,
	0xBB584DF3, 0x3FA1AFAB, 0xE34B6073, 0x6A513227, 0xD61403AE, 0x7E8F251C, 0xD750011E, 0xEC9AC42B,
	0xC8E83027, 0xA7BC16FB, 0x99132CDF, 0xBA0CB120, 0x60950867, 0x507893EF, 0x81F44001, 0xEE743047,
	0x7E24BD61, 0xA56F09E2, 0x9AF73F23, 0xC30E5110, 0x8BDD6CD0, 0xCD337E9A, 0x32AFF6AC, 0xBB0D0F41,
	0x05275CA1, 0x37764E7D, 0x43719ADA, 0x03A4D703, 0x873C3114, 0x11F3170F, 0x1A498953, 0x795BB7BB,
	0x9A409733, 0x9B43C78D, 0xEA8DF8EA, 0x052262C6, 0x33A691E8, 0xE9C26A99, 0x9C27AC47, 0x4D1FB35F,
	0x6B8953FE, 0x8075834D, 0xC650BE04, 0xC5A83195, 0xF6238D3C, 0x25566A6F, 0x60F9F939, 0x0C35E511,
	0x0E2C87B3, 0x8AE5D606, 0xEC9101A2, 0x0F173C0E, 0xBCC749AF, 0x4EB1B990, 0xEA2896A2, 0xB6609DE6,
	0x63867578, 0x6F52F7D9, 0x99ACA0E6, 0xAC90C66B, 0x6C89669D, 0x409A2A27, 0x263AC5E2, 0xE0641CBB,
	0x3F8254DF, 0x6387188F, 0x72AF5E35, 0x8D31ED80, 0x93E89C1E, 0xAB820E1E, 0x438D7C0B, 0xF3F90A7D,
	0xACF82F12, 0x58E1DA8C, 0x6E542315, 0x5D7EBF91, 0xC4E6B381, 0x14EFC4FA, 0x91743B0F, 0x3E230614,
	0xC66AA2D6, 0xDCBBD570, 0xBF3EBD97, 0x40C90EC6, 0x4A8F7A58, 0x3A5C8CDF, 0xDA9ED3C5, 0xC0B36AAE,
	0x4DE90B86, 0x43F13B53, 0x0F085C6C, 0x2837C847, 0xDB7BAFD0, 0xDD11C0BE, 0xDD6CD4FB, 0x9929A527,
	0x2B0DB610, 0x61C67BA8, 0x636E6E38, 0xCE0A5D71, 0x667A6AE6, 0xA7BCE8C2, 0xD954B015, 0xEE1C8963,
	0x85240A71, 0xF958F654, 0x81F4B989, 0x80292695, 0x84835123, 0x27BF7C2B, 0xE6187409, 0xBAEB5141,
	0x2C6BCF43, 0x9982328D, 0xC692EE56, 0x45160A05, 0xDFFC8621, 0xC6DF10B3, 0x7308598E, 0x82BBA407,
	0x57864395, 0x4E98B190, 0xB0245AD4, 0x078DC365, 0xEFE42E91, 0x60DD6C8B, 0x980A4601, 0x1BBA3649,
	0x339F5C45, 0xE772E1CF, 0xC19AE74F, 0x38A46290, 0xE2DD39E7, 0x3751DBE1, 0x845A275A, 0xD85B32C9,
	0x03E57F89, 0x38BCF1D1, 0xC866740E, 0xC5E398EE, 0xD3D6B091, 0xC3144360, 0xDA6F07DE, 0xCD89AC04,
	0x1A7C6916, 0x0A80BE7B, 0x896AB41D, 0x5727E8F9, 0xA3F765E9, 0xC2BBB857, 0x77020A37, 0x0EB64CF2,
	0x78E4DD7B, 0x48D1B1CE, 0x749F298B, 0x2E7F92E9, 0x88BDDCB4, 0x643C8279, 0x0B7F2C7A, 0xF3823D37,
	0x1DC63CA5, 0xF995304F, 0x1E8C6972, 0xF96FADBC, 0x9C46B047, 0x2363526E, 0x07812A5C, 0xEC708B97,
	0x3C8B76FD, 0xE810F9D6, 0x6A1047EF, 0x84968715, 0xD3D30D0A, 0x3E7FC7C1, 0xAC1D9EFC, 0xC61DA313,
	0x33BA27ED, 0xE9EA0FA1, 0x5F06C497, 0x83D662B0, 0xE443623B, 0x36C7B601, 0xF3ED7084, 0x0B95EC49,
	0x54A69F8A, 0x251CFC5E, 0x78795754, 0x223893A3, 0x51793B61, 0x7C1B1333, 0xCF0B9C30, 0x701786B2,
	0xBBD350FB, 0x1F76AB32, 0x0F08BE32, 0x8830E9C6, 0x5E8E5F94, 0x77BD8792, 0x86A8A651, 0xED2AE2BA,
	0x793C6F0A, 0x204A4055, 0xA2A9CC71, 0xF7945186, 0x7A10B461, 0xBF495C8C, 0xDDAE9C86, 0xCD926FFB,
	0x63E2E9A7, 0x9491291E, 0x215BB146, 0xE784F614, 0x7B2AF6FF, 0xB48EAFAC, 0x15A582DD, 0x7380DA16,
	0xB1F190BC, 0x06331BCD, 0xC83F693B, 0xC1E31349, 0x909B341C, 0x13F9EB21, 0xE16AEC72, 0x89AE2C3E,
	0x8599EEA5, 0x274DE7AD, 0xB979937E, 0x66915899, 0x04560738, 0x38D93C45, 0x4B24DC46, 0x56783B43,
	0x7FD2835B, 0x4159F6BD, 0x4E6CDE64, 0x3ADDF625, 0xB65EBB8E, 0x6E16573B, 0x6D41E7B3, 0x94514BC0,
	0xA4EF61F2, 0xB706AFBD, 0xB9BB3EA2, 0x1E6C892E, 0xF2A6AF50, 0x23818CBE, 0xD94E39CA, 0xD8D2FDE7,
	0xB216A613, 0xD0DB0F0B, 0x63D48363, 0xC48ADD86, 0x6F12BA5C, 0x8F5DAE12, 0x73A93974, 0x3574FA79,
	0xFE38E85E, 0xBF74B532, 0xA8ECCBE6, 0xDE8B62DD, 0x986550CC, 0x7D04C932, 0xF08E91BB, 0x697EB90A,
	0x43E88EBB, 0xEB2780D7, 0xBA0A84CF, 0x7D555155, 0xF0265F66, 0x5CFB1710, 0x77DC6E08, 0x789554DC,
	0x1A5539EB, 0xC202F3CC, 0x4AE49B0F, 0xF698EBD1, 0x1E588C27, 0x00D3C30C, 0xCFB4AD99, 0xA9C99A32,
	0x5605782B, 0xF23857FE, 0x30CD9E6C, 0xD315558F, 0xBB7011CA, 0x2C38052B, 0xBFA7779B, 0xE76C0CE0,
	0x982D7A41, 0x204BA046, 0x14828DF4, 0xC7F046C9, 0x04994068, 0x95076F1D, 0xFE2A37EF, 0x3A594F3E,
	0xDBDE6235, 0xC3342AEE, 0x2FD985A6, 0x69A7A9A6, 0xC59B9B33, 0xB2982361, 0x45335D16, 0x5696FB95,
	0x96A1284F, 0x7D93447A, 0x563EC3FE, 0x35A24FF1, 0x228F960B, 0xC294D089, 0x084EABCD, 0x2E3E6828,
	0x4160E268, 0xD31D53A1, 0x2F579790, 0x4879397A, 0x0BE73C37, 0x48DFAE9E, 0x30B21796, 0x1EB7AEA8,
	0x58925DCB, 0xF9715A29, 0x89671974, 0x8B8A2EEB, 0xEC6A3E98, 0x3DDF0D69, 0x7FD745C2, 0xFF6039C3,
	0x9EE0F7D5, 0x2FEA75F8, 0x0E9756C7, 0xBECEB46F, 0x0B6693DF, 0xD8732220, 0x8291A555, 0x73C2022B,
	0x5675320B, 0xA79BAD02, 0xD2CE503F, 0x204D4F6A, 0xAA755C6E, 0x9F3FB296, 0x5A942190, 0x150DFC6B,
	0xBCDBBDA0, 0x5E73AD9A, 0x31F8F4EB, 0xD1BC6A06, 0x66A86082, 0xA0507E69, 0x60D834E2, 0x1F03D6EB,
	0xC03E1D54, 0x660669C4, 0x1D9C6DD3, 0xE7842A0F, 0x51322847, 0x39C6BC99, 0x50A603F0, 0x89F04C4B,
	0x26E881FD, 0x34CDFD77, 0xB6B1C082, 0x4CDB9E5E, 0x0F795DA5, 0x52306768, 0x4A74CEFD, 0x5A67F01C,
	0x84C4942F, 0x61A9344E, 0xE925B21F, 0x2A847FD6, 0xBF17D7FD, 0x0A42F52F, 0x1CC8D8A8, 0x45E10C4F,
	0x656592FD, 0x94F7DF5F, 0xCB721C34, 0xD117139C, 0x593C5973, 0x43DD2549, 0x1C5723A9, 0x0A1DCFC8,
	0x33B5A712, 0xEA15EC0D, 0x00F19D35, 0x62CA4221, 0x95C6FB3E, 0xCAA94BD8, 0x6E452582, 0x5C924269,
	0x97FE3047, 0xD7FD5481, 0xE3F7626F, 0x40FA3E96, 0x15CED1E1, 0xFD37A46E, 0x24138A43, 0xE95B4C0C,
	0x87577661, 0x052E168D, 0x332DD8C8, 0x34A9C616, 0x3EBAE509, 0x76B0FF25, 0xC5CF5236, 0x9B9486FA,
	0x035EBF9C, 0x11A8F19A, 0x8500E8AD, 0x9A7209AE, 0xED1115CA, 0x6FCE7388, 0x761AC09A, 0x895C5262,
	0x7C51197E, 0xB9067568, 0xB302E596, 0x65072AB5, 0x49F7FE70, 0x0E4CCE4D, 0x9ECD00AE, 0x245B9C72,
	0xAA33F381, 0x16620908, 0xFCBB5418, 0x4FE286D0, 0x5E8E24D4, 0x3A0F54FA, 0x2AA0C4DE, 0xB3127A27,
	0x39AE6E44, 0x3315F853, 0xB724677B, 0x3499EE43, 0x4F9F4FDC, 0x03ED99D4, 0xFD9AEB5A, 0x3F903B2E,
	0x216B97F1, 0xC3A21CF5, 0x151625AC, 0x7EBF9E4D, 0x98DEB241, 0x27B4160B, 0x0CD509BC, 0x9961457B,
	0xBFE2FC3E, 0xA05B168A, 0x31484B37, 0x5CDB3DD8, 0xBF3F463D, 0x7E910A64, 0xF4231DD1, 0x88AC4A91,
	0x3944E9DB, 0xCFF28D14, 0x6A8CD6B4, 0xD4CD111B, 0xF0B0077C, 0x4C6A6305, 0x32262644, 0xFE22B85F,
	0x0E0F7D56, 0xF87999DF, 0x1ED20247, 0x3283341D, 0x3173A085, 0x618E611D, 0xFF8BAE7A, 0x6571305F,
	0x3224054D, 0xF050FD80, 0xCB425FF7, 0x6C89470A, 0x05B3420D, 0x666FC06D, 0x43384087, 0x86CF556F,
	0xB7447E70, 0x85A8DE64, 0x705ABFF1, 0xD601E0D6, 0x5055B994, 0xF64BE93E, 0x2E84804B, 0xBEDCFC7A,
	0x975F9103, 0x5FBFF5FB, 0xD80756BC, 0x468AE9E4, 0x651B7DA0, 0xAFFBCCAD, 0x6560D0AD, 0x62FD5FE3,
	0xD78C87FB, 0x59A7F89D, 0x3FD91D0B, 0x03331933, 0x2C8DFBFC, 0xA4C40330, 0x4BF8094E, 0x7A6D9C7A,
	0xE66969BB, 0xC1ED87B3, 0xE8978611, 0x79ED905E, 0xB222F0B1, 0xDCA1492B, 0x881635E9, 0x58167F11,
	0x00CF0C7D, 0x9C9DCC99, 0x861C438F, 0x81EA4DEC, 0xBFB3FF54, 0xF8C8524F, 0xDAE5532C, 0xDC4203FC,
	0x2B01A8E0, 0xFD23CF5B, 0x828DFB8F, 0xF0682F3F, 0x42EB88B8, 0xC34A38FD, 0xE0748607, 0x08BB0F33,
	0xD73BF420, 0x39344E5C, 0x2F20E903, 0xF075F12C, 0x2DE9C041, 0x9473E2B5, 0x84DC441A, 0xDD982DB7,
	0xBD2076B8, 0xBDB689B9, 0x0AFAF815, 0x4D5CAC77, 0x910DEF34, 0x6C9ED613, 0x408C66F8, 0x1CC3301D,
	0x646C83EF, 0xC8192783, 0xC29FE442, 0x81C33672, 0xCDFF02A4, 0x3CA821E5, 0xADBD2F8C, 0x1FF2E3F2,
	0xA3F7F50B, 0x65239309, 0xBD684201, 0x9455D53C, 0xCFB84225, 0xC0A1CBE4, 0x9EDD19CD, 0xE475652A,
	0x7E9B5F49, 0x9940AA0E, 0xF9F103FE, 0x8A6FD6FB, 0xFDE8A083, 0x564D7820, 0x71E3FB35, 0x619DC455,
	0x4C351ED2, 0x343F00B4, 0x4D94E895, 0xD2B1ED20, 0xA6EB44EB, 0x35BCD76A, 0xEF35CCE9, 0x78F94334,
	0xAA975478, 0xE244EC88, 0x5EC9F67B, 0x13A47731, 0xA7958E8C, 0xD504DA1C, 0x4D2E28AD, 0x5B135F1C,
	0xC4F65C35, 0xC57CAE0D, 0x50AA73BA, 0xA9671079, 0x32444C4D, 0x463A5DF0, 0x85488557, 0x471EB85E,
	0x97A572ED, 0xC9A945C6, 0x91401306, 0xECA21705, 0x405FC7B2, 0xA971240A, 0x638A0A53, 0xF512F8C6,
	0xDE3B80E0, 0x590842A4, 0x8C3E035B, 0x4B2E5A1A, 0x6A51FE0D, 0xC7040D62, 0x755E6884, 0xEFEE34E9,
	0x46E1751C, 0x87F18DA7, 0xAF482D2F, 0xB6654D04, 0xD9A1CBF0, 0x3A4D6840, 0x0242FD95, 0x112EA8B9,
	0x44204CB9, 0xF6FA63CE, 0x35783B7C, 0x9D4DCF95, 0xFFC10ECF, 0x7CB495BD, 0x3A1051A7, 0x6C638758,
	0x94009B8B, 0x0A0C35D4, 0x5D1DE984, 0x5292B58C, 0x51195656, 0x53F5FA1B, 0xF247F96B, 0xC7DBEF1F,
	0x3DF67237, 0xA9370DCC, 0x424772D6, 0x34543AFF, 0x610F3742, 0xF7781C44, 0xF4A9A63C, 0xF68CBF10,
	0x4142217E, 0x2C09FA18, 0xC5273555, 0x4B635F28, 0x7C70875E, 0x7EBE1A3A, 0x316E0AA7, 0x2ECE214A,
	0x2A5F9E99, 0xE216CB1B, 0xBF566C8D, 0x7BC1CB50, 0xA847C3F8, 0xD295D833, 0x5FCC1F6D, 0x4D71F573,
	0xDC0BF895, 0x3359D68D, 0x6AFDD952, 0xCAC76C5E, 0x28D30BDF, 0x264B124E, 0xD2B213C5, 0xFCD347C8,
	0x8664F857, 0x0F333171, 0xCE210AFD, 0x3670A224, 0xCF42A799, 0xC303916A, 0x6E987C4A, 0x95ECEF11,
	0x4E0C3EB3, 0xEDE69D33, 0xDB9F5FA8, 0x376C58D2, 0x3DB32CA7, 0xFEE9E382, 0xA211829C, 0x93DDFAC0,
	0xF304BB0E, 0xE6F44B62, 0xE687467C, 0xD5E0E4A4, 0x2E2D9823, 0xD3BD4287, 0x27870718, 0x55B67E62,
	0xFB60C770, 0x7D0B7409, 0xFCB97E9F, 0xF67FE371, 0x0AD24FC3, 0x3236970A, 0xF5885B17, 0x849143A2,
	0xAE665CF4, 0xF4045590, 0xFA82B70D, 0x9CCB53AA, 0xF32088B5, 0x37BE2BE1, 0x89278FAF, 0x6DB085E2,
	0x55CF382A, 0x5A030BE5, 0x590B15AD, 0x276A378F, 0xB031476F, 0xE4EDA0ED, 0x7DB257FB, 0x0801DBB6,
	0xB5D835CD, 0xC363ED65, 0x7A3F2CE7, 0x42093951, 0x64ECB91B, 0x87B94883, 0x51B09195, 0xD0445F17,
	0x9F33607A, 0x35B02261, 0x3499A48A, 0x33BA1E03, 0xCCCF1158, 0x7CCC9BA7, 0xD5BB89AA, 0xF58E5169,
	0x971A7B46, 0x5782FAE3, 0x271C1DA9, 0x8C772EB4, 0x410CA920, 0xCF772DA7, 0xD86E4B03, 0xA5BB333B,
	0x05C97C47, 0x489906B3, 0x2680F85D, 0x05643078, 0xB6ECD2CE, 0x4C81128E, 0x1937B1CF, 0x87E5D77B,
	0x99E1069E, 0x8ECA7809, 0x6E522AF2, 0x966C1046, 0xD9684117, 0xA2DF2613, 0x2F5C1961, 0xEA5BE70C,
	0x078A844D, 0x058E79E4, 0x2F48F99A, 0xA9CEFA7C, 0xACB1BD67, 0x001335A0, 0xA322B71F, 0x06053902,
	0xA162FD8E, 0x6B22692E, 0x2621675B, 0x79FDD9F4, 0xF90DF805, 0x9E5C24DE, 0xB66CC1C1, 0xAF018733,
	0x8B076011, 0x90C8380A, 0x3F714F20, 0x1F086EBC, 0x415D18A7, 0xCBFBA020, 0xF15F683A, 0x2E62CE2B,
	0x719C8FD5, 0xFBDC4C0B, 0xFC225A3D, 0x9A68C5A8, 0x09CD5D40, 0xB87CCD56, 0x1B71B3DE, 0x577BC9B6,
	0x8DB6927D, 0x2332D0D7, 0x45941828, 0x7FC2847D, 0xED81F2AC, 0x8A0EF7DF, 0x058868CE, 0x77CDAE6C,
	0xBA969336, 0x30A1AD10, 0x227A9BB4, 0x75BC3486, 0xEAFFD6AB, 0xA3D64538, 0x1435F582, 0xD8EC3E2B,
	0xD43A646F, 0x7543C65F, 0xBC93724C, 0x8EE35D79, 0x252EA9C3, 0x5AEFA9C4, 0x916D99D5, 0xAD9B04D7,
	0xCF310714, 0x1348DB11, 0x603C6501, 0x248A5B3C, 0x70E63831, 0x88F9FED7, 0x353C32BD, 0xD3F92FAC,
	0x024A135B, 0xBFDAB90B, 0x6744B52D, 0x662D5754, 0xF748AA00, 0xE558013D, 0x3BB12239, 0x3F5F54D9,
	0x0408DFFE, 0x7B8AFAFE, 0x909AFF55, 0x42A72EBC, 0x97665859, 0x61FDF1D4, 0x1F87A1F6, 0xB64DD465,
	0xFB9D4D90, 0xD25DC9CC, 0x82ACD04A, 0xA8DB07C8, 0x8377DC01, 0x661074EA, 0xB177D772, 0xF4DCEB86,
	0x7ACED500, 0xA455F1C3, 0xC54B7078, 0x81A1E944, 0x0359EC89, 0xB0EABD45, 0x1905CCED, 0xE4C87A65,
	0xE0654836, 0x9586FCD2, 0xD8B9A2F9, 0x4CBA97F3, 0x4E3C9121, 0x4B4B4FDF, 0xF5818E66, 0x26331C0B,
	0x509ED43D, 0x622EF64E, 0xF664FEF0, 0xBC652D1F, 0x97DB19A8, 0xE9B5AE9B, 0xDBCC463A, 0xE4CAB217,
	0x685677B1, 0x475AF6EF, 0x73BDC446, 0x0A953637, 0xFEA4091C, 0xC4266699, 0x90DC48EE, 0xFBACD662,
	0x98DB1634, 0xD64CDE6D, 0x890847E9, 0xD4DD50BC, 0x47B365D0, 0xCD684204, 0xBADD0CCC, 0x4FC820BA,
	0x88648A38, 0x4D38FB37, 0xA96EDE42, 0x379BA912, 0x242041E3, 0x232CE9EA, 0x87CD4BF6, 0xA68A9260,
	0x6CA81D30, 0x0504524E, 0x4E6BDC66, 0x7A294A7A, 0xC6184FE5, 0x372D95C8, 0xB59BFEAB, 0x7927583F,
	0x57B394E0, 0xDD2ABCB4, 0xC45A68CE, 0xEE2E0F23, 0xFEA4F136, 0x80F5DE01, 0xC137682D, 0x71D3F1EA,
	0x827C5056, 0x2C024DF7, 0x42073E93, 0x5BF3DB38, 0xBDF363F1, 0x111D0E6A, 0x92029301, 0x4DF6FB39,
	0xDE3790CF, 0x3A516107, 0x904B09C7, 0x4A750A12, 0x9A80388C, 0x023F6C4C, 0x167B03A4, 0xB6D56A23,
	0x9FF689AB, 0xB34A898F, 0xCA83C801, 0xC114B742, 0xE4BCC2D4, 0x7452E35D, 0xD9A9FCA3, 0xB5848366,
	0x8F3F2887, 0xCD0B0C95, 0xB8D585B7, 0x9B12FCC9, 0x69706E2F, 0xCA6CD552, 0xD18C6225, 0x6C2AC038,
	0x743A7EEB, 0xC23D4233, 0x95362049, 0x054F7246, 0xA4863288, 0x9C986413, 0xE9E9B66B, 0x90C93D6C,
	0xF30CCAD4, 0x9E0793E7, 0x1B135B86, 0x87F6140A, 0xC90D5C88, 0x3B2AE2F4, 0x175620F5, 0xC29E53BA,
	0xAEE380AD, 0x2320563D, 0xF7E1AC55, 0x93DFB63D, 0x5EA9BAFE, 0x12589E40, 0x71701A58, 0x89437D78,
	0x76B43DEC, 0xE3A846B4, 0x5799C04F, 0xF5182283, 0x5F2578F9, 0x124846FC, 0xC647716D, 0xA6542A2B,
	0xB453E039, 0xD2DCEA3F, 0x7C39EEF8, 0xE638A8B5, 0xDDE76ABC, 0xA99EFD22, 0x1A927665, 0xD4A9300A,
	0xB4CAEFBD, 0xE3BD2203, 0x177F6D7B, 0x6DC1B01A, 0xD1F53D8E, 0x33BFB10B, 0x16E60A24, 0x16212369,
	0x29DB13B9, 0xF496D8AC, 0x50CAA205, 0x580F9561, 0x9484D37C, 0x4721192B, 0xBBC8DE32, 0x759D0167,
	0x51144D57, 0xBA0ED63B, 0xBF990127, 0x39A579EC, 0x18A3E946, 0x75CC938B, 0xBC306060, 0x676B78B1
};

}
