# Класс CDnn

<!-- TOC -->

- [Класс CDnn](#класс-cdnn)
    - [Конструктор](#конструктор)
    - [Работа со слоями](#работа-со-слоями)
        - [Добавление слоя](#добавление-слоя)
        - [Проверка на наличие слоя](#проверка-на-наличие-слоя)
        - [Получение слоя](#получение-слоя)
        - [Удаление слоя](#удаление-слоя)
        - [Получение списка всех слоёв](#получение-списка-всех-слоёв)
        - [Удаление всех слоёв](#удаление-всех-слоёв)
    - [Обучение сети](#обучение-сети)
        - [Инициализация весов](#инициализация-весов)
        - [Метод оптимизации](#метод-оптимизации)
        - [Запуск с обучением](#запуск-с-обучением)
    - [Запуск без обучения](#запуск-без-обучения)
    - [Сериализация](#сериализация)
    - [Логирование](#логирование)

<!-- /TOC -->

Класс, реализующий нейронную сеть. Нейронная сеть представляет собой направленный граф из слоёв, в котором каждый слой, начиная от входных (Source) слоев и заканчивая выходными (Sink), производит вычисления над блобами данных.

## Конструктор

```c++
CDnn::CDnn( IMathEngine& mathEngine, CRandom& random )
```

Через конструктор задаётся используемый [вычислительный движок](MathEngine.md) и используемый генератор случайных чисел. Обратите внимание, что обязательно использовать один и тот же вычислительный движок для сети и всех её слоёв.

## Работа со слоями

### Добавление слоя

```c++
void AddLayer( CBaseLayer& layer );
```

Добавление слоя к нейронной сети. Если этот слой уже добавлен в какую-либо сеть, или в текущей сети уже есть слой с таким именем, то будет выброшено исключение.

### Проверка на наличие слоя

```c++
virtual bool HasLayer( const char* name ) const;
```

Проверка на наличие слоя с таким именем в сети.

### Получение слоя

```c++
virtual CPtr<CBaseLayer> GetLayer( const char* name );
virtual CPtr<const CBaseLayer> GetLayer( const char* name ) const;
```

Получение указателя на конкретный слой. Если слой с таким именем отсутствует, выбрасывается исключение.

### Удаление слоя

```c++
void DeleteLayer( const char* name );
void DeleteLayer( CBaseLayer& layer );
```

Удаляет указанный слой из сети. Если он не принадлежит данной сети, будет выброшено исключение.

### Получение списка всех слоёв

```c++
virtual void GetLayerList( CArray<const char*>& layerList ) const;
```

Получение списка имён имеющихся слоёв в сети.

### Удаление всех слоёв

```c++
void CDnn::DeleteAllLayers();
```

Удаление всех слоев сети.

## Обучение сети

### Инициализация весов

```c++
CPtr<CDnnInitializer> GetInitializer();
void SetInitializer( const CPtr<CDnnInitializer>& initializer );
```

Класс, инициализирующий веса слоёв перед первой итерацией обучения. По умолчанию используется инициализация Xavier.

### Метод оптимизации

```c++
CDnnSolver* GetSolver();
const CDnnSolver* GetSolver() const;
void SetSolver( CDnnSolver* solver );
```

Класс, реализующий метод оптимизации обучаемых параметров слоёв.

### Запуск с обучением

```c++
void RunAndLearnOnce();
```

Запуск одной итерации с обучением. После этого метода можно извлекать данные из блобов [выходных слоёв](IOLayers/SinkLayer.md).

## Запуск без обучения

```c++
void RunOnce();
```

Произвести вычисления сети. После этого метода можно извлекать данные из блобов [выходных слоёв](IOLayers/SinkLayer.md).

## Сериализация

```c++
void Serialize( CArchive& archive );
```

Сериализация сети. Если архив был открыт для записи, то произойдет запись сети в архив. Если архив был открыт для чтения, то из сети будут удалены все слои, а затем сеть будет заново считана из архива.

```c++
void SerializeCheckpoint( CArchive& archive );
```

Сериализация сети и всей дополнительной информации, необходимой для возобновления обучения (истории градиентов и т.д.).

При загрузке создает новый объект метода оптимизации, указатель на который можно получить методом `CDnn::GetSolver`.

## Логирование

```c++
CTextStream* GetLog();
void SetLog( CTextStream* newLog );
```

Получение/установка используемого лога.

```c++
int GetLogFrequency() const;
void SetLogFrequency( int logFrequency );
```

Получение/установка частоты логирования. По умолчанию, логируется каждая 100-я итерация `RunOnce` или `RunAndLearnOnce`.
