#include "blowfish.h"

BLOWFISH::BLOWFISH(std::string hexKey)
{
	IvSet = false;
	if (hexKey.length() % 2 != 0)
		throw 2;

	bytee* key = new bytee[hexKey.length() / 2];//

	for (int i = 0; i < hexKey.length() / 2; i++)
	{
		key[i] = hex2dec(hexKey[i * 2]) * 16 + hex2dec(hexKey[i * 2 + 1]);
	}

	SetupKey(key, hexKey.length() / 2);
}

int BLOWFISH::hex2dec(char hex)
{
	if ('a' <= hex && hex <= 'f')
		return 10 + (hex - 'a');
	if ('A' <= hex && hex <= 'F')
		return 10 + (hex - 'A');
	return hex - '0';
}

BLOWFISH::BLOWFISH(bytee* cipherKey, int keyLength)
{
	IvSet = false;
	SetupKey(cipherKey, keyLength);
}

bytee* BLOWFISH::Encrypt_ECB(bytee* data, int length, int* newlength)
{
	return Crypt_ECB(data, length, newlength, &BLOWFISH::Encrypt_Block, false);
}

bytee* BLOWFISH::Decrypt_ECB(bytee* data, int length, int* newlength)
{
	return Crypt_ECB(data, length, newlength, &BLOWFISH::Decrypt_Block, true);
}

bytee* BLOWFISH::Encrypt_CBC(bytee* data, int length, int* newlength)
{
	return Crypt_CBC(data, length, newlength, &BLOWFISH::Encrypt_Block, false);
}

bytee* BLOWFISH::Decrypt_CBC(bytee* data, int length, int* newlength)
{
	return Crypt_CBC(data, length, newlength, &BLOWFISH::Decrypt_Block, true);
}

std::string BLOWFISH::Encrypt_CBC(std::string data)
{
	bytee* binaryData = new bytee[data.length()];
	for (int i = 0; i < data.length(); i++)
		binaryData[i] = data[i];
	int newlen = 0;
	bytee* result = Encrypt_CBC(binaryData, data.length(), &newlen);
	std::string encoded = "";
	for (int i = 0; i < newlen; i++)
		encoded += byteToHex(result[i]);
	delete[] result;
	delete[] binaryData;
	return encoded;
}

std::string BLOWFISH::Decrypt_CBC(std::string data)
{
	if (data.length() % 2 != 0)
		throw 2;

	bytee* binaryData = new bytee[data.length() / 2];//

	for (int i = 0; i < data.length() / 2; i++)
	{
		binaryData[i] = hex2dec(data[i * 2]) * 16 + hex2dec(data[i * 2 + 1]);
	}
	int len = 0;
	bytee* cryptresult = Decrypt_CBC(binaryData, data.length() / 2, &len);
	std::string result = "";
	for (int i = 0; i < len; i++)
		result += cryptresult[i];
	delete[] cryptresult;
	return result;
}

std::string BLOWFISH::byteToHex(unsigned char x)
{
	char hex[17] = "0123456789ABCDEF";
	std::string result = "";
	result += hex[x / 16];
	result += hex[x % 16];
	return result;
}

bytee* BLOWFISH::padData(bytee* data, int length, int* paddedLength, bool decrypt, bool IvSpace = false)
{
	int offset = 0;
	int dataoffset = 0;
	if (decrypt)
	{
		if (length % 8 != 0) throw 8;
		*paddedLength = length;
	}
	else
	{
		//if IvSpace, leave a blank block at the front
		*paddedLength = 8 + (length % 8 == 0 ? length : length + 8 - (length % 8)) + (IvSpace ? 8 : 0); //pad the data to a multiple of 8 plus one block
		if (IvSpace)
			offset = 8;
	}

	//fill the new array with the data
	bytee* pData = new bytee[*paddedLength];
	for (int i = 0; i < length; i++)
		pData[offset + i] = data[i + dataoffset];

	//add the padding character to the end
	for (int i = length + offset; i < *paddedLength; i++)
		pData[i] = (pData[length - 1 + offset] ^ 0xCC); //fill the padding with a character that is different from the last character in the plaintext, so we can find the end later

	return pData;
}

int BLOWFISH::findPaddingEnd(bytee* data, int length)
{
	int i = length;
	while (data[i - 1] == data[length - 1]) //find the first character from the back that isnt the same as the last character
	{
		i--;
	}
	return i; //retun the length without the padding
}

bytee* BLOWFISH::Crypt_ECB(bytee* data, int length, int* newlength, void (BLOWFISH::* CryptBlock)(bytee*, int), bool decrypt)
{
	bytee* pData;
	pData = padData(data, length, newlength, decrypt); //this loads the IV from the front of the ciphertext

	for (int i = 0; i < *newlength; i += 8) //run the encryption
	{
		(this->*CryptBlock)(pData, i);
	}

	if (decrypt) //if we are decrypting, we have to find where the data ends.
	{
		*newlength = findPaddingEnd(pData, *newlength);
	}
	return pData;
}

bytee* BLOWFISH::Crypt_CBC(bytee* data, int length, int* newlength, void (BLOWFISH::* CryptBlock)(bytee*, int), bool decrypt)
{
	bytee* pData;
	if (!decrypt && !IvSet)
		SetRandomIV();
	IvSet = false; // don't re-use an IV
	pData = padData(data, length, newlength, decrypt, true);

	if (!decrypt)
	{
		//padData leaves an 8 bytee block at the beggining so we can save the IV
		for (int i = 0; i < 8; i++)
			pData[i] = IV[i];
	}
	else
	{
		for (int i = 0; i < 8; i++)
			IV[i] = pData[i];
	}
	bytee nextIV[8];
	for (int i = 8; i < *newlength; i += 8) //run the encryption
	{
		if (!decrypt)
		{
			for (int k = 0; k < 8; k++)
				pData[k + i] ^= pData[k + i - 8]; //the previous block contains the initialization vector
		}
		else
		{
			for (int k = 0; k < 8; k++)
				nextIV[k] = pData[k + i];
		}
		(this->*CryptBlock)(pData, i);

		if (decrypt)
		{
			for (int k = 0; k < 8; k++)
			{
				pData[i + k] ^= IV[k];
				IV[k] = nextIV[k];
			}
		}
	}

	if (decrypt) //if we are decrypting, we have to find where the data ends, and remove the IV
	{
		*newlength = findPaddingEnd(pData, *newlength) - 8;
		bytee* noIV = new bytee[*newlength];
		for (int i = 0; i < *newlength; i++)
			noIV[i] = pData[i + 8];
		delete[] pData;
		pData = noIV;
	}
	return pData;
}

void BLOWFISH::SetRandomIV()
{
#ifdef _WIN32
	//WIN32 CSPRNG thanks to: http://www.tomhandal.com/DevBlog/2010/03/17/cryptographically-random-bytees-in-microsoft-windows/
	HCRYPTPROV hCryptCtx = NULL;
	CryptAcquireContext(&hCryptCtx, NULL, MS_DEF_PROV, PROV_RSA_FULL, CRYPT_VERIFYCONTEXT);
	CryptGenRandom(hCryptCtx, 8, IV);
	CryptReleaseContext(hCryptCtx, 0);
#else
	std::ifstream devRand("/dev/urandom", std::ios::in | std::ios::binary);
	if (!devRand.read((char*)&IV, 8))
	{
		throw 1;
	}
#endif
	IvSet = true;
}

void BLOWFISH::SetIV(bytee* newIV)
{
	IvSet = true;
	for (int i = 0; i < 8; i++)
		IV[i] = newIV[i];
}

bytee* BLOWFISH::GetIV()
{
	bytee* returnIV = new bytee[8];
	for (int i = 0; i < 8; i++)
		returnIV[i] = IV[i];
	return returnIV;
}

void BLOWFISH::Encrypt_Block(bytee* block, int offset)
{
	setblock(block, offset);
	encipher();
	getblock(block, offset);
}

void BLOWFISH::Decrypt_Block(bytee* block, int offset)
{
	setblock(block, offset);
	decipher();
	getblock(block, offset);
}

void BLOWFISH::setblock(bytee* block, int offset)
{
	//TODO: CHECK ENDIANNESS
	xr_par = 0; xl_par = 0;
	for (int i = 0; i < 4; i++)
	{
		xl_par = (xl_par << 8) + block[offset + i];
		xr_par = (xr_par << 8) + block[4 + offset + i];
	}
}

void BLOWFISH::getblock(bytee* block, int offset)
{
	//TODO: CHECK ENDIANNESS
	unsigned int xl = xl_par;
	unsigned int xr = xr_par;
	for (int i = 3; i >= 0; i--)
	{
		block[i + offset] = xl % 256;
		block[i + 4 + offset] = xr % 256;
		xr = xr >> 8;
		xl = xl >> 8;
	}
}

void BLOWFISH::SetupKey(bytee* cipherKey, int length)
{
	if (length > 56)
	{
		throw 56;
	}

	bytee* key = new bytee[length]; //

	for (int i = 0; i < length; i++)
		key[i] = cipherKey[i];

	int j = 0;
	unsigned int d;
	for (int i = 0; i < 18; i++)
	{
		d = (((key[j % length] * 256 + key[(j + 1) % length]) * 256 + key[(j + 2) % length]) * 256 + key[(j + 3) % length]);
		p[i] ^= d;
		j = (j + 4) % length;
	}

	xl_par = 0;
	xr_par = 0;

	for (int i = 0; i < 18; i += 2)
	{
		encipher();
		p[i] = xl_par;
		p[i + 1] = xr_par;
	}

	for (int i = 0; i < 256; i += 2)
	{
		encipher();
		s0[i] = xl_par;
		s0[i + 1] = xr_par;
	}

	for (int i = 0; i < 256; i += 2)
	{
		encipher();
		s1[i] = xl_par;
		s1[i + 1] = xr_par;
	}

	for (int i = 0; i < 256; i += 2)
	{
		encipher();
		s2[i] = xl_par;
		s2[i + 1] = xr_par;
	}

	for (int i = 0; i < 256; i += 2)
	{
		encipher();
		s3[i] = xl_par;
		s3[i + 1] = xr_par;
	}

}

void BLOWFISH::encipher()
{
	xl_par ^= p[0];
	for (int i = 0; i < ROUNDS; i += 2)
	{
		xr_par = round(xr_par, xl_par, i + 1);
		xl_par = round(xl_par, xr_par, i + 2);
	}
	xr_par ^= p[ROUNDS + 1];

	unsigned int swap = xl_par;
	xl_par = xr_par;
	xr_par = swap;
}

void BLOWFISH::decipher()
{
	xl_par ^= p[ROUNDS + 1];
	for (int i = ROUNDS; i > 0; i -= 2)
	{
		xr_par = round(xr_par, xl_par, i);
		xl_par = round(xl_par, xr_par, i - 1);
	}
	xr_par ^= p[0];

	unsigned int swap = xl_par;
	xl_par = xr_par;
	xr_par = swap;
}

unsigned int BLOWFISH::round(unsigned int a, unsigned int b, unsigned int n)
{
	//TODO: CHECK ENDIANNESS
	unsigned int x1 = (s0[(b >> 24) % 256] + s1[(b >> 16) % 256]) ^ s2[(b >> 8) % 256];
	unsigned int x2 = x1 + s3[b % 256];
	unsigned int x3 = x2 ^ p[n];
	return x3 ^ a;
}

unsigned int BLOWFISH::s0[] = {
				0xd1310ba6,0x98dfb5fd,0x211d72db,0xd01adfb8,0xb8e1afed,0x6a267e96,
				0xba7c9046,0xf12c7f91,0x24a19948,0xa7916cf8,0x18014fe2,0x958eb416,
				0x636920d8,0x91574e61,0xb458fea4,0xe3933d7e,0x1d95748f,0x928eb658,
				0x918bc5f8,0x92154aee,0x9b54a41d,0xd25a59b6,0x9c305f31,0x2a4f6014,
				0x5fd1b024,0x286085f0,0xda417918,0xb8da78ef,0x9e79dcb0,0x603a180e,
				0x6c9e0e8b,0xb01e8a3e,0xd71577c2,0xbd314b28,0x98a4ffda,0x65605c60,
				0xf65525e3,0xba55bc94,0x67489862,0x63e81440,0x65ca396a,0x2bbb10b6,
				0xb4cc5c34,0x2141e8ce,0xb15486af,0x9c72e994,0xa7ee1412,0x636fbc2a,
				0x2ba9c55d,0x941831f6,0xde5c3e16,0x9b87931e,0xbfd6ba34,0x6c24cf5c,
				0x9a325382,0x28958678,0x4b8e3898,0x6b4cc9af,0xd4bfe81b,0x66282194,
				0x61d809cc,0xfb21a992,0x487cfd60,0x6dec8032,0x118455fd,0xf98575b2,
				0xdc262302,0xfb651b88,0x23893e82,0xd396fdc6,0x1f6d6114,0x93e34231,
				0x2e0b4482,0xb4842004,0x69c8f04a,0x9e1f9b5e,0x21c66842,0xf6e96c9a,
				0x670c9c62,0xccd388f0,0x6a51a0d2,0xd8542f68,0x960fa728,0xcc5133a4,
				0x6eef0b6c,0x237a3be4,0xba3bf050,0x9efb2a98,0xb1f1651d,0x49af0176,
				0x66ca593e,0x92430e88,0x9cee8611,0x456f9fb4,0x9d84a5c4,0x4b8b5ebe,
				0xf06f75d8,0x95c12074,0x401a449f,0x66c16bb6,0x4ed3bb62,0x463f7706,
				0x2bfedf72,0x429b023d,0x47d0d724,0xd00a1248,0xdb0fead4,0x49f1c09b,
				0x175372c1,0x90991b7b,0x25d479d8,0xf6e8def8,0xf3fe501a,0xb6794c3b,
				0x976ce0bd,0x14c006ba,0xd1a94fb6,0x409f60c4,0x6e5c9ec2,0x296a2464,
				0x68fb6faf,0x4e6c53b6,0x2339b2eb,0x4b52ec6f,0x6db4511f,0x9a70952c,
				0xdc814544,0xbf5ebd01,0xbee3d004,0xde334afd,0x6604f808,0x292e4cc4,
				0xd0cba858,0x45c8740f,0xd20b5f31,0xb9d3fbdb,0x6579c0bd,0x2a60320a,
				0xd6a100c6,0x402c7271,0x6794f5fe,0xfb1fa3cc,0x9ea5e9f8,0xda7222f8,
				0x4c7516df,0xfd616b16,0x2f501ec8,0xbd0552bc,0x423db5fa,0xfd238760,
				0x63317b48,0x4e00df82,0x9e5c57cc,0xda6f8ca0,0x2a87562e,0xdf1769db,
				0x5f42a8f6,0x287e11c4,0xbc6732c6,0x9c4f5574,0x695b27b0,0xccca58c8,
				0xf111a35d,0xb8f011a0,0x20fa3d98,0xfd2183b8,0x4ab4b56c,0x2ee1d35b,
				0x9a53e471,0xb6f84566,0xd28e49bc,0x4bfb9790,0xf1ee4fda,0xb4cb7e34,
				0x62fb1342,0xdee4c6e8,0x1120cada,0x46774c02,0xd07e9efe,0x2bf11fb4,
				0x95dbda4d,0xbe909198,0xfbbd8e72,0x6b935fa0,0xd08ed1d0,0xbb4725e0,
				0x9e3c5b2f,0x9e7594b8,0x9116e2fb,0x4f122b64,0x9888b812,0x900df01c,
				0x4fa5fea0,0x688b431c,0xd1c11192,0xa7a8c1ad,0x24f4f218,0xbe0e1778,
				0xfa752dfe,0x9b021fa2,0xf5a0cc0f,0xb56f74e8,0x28fdf3d6,0xde89e291,
				0xb4a84fe0,0xfd13e0b8,0x9cc43b82,0xd2ada8d1,0x265fa266,0x90957706,
				0x93cc7314,0x211a1478,0xf6ad2066,0x97b5fa86,0xd75442f6,0xfb9d35cf,
				0xfbcdaf0c,0x9a7e89a0,0xd6411bd4,0xbe1e7e41,0x10250e2d,0x2071a75e,
				0x226800cc,0x67b8e0af,0x2464369b,0xf009b91e,0x6563911d,0x69dfa6bb,
				0x98c14381,0xd95a537f,0x2075fba2,0x12e5b9c6,0x93260376,0x6295cfa1,
				0x21c81968,0x4e734a42,0xa7472dca,0x9b14a94a,0x2b510052,0x9a532916,
				0xd60f573f,0xbc9bc6e4,0x2b60a476,0x91e67400,0x18ba6fb6,0x671be91f,
				0x4f96ec6b,0x2a0ee916,0xb6636522,0xf7b9f9b6,0x1134052e,0x5f855664,
				0x63b025fd,0xb99f8fa2,0x18ba4791,0x6e85076a };

unsigned int BLOWFISH::s1[] = {
				0x4b7a70e1,0xb5a72944,0xdb75092e,0xd4192624,0xbd6ea6b0,0x49a7df7d,
				0x9cee60b8,0x9fedb266,0xb4bb8c72,0x699a1711,0x6664526c,0xd2b19ee2,
				0x293602a6,0x95094c21,0xb0591340,0xe3183a3e,0x4f54989a,0x6b429d66,
				0x6b8fe4d6,0x99f73fd6,0xb1d29c08,0x11e830f6,0x4d2d38e6,0xf0255dc2,
				0x4cee2086,0x9470eb26,0x6382e9c6,0x121ecc5e,0x19686a7f,0x4ebaeb41,
				0x4c971814,0x6b6a70a2,0x687f3584,0x62a0e286,0xb79c5306,0xba500738,
				0x4e07841c,0x9fdeae5c,0x9e7d44ec,0x67164fb8,0xb03ada38,0xf0500c0d,
				0xf01c1f04,0x1200a711,0xbe0cf51a,0x4cb574b2,0x25837a58,0xdc0921bd,
				0xd19113f1,0x9ca92116,0x94324774,0x22f54702,0x4ae5e582,0x47c2dadc,
				0xd8b57634,0x9af3eea8,0xb9446146,0x1fd0030e,0xb4c8c73e,0xb4751e42,
				0x4f38cd91,0x4bea0e2f,0x4280cca2,0x283ea732,0x4e548a78,0x4f6db908,
				0x6e320d04,0xf60a04bf,0x2cb81290,0x24977c71,0x6679b072,0xbcaf89af,
				0xde9a771f,0xd9930810,0xa78bae12,0xdccf34fe,0x6512721f,0x2e6b7124,
				0x601aeee6,0x9f84cd88,0x9a584718,0x9408da18,0xbc9f9bcc,0xf94b7d8c,
				0xb47aec3a,0xdb851dfa,0x63094366,0xd464c3d2,0x111c1848,0x4215d908,
				0xee433a78,0x24c2ba16,0x22a14d44,0x2a65c452,0x60940002,0x233ae4ee,
				0x91d1189e,0x20314e56,0x91fd77d6,0x6f11199b,0x1435564f,0xd7a3c76b,
				0x4c11183b,0x6924a501,0x4f8fe6ed,0x97f1fbfa,0x9ebbc4fc,0x2e153c6e,
				0x96e34570,0xfae96fb2,0x960e5e0a,0x6a3e2bc4,0x971fe71c,0x4e3d06fa,
				0x2965dcb1,0x99e71d0f,0x903e89d6,0x6266c826,0x2e4cc978,0x9c10a76a,
				0xd6150eba,0x94e2ea78,0xb5b43c54,0x2e0a2de3,0x4ff74ea8,0x461d2a7d,
				0x2939260f,0x29c27960,0x6223a708,0xf71312b6,0xfbadfe6e,0xffd31f66,
				0xf3bc4596,0xb67bc884,0xb17f37d2,0x118c1128,0xd332eeef,0xbe6c5bb6,
				0x65582186,0x68bc9802,0xfecea50f,0xdb2f953b,0x2aef7dad,0x6b6e2f84,
				0x2521b628,0x29076170,0xb4ee4776,0x619f1510,0x23cca830,0xfb61bd96,
				0x1334fe1e,0xba0363cf,0xb5735c90,0x4c70a231,0x5f9e9e0b,0xdbbbde14,
				0xfecc86bc,0x60622ca8,0x9cbc5cbc,0xb2f3846e,0x648b1eaf,0x29bdf0ca,
				0xb02369b1,0x655fdc50,0x40685a32,0x4c2bc4b4,0x419ee9d6,0xd021b8f8,
				0x9b540b11,0x975fa091,0x95f7997e,0x623d7da8,0xf837889a,0x97e32d78,
				0x21ed935f,0x26681282,0x1e358821,0xd7e61fd6,0x96dedfa2,0x9858ba91,
				0x67f584a6,0x2b227264,0x9b83c311,0x2fd24696,0xeea70aeb,0x632e3054,
				0x9fd948e4,0x6dbc3128,0x68eb4fef,0x44c611ea,0xfe28ed62,0xfe7c3c74,
				0x6d4a14d1,0xf864b7e4,0x42105d14,0x203e13e0,0x45eee2b6,0xa7bbbcea,
				0xdb6c4f16,0xffdb4fd0,0xd742e342,0x116fdcb6,0x654f3b1d,0x41cd2106,
				0xd81e799e,0x96854dc8,0xe34b476a,0x4d816250,0xdf62a14f,0x6b8d2646,
				0xb48883a0,0xd1c7b6a4,0x9f1524c4,0x69cb7492,0x47848a0b,0x6692b286,
				0x195ccf00,0xbd19489d,0x2462b174,0x23820e00,0x68428d2a,0x1c55f5ea,
				0x2dade33e,0x233f7062,0x4372f092,0x9d937e42,0xd65fec4f,0x6c223bdb,
				0x9cde3751,0xdbee7460,0x40854fa8,0xde77326e,0xb6078084,0x29f8509e,
				0xf8efd856,0x61d99736,0xb969a7bb,0x5f0c06c2,0x6a04bcb4,0x900bcadc,
				0x9e447a2e,0xd3453484,0xfee56706,0x1e1e9ec1,0xdb73dbd4,0x205588cd,
				0x675fda71,0xf3674340,0x5fc43466,0x913e38d8,0x4d28f89e,0xf16d1120,
				0x253e21e8,0x9fb03d4a,0xf6e394fb,0xdb83adf7 };

unsigned int BLOWFISH::s2[] = {
				0xf935fa68,0x948140f8,0xf64c261c,0x94692934,0x411520f8,0x9602d4f8,
				0xbce36b2e,0xd4a20068,0xd4082472,0x4320e36a,0x43b7d4b8,0x600061af,
				0x2e39f62e,0x97244546,0x24214f74,0xbf8b8840,0x4d95b41d,0x96b591af,
				0x90e3eed4,0x66a02e36,0xbfbc09ec,0x13bd9786,0x9ffd6ee0,0x41cb8504,
				0x96eb27b4,0x65fd3942,0xda2547e6,0xccca0a9a,0x28507826,0x630429e3,
				0x1a2c86da,0xf9b66dfb,0x68dc1462,0xd7486900,0x680ec0a4,0x27a18dee,
				0x4f311ea2,0xf887ad8c,0xb58ce006,0x9ae3d6b6,0xbfde1e7c,0xd3375fec,
				0xde78a391,0x406b2a42,0x20fe9e36,0xd9f385b1,0xfe39d7bc,0x4b124e8b,
				0x2dc9faf8,0x4b6d1856,0x26a36632,0xfae397b2,0x4a6efa74,0xee5b4332,
				0x6841e7f8,0xda7820fb,0xfb0af54e,0xd8fea798,0x454056fd,0xba489528,
				0x65533a3a,0x20838d88,0xfe6ba9b8,0xd096954b,0x65a867bc,0xb1159a58,
				0xdca92964,0x99e1da74,0xb62a4a56,0x4f3125f1,0x6ee37e1c,0x9029317c,
				0xfdf8e802,0x14272f70,0x90cc155c,0x15282ce4,0x95c11548,0xe3c66d22,
				0x48c1133f,0xd70f86dc,0x17f9c9ee,0x41041f0f,0x404779a4,0x6d886e18,
				0x425f51eb,0x5f9bc0d2,0x4fbcc18f,0x41113564,0x257b7834,0x602a9c60,
				0xd118e8a4,0x2f636c1b,0x1e12b4c2,0x12e1329e,0xbf664fd2,0xdad18116,
				0x6b2395e0,0x433e92e2,0x4b240b62,0xfebeb922,0x95b2a20e,0xf6ba0d91,
				0xde720c8c,0x2da2f728,0xd0127846,0x95b794fd,0x647d0862,0xf7ccf5f0,
				0x6449a36f,0x977d48fa,0xd39dfd28,0xf33e8d1e,0x1a476342,0x992e1174,
				0x4a6f6ebc,0xe3f8fd38,0xb812dc60,0xb1ebeef8,0x991be14c,0xdb6e6b0d,
				0xd67b5510,0x6d672c38,0x2765d43b,0xdcd0e804,0xf1290dc8,0xdc0011a4,
				0xb5390f92,0x690fed0b,0x667b911b,0xdedb7d9c,0xb091cf0b,0xd9155ea4,
				0xcc132f88,0x615bad24,0x9b9479bf,0x963bd6eb,0x47392eb4,0xdc115971,
				0x9026e298,0xe32e312d,0x6842ada8,0xd66a2a7b,0x22754ccc,0x982ef11c,
				0x6a124238,0xb79251e8,0x16a1cce6,0x4bfb6350,0x2a6b1018,0x21caedfa,
				0x4d25bee8,0x4fe1c3c1,0x44421651,0x1a121386,0xd90cec6e,0x5fbcea2a,
				0x64af674e,0xda86a85f,0xbebfe988,0x64e4c3fe,0x9dbc8058,0xf0f7c086,
				0x60787bf8,0x6003604d,0xd1fd8346,0xf6381fb0,0x9745ae04,0xd736b4cc,
				0x93426a74,0xf01ebc72,0xb0804188,0x4c005e5f,0x97a057be,0xbde8ae24,
				0x65464291,0xbf582e62,0x4e58e38f,0x4feefda2,0xe374ef38,0x9789bdc2,
				0x6366f9c4,0xd8a78e74,0xb4754f56,0x46b4d9b1,0x9aeb2662,0x9b1eef84,
				0x946a0e71,0x915f95e2,0x466e598e,0x20b45770,0x9c5f5592,0xd902de4c,
				0xb90bfde2,0xcc8205d0,0x21a86248,0x9574a99e,0xb77f19b6,0xf0a9dc01,
				0x662d09a2,0xd4324634,0xf85a1f02,0x19f0be8c,0x4a99a026,0x2d6efe10,
				0x2bc93d1d,0x1ba5a4df,0xb1864f0f,0x2868f161,0xdcb7da84,0x673906fe,
				0xb1e2ce9b,0x4b4d7f52,0x60115e02,0xb70683fa,0xb002b5c4,0x1de6d028,
				0x9af88c28,0x973f8642,0xd3604c06,0x61a806b6,0xf0177a28,0xd0f586e0,
				0x106058bb,0x40dc7d62,0x21e69ed8,0x2338ea64,0x63c2ee94,0xd2c21634,
				0xcccbee56,0x90bcb6de,0xfbb47da2,0xde591d76,0x6f05e401,0x4b7c0188,
				0x49720a3d,0x9c927c24,0x96e3725f,0x924d9db1,0x2fd15cc4,0xd39eb8b4,
				0xf5f45578,0x18b4a5b6,0xd83d7cd4,0x4dad0b44,0x2e50ef5e,0xb161e6f8,
				0xb28514d1,0x6c51133c,0x6f5fc7e8,0x66e14ec4,0x462bcb4e,0xeec6c838,
				0xd79a3234,0x92638212,0x670efa8e,0x406000e0 };

unsigned int BLOWFISH::s3[] = {
				0x4a39ce38,0xd3faf5cf,0xccc27738,0x6f5f2d1b,0x6cb0679e,0x4fa33742,
				0xd3822740,0x99bc9cce,0x5f118e9d,0xbf0f7316,0xd62d1c7e,0xd700c47b,
				0xb78c1b6b,0x21a19046,0xb26eb1be,0x6a366eb4,0x6748bc2f,0xbc946e71,
				0xd6a376d2,0x6549c2c8,0x630118ee,0x468eee7d,0x5f730a1d,0x4cd04dc6,
				0x2939ccdb,0xb9ba4650,0xbc9526e8,0xbe5ee304,0xb1fa5ff0,0x6a25f19a,
				0x63ef8ce2,0x9a86ee22,0xd089c2b8,0x43242ef6,0xb51e03bb,0x9c4fd0a4,
				0x93c061ba,0x9be96a4d,0x9fe51550,0xba645bd6,0x2826a2f1,0xb73a3ae2,
				0x4ba99586,0x115562e1,0xd72fefd4,0xf752f7da,0x4f046f61,0x97fa0a51,
				0x90e4a916,0x97b08602,0x9b09e6ad,0x4a7ee594,0xf990f5fa,0x9e34d798,
				0x2cf0b7d1,0x122b8b52,0x965ffd3a,0x117da67d,0xd1cf3ed6,0x9c7d2d28,
				0x2f94f5cf,0xbd4fb89b,0x6ad6b472,0x6a88f54c,0xf029fd72,0xf019a5e6,
				0x47b0fdfd,0xfd93fa9b,0xf8d3c48d,0x283b57cc,0xf85f6621,0x99132e28,
				0x985f0192,0xfd756056,0xf7960e44,0xf3d35e8c,0x25056ee4,0x98e36dba,
				0x13a16126,0x1564f0bd,0xd3eb9e16,0x4c9057a2,0x97271aec,0xb93a072a,
				0x2a7f6d9b,0x2e6321f6,0xf59c66fb,0x26dcf311,0x9533d928,0xb155fdf6,
				0x13563482,0x9bca3ccc,0x28517712,0xd20ad9f8,0xcccc5168,0xdcad925f,
				0x4de81752,0x4830dc8e,0x4795f862,0x9320f992,0xfa7a90c2,0xfa7e7bce,
				0x6121ce64,0x974fbe32,0xb8b6e37e,0xd3293d46,0x48de5361,0x6413e680,
				0xb2ae0810,0xee6db224,0x69852dfd,0x19072166,0xa79a460a,0x6445c0ee,
				0x686cdecf,0x2c20c8ae,0x6ccef7ee,0x2b588d40,0xdcd2017f,0x6cc4e3cc,
				0xeea26a7e,0x4a591146,0x4e350a44,0xbcb4cee6,0x92efdea8,0xfa6484cc,
				0x9d6612ae,0xbf3c6e38,0xd29be464,0x642f5d9e,0xbec2771b,0xf64e6370,
				0x940e0d8d,0xf75b1358,0xf8721672,0xbf5375fd,0x4040cb08,0x4eb4e2cc,
				0x44d2466a,0x1115af84,0xf1b00428,0x95983a1d,0x16b89fb4,0xde6ea048,
				0x6f3f3b82,0x4520bc82,0x111a1d4b,0x277227f8,0x611560b2,0xf7933fdc,
				0xcc3a792b,0x444525bd,0xb08839e2,0x61ce794b,0x2f32c9b8,0xb01fbfd1,
				0xf01cc87e,0xbcc7d1f6,0xdf0111c4,0xb1e8bbc8,0x2a908741,0xd44fbd9a,
				0xd0dadecb,0x5f0ada38,0x1339c32a,0xd6913668,0x9df9317c,0xf0b12b4f,
				0xf79e59b8,0x43f5cc3a,0x4f5f1911,0x27d9459c,0xbf97222c,0x25e6b42a,
				0x1f91b472,0x9b941526,0xfae59362,0xdeb69ceb,0xd2a86451,0x22bbb8d2,
				0xb6c1075e,0xf3056a0c,0x20d25066,0xdb03a442,0xf0ec6e0e,0x2698da7b,
				0x4c98a0be,0x4278e964,0x9f1f9532,0xf0d392df,0xd3a0342b,0x99714f1e,
				0x2b0a7442,0x4ba3348c,0x5fbe7120,0xd37632d8,0xdf359f8d,0x9b9924fe,
				0xf60b6e38,0x1fe3f11d,0xf54cda54,0x2edad892,0xde6279cf,0xee3e7e6f,
				0x2618b166,0xfd2c1d06,0x948fd2c6,0xf6fb2291,0xf523f358,0xb6327624,
				0x93a83532,0x66cccd02,0xbcf08162,0x6a75ecc6,0x6e163698,0x98d273cc,
				0xde966292,0x91b949d0,0x4c50901b,0x91c65614,0xf6c6c7bd,0x427a140a,
				0x45e1d006,0xd34f7b9a,0xd9bb53fd,0x62a80f00,0xcc25bfe2,0x45bee2f6,
				0x91126906,0xb2040222,0xb6cbcf7c,0xee769c2b,0x63113ec0,0x2640e3d4,
				0x48fdcd60,0x2547adf0,0xba38209c,0xf746ce76,0x97afa1c6,0x20756060,
				0x95cbfe4e,0x9ae88ee8,0x9bbaf9b0,0x4cf9bb7e,0x2948c25c,0x12fb8a8c,
				0x11c36ae4,0xd6ebe1f1,0x90d4f861,0xb65cdea0,0x4f09252d,0xd208e69f,
				0xb74e6132,0xde77e25b,0x678fdfe4,0x4fd372e6 };

unsigned int BLOWFISH::p[] = {
				0x243f6a88,0x95a308d4,0x23198a2e,0x13707344,0xb4093822,0x299f31d0,
				0x182efa98,0xb44e6c81,0x452821e6,0x48d01378,0xbe5466cf,0x44e90c6c,
				0xd0fd29b8,0xd97c50ee,0x4f845fb6,0xb5470918,0x92165fd1,0x9979fb1b,

				//240 Aeeitional hex digits of PI for increased rounds versions
				//Starting at ((256 * 4 + 18) * 8)th hex digit of PI
				0xb83fdb02, 0x2002397a, 0x6ec6fb5b, 0x11cfd4ee, 0x4cbf5ed2, 0xe33fe582,
				0x4ee3e824, 0x2d152af0, 0xf718c970, 0x69bd9820, 0x2e3a9d62, 0xf7a529ba,
				0x99e1248d, 0x4bf88656, 0x5f114d0e, 0xbc4cee16, 0x134d8a31, 0x20e47882,
				0xf9ae8fbd, 0xf3bcdc1f, 0x6da51e52, 0x6db2bae2, 0x11f86e7a, 0x6d9c68a1,
				0x2708b4d1, 0x293cbc0c, 0xb03c86f8, 0xb8ad2c2f, 0x10424eeb, 0xdfdb452d,
				0x99cc71b4, 0x5f9c7f92, 0x9f0622bc, 0x6d8a08b2, 0x934d2132, 0x6884ca82,
				0xf3bbcbe3, 0x97864ffa, 0x2cbc6e3d, 0xde535ad2, 0x4f0fd608, 0xd6b8e14f,
				0x6eb4388e, 0x975014a6, 0x656665f8, 0xb64a43e4, 0xba383d02, 0xb2e41071,
				0x9eb2986f, 0x909e0ca4, 0x2f7a7778, 0x2c126030, 0x95088718, 0xd4e7d1bd,
				0x406511ce, 0x9392fd8a, 0xba36d12b, 0xb4c8c9d0, 0x994fb0b8, 0x24f96818,
				0xf9a53998, 0xb0a178c6, 0x2684a81e, 0x9ae972f6, 0xb8425eb6, 0x9a29d486,
				0x651bd711, 0xbf32c181, 0x5f145506, 0xdc815f3e, 0x48424eda, 0xb796ee36,
				0xb0498f04, 0x667deede, 0x13fd0bc4, 0xd497733d, 0x6316a892, 0x40a88b4c,
				0x9604440a, 0xdeeb893a, 0x9725b82b, 0x1e1ef69d, 0x402a5c8e, 0xf7b84def,
				0x6a31b096, 0xd9ebf88d, 0x612d788e, 0x9e4002ee, 0x97e02af6, 0xd358a1cc,
				0x12e8d7af, 0xdf9fb0e8, 0x990e942a, 0x4a7c1bca, 0xd611a7af, 0x9df796f1,
				0x421cc994, 0x1174a8a8, 0xfd22162c, 0xd111cc91, 0xdbb85f52, 0xb45fe44b,
				0xb4eee3ec, 0xb80dc501, 0x1393ee4f, 0x92523d32, 0xd48e3a1c, 0x224eb65e,
				0x6052c3a4, 0x2109c32f, 0x152ee388, 0xfd9f7ea1, 0x91c62f98, 0x97b55ba0,
				0x250cbca4, 0x4aec6526, 0xdf318384, 0x43a9ce26, 0x9362ad8b, 0x1134140b,
				0x9df5cf82, 0x2e911551, 0x267f0564, 0x4812e3e0, 0x688a52b0, 0xdcc8e944,
				0x115b16a4, 0x93c4eda2, 0x9db4feea, 0xf54bccce, 0x9773e3d2, 0x5f31dcd0,
				0x65c46721, 0x62774f3a, 0x67ca6bc0, 0x467d3a3b, 0x24778426, 0xb7991e9a,
				0xee825c26, 0xe352c8ee, 0xb4fdde1e, 0x94833ae3, 0x61211d04, 0x2732c132,
				0xdcadb248, 0xf606be8c, 0x912a794f, 0x98b4ef31, 0x4a9b4dc6, 0x5f755161,
				0x2116994f, 0x49829cb0, 0x21016574, 0x4343cbeb, 0x61d3d0b4, 0x44f30aef,
				0xb8ae7376, 0x2a3a1c9d, 0xb4b70914, 0xd6bc250c, 0x953b7328, 0x495f948f,
				0xd2a4ed8e, 0x6cf751e4, 0xd320cc76, 0xd9cbb0b4, 0x9ba56262, 0x4e84b03f,
				0xfea8076e, 0x94a07fe6, 0x9039e00c, 0x4611daf8, 0x13731358, 0xb9e671b1,
				0xdfd4ce1c, 0xb25b10ed, 0x4ee35fb2, 0xb44fb480, 0x4634f571, 0x25efd400,
				0xb9f5f5ea, 0x928932df, 0x16041d06, 0x6d31f502, 0x5f39c2e4, 0x2b89d9db,
				0x6bcc0a98, 0xd05bfd6f, 0x2b250622, 0x2e21be0e, 0x60973b04, 0xb45f4a68,
				0xb54fe638, 0xb6ed6616, 0x981a910a, 0x6d92928d, 0xbc6b4698, 0xf73c63ad,
				0x456edf5f, 0x457a8146, 0x61875a64, 0xee30994f, 0x69b5f18a, 0x9c73ee0b,
				0x6e57368f, 0x6c79e3cc, 0x9a595926, 0xbbc49ec6, 0x9fd8b4fb, 0x9016cbdb,
				0x9ccc1e38, 0x6982c712, 0x95c7da7a, 0x68811478, 0xee67fad2, 0xd764d9b4,
				0xd8102950, 0x6cd09da6, 0x2cc1f148, 0x95167d80, 0x1367046d, 0xbf1dfda2,
				0xb2247b24, 0x21301a54, 0x991d99c6, 0x9a4fb7cf, 0x277449a4, 0x19e57492,
				0x45c9a57e, 0x6e7f500a, 0xb9a62a8a, 0x5f242a6b, 0xb1337851, 0x9cda3346,
				0x24874048, 0x4328ba08, 0xfb815f1f, 0x4248896a, 0x9007d85d, 0x1f6e8eea,
				0x9250bdaf, 0xde2ee042, 0x997ee022, 0x6f003612, 0x4ba18f90, 0x26314076,
				0x9824035a, 0x4b57e2d6, 0x9e78aed2, 0xf90dc600
};