
### NDArray IO

#### 1. Save arrays to .npy files
- Python
  Numpy can read and write text or binary data on disk.
  NumPy introduces a simple file format for ndarray objects: npy.
  .npy files are used to store data, graphics, dtype and other information required to rebuild ndarray.
  Common IO functions include:
  The load() and save() functions are the two main functions for reading and writing file array data. By default, arrays are saved in an uncompressed raw binary format in files with the extension .npy.
  The savez() function is used to write multiple arrays to a file. By default, arrays are saved in an uncompressed raw binary format in files with the extension .npz.


```text
import numpy as np

a = np.array([1,2,3,4,5])

#Save to outfile.npy
np.save('outfile.npy',a)

#We can view the file content:
$ cat outfile.npy
?NUMPYv{'descr': '<i8', 'fortran_order': False, 'shape': (5,), }

#The file is gibberish because it is the data after Numpy's proprietary binary format.
```

- Java
```text
NDArray a = manager.create(new int[]{1,2,3,4,5});
NDList encoded = new NDList(a);
encoded.encode();
OutputStream os = Files.newOutputStream(Paths.get("src/test/resources/outfile.npy"));
encoded.encode(os, true);

#The output is as follows:
src/test/resources/outfile.npy
```

#### 2. Read .npy files
- Python
  We can use the load() function to read the data and display it normally:
```text
import numpy as np 
 
b = np.load('outfile.npy')  
print (b)

#The output is as follows:
[1 2 3 4 5]
```

- Java
```text
byte[] data = readFile("arr.npy");
NDList decoded = NDList.decode(manager, data);
NDArray array = decoded.get(0);
System.out.println(array.toDebugString(100, 10, 100, 100));

#The output is as follows：
ND: (5) cpu() int32
[ 1,  2,  3,  4,  5]
```

#### 3. Save multiple arrays to .npz files
- Python
  The numpy.savez() function saves multiple arrays to files with the extension .npz.

```text
import numpy as np 
 
a = np.array([[1,2,3],[4,5,6]])
b = np.arange(0, 1.0, 0.1)

np.savez("runoob.npz", a, b)

# The output is as follows:
runoob.npz
```

- Java
```text
a = manager.create(new int[][]{{1, 2, 3}, {4, 5, 6}});
NDArray b = manager.arange(0f, 1f, 0.1f);
encoded = new NDList(a, b);
encoded.encode();
os = Files.newOutputStream(Paths.get("src/test/resources/runoob.npz"));
encoded.encode(os, true);
        
# The output is as follows:
src/test/resources/runoob.npz
```

#### 4. Read .npz files
- Python
```text
import numpy as np

r = np.load("runoob.npz")
print(r.files) #View the names of each array
print(r["arr_0"]) #Array a
print(r["arr_1"]) #Array b

#The output is as follows:
['sin_array', 'arr_0', 'arr_1']
[[1 2 3]
 [4 5 6]]
[0.  0.1 0.2 0.3 0.4 0.5 0.6 0.7 0.8 0.9]
```

- Java
```text
data = readFile("runoob.npz");
decoded = NDList.decode(manager, data);
a = decoded.get(0);
b = decoded.get(1);
System.out.println(a.toDebugString(100, 10, 100, 100));
System.out.println(b.toDebugString(100, 10, 100, 100));

#The output is as follows:
ND: (2, 3) cpu() int32
[[ 1,  2,  3],
 [ 4,  5,  6],
]

ND: (10) cpu() float32
[0. , 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9]

```



### Code download link:
[Github](https://github.com/mymagicpower/AIAS/blob/main/0_tutorials/ndarray_lessons/src/main/java/me/aias/example/No10IOExample.java)    


