/*
 * Copyright (c) 2013, Kevin Läufer
 * Copyright (c) 2013-2014, Sascha Schade
 * Copyright (c) 2013-2018, Niklas Hauser
 * Copyright (c) 2018, Sebastian Birke
 *
 * This file is part of the modm project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */
// ----------------------------------------------------------------------------

#include <modm/board.hpp>
#include <modm/processing/timer.hpp>
#include <modm/debug/logger.hpp>

/**
 * Example of CAN Hardware on STM32 F469 Discovery Board (with display).
 *
 * Connect PB5 / PB13 to a CAN transceiver which is connected to a CAN bus.
 *
 * Tested in hardware (F469I-DISCO) on 2018-08-21 by Sebastian Birke.
 */

// Set the log level
#undef	MODM_LOG_LEVEL
#define	MODM_LOG_LEVEL modm::log::DEBUG

static void
displayMessage(const modm::can::Message& message)
{
	static uint32_t receiveCounter = 0;
	receiveCounter++;

	MODM_LOG_INFO<< "id  =" << message.getIdentifier();
	if (message.isExtended()) {
		MODM_LOG_INFO<< " extended";
	}
	else {
		MODM_LOG_INFO<< " standard";
	}
	if (message.isRemoteTransmitRequest()) {
		MODM_LOG_INFO<< ", rtr";
	}
	MODM_LOG_INFO<< modm::endl;

	MODM_LOG_INFO<< "dlc =" << message.getDataLengthCode() << modm::endl;
	MODM_LOG_INFO<< "length =" << message.getLength() << modm::endl;
	if (!message.isRemoteTransmitRequest())
	{
		MODM_LOG_INFO << "data=";
		for (uint32_t i = 0; i < message.getLength(); ++i) {
			MODM_LOG_INFO<< modm::hex << message.data[i] << modm::ascii << ' ';
		}
		MODM_LOG_INFO<< modm::endl;
	}
	MODM_LOG_INFO<< "# received=" << receiveCounter << modm::endl;
}

// ----------------------------------------------------------------------------
int
main()
{
	Board::initialize();
	Board::LedD13::setOutput(modm::Gpio::Low);

	Board::LedGreen::set();

	MODM_LOG_INFO << "CAN Test Program" << modm::endl;

	/* Split filter bank, otherwise Can2 won't work */
	CanFilter::setStartFilterBankForCan2(14);

	MODM_LOG_INFO << "Initializing Can ..." << modm::endl;
	// Initialize Can
	Can2::connect<GpioB13::Tx, GpioB5::Rx>(Gpio::InputType::PullUp);
	Can2::initialize<Board::SystemClock, 125_kbps>(9);

	MODM_LOG_INFO << "Setting up Filter for Can ..." << modm::endl;
	// Receive every message
	CanFilter::setFilter(14, CanFilter::FIFO0,
			CanFilter::ExtendedIdentifier(0),
			CanFilter::ExtendedFilterMask(0));

	// Send a message
	MODM_LOG_INFO << "Sending message on Can ..." << modm::endl;
	modm::can::Message msg1(1, 1);
	msg1.setExtended(true);
	msg1.data[0] = 0x11;
	Can2::sendMessage(msg1);

	modm::ShortPeriodicTimer pTimer(100ms);

	while (true)
	{
		if (Can2::isMessageAvailable())
		{
			MODM_LOG_INFO << "Can2: Message is available..." << modm::endl;
			modm::can::Message message;
			Can2::getMessage(message);
			displayMessage(message);
			Board::LedGreen::toggle();
		}

		if (pTimer.execute()) {
			Board::LedRed::toggle();

			static uint8_t idx = 0;
			modm::can::Message msg1(1, 1);
			msg1.setExtended(true);
			msg1.data[0] = idx;
			Can2::sendMessage(msg1);

			++idx;
		}
	}

	return 0;
}
