/*
 * JGrapes Event Driven Framework
 * Copyright (C) 2018,2022 Michael N. Lipp
 * 
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Affero General Public License as published by 
 * the Free Software Foundation; either version 3 of the License, or 
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License 
 * for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License along 
 * with this program; if not, see <http://www.gnu.org/licenses/>.
 */

package org.jgrapes.webconsole.base.freemarker;

import freemarker.cache.ClassTemplateLoader;
import freemarker.cache.MultiTemplateLoader;
import freemarker.cache.TemplateLoader;
import freemarker.template.Configuration;
import freemarker.template.SimpleScalar;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import freemarker.template.TemplateExceptionHandler;
import freemarker.template.TemplateMethodModelEx;
import freemarker.template.TemplateModel;
import freemarker.template.TemplateModelException;
import java.io.IOException;
import java.io.Writer;
import java.net.URI;
import java.text.Collator;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.UUID;
import org.jdrupes.httpcodec.protocols.http.HttpConstants.HttpStatus;
import org.jdrupes.httpcodec.protocols.http.HttpField;
import org.jdrupes.httpcodec.protocols.http.HttpResponse;
import org.jdrupes.httpcodec.types.MediaType;
import org.jgrapes.core.Channel;
import org.jgrapes.http.LanguageSelector.Selection;
import org.jgrapes.http.events.Request;
import org.jgrapes.http.events.Response;
import org.jgrapes.io.IOSubchannel;
import org.jgrapes.io.util.ByteBufferWriter;
import org.jgrapes.webconsole.base.ConsoleWeblet;

/**
 * A console weblet that uses a freemarker template to generate
 * the HTML source for the console page.  
 */
public abstract class FreeMarkerConsoleWeblet extends ConsoleWeblet {

    public static final String UTF_8 = "utf-8";
    /**
     * Initialized with a base FreeMarker configuration.
     */
    protected Configuration freeMarkerConfig;

    /**
     * Instantiates a new free marker console weblet.
     *
     * @param webletChannel the weblet channel
     * @param consoleChannel the console channel
     * @param consolePrefix the console prefix
     */
    @SuppressWarnings("PMD.AvoidInstantiatingObjectsInLoops")
    public FreeMarkerConsoleWeblet(Channel webletChannel,
            Channel consoleChannel, URI consolePrefix) {
        super(webletChannel, consoleChannel, consolePrefix);
        freeMarkerConfig = new Configuration(Configuration.VERSION_2_3_26);
        List<TemplateLoader> loaders = new ArrayList<>();
        Class<?> clazz = getClass();
        while (!clazz.equals(FreeMarkerConsoleWeblet.class)) {
            loaders.add(new ClassTemplateLoader(clazz.getClassLoader(),
                clazz.getPackage().getName().replace('.', '/')));
            clazz = clazz.getSuperclass();
        }
        freeMarkerConfig.setTemplateLoader(
            new MultiTemplateLoader(loaders.toArray(new TemplateLoader[0])));
        freeMarkerConfig.setDefaultEncoding(UTF_8);
        freeMarkerConfig.setTemplateExceptionHandler(
            TemplateExceptionHandler.RETHROW_HANDLER);
        freeMarkerConfig.setLogTemplateExceptions(false);
    }

    /**
     * Prepend a class template loader to the list of loaders 
     * derived from the class hierarchy.
     *
     * @param classloader the class loader
     * @param path the path
     * @return the free marker console weblet
     */
    public FreeMarkerConsoleWeblet
            prependClassTemplateLoader(ClassLoader classloader, String path) {
        List<TemplateLoader> loaders = new ArrayList<>();
        loaders.add(new ClassTemplateLoader(classloader, path));
        MultiTemplateLoader oldLoader
            = (MultiTemplateLoader) freeMarkerConfig.getTemplateLoader();
        for (int i = 0; i < oldLoader.getTemplateLoaderCount(); i++) {
            loaders.add(oldLoader.getTemplateLoader(i));
        }
        freeMarkerConfig.setTemplateLoader(
            new MultiTemplateLoader(loaders.toArray(new TemplateLoader[0])));
        return this;
    }

    /**
     * Convenience version of 
     * {@link #prependClassTemplateLoader(ClassLoader, String)} that derives
     * the path from the class's package name.
     *
     * @param clazz the clazz
     * @return the free marker console weblet
     */
    public FreeMarkerConsoleWeblet prependClassTemplateLoader(Class<?> clazz) {
        return prependClassTemplateLoader(clazz.getClassLoader(),
            clazz.getPackage().getName().replace('.', '/'));
    }

    /**
     * Creates the console base model.
     *
     * @return the base model
     */
    @SuppressWarnings("PMD.UseConcurrentHashMap")
    protected Map<String, Object> createConsoleBaseModel() {
        // Create console model
        Map<String, Object> consoleModel = new HashMap<>();
        consoleModel.put("consoleType", getClass().getName());
        consoleModel.put("renderSupport", renderSupport());
        consoleModel.put("useMinifiedResources", useMinifiedResources());
        consoleModel.put("minifiedExtension",
            useMinifiedResources() ? ".min" : "");
        consoleModel.put("connectionRefreshInterval",
            connectionRefreshInterval());
        consoleModel.put("connectionInactivityTimeout",
            connectionInactivityTimeout());
        return consoleModel;
    }

    /**
     * Invoked by {@link #renderConsole renderConsole} 
     * to expand the {@link #createConsoleBaseModel()
     * base model} with information from the current event.
     *
     * @param model the model
     * @param event the event
     * @param consoleConnectionId the console connection id
     * @return the map
     */
    @SuppressWarnings("PMD.DataflowAnomalyAnalysis")
    protected Map<String, Object> expandConsoleModel(
            Map<String, Object> model, Request.In.Get event,
            UUID consoleConnectionId) {
        // WebConsole Connection UUID
        model.put("consoleConnectionId", consoleConnectionId.toString());

        // Add locale
        final Locale locale = event.associated(Selection.class).map(
            sel -> sel.get()[0]).orElse(Locale.getDefault());
        model.put("locale", locale);

        // Add supported languages
        model.put("supportedLanguages", new TemplateMethodModelEx() {
            private Object cachedResult;

            @Override
            public Object exec(@SuppressWarnings("rawtypes") List arguments)
                    throws TemplateModelException {
                if (cachedResult != null) {
                    return cachedResult;
                }
                final Collator coll = Collator.getInstance(locale);
                final Comparator<LanguageInfo> comp = new Comparator<>() {
                    @Override
                    public int compare(LanguageInfo o1, LanguageInfo o2) { // NOPMD
                        return coll.compare(o1.getLabel(), o2.getLabel());
                    }
                };
                return cachedResult = supportedLocales().entrySet().stream()
                    .map(entry -> new LanguageInfo(entry.getKey(),
                        entry.getValue()))
                    .sorted(comp).toArray(size -> new LanguageInfo[size]);
            }
        });

        final ResourceBundle baseResources = consoleResourceBundle(locale);
        model.put("_", new TemplateMethodModelEx() {
            @Override
            public Object exec(@SuppressWarnings("rawtypes") List arguments)
                    throws TemplateModelException {
                @SuppressWarnings("unchecked")
                List<TemplateModel> args = (List<TemplateModel>) arguments;
                if (!(args.get(0) instanceof SimpleScalar)) {
                    throw new TemplateModelException("Not a string.");
                }
                String key = ((SimpleScalar) args.get(0)).getAsString();
                try {
                    return baseResources.getString(key);
                } catch (MissingResourceException e) { // NOPMD
                    // no luck
                }
                return key;
            }
        });
        return model;
    }

    /**
     * Render the console page using the freemarker template 
     * "console.ftl.html". The template for the console page
     * (and all included templates) are loaded using the 
     * a list of template class loaders created as follows:
     * 
     *  1. Start with the actual type of the console conlet.
     *  
     *  2. Using the current type, add a freemarker class template loader
     *    {@link ClassTemplateLoader#ClassTemplateLoader(ClassLoader, String)}
     *    that uses the package name as path (all dots replaced with
     *    slashes).
     *    
     *  3. Repeat step 2 with the super class of the current type
     *     until type {@link FreeMarkerConsoleWeblet} is reached.
     *     
     *  This approach allows a (base) console weblet to provide a
     *  console page template that includes another template
     *  e.g. "footer.ftl.html" to provide a specific part of the 
     *  console page. A derived console weblet can then provide its
     *  own "footer.ftl.html" and thus override the version from the
     *  base class(es).
     * 
     * @param event the event
     * @param channel the channel
     * @throws IOException Signals that an I/O exception has occurred.
     * @throws InterruptedException the interrupted exception
     */
    @Override
    protected void renderConsole(Request.In.Get event, IOSubchannel channel,
            UUID consoleConnectionId) throws IOException, InterruptedException {
        event.setResult(true);
        event.stop();

        // Prepare response
        HttpResponse response = event.httpRequest().response().get();
        MediaType mediaType = MediaType.builder().setType("text", "html")
            .setParameter("charset", UTF_8).build();
        response.setField(HttpField.CONTENT_TYPE, mediaType);
        response.setStatus(HttpStatus.OK);
        response.setHasPayload(true);
        channel.respond(new Response(response));
        try (@SuppressWarnings("resource")
        Writer out = new ByteBufferWriter(channel).suppressClose()) {
            Template tpl = freeMarkerConfig.getTemplate("console.ftl.html");
            Map<String, Object> consoleModel = expandConsoleModel(
                createConsoleBaseModel(), event, consoleConnectionId);
            tpl.process(consoleModel, out);
        } catch (TemplateException e) {
            throw new IOException(e);
        }
    }

}
