/*
 * JGrapes Event Driven Framework
 * Copyright (C) 2017-2022 Michael N. Lipp
 * 
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Affero General Public License as published by 
 * the Free Software Foundation; either version 3 of the License, or 
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License 
 * for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License along 
 * with this program; if not, see <http://www.gnu.org/licenses/>.
 */

package org.jgrapes.webconsole.base;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;
import java.io.Serializable;
import java.io.StringWriter;
import java.net.URL;
import java.nio.CharBuffer;
import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.UUID;
import java.util.WeakHashMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.jgrapes.core.Channel;
import org.jgrapes.core.Component;
import org.jgrapes.core.Components;
import org.jgrapes.core.Components.Timer;
import org.jgrapes.core.Event;
import org.jgrapes.core.annotation.Handler;
import org.jgrapes.core.annotation.HandlerDefinition.ChannelReplacements;
import org.jgrapes.core.events.Detached;
import org.jgrapes.http.Session;
import org.jgrapes.io.IOSubchannel;
import org.jgrapes.io.events.Closed;
import org.jgrapes.webconsole.base.Conlet.RenderMode;
import org.jgrapes.webconsole.base.events.AddConletRequest;
import org.jgrapes.webconsole.base.events.AddConletType;
import org.jgrapes.webconsole.base.events.ConletDeleted;
import org.jgrapes.webconsole.base.events.ConletResourceRequest;
import org.jgrapes.webconsole.base.events.ConsoleReady;
import org.jgrapes.webconsole.base.events.DeleteConlet;
import org.jgrapes.webconsole.base.events.NotifyConletModel;
import org.jgrapes.webconsole.base.events.NotifyConletView;
import org.jgrapes.webconsole.base.events.RenderConlet;
import org.jgrapes.webconsole.base.events.RenderConletRequest;
import org.jgrapes.webconsole.base.events.RenderConletRequestBase;
import org.jgrapes.webconsole.base.events.SetLocale;
import org.jgrapes.webconsole.base.events.UpdateConletType;

/**
 * Provides a base class for implementing web console components.
 * The class provides the following support functions:
 *  * "Translate" the conlet related events to invocations
 *    of abstract methods. This is mainly a prerequisite
 *    for implementing the other support functions.
 *  * Optionally manage state for a conlet instance.
 *  * Optionally track the existing previews or views of
 *    a conlet, thus allowing the server side to send update
 *    events (usually when the state changes on the server side).
 *  * Optionally refresh existing previews or views periodically
 * 
 * # Event handling
 * 
 * The following diagrams show the events exchanged between
 * the {@link WebConsole} and a web console component from the 
 * web console component's perspective. If applicable, they also show 
 * how the events are translated by the {@link AbstractConlet} to invocations 
 * of the abstract methods that have to be implemented by the
 * derived class (the web console component that provides
 * a specific web console component type).
 * 
 * ## ConsoleReady
 * 
 * ![Add web console component type handling](AddConletTypeHandling.svg)
 * 
 * From the web console's page point of view, a web console component 
 * consists of CSS and JavaScript that is added to the console page by
 * {@link AddConletType} events and HTML that is provided by 
 * {@link RenderConlet} events (see below). These events must 
 * therefore be generated by a web console component. 
 * 
 * The {@link AbstractConlet} does not provide support for generating 
 * an {@link AddConletType} event. The handler for the 
 * {@link ConsoleReady} that generates this event must be implemented by
 * the derived class itself.
 * 
 * ## AddConletRequest
 * 
 * ![Add web console component handling](AddConletHandling.svg)
 * 
 * The {@link AddConletRequest} indicates that a new web console component
 * instance of a given type should be added to the page. The
 * {@link AbstractConlet} checks the type requested, and if
 * it matches, invokes {@link #generateInstanceId generateInstanceId}
 * and {@link #createNewState createNewState}.
 * If the conlet has associated state, the information is saved with
 * {@link #putInSession putInSession}. Then 
 * {@link #doRenderConlet doRenderConlet} is invoked, which must 
 * render the conlet in the browser. Information about the rendered views
 * is returned and used to track the views.
 * 
 * Method {@link #doRenderConlet doRenderConlet} renders the preview 
 * or view by firing a {@link RenderConlet} event that provides to 
 * the console page the HTML that represents the web console 
 * component on the page. The HTML may be generated using and thus 
 * depending on the component state.
 * Alternatively, state independent HTML may be provided followed 
 * by a {@link NotifyConletView} event that updates
 * the HTML (using JavaScript) on the console page. The latter approach
 * is preferred if the model changes frequently and updating the
 * rendered representation is more efficient than providing a new one.
 * 
 * ## RenderConletRequest
 * 
 * ![Render web console component handling](RenderConletHandling.svg)
 * 
 * A {@link RenderConletRequest} event indicates that the web console page
 * needs the HTML for displaying a web console component. This may be caused
 * by e.g. the initial display, by a refresh or by requesting a full 
 * page view from the preview.
 * 
 * Upon receiving such an event, the {@link AbstractConlet}
 * checks if it has state information for the component id
 * requested. If not, it calls {@link #recreateState recreateState} 
 * which allows the conlet to e.g. retrieve state information from
 * a backing store.
 * 
 * Once state information has been obtained, the method 
 * continues as when adding a new conlet by invoking
 * {@link #doRenderConlet doRenderConlet}.
 * 
 * ## ConletDeleted
 * 
 * ![Web console component deleted handling](ConletDeletedHandling.svg)
 * 
 * When the {@link AbstractConlet} receives a {@link ConletDeleted}
 * event, it updates the information about the shown conlet views. If
 * the conlet is no longer used in the browser (no views remain),
 * it deletes the state information from the session. In any case, it
 * invokes {@link #doConletDeleted doConletDeleted} with the 
 * state information.
 * 
 * ## NotifyConletModel
 * 
 * ![Notify web console component model handling](NotifyConletModelHandling.svg)
 * 
 * If the web console component views include input elements, actions 
 * on these elements may result in {@link NotifyConletModel} events from
 * the web console page to the web console. When the {@link AbstractConlet}
 * receives such events, it retrieves any existing state information.
 * It then invokes {@link #doUpdateConletState doUpdateConletState}
 * with the retrieved information. The web console component usually
 * responds with a {@link NotifyConletView} event. However, it can
 * also re-render the complete conlet view.
 * 
 * Support for unsolicited updates
 * -------------------------------
 * 
 * The class tracks the relationship between the known
 * {@link ConsoleConnection}s and the web console components displayed 
 * in the console pages. The information is available from
 * {@link #conletInfosByConsoleConnection conletInfosByConsoleConnection}.
 * It can e.g. be used to send events to the web console(s) in response 
 * to an event on the server side.
 *
 * @param <S> the type of the conlet's state information
 * 
 * @startuml AddConletTypeHandling.svg
 * hide footbox
 * 
 * activate WebConsole
 * WebConsole -> Conlet: ConsoleReady
 * deactivate WebConsole
 * activate Conlet
 * Conlet -> WebConsole: AddConletType 
 * deactivate Conlet
 * activate WebConsole
 * deactivate WebConsole
 * @enduml
 * 
 * @startuml AddConletHandling.svg
 * hide footbox
 * 
 * activate WebConsole
 * WebConsole -> Conlet: AddConletRequest
 * deactivate WebConsole
 * activate Conlet
 * Conlet -> Conlet: generateInstanceId
 * activate Conlet
 * deactivate Conlet
 * Conlet -> Conlet: createNewState
 * activate Conlet
 * deactivate Conlet
 * opt if state
 *     Conlet -> Conlet: putInSession
 *     activate Conlet
 *     deactivate Conlet
 * end opt
 * Conlet -> Conlet: doRenderConlet
 * activate Conlet
 * Conlet -> WebConsole: RenderConlet
 * activate WebConsole
 * deactivate WebConsole
 * opt
 *     Conlet -> WebConsole: NotifyConletView
 *     activate WebConsole
 *     deactivate WebConsole
 * end opt
 * deactivate Conlet
 * Conlet -> Conlet: start conlet tracking
 * @enduml
 * 
 * @startuml RenderConletHandling.svg
 * hide footbox
 * 
 * activate WebConsole
 * WebConsole -> Conlet: RenderConletRequest
 * deactivate WebConsole
 * activate Conlet
 * Conlet -> Conlet: stateFromSession
 * activate Conlet
 * deactivate Conlet
 * opt if not found
 *     Conlet -> Conlet: recreateState
 *     activate Conlet
 *     deactivate Conlet
 *     opt if state
 *         Conlet -> Conlet: putInSession
 *         activate Conlet
 *         deactivate Conlet
 *     end opt
 * end opt
 * Conlet -> Conlet: doRenderConlet
 * activate Conlet
 * Conlet -> WebConsole: RenderConlet 
 * activate WebConsole
 * deactivate WebConsole
 * opt 
 *     Conlet -> WebConsole: NotifyConletView
 * activate WebConsole
 * deactivate WebConsole
 * end opt 
 * deactivate Conlet
 * Conlet -> Conlet: update conlet tracking
 * @enduml
 * 
 * @startuml NotifyConletModelHandling.svg
 * hide footbox
 * 
 * activate WebConsole
 * WebConsole -> Conlet: NotifyConletModel
 * deactivate WebConsole
 * activate Conlet
 * Conlet -> Conlet: stateFromSession
 * activate Conlet
 * deactivate Conlet
 * opt if not found
 *     Conlet -> Conlet: recreateState
 *     activate Conlet
 *     deactivate Conlet
 *     opt if state
 *         Conlet -> Conlet: putInSession
 *         activate Conlet
 *         deactivate Conlet
 *     end opt
 * end opt
 * Conlet -> Conlet: doUpdateConletState
 * activate Conlet
 * opt
 *     Conlet -> WebConsole: RenderConlet
 * end opt 
 * opt 
 *     Conlet -> WebConsole: NotifyConletView
 * end opt 
 * deactivate Conlet
 * deactivate Conlet
 * @enduml
 * 
 * @startuml ConletDeletedHandling.svg
 * hide footbox
 * 
 * activate WebConsole
 * WebConsole -> Conlet: ConletDeleted
 * deactivate WebConsole
 * activate Conlet
 * Conlet -> Conlet: stateFromSession
 * activate Conlet
 * deactivate Conlet
 * alt all views deleted
 *     Conlet -> Conlet: removeState
 *     activate Conlet
 *     deactivate Conlet
 *     Conlet -> Conlet: stop conlet tracking
 * else
 *     Conlet -> Conlet: update conlet tracking
 * end alt
 * Conlet -> Conlet: doConletDeleted
 * activate Conlet
 * deactivate Conlet
 * deactivate Conlet
 * @enduml
 */
@SuppressWarnings({ "PMD.TooManyMethods",
    "PMD.EmptyMethodInAbstractClassShouldBeAbstract", "PMD.GodClass",
    "PMD.ExcessiveImports", "PMD.CouplingBetweenObjects" })
public abstract class AbstractConlet<S> extends Component {

    /** Separator used between type and instance when generating the id. */
    public static final String TYPE_INSTANCE_SEPARATOR = "~";
    @SuppressWarnings({ "PMD.FieldNamingConventions",
        "PMD.VariableNamingConventions", "PMD.UseConcurrentHashMap",
        "PMD.AvoidDuplicateLiterals" })
    private static final Map<Class<?>,
            Map<Locale, ResourceBundle>> supportedLocales
                = Collections.synchronizedMap(new WeakHashMap<>());
    @SuppressWarnings({ "PMD.FieldNamingConventions",
        "PMD.VariableNamingConventions", "PMD.UseConcurrentHashMap" })
    private static final Map<Class<?>,
            Map<Locale, ResourceBundle>> l10nBundles
                = Collections.synchronizedMap(new WeakHashMap<>());
    @SuppressWarnings("PMD.LongVariable")
    private Map<ConsoleConnection,
            Map<String, ConletTrackingInfo>> conletInfosByConsoleConnection;
    private Duration refreshInterval;
    private Supplier<Event<?>> refreshEventSupplier;
    private Timer refreshTimer;

    /**
     * Extract the conlet type from a conlet id.
     *
     * @param conletId the conlet id
     * @return the type or {@code null} the conlet id does not contain
     * a {@link TYPE_INSTANCE_SEPARATOR}
     */
    public static String typeFromId(String conletId) {
        int sep = conletId.indexOf(TYPE_INSTANCE_SEPARATOR);
        if (sep < 0) {
            return null;
        }
        return conletId.substring(0, sep);
    }

    /**
     * Creates a new component that listens for new events
     * on the given channel.
     * 
     * @param channel the channel to listen on
     */
    public AbstractConlet(Channel channel) {
        this(channel, null);
    }

    /**
     * Like {@link #AbstractConlet(Channel)}, but supports
     * the specification of channel replacements.
     *
     * @param channel the channel to listen on
     * @param channelReplacements the channel replacements (see
     * {@link Component})
     */
    @SuppressWarnings("PMD.LooseCoupling")
    public AbstractConlet(Channel channel,
            ChannelReplacements channelReplacements) {
        super(channel, channelReplacements);
        conletInfosByConsoleConnection
            = Collections.synchronizedMap(new WeakHashMap<>());
    }

    /**
     * If set to a value different from `null` causes an event
     * from the given supplier to be fired on all tracked web console
     * connections periodically.
     *
     * @param interval the refresh interval
     * @param supplier the supplier
     * @return the web console component for easy chaining
     */
    @SuppressWarnings("PMD.LinguisticNaming")
    public AbstractConlet<S> setPeriodicRefresh(
            Duration interval, Supplier<Event<?>> supplier) {
        refreshInterval = interval;
        refreshEventSupplier = supplier;
        if (refreshTimer != null) {
            refreshTimer.cancel();
            refreshTimer = null;
        }
        updateRefresh();
        return this;
    }

    private void updateRefresh() {
        if (refreshInterval == null
            || conletIdsByConsoleConnection().isEmpty()) {
            // At least one of the prerequisites is missing, terminate
            if (refreshTimer != null) {
                refreshTimer.cancel();
                refreshTimer = null;
            }
            return;
        }
        if (refreshTimer != null) {
            // Already running.
            return;
        }
        refreshTimer = Components.schedule(tmr -> {
            tmr.reschedule(tmr.scheduledFor().plus(refreshInterval));
            fire(refreshEventSupplier.get(), trackedConnections());
        }, Instant.now().plus(refreshInterval));
    }

    /**
     * Returns the web console component type. The default implementation
     * returns the name of the class.
     * 
     * @return the type
     */
    protected String type() {
        return getClass().getName();
    }

    /**
     * A default handler for resource requests. Checks that the request
     * is directed at this web console component, and calls 
     * {@link #doGetResource}.
     * 
     * @param event the resource request event
     * @param channel the channel that the request was recived on
     */
    @Handler
    public final void onConletResourceRequest(
            ConletResourceRequest event, IOSubchannel channel) {
        // For me?
        if (!event.conletClass().equals(type())) {
            return;
        }
        doGetResource(event, channel);
    }

    /**
     * The default implementation searches for a file with the 
     * requested resource URI in the web console component's class 
     * path and sets its {@link URL} as result if found.
     * 
     * @param event the event. The result will be set to
     * `true` on success
     * @param channel the channel
     */
    protected void doGetResource(ConletResourceRequest event,
            IOSubchannel channel) {
        URL resourceUrl = this.getClass().getResource(
            event.resourceUri().getPath());
        if (resourceUrl == null) {
            return;
        }
        event.setResult(new ResourceByUrl(event, resourceUrl));
        event.stop();
    }

    /**
     * Provides a resource bundle for localization.
     * The default implementation looks up a bundle using the
     * package name plus "l10n" as base name. Note that the bundle 
     * returned for a given locale may be the fallback bundle.
     * 
     * @return the resource bundle
     */
    protected ResourceBundle resourceBundle(Locale locale) {
        return ResourceBundle.getBundle(
            getClass().getPackage().getName() + ".l10n", locale,
            getClass().getClassLoader(),
            ResourceBundle.Control.getNoFallbackControl(
                ResourceBundle.Control.FORMAT_DEFAULT));
    }

    /**
     * Returns bundles for the given locales. 
     * 
     * The default implementation uses {@link #resourceBundle(Locale)} 
     * to lookup the bundles. The method is guaranteed to return a 
     * bundle for each requested locale even if it is only the fallback 
     * bundle. The evaluated results are cached for the conlet class.
     *
     * @param toGet the locales to get bundles for
     * @return the map with locales and bundles
     */
    @SuppressWarnings({ "PMD.AvoidInstantiatingObjectsInLoops",
        "PMD.AvoidSynchronizedStatement", "PMD.AvoidDuplicateLiterals" })
    protected Map<Locale, ResourceBundle> l10nBundles(Set<Locale> toGet) {
        @SuppressWarnings("PMD.UseConcurrentHashMap")
        Map<Locale, ResourceBundle> result = new HashMap<>();
        for (Locale locale : toGet) {
            ResourceBundle bundle;
            synchronized (l10nBundles) {
                // Due to the nested computeIfAbsent, it is not sufficient
                // that l10nBundels is thread safe.
                bundle = l10nBundles
                    .computeIfAbsent(getClass(),
                        cls -> new ConcurrentHashMap<>())
                    .computeIfAbsent(locale, l -> resourceBundle(locale));
            }
            result.put(locale, bundle);
        }
        return Collections.unmodifiableMap(result);
    }

    /**
     * Provides localizations for the given key for all requested locales.
     * 
     * The default implementation uses {@link #l10nBundles(Set)} to obtain
     * the localizations.
     *
     * @param locales the requested locales
     * @param key the key
     * @return the result
     */
    protected Map<Locale, String> localizations(Set<Locale> locales,
            String key) {
        @SuppressWarnings("PMD.UseConcurrentHashMap")
        Map<Locale, String> result = new HashMap<>();
        Map<Locale, ResourceBundle> bundles = l10nBundles(locales);
        for (Map.Entry<Locale, ResourceBundle> entry : bundles.entrySet()) {
            result.put(entry.getKey(), entry.getValue().getString(key));
        }
        return result;
    }

    /**
     * Returns the supported locales and the associated bundles.
     * 
     * The default implementation invokes {@link #resourceBundle(Locale)}
     * with all available locales and drops results with fallback bundles.
     * The evaluated results are cached for the conlet class.
     *
     * @return the result
     */
    @SuppressWarnings("PMD.AvoidLiteralsInIfCondition")
    protected Map<Locale, ResourceBundle> supportedLocales() {
        return supportedLocales.computeIfAbsent(getClass(), cls -> {
            ResourceBundle.clearCache(cls.getClassLoader());
            @SuppressWarnings("PMD.UseConcurrentHashMap")
            Map<Locale, ResourceBundle> bundles = new HashMap<>();
            for (Locale locale : Locale.getAvailableLocales()) {
                if ("".equals(locale.getLanguage())) {
                    continue;
                }
                ResourceBundle bundle = resourceBundle(locale);
                if (bundle.getLocale().equals(locale)) {
                    bundles.put(locale, bundle);
                }
            }
            return bundles;
        });
    }

    /**
     * Create the instance specific part of a conlet id. The default
     * implementation generates a UUID. Derived classes override this
     * method if e.g. the instance specific part must include a key that
     * associates the conlet's state with some backing store. 
     * 
     * @param event the event that triggered the creation of a new conlet,
     * which may contain required information 
     * (see {@link AddConletRequest#properties()})
     * @param connection the console connection; usually not required 
     * but provided as context
     *
     * @return the web console component id
     */
    protected String generateInstanceId(AddConletRequest event,
            ConsoleConnection connection) {
        return UUID.randomUUID().toString();
    }

    /**
     * Creates an instance of the type that represents the conlet's state,
     * initialized with default values. The default implementation returns 
     * {@link Optional#isEmpty()}, thus indicating that no state 
     * information is needed or available.
     * 
     * This method should always be overridden if conlet instances
     * have associated state.
     *
     * @param event the event, which may contain required information 
     * (see {@link AddConletRequest#properties()})
     * @param connection the console connection, sometimes required to
     * send events to components that provide a backing store
     * @param conletId the conlet id calculated as
     * `type() + TYPE_INSTANCE_SEPARATOR + generateInstanceId(...)` 
     * @return the state representation or {@link Optional#empty()} if none is
     * required
     * @throws Exception if an exception occurs
     */
    @SuppressWarnings({ "PMD.SignatureDeclareThrowsException",
        "PMD.AvoidDuplicateLiterals" })
    protected Optional<S> createStateRepresentation(Event<?> event,
            ConsoleConnection connection, String conletId) throws Exception {
        return Optional.empty();
    }

    /**
     * Called by {@link #onAddConletRequest}
     * when a new conlet instance is created in the browser. The default
     * implementation simply invokes {@link 
     * #createStateRepresentation} and returns its result. If state
     * is provided, it is put in the browser session by the invoker.
     * 
     * This method should only be overridden if the event has associated
     * information (see {@link AddConletRequest#addProperty}) that
     * can be used to initialize the state with information that differs
     * from the defaults used by {@link #createStateRepresentation}.
     *
     * @param event the event 
     * @param connection the console connection
     * @param conletId the conlet id
     * @return the state representation or {@link Optional#empty()} if none is
     * required
     * @throws Exception if an exception occurs
     */
    @SuppressWarnings({ "PMD.SignatureDeclareThrowsException",
        "PMD.AvoidDuplicateLiterals" })
    protected Optional<S> createNewState(
            AddConletRequest event, ConsoleConnection connection,
            String conletId) throws Exception {
        return createStateRepresentation(event, connection, conletId);
    }

    /**
     * Called when a previously created conlet (with associated state)
     * is rendered in a new browser session for the first time. The 
     * default implementation simply invokes 
     * {@link #createStateRepresentation createStateRepresentation}
     * and returns its result. Conlets with long-term state should
     * retrieve their state from some storage. If state is returned,
     * it is put in the browser session by the invoker.
     *
     * @param event the event 
     * @param connection the console connection
     * @param conletId the conlet id
     * @return the state representation or {@link Optional#empty()} if none is
     * required
     * @throws Exception if an exception occurs
     */
    @SuppressWarnings({ "PMD.SignatureDeclareThrowsException",
        "PMD.AvoidDuplicateLiterals" })
    protected Optional<S> recreateState(
            Event<?> event, ConsoleConnection connection,
            String conletId) throws Exception {
        return createStateRepresentation(event, connection, conletId);
    }

    /**
     * Returns the tracked connections and conlet ids as map.
     * 
     * If you need a particular connection's web console component ids, you 
     * should prefer {@link #conletIds(ConsoleConnection)} over calling
     * this method with `get(consoleConnection)` appended.
     * 
     * @return the result
     */
    protected Map<ConsoleConnection, Set<String>>
            conletIdsByConsoleConnection() {
        return conletInfosByConsoleConnection.entrySet().stream()
            .collect(Collectors.toMap(Entry::getKey,
                e -> new HashSet<>(e.getValue().keySet())));
    }

    /**
     * Returns the tracked connections. This is effectively
     * `conletInfosByConsoleConnection().keySet()` converted to
     * an array. This representation is especially useful 
     * when the web console connections are used as argument for 
     * {@link #fire(Event, Channel...)}.
     *
     * @return the web console connections
     */
    protected ConsoleConnection[] trackedConnections() {
        Set<ConsoleConnection> connections = new HashSet<>(
            conletInfosByConsoleConnection.keySet());
        return connections.toArray(new ConsoleConnection[0]);
    }

    /**
     * Returns the set of web console component ids associated with the 
     * console connection as a {@link Set}. If no web console components 
     * have registered yet, an empty set is returned.
     * 
     * @param connection the console connection
     * @return the set
     */
    protected Set<String> conletIds(ConsoleConnection connection) {
        return new HashSet<>(conletInfosByConsoleConnection.getOrDefault(
            connection, Collections.emptyMap()).keySet());
    }

    /**
     * Returns a map of all conlet ids and the modes in which 
     * views are currently rendered. 
     *
     * @param connection the console connection
     * @return the map
     */
    protected Map<String, Set<RenderMode>>
            conletViews(ConsoleConnection connection) {
        return conletInfosByConsoleConnection.getOrDefault(
            connection, Collections.emptyMap()).entrySet().stream()
            .collect(Collectors.toMap(Entry::getKey,
                e -> e.getValue().renderedAs));
    }

    /**
     * Track the given web console component from the given connection. 
     * This is invoked by 
     * {@link #onAddConletRequest(AddConletRequest, ConsoleConnection)} and
     * {@link #onRenderConletRequest(RenderConletRequest, ConsoleConnection)}.
     * It needs only be invoked if either method is overridden.
     *
     * @param connection the web console connection
     * @param conletId the conlet id
     * @param info the info to be added if currently untracked. If `null`,
     * a new {@link ConletTrackingInfo} is created and added
     * @return the conlet tracking info
     */
    @SuppressWarnings("PMD.AvoidSynchronizedStatement")
    protected ConletTrackingInfo trackConlet(ConsoleConnection connection,
            String conletId, ConletTrackingInfo info) {
        ConletTrackingInfo result;
        synchronized (conletInfosByConsoleConnection) {
            Map<String, ConletTrackingInfo> infos
                = conletInfosByConsoleConnection.computeIfAbsent(connection,
                    newKey -> new ConcurrentHashMap<>());
            result = infos.computeIfAbsent(conletId,
                key -> Optional.ofNullable(info)
                    .orElse(new ConletTrackingInfo(conletId)));
        }
        updateRefresh();
        return result;
    }

    /**
     * Helper that provides the storage spaces for this 
     * conlet type in the session.
     *
     * @param session the session
     * @return the spaces, non-transient first
     */
    @SuppressWarnings({ "unchecked", "PMD.AvoidSynchronizedStatement" })
    private Stream<Map<String, S>> typeContexts(Session session) {
        synchronized (session) {
            return List.of(session, session.transientData()).stream()
                .map(context -> ((Map<Class<?>,
                        Map<String, S>>) (Object) context).computeIfAbsent(
                            AbstractConlet.class,
                            k -> new ConcurrentHashMap<>()));
        }
    }

    /**
     * Puts the given web console component state in the session using the 
     * {@link #type()} and the given web console component id as keys.
     * If the state representation implements {@link Serializable},
     * the information is put in the session, else it is put in the
     * session's {@link Session#transientData()}.
     * 
     * @param session the session to use
     * @param conletId the web console component id
     * @param conletState the web console component state
     * @return the component state
     */
    @SuppressWarnings("PMD.AvoidSynchronizedStatement")
    protected S putInSession(Session session, String conletId, S conletState) {
        synchronized (session) {
            var storages = typeContexts(session);
            if (!(conletState instanceof Serializable)) {
                storages = storages.skip(1);
            }
            storages.findFirst().get().put(conletId, conletState);
            return conletState;
        }
    }

    /**
     * Returns the state of this web console component's type 
     * with the given id from the session.
     *
     * @param session the session to use
     * @param conletId the web console component id
     * @return the web console component state
     */
    @SuppressWarnings("PMD.AvoidSynchronizedStatement")
    protected Optional<S> stateFromSession(Session session, String conletId) {
        synchronized (session) {
            return typeContexts(session).map(storage -> storage.get(conletId))
                .filter(data -> data != null).findFirst();
        }
    }

    /**
     * Returns all conlet ids and conlet states of this web console 
     * component's type from the session.
     *
     * @param session the console connection
     * @return the states
     */
    @SuppressWarnings("PMD.AvoidSynchronizedStatement")
    protected Collection<Map.Entry<String, S>>
            statesFromSession(Session session) {
        synchronized (session) {
            return typeContexts(session).flatMap(storage -> storage.entrySet()
                .stream()).collect(Collectors.toList());
        }
    }

    /**
     * Removes the web console component state of the 
     * web console component with the given id from the session. 
     * 
     * @param session the session to use
     * @param conletId the web console component id
     * @return the removed state if state existed
     */
    @SuppressWarnings("PMD.AvoidSynchronizedStatement")
    protected Optional<S> removeState(Session session, String conletId) {
        synchronized (session) {
            return typeContexts(session)
                .map(storage -> storage.remove(conletId))
                .filter(data -> data != null).findFirst();
        }
    }

    /**
     * Checks if the request applies to this component. If so, stops the 
     * event, requests a new conlet id (see {@link #generateInstanceId}). 
     * Stops processing if state for this id already exists (singleton).
     * Otherwise requests new state information 
     * (see {@link #createNewState}) and saves it in the session 
     * (see {@link #putInSession}). Finally {@link #doRenderConlet} is 
     * called and its result is passed to {@link #trackConlet}.
     *
     * @param event the event
     * @param connection the channel
     * @throws Exception the exception
     */
    @Handler
    @SuppressWarnings({ "PMD.SignatureDeclareThrowsException",
        "PMD.AvoidDuplicateLiterals" })
    public final void onAddConletRequest(AddConletRequest event,
            ConsoleConnection connection) throws Exception {
        if (!event.conletType().equals(type())) {
            return;
        }
        event.stop();
        String conletId = type() + TYPE_INSTANCE_SEPARATOR
            + generateInstanceId(event, connection);

        // Check if state already exists (indicates singleton), may not be
        // added again. Only "content conlets" can already have state.
        if (!event.renderAs().contains(RenderMode.Content)
            && stateFromSession(connection.session(), conletId).isPresent()) {
            logger.finer(() -> String.format("Method generateInstanceId "
                + "returns existing id %s when adding conlet.", conletId));
            return;
        }

        // Create new state and track conlet.
        Optional<S> state = createNewState(event, connection, conletId);
        state.ifPresent(s -> putInSession(
            connection.session(), conletId, s));
        event.setResult(conletId);
        trackConlet(connection, conletId, new ConletTrackingInfo(conletId)
            .addModes(doRenderConlet(event, connection, conletId,
                state.orElse(null))));
    }

    /**
     * Checks if the request applies to this component. If so, stops 
     * the event. If the conlet is completely removed from the browser,
     * removes the web console component state from the 
     * browser session. In all cases, it calls {@link #doConletDeleted} 
     * with the state.
     * 
     * @param event the event
     * @param connection the web console connection
     * @throws Exception the exception
     */
    @Handler
    @SuppressWarnings("PMD.SignatureDeclareThrowsException")
    public final void onConletDeleted(ConletDeleted event,
            ConsoleConnection connection) throws Exception {
        if (!type().equals(typeFromId(event.conletId()))) {
            return;
        }
        String conletId = event.conletId();
        Optional<S> model = stateFromSession(connection.session(), conletId);
        var trackingInfo = trackConlet(connection, conletId, null)
            .removeModes(event.renderModes());
        if (trackingInfo.renderedAs().isEmpty()
            || event.renderModes().isEmpty()) {
            removeState(connection.session(), conletId);
            for (Iterator<Entry<ConsoleConnection, Map<String,
                    ConletTrackingInfo>>> csi = conletInfosByConsoleConnection
                        .entrySet().iterator();
                    csi.hasNext();) {
                Map<String, ConletTrackingInfo> infos = csi.next().getValue();
                infos.remove(conletId);
                if (infos.isEmpty()) {
                    csi.remove();
                }
            }
            updateRefresh();
        } else {
            trackConlet(connection, conletId, null)
                .removeModes(event.renderModes());
        }
        event.stop();
        doConletDeleted(event, connection, event.conletId(),
            model.orElse(null));
    }

    /**
     * Called by {@link #onConletDeleted} to propagate the event to derived
     * classes.
     *
     * @param event the event
     * @param channel the channel
     * @param conletId the web console component id
     * @param conletState the conlet's state; may be `null` if the
     * conlet doesn't have associated state information
     * @throws Exception if a problem occurs
     */
    @SuppressWarnings("PMD.SignatureDeclareThrowsException")
    protected void doConletDeleted(ConletDeleted event,
            ConsoleConnection channel,
            String conletId, S conletState)
            throws Exception {
        // May be defined by derived class.
    }

    /**
     * Checks if the request applies to this component by verifying 
     * if the component id starts with {@link #type()} 
     * plus {@link #TYPE_INSTANCE_SEPARATOR}.
     * If the id matches, sets the event's result to `true`, stops the 
     * event and tries to retrieve the model from the session. If this
     * fails, {@link #recreateState} is called as another attempt to
     * obtain state information.
     *  
     * Finally, {@link #doRenderConlet} is called and the result is added
     * to the tracking information. 
     *
     * @param event the event
     * @param connection the web console connection
     * @throws Exception the exception
     */
    @Handler
    @SuppressWarnings("PMD.SignatureDeclareThrowsException")
    public final void onRenderConletRequest(RenderConletRequest event,
            ConsoleConnection connection) throws Exception {
        if (!type().equals(typeFromId(event.conletId()))) {
            return;
        }
        Optional<S> state = stateFromSession(
            connection.session(), event.conletId());
        if (state.isEmpty()) {
            state = recreateState(event, connection, event.conletId());
            state.ifPresent(s -> putInSession(connection.session(),
                event.conletId(), s));
        }
        event.setResult(true);
        event.stop();
        Set<RenderMode> rendered = doRenderConlet(
            event, connection, event.conletId(), state.orElse(null));
        trackConlet(connection, event.conletId(), null).addModes(rendered);
    }

    /**
     * Called by 
     * {@link #onAddConletRequest(AddConletRequest, ConsoleConnection)} and
     * {@link #onRenderConletRequest(RenderConletRequest, ConsoleConnection)} 
     * to complete rendering the web console component.
     * 
     * The 
     *
     * @param event the event
     * @param channel the channel
     * @param conletId the component id
     * @param conletState the conlet's state; may be `null` if the
     * conlet doesn't have associated state information
     * @return the rendered modes
     * @throws Exception the exception
     */
    @SuppressWarnings("PMD.SignatureDeclareThrowsException")
    protected abstract Set<RenderMode> doRenderConlet(
            RenderConletRequestBase<?> event, ConsoleConnection channel,
            String conletId, S conletState)
            throws Exception;

    /**
     * Invokes {@link #doSetLocale(SetLocale, ConsoleConnection, String)}
     * for each web console component in the console connection.
     * 
     * If the vent has the reload flag set, does nothing.
     * 
     * The default implementation fires a 
     *
     * @param event the event
     * @param connection the web console connection
     * @throws Exception the exception
     */
    @Handler
    @SuppressWarnings("PMD.SignatureDeclareThrowsException")
    public void onSetLocale(SetLocale event, ConsoleConnection connection)
            throws Exception {
        if (event.reload()) {
            return;
        }
        for (String conletId : conletIds(connection)) {
            if (!doSetLocale(event, connection, conletId)) {
                event.forceReload();
                break;
            }
        }
    }

    /**
     * Called by {@link #onSetLocale(SetLocale, ConsoleConnection)} for
     * each web console component in the console connection. Derived 
     * classes must send events for updating the representation to 
     * match the new locale.
     * 
     * If the method returns `false` this indicates that the representation 
     * cannot be updated without reloading the web console page.
     * 
     * The default implementation fires a {@link RenderConletRequest}
     * with tracked render modes (one of or both {@link RenderMode#Preview}
     * and {@link RenderMode#View}), thus updating the known representations.
     * (Assuming that "Edit" and "Help" modes are represented with modal 
     * dialogs and therefore locale changes aren't possible while these are 
     * open.) 
     *
     * @param event the event
     * @param channel the channel
     * @param conletId the web console component id
     * @return true, if adaption to new locale without reload is possible
     * @throws Exception the exception
     */
    @SuppressWarnings("PMD.SignatureDeclareThrowsException")
    protected boolean doSetLocale(SetLocale event, ConsoleConnection channel,
            String conletId) throws Exception {
        fire(new RenderConletRequest(event.renderSupport(), conletId,
            trackConlet(channel, conletId, null).renderedAs()),
            channel);
        return true;
    }

    /**
     * If {@link #stateFromSession(Session, String)} returns a model,
     * calls {@link #doUpdateConletState} with the model. 
     *
     * @param event the event
     * @param connection the connection
     * @throws Exception the exception
     */
    @Handler
    @SuppressWarnings("PMD.SignatureDeclareThrowsException")
    public final void onNotifyConletModel(NotifyConletModel event,
            ConsoleConnection connection) throws Exception {
        if (!type().equals(typeFromId(event.conletId()))) {
            return;
        }
        Optional<S> state
            = stateFromSession(connection.session(), event.conletId());
        if (state.isEmpty()) {
            state = recreateState(event, connection, event.conletId());
            state.ifPresent(s -> putInSession(connection.session(),
                event.conletId(), s));
        }
        doUpdateConletState(event, connection, state.orElse(null));
    }

    /**
     * Called by {@link #onNotifyConletModel} to complete handling
     * the notification. The default implementation does nothing.
     * 
     * @param event the event
     * @param channel the channel
     * @param conletState the conlet's state; may be `null` if the
     * conlet doesn't have associated state information
     */
    @SuppressWarnings("PMD.SignatureDeclareThrowsException")
    protected void doUpdateConletState(NotifyConletModel event,
            ConsoleConnection channel, S conletState) throws Exception {
        // Default is to do nothing.
    }

    /**
     * Removes the {@link ConsoleConnection} from the set of tracked 
     * connections. If derived web console components need to perform 
     * extra actions when a console connection is closed, they have to 
     * override {@link #afterOnClosed(Closed, ConsoleConnection)}.
     * 
     * @param event the closed event
     * @param connection the web console connection
     */
    @Handler
    public final void onClosed(Closed<?> event, ConsoleConnection connection) {
        conletInfosByConsoleConnection.remove(connection);
        updateRefresh();
        afterOnClosed(event, connection);
    }

    /**
     * Invoked by {@link #onClosed(Closed, ConsoleConnection)} after
     * the web console connection has been removed from the set of
     * tracked connections. The default implementation does
     * nothing.
     * 
     * @param event the closed event
     * @param connection the web console connection
     */
    protected void afterOnClosed(Closed<?> event,
            ConsoleConnection connection) {
        // Default is to do nothing.
    }

    /**
     * Calls {@link #doRemoveConletType()} if this component
     * is detached.
     *
     * @param event the event
     */
    @Handler
    public void onDetached(Detached event) {
        if (!equals(event.node())) {
            return;
        }
        doRemoveConletType();
    }

    /**
     * Iterates over all connections and fires {@link DeleteConlet}
     * events for all known conlets and a {@link UpdateConletType} 
     * (with no render modes) event.
     */
    protected void doRemoveConletType() {
        conletIdsByConsoleConnection().forEach((connection, conletIds) -> {
            conletIds.forEach(conletId -> {
                connection.respond(
                    new DeleteConlet(conletId, RenderMode.basicModes));
            });
            connection.respond(new UpdateConletType(type()));
        });
    }

    /**
     * The information tracked about web console components that are
     * used by the console. It includes the component's id and the
     * currently rendered views (only preview and view are tracked,
     * with "deletable preview" mapped to "preview").
     */
    protected static class ConletTrackingInfo {
        private final String conletId;
        private final Set<RenderMode> renderedAs;

        /**
         * Instantiates a new conlet tracking info.
         *
         * @param conletId the conlet id
         */
        public ConletTrackingInfo(String conletId) {
            this.conletId = conletId;
            renderedAs = new HashSet<>();
        }

        /**
         * Returns the conlet id.
         *
         * @return the id
         */
        public String conletId() {
            return conletId;
        }

        /**
         * The render modes current used.
         *
         * @return the render modes
         */
        public Set<RenderMode> renderedAs() {
            return renderedAs;
        }

        /**
         * Adds the given modes.
         *
         * @param modes the modes
         * @return the conlet tracking info
         */
        public ConletTrackingInfo addModes(Set<RenderMode> modes) {
            if (modes.contains(RenderMode.Preview)) {
                renderedAs.add(RenderMode.Preview);
            }
            if (modes.contains(RenderMode.View)) {
                renderedAs.add(RenderMode.View);
            }
            return this;
        }

        /**
         * Removes the given modes.
         *
         * @param modes the modes
         * @return the conlet tracking info
         */
        public ConletTrackingInfo removeModes(Set<RenderMode> modes) {
            renderedAs.removeAll(modes);
            return this;
        }

        @Override
        public int hashCode() {
            return conletId.hashCode();
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (getClass() != obj.getClass()) {
                return false;
            }
            ConletTrackingInfo other = (ConletTrackingInfo) obj;
            if (conletId == null) {
                if (other.conletId != null) {
                    return false;
                }
            } else if (!conletId.equals(other.conletId)) {
                return false;
            }
            return true;
        }
    }

    /**
     * Returns a future string providing the result
     * from reading everything from the provided reader. 
     *
     * @param request the request, used to obtain the 
     * {@link ExecutorService} service related with the request being
     * processed
     * @param contentReader the reader
     * @return the future
     */
    public Future<String> readContent(RenderConletRequestBase<?> request,
            Reader contentReader) {
        return readContent(
            request.processedBy().map(pby -> pby.executorService())
                .orElse(Components.defaultExecutorService()),
            contentReader);
    }

    /**
     * Returns a future string providing the result
     * from reading everything from the provided reader. 
     *
     * @param execSvc the executor service for reading the content
     * @param contentReader the reader
     * @return the future
     */
    public Future<String> readContent(ExecutorService execSvc,
            Reader contentReader) {
        return execSvc.submit(() -> {
            StringWriter content = new StringWriter();
            CharBuffer buffer = CharBuffer.allocate(8192);
            try (Reader rdr = new BufferedReader(contentReader)) {
                while (true) {
                    if (rdr.read(buffer) < 0) {
                        break;
                    }
                    buffer.flip();
                    content.append(buffer);
                    buffer.clear();
                }
            } catch (IOException e) {
                throw new IllegalStateException(e);
            }
            return content.toString();
        });
    }

}
