/**
 * Copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* eslint-disable no-console */

import fs from 'fs';
import * as playwright from '../..';
import type { BrowserType } from '../client/browserType';
import type { LaunchServerOptions } from '../client/types';
import { createPlaywright, DispatcherConnection, RootDispatcher, PlaywrightDispatcher } from '../server';
import { PipeTransport } from '../protocol/transport';
import { PlaywrightServer } from '../remote/playwrightServer';
import { gracefullyProcessExitDoNotHang } from '../utils/processLauncher';

export function printApiJson() {
  // Note: this file is generated by build-playwright-driver.sh
  console.log(JSON.stringify(require('../../api.json')));
}

export function runDriver() {
  const dispatcherConnection = new DispatcherConnection();
  new RootDispatcher(dispatcherConnection, async (rootScope, { sdkLanguage }) => {
    const playwright = createPlaywright({ sdkLanguage });
    return new PlaywrightDispatcher(rootScope, playwright);
  });
  const transport = new PipeTransport(process.stdout, process.stdin);
  transport.onmessage = (message: string) => dispatcherConnection.dispatch(JSON.parse(message));
  // Certain Language Binding JSON parsers (e.g. .NET) do not like strings with lone surrogates.
  const isJavaScriptLanguageBinding = !process.env.PW_LANG_NAME || process.env.PW_LANG_NAME === 'javascript';
  const replacer = !isJavaScriptLanguageBinding && (String.prototype as any).toWellFormed ? (key: string, value: any): any => {
    if (typeof value === 'string')
      return value.toWellFormed();
    return value;
  } : undefined;
  dispatcherConnection.onmessage = message => transport.send(JSON.stringify(message, replacer));
  transport.onclose = () => {
    // Drop any messages during shutdown on the floor.
    dispatcherConnection.onmessage = () => {};
    gracefullyProcessExitDoNotHang(0);
  };
  // Ignore the SIGINT signal in the driver process so the parent can gracefully close the connection.
  // We still will destruct everything (close browsers and exit) when the transport pipe closes.
  process.on('SIGINT', () => {
    // Keep the process running.
  });
}

export type RunServerOptions = {
  port?: number,
  host?: string,
  path?: string,
  extension?: boolean,
  maxConnections?: number,
  browserProxyMode?: 'client' | 'tether',
  ownedByTetherClient?: boolean,
};

export async function runServer(options: RunServerOptions) {
  const {
    port,
    host,
    path = '/',
    maxConnections = Infinity,
    extension,
  } = options;
  const server = new PlaywrightServer({ mode: extension ? 'extension' : 'default', path, maxConnections });
  const wsEndpoint = await server.listen(port, host);
  process.on('exit', () => server.close().catch(console.error));
  console.log('Listening on ' + wsEndpoint);
  process.stdin.on('close', () => gracefullyProcessExitDoNotHang(0));
}

export async function launchBrowserServer(browserName: string, configFile?: string) {
  let options: LaunchServerOptions = {};
  if (configFile)
    options = JSON.parse(fs.readFileSync(configFile).toString());
  const browserType = (playwright as any)[browserName] as BrowserType;
  const server = await browserType.launchServer(options);
  console.log(server.wsEndpoint());
}
