import { ObservablePromise } from "../../utils/ObservablePromise";

let npmVersionsObservable: ObservablePromise<string[]> | undefined = undefined;

export function getNpmVersionsSync(
	currentVersion: string | undefined
): string[] {
	if (!npmVersionsObservable) {
		npmVersionsObservable = new ObservablePromise(loadNpmVersions());
	}
	return (
		npmVersionsObservable.value || (currentVersion ? [currentVersion] : [])
	);
}

let npmVersionsPromise: Promise<string[]> | undefined;

export async function getNpmVersions(): Promise<string[]> {
	getNpmVersionsSync(undefined);
	return npmVersionsPromise!;
}

getNpmVersions();

async function loadNpmVersions(): Promise<string[]> {
	if (npmVersionsPromise === undefined) {
		npmVersionsPromise = _getNpmVersions();
	}
	return npmVersionsPromise;
}

async function _getNpmVersions(): Promise<string[]> {
	// newest first
	const data = (await (
		await fetch("https://registry.npmjs.org/monaco-editor")
	).json()) as { versions: Record<string, {}> };

	const versions = Object.keys(data.versions).reverse();

	const brokenVersions = new Set([
		"0.10.1",
		"0.10.0",
		"0.9.0",
		"0.8.3",
		"0.8.2",
		"0.8.1",
		"0.8.0",
		"0.7.0",
		"0.6.1",
		"0.6.0",
		"0.5.3",
		"0.5.1",
		"0.5.0",
		"0.4.0",
		"0.3.1",
		"0.2.4",
		"0.2.3",
		"0.0.1",
	]);

	const validVersions = versions.filter((v) => !brokenVersions.has(v));

	validVersions.sort((v1, v2) => -compareSemanticVersions(v1, v2));

	return validVersions;
}

function compareSemanticVersions(version1: string, version2: string): number {
	const v1 = version1.split(/[.]/);
	const v2 = version2.split(/[.]/);
	for (let i = 0; i < Math.min(v1.length, v2.length); i++) {
		const isNumber = /^\d+$/.test(v1[i]) && /^\d+$/.test(v2[i]);

		if (isNumber) {
			const n1 = parseInt(v1[i]);
			const n2 = parseInt(v2[i]);
			if (n1 < n2) {
				return -1;
			}
			if (n1 > n2) {
				return 1;
			}
		} else {
			const compare = v1[i].localeCompare(v2[i]);
			if (compare !== 0) {
				return compare;
			}
		}
	}

	return v1.length - v2.length;
}

export async function getVsCodeCommitId(
	monacoEditorVersion: string
): Promise<string | undefined> {
	// TODO https://cdn.jsdelivr.net/npm/monaco-editor@${monacoEditorVersion}/package.json
	let commitId: string | undefined =
		knownVersionVsCodeCommitIds[monacoEditorVersion];
	if (!commitId) {
		const json = (await (
			await fetch(
				`https://cdn.jsdelivr.net/npm/monaco-editor@${monacoEditorVersion}/package.json`
			)
		).json()) as { vscodeCommitId: string | undefined };
		commitId = json.vscodeCommitId;
	}
	return commitId;
}

const knownVersionVsCodeCommitIds: Record<string, string> = {
	"0.11.0": "d0eaa02c0aa629c1707720c2c0fd4d55aa7eb99f",
	"0.11.1": "7c849cb6d2b9ded1a4393a821745f6ba294b3e16",
	"0.12.0": "97524812b8cfb6f1c60ddaca5363397f3a4480f1",
	"0.14.1": "399a80f42ae75cb0efede8d828842f157b3178b3",
	"0.14.2": "dd2a752c09bd6c9f331413feb5865d4fe5f7d804",
	"0.14.3": "f6cf2af556ee0dd15a27e90960787a4e12dcc586",
	"0.16.0": "57f34ad851fd07c0bd4d83da8bb587d806d42ce7",
	"0.16.1": "57f34ad851fd07c0bd4d83da8bb587d806d42ce7",
	"0.16.2": "57f34ad851fd07c0bd4d83da8bb587d806d42ce7",
	"0.18.0": "b5251d43fa8049c32262730e71342ea8fc82da3a",
	"0.19.2": "67f9228713b338f0e79c671dc2356d6427917ff3",
	"0.21.0": "2aa5b9f6fa3fe227491671332210e5165422bea6",
	"0.21.1": "0e297b112830bd5be9cf51a848d9314c3887ec42",
	"0.21.2": "0e297b112830bd5be9cf51a848d9314c3887ec42",
	"0.21.3": "0e297b112830bd5be9cf51a848d9314c3887ec42",
	"0.22.0": "38783a261a6fbf42e60357ca95760519d2076bd9",
	"0.22.1": "38783a261a6fbf42e60357ca95760519d2076bd9",
	"0.22.2": "38783a261a6fbf42e60357ca95760519d2076bd9",
	"0.22.3": "38783a261a6fbf42e60357ca95760519d2076bd9",
	"0.23.0": "49b5e3ba0aef430e24518cf2ebdb0e11edc7f48b",
	"0.29.0": "7d441890e2d441ef06480a5fdc173054ef6e9c79",
	"0.29.1": "7d441890e2d441ef06480a5fdc173054ef6e9c79",
	"0.30.0": "7bd792a094ff1870b8dcd76b68f702db9d4a6a5b",
	"0.30.1": "7bd792a094ff1870b8dcd76b68f702db9d4a6a5b",
	"0.31.0": "7db1a2b88f7557e0a43fec75b6ba7e50b3e9f77e",
	"0.31.0-dev.20211123": "0231d1aa45ba3a32aedafddc988a79363b9c9c6f",
	"0.31.0-dev.20211124": "cdd975671e18fe63b9747b2a506baffd624637dc",
	"0.31.0-dev.20211125": "1e473b624f088fc05269891170a8ffa1c84a35a6",
	"0.31.0-dev.20211126": "d18d093403b12a65350c58a7b0d5771cc1f42aba",
	"0.31.0-dev.20211127": "2eacc42cf3ec8975fc2d1c7089531c44e034a0fd",
	"0.31.0-dev.20211128": "d3a5e09ad2570957ebf0a452cf25c02b8d897769",
	"0.31.0-dev.20211129": "368154bd0b42e3b198decd6e09144cfb30d431e7",
	"0.31.0-dev.20211130": "b9cf83f70bb6b4697b544a97c7056dfd28fccc5f",
	"0.31.0-dev.20211201": "0cc0904c565399781defa830facf43141db8b6f3",
	"0.31.0-dev.20211202": "ff970e88e5240b8a3bb28fe5ce105ef44ffcd4ca",
	"0.31.0-dev.20211203": "cca3e817a2608225d4ae8715308195c91f765bad",
	"0.31.0-dev.20211204": "d09289a2b6dd0e56b2a264a8e9380685498e10d6",
	"0.31.0-dev.20211205": "d09289a2b6dd0e56b2a264a8e9380685498e10d6",
	"0.31.0-dev.20211206": "d09289a2b6dd0e56b2a264a8e9380685498e10d6",
	"0.31.0-dev.20211207": "470cee7f66e0284846ae902308cc35ef69ab7d59",
	"0.31.0-dev.20211208": "530480dcaa227f7c5a8442f47042f4c17f6035fc",
	"0.31.0-dev.20211209": "15415b6de68f3fcfefcd264ac37cc3c3d54a378b",
	"0.31.0-dev.20211210": "a1842c8f633237d6d850ada5772359ad32480655",
	"0.32.0-dev.20211211": "33e3cb4e3547417cdaff5f17e3dc0d3cfab8ff0e",
	"0.32.0-dev.20211212": "ff8f37a79626ede0265788192c406c95131dd7c5",
	"0.32.0-dev.20211213": "ff8f37a79626ede0265788192c406c95131dd7c5",
	"0.32.0-dev.20211214": "f69f6eaddf8a73a904bdec39c5dd3d70213e06dd",
	"0.32.0-dev.20211215": "7e266b2c427222011a66486b2d7ac3e4a9953f11",
	"0.32.0-dev.20211216": "86cbe29ceccef7fd5ca207ccdc90423e8e325946",
	"0.32.0-dev.20211217": "ca17de5011403e977648e4527c85724ab31ceb43",
	"0.32.0-dev.20211218": "807bf598bea406dcb272a9fced54697986e87768",
	"0.32.0-dev.20211219": "807bf598bea406dcb272a9fced54697986e87768",
	"0.32.0-dev.20211220": "807bf598bea406dcb272a9fced54697986e87768",
	"0.32.0-dev.20211221": "3a73bf805c0396d9e578f73586b90de63811dca2",
	"0.32.0-dev.20211222": "59c4f825c01fb9e511cafc35b25514420f3b0e19",
	"0.32.0-dev.20211223": "21c73fe3edb90ccb5f72cdcb934b1eb6786531fc",
	"0.32.0-dev.20211229": "3da17db4d9dc3bd271641415ac1272e9aba7639f",
	"0.32.0-dev.20211230": "4d4c4acd5f20f92d90f632f96e399ae43db2bf17",
	"0.32.0-dev.20211231": "0f265a3651ab227764e30dfe50ae967d34e5890d",
	"0.32.0-dev.20220101": "25594310c07af385d21c5fabc7bb3bd70d92175b",
	"0.32.0-dev.20220102": "25594310c07af385d21c5fabc7bb3bd70d92175b",
	"0.32.0-dev.20220103": "faaeb6be4130ccd9d0e4379f6bde4ab919b23990",
	"0.32.0-dev.20220104": "e5d7e8f1e61fcbff3f4674d959d1436596aa51af",
	"0.32.0-dev.20220105": "147e502c3de33d40124c00a85e58537646088fe9",
	"0.32.0-dev.20220106": "d622a8d39fa45826d43208e94894ea45126bede4",
	"0.32.0-dev.20220110": "01a4170028fb6e7b9079942fadb7d4398fc4013c",
	"0.32.0-dev.20220111": "8d53445b8d30d0dc80e2fa2af0ecd82dbb746d7b",
	"0.32.0-dev.20220112": "f60e94bc836b246f6cde800267ca509c3a874986",
	"0.32.0-dev.20220113": "4b42b79bdc97bdbbc979816377dcf4094e5146cd",
	"0.32.0-dev.20220124": "87123c49a1ecda93fde81eba6a6c33a2aef63c74",
	"0.32.0-dev.20220125": "2a3b916cc07c2101889f1d882c6a63c7c87ee209",
	"0.32.0-dev.20220126": "4454a14dd28cd93848b138a5da412d7442d04047",
	"0.32.0-dev.20220128": "b8178e3e2f3f620ebdd31dfbfba937ff1b12ee04",
	"0.32.0-dev.20220129": "f3b97af4d9f5dbffc41d8ce224f460977a83334d",
	"0.32.0-dev.20220130": "584d5348da59a02764b1731d9dc0b2c1c272e1e5",
	"0.32.0-dev.20220131": "84163149f3aa4f57b4407199cae52af18f0d8269",
	"0.32.0-dev.20220201": "bd0040c74ded30bef15bd95a0993a574281338d4",
	"0.32.0-dev.20220202": "482daf0366fcde6e079826df895566987af8f547",
	"0.32.0-dev.20220203": "a3d103b87454b716f8c467006774606d00a342c7",
	"0.33.0-dev.20220204": "3b18d5ddb27d0ae267e5e4c0f414c5b48bfac81e",
	"0.33.0-dev.20220207": "49a6db23d96a591510ff7ce9038aed1c2292e351",
	"0.33.0-dev.20220208": "25e06c623e94981371351f767a6071bc70bc7006",
	"0.33.0-dev.20220214": "888677ed95e58f33e8a9c16d6ee259b1e35e190b",
	"0.33.0-dev.20220215": "e8cfc68ce46c5868d53af90064a731233e80c829",
	"0.33.0-dev.20220216": "78ce76b524700cce84521accddd585e9c659d59b",
	"0.33.0-dev.20220217": "86200d542474b26aa28034e0fd5789c45c747d8b",
	"0.33.0-dev.20220218": "82a65e1a8d696e07a922454333f6c7323d06a573",
	"0.33.0-dev.20220220": "68fa627d9aafda0c9722e5d8af40bf8789d4467e",
	"0.33.0-dev.20220221": "a0da9978b10ae0b9307c7f545b50a8500efff7eb",
	"0.33.0-dev.20220222": "34cef6b03c744c2de41fedd8ef70f4ccb8ea7b60",
	"0.33.0-dev.20220223": "32456dc9e7d3194a94dd3ee0322ccf47e461d09c",
	"0.33.0-dev.20220224": "d11dc8c2bff16ed9b4151fc9135819e924d62cd0",
	"0.33.0-dev.20220226": "07601513c73783f8ae2c2e802d9042b0feca4ed8",
	"0.33.0-dev.20220227": "7e57af782039727e21d8e63ce0c671ce9004286b",
	"0.33.0-dev.20220228": "da17c722dac528175480d65f2f96b6c36b81e402",
	"0.33.0-dev.20220301": "4380f8ee6ad9d6062797ed9d29274b7727516b63",
	"0.33.0-dev.20220302": "6f906f35e6ba72b8bc11a46d04e8c0840469dc2a",
	"0.33.0-dev.20220303": "0a94615cd30e430a5991794cb315c2d7081e4c8d",
	"0.33.0-dev.20220306": "b9c773e03e2e3773072ef3f0929bb53351d15a41",
	"0.33.0-dev.20220307": "ce3f26732fdce460f16a439b9bc80a70983bfbc4",
	"0.33.0-dev.20220309": "e4495f7aea4738e675742ff97ec1ef0988edfb50",
	"0.33.0-dev.20220310": "b867e992ceb2f16c44afdcdccc285c65651b9d08",
	"0.33.0-dev.20220311": "5883696c82b441689967ee28e0fa8badbad59041",
	"0.34.0-dev.20220312": "8ca417c5ab3657efe2c0494f5dc288417d937fe4",
	"0.34.0-dev.20220313": "439cf5a431e88587fca0291443eb049258e6c0fa",
	"0.34.0-dev.20220314": "8be0e97a32f9e7c4ca824b5a75c11d47095ab0e6",
	"0.34.0-dev.20220315": "4d8406b8cdcdaf0d21d5c8c4bd9c4d986d02ad65",
	"0.34.0-dev.20220316": "4261e7ccd2e4eacf00af9f3a961df2e33035ed76",
	"0.34.0-dev.20220317": "e767f79f1f87ee4c2e75c267f886909396955812",
	"0.34.0-dev.20220318": "27b1a33d0cd2f91c59846dd30d2dc4fbd4ac6cf3",
	"0.34.0-dev.20220319": "801ad86ce10939dbca554f764e6b767223be8973",
	"0.34.0-dev.20220320": "cb524368c7bbde160fa17306c508868bb1cf654a",
	"0.34.0-dev.20220321": "b442b62f8642be2aba089a6832a86563a4bcc6f1",
	"0.34.0-dev.20220322": "b0966eddca8cc86d3051213b6bf8327ef67ecf6d",
	"0.34.0-dev.20220323": "11e83feb5840a2e9cea8708b1aeeb31137fcb6a2",
	"0.34.0-dev.20220324": "001fb95c3d53ab7cc2ee2d28298808790290aff6",
	"0.34.0-dev.20220325": "b4a4a001d625eb8dcc171aaacf6d0d7fd5c97986",
	"0.34.0-dev.20220326": "1cc6ee1c711858b4f4af204d0044c62b70828ab8",
	"0.34.0-dev.20220327": "1cc6ee1c711858b4f4af204d0044c62b70828ab8",
	"0.34.0-dev.20220328": "dc44ff664989a8653528a5d61a56a42d89a8c838",
	"0.34.0-dev.20220329": "6a19508c8711a09428186e86a1e4d4f406d15680",
	"0.34.0-dev.20220330": "43d0483daee5a528301bdc355a5e3c55453ca4b0",
	"0.34.0-dev.20220331": "6b5f78e4ff09b0c34c45946ba73298802a66326c",
	"0.34.0-dev.20220401": "a51801b4654c5b38472d6142a494a56fb6640225",
	"0.34.0-dev.20220402": "ae0f691839eff7dc0354224f7cff9ed3a128cf54",
	"0.34.0-dev.20220403": "24e73f3551a59166d34d0deecbb623c24af67645",
	"0.34.0-dev.20220404": "e8aefb5a41859bf86f2d36e3616b9086cb8b9b46",
	"0.34.0-dev.20220405": "6508a29d23f1c6a8a32e0a01051e598e4b6091de",
	"0.34.0-dev.20220406": "b261d64b68c6c6a5d261f7088d59dedba51a04d3",
	"0.34.0-dev.20220407": "4e69b30b4c6618e99ffc831bb9441c3e65c6596e",
	"0.34.0-dev.20220408": "497e420d238db172ce1e9b8d834a4aa74f1401e9",
	"0.34.0-dev.20220409": "722855c138ba22d2220c81eb91f3f50f255411eb",
	"0.34.0-dev.20220410": "e358a02a350d6fcccb8810c4f1946ec0bae16c24",
	"0.34.0-dev.20220411": "eab68392736d0766f4d7ea00b421d125aac2d57f",
	"0.34.0-dev.20220412": "0cad0800701fdf67474d20d5bc1b7cf42413c36c",
	"0.34.0-dev.20220413": "fa8d1063f6ab829e848575cf402d8bca74bcc2d4",
	"0.34.0-dev.20220414": "69225f5efec44cd8e26761fca14bd72e9ef6c610",
	"0.34.0-dev.20220415": "a9288be67bab4e9c4dbe62207f01fafc2bdaadcb",
	"0.34.0-dev.20220416": "e6d98a13fcd3bf928d0e755eaae873ac200a98bd",
	"0.34.0-dev.20220418": "e6d98a13fcd3bf928d0e755eaae873ac200a98bd",
	"0.34.0-dev.20220419": "a9010d094a33ed219fda67a65bc6ca636e7f3920",
	"0.34.0-dev.20220420": "89d10043719b48a6e6e5978fe61e93fd2c5a5d82",
	"0.34.0-dev.20220421": "e43b54417f9780b6d470fd173060b308c3429e67",
	"0.34.0-dev.20220422": "ff3d5bb4911dbfd3f3f2851d352132bea32a71e3",
	"0.34.0-dev.20220423": "c22cb87311b5eb1a3bf5600d18733f7485355dc0",
	"0.34.0-dev.20220424": "a6d76c2abb2391e376f66fb67201f831957b7d44",
	"0.34.0-dev.20220425": "fef2fb9a75d6fbd20645e4ab75e9b8d49d3a28c8",
	"0.34.0-dev.20220426": "d006f6d38b3337a17f3cd1fed96c4c308113cdc9",
	"0.34.0-dev.20220427": "685c4823ee24a926e1aefbe25b925057c684294c",
	"0.34.0-dev.20220428": "0cb3913490cb51155caa94fb3dd5cdd542bd8d0f",
	"0.34.0-dev.20220429": "6fd33cc509b45be16f59fd50a1993708e497b014",
	"0.34.0-dev.20220430": "3f4cf3d89c50912d383aa9eab80c9d2dfde669b6",
	"0.34.0-dev.20220502": "f90dd46fc33128827966ee1ef44b29119d36632f",
	"0.34.0-dev.20220503": "39b924d01e8758e00e73d18d4a6a698bbfd9c0fd",
	"0.34.0-dev.20220504": "3be2b5d9524ef9d6fda0c2751ec2d02a181ea6f4",
	"0.34.0-dev.20220505": "3b08d52d0b20df1de3675989befdb39e8dcdd4bd",
	"0.34.0-dev.20220506": "e38b831940a9de39eafc29327492dda092104177",
	"0.34.0-dev.20220507": "2f2ad360ee5bd1f6329d8764cc97a0b290db51ca",
	"0.34.0-dev.20220508": "4a774a5fb42ac9ffa83f0e7ab10deda7a3add8e2",
	"0.34.0-dev.20220509": "4a774a5fb42ac9ffa83f0e7ab10deda7a3add8e2",
	"0.34.0-dev.20220510": "c252f1158d19a52e344e2262c7dc84ff158c094b",
	"0.34.0-dev.20220511": "194eb68a7f1a5569c2a473e2b406742632588b29",
	"0.34.0-dev.20220512": "d209fc34bf8d9a11f730dad4bd3ee5ddbe41ed17",
	"0.34.0-dev.20220513": "5eb8bee36789e343fdab8326744f93ef22c2cab9",
	"0.34.0-dev.20220514": "c23f0305dbf82b2319b198f4dbf3c5d5bc522f15",
	"0.34.0-dev.20220515": "c23f0305dbf82b2319b198f4dbf3c5d5bc522f15",
	"0.34.0-dev.20220516": "bd99c998a14e34dc17a3a745bb60f00a13391564",
	"0.34.0-dev.20220517": "2f19fc5c580b56c4f1e40215432df4c682b3d8a5",
	"0.34.0-dev.20220518": "95b8607540917934d215aa6e2dc8e74cf4db8ad3",
	"0.34.0-dev.20220519": "60e199cda4988e09fd78b1eeb9dce5e9e87e8a1d",
	"0.34.0-dev.20220520": "e9393b096d90a5ac49b1528d1ec60483fc6b993e",
	"0.34.0-dev.20220521": "3649387f3fc1f8bbb9d2f58dc1fe66dd01f50a6a",
	"0.34.0-dev.20220522": "93ec6bd572c0530fe4133d1620b1eed8156e7307",
	"0.34.0-dev.20220523": "d3af646dd02d1c73ecf15b4193781b771cce1e6a",
	"0.34.0-dev.20220524": "6099f88ce75a807021ebe5c56df953be0e52ac93",
	"0.34.0-dev.20220525": "d891b49fc0f8fc5e00591657ca225975952b09ca",
	"0.34.0-dev.20220526": "528ee1ae3daabe30c1307cf9dcd6e77eb96094bc",
	"0.34.0-dev.20220528": "17c4bd39f0fc49886004a2685b8350ab541a1e4d",
	"0.34.0-dev.20220529": "61347c8d4e39de751699bddd97d5334128967cf4",
	"0.34.0-dev.20220530": "58555a1d6ef6b71e08a3679754bae100cc2e759e",
	"0.34.0-dev.20220531": "9a2e8df8110c91cb7dc9271a5f20f366a115528a",
	"0.34.0-dev.20220601": "ccf48c8761abed18c0a92795cb38321a4ad61660",
	"0.34.0-dev.20220602": "83575b8e1d9658d804cc72ee0632217d03a9cb85",
	"0.34.0-dev.20220603": "05de94e4c547a8953d3c002fe034fa4589ebfa97",
	"0.34.0-dev.20220604": "55a970c85af9b6b350d65b50a174c6c6b56ff99c",
	"0.34.0-dev.20220605": "55a970c85af9b6b350d65b50a174c6c6b56ff99c",
	"0.34.0-dev.20220606": "9919c760c21316dcd0efa535db4fd32ef3e8051c",
	"0.34.0-dev.20220607": "714456b9b996e969b77a66179e778abbf49ab896",
	"0.34.0-dev.20220608": "2c0a136079200e864c62989c2bd06e6fc2d7aee0",
	"0.34.0-dev.20220609": "3622010b64abbdf8bab57bc0d163f74fe10f3787",
	"0.34.0-dev.20220610": "dce493cb6e36346ef2714e82c42ce14fc461b15c",
	"0.34.0-dev.20220611": "1ccfebef48170e315209f97154be7bb3d4d4e315",
	"0.34.0-dev.20220612": "210a7c0ec0330275e3a4d85415a1e85d94e96f1b",
	"0.34.0-dev.20220613": "e367b12f19c0c28ac35e1713544867ca35f326c2",
	"0.34.0-dev.20220615": "ee1f284f5f0e4d0212f6c6406c55ebe0c8ac3162",
	"0.34.0-dev.20220616": "6f7c824a826ff0ccaf4de05d6fe0aac3be7bc136",
	"0.34.0-dev.20220617": "3d4a1bc11ecddeafb786d47452f40adafa737d59",
	"0.34.0-dev.20220618": "0203aabaa509499495cb70ce8ca06d5cc9b51462",
	"0.34.0-dev.20220619": "5a175207de7c6d0821b3318cfdbe73bb9448ed57",
	"0.34.0-dev.20220620": "4c72dedb4ad283a569f83a7389468c3ae2c642c3",
	"0.34.0-dev.20220622": "d6e60d30518743f5f338f4b01223bfd147d84fa4",
	"0.34.0-dev.20220623": "4b48304259402b5dd5761389b6f17600f51f2d18",
	"0.34.0-dev.20220624": "45f9ce71fc0c226849fd7678a45c992f9adf9a57",
	"0.34.0-dev.20220625": "09c5a6ed48269c3af355a6b9722e6eb36111a469",
	"0.34.0-dev.20220626": "bc403a0b71f0013c7d3cdb2196e9acff32650095",
	"0.34.0-dev.20220627": "a3ee1bfbd9e2877fd0be8d84804332c76f9c0333",
	"0.34.0-dev.20220628": "5feeb83b0c0d57403f02a740b53bd02759561745",
	"0.34.0-dev.20220629": "ff744e89a63dcc65809d7686e1dc92946e227cc1",
	"0.34.0-dev.20220630": "f1abeeab7e51df997bbb0f1ff900321ad2f21ac0",
	"0.34.0-dev.20220701": "c7c3174e136c091ace576fc1cab0bd10caa486a4",
	"0.34.0-dev.20220702": "562ec1e22c1f5a5038e092f2de2a894234abe366",
	"0.34.0-dev.20220703": "80f4dd5bcfe3671f42b3ad57df866f5abebf0233",
	"0.34.0-dev.20220704": "2fcd34a41b11b49f0d0f778752a117de292e25e0",
	"0.34.0-dev.20220705": "866f22e2d17f5da33803b4f41dc681f5a5a3ef5f",
	"0.34.0-dev.20220706": "2d0d5b28a6d9c9233d46d27252f79a728a25d30c",
	"0.34.0-dev.20220720": "d0769c7e7f8cfbf6481c9196a197021a833176fc",
	"0.34.0-dev.20220721": "e0e7ad368e5c2a0ae28469d0292ef994d7a49b97",
	"0.34.0-dev.20220722": "13ba7bb446a638d37ebccb1a7d74e31c32bb9790",
	"0.34.0-dev.20220723": "a3c4e5ffcee85677f1dd7e208cbd32bc84fee4d5",
	"0.34.0-dev.20220724": "28fa84f03524d0ec6a3640f9e322db3d11b8dee5",
	"0.34.0-dev.20220725": "21ca44d35b0291a677188c70c70950aec47cb3ce",
	"0.34.0-dev.20220726": "d2a8be73559a3cdc86260973994f5cc42ca9044a",
	"0.34.0-dev.20220727": "13b1427953e2d1b0e691cb60cdd2163a1a78d027",
	"0.34.0-dev.20220729": "743b016722db90df977feecde0a4b3b4f58c2a4c",
	"0.34.0-dev.20220730": "07feb2bbd328d7b71d0576c6a0313b4b047c71bd",
	"0.34.0-dev.20220731": "07feb2bbd328d7b71d0576c6a0313b4b047c71bd",
	"0.34.0-dev.20220801": "12896eef4f88fcea42a6b40b00884a77d2772a14",
	"0.34.0-dev.20220802": "0ea1cfd5d252245aef711b552966eb567f4a5653",
	"0.34.0-dev.20220803": "8125126a0324c115c06ca0b4390e1a58604831f3",
	"0.35.0-dev.20220805": "73fd3f11032e7b83c2ae011b5516e6ddd19e3db2",
	"0.35.0-dev.20220806": "cb79d015557d546d0d48cea97d7fc3ac0d3dc590",
	"0.35.0-dev.20220807": "66aef7979d4695d31750ada0748f316b76500a80",
	"0.35.0-dev.20220808": "f1c5243126a1add69272ac9d0d4b4ac1db64f290",
	"0.35.0-dev.20220809": "edb9d7d52891d52510891bcb4c5fbb5dab4209c5",
	"0.35.0-dev.20220810": "dced70bbf36d3c53c08e791da1791ac7fc42519b",
	"0.35.0-dev.20220811": "ef65ac1ba57f57f2a3961bfe94aa20481caca4c6",
	"0.35.0-dev.20220812": "eebdf8174b087979ae6af103f6f2e2f6f9062056",
	"0.35.0-dev.20220813": "1a582f7c079d1eb4c89d4f637da2fc2fcb688b31",
	"0.35.0-dev.20220814": "880ffb665816d3cbeb43be4f7e7beeaf223fe794",
	"0.35.0-dev.20220815": "8b46432df318394789e65aa45423c3983b9db847",
	"0.35.0-dev.20220826": "93e346a44990e2e0132aaddbc944edda0bbd60d7",
	"0.35.0-dev.20220827": "b5a1bb34e9c6b77c2a8ccf54bcdbb4bac4d420cb",
	"0.35.0-dev.20220829": "e18726faa9901880804716bc009476328902a322",
	"0.35.0-dev.20220830": "e9d0bed9e577617a5a13b631c33d54807bcb1af1",
	"0.35.0-dev.20220929": "46afa5b1080944f9e06557930367d8b5475e6453",
	"0.35.0-dev.20221001": "61f8dee571efac71035000f4c57f1c42ada04e46",
	"0.35.0-dev.20221002": "61f8dee571efac71035000f4c57f1c42ada04e46",
	"0.35.0-dev.20221003": "ef802ebe7af412860387ac984682f6c86880af5c",
	"0.35.0-dev.20221004": "c9338005f7210e26c529891f21a85d26a2bebc87",
	"0.35.0-dev.20221005": "a90f6a16799973fe414ea90308cd256ffcad8128",
	"0.35.0-dev.20221006": "a71ef85de11ae41776ba46dd3387c2fff83cf1f2",
	"0.35.0-dev.20221007": "f91b3ac8ca669c68a4532990265b83ecc628678c",
	"0.35.0-dev.20221008": "6483e39fb7b0ae0aeae6bb00ff42dbed3ceb618b",
	"0.35.0-dev.20221009": "c3f5861ca2585a020a974c28aae45d1bce7edb62",
	"0.35.0-dev.20221010": "150b7f9d0ba9d3cdd294a47b164bd22fb5017ed2",
	"0.35.0-dev.20221011": "3f2adbfe3e6d07c60bc4c61b4ff7bdcf1c451792",
	"0.35.0-dev.20221012": "8c946e23f45f39fdd982cff1e1fce02a8e3dc719",
	"0.35.0-dev.20221013": "65270235646614cd51f2152ea15da576ba83a1c3",
	"0.35.0-dev.20221014": "493d27bf48037947094e76fa34998eb45fd2381c",
	"0.35.0-dev.20221015": "cbee57d41509512ba044b7cb00210a590c187679",
	"0.35.0-dev.20221016": "81d2ccccc9c153e088df3f3805f8255d36f82f90",
	"0.35.0-dev.20221017": "f9a0fdeff2e704b37f3ec19a2921ee3e38359073",
	"0.35.0-dev.20221018": "b879d03f5c3f91274a45cf49cb6e8f4ac1182cca",
	"0.35.0-dev.20221019": "9d3fbb3d9a50055be0a8c6d721625d02c9de492d",
	"0.35.0-dev.20221020": "929c645c9a3727541c8fb30f4d62765f5c672ff9",
	"0.35.0-dev.20221021": "da1406166702eb790c2629f4c0ced8eaa503cc88",
	"0.35.0-dev.20221022": "2936f3cab8468cc5f4139a23b462313dcb652c9d",
	"0.35.0-dev.20221023": "d1f38520db76f0e80e3cdcbb35b95651afe802ae",
	"0.35.0-dev.20221024": "d1d27e4b50d2c042b9cf29f3ab4d53523ed54dd3",
	"0.35.0-dev.20221026": "b39fd56213e01017975dc246befd4a08435ac6a3",
	"0.35.0-dev.20221027": "9ad069358e9f09d1f49a90db36b73c953505438a",
	"0.35.0-dev.20221028": "6a10de5508f9b4eebbb6769a99f84f6220d96fa7",
	"0.35.0-dev.20221029": "7cb1b9d9f00550ffa914432d7ce8ac85d62655b8",
	"0.35.0-dev.20221030": "3e407526a1e2ff22cacb69c7e353e81a12f41029",
	"0.35.0-dev.20221031": "3e407526a1e2ff22cacb69c7e353e81a12f41029",
	"0.35.0-dev.20221101": "447e61a31fc36ad3c14e7776973b290826c9c40a",
	"0.35.0-dev.20221102": "55a95c7086307225f7562cbb813383a806b45a9f",
	"0.35.0-dev.20221103": "86f0f9bfa268fe21af7efae37d8f0da1dba8137f",
	"0.35.0-dev.20221104": "e764c5b816a99eb8673b2fc4fc028827f202872f",
	"0.35.0-dev.20221105": "5cdde326a1a07afcefaaaad1a20551e3ca4f3072",
	"0.35.0-dev.20221106": "5da2bd6a3da95e2b613e8d104b861d5a65ae47d9",
	"0.35.0-dev.20221107": "f9fb40f63310323cd80e2a06fca36d9df2db9e57",
	"0.35.0-dev.20221108": "d5125cb4c5bebc9fdd4ff9d96ede5157d73aad7b",
	"0.35.0-dev.20221109": "96db0b339360bc72d27d869d81a557540ad5f800",
	"0.35.0-dev.20221110": "9d5741f01a67beea273121615b2d015fc298d9ea",
	"0.35.0-dev.20221111": "9f2be311ff72dc348c794d6d47aeb8cda881ebf4",
	"0.35.0-dev.20221112": "15903434dafa2ecad19ade641ef5f91e46aad3ac",
	"0.35.0-dev.20221113": "964dc545abd63c5f6f6c42755f1c11c2c912a0b5",
	"0.35.0-dev.20221114": "a62879bb46c84d0936798cf06debf5f92b04e4b4",
	"0.35.0-dev.20221116": "39ff7eb93b513976b85b69085414cab11f8ed4cc",
	"0.35.0-dev.20221117": "d9a70c4236a8bde4fe7435df3ecdd333c7bc939c",
	"0.35.0-dev.20221122": "f730160c5d5fe36a7ad8db51f74f40a9d316e8b2",
	"0.35.0-dev.20221123": "58e7c7b8865e3c3ea77055461ffb5d656a7a46af",
	"0.35.0-dev.20221124": "2eca6d38dede985bf6180cca64d43c16a02e9a13",
	"0.35.0-dev.20221126": "fd8c9037af02c02bf0c38af008a69d78f033f942",
	"0.35.0-dev.20221127": "ac084d723b7405591d2110fe374648345ecb8ce6",
	"0.35.0-dev.20221128": "075f04bf3901db78dd02c032420a52e5b6c91835",
	"0.35.0-dev.20221129": "8b9891739b703b50547f506a7e2bc9565e52beff",
	"0.35.0-dev.20221130": "bb5150c70d8f825265869e26bb19aa234d9b855b",
	"0.35.0-dev.20221201": "7f3301d361f4805c404603dd4d76fcfd36347419",
	"0.35.0-dev.20230104": "4acf2d9b46b75748ae687cf3b2952a0799679873",
	"0.35.0-dev.20230105": "7b1f8ffe0f9566e6ce383d007ed21301f0c00bd4",
	"0.35.0-dev.20230106": "c8917bb3274e8d1b2d3b9967395d5b394d0b2d76",
	"0.35.0-dev.20230107": "95bec874d30188cd425cd7ee80f330a59c93c1e6",
	"0.35.0-dev.20230108": "be54d2be91cc7443d0c44ed2a483475b12747b2d",
	"0.35.0-dev.20230109": "eaa9b1b70eaad291980a3e74d1748ca7d9bbd195",
	"0.35.0-dev.20230110": "5b60b8a53e0bdbd8664b7c050b51b1732e30aca7",
	"0.35.0-dev.20230111": "a40ce8aa30c52ef0561ca2d136f8c7d975f73f0d",
	"0.35.0-dev.20230112": "6d40104789d03d41b8866a1e57847dae14c5cf0d",
	"0.35.0-dev.20230117": "4d882322ca0dc62625ebbd5076ebb82a1d250a42",
	"0.35.0-dev.20230118": "ab0f89141e701154afd5728aa893ec94704003bf",
	"0.35.0-dev.20230119": "b3b49029a5ae0d15b15225720fd352e9308d1433",
	"0.35.0-dev.20230120": "514900e23a39ce70e432db4395d892514a376149",
	"0.35.0-dev.20230121": "bcc823272187b5a723631fa64815c79b0299ee88",
	"0.35.0-dev.20230122": "f6c9f22a5bb61eaa307d23671450ec8bc01b0cbd",
	"0.35.0-dev.20230123": "39c7015c93e7f78352dc92a4de46f74c288dc0f3",
	"0.35.0-dev.20230124": "516bd06dadfc4bffed0b1c48794fb71fc128c486",
	"0.35.0-dev.20230125": "5beebeb17067cb39a6f43c231d8f91f555509ff5",
	"0.35.0-dev.20230126": "07d6f5b35fac559504dec65e3073d796b2d25a55",
	"0.35.0-dev.20230127": "acbbdaa41a459aab86072e8d97255cd2ea1d9529",
	"0.35.0-dev.20230128": "e6100dbf2f839a87232100b09af61d2d0717e213",
	"0.35.0-dev.20230129": "339fd630e60b952617b1dfe3083eed2d5fa1c001",
	"0.35.0-dev.20230131": "8db5e04ab2b0cf1e004b19ff0362a45fe0c27ac4",
	"0.35.0-dev.20230201": "70703fe42f5b1d9eede3f2737167ed8c846f3af5",
	"0.35.0-dev.20230202": "3f2dc4202800cc82184fed8fe382cd52049c8358",
	"0.35.0-dev.20230204": "91fccee240d37132a12cc0818d37f6064a93c80f",
	"0.35.0-dev.20230205": "57f26d2907680c37943c60e7aa9bce76d658574e",
	"0.35.0-dev.20230206": "4938e126115c6d70fc8eca4f78dd845aacb2620b",
	"0.35.0-dev.20230208": "5a46f8a57e85478eb51661050ba81a2cc14c568b",
	"0.36.0-dev.20230208": "5a46f8a57e85478eb51661050ba81a2cc14c568b",
	"0.36.0-dev.20230209": "5a46f8a57e85478eb51661050ba81a2cc14c568b",
	"0.31.1": "5a1b4999493d49c857497ad481d73a737439f305",
	"0.32.0": "5554b12acf27056905806867f251c859323ff7e9",
	"0.32.1": "8ad5e3bceab16a4d0856c43a374b511dffb1e795",
	"0.33.0": "c722ca6c7eed3d7987c0d5c3df5c45f6b15e77d1",
	"0.34.0": "4b8a47f3570a4a05ace9d00ae0df044b55befcd5",
	"0.34.1": "0316a754aa4c25208bef91937efbce2ab1e3ce37",
	"0.35.0": "b29a89cbf98df4a8e5079a992c313ce9a1749d55",
};
