/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.core.network;

import java.util.Arrays;
import java.util.Iterator;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.californium.core.coap.CoAP;
import org.eclipse.californium.core.coap.EmptyMessage;
import org.eclipse.californium.core.coap.MessageObserverAdapter;
import org.eclipse.californium.core.coap.Request;
import org.eclipse.californium.core.coap.Response;
import org.eclipse.californium.core.network.Exchange;
import org.eclipse.californium.core.network.ExchangeObserver;
import org.eclipse.californium.core.network.Matcher;
import org.eclipse.californium.core.network.config.NetworkConfig;
import org.eclipse.californium.core.network.deduplication.Deduplicator;
import org.eclipse.californium.core.network.deduplication.DeduplicatorFactory;
import org.eclipse.californium.core.observe.InMemoryObservationStore;
import org.eclipse.californium.core.observe.NotificationListener;
import org.eclipse.californium.core.observe.Observation;
import org.eclipse.californium.core.observe.ObservationStore;
import org.eclipse.californium.core.observe.ObserveRelation;
import org.eclipse.californium.elements.CorrelationContext;

public class UdpMatcher
implements Matcher {
    private static final Logger LOGGER = Logger.getLogger(Matcher.class.getCanonicalName());
    private final ConcurrentHashMap<Exchange.KeyMID, Exchange> exchangesByMID;
    private final ConcurrentHashMap<Exchange.KeyToken, Exchange> exchangesByToken;
    private final ConcurrentHashMap<Exchange.KeyUri, Exchange> ongoingExchanges;
    private final ExchangeObserver exchangeObserver = new ExchangeObserverImpl();
    private final AtomicInteger currendMID;
    private final Deduplicator deduplicator;
    private final boolean useStrictResponseMatching;
    private final int tokenSizeLimit;
    private final Level healthStatusLevel;
    private final int healthStatusInterval;
    private boolean started = false;
    private ScheduledExecutorService executor;
    private NotificationListener notificationListener;
    private final ObservationStore observationStore;

    public UdpMatcher(NetworkConfig config) {
        this(config, null, new InMemoryObservationStore());
    }

    public UdpMatcher(NetworkConfig config, NotificationListener notificationListener, ObservationStore observationStore) {
        this.notificationListener = notificationListener;
        this.observationStore = observationStore;
        this.exchangesByMID = new ConcurrentHashMap();
        this.exchangesByToken = new ConcurrentHashMap();
        this.ongoingExchanges = new ConcurrentHashMap();
        DeduplicatorFactory factory = DeduplicatorFactory.getDeduplicatorFactory();
        this.deduplicator = factory.createDeduplicator(config);
        this.tokenSizeLimit = config.getInt("TOKEN_SIZE_LIMIT");
        this.useStrictResponseMatching = config.getBoolean("USE_STRICT_RESPONSE_MATCHING");
        boolean randomMID = config.getBoolean("USE_RANDOM_MID_START");
        this.currendMID = randomMID ? new AtomicInteger(new Random().nextInt(65536)) : new AtomicInteger(0);
        if (LOGGER.isLoggable(Level.CONFIG)) {
            String msg = "Matcher uses " + "USE_RANDOM_MID_START" + "=" + randomMID + ", " + "TOKEN_SIZE_LIMIT" + "=" + this.tokenSizeLimit + " and " + "USE_STRICT_RESPONSE_MATCHING" + "=" + this.useStrictResponseMatching;
            LOGGER.config(msg);
        }
        this.healthStatusLevel = Level.parse(config.getString("HEALTH_STATUS_PRINT_LEVEL"));
        this.healthStatusInterval = config.getInt("HEALTH_STATUS_INTERVAL");
    }

    @Override
    public synchronized void start() {
        if (this.started) {
            return;
        }
        if (this.executor == null) {
            throw new IllegalStateException("Matcher has no executor to schedule exchange removal");
        }
        this.started = true;
        this.deduplicator.start();
        if (LOGGER.isLoggable(this.healthStatusLevel)) {
            this.executor.scheduleAtFixedRate(new Runnable(){

                @Override
                public void run() {
                    LOGGER.log(UdpMatcher.this.healthStatusLevel, "Matcher state: {0} exchangesByMID, {1} exchangesByToken, {2} ongoingExchanges", new Object[]{UdpMatcher.this.exchangesByMID.size(), UdpMatcher.this.exchangesByToken.size(), UdpMatcher.this.ongoingExchanges.size()});
                }
            }, this.healthStatusInterval, this.healthStatusInterval, TimeUnit.SECONDS);
        }
    }

    @Override
    public synchronized void stop() {
        if (!this.started) {
            return;
        }
        this.started = false;
        this.deduplicator.stop();
        this.clear();
    }

    @Override
    public synchronized void setExecutor(ScheduledExecutorService executor) {
        this.deduplicator.setExecutor(executor);
        this.executor = executor;
    }

    @Override
    public void sendRequest(Exchange exchange, final Request request) {
        Exchange.KeyToken idByToken;
        if (request.getMID() == -1) {
            request.setMID(this.currendMID.getAndIncrement() % 65536);
        }
        Exchange.KeyMID idByMID = new Exchange.KeyMID(request.getMID());
        if (request.getToken() == null) {
            idByToken = this.createUnusedToken();
            request.setToken(idByToken.token);
            if (exchange.getRequest() != null && exchange.getRequest().getToken() == null) {
                exchange.getRequest().setToken(idByToken.token);
            }
        } else {
            idByToken = new Exchange.KeyToken(request.getToken());
            if (!(exchange.getFailedTransmissionCount() > 0 || request.getOptions().hasBlock1() || request.getOptions().hasBlock2() || request.getOptions().hasObserve() || this.exchangesByToken.get(idByToken) == null)) {
                LOGGER.log(Level.WARNING, "Manual token overrides existing open request: {0}", idByToken);
            }
        }
        if (request.getOptions().hasObserve() && request.getOptions().getObserve() == 0 && (!request.getOptions().hasBlock2() || request.getOptions().getBlock2().getNum() == 0 && !request.getOptions().getBlock2().isM())) {
            this.observationStore.add(new Observation(request, null));
            request.addMessageObserver(new MessageObserverAdapter(){

                @Override
                public void onCancel() {
                    UdpMatcher.this.observationStore.remove(request.getToken());
                }

                @Override
                public void onReject() {
                    UdpMatcher.this.observationStore.remove(request.getToken());
                }

                @Override
                public void onTimeout() {
                    UdpMatcher.this.observationStore.remove(request.getToken());
                }
            });
        }
        exchange.setObserver(this.exchangeObserver);
        LOGGER.log(Level.FINE, "Tracking open request using {0}, {1}", new Object[]{idByMID, idByToken});
        this.exchangesByMID.put(idByMID, exchange);
        this.exchangesByToken.put(idByToken, exchange);
    }

    @Override
    public void sendResponse(Exchange exchange, Response response) {
        ObserveRelation relation;
        if (response.getMID() == -1) {
            response.setMID(this.currendMID.getAndIncrement() % 65536);
        }
        response.setToken(exchange.getCurrentRequest().getToken());
        if ((response.getType() == CoAP.Type.CON || response.getType() == CoAP.Type.ACK) && (relation = exchange.getRelation()) != null) {
            this.removeNotificationsOf(relation);
        }
        if (response.getOptions().hasBlock2()) {
            Request request = exchange.getCurrentRequest();
            Exchange.KeyUri idByUri = new Exchange.KeyUri(request.getURI(), response.getDestination().getAddress(), response.getDestinationPort());
            if (exchange.getResponseBlockStatus() != null && !response.getOptions().hasObserve()) {
                if (this.ongoingExchanges.put(idByUri, exchange) == null) {
                    LOGGER.log(Level.FINE, "Ongoing Block2 started late, storing {0} for {1}", new Object[]{idByUri, request});
                } else {
                    LOGGER.log(Level.FINE, "Ongoing Block2 continued, storing {0} for {1}", new Object[]{idByUri, request});
                }
            } else {
                LOGGER.log(Level.FINE, "Ongoing Block2 completed, cleaning up {0} for {1}", new Object[]{idByUri, request});
                this.ongoingExchanges.remove(idByUri);
            }
        }
        if (response.getType() == CoAP.Type.CON || response.getType() == CoAP.Type.NON) {
            Exchange.KeyMID idByMID = new Exchange.KeyMID(response.getMID());
            this.exchangesByMID.put(idByMID, exchange);
        }
        if (response.getType() != CoAP.Type.CON && response.isLast()) {
            exchange.setComplete();
        }
    }

    @Override
    public void sendEmptyMessage(Exchange exchange, EmptyMessage message) {
        message.setToken(new byte[0]);
        if (message.getType() == CoAP.Type.RST && exchange != null) {
            exchange.setComplete();
        }
    }

    @Override
    public Exchange receiveRequest(Request request) {
        Exchange.KeyMID idByMID = Exchange.KeyMID.fromInboundMessage(request);
        if (!request.getOptions().hasBlock1() && !request.getOptions().hasBlock2()) {
            Exchange exchange = new Exchange(request, Exchange.Origin.REMOTE);
            Exchange previous = this.deduplicator.findPrevious(idByMID, exchange);
            if (previous == null) {
                exchange.setObserver(this.exchangeObserver);
                return exchange;
            }
            LOGGER.log(Level.FINER, "Duplicate request: {0}", request);
            request.setDuplicate(true);
            return previous;
        }
        Exchange.KeyUri idByUri = new Exchange.KeyUri(request.getURI(), request.getSource().getAddress(), request.getSourcePort());
        LOGGER.log(Level.FINE, "Looking up ongoing exchange for {0}", idByUri);
        Exchange ongoing = this.ongoingExchanges.get(idByUri);
        if (ongoing != null) {
            Exchange prev = this.deduplicator.findPrevious(idByMID, ongoing);
            if (prev != null) {
                LOGGER.log(Level.FINER, "Duplicate ongoing request: {0}", request);
                request.setDuplicate(true);
            } else if (ongoing.getCurrentResponse() != null && ongoing.getCurrentResponse().getType() != CoAP.Type.ACK && !ongoing.getCurrentResponse().getOptions().hasObserve()) {
                idByMID = new Exchange.KeyMID(ongoing.getCurrentResponse().getMID());
                LOGGER.log(Level.FINE, "Ongoing exchange got new request, cleaning up {0}", idByMID);
                this.exchangesByMID.remove(idByMID);
            }
            return ongoing;
        }
        Exchange exchange = new Exchange(request, Exchange.Origin.REMOTE);
        Exchange previous = this.deduplicator.findPrevious(idByMID, exchange);
        if (previous == null) {
            LOGGER.log(Level.FINER, "New ongoing request, storing {0} for {1}", new Object[]{idByUri, request});
            exchange.setObserver(this.exchangeObserver);
            this.ongoingExchanges.put(idByUri, exchange);
            return exchange;
        }
        LOGGER.log(Level.FINER, "Duplicate initial request: {0}", request);
        request.setDuplicate(true);
        return previous;
    }

    @Override
    public Exchange receiveResponse(Response response, CorrelationContext responseContext) {
        Exchange prev;
        Observation obs;
        Exchange.KeyMID idByMID = response.getType() == CoAP.Type.ACK ? new Exchange.KeyMID(response.getMID()) : Exchange.KeyMID.fromInboundMessage(response);
        Exchange.KeyToken idByToken = new Exchange.KeyToken(response.getToken());
        Exchange exchange = this.exchangesByToken.get(idByToken);
        if (exchange == null && this.observationStore != null && (obs = this.observationStore.get(response.getToken())) != null) {
            final Request request = obs.getRequest();
            request.setDestination(response.getSource());
            request.setDestinationPort(response.getSourcePort());
            exchange = new Exchange(request, Exchange.Origin.LOCAL, obs.getContext());
            exchange.setRequest(request);
            exchange.setObserver(this.exchangeObserver);
            request.addMessageObserver(new MessageObserverAdapter(){

                @Override
                public void onTimeout() {
                    UdpMatcher.this.observationStore.remove(request.getToken());
                }

                @Override
                public void onResponse(Response response) {
                    UdpMatcher.this.notificationListener.onNotification(request, response);
                }

                @Override
                public void onReject() {
                    UdpMatcher.this.observationStore.remove(request.getToken());
                }

                @Override
                public void onCancel() {
                    UdpMatcher.this.observationStore.remove(request.getToken());
                }
            });
        }
        if (exchange == null) {
            if (response.getType() != CoAP.Type.ACK) {
                prev = this.deduplicator.find(idByMID);
                if (prev != null) {
                    LOGGER.log(Level.FINER, "Received response for already completed exchange: {0}", response);
                    response.setDuplicate(true);
                    return prev;
                }
            } else {
                LOGGER.log(Level.FINER, "Discarding unmatchable piggy-backed response from [{0}:{1}]: {2}", new Object[]{response.getSource(), response.getSourcePort(), response});
            }
            return null;
        }
        if (this.isResponseRelatedToRequest(exchange, responseContext)) {
            prev = this.deduplicator.findPrevious(idByMID, exchange);
            if (prev != null) {
                LOGGER.log(Level.FINER, "Received duplicate response for open exchange: {0}", response);
                response.setDuplicate(true);
            } else {
                idByMID = new Exchange.KeyMID(exchange.getCurrentRequest().getMID());
                this.exchangesByMID.remove(idByMID);
                LOGGER.log(Level.FINE, "Closed open request [{0}]", idByMID);
            }
            if (response.getType() == CoAP.Type.ACK && exchange.getCurrentRequest().getMID() != response.getMID()) {
                LOGGER.log(Level.WARNING, "Possible MID reuse before lifetime end for token [{0}], expected MID {1} but received {2}", new Object[]{response.getTokenString(), exchange.getCurrentRequest().getMID(), response.getMID()});
            }
            return exchange;
        }
        LOGGER.log(Level.INFO, "Ignoring potentially forged response for token {0} with non-matching correlation context", idByToken);
        return null;
    }

    private boolean isResponseRelatedToRequest(Exchange exchange, CorrelationContext responseContext) {
        if (exchange.getCorrelationContext() == null) {
            return true;
        }
        if (exchange.getCorrelationContext().get("DTLS_SESSION_ID") != null) {
            if (this.useStrictResponseMatching) {
                return this.isResponseStrictlyRelatedToDtlsRequest(exchange.getCorrelationContext(), responseContext);
            }
            return this.isResponseRelatedToDtlsRequest(exchange.getCorrelationContext(), responseContext);
        }
        return exchange.getCorrelationContext().equals(responseContext);
    }

    private boolean isResponseRelatedToDtlsRequest(CorrelationContext requestContext, CorrelationContext responseContext) {
        if (responseContext == null) {
            return false;
        }
        return requestContext.get("DTLS_SESSION_ID").equals(responseContext.get("DTLS_SESSION_ID")) && requestContext.get("DTLS_CIPHER").equals(responseContext.get("DTLS_CIPHER"));
    }

    private boolean isResponseStrictlyRelatedToDtlsRequest(CorrelationContext requestContext, CorrelationContext responseContext) {
        if (responseContext == null) {
            return false;
        }
        return requestContext.get("DTLS_SESSION_ID").equals(responseContext.get("DTLS_SESSION_ID")) && requestContext.get("DTLS_EPOCH").equals(responseContext.get("DTLS_EPOCH")) && requestContext.get("DTLS_CIPHER").equals(responseContext.get("DTLS_CIPHER"));
    }

    @Override
    public Exchange receiveEmptyMessage(EmptyMessage message) {
        Exchange.KeyMID idByMID = new Exchange.KeyMID(message.getMID());
        Exchange exchange = this.exchangesByMID.get(idByMID);
        if (exchange != null) {
            LOGGER.log(Level.FINE, "Exchange got reply: Cleaning up {0}", idByMID);
            this.exchangesByMID.remove(idByMID);
        } else {
            LOGGER.log(Level.FINE, "Ignoring unmatchable empty message from {0}:{1}: {2}", new Object[]{message.getSource(), message.getSourcePort(), message});
        }
        return exchange;
    }

    @Override
    public void clear() {
        this.exchangesByMID.clear();
        this.exchangesByToken.clear();
        this.ongoingExchanges.clear();
        this.deduplicator.clear();
    }

    private void removeNotificationsOf(ObserveRelation relation) {
        LOGGER.fine("Remove all remaining NON-notifications of observe relation");
        Iterator<Response> iterator = relation.getNotificationIterator();
        while (iterator.hasNext()) {
            Response previous = iterator.next();
            Exchange.KeyMID idByMID = new Exchange.KeyMID(previous.getMID(), null, 0);
            this.exchangesByMID.remove(idByMID);
            iterator.remove();
        }
    }

    private Exchange.KeyToken createUnusedToken() {
        byte[] token;
        Exchange.KeyToken result;
        ThreadLocalRandom random = ThreadLocalRandom.current();
        do {
            token = new byte[((Random)random).nextInt(this.tokenSizeLimit) + 1];
            random.nextBytes(token);
        } while (this.exchangesByToken.get(result = new Exchange.KeyToken(token)) != null && this.observationStore.get(token) != null);
        return result;
    }

    @Override
    public void cancelObserve(byte[] token) {
        for (Map.Entry<Exchange.KeyToken, Exchange> key : this.exchangesByToken.entrySet()) {
            Request cachedRequest = key.getValue().getRequest();
            if (cachedRequest == null || !Arrays.equals(token, cachedRequest.getToken())) continue;
            cachedRequest.cancel();
        }
        this.observationStore.remove(token);
    }

    private class ExchangeObserverImpl
    implements ExchangeObserver {
        private ExchangeObserverImpl() {
        }

        @Override
        public void completed(Exchange exchange) {
            if (exchange.getOrigin() == Exchange.Origin.LOCAL) {
                Exchange.KeyMID idByMID = new Exchange.KeyMID(exchange.getCurrentRequest().getMID());
                Exchange.KeyToken idByToken = new Exchange.KeyToken(exchange.getCurrentRequest().getToken());
                UdpMatcher.this.exchangesByToken.remove(idByToken);
                UdpMatcher.this.exchangesByMID.remove(idByMID);
            } else {
                ObserveRelation relation;
                Request request;
                Response response = exchange.getCurrentResponse();
                if (response != null && response.getType() != CoAP.Type.ACK) {
                    Exchange.KeyMID midKey = new Exchange.KeyMID(response.getMID(), null, 0);
                    UdpMatcher.this.exchangesByMID.remove(midKey);
                }
                if ((request = exchange.getCurrentRequest()) != null && (request.getOptions().hasBlock1() || response.getOptions().hasBlock2())) {
                    Exchange.KeyUri uriKey = new Exchange.KeyUri(request.getURI(), request.getSource().getAddress(), request.getSourcePort());
                    LOGGER.log(Level.FINE, "Remote ongoing completed, cleaning up ", uriKey);
                    UdpMatcher.this.ongoingExchanges.remove(uriKey);
                }
                if ((relation = exchange.getRelation()) != null) {
                    UdpMatcher.this.removeNotificationsOf(relation);
                }
            }
        }

        @Override
        public void contextEstablished(Exchange exchange) {
            if (exchange.getRequest() != null) {
                UdpMatcher.this.observationStore.setContext(exchange.getRequest().getToken(), exchange.getCorrelationContext());
            }
        }
    }
}

