<?php
/**
 * Emoji detector for PHP.
 *
 * Unicode version: 15.0
 *
 * @author MaximAL
 * @date 2020-01-03
 * @time 17:00
 * @since 2023-05-04 Last parsing.
 * @since 2022-08-12 Latest data.
 * @since 2020-01-03 First version.
 *
 * @copyright ©  MaximAL, Sijeko  2020—2023
 * @link https://maximals.ru
 * @link https://sijeko.ru
 */

namespace Maximal\Emoji;

/**
 * Class Detector
 *
 * Unicode version: 15.0
 * @package Maximal\Emoji
 */
class Detector
{
	/**
	 * Skin tone character range
	 */
	const SKIN_TONE_RANGE = '\x{1F3FB}-\x{1F3FF}';

	/**
	 * Hair type character range
	 */
	const HAIR_RANGE = '\x{1F9B0}-\x{1F9B3}';

	/**
	 * Skin tone and hair type combined regex
	 */
	const COMPONENTS_REGEX = '/[' . self::SKIN_TONE_RANGE . self::HAIR_RANGE . ']+/ui';

	/**
	 * Detect whether the given string contains emoji characters.
	 * @param string $string Input string
	 * @return bool Returns `true` if the string contains one or more emoji characters and `false` otherwise.
	 */
	public static function containsEmoji(string $string): bool
	{
		// Remove skin tones
		$string = preg_replace('/[' . self::SKIN_TONE_RANGE . ']+/ui', '', $string);
		if (preg_match(self::EMOJI_REGEX, $string)) {
			return true;
		}
		return false;
	}

	/**
	 * Detect whether the given string consists of emoji characters only.
	 * @param string $string Input string
	 * @param bool $ignoreWhitespace `true` to ignore whitespace characters
	 * @return bool Returns `true` if the string consists of emoji characters only and `false` otherwise.
	 */
	public static function onlyEmoji(string $string, bool $ignoreWhitespace = true): bool
	{
		$string = self::removeEmoji($string);
		if ($ignoreWhitespace) {
			$string = preg_replace('/\s+/', '', $string);
		}
		return $string === '';
	}

	/**
	 * Remove all emoji characters from given string.
	 * @param string $string Input string
	 * @return string Returns the string without emoji characters.
	 */
	public static function removeEmoji(string $string): string
	{
		return preg_replace([self::COMPONENTS_REGEX, self::EMOJI_REGEX], ['', ''], $string);
	}

	/**
	 * Get all emojis of the input string.
	 * @param string $string Input string
	 * @return array Returns an array of all emojis of the input string.
	 */
	public static function allEmojis(string $string): array
	{
		// Remove skin tones
		$string = preg_replace('/[' . self::SKIN_TONE_RANGE . ']+/ui', '', $string);
		// Remove `+` in `])+/ui`
		$allRegex = preg_replace('#]\+?\)\+/ui$#', '])/ui', self::EMOJI_REGEX);
		if (preg_match_all($allRegex, $string, $matches)) {
			return $matches[1];
		}
		return [];
	}

	/**
	 * Get starting emojis of the input string.
	 * @param string $string Input string
	 * @param bool $ignoreWhitespace `true` to ignore whitespace characters
	 * @return array Returns an array of starting emojis of the input string.
	 */
	public static function startingEmojis(string $string, bool $ignoreWhitespace = true): array
	{
		// Remove skin tones
		$string = preg_replace('/[' . self::SKIN_TONE_RANGE . ']+/ui', '', $string);
		if ($ignoreWhitespace) {
			$string = preg_replace('/\s+/', '', $string);
		}
		$startingRegex = preg_replace(
			['#^/\(#', '#]\+?\)\+/ui$#'],
			['/^(', '])(.*)/ui'],
			self::EMOJI_REGEX
		);
		$result = [];
		while (preg_match($startingRegex, $string, $match)) {
			$result[] = $match[1];
			$string = $match[2];
		}
		return $result;
	}

	/**
	 * Get information about this class
	 * @return string Returns information about this class.
	 */
	public static function getInfo(): string
	{
		return 'Emoji detector by MaximAL' . PHP_EOL .
			'Unicode version: ' . self::UNICODE_VERSION .
			' as of ' . self::UNICODE_DATA_DATETIME .
			' with ' . self::TOTAL_EMOJI_COUNT . ' unique Emojis.' . PHP_EOL .
			'Parse date and time: ' . self::PARSE_DATETIME;
	}




	/**
	 * Unicode version when emoji characters parsed
	 * @autogenerated Do not edit manually
	 */
	const UNICODE_VERSION = '15.0';

	/**
	 * Total unique emoji count
	 * @autogenerated Do not edit manually
	 */
	const TOTAL_EMOJI_COUNT = 2012;

	/**
	 * Date and time when emoji data released
	 * @autogenerated Do not edit manually
	 */
	const UNICODE_DATA_DATETIME = '2022-08-12T20:24:39Z';

	/**
	 * Date and time when emoji data parsed
	 * @autogenerated Do not edit manually
	 */
	const PARSE_DATETIME = '2023-05-04T15:45:32Z';

	/**
	 * Full Emoji regex (23237 bytes)
	 *
	 * Includes fully-qualified, minimally-qualified, and unqualified sequences.
	 *
	 * @autogenerated Do not edit manually
	 * @see ../tools/parse.php
	 * @link https://unicode.org/Public/emoji/latest/emoji-test.txt
	 */
	const EMOJI_REGEX =
		'/(\x{1F468}\x{200D}\x{2764}\x{FE0F}\x{200D}\x{1F48B}\x{200D}\x{1F468}|\x{1F469}' .
		'\x{200D}\x{2764}\x{FE0F}\x{200D}\x{1F48B}\x{200D}\x{1F468}|\x{1F469}\x{200D}' .
		'\x{2764}\x{FE0F}\x{200D}\x{1F48B}\x{200D}\x{1F469}|\x{1F9D1}\x{200D}\x{2764}' .
		'\x{FE0F}\x{200D}\x{1F48B}\x{200D}\x{1F9D1}|\x{1F3F4}\x{E0067}\x{E0062}\x{E0065}' .
		'\x{E006E}\x{E0067}\x{E007F}|\x{1F3F4}\x{E0067}\x{E0062}\x{E0073}\x{E0063}' .
		'\x{E0074}\x{E007F}|\x{1F3F4}\x{E0067}\x{E0062}\x{E0077}\x{E006C}\x{E0073}' .
		'\x{E007F}|\x{1F468}\x{200D}\x{1F468}\x{200D}\x{1F466}\x{200D}\x{1F466}|' .
		'\x{1F468}\x{200D}\x{1F468}\x{200D}\x{1F467}\x{200D}\x{1F466}|\x{1F468}\x{200D}' .
		'\x{1F468}\x{200D}\x{1F467}\x{200D}\x{1F467}|\x{1F468}\x{200D}\x{1F469}\x{200D}' .
		'\x{1F466}\x{200D}\x{1F466}|\x{1F468}\x{200D}\x{1F469}\x{200D}\x{1F467}\x{200D}' .
		'\x{1F466}|\x{1F468}\x{200D}\x{1F469}\x{200D}\x{1F467}\x{200D}\x{1F467}|' .
		'\x{1F469}\x{200D}\x{1F469}\x{200D}\x{1F466}\x{200D}\x{1F466}|\x{1F469}\x{200D}' .
		'\x{1F469}\x{200D}\x{1F467}\x{200D}\x{1F466}|\x{1F469}\x{200D}\x{1F469}\x{200D}' .
		'\x{1F467}\x{200D}\x{1F467}|\x{1F468}\x{200D}\x{2764}\x{200D}\x{1F48B}\x{200D}' .
		'\x{1F468}|\x{1F469}\x{200D}\x{2764}\x{200D}\x{1F48B}\x{200D}\x{1F468}|\x{1F469}' .
		'\x{200D}\x{2764}\x{200D}\x{1F48B}\x{200D}\x{1F469}|\x{1F9D1}\x{200D}\x{2764}' .
		'\x{200D}\x{1F48B}\x{200D}\x{1F9D1}|\x{1F468}\x{200D}\x{2764}\x{FE0F}\x{200D}' .
		'\x{1F468}|\x{1F469}\x{200D}\x{2764}\x{FE0F}\x{200D}\x{1F468}|\x{1F469}\x{200D}' .
		'\x{2764}\x{FE0F}\x{200D}\x{1F469}|\x{1F9D1}\x{200D}\x{2764}\x{FE0F}\x{200D}' .
		'\x{1F9D1}|\x{1F468}\x{200D}\x{1F466}\x{200D}\x{1F466}|\x{1F468}\x{200D}' .
		'\x{1F467}\x{200D}\x{1F466}|\x{1F468}\x{200D}\x{1F467}\x{200D}\x{1F467}|' .
		'\x{1F468}\x{200D}\x{1F468}\x{200D}\x{1F466}|\x{1F468}\x{200D}\x{1F468}\x{200D}' .
		'\x{1F467}|\x{1F468}\x{200D}\x{1F469}\x{200D}\x{1F466}|\x{1F468}\x{200D}' .
		'\x{1F469}\x{200D}\x{1F467}|\x{1F468}\x{200D}\x{1F91D}\x{200D}\x{1F468}|' .
		'\x{1F469}\x{200D}\x{1F466}\x{200D}\x{1F466}|\x{1F469}\x{200D}\x{1F467}\x{200D}' .
		'\x{1F466}|\x{1F469}\x{200D}\x{1F467}\x{200D}\x{1F467}|\x{1F469}\x{200D}' .
		'\x{1F469}\x{200D}\x{1F466}|\x{1F469}\x{200D}\x{1F469}\x{200D}\x{1F467}|' .
		'\x{1F469}\x{200D}\x{1F91D}\x{200D}\x{1F468}|\x{1F469}\x{200D}\x{1F91D}\x{200D}' .
		'\x{1F469}|\x{1F9D1}\x{200D}\x{1F91D}\x{200D}\x{1F9D1}|\x{1F441}\x{FE0F}' .
		'\x{200D}\x{1F5E8}\x{FE0F}|\x{1F468}\x{200D}\x{2764}\x{200D}\x{1F468}|\x{1F469}' .
		'\x{200D}\x{2764}\x{200D}\x{1F468}|\x{1F469}\x{200D}\x{2764}\x{200D}\x{1F469}' .
		'|\x{1F9D1}\x{200D}\x{2764}\x{200D}\x{1F9D1}|\x{1F3CB}\x{FE0F}\x{200D}\x{2640}' .
		'\x{FE0F}|\x{1F3CB}\x{FE0F}\x{200D}\x{2642}\x{FE0F}|\x{1F3CC}\x{FE0F}\x{200D}' .
		'\x{2640}\x{FE0F}|\x{1F3CC}\x{FE0F}\x{200D}\x{2642}\x{FE0F}|\x{1F3F3}\x{FE0F}' .
		'\x{200D}\x{26A7}\x{FE0F}|\x{1F575}\x{FE0F}\x{200D}\x{2640}\x{FE0F}|\x{1F575}' .
		'\x{FE0F}\x{200D}\x{2642}\x{FE0F}|\x{26F9}\x{FE0F}\x{200D}\x{2640}\x{FE0F}|' .
		'\x{26F9}\x{FE0F}\x{200D}\x{2642}\x{FE0F}|\x{1F3F3}\x{FE0F}\x{200D}\x{1F308}|' .
		'\x{1F441}\x{200D}\x{1F5E8}\x{FE0F}|\x{1F441}\x{FE0F}\x{200D}\x{1F5E8}|\x{1F636}' .
		'\x{200D}\x{1F32B}\x{FE0F}|\x{1F3C3}\x{200D}\x{2640}\x{FE0F}|\x{1F3C3}\x{200D}' .
		'\x{2642}\x{FE0F}|\x{1F3C4}\x{200D}\x{2640}\x{FE0F}|\x{1F3C4}\x{200D}\x{2642}' .
		'\x{FE0F}|\x{1F3CA}\x{200D}\x{2640}\x{FE0F}|\x{1F3CA}\x{200D}\x{2642}\x{FE0F}' .
		'|\x{1F3CB}\x{200D}\x{2640}\x{FE0F}|\x{1F3CB}\x{200D}\x{2642}\x{FE0F}|\x{1F3CB}' .
		'\x{FE0F}\x{200D}\x{2640}|\x{1F3CB}\x{FE0F}\x{200D}\x{2642}|\x{1F3CC}\x{200D}' .
		'\x{2640}\x{FE0F}|\x{1F3CC}\x{200D}\x{2642}\x{FE0F}|\x{1F3CC}\x{FE0F}\x{200D}' .
		'\x{2640}|\x{1F3CC}\x{FE0F}\x{200D}\x{2642}|\x{1F3F3}\x{200D}\x{26A7}\x{FE0F}' .
		'|\x{1F3F3}\x{FE0F}\x{200D}\x{26A7}|\x{1F3F4}\x{200D}\x{2620}\x{FE0F}|\x{1F43B}' .
		'\x{200D}\x{2744}\x{FE0F}|\x{1F468}\x{200D}\x{2695}\x{FE0F}|\x{1F468}\x{200D}' .
		'\x{2696}\x{FE0F}|\x{1F468}\x{200D}\x{2708}\x{FE0F}|\x{1F469}\x{200D}\x{2695}' .
		'\x{FE0F}|\x{1F469}\x{200D}\x{2696}\x{FE0F}|\x{1F469}\x{200D}\x{2708}\x{FE0F}' .
		'|\x{1F46E}\x{200D}\x{2640}\x{FE0F}|\x{1F46E}\x{200D}\x{2642}\x{FE0F}|\x{1F46F}' .
		'\x{200D}\x{2640}\x{FE0F}|\x{1F46F}\x{200D}\x{2642}\x{FE0F}|\x{1F470}\x{200D}' .
		'\x{2640}\x{FE0F}|\x{1F470}\x{200D}\x{2642}\x{FE0F}|\x{1F471}\x{200D}\x{2640}' .
		'\x{FE0F}|\x{1F471}\x{200D}\x{2642}\x{FE0F}|\x{1F473}\x{200D}\x{2640}\x{FE0F}' .
		'|\x{1F473}\x{200D}\x{2642}\x{FE0F}|\x{1F477}\x{200D}\x{2640}\x{FE0F}|\x{1F477}' .
		'\x{200D}\x{2642}\x{FE0F}|\x{1F481}\x{200D}\x{2640}\x{FE0F}|\x{1F481}\x{200D}' .
		'\x{2642}\x{FE0F}|\x{1F482}\x{200D}\x{2640}\x{FE0F}|\x{1F482}\x{200D}\x{2642}' .
		'\x{FE0F}|\x{1F486}\x{200D}\x{2640}\x{FE0F}|\x{1F486}\x{200D}\x{2642}\x{FE0F}' .
		'|\x{1F487}\x{200D}\x{2640}\x{FE0F}|\x{1F487}\x{200D}\x{2642}\x{FE0F}|\x{1F575}' .
		'\x{200D}\x{2640}\x{FE0F}|\x{1F575}\x{200D}\x{2642}\x{FE0F}|\x{1F575}\x{FE0F}' .
		'\x{200D}\x{2640}|\x{1F575}\x{FE0F}\x{200D}\x{2642}|\x{1F645}\x{200D}\x{2640}' .
		'\x{FE0F}|\x{1F645}\x{200D}\x{2642}\x{FE0F}|\x{1F646}\x{200D}\x{2640}\x{FE0F}' .
		'|\x{1F646}\x{200D}\x{2642}\x{FE0F}|\x{1F647}\x{200D}\x{2640}\x{FE0F}|\x{1F647}' .
		'\x{200D}\x{2642}\x{FE0F}|\x{1F64B}\x{200D}\x{2640}\x{FE0F}|\x{1F64B}\x{200D}' .
		'\x{2642}\x{FE0F}|\x{1F64D}\x{200D}\x{2640}\x{FE0F}|\x{1F64D}\x{200D}\x{2642}' .
		'\x{FE0F}|\x{1F64E}\x{200D}\x{2640}\x{FE0F}|\x{1F64E}\x{200D}\x{2642}\x{FE0F}' .
		'|\x{1F6A3}\x{200D}\x{2640}\x{FE0F}|\x{1F6A3}\x{200D}\x{2642}\x{FE0F}|\x{1F6B4}' .
		'\x{200D}\x{2640}\x{FE0F}|\x{1F6B4}\x{200D}\x{2642}\x{FE0F}|\x{1F6B5}\x{200D}' .
		'\x{2640}\x{FE0F}|\x{1F6B5}\x{200D}\x{2642}\x{FE0F}|\x{1F6B6}\x{200D}\x{2640}' .
		'\x{FE0F}|\x{1F6B6}\x{200D}\x{2642}\x{FE0F}|\x{1F926}\x{200D}\x{2640}\x{FE0F}' .
		'|\x{1F926}\x{200D}\x{2642}\x{FE0F}|\x{1F935}\x{200D}\x{2640}\x{FE0F}|\x{1F935}' .
		'\x{200D}\x{2642}\x{FE0F}|\x{1F937}\x{200D}\x{2640}\x{FE0F}|\x{1F937}\x{200D}' .
		'\x{2642}\x{FE0F}|\x{1F938}\x{200D}\x{2640}\x{FE0F}|\x{1F938}\x{200D}\x{2642}' .
		'\x{FE0F}|\x{1F939}\x{200D}\x{2640}\x{FE0F}|\x{1F939}\x{200D}\x{2642}\x{FE0F}' .
		'|\x{1F93C}\x{200D}\x{2640}\x{FE0F}|\x{1F93C}\x{200D}\x{2642}\x{FE0F}|\x{1F93D}' .
		'\x{200D}\x{2640}\x{FE0F}|\x{1F93D}\x{200D}\x{2642}\x{FE0F}|\x{1F93E}\x{200D}' .
		'\x{2640}\x{FE0F}|\x{1F93E}\x{200D}\x{2642}\x{FE0F}|\x{1F9B8}\x{200D}\x{2640}' .
		'\x{FE0F}|\x{1F9B8}\x{200D}\x{2642}\x{FE0F}|\x{1F9B9}\x{200D}\x{2640}\x{FE0F}' .
		'|\x{1F9B9}\x{200D}\x{2642}\x{FE0F}|\x{1F9CD}\x{200D}\x{2640}\x{FE0F}|\x{1F9CD}' .
		'\x{200D}\x{2642}\x{FE0F}|\x{1F9CE}\x{200D}\x{2640}\x{FE0F}|\x{1F9CE}\x{200D}' .
		'\x{2642}\x{FE0F}|\x{1F9CF}\x{200D}\x{2640}\x{FE0F}|\x{1F9CF}\x{200D}\x{2642}' .
		'\x{FE0F}|\x{1F9D1}\x{200D}\x{2695}\x{FE0F}|\x{1F9D1}\x{200D}\x{2696}\x{FE0F}' .
		'|\x{1F9D1}\x{200D}\x{2708}\x{FE0F}|\x{1F9D4}\x{200D}\x{2640}\x{FE0F}|\x{1F9D4}' .
		'\x{200D}\x{2642}\x{FE0F}|\x{1F9D6}\x{200D}\x{2640}\x{FE0F}|\x{1F9D6}\x{200D}' .
		'\x{2642}\x{FE0F}|\x{1F9D7}\x{200D}\x{2640}\x{FE0F}|\x{1F9D7}\x{200D}\x{2642}' .
		'\x{FE0F}|\x{1F9D8}\x{200D}\x{2640}\x{FE0F}|\x{1F9D8}\x{200D}\x{2642}\x{FE0F}' .
		'|\x{1F9D9}\x{200D}\x{2640}\x{FE0F}|\x{1F9D9}\x{200D}\x{2642}\x{FE0F}|\x{1F9DA}' .
		'\x{200D}\x{2640}\x{FE0F}|\x{1F9DA}\x{200D}\x{2642}\x{FE0F}|\x{1F9DB}\x{200D}' .
		'\x{2640}\x{FE0F}|\x{1F9DB}\x{200D}\x{2642}\x{FE0F}|\x{1F9DC}\x{200D}\x{2640}' .
		'\x{FE0F}|\x{1F9DC}\x{200D}\x{2642}\x{FE0F}|\x{1F9DD}\x{200D}\x{2640}\x{FE0F}' .
		'|\x{1F9DD}\x{200D}\x{2642}\x{FE0F}|\x{1F9DE}\x{200D}\x{2640}\x{FE0F}|\x{1F9DE}' .
		'\x{200D}\x{2642}\x{FE0F}|\x{1F9DF}\x{200D}\x{2640}\x{FE0F}|\x{1F9DF}\x{200D}' .
		'\x{2642}\x{FE0F}|\x{2764}\x{FE0F}\x{200D}\x{1F525}|\x{2764}\x{FE0F}\x{200D}' .
		'\x{1FA79}|\x{26F9}\x{200D}\x{2640}\x{FE0F}|\x{26F9}\x{200D}\x{2642}\x{FE0F}|' .
		'\x{26F9}\x{FE0F}\x{200D}\x{2640}|\x{26F9}\x{FE0F}\x{200D}\x{2642}|\x{1F3F3}' .
		'\x{200D}\x{1F308}|\x{1F415}\x{200D}\x{1F9BA}|\x{1F441}\x{200D}\x{1F5E8}|' .
		'\x{1F468}\x{200D}\x{1F33E}|\x{1F468}\x{200D}\x{1F373}|\x{1F468}\x{200D}' .
		'\x{1F37C}|\x{1F468}\x{200D}\x{1F393}|\x{1F468}\x{200D}\x{1F3A4}|\x{1F468}' .
		'\x{200D}\x{1F3A8}|\x{1F468}\x{200D}\x{1F3EB}|\x{1F468}\x{200D}\x{1F3ED}|' .
		'\x{1F468}\x{200D}\x{1F466}|\x{1F468}\x{200D}\x{1F467}|\x{1F468}\x{200D}' .
		'\x{1F4BB}|\x{1F468}\x{200D}\x{1F4BC}|\x{1F468}\x{200D}\x{1F527}|\x{1F468}' .
		'\x{200D}\x{1F52C}|\x{1F468}\x{200D}\x{1F680}|\x{1F468}\x{200D}\x{1F692}|' .
		'\x{1F468}\x{200D}\x{1F9AF}|\x{1F468}\x{200D}\x{1F9B0}|\x{1F468}\x{200D}' .
		'\x{1F9B1}|\x{1F468}\x{200D}\x{1F9B2}|\x{1F468}\x{200D}\x{1F9B3}|\x{1F468}' .
		'\x{200D}\x{1F9BC}|\x{1F468}\x{200D}\x{1F9BD}|\x{1F469}\x{200D}\x{1F33E}|' .
		'\x{1F469}\x{200D}\x{1F373}|\x{1F469}\x{200D}\x{1F37C}|\x{1F469}\x{200D}' .
		'\x{1F393}|\x{1F469}\x{200D}\x{1F3A4}|\x{1F469}\x{200D}\x{1F3A8}|\x{1F469}' .
		'\x{200D}\x{1F3EB}|\x{1F469}\x{200D}\x{1F3ED}|\x{1F469}\x{200D}\x{1F466}|' .
		'\x{1F469}\x{200D}\x{1F467}|\x{1F469}\x{200D}\x{1F4BB}|\x{1F469}\x{200D}' .
		'\x{1F4BC}|\x{1F469}\x{200D}\x{1F527}|\x{1F469}\x{200D}\x{1F52C}|\x{1F469}' .
		'\x{200D}\x{1F680}|\x{1F469}\x{200D}\x{1F692}|\x{1F469}\x{200D}\x{1F9AF}|' .
		'\x{1F469}\x{200D}\x{1F9B0}|\x{1F469}\x{200D}\x{1F9B1}|\x{1F469}\x{200D}' .
		'\x{1F9B2}|\x{1F469}\x{200D}\x{1F9B3}|\x{1F469}\x{200D}\x{1F9BC}|\x{1F469}' .
		'\x{200D}\x{1F9BD}|\x{1F62E}\x{200D}\x{1F4A8}|\x{1F635}\x{200D}\x{1F4AB}|' .
		'\x{1F636}\x{200D}\x{1F32B}|\x{1F9D1}\x{200D}\x{1F33E}|\x{1F9D1}\x{200D}' .
		'\x{1F373}|\x{1F9D1}\x{200D}\x{1F37C}|\x{1F9D1}\x{200D}\x{1F384}|\x{1F9D1}' .
		'\x{200D}\x{1F393}|\x{1F9D1}\x{200D}\x{1F3A4}|\x{1F9D1}\x{200D}\x{1F3A8}|' .
		'\x{1F9D1}\x{200D}\x{1F3EB}|\x{1F9D1}\x{200D}\x{1F3ED}|\x{1F9D1}\x{200D}' .
		'\x{1F4BB}|\x{1F9D1}\x{200D}\x{1F4BC}|\x{1F9D1}\x{200D}\x{1F527}|\x{1F9D1}' .
		'\x{200D}\x{1F52C}|\x{1F9D1}\x{200D}\x{1F680}|\x{1F9D1}\x{200D}\x{1F692}|' .
		'\x{1F9D1}\x{200D}\x{1F9AF}|\x{1F9D1}\x{200D}\x{1F9B0}|\x{1F9D1}\x{200D}' .
		'\x{1F9B1}|\x{1F9D1}\x{200D}\x{1F9B2}|\x{1F9D1}\x{200D}\x{1F9B3}|\x{1F9D1}' .
		'\x{200D}\x{1F9BC}|\x{1F9D1}\x{200D}\x{1F9BD}|\x{1FAF1}\x{200D}\x{1FAF2}|' .
		'\x{1F3C3}\x{200D}\x{2640}|\x{1F3C3}\x{200D}\x{2642}|\x{1F3C4}\x{200D}\x{2640}' .
		'|\x{1F3C4}\x{200D}\x{2642}|\x{1F3CA}\x{200D}\x{2640}|\x{1F3CA}\x{200D}\x{2642}' .
		'|\x{1F3CB}\x{200D}\x{2640}|\x{1F3CB}\x{200D}\x{2642}|\x{1F3CC}\x{200D}\x{2640}' .
		'|\x{1F3CC}\x{200D}\x{2642}|\x{1F3F3}\x{200D}\x{26A7}|\x{1F3F4}\x{200D}\x{2620}' .
		'|\x{1F408}\x{200D}\x{2B1B}|\x{1F426}\x{200D}\x{2B1B}|\x{1F43B}\x{200D}\x{2744}' .
		'|\x{1F468}\x{200D}\x{2695}|\x{1F468}\x{200D}\x{2696}|\x{1F468}\x{200D}\x{2708}' .
		'|\x{1F469}\x{200D}\x{2695}|\x{1F469}\x{200D}\x{2696}|\x{1F469}\x{200D}\x{2708}' .
		'|\x{1F46E}\x{200D}\x{2640}|\x{1F46E}\x{200D}\x{2642}|\x{1F46F}\x{200D}\x{2640}' .
		'|\x{1F46F}\x{200D}\x{2642}|\x{1F470}\x{200D}\x{2640}|\x{1F470}\x{200D}\x{2642}' .
		'|\x{1F471}\x{200D}\x{2640}|\x{1F471}\x{200D}\x{2642}|\x{1F473}\x{200D}\x{2640}' .
		'|\x{1F473}\x{200D}\x{2642}|\x{1F477}\x{200D}\x{2640}|\x{1F477}\x{200D}\x{2642}' .
		'|\x{1F481}\x{200D}\x{2640}|\x{1F481}\x{200D}\x{2642}|\x{1F482}\x{200D}\x{2640}' .
		'|\x{1F482}\x{200D}\x{2642}|\x{1F486}\x{200D}\x{2640}|\x{1F486}\x{200D}\x{2642}' .
		'|\x{1F487}\x{200D}\x{2640}|\x{1F487}\x{200D}\x{2642}|\x{1F575}\x{200D}\x{2640}' .
		'|\x{1F575}\x{200D}\x{2642}|\x{1F645}\x{200D}\x{2640}|\x{1F645}\x{200D}\x{2642}' .
		'|\x{1F646}\x{200D}\x{2640}|\x{1F646}\x{200D}\x{2642}|\x{1F647}\x{200D}\x{2640}' .
		'|\x{1F647}\x{200D}\x{2642}|\x{1F64B}\x{200D}\x{2640}|\x{1F64B}\x{200D}\x{2642}' .
		'|\x{1F64D}\x{200D}\x{2640}|\x{1F64D}\x{200D}\x{2642}|\x{1F64E}\x{200D}\x{2640}' .
		'|\x{1F64E}\x{200D}\x{2642}|\x{1F6A3}\x{200D}\x{2640}|\x{1F6A3}\x{200D}\x{2642}' .
		'|\x{1F6B4}\x{200D}\x{2640}|\x{1F6B4}\x{200D}\x{2642}|\x{1F6B5}\x{200D}\x{2640}' .
		'|\x{1F6B5}\x{200D}\x{2642}|\x{1F6B6}\x{200D}\x{2640}|\x{1F6B6}\x{200D}\x{2642}' .
		'|\x{1F926}\x{200D}\x{2640}|\x{1F926}\x{200D}\x{2642}|\x{1F935}\x{200D}\x{2640}' .
		'|\x{1F935}\x{200D}\x{2642}|\x{1F937}\x{200D}\x{2640}|\x{1F937}\x{200D}\x{2642}' .
		'|\x{1F938}\x{200D}\x{2640}|\x{1F938}\x{200D}\x{2642}|\x{1F939}\x{200D}\x{2640}' .
		'|\x{1F939}\x{200D}\x{2642}|\x{1F93C}\x{200D}\x{2640}|\x{1F93C}\x{200D}\x{2642}' .
		'|\x{1F93D}\x{200D}\x{2640}|\x{1F93D}\x{200D}\x{2642}|\x{1F93E}\x{200D}\x{2640}' .
		'|\x{1F93E}\x{200D}\x{2642}|\x{1F9B8}\x{200D}\x{2640}|\x{1F9B8}\x{200D}\x{2642}' .
		'|\x{1F9B9}\x{200D}\x{2640}|\x{1F9B9}\x{200D}\x{2642}|\x{1F9CD}\x{200D}\x{2640}' .
		'|\x{1F9CD}\x{200D}\x{2642}|\x{1F9CE}\x{200D}\x{2640}|\x{1F9CE}\x{200D}\x{2642}' .
		'|\x{1F9CF}\x{200D}\x{2640}|\x{1F9CF}\x{200D}\x{2642}|\x{1F9D1}\x{200D}\x{2695}' .
		'|\x{1F9D1}\x{200D}\x{2696}|\x{1F9D1}\x{200D}\x{2708}|\x{1F9D4}\x{200D}\x{2640}' .
		'|\x{1F9D4}\x{200D}\x{2642}|\x{1F9D6}\x{200D}\x{2640}|\x{1F9D6}\x{200D}\x{2642}' .
		'|\x{1F9D7}\x{200D}\x{2640}|\x{1F9D7}\x{200D}\x{2642}|\x{1F9D8}\x{200D}\x{2640}' .
		'|\x{1F9D8}\x{200D}\x{2642}|\x{1F9D9}\x{200D}\x{2640}|\x{1F9D9}\x{200D}\x{2642}' .
		'|\x{1F9DA}\x{200D}\x{2640}|\x{1F9DA}\x{200D}\x{2642}|\x{1F9DB}\x{200D}\x{2640}' .
		'|\x{1F9DB}\x{200D}\x{2642}|\x{1F9DC}\x{200D}\x{2640}|\x{1F9DC}\x{200D}\x{2642}' .
		'|\x{1F9DD}\x{200D}\x{2640}|\x{1F9DD}\x{200D}\x{2642}|\x{1F9DE}\x{200D}\x{2640}' .
		'|\x{1F9DE}\x{200D}\x{2642}|\x{1F9DF}\x{200D}\x{2640}|\x{1F9DF}\x{200D}\x{2642}' .
		'|\x{2764}\x{200D}\x{1F525}|\x{2764}\x{200D}\x{1FA79}|\x{0023}\x{FE0F}\x{20E3}' .
		'|\x{002A}\x{FE0F}\x{20E3}|\x{0030}\x{FE0F}\x{20E3}|\x{0031}\x{FE0F}\x{20E3}|' .
		'\x{0032}\x{FE0F}\x{20E3}|\x{0033}\x{FE0F}\x{20E3}|\x{0034}\x{FE0F}\x{20E3}|' .
		'\x{0035}\x{FE0F}\x{20E3}|\x{0036}\x{FE0F}\x{20E3}|\x{0037}\x{FE0F}\x{20E3}|' .
		'\x{0038}\x{FE0F}\x{20E3}|\x{0039}\x{FE0F}\x{20E3}|\x{26F9}\x{200D}\x{2640}|' .
		'\x{26F9}\x{200D}\x{2642}|\x{1F1E6}\x{1F1E8}|\x{1F1E6}\x{1F1E9}|\x{1F1E6}' .
		'\x{1F1EA}|\x{1F1E6}\x{1F1EB}|\x{1F1E6}\x{1F1EC}|\x{1F1E6}\x{1F1EE}|\x{1F1E6}' .
		'\x{1F1F1}|\x{1F1E6}\x{1F1F2}|\x{1F1E6}\x{1F1F4}|\x{1F1E6}\x{1F1F6}|\x{1F1E6}' .
		'\x{1F1F7}|\x{1F1E6}\x{1F1F8}|\x{1F1E6}\x{1F1F9}|\x{1F1E6}\x{1F1FA}|\x{1F1E6}' .
		'\x{1F1FC}|\x{1F1E6}\x{1F1FD}|\x{1F1E6}\x{1F1FF}|\x{1F1E7}\x{1F1E6}|\x{1F1E7}' .
		'\x{1F1E7}|\x{1F1E7}\x{1F1E9}|\x{1F1E7}\x{1F1EA}|\x{1F1E7}\x{1F1EB}|\x{1F1E7}' .
		'\x{1F1EC}|\x{1F1E7}\x{1F1ED}|\x{1F1E7}\x{1F1EE}|\x{1F1E7}\x{1F1EF}|\x{1F1E7}' .
		'\x{1F1F1}|\x{1F1E7}\x{1F1F2}|\x{1F1E7}\x{1F1F3}|\x{1F1E7}\x{1F1F4}|\x{1F1E7}' .
		'\x{1F1F6}|\x{1F1E7}\x{1F1F7}|\x{1F1E7}\x{1F1F8}|\x{1F1E7}\x{1F1F9}|\x{1F1E7}' .
		'\x{1F1FB}|\x{1F1E7}\x{1F1FC}|\x{1F1E7}\x{1F1FE}|\x{1F1E7}\x{1F1FF}|\x{1F1E8}' .
		'\x{1F1E6}|\x{1F1E8}\x{1F1E8}|\x{1F1E8}\x{1F1E9}|\x{1F1E8}\x{1F1EB}|\x{1F1E8}' .
		'\x{1F1EC}|\x{1F1E8}\x{1F1ED}|\x{1F1E8}\x{1F1EE}|\x{1F1E8}\x{1F1F0}|\x{1F1E8}' .
		'\x{1F1F1}|\x{1F1E8}\x{1F1F2}|\x{1F1E8}\x{1F1F3}|\x{1F1E8}\x{1F1F4}|\x{1F1E8}' .
		'\x{1F1F5}|\x{1F1E8}\x{1F1F7}|\x{1F1E8}\x{1F1FA}|\x{1F1E8}\x{1F1FB}|\x{1F1E8}' .
		'\x{1F1FC}|\x{1F1E8}\x{1F1FD}|\x{1F1E8}\x{1F1FE}|\x{1F1E8}\x{1F1FF}|\x{1F1E9}' .
		'\x{1F1EA}|\x{1F1E9}\x{1F1EC}|\x{1F1E9}\x{1F1EF}|\x{1F1E9}\x{1F1F0}|\x{1F1E9}' .
		'\x{1F1F2}|\x{1F1E9}\x{1F1F4}|\x{1F1E9}\x{1F1FF}|\x{1F1EA}\x{1F1E6}|\x{1F1EA}' .
		'\x{1F1E8}|\x{1F1EA}\x{1F1EA}|\x{1F1EA}\x{1F1EC}|\x{1F1EA}\x{1F1ED}|\x{1F1EA}' .
		'\x{1F1F7}|\x{1F1EA}\x{1F1F8}|\x{1F1EA}\x{1F1F9}|\x{1F1EA}\x{1F1FA}|\x{1F1EB}' .
		'\x{1F1EE}|\x{1F1EB}\x{1F1EF}|\x{1F1EB}\x{1F1F0}|\x{1F1EB}\x{1F1F2}|\x{1F1EB}' .
		'\x{1F1F4}|\x{1F1EB}\x{1F1F7}|\x{1F1EC}\x{1F1E6}|\x{1F1EC}\x{1F1E7}|\x{1F1EC}' .
		'\x{1F1E9}|\x{1F1EC}\x{1F1EA}|\x{1F1EC}\x{1F1EB}|\x{1F1EC}\x{1F1EC}|\x{1F1EC}' .
		'\x{1F1ED}|\x{1F1EC}\x{1F1EE}|\x{1F1EC}\x{1F1F1}|\x{1F1EC}\x{1F1F2}|\x{1F1EC}' .
		'\x{1F1F3}|\x{1F1EC}\x{1F1F5}|\x{1F1EC}\x{1F1F6}|\x{1F1EC}\x{1F1F7}|\x{1F1EC}' .
		'\x{1F1F8}|\x{1F1EC}\x{1F1F9}|\x{1F1EC}\x{1F1FA}|\x{1F1EC}\x{1F1FC}|\x{1F1EC}' .
		'\x{1F1FE}|\x{1F1ED}\x{1F1F0}|\x{1F1ED}\x{1F1F2}|\x{1F1ED}\x{1F1F3}|\x{1F1ED}' .
		'\x{1F1F7}|\x{1F1ED}\x{1F1F9}|\x{1F1ED}\x{1F1FA}|\x{1F1EE}\x{1F1E8}|\x{1F1EE}' .
		'\x{1F1E9}|\x{1F1EE}\x{1F1EA}|\x{1F1EE}\x{1F1F1}|\x{1F1EE}\x{1F1F2}|\x{1F1EE}' .
		'\x{1F1F3}|\x{1F1EE}\x{1F1F4}|\x{1F1EE}\x{1F1F6}|\x{1F1EE}\x{1F1F7}|\x{1F1EE}' .
		'\x{1F1F8}|\x{1F1EE}\x{1F1F9}|\x{1F1EF}\x{1F1EA}|\x{1F1EF}\x{1F1F2}|\x{1F1EF}' .
		'\x{1F1F4}|\x{1F1EF}\x{1F1F5}|\x{1F1F0}\x{1F1EA}|\x{1F1F0}\x{1F1EC}|\x{1F1F0}' .
		'\x{1F1ED}|\x{1F1F0}\x{1F1EE}|\x{1F1F0}\x{1F1F2}|\x{1F1F0}\x{1F1F3}|\x{1F1F0}' .
		'\x{1F1F5}|\x{1F1F0}\x{1F1F7}|\x{1F1F0}\x{1F1FC}|\x{1F1F0}\x{1F1FE}|\x{1F1F0}' .
		'\x{1F1FF}|\x{1F1F1}\x{1F1E6}|\x{1F1F1}\x{1F1E7}|\x{1F1F1}\x{1F1E8}|\x{1F1F1}' .
		'\x{1F1EE}|\x{1F1F1}\x{1F1F0}|\x{1F1F1}\x{1F1F7}|\x{1F1F1}\x{1F1F8}|\x{1F1F1}' .
		'\x{1F1F9}|\x{1F1F1}\x{1F1FA}|\x{1F1F1}\x{1F1FB}|\x{1F1F1}\x{1F1FE}|\x{1F1F2}' .
		'\x{1F1E6}|\x{1F1F2}\x{1F1E8}|\x{1F1F2}\x{1F1E9}|\x{1F1F2}\x{1F1EA}|\x{1F1F2}' .
		'\x{1F1EB}|\x{1F1F2}\x{1F1EC}|\x{1F1F2}\x{1F1ED}|\x{1F1F2}\x{1F1F0}|\x{1F1F2}' .
		'\x{1F1F1}|\x{1F1F2}\x{1F1F2}|\x{1F1F2}\x{1F1F3}|\x{1F1F2}\x{1F1F4}|\x{1F1F2}' .
		'\x{1F1F5}|\x{1F1F2}\x{1F1F6}|\x{1F1F2}\x{1F1F7}|\x{1F1F2}\x{1F1F8}|\x{1F1F2}' .
		'\x{1F1F9}|\x{1F1F2}\x{1F1FA}|\x{1F1F2}\x{1F1FB}|\x{1F1F2}\x{1F1FC}|\x{1F1F2}' .
		'\x{1F1FD}|\x{1F1F2}\x{1F1FE}|\x{1F1F2}\x{1F1FF}|\x{1F1F3}\x{1F1E6}|\x{1F1F3}' .
		'\x{1F1E8}|\x{1F1F3}\x{1F1EA}|\x{1F1F3}\x{1F1EB}|\x{1F1F3}\x{1F1EC}|\x{1F1F3}' .
		'\x{1F1EE}|\x{1F1F3}\x{1F1F1}|\x{1F1F3}\x{1F1F4}|\x{1F1F3}\x{1F1F5}|\x{1F1F3}' .
		'\x{1F1F7}|\x{1F1F3}\x{1F1FA}|\x{1F1F3}\x{1F1FF}|\x{1F1F4}\x{1F1F2}|\x{1F1F5}' .
		'\x{1F1E6}|\x{1F1F5}\x{1F1EA}|\x{1F1F5}\x{1F1EB}|\x{1F1F5}\x{1F1EC}|\x{1F1F5}' .
		'\x{1F1ED}|\x{1F1F5}\x{1F1F0}|\x{1F1F5}\x{1F1F1}|\x{1F1F5}\x{1F1F2}|\x{1F1F5}' .
		'\x{1F1F3}|\x{1F1F5}\x{1F1F7}|\x{1F1F5}\x{1F1F8}|\x{1F1F5}\x{1F1F9}|\x{1F1F5}' .
		'\x{1F1FC}|\x{1F1F5}\x{1F1FE}|\x{1F1F6}\x{1F1E6}|\x{1F1F7}\x{1F1EA}|\x{1F1F7}' .
		'\x{1F1F4}|\x{1F1F7}\x{1F1F8}|\x{1F1F7}\x{1F1FA}|\x{1F1F7}\x{1F1FC}|\x{1F1F8}' .
		'\x{1F1E6}|\x{1F1F8}\x{1F1E7}|\x{1F1F8}\x{1F1E8}|\x{1F1F8}\x{1F1E9}|\x{1F1F8}' .
		'\x{1F1EA}|\x{1F1F8}\x{1F1EC}|\x{1F1F8}\x{1F1ED}|\x{1F1F8}\x{1F1EE}|\x{1F1F8}' .
		'\x{1F1EF}|\x{1F1F8}\x{1F1F0}|\x{1F1F8}\x{1F1F1}|\x{1F1F8}\x{1F1F2}|\x{1F1F8}' .
		'\x{1F1F3}|\x{1F1F8}\x{1F1F4}|\x{1F1F8}\x{1F1F7}|\x{1F1F8}\x{1F1F8}|\x{1F1F8}' .
		'\x{1F1F9}|\x{1F1F8}\x{1F1FB}|\x{1F1F8}\x{1F1FD}|\x{1F1F8}\x{1F1FE}|\x{1F1F8}' .
		'\x{1F1FF}|\x{1F1F9}\x{1F1E6}|\x{1F1F9}\x{1F1E8}|\x{1F1F9}\x{1F1E9}|\x{1F1F9}' .
		'\x{1F1EB}|\x{1F1F9}\x{1F1EC}|\x{1F1F9}\x{1F1ED}|\x{1F1F9}\x{1F1EF}|\x{1F1F9}' .
		'\x{1F1F0}|\x{1F1F9}\x{1F1F1}|\x{1F1F9}\x{1F1F2}|\x{1F1F9}\x{1F1F3}|\x{1F1F9}' .
		'\x{1F1F4}|\x{1F1F9}\x{1F1F7}|\x{1F1F9}\x{1F1F9}|\x{1F1F9}\x{1F1FB}|\x{1F1F9}' .
		'\x{1F1FC}|\x{1F1F9}\x{1F1FF}|\x{1F1FA}\x{1F1E6}|\x{1F1FA}\x{1F1EC}|\x{1F1FA}' .
		'\x{1F1F2}|\x{1F1FA}\x{1F1F3}|\x{1F1FA}\x{1F1F8}|\x{1F1FA}\x{1F1FE}|\x{1F1FA}' .
		'\x{1F1FF}|\x{1F1FB}\x{1F1E6}|\x{1F1FB}\x{1F1E8}|\x{1F1FB}\x{1F1EA}|\x{1F1FB}' .
		'\x{1F1EC}|\x{1F1FB}\x{1F1EE}|\x{1F1FB}\x{1F1F3}|\x{1F1FB}\x{1F1FA}|\x{1F1FC}' .
		'\x{1F1EB}|\x{1F1FC}\x{1F1F8}|\x{1F1FD}\x{1F1F0}|\x{1F1FE}\x{1F1EA}|\x{1F1FE}' .
		'\x{1F1F9}|\x{1F1FF}\x{1F1E6}|\x{1F1FF}\x{1F1F2}|\x{1F1FF}\x{1F1FC}|\x{1F170}' .
		'\x{FE0F}|\x{1F171}\x{FE0F}|\x{1F17E}\x{FE0F}|\x{1F17F}\x{FE0F}|\x{1F202}' .
		'\x{FE0F}|\x{1F237}\x{FE0F}|\x{1F321}\x{FE0F}|\x{1F324}\x{FE0F}|\x{1F325}' .
		'\x{FE0F}|\x{1F326}\x{FE0F}|\x{1F327}\x{FE0F}|\x{1F328}\x{FE0F}|\x{1F329}' .
		'\x{FE0F}|\x{1F32A}\x{FE0F}|\x{1F32B}\x{FE0F}|\x{1F32C}\x{FE0F}|\x{1F336}' .
		'\x{FE0F}|\x{1F37D}\x{FE0F}|\x{1F396}\x{FE0F}|\x{1F397}\x{FE0F}|\x{1F399}' .
		'\x{FE0F}|\x{1F39A}\x{FE0F}|\x{1F39B}\x{FE0F}|\x{1F39E}\x{FE0F}|\x{1F39F}' .
		'\x{FE0F}|\x{1F3CB}\x{FE0F}|\x{1F3CC}\x{FE0F}|\x{1F3CD}\x{FE0F}|\x{1F3CE}' .
		'\x{FE0F}|\x{1F3D4}\x{FE0F}|\x{1F3D5}\x{FE0F}|\x{1F3D6}\x{FE0F}|\x{1F3D7}' .
		'\x{FE0F}|\x{1F3D8}\x{FE0F}|\x{1F3D9}\x{FE0F}|\x{1F3DA}\x{FE0F}|\x{1F3DB}' .
		'\x{FE0F}|\x{1F3DC}\x{FE0F}|\x{1F3DD}\x{FE0F}|\x{1F3DE}\x{FE0F}|\x{1F3DF}' .
		'\x{FE0F}|\x{1F3F3}\x{FE0F}|\x{1F3F5}\x{FE0F}|\x{1F3F7}\x{FE0F}|\x{1F43F}' .
		'\x{FE0F}|\x{1F441}\x{FE0F}|\x{1F4FD}\x{FE0F}|\x{1F549}\x{FE0F}|\x{1F54A}' .
		'\x{FE0F}|\x{1F56F}\x{FE0F}|\x{1F570}\x{FE0F}|\x{1F573}\x{FE0F}|\x{1F574}' .
		'\x{FE0F}|\x{1F575}\x{FE0F}|\x{1F576}\x{FE0F}|\x{1F577}\x{FE0F}|\x{1F578}' .
		'\x{FE0F}|\x{1F579}\x{FE0F}|\x{1F587}\x{FE0F}|\x{1F58A}\x{FE0F}|\x{1F58B}' .
		'\x{FE0F}|\x{1F58C}\x{FE0F}|\x{1F58D}\x{FE0F}|\x{1F590}\x{FE0F}|\x{1F5A5}' .
		'\x{FE0F}|\x{1F5A8}\x{FE0F}|\x{1F5B1}\x{FE0F}|\x{1F5B2}\x{FE0F}|\x{1F5BC}' .
		'\x{FE0F}|\x{1F5C2}\x{FE0F}|\x{1F5C3}\x{FE0F}|\x{1F5C4}\x{FE0F}|\x{1F5D1}' .
		'\x{FE0F}|\x{1F5D2}\x{FE0F}|\x{1F5D3}\x{FE0F}|\x{1F5DC}\x{FE0F}|\x{1F5DD}' .
		'\x{FE0F}|\x{1F5DE}\x{FE0F}|\x{1F5E1}\x{FE0F}|\x{1F5E3}\x{FE0F}|\x{1F5E8}' .
		'\x{FE0F}|\x{1F5EF}\x{FE0F}|\x{1F5F3}\x{FE0F}|\x{1F5FA}\x{FE0F}|\x{1F6CB}' .
		'\x{FE0F}|\x{1F6CD}\x{FE0F}|\x{1F6CE}\x{FE0F}|\x{1F6CF}\x{FE0F}|\x{1F6E0}' .
		'\x{FE0F}|\x{1F6E1}\x{FE0F}|\x{1F6E2}\x{FE0F}|\x{1F6E3}\x{FE0F}|\x{1F6E4}' .
		'\x{FE0F}|\x{1F6E5}\x{FE0F}|\x{1F6E9}\x{FE0F}|\x{1F6F0}\x{FE0F}|\x{1F6F3}' .
		'\x{FE0F}|\x{0023}\x{20E3}|\x{002A}\x{20E3}|\x{0030}\x{20E3}|\x{0031}\x{20E3}' .
		'|\x{0032}\x{20E3}|\x{0033}\x{20E3}|\x{0034}\x{20E3}|\x{0035}\x{20E3}|\x{0036}' .
		'\x{20E3}|\x{0037}\x{20E3}|\x{0038}\x{20E3}|\x{0039}\x{20E3}|\x{00A9}\x{FE0F}' .
		'|\x{00AE}\x{FE0F}|\x{203C}\x{FE0F}|\x{2049}\x{FE0F}|\x{2122}\x{FE0F}|\x{2139}' .
		'\x{FE0F}|\x{2194}\x{FE0F}|\x{2195}\x{FE0F}|\x{2196}\x{FE0F}|\x{2197}\x{FE0F}' .
		'|\x{2198}\x{FE0F}|\x{2199}\x{FE0F}|\x{21A9}\x{FE0F}|\x{21AA}\x{FE0F}|\x{2328}' .
		'\x{FE0F}|\x{23CF}\x{FE0F}|\x{23ED}\x{FE0F}|\x{23EE}\x{FE0F}|\x{23EF}\x{FE0F}' .
		'|\x{23F1}\x{FE0F}|\x{23F2}\x{FE0F}|\x{23F8}\x{FE0F}|\x{23F9}\x{FE0F}|\x{23FA}' .
		'\x{FE0F}|\x{24C2}\x{FE0F}|\x{25AA}\x{FE0F}|\x{25AB}\x{FE0F}|\x{25B6}\x{FE0F}' .
		'|\x{25C0}\x{FE0F}|\x{25FB}\x{FE0F}|\x{25FC}\x{FE0F}|\x{2600}\x{FE0F}|\x{2601}' .
		'\x{FE0F}|\x{2602}\x{FE0F}|\x{2603}\x{FE0F}|\x{2604}\x{FE0F}|\x{260E}\x{FE0F}' .
		'|\x{2611}\x{FE0F}|\x{2618}\x{FE0F}|\x{261D}\x{FE0F}|\x{2620}\x{FE0F}|\x{2622}' .
		'\x{FE0F}|\x{2623}\x{FE0F}|\x{2626}\x{FE0F}|\x{262A}\x{FE0F}|\x{262E}\x{FE0F}' .
		'|\x{262F}\x{FE0F}|\x{2638}\x{FE0F}|\x{2639}\x{FE0F}|\x{263A}\x{FE0F}|\x{2640}' .
		'\x{FE0F}|\x{2642}\x{FE0F}|\x{265F}\x{FE0F}|\x{2660}\x{FE0F}|\x{2663}\x{FE0F}' .
		'|\x{2665}\x{FE0F}|\x{2666}\x{FE0F}|\x{2668}\x{FE0F}|\x{267B}\x{FE0F}|\x{267E}' .
		'\x{FE0F}|\x{2692}\x{FE0F}|\x{2694}\x{FE0F}|\x{2695}\x{FE0F}|\x{2696}\x{FE0F}' .
		'|\x{2697}\x{FE0F}|\x{2699}\x{FE0F}|\x{269B}\x{FE0F}|\x{269C}\x{FE0F}|\x{26A0}' .
		'\x{FE0F}|\x{26A7}\x{FE0F}|\x{26B0}\x{FE0F}|\x{26B1}\x{FE0F}|\x{26C8}\x{FE0F}' .
		'|\x{26CF}\x{FE0F}|\x{26D1}\x{FE0F}|\x{26D3}\x{FE0F}|\x{26E9}\x{FE0F}|\x{26F0}' .
		'\x{FE0F}|\x{26F1}\x{FE0F}|\x{26F4}\x{FE0F}|\x{26F7}\x{FE0F}|\x{26F8}\x{FE0F}' .
		'|\x{26F9}\x{FE0F}|\x{2702}\x{FE0F}|\x{2708}\x{FE0F}|\x{2709}\x{FE0F}|\x{270C}' .
		'\x{FE0F}|\x{270D}\x{FE0F}|\x{270F}\x{FE0F}|\x{2712}\x{FE0F}|\x{2714}\x{FE0F}' .
		'|\x{2716}\x{FE0F}|\x{271D}\x{FE0F}|\x{2721}\x{FE0F}|\x{2733}\x{FE0F}|\x{2734}' .
		'\x{FE0F}|\x{2744}\x{FE0F}|\x{2747}\x{FE0F}|\x{2763}\x{FE0F}|\x{2764}\x{FE0F}' .
		'|\x{27A1}\x{FE0F}|\x{2934}\x{FE0F}|\x{2935}\x{FE0F}|\x{2B05}\x{FE0F}|\x{2B06}' .
		'\x{FE0F}|\x{2B07}\x{FE0F}|\x{3030}\x{FE0F}|\x{303D}\x{FE0F}|\x{3297}\x{FE0F}' .
		'|\x{3299}\x{FE0F}|[\x{00A9}\x{00AE}\x{203C}\x{2049}\x{2122}\x{2139}\x{2194}-' .
		'\x{2199}\x{21A9}\x{21AA}\x{231A}\x{231B}\x{2328}\x{23CF}\x{23E9}-\x{23F3}' .
		'\x{23F8}-\x{23FA}\x{24C2}\x{25AA}\x{25AB}\x{25B6}\x{25C0}\x{25FB}-\x{25FE}' .
		'\x{2600}-\x{2604}\x{260E}\x{2611}\x{2614}\x{2615}\x{2618}\x{261D}\x{2620}' .
		'\x{2622}\x{2623}\x{2626}\x{262A}\x{262E}\x{262F}\x{2638}-\x{263A}\x{2640}' .
		'\x{2642}\x{2648}-\x{2653}\x{265F}\x{2660}\x{2663}\x{2665}\x{2666}\x{2668}' .
		'\x{267B}\x{267E}\x{267F}\x{2692}-\x{2697}\x{2699}\x{269B}\x{269C}\x{26A0}' .
		'\x{26A1}\x{26A7}\x{26AA}\x{26AB}\x{26B0}\x{26B1}\x{26BD}\x{26BE}\x{26C4}' .
		'\x{26C5}\x{26C8}\x{26CE}\x{26CF}\x{26D1}\x{26D3}\x{26D4}\x{26E9}\x{26EA}' .
		'\x{26F0}-\x{26F5}\x{26F7}-\x{26FA}\x{26FD}\x{2702}\x{2705}\x{2708}-\x{270D}' .
		'\x{270F}\x{2712}\x{2714}\x{2716}\x{271D}\x{2721}\x{2728}\x{2733}\x{2734}' .
		'\x{2744}\x{2747}\x{274C}\x{274E}\x{2753}-\x{2755}\x{2757}\x{2763}\x{2764}' .
		'\x{2795}-\x{2797}\x{27A1}\x{27B0}\x{27BF}\x{2934}\x{2935}\x{2B05}-\x{2B07}' .
		'\x{2B1B}\x{2B1C}\x{2B50}\x{2B55}\x{3030}\x{303D}\x{3297}\x{3299}\x{1F004}' .
		'\x{1F0CF}\x{1F170}\x{1F171}\x{1F17E}\x{1F17F}\x{1F18E}\x{1F191}-\x{1F19A}' .
		'\x{1F201}\x{1F202}\x{1F21A}\x{1F22F}\x{1F232}-\x{1F23A}\x{1F250}\x{1F251}' .
		'\x{1F300}-\x{1F321}\x{1F324}-\x{1F393}\x{1F396}\x{1F397}\x{1F399}-\x{1F39B}' .
		'\x{1F39E}-\x{1F3F0}\x{1F3F3}-\x{1F3F5}\x{1F3F7}-\x{1F3FA}\x{1F400}-\x{1F4FD}' .
		'\x{1F4FF}-\x{1F53D}\x{1F549}-\x{1F54E}\x{1F550}-\x{1F567}\x{1F56F}\x{1F570}' .
		'\x{1F573}-\x{1F57A}\x{1F587}\x{1F58A}-\x{1F58D}\x{1F590}\x{1F595}\x{1F596}' .
		'\x{1F5A4}\x{1F5A5}\x{1F5A8}\x{1F5B1}\x{1F5B2}\x{1F5BC}\x{1F5C2}-\x{1F5C4}' .
		'\x{1F5D1}-\x{1F5D3}\x{1F5DC}-\x{1F5DE}\x{1F5E1}\x{1F5E3}\x{1F5E8}\x{1F5EF}' .
		'\x{1F5F3}\x{1F5FA}-\x{1F64F}\x{1F680}-\x{1F6C5}\x{1F6CB}-\x{1F6D2}\x{1F6D5}-' .
		'\x{1F6D7}\x{1F6DC}-\x{1F6E5}\x{1F6E9}\x{1F6EB}\x{1F6EC}\x{1F6F0}\x{1F6F3}-' .
		'\x{1F6FC}\x{1F7E0}-\x{1F7EB}\x{1F7F0}\x{1F90C}-\x{1F93A}\x{1F93C}-\x{1F945}' .
		'\x{1F947}-\x{1F9AF}\x{1F9B4}-\x{1F9FF}\x{1FA70}-\x{1FA7C}\x{1FA80}-\x{1FA88}' .
		'\x{1FA90}-\x{1FABD}\x{1FABF}-\x{1FAC5}\x{1FACE}-\x{1FADB}\x{1FAE0}-\x{1FAE8}' .
		'])+/ui';
}
