﻿using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Mvc.Infrastructure;
using Microsoft.AspNetCore.WebUtilities;
using Microsoft.Extensions.DependencyInjection;

namespace OperationResults.AspNetCore;

public static class OperationResultExtensions
{
    public static IActionResult ToResponse(this Result result, HttpContext httpContext, int? successStatusCode = null)
    {
        if (result.Success)
        {
            return new StatusCodeResult(successStatusCode.GetValueOrDefault(StatusCodes.Status204NoContent));
        }

        return Problem(httpContext, result.FailureReason, null, result.ErrorMessage, result.ErrorDetail, result.ValidationErrors);
    }

    public static IActionResult ToResponse(this Result result, HttpContext httpContext, string? routeName, object? routeValues = null, int? successStatusCode = null)
    {
        if (result.Success)
        {
            var createdAtRouteResult = new CreatedAtRouteResult(routeName, routeValues, null)
            {
                StatusCode = successStatusCode.GetValueOrDefault(StatusCodes.Status201Created)
            };

            return createdAtRouteResult;
        }

        return Problem(httpContext, result.FailureReason, null, result.ErrorMessage, result.ErrorDetail, result.ValidationErrors);
    }

    public static IActionResult ToResponse<T>(this Result<T> result, HttpContext httpContext, int? successStatusCode = null)
        => result.ToResponse(httpContext, null, null, successStatusCode);

    public static IActionResult ToResponse<T>(this Result<T> result, HttpContext httpContext, string? routeName, object? routeValues = null, int? successStatusCode = null)
    {
        if (result.Success)
        {
            if (result.Content is not null)
            {
                if (!string.IsNullOrWhiteSpace(routeName))
                {
                    var createdAtRouteResult = new CreatedAtRouteResult(routeName, routeValues, result.Content)
                    {
                        StatusCode = successStatusCode.GetValueOrDefault(StatusCodes.Status201Created)
                    };

                    return createdAtRouteResult;
                }
                else if (result.Content is StreamFileContent streamFileContent)
                {
                    var fileStreamResult = new FileStreamResult(streamFileContent.Content, streamFileContent.ContentType)
                    {
                        FileDownloadName = streamFileContent.DownloadFileName,
                    };

                    return fileStreamResult;
                }
                else if (result.Content is ByteArrayFileContent byteArrayFileContent)
                {
                    var fileContentResult = new FileContentResult(byteArrayFileContent.Content, byteArrayFileContent.ContentType)
                    {
                        FileDownloadName = byteArrayFileContent.DownloadFileName
                    };

                    return fileContentResult;
                }

                var okResult = new ObjectResult(result.Content)
                {
                    StatusCode = successStatusCode.GetValueOrDefault(StatusCodes.Status200OK)
                };

                return okResult;
            }

            return new StatusCodeResult(successStatusCode.GetValueOrDefault(StatusCodes.Status204NoContent));
        }

        return Problem(httpContext, result.FailureReason, result.Content, result.ErrorMessage, result.ErrorDetail, result.ValidationErrors);
    }

    private static IActionResult Problem(HttpContext httpContext, int failureReason, object? content = null, string? title = null, string? detail = null, IEnumerable<ValidationError>? validationErrors = null)
    {
        var options = httpContext.RequestServices.GetService<OperationResultOptions>() ?? new OperationResultOptions();
        var statusCode = options.MapStatusCodes ? options.GetStatusCode(failureReason) : failureReason;

        if (content is not null)
        {
            var objectResult = new ObjectResult(content)
            {
                StatusCode = statusCode
            };

            return objectResult;
        }

        var problemDetailsFactory = httpContext.RequestServices.GetRequiredService<ProblemDetailsFactory>();
        var problemDetails = problemDetailsFactory.CreateProblemDetails(httpContext, statusCode, title ?? ReasonPhrases.GetReasonPhrase(statusCode),
            detail: detail, instance: httpContext.Request.Path);
        problemDetails.Type ??= $"https://httpstatuses.io/{statusCode}";

        if (validationErrors?.Any() ?? false)
        {
            if (options.ErrorResponseFormat == ErrorResponseFormat.Default)
            {
                var errors = validationErrors.GroupBy(v => v.Name).ToDictionary(k => k.Key, v => v.Select(e => e.Message));
                problemDetails.Extensions.Add("errors", errors);
            }
            else
            {
                problemDetails.Extensions.Add("errors", validationErrors);
            }
        }

        var problemDetailsResult = new JsonResult(problemDetails)
        {
            StatusCode = statusCode,
            ContentType = "application/problem+json"
        };

        return problemDetailsResult;
    }
}
