package util

import (
	"errors"
	"fmt"
	"strings"

	"github.com/loft-sh/log"
	"github.com/loft-sh/log/survey"
	"github.com/loft-sh/log/terminal"
	"github.com/spf13/cobra"
)

var (
	NamespaceNameOnlyUseLine   string
	NamespaceNameOnlyValidator cobra.PositionalArgs

	VClusterNameOnlyUseLine string

	VClusterNameOnlyValidator cobra.PositionalArgs
)

var (
	ErrNonInteractive   = errors.New("terminal is not interactive")
	ErrTooManyArguments = errors.New("too many arguments specified")
)

func init() {
	NamespaceNameOnlyUseLine, NamespaceNameOnlyValidator = NamedPositionalArgsValidator(true, true, "NAMESPACE_NAME")
	VClusterNameOnlyUseLine, VClusterNameOnlyValidator = NamedPositionalArgsValidator(true, true, "VCLUSTER_NAME")
}

// NamedPositionalArgsValidator returns a cobra.PositionalArgs that returns a helpful
// error message if the arg number doesn't match.
// It also returns a string that can be appended to the cobra useline
//
// Example output for extra arguments with :
//
//	$ command arg asdf
//	[fatal]  command ARG_1 [flags]
//	Invalid Args: received 2 arguments, expected 1, extra arguments: "asdf"
//	Run with --help for more details
//
// Example output for missing arguments:
//
//	$ command
//	[fatal]  command ARG_1 [flags]
//	Invalid Args: received 0 arguments, expected 1, please specify missing: "ARG_!"
//	Run with --help for more details on arguments
func NamedPositionalArgsValidator(failMissing, failExtra bool, expectedArgs ...string) (string, cobra.PositionalArgs) {
	return " " + strings.Join(expectedArgs, " "), func(cmd *cobra.Command, args []string) error {
		numExpectedArgs := len(expectedArgs)
		numArgs := len(args)
		numMissing := numExpectedArgs - numArgs

		if numMissing == 0 {
			return nil
		}

		// didn't receive as many arguments as expected
		if numMissing > 0 && failMissing {
			// the last numMissing expectedArgs
			missingKeys := strings.Join(expectedArgs[len(expectedArgs)-(numMissing):], ", ")
			return fmt.Errorf("%s\nInvalid Args: received %d arguments, expected %d, please specify missing: %q\nRun with --help for more details on arguments", cmd.UseLine(), numArgs, numExpectedArgs, missingKeys)
		}

		// received more than expected
		if numMissing < 0 && failExtra {
			// received more than expected
			numExtra := -numMissing
			// the last numExtra args
			extraValues := strings.Join(args[len(args)-numExtra:], ", ")
			return fmt.Errorf("%s\nInvalid Args: received %d arguments, expected %d, extra arguments: %q\nRun with --help for more details on arguments", cmd.UseLine(), numArgs, numExpectedArgs, extraValues)
		}

		return nil
	}
}

// PromptForArgs expects that the terminal is interactive and the number of args, matched the number of argNames, in the
// order they should appear and will prompt one by one for the missing args adding them to the args slice and returning
// a new set for a command to use. It returns the args, rather than a nil slice so they're unaltered in error cases.
func PromptForArgs(l log.Logger, args []string, argNames ...string) ([]string, error) {
	if !terminal.IsTerminalIn {
		return args, ErrNonInteractive
	}
	if len(args) > len(argNames) {
		return args, ErrTooManyArguments
	}

	if len(args) == len(argNames) {
		return args, nil
	}

	for i := range argNames[len(args):] {
		answer, err := l.Question(&survey.QuestionOptions{
			Question: fmt.Sprintf("Please specify %s", argNames[i]),
		})
		if err != nil {
			return args, err
		}
		args = append(args, answer)
	}

	return args, nil
}
