# Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
# See https://llvm.org/LICENSE.txt for license information.
# SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
# Also available under a BSD-style license. See LICENSE.

# This file describes the sets of tests expected to fail for each config.
# This information is deliberately kept in a side table, rather than
# in-situ on the test, as a deliberate layering decision: tests should
# have unique keys to identify them and enable side tables of various kinds
# (this includes down into lower parts of the stack, where a side table
# might be used to keep more elaborate sets of testing configurations).

from torch_mlir_e2e_test.test_suite import COMMON_TORCH_MLIR_LOWERING_XFAILS
from torch_mlir._version import torch_version_for_comparison, version

print(f"TORCH_VERSION_FOR_COMPARISON =", torch_version_for_comparison())

LINALG_XFAIL_SET = COMMON_TORCH_MLIR_LOWERING_XFAILS | {
    # lowering to torch backend IR fails due to unsupported op: aten.upsample_[mode/dims].vec
    # these interpolate tests are added specifically to test onnx.Resize.
    "InterpolateDynamicModule_sizes_bilinear",
    "InterpolateDynamicModule_sizes_nearest",
    "InterpolateStaticModule_scales_bilinear_align_corners",
    "InterpolateDynamicModule_scales_recompute_bilinear",
    "ElementwiseFloatTensorGtIntTensorModule_basic",
    "AtenIntMM_basic",
    # unimplemented lowering torch -> linalg for torchvision.deform_conv2d
    # this is added to check the torch.onnx.export -> import_onnx -> torch path
    "DeformConv2D_basic",
    "ReduceAnyDimFloatModule_basic",
    "UnfoldModule_basic",
    # _trilinear is an implementation of einsum, but sets dimensions to zero
    # if a dimension is specified in all expand lists, and not in sumdim list.
    # This is a bug in the implementation of _trilinear in PyTorch.
    "Aten_TrilinearModuleZerodDimBug_basic",
    # missing lowering from aten.pow.Tensor_Tensor for integer result
    "PowIntIntModule_basic",
}

if torch_version_for_comparison() < version.parse("2.5.0.dev"):
    LINALG_XFAIL_SET = LINALG_XFAIL_SET | {
        # Error: 'torch.aten.scaled_dot_product_attention' op expected 8 operands, but found 7
        # WORKS FOR TORCH VERSION 2.5.0.dev20240902, REMOVE WHEN ENABLE_GQA IS PUT IN STABLE
        "ScaledDotProductAttentionBoolMaskModule_basic",
        "ScaledDotProductAttentionDifferentCausalModule_basic",
        "ScaledDotProductAttentionDifferentDynamicCausalModule_basic",
        "ScaledDotProductAttentionDifferentModule_basic",
        "ScaledDotProductAttentionMaskModule_basic",
        "ScaledDotProductAttentionSameCausalModule_basic",
        "ScaledDotProductAttentionSameDynamicModule_basic",
        "ScaledDotProductAttentionSameModule_basic",
    }

LINALG_CRASHING_SET = {
    # Runtime op verification: Out of bounds access
    "AtenDiagEmbedNegOffsetDiag_basic",
    "AtenDiagEmbedNonDefault4DDiag_basic",
    "AtenDiagEmbedOffsetDiag_basic",
    "AtenDiagEmbedRevDimDiag_basic",
    "AtenEmbeddingBagStaticModule_basic",
    "AtenEmbeddingBagSumExample_basic",
    "Aten_EmbeddingBagExample_basic",
    # Runtime op verification: subview is out-of-bounds of the base memref
    "Conv_Transpose1dModule_basic",
    "Conv_Transpose1dStaticModule_basic",
    "Conv_Transpose2dModule_basic",
    "Conv_Transpose2dStaticModule_basic",
    "Conv_Transpose3dModule_basic",
    "Conv_Transpose3dStaticModule_basic",
    "ConvolutionModule2DTransposeStridedStatic_basic",
    "ConvolutionModule2DTransposeStrided_basic",
    "GridSamplerBasic1_basic",
    "GridSamplerBasic2_basic",
    "GridSamplerBasic3_basic",
    "GridSamplerBasic4_basic",
    # Runtime op verification: stride mismatch in memref.cast
    "ReduceAllDimEmpty_basic",
    "TraceUnsignedIntModule_empty",
    "TraceModule_empty",
    # Crashes due to copy to a smaller destination buffer than the source buffer.
    "SliceCopyStartGreaterThanDimSize_Module_basic",
}

TORCHDYNAMO_XFAIL_SET = {
    #### General TorchDynamo/PyTorch errors
    # torch._dynamo.exc.Unsupported: Tensor.item
    "CumsumModule_basic",
    "CumprodModule_basic",
    # TypeError: new_empty(): argument 'size' (position 1) must be tuple of ints, but found element of type NoneType at pos 0
    # RuntimeError: Failed running call_function aten.convolution_backward(...
    # https://github.com/pytorch/pytorch/issues/89629
    "ConvolutionBackwardModule2DPadded_basic",
    "ConvolutionBackwardModule2D_basic",
    # Size result mismatch (exposed by downstream canonicalizer
    # on incompatabile casts).
    # https://github.com/pytorch/pytorch/issues/119407
    "ConvolutionBackwardModule2DStrided_basic",
    # RuntimeError: Index tensor must have the same number of dimensions as self tensor
    # RuntimeError: Failed running call_function aten.nll_loss_backward(...
    # https://github.com/pytorch/pytorch/issues/89630
    "NllLossModuleBackward1DMeanWeight_basic",
    "NllLossModuleBackward1DMean_basic",
    "NllLossModuleBackward1DSumWeight_basic",
    "NllLossModuleBackward1DSum_basic",
    "NllLossModuleBackward1DWeight_basic",
    "NllLossModuleBackward1D_basic",
    # TypeError: uniform() missing 2 required keyword-only arguments: 'dtype' and 'device'
    # RuntimeError: Failed running call_function aten.uniform(...
    # https://github.com/pytorch/torchdynamo/issues/1954
    "UniformNoCorrelationModule_basic",
    #### Torch-MLIR internal compiler errors
    # These are probably due to slightly different ops being recorded by
    # torchdynamo vs. torchscript.
    # No upstream decompositions.
    # %6:4 = torch.operator "aten._embedding_bag_forward_only"(%1, %3, %5, %false, %int0, %false, %none, %false, %int-1) : (!torch.tensor<*,f32>, !torch.tensor<*,si64>, !torch.tensor<*,si64>, !torch.bool, !torch.int, !torch.bool, !torch.none, !torch.bool, !torch.int) -> (!torch.tensor, !torch.tensor, !torch.tensor, !torch.tensor)
    # See also: https://github.com/pytorch/torchdynamo/issues/327
    "AtenEmbeddingBagSumExample_basic",
    # error: unsupported by backend contract: tensor with unknown rank
    # note: see current operation: %1 = "torch.tensor_static_info_cast"(%arg0) : (!torch.vtensor<[5,4,3,2,1],f32>) -> !torch.vtensor<*,f32>
    "ElementwisePreluModule_basic",
    # error: torch._dynamo.exc.BackendCompilerFailed: backend='compiler_fn' raised: AssertionError: Unregistered operation: torch.aten._prelu_kernel
    "ElementwisePreluStaticModule_basic",
    # ERROR: value (Tensor with shape=[2, 3, 6, 10], dtype=torch.float32, min=-1.336e-32, max=+0.9152, mean=+0.4837) is not close to golden value (Tensor with shape=[2, 3, 6, 10], dtype=torch.float32, min=+0.02233, max=+0.9152, mean=+0.4777)
    "UpSampleNearest2dDynamicFactor_basic",
    "ReduceMaxAlongDimUnsignedInt_basic",
    "ReduceMinAlongDimUnsignedInt_basic",
    # ERROR: value (-56) is not equal to golden value (200)
    "AtenIntTensorByteDtypeModule_basic",
    # ERROR: assert isinstance(e, FakeTensor)
    "ElementwiseAddScalar_NumToTensorFloat_Module_basic",
    # ERROR: assert isinstance(e, FakeTensor)
    "RsubInt0d_NumToTensor_Module_basic",
    # ERROR: RuntimeError: Found a custom (non-ATen) operator that either mutates or its inputs: prims::squeeze.. Getting these operators to work with functionalization requires some extra work. For mutable ops you need to register a corresponding out-of-place variant of the op, and you also need to register a Functionalization kernel that performs some boilerplate, telling functionalization to map from the mutable op to the out-of-place op. See a more complete example of how to do this at https://gist.github.com/bdhirsh/7dadbf6296f8f7d1abcf4c482f438aaa.
    "PrimsSqueezeModule_basic",
    "PrimsSqueezeEmptyDimensionsModule_basic",
    "SplitDimStaticModule_basic",
    "SplitDimDynamicModule_basic",
    # ERROR: RuntimeError: Found a custom (non-ATen) operator that either mutates or its inputs: prims::view_of.. Getting these operators to work with functionalization requires some extra work. For mutable ops you need to register a corresponding out-of-place variant of the op, and you also need to register a Functionalization kernel that performs some boilerplate, telling functionalization to map from the mutable op to the out-of-place op. See a more complete example of how to do this at https://gist.github.com/bdhirsh/7dadbf6296f8f7d1abcf4c482f438aaa.
    "PrimsViewOfModule_basic",
    "PrimsViewOfZeroRankModule_basic",
    # See https://github.com/llvm/torch-mlir/pull/2040 and corresponding upstream issue
    # https://github.com/pytorch/pytorch/issues/99752.
    # torch._dynamo.exc.Unsupported: call_function BuiltinVariable(bool) [TensorVariable()] {}
    "TensorToBoolZeroRank_basic",
    "TensorToBool_basic",
    # START tests failing due to: torch._dynamo.exc.Unsupported: call_function BuiltinVariable(float) [TensorVariable()] {}
    "AtenSubFloatModule_basic",
    "AtenMulFloatModule_basic",
    "BoolFloatFalseModule_basic",
    "BoolFloatTrueModule_basic",
    "CeilFloatModule_basic",
    "DivFloatModule_basic",
    "GeFloatIntModule_basic",
    "GeFloatModule_basic",
    "GtFloatIntModule_basic",
    "NeFloatIntModule_basic",
    "SubFloatModule_basic",
    "MulFloatModule_basic",
    "TensorToFloatZeroRank_basic",
    "TensorToFloat_basic",
    # END tests failing due to: torch._dynamo.exc.Unsupported: call_function BuiltinVariable(float) [TensorVariable()] {}
    # START tests failing due to: torch._dynamo.exc.Unsupported: call_function BuiltinVariable(int) [TensorVariable()] {}
    "AddIntModule_basic",
    "AddFloatIntModule_basic",
    "AtenIntTensorCharDtypeModule_basic",
    "BoolIntFalseModule_basic",
    "BoolIntTrueModule_basic",
    "DivIntModule_basic",
    "EqIntModule_basic",
    "GeIntModule_basic",
    "GtIntModule_basic",
    "MulIntModule_basic",
    "NeIntModule_basic",
    "SqrtIntModule_basic",
    "SubIntModule_basic",
    "TensorToIntZeroRank_basic",
    "TensorToInt_basic",
    "UnsafeViewCollapseDynamicWithAtenSizeIntModule_basic",
    "ViewCollapseDynamicWithAtenSizeIntModule_basic",
    # END tests failing due to: torch._dynamo.exc.Unsupported: call_function BuiltinVariable(int) [TensorVariable()] {}
    # ERROR: torch._dynamo.exc.Unsupported: Tensor.item
    "AtenItemIntOpModule_basic",
    "AtenItemFpOpModule_basic",
    # ERROR: torch._dynamo.exc.Unsupported: call_method ListVariable() sort [] {'reverse': ConstantVariable(bool)}
    "SortIntListReverse_basic",
    # ERROR: torch._dynamo.exc.Unsupported: call_method ListVariable() sort [] {}
    "SortIntList_basic",
    # START tests failing due to: torch._dynamo.exc.Unsupported: data dependent operator: aten._local_scalar_dense.default
    "AtenFloatScalarModule_basic",
    "AtenIntBoolOpModule_basic",
    "QuantizedMLP_basic",
    "QuantizedSingleLayer_basic",
    "QuantizedBatchedInputSingleLayer_basic",
    "QuantizedNoLayer_basic",
    "ScalarImplicitFloatModule_basic",
    "ScalarImplicitIntModule_basic",
    # END tests failing due to: torch._dynamo.exc.Unsupported: data dependent operator: aten._local_scalar_dense.default
    # START tests failing due to: torch._dynamo.exc.Unsupported: dynamic shape operator: aten.bincount.default
    "BincountMinlengthModule_basic",
    "BincountModule_basic",
    "BincountStaticSizeModule_basic",
    # END tests failing due to: torch._dynamo.exc.Unsupported: dynamic shape operator: aten.bincount.default
    # ERROR: torch._dynamo.exc.Unsupported: torch.* op returned non-Tensor bool call_function aten.Bool
    "BoolFloatConstantModule_basic",
    "BoolIntConstantModule_basic",
    # ERROR: torch._dynamo.exc.Unsupported: torch.* op returned non-Tensor int call_function aten.size
    "ViewSizeFromOtherTensor_basic",
    # ERROR: torch._dynamo.exc.Unsupported: torch.* op returned non-Tensor bool call_function aten.__contains__
    "ContainsIntList_False",
    "ContainsIntList_True",
    # ERROR: torch._dynamo.exc.Unsupported: torch.* op returned non-Tensor bool call_function aten.all
    "AllBoolFalseModule_basic",
    "AllBoolTrueModule_basic",
    # ERROR: torch._dynamo.exc.Unsupported: torch.* op returned non-Tensor bool call_function aten.any
    "AnyBoolFalseModule_basic",
    "AnyBoolTrueModule_basic",
    # ERROR: torch._dynamo.exc.Unsupported: torch.* op returned non-Tensor float call_function aten.sqrt
    "SqrtIntConstantModule_basic",
    # ERROR: torch._dynamo.exc.Unsupported: torch.* op returned non-Tensor int call_function aten.size
    "BroadcastDynamicDimModule_basic",
    # START tests failing due to: torch._dynamo.exc.Unsupported: torch.* op returned non-Tensor int call_function aten.Int
    "AtenIntBoolOpConstFalseModule_basic",
    "AtenIntBoolOpConstTrueModule_basic",
    "IntFloatModule_basic",
    # END tests failing due to: torch._dynamo.exc.Unsupported: torch.* op returned non-Tensor int call_function aten.Int
    # ERROR: torch._dynamo.exc.Unsupported: torch.* op returned non-Tensor int call_function aten.len
    "LenStrModule_basic",
    # ERROR: torch._dynamo.exc.Unsupported: torch.* op returned non-Tensor int call_function aten.numel
    "NumelModule_basic",
    "NumelZeroRankModule_basic",
    # ERROR: torch._dynamo.exc.Unsupported: torch.* op returned non-Tensor int call_function prim.max
    "PrimMaxIntModule_basic",
    # ERROR: torch._dynamo.exc.Unsupported: torch.* op returned non-Tensor int call_function prim.min
    "PrimMinIntModule_basic",
    "PrimMinIntDynamicModule_basic",
    # START tests failing due to: empty graph in dynamo
    "IsFloatingPointFloat_True",
    "IsFloatingPointInt_False",
    "TorchPrimLoopForLikeModule_basic",
    "TorchPrimLoopWhileLikeModule_basic",
    "ScalarConstantTupleModule_basic",
    # END tests failing due to: empty graph in dynamo
    # ERROR due to: backend never runs because of empty frame
    "ConstantBoolParameterModule_basic",
    # START tests failing due to: 'torch.aten.mul.Tensor' op operand #1 must be Any Torch tensor type, but got '!torch.float'
    "UpSampleNearest2dDynamicSize_basic",
    "UpSampleNearest2dStaticFactor_basic",
    "UpSampleNearest2dStaticSize_basic",
    "UpSampleNearest2d_basic",
    # END tests failing due to: 'torch.aten.mul.Tensor' op operand #1 must be Any Torch tensor type, but got '!torch.float'
    # START tests failing due to: 'torch.aten.add.Tensor' op operand #1 must be Any Torch tensor type, but got '!torch.float'
    "ElementwiseAddScalarFloatModule_basic",
    # END tests failing due to: 'torch.aten.add.Tensor' op operand #1 must be Any Torch tensor type, but got '!torch.float'
    # ERROR: 'torch.aten.add.Tensor' op operand #1 must be Any Torch tensor type, but got '!torch.int'
    "ElementwiseAddScalar_TensorLiteralInt32_Module_basic",
    "HBC_basic",
    # ERROR: 'torch.aten.div.Tensor' op operand #1 must be Any Torch tensor type, but got '!torch.float'
    "ElementwiseDivScalarModule_basic",
    # ERROR: 'torch.aten.div.Tensor' op operand #1 must be Any Torch tensor type, but got '!torch.int'
    "ElementwiseAtenDivIntScalarModule_basic",
    # ERROR: 'torch.aten.sub.Tensor' op operand #1 must be Any Torch tensor type, but got '!torch.float'
    "ElementwiseSubScalarFloatModule_basic",
    "ElementwiseSubScalarIntModule_basic",
    # ERROR: Exception: Unsupported: missing default value for argument 0 in schema for aten.div.Tensor_mode
    "ElementwiseAtenFloorDivideScalarNegativeModule_basic",
    "ElementwiseAtenFloorDivideScalarModule_basic",
    "ElementwiseDivScalarRoundingModeFloorModule_basic",
    "ElementwiseDivScalarRoundingModeTruncModule_basic",
    "ElementwiseDivScalarRoundingModeFloorStaticModule_basic",
    "ElementwiseDivScalarRoundingModeTruncStaticModule_basic",
    "ElementwiseDivScalarRoundingModeFloorIntStaticModule_basic",
    "ElementwiseDivScalarRoundingModeTruncIntStaticModule_basic",
    # ERROR: 'torch.aten.mul.Tensor' op operand #1 must be Any Torch tensor type, but got '!torch.int'
    "AdaptiveAvgPool1dStaticLargerOutput_basic",
    "AdaptiveAvgPool1dGeneralDynamic_basic",
    "AdaptiveAvgPool1dGeneralDynamicNoBatches_basic",
    "AdaptiveAvgPool2dDynamic_basic",
    "AdaptiveAvgPool2dDynamicNoBatch_basic",
    # ERROR: Exception: Unsupported op: get_attr
    "NumToTensorFloatModule_basic",
    "NumToTensorIntModule_basic",
    "TensorFloatModule_basic",
    "TensorIntModule_basic",
    # START tests failing due to: complex floating point ops
    # END tests failing due to: complex floating point ops
    # ERROR: Exception: Unsupported: return type List[Tensor] in schema for aten.unbind.int
    "UnbindIntListUnpack_Module_basic",
    "UnbindIntGetItem_Module_basic",
    # ERROR: torch._dynamo.exc.Unsupported: call_function BuiltinVariable(float) [TensorVariable()] {}
    "ScatterValueFloatModule_basic",
    # ERROR: torch._dynamo.exc.Unsupported: call_function BuiltinVariable(int) [TensorVariable()] {}
    "ScatterValueIntModule_basic",
    # AssertionError: Unregistered operation: torch.aten._unsafe_index_put
    "UnsafeIndexPutHackedTwin1DFloatNonAccumulateModule_basic",
    # AssertionError: Unregistered operation: torch.aten._embedding_bag_forward_only
    "AtenEmbeddingBagStaticModule_basic",
    # Lowering not present for this case
    "ElementwiseToDtypeI64ToUI8Module_basic",
    # torch._dynamo.exc.TorchRuntimeError: Failed running call_function <built-in method add of type object at 0x7f4f8b05a720>(*(FakeTensor(..., size=(3, 4), dtype=torch.int8), 3, 2), **{}): Tensor with dtype torch.int64 is not the expected dtype of torch.int8!
    "ElementwiseAddScalarInt8Module_basic",
    # ERROR: dtype (torch.int64) is not equal to golden dtype (torch.float32)
    "ThresholdBackward2dMixedModule_basic",
    # ERROR: shape (torch.Size([12])) is not equal to golden shape (torch.Size([3, 4]))
    "ArangeStartOutViewModule_basic",
    # Dynamo does not support tracing quantized tensors
    "ElementwiseDequantizePerChannelModule_basic",
    "ElementwiseDequantizePerTensorModule_basic",
    "ElementwiseQuantizePerTensorModule_basic",
    "ElementwiseQuantizePerTensorUIntModule_basic",
    "AtenMmQuint8_basic",
    "AtenMmQint8_basic",
    "AtenMmQMixedSigni8_basic",
    "AtenMatmulQMixedSigni8Transpose_basic",
    "AtenMatmulQMixedSigni8_basic",
    "AtenMatmulQint8MV_basic",
    "AtenMatmulQint8VV_basic",
    "AtenMatmulQint8VM_basic",
    "AtenMatmulQint8_basic",
    "QuantizedReluInt32_basic",
    "QuantizedReluInt8_basic",
    "QuantizedReluUint8_basic",
    "Conv2dQInt8Module_basic",
    "Conv2dQInt8Module_depthwise",
    "Conv2dQInt8Module_grouped",
    "Conv2dQInt8Module_not_depthwise",
    "Conv2dQInt8PerChannelModule_basic",
    "Conv2dQInt8PerChannelModule_depthwise",
    "Conv2dQInt8PerChannelModule_grouped",
    "ConvTranspose2DQInt8_basic",
    # Dynamo not supporting conv_tbc
    "ConvTbcModule_basic",
    "FloatImplicitModule_basic",
    "IntImplicitModule_basic",
    # Others
    "ExponentialModule_basic",
    "GridSamplerBasic1_basic",
    "GridSamplerBasic2_basic",
    "GridSamplerBasic3_basic",
    "FakeQuantizePerTensorAffineModule_basic",
    "FakeQuantizePerTensorAffineDynamicShapeModule_basic",
    "FakeQuantizePerTensorAffineRoundToEvenModule_basic",
}

TORCHDYNAMO_CRASHING_SET = {
    # No upstream decompositions.
    # %6:4 = torch.operator "aten._embedding_bag_forward_only"(%1, %3, %5, %false, %int0, %false, %none, %false, %int-1) : (!torch.tensor<*,f32>, !torch.tensor<*,si64>, !torch.tensor<*,si64>, !torch.bool, !torch.int, !torch.bool, !torch.none, !torch.bool, !torch.int) -> (!torch.tensor, !torch.tensor, !torch.tensor, !torch.tensor)
    # See also: https://github.com/pytorch/torchdynamo/issues/327
    "Aten_EmbeddingBagExample_basic",
    "FullModuleInt3D_basic",
    "ThresholdBackward1dIntModule_basic",
    "ThresholdBackward2dIntModule_basic",
    "ThresholdBackward3dIntModule_basic",
    # See https://github.com/llvm/torch-mlir/issues/2050
    "ElementwiseCloneChannelsLastMemoryFormatModule_basic",
    "ElementwiseCloneContiguousModule_basic",
    "ElementwiseCloneModule_basic",
    "ExpandAsFloatModule_basic",
    "ExpandAsIntModule_basic",
    "ExpandModule_basic",
    "MoveDimIntModule_basic",
    "MoveDimIntNegativeIndexModule_basic",
    "NarrowVerticalTest2_basic",
    "NarrowVerticalTest_basic",
    "NumpyTRank2Module_basic",
    "NumpyTRankNDynamicModule_basic",
    "NumpyTRankNStaticModule_basic",
    "PermuteModule_basic",
    "PermuteNegativeIndexModule_basic",
    "SelectIntNegativeDimAndIndexStaticModule_basic",
    "TestMultipleTensorAndPrimitiveTypesReturn_basic",
    "TModuleRank2_basic",
    "ToCopyModule_basic",
    "TransposeIntModule_basic",
    "TransposeIntNegDimsModule_basic",
    "IndexPutImpl2DNoneIndexStaticModule_basic",
    "MaxPool3dCeilModeTrueModule_basic",
    "MaxPool3dEmptyStrideStaticModule_basic",
    "MaxPool3dLargeDatadModule_basic",
    "MaxPool3dModuleRandomSimple_basic",
    "MaxPool3dModule_basic",
    "MaxPool3dStaticCeilModeTrueModule_basic",
    "MaxPool3dStaticModule_basic",
    # Looks like incorrect fx graph conversion
    "ElementwiseAddScalar_TensorLiteralInt32_Module_basic",
}

FX_IMPORTER_XFAIL_SET = {
    "TimeOutModule_basic",  # this test is expected to time out
    "ReduceAnyDimFloatModule_basic",
    "AddFloatIntModule_basic",
    "AllBoolFalseModule_basic",
    "AllBoolTrueModule_basic",
    "AnyBoolFalseModule_basic",
    "AnyBoolTrueModule_basic",
    "ArangeStartOutViewModule_basic",
    "AtenFloatScalarModule_basic",
    "AtenIntBoolOpConstFalseModule_basic",
    "AtenIntBoolOpConstTrueModule_basic",
    "AtenIntBoolOpModule_basic",
    "AtenIntMM_basic",
    "AtenNonzero1DDynamicModule_basic",  # no lowering for torch.aten.sym_constrain_range_for_size
    "Aten_TrilinearModuleVaryingRanks_basic",
    "Aten_TrilinearModuleZerodDimBug_basic",
    "QuantizedReluInt32_basic",
    "QuantizedReluInt8_basic",
    "QuantizedReluUint8_basic",
    "BincountMinlengthModule_basic",
    "BincountModule_basic",
    "BincountStaticSizeModule_basic",
    "BoolFloatConstantModule_basic",
    "BoolFloatFalseModule_basic",
    "BoolFloatTrueModule_basic",
    "BoolIntConstantModule_basic",
    "BoolIntFalseModule_basic",
    "BoolIntTrueModule_basic",
    "BroadcastDynamicDimModule_basic",
    "CeilFloatModule_basic",
    "ContainsIntList_False",
    "ContainsIntList_True",
    "ConvTbcModule_basic",
    "ConvolutionBackwardModule2DPadded_basic",
    "ConvolutionBackwardModule2DStrided_basic",
    "ConvolutionBackwardModule2D_basic",
    "CumsumModule_basic",
    "CumprodModule_basic",
    "DeformConv2D_basic",
    "DivIntModule_basic",
    "ElementwiseDequantizePerChannelModule_basic",
    "ElementwiseDequantizePerTensorModule_basic",
    "ElementwiseQuantizePerTensorModule_basic",
    "ElementwiseQuantizePerTensorUIntModule_basic",
    "ElementwiseToDtypeI64ToUI8Module_basic",
    "FloatImplicitModule_basic",
    "GeFloatIntModule_basic",
    "GeIntModule_basic",
    "GtFloatIntModule_basic",
    "IntFloatModule_basic",
    "IntImplicitModule_basic",
    "LenStrModule_basic",
    "NativeGroupNormBackwardModule_basic",
    "NeFloatIntModule_basic",
    "NllLossModuleBackward1DMeanWeight_basic",
    "NllLossModuleBackward1DMean_basic",
    "NllLossModuleBackward1DSumWeight_basic",
    "NllLossModuleBackward1DSum_basic",
    "NllLossModuleBackward1DWeight_basic",
    "NllLossModuleBackward1D_basic",
    "NumelModule_basic",
    "NumelZeroRankModule_basic",
    "PowIntIntModule_basic",
    "PrimMaxIntModule_basic",
    "PrimMinIntDynamicModule_basic",
    "PrimMinIntModule_basic",
    "PrimsSqueezeEmptyDimensionsModule_basic",
    "PrimsSqueezeModule_basic",
    "PrimsViewOfModule_basic",
    "PrimsViewOfZeroRankModule_basic",
    "QuantizedBatchedInputSingleLayer_basic",
    "QuantizedMLP_basic",
    "QuantizedNoLayer_basic",
    "QuantizedSingleLayer_basic",
    "ReduceMaxAlongDimUnsignedInt_basic",
    "ReduceMinAlongDimUnsignedInt_basic",
    "SplitDimDynamicModule_basic",
    "SplitDimStaticModule_basic",
    "SqrtIntModule_basic",
    "TensorToBoolZeroRank_basic",
    "TensorToBool_basic",
    "ThresholdBackward2dMixedModule_basic",
    "UnsafeViewCollapseDynamicWithAtenSizeIntModule_basic",
    "UpSampleNearest2dDynamicFactor_basic",
    "ViewCollapseDynamicWithAtenSizeIntModule_basic",
    "ViewSizeFromOtherTensor_basic",
    "ViewDtypeStaticModule_basic",
    "WeightNormInterfaceModule_basic",
    # Error: `aten.as_strided` op is not supported
    "ChunkListUnpackDynamic_Module_basic",
    "ChunkListUnpackUnevenDynamic_Module_basic",
    "ChunkListUnpackUneven_Module_basic",
    "ChunkListUnpack_Module_basic",
    "SplitTensorGetItem_Module_basic",
    "SplitTensorLastSmallerModule_basic",
    "SplitTensorListUnpackModule_basic",
    "SplitTensorNegativeDimModule_basic",
    "SplitWithSizesListUnpackModule_basic",
    "SplitWithSizes_Module_basic",
    "AdaptiveAvgPool1dGeneralDynamic_basic",
    "AdaptiveAvgPool1dStaticEvenMultiple_basic",
    "AdaptiveAvgPool1dStaticLargerOutput_basic",
    "AdaptiveAvgPool2dDynamicNoBatch_basic",
    "AdaptiveAvgPool2dDynamic_basic",
    "AdaptiveMaxPool1dDynamicNoBatch_basic",
    "AdaptiveMaxPool1dDynamic_basic",
    "AdaptiveMaxPool1dStatic_basic",
    "CrossEntropyLossModule_basic",
    "CrossEntropyLossNoReductionModule_basic",
    "IsInfiniteModule_basic",
    "InterpolateDynamicModule_sizes_nearest",
    "IouOfModule_basic",
    "MeshgridIndexingIJ_basic",
    "MeshgridIndexingXY_basic",
    "Meshgrid_basic",
    "ElementwiseSignbitModule_basic",
    "ElementwiseCopysignModule_basic",
    "BernoulliFloatModule_basic",
    "BernoulliTensorModule_basic",
    "UniformModule_basic",
    "UniformStaticShapeModule_basic",
}

FX_IMPORTER_CRASHING_SET = LINALG_CRASHING_SET | {
    "HBC_basic",
    # Runtime op verification: out-of-bounds access
    "_SoftmaxModule_basic",
    "UpSampleNearest2dDynamicFactor_basic",
    "AdaptiveAvgPool1dGeneralDynamicNoBatches_basic",
    "Aten_TrilinearModuleVaryingRanksUnorderedExpands_basic",
    "Aten_TrilinearModuleSumAllDims_basic",
    "Aten_TrilinearModuleSumdims_basic",
}

FX_IMPORTER_STABLEHLO_XFAIL_SET = {
    "AddFloatIntModule_basic",
    "AtenKthvalueDynamicDimsModule_basic",
    "AtenKthvalueFloat64DynamicDimsModule_basic",
    "AtenKthvalueFloat64Module_basic",
    "AtenKthvalueKeepDimModule_basic",
    "AtenKthvalueModule_basic",
    "AtenPolarDoubleModule_basic",
    "AtenPolarFloatModule_basic",
    "DiagonalWithStaticShapeModule_basic",
    "EinsumStaticDiagonalDimensionModule_basic",
    "ElementwiseRemainderScalarModule_Bool_NegativeDivisor_basic",
    "ElementwiseRemainderScalarModule_Float_NegativeDividend_basic",
    "ElementwiseRemainderScalarModule_Float_NegativeDivisor_basic",
    "ElementwiseRemainderScalarModule_Int_Float_NegativeDividend_basic",
    "ElementwiseRemainderScalarModule_Int_NegativeDividend_basic",
    "ElementwiseRemainderScalarModule_Int_NegativeDivisor_basic",
    "ElementwiseRemainderTensorModule_Float_NegativeDivisor_basic",
    "ElementwiseRemainderTensorModule_Int_Float_NegativeDividend_basic",
    "ElementwiseRemainderTensorModule_Int_Float_NegativeDivisor_basic",
    "ElementwiseRemainderTensorModule_Int_NegativeDividend_basic",
    "ElementwiseRemainderTensorModule_Int_NegativeDivisor_basic",
    "MaxPool1dCeilModeTrueModule_basic",
    "MaxPool1dStaticCeilModeTrueModule_basic",
    "MaxUnpool3dModulePad0_basic",
    "MaxUnpool3dModule_basic",
    "MultinomialModule2D_F32",
    "MultinomialModule2D_basic",
    "MultinomialModule_basic",
    "QuantizedReluInt32_basic",
    "QuantizedReluInt8_basic",
    "QuantizedReluUint8_basic",
    "ScatterAddStaticModule_basic",
    "SelectScattertModule_basic",
    "SelectScattertStaticModule_basic",
    "SignAndLogarithmOfDeterminantBatchedModule_F32",
    "SignAndLogarithmOfDeterminantDynamicModule_F32",
    "SignAndLogarithmOfDeterminantModule_F32",
    "SliceCopyEndGreaterThanDimSize_Module_basic",
    "SliceCopyNegative_Module_basic",
    "SliceCopyNonZeroDim_Module_basic",
    "SliceCopyStartGreaterThanDimSize_Module_basic",
    "SliceCopy_Module_basic",
    "SliceScatterModule_basic",
    "SliceScatterNegativeDimModule_basic",
    "SliceScatterNegativeEndModule_basic",
    "SliceScatterStaticModule_basic",
    "SliceScatterStepVariationModule_basic",
    "SliceScatterZeroDimModule_basic",
    "TimeOutModule_basic",
    "WeightNormInterfaceModule_basic",
    "AdaptiveAvgPool3dDynamicNoBatch_basic",
    "AdaptiveAvgPool3dDynamic_basic",
    "AdaptiveMaxPool1dDynamicNoBatch_basic",
    "AdaptiveMaxPool1dDynamic_basic",
    "AdaptiveMaxPool1dStatic_basic",
    "AdaptiveMaxPool2dDynamicNoBatch_basic",
    "AdaptiveMaxPool2dDynamicWithIndices_basic",
    "AdaptiveMaxPool2dDynamic_basic",
    "AdaptiveMaxPool2dStaticWithIndices_basic",
    "AdaptiveMaxPool2dStatic_basic",
    "AdaptiveMaxPool3dDynamicNoBatch_basic",
    "AdaptiveMaxPool3dDynamicWithIndices_basic",
    "AdaptiveMaxPool3dDynamic_basic",
    "AdaptiveMaxPool3dStaticWithIndices_basic",
    "AdaptiveMaxPool3dStatic_basic",
    "AllBoolFalseModule_basic",
    "AllBoolTrueModule_basic",
    "AnyBoolFalseModule_basic",
    "AnyBoolTrueModule_basic",
    "ArangeStartOutViewModule_basic",
    "AtenComplexImagModule_basic",
    "AtenComplexRealModule_basic",
    "AtenComplexViewModule_basic",
    "AtenDiagEmbedDefaultDiag_basic",
    "AtenDiagEmbedDimDiag_basic",
    "AtenDiagEmbedNegOffsetDiag_basic",
    "AtenDiagEmbedNonDefault4DDiag_basic",
    "AtenDiagEmbedOffsetDiag_basic",
    "AtenDiagEmbedRevDimDiag_basic",
    "AtenEmbeddingBagSumExample_basic",
    "AtenFftRfft2DLastDim_basic",
    "AtenFftRfft2DMiddleDim_basic",
    "AtenFloatScalarModule_basic",
    "AtenIntBoolOpConstFalseModule_basic",
    "AtenIntBoolOpConstTrueModule_basic",
    "AtenIntBoolOpModule_basic",
    "AtenItemFpOpModule_basic",
    "AtenMatmulQMixedSigni8Transpose_basic",
    "AtenMatmulQMixedSigni8_basic",
    "AtenMatmulQint8MV_basic",
    "AtenMatmulQint8VM_basic",
    "AtenMatmulQint8VV_basic",
    "AtenMatmulQint8_basic",
    "AtenMmQMixedSigni8_basic",
    "AtenMmQint8_basic",
    "AtenMmQuint8_basic",
    "AtenNonzero1DDynamicModule_basic",
    "AtenRealView128Module_basic",
    "AtenRealView64Module_basic",
    "AtenTopKModule_basic",
    "AtenTopKSmallestModule_basic",
    "Aten_EmbeddingBagExample_basic",
    "Aten_TrilinearModuleVaryingRanks_basic",
    "Aten_TrilinearModuleZerodDimBug_basic",
    "AvgPool2dDivisorOverrideModule_basic",
    "BernoulliTensorModule_basic",
    "BincountMinlengthModule_basic",
    "BincountModule_basic",
    "BincountStaticSizeModule_basic",
    "BoolFloatConstantModule_basic",
    "BoolFloatFalseModule_basic",
    "BoolFloatTrueModule_basic",
    "BoolIntConstantModule_basic",
    "BoolIntFalseModule_basic",
    "BoolIntTrueModule_basic",
    "BroadcastDynamicDimModule_basic",
    "CeilFloatModule_basic",
    "ConstantBoolParameterModule_basic",
    "ContainsIntList_False",
    "ContainsIntList_True",
    "Conv2dQInt8Module_basic",
    "Conv2dQInt8Module_depthwise",
    "Conv2dQInt8Module_grouped",
    "Conv2dQInt8Module_not_depthwise",
    "Conv2dQInt8PerChannelModule_basic",
    "Conv2dQInt8PerChannelModule_depthwise",
    "Conv2dQInt8PerChannelModule_grouped",
    "ConvTbcModule_basic",
    "ConvTranspose2DQInt8_basic",
    "ConvolutionBackwardModule2DPadded_basic",
    "ConvolutionBackwardModule2DStrided_basic",
    "ConvolutionBackwardModule2D_basic",
    "CumsumModule_basic",
    "CumprodModule_basic",
    "CumprodInputDtypeInt32Module_basic",
    "CumprodStaticModule_basic",
    "CumprodStaticNegativeDimModule_basic",
    "DeformConv2D_basic",
    "DeterminantBatchedModule_F32",
    "DeterminantDynamicModule_F32",
    "DeterminantModule_F32",
    "DiagonalModule_basic",
    "DiagonalModule_nonsquare",
    "DiagonalModule_transposed",
    "DiagonalModule_with_dims",
    "DiagonalModule_with_dims_and_offset",
    "DiagonalModule_with_negative_dims",
    "DiagonalModule_with_offset",
    "DivFloatModule_basic",
    "DivIntModule_basic",
    "ElementwiseDequantizePerChannelModule_basic",
    "ElementwiseDequantizePerTensorModule_basic",
    "ElementwiseErfIntModule_basic",
    "ElementwiseMulTensorComplexModule_basic",
    "ElementwiseMulTensorComplexDiffModule_basic",
    "ElementwiseQuantizePerTensorModule_basic",
    "ElementwiseQuantizePerTensorUIntModule_basic",
    "ElementwiseReciprocalIntModule_basic",
    "ElementwiseToDtypeI64ToUI8Module_basic",
    "EqIntModule_basic",
    "Fill_TensorFloat32WithFloat32_basic",
    "Fill_TensorFloat32WithFloat64_basic",
    "Fill_TensorFloat32WithInt64_basic",
    "FloatImplicitModule_basic",
    "GeFloatIntModule_basic",
    "GeFloatModule_basic",
    "GeIntModule_basic",
    "GtFloatIntModule_basic",
    "GtIntModule_basic",
    "HBC_basic",
    "HardtanhBackward_basic",
    "IndexPut1DFloatAccumulateModule_basic",
    "IndexPut1DFloatNonAccumulateModule_basic",
    "IndexPut1DIntAccumulateModule_basic",
    "IndexPut1DIntNonAccumulateModule_basic",
    "IndexPut2DFloatAccumulateModule_basic",
    "IndexPut2DFloatNonAccumulateModule_basic",
    "IndexPut2DIntAccumulateModule_basic",
    "IndexPut2DIntNonAccumulateModule_basic",
    "IndexPut3DFloatAccumulateModule_basic",
    "IndexPut3DFloatNonAccumulateModule_basic",
    "IndexPut3DIntAccumulateModule_basic",
    "IndexPut3DIntNonAccumulateModule_basic",
    "IndexPutHackedTwin1DFloatAccumulateModule_basic",
    "IndexPutHackedTwin1DFloatNonAccumulateModule_basic",
    "IndexPutHackedTwin1DIntAccumulateModule_basic",
    "IndexPutHackedTwin1DIntNonAccumulateModule_basic",
    "IndexPutHackedTwin2DFloatAccumulateModule_basic",
    "IndexPutHackedTwin2DFloatNonAccumulateModule_basic",
    "IndexPutHackedTwin2DIntAccumulateModule_basic",
    "IndexPutHackedTwin2DIntNonAccumulateModule_basic",
    "IndexPutHackedTwin3DFloatAccumulateModule_basic",
    "IndexPutHackedTwin3DFloatNonAccumulateModule_basic",
    "IndexPutHackedTwin3DIntAccumulateModule_basic",
    "IndexPutHackedTwin3DIntNonAccumulateModule_basic",
    "IndexPutImpl1DFloatAccumulateModule_basic",
    "IndexPutImpl1DFloatNonAccumulateModule_basic",
    "IndexPutImpl1DIntAccumulateModule_basic",
    "IndexPutImpl1DIntNonAccumulateModule_basic",
    "IndexPutImpl2DFloatAccumulateModule_basic",
    "IndexPutImpl2DFloatNonAccumulateModule_basic",
    "IndexPutImpl2DImplicitModule_basic",
    "IndexPutImpl2DIndexModule_basic",
    "IndexPutImpl2DNoneIndexStaticModule_basic",
    "IndexPutImpl3DFloatAccumulateModule_basic",
    "IndexPutImpl3DFloatNonAccumulateModule_basic",
    "IndexPutImplIndexWithNoneModule_basic",
    "IndexPutWithNoneAndBroadcastModule_basic",
    "IndexSelectRank0IdxModule_basic",
    "IndexTensorNegativeIndexModule_basic",
    "IntFloatModule_basic",
    "IntImplicitModule_basic",
    "LenStrModule_basic",
    "MaxPool2dCeilModeTrueModule_basic",
    "MaxPool2dStaticCeilModeTrueModule_basic",
    "MaxPool2dStaticCeilModeTrueReduceOutputModule_basic",
    "MaxPool2dWithIndicesBackwardDynamic3DModule_basic",
    "MaxPool2dWithIndicesBackwardDynamic4DModule_basic",
    "MaxPool2dWithIndicesBackwardStatic3DModule_basic",
    "MaxPool2dWithIndicesBackwardStatic4DModule_basic",
    "MaxPool3dCeilModeTrueModule_basic",
    "MaxPool3dStaticCeilModeTrueModule_basic",
    "MaxPool3dWithIndicesAllNegativeValuesModule_basic",
    "MaxPool3dWithIndicesAllOnesModule_basic",
    "MaxPool3dWithIndicesCeilModeTrueModule_basic",
    "MaxPool3dWithIndicesFullSizeKernelModule_basic",
    "MaxPool3dWithIndicesModule_basic",
    "MaxPool3dWithIndicesNonDefaultDilationModule_basic",
    "MaxPool3dWithIndicesNonDefaultPaddingModule_basic",
    "MaxPool3dWithIndicesNonDefaultParamsModule_basic",
    "MaxPool3dWithIndicesNonDefaultStrideModule_basic",
    "MaxPool3dWithIndicesStaticModule_basic",
    "MulFloatModule_basic",
    "NativeGroupNormBackwardModule_basic",
    "NeFloatIntModule_basic",
    "NeIntModule_basic",
    "NllLossModuleBackward1DMeanWeight_basic",
    "NllLossModuleBackward1DMean_basic",
    "NllLossModuleBackward1DSumWeight_basic",
    "NllLossModuleBackward1DSum_basic",
    "NllLossModuleBackward1DWeight_basic",
    "NllLossModuleBackward1D_basic",
    "NllLossModuleBackwardMeanWeight_basic",
    "NllLossModuleBackwardMean_basic",
    "NllLossModuleBackwardSumWeight_basic",
    "NllLossModuleBackwardSum_basic",
    "NllLossModuleBackwardWeight_basic",
    "NllLossModuleBackward_basic",
    "NllLossModuleBackward_ignore_index",
    "NormScalarComplexModule_basic",
    "NormScalarModule_basic",
    "NormalFunctionalModule_basic",
    "NumelModule_basic",
    "NumelZeroRankModule_basic",
    "PrimMaxIntModule_basic",
    "PrimMinIntDynamicModule_basic",
    "PrimMinIntModule_basic",
    "PrimsSqueezeEmptyDimensionsModule_basic",
    "PrimsSqueezeModule_basic",
    "PrimsViewOfModule_basic",
    "PrimsViewOfZeroRankModule_basic",
    "QuantizedBatchedInputSingleLayer_basic",
    "QuantizedMLP_basic",
    "QuantizedNoLayer_basic",
    "QuantizedSingleLayer_basic",
    "RandnDtypeDeviceModule_basic",
    "RandnGeneratorF64Module_basic",
    "RandnGeneratorModule_basic",
    "RandnLikeDtypeModule_basic",
    "RandnLikeModule_basic",
    "RandnModule_basic",
    "ReduceProdDimIntFloatModule_basic",
    "ReflectionPad1dModule2dInput_Right",
    "ReflectionPad1dModule2dInput_basic",
    "ReflectionPad1dModule3dInput_Left",
    "ReflectionPad1dModule3dInput_basic",
    "ReflectionPad2dModule_Bottom",
    "ReflectionPad2dModule_Left",
    "ReflectionPad2dModule_Right",
    "ReflectionPad2dModule_Top",
    "ReflectionPad2dModule_basic",
    "ReplicationPad2dModule_basic",
    "ReplicationPad2dModule_bottom0",
    "ReplicationPad2dModule_left0",
    "ReplicationPad2dModule_right0",
    "ReplicationPad2dModule_top0",
    "ScalarImplicitFloatModule_basic",
    # REMOVE WHEN ENABLE_GQA IS ADDED
    "ScatterReduceFloatMaxModule",
    "ScatterReduceFloatMaxModuleIncludeSelf",
    "ScatterReduceFloatMeanModule",
    "ScatterReduceFloatMeanModuleIncludeSelf",
    "ScatterReduceFloatMinModule",
    "ScatterReduceFloatMinModuleIncludeSelf",
    "ScatterReduceFloatProdModule",
    "ScatterReduceFloatProdModuleIncludeSelf",
    "ScatterReduceFloatSumModule",
    "ScatterReduceFloatSumModuleIncludeSelf",
    "ScatterReduceIntMaxModule",
    "ScatterReduceIntMaxModuleIncludeSelf",
    "ScatterReduceIntMeanModule",
    "ScatterReduceIntMeanModuleIncludeSelf",
    "ScatterReduceIntMinModule",
    "ScatterReduceIntMinModuleIncludeSelf",
    "ScatterReduceIntProdModule",
    "ScatterReduceIntProdModuleIncludeSelf",
    "ScatterReduceIntSumModule",
    "ScatterReduceIntSumModuleIncludeSelf",
    "ScatterSrcModule_basic",
    "ScatterSrcStaticModule_basic",
    "ScatterValueFloatModule_basic",
    "ScatterValueIntModule_basic",
    "SliceOutOfLowerBoundEndIndexModule_basic",
    "SortTensorDescending_basic",
    "SortTensorInteger_basic",
    "SortTensorNegativeDimension_basic",
    "SortTensorSpecificDimension_basic",
    "SortTensor_basic",
    "SplitDimDynamicModule_basic",
    "SplitDimStaticModule_basic",
    "SqrtIntModule_basic",
    "SubFloatModule_basic",
    "TModuleRank0_basic",
    "TensorToBoolZeroRank_basic",
    "TensorToBool_basic",
    "TensorToFloatZeroRank_basic",
    "TensorToFloat_basic",
    "TensorToInt_basic",
    "ThresholdBackward1dFloatModule_basic",
    "ThresholdBackward1dIntModule_basic",
    "ThresholdBackward1dMixedModule_basic",
    "ThresholdBackward2dFloatModule_basic",
    "ThresholdBackward2dIntModule_basic",
    "ThresholdBackward2dMixedModule_basic",
    "ThresholdBackward3dFloatModule_basic",
    "ThresholdBackward3dIntModule_basic",
    "ThresholdBackward3dMixedModule_basic",
    "TraceModule_basic",
    "TraceModule_empty",
    "TraceModule_nonsquare",
    "TraceSignedIntModule_basic",
    "TraceUnsignedIntModule_basic",
    "TraceUnsignedIntModule_empty",
    "UnsafeIndexPutHackedTwin1DFloatNonAccumulateModule_basic",
    "UnsafeViewCollapseDynamicWithAtenSizeIntModule_basic",
    "UpSampleNearest2dBackwardScalesNone_basic",
    "UpSampleNearest2dBackward_basic",
    "ViewCollapseDynamicWithAtenSizeIntModule_basic",
    "ViewSizeFromOtherTensor_basic",
    # Error: `aten.as_strided` op is not supported
    "ChunkListUnpackDynamic_Module_basic",
    "ChunkListUnpackUnevenDynamic_Module_basic",
    "ChunkListUnpackUneven_Module_basic",
    "ChunkListUnpack_Module_basic",
    "SplitTensorGetItem_Module_basic",
    "SplitTensorLastSmallerModule_basic",
    "SplitTensorListUnpackModule_basic",
    "SplitTensorNegativeDimModule_basic",
    "SplitWithSizesListUnpackModule_basic",
    "SplitWithSizes_Module_basic",
    "Unfold_Module_basic",
    "Unfold_Module_Rank_4",
    "Unfold_Module_Rank_Zero_basic",
    "Unfold_Module_Rank_Zero_Size_Zero_basic",
    "Unfold_Module_Dynamic_basic",
    "AdaptiveAvgPool1dGeneralDynamic_basic",
    "AdaptiveAvgPool1dGeneralDynamicNoBatches_basic",
    "AdaptiveAvgPool1dStaticEvenMultiple_basic",
    "AdaptiveAvgPool1dStaticLargerOutput_basic",
    "AdaptiveAvgPool2dDynamicNoBatch_basic",
    "AdaptiveAvgPool2dDynamic_basic",
    "AddIntModule_basic",
    "AtenIntTensorByteDtypeModule_basic",
    "AtenIntTensorCharDtypeModule_basic",
    "AtenItemIntOpModule_basic",
    "CrossEntropyLossModule_basic",
    "CrossEntropyLossNoReductionModule_basic",
    "InterpolateDynamicModule_sizes_nearest",
    "IouOfModule_basic",
    "IscloseStaticModuleTrue_basic",
    "IscloseStaticModule_basic",
    "MeshgridIndexingIJ_basic",
    "MeshgridIndexingXY_basic",
    "Meshgrid_basic",
    "MulIntModule_basic",
    "ReduceFrobeniusNormComplexModule_basic",
    "ScalarImplicitIntModule_basic",
    "ScaledDotProductAttentionBoolMaskModule_basic",
    "ScaledDotProductAttentionDifferentCausalModule_basic",
    "ScaledDotProductAttentionDifferentDynamicCausalModule_basic",
    "ScaledDotProductAttentionDifferentModule_basic",
    "ScaledDotProductAttentionMaskModule_basic",
    "ScaledDotProductAttentionSameCausalModule_basic",
    "ScaledDotProductAttentionSameDynamicModule_basic",
    "ScaledDotProductAttentionSameModule_basic",
    "SubIntModule_basic",
    "TensorToIntZeroRank_basic",
    "UpSampleNearest2dDynamicFactor_basic",
    "UpSampleNearest2dDynamicSize_basic",
    "UpSampleNearest2dStaticFactor_basic",
    "UpSampleNearest2dStaticSize_basic",
    "UpSampleNearest2d_basic",
    "BernoulliFloatModule_basic",
    "UniformModule_basic",
    "UniformStaticShapeModule_basic",
}

FX_IMPORTER_STABLEHLO_CRASHING_SET = {
    "BatchNorm1DModule_basic",
    "BatchNorm2DModule_basic",
    "BatchNorm3DModule_basic",
    "ResNet18Module_basic",
    "ResNet18StaticModule_basic",
    "MobilenetV3Module_basic",
    "Conv2dBiasNoPaddingModule_basic",
    # llvm-project/llvm/include/llvm/ADT/ArrayRef.h:257:
    #  const T &llvm::ArrayRef<long>::operator[](size_t) const [T = long]:
    #  Assertion `Index < Length && "Invalid index!"
    "IndexPutWithNoneAndBroadcastModule_basic",
    # Assertion `newMaterialization.getType() == outputType
    #  materialization callback produced value of incorrect type failed
    "ReduceMaxAlongDimUnsignedInt_basic",
    "ReduceMinAlongDimUnsignedInt_basic",
    "Aten_TrilinearModuleSumdims_basic",
    "Aten_TrilinearModuleSumAllDims_basic",
    "Aten_TrilinearModuleVaryingRanksUnorderedExpands_basic",
    "CrossEntropyLossModule_basic",
    "CrossEntropyLossNoReductionModule_basic",
}

STABLEHLO_PASS_SET = {
    "ReduceAminmaxSingleDim_basic",
    "ReduceAminmaxAllDims_basic",
    "ReduceAmaxEmptyDim_basic",
    "ReduceMinAlongDimNegative_basic",
    "ReduceMinAlongDim_basic",
    "ArgminModule_with_dim",
    "ReduceMinAlongDimSignedInt_basic",
    "ReduceAnyDimFloatModule_basic",
    "MeshgridIndexingIJ_basic",
    "MeshgridIndexingXY_basic",
    "Meshgrid_basic",
    "SplitWithSizes_Module_basic",
    "TensorSplitSections_GetItemModule_basic",
    "TensorSplitSections_ListUnpackModule_basic",
    "EmptyModule_uint8",
    "TypeConversionUint8ToF32Module_basic",
    "Atleast1dModule0dInput_basic",
    "Atleast1dModule1dInput_basic",
    "Atleast2dModule0dInput_basic",
    "Atleast2dModule1dInput_basic",
    "Atleast2dModule2dInput_basic",
    "AtenLinear1D_basic",
    "AtenLinear2D_basic",
    "AtenLinear3DBias_basic",
    "AtenLinearMatVec_basic",
    "AtenLinearVecMatBias_basic",
    "AtenLinearVecMat_basic",
    "ReduceAminSingleDim_basic",
    "AtenDotModule_basic",
    "AdaptiveAvgPool1dNonUnitOutputSizeStaticModule_basic",
    "AdaptiveAvgPool1dUnitOutputSizeStaticModule_basic",
    "AdaptiveAvgPool2dNonUnitOutputSizeStaticModule_basic",
    "AdaptiveAvgPool2dUnitOutputSizeStaticModule_basic",
    "AdaptiveAvgPool2dOutputSizeDivisibleByInputStaticModule_basic",
    "AdaptiveAvgPool2dFixedKernelStrideSizeStaticModule_basic",
    "AddIntModule_basic",
    "AddFloatIntModule_basic",
    "AliasModule_basic",
    "TrueFalseOrBoolOpModule_basic",
    "AllBoolFalseModule_basic",
    "AllBoolTrueModule_basic",
    "AnyBoolFalseModule_basic",
    "AnyBoolTrueModule_basic",
    "ArangeDtypeFloatModule_basic",
    "ArangeDtypeIntModule_basic",
    "ArangeFalsePinMemoryModule_basic",
    "ArangeFloatModule_basic",
    "ArangeIntModule_basic",
    "ArangeNegativeStartFloatModule_basic",
    "ArangeNegativeStartIntModule_basic",
    "ArangeStartFloatModule_basic",
    "ArangeStartIntModule_basic",
    "ArangeStartNegativeStepFloatModule_basic",
    "ArangeStartNegativeStepIntModule_basic",
    "ArangeStartOutDtypeModule_basic",
    "ArangeStartOutModule_basic",
    "ArangeStartStepFloatModule_basic",
    "ArangeStartStepIntModule_basic",
    "ArangeZeroElementOutputModule_basic",
    "ArgmaxModule_with_dim",
    "AtenComplex64Module_basic",
    "AtenFloatScalarModule_basic",
    "AtenHannWindowPeriodicFalseModule_basic",
    "AtenHannWindowPeriodicTrueModule_basic",
    "AtenIntBoolOpConstFalseModule_basic",
    "AtenIntBoolOpConstTrueModule_basic",
    "AtenIntBoolOpModule_basic",
    "AtenIntTensorByteDtypeModule_basic",
    "AtenIntTensorCharDtypeModule_basic",
    "AtenItemIntOpModule_basic",
    "AtenMmFloatTypes_basic",
    "AtenMmIntTypes_basic",
    "AtenIntMM_basic",
    "AtenRoundFloatHalfToEvenModule_basic",
    "AtenRoundFloatModule_basic",
    "AtenRoundIntModule_basic",
    "AtenSubFloatModule_basic",
    "AtenToDeviceModule_basic",
    "AtenTrilStaticModule_basic",
    "AtenTrilWithNegDiagonalStaticModule_basic",
    "AtenTrilWithPosDiagonalStaticModule_basic",
    "Aten_CastFloatModule_basic",
    "Aten_CastLongModule_basic",
    "AvgPool1dStaticModule_basic",
    "AvgPool2dStaticModule_basic",
    "AvgPool2dCountIncludePadFalseStaticModule_basic",
    "AvgPool3dStaticModule_basic",
    "BaddbmmBroadcast1DInputModule_basic",
    "BaddbmmBroadcast2DInputModule_basic",
    "BaddbmmStaticModule_basic",
    "BoolFloatConstantModule_basic",
    "BoolFloatFalseModule_basic",
    "BoolFloatTrueModule_basic",
    "BoolIntConstantModule_basic",
    "BoolIntFalseModule_basic",
    "BoolIntTrueModule_basic",
    "BoolTensorReturnFalseModule_basic",
    "BoolTensorReturnMixedModule_basic",
    "BoolTensorReturnTrueModule_basic",
    "BroadcastListConstructWithMinusOneModule_basic",
    "BroadcastToSameRankStaticModule_basic",
    "BroadcastZeroRankInputStaticModule_basic",
    "CeilFloatModule_basic",
    "ChunkListUnpackUneven_Module_basic",
    "ChunkListUnpack_Module_basic",
    "CloneModule_basic",
    "CollapseAllDimensionsModule_basic",
    "CollapseStaticModule_basic",
    "ConstantBoolParameterModule_basic",
    "ContainsIntList_False",
    "ContainsIntList_True",
    "ContiguousModule_basic",
    "Conv1dDepthwiseWithPaddingDilationStrideStaticModule_basic",
    "Conv2dWithPaddingDilationStrideStaticModule_basic",
    "Conv2dWithPaddingDilationStrideStaticModule_depthwise",
    "Conv2dWithPaddingDilationStrideStaticModule_depthwise_multiplier",
    "Conv2dWithPaddingDilationStrideStaticModule_grouped",
    "Conv2dWithPaddingDilationStrideStaticModule_grouped_multiplier",
    "Convolution2DStaticModule_basic",
    "ConvolutionBackwardModule2DStatic_basic",
    "ConvolutionModule2DTransposeStridedStatic_basic",
    "Conv_Transpose1dStaticModule_basic",
    "Conv_Transpose2dStaticModule_basic",
    "Conv_Transpose3dStaticModule_basic",
    "ConstantPad2dStaticModule_basic",
    "ConstantPadNdModule_basic",
    "ConstantPadNdPartialStaticModule_basic",
    "ConstantPadNdStaticModule_basic",
    "CosineSimilarityStaticBroadcastModule_basic",
    "CosineSimilarityStaticModule_basic",
    "CumsumInputDtypeInt32Module_basic",
    "CumsumStaticModule_basic",
    "CumsumStaticNegativeDimModule_basic",
    "CumprodInputDtypeInt32Module_basic",
    "CumprodStaticModule_basic",
    "CumprodStaticNegativeDimModule_basic",
    "DetachModule_basic",
    "DivFloatModule_basic",
    "DivIntModule_basic",
    "DropoutEvalFloatModule_basic",
    "DropoutEvalIntModule_basic",
    "ElementwiseAbsFloatModule_basic",
    "ElementwiseAbsIntModule_basic",
    "ElementwiseAcoshIntModule_basic",
    "ElementwiseAcoshModule_basic",
    "ElementwiseAsinhIntModule_basic",
    "ElementwiseAsinhModule_basic",
    "ElementwiseAtanhIntModule_basic",
    "ElementwiseAtanhModule_basic",
    "ElementwiseAtan2TensorFloatStaticModule_basic",
    "ElementwiseAtan2TensorIntStaticModule_basic",
    "ElementwiseAtan2FloatIntStaticModule_basic",
    "ElementwiseAddScalar_NumToTensorFloat_Module_basic",
    "ElementwiseAddScalar_TensorLiteralInt32_Module_basic",
    "ElementwiseAtenIsinfOpModule_basic",
    "ElementwiseAtenIsneginfOpModule_basic",
    "ElementwiseAtenIsposinfOpModule_basic",
    "ElementwiseAtenLogicalAndOpPromoteBroadcastStaticShapeModule_basic",
    "ElementwiseAtenLogicalNotOpModule_basic",
    "ElementwiseAtenLogicalNotOpPromoteModule_basic",
    "ElementwiseAtenLogicalOrOpPromoteBroadcastStaticShapeModule_basic",
    "ElementwiseAtenLogicalXorOpPromoteBroadcastStaticShapeModule_basic",
    "ElementwiseAtenWhereSelfModule_basic",
    "ElementwiseBinaryStaticShapeModule_basic",
    "ElementwiseBitwiseAndStaticShapeModule_basic",
    "ElementwiseBitwiseNotInt32Module_basic",
    "ElementwiseBitwiseNotInt64Module_basic",
    "ElementwiseBitwiseOrStaticShapeModule_basic",
    "ElementwiseBitwiseXorStaticShapeModule_basic",
    "ElementwiseBitwiseLeftShiftInt64Module_basic",
    "ElementwiseBitwiseLeftShiftInt8Module_basic",
    "ElementwiseBitwiseRightShiftInt64Module_basic",
    "ElementwiseBitwiseRightShiftInt8Module_basic",
    "ElementwiseCeilModule_basic",
    "ElementwiseCeluStaticModule_basic",
    "ElementwiseClampMaxModule_basic",
    "ElementwiseClampMinModule_basic",
    "ElementwiseClampMinTensorFloatModule_basic",
    "ElementwiseClampMinTensorIntModule_basic",
    "ElementwiseClampModule_basic",
    "ElementwiseClampTensorFloatModule_basic",
    "ElementwiseClampTensorIntModule_basic",
    "ElementwiseClampTensorInt8Module_basic",
    "ElementwiseCloneChannelsLastMemoryFormatModule_basic",
    "ElementwiseCloneContiguousModule_basic",
    "ElementwiseCloneModule_basic",
    "ElementwiseCosModule_basic",
    "ElementwiseDivTensorRoundingModeFloorStaticModule_basic",
    "ElementwiseDivTensorRoundingModeTruncStaticModule_basic",
    "ElementwiseDivTensorRoundingModeFloorIntStaticModule_basic",
    "ElementwiseDivTensorRoundingModeTruncIntStaticModule_basic",
    "ElementwiseDivScalarRoundingModeFloorStaticModule_basic",
    "ElementwiseDivScalarRoundingModeTruncStaticModule_basic",
    "ElementwiseDivScalarRoundingModeFloorIntStaticModule_basic",
    "ElementwiseDivScalarRoundingModeTruncIntStaticModule_basic",
    "ElementwiseErfModule_basic",
    "ElementwiseExpModule_basic",
    "ElementwiseExpm1IntModule_basic",
    "ElementwiseExpm1Module_basic",
    "ElementwiseFloorIntModule_basic",
    "ElementwiseFloorModule_basic",
    "ElementwiseFmodTensor_Float_basic",
    "ElementwiseFmodTensor_Int_Float_basic",
    "ElementwiseFmodTensor_Int_basic",
    "ElementwiseGeluApproximateTanhModule_basic",
    "ElementwiseGeluModule_basic",
    "ElementwiseLeakyReluStaticModule_basic",
    "ElementwiseLogModule_basic",
    "ElementwiseLog10Module_basic",
    "ElementwiseLog2Module_basic",
    "ElementwiseLog10IntModule_basic",
    "ElementwiseLog2IntModule_basic",
    "ElementwiseNanToNumWithNoneModule_Basic",
    "ElementwiseNanToNumModule_Basic",
    "ElementwiseNeFloatTensorStaticModule_basic",
    "ElementwiseNeIntTensorStaticModule_basic",
    "ElementwiseNegModule_basic",
    "ElementwiseOrTensorStaticShapeModule_basic",
    "ElementwiseAndScalarStaticShapeModule_basic",
    "ElementwisePowScalarModule_basic",
    "ElementwisePowTensorBroadcastStaticModule_basic",
    "ElementwisePowTensorStaticModule_basic",
    "ElementwisePreluStaticModule_basic",
    "ElementwiseReciprocalModule_basic",
    "ElementwiseReluModule_basic",
    "ElementwiseRreluWithNoiseEvalStaticModule_basic",
    "ElementwiseRreluWithNoiseTrainStaticModule_basic",
    "RreluWithNoiseBackwardEvalStaticModule_basic",
    "RreluWithNoiseBackwardTrainStaticModule_basic",
    "ElementwiseRemainderTensorModule_Float_basic",
    "ElementwiseRemainderTensorModule_Float_NegativeDividend_basic",
    "ElementwiseRemainderTensorModule_Int_Float_basic",
    "ElementwiseRemainderTensorModule_Int_basic",
    "ElementwiseRreluEvalStaticModule_basic",
    "ElementwiseRreluTrainStaticModule_basic",
    "ElementwiseRsqrtModule_basic",
    "ElementwiseSigmoidModule_basic",
    "ElementwiseSinModule_basic",
    "ElementwiseSpecialExpm1IntModule_basic",
    "ElementwiseSpecialExpm1Module_basic",
    "ElementwiseSqrtModule_basic",
    "ElementwiseTanIntModule_basic",
    "ElementwiseTanModule_basic",
    "ElementwiseTernaryStaticShapeModule_basic",
    "ElementwiseToDtypeF32ToI64Module_basic",
    "ElementwiseToDtypeI64ToI8Module_basic",
    "ElementwiseToDtypeIdentityModule_basic",
    "ElementwiseUnaryModule_basic",
    "EmptyLikeMemoryFormatModule_basic",
    "EmptyLikeModule_defaultDtype",
    "EmptyLikeModule_falsePinMemory",
    "EmptyLikeModule_float",
    "EmptyLikeModule_int",
    "EmptyModule_contiguous",
    "EmptyModule_defaultDtype",
    "EmptyModule_falsePinMemory",
    "EmptyModule_float",
    "EmptyModule_int",
    "EmptyStridedModule_basic",
    "EqIntModule_basic",
    "ExpandAsIntModule_basic",
    "FakeQuantizePerTensorAffineCachemaskModule_basic",
    "FakeQuantizePerTensorAffineModule_basic",
    "FakeQuantizePerTensorAffineRoundToEvenModule_basic",
    "Fill_TensorFloat64WithFloat32Static_basic",
    "Fill_TensorFloat64WithFloat32_basic",
    "Fill_TensorFloat64WithFloat64_basic",
    "Fill_TensorFloat64WithInt64Static_basic",
    "Fill_TensorFloat64WithInt64_basic",
    "FlattenRank0Module_basic",
    "FlipModuleStaticShape_basic",
    "FlipNegativeIndexModule_basic",
    "FullLikeModuleDefaultDtype_basic",
    "FullLikeModuleFalsePinMemory_basic",
    "FullLikeModuleFloat2D_basic",
    "FullLikeModuleFloat3DStatic_basic",
    "FullLikeModuleFloat3D_basic",
    "FullLikeModuleInt2DStatic_basic",
    "FullLikeModuleInt2D_basic",
    "FullLikeModuleInt3D_basic",
    "FullModuleDefaultDtype_basic",
    "FullModuleFalsePinMemory_basic",
    "FullModuleFloat2D_basic",
    "FullModuleFloat3D_basic",
    "FullModuleInt2D_basic",
    "FullModuleInt3D_basic",
    "GeFloatIntModule_basic",
    "GeFloatModule_basic",
    "GeIntModule_basic",
    "GeluBackwardModule_basic",
    "GluStaticModule_basic",
    "GridSamplerBasic1_basic",
    "GridSamplerBasic2_basic",
    "GridSamplerBasic3_basic",
    "GridSamplerBasic4_basic",
    "GtFloatIntModule_basic",
    "GtIntModule_basic",
    "HstackBasicComplexModule_basic",
    "HstackBasicFloatModule_basic",
    "HstackBasicIntFloatModule_basic",
    "HstackBasicIntModule_basic",
    "IndexTensorMultiIndexStaticModule_basic",
    "IndexTensorStaticModule_basic",
    "IntFloatModule_basic",
    "IsFloatingPointFloat_True",
    "IsFloatingPointInt_False",
    "LeakyReluBackwardStaticModule_basic",
    "LenStrModule_basic",
    "LiftFreshCopyModule_basic",
    "LinspaceDtypeModule_basic",
    "LinspaceEmptyModule_basic",
    "LinspaceModule_basic",
    "LinspaceOneSizeModule_basic",
    "LinspaceTwoSizeModule_basic",
    "MaskedFillScalarFloatValueStaticModule_basic",
    "MaskedFillScalarIntValueStaticModule_basic",
    "MaskedFillTensorIntValueStaticModule_basic",
    "MaskedScatterStaticBasic_basic",
    "Matmul4dStatic_basic",
    "Matmul_2d",
    "Matmul_dot",
    "Matmul_matvec",
    "Matmul_vecmat",
    "MatmulStaticBroadcast_basic",
    "MaxPool1dStaticModule_basic",
    "MaxPool1dEmptyStrideStaticModule_basic",
    "MaxPool2dStaticModule_basic",
    "MaxPool2dEmptyStrideStaticModule_basic",
    "MaxPool3dStaticModule_basic",
    "MaxPool3dEmptyStrideStaticModule_basic",
    "MeanDimAllReduceModule_basic",
    "MeanDimEmptyDimModule_basic",
    "MeanDimNoneDimModule_basic",
    "MeanDtypeModule_basic",
    "MeanDynamicSizesModule_basic",
    "MeanModule_basic",
    "Mlp2LayerModuleNoBias_basic",
    "MmDagModule_basic",
    "MmModule_basic",
    "MmModule_chained",
    "MmTanhModule_basic",
    "MoveDimIntModule_basic",
    "MoveDimIntNegativeIndexModule_basic",
    "MulFloatModule_basic",
    "MulFloatModule_basic",
    "MulIntModule_basic",
    "Mv_basic",
    "NarrowHorizontalTest2_basic",
    "NarrowHorizontalTest_basic",
    "NarrowTensorHorizontalModule_basic",
    "NarrowTensorVerticalModule_basic",
    "NarrowVerticalTest2_basic",
    "NarrowVerticalTest_basic",
    "NativeDropoutEvalFloatModule_basic",
    "NeFloatIntModule_basic",
    "NeIntModule_basic",
    "NewEmptyModuleDefaultDtype_basic",
    "NewEmptyModuleFalsePinMemory_basic",
    "NewEmptyModuleFloat2D_basic",
    "NewEmptyModuleFloat3D_basic",
    "NewEmptyModuleInt2D_basic",
    "NewEmptyModuleInt3D_basic",
    "NewEmptyModuleLayoutIntDtype_basic",
    "NewEmptyModuleNonDefaultFloatDtype_basic",
    "NewEmptyModuleNonDefaultIntDtype_basic",
    "NewEmptyStridedModuleDefaultDtype_basic",
    "NewFullModuleDefaultDtype_basic",
    "NewFullModuleFalsePinMemory_basic",
    "NewFullModuleFloat3DStatic_basic",
    "NewFullModuleFloat3D_basic",
    "NewFullModuleInt2D_basic",
    "NewFullModuleInt3D_basic",
    "NewOnesModuleDefaultDtype_basic",
    "NewOnesModuleFalsePinMemory_basic",
    "NewOnesModuleFloat2D_basic",
    "NewOnesModuleFloat3D_basic",
    "NewOnesModuleInt2D_basic",
    "NewOnesModuleInt3D_basic",
    "NewZerosModuleDefaultDtype_basic",
    "NewZerosModuleFalsePinMemory_basic",
    "NewZerosModuleFloat2D_basic",
    "NewZerosModuleFloat3D_basic",
    "NewZerosModuleInt2D_basic",
    "NewZerosModuleInt3D_basic",
    "NewZerosStaticModuleLayoutStrided_basic",
    "NumToTensorFloatModule_basic",
    "NumToTensorIntModule_basic",
    "NumelModule_basic",
    "NumelZeroRankModule_basic",
    "NumpyTRank0Module_basic",
    "NumpyTRank1Module_basic",
    "NumpyTRank2Module_basic",
    "NumpyTRankNDynamicModule_basic",
    "NumpyTRankNStaticModule_basic",
    "OnesLikeModule_defaultDtype",
    "OnesLikeModule_falsePinMemory",
    "OnesLikeModule_float",
    "OnesLikeModule_int",
    "OnesModuleCPUDevice_basic",
    "OnesModuleDefaultDtype_basic",
    "OnesModuleFalsePinMemory_basic",
    "OnesModuleFloat_basic",
    "OnesModuleInt_basic",
    "PadModule_basic",
    "PadWithNoneValModule_basic",
    "Permute0RankModule_basic",
    "PermuteModule_basic",
    "PermuteNegativeIndexModule_basic",
    "PixelShuffleModuleStaticRank3Int64_basic",
    "PixelShuffleModuleStaticRank4Float32_basic",
    "PowIntFloatModule_basic",
    "PrimListUnpackNumMismatchModule_basic",
    "PrimMaxIntModule_basic",
    "PrimMinIntDynamicModule_basic",
    "PrimMinIntModule_basic",
    "PrimsConvertElementTypeModule_basic",
    "PrimsIotaModule_basic",
    "PrimsSqueezeEmptyDimensionsModule_basic",
    "PrimsViewOfModule_basic",
    "PrimsViewOfZeroRankModule_basic",
    "RandIntDtypeModule_basic",
    "RandIntLowDtypeModule_basic",
    "RandIntLowModule_basic",
    "RandIntModule_basic",
    "RandIntPinMemoryModule_basic",
    "ReduceAllFloatModule_basic",
    "ReduceAllIntModule_basic",
    "ReduceAllBoolModule_basic",
    "ReduceAnyFloatModule_basic",
    "ReduceAnyIntModule_basic",
    "ReduceAnyBoolModule_basic",
    "ReduceAmaxMultiDim_basic",
    "ReduceAmaxOutOfOrderDim_basic",
    "ReduceAmaxSingleDim_basic",
    "ReduceFrobeniusNormModule_basic",
    "ReduceMaxAllDims_basic",
    "ReduceMaxAlongDimNegative_basic",
    "ReduceMaxAlongDimSignedInt_basic",
    "ReduceMaxAlongDim_basic",
    "ReduceMaxFloatModule_basic",
    "ReduceMaxSignedIntModule_basic",
    "ReduceMaxUnsignedIntModule_basic",
    "ReduceMinFloatModule_basic",
    "ReduceMinSignedIntModule_basic",
    "ReduceMinUnsignedIntModule_basic",
    "ReduceSumDimIntListDtypeFloatModule_basic",
    "ReduceSumDimIntListDtypeIntModule_basic",
    "ReduceSumDimIntListElementTypeBoolModule_basic",
    "ReduceSumDimIntListEmptyDimModule_basic",
    "ReduceSumDimIntListFloatModule_basic",
    "ReduceSumDimIntListIntModule_basic",
    "ReduceSumDtypeFloatModule_basic",
    "ReduceSumDtypeIntModule_basic",
    "ReduceSumElementTypeBoolModule_basic",
    "ReduceSumFloatModule_basic",
    "ReduceSumSignedIntModule_basic",
    "ReduceSumUnsignedIntModule_basic",
    "ReduceProdFloatModule_basic",
    "ReduceProdDtypeFloatModule_basic",
    "ReduceProdElementTypeBoolModule_basic",
    "ReduceProdUnsignedIntModule_basic",
    "ReduceProdSignedIntModule_basic",
    "ReduceProdDtypeIntModule_basic",
    "RepeatInterleaveSelfIntModule_basic",
    "RepeatInterleaveSelfIntNoDimModule_basic",
    "ReturnThreeTensorFloat32_basic",
    "ReturnTwoTensorF32I64_basic",
    "RollModule_basic",
    "Rot90BasicModule_basic",
    "Rot90MultipleRotationsModule_basic",
    "Rot90NegativeEvenRotationsModule_basic",
    "Rot90NegativeOddRotationsModule_basic",
    "RsubInt0d_NumToTensor_Module_basic",
    "ScalarConstantTupleModule_basic",
    "ScalarImplicitFloatModule_basic",
    "ScalarImplicitIntModule_basic",
    "ScalarTensorDefaultDtypeModule_basic",
    "ScalarTensorFloat32Module_basic",
    "ScalarTensorInt32Module_basic",
    "ScalarTensorInt64Module_basic",
    "SliceModule_basic",
    "SliceNegIdxModule_basic",
    "SliceOutOfLowerBoundStartIndexModule_basic",
    "SliceOutOfUpperBoundIndexModule_basic",
    "SliceOutOfUpperBoundIndexStaticModule_basic",
    "SliceSizeTwoStepModule_basic",
    "SliceStartEqEndModule_basic",
    "SliceStaticModule_basic",
    "SliceStaticComplexInputModule_basic",
    "SliceWholeTensorModule_basic",
    "SortIntListReverse_basic",
    "SortIntList_basic",
    "SplitDimStaticModule_basic",
    "SplitTensorGetItem_Module_basic",
    "SplitTensorLastSmallerModule_basic",
    "SplitTensorListUnpackModule_basic",
    "SplitTensorNegativeDimModule_basic",
    "SplitWithSizesListUnpackModule_basic",
    "SqrtIntConstantModule_basic",
    "SqrtIntModule_basic",
    "SqueezeDimModule_identity",
    "SqueezeDimModule_unitDim",
    "SqueezeModule_allUnitDim",
    "SubFloatModule_basic",
    "SubIntModule_basic",
    "TModuleRank0_basic",
    "TModuleRank1_basic",
    "TModuleRank2_basic",
    "TensorFloatModule_basic",
    "TensorIntModule_basic",
    "TensorLiteralModule_basic",
    "TensorOpaqueLiteralModule_basic",
    "TensorToBoolZeroRank_basic",
    "TensorToFloatZeroRank_basic",
    "TensorToIntZeroRank_basic",
    "TensorsConcatModule_basic",
    "TensorsConcatComplex128FloatModule_basic",
    "TensorsConcatComplex64FloatModule_basic",
    "TensorsConcatComplex128IntModule_basic",
    "TensorsConcatNegativeDimModule_basic",
    "TensorsConcatNegativeDimStaticModule_basic",
    "TensorsConcatPromoteDTypeModule_basic",
    "TensorsConcatStaticModule_basic",
    "TestF16Return_basic",
    "TestMultipleTensorAndPrimitiveTypesReturn_basic",
    "TestMultipleTensorReturn_basic",
    "ToCopyBoolDTypeStaticModule_basic",
    "ToDtypeBoolLayoutNoneStaticModule_basic",
    "ToDtypeLayoutCPUModule_basic",
    "ToDtypeLayoutNoneModule_basic",
    "ToDtypeLayoutStridedModule_basic",
    "TorchPrimLoopForLikeTensorArgModule_basic",
    "TransposeIntModule_basic",
    "TransposeIntNegDimsModule_basic",
    "TriuIndicesModule_basic",
    "TriuIndicesAllZerosModule_basic",
    "TriuIndicesNegativeOffsetModule_basic",
    "TrilIndicesAllZerosModule_basic",
    "TrilIndicesModule_basic",
    "TrilIndicesNegativeOffsetModule_basic",
    "TrilIndicesOfssetGreaterThanRowModule_basic",
    "TupleModule_basic",
    "TypeAsDifferentModule_basic",
    "TypeAsSameModule_basic",
    "TypeConversionF32ToF64Module_basic",
    "TypeConversionF64ToF32Module_basic",
    "TypeConversionI1ToF32Module_basic",
    "TypeConversionI1ToF64Module_basic",
    "TypeConversionI1ToI32Module_basic",
    "TypeConversionI1ToI64Module_basic",
    "TypeConversionI32ToI64Module_basic",
    "TypeConversionI64ToI32Module_basic",
    "UnsafeView1DFoldModule_basic",
    "View1DFoldModule_basic",
    "ZeroFloat32Module_basic",
    "ZeroInt32Module_basic",
    "ZeroInt64Module_basic",
    "ZerosLikeModule_defaultDtype",
    "ZerosLikeModule_falsePinMemory",
    "ZerosLikeModule_float",
    "ZerosLikeModule_int",
    "ZerosModuleDefaultDtype_basic",
    "ZerosModuleFalsePinMemory_basic",
    "ZerosModuleFloat2D_basic",
    "ZerosModuleFloat3D_basic",
    "ZerosModuleInt2D_basic",
    "ZerosModuleInt3D_basic",
    "AtenEmbeddingBagStaticModule_basic",
    "AtenEyeMModuleCPUDevice_basic",
    "AtenEyeMModuleDefaultDtype_basic",
    "AtenEyeMModuleFalsePinMemory_basic",
    "AtenEyeMModuleFloat2D_basic",
    "AtenEyeMModuleInt2D_basic",
    "AtenEyeModuleCPUDevice_basic",
    "AtenEyeModuleDefaultDtype_basic",
    "AtenEyeModuleFalsePinMemory_basic",
    "AtenEyeModuleFloat2D_basic",
    "AtenEyeModuleInt2D_basic",
    "AtenInstanceNormModule_basic",
    "AtenLinalgCrossBroadcast_basic",
    "AtenLinalgCrossCustomDim_basic",
    "AtenLinalgCrossFloat_basic",
    "AtenLinalgCrossInt_basic",
    "AtenLinalgCrossNegativeDim_basic",
    "BucketizeTensorStaticFloatModule_basic",
    "BucketizeTensorStaticModule_basic",
    "DropoutTrainStaticShapeModule_basic",
    "ElementwiseWhereScalarOtherStaticModule_basic",
    "ElementwiseWhereScalarSelfStaticModule_basic",
    "EmbeddingModule1DIndices_basic",
    "EmbeddingModuleF16_basic",
    "EmbeddingModuleI32Static_basic",
    "EmbeddingModuleI32_basic",
    "EmbeddingModuleI64_basic",
    "GatherStaticModule_basic",
    "IndexSelectDynamicIndexSizeModule_basic",
    "IndexSelectNegativeDimModule_basic",
    "IndexSelectSingleIdxModule_basic",
    "IndexSelectTwoIdxModule_basic",
    "IndexSelectWholeDimensionModule_basic",
    "IndexSelectWholeTensorModule_basic",
    "IndexTensorStaticContiguousWithNoneModule_basic",
    "IndexTensorStaticNonContiguousWithNoneModule_basic",
    "LayerNormLastDimModule_basic",
    "LayerNormModule_basic",
    "LayerNormNormalizeOverAllDimsModule_basic",
    "MaxPool2dWithIndicesStaticModule_basic",
    "MeanDimAllReduceKeepdimModule_basic",
    "NativeDropoutTrainStaticShapeModule_basic",
    "NativeLayerNormModule4D_basic",
    "NativeLayerNormModule_basic",
    "NormalizeModule_basic",
    "PrimsSqueezeModule_basic",
    "RandModule_basic",
    "ReduceSumDimIntListKeepDimNegativeDimStaticModule_basic",
    "SelectIntNegativeDimAndIndexStaticModule_basic",
    "SqueezeDimModule_static",
    "SqueezeModule_static",
    "TriuBroadcastModule_basic",
    "TriuModule_basic",
    "UnbindIntGetItem_Module_basic",
    "UnbindIntListUnpack_Module_basic",
    "UniformStaticShapeModule_basic",
    "ArangeStartOutViewModule_basic",
    "ConvolutionBackwardModule2DStrided_basic",
    "EinsumStaticContractRhsModule_basic",
    "EinsumStaticFourDimensionModule_basic",
    "EinsumStaticModule_basic",
    "EinsumStaticWithEllipsisSlicingAndBroadcastModule_basic",
    "EinsumStaticWithEllipsisSlicingModule_basic",
    "FlattenStaticModule_basic",
    "GroupNormModule_basic",
    "GroupNormNoWeightAndBiasModule_basic",
    "NativeGroupNormModule_basic",
    "RepeatModule_basic",
    "ReshapeAliasCollapseModule_basic",
    "ReshapeAliasExpandModule_basic",
    "ReshapeAsModule_basic",
    "ReshapeExpandModule_basic",
    "TileBigDimsSizeModule_basic",
    "TileSmallDimsSizeModule_basic",
    "UnflattenIntNegativeOneDimStaticModule_basic",
    "UnflattenIntNegativeOneSizeStaticModule_basic",
    "UnflattenIntStaticModule_basic",
    "UnflattenStaticModule_basic",
    "UniformNoCorrelationModule_basic",
    "UnsafeViewCollapseModule_basic",
    "UnsafeViewDynamicExpandModule_basic",
    "UnsafeViewExpandModule_basic",
    "ViewCollapseInferredDimModule_basic",
    "ViewCollapseModule_basic",
    "ViewCollapseOnesMiddleModule_basic",
    "ViewDynamicExpandCollapseModule_basic",
    "ViewDynamicExpandModule_basic",
    "ViewExpandCollapseModule_basic",
    "ViewExpandCollapseWithOnesModule_basic",
    "ViewExpandDynamicDimModule_basic",
    "ViewExpandInferredDimModule_basic",
    "ViewExpandModule_basic",
    "ViewExpandOnesBeforeAndAfterModule_basic",
    "ViewExpandOnesMiddleModule_basic",
    "ViewExpandOnesModule_basic",
    "ViewNegativeStaticModule_basic",
    "ViewNoChange1dModule_basic",
    "ViewNoChange2dModule_basic",
    "ViewNoChange3dModule_basic",
    "ViewNoChangeStaticModule_basic",
    "ViewOffsetBackwardTestStaticModule_basic",
    "ViewOffsetTestStaticModule_basic",
    "ViewTwoFiveThreeStaticModule_basic",
    "ViewTwoToThreeStaticModule_basic",
    "ElementwiseLog1pModule_basic",
    "ElementwiseSgnModule_basic",
    "ElementwiseSignIntModule_basic",
    "ElementwiseAcosModule_basic",
    "ElementwiseAsinModule_basic",
    "ElementwiseAtanTensorFloatModule_basic",
    "ElementwiseAcosIntModule_basic",
    "ElementwiseAsinIntModule_basic",
    "ElementwiseAtanTensorIntModule_basic",
    "ElementwiseCosIntModule_basic",
    "ElementwiseExpIntModule_basic",
    "ElementwiseLogIntModule_basic",
    "ElementwiseRsqrtIntModule_basic",
    "ElementwiseSigmoidIntModule_basic",
    "ElementwiseSinIntModule_basic",
    "ElementwiseSqrtIntModule_basic",
    "ElementwiseUnaryIntModule_basic",
    "ElementwiseCoshIntModule_basic",
    "ElementwiseCoshModule_basic",
    "ElementwiseSinhIntModule_basic",
    "ElementwiseSinhModule_basic",
    "ElementwiseTruncIntModule_basic",
    "ElementwiseTruncModule_basic",
    "ElementwiseLogSigmoidModule_basic",
    "ElementwiseHardshrinkStaticModule_basic",
    "ElementwiseSoftshrinkStaticModule_basic",
    "RenormModuleFloat16_basic",
    "RenormModuleFloat32NegativeDim_basic",
    "RenormModuleFloat32_basic",
}

STABLEHLO_CRASHING_SET = {
    "IndexPutWithNoneAndBroadcastModule_basic",
    "ReduceMaxAlongDimUnsignedInt_basic",
    "ReduceMinAlongDimUnsignedInt_basic",
    # LLVM ERROR: Failed to infer result type(s)
    "ElementwiseClampMinTensorFloatModule_basic",
    "ElementwiseClampMinTensorIntModule_basic",
    "ElementwiseClampTensorFloatModule_basic",
    "ElementwiseClampTensorIntModule_basic",
}

TOSA_CRASHING_SET = {
    "ArangeStartOutDtypeModule_basic",
    "ArangeStartOutModule_basic",
    "ScatterSrcStaticModule_basic",
    # Runtime op verification: Out of bounds access
    "ReduceAllDimEmpty_basic",
    # SmallVector unable to grow for ThresholdBackward1d
    "ThresholdBackward1dFloatModule_basic",
    "ThresholdBackward1dIntModule_basic",
    "ThresholdBackward1dMixedModule_basic",
}

FX_IMPORTER_TOSA_CRASHING_SET = {
    "Aten_TrilinearModuleSumAllDims_basic",
    "Aten_TrilinearModuleSumdims_basic",
    "Aten_TrilinearModuleVaryingRanksUnorderedExpands_basic",
    "CrossEntropyLossModule_basic",
    "CrossEntropyLossNoReductionModule_basic",
    "ScatterSrcModule_basic",
    "ScatterSrcStaticModule_basic",
    "HBC_basic",
}

# Write the TOSA set as a "passing" set as it is very early in development
# and very few tests work yet.
TOSA_PASS_SET = {
    "Unfold_Module_Rank_4",
    "Unfold_Module_Rank_Zero_basic",
    "Unfold_Module_basic",
    "ElementwiseErfIntModule_basic",
    "ElementwiseIntTensorLtFloatScalarModule_basic",
    "ElementwiseSigmoidIntModule_basic",
    "ElementwiseTanIntModule_basic",
    "ElementwiseTanModule_basic",
    "ElementwiseUnaryIntModule_basic",
    "PowIntFloatModule_basic",
    "Deg2radModule_basic",
    "ElementwiseIntTensorLtFloatTensorModule_basic",
    "L1LossMeanReductionModule_basic",
    "L1LossNoReductionModule_basic",
    "L1LossSumReductionModule_basic",
    "PixelShuffleModuleStaticRank3Int64_basic",
    "PixelShuffleModuleStaticRank4Float32_basic",
    "RandIntLowModule_basic",
    "RandIntModule_basic",
    "RandIntPinMemoryModule_basic",
    "RenormModuleFloat16_basic",
    "SplitDimStaticModule_basic",
    "Deg2radModule_basic",
    "ElementwiseExpIntModule_basic",
    "ElementwiseLog10IntModule_basic",
    "ElementwiseLog10Module_basic",
    "ElementwiseLog1pModule_basic",
    "ElementwiseLog2IntModule_basic",
    "ElementwiseLogIntModule_basic",
    "ElementwiseLogitModule_basic",
    "ElementwiseMishModule_basic",
    "L1LossMeanReductionModule_basic",
    "L1LossNoReductionModule_basic",
    "L1LossSumReductionModule_basic",
    "RandIntLowModule_basic",
    "RandIntModule_basic",
    "RandIntPinMemoryModule_basic",
    "SoftplusModule_basic",
    "ReflectionPad1dModule2dInput_Right",
    "ReflectionPad1dModule2dInput_basic",
    "ReflectionPad1dModule3dInput_Left",
    "ReflectionPad1dModule3dInput_basic",
    "ReflectionPad2dModule_Bottom",
    "ReflectionPad2dModule_Left",
    "ReflectionPad2dModule_Right",
    "ReflectionPad2dModule_Top",
    "ReflectionPad2dModule_basic",
    "ReplicationPad2dModule_basic",
    "ReplicationPad2dModule_bottom0",
    "ReplicationPad2dModule_left0",
    "ReplicationPad2dModule_right0",
    "ReplicationPad2dModule_top0",
    "AdaptiveAvgPool1dNonUnitOutputSizeStaticModule_basic",
    "AdaptiveAvgPool1dUnitOutputSizeStaticModule_basic",
    "AdaptiveAvgPool2dFixedKernelStrideSizeStaticModule_basic",
    "AdaptiveAvgPool2dNonUnitOutputSizeStaticModule_basic",
    "AdaptiveAvgPool2dOutputSizeDivisibleByInputStaticModule_basic",
    "AdaptiveAvgPool2dUnitOutputSizeStaticModule_basic",
    "ElementwiseAtenLogicalNotOpPromoteModule_basic",
    "ElementwiseCosIntModule_basic",
    "ElementwiseReciprocalIntModule_basic",
    "ElementwiseRsqrtIntModule_basic",
    "ElementwiseSinIntModule_basic",
    "FloatPowerTensorTensorStaticModule_basic",
    "AdaptiveMaxPool1dDimOneStatic_basic",
    "CollapseAllDimensionsModule_basic",
    "CollapseRank1DynamicModule_basic",
    "CollapseStaticModule_basic",
    "ElementwiseClampMinTensorFloatModule_basic",
    "ElementwiseClampMinTensorIntModule_basic",
    "ElementwiseClampTensorFloatModule_basic",
    "ElementwiseClampTensorIntModule_basic",
    "ElementwiseFracModule_basic",
    "ElementwiseLdexpModule_basic",
    "ElementwiseSignbitIntModule_basic",
    "Exp2StaticIntModule_basic",
    "MaxPool1dEmptyStrideStaticModule_basic",
    "MaxPool1dStaticCeilModeTrueModule_basic",
    "MaxPool1dStaticModule_basic",
    "RepeatInterleaveSelfIntModule_basic",
    "RsubIntModule_noalpha_basic",
    "Aten_TrilinearModuleSumAllDims_basic",
    "Aten_TrilinearModuleSumdims_basic",
    "Aten_TrilinearModuleVaryingRanksUnorderedExpands_basic",
    "Aten_TrilinearModuleVaryingRanks_basic",
    "Aten_TrilinearModule_basic",
    "ElementwiseAddBoolModule_basic",
    "Exp2StaticModule_basic",
    "CosineSimilarityStaticBroadcastModule_basic",
    "DropoutTrainStaticShapeModule_basic",
    "ElementwiseAtenLogicalAndOpModule_basic",
    "ElementwiseAtenLogicalAndOpPromoteBroadcastModule_basic",
    "ElementwiseAtenLogicalAndOpPromoteBroadcastStaticShapeModule_basic",
    "ElementwiseBitwiseAndScalarInt8Module_basic",
    "ElementwiseRreluTrainStaticModule_basic",
    "IndexSelectRank0IdxModule_basic",
    "MseLossSumReductionWithDifferentElemTypeModule_basic",
    "NativeDropoutTrainStaticShapeModule_basic",
    "RandIntDtypeModule_basic",
    "RandIntLowDtypeModule_basic",
    "RandModule_basic",
    "ReduceL3NormAllDimsModule_basic",
    "ReduceL3NormKeepDimModule_basic",
    "SliceCopy_Module_basic",
    "Threshold1dIntModule_basic",
    "Threshold2dIntModule_basic",
    "EmptyModule_contiguous",
    "EmptyModule_defaultDtype",
    "EmptyModule_falsePinMemory",
    "EmptyModule_float",
    "EmptyModule_int",
    "EmptyModule_uint8",
    "EmptyStridedModule_basic",
    "NewEmptyModuleDefaultDtype_basic",
    "NewEmptyModuleFalsePinMemory_basic",
    "NewEmptyModuleFloat2D_basic",
    "NewEmptyModuleFloat3D_basic",
    "NewEmptyModuleInt2D_basic",
    "NewEmptyModuleInt3D_basic",
    "NewEmptyModuleLayoutIntDtype_basic",
    "NewEmptyModuleNonDefaultFloatDtype_basic",
    "NewEmptyModuleNonDefaultIntDtype_basic",
    "NewEmptyStridedModuleDefaultDtype_basic",
    "SelectScattertStaticModule_basic",
    "SliceScatterStaticModule_basic",
    "TensorAlloc1dStaticModule_basic",
    "AtenRoundFloatHalfToEvenModule_basic",
    "AtenRoundFloatModule_basic",
    "FakeQuantizePerTensorAffineCachemaskModule_basic",
    "FakeQuantizePerTensorAffineDynamicShapeModule_basic",
    "FakeQuantizePerTensorAffineModule_basic",
    "FakeQuantizePerTensorAffineRoundToEvenModule_basic",
    "Fill_TensorFloat64WithFloat32Static_basic",
    "Fill_TensorFloat64WithInt64Static_basic",
    "FlipModuleStaticShape_basic",
    "FlipModule_basic",
    "FlipNegativeIndexModule_basic",
    "Rot90BasicModule_basic",
    "Rot90DynamicDimsModule_basic",
    "Rot90MultipleRotationsModule_basic",
    "Rot90NegativeEvenRotationsModule_basic",
    "Rot90NegativeOddRotationsModule_basic",
    "AtenLinalgCrossBroadcast_basic",
    "AtenLinalgCrossCustomDim_basic",
    "AtenLinalgCrossFloat_basic",
    "AtenLinalgCrossInt_basic",
    "AtenLinalgCrossNegativeDim_basic",
    "BinaryCrossEntropyWithLogitsStaticModule_basic",
    "IndexSelectNegativeDimModule_basic",
    "IndexSelectSingleIdxModule_basic",
    "IndexSelectTwoIdxModule_basic",
    "IndexSelectWholeDimensionModule_basic",
    "IndexSelectWholeTensorModule_basic",
    "DiagonalWithStaticShapeModule_basic",
    "EinsumStaticDiagonalDimensionModule_basic",
    "ElementwiseAtenFloorDivideBroadcastModule_basic",
    "ElementwiseAtenFloorDivideScalarModule_basic",
    "ElementwiseAtenFloorDivideScalarNegativeModule_basic",
    "ElementwiseAtenFloorDivideTensorNegativeModule_basic",
    "ElementwiseAtenFloorDivideTensorPositiveModule_basic",
    "ElementwiseDivScalarRoundingModeFloorIntStaticModule_basic",
    "ElementwiseDivScalarRoundingModeFloorModule_basic",
    "ElementwiseDivScalarRoundingModeFloorStaticModule_basic",
    "ElementwiseDivScalarRoundingModeTruncIntStaticModule_basic",
    "ElementwiseDivScalarRoundingModeTruncModule_basic",
    "ElementwiseDivScalarRoundingModeTruncStaticModule_basic",
    "ElementwiseDivTensorRoundingModeFloorIntStaticModule_basic",
    "ElementwiseDivTensorRoundingModeFloorModule_basic",
    "ElementwiseDivTensorRoundingModeFloorStaticModule_basic",
    "ElementwiseDivTensorRoundingModeTruncIntStaticModule_basic",
    "ElementwiseDivTensorRoundingModeTruncModule_basic",
    "ElementwiseDivTensorRoundingModeTruncStaticModule_basic",
    "ElementwiseGeFloatTensorModule_basic",
    "ElementwiseGeIntTensorModule_basic",
    "ElementwiseRemainderScalarModule_Bool_NegativeDivisor_basic",
    "ElementwiseRemainderScalarModule_Bool_basic",
    "ElementwiseRemainderScalarModule_Int_NegativeDividend_basic",
    "ElementwiseRemainderScalarModule_Int_NegativeDivisor_basic",
    "ElementwiseRemainderTensorModule_Float_NegativeDividend_basic",
    "ElementwiseRemainderTensorModule_Float_NegativeDivisor_basic",
    "ElementwiseRemainderTensorModule_Float_basic",
    "ElementwiseRemainderTensorModule_Int_Float_NegativeDividend_basic",
    "ElementwiseRemainderTensorModule_Int_Float_NegativeDivisor_basic",
    "ElementwiseRemainderTensorModule_Int_Float_basic",
    "ElementwiseRemainderTensorModule_Int_NegativeDividend_basic",
    "ElementwiseRemainderTensorModule_Int_NegativeDivisor_basic",
    "ElementwiseRemainderTensorModule_Int_basic",
    "TriuBroadcastModule_basic",
    "TriuModule_basic",
    "AtenHannWindowPeriodicFalseModule_basic",
    "AtenHannWindowPeriodicTrueModule_basic",
    "ElementwiseAndScalarModule_basic",
    "ElementwiseAndScalarStaticShapeModule_basic",
    "ElementwiseAtenLogicalNotOpModule_basic",
    "ElementwiseAtenLogicalXorOpModule_basic",
    "ElementwiseAtenLogicalXorOpPromoteBroadcastModule_basic",
    "ElementwiseAtenLogicalXorOpPromoteBroadcastStaticShapeModule_basic",
    "ElementwiseBitwiseAndScalarInt32Module_basic",
    "ElementwiseBitwiseAndScalarInt64Module_basic",
    "ElementwiseBitwiseLeftShiftInt32Module_basic",
    "ElementwiseBitwiseLeftShiftInt64Module_basic",
    "ElementwiseBitwiseLeftShiftInt8Module_basic",
    "ElementwiseBitwiseRightShiftInt32Module_basic",
    "ElementwiseBitwiseRightShiftInt64Module_basic",
    "ElementwiseBitwiseRightShiftInt8Module_basic",
    "ElementwiseCosModule_basic",
    "ElementwiseErfModule_basic",
    "ElementwiseLeFloatIntScalarModule_basic",
    "ElementwiseLeFloatScalarModule_basic",
    "ElementwiseLeFloatTensorNanModule_basic",
    "ElementwiseLeIntScalarModule_basic",
    "ElementwiseLeMixedIntScalarModule_basic",
    "ElementwisePowScalarModule_basic",
    "ElementwisePowTensorBroadcastModule_basic",
    "ElementwisePowTensorBroadcastStaticModule_basic",
    "ElementwisePowTensorModule_basic",
    "ElementwisePowTensorStaticModule_basic",
    "ElementwiseSinModule_basic",
    "ArgminIntModule_basic",
    "ArgminIntModule_multiple_mins",
    "ArgminModule_basic",
    "ArgminModule_keepDim",
    "ReduceAllDimBool_basic",
    "ReduceAllDimFloat_basic",
    "ReduceAllDimInt_basic",
    "ReduceAllFloatModule_basic",
    "ReduceAllIntModule_basic",
    "ReduceAnyFloatModule_basic",
    "ReduceAnyIntModule_basic",
    "ReduceMaxAllDims_basic",
    "ReduceMaxFloatModule_basic",
    "ReduceMaxSignedIntModule_basic",
    "ReduceMaxUnsignedIntModule_basic",
    "ReduceMinFloatModule_basic",
    "ReduceMinSignedIntModule_basic",
    "ReduceMinUnsignedIntModule_basic",
    "ReduceProdDtypeFloatModule_basic",
    "ReduceProdDtypeIntModule_basic",
    "ReduceProdElementTypeBoolModule_basic",
    "ReduceProdFloatModule_basic",
    "ReduceProdSignedIntModule_basic",
    "ReduceProdUnsignedIntModule_basic",
    "ReduceSumDimIntListDtypeFloatModule_basic",
    "ReduceSumDimIntListDtypeIntModule_basic",
    "ReduceSumDimIntListElementTypeBoolModule_basic",
    "ReduceSumDtypeFloatModule_basic",
    "ReduceSumDtypeIntModule_basic",
    "ReduceSumElementTypeBoolModule_basic",
    "AtenTrilStaticModule_basic",
    "AtenTrilWithNegDiagonalStaticModule_basic",
    "AtenTrilWithPosDiagonalStaticModule_basic",
    "ArgmaxKeepdimModule_basic",
    "MeshgridIndexingIJ_basic",
    "MeshgridIndexingXY_basic",
    "Meshgrid_basic",
    "AvgPool2dCountIncludePadFalseStaticModule_basic",
    "TensorSplitSections_GetItemModule_basic",
    "TensorSplitSections_ListUnpackModule_basic",
    "Atleast1dModule0dInput_basic",
    "Atleast1dModule1dInput_basic",
    "Atleast2dModule0dInput_basic",
    "Atleast2dModule1dInput_basic",
    "Atleast2dModule2dInput_basic",
    "AtenLinear2D_basic",
    "AtenLinear3DBias_basic",
    "ElementwiseAddScalar_NumToTensorFloat_Module_basic",
    "ElementwiseDivTensorFloatModule_basic",
    "ElementwiseMulTensorFloatModule_basic",
    "ElementwiseWhereScalarSelfStaticModule_basic",
    "GroupNormModule_basic",
    "GroupNormNoWeightAndBiasModule_basic",
    "NativeGroupNormModule_basic",
    "AtenDotModule_basic",
    "ElementwiseFloatTensorGtIntScalarModule_basic",
    "ElementwiseLogSigmoidModule_basic",
    "ElementwiseTernaryStaticShapeModule_basic",
    "ElementwiseTruncModule_basic",
    "ElementwiseTruncIntModule_basic",
    "ElementwiseSgnModule_basic",
    "ElementwiseSignIntModule_basic",
    "AddCDivModule_basic",
    "AddCDiv_Module_basic",
    "AddCMulModule_basic",
    "AddCMul_Module_basic",
    "Add_Module_basic",
    "AliasModule_basic",
    "ArangeDtypeFloatModule_basic",
    "ArangeIntModule_basic",
    "ArangeNegativeStartIntModule_basic",
    "ArangeStartIntModule_basic",
    "ArangeStartNegativeStepIntModule_basic",
    "ArangeStartOutModule_basic",
    "ArangeStartOutViewModule_basic",
    "ArangeStartStepIntModule_basic",
    "ArangeDtypeIntModule_basic",
    "ArangeFalsePinMemoryModule_basic",
    "ArangeFloatModule_basic",
    "ArangeNegativeStartFloatModule_basic",
    "ArangeStartFloatModule_basic",
    "ArangeStartNegativeStepFloatModule_basic",
    "ArangeStartOutDtypeModule_basic",
    "ArangeStartStepFloatModule_basic",
    "ArgmaxIntModule_basic",
    "ArgmaxIntModule_multiple_maxs",
    "ArgmaxModule_basic",
    "ArgmaxModule_keepDim",
    "AtenComplex64Module_basic",
    "AtenEyeMModuleCPUDevice_basic",
    "AtenEyeMModuleDefaultDtype_basic",
    "AtenEyeMModuleFalsePinMemory_basic",
    "AtenEyeMModuleFloat2D_basic",
    "AtenEyeModuleCPUDevice_basic",
    "AtenEyeModuleDefaultDtype_basic",
    "AtenEyeModuleFalsePinMemory_basic",
    "AtenEyeModuleFloat2D_basic",
    "AtenRoundIntModule_basic",
    "AtenInstanceNormModule_basic",
    "AtenToDeviceModule_basic",
    "Aten_CastFloatModule_basic",
    "TrueFalseOrBoolOpModule_basic",
    "BaddbmmBroadcast1DInputModule_basic",
    "BaddbmmBroadcast2DInputModule_basic",
    "BaddbmmDynamicModule_basic",
    "BaddbmmStaticModule_basic",
    "BaddbmmWithAlphaBetaModule_basic",
    "BaddbmmWithAlphaModule_basic",
    "BaddbmmWithBetaModule_basic",
    "BatchNorm1DModule_basic",
    "BatchNorm1DStaticShapeModule_basic",
    "BatchNorm1DWith2DInputModule_basic",
    "BatchNorm2DModule_basic",
    "BatchNorm3DModule_basic",
    "BmmFloatModule_basic",
    "BoolTensorHandleSignless_basic",
    "BoolTensorReturnFalseModule_basic",
    "BoolTensorReturnMixedModule_basic",
    "BoolTensorReturnTrueModule_basic",
    "BroadcastListConstructWithMinusOneModule_basic",
    "BroadcastToSameRankStaticModule_basic",
    "BroadcastZeroRankInputStaticModule_basic",
    "BucketizeTensorStaticFloatModule_basic",
    "BucketizeTensorStaticModule_basic",
    "CloneModule_basic",
    "ChunkListUnpackUneven_Module_basic",
    "ChunkListUnpack_Module_basic",
    "ConstantBoolParameterModule_basic",
    "ConstantPad2dStaticModule_basic",
    "ConstantPadNdModule_basic",
    "ConstantPadNdPartialStaticModule_basic",
    "ConstantPadNdStaticModule_basic",
    "ContiguousModule_basic",
    "Conv2dBiasNoPaddingModule_basic",
    "Conv2dNoPaddingModule_basic",
    "Conv2dWithPaddingDilationStrideModule_basic",
    "Conv2dWithPaddingDilationStrideStaticModule_basic",
    "Conv2dWithPaddingDilationStrideStaticModule_depthwise",
    "Conv2dWithPaddingDilationStrideStaticModule_depthwise_multiplier",
    "Conv2dWithPaddingModule_basic",
    "Conv2dWithValidPaddingModule_basic",
    "Conv2dWithSamePaddingModule_basic",
    "Convolution2DStaticModule_basic",
    "CosineSimilarityStaticModule_basic",
    "DetachModule_basic",
    "DropoutEvalFloatModule_basic",
    "DropoutEvalIntModule_basic",
    "DropoutModule_basic",
    "EinsumStaticContractRhsModule_basic",
    "EinsumStaticFourDimensionModule_basic",
    "EinsumStaticModule_basic",
    "EinsumStaticWithEllipsisSlicingModule_basic",
    "EinsumStaticWithEllipsisSlicingAndBroadcastModule_basic",
    "ElementwiseAbsFloatModule_basic",
    "ElementwiseAbsIntModule_basic",
    "ElementwiseAddModule_basic",
    "ElementwiseAddScalarFloatModule_basic",
    "ElementwiseAddScalarInt64Module_basic",
    "ElementwiseAddScalarInt8Module_basic",
    "ElementwiseAddScalarIntModule_basic",
    "ElementwiseAddScalar_TensorLiteralInt32_Module_basic",
    "ElementwiseAtenDivIntScalarModule_basic",
    "ElementwiseAtenIsinfOpModule_basic",
    "ElementwiseAtenIsneginfOpModule_basic",
    "ElementwiseAtenIsposinfOpModule_basic",
    "ElementwiseAtenLogicalOrOpBrodcastModule_basic",
    "ElementwiseAtenLogicalOrOpDiffArgs1Module_basic",
    "ElementwiseAtenLogicalOrOpDiffArgs2Module_basic",
    "ElementwiseAtenLogicalOrOpDiffArgs3Module_basic",
    "ElementwiseAtenLogicalOrOpModule_basic",
    "ElementwiseAtenLogicalOrOpNegativeModule_basic",
    "ElementwiseAtenLogicalOrOpPromoteBroadcastStaticShapeModule_basic",
    "ElementwiseAtenLogicalOrOpRandomFloatModule_basic",
    "ElementwiseAtenLogicalOrOpRandomModule_basic",
    "ElementwiseAtenWhereSelfModule_basic",
    "ElementwiseBinaryModule_basic",
    "ElementwiseBinaryStaticShapeModule_basic",
    "ElementwiseBitwiseAndModule_basic",
    "ElementwiseBitwiseAndStaticShapeModule_basic",
    "ElementwiseBitwiseNotInt32Module_basic",
    "ElementwiseBitwiseNotInt64Module_basic",
    "ElementwiseBitwiseOrModule_basic",
    "ElementwiseBitwiseOrStaticShapeModule_basic",
    "ElementwiseBitwiseXorModule_basic",
    "ElementwiseBitwiseXorStaticShapeModule_basic",
    "ElementwiseCeilModule_basic",
    "ElementwiseCeluModule_basic",
    "ElementwiseCeluStaticModule_basic",
    "ElementwiseClampMaxModule_basic",
    "ElementwiseClampMinModule_basic",
    "ElementwiseClampModule_basic",
    "ElementwiseClampTensorInt8Module_basic",
    "ElementwiseCloneChannelsLastMemoryFormatModule_basic",
    "ElementwiseCloneContiguousModule_basic",
    "ElementwiseCloneModule_basic",
    "ElementwiseDivScalarModule_basic",
    "ElementwiseDivTensorIntegerModule_basic",
    "ElementwiseDivTensorUnsignedIntegerModule_basic",
    "ElementwiseDivScalarIntegerModule_basic",
    "ElementwiseDivScalarUnsignedIntegerModule_basic",
    "ElementwiseEluModule_basic",
    "ElementwiseEluNonDefaultModule_basic",
    "ElementwiseEqBoolScalarModule_basic",
    "ElementwiseEqDiffWidthScalarModule_basic",
    "ElementwiseEqFloatScalarModule_basic",
    "ElementwiseEqFloatTensorModule_basic",
    "ElementwiseEqIntScalarModule_basic",
    "ElementwiseEqIntTensorModule_basic",
    "ElementwiseExpModule_basic",
    "ElementwiseFlattenBroadcastModule_basic",
    "ElementwiseFloorIntModule_basic",
    "ElementwiseFloorModule_basic",
    "ElementwiseFmaxModule_basic",
    "ElementwiseFminModule_basic",
    "ElementwiseFmodTensor_Float_basic",
    "ElementwiseFmodTensor_Int_Float_basic",
    "ElementwiseFmodTensor_Int_basic",
    "ElementwiseGeFloatIntScalarModule_basic",
    "ElementwiseGeFloatScalarModule_basic",
    "ElementwiseGeIntScalarModule_basic",
    "ElementwiseGeMixedIntScalarModule_basic",
    "ElementwiseGeluModule_basic",
    "ElementwiseGtFloatScalarModule_basic",
    "ElementwiseGtFloatTensorModule_basic",
    "ElementwiseGtIntScalarModule_basic",
    "ElementwiseGtIntTensorModule_basic",
    "ElementwiseGtMixed2ScalarModule_basic",
    "ElementwiseIsinfModule_basic",
    "ElementwiseAtenIsneginfOpModule_basic",
    "ElementwiseAtenIsposinfOpModule_basic",
    "ElementwiseIsnanModule_basic",
    "ElementwiseLeFloatTensorModule_basic",
    "ElementwiseLeIntTensorModule_basic",
    "ElementwiseLeakyReluModule_basic",
    "ElementwiseLeakyReluModule_basic",
    "ElementwiseLeakyReluStaticModule_basic",
    "ElementwiseLerpScalarIntModule_basic",
    "ElementwiseLerpScalarFloatModule_basic",
    "ElementwiseLog2Module_basic",
    "ElementwiseLogModule_basic",
    "ElementwiseLtDiffWidthScalarModule_basic",
    "ElementwiseLtFloatScalarModule_basic",
    "ElementwiseLtFloatTensorModule_basic",
    "ElementwiseLtIntScalarModule_basic",
    "ElementwiseLtIntTensorModule_basic",
    "ElementwiseMaxOtherIntModule_basic",
    "ElementwiseMaxOtherModule_basic",
    "ElementwiseMaximumIntModule_basic",
    "ElementwiseMaximumModule_basic",
    "ElementwiseMinOtherIntModule_basic",
    "ElementwiseMinOtherModule_basic",
    "ElementwiseMinimumIntModule_basic",
    "ElementwiseMinimumModule_basic",
    "ElementwiseMulScalarModule_basic",
    "ElementwiseMulScalarModule_float",
    "ElementwiseMulScalarModule_float",
    "ElementwiseMulScalarModule_int",
    "ElementwiseMulTensorIntModule_basic",
    "ElementwiseNeFloatScalarModule_basic",
    "ElementwiseNeFloatTensorModule_basic",
    "ElementwiseNeFloatTensorStaticModule_basic",
    "ElementwiseNeIntScalarModule_basic",
    "ElementwiseNeIntTensorModule_basic",
    "ElementwiseNeIntTensorStaticModule_basic",
    "ElementwiseNegModule_basic",
    "ElementwiseOrTensorModule_basic",
    "ElementwiseOrTensorStaticShapeModule_basic",
    "ElementwisePowModule_basic",
    "ElementwisePreluModule_basic",
    "ElementwisePreluStaticModule_basic",
    "ElementwiseRad2DegModule_basic",
    "ElementwiseRad2DegIntModule_basic",
    "ElementwiseReciprocalModule_basic",
    "ElementwiseRelu6Module_basic",
    "ElementwiseReluModule_basic",
    "ElementwiseRreluWithNoiseEvalStaticModule_basic",
    "ElementwiseRemainderScalarModule_Float_NegativeDividend_basic",
    "ElementwiseRemainderScalarModule_Float_NegativeDivisor_basic",
    "ElementwiseRemainderScalarModule_Int_Float_NegativeDividend_basic",
    "ElementwiseRemainderScalarModule_Int_Float_NegativeDivisor_basic",
    "ElementwiseRemainderScalarModule_Float_basic",
    "ElementwiseRemainderScalarModule_Int_Float_basic",
    "ElementwiseRemainderScalarModule_Int_basic",
    "ElementwiseRemainderScalarModule_Int_basic",
    "ElementwiseRreluEvalModule_basic",
    "ElementwiseRreluEvalStaticModule_basic",
    "ElementwiseRsqrtModule_basic",
    "ElementwiseSeluModule_basic",
    "ElementwiseSigmoidModule_basic",
    "ElementwiseSignModule_basic",
    "ElementwiseHardshrinkModule_basic",
    "ElementwiseHardshrinkStaticModule_basic",
    "ElementwiseSoftshrinkModule_basic",
    "ElementwiseSoftshrinkStaticModule_basic",
    "ElementwiseSqrtIntModule_basic",
    "ElementwiseSqrtModule_basic",
    "ElementwiseSubScalarFloatModule_basic",
    "ElementwiseSubScalarIntModule_basic",
    "ElementwiseSubTensorInt8Module_basic",
    "ElementwiseToDtypeIdentityModule_basic",
    "ElementwiseUnaryModule_basic",
    "ElementwiseUnsqueezeBroadcastModule_basic",
    "ElementwiseWhereScalarModule_basic",
    "ElementwiseNanToNumWithNoneModule_Basic",
    "ElementwiseNanToNumModule_Basic",
    "EmbeddingModule1DIndices_basic",
    "EmbeddingModuleI32Static_basic",
    "FlattenRank0Module_basic",
    "FlattenStaticModule_basic",
    "FlattenDynamicModuleCollapseAll_basic",
    "FullLikeModuleFloat3DStatic_basic",
    "FullLikeModuleInt2DStatic_basic",
    "FullModuleDefaultDtype_basic",
    "FullModuleFloat2D_basic",
    "FullModuleFloat3D_basic",
    "FullModuleInt3D_basic",
    "GatherStaticModule_basic",
    "GeluBackwardModule_basic",
    "GluStaticModule_basic",
    "HardTanhIntModule_basic",
    "HardTanhModule_basic",
    "HardsigmoidModule_basic",
    "HardsigmoidRandomModule_basic",
    "HardswishModule_basic",
    "HardswishRandomModule_basic",
    "HardtanhBackward_basic",
    "IndexTensorMultiIndexStaticModule_basic",
    "IndexTensorNegativeIndexModule_basic",
    "IndexTensorStaticModule_basic",
    "IscloseStaticModuleTrue_basic",
    "IscloseStaticModule_basic",
    "LayerNormNormalizeOverAllDimsModule_basic",
    "LeakyReluBackwardModule_basic",
    "LeakyReluBackwardStaticModule_basic",
    "LiftFreshCopyModule_basic",
    "LinalgVectorNormKeepDimModule_basic",
    "LinalgVectorNormModule_basic",
    "LinalgNormKeepDimModule_basic",
    "MaskedFillScalarDefaultModule_basic",
    "MaskedFillScalarIntValueModule_basic",
    "MaskedFillScalarIntValueStaticModule_basic",
    "MaskedFillTensorIntValueStaticModule_basic",
    "Matmul4dStatic_basic",
    "Matmul_3d",
    "Matmul_dot",
    "MatmulStaticBroadcast_basic",
    "MaxPool2dEmptyStrideStaticModule_basic",
    "MaxPool2dStaticCeilModeTrueModule_basic",
    "MaxPool2dStaticCeilModeTrueReduceOutputModule_basic",
    "MaxPool2dStaticModule_basic",
    "MeanModule_basic",
    "MmDagModule_basic",
    "MoveDimIntModule_basic",
    "MoveDimIntModule_basic",
    "MoveDimIntNegativeIndexModule_basic",
    "MseLossNoReductionModule_basic",
    "NativeLayerNormModule4D_basic",
    "NewFullModuleDefaultDtype_basic",
    "NewFullModuleFalsePinMemory_basic",
    "NewFullModuleFloat2D_basic",
    "NewFullModuleFloat3DStatic_basic",
    "NewFullModuleFloat3D_basic",
    "NewFullModuleInt2DStatic_basic",
    "NewOnesModuleDefaultDtype_basic",
    "NewOnesModuleFalsePinMemory_basic",
    "NewOnesModuleFloat2D_basic",
    "NewOnesModuleFloat3D_basic",
    "NewOnesModuleInt2D_basic",
    "NewOnesModuleInt3D_basic",
    "NewZerosModuleDefaultDtype_basic",
    "NewZerosModuleFalsePinMemory_basic",
    "NewZerosModuleFloat2D_basic",
    "NewZerosModuleFloat3D_basic",
    "NewZerosModuleInt2D_basic",
    "NewZerosModuleInt3D_basic",
    "NewZerosStaticModuleLayoutStrided_basic",
    "NormalizeModule_basic",
    "NormScalarOptDimKeepDimModule_basic",
    "NormScalarOptDimModule_basic",
    "NumToTensorFloatModule_basic",
    "NumToTensorIntModule_basic",
    "NumpyTRank1Module_basic",
    "NumpyTRank2Module_basic",
    "NumpyTRankNDynamicModule_basic",
    "NumpyTRankNStaticModule_basic",
    "OnesModuleCPUDevice_basic",
    "OnesModuleDefaultDtype_basic",
    "OnesModuleFalsePinMemory_basic",
    "OnesModuleFloat_basic",
    "OnesModuleInt_basic",
    "PadModule_basic",
    "PadWithNoneValModule_basic",
    "PermuteModule_basic",
    "PermuteNegativeIndexModule_basic",
    "PowFloatFloatModule_basic",
    "PowFloatIntModule_basic",
    "PrimListUnpackNumMismatchModule_basic",
    "PrimsIotaModule_basic",
    "PrimsSqueezeEmptyDimensionsModule_basic",
    "PrimsSqueezeModule_basic",
    "PrimsViewOfModule_basic",
    "PrimsViewOfZeroRankModule_basic",
    "ReduceAllBoolModule_basic",
    "ReduceAnyBoolModule_basic",
    "ReduceSumDimIntListFloatModule_basic",
    "ReduceSumDimIntListIntModule_basic",
    "ReduceSumDimIntListKeepDimFloatModule_basic",
    "ReduceSumDimIntListKeepDimIntModule_basic",
    "ReduceSumDimIntListKeepDimNegativeDimStaticModule_basic",
    "ReduceSumFloatModule_basic",
    "ReduceSumSignedIntModule_basic",
    "ReduceSumUnsignedIntModule_basic",
    "RreluWithNoiseBackwardEvalModule_basic",
    "RreluWithNoiseBackwardEvalStaticModule_basic",
    "RreluWithNoiseBackwardTrainModule_basic",
    "RreluWithNoiseBackwardTrainStaticModule_basic",
    "RepeatModule_basic",
    "RepeatInterleaveSelfIntNoDimModule_basic",
    "ReshapeAliasCollapseModule_basic",
    "ReshapeAliasExpandModule_basic",
    "ReshapeAsModule_basic",
    "ReshapeCollapseModule_basic",
    "ReshapeExpandModule_basic",
    "ReturnThreeTensorFloat32_basic",
    "ReturnTwoTensorF32I64_basic",
    "ResNet18StaticModule_basic",
    "RsubFloatModule_basic",
    "RsubFloatModule_noalpha_basic",
    "RsubInt0d_NumToTensor_Module_basic",
    "RsubIntModule_basic",
    "ScalarTensorDefaultDtypeModule_basic",
    "ScalarTensorFloat32Module_basic",
    "ScalarTensorInt32Module_basic",
    "ScalarTensorInt64Module_basic",
    "SelectIntNegativeDimAndIndexStaticModule_basic",
    "SiluModule_basic",
    "SliceStaticModule_basic",
    "SplitTensorGetItem_Module_basic",
    "SplitTensorLastSmallerModule_basic",
    "SplitTensorListUnpackModule_basic",
    "SplitTensorNegativeDimModule_basic",
    "SplitWithSizesListUnpackModule_basic",
    "SquareModule_basic",
    "SqueezeDimModule_identity",
    "SqueezeDimModule_static",
    "SqueezeDimModule_unitDim",
    "SqueezeModule_allUnitDim",
    "SqueezeModule_broadcast",
    "SqueezeModule_noUnitDim",
    "SqueezeModule_static",
    "TModuleRank0_basic",
    "TModuleRank1_basic",
    "TModuleRank2_basic",
    "TanhBackward_basic",
    "TensorFloatModule_basic",
    "TensorIntModule_basic",
    "TensorLiteralModule_basic",
    "TensorOpaqueLiteralModule_basic",
    "TestF16Return_basic",
    "TestMultipleTensorReturn_basic",
    "Threshold1dFloatModule_basic",
    "Threshold1dIntI32Module_basic",
    "Threshold2dFloatModule_basic",
    "Threshold3dFloatModule_basic",
    "TileBigDimsSizeModule_basic",
    "TileSmallDimsSizeModule_basic",
    "ToCopyBoolDTypeStaticModule_basic",
    "ToDtypeBoolLayoutNoneStaticModule_basic",
    "TransposeIntModule_basic",
    "TransposeIntNegDimsModule_basic",
    "TupleModule_basic",
    "TypeAsSameModule_basic",
    "TypePromotionAlphaWiderModule_basic",
    "TypePromotionDifferentCategoryModule_basic",
    "TypePromotionSameCategoryDifferentWidthModule_basic",
    "TypePromotionSameCategoryZeroRankWider_basic",
    "TypePromotionZeroRankHigherCategoryModule_basic",
    "UnbindIntGetItem_Module_basic",
    "UnbindIntListUnpack_Module_basic",
    "UnflattenIntNegativeOneDimStaticModule_basic",
    "UnflattenIntNegativeOneSizeStaticModule_basic",
    "UnflattenIntStaticModule_basic",
    "UnflattenStaticModule_basic",
    "UnsafeView1DFoldModule_basic",
    "UnsafeViewCollapseModule_basic",
    "UnsafeViewDynamicExpandModule_basic",
    "UnsafeViewExpandModule_basic",
    "View1DFoldModule_basic",
    "ViewCollapseModule_basic",
    "ViewCollapseInferredDimModule_basic",
    "ViewCollapseOnesMiddleModule_basic",
    "ViewDoubleMergeStaticModule_basic",
    "ViewDynamicExpandCollapseModule_basic",
    "ViewDynamicExpandModule_basic",
    "ViewExpandCollapseModule_basic",
    "ViewExpandCollapseWithOnesModule_basic",
    "ViewExpandDynamicDimModule_basic",
    "ViewExpandInferredDimModule_basic",
    "ViewExpandModule_basic",
    "ViewExpandOnesBeforeAndAfterModule_basic",
    "ViewExpandOnesMiddleModule_basic",
    "ViewExpandOnesMiddleOppModule_basic",
    "ViewExpandOnesModule_basic",
    "ViewFiveTestStaticModule_basic",
    "ViewNegativeStaticModule_basic",
    "ViewNoChange1dModule_basic",
    "ViewNoChange2dModule_basic",
    "ViewNoChange3dModule_basic",
    "ViewNoChangeStaticModule_basic",
    "ViewOffsetBackwardTestStaticModule_basic",
    "ViewOffsetTestStaticModule_basic",
    "ViewTwoFiveThreeStaticModule_basic",
    "ViewTwoToThreeStaticModule_basic",
    "ZerosModuleDefaultDtype_basic",
    "ZerosModuleFalsePinMemory_basic",
    "ZerosModuleFloat2D_basic",
    "ZerosModuleFloat3D_basic",
    "ZerosModuleInt2D_basic",
    "ZerosModuleInt3D_basic",
    "_LogSoftmaxModuleStable_basic",
    "LinspaceModule_basic",
    "LinspaceOneSizeModule_basic",
    "LinspaceTwoSizeModule_basic",
    "RenormModuleFloat32NegativeDim_basic",
    "RenormModuleFloat32_basic",
    "IndexTensorStaticContiguousWithNoneModule_basic",
    "IndexTensorStaticNonContiguousWithNoneModule_basic",
}

MAKE_FX_TOSA_CRASHING_SET = TOSA_CRASHING_SET | {
    # Runtime op verification: static result dims in reassoc group do not divide src dim evenly
    "FlattenDynamicModule_basic",
    "ReshapeDynamicModule_basic",
    "ViewFlattenAndExpandModule_basic",
    "ViewSizeDimLedAndFollowedByExpandedOnesModule_basic",
    "ViewSizeDimLedByExpandedOnesModule_basic",
}

MAKE_FX_TOSA_PASS_SET = (
    TOSA_PASS_SET
    | {
        ### Tests additionally passing in make_fx_tosa
        "AdaptiveAvgPool1dStaticEvenMultiple_basic",
        "IsInfiniteModule_basic",
        "AdaptiveAvgPool2dFixedKernelStrideSizeStaticModule_basic",
        "AdaptiveAvgPool2dUnitOutputSizeStaticModule_basic",
        "ResNet18StaticModule_basic",
        "AdaptiveAvgPool1dStaticLargerOutput_basic",
        "ScaledDotProductAttentionBoolMaskModule_basic",
        "ScaledDotProductAttentionDifferentDynamicCausalModule_basic",
        "ArgminIntModule_basic",
        "ArgminIntModule_multiple_mins",
        "ArgminModule_basic",
        "ArgminModule_keepDim",
        "ReduceAllDimBool_basic",
        "ReduceAllDimFloat_basic",
        "ReduceAllDimInt_basic",
        "ReduceAllFloatModule_basic",
        "ReduceAllIntModule_basic",
        "ReduceAnyFloatModule_basic",
        "ReduceAnyIntModule_basic",
        "ReduceMaxAllDims_basic",
        "ReduceMaxFloatModule_basic",
        "ReduceMaxSignedIntModule_basic",
        "ReduceMaxUnsignedIntModule_basic",
        "ReduceMinFloatModule_basic",
        "ReduceMinSignedIntModule_basic",
        "ReduceMinUnsignedIntModule_basic",
        "ReduceProdDtypeFloatModule_basic",
        "ReduceProdDtypeIntModule_basic",
        "ReduceProdElementTypeBoolModule_basic",
        "ReduceProdFloatModule_basic",
        "ReduceProdSignedIntModule_basic",
        "ReduceProdUnsignedIntModule_basic",
        "ReduceSumDimIntListDtypeFloatModule_basic",
        "ReduceSumDimIntListDtypeIntModule_basic",
        "ReduceSumDimIntListElementTypeBoolModule_basic",
        "ReduceSumDtypeFloatModule_basic",
        "ReduceSumDtypeIntModule_basic",
        "ReduceSumElementTypeBoolModule_basic",
        "ScaledDotProductAttentionDifferentModule_basic",
        "ScaledDotProductAttentionMaskModule_basic",
        "ScaledDotProductAttentionSameModule_basic",
        "AvgPool2dCountIncludePadFalseStaticModule_basic",
        "AtenLinear1D_basic",
        "AtenLinearMatVec_basic",
        "AtenLinearVecMatBias_basic",
        "Atleast1dModule0dInput_basic",
        "Atleast1dModule1dInput_basic",
        "Atleast2dModule0dInput_basic",
        "Atleast2dModule1dInput_basic",
        "Atleast2dModule2dInput_basic",
        "MaxPool1dEmptyStrideStaticModule_basic",
        "MaxPool1dStaticCeilModeTrueModule_basic",
        "MaxPool1dStaticModule_basic",
        "AdaptiveAvgPool1dUnitOutputSizeStaticModule_basic",
        "CosineSimilarityModule_basic",
        "NativeGroupNormBackwardModule_basic",
        "ReduceFrobeniusNormKeepDimModule_basic",
        "ReduceFrobeniusNormModule_basic",
        "SliceWholeTensorModule_basic",
        "TensorFloatModule_basic",
        "TensorIntModule_basic",
        "RepeatInterleaveSelfIntModule_basic",
        "TorchPrimLoopForLikeTensorArgModule_basic",
        "ViewSizeDimFollowedByCollapsedOnesModule_basic",
        "ViewSizeDimFollowedByExpandedOnesModule_basic",
        "ViewSizeDimLedAndFollowedByCollapsedOnesModule_basic",
        "ViewSizeDimLedByCollapsedOnesModule_basic",
        "ViewSizeFromOtherTensor_basic",
        "RenormModuleFloat32NegativeDim_basic",
        "RenormModuleFloat32_basic",
        "RreluWithNoiseBackwardEvalModule_basic",
        "RreluWithNoiseBackwardEvalStaticModule_basic",
        "RreluWithNoiseBackwardTrainModule_basic",
        "RreluWithNoiseBackwardTrainStaticModule_basic",
    }
) - {
    ### Test failing in make_fx_tosa but not in tosa
    "AdaptiveMaxPool1dDimOneStatic_basic",
    "FloatPowerTensorTensorStaticModule_basic",
    # Dynamic shape, has extra unsupported broadcast ops
    "Matmul_3d",
    # Unimplemented operator 'aten._index_put_impl_.hacked_twin'
    "IndexPutImpl1DFloatNonAccumulateModule_basic",
    "IndexPutImpl1DIntNonAccumulateModule_basic",
    # RuntimeError: The size of tensor a (7) must match the size of tensor b (3) at non-singleton dimension 1
    "Add_Module_basic",
    # failed to legalize operation 'torch.aten.to.dtype' that was explicitly marked illegal
    "AtenEyeModuleInt2D_basic",
    "AtenEyeMModuleInt2D_basic",
    "Conv2dBiasNoPaddingModule_basic",
    "Conv2dNoPaddingModule_basic",
    "Conv2dWithPaddingDilationStrideModule_basic",
    "Conv2dWithPaddingModule_basic",
    "Conv2dWithSamePaddingModule_basic",
    "Conv2dWithValidPaddingModule_basic",
    # failed to legalize operation 'torch.operator'
    "ElementwisePreluModule_basic",
    "ElementwisePreluStaticModule_basic",
    "ElementwiseLogSigmoidModule_basic",
    # failed to legalize operation 'torch.aten.rrelu_with_noise'
    "ElementwiseRreluEvalModule_basic",
    # incompatible return type failure for tosa.concat.
    "HstackBasicComplexModule_basic",
    "HstackBasicFloatModule_basic",
    "HstackBasicIntFloatModule_basic",
    "HstackBasicIntModule_basic",
    # Shape Related failures
    "PrimListUnpackNumMismatchModule_basic",
    "ReshapeExpandModule_basic",
    "UnsafeViewCollapseModule_basic",
    "UnsafeViewDynamicExpandModule_basic",
    "ViewCollapseModule_basic",
    "ViewDynamicExpandCollapseModule_basic",
    "ViewDynamicExpandModule_basic",
    "ViewExpandDynamicDimModule_basic",
    "ViewNoChange1dModule_basic",
    "ViewNoChange2dModule_basic",
    "ViewNoChange3dModule_basic",
}

if torch_version_for_comparison() < version.parse("2.5.0.dev"):
    MAKE_FX_TOSA_PASS_SET = MAKE_FX_TOSA_PASS_SET | {
        "ScaledDotProductAttentionDifferentModule_basic",
        "ScaledDotProductAttentionMaskModule_basic",
        "ScaledDotProductAttentionSameModule_basic",
    }

LTC_CRASHING_SET = {
    # TODO: update test to move all inputs to the lazy device. Otherwise test fails with:
    # Check failed: lazy_tensor Input tensor is not a lazy tensor: CPUBoolType.
    "HBC_basic",
}

LTC_XFAIL_SET = {
    "TorchPrimLoopForLikeTensorArgModule_basic" "CollapseAllDimensionsModule_basic",
    "CollapseRank1DynamicModule_basic",
    "CollapseStaticModule_basic",
    "CollapsePartialDynamicModule_basic",
    "CollapseFullDynamicModule_basic",
    "SplitDimStaticModule_basic",
    "SplitDimDynamicModule_basic",
    "PixelShuffleModuleStaticRank3Int64_basic",
    "PixelShuffleModuleStaticRank4Float32_basic",
    "PixelShuffleModuleFullDynamic_basic",
    "PixelShuffleModuleSpatiallyDynamic_basic",
    "PixelShuffleModuleSpatiallyStatic_basic",
    "ConvTbcModule_basic",
    "_Convolution2DAllFalseModule_basic",
    "_Convolution2DBenchmarkModule_basic",
    "_Convolution2DCudnnModule_basic",
    "_Convolution2DDeterministicModule_basic",
    "_Convolution2DTF32Module_basic",
    "_ConvolutionDeprecated2DAllFalseModule_basic",
    "_ConvolutionDeprecated2DBenchmarkModule_basic",
    "_ConvolutionDeprecated2DCudnnModule_basic",
    "_ConvolutionDeprecated2DDeterministicModule_basic",
    "MaxPool3dEmptyStrideStaticModule_basic",
    "AddIntModule_basic",
    "AddFloatIntModule_basic",
    "ArangeStartOutViewModule_basic",
    "AtenIntBoolOpModule_basic",
    "BernoulliTensorModule_basic",
    "BincountMinlengthModule_basic",
    "BincountModule_basic",
    "BincountStaticSizeModule_basic",
    "BoolFloatFalseModule_basic",
    "BoolFloatTrueModule_basic",
    "BoolIntFalseModule_basic",
    "BoolIntTrueModule_basic",
    "CeilFloatModule_basic",
    "DivFloatModule_basic",
    "EqIntModule_basic",
    "ExponentialModule_basic",
    "GeFloatIntModule_basic",
    "GeFloatModule_basic",
    "GeIntModule_basic",
    "GtFloatIntModule_basic",
    "GtIntModule_basic",
    "IndexPutImpl1DFloatAccumulateModule_basic",
    "IndexPutImpl1DFloatNonAccumulateModule_basic",
    "IndexPutImpl1DIntAccumulateModule_basic",
    "IndexPutImpl1DIntNonAccumulateModule_basic",
    "IndexPutImpl2DFloatAccumulateModule_basic",
    "IndexPutImpl2DFloatNonAccumulateModule_basic",
    "IndexPutImpl2DIndexModule_basic",
    "IndexPutImpl2DNoneIndexStaticModule_basic",
    "IndexPutImpl3DFloatAccumulateModule_basic",
    "IndexPutImpl3DFloatNonAccumulateModule_basic",
    "IndexPutImplIndexWithNoneModule_basic",
    "Matmul_dot",
    "MulIntModule_basic",
    "DivIntModule_basic",
    "NeFloatIntModule_basic",
    "NeIntModule_basic",
    "QuantizedMLP_basic",
    "QuantizedSingleLayer_basic",
    "QuantizedBatchedInputSingleLayer_basic",
    "ScalarImplicitFloatModule_basic",
    "ScalarImplicitIntModule_basic",
    "SliceEndSleStartModule_basic",
    "SliceOutOfUpperBoundIndexModule_basic",
    "SliceOutOfUpperBoundIndexStaticModule_basic",
    "SliceStartEqEndModule_basic",
    "SqrtIntModule_basic",
    "SubFloatModule_basic",
    "MulFloatModule_basic",
    "SubIntModule_basic",
    "TensorsStackPromoteDTypeModule_basic",
    "TensorToBoolZeroRank_basic",
    "TensorToBool_basic",
    "TensorToFloatZeroRank_basic",
    "TensorToFloat_basic",
    "TensorToIntZeroRank_basic",
    "TensorToInt_basic",
    "UniformModule_basic",
    "UnsafeViewCollapseDynamicWithAtenSizeIntModule_basic",
    "ViewCollapseDynamicWithAtenSizeIntModule_basic",
    "AtenEmbeddingBagSumExample_basic",
    "Aten_EmbeddingBagExample_basic",
    "ElementwiseLogitModule_basic",
    "ElementwiseRemainderScalarModule_Int_Float_basic",
    "ElementwiseRemainderScalarModule_Bool_basic",
    "ElementwiseLerpScalarIntModule_basic",
    "ElementwiseLerpScalarFloatModule_basic",
    "AtenIntTensorByteDtypeModule_basic",
    "AtenIntTensorCharDtypeModule_basic",
    "UpSampleNearest2dBackwardVec_basic",
    "UpSampleNearest2dBackwardOutputSizeNone_basic",
    "ConvolutionBackwardModule2D_basic",
    "ConvolutionBackwardModule2DPadded_basic",
    "VarMeanCorrectionModule_basic",
    "VarMeanCorrectionNoneModule_basic",
    "ElementwisePreluModule_basic",
    "VarMeanBiasedModule_basic",
    "VarMeanUnbiasedModule_basic",
    "RandnLikeModule_basic",
    "RandnLikeDtypeModule_basic",
    "NormalFunctionalModule_basic",
    "BernoulliFloatModule_basic",
    "BernoulliModule_basic",
    "BernoulliPModule_basic",
    "DropoutTrainModule_basic",
    "DropoutTrainStaticShapeModule_basic",
    "NativeDropoutTrainModule_basic",
    "NativeDropoutTrainStaticShapeModule_basic",
    "StdCorrectionKeepDimModule_basic",
    "StdCorrectionNoneModule_basic",
    "VarCorrectionKeepDimModule_basic",
    "VarCorrectionNoneModule_basic",
    "AtenFloatScalarModule_basic",
    "PrimsSqueezeModule_basic",
    "PrimsSqueezeEmptyDimensionsModule_basic",
    "PrimsViewOfModule_basic",
    "PrimsViewOfZeroRankModule_basic",
    "OneHotModule_basic",
    "VarMeanDimModule_basic",
    "VarMeanDimBiasedModule_basic",
    "AtenComplexImagModule_basic",
    "AtenComplexRealModule_basic",
    "AtenComplexViewModule_basic",
    "AtenRealView128Module_basic",
    "AtenRealView64Module_basic",
    "ScatterValueFloatModule_basic",
    "ScatterValueIntModule_basic",
    "UniformStaticShapeModule_basic",
    "AtenEmbeddingBagStaticModule_basic",
    "EmptyStridedModule_basic",
    "EmptyStridedSizeIntStrideModule_basic",
    "ElementwiseBitwiseAndScalarInt64Module_basic",
    "ElementwiseBitwiseAndScalarInt32Module_basic",
    "ElementwiseBitwiseAndScalarInt8Module_basic",
    "Conv1dGroupModule_basic",
    "Conv2dQInt8Module_basic",
    "Conv2dQInt8Module_depthwise",
    "Conv2dQInt8Module_grouped",
    "Conv2dQInt8Module_not_depthwise",
    "Conv2dQInt8PerChannelModule_basic",
    "Conv2dQInt8PerChannelModule_depthwise",
    "Conv2dQInt8PerChannelModule_grouped",
    "ConvTranspose2DQInt8_basic",
}

ONNX_XFAIL_SET = {
    # This test is expected to time out
    "TimeOutModule_basic",
    # Failure - cast error
    "PermuteNegativeIndexModule_basic",
    # Failure - incorrect numerics
    "ReduceAnyDimFloatModule_basic",
    "AvgPool2dDivisorOverrideModule_basic",
    "BroadcastDynamicDimModule_basic",
    "ElementwiseAtan2TensorIntModule_basic",
    "ElementwiseAtan2TensorIntStaticModule_basic",
    "ElementwiseAtenFloorDivideScalarNegativeModule_basic",
    "ElementwiseAtenFloorDivideTensorNegativeModule_basic",
    "ElementwiseLog10IntModule_basic",
    "ElementwiseLog2IntModule_basic",
    "ElementwiseFminModule_basic",
    "ElementwiseFmaxModule_basic",
    "Exp2StaticModule_basic",
    "FloatPowerTensorTensorStaticModule_basic",
    "MultinomialModule2D_basic",
    "MultinomialModule2D_F32",
    "PixelShuffleModuleStaticRank4Float32_basic",
    "SliceCopyEndGreaterThanDimSize_Module_basic",
    "SliceCopyNegative_Module_basic",
    "SliceCopyNonZeroDim_Module_basic",
    "SliceCopy_Module_basic",
    "SliceStaticComplexInputModule_basic",
    "StdCorrectionLargeInputModule_basic",
    "TupleModule_basic",
    "ThresholdStaticModule_basic",
    "VarCorrectionLargeInputModule_basic",
    # Failure - incorrect shape
    "ArangeStartOutDtypeModule_basic",
    "ArangeStartOutViewModule_basic",
    "MoveDimIntNegativeIndexModule_basic",
    "ReduceL3NormKeepDimModule_basic",
    "ViewSizeFromOtherTensor_basic",
    # incorrect shape generated by torch.onnx.export (needs an unsqueeze)
    "MultinomialModule_basic",
    # Failure - onnx_export
    "AdaptiveAvgPool1dGeneralDynamic_basic",
    "AdaptiveAvgPool1dNonUnitOutputSizeDynamicModule_basic",
    "AdaptiveAvgPool1dStaticLargerOutput_basic",
    "AdaptiveAvgPool2dDynamicNoBatch_basic",
    "AdaptiveAvgPool2dDynamic_basic",
    "AdaptiveAvgPool2dNonUnitOutputSizeDynamicModule_basic",
    "AdaptiveAvgPool2dOutputSizeDivisibleByInputDynamicModule_basic",
    "AdaptiveAvgPool2dFixedKernelStrideSizeStaticModule_basic",
    "AdaptiveAvgPool3dDynamicNoBatch_basic",
    "AdaptiveAvgPool3dDynamic_basic",
    "AdaptiveMaxPool1dDynamicNoBatch_basic",
    "AdaptiveMaxPool1dDynamic_basic",
    "AdaptiveMaxPool1dStatic_basic",
    "AdaptiveMaxPool2dDynamicNoBatch_basic",
    "AdaptiveMaxPool2dDynamicWithIndices_basic",
    "AdaptiveMaxPool2dDynamic_basic",
    "AdaptiveMaxPool2dStaticWithIndices_basic",
    "AdaptiveMaxPool2dStatic_basic",
    "AdaptiveMaxPool3dDynamicNoBatch_basic",
    "AdaptiveMaxPool3dDynamicWithIndices_basic",
    "AdaptiveMaxPool3dDynamic_basic",
    "AdaptiveMaxPool3dStaticWithIndices_basic",
    "AdaptiveMaxPool3dStatic_basic",
    "AddCDivModule_basic",
    "AddIntModule_basic",
    "AddFloatIntModule_basic",
    "Add_Module_basic",
    "AllBoolFalseModule_basic",
    "AllBoolTrueModule_basic",
    "AnyBoolFalseModule_basic",
    "AnyBoolTrueModule_basic",
    "AtenComplex64Module_basic",
    "AtenComplexImagModule_basic",
    "AtenComplexRealModule_basic",
    "AtenComplexViewModule_basic",
    "AtenDiagEmbedDefaultDiag_basic",
    "AtenDiagEmbedDimDiag_basic",
    "AtenDiagEmbedNegOffsetDiag_basic",
    "AtenDiagEmbedNonDefault4DDiag_basic",
    "AtenDiagEmbedOffsetDiag_basic",
    "AtenDiagEmbedRevDimDiag_basic",
    "AtenEmbeddingBagStaticModule_basic",
    "AtenEmbeddingBagSumExample_basic",
    "AtenFftRfft2DLastDim_basic",
    "AtenFftRfft2DMiddleDim_basic",
    "AtenFloatScalarModule_basic",
    "AtenIntBoolOpConstFalseModule_basic",
    "AtenIntBoolOpConstTrueModule_basic",
    "AtenIntBoolOpModule_basic",
    "AtenIntTensorByteDtypeModule_basic",
    "AtenIntTensorCharDtypeModule_basic",
    "AtenIntMM_basic",
    "AtenItemFpOpModule_basic",
    "AtenItemIntOpModule_basic",
    "AtenKthvalueModule_basic",
    "AtenKthvalueKeepDimModule_basic",
    "AtenKthvalueDynamicDimsModule_basic",
    "AtenKthvalueFloat64Module_basic",
    "AtenKthvalueFloat64DynamicDimsModule_basic",
    "AtenLinalgCrossDynamic_basic",
    "AtenMatmulQMixedSigni8Transpose_basic",
    "AtenMatmulQMixedSigni8_basic",
    "AtenMatmulQint8MV_basic",
    "AtenMatmulQint8VM_basic",
    "AtenMatmulQint8VV_basic",
    "AtenMatmulQint8_basic",
    "AtenMmQMixedSigni8_basic",
    "AtenMmQint8_basic",
    "AtenMmQuint8_basic",
    "AtenPolarFloatModule_basic",
    "AtenPolarDoubleModule_basic",
    "AtenRealView128Module_basic",
    "AtenRealView64Module_basic",
    "AtenSubFloatModule_basic",
    "AtenTopKModule_basic",
    "AtenTopKSmallestModule_basic",
    "Aten_EmbeddingBagExample_basic",
    "AvgPool2dWithoutPadModule_basic",
    "BatchMlpLayerModule_basic",
    "BincountMinlengthModule_basic",
    "BincountModule_basic",
    "BincountStaticSizeModule_basic",
    "BoolFloatConstantModule_basic",
    "BoolFloatFalseModule_basic",
    "BoolFloatTrueModule_basic",
    "BoolIntConstantModule_basic",
    "BoolIntFalseModule_basic",
    "BoolIntTrueModule_basic",
    "CeilFloatModule_basic",
    "ChunkListUnpackDynamic_Module_basic",
    "ChunkListUnpackUnevenDynamic_Module_basic",
    "CollapseAllDimensionsModule_basic",
    "CollapseFullDynamicModule_basic",
    "CollapsePartialDynamicModule_basic",
    "CollapseRank1DynamicModule_basic",
    "CollapseStaticModule_basic",
    "ColumnStackBasicIntModule_basic",
    "ColumnStack1dModule_basic",
    "ColumnStack0dModule_basic",
    "ConstantBoolParameterModule_basic",
    "ContainsIntList_False",
    "ContainsIntList_True",
    "Conv1dModule_basic",
    "Conv1dWithSamePaddingModule_basic",
    "Conv1dWithValidPaddingModule_basic",
    "Conv1dGroupModule_basic",
    "Conv2dBiasNoPaddingModule_basic",
    "Conv2dModule_basic",
    "Conv2dNoPaddingModule_basic",
    "Conv2dQInt8Module_basic",
    "Conv2dQInt8Module_depthwise",
    "Conv2dQInt8Module_grouped",
    "Conv2dQInt8Module_not_depthwise",
    "Conv2dQInt8PerChannelModule_basic",
    "Conv2dQInt8PerChannelModule_depthwise",
    "Conv2dQInt8PerChannelModule_grouped",
    "Conv2dWithPaddingDilationStrideModule_basic",
    "Conv2dWithPaddingModule_basic",
    "Conv2dWithSamePaddingModule_basic",
    "Conv2dWithValidPaddingModule_basic",
    "Conv3dModule_basic",
    "Conv3dWithSamePaddingModule_basic",
    "Conv3dWithValidPaddingModule_basic",
    "ConvTbcModule_basic",
    "ConvTranspose2DQInt8_basic",
    "Conv_Transpose2dModule_basic",
    "Convolution2DModule_basic",
    "Convolution2DStridedModule_basic",
    "ConvolutionBackwardModule2DPadded_basic",
    "ConvolutionBackwardModule2DStatic_basic",
    "ConvolutionBackwardModule2DStrided_basic",
    "ConvolutionBackwardModule2D_basic",
    "ConvolutionModule2DGroups_basic",
    "ConvolutionModule2DTransposeNonUnitOutputPadding_basic",
    "ConvolutionModule2DTransposeStrided_basic",
    "ConvolutionModule2DTranspose_basic",
    "Deg2radModule_basic",
    "DivFloatModule_basic",
    "DivIntModule_basic",
    "ElementwiseAcoshIntModule_basic",
    "ElementwiseAcoshModule_basic",
    "ElementwiseAsinhIntModule_basic",
    "ElementwiseAsinhModule_basic",
    "ElementwiseAtanhIntModule_basic",
    "ElementwiseAtanhModule_basic",
    "ElementwiseAtenIsneginfOpModule_basic",
    "ElementwiseAtenIsposinfOpModule_basic",
    "ElementwiseBitwiseNotInt32Module_basic",
    "ElementwiseBitwiseNotInt64Module_basic",
    "ElementwiseBitwiseOrModule_basic",
    "ElementwiseBitwiseOrStaticShapeModule_basic",
    "ElementwiseBitwiseRightShiftInt32Module_basic",
    "ElementwiseBitwiseRightShiftInt64Module_basic",
    "ElementwiseBitwiseRightShiftInt8Module_basic",
    "ElementwiseBitwiseXorModule_basic",
    "ElementwiseBitwiseXorStaticShapeModule_basic",
    "ElementwiseSinhIntModule_basic",
    "ElementwiseSinhModule_basic",
    "ElementwiseCoshIntModule_basic",
    "ElementwiseCoshModule_basic",
    "ElementwiseTruncIntModule_basic",
    "ElementwiseTruncModule_basic",
    "ElementwiseDequantizePerChannelModule_basic",
    "ElementwiseDequantizePerTensorModule_basic",
    "ElementwiseDivTensorRoundingModeFloorIntStaticModule_basic",
    "ElementwiseDivTensorRoundingModeTruncIntStaticModule_basic",
    "ElementwiseEluNonDefaultModule_basic",
    "ElementwiseExpm1IntModule_basic",
    "ElementwiseExpm1Module_basic",
    "ElementwiseSpecialExpm1IntModule_basic",
    "ElementwiseSpecialExpm1Module_basic",
    "ElementwiseFmodTensor_Int_basic",
    "ElementwiseCreateComplexModule_basic",
    "ElementwiseMulTensorComplexModule_basic",
    "ElementwiseMulTensorComplexDiffModule_basic",
    "ElementwiseOrTensorModule_basic",
    "ElementwiseOrTensorStaticShapeModule_basic",
    "ElementwiseQuantizePerTensorModule_basic",
    "ElementwiseQuantizePerTensorUIntModule_basic",
    "ElementwiseRad2DegModule_basic",
    "ElementwiseRad2DegIntModule_basic",
    "ElementwiseRemainderTensorModule_Int_basic",
    "ElementwiseRemainderTensorModule_Int_NegativeDividend_basic",
    "ElementwiseRemainderTensorModule_Int_NegativeDivisor_basic",
    "ElementwiseRreluWithNoiseEvalModule_basic",
    "ElementwiseRreluWithNoiseEvalStaticModule_basic",
    "ElementwiseRreluWithNoiseTrainModule_basic",
    "ElementwiseRreluWithNoiseTrainStaticModule_basic",
    "ElementwiseSgnModule_basic",
    "EmptyStridedModule_basic",
    "EmptyStridedSizeIntStrideModule_basic",
    "EqIntModule_basic",
    "ExponentialModule_basic",
    "FakeQuantizePerTensorAffineCachemaskModule_basic",
    "FloatImplicitModule_basic",
    "GeFloatIntModule_basic",
    "GeFloatModule_basic",
    "GeIntModule_basic",
    "GeluBackwardModule_basic",
    "GtFloatIntModule_basic",
    "GtIntModule_basic",
    "HardtanhBackward_basic",
    "HstackBasicComplexModule_basic",
    "HstackBasicFloatModule_basic",
    "HstackBasicIntFloatModule_basic",
    "HstackBasicIntModule_basic",
    "IndexPutImpl1DFloatAccumulateModule_basic",
    "IndexPutImpl1DFloatNonAccumulateModule_basic",
    "IndexPutImpl1DIntAccumulateModule_basic",
    "IndexPutImpl1DIntNonAccumulateModule_basic",
    "IndexPutImpl2DFloatAccumulateModule_basic",
    "IndexPutImpl2DFloatNonAccumulateModule_basic",
    "IndexPutImpl2DImplicitModule_basic",
    "IndexPutImpl2DIndexModule_basic",
    "IndexPutImpl2DNoneIndexStaticModule_basic",
    "IndexPutImpl3DFloatAccumulateModule_basic",
    "IndexPutImpl3DFloatNonAccumulateModule_basic",
    "IndexPutImplIndexWithNoneModule_basic",
    "IndexPutWithNoneAndBroadcastModule_basic",
    "IntFloatModule_basic",
    "IntImplicitModule_basic",
    "IouOfModule_basic",
    "IsFloatingPointFloat_True",
    "IsFloatingPointInt_False",
    "IscloseStaticModuleTrue_basic",
    "IscloseStaticModule_basic",
    "L1LossNoReductionModule_basic",
    "L1LossMeanReductionModule_basic",
    "L1LossSumReductionModule_basic",
    "LeakyReluBackwardModule_basic",
    "LeakyReluBackwardStaticModule_basic",
    "LenStrModule_basic",
    "LiftFreshCopyModule_basic",
    "LinalgNormKeepDimComplexModule_basic",
    "LinalgVectorNormComplexModule_basic",
    "LogSoftmaxBackwardModule_basic",
    "MaxPool1dCeilModeTrueModule_basic",
    "MaxPool1dModule_basic",
    "MaxPool2dCeilModeTrueModule_basic",
    "MaxPool2dModule_basic",
    "MaxPool2dWithIndicesAllOnesModule_basic",
    "MaxPool2dWithIndicesBackwardDynamic3DModule_basic",
    "MaxPool2dWithIndicesBackwardDynamic4DModule_basic",
    "MaxPool2dWithIndicesBackwardStatic3DModule_basic",
    "MaxPool2dWithIndicesBackwardStatic4DModule_basic",
    "MaxPool2dWithIndicesCeilModeTrueModule_basic",
    "MaxPool2dWithIndicesFullSizeKernelModule_basic",
    "MaxPool2dWithIndicesModule_basic",
    "MaxPool2dWithIndicesNonDefaultDilationModule_basic",
    "MaxPool2dWithIndicesNonDefaultParamsModule_basic",
    "MaxPool2dWithIndicesNonDefaultStrideModule_basic",
    "MaxPool3dCeilModeTrueModule_basic",
    "MaxPool3dLargeDatadModule_basic",
    "MaxPool3dModuleRandomSimple_basic",
    "MaxPool3dModule_basic",
    "MaxPool3dWithIndicesAllOnesModule_basic",
    "MaxPool3dWithIndicesCeilModeTrueModule_basic",
    "MaxPool3dWithIndicesFullSizeKernelModule_basic",
    "MaxPool3dWithIndicesModule_basic",
    "MaxPool3dWithIndicesNonDefaultDilationModule_basic",
    "MaxPool3dWithIndicesNonDefaultParamsModule_basic",
    "MaxPool3dWithIndicesNonDefaultStrideModule_basic",
    "MaxUnpool3dModule_basic",
    "MaxUnpool3dModulePad0_basic",
    "MeanDimEmptyDimModule_basic",
    "Mlp1LayerModule_basic",
    "Mlp2LayerModuleNoBias_basic",
    "Mlp2LayerModule_basic",
    "MulFloatModule_basic",
    "MulIntModule_basic",
    "NarrowHorizontalTest2_basic",
    "NarrowHorizontalTest_basic",
    "NarrowTensorHorizontalModule_basic",
    "NarrowTensorVerticalModule_basic",
    "NarrowVerticalTest2_basic",
    "NarrowVerticalTest_basic",
    "NativeBatchNorm1DModule_basic",
    "NativeBatchNorm2DModule_basic",
    "NativeBatchNorm3DModule_basic",
    "NativeBatchNormNoneWeightModule_basic",
    "NativeDropoutEvalFloatModule_basic",
    "NativeGroupNormBackwardModule_basic",
    "NativeGroupNormModule_basic",
    "NativeLayerNormDynamicModule_basic",
    "NeFloatIntModule_basic",
    "NeIntModule_basic",
    "NewEmptyStridedModuleDefaultDtype_basic",
    "NllLossModuleBackward1DMeanWeight_basic",
    "NllLossModuleBackward1DMean_basic",
    "NllLossModuleBackward1DSumWeight_basic",
    "NllLossModuleBackward1DSum_basic",
    "NllLossModuleBackward1DWeight_basic",
    "NllLossModuleBackward1D_basic",
    "NllLossModuleBackwardMeanWeight_basic",
    "NllLossModuleBackwardMean_basic",
    "NllLossModuleBackwardSumWeight_basic",
    "NllLossModuleBackwardSum_basic",
    "NllLossModuleBackwardWeight_basic",
    "NllLossModuleBackward_basic",
    "NllLossModuleBackward_ignore_index",
    "NllLossModule_1D_basic",
    "NllLossModule_basic",
    "NllLossStaticModule_basic",
    "NllLossStaticModule_weight_basic",
    "NllLossModule_ignore_index_out_of_bounds_basic",
    "NllLossModule_mean_basic",
    "NllLossStaticModule_mean_basic",
    "NllLossModule_sum_basic",
    "NllLossStaticModule_sum_basic",
    "NormScalarComplexModule_basic",
    "NormScalarModule_basic",
    "NormScalarOptDimKeepDimComplexModule_basic",
    "NormScalarOptDimKeepDimModule_basic",
    "NormScalarOptDimModule_basic",
    "NormalFunctionalModule_basic",
    "NumToTensorFloatModule_basic",
    "NumToTensorIntModule_basic",
    "NumelModule_basic",
    "NumelZeroRankModule_basic",
    "PixelShuffleModuleFullDynamic_basic",
    "PixelShuffleModuleSpatiallyDynamic_basic",
    "PixelShuffleModuleSpatiallyStatic_basic",
    "PixelShuffleModuleStaticRank3Int64_basic",
    "PowIntIntModule_basic",
    "PrimMaxIntModule_basic",
    "PrimMinIntDynamicModule_basic",
    "PrimMinIntModule_basic",
    "PrimsConvertElementTypeModule_basic",
    "PrimsSqueezeEmptyDimensionsModule_basic",
    "PrimsSqueezeModule_basic",
    "PrimsViewOfModule_basic",
    "PrimsViewOfZeroRankModule_basic",
    "QuantizedReluInt8_basic",
    "QuantizedReluInt32_basic",
    "QuantizedReluUint8_basic",
    "RandIntDtypeModule_basic",
    "RandIntModule_basic",
    "RandIntPinMemoryModule_basic",
    "ReduceFrobeniusNormComplexModule_basic",
    "ReduceL1NormComplexModule_basic",
    "ReduceL2NormComplexModule_basic",
    "ReduceL3NormKeepDimComplexModule_basic",
    "RreluWithNoiseBackwardEvalModule_basic",
    "RreluWithNoiseBackwardEvalStaticModule_basic",
    "RreluWithNoiseBackwardTrainModule_basic",
    "RreluWithNoiseBackwardTrainStaticModule_basic",
    "RreluWithNoiseForwardBackwardModule_basic",
    "ReshapeAliasCollapseModule_basic",
    "ReshapeAliasExpandModule_basic",
    "ReshapeExpandModule_basic",
    "Rot90DynamicDimsModule_basic",
    "SafeSoftmaxModule_basic",
    "SafeSoftmaxNonNoneDtypeModule_basic",
    "ScalarConstantTupleModule_basic",
    "ScalarImplicitFloatModule_basic",
    "ScalarImplicitIntModule_basic",
    "ScatterReduceFloatMaxModule",
    "ScatterReduceFloatMeanModule",
    "ScatterReduceFloatMeanModuleIncludeSelf",
    "ScatterReduceFloatMinModule",
    "ScatterReduceFloatProdModule",
    "ScatterReduceFloatSumModule",
    "ScatterReduceIntMaxModule",
    "ScatterReduceIntMeanModule",
    "ScatterReduceIntMeanModuleIncludeSelf",
    "ScatterReduceIntMinModule",
    "ScatterReduceIntProdModule",
    "ScatterReduceIntSumModule",
    "SelectScattertModule_basic",
    "SelectScattertStaticModule_basic",
    "SignAndLogarithmOfDeterminantModule_F32",
    "SignAndLogarithmOfDeterminantBatchedModule_F32",
    "SignAndLogarithmOfDeterminantDynamicModule_F32",
    "SliceEndSleStartModule_basic",
    "SliceOutOfUpperBoundIndexModule_basic",
    "SliceScatterModule_basic",
    "SliceScatterNegativeDimModule_basic",
    "SliceScatterNegativeEndModule_basic",
    "SliceScatterStaticModule_basic",
    "SliceScatterStepVariationModule_basic",
    "SliceScatterZeroDimModule_basic",
    "SliceStartEqEndModule_basic",
    "SoftmaxBackwardModule_basic",
    "SortIntListReverse_basic",
    "SortIntList_basic",
    "SplitDimDynamicModule_basic",
    "SplitDimStaticModule_basic",
    "SqrtIntConstantModule_basic",
    "SqrtIntModule_basic",
    "SubFloatModule_basic",
    "SubIntModule_basic",
    "TanhBackward_basic",
    "TensorsConcatComplex128FloatModule_basic",
    "TensorsConcatComplex64FloatModule_basic",
    "TensorsConcatComplex128IntModule_basic",
    "TensorToBoolZeroRank_basic",
    "TensorToBool_basic",
    "TensorToFloatZeroRank_basic",
    "TensorToFloat_basic",
    "TensorToIntZeroRank_basic",
    "TensorToInt_basic",
    "TestMultipleTensorAndPrimitiveTypesReturn_basic",
    "Threshold1dFloatModule_basic",
    "Threshold1dIntI32Module_basic",
    "Threshold1dIntModule_basic",
    "Threshold2dFloatModule_basic",
    "Threshold2dIntModule_basic",
    "Threshold3dFloatModule_basic",
    "Threshold3dIntModule_basic",
    "ThresholdBackward1dFloatModule_basic",
    "ThresholdBackward1dIntModule_basic",
    "ThresholdBackward1dMixedModule_basic",
    "ThresholdBackward2dFloatModule_basic",
    "ThresholdBackward2dIntModule_basic",
    "ThresholdBackward2dMixedModule_basic",
    "ThresholdBackward3dFloatModule_basic",
    "ThresholdBackward3dIntModule_basic",
    "ThresholdBackward3dMixedModule_basic",
    "ToCopyBoolDTypeStaticModule_basic",
    "ToCopyModule_basic",
    "ToCopyWithDTypeFalsePinMemoryModule_basic",
    "ToCopyWithDTypeModule_basic",
    "TorchPrimLoopForLikeModule_basic",
    "TorchPrimLoopWhileLikeModule_basic",
    "TraceModule_basic",
    "TraceModule_empty",
    "TraceModule_nonsquare",
    "TraceSignedIntModule_basic",
    "TraceUnsignedIntModule_basic",
    "TraceUnsignedIntModule_empty",
    "UniformModule_basic",
    "UniformNoCorrelationModule_basic",
    "UniformStaticShapeModule_basic",
    "UnsafeIndexPutHackedTwin1DFloatNonAccumulateModule_basic",
    "UnsafeView1DFoldModule_basic",
    "UnsafeViewCollapseDynamicWithAtenSizeIntModule_basic",
    "UnsafeViewCollapseModule_basic",
    "UnsafeViewDynamicExpandModule_basic",
    "UnsafeViewDynamicExpandWithAtenSizeIntModule_basic",
    "UnsafeViewExpandModule_basic",
    "UpSampleNearest2dBackwardScalesNone_basic",
    "UpSampleNearest2dBackward_basic",
    "UpSampleNearest2dDynamicFactor_basic",
    "UpSampleNearest2dStaticFactor_basic",
    "UpSampleNearest2d_basic",
    "ViewCollapseDynamicWithAtenSizeIntModule_basic",
    "ViewCollapseModule_basic",
    "ViewDynamicExpandCollapseModule_basic",
    "ViewDynamicExpandCollapseWithAtenIntModule_basic",
    "ViewDynamicExpandModule_basic",
    "ViewDynamicExpandWithAtenSizeIntModule_basic",
    "ViewExpandDynamicDimModule_basic",
    "ViewNoChange1dModule_basic",
    "ViewNoChange2dModule_basic",
    "ViewNoChange3dModule_basic",
    "WeightNormInterfaceModule_basic",
    "_Convolution2DAllFalseModule_basic",
    "_Convolution2DBenchmarkModule_basic",
    "_Convolution2DCudnnModule_basic",
    "_Convolution2DDeterministicModule_basic",
    "_Convolution2DTF32Module_basic",
    "_ConvolutionDeprecated2DAllFalseModule_basic",
    "_ConvolutionDeprecated2DBenchmarkModule_basic",
    "_ConvolutionDeprecated2DCudnnModule_basic",
    "_ConvolutionDeprecated2DDeterministicModule_basic",
    "_SoftmaxModule_basic",
    # Failure - onnx_import
    # Failure - onnx_lowering: onnx.SplitToSequence
    "ChunkListUnpackUneven_Module_basic",
    "TensorSplitSections_GetItemModule_basic",
    "TensorSplitSections_ListUnpackModule_basic",
    # Failure - onnx_lowering: onnx.AveragePool
    "AdaptiveAvgPool1dGeneralDynamicNoBatches_basic",
    # these diagonal modules are currently failing due to dynamic shape.
    # We are currently testing aten.diagonal using DiagonalWithStaticShapeModule instead.
    # when the issue is fixed, please remove DiagonalWithStaticShapeModule as well as the xfails here.
    "TileBigDimsSizeModule_basic",
    "TileSmallDimsSizeModule_basic",
    # Failure - onnx_lowering: onnx.ReduceProd
    "ReduceProdFloatModule_basic",
    "ReduceProdDtypeFloatModule_basic",
    "ReduceProdElementTypeBoolModule_basic",
    "ReduceProdUnsignedIntModule_basic",
    "ReduceProdSignedIntModule_basic",
    "ReduceProdDtypeIntModule_basic",
    # ERROR: dtype (torch.float32) is not equal to golden dtype (torch.float64)
    "RandnDtypeDeviceModule_basic",
    "RandnGeneratorF64Module_basic",
    "RandnGeneratorModule_basic",
    "RandnModule_basic",
    "RandnLikeModule_basic",
    "BernoulliFloatModule_basic",
    "BernoulliPModule_basic",
    "BernoulliTensorModule_basic",
    # Failure - onnx_lowering: onnx.ReduceProd
    "ReduceProdDimIntFloatModule_basic",
    # Failure - onnx_lowering: onnx.ScatterElements
    "ScatterReduceFloatMaxModuleIncludeSelf",
    "ScatterReduceFloatMinModuleIncludeSelf",
    "ScatterReduceIntMaxModuleIncludeSelf",
    "ScatterReduceIntMinModuleIncludeSelf",
    "ScatterValueFloatModule_basic",
    # Failure - onnx_lowering: onnx.ScatterND
    "IndexPut1DFloatAccumulateModule_basic",
    "IndexPut1DIntAccumulateModule_basic",
    "IndexPut2DFloatAccumulateModule_basic",
    "IndexPut2DIntAccumulateModule_basic",
    "IndexPut3DFloatAccumulateModule_basic",
    "IndexPut3DIntAccumulateModule_basic",
    "IndexPutHackedTwin1DFloatAccumulateModule_basic",
    "IndexPutHackedTwin1DIntAccumulateModule_basic",
    "IndexPutHackedTwin2DFloatAccumulateModule_basic",
    "IndexPutHackedTwin2DIntAccumulateModule_basic",
    "IndexPutHackedTwin3DFloatAccumulateModule_basic",
    "IndexPutHackedTwin3DIntAccumulateModule_basic",
    # RuntimeError: unsupported input type: Device
    "PrimsIotaModule_basic",
    # unimplemented torchvision.deform_conv2d torch->linalg
    "DeformConv2D_basic",
    # Error: 'aten::renorm' to ONNX opset version 17 is not supported.
    "RenormModuleFloat16_basic",
    "RenormModuleFloat32NegativeDim_basic",
    "RenormModuleFloat32_basic",
    "RenormModuleFloat32DynamicDims_basic",
    "Rot90BasicModule_basic",
    "Rot90DynamicDymsModule_basic",
    "Rot90MultipleRotationsModule_basic",
    "Rot90NegativeEvenRotationsModule_basic",
    "Rot90NegativeOddRotationsModule_basic",
    # Failure - unknown
    "BernoulliModule_basic",
    "Conv_Transpose1dModule_basic",
    "Conv_Transpose3dModule_basic",
    "CopyWithDifferentDTypesAndSizesModule_basic",
    "CopyWithDifferentDTypesModule_basic",
    "CosineSimilarityStaticBroadcastModule_basic",
    "CumsumInputDtypeInt32Module_basic",
    "CumprodModule_basic",
    "CumprodInputDtypeInt32Module_basic",
    "CumprodStaticModule_basic",
    "CumprodStaticNegativeDimModule_basic",
    "ElementwiseAcosIntModule_basic",
    "ElementwiseAsinIntModule_basic",
    "ElementwiseAtanTensorIntModule_basic",
    "ElementwiseCosIntModule_basic",
    "ElementwiseDivTensorRoundingModeTruncModule_basic",
    "ElementwiseDivTensorRoundingModeTruncStaticModule_basic",
    "ElementwiseErfIntModule_basic",
    "ElementwiseExpIntModule_basic",
    "ElementwiseLogIntModule_basic",
    "ElementwiseSigmoidIntModule_basic",
    "ElementwiseSinIntModule_basic",
    "ElementwiseTanIntModule_basic",
    "ElementwiseToDtypeI64ToUI8Module_basic",
    "ElementwiseUnaryIntModule_basic",
    "ElementwiseFloatTensorGtIntTensorModule_basic",
    "ElementwiseSignbitModule_basic",
    "ElementwiseSignbitIntModule_basic",
    "ElementwiseFracModule_basic",
    "ElementwiseCopysignModule_basic",
    "ElementwiseLdexpModule_basic",
    "Exp2StaticIntModule_basic",
    "MaskedFillTensorFloatValueModule_basic",
    "NativeDropoutTrainModule_basic",
    "NativeDropoutTrainStaticShapeModule_basic",
    "ReduceAnyFloatModule_basic",
    "ReduceMaxAlongDimUnsignedInt_basic",
    "ReduceMinAlongDimUnsignedInt_basic",
    "UnfoldModule_basic",
    "Unfold_Module_Rank_4",
    "Unfold_Module_Rank_Zero_basic",
    "Unfold_Module_Rank_Zero_Size_Zero_basic",
    "Unfold_Module_Dynamic_basic",
    "ViewDtypeStaticModule_basic",
    "Aten_TrilinearModule_basic",
    "Aten_TrilinearModuleSumdims_basic",
    "Aten_TrilinearModuleSumAllDims_basic",
    "Aten_TrilinearModuleVaryingRanks_basic",
    "Aten_TrilinearModuleVaryingRanksUnorderedExpands_basic",
    "Aten_TrilinearModuleZerodDimBug_basic",
}

if torch_version_for_comparison() < version.parse("2.3.0.dev"):
    ONNX_XFAIL_SET = ONNX_XFAIL_SET | {
        # ERROR: shape (torch.Size([6, 4, 5])) is not equal to golden shape (torch.Size([120]))
        "RepeatInterleaveSelfIntNoDimModule_basic",
    }

if torch_version_for_comparison() < version.parse("2.4.0.dev"):
    ONNX_XFAIL_SET = ONNX_XFAIL_SET | {
        # torch.onnx.errors.UnsupportedOperatorError: Exporting the operator 'aten::bitwise_left_shift' to ONNX opset version 17 is not supported.
        "ElementwiseBitwiseLeftShiftInt32Module_basic",
        "ElementwiseBitwiseLeftShiftInt64Module_basic",
        "ElementwiseBitwiseLeftShiftInt8Module_basic",
        # bitwise and support has been added in torch nightly
        "ElementwiseAndScalarModule_basic",
        "ElementwiseAndScalarStaticShapeModule_basic",
        "ElementwiseBitwiseAndModule_basic",
        "ElementwiseBitwiseAndScalarInt32Module_basic",
        "ElementwiseBitwiseAndScalarInt64Module_basic",
        "ElementwiseBitwiseAndScalarInt8Module_basic",
        "ElementwiseBitwiseAndStaticShapeModule_basic",
    }

if torch_version_for_comparison() >= version.parse("2.5.0.dev"):
    ONNX_XFAIL_SET = ONNX_XFAIL_SET | {
        # ERROR: value (Tensor with shape=[2, 3, 8, 20], dtype=torch.float32, min=+nan, max=+nan, mean=+nan) is not close to golden value (Tensor with shape=[2, 3, 8, 20], dtype=torch.float32, min=-2.394, max=+2.454, mean=-0.02828)
        "ScaledDotProductAttentionBoolMaskModule_basic",
    }

if torch_version_for_comparison() > version.parse("2.5.1"):
    ONNX_XFAIL_SET = ONNX_XFAIL_SET | {
        # error: 'memref.cast' op operand type 'memref<2x6x4x3xf32>' and result type 'memref<2x6x5x3xf32>' are cast incompatible
        # torch.onnx.export produces onnx.MaxPool op with incorrect output shape of 2x6x5x3 instead of 2x6x4x3
        "MaxPool2dStaticCeilModeTrueReduceOutputModule_basic",
    }

if torch_version_for_comparison() < version.parse("2.4.0.dev"):
    STABLEHLO_PASS_SET = STABLEHLO_PASS_SET - {
        "AtenIntMM_basic",
    }
    FX_IMPORTER_STABLEHLO_XFAIL_SET = FX_IMPORTER_STABLEHLO_XFAIL_SET | {
        "AtenIntMM_basic",
    }

if torch_version_for_comparison() > version.parse("2.4.0.dev"):
    STABLEHLO_PASS_SET = STABLEHLO_PASS_SET - {
        "ElementwiseCreateComplexModule_basic",
        "ElementwiseTanIntModule_basic",
        "ElementwiseTanModule_basic",
    }
    FX_IMPORTER_STABLEHLO_XFAIL_SET = FX_IMPORTER_STABLEHLO_XFAIL_SET | {
        "ElementwiseCreateComplexModule_basic",
        "ElementwiseTanIntModule_basic",
        "ElementwiseTanModule_basic",
    }


ONNX_CRASHING_SET = LINALG_CRASHING_SET | {
    "FakeQuantizePerTensorAffineModule_basic",
    "FakeQuantizePerTensorAffineDynamicShapeModule_basic",
    "ElementwisePreluModule_basic",
    "ViewDynamicExpandCollapseWithParallelUnknownDimModule_basic",
    "ScatterReduceFloatProdModuleIncludeSelf",
    "ScatterReduceFloatSumModuleIncludeSelf",
    "ScatterReduceIntProdModuleIncludeSelf",
    "ScatterReduceIntSumModuleIncludeSelf",
    # The following test sporadically stopped producing correct numerics for the golden value in the CI.
    # For now, we are removing the test until this issue has been debugged.
    "QuantizedMLP_basic",
    # Runtime crash: mismatched size for broadcast
    "MaxPool2dWithIndicesAllNegativeValuesModule_basic",
    "MaxPool2dWithIndicesNonDefaultPaddingModule_basic",
    "MaxPool3dWithIndicesAllNegativeValuesModule_basic",
    "MaxPool3dWithIndicesNonDefaultPaddingModule_basic",
    "StdDimEmptyDimModule_basic",
    "StdCorrectionEmptyDimModule_basic",
    "VarCorrectionEmptyDimModule_basic",
    "VarDimEmptyDimModule_basic",
    # Runtime op verification: rank mismatch in memref.cast
    "ViewSizeFromOtherTensor_basic",
}

FX_IMPORTER_TOSA_XFAIL_SET = {
    "UniformModule_basic",
    "UniformStaticShapeModule_basic",
    "AtenFftRfft2DLastDim_basic",
    "AtenFftRfft2DMiddleDim_basic",
    "IsInfiniteModule_basic",
    "LayerNormFwAndBwModule_basic",
    "LayerNormManualFwAndBwModule_basic",
    "SelfAttentionFwAndBwModule_basic",
    "Threshold3dIntModule_basic",
    "ElementwiseCopysignModule_basic",
    "ElementwiseSignbitModule_basic",
    "Aten_TrilinearModuleVaryingRanks_basic",
    "Aten_TrilinearModuleZerodDimBug_basic",
    "MaxPool3dEmptyStrideStaticModule_basic",
    "MaxPool3dLargeDatadModule_basic",
    "MaxPool3dModuleRandomSimple_basic",
    "MaxPool3dModule_basic",
    "MaxPool3dStaticModule_basic",
    "ViewDtypeStaticModule_basic",
    "Unfold_Module_Rank_Zero_Size_Zero_basic",
    "ArangeZeroElementOutputModule_basic",
    "NumpyTRank0Module_basic",
    "Permute0RankModule_basic",
    "SliceOutOfUpperBoundIndexModule_basic",
    "SliceOutOfUpperBoundIndexStaticModule_basic",
    "SliceStartEqEndModule_basic",
    "ElementwiseCreateComplexModule_basic",
    "AtenPolarDoubleModule_basic",
    "AtenPolarFloatModule_basic",
    "HstackBasicComplexModule_basic",
    "AtenIntMM_basic",
    "AtenKthvalueDynamicDimsModule_basic",
    "AtenKthvalueFloat64DynamicDimsModule_basic",
    "AtenKthvalueFloat64Module_basic",
    "AtenKthvalueKeepDimModule_basic",
    "AtenKthvalueModule_basic",
    "AvgPool3dStaticModule_basic",
    "Conv_Transpose1dModule_basic",
    "Conv_Transpose1dStaticModule_basic",
    "Conv_Transpose2dStaticModule_basic",
    "Conv_Transpose3dModule_basic",
    "Conv_Transpose3dStaticModule_basic",
    "IndexPutWithNoneAndBroadcastModule_basic",
    "MaskedScatterStaticBasic_basic",
    "MaxUnpool3dModulePad0_basic",
    "MaxUnpool3dModule_basic",
    "MultinomialModule2D_F32",
    "MultinomialModule2D_basic",
    "MultinomialModule_basic",
    # REMOVE WHEN ENABLE_GQA IS ADDED
    "ScatterAddStaticModule_basic",
    "TensorsConcatComplex128FloatModule_basic",
    "TensorsConcatComplex128IntModule_basic",
    "TensorsConcatComplex64FloatModule_basic",
    "TimeOutModule_basic",
    "TrilIndicesAllZerosModule_basic",
    "TrilIndicesModule_basic",
    "TrilIndicesNegativeOffsetModule_basic",
    "TrilIndicesOfssetGreaterThanRowModule_basic",
    "TriuIndicesAllZerosModule_basic",
    "TriuIndicesModule_basic",
    "TriuIndicesNegativeOffsetModule_basic",
    "TypeConversionUint8ToF32Module_basic",
    "WeightNormInterfaceModule_basic",
    "AdaptiveAvgPool3dDynamicNoBatch_basic",
    "AdaptiveAvgPool3dDynamic_basic",
    "AdaptiveMaxPool1dDynamicNoBatch_basic",
    "AdaptiveMaxPool1dDynamic_basic",
    "AdaptiveMaxPool1dStatic_basic",
    "AdaptiveMaxPool2dDynamicNoBatch_basic",
    "AdaptiveMaxPool2dDynamicWithIndices_basic",
    "AdaptiveMaxPool2dDynamic_basic",
    "AdaptiveMaxPool2dStaticWithIndices_basic",
    "AdaptiveMaxPool2dStatic_basic",
    "AdaptiveMaxPool3dDynamicNoBatch_basic",
    "AdaptiveMaxPool3dDynamicWithIndices_basic",
    "AdaptiveMaxPool3dDynamic_basic",
    "AdaptiveMaxPool3dStaticWithIndices_basic",
    "AdaptiveMaxPool3dStatic_basic",
    "AddIntModule_basic",
    "AddFloatIntModule_basic",
    "AllBoolFalseModule_basic",
    "AllBoolTrueModule_basic",
    "AnyBoolFalseModule_basic",
    "AnyBoolTrueModule_basic",
    "ArangeStartOutViewModule_basic",
    "AtenComplexImagModule_basic",
    "AtenComplexRealModule_basic",
    "AtenComplexViewModule_basic",
    "AtenEmbeddingBagStaticModule_basic",
    "AtenEmbeddingBagSumExample_basic",
    "AtenEyeMModuleInt2D_basic",
    "AtenEyeModuleInt2D_basic",
    "AtenFloatScalarModule_basic",
    "AtenIntBoolOpConstFalseModule_basic",
    "AtenIntBoolOpConstTrueModule_basic",
    "AtenIntBoolOpModule_basic",
    "AtenIntTensorByteDtypeModule_basic",
    "AtenIntTensorCharDtypeModule_basic",
    "AtenItemFpOpModule_basic",
    "AtenItemIntOpModule_basic",
    "AtenMatmulQMixedSigni8Transpose_basic",
    "AtenMatmulQMixedSigni8_basic",
    "AtenMatmulQint8MV_basic",
    "AtenMatmulQint8VM_basic",
    "AtenMatmulQint8VV_basic",
    "AtenMatmulQint8_basic",
    "AtenMmIntTypes_basic",
    "AtenMmQMixedSigni8_basic",
    "AtenMmQint8_basic",
    "AtenMmQuint8_basic",
    "AtenRealView128Module_basic",
    "AtenRealView64Module_basic",
    "AtenSubFloatModule_basic",
    "AtenTopKModule_basic",
    "AtenTopKSmallestModule_basic",
    "Aten_CastLongModule_basic",
    "Aten_EmbeddingBagExample_basic",
    "AvgPool1dFloatModule_basic",
    "AvgPool1dIntModule_basic",
    "AvgPool1dStaticModule_basic",
    "AvgPool2dCeilModeTrueModule_basic",
    "AvgPool2dDivisorOverrideModule_basic",
    "AvgPool2dFloatModule_basic",
    "AvgPool2dIntModule_basic",
    "AvgPool2dStaticModule_basic",
    "BernoulliFloatModule_basic",
    "BernoulliModule_basic",
    "BernoulliOnesModule_basic",
    "BernoulliPModule_basic",
    "BernoulliTensorModule_basic",
    "BernoulliZerosModule_basic",
    "BincountMinlengthModule_basic",
    "BincountModule_basic",
    "BincountStaticSizeModule_basic",
    "BmmIntModule_basic",
    "BoolFloatConstantModule_basic",
    "BoolFloatFalseModule_basic",
    "BoolFloatTrueModule_basic",
    "BoolIntConstantModule_basic",
    "BoolIntFalseModule_basic",
    "BoolIntTrueModule_basic",
    "BroadcastDynamicDimModule_basic",
    "CeilFloatModule_basic",
    "ConstantBoolParameterModule_basic",
    "ContainsIntList_False",
    "ContainsIntList_True",
    "Conv1dModule_basic",
    "Conv1dDepthwiseWithPaddingDilationStrideStaticModule_basic",
    "Conv1dWithSamePaddingModule_basic",
    "Conv1dWithValidPaddingModule_basic",
    "Conv1dGroupModule_basic",
    "Conv2dQInt8Module_basic",
    "Conv2dQInt8Module_depthwise",
    "Conv2dQInt8Module_grouped",
    "Conv2dQInt8Module_not_depthwise",
    "Conv2dQInt8PerChannelModule_basic",
    "Conv2dQInt8PerChannelModule_depthwise",
    "Conv2dQInt8PerChannelModule_grouped",
    "Conv2dWithPaddingDilationStrideStaticModule_grouped",
    "Conv2dWithPaddingDilationStrideStaticModule_grouped_multiplier",
    "Conv3dModule_basic",
    "Conv3dWithSamePaddingModule_basic",
    "Conv3dWithValidPaddingModule_basic",
    "ConvTbcModule_basic",
    "ConvTranspose2DQInt8_basic",
    "Conv_Transpose2dModule_basic",
    "ConvolutionBackwardModule2DPadded_basic",
    "ConvolutionBackwardModule2DStatic_basic",
    "ConvolutionBackwardModule2DStrided_basic",
    "ConvolutionBackwardModule2D_basic",
    "ConvolutionModule2DGroups_basic",
    "ConvolutionModule2DTransposeNonUnitOutputPadding_basic",
    "ConvolutionModule2DTransposeStridedStatic_basic",
    "ConvolutionModule2DTransposeStrided_basic",
    "ConvolutionModule2DTranspose_basic",
    "CopyWithDifferentDTypesModule_basic",
    "CumsumInputDtypeInt32Module_basic",
    "CumsumModule_basic",
    "CumsumStaticModule_basic",
    "CumsumStaticNegativeDimModule_basic",
    "CumprodModule_basic",
    "CumprodInputDtypeInt32Module_basic",
    "CumprodStaticModule_basic",
    "CumprodStaticNegativeDimModule_basic",
    "DeformConv2D_basic",
    "DeterminantBatchedModule_F32",
    "DeterminantDynamicModule_F32",
    "DeterminantModule_F32",
    "DivFloatModule_basic",
    "DivIntModule_basic",
    "ElementwiseAcosIntModule_basic",
    "ElementwiseAcosModule_basic",
    "ElementwiseAcoshIntModule_basic",
    "ElementwiseAcoshModule_basic",
    "ElementwiseAsinIntModule_basic",
    "ElementwiseAsinModule_basic",
    "ElementwiseAsinhIntModule_basic",
    "ElementwiseAsinhModule_basic",
    "ElementwiseAtan2FloatIntModule_basic",
    "ElementwiseAtan2FloatIntStaticModule_basic",
    "ElementwiseAtan2TensorFloatModule_basic",
    "ElementwiseAtan2TensorFloatStaticModule_basic",
    "ElementwiseAtan2TensorIntModule_basic",
    "ElementwiseAtan2TensorIntStaticModule_basic",
    "ElementwiseAtanTensorFloatModule_basic",
    "ElementwiseAtanTensorIntModule_basic",
    "ElementwiseAtanhIntModule_basic",
    "ElementwiseAtanhModule_basic",
    "ElementwiseCoshIntModule_basic",
    "ElementwiseCoshModule_basic",
    "ElementwiseDequantizePerChannelModule_basic",
    "ElementwiseDequantizePerTensorModule_basic",
    "ElementwiseExpm1IntModule_basic",
    "ElementwiseExpm1Module_basic",
    "ElementwiseMulTensorComplexDiffModule_basic",
    "ElementwiseMulTensorComplexModule_basic",
    "ElementwiseQuantizePerTensorModule_basic",
    "ElementwiseQuantizePerTensorUIntModule_basic",
    "ElementwiseSinhIntModule_basic",
    "ElementwiseSinhModule_basic",
    "ElementwiseSpecialExpm1IntModule_basic",
    "ElementwiseSpecialExpm1Module_basic",
    "ElementwiseToDtypeF32ToI64Module_basic",
    "ElementwiseToDtypeI64ToUI8Module_basic",
    "ElementwiseWhereScalarOtherStaticModule_basic",
    "EqIntModule_basic",
    "FloatImplicitModule_basic",
    "FullLikeModuleInt2D_basic",
    "FullLikeModuleInt3D_basic",
    "GeFloatIntModule_basic",
    "GeFloatModule_basic",
    "GeIntModule_basic",
    "GridSamplerBasic1_basic",
    "GridSamplerBasic2_basic",
    "GridSamplerBasic3_basic",
    "GridSamplerBasic4_basic",
    "GtFloatIntModule_basic",
    "GtIntModule_basic",
    "IndexPut1DFloatAccumulateModule_basic",
    "IndexPut1DIntAccumulateModule_basic",
    "IndexPut2DFloatAccumulateModule_basic",
    "IndexPut2DIntAccumulateModule_basic",
    "IndexPut3DFloatAccumulateModule_basic",
    "IndexPut3DIntAccumulateModule_basic",
    "IndexPutHackedTwin1DFloatAccumulateModule_basic",
    "IndexPutHackedTwin1DIntAccumulateModule_basic",
    "IndexPutHackedTwin2DFloatAccumulateModule_basic",
    "IndexPutHackedTwin2DIntAccumulateModule_basic",
    "IndexPutHackedTwin3DFloatAccumulateModule_basic",
    "IndexPutHackedTwin3DIntAccumulateModule_basic",
    "IndexPutImpl1DFloatAccumulateModule_basic",
    "IndexPutImpl1DIntAccumulateModule_basic",
    "IndexPutImpl2DFloatAccumulateModule_basic",
    "IndexPutImpl2DImplicitModule_basic",
    "IndexPutImpl2DIndexModule_basic",
    "IndexPutImpl2DNoneIndexStaticModule_basic",
    "IndexPutImpl3DFloatAccumulateModule_basic",
    "IndexPutImplIndexWithNoneModule_basic",
    "InterpolateDynamicModule_sizes_bilinear",
    "InterpolateDynamicModule_scales_recompute_bilinear",
    "IntFloatModule_basic",
    "IntImplicitModule_basic",
    "IsFloatingPointFloat_True",
    "IsFloatingPointInt_False",
    "LenStrModule_basic",
    "LinalgNormKeepDimComplexModule_basic",
    "LinalgVectorNormComplexModule_basic",
    "LinspaceDtypeModule_basic",
    "LinspaceEmptyModule_basic",
    "MaskedScatterStaticBasic_basic",
    "MaxPool1dCeilModeTrueModule_basic",
    "MaxPool1dModule_basic",
    "MaxPool2dCeilModeTrueModule_basic",
    "MaxPool2dModule_basic",
    "MaxPool2dWithIndicesAllNegativeValuesModule_basic",
    "MaxPool2dWithIndicesAllOnesModule_basic",
    "MaxPool2dWithIndicesBackwardDynamic3DModule_basic",
    "MaxPool2dWithIndicesBackwardDynamic4DModule_basic",
    "MaxPool2dWithIndicesBackwardStatic3DModule_basic",
    "MaxPool2dWithIndicesBackwardStatic4DModule_basic",
    "MaxPool2dWithIndicesCeilModeTrueModule_basic",
    "MaxPool2dWithIndicesFullSizeKernelModule_basic",
    "MaxPool2dWithIndicesModule_basic",
    "MaxPool2dWithIndicesNonDefaultDilationModule_basic",
    "MaxPool2dWithIndicesNonDefaultPaddingModule_basic",
    "MaxPool2dWithIndicesNonDefaultParamsModule_basic",
    "MaxPool2dWithIndicesNonDefaultStrideModule_basic",
    "MaxPool2dWithIndicesStaticModule_basic",
    "MaxPool3dCeilModeTrueModule_basic",
    "MaxPool3dStaticCeilModeTrueModule_basic",
    "MaxPool3dWithIndicesAllNegativeValuesModule_basic",
    "MaxPool3dWithIndicesAllOnesModule_basic",
    "MaxPool3dWithIndicesCeilModeTrueModule_basic",
    "MaxPool3dWithIndicesFullSizeKernelModule_basic",
    "MaxPool3dWithIndicesModule_basic",
    "MaxPool3dWithIndicesNonDefaultDilationModule_basic",
    "MaxPool3dWithIndicesNonDefaultPaddingModule_basic",
    "MaxPool3dWithIndicesNonDefaultParamsModule_basic",
    "MaxPool3dWithIndicesNonDefaultStrideModule_basic",
    "MaxPool3dWithIndicesStaticModule_basic",
    "MeanDimEmptyDimModule_basic",
    "MlGroupNormManualModule_basic",
    "MlGroupNormModule_basic",
    "MlLayerNormManualModule_basic",
    "MlLayerNormModule_basic",
    "MulFloatModule_basic",
    "MulIntModule_basic",
    "NativeBatchNorm1DModule_basic",
    "NativeBatchNorm2DModule_basic",
    "NativeBatchNorm3DModule_basic",
    "NativeBatchNormNoneWeightModule_basic",
    "NativeGroupNormBackwardModule_basic",
    "NeFloatIntModule_basic",
    "NeIntModule_basic",
    "NewFullModuleInt2D_basic",
    "NewFullModuleInt3D_basic",
    "NllLossModuleBackward1DMeanWeight_basic",
    "NllLossModuleBackward1DMean_basic",
    "NllLossModuleBackward1DSumWeight_basic",
    "NllLossModuleBackward1DSum_basic",
    "NllLossModuleBackward1DWeight_basic",
    "NllLossModuleBackward1D_basic",
    "NormScalarComplexModule_basic",
    "NormScalarModule_basic",
    "NormScalarOptDimKeepDimComplexModule_basic",
    "NormalFunctionalModule_basic",
    "NumelModule_basic",
    "NumelZeroRankModule_basic",
    "OnesLikeModule_falsePinMemory",
    "PowIntIntModule_basic",
    "PrimMaxIntModule_basic",
    "PrimMinIntDynamicModule_basic",
    "PrimMinIntModule_basic",
    "PrimsConvertElementTypeModule_basic",
    "PrimsSqueezeEmptyDimensionsModule_basic",
    "PrimsSqueezeModule_basic",
    "PrimsViewOfModule_basic",
    "PrimsViewOfZeroRankModule_basic",
    "QuantizedBatchedInputSingleLayer_basic",
    "QuantizedMLP_basic",
    "QuantizedNoLayer_basic",
    "QuantizedReluInt32_basic",
    "QuantizedReluInt8_basic",
    "QuantizedReluUint8_basic",
    "QuantizedSingleLayer_basic",
    "RandnDtypeDeviceModule_basic",
    "RandnGeneratorF64Module_basic",
    "RandnGeneratorModule_basic",
    "RandnLikeDtypeModule_basic",
    "RandnLikeModule_basic",
    "RandnModule_basic",
    "ReduceAllDimEmpty_basic",
    "ReduceFrobeniusNormComplexModule_basic",
    "ReduceL1NormComplexModule_basic",
    "ReduceL2NormComplexModule_basic",
    "ReduceL3NormKeepDimComplexModule_basic",
    "ReduceMaxAlongDimUnsignedInt_basic",
    "ReduceMinAlongDimUnsignedInt_basic",
    "ReduceSumDimIntListEmptyDimModule_basic",
    "RollModule_basic",
    "ScalarConstantTupleModule_basic",
    "ScalarImplicitFloatModule_basic",
    "ScalarImplicitIntModule_basic",
    "ScatterReduceFloatMaxModule",
    "ScatterReduceFloatMaxModuleIncludeSelf",
    "ScatterReduceFloatMeanModule",
    "ScatterReduceFloatMeanModuleIncludeSelf",
    "ScatterReduceFloatMinModule",
    "ScatterReduceFloatMinModuleIncludeSelf",
    "ScatterReduceFloatProdModule",
    "ScatterReduceFloatProdModuleIncludeSelf",
    "ScatterReduceFloatSumModule",
    "ScatterReduceFloatSumModuleIncludeSelf",
    "ScatterReduceIntMaxModule",
    "ScatterReduceIntMaxModuleIncludeSelf",
    "ScatterReduceIntMeanModule",
    "ScatterReduceIntMeanModuleIncludeSelf",
    "ScatterReduceIntMinModule",
    "ScatterReduceIntMinModuleIncludeSelf",
    "ScatterReduceIntProdModule",
    "ScatterReduceIntProdModuleIncludeSelf",
    "ScatterReduceIntSumModule",
    "ScatterReduceIntSumModuleIncludeSelf",
    "ScatterSrcModule_basic",
    "ScatterSrcStaticModule_basic",
    "ScatterValueFloatModule_basic",
    "ScatterValueIntModule_basic",
    "SignAndLogarithmOfDeterminantModule_F32",
    "SignAndLogarithmOfDeterminantBatchedModule_F32",
    "SignAndLogarithmOfDeterminantDynamicModule_F32",
    "SliceStaticComplexInputModule_basic",
    "SliceCopyStartGreaterThanDimSize_Module_basic",
    "SliceEndSleStartModule_basic",
    "SliceOutOfLowerBoundEndIndexModule_basic",
    "SliceOutOfLowerBoundStartIndexModule_basic",
    "SliceSizeTwoStepModule_basic",
    "SortIntListReverse_basic",
    "SortIntList_basic",
    "SortTensorDescending_basic",
    "SortTensorInteger_basic",
    "SortTensorNegativeDimension_basic",
    "SortTensorSpecificDimension_basic",
    "SortTensor_basic",
    "SplitDimDynamicModule_basic",
    "SplitDimStaticModule_basic",
    "SqrtIntConstantModule_basic",
    "SqrtIntModule_basic",
    "SubFloatModule_basic",
    "SubIntModule_basic",
    "TModuleRank0_basic",
    "TensorToBoolZeroRank_basic",
    "TensorToBool_basic",
    "TensorToFloatZeroRank_basic",
    "TensorToFloat_basic",
    "TensorToIntZeroRank_basic",
    "TensorToInt_basic",
    "TestMultipleTensorAndPrimitiveTypesReturn_basic",
    "ThresholdBackward2dMixedModule_basic",
    "ToCopyWithDTypeFalsePinMemoryModule_basic",
    "ToCopyWithDTypeModule_basic",
    "TorchPrimLoopForLikeModule_basic",
    "TorchPrimLoopWhileLikeModule_basic",
    "TraceModule_empty",
    "TraceUnsignedIntModule_empty",
    "TypeConversionI1ToF64Module_basic",
    "TypeConversionI1ToI32Module_basic",
    "UnsafeViewCollapseDynamicWithAtenSizeIntModule_basic",
    "UpSampleNearest2dBackwardScalesNone_basic",
    "UpSampleNearest2dBackward_basic",
    "ViewCollapseDynamicWithAtenSizeIntModule_basic",
    "ViewSizeFromOtherTensor_basic",
    "VisionTransformerModule_basic",
    "ZerosLikeModule_falsePinMemory",
    # Unexpected failures due to new PyTorch version update
    "AdaptiveAvgPool1dGeneralDynamicNoBatches_basic",
    "AdaptiveAvgPool1dGeneralDynamic_basic",
    "AdaptiveAvgPool1dStaticEvenMultiple_basic",
    "AdaptiveAvgPool1dStaticLargerOutput_basic",
    "AdaptiveAvgPool2dDynamicNoBatch_basic",
    "AdaptiveAvgPool2dDynamic_basic",
    "CrossEntropyLossModule_basic",
    "CrossEntropyLossNoReductionModule_basic",
    "IouOfModule_basic",
    "MeshgridIndexingIJ_basic",
    "MeshgridIndexingXY_basic",
    "Meshgrid_basic",
    "ReduceFrobeniusNormKeepDimModule_basic",
    "ReduceFrobeniusNormModule_basic",
    "ScaledDotProductAttentionBoolMaskModule_basic",
    "ScaledDotProductAttentionDifferentCausalModule_basic",
    "ScaledDotProductAttentionDifferentDynamicCausalModule_basic",
    "ScaledDotProductAttentionDifferentModule_basic",
    "ScaledDotProductAttentionMaskModule_basic",
    "ScaledDotProductAttentionSameCausalModule_basic",
    "ScaledDotProductAttentionSameDynamicModule_basic",
    "ScaledDotProductAttentionSameModule_basic",
}

ONNX_TOSA_CRASHING_SET = {
    "ScatterSrcStaticModule_basic",
    "StdCorrectionEmptyDimModule_basic",
    "StdDimEmptyDimModule_basic",
    "VarCorrectionEmptyDimModule_basic",
    "VarDimEmptyDimModule_basic",
    "ViewSizeFromOtherTensor_basic",
}

ONNX_TOSA_XFAIL_SET = {
    "AtenFftRfft2DLastDim_basic",
    "AtenFftRfft2DMiddleDim_basic",
    "PowFloatIntModule_basic",
    "PowIntFloatModule_basic",
    "PowIntIntModule_basic",
    "ColumnStack0dModule_basic",
    "ColumnStack1dModule_basic",
    "ColumnStackBasicIntModule_basic",
    "Deg2radModule_basic",
    "L1LossMeanReductionModule_basic",
    "L1LossNoReductionModule_basic",
    "L1LossSumReductionModule_basic",
    "FloatPowerTensorTensorStaticModule_basic",
    "IsInfiniteModule_basic",
    "ElementwiseCopysignModule_basic",
    "ElementwiseFracModule_basic",
    "ElementwiseLdexpModule_basic",
    "ElementwiseSignbitIntModule_basic",
    "ElementwiseSignbitModule_basic",
    "Exp2StaticIntModule_basic",
    "NllLossStaticModule_basic",
    "NllLossStaticModule_mean_basic",
    "NllLossStaticModule_sum_basic",
    "NllLossStaticModule_weight_basic",
    "Exp2StaticModule_basic",
    "ElementwiseRreluWithNoiseEvalModule_basic",
    "ElementwiseRreluWithNoiseEvalStaticModule_basic",
    "ElementwiseRreluWithNoiseTrainModule_basic",
    "ElementwiseRreluWithNoiseTrainStaticModule_basic",
    "RreluWithNoiseBackwardEvalModule_basic",
    "RreluWithNoiseBackwardEvalStaticModule_basic",
    "RreluWithNoiseBackwardTrainModule_basic",
    "RreluWithNoiseBackwardTrainStaticModule_basic",
    "RreluWithNoiseForwardBackwardModule_basic",
    "Unfold_Module_Dynamic_basic",
    "Unfold_Module_Rank_4",
    "Unfold_Module_Rank_Zero_Size_Zero_basic",
    "Unfold_Module_Rank_Zero_basic",
    "ViewDtypeStaticModule_basic",
    "ArangeZeroElementOutputModule_basic",
    "LinspaceEmptyModule_basic",
    "RepeatInterleaveSelfIntNoDimModule_basic",
    "SliceOutOfUpperBoundIndexStaticModule_basic",
    "TrilIndicesAllZerosModule_basic",
    "TriuIndicesAllZerosModule_basic",
    "ElementwiseCreateComplexModule_basic",
    "ReduceAllDimFloatModule_basic",
    "ScaledDotProductAttentionDifferentCausalModule_basic",
    "HstackBasicComplexModule_basic",
    "HstackBasicFloatModule_basic",
    "HstackBasicIntFloatModule_basic",
    "HstackBasicIntModule_basic",
    "Rot90BasicModule_basic",
    "Rot90DynamicDimsModule_basic",
    "Rot90MultipleRotationsModule_basic",
    "Rot90NegativeEvenRotationsModule_basic",
    "Rot90NegativeOddRotationsModule_basic",
    "SafeSoftmaxModule_basic",
    "SafeSoftmaxNonNoneDtypeModule_basic",
    "AdaptiveAvgPool2dFixedKernelStrideSizeStaticModule_basic",
    "AdaptiveAvgPool2dNonUnitOutputSizeStaticModule_basic",
    "AdaptiveAvgPool2dOutputSizeDivisibleByInputStaticModule_basic",
    "ArgmaxKeepdimModule_basic",
    "AtenIntMM_basic",
    "AtenKthvalueDynamicDimsModule_basic",
    "AtenKthvalueFloat64DynamicDimsModule_basic",
    "AtenKthvalueFloat64Module_basic",
    "AtenKthvalueKeepDimModule_basic",
    "AtenKthvalueModule_basic",
    "AvgPool2dCountIncludePadFalseStaticModule_basic",
    "AvgPool3dStaticModule_basic",
    "Conv_Transpose1dModule_basic",
    "Conv_Transpose1dStaticModule_basic",
    "Conv_Transpose2dStaticModule_basic",
    "Conv_Transpose3dModule_basic",
    "Conv_Transpose3dStaticModule_basic",
    "ElementwiseFmaxModule_basic",
    "ElementwiseFminModule_basic",
    "ElementwiseGeluApproximateTanhModule_basic",
    "ElementwiseIntTensorLtFloatTensorModule_basic",
    "ElementwiseRad2DegIntModule_basic",
    "ElementwiseRad2DegModule_basic",
    "ElementwiseRemainderScalarModule_Bool_NegativeDivisor_basic",
    "ElementwiseRemainderScalarModule_Int_Float_NegativeDividend_basic",
    "ElementwiseRemainderScalarModule_Int_Float_NegativeDivisor_basic",
    "ElementwiseRemainderTensorModule_Int_Float_NegativeDividend_basic",
    "ElementwiseRemainderTensorModule_Int_Float_NegativeDivisor_basic",
    "ElementwiseRemainderTensorModule_Int_NegativeDividend_basic",
    "ElementwiseRemainderTensorModule_Int_NegativeDivisor_basic",
    "ElementwiseRreluTrainModule_basic",
    "ElementwiseRreluTrainStaticModule_basic",
    "FakeQuantizePerTensorAffineCachemaskModule_basic",
    "IndexPutWithNoneAndBroadcastModule_basic",
    "MaskedScatterStaticBasic_basic",
    "MaxUnpool3dModulePad0_basic",
    "MaxUnpool3dModule_basic",
    "MultinomialModule2D_F32",
    "MultinomialModule2D_basic",
    "MultinomialModule_basic",
    "ReduceAmaxEmptyDim_basic",
    "ReduceAminSingleDim_basic",
    "ReduceAminmaxAllDims_basic",
    "ReduceAminmaxSingleDim_basic",
    "ReduceAnyDimFloatModule_basic",
    "RenormModuleFloat16_basic",
    "RenormModuleFloat32DynamicDims_basic",
    "RenormModuleFloat32NegativeDim_basic",
    "RenormModuleFloat32_basic",
    "ScatterAddStaticModule_basic",
    "TensorSplitSections_GetItemModule_basic",
    "TensorSplitSections_ListUnpackModule_basic",
    "TensorsConcatComplex128FloatModule_basic",
    "TensorsConcatComplex128IntModule_basic",
    "TensorsConcatComplex64FloatModule_basic",
    "TimeOutModule_basic",
    "TypeConversionUint8ToF32Module_basic",
    "UnfoldModule_basic",
    "WeightNormInterfaceModule_basic",
    "AdaptiveAvgPool1dGeneralDynamicNoBatches_basic",
    "AdaptiveAvgPool1dGeneralDynamic_basic",
    "AdaptiveAvgPool1dNonUnitOutputSizeDynamicModule_basic",
    "AdaptiveAvgPool1dNonUnitOutputSizeStaticModule_basic",
    "AdaptiveAvgPool1dStaticEvenMultiple_basic",
    "AdaptiveAvgPool1dStaticLargerOutput_basic",
    "AdaptiveAvgPool1dUnitOutputSizeDynamicModule_basic",
    "AdaptiveAvgPool2dDynamicNoBatch_basic",
    "AdaptiveAvgPool2dDynamic_basic",
    "AdaptiveAvgPool2dNonUnitOutputSizeDynamicModule_basic",
    "AdaptiveAvgPool2dOutputSizeDivisibleByInputDynamicModule_basic",
    "AdaptiveAvgPool2dUnitOutputSizeDynamicModule_basic",
    "AdaptiveAvgPool3dDynamicNoBatch_basic",
    "AdaptiveAvgPool3dDynamic_basic",
    "AdaptiveMaxPool1dDynamicNoBatch_basic",
    "AdaptiveMaxPool1dDynamic_basic",
    "AdaptiveMaxPool1dStatic_basic",
    "AdaptiveMaxPool2dDynamicNoBatch_basic",
    "AdaptiveMaxPool2dDynamicWithIndices_basic",
    "AdaptiveMaxPool2dDynamic_basic",
    "AdaptiveMaxPool2dStaticWithIndices_basic",
    "AdaptiveMaxPool2dStatic_basic",
    "AdaptiveMaxPool3dDynamicNoBatch_basic",
    "AdaptiveMaxPool3dDynamicWithIndices_basic",
    "AdaptiveMaxPool3dDynamic_basic",
    "AdaptiveMaxPool3dStaticWithIndices_basic",
    "AdaptiveMaxPool3dStatic_basic",
    "AddCDivModule_basic",
    "AddIntModule_basic",
    "AddFloatIntModule_basic",
    "AddSizeIntModule_basic",
    "AddSizeIntNegDimModule_basic",
    "Add_MixPModule_basic",
    "Add_Module_basic",
    "AddmmModuleFloat_basic",
    "AddmmModule_broadcastable",
    "AddmmModule_differentRankBroadcastable",
    "AllBoolFalseModule_basic",
    "AllBoolTrueModule_basic",
    "AnyBoolFalseModule_basic",
    "AnyBoolTrueModule_basic",
    "ArangeStartOutDtypeModule_basic",
    "ArangeStartOutViewModule_basic",
    "ArgmaxIntModule_basic",
    "ArgmaxIntModule_multiple_maxs",
    "ArgmaxModule_basic",
    "ArgmaxModule_with_dim",
    "ArgminIntModule_basic",
    "ArgminIntModule_multiple_mins",
    "ArgminModule_basic",
    "ArgminModule_with_dim",
    "AtenComplex64Module_basic",
    "AtenComplexImagModule_basic",
    "AtenComplexRealModule_basic",
    "AtenComplexViewModule_basic",
    "AtenDiagEmbedDefaultDiag_basic",
    "AtenDiagEmbedDimDiag_basic",
    "AtenDiagEmbedNegOffsetDiag_basic",
    "AtenDiagEmbedNonDefault4DDiag_basic",
    "AtenDiagEmbedOffsetDiag_basic",
    "AtenDiagEmbedRevDimDiag_basic",
    "AtenEmbeddingBagStaticModule_basic",
    "AtenEmbeddingBagSumExample_basic",
    "AtenFloatScalarModule_basic",
    "AtenIntBoolOpConstFalseModule_basic",
    "AtenIntBoolOpConstTrueModule_basic",
    "AtenIntBoolOpModule_basic",
    "AtenIntTensorByteDtypeModule_basic",
    "AtenIntTensorCharDtypeModule_basic",
    "AtenItemFpOpModule_basic",
    "AtenItemIntOpModule_basic",
    "AtenLinalgCrossDynamic_basic",
    "AtenMatmulQMixedSigni8Transpose_basic",
    "AtenMatmulQMixedSigni8_basic",
    "AtenMatmulQint8MV_basic",
    "AtenMatmulQint8VM_basic",
    "AtenMatmulQint8VV_basic",
    "AtenMatmulQint8_basic",
    "AtenMmFloatTypes_basic",
    "AtenMmIntTypes_basic",
    "AtenMmQMixedSigni8_basic",
    "AtenMmQint8_basic",
    "AtenMmQuint8_basic",
    "AtenPolarFloatModule_basic",
    "AtenPolarDoubleModule_basic",
    "AtenRealView128Module_basic",
    "AtenRealView64Module_basic",
    "AtenSubFloatModule_basic",
    "AtenTopKModule_basic",
    "AtenTopKSmallestModule_basic",
    "Aten_TrilinearModule_basic",
    "Aten_TrilinearModuleSumdims_basic",
    "Aten_TrilinearModuleSumAllDims_basic",
    "Aten_TrilinearModuleVaryingRanks_basic",
    "Aten_TrilinearModuleVaryingRanksUnorderedExpands_basic",
    "Aten_TrilinearModuleZerodDimBug_basic",
    "AtenTrilModule_basic",
    "AtenTrilWithNegDiagonalModule_basic",
    "AtenTrilWithPosDiagonalModule_basic",
    "AtenTriuModule_basic",
    "AtenTriuWithNegDiagonalModule_basic",
    "AtenTriuWithPosDiagonalModule_basic",
    "Aten_CastLongModule_basic",
    "Aten_EmbeddingBagExample_basic",
    "AvgPool1dFloatModule_basic",
    "AvgPool1dIntModule_basic",
    "AvgPool1dStaticModule_basic",
    "AvgPool2dCeilModeTrueModule_basic",
    "AvgPool2dDivisorOverrideModule_basic",
    "AvgPool2dFloatModule_basic",
    "AvgPool2dIntModule_basic",
    "AvgPool2dStaticModule_basic",
    "AvgPool2dWithoutPadModule_basic",
    "BatchMlpLayerModule_basic",
    "BernoulliFloatModule_basic",
    "BernoulliModule_basic",
    "BernoulliOnesModule_basic",
    "BernoulliPModule_basic",
    "BernoulliTensorModule_basic",
    "BernoulliZerosModule_basic",
    "BincountMinlengthModule_basic",
    "BincountModule_basic",
    "BincountStaticSizeModule_basic",
    "BmmIntModule_basic",
    "BoolFloatConstantModule_basic",
    "BoolFloatFalseModule_basic",
    "BoolFloatTrueModule_basic",
    "BoolIntConstantModule_basic",
    "BoolIntFalseModule_basic",
    "BoolIntTrueModule_basic",
    "BroadcastDynamicDimModule_basic",
    "BroadcastToModule_basic",
    "BucketizeTensorFloatModule_basic",
    "BucketizeTensorModule_basic",
    "BucketizeTensorOutInt32RightModule_basic",
    "CeilFloatModule_basic",
    "ChunkListUnpackDynamic_Module_basic",
    "ChunkListUnpackUnevenDynamic_Module_basic",
    "ChunkListUnpackUneven_Module_basic",
    "CollapseAllDimensionsModule_basic",
    "CollapseFullDynamicModule_basic",
    "CollapsePartialDynamicModule_basic",
    "CollapseRank1DynamicModule_basic",
    "CollapseStaticModule_basic",
    "ConstantBoolParameterModule_basic",
    "ContainsIntList_False",
    "ContainsIntList_True",
    "Conv1dModule_basic",
    "Conv1dWithSamePaddingModule_basic",
    "Conv1dWithValidPaddingModule_basic",
    "Conv1dDepthwiseWithPaddingDilationStrideStaticModule_basic",
    "Conv1dGroupModule_basic",
    "Conv2dBiasNoPaddingModule_basic",
    "Conv2dModule_basic",
    "Conv2dNoPaddingModule_basic",
    "Conv2dQInt8Module_basic",
    "Conv2dQInt8Module_depthwise",
    "Conv2dQInt8Module_grouped",
    "Conv2dQInt8Module_not_depthwise",
    "Conv2dQInt8PerChannelModule_basic",
    "Conv2dQInt8PerChannelModule_depthwise",
    "Conv2dQInt8PerChannelModule_grouped",
    "Conv2dWithPaddingDilationStrideModule_basic",
    "Conv2dWithPaddingDilationStrideStaticModule_grouped",
    "Conv2dWithPaddingDilationStrideStaticModule_grouped_multiplier",
    "Conv2dWithPaddingModule_basic",
    "Conv2dWithSamePaddingModule_basic",
    "Conv2dWithValidPaddingModule_basic",
    "Conv3dModule_basic",
    "Conv3dWithSamePaddingModule_basic",
    "Conv3dWithValidPaddingModule_basic",
    "ConvTbcModule_basic",
    "ConvTranspose2DQInt8_basic",
    "Conv_Transpose2dModule_basic",
    "Convolution2DModule_basic",
    "Convolution2DStridedModule_basic",
    "ConvolutionBackwardModule2DPadded_basic",
    "ConvolutionBackwardModule2DStatic_basic",
    "ConvolutionBackwardModule2DStrided_basic",
    "ConvolutionBackwardModule2D_basic",
    "ConvolutionModule2DGroups_basic",
    "ConvolutionModule2DTransposeNonUnitOutputPadding_basic",
    "ConvolutionModule2DTransposeStridedStatic_basic",
    "ConvolutionModule2DTransposeStrided_basic",
    "ConvolutionModule2DTranspose_basic",
    "CopyModule_basic",
    "CopyWithDifferentDTypesAndSizesModule_basic",
    "CopyWithDifferentDTypesModule_basic",
    "CopyWithDifferentSizesModule_basic",
    "CosineSimilarityStaticBroadcastModule_basic",
    "CrossEntropyLossModule_basic",
    "CrossEntropyLossNoReductionModule_basic",
    "CumsumInputDtypeInt32Module_basic",
    "CumsumModule_basic",
    "CumsumStaticModule_basic",
    "CumsumStaticNegativeDimModule_basic",
    "CumprodModule_basic",
    "CumprodInputDtypeInt32Module_basic",
    "CumprodStaticModule_basic",
    "CumprodStaticNegativeDimModule_basic",
    "DeformConv2D_basic",
    "DeterminantModule_F32",
    "DeterminantBatchedModule_F32",
    "DeterminantDynamicModule_F32",
    "DeterminantModule_F32",
    "DiagonalModule_basic",
    "DiagonalModule_nonsquare",
    "DiagonalModule_transposed",
    "DiagonalModule_with_dims",
    "DiagonalModule_with_dims_and_offset",
    "DiagonalModule_with_negative_dims",
    "DiagonalModule_with_offset",
    "DiagonalWithStaticShapeModule_basic",
    "DivFloatModule_basic",
    "DivIntModule_basic",
    "DropoutTrainModule_basic",
    "DropoutTrainStaticShapeModule_basic",
    "ElementwiseAcosIntModule_basic",
    "ElementwiseAcosModule_basic",
    "ElementwiseAcoshIntModule_basic",
    "ElementwiseAcoshModule_basic",
    "ElementwiseAddScalarInt64Module_basic",
    "ElementwiseAddScalarIntModule_basic",
    "ElementwiseAsinIntModule_basic",
    "ElementwiseAsinModule_basic",
    "ElementwiseAsinhIntModule_basic",
    "ElementwiseAsinhModule_basic",
    "ElementwiseAtan2FloatIntModule_basic",
    "ElementwiseAtan2FloatIntStaticModule_basic",
    "ElementwiseAtan2TensorFloatModule_basic",
    "ElementwiseAtan2TensorFloatStaticModule_basic",
    "ElementwiseAtan2TensorIntModule_basic",
    "ElementwiseAtan2TensorIntStaticModule_basic",
    "ElementwiseAtanTensorFloatModule_basic",
    "ElementwiseAtanTensorIntModule_basic",
    "ElementwiseAtanhIntModule_basic",
    "ElementwiseAtanhModule_basic",
    "ElementwiseAtenDivIntScalarModule_basic",
    "ElementwiseAtenFloorDivideBroadcastModule_basic",
    "ElementwiseAtenFloorDivideScalarModule_basic",
    "ElementwiseAtenFloorDivideScalarNegativeModule_basic",
    "ElementwiseAtenFloorDivideTensorNegativeModule_basic",
    "ElementwiseAtenFloorDivideTensorPositiveModule_basic",
    "ElementwiseAtenIsneginfOpModule_basic",
    "ElementwiseAtenIsposinfOpModule_basic",
    "ElementwiseAtenLogicalAndOpPromoteBroadcastModule_basic",
    "ElementwiseAtenLogicalNotOpPromoteModule_basic",
    "ElementwiseAtenLogicalOrOpBrodcastModule_basic",
    "ElementwiseAtenLogicalOrOpDiffArgs1Module_basic",
    "ElementwiseAtenLogicalOrOpDiffArgs2Module_basic",
    "ElementwiseAtenLogicalOrOpDiffArgs3Module_basic",
    "ElementwiseAtenLogicalOrOpNegativeModule_basic",
    "ElementwiseAtenLogicalOrOpRandomFloatModule_basic",
    "ElementwiseAtenLogicalOrOpRandomModule_basic",
    "ElementwiseAtenLogicalXorOpPromoteBroadcastModule_basic",
    "ElementwiseBitwiseAndModule_basic",
    "ElementwiseBitwiseLeftShiftInt32Module_basic",
    "ElementwiseBitwiseLeftShiftInt64Module_basic",
    "ElementwiseBitwiseLeftShiftInt8Module_basic",
    "ElementwiseBitwiseNotInt32Module_basic",
    "ElementwiseBitwiseNotInt64Module_basic",
    "ElementwiseBitwiseOrModule_basic",
    "ElementwiseBitwiseOrStaticShapeModule_basic",
    "ElementwiseBitwiseRightShiftInt32Module_basic",
    "ElementwiseBitwiseRightShiftInt64Module_basic",
    "ElementwiseBitwiseRightShiftInt8Module_basic",
    "ElementwiseBitwiseXorModule_basic",
    "ElementwiseBitwiseXorStaticShapeModule_basic",
    "ElementwiseCosIntModule_basic",
    "ElementwiseCoshIntModule_basic",
    "ElementwiseCoshModule_basic",
    "ElementwiseDequantizePerChannelModule_basic",
    "ElementwiseDequantizePerTensorModule_basic",
    "ElementwiseDivScalarRoundingModeTruncModule_basic",
    "ElementwiseDivScalarRoundingModeTruncStaticModule_basic",
    "ElementwiseDivTensorFloatModule_basic",
    "ElementwiseDivTensorIntegerModule_basic",
    "ElementwiseDivTensorRoundingModeFloorIntStaticModule_basic",
    "ElementwiseDivTensorRoundingModeFloorModule_basic",
    "ElementwiseDivTensorRoundingModeTruncIntStaticModule_basic",
    "ElementwiseDivTensorRoundingModeTruncModule_basic",
    "ElementwiseDivTensorRoundingModeTruncStaticModule_basic",
    "ElementwiseDivTensorUnsignedIntegerModule_basic",
    "ElementwiseEluNonDefaultModule_basic",
    "ElementwiseEqBoolScalarModule_basic",
    "ElementwiseEqDiffWidthScalarModule_basic",
    "ElementwiseErfIntModule_basic",
    "ElementwiseExpIntModule_basic",
    "ElementwiseExpm1IntModule_basic",
    "ElementwiseExpm1Module_basic",
    "ElementwiseFlattenBroadcastModule_basic",
    "ElementwiseFloatTensorGtIntTensorModule_basic",
    "ElementwiseFmodTensor_Int_Float_basic",
    "ElementwiseFmodTensor_Int_basic",
    "ElementwiseGeMixedIntScalarModule_basic",
    "ElementwiseGtMixed2ScalarModule_basic",
    "ElementwiseIntTensorLtFloatScalarModule_basic",
    "ElementwiseIsinfModule_basic",
    "ElementwiseLeMixedIntScalarModule_basic",
    "ElementwiseLog10IntModule_basic",
    "ElementwiseLog2IntModule_basic",
    "ElementwiseLogIntModule_basic",
    "ElementwiseLtDiffWidthScalarModule_basic",
    "ElementwiseMulScalarModule_basic",
    "ElementwiseMulTensorComplexDiffModule_basic",
    "ElementwiseMulTensorComplexModule_basic",
    "ElementwiseMulTensorFloatModule_basic",
    "ElementwiseMulTensorIntModule_basic",
    "ElementwiseOrTensorModule_basic",
    "ElementwiseOrTensorStaticShapeModule_basic",
    "ElementwiseQuantizePerTensorModule_basic",
    "ElementwiseQuantizePerTensorUIntModule_basic",
    "ElementwiseReciprocalIntModule_basic",
    "ElementwiseRemainderScalarModule_Bool_basic",
    "ElementwiseRemainderScalarModule_Int_Float_basic",
    "ElementwiseRemainderTensorModule_Int_Float_basic",
    "ElementwiseRemainderTensorModule_Int_basic",
    "ElementwiseRsqrtIntModule_basic",
    "ElementwiseSgnModule_basic",
    "ElementwiseSigmoidIntModule_basic",
    "ElementwiseSinIntModule_basic",
    "ElementwiseSinhIntModule_basic",
    "ElementwiseSinhModule_basic",
    "ElementwiseSpecialExpm1IntModule_basic",
    "ElementwiseSpecialExpm1Module_basic",
    "ElementwiseSqrtIntModule_basic",
    "ElementwiseSubScalarIntModule_basic",
    "ElementwiseTanIntModule_basic",
    "ElementwiseTernaryModule_basic",
    "ElementwiseToDtypeF32ToI64Module_basic",
    "ElementwiseToDtypeI64ToI8Module_basic",
    "ElementwiseToDtypeI64ToUI8Module_basic",
    "ElementwiseTruncIntModule_basic",
    "ElementwiseTruncModule_basic",
    "ElementwiseUnaryIntModule_basic",
    "ElementwiseUnsqueezeNegDimsModule_basic",
    "ElementwiseWhereScalarOtherModule_basic",
    "ElementwiseWhereScalarSelfModule_basic",
    "ElementwiseWhereSelfModule_basic",
    "EmbeddingModule1DIndices_basic",
    "EmbeddingModuleF16_basic",
    "EmbeddingModuleI32_basic",
    "EmbeddingModuleI64_basic",
    "EmptyLikeMemoryFormatModule_basic",
    "EmptyLikeModule_defaultDtype",
    "EmptyLikeModule_falsePinMemory",
    "EmptyLikeModule_float",
    "EmptyLikeModule_int",
    "EmptyStridedModule_basic",
    "EmptyStridedSizeIntStrideModule_basic",
    "EqIntModule_basic",
    "ExpandAsFloatModule_basic",
    "ExpandAsIntModule_basic",
    "ExpandModule_basic",
    "ExponentialModule_basic",
    "FakeQuantizePerTensorAffineDynamicShapeModule_basic",
    "FakeQuantizePerTensorAffineModule_basic",
    "FakeQuantizePerTensorAffineRoundToEvenModule_basic",
    "Fill_TensorFloat32WithFloat32_basic",
    "Fill_TensorFloat32WithFloat64_basic",
    "Fill_TensorFloat32WithInt64_basic",
    "Fill_TensorFloat64WithFloat32_basic",
    "Fill_TensorFloat64WithFloat64_basic",
    "Fill_TensorFloat64WithInt64_basic",
    "FlattenDynamicModuleCollapseAll_basic",
    "FlattenDynamicModule_basic",
    "FlattenRank0Module_basic",
    "FlipModuleStaticShape_basic",
    "FlipModule_basic",
    "FlipNegativeIndexModule_basic",
    "FloatImplicitModule_basic",
    "FullLikeModuleDefaultDtype_basic",
    "FullLikeModuleFalsePinMemory_basic",
    "FullLikeModuleFloat2D_basic",
    "FullLikeModuleFloat3D_basic",
    "FullLikeModuleInt2D_basic",
    "FullLikeModuleInt3D_basic",
    "Gather2DInputModdule_basic",
    "GatherModule_basic",
    "GatherNegativeDimModule_basic",
    "GatherRandomIndexModule_basic",
    "GeFloatIntModule_basic",
    "GeFloatModule_basic",
    "GeIntModule_basic",
    "GeluBackwardModule_basic",
    "GridSamplerBasic1_basic",
    "GridSamplerBasic2_basic",
    "GridSamplerBasic3_basic",
    "GridSamplerBasic4_basic",
    "GtFloatIntModule_basic",
    "GtIntModule_basic",
    "HBC_basic",
    "HardtanhBackward_basic",
    "IndexPut1DFloatAccumulateModule_basic",
    "IndexPut1DFloatNonAccumulateModule_basic",
    "IndexPut1DIntAccumulateModule_basic",
    "IndexPut1DIntNonAccumulateModule_basic",
    "IndexPut2DFloatAccumulateModule_basic",
    "IndexPut2DFloatNonAccumulateModule_basic",
    "IndexPut2DIntAccumulateModule_basic",
    "IndexPut2DIntNonAccumulateModule_basic",
    "IndexPut3DFloatAccumulateModule_basic",
    "IndexPut3DFloatNonAccumulateModule_basic",
    "IndexPut3DIntAccumulateModule_basic",
    "IndexPut3DIntNonAccumulateModule_basic",
    "IndexPutHackedTwin1DFloatAccumulateModule_basic",
    "IndexPutHackedTwin1DFloatNonAccumulateModule_basic",
    "IndexPutHackedTwin1DIntAccumulateModule_basic",
    "IndexPutHackedTwin1DIntNonAccumulateModule_basic",
    "IndexPutHackedTwin2DFloatAccumulateModule_basic",
    "IndexPutHackedTwin2DFloatNonAccumulateModule_basic",
    "IndexPutHackedTwin2DIntAccumulateModule_basic",
    "IndexPutHackedTwin2DIntNonAccumulateModule_basic",
    "IndexPutHackedTwin3DFloatAccumulateModule_basic",
    "IndexPutHackedTwin3DFloatNonAccumulateModule_basic",
    "IndexPutHackedTwin3DIntAccumulateModule_basic",
    "IndexPutHackedTwin3DIntNonAccumulateModule_basic",
    "IndexPutImpl1DFloatAccumulateModule_basic",
    "IndexPutImpl1DFloatNonAccumulateModule_basic",
    "IndexPutImpl1DIntAccumulateModule_basic",
    "IndexPutImpl1DIntNonAccumulateModule_basic",
    "IndexPutImpl2DFloatAccumulateModule_basic",
    "IndexPutImpl2DFloatNonAccumulateModule_basic",
    "IndexPutImpl2DImplicitModule_basic",
    "IndexPutImpl2DIndexModule_basic",
    "IndexPutImpl2DNoneIndexStaticModule_basic",
    "IndexPutImpl3DFloatAccumulateModule_basic",
    "IndexPutImpl3DFloatNonAccumulateModule_basic",
    "IndexPutImplIndexWithNoneModule_basic",
    "IndexSelectDynamicIndexSizeModule_basic",
    "IndexSelectDynamicInputSizeModule_basic",
    "IndexSelectDynamicModulebasic",
    "IndexTensorDyanmicInputContiguousWithNoneModule_basic",
    "IndexTensorDyanmicInputNonContiguousWithNoneModule_basic",
    "IndexTensorHackedTwinModule3dInput_basic",
    "IndexTensorHackedTwinModule_basic",
    "IndexTensorHackedTwinMultiInputNonContiguousMultipleStaticDims_basic",
    "IndexTensorModule3dInput_basic",
    "IndexTensorModule_basic",
    "IndexTensorMultiInputContiguousCenter_basic",
    "IndexTensorMultiInputContiguousOneDimDynamic_basic",
    "IndexTensorMultiInputNonContiguousDynamic_basic",
    "IndexTensorMultiInputNonContiguousMultipleStaticDims_basic",
    "IndexTensorMultiInputNonContiguousOneDimDynamic_basic",
    "IndexTensorMultiInputNonContiguous_basic",
    "IndexTensorMultiInputOneDim_basic",
    "IndexTensorMultiInputThreeIndexers_basic",
    "IndexTensorMultiInput_basic",
    "IndexTensorSelectDimModule_basic",
    "InterpolateDynamicModule_sizes_bilinear",
    "InterpolateDynamicModule_sizes_nearest",
    "InterpolateDynamicModule_scales_recompute_bilinear",
    "IntFloatModule_basic",
    "IntImplicitModule_basic",
    "IouOfModule_basic",
    "IsFloatingPointFloat_True",
    "IsFloatingPointInt_False",
    "IscloseStaticModuleTrue_basic",
    "IscloseStaticModule_basic",
    "LeakyReluBackwardModule_basic",
    "LeakyReluBackwardStaticModule_basic",
    "LenStrModule_basic",
    "LiftFreshCopyModule_basic",
    "LinalgNormKeepDimComplexModule_basic",
    "LinalgNormModule_basic",
    "LinalgVectorNormComplexModule_basic",
    "LogSoftmaxBackwardModule_basic",
    "LogSoftmaxIntModule_basic",
    "MaskedFillTensorFloatValueModule_basic",
    "MatmulBroadcastBatchDim_basic",
    "MatmulSingleDynamicBatchDim_basic",
    "Matmul_2d",
    "Matmul_4d",
    "Matmul_matvec",
    "Matmul_vecmat",
    "MaxPool1dCeilModeTrueModule_basic",
    "MaxPool1dModule_basic",
    "MaxPool2dCeilModeTrueModule_basic",
    "MaxPool2dModule_basic",
    "MaxPool2dWithIndicesAllNegativeValuesModule_basic",
    "MaxPool2dWithIndicesAllOnesModule_basic",
    "MaxPool2dWithIndicesBackwardDynamic3DModule_basic",
    "MaxPool2dWithIndicesBackwardDynamic4DModule_basic",
    "MaxPool2dWithIndicesBackwardStatic3DModule_basic",
    "MaxPool2dWithIndicesBackwardStatic4DModule_basic",
    "MaxPool2dWithIndicesCeilModeTrueModule_basic",
    "MaxPool2dWithIndicesFullSizeKernelModule_basic",
    "MaxPool2dWithIndicesModule_basic",
    "MaxPool2dWithIndicesNonDefaultDilationModule_basic",
    "MaxPool2dWithIndicesNonDefaultPaddingModule_basic",
    "MaxPool2dWithIndicesNonDefaultParamsModule_basic",
    "MaxPool2dWithIndicesNonDefaultStrideModule_basic",
    "MaxPool2dWithIndicesStaticModule_basic",
    "MaxPool3dCeilModeTrueModule_basic",
    "MaxPool3dEmptyStrideStaticModule_basic",
    "MaxPool3dLargeDatadModule_basic",
    "MaxPool3dModuleRandomSimple_basic",
    "MaxPool3dModule_basic",
    "MaxPool3dStaticCeilModeTrueModule_basic",
    "MaxPool3dStaticModule_basic",
    "MaxPool3dWithIndicesAllNegativeValuesModule_basic",
    "MaxPool3dWithIndicesAllOnesModule_basic",
    "MaxPool3dWithIndicesCeilModeTrueModule_basic",
    "MaxPool3dWithIndicesFullSizeKernelModule_basic",
    "MaxPool3dWithIndicesModule_basic",
    "MaxPool3dWithIndicesNonDefaultDilationModule_basic",
    "MaxPool3dWithIndicesNonDefaultPaddingModule_basic",
    "MaxPool3dWithIndicesNonDefaultParamsModule_basic",
    "MaxPool3dWithIndicesNonDefaultStrideModule_basic",
    "MaxPool3dWithIndicesStaticModule_basic",
    "MeanDimAllReduceKeepdimModule_basic",
    "MeanDimAllReduceModule_basic",
    "MeanDimDtypeModule_basic",
    "MeanDimEmptyDimModule_basic",
    "MeanDimKeepdimModule_basic",
    "MeanDimModule_basic",
    "MeanDimNegativeModule_basic",
    "MeanDimNoneDimModule_basic",
    "MeanDtypeModule_basic",
    "MeanDynamicSizesModule_basic",
    "Mlp1LayerModule_basic",
    "Mlp2LayerModuleNoBias_basic",
    "Mlp2LayerModule_basic",
    "MmModule_basic",
    "MmModule_chained",
    "MmTanhModule_basic",
    "MobilenetV3Module_basic",
    "MoveDimIntNegativeIndexModule_basic",
    "MseLossMeanReductionModule_basic",
    "MseLossSumReductionWithDifferentElemTypeModule_basic",
    "MulFloatModule_basic",
    "MulIntModule_basic",
    "Mv_basic",
    "NarrowHorizontalTest2_basic",
    "NarrowHorizontalTest_basic",
    "NarrowTensorHorizontalModule_basic",
    "NarrowTensorVerticalModule_basic",
    "NarrowVerticalTest2_basic",
    "NarrowVerticalTest_basic",
    "NativeBatchNorm1DModule_basic",
    "NativeBatchNorm2DModule_basic",
    "NativeBatchNorm3DModule_basic",
    "NativeBatchNormNoneWeightModule_basic",
    "NativeDropoutEvalFloatModule_basic",
    "NativeDropoutTrainModule_basic",
    "NativeDropoutTrainStaticShapeModule_basic",
    "NativeGroupNormBackwardModule_basic",
    "NativeGroupNormModule_basic",
    "NativeLayerNormDynamicModule_basic",
    "NeFloatIntModule_basic",
    "NeIntModule_basic",
    "NewEmptyStridedModuleDefaultDtype_basic",
    "NllLossModuleBackward1DMeanWeight_basic",
    "NllLossModuleBackward1DMean_basic",
    "NllLossModuleBackward1DSumWeight_basic",
    "NllLossModuleBackward1DSum_basic",
    "NllLossModuleBackward1DWeight_basic",
    "NllLossModuleBackward1D_basic",
    "NllLossModuleBackwardMeanWeight_basic",
    "NllLossModuleBackwardMean_basic",
    "NllLossModuleBackwardSumWeight_basic",
    "NllLossModuleBackwardSum_basic",
    "NllLossModuleBackwardWeight_basic",
    "NllLossModuleBackward_basic",
    "NllLossModuleBackward_ignore_index",
    "NllLossModule_1D_basic",
    "NllLossModule_basic",
    "NllLossModule_ignore_index_out_of_bounds_basic",
    "NllLossModule_mean_basic",
    "NllLossModule_sum_basic",
    "NormScalarComplexModule_basic",
    "NormScalarModule_basic",
    "NormScalarOptDimKeepDimComplexModule_basic",
    "NormScalarOptDimKeepDimModule_basic",
    "NormScalarOptDimModule_basic",
    "NormalFunctionalModule_basic",
    "NumToTensorFloatModule_basic",
    "NumToTensorIntModule_basic",
    "NumelModule_basic",
    "NumelZeroRankModule_basic",
    "OneHotModule_basic",
    "OnesLikeModule_defaultDtype",
    "OnesLikeModule_falsePinMemory",
    "OnesLikeModule_float",
    "OnesLikeModule_int",
    "PermuteNegativeIndexModule_basic",
    "PixelShuffleModuleFullDynamic_basic",
    "PixelShuffleModuleSpatiallyDynamic_basic",
    "PixelShuffleModuleSpatiallyStatic_basic",
    "PixelShuffleModuleStaticRank3Int64_basic",
    "PixelShuffleModuleStaticRank4Float32_basic",
    "PrimMaxIntModule_basic",
    "PrimMinIntDynamicModule_basic",
    "PrimMinIntModule_basic",
    "PrimsConvertElementTypeModule_basic",
    "PrimsIotaModule_basic",
    "PrimsSqueezeEmptyDimensionsModule_basic",
    "PrimsSqueezeModule_basic",
    "PrimsViewOfModule_basic",
    "PrimsViewOfZeroRankModule_basic",
    "QuantizedBatchedInputSingleLayer_basic",
    "QuantizedMLP_basic",
    "QuantizedNoLayer_basic",
    "QuantizedReluInt32_basic",
    "QuantizedReluInt8_basic",
    "QuantizedReluUint8_basic",
    "QuantizedSingleLayer_basic",
    "RandIntDtypeModule_basic",
    "RandIntLowDtypeModule_basic",
    "RandIntModule_basic",
    "RandIntPinMemoryModule_basic",
    "RandLikeDtypeModule_basic",
    "RandLikeModule_basic",
    "RandModule_basic",
    "RandnDtypeDeviceModule_basic",
    "RandnGeneratorF64Module_basic",
    "RandnGeneratorModule_basic",
    "RandnLikeDtypeModule_basic",
    "RandnLikeModule_basic",
    "RandnModule_basic",
    "ReduceAllBoolModule_basic",
    "ReduceAllDimBool_basic",
    "ReduceAllDimEmpty_basic",
    "ReduceAllDimFloat_basic",
    "ReduceAllDimInt_basic",
    "ReduceAllFloatModule_basic",
    "ReduceAllIntModule_basic",
    "ReduceAmaxKeepDim_basic",
    "ReduceAmaxMultiDim_basic",
    "ReduceAmaxOutOfOrderDim_basic",
    "ReduceAmaxSingleDim_basic",
    "ReduceAnyBoolModule_basic",
    "ReduceAnyFloatModule_basic",
    "ReduceAnyIntModule_basic",
    "ReduceFrobeniusNormComplexModule_basic",
    "ReduceL1NormComplexModule_basic",
    "ReduceL1NormModule_basic",
    "ReduceL1NormWithDTypeModule_basic",
    "ReduceL2NormComplexModule_basic",
    "ReduceL2NormModule_basic",
    "ReduceL3NormAllDimsModule_basic",
    "ReduceL3NormKeepDimComplexModule_basic",
    "ReduceL3NormKeepDimModule_basic",
    "ReduceLN3NormModule_basic",
    "ReduceMaxAllDims_basic",
    "ReduceMaxAlongDimNegative_basic",
    "ReduceMaxAlongDimSignedInt_basic",
    "ReduceMaxAlongDimUnsignedInt_basic",
    "ReduceMaxAlongDim_basic",
    "ReduceMaxFloatModule_basic",
    "ReduceMaxKeepDimReturnBoth_basic",
    "ReduceMaxKeepDim_basic",
    "ReduceMaxNegativeDim_basic",
    "ReduceMaxSignedIntModule_basic",
    "ReduceMaxUnsignedIntModule_basic",
    "ReduceMinAlongDimNegative_basic",
    "ReduceMinAlongDimSignedInt_basic",
    "ReduceMinAlongDimUnsignedInt_basic",
    "ReduceMinAlongDim_basic",
    "ReduceMinFloatModule_basic",
    "ReduceMinKeepDimReturnBoth_basic",
    "ReduceMinKeepDim_basic",
    "ReduceMinSignedIntModule_basic",
    "ReduceMinUnsignedIntModule_basic",
    "ReduceProdDimIntFloatModule_basic",
    "ReduceProdDtypeFloatModule_basic",
    "ReduceProdDtypeIntModule_basic",
    "ReduceProdElementTypeBoolModule_basic",
    "ReduceProdFloatModule_basic",
    "ReduceProdSignedIntModule_basic",
    "ReduceProdUnsignedIntModule_basic",
    "ReduceSumDimIntListDtypeFloatModule_basic",
    "ReduceSumDimIntListDtypeIntModule_basic",
    "ReduceSumDimIntListElementTypeBoolModule_basic",
    "ReduceSumDtypeFloatModule_basic",
    "ReduceSumDtypeIntModule_basic",
    "ReduceSumElementTypeBoolModule_basic",
    "ResNet18Module_basic",
    "ReshapeAliasCollapseModule_basic",
    "ReshapeAliasExpandModule_basic",
    "ReshapeCollapseModule_basic",
    "ReshapeDynamicModule_basic",
    "ReshapeExpandModule_basic",
    "RollModule_basic",
    "RsubIntModule_noalpha_basic",
    "ScalarConstantTupleModule_basic",
    "ScalarImplicitFloatModule_basic",
    "ScalarImplicitIntModule_basic",
    # REMOVE WHEN ENABLE_GQA IS ADDED
    "ScaledDotProductAttentionBoolMaskModule_basic",
    "ScaledDotProductAttentionSameCausalModule_basic",
    "ScaledDotProductAttentionSameDynamicModule_basic",
    "ScatterReduceFloatMaxModule",
    "ScatterReduceFloatMaxModuleIncludeSelf",
    "ScatterReduceFloatMeanModule",
    "ScatterReduceFloatMeanModuleIncludeSelf",
    "ScatterReduceFloatMinModule",
    "ScatterReduceFloatMinModuleIncludeSelf",
    "ScatterReduceFloatProdModule",
    "ScatterReduceFloatProdModuleIncludeSelf",
    "ScatterReduceFloatSumModule",
    "ScatterReduceFloatSumModuleIncludeSelf",
    "ScatterReduceIntMaxModule",
    "ScatterReduceIntMaxModuleIncludeSelf",
    "ScatterReduceIntMeanModule",
    "ScatterReduceIntMeanModuleIncludeSelf",
    "ScatterReduceIntMinModule",
    "ScatterReduceIntMinModuleIncludeSelf",
    "ScatterReduceIntProdModule",
    "ScatterReduceIntProdModuleIncludeSelf",
    "ScatterReduceIntSumModule",
    "ScatterReduceIntSumModuleIncludeSelf",
    "ScatterSrcModule_basic",
    "ScatterSrcStaticModule_basic",
    "ScatterValueFloatModule_basic",
    "ScatterValueIntModule_basic",
    "SelectIntModule_basic",
    "SelectScattertModule_basic",
    "SelectScattertStaticModule_basic",
    "SignAndLogarithmOfDeterminantModule_F32",
    "SignAndLogarithmOfDeterminantBatchedModule_F32",
    "SignAndLogarithmOfDeterminantDynamicModule_F32",
    "SliceCopyEndGreaterThanDimSize_Module_basic",
    "SliceCopyNegative_Module_basic",
    "SliceCopyNonZeroDim_Module_basic",
    "SliceCopy_Module_basic",
    "SliceEndSleStartModule_basic",
    "SliceModule_basic",
    "SliceStaticComplexInputModule_basic",
    "SliceNegIdxModule_basic",
    "SliceOutOfLowerBoundEndIndexModule_basic",
    "SliceOutOfLowerBoundStartIndexModule_basic",
    "SliceOutOfUpperBoundIndexModule_basic",
    "SliceScatterModule_basic",
    "SliceScatterNegativeDimModule_basic",
    "SliceScatterNegativeEndModule_basic",
    "SliceScatterStaticModule_basic",
    "SliceScatterStepVariationModule_basic",
    "SliceScatterZeroDimModule_basic",
    "SliceSingleIdxModule_basic",
    "SliceSizeTwoStepModule_basic",
    "SliceStartEqEndModule_basic",
    "SoftmaxBackwardModule_basic",
    "SoftmaxIntArgTypeF64Module_basic",
    "SoftmaxIntModule_basic",
    "SoftmaxIntNegDimModule_basic",
    "SoftmaxIntNonNoneDtypeModule_basic",
    "SortIntListReverse_basic",
    "SortIntList_basic",
    "SortTensorDescending_basic",
    "SortTensorInteger_basic",
    "SortTensorNegativeDimension_basic",
    "SortTensorSpecificDimension_basic",
    "SortTensor_basic",
    "SplitDimDynamicModule_basic",
    "SplitDimStaticModule_basic",
    "SplitWithSizes_Module_basic",
    "SqrtIntConstantModule_basic",
    "SqrtIntModule_basic",
    "SqueezeDimModule_dynamic",
    "SqueezeDimModule_negDim",
    "StdBiasedModule_basic",
    "StdCorrectionAllDimReduceModule_basic",
    "StdCorrectionEmptyDimModule_basic",
    "StdCorrectionKeepDimModule_basic",
    "StdCorrectionLargeInputModule_basic",
    "StdCorrectionModule_basic",
    "StdCorrectionNoneModule_basic",
    "StdCorrectionSingleDimReduceModule_basic",
    "StdDimBiasedModule_basic",
    "StdDimEmptyDimModule_basic",
    "StdDimKeepDimFalseModule_basic",
    "StdDimKeepDimTrueModule_basic",
    "StdDimNoneDimModule_basic",
    "StdUnbiasedModule_basic",
    "SubFloatModule_basic",
    "SubIntModule_basic",
    "TanhBackward_basic",
    "TensorToBoolZeroRank_basic",
    "TensorToBool_basic",
    "TensorToFloatZeroRank_basic",
    "TensorToFloat_basic",
    "TensorToIntZeroRank_basic",
    "TensorToInt_basic",
    "TensorsConcatModule_basic",
    "TensorsConcatNegativeDimModule_basic",
    "TensorsConcatPromoteDTypeModule_basic",
    "TensorsStackModule_basic",
    "TensorsStackNegativeDimModule_basic",
    "TensorsStackPromoteDTypeModule_basic",
    "TensorsStackSingleElementListModule_basic",
    "TestMultipleTensorAndPrimitiveTypesReturn_basic",
    "Threshold1dFloatModule_basic",
    "Threshold1dIntI32Module_basic",
    "Threshold1dIntModule_basic",
    "Threshold2dFloatModule_basic",
    "Threshold2dIntModule_basic",
    "Threshold3dFloatModule_basic",
    "Threshold3dIntModule_basic",
    "ThresholdBackward1dFloatModule_basic",
    "ThresholdBackward1dIntModule_basic",
    "ThresholdBackward1dMixedModule_basic",
    "ThresholdBackward2dFloatModule_basic",
    "ThresholdBackward2dIntModule_basic",
    "ThresholdBackward2dMixedModule_basic",
    "ThresholdBackward3dFloatModule_basic",
    "ThresholdBackward3dIntModule_basic",
    "ThresholdBackward3dMixedModule_basic",
    "TileBigDimsSizeModule_basic",
    "TileSmallDimsSizeModule_basic",
    "ToCopyBoolDTypeStaticModule_basic",
    "ToCopyModule_basic",
    "ToCopyWithDTypeFalsePinMemoryModule_basic",
    "ToCopyWithDTypeModule_basic",
    "ToDtypeLayoutCPUModule_basic",
    "ToDtypeLayoutNoneModule_basic",
    "ToDtypeLayoutStridedModule_basic",
    "TorchPrimLoopForLikeModule_basic",
    "TorchPrimLoopWhileLikeModule_basic",
    "TraceModule_basic",
    "TraceModule_empty",
    "TraceModule_nonsquare",
    "TraceSignedIntModule_basic",
    "TraceUnsignedIntModule_basic",
    "TraceUnsignedIntModule_empty",
    "TupleModule_basic",
    "TypeAsDifferentModule_basic",
    "TypeConversionF32ToF64Module_basic",
    "TypeConversionF64ToF32Module_basic",
    "TypeConversionI1ToF32Module_basic",
    "TypeConversionI1ToF64Module_basic",
    "TypeConversionI1ToI32Module_basic",
    "TypeConversionI1ToI64Module_basic",
    "TypeConversionI32ToI64Module_basic",
    "TypeConversionI64ToI32Module_basic",
    "TypePromotionDifferentCategoryModule_basic",
    "TypePromotionSameCategoryDifferentWidthModule_basic",
    "TypePromotionZeroRankHigherCategoryModule_basic",
    "UniformModule_basic",
    "UniformNoCorrelationModule_basic",
    "UniformStaticShapeModule_basic",
    "UnsafeIndexPutHackedTwin1DFloatNonAccumulateModule_basic",
    "UnsafeView1DFoldModule_basic",
    "UnsafeViewCollapseDynamicWithAtenSizeIntModule_basic",
    "UnsafeViewCollapseModule_basic",
    "UnsafeViewDynamicExpandModule_basic",
    "UnsafeViewDynamicExpandWithAtenSizeIntModule_basic",
    "UnsafeViewExpandModule_basic",
    "UpSampleNearest2dBackwardScalesNone_basic",
    "UpSampleNearest2dBackward_basic",
    "UpSampleNearest2dDynamicFactor_basic",
    "UpSampleNearest2dDynamicSize_basic",
    "UpSampleNearest2dStaticFactor_basic",
    "UpSampleNearest2dStaticSize_basic",
    "UpSampleNearest2d_basic",
    "VarBiasedModule_basic",
    "VarCorrectionAllDimReduceModule_basic",
    "VarCorrectionEmptyDimModule_basic",
    "VarCorrectionKeepDimModule_basic",
    "VarCorrectionLargeInputModule_basic",
    "VarCorrectionModule_basic",
    "VarCorrectionNoneModule_basic",
    "VarCorrectionSingleDimReduceModule_basic",
    "VarDimAllDimReduceModule_basic",
    "VarDimBiasedModule_basic",
    "VarDimEmptyDimModule_basic",
    "VarDimModule_basic",
    "VarDimMultiDimModule_basic",
    "VarDimNegativeModule_basic",
    "VarDimNoneDimModule_basic",
    "VarDimSingleDimModule_basic",
    "VarDimUnbiasedModule_basic",
    "VarMeanBiasedModule_basic",
    "VarMeanCorrectionModule_basic",
    "VarMeanCorrectionNoneModule_basic",
    "VarMeanDimBiasedModule_basic",
    "VarMeanDimModule_basic",
    "VarMeanUnbiasedModule_basic",
    "VarUnbiasedModule_basic",
    "View1DFoldModule_basic",
    "ViewCollapseDynamicWithAtenSizeIntModule_basic",
    "ViewCollapseModule_basic",
    "ViewDynamicExpandCollapseModule_basic",
    "ViewDynamicExpandCollapseWithAtenIntModule_basic",
    "ViewDynamicExpandCollapseWithParallelUnknownDimModule_basic",
    "ViewDynamicExpandModule_basic",
    "ViewDynamicExpandWithAtenSizeIntModule_basic",
    "ViewExpandDynamicDimModule_basic",
    "ViewFlattenAndExpandModule_basic",
    "ViewNoChange1dModule_basic",
    "ViewNoChange2dModule_basic",
    "ViewNoChange3dModule_basic",
    "ViewSizeDimFollowedByCollapsedOnesModule_basic",
    "ViewSizeDimFollowedByExpandedOnesModule_basic",
    "ViewSizeDimLedAndFollowedByCollapsedOnesModule_basic",
    "ViewSizeDimLedAndFollowedByExpandedOnesModule_basic",
    "ViewSizeDimLedByCollapsedOnesModule_basic",
    "ViewSizeDimLedByExpandedOnesModule_basic",
    "ViewSizeFromOtherTensor_basic",
    "ZeroFloat32Module_basic",
    "ZeroInt32Module_basic",
    "ZeroInt64Module_basic",
    "ZerosLikeModule_defaultDtype",
    "ZerosLikeModule_falsePinMemory",
    "ZerosLikeModule_float",
    "ZerosLikeModule_int",
    "_Convolution2DAllFalseModule_basic",
    "_Convolution2DBenchmarkModule_basic",
    "_Convolution2DCudnnModule_basic",
    "_Convolution2DDeterministicModule_basic",
    "_Convolution2DTF32Module_basic",
    "_ConvolutionDeprecated2DAllFalseModule_basic",
    "_ConvolutionDeprecated2DBenchmarkModule_basic",
    "_ConvolutionDeprecated2DCudnnModule_basic",
    "_ConvolutionDeprecated2DDeterministicModule_basic",
    "_LogSoftmaxModule_basic",
    "_SoftmaxModule_basic",
}

if torch_version_for_comparison() > version.parse("2.5.1"):
    ONNX_TOSA_XFAIL_SET = ONNX_TOSA_XFAIL_SET | {
        # error: 'memref.cast' op operand type 'memref<2x6x4x3xf32>' and result type 'memref<2x6x5x3xf32>' are cast incompatible
        # torch.onnx.export produces onnx.MaxPool op with incorrect output shape of 2x6x5x3 instead of 2x6x4x3
        "MaxPool2dStaticCeilModeTrueReduceOutputModule_basic",
    }
