#include "zmq_u.hpp"

#ifdef _WIN32
#pragma comment(lib, "ws2_32.lib")
#pragma comment(lib, "iphlpapi.lib")
#endif

//========= begin of address.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "address.hpp"
// ans ignore: #include "ctx.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "tcp_address.hpp"
// ans ignore: #include "udp_address.hpp"
// ans ignore: #include "ipc_address.hpp"
// ans ignore: #include "tipc_address.hpp"

#if defined ZMQ_HAVE_VMCI
// ans ignore: #include "vmci_address.hpp"
#endif

#include <sstream>
#include <string>

zmq::address_t::address_t(const std::string &protocol_, const std::string &address_, ctx_t *parent_)
    : protocol(protocol_), address(address_), parent(parent_)
{
    resolved.dummy = NULL;
}

zmq::address_t::~address_t()
{
    if (protocol == protocol_name::tcp)
    {
        LIBZMQ_DELETE(resolved.tcp_addr);
    }
    else if (protocol == protocol_name::udp)
    {
        LIBZMQ_DELETE(resolved.udp_addr);
    }
#if !defined ZMQ_HAVE_WINDOWS && !defined ZMQ_HAVE_OPENVMS && !defined ZMQ_HAVE_VXWORKS
    else if (protocol == protocol_name::ipc)
    {
        LIBZMQ_DELETE(resolved.ipc_addr);
    }
#endif
#if defined ZMQ_HAVE_TIPC
    else if (protocol == protocol_name::tipc)
    {
        LIBZMQ_DELETE(resolved.tipc_addr);
    }
#endif
#if defined ZMQ_HAVE_VMCI
    else if (protocol == protocol_name::vmci)
    {
        LIBZMQ_DELETE(resolved.vmci_addr);
    }
#endif
}

int zmq::address_t::to_string(std::string &addr_) const
{
    if (protocol == protocol_name::tcp && resolved.tcp_addr)
        return resolved.tcp_addr->to_string(addr_);
    if (protocol == protocol_name::udp && resolved.udp_addr)
        return resolved.udp_addr->to_string(addr_);
#if !defined ZMQ_HAVE_WINDOWS && !defined ZMQ_HAVE_OPENVMS && !defined ZMQ_HAVE_VXWORKS
    if (protocol == protocol_name::ipc && resolved.ipc_addr)
        return resolved.ipc_addr->to_string(addr_);
#endif
#if defined ZMQ_HAVE_TIPC
    if (protocol == protocol_name::tipc && resolved.tipc_addr)
        return resolved.tipc_addr->to_string(addr_);
#endif
#if defined ZMQ_HAVE_VMCI
    if (protocol == protocol_name::vmci && resolved.vmci_addr)
        return resolved.vmci_addr->to_string(addr_);
#endif

    if (!protocol.empty() && !address.empty())
    {
        std::stringstream s;
        s << protocol << "://" << address;
        addr_ = s.str();
        return 0;
    }
    addr_.clear();
    return -1;
}

zmq::zmq_socklen_t zmq::get_socket_address(fd_t fd_, socket_end_t socket_end_, sockaddr_storage *ss_)
{
    zmq_socklen_t sl = static_cast<zmq_socklen_t>(sizeof(*ss_));

    const int rc = socket_end_ == socket_end_local ? getsockname(fd_, reinterpret_cast<struct sockaddr *>(ss_), &sl)
                                                   : getpeername(fd_, reinterpret_cast<struct sockaddr *>(ss_), &sl);

    return rc != 0 ? 0 : sl;
}

//========= end of address.cpp ============

//========= begin of client.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "client.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "msg.hpp"

zmq::client_t::client_t(class ctx_t *parent_, uint32_t tid_, int sid_) : socket_base_t(parent_, tid_, sid_, true)
{
    options.type = ZMQ_CLIENT;
}

zmq::client_t::~client_t() {}

void zmq::client_t::xattach_pipe(pipe_t *pipe_, bool subscribe_to_all_, bool locally_initiated_)
{
    LIBZMQ_UNUSED(subscribe_to_all_);
    LIBZMQ_UNUSED(locally_initiated_);

    zmq_assert(pipe_);

    _fq.attach(pipe_);
    _lb.attach(pipe_);
}

int zmq::client_t::xsend(msg_t *msg_)
{
    //  CLIENT sockets do not allow multipart data (ZMQ_SNDMORE)
    if (msg_->flags() & msg_t::more)
    {
        errno = EINVAL;
        return -1;
    }
    return _lb.sendpipe(msg_, NULL);
}

int zmq::client_t::xrecv(msg_t *msg_)
{
    int rc = _fq.recvpipe(msg_, NULL);

    // Drop any messages with more flag
    while (rc == 0 && msg_->flags() & msg_t::more)
    {
        // drop all frames of the current multi-frame message
        rc = _fq.recvpipe(msg_, NULL);

        while (rc == 0 && msg_->flags() & msg_t::more)
            rc = _fq.recvpipe(msg_, NULL);

        // get the new message
        if (rc == 0)
            rc = _fq.recvpipe(msg_, NULL);
    }

    return rc;
}

bool zmq::client_t::xhas_in() { return _fq.has_in(); }

bool zmq::client_t::xhas_out() { return _lb.has_out(); }

void zmq::client_t::xread_activated(pipe_t *pipe_) { _fq.activated(pipe_); }

void zmq::client_t::xwrite_activated(pipe_t *pipe_) { _lb.activated(pipe_); }

void zmq::client_t::xpipe_terminated(pipe_t *pipe_)
{
    _fq.pipe_terminated(pipe_);
    _lb.pipe_terminated(pipe_);
}

//========= end of client.cpp ============

//========= begin of clock.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "clock.hpp"
// ans ignore: #include "likely.hpp"
// ans ignore: #include "config.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "mutex.hpp"

#include <stddef.h>

#if defined _MSC_VER
#if defined _WIN32_WCE
#include <cmnintrin.h>
#else
#include <intrin.h>
#endif
#endif

#if !defined ZMQ_HAVE_WINDOWS
#include <sys/time.h>
#endif

#if defined HAVE_CLOCK_GETTIME || defined HAVE_GETHRTIME
#include <time.h>
#endif

#if defined ZMQ_HAVE_VXWORKS
// ans ignore: #include "timers.h"
#endif

#if defined ZMQ_HAVE_OSX
int         alt_clock_gettime(int clock_id, timespec *ts)
{
    clock_serv_t    cclock;
    mach_timespec_t mts;
    host_get_clock_service(mach_host_self(), clock_id, &cclock);
    clock_get_time(cclock, &mts);
    mach_port_deallocate(mach_task_self(), cclock);
    ts->tv_sec  = mts.tv_sec;
    ts->tv_nsec = mts.tv_nsec;
    return 0;
}
#endif

#ifdef ZMQ_HAVE_WINDOWS
typedef ULONGLONG (*f_compatible_get_tick_count64)();

static zmq::mutex_t compatible_get_tick_count64_mutex;

ULONGLONG compatible_get_tick_count64()
{
#ifdef ZMQ_HAVE_WINDOWS_UWP
    const ULONGLONG result = ::GetTickCount64();
    return result;
#else
    zmq::scoped_lock_t locker(compatible_get_tick_count64_mutex);

    static DWORD s_wrap       = 0;
    static DWORD s_last_tick  = 0;
    const DWORD  current_tick = ::GetTickCount();

    if (current_tick < s_last_tick)
        ++s_wrap;

    s_last_tick            = current_tick;
    const ULONGLONG result = (static_cast<ULONGLONG>(s_wrap) << 32) + static_cast<ULONGLONG>(current_tick);

    return result;
#endif
}

f_compatible_get_tick_count64 init_compatible_get_tick_count64()
{
    f_compatible_get_tick_count64 func = NULL;
#if !defined ZMQ_HAVE_WINDOWS_UWP

    HMODULE module = ::LoadLibraryA("Kernel32.dll");
    if (module != NULL)
        func = reinterpret_cast<f_compatible_get_tick_count64>(::GetProcAddress(module, "GetTickCount64"));
#endif
    if (func == NULL)
        func = compatible_get_tick_count64;

#if !defined ZMQ_HAVE_WINDOWS_UWP
    ::FreeLibrary(module);
#endif

    return func;
}

static f_compatible_get_tick_count64 my_get_tick_count64 = init_compatible_get_tick_count64();
#endif

const uint64_t usecs_per_msec = 1000;
const uint64_t usecs_per_sec  = 1000000;
const uint64_t nsecs_per_usec = 1000;

zmq::clock_t::clock_t()
    : _last_tsc(rdtsc()),
#ifdef ZMQ_HAVE_WINDOWS
      _last_time(static_cast<uint64_t>((*my_get_tick_count64)()))
#else
      _last_time(now_us() / usecs_per_msec)
#endif
{
}

uint64_t zmq::clock_t::now_us()
{
#if defined ZMQ_HAVE_WINDOWS

    //  Get the high resolution counter's accuracy.
    //  While QueryPerformanceFrequency only needs to be called once, since its
    //  value does not change during runtime, we query it here since this is a
    //  static function. It might make sense to cache it, though.
    LARGE_INTEGER ticks_per_second;
    QueryPerformanceFrequency(&ticks_per_second);

    //  What time is it?
    LARGE_INTEGER tick;
    QueryPerformanceCounter(&tick);

    //  Convert the tick number into the number of seconds
    //  since the system was started.
    const double ticks_div = static_cast<double>(ticks_per_second.QuadPart) / usecs_per_sec;
    return static_cast<uint64_t>(tick.QuadPart / ticks_div);

#elif defined HAVE_CLOCK_GETTIME && (defined CLOCK_MONOTONIC || defined ZMQ_HAVE_VXWORKS)

    //  Use POSIX clock_gettime function to get precise monotonic time.
    struct timespec tv;

#if defined ZMQ_HAVE_OSX && __MAC_OS_X_VERSION_MIN_REQUIRED < 101200 // less than macOS 10.12
    int             rc = alt_clock_gettime(SYSTEM_CLOCK, &tv);
#else
    int rc = clock_gettime(CLOCK_MONOTONIC, &tv);
#endif
    // Fix case where system has clock_gettime but CLOCK_MONOTONIC is not supported.
    // This should be a configuration check, but I looked into it and writing an
    // AC_FUNC_CLOCK_MONOTONIC seems beyond my powers.
    if (rc != 0)
    {
#ifndef ZMQ_HAVE_VXWORKS
        //  Use POSIX gettimeofday function to get precise time.
        struct timeval tv;
        int            rc = gettimeofday(&tv, NULL);
        errno_assert(rc == 0);
        return tv.tv_sec * usecs_per_sec + tv.tv_usec;
#endif
    }
    return tv.tv_sec * usecs_per_sec + tv.tv_nsec / nsecs_per_usec;

#elif defined HAVE_GETHRTIME

    return gethrtime() / nsecs_per_usec;

#else

    //  Use POSIX gettimeofday function to get precise time.
    struct timeval tv;
    int            rc = gettimeofday(&tv, NULL);
    errno_assert(rc == 0);
    return tv.tv_sec * usecs_per_sec + tv.tv_usec;

#endif
}

uint64_t zmq::clock_t::now_ms()
{
    uint64_t tsc = rdtsc();

    //  If TSC is not supported, get precise time and chop off the microseconds.
    if (!tsc)
    {
#ifdef ZMQ_HAVE_WINDOWS
        // Under Windows, now_us is not so reliable since QueryPerformanceCounter
        // does not guarantee that it will use a hardware that offers a monotonic timer.
        // So, lets use GetTickCount when GetTickCount64 is not available with an workaround
        // to its 32 bit limitation.
        return static_cast<uint64_t>((*my_get_tick_count64)());
#else
        return now_us() / usecs_per_msec;
#endif
    }

    //  If TSC haven't jumped back (in case of migration to a different
    //  CPU core) and if not too much time elapsed since last measurement,
    //  we can return cached time value.
    if (likely(tsc - _last_tsc <= (clock_precision / 2) && tsc >= _last_tsc))
        return _last_time;

    _last_tsc = tsc;
#ifdef ZMQ_HAVE_WINDOWS
    _last_time = static_cast<uint64_t>((*my_get_tick_count64)());
#else
    _last_time = now_us() / usecs_per_msec;
#endif
    return _last_time;
}

uint64_t zmq::clock_t::rdtsc()
{
#if (defined _MSC_VER && (defined _M_IX86 || defined _M_X64))
    return __rdtsc();
#elif (defined __GNUC__ && (defined __i386__ || defined __x86_64__))
    uint32_t low, high;
    __asm__ volatile("rdtsc" : "=a"(low), "=d"(high));
    return static_cast<uint64_t>(high) << 32 | low;
#elif (defined __SUNPRO_CC && (__SUNPRO_CC >= 0x5100) && (defined __i386 || defined __amd64 || defined __x86_64))
    union
    {
        uint64_t u64val;
        uint32_t u32val[2];
    } tsc;
    asm("rdtsc" : "=a"(tsc.u32val[0]), "=d"(tsc.u32val[1]));
    return tsc.u64val;
#elif defined(__s390__)
    uint64_t tsc;
    asm("\tstck\t%0\n" : "=Q"(tsc) : : "cc");
    return tsc;
#else
    struct timespec ts;
#if defined ZMQ_HAVE_OSX && __MAC_OS_X_VERSION_MIN_REQUIRED < 101200 // less than macOS 10.12
    alt_clock_gettime(SYSTEM_CLOCK, &ts);
#else
    clock_gettime(CLOCK_MONOTONIC, &ts);
#endif
    return static_cast<uint64_t>(ts.tv_sec) * nsecs_per_usec * usecs_per_sec + ts.tv_nsec;
#endif
}

//========= end of clock.cpp ============

//========= begin of ctx.cpp ============

/*
    Copyright (c) 2007-2017 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
#ifndef ZMQ_HAVE_WINDOWS
#include <unistd.h>
#endif

#include <climits>
#include <limits>
#include <new>
#include <sstream>
#include <string.h>

// ans ignore: #include "ctx.hpp"
// ans ignore: #include "socket_base.hpp"
// ans ignore: #include "io_thread.hpp"
// ans ignore: #include "reaper.hpp"
// ans ignore: #include "pipe.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "msg.hpp"
// ans ignore: #include "random.hpp"

#ifdef ZMQ_HAVE_VMCI
#include <vmci_sockets.h>
#endif

#define ZMQ_CTX_TAG_VALUE_GOOD 0xabadcafe
#define ZMQ_CTX_TAG_VALUE_BAD 0xdeadbeef

int clipped_maxsocket(int max_requested_)
{
    if (max_requested_ >= zmq::poller_t::max_fds() && zmq::poller_t::max_fds() != -1)
        // -1 because we need room for the reaper mailbox.
        max_requested_ = zmq::poller_t::max_fds() - 1;

    return max_requested_;
}

zmq::ctx_t::ctx_t()
    : _tag(ZMQ_CTX_TAG_VALUE_GOOD), _starting(true), _terminating(false), _reaper(NULL),
      _max_sockets(clipped_maxsocket(ZMQ_MAX_SOCKETS_DFLT)), _max_msgsz(INT_MAX), _io_thread_count(ZMQ_IO_THREADS_DFLT),
      _blocky(true), _ipv6(false), _zero_copy(true)
{
#ifdef HAVE_FORK
    _pid = getpid();
#endif
#ifdef ZMQ_HAVE_VMCI
    _vmci_fd     = -1;
    _vmci_family = -1;
#endif

    //  Initialise crypto library, if needed.
    zmq::random_open();
}

bool zmq::ctx_t::check_tag() { return _tag == ZMQ_CTX_TAG_VALUE_GOOD; }

zmq::ctx_t::~ctx_t()
{
    //  Check that there are no remaining _sockets.
    zmq_assert(_sockets.empty());

    //  Ask I/O threads to terminate. If stop signal wasn't sent to I/O
    //  thread subsequent invocation of destructor would hang-up.
    for (io_threads_t::size_type i = 0; i != _io_threads.size(); i++)
    {
        _io_threads[i]->stop();
    }

    //  Wait till I/O threads actually terminate.
    for (io_threads_t::size_type i = 0; i != _io_threads.size(); i++)
    {
        LIBZMQ_DELETE(_io_threads[i]);
    }

    //  Deallocate the reaper thread object.
    LIBZMQ_DELETE(_reaper);

    //  The mailboxes in _slots themselves were deallocated with their
    //  corresponding io_thread/socket objects.

    //  De-initialise crypto library, if needed.
    zmq::random_close();

    //  Remove the tag, so that the object is considered dead.
    _tag = ZMQ_CTX_TAG_VALUE_BAD;
}

bool zmq::ctx_t::valid() const { return _term_mailbox.valid(); }

int zmq::ctx_t::terminate()
{
    _slot_sync.lock();

    bool save_terminating = _terminating;
    _terminating          = false;

    // Connect up any pending inproc connections, otherwise we will hang
    pending_connections_t copy = _pending_connections;
    for (pending_connections_t::iterator p = copy.begin(), end = copy.end(); p != end; ++p)
    {
        zmq::socket_base_t *s = create_socket(ZMQ_PAIR);
        // create_socket might fail eg: out of memory/sockets limit reached
        zmq_assert(s);
        s->bind(p->first.c_str());
        s->close();
    }
    _terminating = save_terminating;

    if (!_starting)
    {
#ifdef HAVE_FORK
        if (_pid != getpid())
        {
            // we are a forked child process. Close all file descriptors
            // inherited from the parent.
            for (sockets_t::size_type i = 0; i != _sockets.size(); i++)
                _sockets[i]->get_mailbox()->forked();

            _term_mailbox.forked();
        }
#endif

        //  Check whether termination was already underway, but interrupted and now
        //  restarted.
        bool restarted = _terminating;
        _terminating   = true;

        //  First attempt to terminate the context.
        if (!restarted)
        {
            //  First send stop command to sockets so that any blocking calls
            //  can be interrupted. If there are no sockets we can ask reaper
            //  thread to stop.
            for (sockets_t::size_type i = 0; i != _sockets.size(); i++)
                _sockets[i]->stop();
            if (_sockets.empty())
                _reaper->stop();
        }
        _slot_sync.unlock();

        //  Wait till reaper thread closes all the sockets.
        command_t cmd;
        int       rc = _term_mailbox.recv(&cmd, -1);
        if (rc == -1 && errno == EINTR)
            return -1;
        errno_assert(rc == 0);
        zmq_assert(cmd.type == command_t::done);
        _slot_sync.lock();
        zmq_assert(_sockets.empty());
    }
    _slot_sync.unlock();

#ifdef ZMQ_HAVE_VMCI
    _vmci_sync.lock();

    VMCISock_ReleaseAFValueFd(_vmci_fd);
    _vmci_family = -1;
    _vmci_fd     = -1;

    _vmci_sync.unlock();
#endif

    //  Deallocate the resources.
    delete this;

    return 0;
}

int zmq::ctx_t::shutdown()
{
    scoped_lock_t locker(_slot_sync);

    if (!_starting && !_terminating)
    {
        _terminating = true;

        //  Send stop command to sockets so that any blocking calls
        //  can be interrupted. If there are no sockets we can ask reaper
        //  thread to stop.
        for (sockets_t::size_type i = 0; i != _sockets.size(); i++)
            _sockets[i]->stop();
        if (_sockets.empty())
            _reaper->stop();
    }

    return 0;
}

int zmq::ctx_t::set(int option_, int optval_)
{
    int rc = 0;
    if (option_ == ZMQ_MAX_SOCKETS && optval_ >= 1 && optval_ == clipped_maxsocket(optval_))
    {
        scoped_lock_t locker(_opt_sync);
        _max_sockets = optval_;
    }
    else if (option_ == ZMQ_IO_THREADS && optval_ >= 0)
    {
        scoped_lock_t locker(_opt_sync);
        _io_thread_count = optval_;
    }
    else if (option_ == ZMQ_IPV6 && optval_ >= 0)
    {
        scoped_lock_t locker(_opt_sync);
        _ipv6 = (optval_ != 0);
    }
    else if (option_ == ZMQ_BLOCKY && optval_ >= 0)
    {
        scoped_lock_t locker(_opt_sync);
        _blocky = (optval_ != 0);
    }
    else if (option_ == ZMQ_MAX_MSGSZ && optval_ >= 0)
    {
        scoped_lock_t locker(_opt_sync);
        _max_msgsz = optval_ < INT_MAX ? optval_ : INT_MAX;
    }
    else if (option_ == ZMQ_ZERO_COPY_RECV && optval_ >= 0)
    {
        scoped_lock_t locker(_opt_sync);
        _zero_copy = (optval_ != 0);
    }
    else
    {
        rc = thread_ctx_t::set(option_, optval_);
    }
    return rc;
}

int zmq::ctx_t::get(int option_)
{
    int rc = 0;
    if (option_ == ZMQ_MAX_SOCKETS)
        rc = _max_sockets;
    else if (option_ == ZMQ_SOCKET_LIMIT)
        rc = clipped_maxsocket(65535);
    else if (option_ == ZMQ_IO_THREADS)
        rc = _io_thread_count;
    else if (option_ == ZMQ_IPV6)
        rc = _ipv6;
    else if (option_ == ZMQ_BLOCKY)
        rc = _blocky;
    else if (option_ == ZMQ_MAX_MSGSZ)
        rc = _max_msgsz;
    else if (option_ == ZMQ_MSG_T_SIZE)
        rc = sizeof(zmq_msg_t);
    else if (option_ == ZMQ_ZERO_COPY_RECV)
    {
        rc = _zero_copy;
    }
    else
    {
        rc = thread_ctx_t::get(option_);
    }
    return rc;
}

bool zmq::ctx_t::start()
{
    //  Initialise the array of mailboxes. Additional two slots are for
    //  zmq_ctx_term thread and reaper thread.
    _opt_sync.lock();
    const int term_and_reaper_threads_count = 2;
    const int mazmq                         = _max_sockets;
    const int ios                           = _io_thread_count;
    _opt_sync.unlock();
    int slot_count = mazmq + ios + term_and_reaper_threads_count;
    try
    {
        _slots.reserve(slot_count);
        _empty_slots.reserve(slot_count - term_and_reaper_threads_count);
    }
    catch (const std::bad_alloc &)
    {
        errno = ENOMEM;
        return false;
    }
    _slots.resize(term_and_reaper_threads_count);

    //  Initialise the infrastructure for zmq_ctx_term thread.
    _slots[term_tid] = &_term_mailbox;

    //  Create the reaper thread.
    _reaper = new (std::nothrow) reaper_t(this, reaper_tid);
    if (!_reaper)
    {
        errno = ENOMEM;
        goto fail_cleanup_slots;
    }
    if (!_reaper->get_mailbox()->valid())
        goto fail_cleanup_reaper;
    _slots[reaper_tid] = _reaper->get_mailbox();
    _reaper->start();

    //  Create I/O thread objects and launch them.
    _slots.resize(slot_count, NULL);

    for (int i = term_and_reaper_threads_count; i != ios + term_and_reaper_threads_count; i++)
    {
        io_thread_t *io_thread = new (std::nothrow) io_thread_t(this, i);
        if (!io_thread)
        {
            errno = ENOMEM;
            goto fail_cleanup_reaper;
        }
        if (!io_thread->get_mailbox()->valid())
        {
            delete io_thread;
            goto fail_cleanup_reaper;
        }
        _io_threads.push_back(io_thread);
        _slots[i] = io_thread->get_mailbox();
        io_thread->start();
    }

    //  In the unused part of the slot array, create a list of empty slots.
    for (int32_t i = static_cast<int32_t>(_slots.size()) - 1;
         i >= static_cast<int32_t>(ios) + term_and_reaper_threads_count; i--)
    {
        _empty_slots.push_back(i);
    }

    _starting = false;
    return true;

fail_cleanup_reaper:
    _reaper->stop();
    delete _reaper;
    _reaper = NULL;

fail_cleanup_slots:
    _slots.clear();
    return false;
}

zmq::socket_base_t *zmq::ctx_t::create_socket(int type_)
{
    scoped_lock_t locker(_slot_sync);

    if (unlikely(_starting))
    {
        if (!start())
            return NULL;
    }

    //  Once zmq_ctx_term() was called, we can't create new sockets.
    if (_terminating)
    {
        errno = ETERM;
        return NULL;
    }

    //  If max_sockets limit was reached, return error.
    if (_empty_slots.empty())
    {
        errno = EMFILE;
        return NULL;
    }

    //  Choose a slot for the socket.
    uint32_t slot = _empty_slots.back();
    _empty_slots.pop_back();

    //  Generate new unique socket ID.
    int sid = (static_cast<int>(max_socket_id.add(1))) + 1;

    //  Create the socket and register its mailbox.
    socket_base_t *s = socket_base_t::create(type_, this, slot, sid);
    if (!s)
    {
        _empty_slots.push_back(slot);
        return NULL;
    }
    _sockets.push_back(s);
    _slots[slot] = s->get_mailbox();

    return s;
}

void zmq::ctx_t::destroy_socket(class socket_base_t *socket_)
{
    scoped_lock_t locker(_slot_sync);

    //  Free the associated thread slot.
    uint32_t tid = socket_->get_tid();
    _empty_slots.push_back(tid);
    _slots[tid] = NULL;

    //  Remove the socket from the list of sockets.
    _sockets.erase(socket_);

    //  If zmq_ctx_term() was already called and there are no more socket
    //  we can ask reaper thread to terminate.
    if (_terminating && _sockets.empty())
        _reaper->stop();
}

zmq::object_t *zmq::ctx_t::get_reaper() { return _reaper; }

zmq::thread_ctx_t::thread_ctx_t()
    : _thread_priority(ZMQ_THREAD_PRIORITY_DFLT), _thread_sched_policy(ZMQ_THREAD_SCHED_POLICY_DFLT)
{
}

void zmq::thread_ctx_t::start_thread(thread_t &thread_, thread_fn *tfn_, void *arg_, const char *name_) const
{
    thread_.setSchedulingParameters(_thread_priority, _thread_sched_policy, _thread_affinity_cpus);

    char namebuf[16] = "";
    snprintf(namebuf, sizeof(namebuf), "%s%sZMQbg%s%s", _thread_name_prefix.empty() ? "" : _thread_name_prefix.c_str(),
             _thread_name_prefix.empty() ? "" : "/", name_ ? "/" : "", name_ ? name_ : "");
    thread_.start(tfn_, arg_, namebuf);
}

int zmq::thread_ctx_t::set(int option_, int optval_)
{
    int rc = 0;
    if (option_ == ZMQ_THREAD_SCHED_POLICY && optval_ >= 0)
    {
        scoped_lock_t locker(_opt_sync);
        _thread_sched_policy = optval_;
    }
    else if (option_ == ZMQ_THREAD_AFFINITY_CPU_ADD && optval_ >= 0)
    {
        scoped_lock_t locker(_opt_sync);
        _thread_affinity_cpus.insert(optval_);
    }
    else if (option_ == ZMQ_THREAD_AFFINITY_CPU_REMOVE && optval_ >= 0)
    {
        scoped_lock_t locker(_opt_sync);
        if (0 == _thread_affinity_cpus.erase(optval_))
        {
            errno = EINVAL;
            rc    = -1;
        }
    }
    else if (option_ == ZMQ_THREAD_NAME_PREFIX && optval_ >= 0)
    {
        std::ostringstream s;
        s << optval_;
        scoped_lock_t locker(_opt_sync);
        _thread_name_prefix = s.str();
    }
    else if (option_ == ZMQ_THREAD_PRIORITY && optval_ >= 0)
    {
        scoped_lock_t locker(_opt_sync);
        _thread_priority = optval_;
    }
    else
    {
        errno = EINVAL;
        rc    = -1;
    }
    return rc;
}

int zmq::thread_ctx_t::get(int option_)
{
    int rc = 0;
    if (option_ == ZMQ_THREAD_SCHED_POLICY)
    {
        scoped_lock_t locker(_opt_sync);
        rc = _thread_sched_policy;
    }
    else if (option_ == ZMQ_THREAD_NAME_PREFIX)
    {
        scoped_lock_t locker(_opt_sync);
        rc = atoi(_thread_name_prefix.c_str());
    }
    else
    {
        errno = EINVAL;
        rc    = -1;
    }
    return rc;
}

void zmq::ctx_t::send_command(uint32_t tid_, const command_t &command_) { _slots[tid_]->send(command_); }

zmq::io_thread_t *zmq::ctx_t::choose_io_thread(uint64_t affinity_)
{
    if (_io_threads.empty())
        return NULL;

    //  Find the I/O thread with minimum load.
    int          min_load           = -1;
    io_thread_t *selected_io_thread = NULL;
    for (io_threads_t::size_type i = 0; i != _io_threads.size(); i++)
    {
        if (!affinity_ || (affinity_ & (uint64_t(1) << i)))
        {
            int load = _io_threads[i]->get_load();
            if (selected_io_thread == NULL || load < min_load)
            {
                min_load           = load;
                selected_io_thread = _io_threads[i];
            }
        }
    }
    return selected_io_thread;
}

int zmq::ctx_t::register_endpoint(const char *addr_, const endpoint_t &endpoint_)
{
    scoped_lock_t locker(_endpoints_sync);

    const bool inserted = _endpoints.ZMQ_MAP_INSERT_OR_EMPLACE(std::string(addr_), endpoint_).second;
    if (!inserted)
    {
        errno = EADDRINUSE;
        return -1;
    }
    return 0;
}

int zmq::ctx_t::unregister_endpoint(const std::string &addr_, socket_base_t *socket_)
{
    scoped_lock_t locker(_endpoints_sync);

    const endpoints_t::iterator it = _endpoints.find(addr_);
    if (it == _endpoints.end() || it->second.socket != socket_)
    {
        errno = ENOENT;
        return -1;
    }

    //  Remove endpoint.
    _endpoints.erase(it);

    return 0;
}

void zmq::ctx_t::unregister_endpoints(socket_base_t *socket_)
{
    scoped_lock_t locker(_endpoints_sync);

    for (endpoints_t::iterator it = _endpoints.begin(), end = _endpoints.end(); it != end;)
    {
        if (it->second.socket == socket_)
#if __cplusplus >= 201103L
            it = _endpoints.erase(it);
#else
            _endpoints.erase(it++);
#endif
        else
            ++it;
    }
}

zmq::endpoint_t zmq::ctx_t::find_endpoint(const char *addr_)
{
    scoped_lock_t locker(_endpoints_sync);

    endpoints_t::iterator it = _endpoints.find(addr_);
    if (it == _endpoints.end())
    {
        errno            = ECONNREFUSED;
        endpoint_t empty = {NULL, options_t()};
        return empty;
    }
    endpoint_t endpoint = it->second;

    //  Increment the command sequence number of the peer so that it won't
    //  get deallocated until "bind" command is issued by the caller.
    //  The subsequent 'bind' has to be called with inc_seqnum parameter
    //  set to false, so that the seqnum isn't incremented twice.
    endpoint.socket->inc_seqnum();

    return endpoint;
}

void zmq::ctx_t::pend_connection(const std::string &addr_, const endpoint_t &endpoint_, pipe_t **pipes_)
{
    scoped_lock_t locker(_endpoints_sync);

    const pending_connection_t pending_connection = {endpoint_, pipes_[0], pipes_[1]};

    endpoints_t::iterator it = _endpoints.find(addr_);
    if (it == _endpoints.end())
    {
        //  Still no bind.
        endpoint_.socket->inc_seqnum();
        _pending_connections.ZMQ_MAP_INSERT_OR_EMPLACE(addr_, pending_connection);
    }
    else
    {
        //  Bind has happened in the mean time, connect directly
        connect_inproc_sockets(it->second.socket, it->second.options, pending_connection, connect_side);
    }
}

void zmq::ctx_t::connect_pending(const char *addr_, zmq::socket_base_t *bind_socket_)
{
    scoped_lock_t locker(_endpoints_sync);

    std::pair<pending_connections_t::iterator, pending_connections_t::iterator> pending =
        _pending_connections.equal_range(addr_);
    for (pending_connections_t::iterator p = pending.first; p != pending.second; ++p)
        connect_inproc_sockets(bind_socket_, _endpoints[addr_].options, p->second, bind_side);

    _pending_connections.erase(pending.first, pending.second);
}

void zmq::ctx_t::connect_inproc_sockets(zmq::socket_base_t *bind_socket_, options_t &bind_options_,
                                        const pending_connection_t &pending_connection_, side side_)
{
    bind_socket_->inc_seqnum();
    pending_connection_.bind_pipe->set_tid(bind_socket_->get_tid());

    if (!bind_options_.recv_routing_id)
    {
        msg_t      msg;
        const bool ok = pending_connection_.bind_pipe->read(&msg);
        zmq_assert(ok);
        const int rc = msg.close();
        errno_assert(rc == 0);
    }

    if (!get_effective_conflate_option(pending_connection_.endpoint.options))
    {
        pending_connection_.connect_pipe->set_hwms_boost(bind_options_.sndhwm, bind_options_.rcvhwm);
        pending_connection_.bind_pipe->set_hwms_boost(pending_connection_.endpoint.options.sndhwm,
                                                      pending_connection_.endpoint.options.rcvhwm);

        pending_connection_.connect_pipe->set_hwms(pending_connection_.endpoint.options.rcvhwm,
                                                   pending_connection_.endpoint.options.sndhwm);
        pending_connection_.bind_pipe->set_hwms(bind_options_.rcvhwm, bind_options_.sndhwm);
    }
    else
    {
        pending_connection_.connect_pipe->set_hwms(-1, -1);
        pending_connection_.bind_pipe->set_hwms(-1, -1);
    }

    if (side_ == bind_side)
    {
        command_t cmd;
        cmd.type           = command_t::bind;
        cmd.args.bind.pipe = pending_connection_.bind_pipe;
        bind_socket_->process_command(cmd);
        bind_socket_->send_inproc_connected(pending_connection_.endpoint.socket);
    }
    else
        pending_connection_.connect_pipe->send_bind(bind_socket_, pending_connection_.bind_pipe, false);

    // When a ctx is terminated all pending inproc connection will be
    // connected, but the socket will already be closed and the pipe will be
    // in waiting_for_delimiter state, which means no more writes can be done
    // and the routing id write fails and causes an assert. Check if the socket
    // is open before sending.
    if (pending_connection_.endpoint.options.recv_routing_id && pending_connection_.endpoint.socket->check_tag())
    {
        send_routing_id(pending_connection_.bind_pipe, bind_options_);
    }
}

#ifdef ZMQ_HAVE_VMCI

int zmq::ctx_t::get_vmci_socket_family()
{
    zmq::scoped_lock_t locker(_vmci_sync);

    if (_vmci_fd == -1)
    {
        _vmci_family = VMCISock_GetAFValueFd(&_vmci_fd);

        if (_vmci_fd != -1)
        {
#ifdef FD_CLOEXEC
            int rc = fcntl(_vmci_fd, F_SETFD, FD_CLOEXEC);
            errno_assert(rc != -1);
#endif
        }
    }

    return _vmci_family;
}

#endif

//  The last used socket ID, or 0 if no socket was used so far. Note that this
//  is a global variable. Thus, even sockets created in different contexts have
//  unique IDs.
zmq::atomic_counter_t zmq::ctx_t::max_socket_id;

//========= end of ctx.cpp ============

//========= begin of curve_client.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"

#ifdef ZMQ_HAVE_CURVE

// ans ignore: #include "msg.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "curve_client.hpp"
// ans ignore: #include "wire.hpp"
// ans ignore: #include "curve_client_tools.hpp"

zmq::curve_client_t::curve_client_t(session_base_t *session_, const options_t &options_)
    : mechanism_base_t(session_, options_),
      curve_mechanism_base_t(session_, options_, "CurveZMQMESSAGEC", "CurveZMQMESSAGES"), _state(send_hello),
      _tools(options_.curve_public_key, options_.curve_secret_key, options_.curve_server_key)
{
}

zmq::curve_client_t::~curve_client_t() {}

int zmq::curve_client_t::next_handshake_command(msg_t *msg_)
{
    int rc = 0;

    switch (_state)
    {
    case send_hello:
        rc = produce_hello(msg_);
        if (rc == 0)
            _state = expect_welcome;
        break;
    case send_initiate:
        rc = produce_initiate(msg_);
        if (rc == 0)
            _state = expect_ready;
        break;
    default:
        errno = EAGAIN;
        rc    = -1;
    }
    return rc;
}

int zmq::curve_client_t::process_handshake_command(msg_t *msg_)
{
    const unsigned char *msg_data = static_cast<unsigned char *>(msg_->data());
    const size_t         msg_size = msg_->size();

    int rc = 0;
    if (curve_client_tools_t::is_handshake_command_welcome(msg_data, msg_size))
        rc = process_welcome(msg_data, msg_size);
    else if (curve_client_tools_t::is_handshake_command_ready(msg_data, msg_size))
        rc = process_ready(msg_data, msg_size);
    else if (curve_client_tools_t::is_handshake_command_error(msg_data, msg_size))
        rc = process_error(msg_data, msg_size);
    else
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNEXPECTED_COMMAND);
        errno = EPROTO;
        rc    = -1;
    }

    if (rc == 0)
    {
        rc = msg_->close();
        errno_assert(rc == 0);
        rc = msg_->init();
        errno_assert(rc == 0);
    }

    return rc;
}

int zmq::curve_client_t::encode(msg_t *msg_)
{
    zmq_assert(_state == connected);
    return curve_mechanism_base_t::encode(msg_);
}

int zmq::curve_client_t::decode(msg_t *msg_)
{
    zmq_assert(_state == connected);
    return curve_mechanism_base_t::decode(msg_);
}

zmq::mechanism_t::status_t zmq::curve_client_t::status() const
{
    if (_state == connected)
        return mechanism_t::ready;
    if (_state == error_received)
        return mechanism_t::error;
    else
        return mechanism_t::handshaking;
}

int zmq::curve_client_t::produce_hello(msg_t *msg_)
{
    int rc = msg_->init_size(200);
    errno_assert(rc == 0);

    rc = _tools.produce_hello(msg_->data(), cn_nonce);
    if (rc == -1)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_CRYPTOGRAPHIC);

        // TODO this is somewhat inconsistent: we call init_size, but we may
        // not close msg_; i.e. we assume that msg_ is initialized but empty
        // (if it were non-empty, calling init_size might cause a leak!)

        // msg_->close ();
        return -1;
    }

    cn_nonce++;

    return 0;
}

int zmq::curve_client_t::process_welcome(const uint8_t *msg_data_, size_t msg_size_)
{
    int rc = _tools.process_welcome(msg_data_, msg_size_, cn_precom);

    if (rc == -1)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_CRYPTOGRAPHIC);

        errno = EPROTO;
        return -1;
    }

    _state = send_initiate;

    return 0;
}

int zmq::curve_client_t::produce_initiate(msg_t *msg_)
{
    const size_t   metadata_length    = basic_properties_len();
    unsigned char *metadata_plaintext = static_cast<unsigned char *>(malloc(metadata_length));
    alloc_assert(metadata_plaintext);

    add_basic_properties(metadata_plaintext, metadata_length);

    size_t msg_size = 113 + 128 + crypto_box_BOXZEROBYTES + metadata_length;
    int    rc       = msg_->init_size(msg_size);
    errno_assert(rc == 0);

    rc = _tools.produce_initiate(msg_->data(), msg_size, cn_nonce, metadata_plaintext, metadata_length);

    free(metadata_plaintext);

    if (-1 == rc)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_CRYPTOGRAPHIC);

        // TODO see comment in produce_hello
        return -1;
    }

    cn_nonce++;

    return 0;
}

int zmq::curve_client_t::process_ready(const uint8_t *msg_data_, size_t msg_size_)
{
    if (msg_size_ < 30)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_READY);
        errno = EPROTO;
        return -1;
    }

    const size_t clen = (msg_size_ - 14) + crypto_box_BOXZEROBYTES;

    uint8_t  ready_nonce[crypto_box_NONCEBYTES];
    uint8_t *ready_plaintext = static_cast<uint8_t *>(malloc(crypto_box_ZEROBYTES + clen));
    alloc_assert(ready_plaintext);
    uint8_t *ready_box = static_cast<uint8_t *>(malloc(crypto_box_BOXZEROBYTES + 16 + clen));
    alloc_assert(ready_box);

    memset(ready_box, 0, crypto_box_BOXZEROBYTES);
    memcpy(ready_box + crypto_box_BOXZEROBYTES, msg_data_ + 14, clen - crypto_box_BOXZEROBYTES);

    memcpy(ready_nonce, "CurveZMQREADY---", 16);
    memcpy(ready_nonce + 16, msg_data_ + 6, 8);
    cn_peer_nonce = get_uint64(msg_data_ + 6);

    int rc = crypto_box_open_afternm(ready_plaintext, ready_box, clen, ready_nonce, cn_precom);
    free(ready_box);

    if (rc != 0)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_CRYPTOGRAPHIC);
        errno = EPROTO;
        return -1;
    }

    rc = parse_metadata(ready_plaintext + crypto_box_ZEROBYTES, clen - crypto_box_ZEROBYTES);
    free(ready_plaintext);

    if (rc == 0)
        _state = connected;
    else
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_INVALID_METADATA);
        errno = EPROTO;
    }

    return rc;
}

int zmq::curve_client_t::process_error(const uint8_t *msg_data_, size_t msg_size_)
{
    if (_state != expect_welcome && _state != expect_ready)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNEXPECTED_COMMAND);
        errno = EPROTO;
        return -1;
    }
    if (msg_size_ < 7)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_ERROR);
        errno = EPROTO;
        return -1;
    }
    const size_t error_reason_len = static_cast<size_t>(msg_data_[6]);
    if (error_reason_len > msg_size_ - 7)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_ERROR);
        errno = EPROTO;
        return -1;
    }
    const char *error_reason = reinterpret_cast<const char *>(msg_data_) + 7;
    handle_error_reason(error_reason, error_reason_len);
    _state = error_received;
    return 0;
}

#endif

//========= end of curve_client.cpp ============

//========= begin of curve_mechanism_base.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "curve_mechanism_base.hpp"
// ans ignore: #include "msg.hpp"
// ans ignore: #include "wire.hpp"
// ans ignore: #include "session_base.hpp"

#ifdef ZMQ_HAVE_CURVE

zmq::curve_mechanism_base_t::curve_mechanism_base_t(session_base_t *session_, const options_t &options_,
                                                    const char *encode_nonce_prefix_, const char *decode_nonce_prefix_)
    : mechanism_base_t(session_, options_), encode_nonce_prefix(encode_nonce_prefix_),
      decode_nonce_prefix(decode_nonce_prefix_), cn_nonce(1), cn_peer_nonce(1)
{
}

int zmq::curve_mechanism_base_t::encode(msg_t *msg_)
{
    const size_t mlen = crypto_box_ZEROBYTES + 1 + msg_->size();

    uint8_t message_nonce[crypto_box_NONCEBYTES];
    memcpy(message_nonce, encode_nonce_prefix, 16);
    put_uint64(message_nonce + 16, cn_nonce);

    uint8_t flags = 0;
    if (msg_->flags() & msg_t::more)
        flags |= 0x01;
    if (msg_->flags() & msg_t::command)
        flags |= 0x02;

    uint8_t *message_plaintext = static_cast<uint8_t *>(malloc(mlen));
    alloc_assert(message_plaintext);

    memset(message_plaintext, 0, crypto_box_ZEROBYTES);
    message_plaintext[crypto_box_ZEROBYTES] = flags;
    memcpy(message_plaintext + crypto_box_ZEROBYTES + 1, msg_->data(), msg_->size());

    uint8_t *message_box = static_cast<uint8_t *>(malloc(mlen));
    alloc_assert(message_box);

    int rc = crypto_box_afternm(message_box, message_plaintext, mlen, message_nonce, cn_precom);
    zmq_assert(rc == 0);

    rc = msg_->close();
    zmq_assert(rc == 0);

    rc = msg_->init_size(16 + mlen - crypto_box_BOXZEROBYTES);
    zmq_assert(rc == 0);

    uint8_t *message = static_cast<uint8_t *>(msg_->data());

    memcpy(message, "\x07MESSAGE", 8);
    memcpy(message + 8, message_nonce + 16, 8);
    memcpy(message + 16, message_box + crypto_box_BOXZEROBYTES, mlen - crypto_box_BOXZEROBYTES);

    free(message_plaintext);
    free(message_box);

    cn_nonce++;

    return 0;
}

int zmq::curve_mechanism_base_t::decode(msg_t *msg_)
{
    int rc = check_basic_command_structure(msg_);
    if (rc == -1)
        return -1;

    const size_t   size    = msg_->size();
    const uint8_t *message = static_cast<uint8_t *>(msg_->data());

    if (size < 8 || memcmp(message, "\x07MESSAGE", 8))
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNEXPECTED_COMMAND);
        errno = EPROTO;
        return -1;
    }

    if (size < 33)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_MESSAGE);
        errno = EPROTO;
        return -1;
    }

    uint8_t message_nonce[crypto_box_NONCEBYTES];
    memcpy(message_nonce, decode_nonce_prefix, 16);
    memcpy(message_nonce + 16, message + 8, 8);
    uint64_t nonce = get_uint64(message + 8);
    if (nonce <= cn_peer_nonce)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_INVALID_SEQUENCE);
        errno = EPROTO;
        return -1;
    }
    cn_peer_nonce = nonce;

    const size_t clen = crypto_box_BOXZEROBYTES + msg_->size() - 16;

    uint8_t *message_plaintext = static_cast<uint8_t *>(malloc(clen));
    alloc_assert(message_plaintext);

    uint8_t *message_box = static_cast<uint8_t *>(malloc(clen));
    alloc_assert(message_box);

    memset(message_box, 0, crypto_box_BOXZEROBYTES);
    memcpy(message_box + crypto_box_BOXZEROBYTES, message + 16, msg_->size() - 16);

    rc = crypto_box_open_afternm(message_plaintext, message_box, clen, message_nonce, cn_precom);
    if (rc == 0)
    {
        rc = msg_->close();
        zmq_assert(rc == 0);

        rc = msg_->init_size(clen - 1 - crypto_box_ZEROBYTES);
        zmq_assert(rc == 0);

        const uint8_t flags = message_plaintext[crypto_box_ZEROBYTES];
        if (flags & 0x01)
            msg_->set_flags(msg_t::more);
        if (flags & 0x02)
            msg_->set_flags(msg_t::command);

        memcpy(msg_->data(), message_plaintext + crypto_box_ZEROBYTES + 1, msg_->size());
    }
    else
    {
        // CURVE I : connection key used for MESSAGE is wrong
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_CRYPTOGRAPHIC);
        errno = EPROTO;
    }
    free(message_plaintext);
    free(message_box);

    return rc;
}

#endif

//========= end of curve_mechanism_base.cpp ============

//========= begin of curve_server.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"

#ifdef ZMQ_HAVE_CURVE

// ans ignore: #include "msg.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "curve_server.hpp"
// ans ignore: #include "wire.hpp"

zmq::curve_server_t::curve_server_t(session_base_t *session_, const std::string &peer_address_,
                                    const options_t &options_)
    : mechanism_base_t(session_, options_),
      zap_client_common_handshake_t(session_, peer_address_, options_, sending_ready),
      curve_mechanism_base_t(session_, options_, "CurveZMQMESSAGES", "CurveZMQMESSAGEC")
{
    int rc;
    //  Fetch our secret key from socket options
    memcpy(_secret_key, options_.curve_secret_key, crypto_box_SECRETKEYBYTES);

    //  Generate short-term key pair
    rc = crypto_box_keypair(_cn_public, _cn_secret);
    zmq_assert(rc == 0);
}

zmq::curve_server_t::~curve_server_t() {}

int zmq::curve_server_t::next_handshake_command(msg_t *msg_)
{
    int rc = 0;

    switch (state)
    {
    case sending_welcome:
        rc = produce_welcome(msg_);
        if (rc == 0)
            state = waiting_for_initiate;
        break;
    case sending_ready:
        rc = produce_ready(msg_);
        if (rc == 0)
            state = ready;
        break;
    case sending_error:
        rc = produce_error(msg_);
        if (rc == 0)
            state = error_sent;
        break;
    default:
        errno = EAGAIN;
        rc    = -1;
        break;
    }
    return rc;
}

int zmq::curve_server_t::process_handshake_command(msg_t *msg_)
{
    int rc = 0;

    switch (state)
    {
    case waiting_for_hello:
        rc = process_hello(msg_);
        break;
    case waiting_for_initiate:
        rc = process_initiate(msg_);
        break;
    default:
        // TODO I think this is not a case reachable with a misbehaving
        // client. It is not an "invalid handshake command", but would be
        // trying to process a handshake command in an invalid state,
        // which is purely under control of this peer.
        // Therefore, it should be changed to zmq_assert (false);

        // CURVE I: invalid handshake command
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNSPECIFIED);
        errno = EPROTO;
        rc    = -1;
        break;
    }
    if (rc == 0)
    {
        rc = msg_->close();
        errno_assert(rc == 0);
        rc = msg_->init();
        errno_assert(rc == 0);
    }
    return rc;
}

int zmq::curve_server_t::encode(msg_t *msg_)
{
    zmq_assert(state == ready);
    return curve_mechanism_base_t::encode(msg_);
}

int zmq::curve_server_t::decode(msg_t *msg_)
{
    zmq_assert(state == ready);
    return curve_mechanism_base_t::decode(msg_);
}

int zmq::curve_server_t::process_hello(msg_t *msg_)
{
    int rc = check_basic_command_structure(msg_);
    if (rc == -1)
        return -1;

    const size_t         size  = msg_->size();
    const uint8_t *const hello = static_cast<uint8_t *>(msg_->data());

    if (size < 6 || memcmp(hello, "\x05HELLO", 6))
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNEXPECTED_COMMAND);
        errno = EPROTO;
        return -1;
    }

    if (size != 200)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_HELLO);
        errno = EPROTO;
        return -1;
    }

    const uint8_t major = hello[6];
    const uint8_t minor = hello[7];

    if (major != 1 || minor != 0)
    {
        // CURVE I: client HELLO has unknown version number
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_HELLO);
        errno = EPROTO;
        return -1;
    }

    //  Save client's short-term public key (C')
    memcpy(_cn_client, hello + 80, 32);

    uint8_t hello_nonce[crypto_box_NONCEBYTES];
    uint8_t hello_plaintext[crypto_box_ZEROBYTES + 64];
    uint8_t hello_box[crypto_box_BOXZEROBYTES + 80];

    memcpy(hello_nonce, "CurveZMQHELLO---", 16);
    memcpy(hello_nonce + 16, hello + 112, 8);
    cn_peer_nonce = get_uint64(hello + 112);

    memset(hello_box, 0, crypto_box_BOXZEROBYTES);
    memcpy(hello_box + crypto_box_BOXZEROBYTES, hello + 120, 80);

    //  Open Box [64 * %x0](C'->S)
    rc = crypto_box_open(hello_plaintext, hello_box, sizeof hello_box, hello_nonce, _cn_client, _secret_key);
    if (rc != 0)
    {
        // CURVE I: cannot open client HELLO -- wrong server key?
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_CRYPTOGRAPHIC);
        errno = EPROTO;
        return -1;
    }

    state = sending_welcome;
    return rc;
}

int zmq::curve_server_t::produce_welcome(msg_t *msg_)
{
    uint8_t cookie_nonce[crypto_secretbox_NONCEBYTES];
    uint8_t cookie_plaintext[crypto_secretbox_ZEROBYTES + 64];
    uint8_t cookie_ciphertext[crypto_secretbox_BOXZEROBYTES + 80];

    //  Create full nonce for encryption
    //  8-byte prefix plus 16-byte random nonce
    memcpy(cookie_nonce, "COOKIE--", 8);
    randombytes(cookie_nonce + 8, 16);

    //  Generate cookie = Box [C' + s'](t)
    memset(cookie_plaintext, 0, crypto_secretbox_ZEROBYTES);
    memcpy(cookie_plaintext + crypto_secretbox_ZEROBYTES, _cn_client, 32);
    memcpy(cookie_plaintext + crypto_secretbox_ZEROBYTES + 32, _cn_secret, 32);

    //  Generate fresh cookie key
    randombytes(_cookie_key, crypto_secretbox_KEYBYTES);

    //  Encrypt using symmetric cookie key
    int rc = crypto_secretbox(cookie_ciphertext, cookie_plaintext, sizeof cookie_plaintext, cookie_nonce, _cookie_key);
    zmq_assert(rc == 0);

    uint8_t welcome_nonce[crypto_box_NONCEBYTES];
    uint8_t welcome_plaintext[crypto_box_ZEROBYTES + 128];
    uint8_t welcome_ciphertext[crypto_box_BOXZEROBYTES + 144];

    //  Create full nonce for encryption
    //  8-byte prefix plus 16-byte random nonce
    memcpy(welcome_nonce, "WELCOME-", 8);
    randombytes(welcome_nonce + 8, crypto_box_NONCEBYTES - 8);

    //  Create 144-byte Box [S' + cookie](S->C')
    memset(welcome_plaintext, 0, crypto_box_ZEROBYTES);
    memcpy(welcome_plaintext + crypto_box_ZEROBYTES, _cn_public, 32);
    memcpy(welcome_plaintext + crypto_box_ZEROBYTES + 32, cookie_nonce + 8, 16);
    memcpy(welcome_plaintext + crypto_box_ZEROBYTES + 48, cookie_ciphertext + crypto_secretbox_BOXZEROBYTES, 80);

    rc = crypto_box(welcome_ciphertext, welcome_plaintext, sizeof welcome_plaintext, welcome_nonce, _cn_client,
                    _secret_key);

    //  TODO I think we should change this back to zmq_assert (rc == 0);
    //  as it was before https://github.com/zeromq/libzmq/pull/1832
    //  The reason given there was that secret_key might be 0ed.
    //  But if it were, we would never get this far, since we could
    //  not have opened the client's hello box with a 0ed key.

    if (rc == -1)
        return -1;

    rc = msg_->init_size(168);
    errno_assert(rc == 0);

    uint8_t *const welcome = static_cast<uint8_t *>(msg_->data());
    memcpy(welcome, "\x07WELCOME", 8);
    memcpy(welcome + 8, welcome_nonce + 8, 16);
    memcpy(welcome + 24, welcome_ciphertext + crypto_box_BOXZEROBYTES, 144);

    return 0;
}

int zmq::curve_server_t::process_initiate(msg_t *msg_)
{
    int rc = check_basic_command_structure(msg_);
    if (rc == -1)
        return -1;

    const size_t   size     = msg_->size();
    const uint8_t *initiate = static_cast<uint8_t *>(msg_->data());

    if (size < 9 || memcmp(initiate, "\x08INITIATE", 9))
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNEXPECTED_COMMAND);
        errno = EPROTO;
        return -1;
    }

    if (size < 257)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_INITIATE);
        errno = EPROTO;
        return -1;
    }

    uint8_t cookie_nonce[crypto_secretbox_NONCEBYTES];
    uint8_t cookie_plaintext[crypto_secretbox_ZEROBYTES + 64];
    uint8_t cookie_box[crypto_secretbox_BOXZEROBYTES + 80];

    //  Open Box [C' + s'](t)
    memset(cookie_box, 0, crypto_secretbox_BOXZEROBYTES);
    memcpy(cookie_box + crypto_secretbox_BOXZEROBYTES, initiate + 25, 80);

    memcpy(cookie_nonce, "COOKIE--", 8);
    memcpy(cookie_nonce + 8, initiate + 9, 16);

    rc = crypto_secretbox_open(cookie_plaintext, cookie_box, sizeof cookie_box, cookie_nonce, _cookie_key);
    if (rc != 0)
    {
        // CURVE I: cannot open client INITIATE cookie
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_CRYPTOGRAPHIC);
        errno = EPROTO;
        return -1;
    }

    //  Check cookie plain text is as expected [C' + s']
    if (memcmp(cookie_plaintext + crypto_secretbox_ZEROBYTES, _cn_client, 32) ||
        memcmp(cookie_plaintext + crypto_secretbox_ZEROBYTES + 32, _cn_secret, 32))
    {
        // TODO this case is very hard to test, as it would require a modified
        //  client that knows the server's secret temporary cookie key

        // CURVE I: client INITIATE cookie is not valid
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_CRYPTOGRAPHIC);
        errno = EPROTO;
        return -1;
    }

    const size_t clen = (size - 113) + crypto_box_BOXZEROBYTES;

    uint8_t  initiate_nonce[crypto_box_NONCEBYTES];
    uint8_t *initiate_plaintext = static_cast<uint8_t *>(malloc(crypto_box_ZEROBYTES + clen));
    alloc_assert(initiate_plaintext);
    uint8_t *initiate_box = static_cast<uint8_t *>(malloc(crypto_box_BOXZEROBYTES + clen));
    alloc_assert(initiate_box);

    //  Open Box [C + vouch + metadata](C'->S')
    memset(initiate_box, 0, crypto_box_BOXZEROBYTES);
    memcpy(initiate_box + crypto_box_BOXZEROBYTES, initiate + 113, clen - crypto_box_BOXZEROBYTES);

    memcpy(initiate_nonce, "CurveZMQINITIATE", 16);
    memcpy(initiate_nonce + 16, initiate + 105, 8);
    cn_peer_nonce = get_uint64(initiate + 105);

    const uint8_t *client_key = initiate_plaintext + crypto_box_ZEROBYTES;

    rc = crypto_box_open(initiate_plaintext, initiate_box, clen, initiate_nonce, _cn_client, _cn_secret);
    if (rc != 0)
    {
        // CURVE I: cannot open client INITIATE
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_CRYPTOGRAPHIC);
        errno = EPROTO;
        rc    = -1;
        goto exit;
    }

    uint8_t vouch_nonce[crypto_box_NONCEBYTES];
    uint8_t vouch_plaintext[crypto_box_ZEROBYTES + 64];
    uint8_t vouch_box[crypto_box_BOXZEROBYTES + 80];

    //  Open Box Box [C',S](C->S') and check contents
    memset(vouch_box, 0, crypto_box_BOXZEROBYTES);
    memcpy(vouch_box + crypto_box_BOXZEROBYTES, initiate_plaintext + crypto_box_ZEROBYTES + 48, 80);

    memcpy(vouch_nonce, "VOUCH---", 8);
    memcpy(vouch_nonce + 8, initiate_plaintext + crypto_box_ZEROBYTES + 32, 16);

    rc = crypto_box_open(vouch_plaintext, vouch_box, sizeof vouch_box, vouch_nonce, client_key, _cn_secret);
    if (rc != 0)
    {
        // CURVE I: cannot open client INITIATE vouch
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_CRYPTOGRAPHIC);
        errno = EPROTO;
        rc    = -1;
        goto exit;
    }

    //  What we decrypted must be the client's short-term public key
    if (memcmp(vouch_plaintext + crypto_box_ZEROBYTES, _cn_client, 32))
    {
        // TODO this case is very hard to test, as it would require a modified
        //  client that knows the server's secret short-term key

        // CURVE I: invalid handshake from client (public key)
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_KEY_EXCHANGE);
        errno = EPROTO;
        rc    = -1;
        goto exit;
    }

    //  Precompute connection secret from client key
    rc = crypto_box_beforenm(cn_precom, _cn_client, _cn_secret);
    zmq_assert(rc == 0);

    //  Given this is a backward-incompatible change, it's behind a socket
    //  option disabled by default.
    if (zap_required() || !options.zap_enforce_domain)
    {
        //  Use ZAP protocol (RFC 27) to authenticate the user.
        rc = session->zap_connect();
        if (rc == 0)
        {
            send_zap_request(client_key);
            state = waiting_for_zap_reply;

            //  TODO actually, it is quite unlikely that we can read the ZAP
            //  reply already, but removing this has some strange side-effect
            //  (probably because the pipe's in_active flag is true until a read
            //  is attempted)
            rc = receive_and_process_zap_reply();
            if (rc == -1)
                goto exit;
        }
        else if (!options.zap_enforce_domain)
        {
            //  This supports the Stonehouse pattern (encryption without
            //  authentication) in legacy mode (domain set but no handler).
            state = sending_ready;
        }
        else
        {
            session->get_socket()->event_handshake_failed_no_detail(session->get_endpoint(), EFAULT);
            rc = -1;
            goto exit;
        }
    }
    else
    {
        //  This supports the Stonehouse pattern (encryption without authentication).
        state = sending_ready;
    }

    rc = parse_metadata(initiate_plaintext + crypto_box_ZEROBYTES + 128, clen - crypto_box_ZEROBYTES - 128);

exit:
    free(initiate_plaintext);
    free(initiate_box);
    return rc;
}

int zmq::curve_server_t::produce_ready(msg_t *msg_)
{
    const size_t metadata_length = basic_properties_len();
    uint8_t      ready_nonce[crypto_box_NONCEBYTES];

    uint8_t *ready_plaintext = static_cast<uint8_t *>(malloc(crypto_box_ZEROBYTES + metadata_length));
    alloc_assert(ready_plaintext);

    //  Create Box [metadata](S'->C')
    memset(ready_plaintext, 0, crypto_box_ZEROBYTES);
    uint8_t *ptr = ready_plaintext + crypto_box_ZEROBYTES;

    ptr += add_basic_properties(ptr, metadata_length);
    const size_t mlen = ptr - ready_plaintext;

    memcpy(ready_nonce, "CurveZMQREADY---", 16);
    put_uint64(ready_nonce + 16, cn_nonce);

    uint8_t *ready_box = static_cast<uint8_t *>(malloc(crypto_box_BOXZEROBYTES + 16 + metadata_length));
    alloc_assert(ready_box);

    int rc = crypto_box_afternm(ready_box, ready_plaintext, mlen, ready_nonce, cn_precom);
    zmq_assert(rc == 0);

    free(ready_plaintext);

    rc = msg_->init_size(14 + mlen - crypto_box_BOXZEROBYTES);
    errno_assert(rc == 0);

    uint8_t *ready = static_cast<uint8_t *>(msg_->data());

    memcpy(ready, "\x05READY", 6);
    //  Short nonce, prefixed by "CurveZMQREADY---"
    memcpy(ready + 6, ready_nonce + 16, 8);
    //  Box [metadata](S'->C')
    memcpy(ready + 14, ready_box + crypto_box_BOXZEROBYTES, mlen - crypto_box_BOXZEROBYTES);
    free(ready_box);

    cn_nonce++;

    return 0;
}

int zmq::curve_server_t::produce_error(msg_t *msg_) const
{
    const size_t expected_status_code_length = 3;
    zmq_assert(status_code.length() == 3);
    const int rc = msg_->init_size(6 + 1 + expected_status_code_length);
    zmq_assert(rc == 0);
    char *msg_data = static_cast<char *>(msg_->data());
    memcpy(msg_data, "\5ERROR", 6);
    msg_data[6] = expected_status_code_length;
    memcpy(msg_data + 7, status_code.c_str(), expected_status_code_length);
    return 0;
}

void zmq::curve_server_t::send_zap_request(const uint8_t *key_)
{
    zap_client_t::send_zap_request("CURVE", 5, key_, crypto_box_PUBLICKEYBYTES);
}

#endif

//========= end of curve_server.cpp ============

//========= begin of dealer.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "dealer.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "msg.hpp"

zmq::dealer_t::dealer_t(class ctx_t *parent_, uint32_t tid_, int sid_)
    : socket_base_t(parent_, tid_, sid_), _probe_router(false)
{
    options.type = ZMQ_DEALER;
}

zmq::dealer_t::~dealer_t() {}

void zmq::dealer_t::xattach_pipe(pipe_t *pipe_, bool subscribe_to_all_, bool locally_initiated_)
{
    LIBZMQ_UNUSED(subscribe_to_all_);
    LIBZMQ_UNUSED(locally_initiated_);

    zmq_assert(pipe_);

    if (_probe_router)
    {
        msg_t probe_msg;
        int   rc = probe_msg.init();
        errno_assert(rc == 0);

        rc = pipe_->write(&probe_msg);
        // zmq_assert (rc) is not applicable here, since it is not a bug.
        LIBZMQ_UNUSED(rc);

        pipe_->flush();

        rc = probe_msg.close();
        errno_assert(rc == 0);
    }

    _fq.attach(pipe_);
    _lb.attach(pipe_);
}

int zmq::dealer_t::xsetsockopt(int option_, const void *optval_, size_t optvallen_)
{
    bool is_int = (optvallen_ == sizeof(int));
    int  value  = 0;
    if (is_int)
        memcpy(&value, optval_, sizeof(int));

    switch (option_)
    {
    case ZMQ_PROBE_ROUTER:
        if (is_int && value >= 0)
        {
            _probe_router = (value != 0);
            return 0;
        }
        break;

    default:
        break;
    }

    errno = EINVAL;
    return -1;
}

int zmq::dealer_t::xsend(msg_t *msg_) { return sendpipe(msg_, NULL); }

int zmq::dealer_t::xrecv(msg_t *msg_) { return recvpipe(msg_, NULL); }

bool zmq::dealer_t::xhas_in() { return _fq.has_in(); }

bool zmq::dealer_t::xhas_out() { return _lb.has_out(); }

void zmq::dealer_t::xread_activated(pipe_t *pipe_) { _fq.activated(pipe_); }

void zmq::dealer_t::xwrite_activated(pipe_t *pipe_) { _lb.activated(pipe_); }

void zmq::dealer_t::xpipe_terminated(pipe_t *pipe_)
{
    _fq.pipe_terminated(pipe_);
    _lb.pipe_terminated(pipe_);
}

int zmq::dealer_t::sendpipe(msg_t *msg_, pipe_t **pipe_) { return _lb.sendpipe(msg_, pipe_); }

int zmq::dealer_t::recvpipe(msg_t *msg_, pipe_t **pipe_) { return _fq.recvpipe(msg_, pipe_); }

//========= end of dealer.cpp ============

//========= begin of decoder_allocators.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "decoder_allocators.hpp"

#include <cmath>

// ans ignore: #include "msg.hpp"

zmq::shared_message_memory_allocator::shared_message_memory_allocator(std::size_t bufsize_)
    : _buf(NULL), _buf_size(0), _max_size(bufsize_), _msg_content(NULL),
      _max_counters(
          static_cast<size_t>(std::ceil(static_cast<double>(_max_size) / static_cast<double>(msg_t::max_vsm_size))))
{
}

zmq::shared_message_memory_allocator::shared_message_memory_allocator(std::size_t bufsize_, std::size_t max_messages_)
    : _buf(NULL), _buf_size(0), _max_size(bufsize_), _msg_content(NULL), _max_counters(max_messages_)
{
}

zmq::shared_message_memory_allocator::~shared_message_memory_allocator() { deallocate(); }

unsigned char *zmq::shared_message_memory_allocator::allocate()
{
    if (_buf)
    {
        // release reference count to couple lifetime to messages
        zmq::atomic_counter_t *c = reinterpret_cast<zmq::atomic_counter_t *>(_buf);

        // if refcnt drops to 0, there are no message using the buffer
        // because either all messages have been closed or only vsm-messages
        // were created
        if (c->sub(1))
        {
            // buffer is still in use as message data. "Release" it and create a new one
            // release pointer because we are going to create a new buffer
            release();
        }
    }

    // if buf != NULL it is not used by any message so we can re-use it for the next run
    if (!_buf)
    {
        // allocate memory for reference counters together with reception buffer
        std::size_t const allocationsize =
            _max_size + sizeof(zmq::atomic_counter_t) + _max_counters * sizeof(zmq::msg_t::content_t);

        _buf = static_cast<unsigned char *>(std::malloc(allocationsize));
        alloc_assert(_buf);

        new (_buf) atomic_counter_t(1);
    }
    else
    {
        // release reference count to couple lifetime to messages
        zmq::atomic_counter_t *c = reinterpret_cast<zmq::atomic_counter_t *>(_buf);
        c->set(1);
    }

    _buf_size    = _max_size;
    _msg_content = reinterpret_cast<zmq::msg_t::content_t *>(_buf + sizeof(atomic_counter_t) + _max_size);
    return _buf + sizeof(zmq::atomic_counter_t);
}

void zmq::shared_message_memory_allocator::deallocate()
{
    zmq::atomic_counter_t *c = reinterpret_cast<zmq::atomic_counter_t *>(_buf);
    if (_buf && !c->sub(1))
    {
        std::free(_buf);
    }
    clear();
}

unsigned char *zmq::shared_message_memory_allocator::release()
{
    unsigned char *b = _buf;
    clear();
    return b;
}

void zmq::shared_message_memory_allocator::clear()
{
    _buf         = NULL;
    _buf_size    = 0;
    _msg_content = NULL;
}

void zmq::shared_message_memory_allocator::inc_ref() { (reinterpret_cast<zmq::atomic_counter_t *>(_buf))->add(1); }

void zmq::shared_message_memory_allocator::call_dec_ref(void *, void *hint_)
{
    zmq_assert(hint_);
    unsigned char *        buf = static_cast<unsigned char *>(hint_);
    zmq::atomic_counter_t *c   = reinterpret_cast<zmq::atomic_counter_t *>(buf);

    if (!c->sub(1))
    {
        c->~atomic_counter_t();
        std::free(buf);
        buf = NULL;
    }
}

std::size_t zmq::shared_message_memory_allocator::size() const { return _buf_size; }

unsigned char *zmq::shared_message_memory_allocator::data() { return _buf + sizeof(zmq::atomic_counter_t); }

//========= end of decoder_allocators.cpp ============

//========= begin of devpoll.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "devpoll.hpp"
#if defined ZMQ_IOTHREAD_POLLER_USE_DEVPOLL

#include <algorithm>
#include <fcntl.h>
#include <limits.h>
#include <sys/devpoll.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

// ans ignore: #include "devpoll.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "config.hpp"
// ans ignore: #include "i_poll_events.hpp"

zmq::devpoll_t::devpoll_t(const zmq::thread_ctx_t &ctx_) : worker_poller_base_t(ctx_)
{
    devpoll_fd = open("/dev/poll", O_RDWR);
    errno_assert(devpoll_fd != -1);
}

zmq::devpoll_t::~devpoll_t()
{
    //  Wait till the worker thread exits.
    stop_worker();

    close(devpoll_fd);
}

void zmq::devpoll_t::devpoll_ctl(fd_t fd_, short events_)
{
    struct pollfd pfd = {fd_, events_, 0};
    ssize_t       rc  = write(devpoll_fd, &pfd, sizeof pfd);
    zmq_assert(rc == sizeof pfd);
}

zmq::devpoll_t::handle_t zmq::devpoll_t::add_fd(fd_t fd_, i_poll_events *reactor_)
{
    check_thread();
    //  If the file descriptor table is too small expand it.
    fd_table_t::size_type sz = fd_table.size();
    if (sz <= (fd_table_t::size_type)fd_)
    {
        fd_table.resize(fd_ + 1);
        while (sz != (fd_table_t::size_type)(fd_ + 1))
        {
            fd_table[sz].valid = false;
            ++sz;
        }
    }

    zmq_assert(!fd_table[fd_].valid);

    fd_table[fd_].events   = 0;
    fd_table[fd_].reactor  = reactor_;
    fd_table[fd_].valid    = true;
    fd_table[fd_].accepted = false;

    devpoll_ctl(fd_, 0);
    pending_list.push_back(fd_);

    //  Increase the load metric of the thread.
    adjust_load(1);

    return fd_;
}

void zmq::devpoll_t::rm_fd(handle_t handle_)
{
    check_thread();
    zmq_assert(fd_table[handle_].valid);

    devpoll_ctl(handle_, POLLREMOVE);
    fd_table[handle_].valid = false;

    //  Decrease the load metric of the thread.
    adjust_load(-1);
}

void zmq::devpoll_t::set_pollin(handle_t handle_)
{
    check_thread();
    devpoll_ctl(handle_, POLLREMOVE);
    fd_table[handle_].events |= POLLIN;
    devpoll_ctl(handle_, fd_table[handle_].events);
}

void zmq::devpoll_t::reset_pollin(handle_t handle_)
{
    check_thread();
    devpoll_ctl(handle_, POLLREMOVE);
    fd_table[handle_].events &= ~((short)POLLIN);
    devpoll_ctl(handle_, fd_table[handle_].events);
}

void zmq::devpoll_t::set_pollout(handle_t handle_)
{
    check_thread();
    devpoll_ctl(handle_, POLLREMOVE);
    fd_table[handle_].events |= POLLOUT;
    devpoll_ctl(handle_, fd_table[handle_].events);
}

void zmq::devpoll_t::reset_pollout(handle_t handle_)
{
    check_thread();
    devpoll_ctl(handle_, POLLREMOVE);
    fd_table[handle_].events &= ~((short)POLLOUT);
    devpoll_ctl(handle_, fd_table[handle_].events);
}

void zmq::devpoll_t::stop() { check_thread(); }

int zmq::devpoll_t::max_fds() { return -1; }

void zmq::devpoll_t::loop()
{
    while (true)
    {
        struct pollfd ev_buf[max_io_events];
        struct dvpoll poll_req;

        for (pending_list_t::size_type i = 0; i < pending_list.size(); i++)
            fd_table[pending_list[i]].accepted = true;
        pending_list.clear();

        //  Execute any due timers.
        int timeout = (int)execute_timers();

        if (get_load() == 0)
        {
            if (timeout == 0)
                break;

            // TODO sleep for timeout
            continue;
        }

        //  Wait for events.
        //  On Solaris, we can retrieve no more then (OPEN_MAX - 1) events.
        poll_req.dp_fds = &ev_buf[0];
#if defined ZMQ_HAVE_SOLARIS
        poll_req.dp_nfds = std::min((int)max_io_events, OPEN_MAX - 1);
#else
        poll_req.dp_nfds = max_io_events;
#endif
        poll_req.dp_timeout = timeout ? timeout : -1;
        int n               = ioctl(devpoll_fd, DP_POLL, &poll_req);
        if (n == -1 && errno == EINTR)
            continue;
        errno_assert(n != -1);

        for (int i = 0; i < n; i++)
        {
            fd_entry_t *fd_ptr = &fd_table[ev_buf[i].fd];
            if (!fd_ptr->valid || !fd_ptr->accepted)
                continue;
            if (ev_buf[i].revents & (POLLERR | POLLHUP))
                fd_ptr->reactor->in_event();
            if (!fd_ptr->valid || !fd_ptr->accepted)
                continue;
            if (ev_buf[i].revents & POLLOUT)
                fd_ptr->reactor->out_event();
            if (!fd_ptr->valid || !fd_ptr->accepted)
                continue;
            if (ev_buf[i].revents & POLLIN)
                fd_ptr->reactor->in_event();
        }
    }
}

#endif

//========= end of devpoll.cpp ============

//========= begin of dgram.cpp ============

/*
    Copyright (c) 2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "dgram.hpp"
// ans ignore: #include "pipe.hpp"
// ans ignore: #include "wire.hpp"
// ans ignore: #include "random.hpp"
// ans ignore: #include "likely.hpp"
// ans ignore: #include "err.hpp"

zmq::dgram_t::dgram_t(class ctx_t *parent_, uint32_t tid_, int sid_)
    : socket_base_t(parent_, tid_, sid_), _pipe(NULL), _last_in(NULL), _more_out(false)
{
    options.type       = ZMQ_DGRAM;
    options.raw_socket = true;
}

zmq::dgram_t::~dgram_t() { zmq_assert(!_pipe); }

void zmq::dgram_t::xattach_pipe(pipe_t *pipe_, bool subscribe_to_all_, bool locally_initiated_)
{
    LIBZMQ_UNUSED(subscribe_to_all_);
    LIBZMQ_UNUSED(locally_initiated_);

    zmq_assert(pipe_);

    //  ZMQ_DGRAM socket can only be connected to a single peer.
    //  The socket rejects any further connection requests.
    if (_pipe == NULL)
        _pipe = pipe_;
    else
        pipe_->terminate(false);
}

void zmq::dgram_t::xpipe_terminated(pipe_t *pipe_)
{
    if (pipe_ == _pipe)
    {
        if (_last_in == _pipe)
        {
            _last_in = NULL;
        }
        _pipe = NULL;
    }
}

void zmq::dgram_t::xread_activated(pipe_t *)
{
    //  There's just one pipe. No lists of active and inactive pipes.
    //  There's nothing to do here.
}

void zmq::dgram_t::xwrite_activated(pipe_t *)
{
    //  There's just one pipe. No lists of active and inactive pipes.
    //  There's nothing to do here.
}

int zmq::dgram_t::xsend(msg_t *msg_)
{
    // If there's no out pipe, just drop it.
    if (!_pipe)
    {
        int rc = msg_->close();
        errno_assert(rc == 0);
        return -1;
    }

    //  If this is the first part of the message it's the ID of the
    //  peer to send the message to.
    if (!_more_out)
    {
        if (!(msg_->flags() & msg_t::more))
        {
            errno = EINVAL;
            return -1;
        }
    }
    else
    {
        //  dgram messages are two part only, reject part if more is set
        if (msg_->flags() & msg_t::more)
        {
            errno = EINVAL;
            return -1;
        }
    }

    // Push the message into the pipe.
    if (!_pipe->write(msg_))
    {
        errno = EAGAIN;
        return -1;
    }

    if (!(msg_->flags() & msg_t::more))
        _pipe->flush();

    // flip the more flag
    _more_out = !_more_out;

    //  Detach the message from the data buffer.
    int rc = msg_->init();
    errno_assert(rc == 0);

    return 0;
}

int zmq::dgram_t::xrecv(msg_t *msg_)
{
    //  Deallocate old content of the message.
    int rc = msg_->close();
    errno_assert(rc == 0);

    if (!_pipe || !_pipe->read(msg_))
    {
        //  Initialise the output parameter to be a 0-byte message.
        rc = msg_->init();
        errno_assert(rc == 0);

        errno = EAGAIN;
        return -1;
    }
    _last_in = _pipe;

    return 0;
}

bool zmq::dgram_t::xhas_in()
{
    if (!_pipe)
        return false;

    return _pipe->check_read();
}

bool zmq::dgram_t::xhas_out()
{
    if (!_pipe)
        return false;

    return _pipe->check_write();
}

//========= end of dgram.cpp ============

//========= begin of dish.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <string.h>

// ans ignore: #include "macros.hpp"
// ans ignore: #include "dish.hpp"
// ans ignore: #include "err.hpp"

zmq::dish_t::dish_t(class ctx_t *parent_, uint32_t tid_, int sid_)
    : socket_base_t(parent_, tid_, sid_, true), _has_message(false)
{
    options.type = ZMQ_DISH;

    //  When socket is being closed down we don't want to wait till pending
    //  subscription commands are sent to the wire.
    options.linger.store(0);

    int rc = _message.init();
    errno_assert(rc == 0);
}

zmq::dish_t::~dish_t()
{
    int rc = _message.close();
    errno_assert(rc == 0);
}

void zmq::dish_t::xattach_pipe(pipe_t *pipe_, bool subscribe_to_all_, bool locally_initiated_)
{
    LIBZMQ_UNUSED(subscribe_to_all_);
    LIBZMQ_UNUSED(locally_initiated_);

    zmq_assert(pipe_);
    _fq.attach(pipe_);
    _dist.attach(pipe_);

    //  Send all the cached subscriptions to the new upstream peer.
    send_subscriptions(pipe_);
}

void zmq::dish_t::xread_activated(pipe_t *pipe_) { _fq.activated(pipe_); }

void zmq::dish_t::xwrite_activated(pipe_t *pipe_) { _dist.activated(pipe_); }

void zmq::dish_t::xpipe_terminated(pipe_t *pipe_)
{
    _fq.pipe_terminated(pipe_);
    _dist.pipe_terminated(pipe_);
}

void zmq::dish_t::xhiccuped(pipe_t *pipe_)
{
    //  Send all the cached subscriptions to the hiccuped pipe.
    send_subscriptions(pipe_);
}

int zmq::dish_t::xjoin(const char *group_)
{
    std::string group = std::string(group_);

    if (group.length() > ZMQ_GROUP_MAX_LENGTH)
    {
        errno = EINVAL;
        return -1;
    }

    //  User cannot join same group twice
    if (!_subscriptions.insert(group).second)
    {
        errno = EINVAL;
        return -1;
    }

    msg_t msg;
    int   rc = msg.init_join();
    errno_assert(rc == 0);

    rc = msg.set_group(group_);
    errno_assert(rc == 0);

    int err = 0;
    rc      = _dist.send_to_all(&msg);
    if (rc != 0)
        err = errno;
    int rc2 = msg.close();
    errno_assert(rc2 == 0);
    if (rc != 0)
        errno = err;
    return rc;
}

int zmq::dish_t::xleave(const char *group_)
{
    std::string group = std::string(group_);

    if (group.length() > ZMQ_GROUP_MAX_LENGTH)
    {
        errno = EINVAL;
        return -1;
    }

    if (0 == _subscriptions.erase(group))
    {
        errno = EINVAL;
        return -1;
    }

    msg_t msg;
    int   rc = msg.init_leave();
    errno_assert(rc == 0);

    rc = msg.set_group(group_);
    errno_assert(rc == 0);

    int err = 0;
    rc      = _dist.send_to_all(&msg);
    if (rc != 0)
        err = errno;
    int rc2 = msg.close();
    errno_assert(rc2 == 0);
    if (rc != 0)
        errno = err;
    return rc;
}

int zmq::dish_t::xsend(msg_t *msg_)
{
    LIBZMQ_UNUSED(msg_);
    errno = ENOTSUP;
    return -1;
}

bool zmq::dish_t::xhas_out()
{
    //  Subscription can be added/removed anytime.
    return true;
}

int zmq::dish_t::xrecv(msg_t *msg_)
{
    //  If there's already a message prepared by a previous call to zmq_poll,
    //  return it straight ahead.
    if (_has_message)
    {
        const int rc = msg_->move(_message);
        errno_assert(rc == 0);
        _has_message = false;
        return 0;
    }

    return xxrecv(msg_);
}

int zmq::dish_t::xxrecv(msg_t *msg_)
{
    do
    {
        //  Get a message using fair queueing algorithm.
        const int rc = _fq.recv(msg_);

        //  If there's no message available, return immediately.
        //  The same when error occurs.
        if (rc != 0)
            return -1;

        //  Skip non matching messages
    } while (0 == _subscriptions.count(std::string(msg_->group())));

    //  Found a matching message
    return 0;
}

bool zmq::dish_t::xhas_in()
{
    //  If there's already a message prepared by a previous call to zmq_poll,
    //  return straight ahead.
    if (_has_message)
        return true;

    const int rc = xxrecv(&_message);
    if (rc != 0)
    {
        errno_assert(errno == EAGAIN);
        return false;
    }

    //  Matching message found
    _has_message = true;
    return true;
}

void zmq::dish_t::send_subscriptions(pipe_t *pipe_)
{
    for (subscriptions_t::iterator it = _subscriptions.begin(), end = _subscriptions.end(); it != end; ++it)
    {
        msg_t msg;
        int   rc = msg.init_join();
        errno_assert(rc == 0);

        rc = msg.set_group(it->c_str());
        errno_assert(rc == 0);

        //  Send it to the pipe.
        pipe_->write(&msg);
        msg.close();
    }

    pipe_->flush();
}

zmq::dish_session_t::dish_session_t(io_thread_t *io_thread_, bool connect_, socket_base_t *socket_,
                                    const options_t &options_, address_t *addr_)
    : session_base_t(io_thread_, connect_, socket_, options_, addr_), _state(group)
{
}

zmq::dish_session_t::~dish_session_t() {}

int zmq::dish_session_t::push_msg(msg_t *msg_)
{
    if (_state == group)
    {
        if ((msg_->flags() & msg_t::more) != msg_t::more)
        {
            errno = EFAULT;
            return -1;
        }

        if (msg_->size() > ZMQ_GROUP_MAX_LENGTH)
        {
            errno = EFAULT;
            return -1;
        }

        _group_msg = *msg_;
        _state     = body;

        int rc = msg_->init();
        errno_assert(rc == 0);
        return 0;
    }
    const char *group_setting = msg_->group();
    int         rc;
    if (group_setting[0] != 0)
        goto has_group;

    //  Set the message group
    rc = msg_->set_group(static_cast<char *>(_group_msg.data()), _group_msg.size());
    errno_assert(rc == 0);

    //  We set the group, so we don't need the group_msg anymore
    rc = _group_msg.close();
    errno_assert(rc == 0);
has_group:
    //  Thread safe socket doesn't support multipart messages
    if ((msg_->flags() & msg_t::more) == msg_t::more)
    {
        errno = EFAULT;
        return -1;
    }

    //  Push message to dish socket
    rc = session_base_t::push_msg(msg_);

    if (rc == 0)
        _state = group;

    return rc;
}

int zmq::dish_session_t::pull_msg(msg_t *msg_)
{
    int rc = session_base_t::pull_msg(msg_);

    if (rc != 0)
        return rc;

    if (!msg_->is_join() && !msg_->is_leave())
        return rc;

    int group_length = static_cast<int>(strlen(msg_->group()));

    msg_t command;
    int   offset;

    if (msg_->is_join())
    {
        rc = command.init_size(group_length + 5);
        errno_assert(rc == 0);
        offset = 5;
        memcpy(command.data(), "\4JOIN", 5);
    }
    else
    {
        rc = command.init_size(group_length + 6);
        errno_assert(rc == 0);
        offset = 6;
        memcpy(command.data(), "\5LEAVE", 6);
    }

    command.set_flags(msg_t::command);
    char *command_data = static_cast<char *>(command.data());

    //  Copy the group
    memcpy(command_data + offset, msg_->group(), group_length);

    //  Close the join message
    rc = msg_->close();
    errno_assert(rc == 0);

    *msg_ = command;

    return 0;
}

void zmq::dish_session_t::reset()
{
    session_base_t::reset();
    _state = group;
}

//========= end of dish.cpp ============

//========= begin of dist.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "dist.hpp"
// ans ignore: #include "pipe.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "msg.hpp"
// ans ignore: #include "likely.hpp"

zmq::dist_t::dist_t() : _matching(0), _active(0), _eligible(0), _more(false) {}

zmq::dist_t::~dist_t() { zmq_assert(_pipes.empty()); }

void zmq::dist_t::attach(pipe_t *pipe_)
{
    //  If we are in the middle of sending a message, we'll add new pipe
    //  into the list of eligible pipes. Otherwise we add it to the list
    //  of active pipes.
    if (_more)
    {
        _pipes.push_back(pipe_);
        _pipes.swap(_eligible, _pipes.size() - 1);
        _eligible++;
    }
    else
    {
        _pipes.push_back(pipe_);
        _pipes.swap(_active, _pipes.size() - 1);
        _active++;
        _eligible++;
    }
}

void zmq::dist_t::match(pipe_t *pipe_)
{
    //  If pipe is already matching do nothing.
    if (_pipes.index(pipe_) < _matching)
        return;

    //  If the pipe isn't eligible, ignore it.
    if (_pipes.index(pipe_) >= _eligible)
        return;

    //  Mark the pipe as matching.
    _pipes.swap(_pipes.index(pipe_), _matching);
    _matching++;
}

void zmq::dist_t::reverse_match()
{
    pipes_t::size_type prev_matching = _matching;

    // Reset matching to 0
    unmatch();

    // Mark all matching pipes as not matching and vice-versa.
    // To do this, push all pipes that are eligible but not
    // matched - i.e. between "matching" and "eligible" -
    // to the beginning of the queue.
    for (pipes_t::size_type i = prev_matching; i < _eligible; ++i)
    {
        _pipes.swap(i, _matching++);
    }
}

void zmq::dist_t::unmatch() { _matching = 0; }

void zmq::dist_t::pipe_terminated(pipe_t *pipe_)
{
    //  Remove the pipe from the list; adjust number of matching, active and/or
    //  eligible pipes accordingly.
    if (_pipes.index(pipe_) < _matching)
    {
        _pipes.swap(_pipes.index(pipe_), _matching - 1);
        _matching--;
    }
    if (_pipes.index(pipe_) < _active)
    {
        _pipes.swap(_pipes.index(pipe_), _active - 1);
        _active--;
    }
    if (_pipes.index(pipe_) < _eligible)
    {
        _pipes.swap(_pipes.index(pipe_), _eligible - 1);
        _eligible--;
    }

    _pipes.erase(pipe_);
}

void zmq::dist_t::activated(pipe_t *pipe_)
{
    //  Move the pipe from passive to eligible state.
    if (_eligible < _pipes.size())
    {
        _pipes.swap(_pipes.index(pipe_), _eligible);
        _eligible++;
    }

    //  If there's no message being sent at the moment, move it to
    //  the active state.
    if (!_more && _active < _pipes.size())
    {
        _pipes.swap(_eligible - 1, _active);
        _active++;
    }
}

int zmq::dist_t::send_to_all(msg_t *msg_)
{
    _matching = _active;
    return send_to_matching(msg_);
}

int zmq::dist_t::send_to_matching(msg_t *msg_)
{
    //  Is this end of a multipart message?
    bool msg_more = (msg_->flags() & msg_t::more) != 0;

    //  Push the message to matching pipes.
    distribute(msg_);

    //  If multipart message is fully sent, activate all the eligible pipes.
    if (!msg_more)
        _active = _eligible;

    _more = msg_more;

    return 0;
}

void zmq::dist_t::distribute(msg_t *msg_)
{
    //  If there are no matching pipes available, simply drop the message.
    if (_matching == 0)
    {
        int rc = msg_->close();
        errno_assert(rc == 0);
        rc = msg_->init();
        errno_assert(rc == 0);
        return;
    }

    if (msg_->is_vsm())
    {
        for (pipes_t::size_type i = 0; i < _matching; ++i)
            if (!write(_pipes[i], msg_))
                --i; //  Retry last write because index will have been swapped
        int rc = msg_->close();
        errno_assert(rc == 0);
        rc = msg_->init();
        errno_assert(rc == 0);
        return;
    }

    //  Add matching-1 references to the message. We already hold one reference,
    //  that's why -1.
    msg_->add_refs(static_cast<int>(_matching) - 1);

    //  Push copy of the message to each matching pipe.
    int failed = 0;
    for (pipes_t::size_type i = 0; i < _matching; ++i)
        if (!write(_pipes[i], msg_))
        {
            ++failed;
            --i; //  Retry last write because index will have been swapped
        }
    if (unlikely(failed))
        msg_->rm_refs(failed);

    //  Detach the original message from the data buffer. Note that we don't
    //  close the message. That's because we've already used all the references.
    int rc = msg_->init();
    errno_assert(rc == 0);
}

bool zmq::dist_t::has_out() { return true; }

bool zmq::dist_t::write(pipe_t *pipe_, msg_t *msg_)
{
    if (!pipe_->write(msg_))
    {
        _pipes.swap(_pipes.index(pipe_), _matching - 1);
        _matching--;
        _pipes.swap(_pipes.index(pipe_), _active - 1);
        _active--;
        _pipes.swap(_active, _eligible - 1);
        _eligible--;
        return false;
    }
    if (!(msg_->flags() & msg_t::more))
        pipe_->flush();
    return true;
}

bool zmq::dist_t::check_hwm()
{
    for (pipes_t::size_type i = 0; i < _matching; ++i)
        if (!_pipes[i]->check_hwm())
            return false;

    return true;
}

//========= end of dist.cpp ============

//========= begin of endpoint.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "endpoint.hpp"

zmq::endpoint_uri_pair_t zmq::make_unconnected_connect_endpoint_pair(const std::string &endpoint_)
{
    return endpoint_uri_pair_t(std::string(), endpoint_, endpoint_type_connect);
}

zmq::endpoint_uri_pair_t zmq::make_unconnected_bind_endpoint_pair(const std::string &endpoint_)
{
    return endpoint_uri_pair_t(endpoint_, std::string(), endpoint_type_bind);
}

//========= end of endpoint.cpp ============

//========= begin of epoll.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#if defined ZMQ_IOTHREAD_POLLER_USE_EPOLL
// ans ignore: #include "epoll.hpp"

#if !defined ZMQ_HAVE_WINDOWS
#include <unistd.h>
#endif

#include <algorithm>
#include <new>
#include <signal.h>
#include <stdlib.h>
#include <string.h>

// ans ignore: #include "macros.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "config.hpp"
// ans ignore: #include "i_poll_events.hpp"

#ifdef ZMQ_HAVE_WINDOWS
const zmq::epoll_t::epoll_fd_t zmq::epoll_t::epoll_retired_fd = INVALID_HANDLE_VALUE;
#endif

zmq::epoll_t::epoll_t(const zmq::thread_ctx_t &ctx_) : worker_poller_base_t(ctx_)
{
#ifdef ZMQ_IOTHREAD_POLLER_USE_EPOLL_CLOEXEC
    //  Setting this option result in sane behaviour when exec() functions
    //  are used. Old sockets are closed and don't block TCP ports, avoid
    //  leaks, etc.
    _epoll_fd = epoll_create1(EPOLL_CLOEXEC);
#else
    _epoll_fd = epoll_create(1);
#endif
    errno_assert(_epoll_fd != epoll_retired_fd);
}

zmq::epoll_t::~epoll_t()
{
    //  Wait till the worker thread exits.
    stop_worker();

#ifdef ZMQ_HAVE_WINDOWS
    epoll_close(_epoll_fd);
#else
    close(_epoll_fd);
#endif
    for (retired_t::iterator it = _retired.begin(), end = _retired.end(); it != end; ++it)
    {
        LIBZMQ_DELETE(*it);
    }
}

zmq::epoll_t::handle_t zmq::epoll_t::add_fd(fd_t fd_, i_poll_events *events_)
{
    check_thread();
    poll_entry_t *pe = new (std::nothrow) poll_entry_t;
    alloc_assert(pe);

    //  The memset is not actually needed. It's here to prevent debugging
    //  tools to complain about using uninitialised memory.
    memset(pe, 0, sizeof(poll_entry_t));

    pe->fd          = fd_;
    pe->ev.events   = 0;
    pe->ev.data.ptr = pe;
    pe->events      = events_;

    int rc = epoll_ctl(_epoll_fd, EPOLL_CTL_ADD, fd_, &pe->ev);
    errno_assert(rc != -1);

    //  Increase the load metric of the thread.
    adjust_load(1);

    return pe;
}

void zmq::epoll_t::rm_fd(handle_t handle_)
{
    check_thread();
    poll_entry_t *pe = static_cast<poll_entry_t *>(handle_);
    int           rc = epoll_ctl(_epoll_fd, EPOLL_CTL_DEL, pe->fd, &pe->ev);
    errno_assert(rc != -1);
    pe->fd = retired_fd;
    _retired.push_back(pe);

    //  Decrease the load metric of the thread.
    adjust_load(-1);
}

void zmq::epoll_t::set_pollin(handle_t handle_)
{
    check_thread();
    poll_entry_t *pe = static_cast<poll_entry_t *>(handle_);
    pe->ev.events |= EPOLLIN;
    int rc = epoll_ctl(_epoll_fd, EPOLL_CTL_MOD, pe->fd, &pe->ev);
    errno_assert(rc != -1);
}

void zmq::epoll_t::reset_pollin(handle_t handle_)
{
    check_thread();
    poll_entry_t *pe = static_cast<poll_entry_t *>(handle_);
    pe->ev.events &= ~(static_cast<short>(EPOLLIN));
    int rc = epoll_ctl(_epoll_fd, EPOLL_CTL_MOD, pe->fd, &pe->ev);
    errno_assert(rc != -1);
}

void zmq::epoll_t::set_pollout(handle_t handle_)
{
    check_thread();
    poll_entry_t *pe = static_cast<poll_entry_t *>(handle_);
    pe->ev.events |= EPOLLOUT;
    int rc = epoll_ctl(_epoll_fd, EPOLL_CTL_MOD, pe->fd, &pe->ev);
    errno_assert(rc != -1);
}

void zmq::epoll_t::reset_pollout(handle_t handle_)
{
    check_thread();
    poll_entry_t *pe = static_cast<poll_entry_t *>(handle_);
    pe->ev.events &= ~(static_cast<short>(EPOLLOUT));
    int rc = epoll_ctl(_epoll_fd, EPOLL_CTL_MOD, pe->fd, &pe->ev);
    errno_assert(rc != -1);
}

void zmq::epoll_t::stop() { check_thread(); }

int zmq::epoll_t::max_fds() { return -1; }

void zmq::epoll_t::loop()
{
    epoll_event ev_buf[max_io_events];

    while (true)
    {
        //  Execute any due timers.
        int timeout = static_cast<int>(execute_timers());

        if (get_load() == 0)
        {
            if (timeout == 0)
                break;

            // TODO sleep for timeout
            continue;
        }

        //  Wait for events.
        int n = epoll_wait(_epoll_fd, &ev_buf[0], max_io_events, timeout ? timeout : -1);
        if (n == -1)
        {
            errno_assert(errno == EINTR);
            continue;
        }

        for (int i = 0; i < n; i++)
        {
            poll_entry_t *pe = (static_cast<poll_entry_t *>(ev_buf[i].data.ptr));

            if (pe->fd == retired_fd)
                continue;
            if (ev_buf[i].events & (EPOLLERR | EPOLLHUP))
                pe->events->in_event();
            if (pe->fd == retired_fd)
                continue;
            if (ev_buf[i].events & EPOLLOUT)
                pe->events->out_event();
            if (pe->fd == retired_fd)
                continue;
            if (ev_buf[i].events & EPOLLIN)
                pe->events->in_event();
        }

        //  Destroy retired event sources.
        for (retired_t::iterator it = _retired.begin(), end = _retired.end(); it != end; ++it)
        {
            LIBZMQ_DELETE(*it);
        }
        _retired.clear();
    }
}

#endif

//========= end of epoll.cpp ============

//========= begin of err.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "macros.hpp"

const char *zmq::errno_to_string(int errno_)
{
    switch (errno_)
    {
#if defined ZMQ_HAVE_WINDOWS
    case ENOTSUP:
        return "Not supported";
    case EPROTONOSUPPORT:
        return "Protocol not supported";
    case ENOBUFS:
        return "No buffer space available";
    case ENETDOWN:
        return "Network is down";
    case EADDRINUSE:
        return "Address in use";
    case EADDRNOTAVAIL:
        return "Address not available";
    case ECONNREFUSED:
        return "Connection refused";
    case EINPROGRESS:
        return "Operation in progress";
#endif
    case EFSM:
        return "Operation cannot be accomplished in current state";
    case ENOCOMPATPROTO:
        return "The protocol is not compatible with the socket type";
    case ETERM:
        return "Context was terminated";
    case EMTHREAD:
        return "No thread available";
    case EHOSTUNREACH:
        return "Host unreachable";
    default:
#if defined _MSC_VER
#pragma warning(push)
#pragma warning(disable : 4996)
#endif
        return strerror(errno_);
#if defined _MSC_VER
#pragma warning(pop)
#endif
    }
}

void zmq::zmq_abort(const char *errmsg_)
{
#if defined ZMQ_HAVE_WINDOWS

    //  Raise STATUS_FATAL_APP_EXIT.
    ULONG_PTR extra_info[1];
    extra_info[0] = (ULONG_PTR)errmsg_;
    RaiseException(0x40000015, EXCEPTION_NONCONTINUABLE, 1, extra_info);
#else
    LIBZMQ_UNUSED(errmsg_);
    print_backtrace();
    abort();
#endif
}

#ifdef ZMQ_HAVE_WINDOWS

const char *zmq::wsa_error() { return wsa_error_no(WSAGetLastError(), NULL); }

const char *zmq::wsa_error_no(int no_, const char *wsae_wouldblock_string_)
{
    //  TODO:  It seems that list of Windows socket errors is longer than this.
    //         Investigate whether there's a way to convert it into the string
    //         automatically (wsaError->HRESULT->string?).
    switch (no_)
    {
    case WSABASEERR:
        return "No Error";
    case WSAEINTR:
        return "Interrupted system call";
    case WSAEBADF:
        return "Bad file number";
    case WSAEACCES:
        return "Permission denied";
    case WSAEFAULT:
        return "Bad address";
    case WSAEINVAL:
        return "Invalid argument";
    case WSAEMFILE:
        return "Too many open files";
    case WSAEWOULDBLOCK:
        return wsae_wouldblock_string_;
    case WSAEINPROGRESS:
        return "Operation now in progress";
    case WSAEALREADY:
        return "Operation already in progress";
    case WSAENOTSOCK:
        return "Socket operation on non-socket";
    case WSAEDESTADDRREQ:
        return "Destination address required";
    case WSAEMSGSIZE:
        return "Message too long";
    case WSAEPROTOTYPE:
        return "Protocol wrong type for socket";
    case WSAENOPROTOOPT:
        return "Bas protocol option";
    case WSAEPROTONOSUPPORT:
        return "Protocol not supported";
    case WSAESOCKTNOSUPPORT:
        return "Socket type not supported";
    case WSAEOPNOTSUPP:
        return "Operation not supported on socket";
    case WSAEPFNOSUPPORT:
        return "Protocol family not supported";
    case WSAEAFNOSUPPORT:
        return "Address family not supported by protocol family";
    case WSAEADDRINUSE:
        return "Address already in use";
    case WSAEADDRNOTAVAIL:
        return "Can't assign requested address";
    case WSAENETDOWN:
        return "Network is down";
    case WSAENETUNREACH:
        return "Network is unreachable";
    case WSAENETRESET:
        return "Net dropped connection or reset";
    case WSAECONNABORTED:
        return "Software caused connection abort";
    case WSAECONNRESET:
        return "Connection reset by peer";
    case WSAENOBUFS:
        return "No buffer space available";
    case WSAEISCONN:
        return "Socket is already connected";
    case WSAENOTCONN:
        return "Socket is not connected";
    case WSAESHUTDOWN:
        return "Can't send after socket shutdown";
    case WSAETOOMANYREFS:
        return "Too many references can't splice";
    case WSAETIMEDOUT:
        return "Connection timed out";
    case WSAECONNREFUSED:
        return "Connection refused";
    case WSAELOOP:
        return "Too many levels of symbolic links";
    case WSAENAMETOOLONG:
        return "File name too long";
    case WSAEHOSTDOWN:
        return "Host is down";
    case WSAEHOSTUNREACH:
        return "No Route to Host";
    case WSAENOTEMPTY:
        return "Directory not empty";
    case WSAEPROCLIM:
        return "Too many processes";
    case WSAEUSERS:
        return "Too many users";
    case WSAEDQUOT:
        return "Disc Quota Exceeded";
    case WSAESTALE:
        return "Stale NFS file handle";
    case WSAEREMOTE:
        return "Too many levels of remote in path";
    case WSASYSNOTREADY:
        return "Network SubSystem is unavailable";
    case WSAVERNOTSUPPORTED:
        return "WINSOCK DLL Version out of range";
    case WSANOTINITIALISED:
        return "Successful WSASTARTUP not yet performed";
    case WSAHOST_NOT_FOUND:
        return "Host not found";
    case WSATRY_AGAIN:
        return "Non-Authoritative Host not found";
    case WSANO_RECOVERY:
        return "Non-Recoverable errors: FORMERR REFUSED NOTIMP";
    case WSANO_DATA:
        return "Valid name no data record of requested";
    default:
        return "error not defined";
    }
}

void zmq::win_error(char *buffer_, size_t buffer_size_)
{
    DWORD errcode = GetLastError();
#if defined _WIN32_WCE
    DWORD rc = FormatMessageW(FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS, NULL, errcode,
                              MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (LPWSTR)buffer_,
                              buffer_size_ / sizeof(wchar_t), NULL);
#else
    DWORD rc =
        FormatMessageA(FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS, NULL, errcode,
                       MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), buffer_, static_cast<DWORD>(buffer_size_), NULL);
#endif
    zmq_assert(rc);
}

int zmq::wsa_error_to_errno(int errcode_)
{
    switch (errcode_)
    {
        //  10004 - Interrupted system call.
    case WSAEINTR:
        return EINTR;
        //  10009 - File handle is not valid.
    case WSAEBADF:
        return EBADF;
        //  10013 - Permission denied.
    case WSAEACCES:
        return EACCES;
        //  10014 - Bad address.
    case WSAEFAULT:
        return EFAULT;
        //  10022 - Invalid argument.
    case WSAEINVAL:
        return EINVAL;
        //  10024 - Too many open files.
    case WSAEMFILE:
        return EMFILE;
        //  10035 - Operation would block.
    case WSAEWOULDBLOCK:
        return EBUSY;
        //  10036 - Operation now in progress.
    case WSAEINPROGRESS:
        return EAGAIN;
        //  10037 - Operation already in progress.
    case WSAEALREADY:
        return EAGAIN;
        //  10038 - Socket operation on non-socket.
    case WSAENOTSOCK:
        return ENOTSOCK;
        //  10039 - Destination address required.
    case WSAEDESTADDRREQ:
        return EFAULT;
        //  10040 - Message too long.
    case WSAEMSGSIZE:
        return EMSGSIZE;
        //  10041 - Protocol wrong type for socket.
    case WSAEPROTOTYPE:
        return EFAULT;
        //  10042 - Bad protocol option.
    case WSAENOPROTOOPT:
        return EINVAL;
        //  10043 - Protocol not supported.
    case WSAEPROTONOSUPPORT:
        return EPROTONOSUPPORT;
        //  10044 - Socket type not supported.
    case WSAESOCKTNOSUPPORT:
        return EFAULT;
        //  10045 - Operation not supported on socket.
    case WSAEOPNOTSUPP:
        return EFAULT;
        //  10046 - Protocol family not supported.
    case WSAEPFNOSUPPORT:
        return EPROTONOSUPPORT;
        //  10047 - Address family not supported by protocol family.
    case WSAEAFNOSUPPORT:
        return EAFNOSUPPORT;
        //  10048 - Address already in use.
    case WSAEADDRINUSE:
        return EADDRINUSE;
        //  10049 - Cannot assign requested address.
    case WSAEADDRNOTAVAIL:
        return EADDRNOTAVAIL;
        //  10050 - Network is down.
    case WSAENETDOWN:
        return ENETDOWN;
        //  10051 - Network is unreachable.
    case WSAENETUNREACH:
        return ENETUNREACH;
        //  10052 - Network dropped connection on reset.
    case WSAENETRESET:
        return ENETRESET;
        //  10053 - Software caused connection abort.
    case WSAECONNABORTED:
        return ECONNABORTED;
        //  10054 - Connection reset by peer.
    case WSAECONNRESET:
        return ECONNRESET;
        //  10055 - No buffer space available.
    case WSAENOBUFS:
        return ENOBUFS;
        //  10056 - Socket is already connected.
    case WSAEISCONN:
        return EFAULT;
        //  10057 - Socket is not connected.
    case WSAENOTCONN:
        return ENOTCONN;
        //  10058 - Can't send after socket shutdown.
    case WSAESHUTDOWN:
        return EFAULT;
        //  10059 - Too many references can't splice.
    case WSAETOOMANYREFS:
        return EFAULT;
        //  10060 - Connection timed out.
    case WSAETIMEDOUT:
        return ETIMEDOUT;
        //  10061 - Connection refused.
    case WSAECONNREFUSED:
        return ECONNREFUSED;
        //  10062 - Too many levels of symbolic links.
    case WSAELOOP:
        return EFAULT;
        //  10063 - File name too long.
    case WSAENAMETOOLONG:
        return EFAULT;
        //  10064 - Host is down.
    case WSAEHOSTDOWN:
        return EAGAIN;
        //  10065 - No route to host.
    case WSAEHOSTUNREACH:
        return EHOSTUNREACH;
        //  10066 - Directory not empty.
    case WSAENOTEMPTY:
        return EFAULT;
        //  10067 - Too many processes.
    case WSAEPROCLIM:
        return EFAULT;
        //  10068 - Too many users.
    case WSAEUSERS:
        return EFAULT;
        //  10069 - Disc Quota Exceeded.
    case WSAEDQUOT:
        return EFAULT;
        //  10070 - Stale NFS file handle.
    case WSAESTALE:
        return EFAULT;
        //  10071 - Too many levels of remote in path.
    case WSAEREMOTE:
        return EFAULT;
        //  10091 - Network SubSystem is unavailable.
    case WSASYSNOTREADY:
        return EFAULT;
        //  10092 - WINSOCK DLL Version out of range.
    case WSAVERNOTSUPPORTED:
        return EFAULT;
        //  10093 - Successful WSASTARTUP not yet performed.
    case WSANOTINITIALISED:
        return EFAULT;
        //  11001 - Host not found.
    case WSAHOST_NOT_FOUND:
        return EFAULT;
        //  11002 - Non-Authoritative Host not found.
    case WSATRY_AGAIN:
        return EFAULT;
        //  11003 - Non-Recoverable errors: FORMERR REFUSED NOTIMP.
    case WSANO_RECOVERY:
        return EFAULT;
        //  11004 - Valid name no data record of requested.
    case WSANO_DATA:
        return EFAULT;
    default:
        wsa_assert(false);
    }
    //  Not reachable
    return 0;
}

#endif

#if defined(HAVE_LIBUNWIND) && !defined(__SUNPRO_CC)

#define UNW_LOCAL_ONLY
#include <cxxabi.h>
#include <dlfcn.h>
#include <libunwind.h>
// ans ignore: #include "mutex.hpp"

void zmq::print_backtrace(void)
{
    static zmq::mutex_t mtx;
    mtx.lock();
    Dl_info       dl_info;
    unw_cursor_t  cursor;
    unw_context_t ctx;
    unsigned      frame_n = 0;

    unw_getcontext(&ctx);
    unw_init_local(&cursor, &ctx);

    while (unw_step(&cursor) > 0)
    {
        unw_word_t      offset;
        unw_proc_info_t p_info;
        const char *    file_name;
        char *          demangled_name;
        char            func_name[256] = "";
        void *          addr;
        int             rc;

        if (unw_get_proc_info(&cursor, &p_info))
            break;

        rc = unw_get_proc_name(&cursor, func_name, 256, &offset);
        if (rc == -UNW_ENOINFO)
            strcpy(func_name, "?");

        addr = (void *)(p_info.start_ip + offset);

        if (dladdr(addr, &dl_info) && dl_info.dli_fname)
            file_name = dl_info.dli_fname;
        else
            file_name = "?";

        demangled_name = abi::__cxa_demangle(func_name, NULL, NULL, &rc);

        printf("#%u  %p in %s (%s+0x%lx)\n", frame_n++, addr, file_name, rc ? func_name : demangled_name,
               (unsigned long)offset);
        free(demangled_name);
    }
    puts("");

    fflush(stdout);
    mtx.unlock();
}

#else

void zmq::print_backtrace() {}

#endif

//========= end of err.cpp ============

//========= begin of fq.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "fq.hpp"
// ans ignore: #include "pipe.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "msg.hpp"

zmq::fq_t::fq_t() : _active(0), _last_in(NULL), _current(0), _more(false) {}

zmq::fq_t::~fq_t() { zmq_assert(_pipes.empty()); }

void zmq::fq_t::attach(pipe_t *pipe_)
{
    _pipes.push_back(pipe_);
    _pipes.swap(_active, _pipes.size() - 1);
    _active++;
}

void zmq::fq_t::pipe_terminated(pipe_t *pipe_)
{
    const pipes_t::size_type index = _pipes.index(pipe_);

    //  Remove the pipe from the list; adjust number of active pipes
    //  accordingly.
    if (index < _active)
    {
        _active--;
        _pipes.swap(index, _active);
        if (_current == _active)
            _current = 0;
    }
    _pipes.erase(pipe_);

    if (_last_in == pipe_)
    {
        _last_in = NULL;
    }
}

void zmq::fq_t::activated(pipe_t *pipe_)
{
    //  Move the pipe to the list of active pipes.
    _pipes.swap(_pipes.index(pipe_), _active);
    _active++;
}

int zmq::fq_t::recv(msg_t *msg_) { return recvpipe(msg_, NULL); }

int zmq::fq_t::recvpipe(msg_t *msg_, pipe_t **pipe_)
{
    //  Deallocate old content of the message.
    int rc = msg_->close();
    errno_assert(rc == 0);

    //  Round-robin over the pipes to get the next message.
    while (_active > 0)
    {
        //  Try to fetch new message. If we've already read part of the message
        //  subsequent part should be immediately available.
        bool fetched = _pipes[_current]->read(msg_);

        //  Note that when message is not fetched, current pipe is deactivated
        //  and replaced by another active pipe. Thus we don't have to increase
        //  the 'current' pointer.
        if (fetched)
        {
            if (pipe_)
                *pipe_ = _pipes[_current];
            _more = (msg_->flags() & msg_t::more) != 0;
            if (!_more)
            {
                _last_in = _pipes[_current];
                _current = (_current + 1) % _active;
            }
            return 0;
        }

        //  Check the atomicity of the message.
        //  If we've already received the first part of the message
        //  we should get the remaining parts without blocking.
        zmq_assert(!_more);

        _active--;
        _pipes.swap(_current, _active);
        if (_current == _active)
            _current = 0;
    }

    //  No message is available. Initialise the output parameter
    //  to be a 0-byte message.
    rc = msg_->init();
    errno_assert(rc == 0);
    errno = EAGAIN;
    return -1;
}

bool zmq::fq_t::has_in()
{
    //  There are subsequent parts of the partly-read message available.
    if (_more)
        return true;

    //  Note that messing with current doesn't break the fairness of fair
    //  queueing algorithm. If there are no messages available current will
    //  get back to its original value. Otherwise it'll point to the first
    //  pipe holding messages, skipping only pipes with no messages available.
    while (_active > 0)
    {
        if (_pipes[_current]->check_read())
            return true;

        //  Deactivate the pipe.
        _active--;
        _pipes.swap(_current, _active);
        if (_current == _active)
            _current = 0;
    }

    return false;
}

//========= end of fq.cpp ============

//========= begin of gather.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "gather.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "msg.hpp"
// ans ignore: #include "pipe.hpp"

zmq::gather_t::gather_t(class ctx_t *parent_, uint32_t tid_, int sid_) : socket_base_t(parent_, tid_, sid_, true)
{
    options.type = ZMQ_GATHER;
}

zmq::gather_t::~gather_t() {}

void zmq::gather_t::xattach_pipe(pipe_t *pipe_, bool subscribe_to_all_, bool locally_initiated_)
{
    LIBZMQ_UNUSED(subscribe_to_all_);
    LIBZMQ_UNUSED(locally_initiated_);

    zmq_assert(pipe_);
    _fq.attach(pipe_);
}

void zmq::gather_t::xread_activated(pipe_t *pipe_) { _fq.activated(pipe_); }

void zmq::gather_t::xpipe_terminated(pipe_t *pipe_) { _fq.pipe_terminated(pipe_); }

int zmq::gather_t::xrecv(msg_t *msg_)
{
    int rc = _fq.recvpipe(msg_, NULL);

    // Drop any messages with more flag
    while (rc == 0 && msg_->flags() & msg_t::more)
    {
        // drop all frames of the current multi-frame message
        rc = _fq.recvpipe(msg_, NULL);

        while (rc == 0 && msg_->flags() & msg_t::more)
            rc = _fq.recvpipe(msg_, NULL);

        // get the new message
        if (rc == 0)
            rc = _fq.recvpipe(msg_, NULL);
    }

    return rc;
}

bool zmq::gather_t::xhas_in() { return _fq.has_in(); }

//========= end of gather.cpp ============

//========= begin of gssapi_client.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"

#ifdef HAVE_LIBGSSAPI_KRB5

#include <string.h>
#include <string>

// ans ignore: #include "msg.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "gssapi_client.hpp"
// ans ignore: #include "wire.hpp"

zmq::gssapi_client_t::gssapi_client_t(session_base_t *session_, const options_t &options_)
    : mechanism_base_t(session_, options_), gssapi_mechanism_base_t(session_, options_), state(call_next_init),
      token_ptr(GSS_C_NO_BUFFER), mechs(), security_context_established(false)
{
    const std::string::size_type service_size = options_.gss_service_principal.size();
    service_name                              = static_cast<char *>(malloc(service_size + 1));
    assert(service_name);
    memcpy(service_name, options_.gss_service_principal.c_str(), service_size + 1);

    service_name_type = convert_nametype(options_.gss_service_principal_nt);
    maj_stat          = GSS_S_COMPLETE;
    if (!options_.gss_principal.empty())
    {
        const std::string::size_type principal_size = options_.gss_principal.size();
        principal_name                              = static_cast<char *>(malloc(principal_size + 1));
        assert(principal_name);
        memcpy(principal_name, options_.gss_principal.c_str(), principal_size + 1);

        gss_OID name_type = convert_nametype(options_.gss_principal_nt);
        if (acquire_credentials(principal_name, &cred, name_type) != 0)
            maj_stat = GSS_S_FAILURE;
    }

    mechs.elements = NULL;
    mechs.count    = 0;
}

zmq::gssapi_client_t::~gssapi_client_t()
{
    if (service_name)
        free(service_name);
    if (cred)
        gss_release_cred(&min_stat, &cred);
}

int zmq::gssapi_client_t::next_handshake_command(msg_t *msg_)
{
    if (state == send_ready)
    {
        int rc = produce_ready(msg_);
        if (rc == 0)
            state = connected;

        return rc;
    }

    if (state != call_next_init)
    {
        errno = EAGAIN;
        return -1;
    }

    if (initialize_context() < 0)
        return -1;

    if (produce_next_token(msg_) < 0)
        return -1;

    if (maj_stat != GSS_S_CONTINUE_NEEDED && maj_stat != GSS_S_COMPLETE)
        return -1;

    if (maj_stat == GSS_S_COMPLETE)
    {
        security_context_established = true;
        state                        = recv_ready;
    }
    else
        state = recv_next_token;

    return 0;
}

int zmq::gssapi_client_t::process_handshake_command(msg_t *msg_)
{
    if (state == recv_ready)
    {
        int rc = process_ready(msg_);
        if (rc == 0)
            state = send_ready;

        return rc;
    }

    if (state != recv_next_token)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNEXPECTED_COMMAND);
        errno = EPROTO;
        return -1;
    }

    if (process_next_token(msg_) < 0)
        return -1;

    if (maj_stat != GSS_S_COMPLETE && maj_stat != GSS_S_CONTINUE_NEEDED)
        return -1;

    state = call_next_init;

    errno_assert(msg_->close() == 0);
    errno_assert(msg_->init() == 0);

    return 0;
}

int zmq::gssapi_client_t::encode(msg_t *msg_)
{
    zmq_assert(state == connected);

    if (do_encryption)
        return encode_message(msg_);

    return 0;
}

int zmq::gssapi_client_t::decode(msg_t *msg_)
{
    zmq_assert(state == connected);

    if (do_encryption)
        return decode_message(msg_);

    return 0;
}

zmq::mechanism_t::status_t zmq::gssapi_client_t::status() const
{
    return state == connected ? mechanism_t::ready : mechanism_t::handshaking;
}

int zmq::gssapi_client_t::initialize_context()
{
    // principal was specified but credentials could not be acquired
    if (principal_name != NULL && cred == NULL)
        return -1;

    // First time through, import service_name into target_name
    if (target_name == GSS_C_NO_NAME)
    {
        send_tok.value  = service_name;
        send_tok.length = strlen(service_name) + 1;
        OM_uint32 maj   = gss_import_name(&min_stat, &send_tok, service_name_type, &target_name);

        if (maj != GSS_S_COMPLETE)
            return -1;
    }

    maj_stat = gss_init_sec_context(&init_sec_min_stat, cred, &context, target_name, mechs.elements, gss_flags, 0, NULL,
                                    token_ptr, NULL, &send_tok, &ret_flags, NULL);

    if (token_ptr != GSS_C_NO_BUFFER)
        free(recv_tok.value);

    return 0;
}

int zmq::gssapi_client_t::produce_next_token(msg_t *msg_)
{
    if (send_tok.length != 0)
    { // Server expects another token
        if (produce_initiate(msg_, send_tok.value, send_tok.length) < 0)
        {
            gss_release_buffer(&min_stat, &send_tok);
            gss_release_name(&min_stat, &target_name);
            return -1;
        }
    }
    gss_release_buffer(&min_stat, &send_tok);

    if (maj_stat != GSS_S_COMPLETE && maj_stat != GSS_S_CONTINUE_NEEDED)
    {
        gss_release_name(&min_stat, &target_name);
        if (context != GSS_C_NO_CONTEXT)
            gss_delete_sec_context(&min_stat, &context, GSS_C_NO_BUFFER);
        return -1;
    }

    return 0;
}

int zmq::gssapi_client_t::process_next_token(msg_t *msg_)
{
    if (maj_stat == GSS_S_CONTINUE_NEEDED)
    {
        if (process_initiate(msg_, &recv_tok.value, recv_tok.length) < 0)
        {
            gss_release_name(&min_stat, &target_name);
            return -1;
        }
        token_ptr = &recv_tok;
    }

    return 0;
}

#endif

//========= end of gssapi_client.cpp ============

//========= begin of gssapi_mechanism_base.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"

#ifdef HAVE_LIBGSSAPI_KRB5

#include <string.h>
#include <string>

// ans ignore: #include "msg.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "gssapi_mechanism_base.hpp"
// ans ignore: #include "wire.hpp"

zmq::gssapi_mechanism_base_t::gssapi_mechanism_base_t(session_base_t *session_, const options_t &options_)
    : mechanism_base_t(session_, options_), send_tok(), recv_tok(),
      /// FIXME remove? in_buf (),
      target_name(GSS_C_NO_NAME), principal_name(NULL), maj_stat(GSS_S_COMPLETE), min_stat(0), init_sec_min_stat(0),
      ret_flags(0), gss_flags(GSS_C_MUTUAL_FLAG | GSS_C_REPLAY_FLAG), cred(GSS_C_NO_CREDENTIAL),
      context(GSS_C_NO_CONTEXT), do_encryption(!options_.gss_plaintext)
{
}

zmq::gssapi_mechanism_base_t::~gssapi_mechanism_base_t()
{
    if (target_name)
        gss_release_name(&min_stat, &target_name);
    if (context)
        gss_delete_sec_context(&min_stat, &context, GSS_C_NO_BUFFER);
}

int zmq::gssapi_mechanism_base_t::encode_message(msg_t *msg_)
{
    // Wrap the token value
    int             state;
    gss_buffer_desc plaintext;
    gss_buffer_desc wrapped;

    uint8_t flags = 0;
    if (msg_->flags() & msg_t::more)
        flags |= 0x01;
    if (msg_->flags() & msg_t::command)
        flags |= 0x02;

    uint8_t *plaintext_buffer = static_cast<uint8_t *>(malloc(msg_->size() + 1));
    alloc_assert(plaintext_buffer);

    plaintext_buffer[0] = flags;
    memcpy(plaintext_buffer + 1, msg_->data(), msg_->size());

    plaintext.value  = plaintext_buffer;
    plaintext.length = msg_->size() + 1;

    maj_stat = gss_wrap(&min_stat, context, 1, GSS_C_QOP_DEFAULT, &plaintext, &state, &wrapped);

    zmq_assert(maj_stat == GSS_S_COMPLETE);
    zmq_assert(state);

    // Re-initialize msg_ for wrapped text
    int rc = msg_->close();
    zmq_assert(rc == 0);

    rc = msg_->init_size(8 + 4 + wrapped.length);
    zmq_assert(rc == 0);

    uint8_t *ptr = static_cast<uint8_t *>(msg_->data());

    // Add command string
    memcpy(ptr, "\x07MESSAGE", 8);
    ptr += 8;

    // Add token length
    put_uint32(ptr, static_cast<uint32_t>(wrapped.length));
    ptr += 4;

    // Add wrapped token value
    memcpy(ptr, wrapped.value, wrapped.length);
    ptr += wrapped.length;

    gss_release_buffer(&min_stat, &wrapped);

    return 0;
}

int zmq::gssapi_mechanism_base_t::decode_message(msg_t *msg_)
{
    const uint8_t *ptr        = static_cast<uint8_t *>(msg_->data());
    size_t         bytes_left = msg_->size();

    int rc = check_basic_command_structure(msg_);
    if (rc == -1)
        return rc;

    // Get command string
    if (bytes_left < 8 || memcmp(ptr, "\x07MESSAGE", 8))
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNEXPECTED_COMMAND);
        errno = EPROTO;
        return -1;
    }
    ptr += 8;
    bytes_left -= 8;

    // Get token length
    if (bytes_left < 4)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_MESSAGE);
        errno = EPROTO;
        return -1;
    }
    gss_buffer_desc wrapped;
    wrapped.length = get_uint32(ptr);
    ptr += 4;
    bytes_left -= 4;

    // Get token value
    if (bytes_left < wrapped.length)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_MESSAGE);
        errno = EPROTO;
        return -1;
    }
    // TODO: instead of malloc/memcpy, can we just do: wrapped.value = ptr;
    const size_t alloc_length = wrapped.length ? wrapped.length : 1;
    wrapped.value             = static_cast<char *>(malloc(alloc_length));
    alloc_assert(wrapped.value);

    if (wrapped.length)
    {
        memcpy(wrapped.value, ptr, wrapped.length);
        ptr += wrapped.length;
        bytes_left -= wrapped.length;
    }

    // Unwrap the token value
    int             state;
    gss_buffer_desc plaintext;
    maj_stat = gss_unwrap(&min_stat, context, &wrapped, &plaintext, &state, (gss_qop_t *)NULL);

    if (maj_stat != GSS_S_COMPLETE)
    {
        gss_release_buffer(&min_stat, &plaintext);
        free(wrapped.value);
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_CRYPTOGRAPHIC);
        errno = EPROTO;
        return -1;
    }
    zmq_assert(state);

    // Re-initialize msg_ for plaintext
    rc = msg_->close();
    zmq_assert(rc == 0);

    rc = msg_->init_size(plaintext.length - 1);
    zmq_assert(rc == 0);

    const uint8_t flags = static_cast<char *>(plaintext.value)[0];
    if (flags & 0x01)
        msg_->set_flags(msg_t::more);
    if (flags & 0x02)
        msg_->set_flags(msg_t::command);

    memcpy(msg_->data(), static_cast<char *>(plaintext.value) + 1, plaintext.length - 1);

    gss_release_buffer(&min_stat, &plaintext);
    free(wrapped.value);

    if (bytes_left > 0)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_MESSAGE);
        errno = EPROTO;
        return -1;
    }

    return 0;
}

int zmq::gssapi_mechanism_base_t::produce_initiate(msg_t *msg_, void *token_value_, size_t token_length_)
{
    zmq_assert(token_value_);
    zmq_assert(token_length_ <= 0xFFFFFFFFUL);

    const size_t command_size = 9 + 4 + token_length_;

    const int rc = msg_->init_size(command_size);
    errno_assert(rc == 0);

    uint8_t *ptr = static_cast<uint8_t *>(msg_->data());

    // Add command string
    memcpy(ptr, "\x08INITIATE", 9);
    ptr += 9;

    // Add token length
    put_uint32(ptr, static_cast<uint32_t>(token_length_));
    ptr += 4;

    // Add token value
    memcpy(ptr, token_value_, token_length_);
    ptr += token_length_;

    return 0;
}

int zmq::gssapi_mechanism_base_t::process_initiate(msg_t *msg_, void **token_value_, size_t &token_length_)
{
    zmq_assert(token_value_);

    const uint8_t *ptr        = static_cast<uint8_t *>(msg_->data());
    size_t         bytes_left = msg_->size();

    int rc = check_basic_command_structure(msg_);
    if (rc == -1)
        return rc;

    // Get command string
    if (bytes_left < 9 || memcmp(ptr, "\x08INITIATE", 9))
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNEXPECTED_COMMAND);
        errno = EPROTO;
        return -1;
    }
    ptr += 9;
    bytes_left -= 9;

    // Get token length
    if (bytes_left < 4)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_INITIATE);
        errno = EPROTO;
        return -1;
    }
    token_length_ = get_uint32(ptr);
    ptr += 4;
    bytes_left -= 4;

    // Get token value
    if (bytes_left < token_length_)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_INITIATE);
        errno = EPROTO;
        return -1;
    }

    *token_value_ = static_cast<char *>(malloc(token_length_ ? token_length_ : 1));
    alloc_assert(*token_value_);

    if (token_length_)
    {
        memcpy(*token_value_, ptr, token_length_);
        ptr += token_length_;
        bytes_left -= token_length_;
    }

    if (bytes_left > 0)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_INITIATE);
        errno = EPROTO;
        return -1;
    }

    return 0;
}

int zmq::gssapi_mechanism_base_t::produce_ready(msg_t *msg_)
{
    make_command_with_basic_properties(msg_, "\5READY", 6);

    if (do_encryption)
        return encode_message(msg_);

    return 0;
}

int zmq::gssapi_mechanism_base_t::process_ready(msg_t *msg_)
{
    if (do_encryption)
    {
        const int rc = decode_message(msg_);
        if (rc != 0)
            return rc;
    }

    const unsigned char *ptr        = static_cast<unsigned char *>(msg_->data());
    size_t               bytes_left = msg_->size();

    int rc = check_basic_command_structure(msg_);
    if (rc == -1)
        return rc;

    if (bytes_left < 6 || memcmp(ptr, "\x05READY", 6))
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNEXPECTED_COMMAND);
        errno = EPROTO;
        return -1;
    }
    ptr += 6;
    bytes_left -= 6;
    rc = parse_metadata(ptr, bytes_left);
    if (rc == -1)
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_INVALID_METADATA);

    return rc;
}

const gss_OID zmq::gssapi_mechanism_base_t::convert_nametype(int zmq_nametype)
{
    switch (zmq_nametype)
    {
    case ZMQ_GSSAPI_NT_HOSTBASED:
        return GSS_C_NT_HOSTBASED_SERVICE;
    case ZMQ_GSSAPI_NT_USER_NAME:
        return GSS_C_NT_USER_NAME;
    case ZMQ_GSSAPI_NT_KRB5_PRINCIPAL:
#ifdef GSS_KRB5_NT_PRINCIPAL_NAME
        return (gss_OID)GSS_KRB5_NT_PRINCIPAL_NAME;
#else
        return GSS_C_NT_USER_NAME;
#endif
    }
    return NULL;
}

int zmq::gssapi_mechanism_base_t::acquire_credentials(char *service_name_, gss_cred_id_t *cred_, gss_OID name_type_)
{
    OM_uint32  maj_stat;
    OM_uint32  min_stat;
    gss_name_t server_name;

    gss_buffer_desc name_buf;
    name_buf.value  = service_name_;
    name_buf.length = strlen((char *)name_buf.value) + 1;

    maj_stat = gss_import_name(&min_stat, &name_buf, name_type_, &server_name);

    if (maj_stat != GSS_S_COMPLETE)
        return -1;

    maj_stat = gss_acquire_cred(&min_stat, server_name, 0, GSS_C_NO_OID_SET, GSS_C_BOTH, cred_, NULL, NULL);

    if (maj_stat != GSS_S_COMPLETE)
        return -1;

    gss_release_name(&min_stat, &server_name);

    return 0;
}

#endif

//========= end of gssapi_mechanism_base.cpp ============

//========= begin of gssapi_server.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"

#ifdef HAVE_LIBGSSAPI_KRB5

#include <string.h>
#include <string>

// ans ignore: #include "msg.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "gssapi_server.hpp"
// ans ignore: #include "wire.hpp"

#include <gssapi/gssapi.h>

zmq::gssapi_server_t::gssapi_server_t(session_base_t *session_, const std::string &peer_address_,
                                      const options_t &options_)
    : mechanism_base_t(session_, options_), gssapi_mechanism_base_t(session_, options_),
      zap_client_t(session_, peer_address_, options_), session(session_), peer_address(peer_address_),
      state(recv_next_token), security_context_established(false)
{
    maj_stat = GSS_S_CONTINUE_NEEDED;
    if (!options_.gss_principal.empty())
    {
        const std::string::size_type principal_size = options_.gss_principal.size();
        principal_name                              = static_cast<char *>(malloc(principal_size + 1));
        assert(principal_name);
        memcpy(principal_name, options_.gss_principal.c_str(), principal_size + 1);
        gss_OID name_type = convert_nametype(options_.gss_principal_nt);
        if (acquire_credentials(principal_name, &cred, name_type) != 0)
            maj_stat = GSS_S_FAILURE;
    }
}

zmq::gssapi_server_t::~gssapi_server_t()
{
    if (cred)
        gss_release_cred(&min_stat, &cred);

    if (target_name)
        gss_release_name(&min_stat, &target_name);
}

int zmq::gssapi_server_t::next_handshake_command(msg_t *msg_)
{
    if (state == send_ready)
    {
        int rc = produce_ready(msg_);
        if (rc == 0)
            state = recv_ready;

        return rc;
    }

    if (state != send_next_token)
    {
        errno = EAGAIN;
        return -1;
    }

    if (produce_next_token(msg_) < 0)
        return -1;

    if (maj_stat != GSS_S_CONTINUE_NEEDED && maj_stat != GSS_S_COMPLETE)
        return -1;

    if (maj_stat == GSS_S_COMPLETE)
    {
        security_context_established = true;
    }

    state = recv_next_token;

    return 0;
}

int zmq::gssapi_server_t::process_handshake_command(msg_t *msg_)
{
    if (state == recv_ready)
    {
        int rc = process_ready(msg_);
        if (rc == 0)
            state = connected;

        return rc;
    }

    if (state != recv_next_token)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNEXPECTED_COMMAND);
        errno = EPROTO;
        return -1;
    }

    if (security_context_established)
    {
        //  Use ZAP protocol (RFC 27) to authenticate the user.
        //  Note that rc will be -1 only if ZAP is not set up, but if it was
        //  requested and it does not work properly the program will abort.
        bool expecting_zap_reply = false;
        int  rc                  = session->zap_connect();
        if (rc == 0)
        {
            send_zap_request();
            rc = receive_and_process_zap_reply();
            if (rc != 0)
            {
                if (rc == -1)
                    return -1;
                expecting_zap_reply = true;
            }
        }
        state = expecting_zap_reply ? expect_zap_reply : send_ready;
        return 0;
    }

    if (process_next_token(msg_) < 0)
        return -1;

    accept_context();
    state = send_next_token;

    errno_assert(msg_->close() == 0);
    errno_assert(msg_->init() == 0);

    return 0;
}

void zmq::gssapi_server_t::send_zap_request()
{
    gss_buffer_desc principal;
    gss_display_name(&min_stat, target_name, &principal, NULL);
    zap_client_t::send_zap_request("GSSAPI", 6, reinterpret_cast<const uint8_t *>(principal.value), principal.length);

    gss_release_buffer(&min_stat, &principal);
}

int zmq::gssapi_server_t::encode(msg_t *msg_)
{
    zmq_assert(state == connected);

    if (do_encryption)
        return encode_message(msg_);

    return 0;
}

int zmq::gssapi_server_t::decode(msg_t *msg_)
{
    zmq_assert(state == connected);

    if (do_encryption)
        return decode_message(msg_);

    return 0;
}

int zmq::gssapi_server_t::zap_msg_available()
{
    if (state != expect_zap_reply)
    {
        errno = EFSM;
        return -1;
    }
    const int rc = receive_and_process_zap_reply();
    if (rc == 0)
        state = send_ready;
    return rc == -1 ? -1 : 0;
}

zmq::mechanism_t::status_t zmq::gssapi_server_t::status() const
{
    return state == connected ? mechanism_t::ready : mechanism_t::handshaking;
}

int zmq::gssapi_server_t::produce_next_token(msg_t *msg_)
{
    if (send_tok.length != 0)
    { // Client expects another token
        if (produce_initiate(msg_, send_tok.value, send_tok.length) < 0)
            return -1;
        gss_release_buffer(&min_stat, &send_tok);
    }

    if (maj_stat != GSS_S_COMPLETE && maj_stat != GSS_S_CONTINUE_NEEDED)
    {
        gss_release_name(&min_stat, &target_name);
        if (context != GSS_C_NO_CONTEXT)
            gss_delete_sec_context(&min_stat, &context, GSS_C_NO_BUFFER);
        return -1;
    }

    return 0;
}

int zmq::gssapi_server_t::process_next_token(msg_t *msg_)
{
    if (maj_stat == GSS_S_CONTINUE_NEEDED)
    {
        if (process_initiate(msg_, &recv_tok.value, recv_tok.length) < 0)
        {
            if (target_name != GSS_C_NO_NAME)
                gss_release_name(&min_stat, &target_name);
            return -1;
        }
    }

    return 0;
}

void zmq::gssapi_server_t::accept_context()
{
    maj_stat = gss_accept_sec_context(&init_sec_min_stat, &context, cred, &recv_tok, GSS_C_NO_CHANNEL_BINDINGS,
                                      &target_name, &doid, &send_tok, &ret_flags, NULL, NULL);

    if (recv_tok.value)
    {
        free(recv_tok.value);
        recv_tok.value = NULL;
    }
}

#endif

//========= end of gssapi_server.cpp ============

//========= begin of io_object.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "io_object.hpp"
// ans ignore: #include "io_thread.hpp"
// ans ignore: #include "err.hpp"

zmq::io_object_t::io_object_t(io_thread_t *io_thread_) : _poller(NULL)
{
    if (io_thread_)
        plug(io_thread_);
}

zmq::io_object_t::~io_object_t() {}

void zmq::io_object_t::plug(io_thread_t *io_thread_)
{
    zmq_assert(io_thread_);
    zmq_assert(!_poller);

    //  Retrieve the poller from the thread we are running in.
    _poller = io_thread_->get_poller();
}

void zmq::io_object_t::unplug()
{
    zmq_assert(_poller);

    //  Forget about old poller in preparation to be migrated
    //  to a different I/O thread.
    _poller = NULL;
}

zmq::io_object_t::handle_t zmq::io_object_t::add_fd(fd_t fd_) { return _poller->add_fd(fd_, this); }

void zmq::io_object_t::rm_fd(handle_t handle_) { _poller->rm_fd(handle_); }

void zmq::io_object_t::set_pollin(handle_t handle_) { _poller->set_pollin(handle_); }

void zmq::io_object_t::reset_pollin(handle_t handle_) { _poller->reset_pollin(handle_); }

void zmq::io_object_t::set_pollout(handle_t handle_) { _poller->set_pollout(handle_); }

void zmq::io_object_t::reset_pollout(handle_t handle_) { _poller->reset_pollout(handle_); }

void zmq::io_object_t::add_timer(int timeout_, int id_) { _poller->add_timer(timeout_, this, id_); }

void zmq::io_object_t::cancel_timer(int id_) { _poller->cancel_timer(this, id_); }

void zmq::io_object_t::in_event() { zmq_assert(false); }

void zmq::io_object_t::out_event() { zmq_assert(false); }

void zmq::io_object_t::timer_event(int) { zmq_assert(false); }

//========= end of io_object.cpp ============

//========= begin of io_thread.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"

#include <new>

// ans ignore: #include "macros.hpp"
// ans ignore: #include "io_thread.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "ctx.hpp"

zmq::io_thread_t::io_thread_t(ctx_t *ctx_, uint32_t tid_)
    : object_t(ctx_, tid_), _mailbox_handle(static_cast<poller_t::handle_t>(NULL))
{
    _poller = new (std::nothrow) poller_t(*ctx_);
    alloc_assert(_poller);

    if (_mailbox.get_fd() != retired_fd)
    {
        _mailbox_handle = _poller->add_fd(_mailbox.get_fd(), this);
        _poller->set_pollin(_mailbox_handle);
    }
}

zmq::io_thread_t::~io_thread_t() { LIBZMQ_DELETE(_poller); }

void zmq::io_thread_t::start()
{
    char name[16] = "";
    snprintf(name, sizeof(name), "IO/%u", get_tid() - zmq::ctx_t::reaper_tid - 1);
    //  Start the underlying I/O thread.
    _poller->start(name);
}

void zmq::io_thread_t::stop() { send_stop(); }

zmq::mailbox_t *zmq::io_thread_t::get_mailbox() { return &_mailbox; }

int zmq::io_thread_t::get_load() { return _poller->get_load(); }

void zmq::io_thread_t::in_event()
{
    //  TODO: Do we want to limit number of commands I/O thread can
    //  process in a single go?

    command_t cmd;
    int       rc = _mailbox.recv(&cmd, 0);

    while (rc == 0 || errno == EINTR)
    {
        if (rc == 0)
            cmd.destination->process_command(cmd);
        rc = _mailbox.recv(&cmd, 0);
    }

    errno_assert(rc != 0 && errno == EAGAIN);
}

void zmq::io_thread_t::out_event()
{
    //  We are never polling for POLLOUT here. This function is never called.
    zmq_assert(false);
}

void zmq::io_thread_t::timer_event(int)
{
    //  No timers here. This function is never called.
    zmq_assert(false);
}

zmq::poller_t *zmq::io_thread_t::get_poller()
{
    zmq_assert(_poller);
    return _poller;
}

void zmq::io_thread_t::process_stop()
{
    zmq_assert(_mailbox_handle);
    _poller->rm_fd(_mailbox_handle);
    _poller->stop();
}

//========= end of io_thread.cpp ============

//========= begin of ip.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "ip.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "config.hpp"
// ans ignore: #include "address.hpp"

#if !defined ZMQ_HAVE_WINDOWS
#include <fcntl.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <sys/socket.h>
#include <sys/types.h>
#else
// ans ignore: #include "tcp.hpp"
#endif

#if defined ZMQ_HAVE_OPENVMS || defined ZMQ_HAVE_VXWORKS
#include <ioctl.h>
#endif

#if defined ZMQ_HAVE_VXWORKS
#include <ioLib.h>
#include <sockLib.h>
#include <unistd.h>
#endif

#if defined ZMQ_HAVE_EVENTFD
#include <sys/eventfd.h>
#endif

#if defined ZMQ_HAVE_OPENPGM
#ifdef ZMQ_HAVE_WINDOWS
#define __PGM_WININT_H__
#endif

#include <pgm/pgm.h>
#endif

#ifdef __APPLE__
#include <TargetConditionals.h>
#endif

zmq::fd_t zmq::open_socket(int domain_, int type_, int protocol_)
{
    int rc;

    //  Setting this option result in sane behaviour when exec() functions
    //  are used. Old sockets are closed and don't block TCP ports etc.
#if defined ZMQ_HAVE_SOCK_CLOEXEC
    type_ |= SOCK_CLOEXEC;
#endif

#if defined ZMQ_HAVE_WINDOWS && defined WSA_FLAG_NO_HANDLE_INHERIT
    // if supported, create socket with WSA_FLAG_NO_HANDLE_INHERIT, such that
    // the race condition in making it non-inheritable later is avoided
    const fd_t s = WSASocket(domain_, type_, protocol_, NULL, 0, WSA_FLAG_OVERLAPPED || WSA_FLAG_NO_HANDLE_INHERIT);
#else
    const fd_t s        = socket(domain_, type_, protocol_);
#endif
    if (s == retired_fd)
    {
#ifdef ZMQ_HAVE_WINDOWS
        errno = wsa_error_to_errno(WSAGetLastError());
#endif
        return retired_fd;
    }

    make_socket_noninheritable(s);

    //  Socket is not yet connected so EINVAL is not a valid networking error
    rc = zmq::set_nosigpipe(s);
    errno_assert(rc == 0);

    return s;
}

void zmq::unblock_socket(fd_t s_)
{
#if defined ZMQ_HAVE_WINDOWS
    u_long nonblock = 1;
    int    rc       = ioctlsocket(s_, FIONBIO, &nonblock);
    wsa_assert(rc != SOCKET_ERROR);
#elif defined ZMQ_HAVE_OPENVMS || defined ZMQ_HAVE_VXWORKS
    int        nonblock = 1;
    int        rc       = ioctl(s_, FIONBIO, &nonblock);
    errno_assert(rc != -1);
#else
    int flags = fcntl(s_, F_GETFL, 0);
    if (flags == -1)
        flags = 0;
    int rc = fcntl(s_, F_SETFL, flags | O_NONBLOCK);
    errno_assert(rc != -1);
#endif
}

void zmq::enable_ipv4_mapping(fd_t s_)
{
    LIBZMQ_UNUSED(s_);

#if defined IPV6_V6ONLY && !defined ZMQ_HAVE_OPENBSD
#ifdef ZMQ_HAVE_WINDOWS
    DWORD flag = 0;
#else
    int flag = 0;
#endif
    int rc = setsockopt(s_, IPPROTO_IPV6, IPV6_V6ONLY, reinterpret_cast<char *>(&flag), sizeof(flag));
#ifdef ZMQ_HAVE_WINDOWS
    wsa_assert(rc != SOCKET_ERROR);
#else
    errno_assert(rc == 0);
#endif
#endif
}

int zmq::get_peer_ip_address(fd_t sockfd_, std::string &ip_addr_)
{
    struct sockaddr_storage ss;

    zmq_socklen_t addrlen = get_socket_address(sockfd_, socket_end_remote, &ss);

    if (addrlen == 0)
    {
#ifdef ZMQ_HAVE_WINDOWS
        const int last_error = WSAGetLastError();
        wsa_assert(last_error != WSANOTINITIALISED && last_error != WSAEFAULT && last_error != WSAEINPROGRESS &&
                   last_error != WSAENOTSOCK);
#elif !defined(TARGET_OS_IPHONE) || !TARGET_OS_IPHONE
        errno_assert(errno != EBADF && errno != EFAULT && errno != ENOTSOCK);
#else
        errno_assert(errno != EFAULT && errno != ENOTSOCK);
#endif
        return 0;
    }

    char host[NI_MAXHOST];
    int rc = getnameinfo(reinterpret_cast<struct sockaddr *>(&ss), addrlen, host, sizeof host, NULL, 0, NI_NUMERICHOST);
    if (rc != 0)
        return 0;

    ip_addr_ = host;

    union
    {
        struct sockaddr         sa;
        struct sockaddr_storage sa_stor;
    } u;

    u.sa_stor = ss;
    return static_cast<int>(u.sa.sa_family);
}

void zmq::set_ip_type_of_service(fd_t s_, int iptos_)
{
    int rc = setsockopt(s_, IPPROTO_IP, IP_TOS, reinterpret_cast<char *>(&iptos_), sizeof(iptos_));

#ifdef ZMQ_HAVE_WINDOWS
    wsa_assert(rc != SOCKET_ERROR);
#else
    errno_assert(rc == 0);
#endif

    //  Windows and Hurd do not support IPV6_TCLASS
#if !defined(ZMQ_HAVE_WINDOWS) && defined(IPV6_TCLASS)
    rc = setsockopt(s_, IPPROTO_IPV6, IPV6_TCLASS, reinterpret_cast<char *>(&iptos_), sizeof(iptos_));

    //  If IPv6 is not enabled ENOPROTOOPT will be returned on Linux and
    //  EINVAL on OSX
    if (rc == -1)
    {
        errno_assert(errno == ENOPROTOOPT || errno == EINVAL);
    }
#endif
}

int zmq::set_nosigpipe(fd_t s_)
{
#ifdef SO_NOSIGPIPE
    //  Make sure that SIGPIPE signal is not generated when writing to a
    //  connection that was already closed by the peer.
    //  As per POSIX spec, EINVAL will be returned if the socket was valid but
    //  the connection has been reset by the peer. Return an error so that the
    //  socket can be closed and the connection retried if necessary.
    int set = 1;
    int rc  = setsockopt(s_, SOL_SOCKET, SO_NOSIGPIPE, &set, sizeof(int));
    if (rc != 0 && errno == EINVAL)
        return -1;
    errno_assert(rc == 0);
#else
    LIBZMQ_UNUSED(s_);
#endif

    return 0;
}

int zmq::bind_to_device(fd_t s_, const std::string &bound_device_)
{
#ifdef ZMQ_HAVE_SO_BINDTODEVICE
    int rc = setsockopt(s_, SOL_SOCKET, SO_BINDTODEVICE, bound_device_.c_str(), bound_device_.length());

#ifdef ZMQ_HAVE_WINDOWS
    if (rc != SOCKET_ERROR)
        return 0;
    const int lastError = WSAGetLastError();
    errno               = wsa_error_to_errno(lastError);
    wsa_assert(lastError != WSAENOTSOCK);
    return -1;
#else
    if (rc == 0)
        return 0;
    errno_assert(errno != ENOTSOCK);
    return -1;
#endif
#else
    LIBZMQ_UNUSED(s_);
    LIBZMQ_UNUSED(bound_device_);

    errno = ENOTSUP;
    return -1;
#endif
}

bool zmq::initialize_network()
{
#if defined ZMQ_HAVE_OPENPGM

    //  Init PGM transport. Ensure threading and timer are enabled. Find PGM
    //  protocol ID. Note that if you want to use gettimeofday and sleep for
    //  openPGM timing, set environment variables PGM_TIMER to "GTOD" and
    //  PGM_SLEEP to "USLEEP".
    pgm_error_t *pgm_error = NULL;
    const bool   ok        = pgm_init(&pgm_error);
    if (ok != TRUE)
    {
        //  Invalid parameters don't set pgm_error_t
        zmq_assert(pgm_error != NULL);
        if (pgm_error->domain == PGM_ERROR_DOMAIN_TIME && (pgm_error->code == PGM_ERROR_FAILED))
        {
            //  Failed to access RTC or HPET device.
            pgm_error_free(pgm_error);
            errno = EINVAL;
            return false;
        }

        //  PGM_ERROR_DOMAIN_ENGINE: WSAStartup errors or missing WSARecvMsg.
        zmq_assert(false);
    }
#endif

#ifdef ZMQ_HAVE_WINDOWS
    //  Intialise Windows sockets. Note that WSAStartup can be called multiple
    //  times given that WSACleanup will be called for each WSAStartup.

    WORD    version_requested = MAKEWORD(2, 2);
    WSADATA wsa_data;
    int     rc = WSAStartup(version_requested, &wsa_data);
    zmq_assert(rc == 0);
    zmq_assert(LOBYTE(wsa_data.wVersion) == 2 && HIBYTE(wsa_data.wVersion) == 2);
#endif

    return true;
}

void zmq::shutdown_network()
{
#ifdef ZMQ_HAVE_WINDOWS
    //  On Windows, uninitialise socket layer.
    int rc = WSACleanup();
    wsa_assert(rc != SOCKET_ERROR);
#endif

#if defined ZMQ_HAVE_OPENPGM
    //  Shut down the OpenPGM library.
    if (pgm_shutdown() != TRUE)
        zmq_assert(false);
#endif
}

#if defined ZMQ_HAVE_WINDOWS
static void tune_socket(const SOCKET socket_)
{
    BOOL tcp_nodelay = 1;
    int  rc = setsockopt(socket_, IPPROTO_TCP, TCP_NODELAY, reinterpret_cast<char *>(&tcp_nodelay), sizeof tcp_nodelay);
    wsa_assert(rc != SOCKET_ERROR);

    zmq::tcp_tune_loopback_fast_path(socket_);
}
#endif

int zmq::make_fdpair(fd_t *r_, fd_t *w_)
{
#if defined ZMQ_HAVE_EVENTFD
    int flags = 0;
#if defined ZMQ_HAVE_EVENTFD_CLOEXEC
    //  Setting this option result in sane behaviour when exec() functions
    //  are used. Old sockets are closed and don't block TCP ports, avoid
    //  leaks, etc.
    flags |= EFD_CLOEXEC;
#endif
    fd_t fd = eventfd(0, flags);
    if (fd == -1)
    {
        errno_assert(errno == ENFILE || errno == EMFILE);
        *w_ = *r_ = -1;
        return -1;
    }
    else
    {
        *w_ = *r_ = fd;
        return 0;
    }

#elif defined ZMQ_HAVE_WINDOWS
#if !defined _WIN32_WCE && !defined ZMQ_HAVE_WINDOWS_UWP
    //  Windows CE does not manage security attributes
    SECURITY_DESCRIPTOR sd;
    SECURITY_ATTRIBUTES sa;
    memset(&sd, 0, sizeof sd);
    memset(&sa, 0, sizeof sa);

    InitializeSecurityDescriptor(&sd, SECURITY_DESCRIPTOR_REVISION);
    SetSecurityDescriptorDacl(&sd, TRUE, 0, FALSE);

    sa.nLength              = sizeof(SECURITY_ATTRIBUTES);
    sa.lpSecurityDescriptor = &sd;
#endif

    //  This function has to be in a system-wide critical section so that
    //  two instances of the library don't accidentally create signaler
    //  crossing the process boundary.
    //  We'll use named event object to implement the critical section.
    //  Note that if the event object already exists, the CreateEvent requests
    //  EVENT_ALL_ACCESS access right. If this fails, we try to open
    //  the event object asking for SYNCHRONIZE access only.
    HANDLE sync = NULL;

    //  Create critical section only if using fixed signaler port
    //  Use problematic Event implementation for compatibility if using old port 5905.
    //  Otherwise use Mutex implementation.
    int event_signaler_port = 5905;

    if (signaler_port == event_signaler_port)
    {
#if !defined _WIN32_WCE && !defined ZMQ_HAVE_WINDOWS_UWP
        sync = CreateEventW(&sa, FALSE, TRUE, L"Global\\zmq-signaler-port-sync");
#else
        sync = CreateEventW(NULL, FALSE, TRUE, L"Global\\zmq-signaler-port-sync");
#endif
        if (sync == NULL && GetLastError() == ERROR_ACCESS_DENIED)
            sync = OpenEventW(SYNCHRONIZE | EVENT_MODIFY_STATE, FALSE, L"Global\\zmq-signaler-port-sync");

        win_assert(sync != NULL);
    }
    else if (signaler_port != 0)
    {
        wchar_t mutex_name[MAX_PATH];
#ifdef __MINGW32__
        _snwprintf(mutex_name, MAX_PATH, L"Global\\zmq-signaler-port-%d", signaler_port);
#else
        swprintf(mutex_name, MAX_PATH, L"Global\\zmq-signaler-port-%d", signaler_port);
#endif

#if !defined _WIN32_WCE && !defined ZMQ_HAVE_WINDOWS_UWP
        sync = CreateMutexW(&sa, FALSE, mutex_name);
#else
        sync = CreateMutexW(NULL, FALSE, mutex_name);
#endif
        if (sync == NULL && GetLastError() == ERROR_ACCESS_DENIED)
            sync = OpenMutexW(SYNCHRONIZE, FALSE, mutex_name);

        win_assert(sync != NULL);
    }

    //  Windows has no 'socketpair' function. CreatePipe is no good as pipe
    //  handles cannot be polled on. Here we create the socketpair by hand.
    *w_ = INVALID_SOCKET;
    *r_ = INVALID_SOCKET;

    //  Create listening socket.
    SOCKET listener;
    listener = open_socket(AF_INET, SOCK_STREAM, 0);
    wsa_assert(listener != INVALID_SOCKET);

    //  Set SO_REUSEADDR and TCP_NODELAY on listening socket.
    BOOL so_reuseaddr = 1;
    int  rc =
        setsockopt(listener, SOL_SOCKET, SO_REUSEADDR, reinterpret_cast<char *>(&so_reuseaddr), sizeof so_reuseaddr);
    wsa_assert(rc != SOCKET_ERROR);

    tune_socket(listener);

    //  Init sockaddr to signaler port.
    struct sockaddr_in addr;
    memset(&addr, 0, sizeof addr);
    addr.sin_family      = AF_INET;
    addr.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
    addr.sin_port        = htons(signaler_port);

    //  Create the writer socket.
    *w_ = open_socket(AF_INET, SOCK_STREAM, 0);
    wsa_assert(*w_ != INVALID_SOCKET);

    if (sync != NULL)
    {
        //  Enter the critical section.
        DWORD dwrc = WaitForSingleObject(sync, INFINITE);
        zmq_assert(dwrc == WAIT_OBJECT_0 || dwrc == WAIT_ABANDONED);
    }

    //  Bind listening socket to signaler port.
    rc = bind(listener, reinterpret_cast<const struct sockaddr *>(&addr), sizeof addr);

    if (rc != SOCKET_ERROR && signaler_port == 0)
    {
        //  Retrieve ephemeral port number
        int addrlen = sizeof addr;
        rc          = getsockname(listener, reinterpret_cast<struct sockaddr *>(&addr), &addrlen);
    }

    //  Listen for incoming connections.
    if (rc != SOCKET_ERROR)
        rc = listen(listener, 1);

    //  Connect writer to the listener.
    if (rc != SOCKET_ERROR)
        rc = connect(*w_, reinterpret_cast<struct sockaddr *>(&addr), sizeof addr);

    //  Set TCP_NODELAY on writer socket.
    tune_socket(*w_);

    //  Accept connection from writer.
    if (rc != SOCKET_ERROR)
        *r_ = accept(listener, NULL, NULL);

    //  Send/receive large chunk to work around TCP slow start
    //  This code is a workaround for #1608
    if (*r_ != INVALID_SOCKET)
    {
        size_t         dummy_size = 1024 * 1024; //  1M to overload default receive buffer
        unsigned char *dummy      = static_cast<unsigned char *>(malloc(dummy_size));
        wsa_assert(dummy);

        int still_to_send = static_cast<int>(dummy_size);
        int still_to_recv = static_cast<int>(dummy_size);
        while (still_to_send || still_to_recv)
        {
            int nbytes;
            if (still_to_send > 0)
            {
                nbytes = ::send(*w_, reinterpret_cast<char *>(dummy + dummy_size - still_to_send), still_to_send, 0);
                wsa_assert(nbytes != SOCKET_ERROR);
                still_to_send -= nbytes;
            }
            nbytes = ::recv(*r_, reinterpret_cast<char *>(dummy + dummy_size - still_to_recv), still_to_recv, 0);
            wsa_assert(nbytes != SOCKET_ERROR);
            still_to_recv -= nbytes;
        }
        free(dummy);
    }

    //  Save errno if error occurred in bind/listen/connect/accept.
    int saved_errno = 0;
    if (*r_ == INVALID_SOCKET)
        saved_errno = WSAGetLastError();

    //  We don't need the listening socket anymore. Close it.
    rc = closesocket(listener);
    wsa_assert(rc != SOCKET_ERROR);

    if (sync != NULL)
    {
        //  Exit the critical section.
        BOOL brc;
        if (signaler_port == event_signaler_port)
            brc = SetEvent(sync);
        else
            brc = ReleaseMutex(sync);
        win_assert(brc != 0);

        //  Release the kernel object
        brc = CloseHandle(sync);
        win_assert(brc != 0);
    }

    if (*r_ != INVALID_SOCKET)
    {
        make_socket_noninheritable(*r_);
        return 0;
    }
    //  Cleanup writer if connection failed
    if (*w_ != INVALID_SOCKET)
    {
        rc = closesocket(*w_);
        wsa_assert(rc != SOCKET_ERROR);
        *w_ = INVALID_SOCKET;
    }
    //  Set errno from saved value
    errno = wsa_error_to_errno(saved_errno);
    return -1;

#elif defined ZMQ_HAVE_OPENVMS

    //  Whilst OpenVMS supports socketpair - it maps to AF_INET only.  Further,
    //  it does not set the socket options TCP_NODELAY and TCP_NODELACK which
    //  can lead to performance problems.
    //
    //  The bug will be fixed in V5.6 ECO4 and beyond.  In the meantime, we'll
    //  create the socket pair manually.
    struct sockaddr_in lcladdr;
    memset(&lcladdr, 0, sizeof lcladdr);
    lcladdr.sin_family      = AF_INET;
    lcladdr.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
    lcladdr.sin_port        = 0;

    int listener = open_socket(AF_INET, SOCK_STREAM, 0);
    errno_assert(listener != -1);

    int on = 1;
    int rc = setsockopt(listener, IPPROTO_TCP, TCP_NODELAY, &on, sizeof on);
    errno_assert(rc != -1);

    rc = setsockopt(listener, IPPROTO_TCP, TCP_NODELACK, &on, sizeof on);
    errno_assert(rc != -1);

    rc = bind(listener, (struct sockaddr *)&lcladdr, sizeof lcladdr);
    errno_assert(rc != -1);

    socklen_t lcladdr_len = sizeof lcladdr;

    rc = getsockname(listener, (struct sockaddr *)&lcladdr, &lcladdr_len);
    errno_assert(rc != -1);

    rc = listen(listener, 1);
    errno_assert(rc != -1);

    *w_ = open_socket(AF_INET, SOCK_STREAM, 0);
    errno_assert(*w_ != -1);

    rc = setsockopt(*w_, IPPROTO_TCP, TCP_NODELAY, &on, sizeof on);
    errno_assert(rc != -1);

    rc = setsockopt(*w_, IPPROTO_TCP, TCP_NODELACK, &on, sizeof on);
    errno_assert(rc != -1);

    rc = connect(*w_, (struct sockaddr *)&lcladdr, sizeof lcladdr);
    errno_assert(rc != -1);

    *r_ = accept(listener, NULL, NULL);
    errno_assert(*r_ != -1);

    close(listener);

    return 0;
#elif defined ZMQ_HAVE_VXWORKS
    struct sockaddr_in lcladdr;
    memset(&lcladdr, 0, sizeof lcladdr);
    lcladdr.sin_family      = AF_INET;
    lcladdr.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
    lcladdr.sin_port        = 0;

    int listener = open_socket(AF_INET, SOCK_STREAM, 0);
    errno_assert(listener != -1);

    int on = 1;
    int rc = setsockopt(listener, IPPROTO_TCP, TCP_NODELAY, (char *)&on, sizeof on);
    errno_assert(rc != -1);

    rc = bind(listener, (struct sockaddr *)&lcladdr, sizeof lcladdr);
    errno_assert(rc != -1);

    socklen_t lcladdr_len = sizeof lcladdr;

    rc = getsockname(listener, (struct sockaddr *)&lcladdr, (int *)&lcladdr_len);
    errno_assert(rc != -1);

    rc = listen(listener, 1);
    errno_assert(rc != -1);

    *w_ = open_socket(AF_INET, SOCK_STREAM, 0);
    errno_assert(*w_ != -1);

    rc = setsockopt(*w_, IPPROTO_TCP, TCP_NODELAY, (char *)&on, sizeof on);
    errno_assert(rc != -1);

    rc = connect(*w_, (struct sockaddr *)&lcladdr, sizeof lcladdr);
    errno_assert(rc != -1);

    *r_ = accept(listener, NULL, NULL);
    errno_assert(*r_ != -1);

    close(listener);

    return 0;
#else
    // All other implementations support socketpair()
    int sv[2];
    int type = SOCK_STREAM;
    //  Setting this option result in sane behaviour when exec() functions
    //  are used. Old sockets are closed and don't block TCP ports, avoid
    //  leaks, etc.
#if defined ZMQ_HAVE_SOCK_CLOEXEC
    type |= SOCK_CLOEXEC;
#endif
    int rc = socketpair(AF_UNIX, type, 0, sv);
    if (rc == -1)
    {
        errno_assert(errno == ENFILE || errno == EMFILE);
        *w_ = *r_ = -1;
        return -1;
    }
    else
    {
        make_socket_noninheritable(sv[0]);
        make_socket_noninheritable(sv[1]);

        *w_ = sv[0];
        *r_ = sv[1];
        return 0;
    }
#endif
}

void zmq::make_socket_noninheritable(fd_t sock_)
{
#if defined ZMQ_HAVE_WINDOWS && !defined _WIN32_WCE && !defined ZMQ_HAVE_WINDOWS_UWP
    //  On Windows, preventing sockets to be inherited by child processes.
    const BOOL brc = SetHandleInformation(reinterpret_cast<HANDLE>(sock_), HANDLE_FLAG_INHERIT, 0);
    win_assert(brc);
#elif (!defined ZMQ_HAVE_SOCK_CLOEXEC || !defined HAVE_ACCEPT4) && defined FD_CLOEXEC
    //  If there 's no SOCK_CLOEXEC, let's try the second best option.
    //  Race condition can cause socket not to be closed (if fork happens
    //  between accept and this point).
    const int rc = fcntl(sock_, F_SETFD, FD_CLOEXEC);
    errno_assert(rc != -1);
#else
    LIBZMQ_UNUSED(sock_);
#endif
}

//========= end of ip.cpp ============

//========= begin of ipc_address.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "ipc_address.hpp"

#if !defined ZMQ_HAVE_WINDOWS && !defined ZMQ_HAVE_OPENVMS && !defined ZMQ_HAVE_VXWORKS

// ans ignore: #include "err.hpp"

#include <string>

#ifndef HAVE_STRNLEN
static size_t strnlen(const char *s, size_t len)
{
    for (size_t i = 0; i < len; i++)
    {
        if (s[i] == '\0')
            return i + 1;
    }

    return len;
}
#endif

zmq::ipc_address_t::ipc_address_t() { memset(&_address, 0, sizeof _address); }

zmq::ipc_address_t::ipc_address_t(const sockaddr *sa_, socklen_t sa_len_) : _addrlen(sa_len_)
{
    zmq_assert(sa_ && sa_len_ > 0);

    memset(&_address, 0, sizeof _address);
    if (sa_->sa_family == AF_UNIX)
        memcpy(&_address, sa_, sa_len_);
}

zmq::ipc_address_t::~ipc_address_t() {}

int zmq::ipc_address_t::resolve(const char *path_)
{
    const size_t path_len = strlen(path_);
    if (path_len >= sizeof _address.sun_path)
    {
        errno = ENAMETOOLONG;
        return -1;
    }
    if (path_[0] == '@' && !path_[1])
    {
        errno = EINVAL;
        return -1;
    }

    _address.sun_family = AF_UNIX;
    memcpy(_address.sun_path, path_, path_len + 1);
    /* Abstract sockets start with '\0' */
    if (path_[0] == '@')
        *_address.sun_path = '\0';

    _addrlen = offsetof(sockaddr_un, sun_path) + path_len;
    return 0;
}

int zmq::ipc_address_t::to_string(std::string &addr_) const
{
    if (_address.sun_family != AF_UNIX)
    {
        addr_.clear();
        return -1;
    }

    const char prefix[] = "ipc://";
    char       buf[sizeof prefix + sizeof _address.sun_path];
    char *     pos = buf;
    memcpy(pos, prefix, sizeof prefix - 1);
    pos += sizeof prefix - 1;
    const char *src_pos = _address.sun_path;
    if (!_address.sun_path[0] && _address.sun_path[1])
    {
        *pos++ = '@';
        src_pos++;
    }
    // according to http://man7.org/linux/man-pages/man7/unix.7.html, NOTES
    // section, address.sun_path might not always be null-terminated; therefore,
    // we calculate the length based of addrlen
    const size_t src_len = strnlen(src_pos, _addrlen - offsetof(sockaddr_un, sun_path) - (src_pos - _address.sun_path));
    memcpy(pos, src_pos, src_len);
    addr_.assign(buf, pos - buf + src_len);
    return 0;
}

const sockaddr *zmq::ipc_address_t::addr() const { return reinterpret_cast<const sockaddr *>(&_address); }

socklen_t zmq::ipc_address_t::addrlen() const { return _addrlen; }

#endif

//========= end of ipc_address.cpp ============

//========= begin of ipc_connecter.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "ipc_connecter.hpp"

#if !defined ZMQ_HAVE_WINDOWS && !defined ZMQ_HAVE_OPENVMS && !defined ZMQ_HAVE_VXWORKS

#include <new>
#include <string>

// ans ignore: #include "stream_engine.hpp"
// ans ignore: #include "io_thread.hpp"
// ans ignore: #include "random.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "ip.hpp"
// ans ignore: #include "address.hpp"
// ans ignore: #include "ipc_address.hpp"
// ans ignore: #include "session_base.hpp"

#include <sys/socket.h>
#include <sys/types.h>
#include <sys/un.h>
#include <unistd.h>

zmq::ipc_connecter_t::ipc_connecter_t(class io_thread_t *io_thread_, class session_base_t *session_,
                                      const options_t &options_, address_t *addr_, bool delayed_start_)
    : stream_connecter_base_t(io_thread_, session_, options_, addr_, delayed_start_)
{
    zmq_assert(_addr->protocol == protocol_name::ipc);
}

void zmq::ipc_connecter_t::out_event()
{
    fd_t fd = connect();
    rm_handle();

    //  Handle the error condition by attempt to reconnect.
    if (fd == retired_fd)
    {
        close();
        add_reconnect_timer();
        return;
    }

    create_engine(fd, get_socket_name<ipc_address_t>(fd, socket_end_local));
}

void zmq::ipc_connecter_t::start_connecting()
{
    //  Open the connecting socket.
    int rc = open();

    //  Connect may succeed in synchronous manner.
    if (rc == 0)
    {
        _handle = add_fd(_s);
        out_event();
    }

    //  Connection establishment may be delayed. Poll for its completion.
    else if (rc == -1 && errno == EINPROGRESS)
    {
        _handle = add_fd(_s);
        set_pollout(_handle);
        _socket->event_connect_delayed(make_unconnected_connect_endpoint_pair(_endpoint), zmq_errno());

        // TODO, tcp_connecter_t adds a connect timer in this case; maybe this
        // should be done here as well (and then this could be pulled up to
        // stream_connecter_base_t).
    }

    //  Handle any other error condition by eventual reconnect.
    else
    {
        if (_s != retired_fd)
            close();
        add_reconnect_timer();
    }
}

int zmq::ipc_connecter_t::open()
{
    zmq_assert(_s == retired_fd);

    //  Create the socket.
    _s = open_socket(AF_UNIX, SOCK_STREAM, 0);
    if (_s == -1)
        return -1;

    //  Set the non-blocking flag.
    unblock_socket(_s);

    //  Connect to the remote peer.
    int rc = ::connect(_s, _addr->resolved.ipc_addr->addr(), _addr->resolved.ipc_addr->addrlen());

    //  Connect was successful immediately.
    if (rc == 0)
        return 0;

    //  Translate other error codes indicating asynchronous connect has been
    //  launched to a uniform EINPROGRESS.
    if (rc == -1 && errno == EINTR)
    {
        errno = EINPROGRESS;
        return -1;
    }

    //  Forward the error.
    return -1;
}

zmq::fd_t zmq::ipc_connecter_t::connect()
{
    //  Following code should handle both Berkeley-derived socket
    //  implementations and Solaris.
    int           err = 0;
    zmq_socklen_t len = static_cast<zmq_socklen_t>(sizeof(err));
    int           rc  = getsockopt(_s, SOL_SOCKET, SO_ERROR, reinterpret_cast<char *>(&err), &len);
    if (rc == -1)
    {
        if (errno == ENOPROTOOPT)
            errno = 0;
        err = errno;
    }
    if (err != 0)
    {
        //  Assert if the error was caused by 0MQ bug.
        //  Networking problems are OK. No need to assert.
        errno = err;
        errno_assert(errno == ECONNREFUSED || errno == ECONNRESET || errno == ETIMEDOUT || errno == EHOSTUNREACH ||
                     errno == ENETUNREACH || errno == ENETDOWN);

        return retired_fd;
    }

    fd_t result = _s;
    _s          = retired_fd;
    return result;
}

#endif

//========= end of ipc_connecter.cpp ============

//========= begin of ipc_listener.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "ipc_listener.hpp"

#if !defined ZMQ_HAVE_WINDOWS && !defined ZMQ_HAVE_OPENVMS && !defined ZMQ_HAVE_VXWORKS

#include <new>

#include <string.h>

// ans ignore: #include "ipc_address.hpp"
// ans ignore: #include "io_thread.hpp"
// ans ignore: #include "config.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "ip.hpp"
// ans ignore: #include "socket_base.hpp"
// ans ignore: #include "address.hpp"

#include <fcntl.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/un.h>
#include <unistd.h>

#ifdef ZMQ_HAVE_LOCAL_PEERCRED
#include <sys/types.h>
#include <sys/ucred.h>
#endif
#ifdef ZMQ_HAVE_SO_PEERCRED
#include <grp.h>
#include <pwd.h>
#include <sys/types.h>
#if defined ZMQ_HAVE_OPENBSD
#define ucred sockpeercred
#endif
#endif

const char *zmq::ipc_listener_t::tmp_env_vars[] = {
    "TMPDIR", "TEMPDIR", "TMP",
    0 // Sentinel
};

int zmq::ipc_listener_t::create_wildcard_address(std::string &path_, std::string &file_)
{
    std::string tmp_path;

    // If TMPDIR, TEMPDIR, or TMP are available and are directories, create
    // the socket directory there.
    const char **tmp_env = tmp_env_vars;
    while (tmp_path.empty() && *tmp_env != 0)
    {
        char *      tmpdir = getenv(*tmp_env);
        struct stat statbuf;

        // Confirm it is actually a directory before trying to use
        if (tmpdir != 0 && ::stat(tmpdir, &statbuf) == 0 && S_ISDIR(statbuf.st_mode))
        {
            tmp_path.assign(tmpdir);
            if (*(tmp_path.rbegin()) != '/')
            {
                tmp_path.push_back('/');
            }
        }

        // Try the next environment variable
        ++tmp_env;
    }

    // Append a directory name
    tmp_path.append("tmpXXXXXX");

    // We need room for tmp_path + trailing NUL
    std::vector<char> buffer(tmp_path.length() + 1);
    strcpy(&buffer[0], tmp_path.c_str());

#ifdef HAVE_MKDTEMP
    // Create the directory.  POSIX requires that mkdtemp() creates the
    // directory with 0700 permissions, meaning the only possible race
    // with socket creation could be the same user.  However, since
    // each socket is created in a directory created by mkdtemp(), and
    // mkdtemp() guarantees a unique directory name, there will be no
    // collision.
    if (mkdtemp(&buffer[0]) == 0)
    {
        return -1;
    }

    path_.assign(&buffer[0]);
    file_ = path_ + "/socket";
#else
    LIBZMQ_UNUSED(path_);
    int fd = mkstemp(&buffer[0]);
    if (fd == -1)
        return -1;
    ::close(fd);

    file_.assign(&buffer[0]);
#endif

    return 0;
}

zmq::ipc_listener_t::ipc_listener_t(io_thread_t *io_thread_, socket_base_t *socket_, const options_t &options_)
    : stream_listener_base_t(io_thread_, socket_, options_), _has_file(false)
{
}

void zmq::ipc_listener_t::in_event()
{
    fd_t fd = accept();

    //  If connection was reset by the peer in the meantime, just ignore it.
    //  TODO: Handle specific errors like ENFILE/EMFILE etc.
    if (fd == retired_fd)
    {
        _socket->event_accept_failed(make_unconnected_bind_endpoint_pair(_endpoint), zmq_errno());
        return;
    }

    //  Create the engine object for this connection.
    create_engine(fd);
}

std::string zmq::ipc_listener_t::get_socket_name(zmq::fd_t fd_, socket_end_t socket_end_) const
{
    return zmq::get_socket_name<ipc_address_t>(fd_, socket_end_);
}

int zmq::ipc_listener_t::set_local_address(const char *addr_)
{
    //  Create addr on stack for auto-cleanup
    std::string addr(addr_);

    //  Allow wildcard file
    if (options.use_fd == -1 && addr[0] == '*')
    {
        if (create_wildcard_address(_tmp_socket_dirname, addr) < 0)
        {
            return -1;
        }
    }

    //  Get rid of the file associated with the UNIX domain socket that
    //  may have been left behind by the previous run of the application.
    //  MUST NOT unlink if the FD is managed by the user, or it will stop
    //  working after the first client connects. The user will take care of
    //  cleaning up the file after the service is stopped.
    if (options.use_fd == -1)
    {
        ::unlink(addr.c_str());
    }
    _filename.clear();

    //  Initialise the address structure.
    ipc_address_t address;
    int           rc = address.resolve(addr.c_str());
    if (rc != 0)
    {
        if (!_tmp_socket_dirname.empty())
        {
            // We need to preserve errno to return to the user
            int tmp_errno = errno;
            ::rmdir(_tmp_socket_dirname.c_str());
            _tmp_socket_dirname.clear();
            errno = tmp_errno;
        }
        return -1;
    }

    address.to_string(_endpoint);

    if (options.use_fd != -1)
    {
        _s = options.use_fd;
    }
    else
    {
        //  Create a listening socket.
        _s = open_socket(AF_UNIX, SOCK_STREAM, 0);
        if (_s == -1)
        {
            if (!_tmp_socket_dirname.empty())
            {
                // We need to preserve errno to return to the user
                int tmp_errno = errno;
                ::rmdir(_tmp_socket_dirname.c_str());
                _tmp_socket_dirname.clear();
                errno = tmp_errno;
            }
            return -1;
        }

        //  Bind the socket to the file path.
        rc = bind(_s, const_cast<sockaddr *>(address.addr()), address.addrlen());
        if (rc != 0)
            goto error;

        //  Listen for incoming connections.
        rc = listen(_s, options.backlog);
        if (rc != 0)
            goto error;
    }

    _filename = ZMQ_MOVE(addr);
    _has_file = true;

    _socket->event_listening(make_unconnected_bind_endpoint_pair(_endpoint), _s);
    return 0;

error:
    int err = errno;
    close();
    errno = err;
    return -1;
}

int zmq::ipc_listener_t::close()
{
    zmq_assert(_s != retired_fd);
    int fd_for_event = _s;
    int rc           = ::close(_s);
    errno_assert(rc == 0);

    _s = retired_fd;

    if (_has_file && options.use_fd == -1)
    {
        if (!_tmp_socket_dirname.empty())
        {
            //  TODO review this behaviour, it is inconsistent with the use of
            //  unlink in open since 656cdb959a7482c45db979c1d08ede585d12e315;
            //  however, we must at least remove the file before removing the
            //  directory, otherwise it will always fail
            rc = ::unlink(_filename.c_str());

            if (rc == 0)
            {
                rc = ::rmdir(_tmp_socket_dirname.c_str());
                _tmp_socket_dirname.clear();
            }
        }

        if (rc != 0)
        {
            _socket->event_close_failed(make_unconnected_bind_endpoint_pair(_endpoint), zmq_errno());
            return -1;
        }
    }

    _socket->event_closed(make_unconnected_bind_endpoint_pair(_endpoint), fd_for_event);
    return 0;
}

#if defined ZMQ_HAVE_SO_PEERCRED

bool zmq::ipc_listener_t::filter(fd_t sock_)
{
    if (options.ipc_uid_accept_filters.empty() && options.ipc_pid_accept_filters.empty() &&
        options.ipc_gid_accept_filters.empty())
        return true;

    struct ucred cred;
    socklen_t    size = sizeof(cred);

    if (getsockopt(sock_, SOL_SOCKET, SO_PEERCRED, &cred, &size))
        return false;
    if (options.ipc_uid_accept_filters.find(cred.uid) != options.ipc_uid_accept_filters.end() ||
        options.ipc_gid_accept_filters.find(cred.gid) != options.ipc_gid_accept_filters.end() ||
        options.ipc_pid_accept_filters.find(cred.pid) != options.ipc_pid_accept_filters.end())
        return true;

    struct passwd *pw;
    struct group * gr;

    if (!(pw = getpwuid(cred.uid)))
        return false;
    for (options_t::ipc_gid_accept_filters_t::const_iterator it = options.ipc_gid_accept_filters.begin();
         it != options.ipc_gid_accept_filters.end(); it++)
    {
        if (!(gr = getgrgid(*it)))
            continue;
        for (char **mem = gr->gr_mem; *mem; mem++)
        {
            if (!strcmp(*mem, pw->pw_name))
                return true;
        }
    }
    return false;
}

#elif defined ZMQ_HAVE_LOCAL_PEERCRED

bool zmq::ipc_listener_t::filter(fd_t sock_)
{
    if (options.ipc_uid_accept_filters.empty() && options.ipc_gid_accept_filters.empty())
        return true;

    struct xucred cred;
    socklen_t size = sizeof(cred);

    if (getsockopt(sock_, 0, LOCAL_PEERCRED, &cred, &size))
        return false;
    if (cred.cr_version != XUCRED_VERSION)
        return false;
    if (options.ipc_uid_accept_filters.find(cred.cr_uid) != options.ipc_uid_accept_filters.end())
        return true;
    for (int i = 0; i < cred.cr_ngroups; i++)
    {
        if (options.ipc_gid_accept_filters.find(cred.cr_groups[i]) != options.ipc_gid_accept_filters.end())
            return true;
    }

    return false;
}

#endif

zmq::fd_t zmq::ipc_listener_t::accept()
{
    //  Accept one connection and deal with different failure modes.
    //  The situation where connection cannot be accepted due to insufficient
    //  resources is considered valid and treated by ignoring the connection.
    zmq_assert(_s != retired_fd);
#if defined ZMQ_HAVE_SOCK_CLOEXEC && defined HAVE_ACCEPT4
    fd_t sock = ::accept4(_s, NULL, NULL, SOCK_CLOEXEC);
#else
    fd_t sock = ::accept(_s, NULL, NULL);
#endif
    if (sock == -1)
    {
        errno_assert(errno == EAGAIN || errno == EWOULDBLOCK || errno == EINTR || errno == ECONNABORTED ||
                     errno == EPROTO || errno == ENFILE);
        return retired_fd;
    }

    make_socket_noninheritable(sock);

    // IPC accept() filters
#if defined ZMQ_HAVE_SO_PEERCRED || defined ZMQ_HAVE_LOCAL_PEERCRED
    if (!filter(sock))
    {
        int rc = ::close(sock);
        errno_assert(rc == 0);
        return retired_fd;
    }
#endif

    if (zmq::set_nosigpipe(sock))
    {
#ifdef ZMQ_HAVE_WINDOWS
        int rc = closesocket(sock);
        wsa_assert(rc != SOCKET_ERROR);
#else
        int rc = ::close(sock);
        errno_assert(rc == 0);
#endif
        return retired_fd;
    }

    return sock;
}

#endif

//========= end of ipc_listener.cpp ============

//========= begin of ip_resolver.cpp ============

// ans ignore: #include "precompiled.hpp"
#include <cstring>
#include <string>

// ans ignore: #include "macros.hpp"
// ans ignore: #include "stdint.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "ip.hpp"

#ifndef ZMQ_HAVE_WINDOWS
#include <arpa/inet.h>
#include <ctype.h>
#include <net/if.h>
#include <netdb.h>
#include <netinet/tcp.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#endif

// ans ignore: #include "ip_resolver.hpp"

int zmq::ip_addr_t::family() const { return generic.sa_family; }

bool zmq::ip_addr_t::is_multicast() const
{
    if (family() == AF_INET)
    {
        //  IPv4 Multicast: address MSBs are 1110
        //  Range: 224.0.0.0 - 239.255.255.255
        return IN_MULTICAST(ntohl(ipv4.sin_addr.s_addr));
    }
    //  IPv6 Multicast: ff00::/8
    return IN6_IS_ADDR_MULTICAST(&ipv6.sin6_addr) != 0;
}

uint16_t zmq::ip_addr_t::port() const
{
    if (family() == AF_INET6)
    {
        return ntohs(ipv6.sin6_port);
    }
    return ntohs(ipv4.sin_port);
}

const struct sockaddr *zmq::ip_addr_t::as_sockaddr() const { return &generic; }

zmq::zmq_socklen_t zmq::ip_addr_t::sockaddr_len() const
{
    return static_cast<zmq_socklen_t>(family() == AF_INET6 ? sizeof(ipv6) : sizeof(ipv4));
}

void zmq::ip_addr_t::set_port(uint16_t port_)
{
    if (family() == AF_INET6)
    {
        ipv6.sin6_port = htons(port_);
    }
    else
    {
        ipv4.sin_port = htons(port_);
    }
}

//  Construct an "ANY" address for the given family
zmq::ip_addr_t zmq::ip_addr_t::any(int family_)
{
    ip_addr_t addr;

    if (family_ == AF_INET)
    {
        sockaddr_in *ip4_addr = &addr.ipv4;
        memset(ip4_addr, 0, sizeof(*ip4_addr));
        ip4_addr->sin_family      = AF_INET;
        ip4_addr->sin_addr.s_addr = htonl(INADDR_ANY);
    }
    else if (family_ == AF_INET6)
    {
        sockaddr_in6 *ip6_addr = &addr.ipv6;

        memset(ip6_addr, 0, sizeof(*ip6_addr));
        ip6_addr->sin6_family = AF_INET6;
#ifdef ZMQ_HAVE_VXWORKS
        struct in6_addr newaddr = IN6ADDR_ANY_INIT;
        memcpy(&ip6_addr->sin6_addr, &newaddr, sizeof(in6_addr));
#else
        memcpy(&ip6_addr->sin6_addr, &in6addr_any, sizeof(in6addr_any));
#endif
    }
    else
    {
        assert(0 == "unsupported address family");
    }

    return addr;
}

zmq::ip_resolver_options_t::ip_resolver_options_t()
    : _bindable_wanted(false), _nic_name_allowed(false), _ipv6_wanted(false), _port_expected(false), _dns_allowed(false)
{
}

zmq::ip_resolver_options_t &zmq::ip_resolver_options_t::bindable(bool bindable_)
{
    _bindable_wanted = bindable_;

    return *this;
}

zmq::ip_resolver_options_t &zmq::ip_resolver_options_t::allow_nic_name(bool allow_)
{
    _nic_name_allowed = allow_;

    return *this;
}

zmq::ip_resolver_options_t &zmq::ip_resolver_options_t::ipv6(bool ipv6_)
{
    _ipv6_wanted = ipv6_;

    return *this;
}

//  If true we expect that the host will be followed by a colon and a port
//  number or service name
zmq::ip_resolver_options_t &zmq::ip_resolver_options_t::expect_port(bool expect_)
{
    _port_expected = expect_;

    return *this;
}

zmq::ip_resolver_options_t &zmq::ip_resolver_options_t::allow_dns(bool allow_)
{
    _dns_allowed = allow_;

    return *this;
}

bool zmq::ip_resolver_options_t::bindable() { return _bindable_wanted; }

bool zmq::ip_resolver_options_t::allow_nic_name() { return _nic_name_allowed; }

bool zmq::ip_resolver_options_t::ipv6() { return _ipv6_wanted; }

bool zmq::ip_resolver_options_t::expect_port() { return _port_expected; }

bool zmq::ip_resolver_options_t::allow_dns() { return _dns_allowed; }

zmq::ip_resolver_t::ip_resolver_t(ip_resolver_options_t opts_) : _options(opts_) {}

int zmq::ip_resolver_t::resolve(ip_addr_t *ip_addr_, const char *name_)
{
    std::string addr;
    uint16_t    port;

    if (_options.expect_port())
    {
        //  We expect 'addr:port'. It's important to use str*r*chr to only get
        //  the latest colon since IPv6 addresses use colons as delemiters.
        const char *delim = strrchr(name_, ':');

        if (delim == NULL)
        {
            errno = EINVAL;
            return -1;
        }

        addr                 = std::string(name_, delim - name_);
        std::string port_str = std::string(delim + 1);

        if (port_str == "*")
        {
            if (_options.bindable())
            {
                //  Resolve wildcard to 0 to allow autoselection of port
                port = 0;
            }
            else
            {
                errno = EINVAL;
                return -1;
            }
        }
        else if (port_str == "0")
        {
            //  Using "0" for a bind address is equivalent to using "*". For a
            //  connectable address it could be used to connect to port 0.
            port = 0;
        }
        else
        {
            //  Parse the port number (0 is not a valid port).
            port = static_cast<uint16_t>(atoi(port_str.c_str()));
            if (port == 0)
            {
                errno = EINVAL;
                return -1;
            }
        }
    }
    else
    {
        addr = std::string(name_);
        port = 0;
    }

    //  Trim any square brackets surrounding the address. Used for
    //  IPv6 addresses to remove the confusion with the port
    //  delimiter.
    //  TODO Should we validate that the brackets are present if
    //  'addr' contains ':' ?
    const size_t brackets_length = 2;
    if (addr.size() >= brackets_length && addr[0] == '[' && addr[addr.size() - 1] == ']')
    {
        addr = addr.substr(1, addr.size() - brackets_length);
    }

    //  Look for an interface name / zone_id in the address
    //  Reference: https://tools.ietf.org/html/rfc4007
    std::size_t pos     = addr.rfind('%');
    uint32_t    zone_id = 0;

    if (pos != std::string::npos)
    {
        std::string if_str = addr.substr(pos + 1);
        addr               = addr.substr(0, pos);

        if (isalpha(if_str.at(0)))
        {
            zone_id = do_if_nametoindex(if_str.c_str());
        }
        else
        {
            zone_id = static_cast<uint32_t>(atoi(if_str.c_str()));
        }

        if (zone_id == 0)
        {
            errno = EINVAL;
            return -1;
        }
    }

    bool        resolved = false;
    const char *addr_str = addr.c_str();

    if (_options.bindable() && addr == "*")
    {
        //  Return an ANY address
        *ip_addr_ = ip_addr_t::any(_options.ipv6() ? AF_INET6 : AF_INET);
        resolved  = true;
    }

    if (!resolved && _options.allow_nic_name())
    {
        //  Try to resolve the string as a NIC name.
        int rc = resolve_nic_name(ip_addr_, addr_str);

        if (rc == 0)
        {
            resolved = true;
        }
        else if (errno != ENODEV)
        {
            return rc;
        }
    }

    if (!resolved)
    {
        int rc = resolve_getaddrinfo(ip_addr_, addr_str);

        if (rc != 0)
        {
            return rc;
        }
        resolved = true;
    }

    //  Store the port into the structure. We could get 'getaddrinfo' to do it
    //  for us but since we don't resolve service names it's a bit overkill and
    //  we'd still have to do it manually when the address is resolved by
    //  'resolve_nic_name'
    ip_addr_->set_port(port);

    if (ip_addr_->family() == AF_INET6)
    {
        ip_addr_->ipv6.sin6_scope_id = zone_id;
    }

    assert(resolved == true);
    return 0;
}

int zmq::ip_resolver_t::resolve_getaddrinfo(ip_addr_t *ip_addr_, const char *addr_)
{
#if defined ZMQ_HAVE_OPENVMS && defined __ia64
    __addrinfo64 *res = NULL;
    __addrinfo64  req;
#else
    addrinfo *res = NULL;
    addrinfo  req;
#endif

    memset(&req, 0, sizeof(req));

    //  Choose IPv4 or IPv6 protocol family. Note that IPv6 allows for
    //  IPv4-in-IPv6 addresses.
    req.ai_family = _options.ipv6() ? AF_INET6 : AF_INET;

    //  Arbitrary, not used in the output, but avoids duplicate results.
    req.ai_socktype = SOCK_STREAM;

    req.ai_flags = 0;

    if (_options.bindable())
    {
        req.ai_flags |= AI_PASSIVE;
    }

    if (!_options.allow_dns())
    {
        req.ai_flags |= AI_NUMERICHOST;
    }

#if defined AI_V4MAPPED
    //  In this API we only require IPv4-mapped addresses when
    //  no native IPv6 interfaces are available (~AI_ALL).
    //  This saves an additional DNS roundtrip for IPv4 addresses.
    if (req.ai_family == AF_INET6)
    {
        req.ai_flags |= AI_V4MAPPED;
    }
#endif

    //  Resolve the literal address. Some of the error info is lost in case
    //  of error, however, there's no way to report EAI errors via errno.
    int rc = do_getaddrinfo(addr_, NULL, &req, &res);

#if defined AI_V4MAPPED
    // Some OS do have AI_V4MAPPED defined but it is not supported in getaddrinfo()
    // returning EAI_BADFLAGS. Detect this and retry
    if (rc == EAI_BADFLAGS && (req.ai_flags & AI_V4MAPPED))
    {
        req.ai_flags &= ~AI_V4MAPPED;
        rc = do_getaddrinfo(addr_, NULL, &req, &res);
    }
#endif

#if defined ZMQ_HAVE_WINDOWS
    //  Resolve specific case on Windows platform when using IPv4 address
    //  with ZMQ_IPv6 socket option.
    if ((req.ai_family == AF_INET6) && (rc == WSAHOST_NOT_FOUND))
    {
        req.ai_family = AF_INET;
        rc            = do_getaddrinfo(addr_, NULL, &req, &res);
    }
#endif

    if (rc)
    {
        switch (rc)
        {
        case EAI_MEMORY:
            errno = ENOMEM;
            break;
        default:
            if (_options.bindable())
            {
                errno = ENODEV;
            }
            else
            {
                errno = EINVAL;
            }
            break;
        }
        return -1;
    }

    //  Use the first result.
    zmq_assert(res != NULL);
    zmq_assert((size_t)res->ai_addrlen <= sizeof(*ip_addr_));
    memcpy(ip_addr_, res->ai_addr, res->ai_addrlen);

    //  Cleanup getaddrinfo after copying the possibly referenced result.
    do_freeaddrinfo(res);

    return 0;
}

#ifdef ZMQ_HAVE_SOLARIS
#include <sys/sockio.h>

//  On Solaris platform, network interface name can be queried by ioctl.
int zmq::ip_resolver_t::resolve_nic_name(ip_addr_t *ip_addr_, const char *nic_)
{
    //  Create a socket.
    const int fd = open_socket(AF_INET, SOCK_DGRAM, 0);
    errno_assert(fd != -1);

    //  Retrieve number of interfaces.
    lifnum ifn;
    ifn.lifn_family = AF_INET;
    ifn.lifn_flags  = 0;
    int rc          = ioctl(fd, SIOCGLIFNUM, (char *)&ifn);
    errno_assert(rc != -1);

    //  Allocate memory to get interface names.
    const size_t ifr_size = sizeof(struct lifreq) * ifn.lifn_count;
    char *       ifr      = (char *)malloc(ifr_size);
    alloc_assert(ifr);

    //  Retrieve interface names.
    lifconf ifc;
    ifc.lifc_family = AF_INET;
    ifc.lifc_flags  = 0;
    ifc.lifc_len    = ifr_size;
    ifc.lifc_buf    = ifr;
    rc              = ioctl(fd, SIOCGLIFCONF, (char *)&ifc);
    errno_assert(rc != -1);

    //  Find the interface with the specified name and AF_INET family.
    bool    found = false;
    lifreq *ifrp  = ifc.lifc_req;
    for (int n = 0; n < (int)(ifc.lifc_len / sizeof(lifreq)); n++, ifrp++)
    {
        if (!strcmp(nic_, ifrp->lifr_name))
        {
            rc = ioctl(fd, SIOCGLIFADDR, (char *)ifrp);
            errno_assert(rc != -1);
            if (ifrp->lifr_addr.ss_family == AF_INET)
            {
                ip_addr_->ipv4 = *(sockaddr_in *)&ifrp->lifr_addr;
                found          = true;
                break;
            }
        }
    }

    //  Clean-up.
    free(ifr);
    close(fd);

    if (!found)
    {
        errno = ENODEV;
        return -1;
    }
    return 0;
}

#elif defined ZMQ_HAVE_AIX || defined ZMQ_HAVE_HPUX || defined ZMQ_HAVE_ANDROID || defined ZMQ_HAVE_VXWORKS
#include <sys/ioctl.h>
#ifdef ZMQ_HAVE_VXWORKS
#include <ioLib.h>
#endif

int zmq::ip_resolver_t::resolve_nic_name(ip_addr_t *ip_addr_, const char *nic_)
{
#if defined ZMQ_HAVE_AIX || defined ZMQ_HAVE_HPUX
    // IPv6 support not implemented for AIX or HP/UX.
    if (_options.ipv6())
    {
        errno = ENODEV;
        return -1;
    }
#endif

    //  Create a socket.
    const int sd = open_socket(_options.ipv6() ? AF_INET6 : AF_INET, SOCK_DGRAM, 0);
    errno_assert(sd != -1);

    struct ifreq ifr;

    //  Copy interface name for ioctl get.
    strncpy(ifr.ifr_name, nic_, sizeof(ifr.ifr_name));

    //  Fetch interface address.
    const int rc = ioctl(sd, SIOCGIFADDR, (caddr_t)&ifr, sizeof(ifr));

    //  Clean up.
    close(sd);

    if (rc == -1)
    {
        errno = ENODEV;
        return -1;
    }

    const int family = ifr.ifr_addr.sa_family;
    if (family == (_options.ipv6() ? AF_INET6 : AF_INET) && !strcmp(nic_, ifr.ifr_name))
    {
        memcpy(ip_addr_, &ifr.ifr_addr, (family == AF_INET) ? sizeof(struct sockaddr_in) : sizeof(struct sockaddr_in6));
    }
    else
    {
        errno = ENODEV;
        return -1;
    }

    return 0;
}

#elif ((defined ZMQ_HAVE_LINUX || defined ZMQ_HAVE_FREEBSD || defined ZMQ_HAVE_OSX || defined ZMQ_HAVE_OPENBSD ||      \
        defined ZMQ_HAVE_QNXNTO || defined ZMQ_HAVE_NETBSD || defined ZMQ_HAVE_DRAGONFLY || defined ZMQ_HAVE_GNU) &&   \
       defined ZMQ_HAVE_IFADDRS)

#include <ifaddrs.h>

//  On these platforms, network interface name can be queried
//  using getifaddrs function.
int zmq::ip_resolver_t::resolve_nic_name(ip_addr_t *ip_addr_, const char *nic_)
{
    //  Get the addresses.
    ifaddrs *ifa = NULL;
    int rc = 0;
    const int max_attempts = 10;
    const int backoff_msec = 1;
    for (int i = 0; i < max_attempts; i++)
    {
        rc = getifaddrs(&ifa);
        if (rc == 0 || (rc < 0 && errno != ECONNREFUSED))
            break;
        usleep((backoff_msec << i) * 1000);
    }

    if (rc != 0 && ((errno == EINVAL) || (errno == EOPNOTSUPP)))
    {
        // Windows Subsystem for Linux compatibility
        errno = ENODEV;
        return -1;
    }
    errno_assert(rc == 0);
    zmq_assert(ifa != NULL);

    //  Find the corresponding network interface.
    bool found = false;
    for (ifaddrs *ifp = ifa; ifp != NULL; ifp = ifp->ifa_next)
    {
        if (ifp->ifa_addr == NULL)
            continue;

        const int family = ifp->ifa_addr->sa_family;
        if (family == (_options.ipv6() ? AF_INET6 : AF_INET) && !strcmp(nic_, ifp->ifa_name))
        {
            memcpy(ip_addr_, ifp->ifa_addr,
                   (family == AF_INET) ? sizeof(struct sockaddr_in) : sizeof(struct sockaddr_in6));
            found = true;
            break;
        }
    }

    //  Clean-up;
    freeifaddrs(ifa);

    if (!found)
    {
        errno = ENODEV;
        return -1;
    }
    return 0;
}

#elif (defined ZMQ_HAVE_WINDOWS)

#include <netioapi.h>

int zmq::ip_resolver_t::get_interface_name(unsigned long index_, char **dest_) const
{
#ifdef ZMQ_HAVE_WINDOWS_UWP
    char *buffer = (char *)malloc(1024);
#else
    char *buffer = static_cast<char *>(malloc(IF_MAX_STRING_SIZE));
#endif
    alloc_assert(buffer);

    char *if_name_result = NULL;

#if !defined ZMQ_HAVE_WINDOWS_TARGET_XP && !defined ZMQ_HAVE_WINDOWS_UWP
    if_name_result = if_indextoname(index_, buffer);
#endif

    if (if_name_result == NULL)
    {
        free(buffer);
        return -1;
    }

    *dest_ = buffer;
    return 0;
}

int zmq::ip_resolver_t::wchar_to_utf8(const WCHAR *src_, char **dest_) const
{
    int rc;
    int buffer_len = WideCharToMultiByte(CP_UTF8, 0, src_, -1, NULL, 0, NULL, 0);

    char *buffer = static_cast<char *>(malloc(buffer_len));
    alloc_assert(buffer);

    rc = WideCharToMultiByte(CP_UTF8, 0, src_, -1, buffer, buffer_len, NULL, 0);

    if (rc == 0)
    {
        free(buffer);
        return -1;
    }

    *dest_ = buffer;
    return 0;
}

int zmq::ip_resolver_t::resolve_nic_name(ip_addr_t *ip_addr_, const char *nic_)
{
    int rc;
    bool found = false;
    const int max_attempts = 10;

    int iterations = 0;
    IP_ADAPTER_ADDRESSES *addresses;
    unsigned long out_buf_len = sizeof(IP_ADAPTER_ADDRESSES);

    do
    {
        addresses = static_cast<IP_ADAPTER_ADDRESSES *>(malloc(out_buf_len));
        alloc_assert(addresses);

        rc = GetAdaptersAddresses(AF_UNSPEC, GAA_FLAG_SKIP_ANYCAST | GAA_FLAG_SKIP_MULTICAST | GAA_FLAG_SKIP_DNS_SERVER,
                                  NULL, addresses, &out_buf_len);
        if (rc == ERROR_BUFFER_OVERFLOW)
        {
            free(addresses);
            addresses = NULL;
        }
        else
        {
            break;
        }
        iterations++;
    } while ((rc == ERROR_BUFFER_OVERFLOW) && (iterations < max_attempts));

    if (rc == 0)
    {
        for (const IP_ADAPTER_ADDRESSES *current_addresses = addresses; current_addresses;
             current_addresses = current_addresses->Next)
        {
            char *if_name = NULL;
            char *if_friendly_name = NULL;

            const int str_rc1 = get_interface_name(current_addresses->IfIndex, &if_name);
            const int str_rc2 = wchar_to_utf8(current_addresses->FriendlyName, &if_friendly_name);

            //  Find a network adapter by its "name" or "friendly name"
            if (((str_rc1 == 0) && (!strcmp(nic_, if_name))) || ((str_rc2 == 0) && (!strcmp(nic_, if_friendly_name))))
            {
                //  Iterate over all unicast addresses bound to the current network interface
                for (const IP_ADAPTER_UNICAST_ADDRESS *current_unicast_address = current_addresses->FirstUnicastAddress;
                     current_unicast_address; current_unicast_address = current_unicast_address->Next)
                {
                    const ADDRESS_FAMILY family = current_unicast_address->Address.lpSockaddr->sa_family;

                    if (family == (_options.ipv6() ? AF_INET6 : AF_INET))
                    {
                        memcpy(ip_addr_, current_unicast_address->Address.lpSockaddr,
                               (family == AF_INET) ? sizeof(struct sockaddr_in) : sizeof(struct sockaddr_in6));
                        found = true;
                        break;
                    }
                }

                if (found)
                    break;
            }

            if (str_rc1 == 0)
                free(if_name);
            if (str_rc2 == 0)
                free(if_friendly_name);
        }

        free(addresses);
    }

    if (!found)
    {
        errno = ENODEV;
        return -1;
    }
    return 0;
}

#else

//  On other platforms we assume there are no sane interface names.
int zmq::ip_resolver_t::resolve_nic_name(ip_addr_t *ip_addr_, const char *nic_)
{
    LIBZMQ_UNUSED(ip_addr_);
    LIBZMQ_UNUSED(nic_);

    errno = ENODEV;
    return -1;
}

#endif

int zmq::ip_resolver_t::do_getaddrinfo(const char *node_, const char *service_, const struct addrinfo *hints_,
                                       struct addrinfo **res_)
{
    return getaddrinfo(node_, service_, hints_, res_);
}

void zmq::ip_resolver_t::do_freeaddrinfo(struct addrinfo *res_) { freeaddrinfo(res_); }

unsigned int zmq::ip_resolver_t::do_if_nametoindex(const char *ifname_)
{
#if !defined ZMQ_HAVE_WINDOWS_TARGET_XP && !defined ZMQ_HAVE_WINDOWS_UWP && !defined ZMQ_HAVE_VXWORKS
    return if_nametoindex(ifname_);
#else
    // The function 'if_nametoindex' is not supported on Windows XP.
    // If we are targeting XP using a vxxx_xp toolset then fail.
    // This is brutal as this code could be run on later windows clients
    // meaning the IPv6 zone_id cannot have an interface name.
    // This could be fixed with a runtime check.
    return 0;
#endif
}

//========= end of ip_resolver.cpp ============

//========= begin of kqueue.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "kqueue.hpp"
#if defined ZMQ_IOTHREAD_POLLER_USE_KQUEUE

#include <algorithm>
#include <new>
#include <stdlib.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

// ans ignore: #include "macros.hpp"
// ans ignore: #include "kqueue.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "config.hpp"
// ans ignore: #include "i_poll_events.hpp"
// ans ignore: #include "likely.hpp"

//  NetBSD defines (struct kevent).udata as intptr_t, everyone else
//  as void *.
#if defined ZMQ_HAVE_NETBSD
#define kevent_udata_t intptr_t
#else
#define kevent_udata_t void *
#endif

zmq::kqueue_t::kqueue_t(const zmq::thread_ctx_t &ctx_) : worker_poller_base_t(ctx_)
{
    //  Create event queue
    kqueue_fd = kqueue();
    errno_assert(kqueue_fd != -1);
#ifdef HAVE_FORK
    pid = getpid();
#endif
}

zmq::kqueue_t::~kqueue_t()
{
    stop_worker();
    close(kqueue_fd);
}

void zmq::kqueue_t::kevent_add(fd_t fd_, short filter_, void *udata_)
{
    check_thread();
    struct kevent ev;

    EV_SET(&ev, fd_, filter_, EV_ADD, 0, 0, (kevent_udata_t)udata_);
    int rc = kevent(kqueue_fd, &ev, 1, NULL, 0, NULL);
    errno_assert(rc != -1);
}

void zmq::kqueue_t::kevent_delete(fd_t fd_, short filter_)
{
    struct kevent ev;

    EV_SET(&ev, fd_, filter_, EV_DELETE, 0, 0, 0);
    int rc = kevent(kqueue_fd, &ev, 1, NULL, 0, NULL);
    errno_assert(rc != -1);
}

zmq::kqueue_t::handle_t zmq::kqueue_t::add_fd(fd_t fd_, i_poll_events *reactor_)
{
    check_thread();
    poll_entry_t *pe = new (std::nothrow) poll_entry_t;
    alloc_assert(pe);

    pe->fd           = fd_;
    pe->flag_pollin  = 0;
    pe->flag_pollout = 0;
    pe->reactor      = reactor_;

    adjust_load(1);

    return pe;
}

void zmq::kqueue_t::rm_fd(handle_t handle_)
{
    check_thread();
    poll_entry_t *pe = (poll_entry_t *)handle_;
    if (pe->flag_pollin)
        kevent_delete(pe->fd, EVFILT_READ);
    if (pe->flag_pollout)
        kevent_delete(pe->fd, EVFILT_WRITE);
    pe->fd = retired_fd;
    retired.push_back(pe);

    adjust_load(-1);
}

void zmq::kqueue_t::set_pollin(handle_t handle_)
{
    check_thread();
    poll_entry_t *pe = (poll_entry_t *)handle_;
    if (likely(!pe->flag_pollin))
    {
        pe->flag_pollin = true;
        kevent_add(pe->fd, EVFILT_READ, pe);
    }
}

void zmq::kqueue_t::reset_pollin(handle_t handle_)
{
    check_thread();
    poll_entry_t *pe = (poll_entry_t *)handle_;
    if (likely(pe->flag_pollin))
    {
        pe->flag_pollin = false;
        kevent_delete(pe->fd, EVFILT_READ);
    }
}

void zmq::kqueue_t::set_pollout(handle_t handle_)
{
    check_thread();
    poll_entry_t *pe = (poll_entry_t *)handle_;
    if (likely(!pe->flag_pollout))
    {
        pe->flag_pollout = true;
        kevent_add(pe->fd, EVFILT_WRITE, pe);
    }
}

void zmq::kqueue_t::reset_pollout(handle_t handle_)
{
    check_thread();
    poll_entry_t *pe = (poll_entry_t *)handle_;
    if (likely(pe->flag_pollout))
    {
        pe->flag_pollout = false;
        kevent_delete(pe->fd, EVFILT_WRITE);
    }
}

void zmq::kqueue_t::stop() {}

int zmq::kqueue_t::max_fds() { return -1; }

void zmq::kqueue_t::loop()
{
    while (true)
    {
        //  Execute any due timers.
        int timeout = (int)execute_timers();

        if (get_load() == 0)
        {
            if (timeout == 0)
                break;

            // TODO sleep for timeout
            continue;
        }

        //  Wait for events.
        struct kevent ev_buf[max_io_events];
        timespec      ts = {timeout / 1000, (timeout % 1000) * 1000000};
        int           n  = kevent(kqueue_fd, NULL, 0, &ev_buf[0], max_io_events, timeout ? &ts : NULL);
#ifdef HAVE_FORK
        if (unlikely(pid != getpid()))
        {
            // printf("zmq::kqueue_t::loop aborting on forked child %d\n", (int)getpid());
            // simply exit the loop in a forked process.
            return;
        }
#endif
        if (n == -1)
        {
            errno_assert(errno == EINTR);
            continue;
        }

        for (int i = 0; i < n; i++)
        {
            poll_entry_t *pe = (poll_entry_t *)ev_buf[i].udata;

            if (pe->fd == retired_fd)
                continue;
            if (ev_buf[i].flags & EV_EOF)
                pe->reactor->in_event();
            if (pe->fd == retired_fd)
                continue;
            if (ev_buf[i].filter == EVFILT_WRITE)
                pe->reactor->out_event();
            if (pe->fd == retired_fd)
                continue;
            if (ev_buf[i].filter == EVFILT_READ)
                pe->reactor->in_event();
        }

        //  Destroy retired event sources.
        for (retired_t::iterator it = retired.begin(); it != retired.end(); ++it)
        {
            LIBZMQ_DELETE(*it);
        }
        retired.clear();
    }
}

#endif

//========= end of kqueue.cpp ============

//========= begin of lb.cpp ============

/*
    Copyright (c) 2007-2018 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "lb.hpp"
// ans ignore: #include "pipe.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "msg.hpp"

zmq::lb_t::lb_t() : _active(0), _current(0), _more(false), _dropping(false) {}

zmq::lb_t::~lb_t() { zmq_assert(_pipes.empty()); }

void zmq::lb_t::attach(pipe_t *pipe_)
{
    _pipes.push_back(pipe_);
    activated(pipe_);
}

void zmq::lb_t::pipe_terminated(pipe_t *pipe_)
{
    pipes_t::size_type index = _pipes.index(pipe_);

    //  If we are in the middle of multipart message and current pipe
    //  have disconnected, we have to drop the remainder of the message.
    if (index == _current && _more)
        _dropping = true;

    //  Remove the pipe from the list; adjust number of active pipes
    //  accordingly.
    if (index < _active)
    {
        _active--;
        _pipes.swap(index, _active);
        if (_current == _active)
            _current = 0;
    }
    _pipes.erase(pipe_);
}

void zmq::lb_t::activated(pipe_t *pipe_)
{
    //  Move the pipe to the list of active pipes.
    _pipes.swap(_pipes.index(pipe_), _active);
    _active++;
}

int zmq::lb_t::send(msg_t *msg_) { return sendpipe(msg_, NULL); }

int zmq::lb_t::sendpipe(msg_t *msg_, pipe_t **pipe_)
{
    //  Drop the message if required. If we are at the end of the message
    //  switch back to non-dropping mode.
    if (_dropping)
    {
        _more     = (msg_->flags() & msg_t::more) != 0;
        _dropping = _more;

        int rc = msg_->close();
        errno_assert(rc == 0);
        rc = msg_->init();
        errno_assert(rc == 0);
        return 0;
    }

    while (_active > 0)
    {
        if (_pipes[_current]->write(msg_))
        {
            if (pipe_)
                *pipe_ = _pipes[_current];
            break;
        }

        // If send fails for multi-part msg rollback other
        // parts sent earlier and return EAGAIN.
        // Application should handle this as suitable
        if (_more)
        {
            _pipes[_current]->rollback();
            // At this point the pipe is already being deallocated
            // and the first N frames are unreachable (_outpipe is
            // most likely already NULL so rollback won't actually do
            // anything and they can't be un-written to deliver later).
            // Return EFAULT to socket_base caller to drop current message
            // and any other subsequent frames to avoid them being
            // "stuck" and received when a new client reconnects, which
            // would break atomicity of multi-part messages (in blocking mode
            // socket_base just tries again and again to send the same message)
            // Note that given dropping mode returns 0, the user will
            // never know that the message could not be delivered, but
            // can't really fix it without breaking backward compatibility.
            // -2/EAGAIN will make sure socket_base caller does not re-enter
            // immediately or after a short sleep in blocking mode.
            _dropping = (msg_->flags() & msg_t::more) != 0;
            _more     = false;
            errno     = EAGAIN;
            return -2;
        }

        _active--;
        if (_current < _active)
            _pipes.swap(_current, _active);
        else
            _current = 0;
    }

    //  If there are no pipes we cannot send the message.
    if (_active == 0)
    {
        errno = EAGAIN;
        return -1;
    }

    //  If it's final part of the message we can flush it downstream and
    //  continue round-robining (load balance).
    _more = (msg_->flags() & msg_t::more) != 0;
    if (!_more)
    {
        _pipes[_current]->flush();

        if (++_current >= _active)
            _current = 0;
    }

    //  Detach the message from the data buffer.
    int rc = msg_->init();
    errno_assert(rc == 0);

    return 0;
}

bool zmq::lb_t::has_out()
{
    //  If one part of the message was already written we can definitely
    //  write the rest of the message.
    if (_more)
        return true;

    while (_active > 0)
    {
        //  Check whether a pipe has room for another message.
        if (_pipes[_current]->check_write())
            return true;

        //  Deactivate the pipe.
        _active--;
        _pipes.swap(_current, _active);
        if (_current == _active)
            _current = 0;
    }

    return false;
}

//========= end of lb.cpp ============

//========= begin of mailbox.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "mailbox.hpp"
// ans ignore: #include "err.hpp"

zmq::mailbox_t::mailbox_t()
{
    //  Get the pipe into passive state. That way, if the users starts by
    //  polling on the associated file descriptor it will get woken up when
    //  new command is posted.
    const bool ok = _cpipe.check_read();
    zmq_assert(!ok);
    _active = false;
}

zmq::mailbox_t::~mailbox_t()
{
    //  TODO: Retrieve and deallocate commands inside the _cpipe.

    // Work around problem that other threads might still be in our
    // send() method, by waiting on the mutex before disappearing.
    _sync.lock();
    _sync.unlock();
}

zmq::fd_t zmq::mailbox_t::get_fd() const { return _signaler.get_fd(); }

void zmq::mailbox_t::send(const command_t &cmd_)
{
    _sync.lock();
    _cpipe.write(cmd_, false);
    const bool ok = _cpipe.flush();
    _sync.unlock();
    if (!ok)
        _signaler.send();
}

int zmq::mailbox_t::recv(command_t *cmd_, int timeout_)
{
    //  Try to get the command straight away.
    if (_active)
    {
        if (_cpipe.read(cmd_))
            return 0;

        //  If there are no more commands available, switch into passive state.
        _active = false;
    }

    //  Wait for signal from the command sender.
    int rc = _signaler.wait(timeout_);
    if (rc == -1)
    {
        errno_assert(errno == EAGAIN || errno == EINTR);
        return -1;
    }

    //  Receive the signal.
    rc = _signaler.recv_failable();
    if (rc == -1)
    {
        errno_assert(errno == EAGAIN);
        return -1;
    }

    //  Switch into active state.
    _active = true;

    //  Get a command.
    const bool ok = _cpipe.read(cmd_);
    zmq_assert(ok);
    return 0;
}

bool zmq::mailbox_t::valid() const { return _signaler.valid(); }

//========= end of mailbox.cpp ============

//========= begin of mailbox_safe.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "mailbox_safe.hpp"
// ans ignore: #include "clock.hpp"
// ans ignore: #include "err.hpp"

#include <algorithm>

zmq::mailbox_safe_t::mailbox_safe_t(mutex_t *sync_) : _sync(sync_)
{
    //  Get the pipe into passive state. That way, if the users starts by
    //  polling on the associated file descriptor it will get woken up when
    //  new command is posted.
    const bool ok = _cpipe.check_read();
    zmq_assert(!ok);
}

zmq::mailbox_safe_t::~mailbox_safe_t()
{
    //  TODO: Retrieve and deallocate commands inside the cpipe.

    // Work around problem that other threads might still be in our
    // send() method, by waiting on the mutex before disappearing.
    _sync->lock();
    _sync->unlock();
}

void zmq::mailbox_safe_t::add_signaler(signaler_t *signaler_) { _signalers.push_back(signaler_); }

void zmq::mailbox_safe_t::remove_signaler(signaler_t *signaler_)
{
    // TODO: make a copy of array and signal outside the lock
    const std::vector<zmq::signaler_t *>::iterator end = _signalers.end();
    std::vector<signaler_t *>::iterator            it  = std::find(_signalers.begin(), end, signaler_);

    if (it != end)
        _signalers.erase(it);
}

void zmq::mailbox_safe_t::clear_signalers() { _signalers.clear(); }

void zmq::mailbox_safe_t::send(const command_t &cmd_)
{
    _sync->lock();
    _cpipe.write(cmd_, false);
    const bool ok = _cpipe.flush();

    if (!ok)
    {
        _cond_var.broadcast();

        for (std::vector<signaler_t *>::iterator it = _signalers.begin(), end = _signalers.end(); it != end; ++it)
        {
            (*it)->send();
        }
    }

    _sync->unlock();
}

int zmq::mailbox_safe_t::recv(command_t *cmd_, int timeout_)
{
    //  Try to get the command straight away.
    if (_cpipe.read(cmd_))
        return 0;

    //  If the timeout is zero, it will be quicker to release the lock, giving other a chance to send a command
    //  and immediately relock it.
    if (timeout_ == 0)
    {
        _sync->unlock();
        _sync->lock();
    }
    else
    {
        //  Wait for signal from the command sender.
        int rc = _cond_var.wait(_sync, timeout_);
        if (rc == -1)
        {
            errno_assert(errno == EAGAIN || errno == EINTR);
            return -1;
        }
    }

    //  Another thread may already fetch the command
    const bool ok = _cpipe.read(cmd_);

    if (!ok)
    {
        errno = EAGAIN;
        return -1;
    }

    return 0;
}

//========= end of mailbox_safe.cpp ============

//========= begin of mechanism.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <limits.h>
#include <string.h>

// ans ignore: #include "mechanism.hpp"
// ans ignore: #include "options.hpp"
// ans ignore: #include "msg.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "wire.hpp"
// ans ignore: #include "session_base.hpp"

zmq::mechanism_t::mechanism_t(const options_t &options_) : options(options_) {}

zmq::mechanism_t::~mechanism_t() {}

void zmq::mechanism_t::set_peer_routing_id(const void *id_ptr_, size_t id_size_)
{
    _routing_id.set(static_cast<const unsigned char *>(id_ptr_), id_size_);
}

void zmq::mechanism_t::peer_routing_id(msg_t *msg_)
{
    const int rc = msg_->init_size(_routing_id.size());
    errno_assert(rc == 0);
    memcpy(msg_->data(), _routing_id.data(), _routing_id.size());
    msg_->set_flags(msg_t::routing_id);
}

void zmq::mechanism_t::set_user_id(const void *user_id_, size_t size_)
{
    _user_id.set(static_cast<const unsigned char *>(user_id_), size_);
    _zap_properties.ZMQ_MAP_INSERT_OR_EMPLACE(std::string(ZMQ_MSG_PROPERTY_USER_ID),
                                              std::string(reinterpret_cast<const char *>(user_id_), size_));
}

const zmq::blob_t &zmq::mechanism_t::get_user_id() const { return _user_id; }

const char socket_type_pair[]   = "PAIR";
const char socket_type_pub[]    = "PUB";
const char socket_type_sub[]    = "SUB";
const char socket_type_req[]    = "REQ";
const char socket_type_rep[]    = "REP";
const char socket_type_dealer[] = "DEALER";
const char socket_type_router[] = "ROUTER";
const char socket_type_pull[]   = "PULL";
const char socket_type_push[]   = "PUSH";
const char socket_type_xpub[]   = "XPUB";
const char socket_type_xsub[]   = "XSUB";
const char socket_type_stream[] = "STREAM";
#ifdef ZMQ_BUILD_DRAFT_API
const char socket_type_server[]  = "SERVER";
const char socket_type_client[]  = "CLIENT";
const char socket_type_radio[]   = "RADIO";
const char socket_type_dish[]    = "DISH";
const char socket_type_gather[]  = "GATHER";
const char socket_type_scatter[] = "SCATTER";
const char socket_type_dgram[]   = "DGRAM";
#endif

const char *zmq::mechanism_t::socket_type_string(int socket_type_) const
{
    // TODO the order must of the names must correspond to the values resp. order of ZMQ_* socket type definitions in
    // zmq.h!
    static const char *names[] = {socket_type_pair,   socket_type_pub,     socket_type_sub,    socket_type_req,
                                  socket_type_rep,    socket_type_dealer,  socket_type_router, socket_type_pull,
                                  socket_type_push,   socket_type_xpub,    socket_type_xsub,   socket_type_stream,
#ifdef ZMQ_BUILD_DRAFT_API
                                  socket_type_server, socket_type_client,  socket_type_radio,  socket_type_dish,
                                  socket_type_gather, socket_type_scatter, socket_type_dgram
#endif
    };
    static const size_t names_count = sizeof(names) / sizeof(names[0]);
    zmq_assert(socket_type_ >= 0 && socket_type_ < (int)names_count);
    return names[socket_type_];
}

const size_t name_len_size  = sizeof(unsigned char);
const size_t value_len_size = sizeof(uint32_t);

static size_t property_len(size_t name_len_, size_t value_len_)
{
    return name_len_size + name_len_ + value_len_size + value_len_;
}

static size_t name_len(const char *name_)
{
    const size_t name_len = strlen(name_);
    zmq_assert(name_len <= UCHAR_MAX);
    return name_len;
}

size_t zmq::mechanism_t::add_property(unsigned char *ptr_, size_t ptr_capacity_, const char *name_, const void *value_,
                                      size_t value_len_)
{
    const size_t name_len  = ::name_len(name_);
    const size_t total_len = ::property_len(name_len, value_len_);
    zmq_assert(total_len <= ptr_capacity_);

    *ptr_ = static_cast<unsigned char>(name_len);
    ptr_ += name_len_size;
    memcpy(ptr_, name_, name_len);
    ptr_ += name_len;
    zmq_assert(value_len_ <= 0x7FFFFFFF);
    put_uint32(ptr_, static_cast<uint32_t>(value_len_));
    ptr_ += value_len_size;
    memcpy(ptr_, value_, value_len_);

    return total_len;
}

size_t zmq::mechanism_t::property_len(const char *name_, size_t value_len_)
{
    return ::property_len(name_len(name_), value_len_);
}

#define ZMTP_PROPERTY_SOCKET_TYPE "Socket-Type"
#define ZMTP_PROPERTY_IDENTITY "Identity"

size_t zmq::mechanism_t::add_basic_properties(unsigned char *ptr_, size_t ptr_capacity_) const
{
    unsigned char *ptr = ptr_;

    //  Add socket type property
    const char *socket_type = socket_type_string(options.type);
    ptr += add_property(ptr, ptr_capacity_, ZMTP_PROPERTY_SOCKET_TYPE, socket_type, strlen(socket_type));

    //  Add identity (aka routing id) property
    if (options.type == ZMQ_REQ || options.type == ZMQ_DEALER || options.type == ZMQ_ROUTER)
    {
        ptr += add_property(ptr, ptr_capacity_ - (ptr - ptr_), ZMTP_PROPERTY_IDENTITY, options.routing_id,
                            options.routing_id_size);
    }

    for (std::map<std::string, std::string>::const_iterator it  = options.app_metadata.begin(),
                                                            end = options.app_metadata.end();
         it != end; ++it)
    {
        ptr += add_property(ptr, ptr_capacity_ - (ptr - ptr_), it->first.c_str(), it->second.c_str(),
                            strlen(it->second.c_str()));
    }

    return ptr - ptr_;
}

size_t zmq::mechanism_t::basic_properties_len() const
{
    const char *socket_type = socket_type_string(options.type);
    size_t      meta_len    = 0;

    for (std::map<std::string, std::string>::const_iterator it  = options.app_metadata.begin(),
                                                            end = options.app_metadata.end();
         it != end; ++it)
    {
        meta_len += property_len(it->first.c_str(), strlen(it->second.c_str()));
    }

    return property_len(ZMTP_PROPERTY_SOCKET_TYPE, strlen(socket_type)) + meta_len +
           ((options.type == ZMQ_REQ || options.type == ZMQ_DEALER || options.type == ZMQ_ROUTER)
                ? property_len(ZMTP_PROPERTY_IDENTITY, options.routing_id_size)
                : 0);
}

void zmq::mechanism_t::make_command_with_basic_properties(msg_t *msg_, const char *prefix_, size_t prefix_len_) const
{
    const size_t command_size = prefix_len_ + basic_properties_len();
    const int    rc           = msg_->init_size(command_size);
    errno_assert(rc == 0);

    unsigned char *ptr = static_cast<unsigned char *>(msg_->data());

    //  Add prefix
    memcpy(ptr, prefix_, prefix_len_);
    ptr += prefix_len_;

    add_basic_properties(ptr, command_size - (ptr - static_cast<unsigned char *>(msg_->data())));
}

int zmq::mechanism_t::parse_metadata(const unsigned char *ptr_, size_t length_, bool zap_flag_)
{
    size_t bytes_left = length_;

    while (bytes_left > 1)
    {
        const size_t name_length = static_cast<size_t>(*ptr_);
        ptr_ += name_len_size;
        bytes_left -= name_len_size;
        if (bytes_left < name_length)
            break;

        const std::string name = std::string(reinterpret_cast<const char *>(ptr_), name_length);
        ptr_ += name_length;
        bytes_left -= name_length;
        if (bytes_left < value_len_size)
            break;

        const size_t value_length = static_cast<size_t>(get_uint32(ptr_));
        ptr_ += value_len_size;
        bytes_left -= value_len_size;
        if (bytes_left < value_length)
            break;

        const uint8_t *value = ptr_;
        ptr_ += value_length;
        bytes_left -= value_length;

        if (name == ZMTP_PROPERTY_IDENTITY && options.recv_routing_id)
            set_peer_routing_id(value, value_length);
        else if (name == ZMTP_PROPERTY_SOCKET_TYPE)
        {
            if (!check_socket_type(reinterpret_cast<const char *>(value), value_length))
            {
                errno = EINVAL;
                return -1;
            }
        }
        else
        {
            const int rc = property(name, value, value_length);
            if (rc == -1)
                return -1;
        }
        (zap_flag_ ? _zap_properties : _zmtp_properties)
            .ZMQ_MAP_INSERT_OR_EMPLACE(name, std::string(reinterpret_cast<const char *>(value), value_length));
    }
    if (bytes_left > 0)
    {
        errno = EPROTO;
        return -1;
    }
    return 0;
}

int zmq::mechanism_t::property(const std::string & /* name_ */, const void * /* value_ */, size_t /* length_ */)
{
    //  Default implementation does not check
    //  property values and returns 0 to signal success.
    return 0;
}

template <size_t N>
static bool strequals(const char *actual_type_, const size_t actual_len_, const char (&expected_type_)[N])
{
    return actual_len_ == N - 1 && memcmp(actual_type_, expected_type_, N - 1) == 0;
}

bool zmq::mechanism_t::check_socket_type(const char *type_, const size_t len_) const
{
    switch (options.type)
    {
    case ZMQ_REQ:
        return strequals(type_, len_, socket_type_rep) || strequals(type_, len_, socket_type_router);
    case ZMQ_REP:
        return strequals(type_, len_, socket_type_req) || strequals(type_, len_, socket_type_dealer);
    case ZMQ_DEALER:
        return strequals(type_, len_, socket_type_rep) || strequals(type_, len_, socket_type_dealer) ||
               strequals(type_, len_, socket_type_router);
    case ZMQ_ROUTER:
        return strequals(type_, len_, socket_type_req) || strequals(type_, len_, socket_type_dealer) ||
               strequals(type_, len_, socket_type_router);
    case ZMQ_PUSH:
        return strequals(type_, len_, socket_type_pull);
    case ZMQ_PULL:
        return strequals(type_, len_, socket_type_push);
    case ZMQ_PUB:
        return strequals(type_, len_, socket_type_sub) || strequals(type_, len_, socket_type_xsub);
    case ZMQ_SUB:
        return strequals(type_, len_, socket_type_pub) || strequals(type_, len_, socket_type_xpub);
    case ZMQ_XPUB:
        return strequals(type_, len_, socket_type_sub) || strequals(type_, len_, socket_type_xsub);
    case ZMQ_XSUB:
        return strequals(type_, len_, socket_type_pub) || strequals(type_, len_, socket_type_xpub);
    case ZMQ_PAIR:
        return strequals(type_, len_, socket_type_pair);
#ifdef ZMQ_BUILD_DRAFT_API
    case ZMQ_SERVER:
        return strequals(type_, len_, socket_type_client);
    case ZMQ_CLIENT:
        return strequals(type_, len_, socket_type_server);
    case ZMQ_RADIO:
        return strequals(type_, len_, socket_type_dish);
    case ZMQ_DISH:
        return strequals(type_, len_, socket_type_radio);
    case ZMQ_GATHER:
        return strequals(type_, len_, socket_type_scatter);
    case ZMQ_SCATTER:
        return strequals(type_, len_, socket_type_gather);
    case ZMQ_DGRAM:
        return strequals(type_, len_, socket_type_dgram);
#endif
    default:
        break;
    }
    return false;
}

//========= end of mechanism.cpp ============

//========= begin of mechanism_base.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"

// ans ignore: #include "mechanism_base.hpp"
// ans ignore: #include "session_base.hpp"

zmq::mechanism_base_t::mechanism_base_t(session_base_t *const session_, const options_t &options_)
    : mechanism_t(options_), session(session_)
{
}

int zmq::mechanism_base_t::check_basic_command_structure(msg_t *msg_)
{
    if (msg_->size() <= 1 || msg_->size() <= (static_cast<uint8_t *>(msg_->data()))[0])
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_UNSPECIFIED);
        errno = EPROTO;
        return -1;
    }
    return 0;
}

void zmq::mechanism_base_t::handle_error_reason(const char *error_reason_, size_t error_reason_len_)
{
    const size_t status_code_len         = 3;
    const char   zero_digit              = '0';
    const size_t significant_digit_index = 0;
    const size_t first_zero_digit_index  = 1;
    const size_t second_zero_digit_index = 2;
    const int    factor                  = 100;
    if (error_reason_len_ == status_code_len && error_reason_[first_zero_digit_index] == zero_digit &&
        error_reason_[second_zero_digit_index] == zero_digit && error_reason_[significant_digit_index] >= '3' &&
        error_reason_[significant_digit_index] <= '5')
    {
        // it is a ZAP error status code (300, 400 or 500), so emit an authentication failure event
        session->get_socket()->event_handshake_failed_auth(
            session->get_endpoint(), (error_reason_[significant_digit_index] - zero_digit) * factor);
    }
    else
    {
        // this is a violation of the ZAP protocol
        // TODO zmq_assert in this case?
    }
}

bool zmq::mechanism_base_t::zap_required() const { return !options.zap_domain.empty(); }

//========= end of mechanism_base.cpp ============

//========= begin of metadata.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "metadata.hpp"

zmq::metadata_t::metadata_t(const dict_t &dict_) : _ref_cnt(1), _dict(dict_) {}

const char *zmq::metadata_t::get(const std::string &property_) const
{
    dict_t::const_iterator it = _dict.find(property_);
    if (it == _dict.end())
    {
        /** \todo remove this when support for the deprecated name "Identity" is dropped */
        if (property_ == "Identity")
            return get(ZMQ_MSG_PROPERTY_ROUTING_ID);

        return NULL;
    }
    return it->second.c_str();
}

void zmq::metadata_t::add_ref() { _ref_cnt.add(1); }

bool zmq::metadata_t::drop_ref() { return !_ref_cnt.sub(1); }

//========= end of metadata.cpp ============

//========= begin of msg.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "msg.hpp"

#include <new>
#include <stdlib.h>
#include <string.h>

// ans ignore: #include "stdint.hpp"
// ans ignore: #include "likely.hpp"
// ans ignore: #include "metadata.hpp"
// ans ignore: #include "err.hpp"

//  Check whether the sizes of public representation of the message (zmq_msg_t)
//  and private representation of the message (zmq::msg_t) match.

typedef char zmq_msg_size_check[2 * ((sizeof(zmq::msg_t) == sizeof(zmq_msg_t)) != 0) - 1];

bool zmq::msg_t::check() const { return _u.base.type >= type_min && _u.base.type <= type_max; }

int zmq::msg_t::init(void *data_, size_t size_, msg_free_fn *ffn_, void *hint_, content_t *content_)
{
    if (size_ < max_vsm_size)
    {
        const int rc = init_size(size_);

        if (rc != -1)
        {
            memcpy(data(), data_, size_);
            return 0;
        }
        return -1;
    }
    if (content_)
    {
        return init_external_storage(content_, data_, size_, ffn_, hint_);
    }
    return init_data(data_, size_, ffn_, hint_);
}

int zmq::msg_t::init()
{
    _u.vsm.metadata   = NULL;
    _u.vsm.type       = type_vsm;
    _u.vsm.flags      = 0;
    _u.vsm.size       = 0;
    _u.vsm.group[0]   = '\0';
    _u.vsm.routing_id = 0;
    return 0;
}

int zmq::msg_t::init_size(size_t size_)
{
    if (size_ <= max_vsm_size)
    {
        _u.vsm.metadata   = NULL;
        _u.vsm.type       = type_vsm;
        _u.vsm.flags      = 0;
        _u.vsm.size       = static_cast<unsigned char>(size_);
        _u.vsm.group[0]   = '\0';
        _u.vsm.routing_id = 0;
    }
    else
    {
        _u.lmsg.metadata   = NULL;
        _u.lmsg.type       = type_lmsg;
        _u.lmsg.flags      = 0;
        _u.lmsg.group[0]   = '\0';
        _u.lmsg.routing_id = 0;
        _u.lmsg.content    = NULL;
        if (sizeof(content_t) + size_ > size_)
            _u.lmsg.content = static_cast<content_t *>(malloc(sizeof(content_t) + size_));
        if (unlikely(!_u.lmsg.content))
        {
            errno = ENOMEM;
            return -1;
        }

        _u.lmsg.content->data = _u.lmsg.content + 1;
        _u.lmsg.content->size = size_;
        _u.lmsg.content->ffn  = NULL;
        _u.lmsg.content->hint = NULL;
        new (&_u.lmsg.content->refcnt) zmq::atomic_counter_t();
    }
    return 0;
}

int zmq::msg_t::init_external_storage(content_t *content_, void *data_, size_t size_, msg_free_fn *ffn_, void *hint_)
{
    zmq_assert(NULL != data_);
    zmq_assert(NULL != content_);

    _u.zclmsg.metadata   = NULL;
    _u.zclmsg.type       = type_zclmsg;
    _u.zclmsg.flags      = 0;
    _u.zclmsg.group[0]   = '\0';
    _u.zclmsg.routing_id = 0;

    _u.zclmsg.content       = content_;
    _u.zclmsg.content->data = data_;
    _u.zclmsg.content->size = size_;
    _u.zclmsg.content->ffn  = ffn_;
    _u.zclmsg.content->hint = hint_;
    new (&_u.zclmsg.content->refcnt) zmq::atomic_counter_t();

    return 0;
}

int zmq::msg_t::init_data(void *data_, size_t size_, msg_free_fn *ffn_, void *hint_)
{
    //  If data is NULL and size is not 0, a segfault
    //  would occur once the data is accessed
    zmq_assert(data_ != NULL || size_ == 0);

    //  Initialize constant message if there's no need to deallocate
    if (ffn_ == NULL)
    {
        _u.cmsg.metadata   = NULL;
        _u.cmsg.type       = type_cmsg;
        _u.cmsg.flags      = 0;
        _u.cmsg.data       = data_;
        _u.cmsg.size       = size_;
        _u.cmsg.group[0]   = '\0';
        _u.cmsg.routing_id = 0;
    }
    else
    {
        _u.lmsg.metadata   = NULL;
        _u.lmsg.type       = type_lmsg;
        _u.lmsg.flags      = 0;
        _u.lmsg.group[0]   = '\0';
        _u.lmsg.routing_id = 0;
        _u.lmsg.content    = static_cast<content_t *>(malloc(sizeof(content_t)));
        if (!_u.lmsg.content)
        {
            errno = ENOMEM;
            return -1;
        }

        _u.lmsg.content->data = data_;
        _u.lmsg.content->size = size_;
        _u.lmsg.content->ffn  = ffn_;
        _u.lmsg.content->hint = hint_;
        new (&_u.lmsg.content->refcnt) zmq::atomic_counter_t();
    }
    return 0;
}

int zmq::msg_t::init_delimiter()
{
    _u.delimiter.metadata   = NULL;
    _u.delimiter.type       = type_delimiter;
    _u.delimiter.flags      = 0;
    _u.delimiter.group[0]   = '\0';
    _u.delimiter.routing_id = 0;
    return 0;
}

int zmq::msg_t::init_join()
{
    _u.base.metadata   = NULL;
    _u.base.type       = type_join;
    _u.base.flags      = 0;
    _u.base.group[0]   = '\0';
    _u.base.routing_id = 0;
    return 0;
}

int zmq::msg_t::init_leave()
{
    _u.base.metadata   = NULL;
    _u.base.type       = type_leave;
    _u.base.flags      = 0;
    _u.base.group[0]   = '\0';
    _u.base.routing_id = 0;
    return 0;
}

int zmq::msg_t::close()
{
    //  Check the validity of the message.
    if (unlikely(!check()))
    {
        errno = EFAULT;
        return -1;
    }

    if (_u.base.type == type_lmsg)
    {
        //  If the content is not shared, or if it is shared and the reference
        //  count has dropped to zero, deallocate it.
        if (!(_u.lmsg.flags & msg_t::shared) || !_u.lmsg.content->refcnt.sub(1))
        {
            //  We used "placement new" operator to initialize the reference
            //  counter so we call the destructor explicitly now.
            _u.lmsg.content->refcnt.~atomic_counter_t();

            if (_u.lmsg.content->ffn)
                _u.lmsg.content->ffn(_u.lmsg.content->data, _u.lmsg.content->hint);
            free(_u.lmsg.content);
        }
    }

    if (is_zcmsg())
    {
        zmq_assert(_u.zclmsg.content->ffn);

        //  If the content is not shared, or if it is shared and the reference
        //  count has dropped to zero, deallocate it.
        if (!(_u.zclmsg.flags & msg_t::shared) || !_u.zclmsg.content->refcnt.sub(1))
        {
            //  We used "placement new" operator to initialize the reference
            //  counter so we call the destructor explicitly now.
            _u.zclmsg.content->refcnt.~atomic_counter_t();

            _u.zclmsg.content->ffn(_u.zclmsg.content->data, _u.zclmsg.content->hint);
        }
    }

    if (_u.base.metadata != NULL)
    {
        if (_u.base.metadata->drop_ref())
        {
            LIBZMQ_DELETE(_u.base.metadata);
        }
        _u.base.metadata = NULL;
    }

    //  Make the message invalid.
    _u.base.type = 0;

    return 0;
}

int zmq::msg_t::move(msg_t &src_)
{
    //  Check the validity of the source.
    if (unlikely(!src_.check()))
    {
        errno = EFAULT;
        return -1;
    }

    int rc = close();
    if (unlikely(rc < 0))
        return rc;

    *this = src_;

    rc = src_.init();
    if (unlikely(rc < 0))
        return rc;

    return 0;
}

int zmq::msg_t::copy(msg_t &src_)
{
    //  Check the validity of the source.
    if (unlikely(!src_.check()))
    {
        errno = EFAULT;
        return -1;
    }

    int rc = close();
    if (unlikely(rc < 0))
        return rc;

    // The initial reference count, when a non-shared message is initially
    // shared (between the original and the copy we create here).
    const atomic_counter_t::integer_t initial_shared_refcnt = 2;

    if (src_.is_lmsg() || src_.is_zcmsg())
    {
        //  One reference is added to shared messages. Non-shared messages
        //  are turned into shared messages.
        if (src_.flags() & msg_t::shared)
            src_.refcnt()->add(1);
        else
        {
            src_.set_flags(msg_t::shared);
            src_.refcnt()->set(initial_shared_refcnt);
        }
    }

    if (src_._u.base.metadata != NULL)
        src_._u.base.metadata->add_ref();

    *this = src_;

    return 0;
}

void *zmq::msg_t::data()
{
    //  Check the validity of the message.
    zmq_assert(check());

    switch (_u.base.type)
    {
    case type_vsm:
        return _u.vsm.data;
    case type_lmsg:
        return _u.lmsg.content->data;
    case type_cmsg:
        return _u.cmsg.data;
    case type_zclmsg:
        return _u.zclmsg.content->data;
    default:
        zmq_assert(false);
        return NULL;
    }
}

size_t zmq::msg_t::size() const
{
    //  Check the validity of the message.
    zmq_assert(check());

    switch (_u.base.type)
    {
    case type_vsm:
        return _u.vsm.size;
    case type_lmsg:
        return _u.lmsg.content->size;
    case type_zclmsg:
        return _u.zclmsg.content->size;
    case type_cmsg:
        return _u.cmsg.size;
    default:
        zmq_assert(false);
        return 0;
    }
}

unsigned char zmq::msg_t::flags() const { return _u.base.flags; }

void zmq::msg_t::set_flags(unsigned char flags_) { _u.base.flags |= flags_; }

void zmq::msg_t::reset_flags(unsigned char flags_) { _u.base.flags &= ~flags_; }

zmq::metadata_t *zmq::msg_t::metadata() const { return _u.base.metadata; }

void zmq::msg_t::set_metadata(zmq::metadata_t *metadata_)
{
    assert(metadata_ != NULL);
    assert(_u.base.metadata == NULL);
    metadata_->add_ref();
    _u.base.metadata = metadata_;
}

void zmq::msg_t::reset_metadata()
{
    if (_u.base.metadata)
    {
        if (_u.base.metadata->drop_ref())
        {
            LIBZMQ_DELETE(_u.base.metadata);
        }
        _u.base.metadata = NULL;
    }
}

bool zmq::msg_t::is_routing_id() const { return (_u.base.flags & routing_id) == routing_id; }

bool zmq::msg_t::is_credential() const { return (_u.base.flags & credential) == credential; }

bool zmq::msg_t::is_delimiter() const { return _u.base.type == type_delimiter; }

bool zmq::msg_t::is_vsm() const { return _u.base.type == type_vsm; }

bool zmq::msg_t::is_cmsg() const { return _u.base.type == type_cmsg; }

bool zmq::msg_t::is_lmsg() const { return _u.base.type == type_lmsg; }

bool zmq::msg_t::is_zcmsg() const { return _u.base.type == type_zclmsg; }

bool zmq::msg_t::is_join() const { return _u.base.type == type_join; }

bool zmq::msg_t::is_leave() const { return _u.base.type == type_leave; }

bool zmq::msg_t::is_ping() const { return (_u.base.flags & CMD_TYPE_MASK) == ping; }

bool zmq::msg_t::is_pong() const { return (_u.base.flags & CMD_TYPE_MASK) == pong; }

size_t zmq::msg_t::command_body_size() const
{
    if (this->is_ping() || this->is_pong())
        return this->size() - ping_cmd_name_size;
    if (this->is_subscribe())
        return this->size() - sub_cmd_name_size;
    if (this->is_cancel())
        return this->size() - cancel_cmd_name_size;

    return 0;
}

void *zmq::msg_t::command_body()
{
    unsigned char *data = NULL;
    if (this->is_ping() || this->is_pong())
        data = static_cast<unsigned char *>(this->data()) + ping_cmd_name_size;
    if (this->is_subscribe())
        data = static_cast<unsigned char *>(this->data()) + sub_cmd_name_size;
    if (this->is_cancel())
        data = static_cast<unsigned char *>(this->data()) + cancel_cmd_name_size;

    return data;
}

void zmq::msg_t::add_refs(int refs_)
{
    zmq_assert(refs_ >= 0);

    //  Operation not supported for messages with metadata.
    zmq_assert(_u.base.metadata == NULL);

    //  No copies required.
    if (!refs_)
        return;

    //  VSMs, CMSGS and delimiters can be copied straight away. The only
    //  message type that needs special care are long messages.
    if (_u.base.type == type_lmsg || is_zcmsg())
    {
        if (_u.base.flags & msg_t::shared)
            refcnt()->add(refs_);
        else
        {
            refcnt()->set(refs_ + 1);
            _u.base.flags |= msg_t::shared;
        }
    }
}

bool zmq::msg_t::rm_refs(int refs_)
{
    zmq_assert(refs_ >= 0);

    //  Operation not supported for messages with metadata.
    zmq_assert(_u.base.metadata == NULL);

    //  No copies required.
    if (!refs_)
        return true;

    //  If there's only one reference close the message.
    if ((_u.base.type != type_zclmsg && _u.base.type != type_lmsg) || !(_u.base.flags & msg_t::shared))
    {
        close();
        return false;
    }

    //  The only message type that needs special care are long and zcopy messages.
    if (_u.base.type == type_lmsg && !_u.lmsg.content->refcnt.sub(refs_))
    {
        //  We used "placement new" operator to initialize the reference
        //  counter so we call the destructor explicitly now.
        _u.lmsg.content->refcnt.~atomic_counter_t();

        if (_u.lmsg.content->ffn)
            _u.lmsg.content->ffn(_u.lmsg.content->data, _u.lmsg.content->hint);
        free(_u.lmsg.content);

        return false;
    }

    if (is_zcmsg() && !_u.zclmsg.content->refcnt.sub(refs_))
    {
        // storage for rfcnt is provided externally
        if (_u.zclmsg.content->ffn)
        {
            _u.zclmsg.content->ffn(_u.zclmsg.content->data, _u.zclmsg.content->hint);
        }

        return false;
    }

    return true;
}

uint32_t zmq::msg_t::get_routing_id() { return _u.base.routing_id; }

int zmq::msg_t::set_routing_id(uint32_t routing_id_)
{
    if (routing_id_)
    {
        _u.base.routing_id = routing_id_;
        return 0;
    }
    errno = EINVAL;
    return -1;
}

int zmq::msg_t::reset_routing_id()
{
    _u.base.routing_id = 0;
    return 0;
}

const char *zmq::msg_t::group() { return _u.base.group; }

int zmq::msg_t::set_group(const char *group_) { return set_group(group_, ZMQ_GROUP_MAX_LENGTH); }

int zmq::msg_t::set_group(const char *group_, size_t length_)
{
    if (length_ > ZMQ_GROUP_MAX_LENGTH)
    {
        errno = EINVAL;
        return -1;
    }

    strncpy(_u.base.group, group_, length_);
    _u.base.group[length_] = '\0';

    return 0;
}

zmq::atomic_counter_t *zmq::msg_t::refcnt()
{
    switch (_u.base.type)
    {
    case type_lmsg:
        return &_u.lmsg.content->refcnt;
    case type_zclmsg:
        return &_u.zclmsg.content->refcnt;
    default:
        zmq_assert(false);
        return NULL;
    }
}

//========= end of msg.cpp ============

//========= begin of mtrie.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "mtrie.hpp"
// ans ignore: #include "generic_mtrie_impl.hpp"

namespace zmq
{
template class generic_mtrie_t<pipe_t>;
}

//========= end of mtrie.cpp ============

//========= begin of norm_engine.cpp ============

// ans ignore: #include "precompiled.hpp"

// ans ignore: #include "platform.hpp"

#if defined ZMQ_HAVE_NORM

// ans ignore: #include "norm_engine.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "v2_protocol.hpp"

zmq::norm_engine_t::norm_engine_t(io_thread_t *parent_, const options_t &options_)
    : io_object_t(parent_), zmq_session(NULL), options(options_), norm_instance(NORM_INSTANCE_INVALID),
      norm_session(NORM_SESSION_INVALID), is_sender(false), is_receiver(false), zmq_encoder(0),
      norm_tx_stream(NORM_OBJECT_INVALID), tx_first_msg(true), tx_more_bit(false), zmq_output_ready(false),
      norm_tx_ready(false), tx_index(0), tx_len(0), zmq_input_ready(false)
{
    int rc = tx_msg.init();
    errno_assert(0 == rc);
}

zmq::norm_engine_t::~norm_engine_t()
{
    shutdown(); // in case it was not already called
}

int zmq::norm_engine_t::init(const char *network_, bool send, bool recv)
{
    // Parse the "network_" address int "iface", "addr", and "port"
    // norm endpoint format: [id,][<iface>;]<addr>:<port>
    // First, look for optional local NormNodeId
    // (default NORM_NODE_ANY causes NORM to use host IP addr for NormNodeId)
    NormNodeId  localId  = NORM_NODE_ANY;
    const char *ifacePtr = strchr(network_, ',');
    if (NULL != ifacePtr)
    {
        size_t idLen = ifacePtr - network_;
        if (idLen > 31)
            idLen = 31;
        char idText[32];
        strncpy(idText, network_, idLen);
        idText[idLen] = '\0';
        localId       = (NormNodeId)atoi(idText);
        ifacePtr++;
    }
    else
    {
        ifacePtr = network_;
    }

    // Second, look for optional multicast ifaceName
    char        ifaceName[256];
    const char *addrPtr = strchr(ifacePtr, ';');
    if (NULL != addrPtr)
    {
        size_t ifaceLen = addrPtr - ifacePtr;
        if (ifaceLen > 255)
            ifaceLen = 255; // return error instead?
        strncpy(ifaceName, ifacePtr, ifaceLen);
        ifaceName[ifaceLen] = '\0';
        ifacePtr            = ifaceName;
        addrPtr++;
    }
    else
    {
        addrPtr  = ifacePtr;
        ifacePtr = NULL;
    }

    // Finally, parse IP address and port number
    const char *portPtr = strrchr(addrPtr, ':');
    if (NULL == portPtr)
    {
        errno = EINVAL;
        return -1;
    }

    char   addr[256];
    size_t addrLen = portPtr - addrPtr;
    if (addrLen > 255)
        addrLen = 255;
    strncpy(addr, addrPtr, addrLen);
    addr[addrLen] = '\0';
    portPtr++;
    unsigned short portNumber = atoi(portPtr);

    if (NORM_INSTANCE_INVALID == norm_instance)
    {
        if (NORM_INSTANCE_INVALID == (norm_instance = NormCreateInstance()))
        {
            // errno set by whatever caused NormCreateInstance() to fail
            return -1;
        }
    }

    // TBD - What do we use for our local NormNodeId?
    //       (for now we use automatic, IP addr based assignment or passed in 'id')
    //       a) Use ZMQ Identity somehow?
    //       b) Add function to use iface addr
    //       c) Randomize and implement a NORM session layer
    //          conflict detection/resolution protocol

    norm_session = NormCreateSession(norm_instance, addr, portNumber, localId);
    if (NORM_SESSION_INVALID == norm_session)
    {
        int savedErrno = errno;
        NormDestroyInstance(norm_instance);
        norm_instance = NORM_INSTANCE_INVALID;
        errno         = savedErrno;
        return -1;
    }
    // There's many other useful NORM options that could be applied here
    if (NormIsUnicastAddress(addr))
    {
        NormSetDefaultUnicastNack(norm_session, true);
    }
    else
    {
        // These only apply for multicast sessions
        // NormSetTTL(norm_session, options.multicast_hops);  // ZMQ default is 1
        NormSetTTL(norm_session,
                   255); // since the ZMQ_MULTICAST_HOPS socket option isn't well-supported
        NormSetRxPortReuse(norm_session,
                           true); // port reuse doesn't work for non-connected unicast
        NormSetLoopback(norm_session,
                        true); // needed when multicast users on same machine
        if (NULL != ifacePtr)
        {
            // Note a bad interface may not be caught until sender or receiver start
            // (Since sender/receiver is not yet started, this always succeeds here)
            NormSetMulticastInterface(norm_session, ifacePtr);
        }
    }

    if (recv)
    {
        // The alternative NORM_SYNC_CURRENT here would provide "instant"
        // receiver sync to the sender's _current_ message transmission.
        // NORM_SYNC_STREAM tries to get everything the sender has cached/buffered
        NormSetDefaultSyncPolicy(norm_session, NORM_SYNC_STREAM);
        if (!NormStartReceiver(norm_session, 2 * 1024 * 1024))
        {
            // errno set by whatever failed
            int savedErrno = errno;
            NormDestroyInstance(norm_instance); // session gets closed, too
            norm_session  = NORM_SESSION_INVALID;
            norm_instance = NORM_INSTANCE_INVALID;
            errno         = savedErrno;
            return -1;
        }
        is_receiver = true;
    }

    if (send)
    {
        // Pick a random sender instance id (aka norm sender session id)
        NormSessionId instanceId = NormGetRandomSessionId();
        // TBD - provide "options" for some NORM sender parameters
        if (!NormStartSender(norm_session, instanceId, 2 * 1024 * 1024, 1400, 16, 4))
        {
            // errno set by whatever failed
            int savedErrno = errno;
            NormDestroyInstance(norm_instance); // session gets closed, too
            norm_session  = NORM_SESSION_INVALID;
            norm_instance = NORM_INSTANCE_INVALID;
            errno         = savedErrno;
            return -1;
        }
        NormSetCongestionControl(norm_session, true);
        norm_tx_ready = true;
        is_sender     = true;
        if (NORM_OBJECT_INVALID == (norm_tx_stream = NormStreamOpen(norm_session, 2 * 1024 * 1024)))
        {
            // errno set by whatever failed
            int savedErrno = errno;
            NormDestroyInstance(norm_instance); // session gets closed, too
            norm_session  = NORM_SESSION_INVALID;
            norm_instance = NORM_INSTANCE_INVALID;
            errno         = savedErrno;
            return -1;
        }
    }

    // NormSetMessageTrace(norm_session, true);
    // NormSetDebugLevel(3);
    // NormOpenDebugLog(norm_instance, "normLog.txt");

    return 0; // no error
} // end zmq::norm_engine_t::init()

void zmq::norm_engine_t::shutdown()
{
    // TBD - implement a more graceful shutdown option
    if (is_receiver)
    {
        NormStopReceiver(norm_session);

        // delete any active NormRxStreamState
        rx_pending_list.Destroy();
        rx_ready_list.Destroy();
        msg_ready_list.Destroy();

        is_receiver = false;
    }
    if (is_sender)
    {
        NormStopSender(norm_session);
        is_sender = false;
    }
    if (NORM_SESSION_INVALID != norm_session)
    {
        NormDestroySession(norm_session);
        norm_session = NORM_SESSION_INVALID;
    }
    if (NORM_INSTANCE_INVALID != norm_instance)
    {
        NormStopInstance(norm_instance);
        NormDestroyInstance(norm_instance);
        norm_instance = NORM_INSTANCE_INVALID;
    }
} // end zmq::norm_engine_t::shutdown()

void zmq::norm_engine_t::plug(io_thread_t *io_thread_, session_base_t *session_)
{
    // TBD - we may assign the NORM engine to an io_thread in the future???
    zmq_session = session_;
    if (is_sender)
        zmq_output_ready = true;
    if (is_receiver)
        zmq_input_ready = true;

    fd_t normDescriptor    = NormGetDescriptor(norm_instance);
    norm_descriptor_handle = add_fd(normDescriptor);
    // Set POLLIN for notification of pending NormEvents
    set_pollin(norm_descriptor_handle);

    if (is_sender)
        send_data();

} // end zmq::norm_engine_t::init()

void zmq::norm_engine_t::unplug()
{
    rm_fd(norm_descriptor_handle);

    zmq_session = NULL;
} // end zmq::norm_engine_t::unplug()

void zmq::norm_engine_t::terminate()
{
    unplug();
    shutdown();
    delete this;
}

void zmq::norm_engine_t::restart_output()
{
    // There's new message data available from the session
    zmq_output_ready = true;
    if (norm_tx_ready)
        send_data();

} // end zmq::norm_engine_t::restart_output()

void zmq::norm_engine_t::send_data()
{
    // Here we write as much as is available or we can
    while (zmq_output_ready && norm_tx_ready)
    {
        if (0 == tx_len)
        {
            // Our tx_buffer needs data to send
            // Get more data from encoder
            size_t         space  = BUFFER_SIZE;
            unsigned char *bufPtr = (unsigned char *)tx_buffer;
            tx_len                = zmq_encoder.encode(&bufPtr, space);
            if (0 == tx_len)
            {
                if (tx_first_msg)
                {
                    // We don't need to mark eom/flush until a message is sent
                    tx_first_msg = false;
                }
                else
                {
                    // A prior message was completely written to stream, so
                    // mark end-of-message and possibly flush (to force packet transmission,
                    // even if it's not a full segment so message gets delivered quickly)
                    // NormStreamMarkEom(norm_tx_stream);  // the flush below marks eom
                    // Note NORM_FLUSH_ACTIVE makes NORM fairly chatty for low duty cycle messaging
                    // but makes sure content is delivered quickly.  Positive acknowledgements
                    // with flush override would make NORM more succinct here
                    NormStreamFlush(norm_tx_stream, true, NORM_FLUSH_ACTIVE);
                }
                // Need to pull and load a new message to send
                if (-1 == zmq_session->pull_msg(&tx_msg))
                {
                    // We need to wait for "restart_output()" to be called by ZMQ
                    zmq_output_ready = false;
                    break;
                }
                zmq_encoder.load_msg(&tx_msg);
                // Should we write message size header for NORM to use? Or expect NORM
                // receiver to decode ZMQ message framing format(s)?
                // OK - we need to use a byte to denote when the ZMQ frame is the _first_
                //      frame of a message so it can be decoded properly when a receiver
                //      'syncs' mid-stream.  We key off the the state of the 'more_flag'
                //      I.e.,If  more_flag _was_ false previously, this is the first
                //      frame of a ZMQ message.
                if (tx_more_bit)
                    tx_buffer[0] = (char)0xff; // this is not first frame of message
                else
                    tx_buffer[0] = 0x00; // this is first frame of message
                tx_more_bit = (0 != (tx_msg.flags() & msg_t::more));
                // Go ahead an get a first chunk of the message
                bufPtr++;
                space--;
                tx_len   = 1 + zmq_encoder.encode(&bufPtr, space);
                tx_index = 0;
            }
        }
        // Do we have data in our tx_buffer pending
        if (tx_index < tx_len)
        {
            // We have data in our tx_buffer to send, so write it to the stream
            tx_index += NormStreamWrite(norm_tx_stream, tx_buffer + tx_index, tx_len - tx_index);
            if (tx_index < tx_len)
            {
                // NORM stream buffer full, wait for NORM_TX_QUEUE_VACANCY
                norm_tx_ready = false;
                break;
            }
            tx_len = 0; // all buffered data was written
        }
    } // end while (zmq_output_ready && norm_tx_ready)
} // end zmq::norm_engine_t::send_data()

void zmq::norm_engine_t::in_event()
{
    // This means a NormEvent is pending, so call NormGetNextEvent() and handle
    NormEvent event;
    if (!NormGetNextEvent(norm_instance, &event))
    {
        // NORM has died before we unplugged?!
        zmq_assert(false);
        return;
    }

    switch (event.type)
    {
    case NORM_TX_QUEUE_VACANCY:
    case NORM_TX_QUEUE_EMPTY:
        if (!norm_tx_ready)
        {
            norm_tx_ready = true;
            send_data();
        }
        break;

    case NORM_RX_OBJECT_NEW:
        // break;
    case NORM_RX_OBJECT_UPDATED:
        recv_data(event.object);
        break;

    case NORM_RX_OBJECT_ABORTED:
    {
        NormRxStreamState *rxState = (NormRxStreamState *)NormObjectGetUserData(event.object);
        if (NULL != rxState)
        {
            // Remove the state from the list it's in
            // This is now unnecessary since deletion takes care of list removal
            // but in the interest of being clear ...
            NormRxStreamState::List *list = rxState->AccessList();
            if (NULL != list)
                list->Remove(*rxState);
        }
        delete rxState;
        break;
    }
    case NORM_REMOTE_SENDER_INACTIVE:
        // Here we free resources used for this formerly active sender.
        // Note w/ NORM_SYNC_STREAM, if sender reactivates, we may
        //  get some messages delivered twice.  NORM_SYNC_CURRENT would
        // mitigate that but might miss data at startup. Always tradeoffs.
        // Instead of immediately deleting, we could instead initiate a
        // user configurable timeout here to wait some amount of time
        // after this event to declare the remote sender truly dead
        // and delete its state???
        NormNodeDelete(event.sender);
        break;

    default:
        // We ignore some NORM events
        break;
    }
} // zmq::norm_engine_t::in_event()

bool zmq::norm_engine_t::restart_input()
{
    // TBD - should we check/assert that zmq_input_ready was false???
    zmq_input_ready = true;
    // Process any pending received messages
    if (!msg_ready_list.IsEmpty())
        recv_data(NORM_OBJECT_INVALID);

    return true;
} // end zmq::norm_engine_t::restart_input()

void zmq::norm_engine_t::recv_data(NormObjectHandle object)
{
    if (NORM_OBJECT_INVALID != object)
    {
        // Call result of NORM_RX_OBJECT_UPDATED notification
        // This is a rx_ready indication for a new or existing rx stream
        // First, determine if this is a stream we already know
        zmq_assert(NORM_OBJECT_STREAM == NormObjectGetType(object));
        // Since there can be multiple senders (publishers), we keep
        // state for each separate rx stream.
        NormRxStreamState *rxState = (NormRxStreamState *)NormObjectGetUserData(object);
        if (NULL == rxState)
        {
            // This is a new stream, so create rxState with zmq decoder, etc
            rxState = new (std::nothrow)
                NormRxStreamState(object, options.maxmsgsize, options.zero_copy, options.in_batch_size);
            errno_assert(rxState);

            if (!rxState->Init())
            {
                errno_assert(false);
                delete rxState;
                return;
            }
            NormObjectSetUserData(object, rxState);
        }
        else if (!rxState->IsRxReady())
        {
            // Existing non-ready stream, so remove from pending
            // list to be promoted to rx_ready_list ...
            rx_pending_list.Remove(*rxState);
        }
        if (!rxState->IsRxReady())
        {
            // TBD - prepend up front for immediate service?
            rxState->SetRxReady(true);
            rx_ready_list.Append(*rxState);
        }
    }
    // This loop repeats until we've read all data available from "rx ready" inbound streams
    // and pushed any accumulated messages we can up to the zmq session.
    while (!rx_ready_list.IsEmpty() || (zmq_input_ready && !msg_ready_list.IsEmpty()))
    {
        // Iterate through our rx_ready streams, reading data into the decoder
        // (This services incoming "rx ready" streams in a round-robin fashion)
        NormRxStreamState::List::Iterator iterator(rx_ready_list);
        NormRxStreamState *               rxState;
        while (NULL != (rxState = iterator.GetNextItem()))
        {
            switch (rxState->Decode())
            {
            case 1: // msg completed
                // Complete message decoded, move this stream to msg_ready_list
                // to push the message up to the session below.  Note the stream
                // will be returned to the "rx_ready_list" after that's done
                rx_ready_list.Remove(*rxState);
                msg_ready_list.Append(*rxState);
                continue;

            case -1: // decoding error (shouldn't happen w/ NORM, but ...)
                // We need to re-sync this stream (decoder buffer was reset)
                rxState->SetSync(false);
                break;

            default: // 0 - need more data
                break;
            }
            // Get more data from this stream
            NormObjectHandle stream = rxState->GetStreamHandle();
            // First, make sure we're in sync ...
            while (!rxState->InSync())
            {
                // seek NORM message start
                if (!NormStreamSeekMsgStart(stream))
                {
                    // Need to wait for more data
                    break;
                }
                // read message 'flag' byte to see if this it's a 'final' frame
                char         syncFlag;
                unsigned int numBytes = 1;
                if (!NormStreamRead(stream, &syncFlag, &numBytes))
                {
                    // broken stream (shouldn't happen after seek msg start?)
                    zmq_assert(false);
                    continue;
                }
                if (0 == numBytes)
                {
                    // This probably shouldn't happen either since we found msg start
                    // Need to wait for more data
                    break;
                }
                if (0 == syncFlag)
                    rxState->SetSync(true);
                // else keep seeking ...
            } // end while(!rxState->InSync())
            if (!rxState->InSync())
            {
                // Need more data for this stream, so remove from "rx ready"
                // list and iterate to next "rx ready" stream
                rxState->SetRxReady(false);
                // Move from rx_ready_list to rx_pending_list
                rx_ready_list.Remove(*rxState);
                rx_pending_list.Append(*rxState);
                continue;
            }
            // Now we're actually ready to read data from the NORM stream to the zmq_decoder
            // the underlying zmq_decoder->get_buffer() call sets how much is needed.
            unsigned int numBytes = rxState->GetBytesNeeded();
            if (!NormStreamRead(stream, rxState->AccessBuffer(), &numBytes))
            {
                // broken NORM stream, so re-sync
                rxState->Init(); // TBD - check result
                // This will retry syncing, and getting data from this stream
                // since we don't increment the "it" iterator
                continue;
            }
            rxState->IncrementBufferCount(numBytes);
            if (0 == numBytes)
            {
                // All the data available has been read
                // Need to wait for NORM_RX_OBJECT_UPDATED for this stream
                rxState->SetRxReady(false);
                // Move from rx_ready_list to rx_pending_list
                rx_ready_list.Remove(*rxState);
                rx_pending_list.Append(*rxState);
            }
        } // end while(NULL != (rxState = iterator.GetNextItem()))

        if (zmq_input_ready)
        {
            // At this point, we've made a pass through the "rx_ready" stream list
            // Now make a pass through the "msg_pending" list (if the zmq session
            // ready for more input).  This may possibly return streams back to
            // the "rx ready" stream list after their pending message is handled
            NormRxStreamState::List::Iterator iterator(msg_ready_list);
            NormRxStreamState *               rxState;
            while (NULL != (rxState = iterator.GetNextItem()))
            {
                msg_t *msg = rxState->AccessMsg();
                int    rc  = zmq_session->push_msg(msg);
                if (-1 == rc)
                {
                    if (EAGAIN == errno)
                    {
                        // need to wait until session calls "restart_input()"
                        zmq_input_ready = false;
                        break;
                    }
                    else
                    {
                        // session rejected message?
                        // TBD - handle this better
                        zmq_assert(false);
                    }
                }
                // else message was accepted.
                msg_ready_list.Remove(*rxState);
                if (rxState->IsRxReady()) // Move back to "rx_ready" list to read more data
                    rx_ready_list.Append(*rxState);
                else // Move back to "rx_pending" list until NORM_RX_OBJECT_UPDATED
                    msg_ready_list.Append(*rxState);
            } // end while(NULL != (rxState = iterator.GetNextItem()))
        }     // end if (zmq_input_ready)
    }         // end while ((!rx_ready_list.empty() || (zmq_input_ready && !msg_ready_list.empty()))

    // Alert zmq of the messages we have pushed up
    zmq_session->flush();

} // end zmq::norm_engine_t::recv_data()

zmq::norm_engine_t::NormRxStreamState::NormRxStreamState(NormObjectHandle normStream, int64_t maxMsgSize, bool zeroCopy,
                                                         int inBatchSize)
    : norm_stream(normStream), max_msg_size(maxMsgSize), zero_copy(zeroCopy), in_batch_size(inBatchSize),
      in_sync(false), rx_ready(false), zmq_decoder(NULL), skip_norm_sync(false), buffer_ptr(NULL), buffer_size(0),
      buffer_count(0), prev(NULL), next(NULL), list(NULL)
{
}

zmq::norm_engine_t::NormRxStreamState::~NormRxStreamState()
{
    if (NULL != zmq_decoder)
    {
        delete zmq_decoder;
        zmq_decoder = NULL;
    }
    if (NULL != list)
    {
        list->Remove(*this);
        list = NULL;
    }
}

bool zmq::norm_engine_t::NormRxStreamState::Init()
{
    in_sync        = false;
    skip_norm_sync = false;
    if (NULL != zmq_decoder)
        delete zmq_decoder;
    zmq_decoder = new (std::nothrow) v2_decoder_t(in_batch_size, max_msg_size, zero_copy);
    alloc_assert(zmq_decoder);
    if (NULL != zmq_decoder)
    {
        buffer_count = 0;
        buffer_size  = 0;
        zmq_decoder->get_buffer(&buffer_ptr, &buffer_size);
        return true;
    }
    else
    {
        return false;
    }
} // end zmq::norm_engine_t::NormRxStreamState::Init()

// This decodes any pending data sitting in our stream decoder buffer
// It returns 1 upon message completion, -1 on error, 1 on msg completion
int zmq::norm_engine_t::NormRxStreamState::Decode()
{
    // If we have pending bytes to decode, process those first
    while (buffer_count > 0)
    {
        // There's pending data for the decoder to decode
        size_t processed = 0;

        // This a bit of a kludgy approach used to weed
        // out the NORM ZMQ message transport "syncFlag" byte
        // from the ZMQ message stream being decoded (but it works!)
        if (skip_norm_sync)
        {
            buffer_ptr++;
            buffer_count--;
            skip_norm_sync = false;
        }

        int rc = zmq_decoder->decode(buffer_ptr, buffer_count, processed);
        buffer_ptr += processed;
        buffer_count -= processed;
        switch (rc)
        {
        case 1:
            // msg completed
            if (0 == buffer_count)
            {
                buffer_size = 0;
                zmq_decoder->get_buffer(&buffer_ptr, &buffer_size);
            }
            skip_norm_sync = true;
            return 1;
        case -1:
            // decoder error (reset decoder and state variables)
            in_sync        = false;
            skip_norm_sync = false; // will get consumed by norm sync check
            Init();
            break;

        case 0:
            // need more data, keep decoding until buffer exhausted
            break;
        }
    }
    // Reset buffer pointer/count for next read
    buffer_count = 0;
    buffer_size  = 0;
    zmq_decoder->get_buffer(&buffer_ptr, &buffer_size);
    return 0; //  need more data

} // end zmq::norm_engine_t::NormRxStreamState::Decode()

zmq::norm_engine_t::NormRxStreamState::List::List() : head(NULL), tail(NULL) {}

zmq::norm_engine_t::NormRxStreamState::List::~List() { Destroy(); }

void zmq::norm_engine_t::NormRxStreamState::List::Destroy()
{
    NormRxStreamState *item = head;
    while (NULL != item)
    {
        Remove(*item);
        delete item;
        item = head;
    }
} // end zmq::norm_engine_t::NormRxStreamState::List::Destroy()

void zmq::norm_engine_t::NormRxStreamState::List::Append(NormRxStreamState &item)
{
    item.prev = tail;
    if (NULL != tail)
        tail->next = &item;
    else
        head = &item;
    item.next = NULL;
    tail      = &item;
    item.list = this;
} // end zmq::norm_engine_t::NormRxStreamState::List::Append()

void zmq::norm_engine_t::NormRxStreamState::List::Remove(NormRxStreamState &item)
{
    if (NULL != item.prev)
        item.prev->next = item.next;
    else
        head = item.next;
    if (NULL != item.next)
        item.next->prev = item.prev;
    else
        tail = item.prev;
    item.prev = item.next = NULL;
    item.list             = NULL;
} // end zmq::norm_engine_t::NormRxStreamState::List::Remove()

zmq::norm_engine_t::NormRxStreamState::List::Iterator::Iterator(const List &list) : next_item(list.head) {}

zmq::norm_engine_t::NormRxStreamState *zmq::norm_engine_t::NormRxStreamState::List::Iterator::GetNextItem()
{
    NormRxStreamState *nextItem = next_item;
    if (NULL != nextItem)
        next_item = nextItem->next;
    return nextItem;
} // end zmq::norm_engine_t::NormRxStreamState::List::Iterator::GetNextItem()

const zmq::endpoint_uri_pair_t &zmq::norm_engine_t::get_endpoint() const { return _empty_endpoint; }

#endif // ZMQ_HAVE_NORM

//========= end of norm_engine.cpp ============

//========= begin of null_mechanism.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"

#include <stddef.h>
#include <stdlib.h>
#include <string.h>

// ans ignore: #include "err.hpp"
// ans ignore: #include "msg.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "null_mechanism.hpp"

const char   error_command_name[]   = "\5ERROR";
const size_t error_command_name_len = sizeof(error_command_name) - 1;
const size_t error_reason_len_size  = 1;

const char   ready_command_name[]   = "\5READY";
const size_t ready_command_name_len = sizeof(ready_command_name) - 1;

zmq::null_mechanism_t::null_mechanism_t(session_base_t *session_, const std::string &peer_address_,
                                        const options_t &options_)
    : mechanism_base_t(session_, options_), zap_client_t(session_, peer_address_, options_), _ready_command_sent(false),
      _error_command_sent(false), _ready_command_received(false), _error_command_received(false),
      _zap_request_sent(false), _zap_reply_received(false)
{
}

zmq::null_mechanism_t::~null_mechanism_t() {}

int zmq::null_mechanism_t::next_handshake_command(msg_t *msg_)
{
    if (_ready_command_sent || _error_command_sent)
    {
        errno = EAGAIN;
        return -1;
    }

    if (zap_required() && !_zap_reply_received)
    {
        if (_zap_request_sent)
        {
            errno = EAGAIN;
            return -1;
        }
        //  Given this is a backward-incompatible change, it's behind a socket
        //  option disabled by default.
        int rc = session->zap_connect();
        if (rc == -1 && options.zap_enforce_domain)
        {
            session->get_socket()->event_handshake_failed_no_detail(session->get_endpoint(), EFAULT);
            return -1;
        }
        if (rc == 0)
        {
            send_zap_request();
            _zap_request_sent = true;

            //  TODO actually, it is quite unlikely that we can read the ZAP
            //  reply already, but removing this has some strange side-effect
            //  (probably because the pipe's in_active flag is true until a read
            //  is attempted)
            rc = receive_and_process_zap_reply();
            if (rc != 0)
                return -1;

            _zap_reply_received = true;
        }
    }

    if (_zap_reply_received && status_code != "200")
    {
        _error_command_sent = true;
        if (status_code != "300")
        {
            const size_t status_code_len = 3;
            const int    rc = msg_->init_size(error_command_name_len + error_reason_len_size + status_code_len);
            zmq_assert(rc == 0);
            unsigned char *msg_data = static_cast<unsigned char *>(msg_->data());
            memcpy(msg_data, error_command_name, error_command_name_len);
            msg_data += error_command_name_len;
            *msg_data = status_code_len;
            msg_data += error_reason_len_size;
            memcpy(msg_data, status_code.c_str(), status_code_len);
            return 0;
        }
        errno = EAGAIN;
        return -1;
    }

    make_command_with_basic_properties(msg_, ready_command_name, ready_command_name_len);

    _ready_command_sent = true;

    return 0;
}

int zmq::null_mechanism_t::process_handshake_command(msg_t *msg_)
{
    if (_ready_command_received || _error_command_received)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNEXPECTED_COMMAND);
        errno = EPROTO;
        return -1;
    }

    const unsigned char *cmd_data  = static_cast<unsigned char *>(msg_->data());
    const size_t         data_size = msg_->size();

    int rc = 0;
    if (data_size >= ready_command_name_len && !memcmp(cmd_data, ready_command_name, ready_command_name_len))
        rc = process_ready_command(cmd_data, data_size);
    else if (data_size >= error_command_name_len && !memcmp(cmd_data, error_command_name, error_command_name_len))
        rc = process_error_command(cmd_data, data_size);
    else
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNEXPECTED_COMMAND);
        errno = EPROTO;
        rc    = -1;
    }

    if (rc == 0)
    {
        rc = msg_->close();
        errno_assert(rc == 0);
        rc = msg_->init();
        errno_assert(rc == 0);
    }
    return rc;
}

int zmq::null_mechanism_t::process_ready_command(const unsigned char *cmd_data_, size_t data_size_)
{
    _ready_command_received = true;
    return parse_metadata(cmd_data_ + ready_command_name_len, data_size_ - ready_command_name_len);
}

int zmq::null_mechanism_t::process_error_command(const unsigned char *cmd_data_, size_t data_size_)
{
    const size_t fixed_prefix_size = error_command_name_len + error_reason_len_size;
    if (data_size_ < fixed_prefix_size)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_ERROR);

        errno = EPROTO;
        return -1;
    }
    const size_t error_reason_len = static_cast<size_t>(cmd_data_[error_command_name_len]);
    if (error_reason_len > data_size_ - fixed_prefix_size)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_ERROR);

        errno = EPROTO;
        return -1;
    }
    const char *error_reason = reinterpret_cast<const char *>(cmd_data_) + fixed_prefix_size;
    handle_error_reason(error_reason, error_reason_len);
    _error_command_received = true;
    return 0;
}

int zmq::null_mechanism_t::zap_msg_available()
{
    if (_zap_reply_received)
    {
        errno = EFSM;
        return -1;
    }
    const int rc = receive_and_process_zap_reply();
    if (rc == 0)
        _zap_reply_received = true;
    return rc == -1 ? -1 : 0;
}

zmq::mechanism_t::status_t zmq::null_mechanism_t::status() const
{
    if (_ready_command_sent && _ready_command_received)
        return ready;

    const bool command_sent     = _ready_command_sent || _error_command_sent;
    const bool command_received = _ready_command_received || _error_command_received;
    return command_sent && command_received ? error : handshaking;
}

void zmq::null_mechanism_t::send_zap_request() { zap_client_t::send_zap_request("NULL", 4, NULL, NULL, 0); }

//========= end of null_mechanism.cpp ============

//========= begin of object.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <stdarg.h>
#include <string.h>

// ans ignore: #include "object.hpp"
// ans ignore: #include "ctx.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "pipe.hpp"
// ans ignore: #include "io_thread.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "socket_base.hpp"

zmq::object_t::object_t(ctx_t *ctx_, uint32_t tid_) : _ctx(ctx_), _tid(tid_) {}

zmq::object_t::object_t(object_t *parent_) : _ctx(parent_->_ctx), _tid(parent_->_tid) {}

zmq::object_t::~object_t() {}

uint32_t zmq::object_t::get_tid() { return _tid; }

void zmq::object_t::set_tid(uint32_t id_) { _tid = id_; }

zmq::ctx_t *zmq::object_t::get_ctx() { return _ctx; }

void zmq::object_t::process_command(command_t &cmd_)
{
    switch (cmd_.type)
    {
    case command_t::activate_read:
        process_activate_read();
        break;

    case command_t::activate_write:
        process_activate_write(cmd_.args.activate_write.msgs_read);
        break;

    case command_t::stop:
        process_stop();
        break;

    case command_t::plug:
        process_plug();
        process_seqnum();
        break;

    case command_t::own:
        process_own(cmd_.args.own.object);
        process_seqnum();
        break;

    case command_t::attach:
        process_attach(cmd_.args.attach.engine);
        process_seqnum();
        break;

    case command_t::bind:
        process_bind(cmd_.args.bind.pipe);
        process_seqnum();
        break;

    case command_t::hiccup:
        process_hiccup(cmd_.args.hiccup.pipe);
        break;

    case command_t::pipe_peer_stats:
        process_pipe_peer_stats(cmd_.args.pipe_peer_stats.queue_count, cmd_.args.pipe_peer_stats.socket_base,
                                cmd_.args.pipe_peer_stats.endpoint_pair);
        break;

    case command_t::pipe_stats_publish:
        process_pipe_stats_publish(cmd_.args.pipe_stats_publish.outbound_queue_count,
                                   cmd_.args.pipe_stats_publish.inbound_queue_count,
                                   cmd_.args.pipe_stats_publish.endpoint_pair);
        break;

    case command_t::pipe_term:
        process_pipe_term();
        break;

    case command_t::pipe_term_ack:
        process_pipe_term_ack();
        break;

    case command_t::pipe_hwm:
        process_pipe_hwm(cmd_.args.pipe_hwm.inhwm, cmd_.args.pipe_hwm.outhwm);
        break;

    case command_t::term_req:
        process_term_req(cmd_.args.term_req.object);
        break;

    case command_t::term:
        process_term(cmd_.args.term.linger);
        break;

    case command_t::term_ack:
        process_term_ack();
        break;

    case command_t::term_endpoint:
        process_term_endpoint(cmd_.args.term_endpoint.endpoint);
        break;

    case command_t::reap:
        process_reap(cmd_.args.reap.socket);
        break;

    case command_t::reaped:
        process_reaped();
        break;

    case command_t::inproc_connected:
        process_seqnum();
        break;

    case command_t::done:
    default:
        zmq_assert(false);
    }
}

int zmq::object_t::register_endpoint(const char *addr_, const endpoint_t &endpoint_)
{
    return _ctx->register_endpoint(addr_, endpoint_);
}

int zmq::object_t::unregister_endpoint(const std::string &addr_, socket_base_t *socket_)
{
    return _ctx->unregister_endpoint(addr_, socket_);
}

void zmq::object_t::unregister_endpoints(socket_base_t *socket_) { return _ctx->unregister_endpoints(socket_); }

zmq::endpoint_t zmq::object_t::find_endpoint(const char *addr_) { return _ctx->find_endpoint(addr_); }

void zmq::object_t::pend_connection(const std::string &addr_, const endpoint_t &endpoint_, pipe_t **pipes_)
{
    _ctx->pend_connection(addr_, endpoint_, pipes_);
}

void zmq::object_t::connect_pending(const char *addr_, zmq::socket_base_t *bind_socket_)
{
    return _ctx->connect_pending(addr_, bind_socket_);
}

void zmq::object_t::destroy_socket(socket_base_t *socket_) { _ctx->destroy_socket(socket_); }

zmq::io_thread_t *zmq::object_t::choose_io_thread(uint64_t affinity_) { return _ctx->choose_io_thread(affinity_); }

void zmq::object_t::send_stop()
{
    //  'stop' command goes always from administrative thread to
    //  the current object.
    command_t cmd;
    cmd.destination = this;
    cmd.type        = command_t::stop;
    _ctx->send_command(_tid, cmd);
}

void zmq::object_t::send_plug(own_t *destination_, bool inc_seqnum_)
{
    if (inc_seqnum_)
        destination_->inc_seqnum();

    command_t cmd;
    cmd.destination = destination_;
    cmd.type        = command_t::plug;
    send_command(cmd);
}

void zmq::object_t::send_own(own_t *destination_, own_t *object_)
{
    destination_->inc_seqnum();
    command_t cmd;
    cmd.destination     = destination_;
    cmd.type            = command_t::own;
    cmd.args.own.object = object_;
    send_command(cmd);
}

void zmq::object_t::send_attach(session_base_t *destination_, i_engine *engine_, bool inc_seqnum_)
{
    if (inc_seqnum_)
        destination_->inc_seqnum();

    command_t cmd;
    cmd.destination        = destination_;
    cmd.type               = command_t::attach;
    cmd.args.attach.engine = engine_;
    send_command(cmd);
}

void zmq::object_t::send_bind(own_t *destination_, pipe_t *pipe_, bool inc_seqnum_)
{
    if (inc_seqnum_)
        destination_->inc_seqnum();

    command_t cmd;
    cmd.destination    = destination_;
    cmd.type           = command_t::bind;
    cmd.args.bind.pipe = pipe_;
    send_command(cmd);
}

void zmq::object_t::send_activate_read(pipe_t *destination_)
{
    command_t cmd;
    cmd.destination = destination_;
    cmd.type        = command_t::activate_read;
    send_command(cmd);
}

void zmq::object_t::send_activate_write(pipe_t *destination_, uint64_t msgs_read_)
{
    command_t cmd;
    cmd.destination                   = destination_;
    cmd.type                          = command_t::activate_write;
    cmd.args.activate_write.msgs_read = msgs_read_;
    send_command(cmd);
}

void zmq::object_t::send_hiccup(pipe_t *destination_, void *pipe_)
{
    command_t cmd;
    cmd.destination      = destination_;
    cmd.type             = command_t::hiccup;
    cmd.args.hiccup.pipe = pipe_;
    send_command(cmd);
}

void zmq::object_t::send_pipe_peer_stats(pipe_t *destination_, uint64_t queue_count_, own_t *socket_base_,
                                         endpoint_uri_pair_t *endpoint_pair_)
{
    command_t cmd;
    cmd.destination                        = destination_;
    cmd.type                               = command_t::pipe_peer_stats;
    cmd.args.pipe_peer_stats.queue_count   = queue_count_;
    cmd.args.pipe_peer_stats.socket_base   = socket_base_;
    cmd.args.pipe_peer_stats.endpoint_pair = endpoint_pair_;
    send_command(cmd);
}

void zmq::object_t::send_pipe_stats_publish(own_t *destination_, uint64_t outbound_queue_count_,
                                            uint64_t inbound_queue_count_, endpoint_uri_pair_t *endpoint_pair_)
{
    command_t cmd;
    cmd.destination                                  = destination_;
    cmd.type                                         = command_t::pipe_stats_publish;
    cmd.args.pipe_stats_publish.outbound_queue_count = outbound_queue_count_;
    cmd.args.pipe_stats_publish.inbound_queue_count  = inbound_queue_count_;
    cmd.args.pipe_stats_publish.endpoint_pair        = endpoint_pair_;
    send_command(cmd);
}

void zmq::object_t::send_pipe_term(pipe_t *destination_)
{
    command_t cmd;
    cmd.destination = destination_;
    cmd.type        = command_t::pipe_term;
    send_command(cmd);
}

void zmq::object_t::send_pipe_term_ack(pipe_t *destination_)
{
    command_t cmd;
    cmd.destination = destination_;
    cmd.type        = command_t::pipe_term_ack;
    send_command(cmd);
}

void zmq::object_t::send_pipe_hwm(pipe_t *destination_, int inhwm_, int outhwm_)
{
    command_t cmd;
    cmd.destination          = destination_;
    cmd.type                 = command_t::pipe_hwm;
    cmd.args.pipe_hwm.inhwm  = inhwm_;
    cmd.args.pipe_hwm.outhwm = outhwm_;
    send_command(cmd);
}

void zmq::object_t::send_term_req(own_t *destination_, own_t *object_)
{
    command_t cmd;
    cmd.destination          = destination_;
    cmd.type                 = command_t::term_req;
    cmd.args.term_req.object = object_;
    send_command(cmd);
}

void zmq::object_t::send_term(own_t *destination_, int linger_)
{
    command_t cmd;
    cmd.destination      = destination_;
    cmd.type             = command_t::term;
    cmd.args.term.linger = linger_;
    send_command(cmd);
}

void zmq::object_t::send_term_ack(own_t *destination_)
{
    command_t cmd;
    cmd.destination = destination_;
    cmd.type        = command_t::term_ack;
    send_command(cmd);
}

void zmq::object_t::send_term_endpoint(own_t *destination_, std::string *endpoint_)
{
    command_t cmd;
    cmd.destination                 = destination_;
    cmd.type                        = command_t::term_endpoint;
    cmd.args.term_endpoint.endpoint = endpoint_;
    send_command(cmd);
}

void zmq::object_t::send_reap(class socket_base_t *socket_)
{
    command_t cmd;
    cmd.destination      = _ctx->get_reaper();
    cmd.type             = command_t::reap;
    cmd.args.reap.socket = socket_;
    send_command(cmd);
}

void zmq::object_t::send_reaped()
{
    command_t cmd;
    cmd.destination = _ctx->get_reaper();
    cmd.type        = command_t::reaped;
    send_command(cmd);
}

void zmq::object_t::send_inproc_connected(zmq::socket_base_t *socket_)
{
    command_t cmd;
    cmd.destination = socket_;
    cmd.type        = command_t::inproc_connected;
    send_command(cmd);
}

void zmq::object_t::send_done()
{
    command_t cmd;
    cmd.destination = NULL;
    cmd.type        = command_t::done;
    _ctx->send_command(ctx_t::term_tid, cmd);
}

void zmq::object_t::process_stop() { zmq_assert(false); }

void zmq::object_t::process_plug() { zmq_assert(false); }

void zmq::object_t::process_own(own_t *) { zmq_assert(false); }

void zmq::object_t::process_attach(i_engine *) { zmq_assert(false); }

void zmq::object_t::process_bind(pipe_t *) { zmq_assert(false); }

void zmq::object_t::process_activate_read() { zmq_assert(false); }

void zmq::object_t::process_activate_write(uint64_t) { zmq_assert(false); }

void zmq::object_t::process_hiccup(void *) { zmq_assert(false); }

void zmq::object_t::process_pipe_peer_stats(uint64_t, own_t *, endpoint_uri_pair_t *) { zmq_assert(false); }

void zmq::object_t::process_pipe_stats_publish(uint64_t, uint64_t, endpoint_uri_pair_t *) { zmq_assert(false); }

void zmq::object_t::process_pipe_term() { zmq_assert(false); }

void zmq::object_t::process_pipe_term_ack() { zmq_assert(false); }

void zmq::object_t::process_pipe_hwm(int, int) { zmq_assert(false); }

void zmq::object_t::process_term_req(own_t *) { zmq_assert(false); }

void zmq::object_t::process_term(int) { zmq_assert(false); }

void zmq::object_t::process_term_ack() { zmq_assert(false); }

void zmq::object_t::process_term_endpoint(std::string *) { zmq_assert(false); }

void zmq::object_t::process_reap(class socket_base_t *) { zmq_assert(false); }

void zmq::object_t::process_reaped() { zmq_assert(false); }

void zmq::object_t::process_seqnum() { zmq_assert(false); }

void zmq::object_t::send_command(command_t &cmd_) { _ctx->send_command(cmd_.destination->get_tid(), cmd_); }

//========= end of object.cpp ============

//========= begin of options.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <limits.h>
#include <set>
#include <string.h>

// ans ignore: #include "options.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "macros.hpp"

#ifndef ZMQ_HAVE_WINDOWS
#include <net/if.h>
#endif

#if defined IFNAMSIZ
#define BINDDEVSIZ IFNAMSIZ
#else
#define BINDDEVSIZ 16
#endif

static int sockopt_invalid()
{
#if defined(ZMQ_ACT_MILITANT)
    zmq_assert(false);
#endif
    errno = EINVAL;
    return -1;
}

int zmq::do_getsockopt(void *const optval_, size_t *const optvallen_, const std::string &value_)
{
    return do_getsockopt(optval_, optvallen_, value_.c_str(), value_.size() + 1);
}

int zmq::do_getsockopt(void *const optval_, size_t *const optvallen_, const void *value_, const size_t value_len_)
{
    // TODO behaviour is inconsistent with options_t::getsockopt; there, an
    // *exact* length match is required except for string-like (but not the
    // CURVE keys!) (and therefore null-ing remaining memory is a no-op, see
    // comment below)
    if (*optvallen_ < value_len_)
    {
        return sockopt_invalid();
    }
    memcpy(optval_, value_, value_len_);
    // TODO why is the remaining memory null-ed?
    memset(static_cast<char *>(optval_) + value_len_, 0, *optvallen_ - value_len_);
    *optvallen_ = value_len_;
    return 0;
}

#ifdef ZMQ_HAVE_CURVE
static int do_getsockopt_curve_key(void *const optval_, size_t *const optvallen_,
                                   const uint8_t (&curve_key_)[CURVE_KEYSIZE])
{
    if (*optvallen_ == CURVE_KEYSIZE)
    {
        memcpy(optval_, curve_key_, CURVE_KEYSIZE);
        return 0;
    }
    if (*optvallen_ == CURVE_KEYSIZE_Z85 + 1)
    {
        zmq_z85_encode(static_cast<char *>(optval_), curve_key_, CURVE_KEYSIZE);
        return 0;
    }
    return sockopt_invalid();
}
#endif

template <typename T> int do_setsockopt(const void *const optval_, const size_t optvallen_, T *const out_value_)
{
    if (optvallen_ == sizeof(T))
    {
        memcpy(out_value_, optval_, sizeof(T));
        return 0;
    }
    return sockopt_invalid();
}

int zmq::do_setsockopt_int_as_bool_strict(const void *const optval_, const size_t optvallen_, bool *const out_value_)
{
    // TODO handling of values other than 0 or 1 is not consistent,
    // here it is disallowed, but for other options such as
    // ZMQ_ROUTER_RAW any positive value is accepted
    int value = -1;
    if (do_setsockopt(optval_, optvallen_, &value) == -1)
        return -1;
    if (value == 0 || value == 1)
    {
        *out_value_ = (value != 0);
        return 0;
    }
    return sockopt_invalid();
}

int zmq::do_setsockopt_int_as_bool_relaxed(const void *const optval_, const size_t optvallen_, bool *const out_value_)
{
    int value = -1;
    if (do_setsockopt(optval_, optvallen_, &value) == -1)
        return -1;
    *out_value_ = (value != 0);
    return 0;
}

static int do_setsockopt_string_allow_empty_strict(const void *const optval_, const size_t optvallen_,
                                                   std::string *const out_value_, const size_t max_len_)
{
    // TODO why is optval_ != NULL not allowed in case of optvallen_== 0?
    // TODO why are empty strings allowed for some socket options, but not for others?
    if (optval_ == NULL && optvallen_ == 0)
    {
        out_value_->clear();
        return 0;
    }
    if (optval_ != NULL && optvallen_ > 0 && optvallen_ <= max_len_)
    {
        out_value_->assign(static_cast<const char *>(optval_), optvallen_);
        return 0;
    }
    return sockopt_invalid();
}

static int do_setsockopt_string_allow_empty_relaxed(const void *const optval_, const size_t optvallen_,
                                                    std::string *const out_value_, const size_t max_len_)
{
    // TODO use either do_setsockopt_string_allow_empty_relaxed or
    // do_setsockopt_string_allow_empty_strict everywhere
    if (optvallen_ > 0 && optvallen_ <= max_len_)
    {
        out_value_->assign(static_cast<const char *>(optval_), optvallen_);
        return 0;
    }
    return sockopt_invalid();
}

template <typename T> int do_setsockopt_set(const void *const optval_, const size_t optvallen_, std::set<T> *const set_)
{
    if (optvallen_ == 0 && optval_ == NULL)
    {
        set_->clear();
        return 0;
    }
    if (optvallen_ == sizeof(T) && optval_ != NULL)
    {
        set_->insert(*(static_cast<const T *>(optval_)));
        return 0;
    }
    return sockopt_invalid();
}

// TODO why is 1000 a sensible default?
const int default_hwm = 1000;

zmq::options_t::options_t()
    : sndhwm(default_hwm), rcvhwm(default_hwm), affinity(0), routing_id_size(0), rate(100), recovery_ivl(10000),
      multicast_hops(1), multicast_maxtpdu(1500), sndbuf(-1), rcvbuf(-1), tos(0), type(-1), linger(-1),
      connect_timeout(0), tcp_maxrt(0), reconnect_ivl(100), reconnect_ivl_max(0), backlog(100), maxmsgsize(-1),
      rcvtimeo(-1), sndtimeo(-1), ipv6(false), immediate(0), filter(false), invert_matching(false),
      recv_routing_id(false), raw_socket(false), raw_notify(true), tcp_keepalive(-1), tcp_keepalive_cnt(-1),
      tcp_keepalive_idle(-1), tcp_keepalive_intvl(-1), mechanism(ZMQ_NULL), as_server(0),
      gss_principal_nt(ZMQ_GSSAPI_NT_HOSTBASED), gss_service_principal_nt(ZMQ_GSSAPI_NT_HOSTBASED),
      gss_plaintext(false), socket_id(0), conflate(false), handshake_ivl(30000), connected(false), heartbeat_ttl(0),
      heartbeat_interval(0), heartbeat_timeout(-1), use_fd(-1), zap_enforce_domain(false), loopback_fastpath(false),
      multicast_loop(true), in_batch_size(8192), out_batch_size(8192), zero_copy(true), router_notify(0),
      monitor_event_version(1)
{
    memset(curve_public_key, 0, CURVE_KEYSIZE);
    memset(curve_secret_key, 0, CURVE_KEYSIZE);
    memset(curve_server_key, 0, CURVE_KEYSIZE);
#if defined ZMQ_HAVE_VMCI
    vmci_buffer_size     = 0;
    vmci_buffer_min_size = 0;
    vmci_buffer_max_size = 0;
    vmci_connect_timeout = -1;
#endif
}

int zmq::options_t::set_curve_key(uint8_t *destination_, const void *optval_, size_t optvallen_)
{
    switch (optvallen_)
    {
    case CURVE_KEYSIZE:
        memcpy(destination_, optval_, optvallen_);
        mechanism = ZMQ_CURVE;
        return 0;

    case CURVE_KEYSIZE_Z85 + 1:
        if (zmq_z85_decode(destination_, reinterpret_cast<const char *>(optval_)))
        {
            mechanism = ZMQ_CURVE;
            return 0;
        }
        break;

    case CURVE_KEYSIZE_Z85:
        char z85_key[CURVE_KEYSIZE_Z85 + 1];
        memcpy(z85_key, reinterpret_cast<const char *>(optval_), optvallen_);
        z85_key[CURVE_KEYSIZE_Z85] = 0;
        if (zmq_z85_decode(destination_, z85_key))
        {
            mechanism = ZMQ_CURVE;
            return 0;
        }
        break;

    default:
        break;
    }
    return -1;
}

const int deciseconds_per_millisecond = 100;

int zmq::options_t::setsockopt(int option_, const void *optval_, size_t optvallen_)
{
    bool is_int = (optvallen_ == sizeof(int));
    int  value  = 0;
    if (is_int)
        memcpy(&value, optval_, sizeof(int));
#if defined(ZMQ_ACT_MILITANT)
    bool malformed = true; //  Did caller pass a bad option value?
#endif

    switch (option_)
    {
    case ZMQ_SNDHWM:
        if (is_int && value >= 0)
        {
            sndhwm = value;
            return 0;
        }
        break;

    case ZMQ_RCVHWM:
        if (is_int && value >= 0)
        {
            rcvhwm = value;
            return 0;
        }
        break;

    case ZMQ_AFFINITY:
        return do_setsockopt(optval_, optvallen_, &affinity);

    case ZMQ_ROUTING_ID:
        //  Routing id is any binary string from 1 to 255 octets
        if (optvallen_ > 0 && optvallen_ <= UCHAR_MAX)
        {
            routing_id_size = static_cast<unsigned char>(optvallen_);
            memcpy(routing_id, optval_, routing_id_size);
            return 0;
        }
        break;

    case ZMQ_RATE:
        if (is_int && value > 0)
        {
            rate = value;
            return 0;
        }
        break;

    case ZMQ_RECOVERY_IVL:
        if (is_int && value >= 0)
        {
            recovery_ivl = value;
            return 0;
        }
        break;

    case ZMQ_SNDBUF:
        if (is_int && value >= -1)
        {
            sndbuf = value;
            return 0;
        }
        break;

    case ZMQ_RCVBUF:
        if (is_int && value >= -1)
        {
            rcvbuf = value;
            return 0;
        }
        break;

    case ZMQ_TOS:
        if (is_int && value >= 0)
        {
            tos = value;
            return 0;
        }
        break;

    case ZMQ_LINGER:
        if (is_int && value >= -1)
        {
            linger.store(value);
            return 0;
        }
        break;

    case ZMQ_CONNECT_TIMEOUT:
        if (is_int && value >= 0)
        {
            connect_timeout = value;
            return 0;
        }
        break;

    case ZMQ_TCP_MAXRT:
        if (is_int && value >= 0)
        {
            tcp_maxrt = value;
            return 0;
        }
        break;

    case ZMQ_RECONNECT_IVL:
        if (is_int && value >= -1)
        {
            reconnect_ivl = value;
            return 0;
        }
        break;

    case ZMQ_RECONNECT_IVL_MAX:
        if (is_int && value >= 0)
        {
            reconnect_ivl_max = value;
            return 0;
        }
        break;

    case ZMQ_BACKLOG:
        if (is_int && value >= 0)
        {
            backlog = value;
            return 0;
        }
        break;

    case ZMQ_MAXMSGSIZE:
        return do_setsockopt(optval_, optvallen_, &maxmsgsize);

    case ZMQ_MULTICAST_HOPS:
        if (is_int && value > 0)
        {
            multicast_hops = value;
            return 0;
        }
        break;

    case ZMQ_MULTICAST_MAXTPDU:
        if (is_int && value > 0)
        {
            multicast_maxtpdu = value;
            return 0;
        }
        break;

    case ZMQ_RCVTIMEO:
        if (is_int && value >= -1)
        {
            rcvtimeo = value;
            return 0;
        }
        break;

    case ZMQ_SNDTIMEO:
        if (is_int && value >= -1)
        {
            sndtimeo = value;
            return 0;
        }
        break;

    /*  Deprecated in favor of ZMQ_IPV6  */
    case ZMQ_IPV4ONLY:
    {
        bool value;
        int  rc = do_setsockopt_int_as_bool_strict(optval_, optvallen_, &value);
        if (rc == 0)
            ipv6 = !value;
        return rc;
    }

    /*  To replace the somewhat surprising IPV4ONLY */
    case ZMQ_IPV6:
        return do_setsockopt_int_as_bool_strict(optval_, optvallen_, &ipv6);

    case ZMQ_SOCKS_PROXY:
        return do_setsockopt_string_allow_empty_strict(optval_, optvallen_, &socks_proxy_address, SIZE_MAX);

    case ZMQ_SOCKS_USERNAME:
        /* Make empty string or NULL equivalent. */
        if (optval_ == NULL || optvallen_ == 0)
        {
            socks_proxy_username.clear();
            return 0;
        }
        else
        {
            return do_setsockopt_string_allow_empty_strict(optval_, optvallen_, &socks_proxy_username, 255);
        }
    case ZMQ_SOCKS_PASSWORD:
        /* Make empty string or NULL equivalent. */
        if (optval_ == NULL || optvallen_ == 0)
        {
            socks_proxy_password.clear();
            return 0;
        }
        else
        {
            return do_setsockopt_string_allow_empty_strict(optval_, optvallen_, &socks_proxy_password, 255);
        }
    case ZMQ_TCP_KEEPALIVE:
        if (is_int && (value == -1 || value == 0 || value == 1))
        {
            tcp_keepalive = value;
            return 0;
        }
        break;

    case ZMQ_TCP_KEEPALIVE_CNT:
        if (is_int && (value == -1 || value >= 0))
        {
            tcp_keepalive_cnt = value;
            return 0;
        }
        break;

    case ZMQ_TCP_KEEPALIVE_IDLE:
        if (is_int && (value == -1 || value >= 0))
        {
            tcp_keepalive_idle = value;
            return 0;
        }
        break;

    case ZMQ_TCP_KEEPALIVE_INTVL:
        if (is_int && (value == -1 || value >= 0))
        {
            tcp_keepalive_intvl = value;
            return 0;
        }
        break;

    case ZMQ_IMMEDIATE:
        // TODO why is immediate not bool (and called non_immediate, as its meaning appears to be reversed)
        if (is_int && (value == 0 || value == 1))
        {
            immediate = value;
            return 0;
        }
        break;

    case ZMQ_TCP_ACCEPT_FILTER:
    {
        std::string filter_str;
        int         rc = do_setsockopt_string_allow_empty_strict(optval_, optvallen_, &filter_str, UCHAR_MAX);
        if (rc == 0)
        {
            if (filter_str.empty())
            {
                tcp_accept_filters.clear();
            }
            else
            {
                tcp_address_mask_t mask;
                rc = mask.resolve(filter_str.c_str(), ipv6);
                if (rc == 0)
                {
                    tcp_accept_filters.push_back(mask);
                }
            }
        }
        return rc;
    }

#if defined ZMQ_HAVE_SO_PEERCRED || defined ZMQ_HAVE_LOCAL_PEERCRED
    case ZMQ_IPC_FILTER_UID:
        return do_setsockopt_set(optval_, optvallen_, &ipc_uid_accept_filters);

    case ZMQ_IPC_FILTER_GID:
        return do_setsockopt_set(optval_, optvallen_, &ipc_gid_accept_filters);
#endif

#if defined ZMQ_HAVE_SO_PEERCRED
    case ZMQ_IPC_FILTER_PID:
        return do_setsockopt_set(optval_, optvallen_, &ipc_pid_accept_filters);
#endif

    case ZMQ_PLAIN_SERVER:
        if (is_int && (value == 0 || value == 1))
        {
            as_server = value;
            mechanism = value ? ZMQ_PLAIN : ZMQ_NULL;
            return 0;
        }
        break;

    case ZMQ_PLAIN_USERNAME:
        if (optvallen_ == 0 && optval_ == NULL)
        {
            mechanism = ZMQ_NULL;
            return 0;
        }
        else if (optvallen_ > 0 && optvallen_ <= UCHAR_MAX && optval_ != NULL)
        {
            plain_username.assign(static_cast<const char *>(optval_), optvallen_);
            as_server = 0;
            mechanism = ZMQ_PLAIN;
            return 0;
        }
        break;

    case ZMQ_PLAIN_PASSWORD:
        if (optvallen_ == 0 && optval_ == NULL)
        {
            mechanism = ZMQ_NULL;
            return 0;
        }
        else if (optvallen_ > 0 && optvallen_ <= UCHAR_MAX && optval_ != NULL)
        {
            plain_password.assign(static_cast<const char *>(optval_), optvallen_);
            as_server = 0;
            mechanism = ZMQ_PLAIN;
            return 0;
        }
        break;

    case ZMQ_ZAP_DOMAIN:
        return do_setsockopt_string_allow_empty_relaxed(optval_, optvallen_, &zap_domain, UCHAR_MAX);
        break;

        //  If curve encryption isn't built, these options provoke EINVAL
#ifdef ZMQ_HAVE_CURVE
    case ZMQ_CURVE_SERVER:
        if (is_int && (value == 0 || value == 1))
        {
            as_server = value;
            mechanism = value ? ZMQ_CURVE : ZMQ_NULL;
            return 0;
        }
        break;

    case ZMQ_CURVE_PUBLICKEY:
        if (0 == set_curve_key(curve_public_key, optval_, optvallen_))
        {
            return 0;
        }
        break;

    case ZMQ_CURVE_SECRETKEY:
        if (0 == set_curve_key(curve_secret_key, optval_, optvallen_))
        {
            return 0;
        }
        break;

    case ZMQ_CURVE_SERVERKEY:
        if (0 == set_curve_key(curve_server_key, optval_, optvallen_))
        {
            as_server = 0;
            return 0;
        }
        break;
#endif

    case ZMQ_CONFLATE:
        return do_setsockopt_int_as_bool_strict(optval_, optvallen_, &conflate);

        //  If libgssapi isn't installed, these options provoke EINVAL
#ifdef HAVE_LIBGSSAPI_KRB5
    case ZMQ_GSSAPI_SERVER:
        if (is_int && (value == 0 || value == 1))
        {
            as_server = value;
            mechanism = ZMQ_GSSAPI;
            return 0;
        }
        break;

    case ZMQ_GSSAPI_PRINCIPAL:
        if (optvallen_ > 0 && optvallen_ <= UCHAR_MAX && optval_ != NULL)
        {
            gss_principal.assign((const char *)optval_, optvallen_);
            mechanism = ZMQ_GSSAPI;
            return 0;
        }
        break;

    case ZMQ_GSSAPI_SERVICE_PRINCIPAL:
        if (optvallen_ > 0 && optvallen_ <= UCHAR_MAX && optval_ != NULL)
        {
            gss_service_principal.assign((const char *)optval_, optvallen_);
            mechanism = ZMQ_GSSAPI;
            as_server = 0;
            return 0;
        }
        break;

    case ZMQ_GSSAPI_PLAINTEXT:
        return do_setsockopt_int_as_bool_strict(optval_, optvallen_, &gss_plaintext);

    case ZMQ_GSSAPI_PRINCIPAL_NAMETYPE:
        if (is_int && (value == ZMQ_GSSAPI_NT_HOSTBASED || value == ZMQ_GSSAPI_NT_USER_NAME ||
                       value == ZMQ_GSSAPI_NT_KRB5_PRINCIPAL))
        {
            gss_principal_nt = value;
            return 0;
        }
        break;

    case ZMQ_GSSAPI_SERVICE_PRINCIPAL_NAMETYPE:
        if (is_int && (value == ZMQ_GSSAPI_NT_HOSTBASED || value == ZMQ_GSSAPI_NT_USER_NAME ||
                       value == ZMQ_GSSAPI_NT_KRB5_PRINCIPAL))
        {
            gss_service_principal_nt = value;
            return 0;
        }
        break;
#endif

    case ZMQ_HANDSHAKE_IVL:
        if (is_int && value >= 0)
        {
            handshake_ivl = value;
            return 0;
        }
        break;

    case ZMQ_INVERT_MATCHING:
        return do_setsockopt_int_as_bool_relaxed(optval_, optvallen_, &invert_matching);

    case ZMQ_HEARTBEAT_IVL:
        if (is_int && value >= 0)
        {
            heartbeat_interval = value;
            return 0;
        }
        break;

    case ZMQ_HEARTBEAT_TTL:
        // Convert this to deciseconds from milliseconds
        value = value / deciseconds_per_millisecond;
        if (is_int && value >= 0 && value <= UINT16_MAX)
        {
            heartbeat_ttl = static_cast<uint16_t>(value);
            return 0;
        }
        break;

    case ZMQ_HEARTBEAT_TIMEOUT:
        if (is_int && value >= 0)
        {
            heartbeat_timeout = value;
            return 0;
        }
        break;

#ifdef ZMQ_HAVE_VMCI
    case ZMQ_VMCI_BUFFER_SIZE:
        return do_setsockopt(optval_, optvallen_, &vmci_buffer_size);

    case ZMQ_VMCI_BUFFER_MIN_SIZE:
        return do_setsockopt(optval_, optvallen_, &vmci_buffer_min_size);

    case ZMQ_VMCI_BUFFER_MAX_SIZE:
        return do_setsockopt(optval_, optvallen_, &vmci_buffer_max_size);

    case ZMQ_VMCI_CONNECT_TIMEOUT:
        return do_setsockopt(optval_, optvallen_, &vmci_connect_timeout);
#endif

    case ZMQ_USE_FD:
        if (is_int && value >= -1)
        {
            use_fd = value;
            return 0;
        }
        break;

    case ZMQ_BINDTODEVICE:
        return do_setsockopt_string_allow_empty_strict(optval_, optvallen_, &bound_device, BINDDEVSIZ);

    case ZMQ_ZAP_ENFORCE_DOMAIN:
        return do_setsockopt_int_as_bool_relaxed(optval_, optvallen_, &zap_enforce_domain);

    case ZMQ_LOOPBACK_FASTPATH:
        return do_setsockopt_int_as_bool_relaxed(optval_, optvallen_, &loopback_fastpath);

    case ZMQ_METADATA:
        if (optvallen_ > 0 && !is_int)
        {
            const std::string s(static_cast<const char *>(optval_));
            const size_t      pos = s.find(':');
            if (pos != std::string::npos && pos != 0 && pos != s.length() - 1)
            {
                const std::string key = s.substr(0, pos);
                if (key.compare(0, 2, "X-") == 0 && key.length() <= UCHAR_MAX)
                {
                    std::string val = s.substr(pos + 1, s.length());
                    app_metadata.insert(std::pair<std::string, std::string>(key, val));
                    return 0;
                }
            }
        }
        errno = EINVAL;
        return -1;

    case ZMQ_MULTICAST_LOOP:
        return do_setsockopt_int_as_bool_relaxed(optval_, optvallen_, &multicast_loop);

#ifdef ZMQ_BUILD_DRAFT_API
    case ZMQ_IN_BATCH_SIZE:
        if (is_int && value > 0)
        {
            in_batch_size = value;
            return 0;
        }
        break;

    case ZMQ_OUT_BATCH_SIZE:
        if (is_int && value > 0)
        {
            out_batch_size = value;
            return 0;
        }
        break;
#endif

    default:
#if defined(ZMQ_ACT_MILITANT)
        //  There are valid scenarios for probing with unknown socket option
        //  values, e.g. to check if security is enabled or not. This will not
        //  provoke a militant assert. However, passing bad values to a valid
        //  socket option will, if ZMQ_ACT_MILITANT is defined.
        malformed = false;
#endif
        break;
    }

    // TODO mechanism should either be set explicitly, or determined when
    // connecting. currently, it depends on the order of setsockopt calls
    // if there is some inconsistency, which is confusing. in addition,
    // the assumed or set mechanism should be queryable (as a socket option)

#if defined(ZMQ_ACT_MILITANT)
    //  There is no valid use case for passing an error back to the application
    //  when it sent malformed arguments to a socket option. Use ./configure
    //  --with-militant to enable this checking.
    if (malformed)
        zmq_assert(false);
#endif
    errno = EINVAL;
    return -1;
}

int zmq::options_t::getsockopt(int option_, void *optval_, size_t *optvallen_) const
{
    const bool is_int = (*optvallen_ == sizeof(int));
    int *      value  = static_cast<int *>(optval_);
#if defined(ZMQ_ACT_MILITANT)
    bool malformed = true; //  Did caller pass a bad option value?
#endif

    switch (option_)
    {
    case ZMQ_SNDHWM:
        if (is_int)
        {
            *value = sndhwm;
            return 0;
        }
        break;

    case ZMQ_RCVHWM:
        if (is_int)
        {
            *value = rcvhwm;
            return 0;
        }
        break;

    case ZMQ_AFFINITY:
        if (*optvallen_ == sizeof(uint64_t))
        {
            *(static_cast<uint64_t *>(optval_)) = affinity;
            return 0;
        }
        break;

    case ZMQ_ROUTING_ID:
        return do_getsockopt(optval_, optvallen_, routing_id, routing_id_size);
        break;

    case ZMQ_RATE:
        if (is_int)
        {
            *value = rate;
            return 0;
        }
        break;

    case ZMQ_RECOVERY_IVL:
        if (is_int)
        {
            *value = recovery_ivl;
            return 0;
        }
        break;

    case ZMQ_SNDBUF:
        if (is_int)
        {
            *value = sndbuf;
            return 0;
        }
        break;

    case ZMQ_RCVBUF:
        if (is_int)
        {
            *value = rcvbuf;
            return 0;
        }
        break;

    case ZMQ_TOS:
        if (is_int)
        {
            *value = tos;
            return 0;
        }
        break;

    case ZMQ_TYPE:
        if (is_int)
        {
            *value = type;
            return 0;
        }
        break;

    case ZMQ_LINGER:
        if (is_int)
        {
            *value = linger.load();
            return 0;
        }
        break;

    case ZMQ_CONNECT_TIMEOUT:
        if (is_int)
        {
            *value = connect_timeout;
            return 0;
        }
        break;

    case ZMQ_TCP_MAXRT:
        if (is_int)
        {
            *value = tcp_maxrt;
            return 0;
        }
        break;

    case ZMQ_RECONNECT_IVL:
        if (is_int)
        {
            *value = reconnect_ivl;
            return 0;
        }
        break;

    case ZMQ_RECONNECT_IVL_MAX:
        if (is_int)
        {
            *value = reconnect_ivl_max;
            return 0;
        }
        break;

    case ZMQ_BACKLOG:
        if (is_int)
        {
            *value = backlog;
            return 0;
        }
        break;

    case ZMQ_MAXMSGSIZE:
        if (*optvallen_ == sizeof(int64_t))
        {
            *(static_cast<int64_t *>(optval_)) = maxmsgsize;
            *optvallen_                        = sizeof(int64_t);
            return 0;
        }
        break;

    case ZMQ_MULTICAST_HOPS:
        if (is_int)
        {
            *value = multicast_hops;
            return 0;
        }
        break;

    case ZMQ_MULTICAST_MAXTPDU:
        if (is_int)
        {
            *value = multicast_maxtpdu;
            return 0;
        }
        break;

    case ZMQ_RCVTIMEO:
        if (is_int)
        {
            *value = rcvtimeo;
            return 0;
        }
        break;

    case ZMQ_SNDTIMEO:
        if (is_int)
        {
            *value = sndtimeo;
            return 0;
        }
        break;

    case ZMQ_IPV4ONLY:
        if (is_int)
        {
            *value = 1 - ipv6;
            return 0;
        }
        break;

    case ZMQ_IPV6:
        if (is_int)
        {
            *value = ipv6;
            return 0;
        }
        break;

    case ZMQ_IMMEDIATE:
        if (is_int)
        {
            *value = immediate;
            return 0;
        }
        break;

    case ZMQ_SOCKS_PROXY:
        return do_getsockopt(optval_, optvallen_, socks_proxy_address);
        break;

    case ZMQ_SOCKS_USERNAME:
        return do_getsockopt(optval_, optvallen_, socks_proxy_username);
        break;

    case ZMQ_SOCKS_PASSWORD:
        return do_getsockopt(optval_, optvallen_, socks_proxy_password);
        break;

    case ZMQ_TCP_KEEPALIVE:
        if (is_int)
        {
            *value = tcp_keepalive;
            return 0;
        }
        break;

    case ZMQ_TCP_KEEPALIVE_CNT:
        if (is_int)
        {
            *value = tcp_keepalive_cnt;
            return 0;
        }
        break;

    case ZMQ_TCP_KEEPALIVE_IDLE:
        if (is_int)
        {
            *value = tcp_keepalive_idle;
            return 0;
        }
        break;

    case ZMQ_TCP_KEEPALIVE_INTVL:
        if (is_int)
        {
            *value = tcp_keepalive_intvl;
            return 0;
        }
        break;

    case ZMQ_MECHANISM:
        if (is_int)
        {
            *value = mechanism;
            return 0;
        }
        break;

    case ZMQ_PLAIN_SERVER:
        if (is_int)
        {
            *value = as_server && mechanism == ZMQ_PLAIN;
            return 0;
        }
        break;

    case ZMQ_PLAIN_USERNAME:
        return do_getsockopt(optval_, optvallen_, plain_username);
        break;

    case ZMQ_PLAIN_PASSWORD:
        return do_getsockopt(optval_, optvallen_, plain_password);
        break;

    case ZMQ_ZAP_DOMAIN:
        return do_getsockopt(optval_, optvallen_, zap_domain);
        break;

        //  If curve encryption isn't built, these options provoke EINVAL
#ifdef ZMQ_HAVE_CURVE
    case ZMQ_CURVE_SERVER:
        if (is_int)
        {
            *value = as_server && mechanism == ZMQ_CURVE;
            return 0;
        }
        break;

    case ZMQ_CURVE_PUBLICKEY:
        return do_getsockopt_curve_key(optval_, optvallen_, curve_public_key);
        break;

    case ZMQ_CURVE_SECRETKEY:
        return do_getsockopt_curve_key(optval_, optvallen_, curve_secret_key);
        break;

    case ZMQ_CURVE_SERVERKEY:
        return do_getsockopt_curve_key(optval_, optvallen_, curve_server_key);
        break;
#endif

    case ZMQ_CONFLATE:
        if (is_int)
        {
            *value = conflate;
            return 0;
        }
        break;

        //  If libgssapi isn't installed, these options provoke EINVAL
#ifdef HAVE_LIBGSSAPI_KRB5
    case ZMQ_GSSAPI_SERVER:
        if (is_int)
        {
            *value = as_server && mechanism == ZMQ_GSSAPI;
            return 0;
        }
        break;

    case ZMQ_GSSAPI_PRINCIPAL:
        return do_getsockopt(optval_, optvallen_, gss_principal);
        break;

    case ZMQ_GSSAPI_SERVICE_PRINCIPAL:
        return do_getsockopt(optval_, optvallen_, gss_service_principal);
        break;

    case ZMQ_GSSAPI_PLAINTEXT:
        if (is_int)
        {
            *value = gss_plaintext;
            return 0;
        }
        break;

    case ZMQ_GSSAPI_PRINCIPAL_NAMETYPE:
        if (is_int)
        {
            *value = gss_principal_nt;
            return 0;
        }
        break;
    case ZMQ_GSSAPI_SERVICE_PRINCIPAL_NAMETYPE:
        if (is_int)
        {
            *value = gss_service_principal_nt;
            return 0;
        }
        break;
#endif

    case ZMQ_HANDSHAKE_IVL:
        if (is_int)
        {
            *value = handshake_ivl;
            return 0;
        }
        break;

    case ZMQ_INVERT_MATCHING:
        if (is_int)
        {
            *value = invert_matching;
            return 0;
        }
        break;

    case ZMQ_HEARTBEAT_IVL:
        if (is_int)
        {
            *value = heartbeat_interval;
            return 0;
        }
        break;

    case ZMQ_HEARTBEAT_TTL:
        if (is_int)
        {
            // Convert the internal deciseconds value to milliseconds
            *value = heartbeat_ttl * 100;
            return 0;
        }
        break;

    case ZMQ_HEARTBEAT_TIMEOUT:
        if (is_int)
        {
            *value = heartbeat_timeout;
            return 0;
        }
        break;

    case ZMQ_USE_FD:
        if (is_int)
        {
            *value = use_fd;
            return 0;
        }
        break;

    case ZMQ_BINDTODEVICE:
        return do_getsockopt(optval_, optvallen_, bound_device);
        break;

    case ZMQ_ZAP_ENFORCE_DOMAIN:
        if (is_int)
        {
            *value = zap_enforce_domain;
            return 0;
        }
        break;

    case ZMQ_LOOPBACK_FASTPATH:
        if (is_int)
        {
            *value = loopback_fastpath;
            return 0;
        }
        break;

    case ZMQ_MULTICAST_LOOP:
        if (is_int)
        {
            *value = multicast_loop;
            return 0;
        }
        break;

#ifdef ZMQ_BUILD_DRAFT_API
    case ZMQ_ROUTER_NOTIFY:
        if (is_int)
        {
            *value = router_notify;
            return 0;
        }
        break;
    case ZMQ_IN_BATCH_SIZE:
        if (is_int)
        {
            *value = in_batch_size;
            return 0;
        }
        break;

    case ZMQ_OUT_BATCH_SIZE:
        if (is_int)
        {
            *value = out_batch_size;
            return 0;
        }
        break;
#endif

    default:
#if defined(ZMQ_ACT_MILITANT)
        malformed = false;
#endif
        break;
    }
#if defined(ZMQ_ACT_MILITANT)
    if (malformed)
        zmq_assert(false);
#endif
    errno = EINVAL;
    return -1;
}

//========= end of options.cpp ============

//========= begin of own.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "own.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "io_thread.hpp"

zmq::own_t::own_t(class ctx_t *parent_, uint32_t tid_)
    : object_t(parent_, tid_), _terminating(false), _sent_seqnum(0), _processed_seqnum(0), _owner(NULL), _term_acks(0)
{
}

zmq::own_t::own_t(io_thread_t *io_thread_, const options_t &options_)
    : object_t(io_thread_), options(options_), _terminating(false), _sent_seqnum(0), _processed_seqnum(0), _owner(NULL),
      _term_acks(0)
{
}

zmq::own_t::~own_t() {}

void zmq::own_t::set_owner(own_t *owner_)
{
    zmq_assert(!_owner);
    _owner = owner_;
}

void zmq::own_t::inc_seqnum()
{
    //  This function may be called from a different thread!
    _sent_seqnum.add(1);
}

void zmq::own_t::process_seqnum()
{
    //  Catch up with counter of processed commands.
    _processed_seqnum++;

    //  We may have catched up and still have pending terms acks.
    check_term_acks();
}

void zmq::own_t::launch_child(own_t *object_)
{
    //  Specify the owner of the object.
    object_->set_owner(this);

    //  Plug the object into the I/O thread.
    send_plug(object_);

    //  Take ownership of the object.
    send_own(this, object_);
}

void zmq::own_t::term_child(own_t *object_) { process_term_req(object_); }

void zmq::own_t::process_term_req(own_t *object_)
{
    //  When shutting down we can ignore termination requests from owned
    //  objects. The termination request was already sent to the object.
    if (_terminating)
        return;

    //  If not found, we assume that termination request was already sent to
    //  the object so we can safely ignore the request.
    if (0 == _owned.erase(object_))
        return;

    //  If I/O object is well and alive let's ask it to terminate.
    register_term_acks(1);

    //  Note that this object is the root of the (partial shutdown) thus, its
    //  value of linger is used, rather than the value stored by the children.
    send_term(object_, options.linger.load());
}

void zmq::own_t::process_own(own_t *object_)
{
    //  If the object is already being shut down, new owned objects are
    //  immediately asked to terminate. Note that linger is set to zero.
    if (_terminating)
    {
        register_term_acks(1);
        send_term(object_, 0);
        return;
    }

    //  Store the reference to the owned object.
    _owned.insert(object_);
}

void zmq::own_t::terminate()
{
    //  If termination is already underway, there's no point
    //  in starting it anew.
    if (_terminating)
        return;

    //  As for the root of the ownership tree, there's no one to terminate it,
    //  so it has to terminate itself.
    if (!_owner)
    {
        process_term(options.linger.load());
        return;
    }

    //  If I am an owned object, I'll ask my owner to terminate me.
    send_term_req(_owner, this);
}

bool zmq::own_t::is_terminating() { return _terminating; }

void zmq::own_t::process_term(int linger_)
{
    //  Double termination should never happen.
    zmq_assert(!_terminating);

    //  Send termination request to all owned objects.
    for (owned_t::iterator it = _owned.begin(), end = _owned.end(); it != end; ++it)
        send_term(*it, linger_);
    register_term_acks(static_cast<int>(_owned.size()));
    _owned.clear();

    //  Start termination process and check whether by chance we cannot
    //  terminate immediately.
    _terminating = true;
    check_term_acks();
}

void zmq::own_t::register_term_acks(int count_) { _term_acks += count_; }

void zmq::own_t::unregister_term_ack()
{
    zmq_assert(_term_acks > 0);
    _term_acks--;

    //  This may be a last ack we are waiting for before termination...
    check_term_acks();
}

void zmq::own_t::process_term_ack() { unregister_term_ack(); }

void zmq::own_t::check_term_acks()
{
    if (_terminating && _processed_seqnum == _sent_seqnum.get() && _term_acks == 0)
    {
        //  Sanity check. There should be no active children at this point.
        zmq_assert(_owned.empty());

        //  The root object has nobody to confirm the termination to.
        //  Other nodes will confirm the termination to the owner.
        if (_owner)
            send_term_ack(_owner);

        //  Deallocate the resources.
        process_destroy();
    }
}

void zmq::own_t::process_destroy() { delete this; }

//========= end of own.cpp ============

//========= begin of pair.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "pair.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "pipe.hpp"
// ans ignore: #include "msg.hpp"

zmq::pair_t::pair_t(class ctx_t *parent_, uint32_t tid_, int sid_)
    : socket_base_t(parent_, tid_, sid_), _pipe(NULL), _last_in(NULL)
{
    options.type = ZMQ_PAIR;
}

zmq::pair_t::~pair_t() { zmq_assert(!_pipe); }

void zmq::pair_t::xattach_pipe(pipe_t *pipe_, bool subscribe_to_all_, bool locally_initiated_)
{
    LIBZMQ_UNUSED(subscribe_to_all_);
    LIBZMQ_UNUSED(locally_initiated_);

    zmq_assert(pipe_ != NULL);

    //  ZMQ_PAIR socket can only be connected to a single peer.
    //  The socket rejects any further connection requests.
    if (_pipe == NULL)
        _pipe = pipe_;
    else
        pipe_->terminate(false);
}

void zmq::pair_t::xpipe_terminated(pipe_t *pipe_)
{
    if (pipe_ == _pipe)
    {
        if (_last_in == _pipe)
        {
            _last_in = NULL;
        }
        _pipe = NULL;
    }
}

void zmq::pair_t::xread_activated(pipe_t *)
{
    //  There's just one pipe. No lists of active and inactive pipes.
    //  There's nothing to do here.
}

void zmq::pair_t::xwrite_activated(pipe_t *)
{
    //  There's just one pipe. No lists of active and inactive pipes.
    //  There's nothing to do here.
}

int zmq::pair_t::xsend(msg_t *msg_)
{
    if (!_pipe || !_pipe->write(msg_))
    {
        errno = EAGAIN;
        return -1;
    }

    if (!(msg_->flags() & msg_t::more))
        _pipe->flush();

    //  Detach the original message from the data buffer.
    int rc = msg_->init();
    errno_assert(rc == 0);

    return 0;
}

int zmq::pair_t::xrecv(msg_t *msg_)
{
    //  Deallocate old content of the message.
    int rc = msg_->close();
    errno_assert(rc == 0);

    if (!_pipe || !_pipe->read(msg_))
    {
        //  Initialise the output parameter to be a 0-byte message.
        rc = msg_->init();
        errno_assert(rc == 0);

        errno = EAGAIN;
        return -1;
    }
    _last_in = _pipe;
    return 0;
}

bool zmq::pair_t::xhas_in()
{
    if (!_pipe)
        return false;

    return _pipe->check_read();
}

bool zmq::pair_t::xhas_out()
{
    if (!_pipe)
        return false;

    return _pipe->check_write();
}

//========= end of pair.cpp ============

//========= begin of pgm_receiver.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"

#if defined ZMQ_HAVE_OPENPGM

#include <new>

// ans ignore: #include "pgm_receiver.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "v1_decoder.hpp"
// ans ignore: #include "stdint.hpp"
// ans ignore: #include "wire.hpp"
// ans ignore: #include "err.hpp"

zmq::pgm_receiver_t::pgm_receiver_t(class io_thread_t *parent_, const options_t &options_)
    : io_object_t(parent_), has_rx_timer(false), pgm_socket(true, options_), options(options_), session(NULL),
      active_tsi(NULL), insize(0)
{
}

zmq::pgm_receiver_t::~pgm_receiver_t()
{
    //  Destructor should not be called before unplug.
    zmq_assert(peers.empty());
}

int zmq::pgm_receiver_t::init(bool udp_encapsulation_, const char *network_)
{
    return pgm_socket.init(udp_encapsulation_, network_);
}

void zmq::pgm_receiver_t::plug(io_thread_t *io_thread_, session_base_t *session_)
{
    LIBZMQ_UNUSED(io_thread_);
    //  Retrieve PGM fds and start polling.
    fd_t socket_fd       = retired_fd;
    fd_t waiting_pipe_fd = retired_fd;
    pgm_socket.get_receiver_fds(&socket_fd, &waiting_pipe_fd);
    socket_handle = add_fd(socket_fd);
    pipe_handle   = add_fd(waiting_pipe_fd);
    set_pollin(pipe_handle);
    set_pollin(socket_handle);

    session = session_;

    //  If there are any subscriptions already queued in the session, drop them.
    drop_subscriptions();
}

void zmq::pgm_receiver_t::unplug()
{
    //  Delete decoders.
    for (peers_t::iterator it = peers.begin(), end = peers.end(); it != end; ++it)
    {
        if (it->second.decoder != NULL)
        {
            LIBZMQ_DELETE(it->second.decoder);
        }
    }
    peers.clear();
    active_tsi = NULL;

    if (has_rx_timer)
    {
        cancel_timer(rx_timer_id);
        has_rx_timer = false;
    }

    rm_fd(socket_handle);
    rm_fd(pipe_handle);

    session = NULL;
}

void zmq::pgm_receiver_t::terminate()
{
    unplug();
    delete this;
}

void zmq::pgm_receiver_t::restart_output() { drop_subscriptions(); }

bool zmq::pgm_receiver_t::restart_input()
{
    zmq_assert(session != NULL);
    zmq_assert(active_tsi != NULL);

    const peers_t::iterator it = peers.find(*active_tsi);
    zmq_assert(it != peers.end());
    zmq_assert(it->second.joined);

    //  Push the pending message into the session.
    int rc = session->push_msg(it->second.decoder->msg());
    errno_assert(rc == 0);

    if (insize > 0)
    {
        rc = process_input(it->second.decoder);
        if (rc == -1)
        {
            //  HWM reached; we will try later.
            if (errno == EAGAIN)
            {
                session->flush();
                return true;
            }
            //  Data error. Delete message decoder, mark the
            //  peer as not joined and drop remaining data.
            it->second.joined = false;
            LIBZMQ_DELETE(it->second.decoder);
            insize = 0;
        }
    }

    //  Resume polling.
    set_pollin(pipe_handle);
    set_pollin(socket_handle);

    active_tsi = NULL;
    in_event();

    return true;
}

const zmq::endpoint_uri_pair_t &zmq::pgm_receiver_t::get_endpoint() const { return _empty_endpoint; }

void zmq::pgm_receiver_t::in_event()
{
    // If active_tsi is not null, there is a pending restart_input.
    // Keep the internal state as is so that restart_input would process the right data
    if (active_tsi)
    {
        return;
    }

    // Read data from the underlying pgm_socket.
    const pgm_tsi_t *tsi = NULL;

    if (has_rx_timer)
    {
        cancel_timer(rx_timer_id);
        has_rx_timer = false;
    }

    //  TODO: This loop can effectively block other engines in the same I/O
    //  thread in the case of high load.
    while (true)
    {
        //  Get new batch of data.
        //  Note the workaround made not to break strict-aliasing rules.
        insize           = 0;
        void *  tmp      = NULL;
        ssize_t received = pgm_socket.receive(&tmp, &tsi);

        //  No data to process. This may happen if the packet received is
        //  neither ODATA nor ODATA.
        if (received == 0)
        {
            if (errno == ENOMEM || errno == EBUSY)
            {
                const long timeout = pgm_socket.get_rx_timeout();
                add_timer(timeout, rx_timer_id);
                has_rx_timer = true;
            }
            break;
        }

        //  Find the peer based on its TSI.
        peers_t::iterator it = peers.find(*tsi);

        //  Data loss. Delete decoder and mark the peer as disjoint.
        if (received == -1)
        {
            if (it != peers.end())
            {
                it->second.joined = false;
                if (it->second.decoder != NULL)
                {
                    LIBZMQ_DELETE(it->second.decoder);
                }
            }
            break;
        }

        //  New peer. Add it to the list of know but unjoint peers.
        if (it == peers.end())
        {
            peer_info_t peer_info = {false, NULL};
            it                    = peers.ZMQ_MAP_INSERT_OR_EMPLACE(*tsi, peer_info).first;
        }

        insize = static_cast<size_t>(received);
        inpos  = (unsigned char *)tmp;

        //  Read the offset of the fist message in the current packet.
        zmq_assert(insize >= sizeof(uint16_t));
        uint16_t offset = get_uint16(inpos);
        inpos += sizeof(uint16_t);
        insize -= sizeof(uint16_t);

        //  Join the stream if needed.
        if (!it->second.joined)
        {
            //  There is no beginning of the message in current packet.
            //  Ignore the data.
            if (offset == 0xffff)
                continue;

            zmq_assert(offset <= insize);
            zmq_assert(it->second.decoder == NULL);

            //  We have to move data to the beginning of the first message.
            inpos += offset;
            insize -= offset;

            //  Mark the stream as joined.
            it->second.joined = true;

            //  Create and connect decoder for the peer.
            it->second.decoder = new (std::nothrow) v1_decoder_t(0, options.maxmsgsize);
            alloc_assert(it->second.decoder);
        }

        int rc = process_input(it->second.decoder);
        if (rc == -1)
        {
            if (errno == EAGAIN)
            {
                active_tsi = tsi;

                //  Stop polling.
                reset_pollin(pipe_handle);
                reset_pollin(socket_handle);

                break;
            }

            it->second.joined = false;
            LIBZMQ_DELETE(it->second.decoder);
            insize = 0;
        }
    }

    //  Flush any messages decoder may have produced.
    session->flush();
}

int zmq::pgm_receiver_t::process_input(v1_decoder_t *decoder)
{
    zmq_assert(session != NULL);

    while (insize > 0)
    {
        size_t n  = 0;
        int    rc = decoder->decode(inpos, insize, n);
        if (rc == -1)
            return -1;
        inpos += n;
        insize -= n;
        if (rc == 0)
            break;
        rc = session->push_msg(decoder->msg());
        if (rc == -1)
        {
            errno_assert(errno == EAGAIN);
            return -1;
        }
    }
    return 0;
}

void zmq::pgm_receiver_t::timer_event(int token)
{
    zmq_assert(token == rx_timer_id);

    //  Timer cancels on return by poller_base.
    has_rx_timer = false;
    in_event();
}

void zmq::pgm_receiver_t::drop_subscriptions()
{
    msg_t msg;
    msg.init();
    while (session->pull_msg(&msg) == 0)
        msg.close();
}

#endif

//========= end of pgm_receiver.cpp ============

//========= begin of pgm_sender.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"

#if defined ZMQ_HAVE_OPENPGM

#include <stdlib.h>

// ans ignore: #include "io_thread.hpp"
// ans ignore: #include "pgm_sender.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "wire.hpp"
// ans ignore: #include "stdint.hpp"
// ans ignore: #include "macros.hpp"

zmq::pgm_sender_t::pgm_sender_t(io_thread_t *parent_, const options_t &options_)
    : io_object_t(parent_), has_tx_timer(false), has_rx_timer(false), session(NULL), encoder(0), more_flag(false),
      pgm_socket(false, options_), options(options_), handle(static_cast<handle_t>(NULL)),
      uplink_handle(static_cast<handle_t>(NULL)), rdata_notify_handle(static_cast<handle_t>(NULL)),
      pending_notify_handle(static_cast<handle_t>(NULL)), out_buffer(NULL), out_buffer_size(0), write_size(0)
{
    int rc = msg.init();
    errno_assert(rc == 0);
}

int zmq::pgm_sender_t::init(bool udp_encapsulation_, const char *network_)
{
    int rc = pgm_socket.init(udp_encapsulation_, network_);
    if (rc != 0)
        return rc;

    out_buffer_size = pgm_socket.get_max_tsdu_size();
    out_buffer      = (unsigned char *)malloc(out_buffer_size);
    alloc_assert(out_buffer);

    return rc;
}

void zmq::pgm_sender_t::plug(io_thread_t *io_thread_, session_base_t *session_)
{
    LIBZMQ_UNUSED(io_thread_);
    //  Allocate 2 fds for PGM socket.
    fd_t downlink_socket_fd = retired_fd;
    fd_t uplink_socket_fd   = retired_fd;
    fd_t rdata_notify_fd    = retired_fd;
    fd_t pending_notify_fd  = retired_fd;

    session = session_;

    //  Fill fds from PGM transport and add them to the poller.
    pgm_socket.get_sender_fds(&downlink_socket_fd, &uplink_socket_fd, &rdata_notify_fd, &pending_notify_fd);

    handle                = add_fd(downlink_socket_fd);
    uplink_handle         = add_fd(uplink_socket_fd);
    rdata_notify_handle   = add_fd(rdata_notify_fd);
    pending_notify_handle = add_fd(pending_notify_fd);

    //  Set POLLIN. We wont never want to stop polling for uplink = we never
    //  want to stop processing NAKs.
    set_pollin(uplink_handle);
    set_pollin(rdata_notify_handle);
    set_pollin(pending_notify_handle);

    //  Set POLLOUT for downlink_socket_handle.
    set_pollout(handle);
}

void zmq::pgm_sender_t::unplug()
{
    if (has_rx_timer)
    {
        cancel_timer(rx_timer_id);
        has_rx_timer = false;
    }

    if (has_tx_timer)
    {
        cancel_timer(tx_timer_id);
        has_tx_timer = false;
    }

    rm_fd(handle);
    rm_fd(uplink_handle);
    rm_fd(rdata_notify_handle);
    rm_fd(pending_notify_handle);
    session = NULL;
}

void zmq::pgm_sender_t::terminate()
{
    unplug();
    delete this;
}

void zmq::pgm_sender_t::restart_output()
{
    set_pollout(handle);
    out_event();
}

bool zmq::pgm_sender_t::restart_input()
{
    zmq_assert(false);
    return true;
}

const zmq::endpoint_uri_pair_t &zmq::pgm_sender_t::get_endpoint() const { return _empty_endpoint; }

zmq::pgm_sender_t::~pgm_sender_t()
{
    int rc = msg.close();
    errno_assert(rc == 0);

    if (out_buffer)
    {
        free(out_buffer);
        out_buffer = NULL;
    }
}

void zmq::pgm_sender_t::in_event()
{
    if (has_rx_timer)
    {
        cancel_timer(rx_timer_id);
        has_rx_timer = false;
    }

    //  In-event on sender side means NAK or SPMR receiving from some peer.
    pgm_socket.process_upstream();
    if (errno == ENOMEM || errno == EBUSY)
    {
        const long timeout = pgm_socket.get_rx_timeout();
        add_timer(timeout, rx_timer_id);
        has_rx_timer = true;
    }
}

void zmq::pgm_sender_t::out_event()
{
    //  POLLOUT event from send socket. If write buffer is empty,
    //  try to read new data from the encoder.
    if (write_size == 0)
    {
        //  First two bytes (sizeof uint16_t) are used to store message
        //  offset in following steps. Note that by passing our buffer to
        //  the get data function we prevent it from returning its own buffer.
        unsigned char *bf     = out_buffer + sizeof(uint16_t);
        size_t         bfsz   = out_buffer_size - sizeof(uint16_t);
        uint16_t       offset = 0xffff;

        size_t bytes = encoder.encode(&bf, bfsz);
        while (bytes < bfsz)
        {
            if (!more_flag && offset == 0xffff)
                offset = static_cast<uint16_t>(bytes);
            int rc = session->pull_msg(&msg);
            if (rc == -1)
                break;
            more_flag = msg.flags() & msg_t::more;
            encoder.load_msg(&msg);
            bf = out_buffer + sizeof(uint16_t) + bytes;
            bytes += encoder.encode(&bf, bfsz - bytes);
        }

        //  If there are no data to write stop polling for output.
        if (bytes == 0)
        {
            reset_pollout(handle);
            return;
        }

        write_size = sizeof(uint16_t) + bytes;

        //  Put offset information in the buffer.
        put_uint16(out_buffer, offset);
    }

    if (has_tx_timer)
    {
        cancel_timer(tx_timer_id);
        set_pollout(handle);
        has_tx_timer = false;
    }

    //  Send the data.
    size_t nbytes = pgm_socket.send(out_buffer, write_size);

    //  We can write either all data or 0 which means rate limit reached.
    if (nbytes == write_size)
        write_size = 0;
    else
    {
        zmq_assert(nbytes == 0);

        if (errno == ENOMEM)
        {
            // Stop polling handle and wait for tx timeout
            const long timeout = pgm_socket.get_tx_timeout();
            add_timer(timeout, tx_timer_id);
            reset_pollout(handle);
            has_tx_timer = true;
        }
        else
            errno_assert(errno == EBUSY);
    }
}

void zmq::pgm_sender_t::timer_event(int token)
{
    //  Timer cancels on return by poller_base.
    if (token == rx_timer_id)
    {
        has_rx_timer = false;
        in_event();
    }
    else if (token == tx_timer_id)
    {
        // Restart polling handle and retry sending
        has_tx_timer = false;
        set_pollout(handle);
        out_event();
    }
    else
        zmq_assert(false);
}

#endif

//========= end of pgm_sender.cpp ============

//========= begin of pgm_socket.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"

#ifdef ZMQ_HAVE_OPENPGM

#ifdef ZMQ_HAVE_LINUX
#include <poll.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <string>

// ans ignore: #include "options.hpp"
// ans ignore: #include "pgm_socket.hpp"
// ans ignore: #include "config.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "random.hpp"
// ans ignore: #include "stdint.hpp"

#ifndef MSG_ERRQUEUE
#define MSG_ERRQUEUE 0x2000
#endif

zmq::pgm_socket_t::pgm_socket_t(bool receiver_, const options_t &options_)
    : sock(NULL), options(options_), receiver(receiver_), pgm_msgv(NULL), pgm_msgv_len(0), nbytes_rec(0),
      nbytes_processed(0), pgm_msgv_processed(0)
{
}

//  Resolve PGM socket address.
//  network_ of the form <interface & multicast group decls>:<IP port>
//  e.g. eth0;239.192.0.1:7500
//       link-local;224.250.0.1,224.250.0.2;224.250.0.3:8000
//       ;[fe80::1%en0]:7500
int zmq::pgm_socket_t::init_address(const char *network_, struct pgm_addrinfo_t **res, uint16_t *port_number)
{
    //  Parse port number, start from end for IPv6
    const char *port_delim = strrchr(network_, ':');
    if (!port_delim)
    {
        errno = EINVAL;
        return -1;
    }

    *port_number = atoi(port_delim + 1);

    char network[256];
    if (port_delim - network_ >= (int)sizeof(network) - 1)
    {
        errno = EINVAL;
        return -1;
    }
    memset(network, '\0', sizeof(network));
    memcpy(network, network_, port_delim - network_);

    pgm_error_t *         pgm_error = NULL;
    struct pgm_addrinfo_t hints;

    memset(&hints, 0, sizeof(hints));
    hints.ai_family = AF_UNSPEC;
    if (!pgm_getaddrinfo(network, NULL, res, &pgm_error))
    {
        //  Invalid parameters don't set pgm_error_t.
        zmq_assert(pgm_error != NULL);
        if (pgm_error->domain == PGM_ERROR_DOMAIN_IF &&

            //  NB: cannot catch EAI_BADFLAGS.
            (pgm_error->code != PGM_ERROR_SERVICE && pgm_error->code != PGM_ERROR_SOCKTNOSUPPORT))
        {
            //  User, host, or network configuration or transient error.
            pgm_error_free(pgm_error);
            errno = EINVAL;
            return -1;
        }

        //  Fatal OpenPGM internal error.
        zmq_assert(false);
    }
    return 0;
}

//  Create, bind and connect PGM socket.
int zmq::pgm_socket_t::init(bool udp_encapsulation_, const char *network_)
{
    //  Can not open transport before destroying old one.
    zmq_assert(sock == NULL);
    zmq_assert(options.rate > 0);

    //  Zero counter used in msgrecv.
    nbytes_rec         = 0;
    nbytes_processed   = 0;
    pgm_msgv_processed = 0;

    uint16_t               port_number;
    struct pgm_addrinfo_t *res = NULL;
    sa_family_t            sa_family;

    pgm_error_t *pgm_error = NULL;

    if (init_address(network_, &res, &port_number) < 0)
    {
        goto err_abort;
    }

    zmq_assert(res != NULL);

    //  Pick up detected IP family.
    sa_family = res->ai_send_addrs[0].gsr_group.ss_family;

    //  Create IP/PGM or UDP/PGM socket.
    if (udp_encapsulation_)
    {
        if (!pgm_socket(&sock, sa_family, SOCK_SEQPACKET, IPPROTO_UDP, &pgm_error))
        {
            //  Invalid parameters don't set pgm_error_t.
            zmq_assert(pgm_error != NULL);
            if (pgm_error->domain == PGM_ERROR_DOMAIN_SOCKET &&
                (pgm_error->code != PGM_ERROR_BADF && pgm_error->code != PGM_ERROR_FAULT &&
                 pgm_error->code != PGM_ERROR_NOPROTOOPT && pgm_error->code != PGM_ERROR_FAILED))

                //  User, host, or network configuration or transient error.
                goto err_abort;

            //  Fatal OpenPGM internal error.
            zmq_assert(false);
        }

        //  All options are of data type int
        const int encapsulation_port = port_number;
        if (!pgm_setsockopt(sock, IPPROTO_PGM, PGM_UDP_ENCAP_UCAST_PORT, &encapsulation_port,
                            sizeof(encapsulation_port)))
            goto err_abort;
        if (!pgm_setsockopt(sock, IPPROTO_PGM, PGM_UDP_ENCAP_MCAST_PORT, &encapsulation_port,
                            sizeof(encapsulation_port)))
            goto err_abort;
    }
    else
    {
        if (!pgm_socket(&sock, sa_family, SOCK_SEQPACKET, IPPROTO_PGM, &pgm_error))
        {
            //  Invalid parameters don't set pgm_error_t.
            zmq_assert(pgm_error != NULL);
            if (pgm_error->domain == PGM_ERROR_DOMAIN_SOCKET &&
                (pgm_error->code != PGM_ERROR_BADF && pgm_error->code != PGM_ERROR_FAULT &&
                 pgm_error->code != PGM_ERROR_NOPROTOOPT && pgm_error->code != PGM_ERROR_FAILED))

                //  User, host, or network configuration or transient error.
                goto err_abort;

            //  Fatal OpenPGM internal error.
            zmq_assert(false);
        }
    }

    {
        const int rcvbuf = (int)options.rcvbuf;
        if (rcvbuf >= 0)
        {
            if (!pgm_setsockopt(sock, SOL_SOCKET, SO_RCVBUF, &rcvbuf, sizeof(rcvbuf)))
                goto err_abort;
        }

        const int sndbuf = (int)options.sndbuf;
        if (sndbuf >= 0)
        {
            if (!pgm_setsockopt(sock, SOL_SOCKET, SO_SNDBUF, &sndbuf, sizeof(sndbuf)))
                goto err_abort;
        }

        const int max_tpdu = (int)options.multicast_maxtpdu;
        if (!pgm_setsockopt(sock, IPPROTO_PGM, PGM_MTU, &max_tpdu, sizeof(max_tpdu)))
            goto err_abort;
    }

    if (receiver)
    {
        const int recv_only = 1, rxw_max_tpdu = (int)options.multicast_maxtpdu, rxw_sqns = compute_sqns(rxw_max_tpdu),
                  peer_expiry = pgm_secs(300), spmr_expiry = pgm_msecs(25), nak_bo_ivl = pgm_msecs(50),
                  nak_rpt_ivl = pgm_msecs(200), nak_rdata_ivl = pgm_msecs(200), nak_data_retries = 50,
                  nak_ncf_retries = 50;

        if (!pgm_setsockopt(sock, IPPROTO_PGM, PGM_RECV_ONLY, &recv_only, sizeof(recv_only)) ||
            !pgm_setsockopt(sock, IPPROTO_PGM, PGM_RXW_SQNS, &rxw_sqns, sizeof(rxw_sqns)) ||
            !pgm_setsockopt(sock, IPPROTO_PGM, PGM_PEER_EXPIRY, &peer_expiry, sizeof(peer_expiry)) ||
            !pgm_setsockopt(sock, IPPROTO_PGM, PGM_SPMR_EXPIRY, &spmr_expiry, sizeof(spmr_expiry)) ||
            !pgm_setsockopt(sock, IPPROTO_PGM, PGM_NAK_BO_IVL, &nak_bo_ivl, sizeof(nak_bo_ivl)) ||
            !pgm_setsockopt(sock, IPPROTO_PGM, PGM_NAK_RPT_IVL, &nak_rpt_ivl, sizeof(nak_rpt_ivl)) ||
            !pgm_setsockopt(sock, IPPROTO_PGM, PGM_NAK_RDATA_IVL, &nak_rdata_ivl, sizeof(nak_rdata_ivl)) ||
            !pgm_setsockopt(sock, IPPROTO_PGM, PGM_NAK_DATA_RETRIES, &nak_data_retries, sizeof(nak_data_retries)) ||
            !pgm_setsockopt(sock, IPPROTO_PGM, PGM_NAK_NCF_RETRIES, &nak_ncf_retries, sizeof(nak_ncf_retries)))
            goto err_abort;
    }
    else
    {
        const int send_only = 1, max_rte = (int)((options.rate * 1000) / 8),
                  txw_max_tpdu = (int)options.multicast_maxtpdu, txw_sqns = compute_sqns(txw_max_tpdu),
                  ambient_spm     = pgm_secs(30),
                  heartbeat_spm[] = {pgm_msecs(100), pgm_msecs(100), pgm_msecs(100), pgm_msecs(100), pgm_msecs(1300),
                                     pgm_secs(7),    pgm_secs(16),   pgm_secs(25),   pgm_secs(30)};

        if (!pgm_setsockopt(sock, IPPROTO_PGM, PGM_SEND_ONLY, &send_only, sizeof(send_only)) ||
            !pgm_setsockopt(sock, IPPROTO_PGM, PGM_ODATA_MAX_RTE, &max_rte, sizeof(max_rte)) ||
            !pgm_setsockopt(sock, IPPROTO_PGM, PGM_TXW_SQNS, &txw_sqns, sizeof(txw_sqns)) ||
            !pgm_setsockopt(sock, IPPROTO_PGM, PGM_AMBIENT_SPM, &ambient_spm, sizeof(ambient_spm)) ||
            !pgm_setsockopt(sock, IPPROTO_PGM, PGM_HEARTBEAT_SPM, &heartbeat_spm, sizeof(heartbeat_spm)))
            goto err_abort;
    }

    //  PGM transport GSI.
    struct pgm_sockaddr_t addr;

    memset(&addr, 0, sizeof(addr));
    addr.sa_port       = port_number;
    addr.sa_addr.sport = DEFAULT_DATA_SOURCE_PORT;

    //  Create random GSI.
    uint32_t buf[2];
    buf[0] = generate_random();
    buf[1] = generate_random();
    if (!pgm_gsi_create_from_data(&addr.sa_addr.gsi, (uint8_t *)buf, 8))
        goto err_abort;

    //  Bind a transport to the specified network devices.
    struct pgm_interface_req_t if_req;
    memset(&if_req, 0, sizeof(if_req));
    if_req.ir_interface = res->ai_recv_addrs[0].gsr_interface;
    if_req.ir_scope_id  = 0;
    if (AF_INET6 == sa_family)
    {
        struct sockaddr_in6 sa6;
        memcpy(&sa6, &res->ai_recv_addrs[0].gsr_group, sizeof(sa6));
        if_req.ir_scope_id = sa6.sin6_scope_id;
    }
    if (!pgm_bind3(sock, &addr, sizeof(addr), &if_req, sizeof(if_req), &if_req, sizeof(if_req), &pgm_error))
    {
        //  Invalid parameters don't set pgm_error_t.
        zmq_assert(pgm_error != NULL);
        if ((pgm_error->domain == PGM_ERROR_DOMAIN_SOCKET || pgm_error->domain == PGM_ERROR_DOMAIN_IF) &&
            (pgm_error->code != PGM_ERROR_INVAL && pgm_error->code != PGM_ERROR_BADF &&
             pgm_error->code != PGM_ERROR_FAULT))

            //  User, host, or network configuration or transient error.
            goto err_abort;

        //  Fatal OpenPGM internal error.
        zmq_assert(false);
    }

    //  Join IP multicast groups.
    for (unsigned i = 0; i < res->ai_recv_addrs_len; i++)
    {
        if (!pgm_setsockopt(sock, IPPROTO_PGM, PGM_JOIN_GROUP, &res->ai_recv_addrs[i], sizeof(struct group_req)))
            goto err_abort;
    }
    if (!pgm_setsockopt(sock, IPPROTO_PGM, PGM_SEND_GROUP, &res->ai_send_addrs[0], sizeof(struct group_req)))
        goto err_abort;

    pgm_freeaddrinfo(res);
    res = NULL;

    //  Set IP level parameters.
    {
        // Multicast loopback disabled by default
        const int multicast_loop = 0;
        if (!pgm_setsockopt(sock, IPPROTO_PGM, PGM_MULTICAST_LOOP, &multicast_loop, sizeof(multicast_loop)))
            goto err_abort;

        const int multicast_hops = options.multicast_hops;
        if (!pgm_setsockopt(sock, IPPROTO_PGM, PGM_MULTICAST_HOPS, &multicast_hops, sizeof(multicast_hops)))
            goto err_abort;

        //  Expedited Forwarding PHB for network elements, no ECN.
        //  Ignore return value due to varied runtime support.
        const int dscp = 0x2e << 2;
        if (AF_INET6 != sa_family)
            pgm_setsockopt(sock, IPPROTO_PGM, PGM_TOS, &dscp, sizeof(dscp));

        const int nonblocking = 1;
        if (!pgm_setsockopt(sock, IPPROTO_PGM, PGM_NOBLOCK, &nonblocking, sizeof(nonblocking)))
            goto err_abort;
    }

    //  Connect PGM transport to start state machine.
    if (!pgm_connect(sock, &pgm_error))
    {
        //  Invalid parameters don't set pgm_error_t.
        zmq_assert(pgm_error != NULL);
        goto err_abort;
    }

    //  For receiver transport preallocate pgm_msgv array.
    if (receiver)
    {
        zmq_assert(options.in_batch_size > 0);
        size_t max_tsdu_size = get_max_tsdu_size();
        pgm_msgv_len         = (int)options.in_batch_size / max_tsdu_size;
        if ((int)options.in_batch_size % max_tsdu_size)
            pgm_msgv_len++;
        zmq_assert(pgm_msgv_len);

        pgm_msgv = (pgm_msgv_t *)malloc(sizeof(pgm_msgv_t) * pgm_msgv_len);
        alloc_assert(pgm_msgv);
    }

    return 0;

err_abort:
    if (sock != NULL)
    {
        pgm_close(sock, FALSE);
        sock = NULL;
    }
    if (res != NULL)
    {
        pgm_freeaddrinfo(res);
        res = NULL;
    }
    if (pgm_error != NULL)
    {
        pgm_error_free(pgm_error);
        pgm_error = NULL;
    }
    errno = EINVAL;
    return -1;
}

zmq::pgm_socket_t::~pgm_socket_t()
{
    if (pgm_msgv)
        free(pgm_msgv);
    if (sock)
        pgm_close(sock, TRUE);
}

//  Get receiver fds. receive_fd_ is signaled for incoming packets,
//  waiting_pipe_fd_ is signaled for state driven events and data.
void zmq::pgm_socket_t::get_receiver_fds(fd_t *receive_fd_, fd_t *waiting_pipe_fd_)
{
    socklen_t socklen;
    bool      rc;

    zmq_assert(receive_fd_);
    zmq_assert(waiting_pipe_fd_);

    socklen = sizeof(*receive_fd_);
    rc      = pgm_getsockopt(sock, IPPROTO_PGM, PGM_RECV_SOCK, receive_fd_, &socklen);
    zmq_assert(rc);
    zmq_assert(socklen == sizeof(*receive_fd_));

    socklen = sizeof(*waiting_pipe_fd_);
    rc      = pgm_getsockopt(sock, IPPROTO_PGM, PGM_PENDING_SOCK, waiting_pipe_fd_, &socklen);
    zmq_assert(rc);
    zmq_assert(socklen == sizeof(*waiting_pipe_fd_));
}

//  Get fds and store them into user allocated memory.
//  send_fd is for non-blocking send wire notifications.
//  receive_fd_ is for incoming back-channel protocol packets.
//  rdata_notify_fd_ is raised for waiting repair transmissions.
//  pending_notify_fd_ is for state driven events.
void zmq::pgm_socket_t::get_sender_fds(fd_t *send_fd_, fd_t *receive_fd_, fd_t *rdata_notify_fd_,
                                       fd_t *pending_notify_fd_)
{
    socklen_t socklen;
    bool      rc;

    zmq_assert(send_fd_);
    zmq_assert(receive_fd_);
    zmq_assert(rdata_notify_fd_);
    zmq_assert(pending_notify_fd_);

    socklen = sizeof(*send_fd_);
    rc      = pgm_getsockopt(sock, IPPROTO_PGM, PGM_SEND_SOCK, send_fd_, &socklen);
    zmq_assert(rc);
    zmq_assert(socklen == sizeof(*receive_fd_));

    socklen = sizeof(*receive_fd_);
    rc      = pgm_getsockopt(sock, IPPROTO_PGM, PGM_RECV_SOCK, receive_fd_, &socklen);
    zmq_assert(rc);
    zmq_assert(socklen == sizeof(*receive_fd_));

    socklen = sizeof(*rdata_notify_fd_);
    rc      = pgm_getsockopt(sock, IPPROTO_PGM, PGM_REPAIR_SOCK, rdata_notify_fd_, &socklen);
    zmq_assert(rc);
    zmq_assert(socklen == sizeof(*rdata_notify_fd_));

    socklen = sizeof(*pending_notify_fd_);
    rc      = pgm_getsockopt(sock, IPPROTO_PGM, PGM_PENDING_SOCK, pending_notify_fd_, &socklen);
    zmq_assert(rc);
    zmq_assert(socklen == sizeof(*pending_notify_fd_));
}

//  Send one APDU, transmit window owned memory.
//  data_len_ must be less than one TPDU.
size_t zmq::pgm_socket_t::send(unsigned char *data_, size_t data_len_)
{
    size_t nbytes = 0;

    const int status = pgm_send(sock, data_, data_len_, &nbytes);

    //  We have to write all data as one packet.
    if (nbytes > 0)
    {
        zmq_assert(status == PGM_IO_STATUS_NORMAL);
        zmq_assert(nbytes == data_len_);
    }
    else
    {
        zmq_assert(status == PGM_IO_STATUS_RATE_LIMITED || status == PGM_IO_STATUS_WOULD_BLOCK);

        if (status == PGM_IO_STATUS_RATE_LIMITED)
            errno = ENOMEM;
        else
            errno = EBUSY;
    }

    //  Save return value.
    last_tx_status = status;

    return nbytes;
}

long zmq::pgm_socket_t::get_rx_timeout()
{
    if (last_rx_status != PGM_IO_STATUS_RATE_LIMITED && last_rx_status != PGM_IO_STATUS_TIMER_PENDING)
        return -1;

    struct timeval tv;
    socklen_t      optlen = sizeof(tv);
    const bool     rc =
        pgm_getsockopt(sock, IPPROTO_PGM,
                       last_rx_status == PGM_IO_STATUS_RATE_LIMITED ? PGM_RATE_REMAIN : PGM_TIME_REMAIN, &tv, &optlen);
    zmq_assert(rc);

    const long timeout = (tv.tv_sec * 1000) + (tv.tv_usec / 1000);

    return timeout;
}

long zmq::pgm_socket_t::get_tx_timeout()
{
    if (last_tx_status != PGM_IO_STATUS_RATE_LIMITED)
        return -1;

    struct timeval tv;
    socklen_t      optlen = sizeof(tv);
    const bool     rc     = pgm_getsockopt(sock, IPPROTO_PGM, PGM_RATE_REMAIN, &tv, &optlen);
    zmq_assert(rc);

    const long timeout = (tv.tv_sec * 1000) + (tv.tv_usec / 1000);

    return timeout;
}

//  Return max TSDU size without fragmentation from current PGM transport.
size_t zmq::pgm_socket_t::get_max_tsdu_size()
{
    int       max_tsdu = 0;
    socklen_t optlen   = sizeof(max_tsdu);

    bool rc = pgm_getsockopt(sock, IPPROTO_PGM, PGM_MSS, &max_tsdu, &optlen);
    zmq_assert(rc);
    zmq_assert(optlen == sizeof(max_tsdu));
    return (size_t)max_tsdu;
}

//  pgm_recvmsgv is called to fill the pgm_msgv array up to  pgm_msgv_len.
//  In subsequent calls data from pgm_msgv structure are returned.
ssize_t zmq::pgm_socket_t::receive(void **raw_data_, const pgm_tsi_t **tsi_)
{
    size_t raw_data_len = 0;

    //  We just sent all data from pgm_transport_recvmsgv up
    //  and have to return 0 that another engine in this thread is scheduled.
    if (nbytes_rec == nbytes_processed && nbytes_rec > 0)
    {
        //  Reset all the counters.
        nbytes_rec         = 0;
        nbytes_processed   = 0;
        pgm_msgv_processed = 0;
        errno              = EAGAIN;
        return 0;
    }

    //  If we have are going first time or if we have processed all pgm_msgv_t
    //  structure previously read from the pgm socket.
    if (nbytes_rec == nbytes_processed)
    {
        //  Check program flow.
        zmq_assert(pgm_msgv_processed == 0);
        zmq_assert(nbytes_processed == 0);
        zmq_assert(nbytes_rec == 0);

        //  Receive a vector of Application Protocol Domain Unit's (APDUs)
        //  from the transport.
        pgm_error_t *pgm_error = NULL;

        const int status = pgm_recvmsgv(sock, pgm_msgv, pgm_msgv_len, MSG_ERRQUEUE, &nbytes_rec, &pgm_error);

        //  Invalid parameters.
        zmq_assert(status != PGM_IO_STATUS_ERROR);

        last_rx_status = status;

        //  In a case when no ODATA/RDATA fired POLLIN event (SPM...)
        //  pgm_recvmsg returns PGM_IO_STATUS_TIMER_PENDING.
        if (status == PGM_IO_STATUS_TIMER_PENDING)
        {
            zmq_assert(nbytes_rec == 0);

            //  In case if no RDATA/ODATA caused POLLIN 0 is
            //  returned.
            nbytes_rec = 0;
            errno      = EBUSY;
            return 0;
        }

        //  Send SPMR, NAK, ACK is rate limited.
        if (status == PGM_IO_STATUS_RATE_LIMITED)
        {
            zmq_assert(nbytes_rec == 0);

            //  In case if no RDATA/ODATA caused POLLIN 0 is returned.
            nbytes_rec = 0;
            errno      = ENOMEM;
            return 0;
        }

        //  No peers and hence no incoming packets.
        if (status == PGM_IO_STATUS_WOULD_BLOCK)
        {
            zmq_assert(nbytes_rec == 0);

            //  In case if no RDATA/ODATA caused POLLIN 0 is returned.
            nbytes_rec = 0;
            errno      = EAGAIN;
            return 0;
        }

        //  Data loss.
        if (status == PGM_IO_STATUS_RESET)
        {
            struct pgm_sk_buff_t *skb = pgm_msgv[0].msgv_skb[0];

            //  Save lost data TSI.
            *tsi_      = &skb->tsi;
            nbytes_rec = 0;

            //  In case of dala loss -1 is returned.
            errno = EINVAL;
            pgm_free_skb(skb);
            return -1;
        }

        zmq_assert(status == PGM_IO_STATUS_NORMAL);
    }
    else
    {
        zmq_assert(pgm_msgv_processed <= pgm_msgv_len);
    }

    // Zero byte payloads are valid in PGM, but not 0MQ protocol.
    zmq_assert(nbytes_rec > 0);

    // Only one APDU per pgm_msgv_t structure is allowed.
    zmq_assert(pgm_msgv[pgm_msgv_processed].msgv_len == 1);

    struct pgm_sk_buff_t *skb = pgm_msgv[pgm_msgv_processed].msgv_skb[0];

    //  Take pointers from pgm_msgv_t structure.
    *raw_data_   = skb->data;
    raw_data_len = skb->len;

    //  Save current TSI.
    *tsi_ = &skb->tsi;

    //  Move the the next pgm_msgv_t structure.
    pgm_msgv_processed++;
    zmq_assert(pgm_msgv_processed <= pgm_msgv_len);
    nbytes_processed += raw_data_len;

    return raw_data_len;
}

void zmq::pgm_socket_t::process_upstream()
{
    pgm_msgv_t dummy_msg;

    size_t       dummy_bytes = 0;
    pgm_error_t *pgm_error   = NULL;

    const int status = pgm_recvmsgv(sock, &dummy_msg, 1, MSG_ERRQUEUE, &dummy_bytes, &pgm_error);

    //  Invalid parameters.
    zmq_assert(status != PGM_IO_STATUS_ERROR);

    //  No data should be returned.
    zmq_assert(dummy_bytes == 0 && (status == PGM_IO_STATUS_TIMER_PENDING || status == PGM_IO_STATUS_RATE_LIMITED ||
                                    status == PGM_IO_STATUS_WOULD_BLOCK));

    last_rx_status = status;

    if (status == PGM_IO_STATUS_TIMER_PENDING)
        errno = EBUSY;
    else if (status == PGM_IO_STATUS_RATE_LIMITED)
        errno = ENOMEM;
    else
        errno = EAGAIN;
}

int zmq::pgm_socket_t::compute_sqns(int tpdu_)
{
    //  Convert rate into B/ms.
    uint64_t rate = uint64_t(options.rate) / 8;

    //  Compute the size of the buffer in bytes.
    uint64_t size = uint64_t(options.recovery_ivl) * rate;

    //  Translate the size into number of packets.
    uint64_t sqns = size / tpdu_;

    //  Buffer should be able to hold at least one packet.
    if (sqns == 0)
        sqns = 1;

    return (int)sqns;
}

#endif

//========= end of pgm_socket.cpp ============

//========= begin of pipe.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <new>
#include <stddef.h>

// ans ignore: #include "macros.hpp"
// ans ignore: #include "pipe.hpp"
// ans ignore: #include "err.hpp"

// ans ignore: #include "ypipe.hpp"
// ans ignore: #include "ypipe_conflate.hpp"

int zmq::pipepair(class object_t *parents_[2], class pipe_t *pipes_[2], int hwms_[2], bool conflate_[2])
{
    //   Creates two pipe objects. These objects are connected by two ypipes,
    //   each to pass messages in one direction.

    typedef ypipe_t<msg_t, message_pipe_granularity> upipe_normal_t;
    typedef ypipe_conflate_t<msg_t>                  upipe_conflate_t;

    pipe_t::upipe_t *upipe1;
    if (conflate_[0])
        upipe1 = new (std::nothrow) upipe_conflate_t();
    else
        upipe1 = new (std::nothrow) upipe_normal_t();
    alloc_assert(upipe1);

    pipe_t::upipe_t *upipe2;
    if (conflate_[1])
        upipe2 = new (std::nothrow) upipe_conflate_t();
    else
        upipe2 = new (std::nothrow) upipe_normal_t();
    alloc_assert(upipe2);

    pipes_[0] = new (std::nothrow) pipe_t(parents_[0], upipe1, upipe2, hwms_[1], hwms_[0], conflate_[0]);
    alloc_assert(pipes_[0]);
    pipes_[1] = new (std::nothrow) pipe_t(parents_[1], upipe2, upipe1, hwms_[0], hwms_[1], conflate_[1]);
    alloc_assert(pipes_[1]);

    pipes_[0]->set_peer(pipes_[1]);
    pipes_[1]->set_peer(pipes_[0]);

    return 0;
}

void zmq::send_routing_id(pipe_t *pipe_, const options_t &options_)
{
    zmq::msg_t id;
    const int  rc = id.init_size(options_.routing_id_size);
    errno_assert(rc == 0);
    memcpy(id.data(), options_.routing_id, options_.routing_id_size);
    id.set_flags(zmq::msg_t::routing_id);
    const bool written = pipe_->write(&id);
    zmq_assert(written);
    pipe_->flush();
}

zmq::pipe_t::pipe_t(object_t *parent_, upipe_t *inpipe_, upipe_t *outpipe_, int inhwm_, int outhwm_, bool conflate_)
    : object_t(parent_), _in_pipe(inpipe_), _out_pipe(outpipe_), _in_active(true), _out_active(true), _hwm(outhwm_),
      _lwm(compute_lwm(inhwm_)), _in_hwm_boost(-1), _out_hwm_boost(-1), _msgs_read(0), _msgs_written(0),
      _peers_msgs_read(0), _peer(NULL), _sink(NULL), _state(active), _delay(true), _server_socket_routing_id(0),
      _conflate(conflate_)
{
}

zmq::pipe_t::~pipe_t() {}

void zmq::pipe_t::set_peer(pipe_t *peer_)
{
    //  Peer can be set once only.
    zmq_assert(!_peer);
    _peer = peer_;
}

void zmq::pipe_t::set_event_sink(i_pipe_events *sink_)
{
    // Sink can be set once only.
    zmq_assert(!_sink);
    _sink = sink_;
}

void zmq::pipe_t::set_server_socket_routing_id(uint32_t server_socket_routing_id_)
{
    _server_socket_routing_id = server_socket_routing_id_;
}

uint32_t zmq::pipe_t::get_server_socket_routing_id() const { return _server_socket_routing_id; }

void zmq::pipe_t::set_router_socket_routing_id(const blob_t &router_socket_routing_id_)
{
    _router_socket_routing_id.set_deep_copy(router_socket_routing_id_);
}

const zmq::blob_t &zmq::pipe_t::get_routing_id() const { return _router_socket_routing_id; }

bool zmq::pipe_t::check_read()
{
    if (unlikely(!_in_active))
        return false;
    if (unlikely(_state != active && _state != waiting_for_delimiter))
        return false;

    //  Check if there's an item in the pipe.
    if (!_in_pipe->check_read())
    {
        _in_active = false;
        return false;
    }

    //  If the next item in the pipe is message delimiter,
    //  initiate termination process.
    if (_in_pipe->probe(is_delimiter))
    {
        msg_t      msg;
        const bool ok = _in_pipe->read(&msg);
        zmq_assert(ok);
        process_delimiter();
        return false;
    }

    return true;
}

bool zmq::pipe_t::read(msg_t *msg_)
{
    if (unlikely(!_in_active))
        return false;
    if (unlikely(_state != active && _state != waiting_for_delimiter))
        return false;

    for (bool payload_read = false; !payload_read;)
    {
        if (!_in_pipe->read(msg_))
        {
            _in_active = false;
            return false;
        }

        //  If this is a credential, ignore it and receive next message.
        if (unlikely(msg_->is_credential()))
        {
            const int rc = msg_->close();
            zmq_assert(rc == 0);
        }
        else
            payload_read = true;
    }

    //  If delimiter was read, start termination process of the pipe.
    if (msg_->is_delimiter())
    {
        process_delimiter();
        return false;
    }

    if (!(msg_->flags() & msg_t::more) && !msg_->is_routing_id())
        _msgs_read++;

    if (_lwm > 0 && _msgs_read % _lwm == 0)
        send_activate_write(_peer, _msgs_read);

    return true;
}

bool zmq::pipe_t::check_write()
{
    if (unlikely(!_out_active || _state != active))
        return false;

    const bool full = !check_hwm();

    if (unlikely(full))
    {
        _out_active = false;
        return false;
    }

    return true;
}

bool zmq::pipe_t::write(msg_t *msg_)
{
    if (unlikely(!check_write()))
        return false;

    const bool more          = (msg_->flags() & msg_t::more) != 0;
    const bool is_routing_id = msg_->is_routing_id();
    _out_pipe->write(*msg_, more);
    if (!more && !is_routing_id)
        _msgs_written++;

    return true;
}

void zmq::pipe_t::rollback() const
{
    //  Remove incomplete message from the outbound pipe.
    msg_t msg;
    if (_out_pipe)
    {
        while (_out_pipe->unwrite(&msg))
        {
            zmq_assert(msg.flags() & msg_t::more);
            const int rc = msg.close();
            errno_assert(rc == 0);
        }
    }
}

void zmq::pipe_t::flush()
{
    //  The peer does not exist anymore at this point.
    if (_state == term_ack_sent)
        return;

    if (_out_pipe && !_out_pipe->flush())
        send_activate_read(_peer);
}

void zmq::pipe_t::process_activate_read()
{
    if (!_in_active && (_state == active || _state == waiting_for_delimiter))
    {
        _in_active = true;
        _sink->read_activated(this);
    }
}

void zmq::pipe_t::process_activate_write(uint64_t msgs_read_)
{
    //  Remember the peer's message sequence number.
    _peers_msgs_read = msgs_read_;

    if (!_out_active && _state == active)
    {
        _out_active = true;
        _sink->write_activated(this);
    }
}

void zmq::pipe_t::process_hiccup(void *pipe_)
{
    //  Destroy old outpipe. Note that the read end of the pipe was already
    //  migrated to this thread.
    zmq_assert(_out_pipe);
    _out_pipe->flush();
    msg_t msg;
    while (_out_pipe->read(&msg))
    {
        if (!(msg.flags() & msg_t::more))
            _msgs_written--;
        const int rc = msg.close();
        errno_assert(rc == 0);
    }
    LIBZMQ_DELETE(_out_pipe);

    //  Plug in the new outpipe.
    zmq_assert(pipe_);
    _out_pipe   = static_cast<upipe_t *>(pipe_);
    _out_active = true;

    //  If appropriate, notify the user about the hiccup.
    if (_state == active)
        _sink->hiccuped(this);
}

void zmq::pipe_t::process_pipe_term()
{
    zmq_assert(_state == active || _state == delimiter_received || _state == term_req_sent1);

    //  This is the simple case of peer-induced termination. If there are no
    //  more pending messages to read, or if the pipe was configured to drop
    //  pending messages, we can move directly to the term_ack_sent state.
    //  Otherwise we'll hang up in waiting_for_delimiter state till all
    //  pending messages are read.
    if (_state == active)
    {
        if (_delay)
            _state = waiting_for_delimiter;
        else
        {
            _state    = term_ack_sent;
            _out_pipe = NULL;
            send_pipe_term_ack(_peer);
        }
    }

    //  Delimiter happened to arrive before the term command. Now we have the
    //  term command as well, so we can move straight to term_ack_sent state.
    else if (_state == delimiter_received)
    {
        _state    = term_ack_sent;
        _out_pipe = NULL;
        send_pipe_term_ack(_peer);
    }

    //  This is the case where both ends of the pipe are closed in parallel.
    //  We simply reply to the request by ack and continue waiting for our
    //  own ack.
    else if (_state == term_req_sent1)
    {
        _state    = term_req_sent2;
        _out_pipe = NULL;
        send_pipe_term_ack(_peer);
    }
}

void zmq::pipe_t::process_pipe_term_ack()
{
    //  Notify the user that all the references to the pipe should be dropped.
    zmq_assert(_sink);
    _sink->pipe_terminated(this);

    //  In term_ack_sent and term_req_sent2 states there's nothing to do.
    //  Simply deallocate the pipe. In term_req_sent1 state we have to ack
    //  the peer before deallocating this side of the pipe.
    //  All the other states are invalid.
    if (_state == term_req_sent1)
    {
        _out_pipe = NULL;
        send_pipe_term_ack(_peer);
    }
    else
        zmq_assert(_state == term_ack_sent || _state == term_req_sent2);

    //  We'll deallocate the inbound pipe, the peer will deallocate the outbound
    //  pipe (which is an inbound pipe from its point of view).
    //  First, delete all the unread messages in the pipe. We have to do it by
    //  hand because msg_t doesn't have automatic destructor. Then deallocate
    //  the ypipe itself.

    if (!_conflate)
    {
        msg_t msg;
        while (_in_pipe->read(&msg))
        {
            const int rc = msg.close();
            errno_assert(rc == 0);
        }
    }

    LIBZMQ_DELETE(_in_pipe);

    //  Deallocate the pipe object
    delete this;
}

void zmq::pipe_t::process_pipe_hwm(int inhwm_, int outhwm_) { set_hwms(inhwm_, outhwm_); }

void zmq::pipe_t::set_nodelay() { this->_delay = false; }

void zmq::pipe_t::terminate(bool delay_)
{
    //  Overload the value specified at pipe creation.
    _delay = delay_;

    //  If terminate was already called, we can ignore the duplicate invocation.
    if (_state == term_req_sent1 || _state == term_req_sent2)
    {
        return;
    }
    //  If the pipe is in the final phase of async termination, it's going to
    //  closed anyway. No need to do anything special here.
    if (_state == term_ack_sent)
    {
        return;
    }
    //  The simple sync termination case. Ask the peer to terminate and wait
    //  for the ack.
    if (_state == active)
    {
        send_pipe_term(_peer);
        _state = term_req_sent1;
    }
    //  There are still pending messages available, but the user calls
    //  'terminate'. We can act as if all the pending messages were read.
    else if (_state == waiting_for_delimiter && !_delay)
    {
        //  Drop any unfinished outbound messages.
        rollback();
        _out_pipe = NULL;
        send_pipe_term_ack(_peer);
        _state = term_ack_sent;
    }
    //  If there are pending messages still available, do nothing.
    else if (_state == waiting_for_delimiter)
    {
    }
    //  We've already got delimiter, but not term command yet. We can ignore
    //  the delimiter and ack synchronously terminate as if we were in
    //  active state.
    else if (_state == delimiter_received)
    {
        send_pipe_term(_peer);
        _state = term_req_sent1;
    }
    //  There are no other states.
    else
    {
        zmq_assert(false);
    }

    //  Stop outbound flow of messages.
    _out_active = false;

    if (_out_pipe)
    {
        //  Drop any unfinished outbound messages.
        rollback();

        //  Write the delimiter into the pipe. Note that watermarks are not
        //  checked; thus the delimiter can be written even when the pipe is full.
        msg_t msg;
        msg.init_delimiter();
        _out_pipe->write(msg, false);
        flush();
    }
}

bool zmq::pipe_t::is_delimiter(const msg_t &msg_) { return msg_.is_delimiter(); }

int zmq::pipe_t::compute_lwm(int hwm_)
{
    //  Compute the low water mark. Following point should be taken
    //  into consideration:
    //
    //  1. LWM has to be less than HWM.
    //  2. LWM cannot be set to very low value (such as zero) as after filling
    //     the queue it would start to refill only after all the messages are
    //     read from it and thus unnecessarily hold the progress back.
    //  3. LWM cannot be set to very high value (such as HWM-1) as it would
    //     result in lock-step filling of the queue - if a single message is
    //     read from a full queue, writer thread is resumed to write exactly one
    //     message to the queue and go back to sleep immediately. This would
    //     result in low performance.
    //
    //  Given the 3. it would be good to keep HWM and LWM as far apart as
    //  possible to reduce the thread switching overhead to almost zero.
    //  Let's make LWM 1/2 of HWM.
    const int result = (hwm_ + 1) / 2;

    return result;
}

void zmq::pipe_t::process_delimiter()
{
    zmq_assert(_state == active || _state == waiting_for_delimiter);

    if (_state == active)
        _state = delimiter_received;
    else
    {
        _out_pipe = NULL;
        send_pipe_term_ack(_peer);
        _state = term_ack_sent;
    }
}

void zmq::pipe_t::hiccup()
{
    //  If termination is already under way do nothing.
    if (_state != active)
        return;

    //  We'll drop the pointer to the inpipe. From now on, the peer is
    //  responsible for deallocating it.

    //  Create new inpipe.
    _in_pipe = _conflate ? static_cast<upipe_t *>(new (std::nothrow) ypipe_conflate_t<msg_t>())
                         : new (std::nothrow) ypipe_t<msg_t, message_pipe_granularity>();

    alloc_assert(_in_pipe);
    _in_active = true;

    //  Notify the peer about the hiccup.
    send_hiccup(_peer, _in_pipe);
}

void zmq::pipe_t::set_hwms(int inhwm_, int outhwm_)
{
    int in  = inhwm_ + std::max(_in_hwm_boost, 0);
    int out = outhwm_ + std::max(_out_hwm_boost, 0);

    // if either send or recv side has hwm <= 0 it means infinite so we should set hwms infinite
    if (inhwm_ <= 0 || _in_hwm_boost == 0)
        in = 0;

    if (outhwm_ <= 0 || _out_hwm_boost == 0)
        out = 0;

    _lwm = compute_lwm(in);
    _hwm = out;
}

void zmq::pipe_t::set_hwms_boost(int inhwmboost_, int outhwmboost_)
{
    _in_hwm_boost  = inhwmboost_;
    _out_hwm_boost = outhwmboost_;
}

bool zmq::pipe_t::check_hwm() const
{
    const bool full = _hwm > 0 && _msgs_written - _peers_msgs_read >= uint64_t(_hwm);
    return !full;
}

void zmq::pipe_t::send_hwms_to_peer(int inhwm_, int outhwm_) { send_pipe_hwm(_peer, inhwm_, outhwm_); }

void zmq::pipe_t::set_endpoint_pair(zmq::endpoint_uri_pair_t endpoint_pair_)
{
    _endpoint_pair = ZMQ_MOVE(endpoint_pair_);
}

const zmq::endpoint_uri_pair_t &zmq::pipe_t::get_endpoint_pair() const { return _endpoint_pair; }

void zmq::pipe_t::send_stats_to_peer(own_t *socket_base_)
{
    endpoint_uri_pair_t *ep = new (std::nothrow) endpoint_uri_pair_t(_endpoint_pair);
    send_pipe_peer_stats(_peer, _msgs_written - _peers_msgs_read, socket_base_, ep);
}

void zmq::pipe_t::process_pipe_peer_stats(uint64_t queue_count_, own_t *socket_base_,
                                          endpoint_uri_pair_t *endpoint_pair_)
{
    send_pipe_stats_publish(socket_base_, queue_count_, _msgs_written - _peers_msgs_read, endpoint_pair_);
}

//========= end of pipe.cpp ============

//========= begin of plain_client.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"

#include <limits.h>
#include <string>

// ans ignore: #include "msg.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "plain_client.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "plain_common.hpp"

zmq::plain_client_t::plain_client_t(session_base_t *const session_, const options_t &options_)
    : mechanism_base_t(session_, options_), _state(sending_hello)
{
}

zmq::plain_client_t::~plain_client_t() {}

int zmq::plain_client_t::next_handshake_command(msg_t *msg_)
{
    int rc = 0;

    switch (_state)
    {
    case sending_hello:
        produce_hello(msg_);
        _state = waiting_for_welcome;
        break;
    case sending_initiate:
        produce_initiate(msg_);
        _state = waiting_for_ready;
        break;
    default:
        errno = EAGAIN;
        rc    = -1;
    }
    return rc;
}

int zmq::plain_client_t::process_handshake_command(msg_t *msg_)
{
    const unsigned char *cmd_data  = static_cast<unsigned char *>(msg_->data());
    const size_t         data_size = msg_->size();

    int rc = 0;
    if (data_size >= welcome_prefix_len && !memcmp(cmd_data, welcome_prefix, welcome_prefix_len))
        rc = process_welcome(cmd_data, data_size);
    else if (data_size >= ready_prefix_len && !memcmp(cmd_data, ready_prefix, ready_prefix_len))
        rc = process_ready(cmd_data, data_size);
    else if (data_size >= error_prefix_len && !memcmp(cmd_data, error_prefix, error_prefix_len))
        rc = process_error(cmd_data, data_size);
    else
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNEXPECTED_COMMAND);
        errno = EPROTO;
        rc    = -1;
    }

    if (rc == 0)
    {
        rc = msg_->close();
        errno_assert(rc == 0);
        rc = msg_->init();
        errno_assert(rc == 0);
    }

    return rc;
}

zmq::mechanism_t::status_t zmq::plain_client_t::status() const
{
    switch (_state)
    {
    case ready:
        return mechanism_t::ready;
    case error_command_received:
        return mechanism_t::error;
    default:
        return mechanism_t::handshaking;
    }
}

void zmq::plain_client_t::produce_hello(msg_t *msg_) const
{
    const std::string username = options.plain_username;
    zmq_assert(username.length() <= UCHAR_MAX);

    const std::string password = options.plain_password;
    zmq_assert(password.length() <= UCHAR_MAX);

    const size_t command_size =
        hello_prefix_len + brief_len_size + username.length() + brief_len_size + password.length();

    const int rc = msg_->init_size(command_size);
    errno_assert(rc == 0);

    unsigned char *ptr = static_cast<unsigned char *>(msg_->data());
    memcpy(ptr, hello_prefix, hello_prefix_len);
    ptr += hello_prefix_len;

    *ptr++ = static_cast<unsigned char>(username.length());
    memcpy(ptr, username.c_str(), username.length());
    ptr += username.length();

    *ptr++ = static_cast<unsigned char>(password.length());
    memcpy(ptr, password.c_str(), password.length());
}

int zmq::plain_client_t::process_welcome(const unsigned char *cmd_data_, size_t data_size_)
{
    LIBZMQ_UNUSED(cmd_data_);

    if (_state != waiting_for_welcome)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNEXPECTED_COMMAND);
        errno = EPROTO;
        return -1;
    }
    if (data_size_ != welcome_prefix_len)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_WELCOME);
        errno = EPROTO;
        return -1;
    }
    _state = sending_initiate;
    return 0;
}

void zmq::plain_client_t::produce_initiate(msg_t *msg_) const
{
    make_command_with_basic_properties(msg_, initiate_prefix, initiate_prefix_len);
}

int zmq::plain_client_t::process_ready(const unsigned char *cmd_data_, size_t data_size_)
{
    if (_state != waiting_for_ready)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNEXPECTED_COMMAND);
        errno = EPROTO;
        return -1;
    }
    const int rc = parse_metadata(cmd_data_ + ready_prefix_len, data_size_ - ready_prefix_len);
    if (rc == 0)
        _state = ready;
    else
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_INVALID_METADATA);

    return rc;
}

int zmq::plain_client_t::process_error(const unsigned char *cmd_data_, size_t data_size_)
{
    if (_state != waiting_for_welcome && _state != waiting_for_ready)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNEXPECTED_COMMAND);
        errno = EPROTO;
        return -1;
    }
    const size_t start_of_error_reason = error_prefix_len + brief_len_size;
    if (data_size_ < start_of_error_reason)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_ERROR);
        errno = EPROTO;
        return -1;
    }
    const size_t error_reason_len = static_cast<size_t>(cmd_data_[error_prefix_len]);
    if (error_reason_len > data_size_ - start_of_error_reason)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_ERROR);
        errno = EPROTO;
        return -1;
    }
    const char *error_reason = reinterpret_cast<const char *>(cmd_data_) + start_of_error_reason;
    handle_error_reason(error_reason, error_reason_len);
    _state = error_command_received;
    return 0;
}

//========= end of plain_client.cpp ============

//========= begin of plain_server.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"

#include <string>

// ans ignore: #include "msg.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "plain_server.hpp"
// ans ignore: #include "wire.hpp"
// ans ignore: #include "plain_common.hpp"

zmq::plain_server_t::plain_server_t(session_base_t *session_, const std::string &peer_address_,
                                    const options_t &options_)
    : mechanism_base_t(session_, options_),
      zap_client_common_handshake_t(session_, peer_address_, options_, sending_welcome)
{
    //  Note that there is no point to PLAIN if ZAP is not set up to handle the
    //  username and password, so if ZAP is not configured it is considered a
    //  failure.
    //  Given this is a backward-incompatible change, it's behind a socket
    //  option disabled by default.
    if (options.zap_enforce_domain)
        zmq_assert(zap_required());
}

zmq::plain_server_t::~plain_server_t() {}

int zmq::plain_server_t::next_handshake_command(msg_t *msg_)
{
    int rc = 0;

    switch (state)
    {
    case sending_welcome:
        produce_welcome(msg_);
        state = waiting_for_initiate;
        break;
    case sending_ready:
        produce_ready(msg_);
        state = ready;
        break;
    case sending_error:
        produce_error(msg_);
        state = error_sent;
        break;
    default:
        errno = EAGAIN;
        rc    = -1;
    }
    return rc;
}

int zmq::plain_server_t::process_handshake_command(msg_t *msg_)
{
    int rc = 0;

    switch (state)
    {
    case waiting_for_hello:
        rc = process_hello(msg_);
        break;
    case waiting_for_initiate:
        rc = process_initiate(msg_);
        break;
    default:
        //  TODO see comment in curve_server_t::process_handshake_command
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNSPECIFIED);
        errno = EPROTO;
        rc    = -1;
        break;
    }
    if (rc == 0)
    {
        rc = msg_->close();
        errno_assert(rc == 0);
        rc = msg_->init();
        errno_assert(rc == 0);
    }
    return rc;
}

int zmq::plain_server_t::process_hello(msg_t *msg_)
{
    int rc = check_basic_command_structure(msg_);
    if (rc == -1)
        return -1;

    const char *ptr        = static_cast<char *>(msg_->data());
    size_t      bytes_left = msg_->size();

    if (bytes_left < hello_prefix_len || memcmp(ptr, hello_prefix, hello_prefix_len) != 0)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNEXPECTED_COMMAND);
        errno = EPROTO;
        return -1;
    }
    ptr += hello_prefix_len;
    bytes_left -= hello_prefix_len;

    if (bytes_left < 1)
    {
        //  PLAIN I: invalid PLAIN client, did not send username
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_HELLO);
        errno = EPROTO;
        return -1;
    }
    const uint8_t username_length = *ptr++;
    bytes_left -= sizeof(username_length);

    if (bytes_left < username_length)
    {
        //  PLAIN I: invalid PLAIN client, sent malformed username
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_HELLO);
        errno = EPROTO;
        return -1;
    }
    const std::string username = std::string(ptr, username_length);
    ptr += username_length;
    bytes_left -= username_length;
    if (bytes_left < 1)
    {
        //  PLAIN I: invalid PLAIN client, did not send password
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_HELLO);
        errno = EPROTO;
        return -1;
    }

    const uint8_t password_length = *ptr++;
    bytes_left -= sizeof(password_length);
    if (bytes_left != password_length)
    {
        //  PLAIN I: invalid PLAIN client, sent malformed password or
        //  extraneous data
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_MALFORMED_COMMAND_HELLO);
        errno = EPROTO;
        return -1;
    }

    const std::string password = std::string(ptr, password_length);

    //  Use ZAP protocol (RFC 27) to authenticate the user.
    rc = session->zap_connect();
    if (rc != 0)
    {
        session->get_socket()->event_handshake_failed_no_detail(session->get_endpoint(), EFAULT);
        return -1;
    }

    send_zap_request(username, password);
    state = waiting_for_zap_reply;

    //  TODO actually, it is quite unlikely that we can read the ZAP
    //  reply already, but removing this has some strange side-effect
    //  (probably because the pipe's in_active flag is true until a read
    //  is attempted)
    return receive_and_process_zap_reply() == -1 ? -1 : 0;
}

void zmq::plain_server_t::produce_welcome(msg_t *msg_) const
{
    const int rc = msg_->init_size(welcome_prefix_len);
    errno_assert(rc == 0);
    memcpy(msg_->data(), welcome_prefix, welcome_prefix_len);
}

int zmq::plain_server_t::process_initiate(msg_t *msg_)
{
    const unsigned char *ptr        = static_cast<unsigned char *>(msg_->data());
    const size_t         bytes_left = msg_->size();

    if (bytes_left < initiate_prefix_len || memcmp(ptr, initiate_prefix, initiate_prefix_len) != 0)
    {
        session->get_socket()->event_handshake_failed_protocol(session->get_endpoint(),
                                                               ZMQ_PROTOCOL_ERROR_ZMTP_UNEXPECTED_COMMAND);
        errno = EPROTO;
        return -1;
    }
    const int rc = parse_metadata(ptr + initiate_prefix_len, bytes_left - initiate_prefix_len);
    if (rc == 0)
        state = sending_ready;
    return rc;
}

void zmq::plain_server_t::produce_ready(msg_t *msg_) const
{
    make_command_with_basic_properties(msg_, ready_prefix, ready_prefix_len);
}

void zmq::plain_server_t::produce_error(msg_t *msg_) const
{
    const char expected_status_code_len = 3;
    zmq_assert(status_code.length() == static_cast<size_t>(expected_status_code_len));
    const size_t status_code_len_size = sizeof(expected_status_code_len);
    const int    rc = msg_->init_size(error_prefix_len + status_code_len_size + expected_status_code_len);
    zmq_assert(rc == 0);
    char *msg_data = static_cast<char *>(msg_->data());
    memcpy(msg_data, error_prefix, error_prefix_len);
    msg_data[error_prefix_len] = expected_status_code_len;
    memcpy(msg_data + error_prefix_len + status_code_len_size, status_code.c_str(), status_code.length());
}

void zmq::plain_server_t::send_zap_request(const std::string &username_, const std::string &password_)
{
    const uint8_t *credentials[]          = {reinterpret_cast<const uint8_t *>(username_.c_str()),
                                    reinterpret_cast<const uint8_t *>(password_.c_str())};
    size_t         credentials_sizes[]    = {username_.size(), password_.size()};
    const char     plain_mechanism_name[] = "PLAIN";
    zap_client_t::send_zap_request(plain_mechanism_name, sizeof(plain_mechanism_name) - 1, credentials,
                                   credentials_sizes, sizeof(credentials) / sizeof(credentials[0]));
}

//========= end of plain_server.cpp ============

//========= begin of poll.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "poll.hpp"
#if defined ZMQ_IOTHREAD_POLLER_USE_POLL

#include <algorithm>
#include <poll.h>
#include <sys/time.h>
#include <sys/types.h>

// ans ignore: #include "poll.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "config.hpp"
// ans ignore: #include "i_poll_events.hpp"

zmq::poll_t::poll_t(const zmq::thread_ctx_t &ctx_) : worker_poller_base_t(ctx_), retired(false) {}

zmq::poll_t::~poll_t() { stop_worker(); }

zmq::poll_t::handle_t zmq::poll_t::add_fd(fd_t fd_, i_poll_events *events_)
{
    check_thread();
    zmq_assert(fd_ != retired_fd);

    //  If the file descriptor table is too small expand it.
    fd_table_t::size_type sz = fd_table.size();
    if (sz <= (fd_table_t::size_type)fd_)
    {
        fd_table.resize(fd_ + 1);
        while (sz != (fd_table_t::size_type)(fd_ + 1))
        {
            fd_table[sz].index = retired_fd;
            ++sz;
        }
    }

    pollfd pfd = {fd_, 0, 0};
    pollset.push_back(pfd);
    zmq_assert(fd_table[fd_].index == retired_fd);

    fd_table[fd_].index  = pollset.size() - 1;
    fd_table[fd_].events = events_;

    //  Increase the load metric of the thread.
    adjust_load(1);

    return fd_;
}

void zmq::poll_t::rm_fd(handle_t handle_)
{
    check_thread();
    fd_t index = fd_table[handle_].index;
    zmq_assert(index != retired_fd);

    //  Mark the fd as unused.
    pollset[index].fd       = retired_fd;
    fd_table[handle_].index = retired_fd;
    retired                 = true;

    //  Decrease the load metric of the thread.
    adjust_load(-1);
}

void zmq::poll_t::set_pollin(handle_t handle_)
{
    check_thread();
    fd_t index = fd_table[handle_].index;
    pollset[index].events |= POLLIN;
}

void zmq::poll_t::reset_pollin(handle_t handle_)
{
    check_thread();
    fd_t index = fd_table[handle_].index;
    pollset[index].events &= ~((short)POLLIN);
}

void zmq::poll_t::set_pollout(handle_t handle_)
{
    check_thread();
    fd_t index = fd_table[handle_].index;
    pollset[index].events |= POLLOUT;
}

void zmq::poll_t::reset_pollout(handle_t handle_)
{
    check_thread();
    fd_t index = fd_table[handle_].index;
    pollset[index].events &= ~((short)POLLOUT);
}

void zmq::poll_t::stop()
{
    check_thread();
    //  no-op... thread is stopped when no more fds or timers are registered
}

int zmq::poll_t::max_fds() { return -1; }

void zmq::poll_t::loop()
{
    while (true)
    {
        //  Execute any due timers.
        int timeout = (int)execute_timers();

        cleanup_retired();

        if (pollset.empty())
        {
            zmq_assert(get_load() == 0);

            if (timeout == 0)
                break;

            // TODO sleep for timeout
            continue;
        }

        //  Wait for events.
        int rc = poll(&pollset[0], static_cast<nfds_t>(pollset.size()), timeout ? timeout : -1);
        if (rc == -1)
        {
            errno_assert(errno == EINTR);
            continue;
        }

        //  If there are no events (i.e. it's a timeout) there's no point
        //  in checking the pollset.
        if (rc == 0)
            continue;

        for (pollset_t::size_type i = 0; i != pollset.size(); i++)
        {
            zmq_assert(!(pollset[i].revents & POLLNVAL));
            if (pollset[i].fd == retired_fd)
                continue;
            if (pollset[i].revents & (POLLERR | POLLHUP))
                fd_table[pollset[i].fd].events->in_event();
            if (pollset[i].fd == retired_fd)
                continue;
            if (pollset[i].revents & POLLOUT)
                fd_table[pollset[i].fd].events->out_event();
            if (pollset[i].fd == retired_fd)
                continue;
            if (pollset[i].revents & POLLIN)
                fd_table[pollset[i].fd].events->in_event();
        }
    }
}

void zmq::poll_t::cleanup_retired()
{
    //  Clean up the pollset and update the fd_table accordingly.
    if (retired)
    {
        pollset_t::size_type i = 0;
        while (i < pollset.size())
        {
            if (pollset[i].fd == retired_fd)
                pollset.erase(pollset.begin() + i);
            else
            {
                fd_table[pollset[i].fd].index = i;
                i++;
            }
        }
        retired = false;
    }
}

#endif

//========= end of poll.cpp ============

//========= begin of poller_base.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "poller_base.hpp"
// ans ignore: #include "i_poll_events.hpp"
// ans ignore: #include "err.hpp"

zmq::poller_base_t::poller_base_t() {}

zmq::poller_base_t::~poller_base_t()
{
    //  Make sure there is no more load on the shutdown.
    zmq_assert(get_load() == 0);
}

int zmq::poller_base_t::get_load() const { return _load.get(); }

void zmq::poller_base_t::adjust_load(int amount_)
{
    if (amount_ > 0)
        _load.add(amount_);
    else if (amount_ < 0)
        _load.sub(-amount_);
}

void zmq::poller_base_t::add_timer(int timeout_, i_poll_events *sink_, int id_)
{
    uint64_t     expiration = _clock.now_ms() + timeout_;
    timer_info_t info       = {sink_, id_};
    _timers.insert(timers_t::value_type(expiration, info));
}

void zmq::poller_base_t::cancel_timer(i_poll_events *sink_, int id_)
{
    //  Complexity of this operation is O(n). We assume it is rarely used.
    for (timers_t::iterator it = _timers.begin(), end = _timers.end(); it != end; ++it)
        if (it->second.sink == sink_ && it->second.id == id_)
        {
            _timers.erase(it);
            return;
        }

    //  Timer not found.
    zmq_assert(false);
}

uint64_t zmq::poller_base_t::execute_timers()
{
    //  Fast track.
    if (_timers.empty())
        return 0;

    //  Get the current time.
    const uint64_t current = _clock.now_ms();

    //   Execute the timers that are already due.
    const timers_t::iterator begin = _timers.begin();
    const timers_t::iterator end   = _timers.end();
    uint64_t                 res   = 0;
    timers_t::iterator       it    = begin;
    for (; it != end; ++it)
    {
        //  If we have to wait to execute the item, same will be true about
        //  all the following items (multimap is sorted). Thus we can stop
        //  checking the subsequent timers.
        if (it->first > current)
        {
            res = it->first - current;
            break;
        }

        //  Trigger the timer.
        it->second.sink->timer_event(it->second.id);
    }

    //  Remove them from the list of active timers.
    _timers.erase(begin, it);

    //  Return the time to wait for the next timer (at least 1ms), or 0, if
    //  there are no more timers.
    return res;
}

zmq::worker_poller_base_t::worker_poller_base_t(const thread_ctx_t &ctx_) : _ctx(ctx_) {}

void zmq::worker_poller_base_t::stop_worker() { _worker.stop(); }

void zmq::worker_poller_base_t::start(const char *name_)
{
    zmq_assert(get_load() > 0);
    _ctx.start_thread(_worker, worker_routine, this, name_);
}

void zmq::worker_poller_base_t::check_thread()
{
#ifdef _DEBUG
    zmq_assert(!_worker.get_started() || _worker.is_current_thread());
#endif
}

void zmq::worker_poller_base_t::worker_routine(void *arg_) { (static_cast<worker_poller_base_t *>(arg_))->loop(); }

//========= end of poller_base.cpp ============

//========= begin of polling_util.cpp ============

/*
    Copyright (c) 2007-2018 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "polling_util.hpp"

#if defined ZMQ_POLL_BASED_ON_POLL
#include <algorithm>
#include <limits.h>

zmq::timeout_t zmq::compute_timeout(const bool first_pass_, const long timeout_, const uint64_t now_,
                                    const uint64_t end_)
{
    if (first_pass_)
        return 0;

    if (timeout_ < 0)
        return -1;

    return static_cast<zmq::timeout_t>(std::min<uint64_t>(end_ - now_, INT_MAX));
}
#endif

//========= end of polling_util.cpp ============

//========= begin of pollset.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "pollset.hpp"
#if defined ZMQ_IOTHREAD_POLLER_USE_POLLSET

#include <algorithm>
#include <new>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

// ans ignore: #include "macros.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "config.hpp"
// ans ignore: #include "i_poll_events.hpp"

zmq::pollset_t::pollset_t(const zmq::thread_ctx_t &ctx_) : ctx(ctx_), stopping(false)
{
    pollset_fd = pollset_create(-1);
    errno_assert(pollset_fd != -1);
}

zmq::pollset_t::~pollset_t()
{
    //  Wait till the worker thread exits.
    worker.stop();

    pollset_destroy(pollset_fd);
    for (retired_t::iterator it = retired.begin(); it != retired.end(); ++it)
        LIBZMQ_DELETE(*it);
}

zmq::pollset_t::handle_t zmq::pollset_t::add_fd(fd_t fd_, i_poll_events *events_)
{
    poll_entry_t *pe = new (std::nothrow) poll_entry_t;
    alloc_assert(pe);

    pe->fd           = fd_;
    pe->flag_pollin  = false;
    pe->flag_pollout = false;
    pe->events       = events_;

    struct poll_ctl pc;
    pc.fd     = fd_;
    pc.cmd    = PS_ADD;
    pc.events = 0;

    int rc = pollset_ctl(pollset_fd, &pc, 1);
    errno_assert(rc != -1);

    //  Increase the load metric of the thread.
    adjust_load(1);

    if (fd_ >= fd_table.size())
    {
        fd_table.resize(fd_ + 1, NULL);
    }
    fd_table[fd_] = pe;
    return pe;
}

void zmq::pollset_t::rm_fd(handle_t handle_)
{
    poll_entry_t *pe = (poll_entry_t *)handle_;

    struct poll_ctl pc;
    pc.fd     = pe->fd;
    pc.cmd    = PS_DELETE;
    pc.events = 0;
    pollset_ctl(pollset_fd, &pc, 1);

    fd_table[pe->fd] = NULL;

    pe->fd = retired_fd;
    retired.push_back(pe);

    //  Decrease the load metric of the thread.
    adjust_load(-1);
}

void zmq::pollset_t::set_pollin(handle_t handle_)
{
    poll_entry_t *pe = (poll_entry_t *)handle_;
    if (likely(!pe->flag_pollin))
    {
        struct poll_ctl pc;
        pc.fd     = pe->fd;
        pc.cmd    = PS_MOD;
        pc.events = POLLIN;

        const int rc = pollset_ctl(pollset_fd, &pc, 1);
        errno_assert(rc != -1);

        pe->flag_pollin = true;
    }
}

void zmq::pollset_t::reset_pollin(handle_t handle_)
{
    poll_entry_t *pe = (poll_entry_t *)handle_;
    if (unlikely(!pe->flag_pollin))
    {
        return;
    }

    struct poll_ctl pc;
    pc.fd     = pe->fd;
    pc.events = 0;

    pc.cmd = PS_DELETE;
    int rc = pollset_ctl(pollset_fd, &pc, 1);

    if (pe->flag_pollout)
    {
        pc.events = POLLOUT;
        pc.cmd    = PS_MOD;
        rc        = pollset_ctl(pollset_fd, &pc, 1);
        errno_assert(rc != -1);
    }

    pe->flag_pollin = false;
}

void zmq::pollset_t::set_pollout(handle_t handle_)
{
    poll_entry_t *pe = (poll_entry_t *)handle_;
    if (likely(!pe->flag_pollout))
    {
        struct poll_ctl pc;
        pc.fd     = pe->fd;
        pc.cmd    = PS_MOD;
        pc.events = POLLOUT;

        const int rc = pollset_ctl(pollset_fd, &pc, 1);
        errno_assert(rc != -1);

        pe->flag_pollout = true;
    }
}

void zmq::pollset_t::reset_pollout(handle_t handle_)
{
    poll_entry_t *pe = (poll_entry_t *)handle_;
    if (unlikely(!pe->flag_pollout))
    {
        return;
    }

    struct poll_ctl pc;
    pc.fd     = pe->fd;
    pc.events = 0;

    pc.cmd = PS_DELETE;
    int rc = pollset_ctl(pollset_fd, &pc, 1);
    errno_assert(rc != -1);

    if (pe->flag_pollin)
    {
        pc.cmd    = PS_MOD;
        pc.events = POLLIN;
        rc        = pollset_ctl(pollset_fd, &pc, 1);
        errno_assert(rc != -1);
    }
    pe->flag_pollout = false;
}

void zmq::pollset_t::start() { ctx.start_thread(worker, worker_routine, this); }

void zmq::pollset_t::stop() { stopping = true; }

int zmq::pollset_t::max_fds() { return -1; }

void zmq::pollset_t::loop()
{
    struct pollfd polldata_array[max_io_events];

    while (!stopping)
    {
        //  Execute any due timers.
        int timeout = (int)execute_timers();

        //  Wait for events.
        int n = pollset_poll(pollset_fd, polldata_array, max_io_events, timeout ? timeout : -1);
        if (n == -1)
        {
            errno_assert(errno == EINTR);
            continue;
        }

        for (int i = 0; i < n; i++)
        {
            poll_entry_t *pe = fd_table[polldata_array[i].fd];
            if (!pe)
                continue;

            if (pe->fd == retired_fd)
                continue;
            if (polldata_array[i].revents & (POLLERR | POLLHUP))
                pe->events->in_event();
            if (pe->fd == retired_fd)
                continue;
            if (polldata_array[i].revents & POLLOUT)
                pe->events->out_event();
            if (pe->fd == retired_fd)
                continue;
            if (polldata_array[i].revents & POLLIN)
                pe->events->in_event();
        }

        //  Destroy retired event sources.
        for (retired_t::iterator it = retired.begin(); it != retired.end(); ++it)
            LIBZMQ_DELETE(*it);
        retired.clear();
    }
}

void zmq::pollset_t::worker_routine(void *arg_) { ((pollset_t *)arg_)->loop(); }

#endif

//========= end of pollset.cpp ============

//========= begin of precompiled.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"

//========= end of precompiled.cpp ============

//========= begin of proxy.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"

#include <stddef.h>
// ans ignore: #include "poller.hpp"
// ans ignore: #include "proxy.hpp"
// ans ignore: #include "likely.hpp"
// ans ignore: #include "msg.hpp"

#if defined ZMQ_POLL_BASED_ON_POLL && !defined ZMQ_HAVE_WINDOWS && !defined ZMQ_HAVE_AIX
#include <poll.h>
#endif

// These headers end up pulling in zmq.h somewhere in their include
// dependency chain
// ans ignore: #include "socket_base.hpp"
// ans ignore: #include "err.hpp"

#ifdef ZMQ_HAVE_POLLER

// ans ignore: #include "socket_poller.hpp"

//  Macros for repetitive code.

//  PROXY_CLEANUP() must not be used before these variables are initialized.
#define PROXY_CLEANUP()                                                                                                \
    do                                                                                                                 \
    {                                                                                                                  \
        delete poller_all;                                                                                             \
        delete poller_in;                                                                                              \
        delete poller_control;                                                                                         \
        delete poller_receive_blocked;                                                                                 \
        delete poller_send_blocked;                                                                                    \
        delete poller_both_blocked;                                                                                    \
        delete poller_frontend_only;                                                                                   \
        delete poller_backend_only;                                                                                    \
    } while (false)

#define CHECK_RC_EXIT_ON_FAILURE()                                                                                     \
    do                                                                                                                 \
    {                                                                                                                  \
        if (rc < 0)                                                                                                    \
        {                                                                                                              \
            PROXY_CLEANUP();                                                                                           \
            return close_and_return(&msg, -1);                                                                         \
        }                                                                                                              \
    } while (false)

#endif //  ZMQ_HAVE_POLLER

// Control socket messages

typedef struct
{
    uint64_t msg_in;
    uint64_t bytes_in;
    uint64_t msg_out;
    uint64_t bytes_out;
} zmq_socket_stats_t;

// Utility functions

int capture(class zmq::socket_base_t *capture_, zmq::msg_t *msg_, int more_ = 0)
{
    //  Copy message to capture socket if any
    if (capture_)
    {
        zmq::msg_t ctrl;
        int        rc = ctrl.init();
        if (unlikely(rc < 0))
            return -1;
        rc = ctrl.copy(*msg_);
        if (unlikely(rc < 0))
            return -1;
        rc = capture_->send(&ctrl, more_ ? ZMQ_SNDMORE : 0);
        if (unlikely(rc < 0))
            return -1;
    }
    return 0;
}

int forward(class zmq::socket_base_t *from_, zmq_socket_stats_t *from_stats_, class zmq::socket_base_t *to_,
            zmq_socket_stats_t *to_stats_, class zmq::socket_base_t *capture_, zmq::msg_t *msg_)
{
    // Forward a burst of messages
    for (unsigned int i = 0; i < zmq::proxy_burst_size; i++)
    {
        int    more;
        size_t moresz;
        size_t complete_msg_size = 0;

        // Forward all the parts of one message
        while (true)
        {
            int rc = from_->recv(msg_, ZMQ_DONTWAIT);
            if (rc < 0)
            {
                if (likely(errno == EAGAIN && i > 0))
                    return 0; // End of burst
                else
                    return -1;
            }

            complete_msg_size += msg_->size();

            moresz = sizeof more;
            rc     = from_->getsockopt(ZMQ_RCVMORE, &more, &moresz);
            if (unlikely(rc < 0))
                return -1;

            //  Copy message to capture socket if any
            rc = capture(capture_, msg_, more);
            if (unlikely(rc < 0))
                return -1;

            rc = to_->send(msg_, more ? ZMQ_SNDMORE : 0);
            if (unlikely(rc < 0))
                return -1;

            if (more == 0)
                break;
        }

        // A multipart message counts as 1 packet:
        from_stats_->msg_in++;
        from_stats_->bytes_in += complete_msg_size;
        to_stats_->msg_out++;
        to_stats_->bytes_out += complete_msg_size;
    }

    return 0;
}

static int loop_and_send_multipart_stat(zmq::socket_base_t *control_, uint64_t stat_, bool first_, bool more_)
{
    int        rc;
    zmq::msg_t msg;

    //  VSM of 8 bytes can't fail to init
    msg.init_size(sizeof(uint64_t));
    memcpy(msg.data(), &stat_, sizeof(uint64_t));

    //  if the first message is handed to the pipe successfully then the HWM
    //  is not full, which means failures are due to interrupts (on Windows pipes
    //  are TCP sockets), so keep retrying
    do
    {
        rc = control_->send(&msg, more_ ? ZMQ_SNDMORE : 0);
    } while (!first_ && rc != 0 && errno == EAGAIN);

    return rc;
}

int reply_stats(class zmq::socket_base_t *control_, zmq_socket_stats_t *frontend_stats_,
                zmq_socket_stats_t *backend_stats_)
{
    // first part: frontend stats - the first send might fail due to HWM
    if (loop_and_send_multipart_stat(control_, frontend_stats_->msg_in, true, true) != 0)
        return -1;

    loop_and_send_multipart_stat(control_, frontend_stats_->bytes_in, false, true);
    loop_and_send_multipart_stat(control_, frontend_stats_->msg_out, false, true);
    loop_and_send_multipart_stat(control_, frontend_stats_->bytes_out, false, true);

    // second part: backend stats
    loop_and_send_multipart_stat(control_, backend_stats_->msg_in, false, true);
    loop_and_send_multipart_stat(control_, backend_stats_->bytes_in, false, true);
    loop_and_send_multipart_stat(control_, backend_stats_->msg_out, false, true);
    loop_and_send_multipart_stat(control_, backend_stats_->bytes_out, false, false);

    return 0;
}

#ifdef ZMQ_HAVE_POLLER

int zmq::proxy(class socket_base_t *frontend_, class socket_base_t *backend_, class socket_base_t *capture_,
               class socket_base_t *control_)
{
    msg_t msg;
    int   rc = msg.init();
    if (rc != 0)
        return -1;

    //  The algorithm below assumes ratio of requests and replies processed
    //  under full load to be 1:1.

    int    more;
    size_t moresz = sizeof(more);

    //  Proxy can be in these three states
    enum
    {
        active,
        paused,
        terminated
    } state = active;

    bool                          frontend_equal_to_backend;
    bool                          frontend_in  = false;
    bool                          frontend_out = false;
    bool                          backend_in   = false;
    bool                          backend_out  = false;
    bool                          control_in   = false;
    zmq::socket_poller_t::event_t events[3];
    zmq_socket_stats_t            frontend_stats;
    zmq_socket_stats_t            backend_stats;
    memset(&frontend_stats, 0, sizeof(frontend_stats));
    memset(&backend_stats, 0, sizeof(backend_stats));

    //  Don't allocate these pollers from stack because they will take more than 900 kB of stack!
    //  On Windows this blows up default stack of 1 MB and aborts the program.
    //  I wanted to use std::shared_ptr here as the best solution but that requires C++11...
    zmq::socket_poller_t *poller_all = new (std::nothrow) zmq::socket_poller_t; //  Poll for everything.
    zmq::socket_poller_t *poller_in  = new (std::nothrow)
        zmq::socket_poller_t; //  Poll only 'ZMQ_POLLIN' on all sockets. Initial blocking poll in loop.
    zmq::socket_poller_t *poller_control =
        new (std::nothrow) zmq::socket_poller_t; //  Poll only for 'ZMQ_POLLIN' on 'control_', when proxy is paused.
    zmq::socket_poller_t *poller_receive_blocked =
        new (std::nothrow) zmq::socket_poller_t; //  All except 'ZMQ_POLLIN' on 'frontend_'.

    //  If frontend_==backend_ 'poller_send_blocked' and 'poller_receive_blocked' are the same, 'ZMQ_POLLIN' is ignored.
    //  In that case 'poller_send_blocked' is not used. We need only 'poller_receive_blocked'.
    //  We also don't need 'poller_both_blocked', 'poller_backend_only' nor 'poller_frontend_only' no need to initialize
    //  it. We save some RAM and time for initialization.
    zmq::socket_poller_t *poller_send_blocked  = NULL; //  All except 'ZMQ_POLLIN' on 'backend_'.
    zmq::socket_poller_t *poller_both_blocked  = NULL; //  All except 'ZMQ_POLLIN' on both 'frontend_' and 'backend_'.
    zmq::socket_poller_t *poller_frontend_only = NULL; //  Only 'ZMQ_POLLIN' and 'ZMQ_POLLOUT' on 'frontend_'.
    zmq::socket_poller_t *poller_backend_only  = NULL; //  Only 'ZMQ_POLLIN' and 'ZMQ_POLLOUT' on 'backend_'.

    if (frontend_ != backend_)
    {
        poller_send_blocked = new (std::nothrow) zmq::socket_poller_t; //  All except 'ZMQ_POLLIN' on 'backend_'.
        poller_both_blocked =
            new (std::nothrow) zmq::socket_poller_t; //  All except 'ZMQ_POLLIN' on both 'frontend_' and 'backend_'.
        poller_frontend_only =
            new (std::nothrow) zmq::socket_poller_t; //  Only 'ZMQ_POLLIN' and 'ZMQ_POLLOUT' on 'frontend_'.
        poller_backend_only =
            new (std::nothrow) zmq::socket_poller_t; //  Only 'ZMQ_POLLIN' and 'ZMQ_POLLOUT' on 'backend_'.
        frontend_equal_to_backend = false;
    }
    else
        frontend_equal_to_backend = true;

    if (poller_all == NULL || poller_in == NULL || poller_control == NULL || poller_receive_blocked == NULL ||
        ((poller_send_blocked == NULL || poller_both_blocked == NULL) && !frontend_equal_to_backend))
    {
        PROXY_CLEANUP();
        return close_and_return(&msg, -1);
    }

    zmq::socket_poller_t *poller_wait = poller_in; //  Poller for blocking wait, initially all 'ZMQ_POLLIN'.

    //  Register 'frontend_' and 'backend_' with pollers.
    rc = poller_all->add(frontend_, NULL,
                         ZMQ_POLLIN | ZMQ_POLLOUT); //  Everything.
    CHECK_RC_EXIT_ON_FAILURE();
    rc = poller_in->add(frontend_, NULL, ZMQ_POLLIN); //  All 'ZMQ_POLLIN's.
    CHECK_RC_EXIT_ON_FAILURE();

    if (frontend_equal_to_backend)
    {
        //  If frontend_==backend_ 'poller_send_blocked' and 'poller_receive_blocked' are the same,
        //  so we don't need 'poller_send_blocked'. We need only 'poller_receive_blocked'.
        //  We also don't need 'poller_both_blocked', no need to initialize it.
        rc = poller_receive_blocked->add(frontend_, NULL, ZMQ_POLLOUT);
        CHECK_RC_EXIT_ON_FAILURE();
    }
    else
    {
        rc = poller_all->add(backend_, NULL,
                             ZMQ_POLLIN | ZMQ_POLLOUT); //  Everything.
        CHECK_RC_EXIT_ON_FAILURE();
        rc = poller_in->add(backend_, NULL, ZMQ_POLLIN); //  All 'ZMQ_POLLIN's.
        CHECK_RC_EXIT_ON_FAILURE();
        rc = poller_both_blocked->add(frontend_, NULL, ZMQ_POLLOUT); //  Waiting only for 'ZMQ_POLLOUT'.
        CHECK_RC_EXIT_ON_FAILURE();
        rc = poller_both_blocked->add(backend_, NULL, ZMQ_POLLOUT); //  Waiting only for 'ZMQ_POLLOUT'.
        CHECK_RC_EXIT_ON_FAILURE();
        rc = poller_send_blocked->add(backend_, NULL,
                                      ZMQ_POLLOUT); //  All except 'ZMQ_POLLIN' on 'backend_'.
        CHECK_RC_EXIT_ON_FAILURE();
        rc = poller_send_blocked->add(frontend_, NULL,
                                      ZMQ_POLLIN | ZMQ_POLLOUT); //  All except 'ZMQ_POLLIN' on 'backend_'.
        CHECK_RC_EXIT_ON_FAILURE();
        rc = poller_receive_blocked->add(frontend_, NULL,
                                         ZMQ_POLLOUT); //  All except 'ZMQ_POLLIN' on 'frontend_'.
        CHECK_RC_EXIT_ON_FAILURE();
        rc = poller_receive_blocked->add(backend_, NULL,
                                         ZMQ_POLLIN | ZMQ_POLLOUT); //  All except 'ZMQ_POLLIN' on 'frontend_'.
        CHECK_RC_EXIT_ON_FAILURE();
        rc = poller_frontend_only->add(frontend_, NULL, ZMQ_POLLIN | ZMQ_POLLOUT);
        CHECK_RC_EXIT_ON_FAILURE();
        rc = poller_backend_only->add(backend_, NULL, ZMQ_POLLIN | ZMQ_POLLOUT);
        CHECK_RC_EXIT_ON_FAILURE();
    }

    //  Register 'control_' with pollers.
    if (control_ != NULL)
    {
        rc = poller_all->add(control_, NULL, ZMQ_POLLIN);
        CHECK_RC_EXIT_ON_FAILURE();
        rc = poller_in->add(control_, NULL, ZMQ_POLLIN);
        CHECK_RC_EXIT_ON_FAILURE();
        rc = poller_control->add(control_, NULL,
                                 ZMQ_POLLIN); //  When proxy is paused we wait only for ZMQ_POLLIN on 'control_' socket.
        CHECK_RC_EXIT_ON_FAILURE();
        rc = poller_receive_blocked->add(control_, NULL, ZMQ_POLLIN);
        CHECK_RC_EXIT_ON_FAILURE();
        if (!frontend_equal_to_backend)
        {
            rc = poller_send_blocked->add(control_, NULL, ZMQ_POLLIN);
            CHECK_RC_EXIT_ON_FAILURE();
            rc = poller_both_blocked->add(control_, NULL, ZMQ_POLLIN);
            CHECK_RC_EXIT_ON_FAILURE();
            rc = poller_frontend_only->add(control_, NULL, ZMQ_POLLIN);
            CHECK_RC_EXIT_ON_FAILURE();
            rc = poller_backend_only->add(control_, NULL, ZMQ_POLLIN);
            CHECK_RC_EXIT_ON_FAILURE();
        }
    }

    bool request_processed, reply_processed;

    while (state != terminated)
    {
        //  Blocking wait initially only for 'ZMQ_POLLIN' - 'poller_wait' points to 'poller_in'.
        //  If one of receiving end's queue is full ('ZMQ_POLLOUT' not available),
        //  'poller_wait' is pointed to 'poller_receive_blocked', 'poller_send_blocked' or 'poller_both_blocked'.
        rc = poller_wait->wait(events, 3, -1);
        if (rc < 0 && errno == EAGAIN)
            rc = 0;
        CHECK_RC_EXIT_ON_FAILURE();

        //  Some of events waited for by 'poller_wait' have arrived, now poll for everything without blocking.
        rc = poller_all->wait(events, 3, 0);
        if (rc < 0 && errno == EAGAIN)
            rc = 0;
        CHECK_RC_EXIT_ON_FAILURE();

        //  Process events.
        for (int i = 0; i < rc; i++)
        {
            if (events[i].socket == frontend_)
            {
                frontend_in  = (events[i].events & ZMQ_POLLIN) != 0;
                frontend_out = (events[i].events & ZMQ_POLLOUT) != 0;
            }
            else
                //  This 'if' needs to be after check for 'frontend_' in order never
                //  to be reached in case frontend_==backend_, so we ensure backend_in=false in that case.
                if (events[i].socket == backend_)
            {
                backend_in  = (events[i].events & ZMQ_POLLIN) != 0;
                backend_out = (events[i].events & ZMQ_POLLOUT) != 0;
            }
            else if (events[i].socket == control_)
                control_in = (events[i].events & ZMQ_POLLIN) != 0;
        }

        //  Process a control command if any.
        if (control_in)
        {
            rc = control_->recv(&msg, 0);
            CHECK_RC_EXIT_ON_FAILURE();
            rc = control_->getsockopt(ZMQ_RCVMORE, &more, &moresz);
            if (unlikely(rc < 0) || more)
            {
                PROXY_CLEANUP();
                return close_and_return(&msg, -1);
            }

            //  Copy message to capture socket if any.
            rc = capture(capture_, &msg);
            CHECK_RC_EXIT_ON_FAILURE();

            if (msg.size() == 5 && memcmp(msg.data(), "PAUSE", 5) == 0)
            {
                state       = paused;
                poller_wait = poller_control;
            }
            else if (msg.size() == 6 && memcmp(msg.data(), "RESUME", 6) == 0)
            {
                state       = active;
                poller_wait = poller_in;
            }
            else
            {
                if (msg.size() == 9 && memcmp(msg.data(), "TERMINATE", 9) == 0)
                    state = terminated;
                else
                {
                    if (msg.size() == 10 && memcmp(msg.data(), "STATISTICS", 10) == 0)
                    {
                        rc = reply_stats(control_, &frontend_stats, &backend_stats);
                        CHECK_RC_EXIT_ON_FAILURE();
                    }
                    else
                    {
                        //  This is an API error, we assert
                        puts("E: invalid command sent to proxy");
                        zmq_assert(false);
                    }
                }
            }
            control_in = false;
        }

        if (state == active)
        {
            //  Process a request, 'ZMQ_POLLIN' on 'frontend_' and 'ZMQ_POLLOUT' on 'backend_'.
            //  In case of frontend_==backend_ there's no 'ZMQ_POLLOUT' event.
            if (frontend_in && (backend_out || frontend_equal_to_backend))
            {
                rc = forward(frontend_, &frontend_stats, backend_, &backend_stats, capture_, &msg);
                CHECK_RC_EXIT_ON_FAILURE();
                request_processed = true;
                frontend_in = backend_out = false;
            }
            else
                request_processed = false;

            //  Process a reply, 'ZMQ_POLLIN' on 'backend_' and 'ZMQ_POLLOUT' on 'frontend_'.
            //  If 'frontend_' and 'backend_' are the same this is not needed because previous processing
            //  covers all of the cases. 'backend_in' is always false if frontend_==backend_ due to
            //  design in 'for' event processing loop.
            if (backend_in && frontend_out)
            {
                rc = forward(backend_, &backend_stats, frontend_, &frontend_stats, capture_, &msg);
                CHECK_RC_EXIT_ON_FAILURE();
                reply_processed = true;
                backend_in = frontend_out = false;
            }
            else
                reply_processed = false;

            if (request_processed || reply_processed)
            {
                //  If request/reply is processed that means we had at least one 'ZMQ_POLLOUT' event.
                //  Enable corresponding 'ZMQ_POLLIN' for blocking wait if any was disabled.
                if (poller_wait != poller_in)
                {
                    if (request_processed)
                    { //  'frontend_' -> 'backend_'
                        if (poller_wait == poller_both_blocked)
                            poller_wait = poller_send_blocked;
                        else if (poller_wait == poller_receive_blocked || poller_wait == poller_frontend_only)
                            poller_wait = poller_in;
                    }
                    if (reply_processed)
                    { //  'backend_' -> 'frontend_'
                        if (poller_wait == poller_both_blocked)
                            poller_wait = poller_receive_blocked;
                        else if (poller_wait == poller_send_blocked || poller_wait == poller_backend_only)
                            poller_wait = poller_in;
                    }
                }
            }
            else
            {
                //  No requests have been processed, there were no 'ZMQ_POLLIN' with corresponding 'ZMQ_POLLOUT' events.
                //  That means that out queue(s) is/are full or one out queue is full and second one has no messages to
                //  process. Disable receiving 'ZMQ_POLLIN' for sockets for which there's no 'ZMQ_POLLOUT', or wait only
                //  on both 'backend_''s or 'frontend_''s 'ZMQ_POLLIN' and 'ZMQ_POLLOUT'.
                if (frontend_in)
                {
                    if (frontend_out)
                        // If frontend_in and frontend_out are true, obviously backend_in and backend_out are both
                        // false. In that case we need to wait for both 'ZMQ_POLLIN' and 'ZMQ_POLLOUT' only on
                        // 'backend_'. We'll never get here in case of frontend_==backend_ because then frontend_out
                        // will always be false.
                        poller_wait = poller_backend_only;
                    else
                    {
                        if (poller_wait == poller_send_blocked)
                            poller_wait = poller_both_blocked;
                        else if (poller_wait == poller_in)
                            poller_wait = poller_receive_blocked;
                    }
                }
                if (backend_in)
                {
                    //  Will never be reached if frontend_==backend_, 'backend_in' will
                    //  always be false due to design in 'for' event processing loop.
                    if (backend_out)
                        // If backend_in and backend_out are true, obviously frontend_in and frontend_out are both
                        // false. In that case we need to wait for both 'ZMQ_POLLIN' and 'ZMQ_POLLOUT' only on
                        // 'frontend_'.
                        poller_wait = poller_frontend_only;
                    else
                    {
                        if (poller_wait == poller_receive_blocked)
                            poller_wait = poller_both_blocked;
                        else if (poller_wait == poller_in)
                            poller_wait = poller_send_blocked;
                    }
                }
            }
        }
    }
    PROXY_CLEANUP();
    return close_and_return(&msg, 0);
}

#else //  ZMQ_HAVE_POLLER

int zmq::proxy(class socket_base_t *frontend_, class socket_base_t *backend_, class socket_base_t *capture_,
               class socket_base_t *control_)
{
    msg_t msg;
    int   rc = msg.init();
    if (rc != 0)
        return -1;

    //  The algorithm below assumes ratio of requests and replies processed
    //  under full load to be 1:1.

    int            more;
    size_t         moresz;
    zmq_pollitem_t items[] = {
        {frontend_, 0, ZMQ_POLLIN, 0}, {backend_, 0, ZMQ_POLLIN, 0}, {control_, 0, ZMQ_POLLIN, 0}};
    int            qt_poll_items = (control_ ? 3 : 2);
    zmq_pollitem_t itemsout[]    = {{frontend_, 0, ZMQ_POLLOUT, 0}, {backend_, 0, ZMQ_POLLOUT, 0}};

    zmq_socket_stats_t frontend_stats;
    memset(&frontend_stats, 0, sizeof(frontend_stats));
    zmq_socket_stats_t backend_stats;
    memset(&backend_stats, 0, sizeof(backend_stats));

    //  Proxy can be in these three states
    enum
    {
        active,
        paused,
        terminated
    } state = active;

    while (state != terminated)
    {
        //  Wait while there are either requests or replies to process.
        rc = zmq_poll(&items[0], qt_poll_items, -1);
        if (unlikely(rc < 0))
            return close_and_return(&msg, -1);

        //  Get the pollout separately because when combining this with pollin it maxes the CPU
        //  because pollout shall most of the time return directly.
        //  POLLOUT is only checked when frontend and backend sockets are not the same.
        if (frontend_ != backend_)
        {
            rc = zmq_poll(&itemsout[0], 2, 0);
            if (unlikely(rc < 0))
            {
                return close_and_return(&msg, -1);
            }
        }

        //  Process a control command if any
        if (control_ && items[2].revents & ZMQ_POLLIN)
        {
            rc = control_->recv(&msg, 0);
            if (unlikely(rc < 0))
                return close_and_return(&msg, -1);

            moresz = sizeof more;
            rc     = control_->getsockopt(ZMQ_RCVMORE, &more, &moresz);
            if (unlikely(rc < 0) || more)
                return close_and_return(&msg, -1);

            //  Copy message to capture socket if any
            rc = capture(capture_, &msg);
            if (unlikely(rc < 0))
                return close_and_return(&msg, -1);

            if (msg.size() == 5 && memcmp(msg.data(), "PAUSE", 5) == 0)
                state = paused;
            else if (msg.size() == 6 && memcmp(msg.data(), "RESUME", 6) == 0)
                state = active;
            else if (msg.size() == 9 && memcmp(msg.data(), "TERMINATE", 9) == 0)
                state = terminated;
            else
            {
                if (msg.size() == 10 && memcmp(msg.data(), "STATISTICS", 10) == 0)
                {
                    rc = reply_stats(control_, &frontend_stats, &backend_stats);
                    if (unlikely(rc < 0))
                        return close_and_return(&msg, -1);
                }
                else
                {
                    //  This is an API error, we assert
                    puts("E: invalid command sent to proxy");
                    zmq_assert(false);
                }
            }
        }
        //  Process a request
        if (state == active && items[0].revents & ZMQ_POLLIN &&
            (frontend_ == backend_ || itemsout[1].revents & ZMQ_POLLOUT))
        {
            rc = forward(frontend_, &frontend_stats, backend_, &backend_stats, capture_, &msg);
            if (unlikely(rc < 0))
                return close_and_return(&msg, -1);
        }
        //  Process a reply
        if (state == active && frontend_ != backend_ && items[1].revents & ZMQ_POLLIN &&
            itemsout[0].revents & ZMQ_POLLOUT)
        {
            rc = forward(backend_, &backend_stats, frontend_, &frontend_stats, capture_, &msg);
            if (unlikely(rc < 0))
                return close_and_return(&msg, -1);
        }
    }

    return close_and_return(&msg, 0);
}

#endif //  ZMQ_HAVE_POLLER

//========= end of proxy.cpp ============

//========= begin of pub.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "pub.hpp"
// ans ignore: #include "pipe.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "msg.hpp"

zmq::pub_t::pub_t(class ctx_t *parent_, uint32_t tid_, int sid_) : xpub_t(parent_, tid_, sid_)
{
    options.type = ZMQ_PUB;
}

zmq::pub_t::~pub_t() {}

void zmq::pub_t::xattach_pipe(pipe_t *pipe_, bool subscribe_to_all_, bool locally_initiated_)
{
    zmq_assert(pipe_);

    //  Don't delay pipe termination as there is no one
    //  to receive the delimiter.
    pipe_->set_nodelay();

    xpub_t::xattach_pipe(pipe_, subscribe_to_all_, locally_initiated_);
}

int zmq::pub_t::xrecv(class msg_t *)
{
    //  Messages cannot be received from PUB socket.
    errno = ENOTSUP;
    return -1;
}

bool zmq::pub_t::xhas_in() { return false; }

//========= end of pub.cpp ============

//========= begin of pull.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "pull.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "msg.hpp"
// ans ignore: #include "pipe.hpp"

zmq::pull_t::pull_t(class ctx_t *parent_, uint32_t tid_, int sid_) : socket_base_t(parent_, tid_, sid_)
{
    options.type = ZMQ_PULL;
}

zmq::pull_t::~pull_t() {}

void zmq::pull_t::xattach_pipe(pipe_t *pipe_, bool subscribe_to_all_, bool locally_initiated_)
{
    LIBZMQ_UNUSED(subscribe_to_all_);
    LIBZMQ_UNUSED(locally_initiated_);

    zmq_assert(pipe_);
    _fq.attach(pipe_);
}

void zmq::pull_t::xread_activated(pipe_t *pipe_) { _fq.activated(pipe_); }

void zmq::pull_t::xpipe_terminated(pipe_t *pipe_) { _fq.pipe_terminated(pipe_); }

int zmq::pull_t::xrecv(msg_t *msg_) { return _fq.recv(msg_); }

bool zmq::pull_t::xhas_in() { return _fq.has_in(); }

//========= end of pull.cpp ============

//========= begin of push.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "push.hpp"
// ans ignore: #include "pipe.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "msg.hpp"

zmq::push_t::push_t(class ctx_t *parent_, uint32_t tid_, int sid_) : socket_base_t(parent_, tid_, sid_)
{
    options.type = ZMQ_PUSH;
}

zmq::push_t::~push_t() {}

void zmq::push_t::xattach_pipe(pipe_t *pipe_, bool subscribe_to_all_, bool locally_initiated_)
{
    LIBZMQ_UNUSED(subscribe_to_all_);
    LIBZMQ_UNUSED(locally_initiated_);

    //  Don't delay pipe termination as there is no one
    //  to receive the delimiter.
    pipe_->set_nodelay();

    zmq_assert(pipe_);
    _lb.attach(pipe_);
}

void zmq::push_t::xwrite_activated(pipe_t *pipe_) { _lb.activated(pipe_); }

void zmq::push_t::xpipe_terminated(pipe_t *pipe_) { _lb.pipe_terminated(pipe_); }

int zmq::push_t::xsend(msg_t *msg_) { return _lb.send(msg_); }

bool zmq::push_t::xhas_out() { return _lb.has_out(); }

//========= end of push.cpp ============

//========= begin of radio.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <string.h>

// ans ignore: #include "radio.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "pipe.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "msg.hpp"

zmq::radio_t::radio_t(class ctx_t *parent_, uint32_t tid_, int sid_)
    : socket_base_t(parent_, tid_, sid_, true), _lossy(true)
{
    options.type = ZMQ_RADIO;
}

zmq::radio_t::~radio_t() {}

void zmq::radio_t::xattach_pipe(pipe_t *pipe_, bool subscribe_to_all_, bool locally_initiated_)
{
    LIBZMQ_UNUSED(subscribe_to_all_);
    LIBZMQ_UNUSED(locally_initiated_);

    zmq_assert(pipe_);

    //  Don't delay pipe termination as there is no one
    //  to receive the delimiter.
    pipe_->set_nodelay();

    _dist.attach(pipe_);

    if (subscribe_to_all_)
        _udp_pipes.push_back(pipe_);
    //  The pipe is active when attached. Let's read the subscriptions from
    //  it, if any.
    else
        xread_activated(pipe_);
}

void zmq::radio_t::xread_activated(pipe_t *pipe_)
{
    //  There are some subscriptions waiting. Let's process them.
    msg_t msg;
    while (pipe_->read(&msg))
    {
        //  Apply the subscription to the trie
        if (msg.is_join() || msg.is_leave())
        {
            std::string group = std::string(msg.group());

            if (msg.is_join())
                _subscriptions.ZMQ_MAP_INSERT_OR_EMPLACE(ZMQ_MOVE(group), pipe_);
            else
            {
                std::pair<subscriptions_t::iterator, subscriptions_t::iterator> range =
                    _subscriptions.equal_range(group);

                for (subscriptions_t::iterator it = range.first; it != range.second; ++it)
                {
                    if (it->second == pipe_)
                    {
                        _subscriptions.erase(it);
                        break;
                    }
                }
            }
        }
        msg.close();
    }
}

void zmq::radio_t::xwrite_activated(pipe_t *pipe_) { _dist.activated(pipe_); }
int  zmq::radio_t::xsetsockopt(int option_, const void *optval_, size_t optvallen_)
{
    if (optvallen_ != sizeof(int) || *static_cast<const int *>(optval_) < 0)
    {
        errno = EINVAL;
        return -1;
    }
    if (option_ == ZMQ_XPUB_NODROP)
        _lossy = (*static_cast<const int *>(optval_) == 0);
    else
    {
        errno = EINVAL;
        return -1;
    }
    return 0;
}

void zmq::radio_t::xpipe_terminated(pipe_t *pipe_)
{
    for (subscriptions_t::iterator it = _subscriptions.begin(), end = _subscriptions.end(); it != end;)
    {
        if (it->second == pipe_)
        {
#if __cplusplus >= 201103L
            it = _subscriptions.erase(it);
#else
            _subscriptions.erase(it++);
#endif
        }
        else
        {
            ++it;
        }
    }

    {
        const udp_pipes_t::iterator end = _udp_pipes.end();
        const udp_pipes_t::iterator it  = std::find(_udp_pipes.begin(), end, pipe_);
        if (it != end)
            _udp_pipes.erase(it);
    }

    _dist.pipe_terminated(pipe_);
}

int zmq::radio_t::xsend(msg_t *msg_)
{
    //  Radio sockets do not allow multipart data (ZMQ_SNDMORE)
    if (msg_->flags() & msg_t::more)
    {
        errno = EINVAL;
        return -1;
    }

    _dist.unmatch();

    std::pair<subscriptions_t::iterator, subscriptions_t::iterator> range =
        _subscriptions.equal_range(std::string(msg_->group()));

    for (subscriptions_t::iterator it = range.first; it != range.second; ++it)
        _dist.match(it->second);

    for (udp_pipes_t::iterator it = _udp_pipes.begin(), end = _udp_pipes.end(); it != end; ++it)
        _dist.match(*it);

    int rc = -1;
    if (_lossy || _dist.check_hwm())
    {
        if (_dist.send_to_matching(msg_) == 0)
        {
            rc = 0; //  Yay, sent successfully
        }
    }
    else
        errno = EAGAIN;

    return rc;
}

bool zmq::radio_t::xhas_out() { return _dist.has_out(); }

int zmq::radio_t::xrecv(msg_t *msg_)
{
    //  Messages cannot be received from PUB socket.
    LIBZMQ_UNUSED(msg_);
    errno = ENOTSUP;
    return -1;
}

bool zmq::radio_t::xhas_in() { return false; }

zmq::radio_session_t::radio_session_t(io_thread_t *io_thread_, bool connect_, socket_base_t *socket_,
                                      const options_t &options_, address_t *addr_)
    : session_base_t(io_thread_, connect_, socket_, options_, addr_), _state(group)
{
}

zmq::radio_session_t::~radio_session_t() {}

int zmq::radio_session_t::push_msg(msg_t *msg_)
{
    if (msg_->flags() & msg_t::command)
    {
        char *       command_data = static_cast<char *>(msg_->data());
        const size_t data_size    = msg_->size();

        int   group_length;
        char *group;

        msg_t join_leave_msg;
        int   rc;

        //  Set the msg type to either JOIN or LEAVE
        if (data_size >= 5 && memcmp(command_data, "\4JOIN", 5) == 0)
        {
            group_length = static_cast<int>(data_size) - 5;
            group        = command_data + 5;
            rc           = join_leave_msg.init_join();
        }
        else if (data_size >= 6 && memcmp(command_data, "\5LEAVE", 6) == 0)
        {
            group_length = static_cast<int>(data_size) - 6;
            group        = command_data + 6;
            rc           = join_leave_msg.init_leave();
        }
        //  If it is not a JOIN or LEAVE just push the message
        else
            return session_base_t::push_msg(msg_);

        errno_assert(rc == 0);

        //  Set the group
        rc = join_leave_msg.set_group(group, group_length);
        errno_assert(rc == 0);

        //  Close the current command
        rc = msg_->close();
        errno_assert(rc == 0);

        //  Push the join or leave command
        *msg_ = join_leave_msg;
        return session_base_t::push_msg(msg_);
    }
    return session_base_t::push_msg(msg_);
}

int zmq::radio_session_t::pull_msg(msg_t *msg_)
{
    if (_state == group)
    {
        int rc = session_base_t::pull_msg(&_pending_msg);
        if (rc != 0)
            return rc;

        const char *group  = _pending_msg.group();
        int         length = static_cast<int>(strlen(group));

        //  First frame is the group
        rc = msg_->init_size(length);
        errno_assert(rc == 0);
        msg_->set_flags(msg_t::more);
        memcpy(msg_->data(), group, length);

        //  Next status is the body
        _state = body;
        return 0;
    }
    *msg_  = _pending_msg;
    _state = group;
    return 0;
}

void zmq::radio_session_t::reset()
{
    session_base_t::reset();
    _state = group;
}

//========= end of radio.cpp ============

//========= begin of radix_tree.cpp ============

/*
    Copyright (c) 2018 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "radix_tree.hpp"

#include <stdlib.h>
#include <string.h>
#include <vector>

node_t::node_t(unsigned char *data_) : _data(data_) {}

uint32_t node_t::refcount()
{
    uint32_t u32;
    memcpy(&u32, _data, sizeof(u32));
    return u32;
}

void node_t::set_refcount(uint32_t value_) { memcpy(_data, &value_, sizeof(value_)); }

uint32_t node_t::prefix_length()
{
    uint32_t u32;
    memcpy(&u32, _data + sizeof(uint32_t), sizeof(u32));
    return u32;
}

void node_t::set_prefix_length(uint32_t value_) { memcpy(_data + sizeof(value_), &value_, sizeof(value_)); }

uint32_t node_t::edgecount()
{
    uint32_t u32;
    memcpy(&u32, _data + 2 * sizeof(uint32_t), sizeof(u32));
    return u32;
}

void node_t::set_edgecount(uint32_t value_) { memcpy(_data + 2 * sizeof(value_), &value_, sizeof(value_)); }

unsigned char *node_t::prefix() { return _data + 3 * sizeof(uint32_t); }

void node_t::set_prefix(const unsigned char *bytes_) { memcpy(prefix(), bytes_, prefix_length()); }

unsigned char *node_t::first_bytes() { return prefix() + prefix_length(); }

void node_t::set_first_bytes(const unsigned char *bytes_) { memcpy(first_bytes(), bytes_, edgecount()); }

unsigned char node_t::first_byte_at(size_t index_)
{
    zmq_assert(index_ < edgecount());
    return first_bytes()[index_];
}

void node_t::set_first_byte_at(size_t index_, unsigned char byte_)
{
    zmq_assert(index_ < edgecount());
    first_bytes()[index_] = byte_;
}

unsigned char *node_t::node_pointers() { return prefix() + prefix_length() + edgecount(); }

void node_t::set_node_pointers(const unsigned char *pointers_)
{
    memcpy(node_pointers(), pointers_, edgecount() * sizeof(void *));
}

node_t node_t::node_at(size_t index_)
{
    zmq_assert(index_ < edgecount());

    unsigned char *data;
    memcpy(&data, node_pointers() + index_ * sizeof(void *), sizeof(data));
    return node_t(data);
}

void node_t::set_node_at(size_t index_, node_t node_)
{
    zmq_assert(index_ < edgecount());
    memcpy(node_pointers() + index_ * sizeof(void *), &node_._data, sizeof(node_._data));
}

void node_t::set_edge_at(size_t index_, unsigned char first_byte_, node_t node_)
{
    set_first_byte_at(index_, first_byte_);
    set_node_at(index_, node_);
}

bool node_t::operator==(node_t other_) const { return _data == other_._data; }

bool node_t::operator!=(node_t other_) const { return !(*this == other_); }

void node_t::resize(size_t prefix_length_, size_t edgecount_)
{
    size_t         node_size = 3 * sizeof(uint32_t) + prefix_length_ + edgecount_ * (1 + sizeof(void *));
    unsigned char *new_data  = static_cast<unsigned char *>(realloc(_data, node_size));
    zmq_assert(new_data);
    _data = new_data;
    set_prefix_length(static_cast<uint32_t>(prefix_length_));
    set_edgecount(static_cast<uint32_t>(edgecount_));
}

node_t make_node(size_t refcount_, size_t prefix_length_, size_t edgecount_)
{
    size_t node_size = 3 * sizeof(uint32_t) + prefix_length_ + edgecount_ * (1 + sizeof(void *));

    unsigned char *data = static_cast<unsigned char *>(malloc(node_size));
    zmq_assert(data);

    node_t node(data);
    node.set_refcount(static_cast<uint32_t>(refcount_));
    node.set_prefix_length(static_cast<uint32_t>(prefix_length_));
    node.set_edgecount(static_cast<uint32_t>(edgecount_));
    return node;
}

// ----------------------------------------------------------------------

zmq::radix_tree_t::radix_tree_t() : _root(make_node(0, 0, 0)), _size(0) {}

static void free_nodes(node_t node_)
{
    for (size_t i = 0; i < node_.edgecount(); ++i)
        free_nodes(node_.node_at(i));
    free(node_._data);
}

zmq::radix_tree_t::~radix_tree_t() { free_nodes(_root); }

match_result_t::match_result_t(size_t key_bytes_matched_, size_t prefix_bytes_matched_, size_t edge_index_,
                               size_t parent_edge_index_, node_t current_, node_t parent_, node_t grandparent_)
    : _key_bytes_matched(key_bytes_matched_), _prefix_bytes_matched(prefix_bytes_matched_), _edge_index(edge_index_),
      _parent_edge_index(parent_edge_index_), _current_node(current_), _parent_node(parent_),
      _grandparent_node(grandparent_)
{
}

match_result_t zmq::radix_tree_t::match(const unsigned char *key_, size_t key_size_, bool is_lookup_ = false) const
{
    zmq_assert(key_);

    // Node we're currently at in the traversal and its predecessors.
    node_t current_node     = _root;
    node_t parent_node      = current_node;
    node_t grandparent_node = current_node;
    // Index of the next byte to match in the key.
    size_t key_byte_index = 0;
    // Index of the next byte to match in the current node's prefix.
    size_t prefix_byte_index = 0;
    // Index of the edge from parent to current node.
    size_t edge_index = 0;
    // Index of the edge from grandparent to parent.
    size_t parent_edge_index = 0;

    while (current_node.prefix_length() > 0 || current_node.edgecount() > 0)
    {
        for (prefix_byte_index = 0; prefix_byte_index < current_node.prefix_length() && key_byte_index < key_size_;
             ++prefix_byte_index, ++key_byte_index)
        {
            if (current_node.prefix()[prefix_byte_index] != key_[key_byte_index])
                break;
        }

        // Even if a prefix of the key matches and we're doing a
        // lookup, this means we've found a matching subscription.
        if (is_lookup_ && prefix_byte_index == current_node.prefix_length() && current_node.refcount() > 0)
        {
            key_byte_index = key_size_;
            break;
        }

        // There was a mismatch or we've matched the whole key, so
        // there's nothing more to do.
        if (prefix_byte_index != current_node.prefix_length() || key_byte_index == key_size_)
            break;

        // We need to match the rest of the key. Check if there's an
        // outgoing edge from this node.
        node_t next_node = current_node;
        for (size_t i = 0; i < current_node.edgecount(); ++i)
        {
            if (current_node.first_byte_at(i) == key_[key_byte_index])
            {
                parent_edge_index = edge_index;
                edge_index        = i;
                next_node         = current_node.node_at(i);
                break;
            }
        }

        if (next_node == current_node)
            break; // No outgoing edge.
        grandparent_node = parent_node;
        parent_node      = current_node;
        current_node     = next_node;
    }

    return match_result_t(key_byte_index, prefix_byte_index, edge_index, parent_edge_index, current_node, parent_node,
                          grandparent_node);
}

bool zmq::radix_tree_t::add(const unsigned char *key_, size_t key_size_)
{
    match_result_t match_result         = match(key_, key_size_);
    size_t         key_bytes_matched    = match_result._key_bytes_matched;
    size_t         prefix_bytes_matched = match_result._prefix_bytes_matched;
    size_t         edge_index           = match_result._edge_index;
    node_t         current_node         = match_result._current_node;
    node_t         parent_node          = match_result._parent_node;

    if (key_bytes_matched != key_size_)
    {
        // Not all characters match, we might have to split the node.
        if (key_bytes_matched == 0 || prefix_bytes_matched == current_node.prefix_length())
        {
            // The mismatch is at one of the outgoing edges, so we
            // create an edge from the current node to a new leaf node
            // that has the rest of the key as the prefix.
            node_t key_node = make_node(1, key_size_ - key_bytes_matched, 0);
            key_node.set_prefix(key_ + key_bytes_matched);

            // Reallocate for one more edge.
            current_node.resize(current_node.prefix_length(), current_node.edgecount() + 1);

            // Make room for the new edge. We need to shift the chunk
            // of node pointers one byte to the right. Since resize()
            // increments the edgecount by 1, node_pointers() tells us the
            // destination address. The chunk of node pointers starts
            // at one byte to the left of this destination.
            //
            // Since the regions can overlap, we use memmove.
            memmove(current_node.node_pointers(), current_node.node_pointers() - 1,
                    (current_node.edgecount() - 1) * sizeof(void *));

            // Add an edge to the new node.
            current_node.set_edge_at(current_node.edgecount() - 1, key_[key_bytes_matched], key_node);

            // We need to update all pointers to the current node
            // after the call to resize().
            if (current_node.prefix_length() == 0)
                _root._data = current_node._data;
            else
                parent_node.set_node_at(edge_index, current_node);
            ++_size;
            return true;
        }

        // There was a mismatch, so we need to split this node.
        //
        // Create two nodes that will be reachable from the parent.
        // One node will have the rest of the characters from the key,
        // and the other node will have the rest of the characters
        // from the current node's prefix.
        node_t key_node   = make_node(1, key_size_ - key_bytes_matched, 0);
        node_t split_node = make_node(current_node.refcount(), current_node.prefix_length() - prefix_bytes_matched,
                                      current_node.edgecount());

        // Copy the prefix chunks to the new nodes.
        key_node.set_prefix(key_ + key_bytes_matched);
        split_node.set_prefix(current_node.prefix() + prefix_bytes_matched);

        // Copy the current node's edges to the new node.
        split_node.set_first_bytes(current_node.first_bytes());
        split_node.set_node_pointers(current_node.node_pointers());

        // Resize the current node to accommodate a prefix comprising
        // the matched characters and 2 outgoing edges to the above
        // nodes. Set the refcount to 0 since this node doesn't hold a
        // key.
        current_node.resize(prefix_bytes_matched, 2);
        current_node.set_refcount(0);

        // Add links to the new nodes. We don't need to copy the
        // prefix since resize() retains it in the current node.
        current_node.set_edge_at(0, key_node.prefix()[0], key_node);
        current_node.set_edge_at(1, split_node.prefix()[0], split_node);

        ++_size;
        parent_node.set_node_at(edge_index, current_node);
        return true;
    }

    // All characters in the key match, but we still might need to split.
    if (prefix_bytes_matched != current_node.prefix_length())
    {
        // All characters in the key match, but not all characters
        // from the current node's prefix match.

        // Create a node that contains the rest of the characters from
        // the current node's prefix and the outgoing edges from the
        // current node.
        node_t split_node = make_node(current_node.refcount(), current_node.prefix_length() - prefix_bytes_matched,
                                      current_node.edgecount());
        split_node.set_prefix(current_node.prefix() + prefix_bytes_matched);
        split_node.set_first_bytes(current_node.first_bytes());
        split_node.set_node_pointers(current_node.node_pointers());

        // Resize the current node to hold only the matched characters
        // from its prefix and one edge to the new node.
        current_node.resize(prefix_bytes_matched, 1);

        // Add an edge to the split node and set the refcount to 1
        // since this key wasn't inserted earlier. We don't need to
        // set the prefix because the first `prefix_bytes_matched` bytes
        // in the prefix are preserved by resize().
        current_node.set_edge_at(0, split_node.prefix()[0], split_node);
        current_node.set_refcount(1);

        ++_size;
        parent_node.set_node_at(edge_index, current_node);
        return true;
    }

    zmq_assert(key_bytes_matched == key_size_);
    zmq_assert(prefix_bytes_matched == current_node.prefix_length());

    ++_size;
    current_node.set_refcount(current_node.refcount() + 1);
    return current_node.refcount() == 1;
}

bool zmq::radix_tree_t::rm(const unsigned char *key_, size_t key_size_)
{
    match_result_t match_result         = match(key_, key_size_);
    size_t         key_bytes_matched    = match_result._key_bytes_matched;
    size_t         prefix_bytes_matched = match_result._prefix_bytes_matched;
    size_t         edge_index           = match_result._edge_index;
    size_t         parent_edge_index    = match_result._parent_edge_index;
    node_t         current_node         = match_result._current_node;
    node_t         parent_node          = match_result._parent_node;
    node_t         grandparent_node     = match_result._grandparent_node;

    if (key_bytes_matched != key_size_ || prefix_bytes_matched != current_node.prefix_length() ||
        current_node.refcount() == 0)
        return false;

    current_node.set_refcount(current_node.refcount() - 1);
    --_size;
    if (current_node.refcount() > 0)
        return false;

    // Don't delete the root node.
    if (current_node == _root)
        return true;

    size_t outgoing_edges = current_node.edgecount();
    if (outgoing_edges > 1)
        // This node can't be merged with any other node, so there's
        // nothing more to do.
        return true;

    if (outgoing_edges == 1)
    {
        // Merge this node with the single child node.
        node_t child = current_node.node_at(0);

        // Make room for the child node's prefix and edges. We need to
        // keep the old prefix length since resize() will overwrite
        // it.
        uint32_t old_prefix_length = current_node.prefix_length();
        current_node.resize(old_prefix_length + child.prefix_length(), child.edgecount());

        // Append the child node's prefix to the current node.
        memcpy(current_node.prefix() + old_prefix_length, child.prefix(), child.prefix_length());

        // Copy the rest of child node's data to the current node.
        current_node.set_first_bytes(child.first_bytes());
        current_node.set_node_pointers(child.node_pointers());
        current_node.set_refcount(child.refcount());

        free(child._data);
        parent_node.set_node_at(edge_index, current_node);
        return true;
    }

    if (parent_node.edgecount() == 2 && parent_node.refcount() == 0 && parent_node != _root)
    {
        // Removing this node leaves the parent with one child.
        // If the parent doesn't hold a key or if it isn't the root,
        // we can merge it with its single child node.
        zmq_assert(edge_index < 2);
        node_t other_child = parent_node.node_at(!edge_index);

        // Make room for the child node's prefix and edges. We need to
        // keep the old prefix length since resize() will overwrite
        // it.
        uint32_t old_prefix_length = parent_node.prefix_length();
        parent_node.resize(old_prefix_length + other_child.prefix_length(), other_child.edgecount());

        // Append the child node's prefix to the current node.
        memcpy(parent_node.prefix() + old_prefix_length, other_child.prefix(), other_child.prefix_length());

        // Copy the rest of child node's data to the current node.
        parent_node.set_first_bytes(other_child.first_bytes());
        parent_node.set_node_pointers(other_child.node_pointers());
        parent_node.set_refcount(other_child.refcount());

        free(current_node._data);
        free(other_child._data);
        grandparent_node.set_node_at(parent_edge_index, parent_node);
        return true;
    }

    // This is a leaf node that doesn't leave its parent with one
    // outgoing edge. Remove the outgoing edge to this node from the
    // parent.
    zmq_assert(outgoing_edges == 0);

    // Replace the edge to the current node with the last edge. An
    // edge consists of a byte and a pointer to the next node. First
    // replace the byte.
    size_t        last_index = parent_node.edgecount() - 1;
    unsigned char last_byte  = parent_node.first_byte_at(last_index);
    node_t        last_node  = parent_node.node_at(last_index);
    parent_node.set_edge_at(edge_index, last_byte, last_node);

    // Move the chunk of pointers one byte to the left, effectively
    // deleting the last byte in the region of first bytes by
    // overwriting it.
    memmove(parent_node.node_pointers() - 1, parent_node.node_pointers(), parent_node.edgecount() * sizeof(void *));

    // Shrink the parent node to the new size, which "deletes" the
    // last pointer in the chunk of node pointers.
    parent_node.resize(parent_node.prefix_length(), parent_node.edgecount() - 1);

    // Nothing points to this node now, so we can reclaim it.
    free(current_node._data);

    if (parent_node.prefix_length() == 0)
        _root._data = parent_node._data;
    else
        grandparent_node.set_node_at(parent_edge_index, parent_node);
    return true;
}

bool zmq::radix_tree_t::check(const unsigned char *key_, size_t key_size_)
{
    if (_root.refcount() > 0)
        return true;

    match_result_t match_result = match(key_, key_size_, true);
    return match_result._key_bytes_matched == key_size_ &&
           match_result._prefix_bytes_matched == match_result._current_node.prefix_length() &&
           match_result._current_node.refcount() > 0;
}

static void visit_keys(node_t node_, std::vector<unsigned char> &                        buffer_,
                       void (*func_)(unsigned char *data, size_t size, void *arg), void *arg_)
{
    for (size_t i = 0; i < node_.prefix_length(); ++i)
        buffer_.push_back(node_.prefix()[i]);

    if (node_.refcount() > 0)
    {
        zmq_assert(!buffer_.empty());
        func_(&buffer_[0], buffer_.size(), arg_);
    }

    for (size_t i = 0; i < node_.edgecount(); ++i)
        visit_keys(node_.node_at(i), buffer_, func_, arg_);
    for (size_t i = 0; i < node_.prefix_length(); ++i)
        buffer_.pop_back();
}

void zmq::radix_tree_t::apply(void (*func_)(unsigned char *data, size_t size, void *arg), void *arg_)
{
    if (_root.refcount() > 0)
        func_(NULL, 0, arg_); // Root node is always empty.

    std::vector<unsigned char> buffer;
    for (size_t i = 0; i < _root.edgecount(); ++i)
        visit_keys(_root.node_at(i), buffer, func_, arg_);
}

size_t zmq::radix_tree_t::size() const { return _size; }

//========= end of radix_tree.cpp ============

//========= begin of random.cpp ============

/*
    Copyright (c) 2007-2017 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <stdlib.h>

#if !defined ZMQ_HAVE_WINDOWS
#include <unistd.h>
#endif

// ans ignore: #include "random.hpp"
// ans ignore: #include "stdint.hpp"
// ans ignore: #include "clock.hpp"
// ans ignore: #include "mutex.hpp"
// ans ignore: #include "macros.hpp"

#if defined(ZMQ_USE_TWEETNACL)
// ans ignore: #include "tweetnacl.h"
#elif defined(ZMQ_USE_LIBSODIUM)
// ans ignore: #include "sodium.h"
#endif

void zmq::seed_random()
{
#if defined ZMQ_HAVE_WINDOWS
    int pid = static_cast<int>(GetCurrentProcessId());
#else
    int pid = static_cast<int>(getpid());
#endif
    srand(static_cast<unsigned int>(clock_t::now_us() + pid));
}

uint32_t zmq::generate_random()
{
    //  Compensate for the fact that rand() returns signed integer.
    uint32_t low  = static_cast<uint32_t>(rand());
    uint32_t high = static_cast<uint32_t>(rand());
    high <<= (sizeof(int) * 8 - 1);
    return high | low;
}

//  When different threads have their own context the file descriptor
//  variable is shared and is subject to race conditions in tweetnacl,
//  that lead to file descriptors leaks. In long-running programs with
//  ephemeral threads this is a problem as it accumulates.
//  thread-local storage cannot be used to initialise the file descriptor
//  as it is perfectly legal to share a context among many threads, each
//  of which might call curve APIs.
//  Also libsodium documentation specifically states that sodium_init
//  must not be called concurrently from multiple threads, for the
//  same reason. Inspecting the code also reveals that the close API is
//  not thread safe.
//  The context class cannot be used with static variables as the curve
//  utility APIs like zmq_curve_keypair also call into the crypto
//  library.
//  The safest solution for all use cases therefore is to have a
//  static lock to serialize calls into an initialiser and a finaliser,
//  using refcounts to make sure that a thread does not close the library
//  while another is still using it. To avoid the static initialization
//  order fiasco, this is done using function-local statics, if the
//  compiler implementation supports thread-safe initialization of those.
//  Otherwise, we fall back to global statics.
//  HOWEVER, this initialisation code imposes ordering constraints, which
//  are not obvious to users of libzmq, and may lead to problems if atexit
//  or similar methods are used for cleanup.
//  In that case, a strict ordering is imposed whereas the contexts MUST
//  be initialised BEFORE registering the cleanup with atexit. CZMQ is an
//  example. Hence we make the choice to restrict this global transition
//  mechanism ONLY to Tweenacl + *NIX (when using /dev/urandom) as it is
//  the less risky option.

//  TODO if there is some other user of libsodium besides libzmq, this must
//    be synchronized by the application. This should probably also be
//    configurable via config.h

//  TODO this should probably be done via config.h
#if __cplusplus >= 201103L || (defined(__cpp_threadsafe_static_init) && __cpp_threadsafe_static_init >= 200806) ||     \
    (defined(_MSC_VER) && _MSC_VER >= 1900)
#define ZMQ_HAVE_THREADSAFE_STATIC_LOCAL_INIT 1
//  TODO this might probably also be set if a sufficiently recent gcc is used
//  without -fno-threadsafe-statics, but this cannot be determined at
//  compile-time, so it must be set via config.h
#else
#define ZMQ_HAVE_THREADSAFE_STATIC_LOCAL_INIT 0
#endif

#if !ZMQ_HAVE_THREADSAFE_STATIC_LOCAL_INIT &&                                                                          \
    (defined(ZMQ_USE_TWEETNACL) && !defined(ZMQ_HAVE_WINDOWS) && !defined(ZMQ_HAVE_GETRANDOM))
static unsigned int random_refcount = 0;
static zmq::mutex_t random_sync;
#endif

static void manage_random(bool init_)
{
#if defined(ZMQ_USE_TWEETNACL) && !defined(ZMQ_HAVE_WINDOWS) && !defined(ZMQ_HAVE_GETRANDOM)

#if ZMQ_HAVE_THREADSAFE_STATIC_LOCAL_INIT
    static int          random_refcount = 0;
    static zmq::mutex_t random_sync;
#endif

    if (init_)
    {
        zmq::scoped_lock_t locker(random_sync);

        if (random_refcount == 0)
        {
            int rc = sodium_init();
            zmq_assert(rc != -1);
        }

        ++random_refcount;
    }
    else
    {
        zmq::scoped_lock_t locker(random_sync);
        --random_refcount;

        if (random_refcount == 0)
        {
            randombytes_close();
        }
    }

#elif defined(ZMQ_USE_LIBSODIUM)
    if (init_)
    {
        int rc = sodium_init();
        zmq_assert(rc != -1);
    }
    else
    {
        randombytes_close();
    }
#else
    LIBZMQ_UNUSED(init_);
#endif
}

void zmq::random_open() { manage_random(true); }

void zmq::random_close() { manage_random(false); }

//========= end of random.cpp ============

//========= begin of raw_decoder.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <stdlib.h>
#include <string.h>

// ans ignore: #include "raw_decoder.hpp"
// ans ignore: #include "err.hpp"

zmq::raw_decoder_t::raw_decoder_t(size_t bufsize_) : _allocator(bufsize_, 1)
{
    const int rc = _in_progress.init();
    errno_assert(rc == 0);
}

zmq::raw_decoder_t::~raw_decoder_t()
{
    const int rc = _in_progress.close();
    errno_assert(rc == 0);
}

void zmq::raw_decoder_t::get_buffer(unsigned char **data_, size_t *size_)
{
    *data_ = _allocator.allocate();
    *size_ = _allocator.size();
}

int zmq::raw_decoder_t::decode(const uint8_t *data_, size_t size_, size_t &bytes_used_)
{
    const int rc =
        _in_progress.init(const_cast<unsigned char *>(data_), size_, shared_message_memory_allocator::call_dec_ref,
                          _allocator.buffer(), _allocator.provide_content());

    // if the buffer serves as memory for a zero-copy message, release it
    // and allocate a new buffer in get_buffer for the next decode
    if (_in_progress.is_zcmsg())
    {
        _allocator.advance_content();
        _allocator.release();
    }

    errno_assert(rc != -1);
    bytes_used_ = size_;
    return 1;
}

//========= end of raw_decoder.cpp ============

//========= begin of raw_encoder.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "encoder.hpp"
// ans ignore: #include "raw_encoder.hpp"
// ans ignore: #include "msg.hpp"

zmq::raw_encoder_t::raw_encoder_t(size_t bufsize_) : encoder_base_t<raw_encoder_t>(bufsize_)
{
    //  Write 0 bytes to the batch and go to message_ready state.
    next_step(NULL, 0, &raw_encoder_t::raw_message_ready, true);
}

zmq::raw_encoder_t::~raw_encoder_t() {}

void zmq::raw_encoder_t::raw_message_ready()
{
    next_step(in_progress()->data(), in_progress()->size(), &raw_encoder_t::raw_message_ready, true);
}

//========= end of raw_encoder.cpp ============

//========= begin of reaper.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "reaper.hpp"
// ans ignore: #include "socket_base.hpp"
// ans ignore: #include "err.hpp"

zmq::reaper_t::reaper_t(class ctx_t *ctx_, uint32_t tid_)
    : object_t(ctx_, tid_), _mailbox_handle(static_cast<poller_t::handle_t>(NULL)), _poller(NULL), _sockets(0),
      _terminating(false)
{
    if (!_mailbox.valid())
        return;

    _poller = new (std::nothrow) poller_t(*ctx_);
    alloc_assert(_poller);

    if (_mailbox.get_fd() != retired_fd)
    {
        _mailbox_handle = _poller->add_fd(_mailbox.get_fd(), this);
        _poller->set_pollin(_mailbox_handle);
    }

#ifdef HAVE_FORK
    _pid = getpid();
#endif
}

zmq::reaper_t::~reaper_t() { LIBZMQ_DELETE(_poller); }

zmq::mailbox_t *zmq::reaper_t::get_mailbox() { return &_mailbox; }

void zmq::reaper_t::start()
{
    zmq_assert(_mailbox.valid());

    //  Start the thread.
    _poller->start("Reaper");
}

void zmq::reaper_t::stop()
{
    if (get_mailbox()->valid())
    {
        send_stop();
    }
}

void zmq::reaper_t::in_event()
{
    while (true)
    {
#ifdef HAVE_FORK
        if (unlikely(_pid != getpid()))
        {
            // printf("zmq::reaper_t::in_event return in child process %d\n", (int)getpid());
            return;
        }
#endif

        //  Get the next command. If there is none, exit.
        command_t cmd;
        int       rc = _mailbox.recv(&cmd, 0);
        if (rc != 0 && errno == EINTR)
            continue;
        if (rc != 0 && errno == EAGAIN)
            break;
        errno_assert(rc == 0);

        //  Process the command.
        cmd.destination->process_command(cmd);
    }
}

void zmq::reaper_t::out_event() { zmq_assert(false); }

void zmq::reaper_t::timer_event(int) { zmq_assert(false); }

void zmq::reaper_t::process_stop()
{
    _terminating = true;

    //  If there are no sockets being reaped finish immediately.
    if (!_sockets)
    {
        send_done();
        _poller->rm_fd(_mailbox_handle);
        _poller->stop();
    }
}

void zmq::reaper_t::process_reap(socket_base_t *socket_)
{
    //  Add the socket to the poller.
    socket_->start_reaping(_poller);

    ++_sockets;
}

void zmq::reaper_t::process_reaped()
{
    --_sockets;

    //  If reaped was already asked to terminate and there are no more sockets,
    //  finish immediately.
    if (!_sockets && _terminating)
    {
        send_done();
        _poller->rm_fd(_mailbox_handle);
        _poller->stop();
    }
}

//========= end of reaper.cpp ============

//========= begin of rep.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "rep.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "msg.hpp"

zmq::rep_t::rep_t(class ctx_t *parent_, uint32_t tid_, int sid_)
    : router_t(parent_, tid_, sid_), _sending_reply(false), _request_begins(true)
{
    options.type = ZMQ_REP;
}

zmq::rep_t::~rep_t() {}

int zmq::rep_t::xsend(msg_t *msg_)
{
    //  If we are in the middle of receiving a request, we cannot send reply.
    if (!_sending_reply)
    {
        errno = EFSM;
        return -1;
    }

    bool more = (msg_->flags() & msg_t::more) != 0;

    //  Push message to the reply pipe.
    int rc = router_t::xsend(msg_);
    if (rc != 0)
        return rc;

    //  If the reply is complete flip the FSM back to request receiving state.
    if (!more)
        _sending_reply = false;

    return 0;
}

int zmq::rep_t::xrecv(msg_t *msg_)
{
    //  If we are in middle of sending a reply, we cannot receive next request.
    if (_sending_reply)
    {
        errno = EFSM;
        return -1;
    }

    //  First thing to do when receiving a request is to copy all the labels
    //  to the reply pipe.
    if (_request_begins)
    {
        while (true)
        {
            int rc = router_t::xrecv(msg_);
            if (rc != 0)
                return rc;

            if ((msg_->flags() & msg_t::more))
            {
                //  Empty message part delimits the traceback stack.
                bool bottom = (msg_->size() == 0);

                //  Push it to the reply pipe.
                rc = router_t::xsend(msg_);
                errno_assert(rc == 0);

                if (bottom)
                    break;
            }
            else
            {
                //  If the traceback stack is malformed, discard anything
                //  already sent to pipe (we're at end of invalid message).
                rc = router_t::rollback();
                errno_assert(rc == 0);
            }
        }
        _request_begins = false;
    }

    //  Get next message part to return to the user.
    int rc = router_t::xrecv(msg_);
    if (rc != 0)
        return rc;

    //  If whole request is read, flip the FSM to reply-sending state.
    if (!(msg_->flags() & msg_t::more))
    {
        _sending_reply  = true;
        _request_begins = true;
    }

    return 0;
}

bool zmq::rep_t::xhas_in()
{
    if (_sending_reply)
        return false;

    return router_t::xhas_in();
}

bool zmq::rep_t::xhas_out()
{
    if (!_sending_reply)
        return false;

    return router_t::xhas_out();
}

//========= end of rep.cpp ============

//========= begin of req.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "req.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "msg.hpp"
// ans ignore: #include "wire.hpp"
// ans ignore: #include "random.hpp"
// ans ignore: #include "likely.hpp"

zmq::req_t::req_t(class ctx_t *parent_, uint32_t tid_, int sid_)
    : dealer_t(parent_, tid_, sid_), _receiving_reply(false), _message_begins(true), _reply_pipe(NULL),
      _request_id_frames_enabled(false), _request_id(generate_random()), _strict(true)
{
    options.type = ZMQ_REQ;
}

zmq::req_t::~req_t() {}

int zmq::req_t::xsend(msg_t *msg_)
{
    //  If we've sent a request and we still haven't got the reply,
    //  we can't send another request unless the strict option is disabled.
    if (_receiving_reply)
    {
        if (_strict)
        {
            errno = EFSM;
            return -1;
        }

        _receiving_reply = false;
        _message_begins  = true;
    }

    //  First part of the request is the request routing id.
    if (_message_begins)
    {
        _reply_pipe = NULL;

        if (_request_id_frames_enabled)
        {
            _request_id++;

            msg_t id;
            int   rc = id.init_size(sizeof(uint32_t));
            memcpy(id.data(), &_request_id, sizeof(uint32_t));
            errno_assert(rc == 0);
            id.set_flags(msg_t::more);

            rc = dealer_t::sendpipe(&id, &_reply_pipe);
            if (rc != 0)
            {
                return -1;
            }
        }

        msg_t bottom;
        int   rc = bottom.init();
        errno_assert(rc == 0);
        bottom.set_flags(msg_t::more);

        rc = dealer_t::sendpipe(&bottom, &_reply_pipe);
        if (rc != 0)
            return -1;
        zmq_assert(_reply_pipe);

        _message_begins = false;

        // Eat all currently available messages before the request is fully
        // sent. This is done to avoid:
        //   REQ sends request to A, A replies, B replies too.
        //   A's reply was first and matches, that is used.
        //   An hour later REQ sends a request to B. B's old reply is used.
        msg_t drop;
        while (true)
        {
            rc = drop.init();
            errno_assert(rc == 0);
            rc = dealer_t::xrecv(&drop);
            if (rc != 0)
                break;
            drop.close();
        }
    }

    bool more = (msg_->flags() & msg_t::more) != 0;

    int rc = dealer_t::xsend(msg_);
    if (rc != 0)
        return rc;

    //  If the request was fully sent, flip the FSM into reply-receiving state.
    if (!more)
    {
        _receiving_reply = true;
        _message_begins  = true;
    }

    return 0;
}

int zmq::req_t::xrecv(msg_t *msg_)
{
    //  If request wasn't send, we can't wait for reply.
    if (!_receiving_reply)
    {
        errno = EFSM;
        return -1;
    }

    //  Skip messages until one with the right first frames is found.
    while (_message_begins)
    {
        //  If enabled, the first frame must have the correct request_id.
        if (_request_id_frames_enabled)
        {
            int rc = recv_reply_pipe(msg_);
            if (rc != 0)
                return rc;

            if (unlikely(!(msg_->flags() & msg_t::more) || msg_->size() != sizeof(_request_id) ||
                         *static_cast<uint32_t *>(msg_->data()) != _request_id))
            {
                //  Skip the remaining frames and try the next message
                while (msg_->flags() & msg_t::more)
                {
                    rc = recv_reply_pipe(msg_);
                    errno_assert(rc == 0);
                }
                continue;
            }
        }

        //  The next frame must be 0.
        // TODO: Failing this check should also close the connection with the peer!
        int rc = recv_reply_pipe(msg_);
        if (rc != 0)
            return rc;

        if (unlikely(!(msg_->flags() & msg_t::more) || msg_->size() != 0))
        {
            //  Skip the remaining frames and try the next message
            while (msg_->flags() & msg_t::more)
            {
                rc = recv_reply_pipe(msg_);
                errno_assert(rc == 0);
            }
            continue;
        }

        _message_begins = false;
    }

    int rc = recv_reply_pipe(msg_);
    if (rc != 0)
        return rc;

    //  If the reply is fully received, flip the FSM into request-sending state.
    if (!(msg_->flags() & msg_t::more))
    {
        _receiving_reply = false;
        _message_begins  = true;
    }

    return 0;
}

bool zmq::req_t::xhas_in()
{
    //  TODO: Duplicates should be removed here.

    if (!_receiving_reply)
        return false;

    return dealer_t::xhas_in();
}

bool zmq::req_t::xhas_out()
{
    if (_receiving_reply && _strict)
        return false;

    return dealer_t::xhas_out();
}

int zmq::req_t::xsetsockopt(int option_, const void *optval_, size_t optvallen_)
{
    bool is_int = (optvallen_ == sizeof(int));
    int  value  = 0;
    if (is_int)
        memcpy(&value, optval_, sizeof(int));

    switch (option_)
    {
    case ZMQ_REQ_CORRELATE:
        if (is_int && value >= 0)
        {
            _request_id_frames_enabled = (value != 0);
            return 0;
        }
        break;

    case ZMQ_REQ_RELAXED:
        if (is_int && value >= 0)
        {
            _strict = (value == 0);
            return 0;
        }
        break;

    default:
        break;
    }

    return dealer_t::xsetsockopt(option_, optval_, optvallen_);
}

void zmq::req_t::xpipe_terminated(pipe_t *pipe_)
{
    if (_reply_pipe == pipe_)
        _reply_pipe = NULL;
    dealer_t::xpipe_terminated(pipe_);
}

int zmq::req_t::recv_reply_pipe(msg_t *msg_)
{
    while (true)
    {
        pipe_t *pipe = NULL;
        int     rc   = dealer_t::recvpipe(msg_, &pipe);
        if (rc != 0)
            return rc;
        if (!_reply_pipe || pipe == _reply_pipe)
            return 0;
    }
}

zmq::req_session_t::req_session_t(io_thread_t *io_thread_, bool connect_, socket_base_t *socket_,
                                  const options_t &options_, address_t *addr_)
    : session_base_t(io_thread_, connect_, socket_, options_, addr_), _state(bottom)
{
}

zmq::req_session_t::~req_session_t() {}

int zmq::req_session_t::push_msg(msg_t *msg_)
{
    //  Ignore commands, they are processed by the engine and should not
    //  affect the state machine.
    if (unlikely(msg_->flags() & msg_t::command))
        return 0;

    switch (_state)
    {
    case bottom:
        if (msg_->flags() == msg_t::more)
        {
            //  In case option ZMQ_CORRELATE is on, allow request_id to be
            //  transfered as first frame (would be too cumbersome to check
            //  whether the option is actually on or not).
            if (msg_->size() == sizeof(uint32_t))
            {
                _state = request_id;
                return session_base_t::push_msg(msg_);
            }
            if (msg_->size() == 0)
            {
                _state = body;
                return session_base_t::push_msg(msg_);
            }
        }
        break;
    case request_id:
        if (msg_->flags() == msg_t::more && msg_->size() == 0)
        {
            _state = body;
            return session_base_t::push_msg(msg_);
        }
        break;
    case body:
        if (msg_->flags() == msg_t::more)
            return session_base_t::push_msg(msg_);
        if (msg_->flags() == 0)
        {
            _state = bottom;
            return session_base_t::push_msg(msg_);
        }
        break;
    }
    errno = EFAULT;
    return -1;
}

void zmq::req_session_t::reset()
{
    session_base_t::reset();
    _state = bottom;
}

//========= end of req.cpp ============

//========= begin of router.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "router.hpp"
// ans ignore: #include "pipe.hpp"
// ans ignore: #include "wire.hpp"
// ans ignore: #include "random.hpp"
// ans ignore: #include "likely.hpp"
// ans ignore: #include "err.hpp"

zmq::router_t::router_t(class ctx_t *parent_, uint32_t tid_, int sid_)
    : routing_socket_base_t(parent_, tid_, sid_), _prefetched(false), _routing_id_sent(false), _current_in(NULL),
      _terminate_current_in(false), _more_in(false), _current_out(NULL), _more_out(false),
      _next_integral_routing_id(generate_random()), _mandatory(false),
      //  raw_socket functionality in ROUTER is deprecated
      _raw_socket(false), _probe_router(false), _handover(false)
{
    options.type            = ZMQ_ROUTER;
    options.recv_routing_id = true;
    options.raw_socket      = false;

    _prefetched_id.init();
    _prefetched_msg.init();
}

zmq::router_t::~router_t()
{
    zmq_assert(_anonymous_pipes.empty());
    _prefetched_id.close();
    _prefetched_msg.close();
}

void zmq::router_t::xattach_pipe(pipe_t *pipe_, bool subscribe_to_all_, bool locally_initiated_)
{
    LIBZMQ_UNUSED(subscribe_to_all_);

    zmq_assert(pipe_);

    if (_probe_router)
    {
        msg_t probe_msg;
        int   rc = probe_msg.init();
        errno_assert(rc == 0);

        rc = pipe_->write(&probe_msg);
        // zmq_assert (rc) is not applicable here, since it is not a bug.
        LIBZMQ_UNUSED(rc);

        pipe_->flush();

        rc = probe_msg.close();
        errno_assert(rc == 0);
    }

    bool routing_id_ok = identify_peer(pipe_, locally_initiated_);
    if (routing_id_ok)
        _fq.attach(pipe_);
    else
        _anonymous_pipes.insert(pipe_);
}

int zmq::router_t::xsetsockopt(int option_, const void *optval_, size_t optvallen_)
{
    const bool is_int = (optvallen_ == sizeof(int));
    int        value  = 0;
    if (is_int)
        memcpy(&value, optval_, sizeof(int));

    switch (option_)
    {
    case ZMQ_ROUTER_RAW:
        if (is_int && value >= 0)
        {
            _raw_socket = (value != 0);
            if (_raw_socket)
            {
                options.recv_routing_id = false;
                options.raw_socket      = true;
            }
            return 0;
        }
        break;

    case ZMQ_ROUTER_MANDATORY:
        if (is_int && value >= 0)
        {
            _mandatory = (value != 0);
            return 0;
        }
        break;

    case ZMQ_PROBE_ROUTER:
        if (is_int && value >= 0)
        {
            _probe_router = (value != 0);
            return 0;
        }
        break;

    case ZMQ_ROUTER_HANDOVER:
        if (is_int && value >= 0)
        {
            _handover = (value != 0);
            return 0;
        }
        break;

#ifdef ZMQ_BUILD_DRAFT_API
    case ZMQ_ROUTER_NOTIFY:
        if (is_int && value >= 0 && value <= (ZMQ_NOTIFY_CONNECT | ZMQ_NOTIFY_DISCONNECT))
        {
            options.router_notify = value;
            return 0;
        }
        break;
#endif

    default:
        return routing_socket_base_t::xsetsockopt(option_, optval_, optvallen_);
    }
    errno = EINVAL;
    return -1;
}

void zmq::router_t::xpipe_terminated(pipe_t *pipe_)
{
    if (0 == _anonymous_pipes.erase(pipe_))
    {
        erase_out_pipe(pipe_);
        _fq.pipe_terminated(pipe_);
        pipe_->rollback();
        if (pipe_ == _current_out)
            _current_out = NULL;
    }
}

void zmq::router_t::xread_activated(pipe_t *pipe_)
{
    std::set<pipe_t *>::iterator it = _anonymous_pipes.find(pipe_);
    if (it == _anonymous_pipes.end())
        _fq.activated(pipe_);
    else
    {
        const bool routing_id_ok = identify_peer(pipe_, false);
        if (routing_id_ok)
        {
            _anonymous_pipes.erase(it);
            _fq.attach(pipe_);
        }
    }
}

int zmq::router_t::xsend(msg_t *msg_)
{
    //  If this is the first part of the message it's the ID of the
    //  peer to send the message to.
    if (!_more_out)
    {
        zmq_assert(!_current_out);

        //  If we have malformed message (prefix with no subsequent message)
        //  then just silently ignore it.
        //  TODO: The connections should be killed instead.
        if (msg_->flags() & msg_t::more)
        {
            _more_out = true;

            //  Find the pipe associated with the routing id stored in the prefix.
            //  If there's no such pipe just silently ignore the message, unless
            //  router_mandatory is set.
            out_pipe_t *out_pipe = lookup_out_pipe(
                blob_t(static_cast<unsigned char *>(msg_->data()), msg_->size(), zmq::reference_tag_t()));

            if (out_pipe)
            {
                _current_out = out_pipe->pipe;

                // Check whether pipe is closed or not
                if (!_current_out->check_write())
                {
                    // Check whether pipe is full or not
                    bool pipe_full   = !_current_out->check_hwm();
                    out_pipe->active = false;
                    _current_out     = NULL;

                    if (_mandatory)
                    {
                        _more_out = false;
                        if (pipe_full)
                            errno = EAGAIN;
                        else
                            errno = EHOSTUNREACH;
                        return -1;
                    }
                }
            }
            else if (_mandatory)
            {
                _more_out = false;
                errno     = EHOSTUNREACH;
                return -1;
            }
        }

        int rc = msg_->close();
        errno_assert(rc == 0);
        rc = msg_->init();
        errno_assert(rc == 0);
        return 0;
    }

    //  Ignore the MORE flag for raw-sock or assert?
    if (options.raw_socket)
        msg_->reset_flags(msg_t::more);

    //  Check whether this is the last part of the message.
    _more_out = (msg_->flags() & msg_t::more) != 0;

    //  Push the message into the pipe. If there's no out pipe, just drop it.
    if (_current_out)
    {
        // Close the remote connection if user has asked to do so
        // by sending zero length message.
        // Pending messages in the pipe will be dropped (on receiving term- ack)
        if (_raw_socket && msg_->size() == 0)
        {
            _current_out->terminate(false);
            int rc = msg_->close();
            errno_assert(rc == 0);
            rc = msg_->init();
            errno_assert(rc == 0);
            _current_out = NULL;
            return 0;
        }

        bool ok = _current_out->write(msg_);
        if (unlikely(!ok))
        {
            // Message failed to send - we must close it ourselves.
            int rc = msg_->close();
            errno_assert(rc == 0);
            // HWM was checked before, so the pipe must be gone. Roll back
            // messages that were piped, for example REP labels.
            _current_out->rollback();
            _current_out = NULL;
        }
        else
        {
            if (!_more_out)
            {
                _current_out->flush();
                _current_out = NULL;
            }
        }
    }
    else
    {
        int rc = msg_->close();
        errno_assert(rc == 0);
    }

    //  Detach the message from the data buffer.
    int rc = msg_->init();
    errno_assert(rc == 0);

    return 0;
}

int zmq::router_t::xrecv(msg_t *msg_)
{
    if (_prefetched)
    {
        if (!_routing_id_sent)
        {
            int rc = msg_->move(_prefetched_id);
            errno_assert(rc == 0);
            _routing_id_sent = true;
        }
        else
        {
            int rc = msg_->move(_prefetched_msg);
            errno_assert(rc == 0);
            _prefetched = false;
        }
        _more_in = (msg_->flags() & msg_t::more) != 0;

        if (!_more_in)
        {
            if (_terminate_current_in)
            {
                _current_in->terminate(true);
                _terminate_current_in = false;
            }
            _current_in = NULL;
        }
        return 0;
    }

    pipe_t *pipe = NULL;
    int     rc   = _fq.recvpipe(msg_, &pipe);

    //  It's possible that we receive peer's routing id. That happens
    //  after reconnection. The current implementation assumes that
    //  the peer always uses the same routing id.
    while (rc == 0 && msg_->is_routing_id())
        rc = _fq.recvpipe(msg_, &pipe);

    if (rc != 0)
        return -1;

    zmq_assert(pipe != NULL);

    //  If we are in the middle of reading a message, just return the next part.
    if (_more_in)
    {
        _more_in = (msg_->flags() & msg_t::more) != 0;

        if (!_more_in)
        {
            if (_terminate_current_in)
            {
                _current_in->terminate(true);
                _terminate_current_in = false;
            }
            _current_in = NULL;
        }
    }
    else
    {
        //  We are at the beginning of a message.
        //  Keep the message part we have in the prefetch buffer
        //  and return the ID of the peer instead.
        rc = _prefetched_msg.move(*msg_);
        errno_assert(rc == 0);
        _prefetched = true;
        _current_in = pipe;

        const blob_t &routing_id = pipe->get_routing_id();
        rc                       = msg_->init_size(routing_id.size());
        errno_assert(rc == 0);
        memcpy(msg_->data(), routing_id.data(), routing_id.size());
        msg_->set_flags(msg_t::more);
        if (_prefetched_msg.metadata())
            msg_->set_metadata(_prefetched_msg.metadata());
        _routing_id_sent = true;
    }

    return 0;
}

int zmq::router_t::rollback()
{
    if (_current_out)
    {
        _current_out->rollback();
        _current_out = NULL;
        _more_out    = false;
    }
    return 0;
}

bool zmq::router_t::xhas_in()
{
    //  If we are in the middle of reading the messages, there are
    //  definitely more parts available.
    if (_more_in)
        return true;

    //  We may already have a message pre-fetched.
    if (_prefetched)
        return true;

    //  Try to read the next message.
    //  The message, if read, is kept in the pre-fetch buffer.
    pipe_t *pipe = NULL;
    int     rc   = _fq.recvpipe(&_prefetched_msg, &pipe);

    //  It's possible that we receive peer's routing id. That happens
    //  after reconnection. The current implementation assumes that
    //  the peer always uses the same routing id.
    //  TODO: handle the situation when the peer changes its routing id.
    while (rc == 0 && _prefetched_msg.is_routing_id())
        rc = _fq.recvpipe(&_prefetched_msg, &pipe);

    if (rc != 0)
        return false;

    zmq_assert(pipe != NULL);

    const blob_t &routing_id = pipe->get_routing_id();
    rc                       = _prefetched_id.init_size(routing_id.size());
    errno_assert(rc == 0);
    memcpy(_prefetched_id.data(), routing_id.data(), routing_id.size());
    _prefetched_id.set_flags(msg_t::more);

    _prefetched      = true;
    _routing_id_sent = false;
    _current_in      = pipe;

    return true;
}

static bool check_pipe_hwm(const zmq::pipe_t &pipe_) { return pipe_.check_hwm(); }

bool zmq::router_t::xhas_out()
{
    //  In theory, ROUTER socket is always ready for writing (except when
    //  MANDATORY is set). Whether actual attempt to write succeeds depends
    //  on whitch pipe the message is going to be routed to.

    if (!_mandatory)
        return true;

    return any_of_out_pipes(check_pipe_hwm);
}

int zmq::router_t::get_peer_state(const void *routing_id_, size_t routing_id_size_) const
{
    int res = 0;

    // TODO remove the const_cast, see comment in lookup_out_pipe
    const blob_t      routing_id_blob(static_cast<unsigned char *>(const_cast<void *>(routing_id_)), routing_id_size_);
    const out_pipe_t *out_pipe = lookup_out_pipe(routing_id_blob);
    if (!out_pipe)
    {
        errno = EHOSTUNREACH;
        return -1;
    }

    if (out_pipe->pipe->check_hwm())
        res |= ZMQ_POLLOUT;

    /** \todo does it make any sense to check the inpipe as well? */

    return res;
}

bool zmq::router_t::identify_peer(pipe_t *pipe_, bool locally_initiated_)
{
    msg_t  msg;
    blob_t routing_id;

    if (locally_initiated_ && connect_routing_id_is_set())
    {
        const std::string connect_routing_id = extract_connect_routing_id();
        routing_id.set(reinterpret_cast<const unsigned char *>(connect_routing_id.c_str()),
                       connect_routing_id.length());
        //  Not allowed to duplicate an existing rid
        zmq_assert(!has_out_pipe(routing_id));
    }
    else if (options.raw_socket)
    { //  Always assign an integral routing id for raw-socket
        unsigned char buf[5];
        buf[0] = 0;
        put_uint32(buf + 1, _next_integral_routing_id++);
        routing_id.set(buf, sizeof buf);
    }
    else if (!options.raw_socket)
    {
        //  Pick up handshake cases and also case where next integral routing id is set
        msg.init();
        bool ok = pipe_->read(&msg);
        if (!ok)
            return false;

        if (msg.size() == 0)
        {
            //  Fall back on the auto-generation
            unsigned char buf[5];
            buf[0] = 0;
            put_uint32(buf + 1, _next_integral_routing_id++);
            routing_id.set(buf, sizeof buf);
            msg.close();
        }
        else
        {
            routing_id.set(static_cast<unsigned char *>(msg.data()), msg.size());
            msg.close();

            //  Try to remove an existing routing id entry to allow the new
            //  connection to take the routing id.
            out_pipe_t *existing_outpipe = lookup_out_pipe(routing_id);

            if (existing_outpipe)
            {
                if (!_handover)
                    //  Ignore peers with duplicate ID
                    return false;

                //  We will allow the new connection to take over this
                //  routing id. Temporarily assign a new routing id to the
                //  existing pipe so we can terminate it asynchronously.
                unsigned char buf[5];
                buf[0] = 0;
                put_uint32(buf + 1, _next_integral_routing_id++);
                blob_t new_routing_id(buf, sizeof buf);

                pipe_t *const old_pipe = existing_outpipe->pipe;

                erase_out_pipe(old_pipe);
                old_pipe->set_router_socket_routing_id(new_routing_id);
                add_out_pipe(ZMQ_MOVE(new_routing_id), old_pipe);

                if (old_pipe == _current_in)
                    _terminate_current_in = true;
                else
                    old_pipe->terminate(true);
            }
        }
    }

    pipe_->set_router_socket_routing_id(routing_id);
    add_out_pipe(ZMQ_MOVE(routing_id), pipe_);

    return true;
}

//========= end of router.cpp ============

//========= begin of scatter.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "scatter.hpp"
// ans ignore: #include "pipe.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "msg.hpp"

zmq::scatter_t::scatter_t(class ctx_t *parent_, uint32_t tid_, int sid_) : socket_base_t(parent_, tid_, sid_, true)
{
    options.type = ZMQ_SCATTER;
}

zmq::scatter_t::~scatter_t() {}

void zmq::scatter_t::xattach_pipe(pipe_t *pipe_, bool subscribe_to_all_, bool locally_initiated_)
{
    LIBZMQ_UNUSED(subscribe_to_all_);
    LIBZMQ_UNUSED(locally_initiated_);

    //  Don't delay pipe termination as there is no one
    //  to receive the delimiter.
    pipe_->set_nodelay();

    zmq_assert(pipe_);
    _lb.attach(pipe_);
}

void zmq::scatter_t::xwrite_activated(pipe_t *pipe_) { _lb.activated(pipe_); }

void zmq::scatter_t::xpipe_terminated(pipe_t *pipe_) { _lb.pipe_terminated(pipe_); }

int zmq::scatter_t::xsend(msg_t *msg_)
{
    //  SCATTER sockets do not allow multipart data (ZMQ_SNDMORE)
    if (msg_->flags() & msg_t::more)
    {
        errno = EINVAL;
        return -1;
    }

    return _lb.send(msg_);
}

bool zmq::scatter_t::xhas_out() { return _lb.has_out(); }

//========= end of scatter.cpp ============

//========= begin of select.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "select.hpp"
#if defined ZMQ_IOTHREAD_POLLER_USE_SELECT

#if defined   ZMQ_HAVE_WINDOWS
#elif defined ZMQ_HAVE_HPUX
#include <sys/param.h>
#include <sys/time.h>
#include <sys/types.h>
#elif defined ZMQ_HAVE_OPENVMS
#include <sys/time.h>
#include <sys/types.h>
#elif defined ZMQ_HAVE_VXWORKS
#include <strings.h>
#include <sys/time.h>
#include <sys/types.h>
#else
#include <sys/select.h>
#endif

// ans ignore: #include "err.hpp"
// ans ignore: #include "config.hpp"
// ans ignore: #include "i_poll_events.hpp"

#include <algorithm>
#include <climits>
#include <limits>

zmq::select_t::select_t(const zmq::thread_ctx_t &ctx_)
    : worker_poller_base_t(ctx_),
#if defined ZMQ_HAVE_WINDOWS
      //  Fine as long as map is not cleared.
      _current_family_entry_it(_family_entries.end())
#else
      _max_fd(retired_fd)
#endif
{
#if defined ZMQ_HAVE_WINDOWS
    for (size_t i = 0; i < fd_family_cache_size; ++i)
        _fd_family_cache[i] = std::make_pair(retired_fd, 0);
#endif
}

zmq::select_t::~select_t() { stop_worker(); }

zmq::select_t::handle_t zmq::select_t::add_fd(fd_t fd_, i_poll_events *events_)
{
    check_thread();
    zmq_assert(fd_ != retired_fd);

    fd_entry_t fd_entry;
    fd_entry.fd     = fd_;
    fd_entry.events = events_;

#if defined ZMQ_HAVE_WINDOWS
    u_short family = get_fd_family(fd_);
    wsa_assert(family != AF_UNSPEC);
    family_entry_t &family_entry = _family_entries[family];
#else
    family_entry_t &family_entry = _family_entry;
#endif
    family_entry.fd_entries.push_back(fd_entry);
    FD_SET(fd_, &family_entry.fds_set.error);

#if !defined ZMQ_HAVE_WINDOWS
    if (fd_ > _max_fd)
        _max_fd = fd_;
#endif

    adjust_load(1);

    return fd_;
}

zmq::select_t::fd_entries_t::iterator zmq::select_t::find_fd_entry_by_handle(fd_entries_t &fd_entries_,
                                                                             handle_t      handle_)
{
    fd_entries_t::iterator fd_entry_it;
    for (fd_entry_it = fd_entries_.begin(); fd_entry_it != fd_entries_.end(); ++fd_entry_it)
        if (fd_entry_it->fd == handle_)
            break;

    return fd_entry_it;
}

void zmq::select_t::trigger_events(const fd_entries_t &fd_entries_, const fds_set_t &local_fds_set_, int event_count_)
{
    //  Size is cached to avoid iteration through recently added descriptors.
    for (fd_entries_t::size_type i = 0, size = fd_entries_.size(); i < size && event_count_ > 0; ++i)
    {
        //  fd_entries_[i] may not be stored, since calls to
        //  in_event/out_event may reallocate the vector

        if (is_retired_fd(fd_entries_[i]))
            continue;

        if (FD_ISSET(fd_entries_[i].fd, &local_fds_set_.read))
        {
            fd_entries_[i].events->in_event();
            --event_count_;
        }

        //  TODO: can the is_retired_fd be true at this point? if it
        //  was retired before, we would already have continued, and I
        //  don't see where it might have been modified
        //  And if rc == 0, we can break instead of continuing
        if (is_retired_fd(fd_entries_[i]) || event_count_ == 0)
            continue;

        if (FD_ISSET(fd_entries_[i].fd, &local_fds_set_.write))
        {
            fd_entries_[i].events->out_event();
            --event_count_;
        }

        //  TODO: same as above
        if (is_retired_fd(fd_entries_[i]) || event_count_ == 0)
            continue;

        if (FD_ISSET(fd_entries_[i].fd, &local_fds_set_.error))
        {
            fd_entries_[i].events->in_event();
            --event_count_;
        }
    }
}

#if defined ZMQ_HAVE_WINDOWS
int         zmq::select_t::try_retire_fd_entry(family_entries_t::iterator family_entry_it_, zmq::fd_t &handle_)
{
    family_entry_t &family_entry = family_entry_it_->second;

    fd_entries_t::iterator fd_entry_it = find_fd_entry_by_handle(family_entry.fd_entries, handle_);

    if (fd_entry_it == family_entry.fd_entries.end())
        return 0;

    fd_entry_t &fd_entry = *fd_entry_it;
    zmq_assert(fd_entry.fd != retired_fd);

    if (family_entry_it_ != _current_family_entry_it)
    {
        //  Family is not currently being iterated and can be safely
        //  modified in-place. So later it can be skipped without
        //  re-verifying its content.
        family_entry.fd_entries.erase(fd_entry_it);
    }
    else
    {
        //  Otherwise mark removed entries as retired. It will be cleaned up
        //  at the end of the iteration. See zmq::select_t::loop
        fd_entry.fd              = retired_fd;
        family_entry.has_retired = true;
    }
    family_entry.fds_set.remove_fd(handle_);
    return 1;
}
#endif

void zmq::select_t::rm_fd(handle_t handle_)
{
    check_thread();
    int retired = 0;
#if defined ZMQ_HAVE_WINDOWS
    u_short family = get_fd_family(handle_);
    if (family != AF_UNSPEC)
    {
        family_entries_t::iterator family_entry_it = _family_entries.find(family);

        retired += try_retire_fd_entry(family_entry_it, handle_);
    }
    else
    {
        //  get_fd_family may fail and return AF_UNSPEC if the socket was not
        //  successfully connected. In that case, we need to look for the
        //  socket in all family_entries.
        family_entries_t::iterator end = _family_entries.end();
        for (family_entries_t::iterator family_entry_it = _family_entries.begin(); family_entry_it != end;
             ++family_entry_it)
        {
            if (retired += try_retire_fd_entry(family_entry_it, handle_))
            {
                break;
            }
        }
    }
#else
    fd_entries_t::iterator fd_entry_it = find_fd_entry_by_handle(_family_entry.fd_entries, handle_);
    assert(fd_entry_it != _family_entry.fd_entries.end());

    zmq_assert(fd_entry_it->fd != retired_fd);
    fd_entry_it->fd = retired_fd;
    _family_entry.fds_set.remove_fd(handle_);

    ++retired;

    if (handle_ == _max_fd)
    {
        _max_fd = retired_fd;
        for (fd_entry_it = _family_entry.fd_entries.begin(); fd_entry_it != _family_entry.fd_entries.end();
             ++fd_entry_it)
            if (fd_entry_it->fd > _max_fd)
                _max_fd = fd_entry_it->fd;
    }

    _family_entry.has_retired = true;
#endif
    zmq_assert(retired == 1);
    adjust_load(-1);
}

void zmq::select_t::set_pollin(handle_t handle_)
{
    check_thread();
#if defined ZMQ_HAVE_WINDOWS
    u_short family = get_fd_family(handle_);
    wsa_assert(family != AF_UNSPEC);
    family_entry_t &family_entry = _family_entries[family];
#else
    family_entry_t &family_entry = _family_entry;
#endif
    FD_SET(handle_, &family_entry.fds_set.read);
}

void zmq::select_t::reset_pollin(handle_t handle_)
{
    check_thread();
#if defined ZMQ_HAVE_WINDOWS
    u_short family = get_fd_family(handle_);
    wsa_assert(family != AF_UNSPEC);
    family_entry_t &family_entry = _family_entries[family];
#else
    family_entry_t &family_entry = _family_entry;
#endif
    FD_CLR(handle_, &family_entry.fds_set.read);
}

void zmq::select_t::set_pollout(handle_t handle_)
{
    check_thread();
#if defined ZMQ_HAVE_WINDOWS
    u_short family = get_fd_family(handle_);
    wsa_assert(family != AF_UNSPEC);
    family_entry_t &family_entry = _family_entries[family];
#else
    family_entry_t &family_entry = _family_entry;
#endif
    FD_SET(handle_, &family_entry.fds_set.write);
}

void zmq::select_t::reset_pollout(handle_t handle_)
{
    check_thread();
#if defined ZMQ_HAVE_WINDOWS
    u_short family = get_fd_family(handle_);
    wsa_assert(family != AF_UNSPEC);
    family_entry_t &family_entry = _family_entries[family];
#else
    family_entry_t &family_entry = _family_entry;
#endif
    FD_CLR(handle_, &family_entry.fds_set.write);
}

void zmq::select_t::stop()
{
    check_thread();
    //  no-op... thread is stopped when no more fds or timers are registered
}

int zmq::select_t::max_fds() { return FD_SETSIZE; }

void zmq::select_t::loop()
{
    while (true)
    {
        //  Execute any due timers.
        int timeout = static_cast<int>(execute_timers());

        cleanup_retired();

#ifdef _WIN32
        if (_family_entries.empty())
        {
#else
        if (_family_entry.fd_entries.empty())
        {
#endif
            zmq_assert(get_load() == 0);

            if (timeout == 0)
                break;

            // TODO sleep for timeout
            continue;
        }

#if defined ZMQ_HAVE_OSX
        struct timeval tv = {(long)(timeout / 1000), timeout % 1000 * 1000};
#else
        struct timeval tv = {static_cast<long>(timeout / 1000), static_cast<long>(timeout % 1000 * 1000)};
#endif

#if defined ZMQ_HAVE_WINDOWS
        /*
            On Windows select does not allow to mix descriptors from different
            service providers. It seems to work for AF_INET and AF_INET6,
            but fails for AF_INET and VMCI. The workaround is to use
            WSAEventSelect and WSAWaitForMultipleEvents to wait, then use
            select to find out what actually changed. WSAWaitForMultipleEvents
            cannot be used alone, because it does not support more than 64 events
            which is not enough.

            To reduce unnecessary overhead, WSA is only used when there are more
            than one family. Moreover, AF_INET and AF_INET6 are considered the same
            family because Windows seems to handle them properly.
            See get_fd_family for details.
        */

        //  If there is just one family, there is no reason to use WSA events.
        int        rc             = 0;
        const bool use_wsa_events = _family_entries.size() > 1;
        if (use_wsa_events)
        {
            // TODO: I don't really understand why we are doing this. If any of
            // the events was signaled, we will call select for each fd_family
            // afterwards. The only benefit is if none of the events was
            // signaled, then we continue early.
            // IMHO, either WSAEventSelect/WSAWaitForMultipleEvents or select
            // should be used, but not both

            wsa_events_t wsa_events;

            for (family_entries_t::iterator family_entry_it = _family_entries.begin();
                 family_entry_it != _family_entries.end(); ++family_entry_it)
            {
                family_entry_t &family_entry = family_entry_it->second;

                for (fd_entries_t::iterator fd_entry_it = family_entry.fd_entries.begin();
                     fd_entry_it != family_entry.fd_entries.end(); ++fd_entry_it)
                {
                    fd_t fd = fd_entry_it->fd;

                    //  http://stackoverflow.com/q/35043420/188530
                    if (FD_ISSET(fd, &family_entry.fds_set.read) && FD_ISSET(fd, &family_entry.fds_set.write))
                        rc = WSAEventSelect(fd, wsa_events.events[3],
                                            FD_READ | FD_ACCEPT | FD_CLOSE | FD_WRITE | FD_CONNECT | FD_OOB);
                    else if (FD_ISSET(fd, &family_entry.fds_set.read))
                        rc = WSAEventSelect(fd, wsa_events.events[0], FD_READ | FD_ACCEPT | FD_CLOSE | FD_OOB);
                    else if (FD_ISSET(fd, &family_entry.fds_set.write))
                        rc = WSAEventSelect(fd, wsa_events.events[1], FD_WRITE | FD_CONNECT | FD_OOB);
                    else if (FD_ISSET(fd, &family_entry.fds_set.error))
                        rc = WSAEventSelect(fd, wsa_events.events[2], FD_OOB);
                    else
                        rc = 0;

                    wsa_assert(rc != SOCKET_ERROR);
                }
            }

            rc = WSAWaitForMultipleEvents(4, wsa_events.events, FALSE, timeout ? timeout : INFINITE, FALSE);
            wsa_assert(rc != (int)WSA_WAIT_FAILED);
            zmq_assert(rc != WSA_WAIT_IO_COMPLETION);

            if (rc == WSA_WAIT_TIMEOUT)
                continue;
        }

        for (_current_family_entry_it = _family_entries.begin(); _current_family_entry_it != _family_entries.end();
             ++_current_family_entry_it)
        {
            family_entry_t &family_entry = _current_family_entry_it->second;

            if (use_wsa_events)
            {
                //  There is no reason to wait again after WSAWaitForMultipleEvents.
                //  Simply collect what is ready.
                struct timeval tv_nodelay = {0, 0};
                select_family_entry(family_entry, 0, true, tv_nodelay);
            }
            else
            {
                select_family_entry(family_entry, 0, timeout > 0, tv);
            }
        }
#else
        select_family_entry(_family_entry, _max_fd + 1, timeout > 0, tv);
#endif
    }
}

void zmq::select_t::select_family_entry(family_entry_t &family_entry_, const int max_fd_, const bool use_timeout_,
                                        struct timeval &tv_)
{
    //  select will fail when run with empty sets.
    fd_entries_t &fd_entries = family_entry_.fd_entries;
    if (fd_entries.empty())
        return;

    fds_set_t local_fds_set = family_entry_.fds_set;
    int       rc =
        select(max_fd_, &local_fds_set.read, &local_fds_set.write, &local_fds_set.error, use_timeout_ ? &tv_ : NULL);

#if defined ZMQ_HAVE_WINDOWS
    wsa_assert(rc != SOCKET_ERROR);
#else
    if (rc == -1)
    {
        errno_assert(errno == EINTR);
        return;
    }
#endif

    trigger_events(fd_entries, local_fds_set, rc);

    cleanup_retired(family_entry_);
}

zmq::select_t::fds_set_t::fds_set_t()
{
    FD_ZERO(&read);
    FD_ZERO(&write);
    FD_ZERO(&error);
}

zmq::select_t::fds_set_t::fds_set_t(const fds_set_t &other_)
{
#if defined ZMQ_HAVE_WINDOWS
    // On Windows we don't need to copy the whole fd_set.
    // SOCKETS are continuous from the beginning of fd_array in fd_set.
    // We just need to copy fd_count elements of fd_array.
    // We gain huge memcpy() improvement if number of used SOCKETs is much lower than FD_SETSIZE.
    memcpy(&read, &other_.read, (char *)(other_.read.fd_array + other_.read.fd_count) - (char *)&other_.read);
    memcpy(&write, &other_.write, (char *)(other_.write.fd_array + other_.write.fd_count) - (char *)&other_.write);
    memcpy(&error, &other_.error, (char *)(other_.error.fd_array + other_.error.fd_count) - (char *)&other_.error);
#else
    memcpy(&read, &other_.read, sizeof other_.read);
    memcpy(&write, &other_.write, sizeof other_.write);
    memcpy(&error, &other_.error, sizeof other_.error);
#endif
}

zmq::select_t::fds_set_t &zmq::select_t::fds_set_t::operator=(const fds_set_t &other_)
{
#if defined ZMQ_HAVE_WINDOWS
    // On Windows we don't need to copy the whole fd_set.
    // SOCKETS are continuous from the beginning of fd_array in fd_set.
    // We just need to copy fd_count elements of fd_array.
    // We gain huge memcpy() improvement if number of used SOCKETs is much lower than FD_SETSIZE.
    memcpy(&read, &other_.read, (char *)(other_.read.fd_array + other_.read.fd_count) - (char *)&other_.read);
    memcpy(&write, &other_.write, (char *)(other_.write.fd_array + other_.write.fd_count) - (char *)&other_.write);
    memcpy(&error, &other_.error, (char *)(other_.error.fd_array + other_.error.fd_count) - (char *)&other_.error);
#else
    memcpy(&read, &other_.read, sizeof other_.read);
    memcpy(&write, &other_.write, sizeof other_.write);
    memcpy(&error, &other_.error, sizeof other_.error);
#endif
    return *this;
}

void zmq::select_t::fds_set_t::remove_fd(const fd_t &fd_)
{
    FD_CLR(fd_, &read);
    FD_CLR(fd_, &write);
    FD_CLR(fd_, &error);
}

bool zmq::select_t::cleanup_retired(family_entry_t &family_entry_)
{
    if (family_entry_.has_retired)
    {
        family_entry_.has_retired = false;
        family_entry_.fd_entries.erase(
            std::remove_if(family_entry_.fd_entries.begin(), family_entry_.fd_entries.end(), is_retired_fd),
            family_entry_.fd_entries.end());
    }
    return family_entry_.fd_entries.empty();
}

void zmq::select_t::cleanup_retired()
{
#ifdef _WIN32
    for (family_entries_t::iterator it = _family_entries.begin(); it != _family_entries.end();)
    {
        if (cleanup_retired(it->second))
            it = _family_entries.erase(it);
        else
            ++it;
    }
#else
    cleanup_retired(_family_entry);
#endif
}

bool zmq::select_t::is_retired_fd(const fd_entry_t &entry_) { return entry_.fd == retired_fd; }

zmq::select_t::family_entry_t::family_entry_t() : has_retired(false) {}

#if defined ZMQ_HAVE_WINDOWS
u_short     zmq::select_t::get_fd_family(fd_t fd_)
{
    // cache the results of determine_fd_family, as this is frequently called
    // for the same sockets, and determine_fd_family is expensive
    size_t i;
    for (i = 0; i < fd_family_cache_size; ++i)
    {
        const std::pair<fd_t, u_short> &entry = _fd_family_cache[i];
        if (entry.first == fd_)
        {
            return entry.second;
        }
        if (entry.first == retired_fd)
            break;
    }

    std::pair<fd_t, u_short> res = std::make_pair(fd_, determine_fd_family(fd_));
    if (i < fd_family_cache_size)
    {
        _fd_family_cache[i] = res;
    }
    else
    {
        // just overwrite a random entry
        // could be optimized by some LRU strategy
        _fd_family_cache[rand() % fd_family_cache_size] = res;
    }

    return res.second;
}

u_short zmq::select_t::determine_fd_family(fd_t fd_)
{
    //  Use sockaddr_storage instead of sockaddr to accommodate different structure sizes
    sockaddr_storage addr      = {0};
    int              addr_size = sizeof addr;

    int type;
    int type_length = sizeof(int);

    int rc = getsockopt(fd_, SOL_SOCKET, SO_TYPE, reinterpret_cast<char *>(&type), &type_length);

    if (rc == 0)
    {
        if (type == SOCK_DGRAM)
            return AF_INET;

        rc = getsockname(fd_, reinterpret_cast<sockaddr *>(&addr), &addr_size);

        //  AF_INET and AF_INET6 can be mixed in select
        //  TODO: If proven otherwise, should simply return addr.sa_family
        if (rc != SOCKET_ERROR)
            return addr.ss_family == AF_INET6 ? AF_INET : addr.ss_family;
    }

    return AF_UNSPEC;
}

zmq::select_t::wsa_events_t::wsa_events_t()
{
    events[0] = WSACreateEvent();
    wsa_assert(events[0] != WSA_INVALID_EVENT);
    events[1] = WSACreateEvent();
    wsa_assert(events[1] != WSA_INVALID_EVENT);
    events[2] = WSACreateEvent();
    wsa_assert(events[2] != WSA_INVALID_EVENT);
    events[3] = WSACreateEvent();
    wsa_assert(events[3] != WSA_INVALID_EVENT);
}

zmq::select_t::wsa_events_t::~wsa_events_t()
{
    wsa_assert(WSACloseEvent(events[0]));
    wsa_assert(WSACloseEvent(events[1]));
    wsa_assert(WSACloseEvent(events[2]));
    wsa_assert(WSACloseEvent(events[3]));
}
#endif

#endif

//========= end of select.cpp ============

//========= begin of server.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "server.hpp"
// ans ignore: #include "pipe.hpp"
// ans ignore: #include "wire.hpp"
// ans ignore: #include "random.hpp"
// ans ignore: #include "likely.hpp"
// ans ignore: #include "err.hpp"

zmq::server_t::server_t(class ctx_t *parent_, uint32_t tid_, int sid_)
    : socket_base_t(parent_, tid_, sid_, true), _next_routing_id(generate_random())
{
    options.type = ZMQ_SERVER;
}

zmq::server_t::~server_t() { zmq_assert(_out_pipes.empty()); }

void zmq::server_t::xattach_pipe(pipe_t *pipe_, bool subscribe_to_all_, bool locally_initiated_)
{
    LIBZMQ_UNUSED(subscribe_to_all_);
    LIBZMQ_UNUSED(locally_initiated_);

    zmq_assert(pipe_);

    uint32_t routing_id = _next_routing_id++;
    if (!routing_id)
        routing_id = _next_routing_id++; //  Never use Routing ID zero

    pipe_->set_server_socket_routing_id(routing_id);
    //  Add the record into output pipes lookup table
    outpipe_t outpipe = {pipe_, true};
    bool      ok      = _out_pipes.ZMQ_MAP_INSERT_OR_EMPLACE(routing_id, outpipe).second;
    zmq_assert(ok);

    _fq.attach(pipe_);
}

void zmq::server_t::xpipe_terminated(pipe_t *pipe_)
{
    out_pipes_t::iterator it = _out_pipes.find(pipe_->get_server_socket_routing_id());
    zmq_assert(it != _out_pipes.end());
    _out_pipes.erase(it);
    _fq.pipe_terminated(pipe_);
}

void zmq::server_t::xread_activated(pipe_t *pipe_) { _fq.activated(pipe_); }

void zmq::server_t::xwrite_activated(pipe_t *pipe_)
{
    const out_pipes_t::iterator end = _out_pipes.end();
    out_pipes_t::iterator       it;
    for (it = _out_pipes.begin(); it != end; ++it)
        if (it->second.pipe == pipe_)
            break;

    zmq_assert(it != _out_pipes.end());
    zmq_assert(!it->second.active);
    it->second.active = true;
}

int zmq::server_t::xsend(msg_t *msg_)
{
    //  SERVER sockets do not allow multipart data (ZMQ_SNDMORE)
    if (msg_->flags() & msg_t::more)
    {
        errno = EINVAL;
        return -1;
    }
    //  Find the pipe associated with the routing stored in the message.
    uint32_t              routing_id = msg_->get_routing_id();
    out_pipes_t::iterator it         = _out_pipes.find(routing_id);

    if (it != _out_pipes.end())
    {
        if (!it->second.pipe->check_write())
        {
            it->second.active = false;
            errno             = EAGAIN;
            return -1;
        }
    }
    else
    {
        errno = EHOSTUNREACH;
        return -1;
    }

    //  Message might be delivered over inproc, so we reset routing id
    int rc = msg_->reset_routing_id();
    errno_assert(rc == 0);

    bool ok = it->second.pipe->write(msg_);
    if (unlikely(!ok))
    {
        // Message failed to send - we must close it ourselves.
        rc = msg_->close();
        errno_assert(rc == 0);
    }
    else
        it->second.pipe->flush();

    //  Detach the message from the data buffer.
    rc = msg_->init();
    errno_assert(rc == 0);

    return 0;
}

int zmq::server_t::xrecv(msg_t *msg_)
{
    pipe_t *pipe = NULL;
    int     rc   = _fq.recvpipe(msg_, &pipe);

    // Drop any messages with more flag
    while (rc == 0 && msg_->flags() & msg_t::more)
    {
        // drop all frames of the current multi-frame message
        rc = _fq.recvpipe(msg_, NULL);

        while (rc == 0 && msg_->flags() & msg_t::more)
            rc = _fq.recvpipe(msg_, NULL);

        // get the new message
        if (rc == 0)
            rc = _fq.recvpipe(msg_, &pipe);
    }

    if (rc != 0)
        return rc;

    zmq_assert(pipe != NULL);

    uint32_t routing_id = pipe->get_server_socket_routing_id();
    msg_->set_routing_id(routing_id);

    return 0;
}

bool zmq::server_t::xhas_in() { return _fq.has_in(); }

bool zmq::server_t::xhas_out()
{
    //  In theory, SERVER socket is always ready for writing. Whether actual
    //  attempt to write succeeds depends on which pipe the message is going
    //  to be routed to.
    return true;
}

//========= end of server.cpp ============

//========= begin of session_base.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "i_engine.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "pipe.hpp"
// ans ignore: #include "likely.hpp"
// ans ignore: #include "tcp_connecter.hpp"
// ans ignore: #include "ipc_connecter.hpp"
// ans ignore: #include "tipc_connecter.hpp"
// ans ignore: #include "socks_connecter.hpp"
// ans ignore: #include "vmci_connecter.hpp"
// ans ignore: #include "pgm_sender.hpp"
// ans ignore: #include "pgm_receiver.hpp"
// ans ignore: #include "address.hpp"
// ans ignore: #include "norm_engine.hpp"
// ans ignore: #include "udp_engine.hpp"

// ans ignore: #include "ctx.hpp"
// ans ignore: #include "req.hpp"
// ans ignore: #include "radio.hpp"
// ans ignore: #include "dish.hpp"

zmq::session_base_t *zmq::session_base_t::create(class io_thread_t *io_thread_, bool active_,
                                                 class socket_base_t *socket_, const options_t &options_,
                                                 address_t *addr_)
{
    session_base_t *s = NULL;
    switch (options_.type)
    {
    case ZMQ_REQ:
        s = new (std::nothrow) req_session_t(io_thread_, active_, socket_, options_, addr_);
        break;
    case ZMQ_RADIO:
        s = new (std::nothrow) radio_session_t(io_thread_, active_, socket_, options_, addr_);
        break;
    case ZMQ_DISH:
        s = new (std::nothrow) dish_session_t(io_thread_, active_, socket_, options_, addr_);
        break;
    case ZMQ_DEALER:
    case ZMQ_REP:
    case ZMQ_ROUTER:
    case ZMQ_PUB:
    case ZMQ_XPUB:
    case ZMQ_SUB:
    case ZMQ_XSUB:
    case ZMQ_PUSH:
    case ZMQ_PULL:
    case ZMQ_PAIR:
    case ZMQ_STREAM:
    case ZMQ_SERVER:
    case ZMQ_CLIENT:
    case ZMQ_GATHER:
    case ZMQ_SCATTER:
    case ZMQ_DGRAM:
        s = new (std::nothrow) session_base_t(io_thread_, active_, socket_, options_, addr_);
        break;
    default:
        errno = EINVAL;
        return NULL;
    }
    alloc_assert(s);
    return s;
}

zmq::session_base_t::session_base_t(class io_thread_t *io_thread_, bool active_, class socket_base_t *socket_,
                                    const options_t &options_, address_t *addr_)
    : own_t(io_thread_, options_), io_object_t(io_thread_), _active(active_), _pipe(NULL), _zap_pipe(NULL),
      _incomplete_in(false), _pending(false), _engine(NULL), _socket(socket_), _io_thread(io_thread_),
      _has_linger_timer(false), _addr(addr_)
{
}

const zmq::endpoint_uri_pair_t &zmq::session_base_t::get_endpoint() const { return _engine->get_endpoint(); }

zmq::session_base_t::~session_base_t()
{
    zmq_assert(!_pipe);
    zmq_assert(!_zap_pipe);

    //  If there's still a pending linger timer, remove it.
    if (_has_linger_timer)
    {
        cancel_timer(linger_timer_id);
        _has_linger_timer = false;
    }

    //  Close the engine.
    if (_engine)
        _engine->terminate();

    LIBZMQ_DELETE(_addr);
}

void zmq::session_base_t::attach_pipe(pipe_t *pipe_)
{
    zmq_assert(!is_terminating());
    zmq_assert(!_pipe);
    zmq_assert(pipe_);
    _pipe = pipe_;
    _pipe->set_event_sink(this);
}

int zmq::session_base_t::pull_msg(msg_t *msg_)
{
    if (!_pipe || !_pipe->read(msg_))
    {
        errno = EAGAIN;
        return -1;
    }

    _incomplete_in = (msg_->flags() & msg_t::more) != 0;

    return 0;
}

int zmq::session_base_t::push_msg(msg_t *msg_)
{
    //  pass subscribe/cancel to the sockets
    if ((msg_->flags() & msg_t::command) && !msg_->is_subscribe() && !msg_->is_cancel())
        return 0;
    if (_pipe && _pipe->write(msg_))
    {
        int rc = msg_->init();
        errno_assert(rc == 0);
        return 0;
    }

    errno = EAGAIN;
    return -1;
}

int zmq::session_base_t::read_zap_msg(msg_t *msg_)
{
    if (_zap_pipe == NULL)
    {
        errno = ENOTCONN;
        return -1;
    }

    if (!_zap_pipe->read(msg_))
    {
        errno = EAGAIN;
        return -1;
    }

    return 0;
}

int zmq::session_base_t::write_zap_msg(msg_t *msg_)
{
    if (_zap_pipe == NULL || !_zap_pipe->write(msg_))
    {
        errno = ENOTCONN;
        return -1;
    }

    if ((msg_->flags() & msg_t::more) == 0)
        _zap_pipe->flush();

    const int rc = msg_->init();
    errno_assert(rc == 0);
    return 0;
}

void zmq::session_base_t::reset() {}

void zmq::session_base_t::flush()
{
    if (_pipe)
        _pipe->flush();
}

void zmq::session_base_t::rollback()
{
    if (_pipe)
        _pipe->rollback();
}

void zmq::session_base_t::clean_pipes()
{
    zmq_assert(_pipe != NULL);

    //  Get rid of half-processed messages in the out pipe. Flush any
    //  unflushed messages upstream.
    _pipe->rollback();
    _pipe->flush();

    //  Remove any half-read message from the in pipe.
    while (_incomplete_in)
    {
        msg_t msg;
        int   rc = msg.init();
        errno_assert(rc == 0);
        rc = pull_msg(&msg);
        errno_assert(rc == 0);
        rc = msg.close();
        errno_assert(rc == 0);
    }
}

void zmq::session_base_t::pipe_terminated(pipe_t *pipe_)
{
    // Drop the reference to the deallocated pipe if required.
    zmq_assert(pipe_ == _pipe || pipe_ == _zap_pipe || _terminating_pipes.count(pipe_) == 1);

    if (pipe_ == _pipe)
    {
        // If this is our current pipe, remove it
        _pipe = NULL;
        if (_has_linger_timer)
        {
            cancel_timer(linger_timer_id);
            _has_linger_timer = false;
        }
    }
    else if (pipe_ == _zap_pipe)
        _zap_pipe = NULL;
    else
        // Remove the pipe from the detached pipes set
        _terminating_pipes.erase(pipe_);

    if (!is_terminating() && options.raw_socket)
    {
        if (_engine)
        {
            _engine->terminate();
            _engine = NULL;
        }
        terminate();
    }

    //  If we are waiting for pending messages to be sent, at this point
    //  we are sure that there will be no more messages and we can proceed
    //  with termination safely.
    if (_pending && !_pipe && !_zap_pipe && _terminating_pipes.empty())
    {
        _pending = false;
        own_t::process_term(0);
    }
}

void zmq::session_base_t::read_activated(pipe_t *pipe_)
{
    // Skip activating if we're detaching this pipe
    if (unlikely(pipe_ != _pipe && pipe_ != _zap_pipe))
    {
        zmq_assert(_terminating_pipes.count(pipe_) == 1);
        return;
    }

    if (unlikely(_engine == NULL))
    {
        _pipe->check_read();
        return;
    }

    if (likely(pipe_ == _pipe))
        _engine->restart_output();
    else
    {
        // i.e. pipe_ == zap_pipe
        _engine->zap_msg_available();
    }
}

void zmq::session_base_t::write_activated(pipe_t *pipe_)
{
    // Skip activating if we're detaching this pipe
    if (_pipe != pipe_)
    {
        zmq_assert(_terminating_pipes.count(pipe_) == 1);
        return;
    }

    if (_engine)
        _engine->restart_input();
}

void zmq::session_base_t::hiccuped(pipe_t *)
{
    //  Hiccups are always sent from session to socket, not the other
    //  way round.
    zmq_assert(false);
}

zmq::socket_base_t *zmq::session_base_t::get_socket() { return _socket; }

void zmq::session_base_t::process_plug()
{
    if (_active)
        start_connecting(false);
}

//  This functions can return 0 on success or -1 and errno=ECONNREFUSED if ZAP
//  is not setup (IE: inproc://zeromq.zap.01 does not exist in the same context)
//  or it aborts on any other error. In other words, either ZAP is not
//  configured or if it is configured it MUST be configured correctly and it
//  MUST work, otherwise authentication cannot be guaranteed and it would be a
//  security flaw.
int zmq::session_base_t::zap_connect()
{
    if (_zap_pipe != NULL)
        return 0;

    endpoint_t peer = find_endpoint("inproc://zeromq.zap.01");
    if (peer.socket == NULL)
    {
        errno = ECONNREFUSED;
        return -1;
    }
    zmq_assert(peer.options.type == ZMQ_REP || peer.options.type == ZMQ_ROUTER || peer.options.type == ZMQ_SERVER);

    //  Create a bi-directional pipe that will connect
    //  session with zap socket.
    object_t *parents[2]   = {this, peer.socket};
    pipe_t *  new_pipes[2] = {NULL, NULL};
    int       hwms[2]      = {0, 0};
    bool      conflates[2] = {false, false};
    int       rc           = pipepair(parents, new_pipes, hwms, conflates);
    errno_assert(rc == 0);

    //  Attach local end of the pipe to this socket object.
    _zap_pipe = new_pipes[0];
    _zap_pipe->set_nodelay();
    _zap_pipe->set_event_sink(this);

    send_bind(peer.socket, new_pipes[1], false);

    //  Send empty routing id if required by the peer.
    if (peer.options.recv_routing_id)
    {
        msg_t id;
        rc = id.init();
        errno_assert(rc == 0);
        id.set_flags(msg_t::routing_id);
        bool ok = _zap_pipe->write(&id);
        zmq_assert(ok);
        _zap_pipe->flush();
    }

    return 0;
}

bool zmq::session_base_t::zap_enabled() { return (options.mechanism != ZMQ_NULL || !options.zap_domain.empty()); }

void zmq::session_base_t::process_attach(i_engine *engine_)
{
    zmq_assert(engine_ != NULL);

    //  Create the pipe if it does not exist yet.
    if (!_pipe && !is_terminating())
    {
        object_t *parents[2] = {this, _socket};
        pipe_t *  pipes[2]   = {NULL, NULL};

        const bool conflate = get_effective_conflate_option(options);

        int  hwms[2]      = {conflate ? -1 : options.rcvhwm, conflate ? -1 : options.sndhwm};
        bool conflates[2] = {conflate, conflate};
        int  rc           = pipepair(parents, pipes, hwms, conflates);
        errno_assert(rc == 0);

        //  Plug the local end of the pipe.
        pipes[0]->set_event_sink(this);

        //  Remember the local end of the pipe.
        zmq_assert(!_pipe);
        _pipe = pipes[0];

        //  The endpoints strings are not set on bind, set them here so that
        //  events can use them.
        pipes[0]->set_endpoint_pair(engine_->get_endpoint());
        pipes[1]->set_endpoint_pair(engine_->get_endpoint());

        //  Ask socket to plug into the remote end of the pipe.
        send_bind(_socket, pipes[1]);
    }

    //  Plug in the engine.
    zmq_assert(!_engine);
    _engine = engine_;
    _engine->plug(_io_thread, this);
}

void zmq::session_base_t::engine_error(zmq::stream_engine_t::error_reason_t reason_)
{
    //  Engine is dead. Let's forget about it.
    _engine = NULL;

    //  Remove any half-done messages from the pipes.
    if (_pipe)
        clean_pipes();

    zmq_assert(reason_ == stream_engine_t::connection_error || reason_ == stream_engine_t::timeout_error ||
               reason_ == stream_engine_t::protocol_error);

    switch (reason_)
    {
    case stream_engine_t::timeout_error:
        /* FALLTHROUGH */
    case stream_engine_t::connection_error:
        if (_active)
        {
            reconnect();
            break;
        }
        /* FALLTHROUGH */
    case stream_engine_t::protocol_error:
        if (_pending)
        {
            if (_pipe)
                _pipe->terminate(false);
            if (_zap_pipe)
                _zap_pipe->terminate(false);
        }
        else
        {
            terminate();
        }
        break;
    }

    //  Just in case there's only a delimiter in the pipe.
    if (_pipe)
        _pipe->check_read();

    if (_zap_pipe)
        _zap_pipe->check_read();
}

void zmq::session_base_t::process_term(int linger_)
{
    zmq_assert(!_pending);

    //  If the termination of the pipe happens before the term command is
    //  delivered there's nothing much to do. We can proceed with the
    //  standard termination immediately.
    if (!_pipe && !_zap_pipe && _terminating_pipes.empty())
    {
        own_t::process_term(0);
        return;
    }

    _pending = true;

    if (_pipe != NULL)
    {
        //  If there's finite linger value, delay the termination.
        //  If linger is infinite (negative) we don't even have to set
        //  the timer.
        if (linger_ > 0)
        {
            zmq_assert(!_has_linger_timer);
            add_timer(linger_, linger_timer_id);
            _has_linger_timer = true;
        }

        //  Start pipe termination process. Delay the termination till all messages
        //  are processed in case the linger time is non-zero.
        _pipe->terminate(linger_ != 0);

        //  TODO: Should this go into pipe_t::terminate ?
        //  In case there's no engine and there's only delimiter in the
        //  pipe it wouldn't be ever read. Thus we check for it explicitly.
        if (!_engine)
            _pipe->check_read();
    }

    if (_zap_pipe != NULL)
        _zap_pipe->terminate(false);
}

void zmq::session_base_t::timer_event(int id_)
{
    //  Linger period expired. We can proceed with termination even though
    //  there are still pending messages to be sent.
    zmq_assert(id_ == linger_timer_id);
    _has_linger_timer = false;

    //  Ask pipe to terminate even though there may be pending messages in it.
    zmq_assert(_pipe);
    _pipe->terminate(false);
}

void zmq::session_base_t::reconnect()
{
    //  For delayed connect situations, terminate the pipe
    //  and reestablish later on
    if (_pipe && options.immediate == 1 && _addr->protocol != "pgm" && _addr->protocol != "epgm" &&
        _addr->protocol != "norm" && _addr->protocol != protocol_name::udp)
    {
        _pipe->hiccup();
        _pipe->terminate(false);
        _terminating_pipes.insert(_pipe);
        _pipe = NULL;

        if (_has_linger_timer)
        {
            cancel_timer(linger_timer_id);
            _has_linger_timer = false;
        }
    }

    reset();

    //  Reconnect.
    if (options.reconnect_ivl != -1)
        start_connecting(true);
    else
    {
        std::string *ep = new (std::string);
        _addr->to_string(*ep);
        send_term_endpoint(_socket, ep);
    }

    //  For subscriber sockets we hiccup the inbound pipe, which will cause
    //  the socket object to resend all the subscriptions.
    if (_pipe && (options.type == ZMQ_SUB || options.type == ZMQ_XSUB || options.type == ZMQ_DISH))
        _pipe->hiccup();
}

zmq::session_base_t::connecter_factory_entry_t zmq::session_base_t::_connecter_factories[] = {
    connecter_factory_entry_t(protocol_name::tcp, &zmq::session_base_t::create_connecter_tcp),
#if !defined ZMQ_HAVE_WINDOWS && !defined ZMQ_HAVE_OPENVMS && !defined ZMQ_HAVE_VXWORKS
    connecter_factory_entry_t(protocol_name::ipc, &zmq::session_base_t::create_connecter_ipc),
#endif
#if defined ZMQ_HAVE_TIPC
    connecter_factory_entry_t(protocol_name::tipc, &zmq::session_base_t::create_connecter_tipc),
#endif
#if defined ZMQ_HAVE_VMCI
    connecter_factory_entry_t(protocol_name::vmci, &zmq::session_base_t::create_connecter_vmci),
#endif
};

zmq::session_base_t::connecter_factory_map_t zmq::session_base_t::_connecter_factories_map(
    _connecter_factories, _connecter_factories + sizeof(_connecter_factories) / sizeof(_connecter_factories[0]));

zmq::session_base_t::start_connecting_entry_t zmq::session_base_t::_start_connecting_entries[] = {
    start_connecting_entry_t(protocol_name::udp, &zmq::session_base_t::start_connecting_udp),
#if defined ZMQ_HAVE_OPENPGM
    start_connecting_entry_t("pgm", &zmq::session_base_t::start_connecting_pgm),
    start_connecting_entry_t("epgm", &zmq::session_base_t::start_connecting_pgm),
#endif
#if defined ZMQ_HAVE_NORM
    start_connecting_entry_t("norm", &zmq::session_base_t::start_connecting_norm),
#endif
};

zmq::session_base_t::start_connecting_map_t zmq::session_base_t::_start_connecting_map(
    _start_connecting_entries,
    _start_connecting_entries + sizeof(_start_connecting_entries) / sizeof(_start_connecting_entries[0]));

void zmq::session_base_t::start_connecting(bool wait_)
{
    zmq_assert(_active);

    //  Choose I/O thread to run connecter in. Given that we are already
    //  running in an I/O thread, there must be at least one available.
    io_thread_t *io_thread = choose_io_thread(options.affinity);
    zmq_assert(io_thread);

    //  Create the connecter object.
    const connecter_factory_map_t::const_iterator connecter_factories_it =
        _connecter_factories_map.find(_addr->protocol);
    if (connecter_factories_it != _connecter_factories_map.end())
    {
        own_t *connecter = (this->*connecter_factories_it->second)(io_thread, wait_);

        alloc_assert(connecter);
        launch_child(connecter);
        return;
    }
    const start_connecting_map_t::const_iterator start_connecting_it = _start_connecting_map.find(_addr->protocol);
    if (start_connecting_it != _start_connecting_map.end())
    {
        (this->*start_connecting_it->second)(io_thread);
        return;
    }

    zmq_assert(false);
}

#if defined ZMQ_HAVE_VMCI
zmq::own_t *zmq::session_base_t::create_connecter_vmci(io_thread_t *io_thread_, bool wait_)
{
    return new (std::nothrow) vmci_connecter_t(io_thread_, this, options, _addr, wait_);
}
#endif

#if defined ZMQ_HAVE_TIPC
zmq::own_t *zmq::session_base_t::create_connecter_tipc(io_thread_t *io_thread_, bool wait_)
{
    return new (std::nothrow) tipc_connecter_t(io_thread_, this, options, _addr, wait_);
}
#endif

#if !defined ZMQ_HAVE_WINDOWS && !defined ZMQ_HAVE_OPENVMS && !defined ZMQ_HAVE_VXWORKS
zmq::own_t *zmq::session_base_t::create_connecter_ipc(io_thread_t *io_thread_, bool wait_)
{
    return new (std::nothrow) ipc_connecter_t(io_thread_, this, options, _addr, wait_);
}
#endif

zmq::own_t *zmq::session_base_t::create_connecter_tcp(io_thread_t *io_thread_, bool wait_)
{
    if (!options.socks_proxy_address.empty())
    {
        address_t *proxy_address =
            new (std::nothrow) address_t(protocol_name::tcp, options.socks_proxy_address, this->get_ctx());
        alloc_assert(proxy_address);
        socks_connecter_t *connecter =
            new (std::nothrow) socks_connecter_t(io_thread_, this, options, _addr, proxy_address, wait_);
        alloc_assert(connecter);
        if (!options.socks_proxy_username.empty())
        {
            connecter->set_auth_method_basic(options.socks_proxy_username, options.socks_proxy_password);
        }
        return connecter;
    }
    return new (std::nothrow) tcp_connecter_t(io_thread_, this, options, _addr, wait_);
}

#ifdef ZMQ_HAVE_OPENPGM
void zmq::session_base_t::start_connecting_pgm(io_thread_t *io_thread_)
{
    zmq_assert(options.type == ZMQ_PUB || options.type == ZMQ_XPUB || options.type == ZMQ_SUB ||
               options.type == ZMQ_XSUB);

    //  For EPGM transport with UDP encapsulation of PGM is used.
    bool const udp_encapsulation = _addr->protocol == "epgm";

    //  At this point we'll create message pipes to the session straight
    //  away. There's no point in delaying it as no concept of 'connect'
    //  exists with PGM anyway.
    if (options.type == ZMQ_PUB || options.type == ZMQ_XPUB)
    {
        //  PGM sender.
        pgm_sender_t *pgm_sender = new (std::nothrow) pgm_sender_t(io_thread_, options);
        alloc_assert(pgm_sender);

        int rc = pgm_sender->init(udp_encapsulation, _addr->address.c_str());
        errno_assert(rc == 0);

        send_attach(this, pgm_sender);
    }
    else
    {
        //  PGM receiver.
        pgm_receiver_t *pgm_receiver = new (std::nothrow) pgm_receiver_t(io_thread_, options);
        alloc_assert(pgm_receiver);

        int rc = pgm_receiver->init(udp_encapsulation, _addr->address.c_str());
        errno_assert(rc == 0);

        send_attach(this, pgm_receiver);
    }
}
#endif

#ifdef ZMQ_HAVE_NORM
void zmq::session_base_t::start_connecting_norm(io_thread_t *io_thread_)
{
    //  At this point we'll create message pipes to the session straight
    //  away. There's no point in delaying it as no concept of 'connect'
    //  exists with NORM anyway.
    if (options.type == ZMQ_PUB || options.type == ZMQ_XPUB)
    {
        //  NORM sender.
        norm_engine_t *norm_sender = new (std::nothrow) norm_engine_t(io_thread_, options);
        alloc_assert(norm_sender);

        int rc = norm_sender->init(_addr->address.c_str(), true, false);
        errno_assert(rc == 0);

        send_attach(this, norm_sender);
    }
    else
    { // ZMQ_SUB or ZMQ_XSUB

        //  NORM receiver.
        norm_engine_t *norm_receiver = new (std::nothrow) norm_engine_t(io_thread_, options);
        alloc_assert(norm_receiver);

        int rc = norm_receiver->init(_addr->address.c_str(), false, true);
        errno_assert(rc == 0);

        send_attach(this, norm_receiver);
    }
}
#endif

void zmq::session_base_t::start_connecting_udp(io_thread_t * /*io_thread_*/)
{
    zmq_assert(options.type == ZMQ_DISH || options.type == ZMQ_RADIO || options.type == ZMQ_DGRAM);

    udp_engine_t *engine = new (std::nothrow) udp_engine_t(options);
    alloc_assert(engine);

    const bool recv = options.type == ZMQ_DISH || options.type == ZMQ_DGRAM;
    const bool send = options.type == ZMQ_RADIO || options.type == ZMQ_DGRAM;

    const int rc = engine->init(_addr, send, recv);
    errno_assert(rc == 0);

    send_attach(this, engine);
}

//========= end of session_base.cpp ============

//========= begin of signaler.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "poller.hpp"
// ans ignore: #include "polling_util.hpp"

#if defined  ZMQ_POLL_BASED_ON_POLL
#if !defined ZMQ_HAVE_WINDOWS && !defined ZMQ_HAVE_AIX
#include <poll.h>
#endif
#elif defined ZMQ_POLL_BASED_ON_SELECT
#if defined   ZMQ_HAVE_WINDOWS
#elif defined ZMQ_HAVE_HPUX
#include <sys/param.h>
#include <sys/time.h>
#include <sys/types.h>
#elif defined ZMQ_HAVE_OPENVMS
#include <sys/time.h>
#include <sys/types.h>
#elif defined ZMQ_HAVE_VXWORKS
#include <sockLib.h>
#include <strings.h>
#include <sys/time.h>
#include <sys/types.h>
#else
#include <sys/select.h>
#endif
#endif

// ans ignore: #include "signaler.hpp"
// ans ignore: #include "likely.hpp"
// ans ignore: #include "stdint.hpp"
// ans ignore: #include "config.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "fd.hpp"
// ans ignore: #include "ip.hpp"
// ans ignore: #include "tcp.hpp"

#if !defined ZMQ_HAVE_WINDOWS
#include <netinet/tcp.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <unistd.h>
#endif

#if !defined(ZMQ_HAVE_WINDOWS)
// Helper to sleep for specific number of milliseconds (or until signal)
//
static int sleep_ms(unsigned int ms_)
{
    if (ms_ == 0)
        return 0;
#if defined ZMQ_HAVE_ANDROID
    usleep(ms_ * 1000);
    return 0;
#elif defined ZMQ_HAVE_VXWORKS
    struct timespec ns_;
    ns_.tv_sec = ms_ / 1000;
    ns_.tv_nsec = ms_ % 1000 * 1000000;
    return nanosleep(&ns_, 0);
#else
    return usleep(ms_ * 1000);
#endif
}

// Helper to wait on close(), for non-blocking sockets, until it completes
// If EAGAIN is received, will sleep briefly (1-100ms) then try again, until
// the overall timeout is reached.
//
static int close_wait_ms(int fd_, unsigned int max_ms_ = 2000)
{
    unsigned int       ms_so_far   = 0;
    const unsigned int min_step_ms = 1;
    const unsigned int max_step_ms = 100;
    const unsigned int step_ms     = std::min(std::max(min_step_ms, max_ms_ / 10), max_step_ms);

    int rc = 0; // do not sleep on first attempt
    do
    {
        if (rc == -1 && errno == EAGAIN)
        {
            sleep_ms(step_ms);
            ms_so_far += step_ms;
        }
        rc = close(fd_);
    } while (ms_so_far < max_ms_ && rc == -1 && errno == EAGAIN);

    return rc;
}
#endif

zmq::signaler_t::signaler_t()
{
    //  Create the socketpair for signaling.
    if (make_fdpair(&_r, &_w) == 0)
    {
        unblock_socket(_w);
        unblock_socket(_r);
    }
#ifdef HAVE_FORK
    pid = getpid();
#endif
}

// This might get run after some part of construction failed, leaving one or
// both of _r and _w retired_fd.
zmq::signaler_t::~signaler_t()
{
#if defined ZMQ_HAVE_EVENTFD
    if (_r == retired_fd)
        return;
    int rc = close_wait_ms(_r);
    errno_assert(rc == 0);
#elif defined ZMQ_HAVE_WINDOWS
    if (_w != retired_fd)
    {
        const struct linger so_linger = {1, 0};
        int rc = setsockopt(_w, SOL_SOCKET, SO_LINGER, reinterpret_cast<const char *>(&so_linger), sizeof so_linger);
        //  Only check shutdown if WSASTARTUP was previously done
        if (rc == 0 || WSAGetLastError() != WSANOTINITIALISED)
        {
            wsa_assert(rc != SOCKET_ERROR);
            rc = closesocket(_w);
            wsa_assert(rc != SOCKET_ERROR);
            if (_r == retired_fd)
                return;
            rc = closesocket(_r);
            wsa_assert(rc != SOCKET_ERROR);
        }
    }
#else
    if (_w != retired_fd)
    {
        int rc = close_wait_ms(_w);
        errno_assert(rc == 0);
    }
    if (_r != retired_fd)
    {
        int rc = close_wait_ms(_r);
        errno_assert(rc == 0);
    }
#endif
}

zmq::fd_t zmq::signaler_t::get_fd() const { return _r; }

void zmq::signaler_t::send()
{
#if defined HAVE_FORK
    if (unlikely(pid != getpid()))
    {
        // printf("Child process %d signaler_t::send returning without sending #1\n", getpid());
        return; // do not send anything in forked child context
    }
#endif
#if defined ZMQ_HAVE_EVENTFD
    const uint64_t inc = 1;
    ssize_t        sz  = write(_w, &inc, sizeof(inc));
    errno_assert(sz == sizeof(inc));
#elif defined ZMQ_HAVE_WINDOWS
    unsigned char dummy  = 0;
    const int     nbytes = ::send(_w, reinterpret_cast<char *>(&dummy), sizeof(dummy), 0);
    wsa_assert(nbytes != SOCKET_ERROR);
    zmq_assert(nbytes == sizeof(dummy));
#elif defined ZMQ_HAVE_VXWORKS
    unsigned char dummy = 0;
    while (true)
    {
        ssize_t nbytes = ::send(_w, (char *)&dummy, sizeof(dummy), 0);
        if (unlikely(nbytes == -1 && errno == EINTR))
            continue;
#if defined(HAVE_FORK)
        if (unlikely(pid != getpid()))
        {
            // printf("Child process %d signaler_t::send returning without sending #2\n", getpid());
            errno = EINTR;
            break;
        }
#endif
        zmq_assert(nbytes == sizeof dummy);
        break;
    }
#else
    unsigned char dummy = 0;
    while (true)
    {
        ssize_t nbytes = ::send(_w, &dummy, sizeof(dummy), 0);
        if (unlikely(nbytes == -1 && errno == EINTR))
            continue;
#if defined(HAVE_FORK)
        if (unlikely(pid != getpid()))
        {
            // printf("Child process %d signaler_t::send returning without sending #2\n", getpid());
            errno = EINTR;
            break;
        }
#endif
        zmq_assert(nbytes == sizeof dummy);
        break;
    }
#endif
}

int zmq::signaler_t::wait(int timeout_)
{
#ifdef HAVE_FORK
    if (unlikely(pid != getpid()))
    {
        // we have forked and the file descriptor is closed. Emulate an interrupt
        // response.
        // printf("Child process %d signaler_t::wait returning simulating interrupt #1\n", getpid());
        errno = EINTR;
        return -1;
    }
#endif

#ifdef ZMQ_POLL_BASED_ON_POLL
    struct pollfd pfd;
    pfd.fd       = _r;
    pfd.events   = POLLIN;
    const int rc = poll(&pfd, 1, timeout_);
    if (unlikely(rc < 0))
    {
        errno_assert(errno == EINTR);
        return -1;
    }
    if (unlikely(rc == 0))
    {
        errno = EAGAIN;
        return -1;
    }
#ifdef HAVE_FORK
    if (unlikely(pid != getpid()))
    {
        // we have forked and the file descriptor is closed. Emulate an interrupt
        // response.
        // printf("Child process %d signaler_t::wait returning simulating interrupt #2\n", getpid());
        errno = EINTR;
        return -1;
    }
#endif
    zmq_assert(rc == 1);
    zmq_assert(pfd.revents & POLLIN);
    return 0;

#elif defined ZMQ_POLL_BASED_ON_SELECT

    optimized_fd_set_t fds(1);
    FD_ZERO(fds.get());
    FD_SET(_r, fds.get());
    struct timeval timeout;
    if (timeout_ >= 0)
    {
        timeout.tv_sec  = timeout_ / 1000;
        timeout.tv_usec = timeout_ % 1000 * 1000;
    }
#ifdef ZMQ_HAVE_WINDOWS
    int rc = select(0, fds.get(), NULL, NULL, timeout_ >= 0 ? &timeout : NULL);
    wsa_assert(rc != SOCKET_ERROR);
#else
    int rc = select(_r + 1, fds.get(), NULL, NULL, timeout_ >= 0 ? &timeout : NULL);
    if (unlikely(rc < 0))
    {
        errno_assert(errno == EINTR);
        return -1;
    }
#endif
    if (unlikely(rc == 0))
    {
        errno = EAGAIN;
        return -1;
    }
    zmq_assert(rc == 1);
    return 0;

#else
#error
#endif
}

void zmq::signaler_t::recv()
{
//  Attempt to read a signal.
#if defined ZMQ_HAVE_EVENTFD
    uint64_t dummy;
    ssize_t  sz = read(_r, &dummy, sizeof(dummy));
    errno_assert(sz == sizeof(dummy));

    //  If we accidentally grabbed the next signal(s) along with the current
    //  one, return it back to the eventfd object.
    if (unlikely(dummy > 1))
    {
        const uint64_t inc = dummy - 1;
        ssize_t        sz2 = write(_w, &inc, sizeof(inc));
        errno_assert(sz2 == sizeof(inc));
        return;
    }

    zmq_assert(dummy == 1);
#else
    unsigned char dummy;
#if defined   ZMQ_HAVE_WINDOWS
    const int     nbytes = ::recv(_r, reinterpret_cast<char *>(&dummy), sizeof(dummy), 0);
    wsa_assert(nbytes != SOCKET_ERROR);
#elif defined ZMQ_HAVE_VXWORKS
    ssize_t nbytes = ::recv(_r, (char *)&dummy, sizeof(dummy), 0);
    errno_assert(nbytes >= 0);
#else
    ssize_t nbytes = ::recv(_r, &dummy, sizeof(dummy), 0);
    errno_assert(nbytes >= 0);
#endif
    zmq_assert(nbytes == sizeof(dummy));
    zmq_assert(dummy == 0);
#endif
}

int zmq::signaler_t::recv_failable()
{
//  Attempt to read a signal.
#if defined ZMQ_HAVE_EVENTFD
    uint64_t dummy;
    ssize_t  sz = read(_r, &dummy, sizeof(dummy));
    if (sz == -1)
    {
        errno_assert(errno == EAGAIN);
        return -1;
    }
    errno_assert(sz == sizeof(dummy));

    //  If we accidentally grabbed the next signal(s) along with the current
    //  one, return it back to the eventfd object.
    if (unlikely(dummy > 1))
    {
        const uint64_t inc = dummy - 1;
        ssize_t        sz2 = write(_w, &inc, sizeof(inc));
        errno_assert(sz2 == sizeof(inc));
        return 0;
    }

    zmq_assert(dummy == 1);

#else
    unsigned char dummy;
#if defined   ZMQ_HAVE_WINDOWS
    const int     nbytes = ::recv(_r, reinterpret_cast<char *>(&dummy), sizeof(dummy), 0);
    if (nbytes == SOCKET_ERROR)
    {
        const int last_error = WSAGetLastError();
        if (last_error == WSAEWOULDBLOCK)
        {
            errno = EAGAIN;
            return -1;
        }
        wsa_assert(last_error == WSAEWOULDBLOCK);
    }
#elif defined ZMQ_HAVE_VXWORKS
    ssize_t nbytes = ::recv(_r, (char *)&dummy, sizeof(dummy), 0);
    if (nbytes == -1)
    {
        if (errno == EAGAIN || errno == EWOULDBLOCK || errno == EINTR)
        {
            errno = EAGAIN;
            return -1;
        }
        errno_assert(errno == EAGAIN || errno == EWOULDBLOCK || errno == EINTR);
    }
#else
    ssize_t nbytes = ::recv(_r, &dummy, sizeof(dummy), 0);
    if (nbytes == -1)
    {
        if (errno == EAGAIN || errno == EWOULDBLOCK || errno == EINTR)
        {
            errno = EAGAIN;
            return -1;
        }
        errno_assert(errno == EAGAIN || errno == EWOULDBLOCK || errno == EINTR);
    }
#endif
    zmq_assert(nbytes == sizeof(dummy));
    zmq_assert(dummy == 0);
#endif
    return 0;
}

bool zmq::signaler_t::valid() const { return _w != retired_fd; }

#ifdef HAVE_FORK
void zmq::signaler_t::forked()
{
    //  Close file descriptors created in the parent and create new pair
    close(_r);
    close(_w);
    make_fdpair(&_r, &_w);
}
#endif

//========= end of signaler.cpp ============

//========= begin of socket_base.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <algorithm>
#include <limits>
#include <new>
#include <string>

// ans ignore: #include "macros.hpp"

#if defined ZMQ_HAVE_WINDOWS
#if defined _MSC_VER
#if defined _WIN32_WCE
#include <cmnintrin.h>
#else
#include <intrin.h>
#endif
#endif
#else
#include <ctype.h>
#include <unistd.h>
#endif

// ans ignore: #include "socket_base.hpp"
// ans ignore: #include "tcp_listener.hpp"
// ans ignore: #include "ipc_listener.hpp"
// ans ignore: #include "tipc_listener.hpp"
// ans ignore: #include "tcp_connecter.hpp"
// ans ignore: #include "io_thread.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "config.hpp"
// ans ignore: #include "pipe.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "ctx.hpp"
// ans ignore: #include "likely.hpp"
// ans ignore: #include "msg.hpp"
// ans ignore: #include "address.hpp"
// ans ignore: #include "ipc_address.hpp"
// ans ignore: #include "tcp_address.hpp"
// ans ignore: #include "udp_address.hpp"
// ans ignore: #include "tipc_address.hpp"
// ans ignore: #include "mailbox.hpp"
// ans ignore: #include "mailbox_safe.hpp"

#if defined ZMQ_HAVE_VMCI
// ans ignore: #include "vmci_address.hpp"
// ans ignore: #include "vmci_listener.hpp"
#endif

#ifdef ZMQ_HAVE_OPENPGM
// ans ignore: #include "pgm_socket.hpp"
#endif

// ans ignore: #include "pair.hpp"
// ans ignore: #include "pub.hpp"
// ans ignore: #include "sub.hpp"
// ans ignore: #include "req.hpp"
// ans ignore: #include "rep.hpp"
// ans ignore: #include "pull.hpp"
// ans ignore: #include "push.hpp"
// ans ignore: #include "dealer.hpp"
// ans ignore: #include "router.hpp"
// ans ignore: #include "xpub.hpp"
// ans ignore: #include "xsub.hpp"
// ans ignore: #include "stream.hpp"
// ans ignore: #include "server.hpp"
// ans ignore: #include "client.hpp"
// ans ignore: #include "radio.hpp"
// ans ignore: #include "dish.hpp"
// ans ignore: #include "gather.hpp"
// ans ignore: #include "scatter.hpp"
// ans ignore: #include "dgram.hpp"

void zmq::socket_base_t::inprocs_t::emplace(const char *endpoint_uri_, pipe_t *pipe_)
{
    _inprocs.ZMQ_MAP_INSERT_OR_EMPLACE(std::string(endpoint_uri_), pipe_);
}

int zmq::socket_base_t::inprocs_t::erase_pipes(const std::string &endpoint_uri_str_)
{
    const std::pair<map_t::iterator, map_t::iterator> range = _inprocs.equal_range(endpoint_uri_str_);
    if (range.first == range.second)
    {
        errno = ENOENT;
        return -1;
    }

    for (map_t::iterator it = range.first; it != range.second; ++it)
        it->second->terminate(true);
    _inprocs.erase(range.first, range.second);
    return 0;
}

void zmq::socket_base_t::inprocs_t::erase_pipe(pipe_t *pipe_)
{
    for (map_t::iterator it = _inprocs.begin(), end = _inprocs.end(); it != end; ++it)
        if (it->second == pipe_)
        {
            _inprocs.erase(it);
            break;
        }
}

bool zmq::socket_base_t::check_tag() const { return _tag == 0xbaddecaf; }

bool zmq::socket_base_t::is_thread_safe() const { return _thread_safe; }

zmq::socket_base_t *zmq::socket_base_t::create(int type_, class ctx_t *parent_, uint32_t tid_, int sid_)
{
    socket_base_t *s = NULL;
    switch (type_)
    {
    case ZMQ_PAIR:
        s = new (std::nothrow) pair_t(parent_, tid_, sid_);
        break;
    case ZMQ_PUB:
        s = new (std::nothrow) pub_t(parent_, tid_, sid_);
        break;
    case ZMQ_SUB:
        s = new (std::nothrow) sub_t(parent_, tid_, sid_);
        break;
    case ZMQ_REQ:
        s = new (std::nothrow) req_t(parent_, tid_, sid_);
        break;
    case ZMQ_REP:
        s = new (std::nothrow) rep_t(parent_, tid_, sid_);
        break;
    case ZMQ_DEALER:
        s = new (std::nothrow) dealer_t(parent_, tid_, sid_);
        break;
    case ZMQ_ROUTER:
        s = new (std::nothrow) router_t(parent_, tid_, sid_);
        break;
    case ZMQ_PULL:
        s = new (std::nothrow) pull_t(parent_, tid_, sid_);
        break;
    case ZMQ_PUSH:
        s = new (std::nothrow) push_t(parent_, tid_, sid_);
        break;
    case ZMQ_XPUB:
        s = new (std::nothrow) xpub_t(parent_, tid_, sid_);
        break;
    case ZMQ_XSUB:
        s = new (std::nothrow) xsub_t(parent_, tid_, sid_);
        break;
    case ZMQ_STREAM:
        s = new (std::nothrow) stream_t(parent_, tid_, sid_);
        break;
    case ZMQ_SERVER:
        s = new (std::nothrow) server_t(parent_, tid_, sid_);
        break;
    case ZMQ_CLIENT:
        s = new (std::nothrow) client_t(parent_, tid_, sid_);
        break;
    case ZMQ_RADIO:
        s = new (std::nothrow) radio_t(parent_, tid_, sid_);
        break;
    case ZMQ_DISH:
        s = new (std::nothrow) dish_t(parent_, tid_, sid_);
        break;
    case ZMQ_GATHER:
        s = new (std::nothrow) gather_t(parent_, tid_, sid_);
        break;
    case ZMQ_SCATTER:
        s = new (std::nothrow) scatter_t(parent_, tid_, sid_);
        break;
    case ZMQ_DGRAM:
        s = new (std::nothrow) dgram_t(parent_, tid_, sid_);
        break;
    default:
        errno = EINVAL;
        return NULL;
    }

    alloc_assert(s);

    if (s->_mailbox == NULL)
    {
        s->_destroyed = true;
        LIBZMQ_DELETE(s);
        return NULL;
    }

    return s;
}

zmq::socket_base_t::socket_base_t(ctx_t *parent_, uint32_t tid_, int sid_, bool thread_safe_)
    : own_t(parent_, tid_), _tag(0xbaddecaf), _ctx_terminated(false), _destroyed(false), _poller(NULL),
      _handle(static_cast<poller_t::handle_t>(NULL)), _last_tsc(0), _ticks(0), _rcvmore(false), _monitor_socket(NULL),
      _monitor_events(0), _thread_safe(thread_safe_), _reaper_signaler(NULL), _sync(), _monitor_sync()
{
    options.socket_id = sid_;
    options.ipv6      = (parent_->get(ZMQ_IPV6) != 0);
    options.linger.store(parent_->get(ZMQ_BLOCKY) ? -1 : 0);
    options.zero_copy = parent_->get(ZMQ_ZERO_COPY_RECV) != 0;

    if (_thread_safe)
    {
        _mailbox = new (std::nothrow) mailbox_safe_t(&_sync);
        zmq_assert(_mailbox);
    }
    else
    {
        mailbox_t *m = new (std::nothrow) mailbox_t();
        zmq_assert(m);

        if (m->get_fd() != retired_fd)
            _mailbox = m;
        else
        {
            LIBZMQ_DELETE(m);
            _mailbox = NULL;
        }
    }
}

int zmq::socket_base_t::get_peer_state(const void *routing_id_, size_t routing_id_size_) const
{
    LIBZMQ_UNUSED(routing_id_);
    LIBZMQ_UNUSED(routing_id_size_);

    //  Only ROUTER sockets support this
    errno = ENOTSUP;
    return -1;
}

zmq::socket_base_t::~socket_base_t()
{
    if (_mailbox)
        LIBZMQ_DELETE(_mailbox);

    if (_reaper_signaler)
        LIBZMQ_DELETE(_reaper_signaler);

    scoped_lock_t lock(_monitor_sync);
    stop_monitor();

    zmq_assert(_destroyed);
}

zmq::i_mailbox *zmq::socket_base_t::get_mailbox() const { return _mailbox; }

void zmq::socket_base_t::stop()
{
    //  Called by ctx when it is terminated (zmq_ctx_term).
    //  'stop' command is sent from the threads that called zmq_ctx_term to
    //  the thread owning the socket. This way, blocking call in the
    //  owner thread can be interrupted.
    send_stop();
}

// TODO consider renaming protocol_ to scheme_ in conformance with RFC 3986
// terminology, but this requires extensive changes to be consistent
int zmq::socket_base_t::parse_uri(const char *uri_, std::string &protocol_, std::string &path_)
{
    zmq_assert(uri_ != NULL);

    std::string                  uri(uri_);
    const std::string::size_type pos = uri.find("://");
    if (pos == std::string::npos)
    {
        errno = EINVAL;
        return -1;
    }
    protocol_ = uri.substr(0, pos);
    path_     = uri.substr(pos + 3);

    if (protocol_.empty() || path_.empty())
    {
        errno = EINVAL;
        return -1;
    }
    return 0;
}

int zmq::socket_base_t::check_protocol(const std::string &protocol_) const
{
    //  First check out whether the protocol is something we are aware of.
    if (protocol_ != protocol_name::inproc
#if !defined ZMQ_HAVE_WINDOWS && !defined ZMQ_HAVE_OPENVMS && !defined ZMQ_HAVE_VXWORKS
        && protocol_ != protocol_name::ipc
#endif
        && protocol_ != protocol_name::tcp
#if defined ZMQ_HAVE_OPENPGM
        //  pgm/epgm transports only available if 0MQ is compiled with OpenPGM.
        && protocol_ != "pgm" &&
        protocol_ != "epgm"
#endif
#if defined ZMQ_HAVE_TIPC
        // TIPC transport is only available on Linux.
        && protocol_ != protocol_name::tipc
#endif
#if defined ZMQ_HAVE_NORM
        && protocol_ != "norm"
#endif
#if defined ZMQ_HAVE_VMCI
        && protocol_ != protocol_name::vmci
#endif
        && protocol_ != protocol_name::udp)
    {
        errno = EPROTONOSUPPORT;
        return -1;
    }

    //  Check whether socket type and transport protocol match.
    //  Specifically, multicast protocols can't be combined with
    //  bi-directional messaging patterns (socket types).
#if defined ZMQ_HAVE_OPENPGM || defined ZMQ_HAVE_NORM
    if ((protocol_ == "pgm" || protocol_ == "epgm" || protocol_ == "norm") && options.type != ZMQ_PUB &&
        options.type != ZMQ_SUB && options.type != ZMQ_XPUB && options.type != ZMQ_XSUB)
    {
        errno = ENOCOMPATPROTO;
        return -1;
    }
#endif

    if (protocol_ == protocol_name::udp &&
        (options.type != ZMQ_DISH && options.type != ZMQ_RADIO && options.type != ZMQ_DGRAM))
    {
        errno = ENOCOMPATPROTO;
        return -1;
    }

    //  Protocol is available.
    return 0;
}

void zmq::socket_base_t::attach_pipe(pipe_t *pipe_, bool subscribe_to_all_, bool locally_initiated_)
{
    //  First, register the pipe so that we can terminate it later on.
    pipe_->set_event_sink(this);
    _pipes.push_back(pipe_);

    //  Let the derived socket type know about new pipe.
    xattach_pipe(pipe_, subscribe_to_all_, locally_initiated_);

    //  If the socket is already being closed, ask any new pipes to terminate
    //  straight away.
    if (is_terminating())
    {
        register_term_acks(1);
        pipe_->terminate(false);
    }
}

int zmq::socket_base_t::setsockopt(int option_, const void *optval_, size_t optvallen_)
{
    scoped_optional_lock_t sync_lock(_thread_safe ? &_sync : NULL);

    if (unlikely(_ctx_terminated))
    {
        errno = ETERM;
        return -1;
    }

    //  First, check whether specific socket type overloads the option.
    int rc = xsetsockopt(option_, optval_, optvallen_);
    if (rc == 0 || errno != EINVAL)
    {
        return rc;
    }

    //  If the socket type doesn't support the option, pass it to
    //  the generic option parser.
    rc = options.setsockopt(option_, optval_, optvallen_);
    update_pipe_options(option_);

    return rc;
}

int zmq::socket_base_t::getsockopt(int option_, void *optval_, size_t *optvallen_)
{
    scoped_optional_lock_t sync_lock(_thread_safe ? &_sync : NULL);

    if (unlikely(_ctx_terminated))
    {
        errno = ETERM;
        return -1;
    }

    if (option_ == ZMQ_RCVMORE)
    {
        return do_getsockopt<int>(optval_, optvallen_, _rcvmore ? 1 : 0);
    }

    if (option_ == ZMQ_FD)
    {
        if (_thread_safe)
        {
            // thread safe socket doesn't provide file descriptor
            errno = EINVAL;
            return -1;
        }

        return do_getsockopt<fd_t>(optval_, optvallen_, (static_cast<mailbox_t *>(_mailbox))->get_fd());
    }

    if (option_ == ZMQ_EVENTS)
    {
        const int rc = process_commands(0, false);
        if (rc != 0 && (errno == EINTR || errno == ETERM))
        {
            return -1;
        }
        errno_assert(rc == 0);

        return do_getsockopt<int>(optval_, optvallen_, (has_out() ? ZMQ_POLLOUT : 0) | (has_in() ? ZMQ_POLLIN : 0));
    }

    if (option_ == ZMQ_LAST_ENDPOINT)
    {
        return do_getsockopt(optval_, optvallen_, _last_endpoint);
    }

    if (option_ == ZMQ_THREAD_SAFE)
    {
        return do_getsockopt<int>(optval_, optvallen_, _thread_safe ? 1 : 0);
    }

    return options.getsockopt(option_, optval_, optvallen_);
}

int zmq::socket_base_t::join(const char *group_)
{
    scoped_optional_lock_t sync_lock(_thread_safe ? &_sync : NULL);

    return xjoin(group_);
}

int zmq::socket_base_t::leave(const char *group_)
{
    scoped_optional_lock_t sync_lock(_thread_safe ? &_sync : NULL);

    return xleave(group_);
}

void zmq::socket_base_t::add_signaler(signaler_t *s_)
{
    zmq_assert(_thread_safe);

    scoped_lock_t sync_lock(_sync);
    (static_cast<mailbox_safe_t *>(_mailbox))->add_signaler(s_);
}

void zmq::socket_base_t::remove_signaler(signaler_t *s_)
{
    zmq_assert(_thread_safe);

    scoped_lock_t sync_lock(_sync);
    (static_cast<mailbox_safe_t *>(_mailbox))->remove_signaler(s_);
}

int zmq::socket_base_t::bind(const char *endpoint_uri_)
{
    scoped_optional_lock_t sync_lock(_thread_safe ? &_sync : NULL);

    if (unlikely(_ctx_terminated))
    {
        errno = ETERM;
        return -1;
    }

    //  Process pending commands, if any.
    int rc = process_commands(0, false);
    if (unlikely(rc != 0))
    {
        return -1;
    }

    //  Parse endpoint_uri_ string.
    std::string protocol;
    std::string address;
    if (parse_uri(endpoint_uri_, protocol, address) || check_protocol(protocol))
    {
        return -1;
    }

    if (protocol == protocol_name::inproc)
    {
        const endpoint_t endpoint = {this, options};
        rc                        = register_endpoint(endpoint_uri_, endpoint);
        if (rc == 0)
        {
            connect_pending(endpoint_uri_, this);
            _last_endpoint.assign(endpoint_uri_);
            options.connected = true;
        }
        return rc;
    }

    if (protocol == "pgm" || protocol == "epgm" || protocol == "norm")
    {
        //  For convenience's sake, bind can be used interchangeable with
        //  connect for PGM, EPGM, NORM transports.
        rc = connect(endpoint_uri_);
        if (rc != -1)
            options.connected = true;
        return rc;
    }

    if (protocol == protocol_name::udp)
    {
        if (!(options.type == ZMQ_DGRAM || options.type == ZMQ_DISH))
        {
            errno = ENOCOMPATPROTO;
            return -1;
        }

        //  Choose the I/O thread to run the session in.
        io_thread_t *io_thread = choose_io_thread(options.affinity);
        if (!io_thread)
        {
            errno = EMTHREAD;
            return -1;
        }

        address_t *paddr = new (std::nothrow) address_t(protocol, address, this->get_ctx());
        alloc_assert(paddr);

        paddr->resolved.udp_addr = new (std::nothrow) udp_address_t();
        alloc_assert(paddr->resolved.udp_addr);
        rc = paddr->resolved.udp_addr->resolve(address.c_str(), true, options.ipv6);
        if (rc != 0)
        {
            LIBZMQ_DELETE(paddr);
            return -1;
        }

        session_base_t *session = session_base_t::create(io_thread, true, this, options, paddr);
        errno_assert(session);

        //  Create a bi-directional pipe.
        object_t *parents[2]   = {this, session};
        pipe_t *  new_pipes[2] = {NULL, NULL};

        int  hwms[2]      = {options.sndhwm, options.rcvhwm};
        bool conflates[2] = {false, false};
        rc                = pipepair(parents, new_pipes, hwms, conflates);
        errno_assert(rc == 0);

        //  Attach local end of the pipe to the socket object.
        attach_pipe(new_pipes[0], true, true);
        pipe_t *const newpipe = new_pipes[0];

        //  Attach remote end of the pipe to the session object later on.
        session->attach_pipe(new_pipes[1]);

        //  Save last endpoint URI
        paddr->to_string(_last_endpoint);

        //  TODO shouldn't this use _last_endpoint instead of endpoint_uri_? as in the other cases
        add_endpoint(endpoint_uri_pair_t(endpoint_uri_, std::string(), endpoint_type_none),
                     static_cast<own_t *>(session), newpipe);

        return 0;
    }

    //  Remaining transports require to be run in an I/O thread, so at this
    //  point we'll choose one.
    io_thread_t *io_thread = choose_io_thread(options.affinity);
    if (!io_thread)
    {
        errno = EMTHREAD;
        return -1;
    }

    if (protocol == protocol_name::tcp)
    {
        tcp_listener_t *listener = new (std::nothrow) tcp_listener_t(io_thread, this, options);
        alloc_assert(listener);
        rc = listener->set_local_address(address.c_str());
        if (rc != 0)
        {
            LIBZMQ_DELETE(listener);
            event_bind_failed(make_unconnected_bind_endpoint_pair(address), zmq_errno());
            return -1;
        }

        // Save last endpoint URI
        listener->get_local_address(_last_endpoint);

        add_endpoint(make_unconnected_bind_endpoint_pair(_last_endpoint), static_cast<own_t *>(listener), NULL);
        options.connected = true;
        return 0;
    }

#if !defined ZMQ_HAVE_WINDOWS && !defined ZMQ_HAVE_OPENVMS && !defined ZMQ_HAVE_VXWORKS
    if (protocol == protocol_name::ipc)
    {
        ipc_listener_t *listener = new (std::nothrow) ipc_listener_t(io_thread, this, options);
        alloc_assert(listener);
        int rc = listener->set_local_address(address.c_str());
        if (rc != 0)
        {
            LIBZMQ_DELETE(listener);
            event_bind_failed(make_unconnected_bind_endpoint_pair(address), zmq_errno());
            return -1;
        }

        // Save last endpoint URI
        listener->get_local_address(_last_endpoint);

        add_endpoint(make_unconnected_bind_endpoint_pair(_last_endpoint), static_cast<own_t *>(listener), NULL);
        options.connected = true;
        return 0;
    }
#endif
#if defined ZMQ_HAVE_TIPC
    if (protocol == protocol_name::tipc)
    {
        tipc_listener_t *listener = new (std::nothrow) tipc_listener_t(io_thread, this, options);
        alloc_assert(listener);
        int rc = listener->set_local_address(address.c_str());
        if (rc != 0)
        {
            LIBZMQ_DELETE(listener);
            event_bind_failed(make_unconnected_bind_endpoint_pair(address), zmq_errno());
            return -1;
        }

        // Save last endpoint URI
        listener->get_local_address(_last_endpoint);

        // TODO shouldn't this use _last_endpoint as in the other cases?
        add_endpoint(make_unconnected_bind_endpoint_pair(endpoint_uri_), static_cast<own_t *>(listener), NULL);
        options.connected = true;
        return 0;
    }
#endif
#if defined ZMQ_HAVE_VMCI
    if (protocol == protocol_name::vmci)
    {
        vmci_listener_t *listener = new (std::nothrow) vmci_listener_t(io_thread, this, options);
        alloc_assert(listener);
        int rc = listener->set_local_address(address.c_str());
        if (rc != 0)
        {
            LIBZMQ_DELETE(listener);
            event_bind_failed(make_unconnected_bind_endpoint_pair(address), zmq_errno());
            return -1;
        }

        listener->get_local_address(_last_endpoint);

        add_endpoint(make_unconnected_bind_endpoint_pair(_last_endpoint), static_cast<own_t *>(listener), NULL);
        options.connected = true;
        return 0;
    }
#endif

    zmq_assert(false);
    return -1;
}

int zmq::socket_base_t::connect(const char *endpoint_uri_)
{
    scoped_optional_lock_t sync_lock(_thread_safe ? &_sync : NULL);

    if (unlikely(_ctx_terminated))
    {
        errno = ETERM;
        return -1;
    }

    //  Process pending commands, if any.
    int rc = process_commands(0, false);
    if (unlikely(rc != 0))
    {
        return -1;
    }

    //  Parse endpoint_uri_ string.
    std::string protocol;
    std::string address;
    if (parse_uri(endpoint_uri_, protocol, address) || check_protocol(protocol))
    {
        return -1;
    }

    if (protocol == protocol_name::inproc)
    {
        //  TODO: inproc connect is specific with respect to creating pipes
        //  as there's no 'reconnect' functionality implemented. Once that
        //  is in place we should follow generic pipe creation algorithm.

        //  Find the peer endpoint.
        const endpoint_t peer = find_endpoint(endpoint_uri_);

        // The total HWM for an inproc connection should be the sum of
        // the binder's HWM and the connector's HWM.
        const int sndhwm =
            peer.socket == NULL
                ? options.sndhwm
                : options.sndhwm != 0 && peer.options.rcvhwm != 0 ? options.sndhwm + peer.options.rcvhwm : 0;
        const int rcvhwm =
            peer.socket == NULL
                ? options.rcvhwm
                : options.rcvhwm != 0 && peer.options.sndhwm != 0 ? options.rcvhwm + peer.options.sndhwm : 0;

        //  Create a bi-directional pipe to connect the peers.
        object_t *parents[2]   = {this, peer.socket == NULL ? this : peer.socket};
        pipe_t *  new_pipes[2] = {NULL, NULL};

        const bool conflate = get_effective_conflate_option(options);

        int  hwms[2]      = {conflate ? -1 : sndhwm, conflate ? -1 : rcvhwm};
        bool conflates[2] = {conflate, conflate};
        rc                = pipepair(parents, new_pipes, hwms, conflates);
        if (!conflate)
        {
            new_pipes[0]->set_hwms_boost(peer.options.sndhwm, peer.options.rcvhwm);
            new_pipes[1]->set_hwms_boost(options.sndhwm, options.rcvhwm);
        }

        errno_assert(rc == 0);

        if (!peer.socket)
        {
            //  The peer doesn't exist yet so we don't know whether
            //  to send the routing id message or not. To resolve this,
            //  we always send our routing id and drop it later if
            //  the peer doesn't expect it.
            send_routing_id(new_pipes[0], options);

            const endpoint_t endpoint = {this, options};
            pend_connection(std::string(endpoint_uri_), endpoint, new_pipes);
        }
        else
        {
            //  If required, send the routing id of the local socket to the peer.
            if (peer.options.recv_routing_id)
            {
                send_routing_id(new_pipes[0], options);
            }

            //  If required, send the routing id of the peer to the local socket.
            if (options.recv_routing_id)
            {
                send_routing_id(new_pipes[1], peer.options);
            }

            //  Attach remote end of the pipe to the peer socket. Note that peer's
            //  seqnum was incremented in find_endpoint function. We don't need it
            //  increased here.
            send_bind(peer.socket, new_pipes[1], false);
        }

        //  Attach local end of the pipe to this socket object.
        attach_pipe(new_pipes[0], false, true);

        // Save last endpoint URI
        _last_endpoint.assign(endpoint_uri_);

        // remember inproc connections for disconnect
        _inprocs.emplace(endpoint_uri_, new_pipes[0]);

        options.connected = true;
        return 0;
    }
    const bool is_single_connect =
        (options.type == ZMQ_DEALER || options.type == ZMQ_SUB || options.type == ZMQ_PUB || options.type == ZMQ_REQ);
    if (unlikely(is_single_connect))
    {
        if (0 != _endpoints.count(endpoint_uri_))
        {
            // There is no valid use for multiple connects for SUB-PUB nor
            // DEALER-ROUTER nor REQ-REP. Multiple connects produces
            // nonsensical results.
            return 0;
        }
    }

    //  Choose the I/O thread to run the session in.
    io_thread_t *io_thread = choose_io_thread(options.affinity);
    if (!io_thread)
    {
        errno = EMTHREAD;
        return -1;
    }

    address_t *paddr = new (std::nothrow) address_t(protocol, address, this->get_ctx());
    alloc_assert(paddr);

    //  Resolve address (if needed by the protocol)
    if (protocol == protocol_name::tcp)
    {
        //  Do some basic sanity checks on tcp:// address syntax
        //  - hostname starts with digit or letter, with embedded '-' or '.'
        //  - IPv6 address may contain hex chars and colons.
        //  - IPv6 link local address may contain % followed by interface name / zone_id
        //    (Reference: https://tools.ietf.org/html/rfc4007)
        //  - IPv4 address may contain decimal digits and dots.
        //  - Address must end in ":port" where port is *, or numeric
        //  - Address may contain two parts separated by ':'
        //  Following code is quick and dirty check to catch obvious errors,
        //  without trying to be fully accurate.
        const char *check = address.c_str();
        if (isalnum(*check) || isxdigit(*check) || *check == '[' || *check == ':')
        {
            check++;
            while (isalnum(*check) || isxdigit(*check) || *check == '.' || *check == '-' || *check == ':' ||
                   *check == '%' || *check == ';' || *check == '[' || *check == ']' || *check == '_' || *check == '*')
            {
                check++;
            }
        }
        //  Assume the worst, now look for success
        rc = -1;
        //  Did we reach the end of the address safely?
        if (*check == 0)
        {
            //  Do we have a valid port string? (cannot be '*' in connect
            check = strrchr(address.c_str(), ':');
            if (check)
            {
                check++;
                if (*check && (isdigit(*check)))
                    rc = 0; //  Valid
            }
        }
        if (rc == -1)
        {
            errno = EINVAL;
            LIBZMQ_DELETE(paddr);
            return -1;
        }
        //  Defer resolution until a socket is opened
        paddr->resolved.tcp_addr = NULL;
    }
#if !defined ZMQ_HAVE_WINDOWS && !defined ZMQ_HAVE_OPENVMS && !defined ZMQ_HAVE_VXWORKS
    else if (protocol == protocol_name::ipc)
    {
        paddr->resolved.ipc_addr = new (std::nothrow) ipc_address_t();
        alloc_assert(paddr->resolved.ipc_addr);
        int rc = paddr->resolved.ipc_addr->resolve(address.c_str());
        if (rc != 0)
        {
            LIBZMQ_DELETE(paddr);
            return -1;
        }
    }
#endif

    if (protocol == protocol_name::udp)
    {
        if (options.type != ZMQ_RADIO)
        {
            errno = ENOCOMPATPROTO;
            LIBZMQ_DELETE(paddr);
            return -1;
        }

        paddr->resolved.udp_addr = new (std::nothrow) udp_address_t();
        alloc_assert(paddr->resolved.udp_addr);
        rc = paddr->resolved.udp_addr->resolve(address.c_str(), false, options.ipv6);
        if (rc != 0)
        {
            LIBZMQ_DELETE(paddr);
            return -1;
        }
    }

    // TBD - Should we check address for ZMQ_HAVE_NORM???

#ifdef ZMQ_HAVE_OPENPGM
    if (protocol == "pgm" || protocol == "epgm")
    {
        struct pgm_addrinfo_t *res         = NULL;
        uint16_t               port_number = 0;
        int                    rc          = pgm_socket_t::init_address(address.c_str(), &res, &port_number);
        if (res != NULL)
            pgm_freeaddrinfo(res);
        if (rc != 0 || port_number == 0)
        {
            return -1;
        }
    }
#endif
#if defined ZMQ_HAVE_TIPC
    else if (protocol == protocol_name::tipc)
    {
        paddr->resolved.tipc_addr = new (std::nothrow) tipc_address_t();
        alloc_assert(paddr->resolved.tipc_addr);
        int rc = paddr->resolved.tipc_addr->resolve(address.c_str());
        if (rc != 0)
        {
            LIBZMQ_DELETE(paddr);
            return -1;
        }
        sockaddr_tipc *saddr = (sockaddr_tipc *)paddr->resolved.tipc_addr->addr();
        // Cannot connect to random Port Identity
        if (saddr->addrtype == TIPC_ADDR_ID && paddr->resolved.tipc_addr->is_random())
        {
            LIBZMQ_DELETE(paddr);
            errno = EINVAL;
            return -1;
        }
    }
#endif
#if defined ZMQ_HAVE_VMCI
    else if (protocol == protocol_name::vmci)
    {
        paddr->resolved.vmci_addr = new (std::nothrow) vmci_address_t(this->get_ctx());
        alloc_assert(paddr->resolved.vmci_addr);
        int rc = paddr->resolved.vmci_addr->resolve(address.c_str());
        if (rc != 0)
        {
            LIBZMQ_DELETE(paddr);
            return -1;
        }
    }
#endif

    //  Create session.
    session_base_t *session = session_base_t::create(io_thread, true, this, options, paddr);
    errno_assert(session);

    //  PGM does not support subscription forwarding; ask for all data to be
    //  sent to this pipe. (same for NORM, currently?)
    const bool subscribe_to_all =
        protocol == "pgm" || protocol == "epgm" || protocol == "norm" || protocol == protocol_name::udp;
    pipe_t *newpipe = NULL;

    if (options.immediate != 1 || subscribe_to_all)
    {
        //  Create a bi-directional pipe.
        object_t *parents[2]   = {this, session};
        pipe_t *  new_pipes[2] = {NULL, NULL};

        const bool conflate = get_effective_conflate_option(options);

        int  hwms[2]      = {conflate ? -1 : options.sndhwm, conflate ? -1 : options.rcvhwm};
        bool conflates[2] = {conflate, conflate};
        rc                = pipepair(parents, new_pipes, hwms, conflates);
        errno_assert(rc == 0);

        //  Attach local end of the pipe to the socket object.
        attach_pipe(new_pipes[0], subscribe_to_all, true);
        newpipe = new_pipes[0];

        //  Attach remote end of the pipe to the session object later on.
        session->attach_pipe(new_pipes[1]);
    }

    //  Save last endpoint URI
    paddr->to_string(_last_endpoint);

    add_endpoint(make_unconnected_connect_endpoint_pair(endpoint_uri_), static_cast<own_t *>(session), newpipe);
    return 0;
}

std::string zmq::socket_base_t::resolve_tcp_addr(std::string endpoint_uri_pair_, const char *tcp_address_)
{
    // The resolved last_endpoint is used as a key in the endpoints map.
    // The address passed by the user might not match in the TCP case due to
    // IPv4-in-IPv6 mapping (EG: tcp://[::ffff:127.0.0.1]:9999), so try to
    // resolve before giving up. Given at this stage we don't know whether a
    // socket is connected or bound, try with both.
    if (_endpoints.find(endpoint_uri_pair_) == _endpoints.end())
    {
        tcp_address_t *tcp_addr = new (std::nothrow) tcp_address_t();
        alloc_assert(tcp_addr);
        int rc = tcp_addr->resolve(tcp_address_, false, options.ipv6);

        if (rc == 0)
        {
            tcp_addr->to_string(endpoint_uri_pair_);
            if (_endpoints.find(endpoint_uri_pair_) == _endpoints.end())
            {
                rc = tcp_addr->resolve(tcp_address_, true, options.ipv6);
                if (rc == 0)
                {
                    tcp_addr->to_string(endpoint_uri_pair_);
                }
            }
        }
        LIBZMQ_DELETE(tcp_addr);
    }
    return endpoint_uri_pair_;
}

void zmq::socket_base_t::add_endpoint(const endpoint_uri_pair_t &endpoint_pair_, own_t *endpoint_, pipe_t *pipe_)
{
    //  Activate the session. Make it a child of this socket.
    launch_child(endpoint_);
    _endpoints.ZMQ_MAP_INSERT_OR_EMPLACE(endpoint_pair_.identifier(), endpoint_pipe_t(endpoint_, pipe_));

    if (pipe_ != NULL)
        pipe_->set_endpoint_pair(endpoint_pair_);
}

int zmq::socket_base_t::term_endpoint(const char *endpoint_uri_)
{
    scoped_optional_lock_t sync_lock(_thread_safe ? &_sync : NULL);

    //  Check whether the context hasn't been shut down yet.
    if (unlikely(_ctx_terminated))
    {
        errno = ETERM;
        return -1;
    }

    //  Check whether endpoint address passed to the function is valid.
    if (unlikely(!endpoint_uri_))
    {
        errno = EINVAL;
        return -1;
    }

    //  Process pending commands, if any, since there could be pending unprocessed process_own()'s
    //  (from launch_child() for example) we're asked to terminate now.
    const int rc = process_commands(0, false);
    if (unlikely(rc != 0))
    {
        return -1;
    }

    //  Parse endpoint_uri_ string.
    std::string uri_protocol;
    std::string uri_path;
    if (parse_uri(endpoint_uri_, uri_protocol, uri_path) || check_protocol(uri_protocol))
    {
        return -1;
    }

    const std::string endpoint_uri_str = std::string(endpoint_uri_);

    // Disconnect an inproc socket
    if (uri_protocol == protocol_name::inproc)
    {
        return unregister_endpoint(endpoint_uri_str, this) == 0 ? 0 : _inprocs.erase_pipes(endpoint_uri_str);
    }

    const std::string resolved_endpoint_uri =
        uri_protocol == protocol_name::tcp ? resolve_tcp_addr(endpoint_uri_str, uri_path.c_str()) : endpoint_uri_str;

    //  Find the endpoints range (if any) corresponding to the endpoint_uri_pair_ string.
    const std::pair<endpoints_t::iterator, endpoints_t::iterator> range = _endpoints.equal_range(resolved_endpoint_uri);
    if (range.first == range.second)
    {
        errno = ENOENT;
        return -1;
    }

    for (endpoints_t::iterator it = range.first; it != range.second; ++it)
    {
        //  If we have an associated pipe, terminate it.
        if (it->second.second != NULL)
            it->second.second->terminate(false);
        term_child(it->second.first);
    }
    _endpoints.erase(range.first, range.second);
    return 0;
}

int zmq::socket_base_t::send(msg_t *msg_, int flags_)
{
    scoped_optional_lock_t sync_lock(_thread_safe ? &_sync : NULL);

    //  Check whether the context hasn't been shut down yet.
    if (unlikely(_ctx_terminated))
    {
        errno = ETERM;
        return -1;
    }

    //  Check whether message passed to the function is valid.
    if (unlikely(!msg_ || !msg_->check()))
    {
        errno = EFAULT;
        return -1;
    }

    //  Process pending commands, if any.
    int rc = process_commands(0, true);
    if (unlikely(rc != 0))
    {
        return -1;
    }

    //  Clear any user-visible flags that are set on the message.
    msg_->reset_flags(msg_t::more);

    //  At this point we impose the flags on the message.
    if (flags_ & ZMQ_SNDMORE)
        msg_->set_flags(msg_t::more);

    msg_->reset_metadata();

    //  Try to send the message using method in each socket class
    rc = xsend(msg_);
    if (rc == 0)
    {
        return 0;
    }
    //  Special case for ZMQ_PUSH: -2 means pipe is dead while a
    //  multi-part send is in progress and can't be recovered, so drop
    //  silently when in blocking mode to keep backward compatibility.
    if (unlikely(rc == -2))
    {
        if (!((flags_ & ZMQ_DONTWAIT) || options.sndtimeo == 0))
        {
            rc = msg_->close();
            errno_assert(rc == 0);
            rc = msg_->init();
            errno_assert(rc == 0);
            return 0;
        }
    }
    if (unlikely(errno != EAGAIN))
    {
        return -1;
    }

    //  In case of non-blocking send we'll simply propagate
    //  the error - including EAGAIN - up the stack.
    if ((flags_ & ZMQ_DONTWAIT) || options.sndtimeo == 0)
    {
        return -1;
    }

    //  Compute the time when the timeout should occur.
    //  If the timeout is infinite, don't care.
    int            timeout = options.sndtimeo;
    const uint64_t end     = timeout < 0 ? 0 : (_clock.now_ms() + timeout);

    //  Oops, we couldn't send the message. Wait for the next
    //  command, process it and try to send the message again.
    //  If timeout is reached in the meantime, return EAGAIN.
    while (true)
    {
        if (unlikely(process_commands(timeout, false) != 0))
        {
            return -1;
        }
        rc = xsend(msg_);
        if (rc == 0)
            break;
        if (unlikely(errno != EAGAIN))
        {
            return -1;
        }
        if (timeout > 0)
        {
            timeout = static_cast<int>(end - _clock.now_ms());
            if (timeout <= 0)
            {
                errno = EAGAIN;
                return -1;
            }
        }
    }

    return 0;
}

int zmq::socket_base_t::recv(msg_t *msg_, int flags_)
{
    scoped_optional_lock_t sync_lock(_thread_safe ? &_sync : NULL);

    //  Check whether the context hasn't been shut down yet.
    if (unlikely(_ctx_terminated))
    {
        errno = ETERM;
        return -1;
    }

    //  Check whether message passed to the function is valid.
    if (unlikely(!msg_ || !msg_->check()))
    {
        errno = EFAULT;
        return -1;
    }

    //  Once every inbound_poll_rate messages check for signals and process
    //  incoming commands. This happens only if we are not polling altogether
    //  because there are messages available all the time. If poll occurs,
    //  ticks is set to zero and thus we avoid this code.
    //
    //  Note that 'recv' uses different command throttling algorithm (the one
    //  described above) from the one used by 'send'. This is because counting
    //  ticks is more efficient than doing RDTSC all the time.
    if (++_ticks == inbound_poll_rate)
    {
        if (unlikely(process_commands(0, false) != 0))
        {
            return -1;
        }
        _ticks = 0;
    }

    //  Get the message.
    int rc = xrecv(msg_);
    if (unlikely(rc != 0 && errno != EAGAIN))
    {
        return -1;
    }

    //  If we have the message, return immediately.
    if (rc == 0)
    {
        extract_flags(msg_);
        return 0;
    }

    //  If the message cannot be fetched immediately, there are two scenarios.
    //  For non-blocking recv, commands are processed in case there's an
    //  activate_reader command already waiting in a command pipe.
    //  If it's not, return EAGAIN.
    if ((flags_ & ZMQ_DONTWAIT) || options.rcvtimeo == 0)
    {
        if (unlikely(process_commands(0, false) != 0))
        {
            return -1;
        }
        _ticks = 0;

        rc = xrecv(msg_);
        if (rc < 0)
        {
            return rc;
        }
        extract_flags(msg_);

        return 0;
    }

    //  Compute the time when the timeout should occur.
    //  If the timeout is infinite, don't care.
    int            timeout = options.rcvtimeo;
    const uint64_t end     = timeout < 0 ? 0 : (_clock.now_ms() + timeout);

    //  In blocking scenario, commands are processed over and over again until
    //  we are able to fetch a message.
    bool block = (_ticks != 0);
    while (true)
    {
        if (unlikely(process_commands(block ? timeout : 0, false) != 0))
        {
            return -1;
        }
        rc = xrecv(msg_);
        if (rc == 0)
        {
            _ticks = 0;
            break;
        }
        if (unlikely(errno != EAGAIN))
        {
            return -1;
        }
        block = true;
        if (timeout > 0)
        {
            timeout = static_cast<int>(end - _clock.now_ms());
            if (timeout <= 0)
            {
                errno = EAGAIN;
                return -1;
            }
        }
    }

    extract_flags(msg_);
    return 0;
}

int zmq::socket_base_t::close()
{
    scoped_optional_lock_t sync_lock(_thread_safe ? &_sync : NULL);

    //  Remove all existing signalers for thread safe sockets
    if (_thread_safe)
        (static_cast<mailbox_safe_t *>(_mailbox))->clear_signalers();

    //  Mark the socket as dead
    _tag = 0xdeadbeef;

    //  Transfer the ownership of the socket from this application thread
    //  to the reaper thread which will take care of the rest of shutdown
    //  process.
    send_reap(this);

    return 0;
}

bool zmq::socket_base_t::has_in() { return xhas_in(); }

bool zmq::socket_base_t::has_out() { return xhas_out(); }

void zmq::socket_base_t::start_reaping(poller_t *poller_)
{
    //  Plug the socket to the reaper thread.
    _poller = poller_;

    fd_t fd;

    if (!_thread_safe)
        fd = (static_cast<mailbox_t *>(_mailbox))->get_fd();
    else
    {
        scoped_optional_lock_t sync_lock(_thread_safe ? &_sync : NULL);

        _reaper_signaler = new (std::nothrow) signaler_t();
        zmq_assert(_reaper_signaler);

        //  Add signaler to the safe mailbox
        fd = _reaper_signaler->get_fd();
        (static_cast<mailbox_safe_t *>(_mailbox))->add_signaler(_reaper_signaler);

        //  Send a signal to make sure reaper handle existing commands
        _reaper_signaler->send();
    }

    _handle = _poller->add_fd(fd, this);
    _poller->set_pollin(_handle);

    //  Initialise the termination and check whether it can be deallocated
    //  immediately.
    terminate();
    check_destroy();
}

int zmq::socket_base_t::process_commands(int timeout_, bool throttle_)
{
    if (timeout_ == 0)
    {
        //  If we are asked not to wait, check whether we haven't processed
        //  commands recently, so that we can throttle the new commands.

        //  Get the CPU's tick counter. If 0, the counter is not available.
        const uint64_t tsc = zmq::clock_t::rdtsc();

        //  Optimised version of command processing - it doesn't have to check
        //  for incoming commands each time. It does so only if certain time
        //  elapsed since last command processing. Command delay varies
        //  depending on CPU speed: It's ~1ms on 3GHz CPU, ~2ms on 1.5GHz CPU
        //  etc. The optimisation makes sense only on platforms where getting
        //  a timestamp is a very cheap operation (tens of nanoseconds).
        if (tsc && throttle_)
        {
            //  Check whether TSC haven't jumped backwards (in case of migration
            //  between CPU cores) and whether certain time have elapsed since
            //  last command processing. If it didn't do nothing.
            if (tsc >= _last_tsc && tsc - _last_tsc <= max_command_delay)
                return 0;
            _last_tsc = tsc;
        }
    }

    //  Check whether there are any commands pending for this thread.
    command_t cmd;
    int       rc = _mailbox->recv(&cmd, timeout_);

    //  Process all available commands.
    while (rc == 0)
    {
        cmd.destination->process_command(cmd);
        rc = _mailbox->recv(&cmd, 0);
    }

    if (errno == EINTR)
        return -1;

    zmq_assert(errno == EAGAIN);

    if (_ctx_terminated)
    {
        errno = ETERM;
        return -1;
    }

    return 0;
}

void zmq::socket_base_t::process_stop()
{
    //  Here, someone have called zmq_ctx_term while the socket was still alive.
    //  We'll remember the fact so that any blocking call is interrupted and any
    //  further attempt to use the socket will return ETERM. The user is still
    //  responsible for calling zmq_close on the socket though!
    scoped_lock_t lock(_monitor_sync);
    stop_monitor();

    _ctx_terminated = true;
}

void zmq::socket_base_t::process_bind(pipe_t *pipe_) { attach_pipe(pipe_); }

void zmq::socket_base_t::process_term(int linger_)
{
    //  Unregister all inproc endpoints associated with this socket.
    //  Doing this we make sure that no new pipes from other sockets (inproc)
    //  will be initiated.
    unregister_endpoints(this);

    //  Ask all attached pipes to terminate.
    for (pipes_t::size_type i = 0; i != _pipes.size(); ++i)
        _pipes[i]->terminate(false);
    register_term_acks(static_cast<int>(_pipes.size()));

    //  Continue the termination process immediately.
    own_t::process_term(linger_);
}

void zmq::socket_base_t::process_term_endpoint(std::string *endpoint_)
{
    term_endpoint(endpoint_->c_str());
    delete endpoint_;
}

void zmq::socket_base_t::process_pipe_stats_publish(uint64_t outbound_queue_count_, uint64_t inbound_queue_count_,
                                                    endpoint_uri_pair_t *endpoint_pair_)
{
    uint64_t values[2] = {outbound_queue_count_, inbound_queue_count_};
    event(*endpoint_pair_, values, 2, ZMQ_EVENT_PIPES_STATS);
    delete endpoint_pair_;
}

/*
 * There are 2 pipes per connection, and the inbound one _must_ be queried from
 * the I/O thread. So ask the outbound pipe, in the application thread, to send
 * a message (pipe_peer_stats) to its peer. The message will carry the outbound
 * pipe stats and endpoint, and the reference to the socket object.
 * The inbound pipe on the I/O thread will then add its own stats and endpoint,
 * and write back a message to the socket object (pipe_stats_publish) which
 * will raise an event with the data.
 */
int zmq::socket_base_t::query_pipes_stats()
{
    {
        scoped_lock_t lock(_monitor_sync);
        if (!(_monitor_events & ZMQ_EVENT_PIPES_STATS))
        {
            errno = EINVAL;
            return -1;
        }
    }
    if (_pipes.size() == 0)
    {
        errno = EAGAIN;
        return -1;
    }
    for (pipes_t::size_type i = 0; i != _pipes.size(); ++i)
    {
        _pipes[i]->send_stats_to_peer(this);
    }

    return 0;
}

void zmq::socket_base_t::update_pipe_options(int option_)
{
    if (option_ == ZMQ_SNDHWM || option_ == ZMQ_RCVHWM)
    {
        for (pipes_t::size_type i = 0; i != _pipes.size(); ++i)
        {
            _pipes[i]->set_hwms(options.rcvhwm, options.sndhwm);
            _pipes[i]->send_hwms_to_peer(options.sndhwm, options.rcvhwm);
        }
    }
}

void zmq::socket_base_t::process_destroy() { _destroyed = true; }

int zmq::socket_base_t::xsetsockopt(int, const void *, size_t)
{
    errno = EINVAL;
    return -1;
}

bool zmq::socket_base_t::xhas_out() { return false; }

int zmq::socket_base_t::xsend(msg_t *)
{
    errno = ENOTSUP;
    return -1;
}

bool zmq::socket_base_t::xhas_in() { return false; }

int zmq::socket_base_t::xjoin(const char *group_)
{
    LIBZMQ_UNUSED(group_);
    errno = ENOTSUP;
    return -1;
}

int zmq::socket_base_t::xleave(const char *group_)
{
    LIBZMQ_UNUSED(group_);
    errno = ENOTSUP;
    return -1;
}

int zmq::socket_base_t::xrecv(msg_t *)
{
    errno = ENOTSUP;
    return -1;
}

void zmq::socket_base_t::xread_activated(pipe_t *) { zmq_assert(false); }
void zmq::socket_base_t::xwrite_activated(pipe_t *) { zmq_assert(false); }

void zmq::socket_base_t::xhiccuped(pipe_t *) { zmq_assert(false); }

void zmq::socket_base_t::in_event()
{
    //  This function is invoked only once the socket is running in the context
    //  of the reaper thread. Process any commands from other threads/sockets
    //  that may be available at the moment. Ultimately, the socket will
    //  be destroyed.
    {
        scoped_optional_lock_t sync_lock(_thread_safe ? &_sync : NULL);

        //  If the socket is thread safe we need to unsignal the reaper signaler
        if (_thread_safe)
            _reaper_signaler->recv();

        process_commands(0, false);
    }
    check_destroy();
}

void zmq::socket_base_t::out_event() { zmq_assert(false); }

void zmq::socket_base_t::timer_event(int) { zmq_assert(false); }

void zmq::socket_base_t::check_destroy()
{
    //  If the object was already marked as destroyed, finish the deallocation.
    if (_destroyed)
    {
        //  Remove the socket from the reaper's poller.
        _poller->rm_fd(_handle);

        //  Remove the socket from the context.
        destroy_socket(this);

        //  Notify the reaper about the fact.
        send_reaped();

        //  Deallocate.
        own_t::process_destroy();
    }
}

void zmq::socket_base_t::read_activated(pipe_t *pipe_) { xread_activated(pipe_); }

void zmq::socket_base_t::write_activated(pipe_t *pipe_) { xwrite_activated(pipe_); }

void zmq::socket_base_t::hiccuped(pipe_t *pipe_)
{
    if (options.immediate == 1)
        pipe_->terminate(false);
    else
        // Notify derived sockets of the hiccup
        xhiccuped(pipe_);
}

void zmq::socket_base_t::pipe_terminated(pipe_t *pipe_)
{
    //  Notify the specific socket type about the pipe termination.
    xpipe_terminated(pipe_);

    // Remove pipe from inproc pipes
    _inprocs.erase_pipe(pipe_);

    //  Remove the pipe from the list of attached pipes and confirm its
    //  termination if we are already shutting down.
    _pipes.erase(pipe_);

    // Remove the pipe from _endpoints (set it to NULL).
    const std::string &identifier = pipe_->get_endpoint_pair().identifier();
    if (!identifier.empty())
    {
        std::pair<endpoints_t::iterator, endpoints_t::iterator> range;
        range = _endpoints.equal_range(identifier);

        for (endpoints_t::iterator it = range.first; it != range.second; ++it)
        {
            if (it->second.second == pipe_)
            {
                it->second.second = NULL;
                break;
            }
        }
    }

    if (is_terminating())
        unregister_term_ack();
}

void zmq::socket_base_t::extract_flags(msg_t *msg_)
{
    //  Test whether routing_id flag is valid for this socket type.
    if (unlikely(msg_->flags() & msg_t::routing_id))
        zmq_assert(options.recv_routing_id);

    //  Remove MORE flag.
    _rcvmore = (msg_->flags() & msg_t::more) != 0;
}

int zmq::socket_base_t::monitor(const char *endpoint_, uint64_t events_, int event_version_, int type_)
{
    scoped_lock_t lock(_monitor_sync);

    if (unlikely(_ctx_terminated))
    {
        errno = ETERM;
        return -1;
    }

    //  Event version 1 supports only first 16 events.
    if (unlikely(event_version_ == 1 && events_ >> 16 != 0))
    {
        errno = EINVAL;
        return -1;
    }

    //  Support deregistering monitoring endpoints as well
    if (endpoint_ == NULL)
    {
        stop_monitor();
        return 0;
    }
    //  Parse endpoint_uri_ string.
    std::string protocol;
    std::string address;
    if (parse_uri(endpoint_, protocol, address) || check_protocol(protocol))
        return -1;

    //  Event notification only supported over inproc://
    if (protocol != protocol_name::inproc)
    {
        errno = EPROTONOSUPPORT;
        return -1;
    }

    // already monitoring. Stop previous monitor before starting new one.
    if (_monitor_socket != NULL)
    {
        stop_monitor(true);
    }

    // Check if the specified socket type is supported. It must be a
    // one-way socket types that support the SNDMORE flag.
    switch (type_)
    {
    case ZMQ_PAIR:
        break;
    case ZMQ_PUB:
        break;
    case ZMQ_PUSH:
        break;
    default:
        errno = EINVAL;
        return -1;
    }

    //  Register events to monitor
    _monitor_events               = events_;
    options.monitor_event_version = event_version_;
    //  Create a monitor socket of the specified type.
    _monitor_socket = zmq_socket(get_ctx(), type_);
    if (_monitor_socket == NULL)
        return -1;

    //  Never block context termination on pending event messages
    int linger = 0;
    int rc     = zmq_setsockopt(_monitor_socket, ZMQ_LINGER, &linger, sizeof(linger));
    if (rc == -1)
        stop_monitor(false);

    //  Spawn the monitor socket endpoint
    rc = zmq_bind(_monitor_socket, endpoint_);
    if (rc == -1)
        stop_monitor(false);
    return rc;
}

void zmq::socket_base_t::event_connected(const endpoint_uri_pair_t &endpoint_uri_pair_, zmq::fd_t fd_)
{
    uint64_t values[1] = {(uint64_t)fd_};
    event(endpoint_uri_pair_, values, 1, ZMQ_EVENT_CONNECTED);
}

void zmq::socket_base_t::event_connect_delayed(const endpoint_uri_pair_t &endpoint_uri_pair_, int err_)
{
    uint64_t values[1] = {(uint64_t)err_};
    event(endpoint_uri_pair_, values, 1, ZMQ_EVENT_CONNECT_DELAYED);
}

void zmq::socket_base_t::event_connect_retried(const endpoint_uri_pair_t &endpoint_uri_pair_, int interval_)
{
    uint64_t values[1] = {(uint64_t)interval_};
    event(endpoint_uri_pair_, values, 1, ZMQ_EVENT_CONNECT_RETRIED);
}

void zmq::socket_base_t::event_listening(const endpoint_uri_pair_t &endpoint_uri_pair_, zmq::fd_t fd_)
{
    uint64_t values[1] = {(uint64_t)fd_};
    event(endpoint_uri_pair_, values, 1, ZMQ_EVENT_LISTENING);
}

void zmq::socket_base_t::event_bind_failed(const endpoint_uri_pair_t &endpoint_uri_pair_, int err_)
{
    uint64_t values[1] = {(uint64_t)err_};
    event(endpoint_uri_pair_, values, 1, ZMQ_EVENT_BIND_FAILED);
}

void zmq::socket_base_t::event_accepted(const endpoint_uri_pair_t &endpoint_uri_pair_, zmq::fd_t fd_)
{
    uint64_t values[1] = {(uint64_t)fd_};
    event(endpoint_uri_pair_, values, 1, ZMQ_EVENT_ACCEPTED);
}

void zmq::socket_base_t::event_accept_failed(const endpoint_uri_pair_t &endpoint_uri_pair_, int err_)
{
    uint64_t values[1] = {(uint64_t)err_};
    event(endpoint_uri_pair_, values, 1, ZMQ_EVENT_ACCEPT_FAILED);
}

void zmq::socket_base_t::event_closed(const endpoint_uri_pair_t &endpoint_uri_pair_, zmq::fd_t fd_)
{
    uint64_t values[1] = {(uint64_t)fd_};
    event(endpoint_uri_pair_, values, 1, ZMQ_EVENT_CLOSED);
}

void zmq::socket_base_t::event_close_failed(const endpoint_uri_pair_t &endpoint_uri_pair_, int err_)
{
    uint64_t values[1] = {(uint64_t)err_};
    event(endpoint_uri_pair_, values, 1, ZMQ_EVENT_CLOSE_FAILED);
}

void zmq::socket_base_t::event_disconnected(const endpoint_uri_pair_t &endpoint_uri_pair_, zmq::fd_t fd_)
{
    uint64_t values[1] = {(uint64_t)fd_};
    event(endpoint_uri_pair_, values, 1, ZMQ_EVENT_DISCONNECTED);
}

void zmq::socket_base_t::event_handshake_failed_no_detail(const endpoint_uri_pair_t &endpoint_uri_pair_, int err_)
{
    uint64_t values[1] = {(uint64_t)err_};
    event(endpoint_uri_pair_, values, 1, ZMQ_EVENT_HANDSHAKE_FAILED_NO_DETAIL);
}

void zmq::socket_base_t::event_handshake_failed_protocol(const endpoint_uri_pair_t &endpoint_uri_pair_, int err_)
{
    uint64_t values[1] = {(uint64_t)err_};
    event(endpoint_uri_pair_, values, 1, ZMQ_EVENT_HANDSHAKE_FAILED_PROTOCOL);
}

void zmq::socket_base_t::event_handshake_failed_auth(const endpoint_uri_pair_t &endpoint_uri_pair_, int err_)
{
    uint64_t values[1] = {(uint64_t)err_};
    event(endpoint_uri_pair_, values, 1, ZMQ_EVENT_HANDSHAKE_FAILED_AUTH);
}

void zmq::socket_base_t::event_handshake_succeeded(const endpoint_uri_pair_t &endpoint_uri_pair_, int err_)
{
    uint64_t values[1] = {(uint64_t)err_};
    event(endpoint_uri_pair_, values, 1, ZMQ_EVENT_HANDSHAKE_SUCCEEDED);
}

void zmq::socket_base_t::event(const endpoint_uri_pair_t &endpoint_uri_pair_, uint64_t values_[],
                               uint64_t values_count_, uint64_t type_)
{
    scoped_lock_t lock(_monitor_sync);
    if (_monitor_events & type_)
    {
        monitor_event(type_, values_, values_count_, endpoint_uri_pair_);
    }
}

//  Send a monitor event
void zmq::socket_base_t::monitor_event(uint64_t event_, uint64_t values_[], uint64_t values_count_,
                                       const endpoint_uri_pair_t &endpoint_uri_pair_) const
{
    // this is a private method which is only called from
    // contexts where the _monitor_sync mutex has been locked before

    if (_monitor_socket)
    {
        zmq_msg_t msg;

        switch (options.monitor_event_version)
        {
        case 1:
        {
            //  The API should not allow to activate unsupported events
            zmq_assert(event_ <= std::numeric_limits<uint16_t>::max());
            //  v1 only allows one value
            zmq_assert(values_count_ == 1);
            zmq_assert(values_[0] <= std::numeric_limits<uint32_t>::max());

            //  Send event and value in first frame
            const uint16_t event = static_cast<uint16_t>(event_);
            const uint32_t value = static_cast<uint32_t>(values_[0]);
            zmq_msg_init_size(&msg, sizeof(event) + sizeof(value));
            uint8_t *data = static_cast<uint8_t *>(zmq_msg_data(&msg));
            //  Avoid dereferencing uint32_t on unaligned address
            memcpy(data + 0, &event, sizeof(event));
            memcpy(data + sizeof(event), &value, sizeof(value));
            zmq_msg_send(&msg, _monitor_socket, ZMQ_SNDMORE);

            const std::string &endpoint_uri = endpoint_uri_pair_.identifier();

            //  Send address in second frame
            zmq_msg_init_size(&msg, endpoint_uri.size());
            memcpy(zmq_msg_data(&msg), endpoint_uri.c_str(), endpoint_uri.size());
            zmq_msg_send(&msg, _monitor_socket, 0);
        }
        break;
        case 2:
        {
            //  Send event in first frame (64bit unsigned)
            zmq_msg_init_size(&msg, sizeof(event_));
            memcpy(zmq_msg_data(&msg), &event_, sizeof(event_));
            zmq_msg_send(&msg, _monitor_socket, ZMQ_SNDMORE);

            //  Send number of values that will follow in second frame
            zmq_msg_init_size(&msg, sizeof(values_count_));
            memcpy(zmq_msg_data(&msg), &values_count_, sizeof(values_count_));
            zmq_msg_send(&msg, _monitor_socket, ZMQ_SNDMORE);

            //  Send values in third-Nth frames (64bit unsigned)
            for (uint64_t i = 0; i < values_count_; ++i)
            {
                zmq_msg_init_size(&msg, sizeof(values_[i]));
                memcpy(zmq_msg_data(&msg), &values_[i], sizeof(values_[i]));
                zmq_msg_send(&msg, _monitor_socket, ZMQ_SNDMORE);
            }

            //  Send local endpoint URI in second-to-last frame (string)
            zmq_msg_init_size(&msg, endpoint_uri_pair_.local.size());
            memcpy(zmq_msg_data(&msg), endpoint_uri_pair_.local.c_str(), endpoint_uri_pair_.local.size());
            zmq_msg_send(&msg, _monitor_socket, ZMQ_SNDMORE);

            //  Send remote endpoint URI in last frame (string)
            zmq_msg_init_size(&msg, endpoint_uri_pair_.remote.size());
            memcpy(zmq_msg_data(&msg), endpoint_uri_pair_.remote.c_str(), endpoint_uri_pair_.remote.size());
            zmq_msg_send(&msg, _monitor_socket, 0);
        }
        break;
        }
    }
}

void zmq::socket_base_t::stop_monitor(bool send_monitor_stopped_event_)
{
    // this is a private method which is only called from
    // contexts where the _monitor_sync mutex has been locked before

    if (_monitor_socket)
    {
        if ((_monitor_events & ZMQ_EVENT_MONITOR_STOPPED) && send_monitor_stopped_event_)
        {
            uint64_t values[1] = {0};
            monitor_event(ZMQ_EVENT_MONITOR_STOPPED, values, 1, endpoint_uri_pair_t());
        }
        zmq_close(_monitor_socket);
        _monitor_socket = NULL;
        _monitor_events = 0;
    }
}

zmq::routing_socket_base_t::routing_socket_base_t(class ctx_t *parent_, uint32_t tid_, int sid_)
    : socket_base_t(parent_, tid_, sid_)
{
}

zmq::routing_socket_base_t::~routing_socket_base_t() { zmq_assert(_out_pipes.empty()); }

int zmq::routing_socket_base_t::xsetsockopt(int option_, const void *optval_, size_t optvallen_)
{
    switch (option_)
    {
    case ZMQ_CONNECT_ROUTING_ID:
        // TODO why isn't it possible to set an empty connect_routing_id
        //   (which is the default value)
        if (optval_ && optvallen_)
        {
            _connect_routing_id.assign(static_cast<const char *>(optval_), optvallen_);
            return 0;
        }
        break;
    }
    errno = EINVAL;
    return -1;
}

void zmq::routing_socket_base_t::xwrite_activated(pipe_t *pipe_)
{
    const out_pipes_t::iterator end = _out_pipes.end();
    out_pipes_t::iterator       it;
    for (it = _out_pipes.begin(); it != end; ++it)
        if (it->second.pipe == pipe_)
            break;

    zmq_assert(it != end);
    zmq_assert(!it->second.active);
    it->second.active = true;
}

std::string zmq::routing_socket_base_t::extract_connect_routing_id()
{
    std::string res = ZMQ_MOVE(_connect_routing_id);
    _connect_routing_id.clear();
    return res;
}

bool zmq::routing_socket_base_t::connect_routing_id_is_set() const { return !_connect_routing_id.empty(); }

void zmq::routing_socket_base_t::add_out_pipe(blob_t routing_id_, pipe_t *pipe_)
{
    //  Add the record into output pipes lookup table
    const out_pipe_t outpipe = {pipe_, true};
    const bool       ok      = _out_pipes.ZMQ_MAP_INSERT_OR_EMPLACE(ZMQ_MOVE(routing_id_), outpipe).second;
    zmq_assert(ok);
}

bool zmq::routing_socket_base_t::has_out_pipe(const blob_t &routing_id_) const
{
    return 0 != _out_pipes.count(routing_id_);
}

zmq::routing_socket_base_t::out_pipe_t *zmq::routing_socket_base_t::lookup_out_pipe(const blob_t &routing_id_)
{
    // TODO we could probably avoid constructor a temporary blob_t to call this function
    out_pipes_t::iterator it = _out_pipes.find(routing_id_);
    return it == _out_pipes.end() ? NULL : &it->second;
}

const zmq::routing_socket_base_t::out_pipe_t *
zmq::routing_socket_base_t::lookup_out_pipe(const blob_t &routing_id_) const
{
    // TODO we could probably avoid constructor a temporary blob_t to call this function
    const out_pipes_t::const_iterator it = _out_pipes.find(routing_id_);
    return it == _out_pipes.end() ? NULL : &it->second;
}

void zmq::routing_socket_base_t::erase_out_pipe(pipe_t *pipe_)
{
    const size_t erased = _out_pipes.erase(pipe_->get_routing_id());
    zmq_assert(erased);
}

zmq::routing_socket_base_t::out_pipe_t zmq::routing_socket_base_t::try_erase_out_pipe(const blob_t &routing_id_)
{
    const out_pipes_t::iterator it  = _out_pipes.find(routing_id_);
    out_pipe_t                  res = {NULL, false};
    if (it != _out_pipes.end())
    {
        res = it->second;
        _out_pipes.erase(it);
    }
    return res;
}

//========= end of socket_base.cpp ============

//========= begin of socket_poller.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "socket_poller.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "polling_util.hpp"
// ans ignore: #include "macros.hpp"

#include <limits.h>

static bool is_thread_safe(zmq::socket_base_t &socket_)
{
    // do not use getsockopt here, since that would fail during context termination
    return socket_.is_thread_safe();
}

zmq::socket_poller_t::socket_poller_t()
    : _tag(0xCAFEBABE), _signaler(NULL)
#if defined ZMQ_POLL_BASED_ON_POLL
      ,
      _pollfds(NULL)
#elif defined ZMQ_POLL_BASED_ON_SELECT
      ,
      _max_fd(0)
#endif
{
    rebuild();
}

zmq::socket_poller_t::~socket_poller_t()
{
    //  Mark the socket_poller as dead
    _tag = 0xdeadbeef;

    for (items_t::iterator it = _items.begin(), end = _items.end(); it != end; ++it)
    {
        // TODO shouldn't this zmq_assert (it->socket->check_tag ()) instead?
        if (it->socket && it->socket->check_tag() && is_thread_safe(*it->socket))
        {
            it->socket->remove_signaler(_signaler);
        }
    }

    if (_signaler != NULL)
    {
        LIBZMQ_DELETE(_signaler);
    }

#if defined ZMQ_POLL_BASED_ON_POLL
    if (_pollfds)
    {
        free(_pollfds);
        _pollfds = NULL;
    }
#endif
}

bool zmq::socket_poller_t::check_tag() { return _tag == 0xCAFEBABE; }

int zmq::socket_poller_t::signaler_fd(fd_t *fd_)
{
    if (_signaler)
    {
        *fd_ = _signaler->get_fd();
        return 0;
    }
    else
    {
        // Only thread-safe socket types are guaranteed to have a signaler.
        errno = EINVAL;
        return -1;
    }
}

int zmq::socket_poller_t::add(socket_base_t *socket_, void *user_data_, short events_)
{
    for (items_t::iterator it = _items.begin(), end = _items.end(); it != end; ++it)
    {
        if (it->socket == socket_)
        {
            errno = EINVAL;
            return -1;
        }
    }

    if (is_thread_safe(*socket_))
    {
        if (_signaler == NULL)
        {
            _signaler = new (std::nothrow) signaler_t();
            if (!_signaler)
            {
                errno = ENOMEM;
                return -1;
            }
            if (!_signaler->valid())
            {
                delete _signaler;
                _signaler = NULL;
                errno     = EMFILE;
                return -1;
            }
        }

        socket_->add_signaler(_signaler);
    }

    item_t item = {
        socket_,
        0,
        user_data_,
        events_
#if defined ZMQ_POLL_BASED_ON_POLL
        ,
        -1
#endif
    };
    try
    {
        _items.push_back(item);
    }
    catch (const std::bad_alloc &)
    {
        errno = ENOMEM;
        return -1;
    }
    _need_rebuild = true;

    return 0;
}

int zmq::socket_poller_t::add_fd(fd_t fd_, void *user_data_, short events_)
{
    for (items_t::iterator it = _items.begin(), end = _items.end(); it != end; ++it)
    {
        if (!it->socket && it->fd == fd_)
        {
            errno = EINVAL;
            return -1;
        }
    }

    item_t item = {
        NULL,
        fd_,
        user_data_,
        events_
#if defined ZMQ_POLL_BASED_ON_POLL
        ,
        -1
#endif
    };
    try
    {
        _items.push_back(item);
    }
    catch (const std::bad_alloc &)
    {
        errno = ENOMEM;
        return -1;
    }
    _need_rebuild = true;

    return 0;
}

int zmq::socket_poller_t::modify(socket_base_t *socket_, short events_)
{
    const items_t::iterator end = _items.end();
    items_t::iterator       it;

    for (it = _items.begin(); it != end; ++it)
    {
        if (it->socket == socket_)
            break;
    }

    if (it == end)
    {
        errno = EINVAL;
        return -1;
    }

    it->events    = events_;
    _need_rebuild = true;

    return 0;
}

int zmq::socket_poller_t::modify_fd(fd_t fd_, short events_)
{
    const items_t::iterator end = _items.end();
    items_t::iterator       it;

    for (it = _items.begin(); it != end; ++it)
    {
        if (!it->socket && it->fd == fd_)
            break;
    }

    if (it == end)
    {
        errno = EINVAL;
        return -1;
    }

    it->events    = events_;
    _need_rebuild = true;

    return 0;
}

int zmq::socket_poller_t::remove(socket_base_t *socket_)
{
    const items_t::iterator end = _items.end();
    items_t::iterator       it;

    for (it = _items.begin(); it != end; ++it)
    {
        if (it->socket == socket_)
            break;
    }

    if (it == end)
    {
        errno = EINVAL;
        return -1;
    }

    _items.erase(it);
    _need_rebuild = true;

    if (is_thread_safe(*socket_))
    {
        socket_->remove_signaler(_signaler);
    }

    return 0;
}

int zmq::socket_poller_t::remove_fd(fd_t fd_)
{
    const items_t::iterator end = _items.end();
    items_t::iterator       it;

    for (it = _items.begin(); it != end; ++it)
    {
        if (!it->socket && it->fd == fd_)
            break;
    }

    if (it == end)
    {
        errno = EINVAL;
        return -1;
    }

    _items.erase(it);
    _need_rebuild = true;

    return 0;
}

int zmq::socket_poller_t::rebuild()
{
    _use_signaler = false;
    _pollset_size = 0;
    _need_rebuild = false;

#if defined ZMQ_POLL_BASED_ON_POLL

    if (_pollfds)
    {
        free(_pollfds);
        _pollfds = NULL;
    }

    for (items_t::iterator it = _items.begin(), end = _items.end(); it != end; ++it)
    {
        if (it->events)
        {
            if (it->socket && is_thread_safe(*it->socket))
            {
                if (!_use_signaler)
                {
                    _use_signaler = true;
                    _pollset_size++;
                }
            }
            else
                _pollset_size++;
        }
    }

    if (_pollset_size == 0)
        return 0;

    _pollfds = static_cast<pollfd *>(malloc(_pollset_size * sizeof(pollfd)));

    if (!_pollfds)
    {
        errno         = ENOMEM;
        _need_rebuild = true;
        return -1;
    }

    int item_nbr = 0;

    if (_use_signaler)
    {
        item_nbr           = 1;
        _pollfds[0].fd     = _signaler->get_fd();
        _pollfds[0].events = POLLIN;
    }

    for (items_t::iterator it = _items.begin(), end = _items.end(); it != end; ++it)
    {
        if (it->events)
        {
            if (it->socket)
            {
                if (!is_thread_safe(*it->socket))
                {
                    size_t fd_size = sizeof(zmq::fd_t);
                    int    rc      = it->socket->getsockopt(ZMQ_FD, &_pollfds[item_nbr].fd, &fd_size);
                    zmq_assert(rc == 0);

                    _pollfds[item_nbr].events = POLLIN;
                    item_nbr++;
                }
            }
            else
            {
                _pollfds[item_nbr].fd     = it->fd;
                _pollfds[item_nbr].events = (it->events & ZMQ_POLLIN ? POLLIN : 0) |
                                            (it->events & ZMQ_POLLOUT ? POLLOUT : 0) |
                                            (it->events & ZMQ_POLLPRI ? POLLPRI : 0);
                it->pollfd_index = item_nbr;
                item_nbr++;
            }
        }
    }

#elif defined ZMQ_POLL_BASED_ON_SELECT

    //  Ensure we do not attempt to select () on more than FD_SETSIZE
    //  file descriptors.
    zmq_assert(_items.size() <= FD_SETSIZE);

    _pollset_in.resize(_items.size());
    _pollset_out.resize(_items.size());
    _pollset_err.resize(_items.size());

    FD_ZERO(_pollset_in.get());
    FD_ZERO(_pollset_out.get());
    FD_ZERO(_pollset_err.get());

    for (items_t::iterator it = _items.begin(), end = _items.end(); it != end; ++it)
    {
        if (it->socket && is_thread_safe(*it->socket) && it->events)
        {
            _use_signaler = true;
            FD_SET(_signaler->get_fd(), _pollset_in.get());
            _pollset_size = 1;
            break;
        }
    }

    _max_fd = 0;

    //  Build the fd_sets for passing to select ().
    for (items_t::iterator it = _items.begin(), end = _items.end(); it != end; ++it)
    {
        if (it->events)
        {
            //  If the poll item is a 0MQ socket we are interested in input on the
            //  notification file descriptor retrieved by the ZMQ_FD socket option.
            if (it->socket)
            {
                if (!is_thread_safe(*it->socket))
                {
                    zmq::fd_t notify_fd;
                    size_t    fd_size = sizeof(zmq::fd_t);
                    int       rc      = it->socket->getsockopt(ZMQ_FD, &notify_fd, &fd_size);
                    zmq_assert(rc == 0);

                    FD_SET(notify_fd, _pollset_in.get());
                    if (_max_fd < notify_fd)
                        _max_fd = notify_fd;

                    _pollset_size++;
                }
            }
            //  Else, the poll item is a raw file descriptor. Convert the poll item
            //  events to the appropriate fd_sets.
            else
            {
                if (it->events & ZMQ_POLLIN)
                    FD_SET(it->fd, _pollset_in.get());
                if (it->events & ZMQ_POLLOUT)
                    FD_SET(it->fd, _pollset_out.get());
                if (it->events & ZMQ_POLLERR)
                    FD_SET(it->fd, _pollset_err.get());
                if (_max_fd < it->fd)
                    _max_fd = it->fd;

                _pollset_size++;
            }
        }
    }

#endif

    return 0;
}

void zmq::socket_poller_t::zero_trail_events(zmq::socket_poller_t::event_t *events_, int n_events_, int found_)
{
    for (int i = found_; i < n_events_; ++i)
    {
        events_[i].socket    = NULL;
        events_[i].fd        = 0;
        events_[i].user_data = NULL;
        events_[i].events    = 0;
    }
}

#if defined   ZMQ_POLL_BASED_ON_POLL
int           zmq::socket_poller_t::check_events(zmq::socket_poller_t::event_t *events_, int n_events_)
#elif defined ZMQ_POLL_BASED_ON_SELECT
int         zmq::socket_poller_t::check_events(zmq::socket_poller_t::event_t *events_, int n_events_, fd_set &inset_,
                                       fd_set &outset_, fd_set &errset_)
#endif
{
    int found = 0;
    for (items_t::iterator it = _items.begin(), end = _items.end(); it != end && found < n_events_; ++it)
    {
        //  The poll item is a 0MQ socket. Retrieve pending events
        //  using the ZMQ_EVENTS socket option.
        if (it->socket)
        {
            size_t   events_size = sizeof(uint32_t);
            uint32_t events;
            if (it->socket->getsockopt(ZMQ_EVENTS, &events, &events_size) == -1)
            {
                return -1;
            }

            if (it->events & events)
            {
                events_[found].socket    = it->socket;
                events_[found].user_data = it->user_data;
                events_[found].events    = it->events & events;
                ++found;
            }
        }
        //  Else, the poll item is a raw file descriptor, simply convert
        //  the events to zmq_pollitem_t-style format.
        else
        {
#if defined ZMQ_POLL_BASED_ON_POLL

            short revents = _pollfds[it->pollfd_index].revents;
            short events  = 0;

            if (revents & POLLIN)
                events |= ZMQ_POLLIN;
            if (revents & POLLOUT)
                events |= ZMQ_POLLOUT;
            if (revents & POLLPRI)
                events |= ZMQ_POLLPRI;
            if (revents & ~(POLLIN | POLLOUT | POLLPRI))
                events |= ZMQ_POLLERR;

#elif defined ZMQ_POLL_BASED_ON_SELECT

            short events = 0;

            if (FD_ISSET(it->fd, &inset_))
                events |= ZMQ_POLLIN;
            if (FD_ISSET(it->fd, &outset_))
                events |= ZMQ_POLLOUT;
            if (FD_ISSET(it->fd, &errset_))
                events |= ZMQ_POLLERR;
#endif // POLL_SELECT

            if (events)
            {
                events_[found].socket    = NULL;
                events_[found].user_data = it->user_data;
                events_[found].fd        = it->fd;
                events_[found].events    = events;
                ++found;
            }
        }
    }

    return found;
}

// Return 0 if timeout is expired otherwise 1
int zmq::socket_poller_t::adjust_timeout(zmq::clock_t &clock_, long timeout_, uint64_t &now_, uint64_t &end_,
                                         bool &first_pass_)
{
    //  If socket_poller_t::timeout is zero, exit immediately whether there
    //  are events or not.
    if (timeout_ == 0)
        return 0;

    //  At this point we are meant to wait for events but there are none.
    //  If timeout is infinite we can just loop until we get some events.
    if (timeout_ < 0)
    {
        if (first_pass_)
            first_pass_ = false;
        return 1;
    }

    //  The timeout is finite and there are no events. In the first pass
    //  we get a timestamp of when the polling have begun. (We assume that
    //  first pass have taken negligible time). We also compute the time
    //  when the polling should time out.
    now_ = clock_.now_ms();
    if (first_pass_)
    {
        end_        = now_ + timeout_;
        first_pass_ = false;
        return 1;
    }

    //  Find out whether timeout have expired.
    if (now_ >= end_)
        return 0;

    return 1;
}

int zmq::socket_poller_t::wait(zmq::socket_poller_t::event_t *events_, int n_events_, long timeout_)
{
    if (_items.empty() && timeout_ < 0)
    {
        errno = EFAULT;
        return -1;
    }

    if (_need_rebuild)
    {
        int rc = rebuild();
        if (rc == -1)
            return -1;
    }

    if (unlikely(_pollset_size == 0))
    {
        // We'll report an error (timed out) as if the list was non-empty and
        // no event occurred within the specified timeout. Otherwise the caller
        // needs to check the return value AND the event to avoid using the
        // nullified event data.
        errno = EAGAIN;
        if (timeout_ == 0)
            return -1;
#if defined ZMQ_HAVE_WINDOWS
        Sleep(timeout_ > 0 ? timeout_ : INFINITE);
        return -1;
#elif defined ZMQ_HAVE_ANDROID
        usleep(timeout_ * 1000);
        return -1;
#elif defined ZMQ_HAVE_OSX
        usleep(timeout_ * 1000);
        errno = EAGAIN;
        return -1;
#elif defined ZMQ_HAVE_VXWORKS
        struct timespec ns_;
        ns_.tv_sec = timeout_ / 1000;
        ns_.tv_nsec = timeout_ % 1000 * 1000000;
        nanosleep(&ns_, 0);
        return -1;
#else
        usleep(timeout_ * 1000);
        return -1;
#endif
    }

#if defined ZMQ_POLL_BASED_ON_POLL
    zmq::clock_t clock;
    uint64_t     now = 0;
    uint64_t     end = 0;

    bool first_pass = true;

    while (true)
    {
        //  Compute the timeout for the subsequent poll.
        int timeout;
        if (first_pass)
            timeout = 0;
        else if (timeout_ < 0)
            timeout = -1;
        else
            timeout = static_cast<int>(std::min<uint64_t>(end - now, INT_MAX));

        //  Wait for events.
        while (true)
        {
            int rc = poll(_pollfds, _pollset_size, timeout);
            if (rc == -1 && errno == EINTR)
            {
                return -1;
            }
            errno_assert(rc >= 0);
            break;
        }

        //  Receive the signal from pollfd
        if (_use_signaler && _pollfds[0].revents & POLLIN)
            _signaler->recv();

        //  Check for the events.
        int found = check_events(events_, n_events_);
        if (found)
        {
            if (found > 0)
                zero_trail_events(events_, n_events_, found);
            return found;
        }

        //  Adjust timeout or break
        if (adjust_timeout(clock, timeout_, now, end, first_pass) == 0)
            break;
    }
    errno = EAGAIN;
    return -1;

#elif defined ZMQ_POLL_BASED_ON_SELECT

    zmq::clock_t clock;
    uint64_t     now = 0;
    uint64_t     end = 0;

    bool first_pass = true;

    optimized_fd_set_t inset(_pollset_size);
    optimized_fd_set_t outset(_pollset_size);
    optimized_fd_set_t errset(_pollset_size);

    while (true)
    {
        //  Compute the timeout for the subsequent poll.
        timeval  timeout;
        timeval *ptimeout;
        if (first_pass)
        {
            timeout.tv_sec  = 0;
            timeout.tv_usec = 0;
            ptimeout        = &timeout;
        }
        else if (timeout_ < 0)
            ptimeout = NULL;
        else
        {
            timeout.tv_sec  = static_cast<long>((end - now) / 1000);
            timeout.tv_usec = static_cast<long>((end - now) % 1000 * 1000);
            ptimeout        = &timeout;
        }

        //  Wait for events. Ignore interrupts if there's infinite timeout.
        while (true)
        {
            memcpy(inset.get(), _pollset_in.get(), valid_pollset_bytes(*_pollset_in.get()));
            memcpy(outset.get(), _pollset_out.get(), valid_pollset_bytes(*_pollset_out.get()));
            memcpy(errset.get(), _pollset_err.get(), valid_pollset_bytes(*_pollset_err.get()));
            const int rc = select(static_cast<int>(_max_fd + 1), inset.get(), outset.get(), errset.get(), ptimeout);
#if defined   ZMQ_HAVE_WINDOWS
            if (unlikely(rc == SOCKET_ERROR))
            {
                errno = wsa_error_to_errno(WSAGetLastError());
                wsa_assert(errno == ENOTSOCK);
                return -1;
            }
#else
            if (unlikely(rc == -1))
            {
                errno_assert(errno == EINTR || errno == EBADF);
                return -1;
            }
#endif
            break;
        }

        if (_use_signaler && FD_ISSET(_signaler->get_fd(), inset.get()))
            _signaler->recv();

        //  Check for the events.
        const int found = check_events(events_, n_events_, *inset.get(), *outset.get(), *errset.get());
        if (found)
        {
            if (found > 0)
                zero_trail_events(events_, n_events_, found);
            return found;
        }

        //  Adjust timeout or break
        if (adjust_timeout(clock, timeout_, now, end, first_pass) == 0)
            break;
    }

    errno = EAGAIN;
    return -1;

#else

    //  Exotic platforms that support neither poll() nor select().
    errno = ENOTSUP;
    return -1;

#endif
}

//========= end of socket_poller.cpp ============

//========= begin of socks.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <sys/types.h>

// ans ignore: #include "err.hpp"
// ans ignore: #include "socks.hpp"
// ans ignore: #include "tcp.hpp"
// ans ignore: #include "blob.hpp"

#ifndef ZMQ_HAVE_WINDOWS
#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>
#endif

zmq::socks_greeting_t::socks_greeting_t(uint8_t method_) : num_methods(1) { methods[0] = method_; }

zmq::socks_greeting_t::socks_greeting_t(const uint8_t *methods_, uint8_t num_methods_) : num_methods(num_methods_)
{
    for (uint8_t i = 0; i < num_methods_; i++)
        methods[i] = methods_[i];
}

zmq::socks_greeting_encoder_t::socks_greeting_encoder_t() : _bytes_encoded(0), _bytes_written(0) {}

void zmq::socks_greeting_encoder_t::encode(const socks_greeting_t &greeting_)
{
    uint8_t *ptr = _buf;

    *ptr++ = 0x05;
    *ptr++ = static_cast<uint8_t>(greeting_.num_methods);
    for (uint8_t i = 0; i < greeting_.num_methods; i++)
        *ptr++ = greeting_.methods[i];

    _bytes_encoded = 2 + greeting_.num_methods;
    _bytes_written = 0;
}

int zmq::socks_greeting_encoder_t::output(fd_t fd_)
{
    const int rc = tcp_write(fd_, _buf + _bytes_written, _bytes_encoded - _bytes_written);
    if (rc > 0)
        _bytes_written += static_cast<size_t>(rc);
    return rc;
}

bool zmq::socks_greeting_encoder_t::has_pending_data() const { return _bytes_written < _bytes_encoded; }

void zmq::socks_greeting_encoder_t::reset() { _bytes_encoded = _bytes_written = 0; }

zmq::socks_choice_t::socks_choice_t(unsigned char method_) : method(method_) {}

zmq::socks_choice_decoder_t::socks_choice_decoder_t() : _bytes_read(0) {}

int zmq::socks_choice_decoder_t::input(fd_t fd_)
{
    zmq_assert(_bytes_read < 2);
    const int rc = tcp_read(fd_, _buf + _bytes_read, 2 - _bytes_read);
    if (rc > 0)
    {
        _bytes_read += static_cast<size_t>(rc);
        if (_buf[0] != 0x05)
            return -1;
    }
    return rc;
}

bool zmq::socks_choice_decoder_t::message_ready() const { return _bytes_read == 2; }

zmq::socks_choice_t zmq::socks_choice_decoder_t::decode()
{
    zmq_assert(message_ready());
    return socks_choice_t(_buf[1]);
}

void zmq::socks_choice_decoder_t::reset() { _bytes_read = 0; }

zmq::socks_basic_auth_request_t::socks_basic_auth_request_t(std::string username_, std::string password_)
    : username(username_), password(password_)
{
    zmq_assert(username_.size() <= UINT8_MAX);
    zmq_assert(password_.size() <= UINT8_MAX);
}

zmq::socks_basic_auth_request_encoder_t::socks_basic_auth_request_encoder_t() : _bytes_encoded(0), _bytes_written(0) {}

void zmq::socks_basic_auth_request_encoder_t::encode(const socks_basic_auth_request_t &req_)
{
    unsigned char *ptr = _buf;
    *ptr++             = 0x01;
    *ptr++             = static_cast<unsigned char>(req_.username.size());
    memcpy(ptr, req_.username.c_str(), req_.username.size());
    ptr += req_.username.size();
    *ptr++ = static_cast<unsigned char>(req_.password.size());
    memcpy(ptr, req_.password.c_str(), req_.password.size());
    ptr += req_.password.size();

    _bytes_encoded = ptr - _buf;
    _bytes_written = 0;
}

int zmq::socks_basic_auth_request_encoder_t::output(fd_t fd_)
{
    const int rc = tcp_write(fd_, _buf + _bytes_written, _bytes_encoded - _bytes_written);
    if (rc > 0)
        _bytes_written += static_cast<size_t>(rc);
    return rc;
}

bool zmq::socks_basic_auth_request_encoder_t::has_pending_data() const { return _bytes_written < _bytes_encoded; }

void zmq::socks_basic_auth_request_encoder_t::reset() { _bytes_encoded = _bytes_written = 0; }

zmq::socks_auth_response_t::socks_auth_response_t(uint8_t response_code_) : response_code(response_code_) {}

zmq::socks_auth_response_decoder_t::socks_auth_response_decoder_t() : _bytes_read(0) {}

int zmq::socks_auth_response_decoder_t::input(fd_t fd_)
{
    zmq_assert(_bytes_read < 2);
    const int rc = tcp_read(fd_, _buf + _bytes_read, 2 - _bytes_read);
    if (rc > 0)
    {
        _bytes_read += static_cast<size_t>(rc);
        if (_buf[0] != 0x01)
            return -1;
    }
    return rc;
}

bool zmq::socks_auth_response_decoder_t::message_ready() const { return _bytes_read == 2; }

zmq::socks_auth_response_t zmq::socks_auth_response_decoder_t::decode()
{
    zmq_assert(message_ready());
    return socks_auth_response_t(_buf[1]);
}

void zmq::socks_auth_response_decoder_t::reset() { _bytes_read = 0; }

zmq::socks_request_t::socks_request_t(uint8_t command_, std::string hostname_, uint16_t port_)
    : command(command_), hostname(ZMQ_MOVE(hostname_)), port(port_)
{
    zmq_assert(hostname.size() <= UINT8_MAX);
}

zmq::socks_request_encoder_t::socks_request_encoder_t() : _bytes_encoded(0), _bytes_written(0) {}

void zmq::socks_request_encoder_t::encode(const socks_request_t &req_)
{
    zmq_assert(req_.hostname.size() <= UINT8_MAX);

    unsigned char *ptr = _buf;
    *ptr++             = 0x05;
    *ptr++             = req_.command;
    *ptr++             = 0x00;

#if defined ZMQ_HAVE_OPENVMS && defined __ia64 && __INITIAL_POINTER_SIZE == 64
    __addrinfo64 hints, *res = NULL;
#else
    addrinfo hints, *res = NULL;
#endif

    memset(&hints, 0, sizeof hints);

    //  Suppress potential DNS lookups.
    hints.ai_flags = AI_NUMERICHOST;

    const int rc = getaddrinfo(req_.hostname.c_str(), NULL, &hints, &res);
    if (rc == 0 && res->ai_family == AF_INET)
    {
        const struct sockaddr_in *sockaddr_in = reinterpret_cast<const struct sockaddr_in *>(res->ai_addr);
        *ptr++                                = 0x01;
        memcpy(ptr, &sockaddr_in->sin_addr, 4);
        ptr += 4;
    }
    else if (rc == 0 && res->ai_family == AF_INET6)
    {
        const struct sockaddr_in6 *sockaddr_in6 = reinterpret_cast<const struct sockaddr_in6 *>(res->ai_addr);
        *ptr++                                  = 0x04;
        memcpy(ptr, &sockaddr_in6->sin6_addr, 16);
        ptr += 16;
    }
    else
    {
        *ptr++ = 0x03;
        *ptr++ = static_cast<unsigned char>(req_.hostname.size());
        memcpy(ptr, req_.hostname.c_str(), req_.hostname.size());
        ptr += req_.hostname.size();
    }

    if (rc == 0)
        freeaddrinfo(res);

    *ptr++ = req_.port / 256;
    *ptr++ = req_.port % 256;

    _bytes_encoded = ptr - _buf;
    _bytes_written = 0;
}

int zmq::socks_request_encoder_t::output(fd_t fd_)
{
    const int rc = tcp_write(fd_, _buf + _bytes_written, _bytes_encoded - _bytes_written);
    if (rc > 0)
        _bytes_written += static_cast<size_t>(rc);
    return rc;
}

bool zmq::socks_request_encoder_t::has_pending_data() const { return _bytes_written < _bytes_encoded; }

void zmq::socks_request_encoder_t::reset() { _bytes_encoded = _bytes_written = 0; }

zmq::socks_response_t::socks_response_t(uint8_t response_code_, std::string address_, uint16_t port_)
    : response_code(response_code_), address(address_), port(port_)
{
}

zmq::socks_response_decoder_t::socks_response_decoder_t() : _bytes_read(0) {}

int zmq::socks_response_decoder_t::input(fd_t fd_)
{
    size_t n = 0;

    if (_bytes_read < 5)
        n = 5 - _bytes_read;
    else
    {
        const uint8_t atyp = _buf[3];
        zmq_assert(atyp == 0x01 || atyp == 0x03 || atyp == 0x04);
        if (atyp == 0x01)
            n = 3 + 2;
        else if (atyp == 0x03)
            n = _buf[4] + 2;
        else if (atyp == 0x04)
            n = 15 + 2;
    }
    const int rc = tcp_read(fd_, _buf + _bytes_read, n);
    if (rc > 0)
    {
        _bytes_read += static_cast<size_t>(rc);
        if (_buf[0] != 0x05)
            return -1;
        if (_bytes_read >= 2)
            if (_buf[1] > 0x08)
                return -1;
        if (_bytes_read >= 3)
            if (_buf[2] != 0x00)
                return -1;
        if (_bytes_read >= 4)
        {
            const uint8_t atyp = _buf[3];
            if (atyp != 0x01 && atyp != 0x03 && atyp != 0x04)
                return -1;
        }
    }
    return rc;
}

bool zmq::socks_response_decoder_t::message_ready() const
{
    if (_bytes_read < 4)
        return false;

    const uint8_t atyp = _buf[3];
    zmq_assert(atyp == 0x01 || atyp == 0x03 || atyp == 0x04);
    if (atyp == 0x01)
        return _bytes_read == 10;
    if (atyp == 0x03)
        return _bytes_read > 4 && _bytes_read == 4 + 1 + _buf[4] + 2u;

    return _bytes_read == 22;
}

zmq::socks_response_t zmq::socks_response_decoder_t::decode()
{
    zmq_assert(message_ready());
    return socks_response_t(_buf[1], "", 0);
}

void zmq::socks_response_decoder_t::reset() { _bytes_read = 0; }

//========= end of socks.cpp ============

//========= begin of socks_connecter.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <new>
#include <string>

// ans ignore: #include "macros.hpp"
// ans ignore: #include "socks_connecter.hpp"
// ans ignore: #include "stream_engine.hpp"
// ans ignore: #include "random.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "ip.hpp"
// ans ignore: #include "tcp.hpp"
// ans ignore: #include "address.hpp"
// ans ignore: #include "tcp_address.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "socks.hpp"

#ifndef ZMQ_HAVE_WINDOWS
#include <sys/socket.h>
#include <sys/types.h>
#include <unistd.h>
#if defined ZMQ_HAVE_VXWORKS
#include <sockLib.h>
#endif
#endif

zmq::socks_connecter_t::socks_connecter_t(class io_thread_t *io_thread_, class session_base_t *session_,
                                          const options_t &options_, address_t *addr_, address_t *proxy_addr_,
                                          bool delayed_start_)
    : stream_connecter_base_t(io_thread_, session_, options_, addr_, delayed_start_), _proxy_addr(proxy_addr_),
      _auth_method(socks_no_auth_required), _status(unplugged)
{
    zmq_assert(_addr->protocol == protocol_name::tcp);
    _proxy_addr->to_string(_endpoint);
}

zmq::socks_connecter_t::~socks_connecter_t() { LIBZMQ_DELETE(_proxy_addr); }

void zmq::socks_connecter_t::set_auth_method_none()
{
    _auth_method = socks_no_auth_required;
    _auth_username.clear();
    _auth_password.clear();
}

void zmq::socks_connecter_t::set_auth_method_basic(const std::string username, const std::string password)
{
    _auth_method   = socks_basic_auth;
    _auth_username = username;
    _auth_password = password;
}

void zmq::socks_connecter_t::in_event()
{
    int expected_status = -1;
    zmq_assert(_status != unplugged);

    if (_status == waiting_for_choice)
    {
        int rc = _choice_decoder.input(_s);
        if (rc == 0 || rc == -1)
            error();
        else if (_choice_decoder.message_ready())
        {
            const socks_choice_t choice = _choice_decoder.decode();
            rc                          = process_server_response(choice);
            if (rc == -1)
                error();
            else
            {
                if (choice.method == socks_basic_auth)
                    expected_status = sending_basic_auth_request;
                else
                    expected_status = sending_request;
            }
        }
    }
    else if (_status == waiting_for_auth_response)
    {
        int rc = _auth_response_decoder.input(_s);
        if (rc == 0 || rc == -1)
            error();
        else if (_auth_response_decoder.message_ready())
        {
            const socks_auth_response_t auth_response = _auth_response_decoder.decode();
            rc                                        = process_server_response(auth_response);
            if (rc == -1)
                error();
            else
            {
                expected_status = sending_request;
            }
        }
    }
    else if (_status == waiting_for_response)
    {
        int rc = _response_decoder.input(_s);
        if (rc == 0 || rc == -1)
            error();
        else if (_response_decoder.message_ready())
        {
            const socks_response_t response = _response_decoder.decode();
            rc                              = process_server_response(response);
            if (rc == -1)
                error();
            else
            {
                rm_handle();
                create_engine(_s, get_socket_name<tcp_address_t>(_s, socket_end_local));
                _s      = -1;
                _status = unplugged;
            }
        }
    }
    else
        error();

    if (expected_status == sending_basic_auth_request)
    {
        _basic_auth_request_encoder.encode(socks_basic_auth_request_t(_auth_username, _auth_password));
        reset_pollin(_handle);
        set_pollout(_handle);
        _status = sending_basic_auth_request;
    }
    else if (expected_status == sending_request)
    {
        std::string hostname;
        uint16_t    port = 0;
        if (parse_address(_addr->address, hostname, port) == -1)
            error();
        else
        {
            _request_encoder.encode(socks_request_t(1, hostname, port));
            reset_pollin(_handle);
            set_pollout(_handle);
            _status = sending_request;
        }
    }
}

void zmq::socks_connecter_t::out_event()
{
    zmq_assert(_status == waiting_for_proxy_connection || _status == sending_greeting ||
               _status == sending_basic_auth_request || _status == sending_request);

    if (_status == waiting_for_proxy_connection)
    {
        const int rc = static_cast<int>(check_proxy_connection());
        if (rc == -1)
            error();
        else
        {
            _greeting_encoder.encode(socks_greeting_t(_auth_method));
            _status = sending_greeting;
        }
    }
    else if (_status == sending_greeting)
    {
        zmq_assert(_greeting_encoder.has_pending_data());
        const int rc = _greeting_encoder.output(_s);
        if (rc == -1 || rc == 0)
            error();
        else if (!_greeting_encoder.has_pending_data())
        {
            reset_pollout(_handle);
            set_pollin(_handle);
            _status = waiting_for_choice;
        }
    }
    else if (_status == sending_basic_auth_request)
    {
        zmq_assert(_basic_auth_request_encoder.has_pending_data());
        const int rc = _basic_auth_request_encoder.output(_s);
        if (rc == -1 || rc == 0)
            error();
        else if (!_basic_auth_request_encoder.has_pending_data())
        {
            reset_pollout(_handle);
            set_pollin(_handle);
            _status = waiting_for_auth_response;
        }
    }
    else
    {
        zmq_assert(_request_encoder.has_pending_data());
        const int rc = _request_encoder.output(_s);
        if (rc == -1 || rc == 0)
            error();
        else if (!_request_encoder.has_pending_data())
        {
            reset_pollout(_handle);
            set_pollin(_handle);
            _status = waiting_for_response;
        }
    }
}

void zmq::socks_connecter_t::start_connecting()
{
    zmq_assert(_status == unplugged);

    //  Open the connecting socket.
    const int rc = connect_to_proxy();

    //  Connect may succeed in synchronous manner.
    if (rc == 0)
    {
        _handle = add_fd(_s);
        set_pollout(_handle);
        _status = sending_greeting;
    }
    //  Connection establishment may be delayed. Poll for its completion.
    else if (errno == EINPROGRESS)
    {
        _handle = add_fd(_s);
        set_pollout(_handle);
        _status = waiting_for_proxy_connection;
        _socket->event_connect_delayed(make_unconnected_connect_endpoint_pair(_endpoint), zmq_errno());
    }
    //  Handle any other error condition by eventual reconnect.
    else
    {
        if (_s != retired_fd)
            close();
        add_reconnect_timer();
    }
}

int zmq::socks_connecter_t::process_server_response(const socks_choice_t &response_)
{
    return response_.method == socks_no_auth_required || response_.method == socks_basic_auth ? 0 : -1;
}

int zmq::socks_connecter_t::process_server_response(const socks_response_t &response_)
{
    return response_.response_code == 0 ? 0 : -1;
}

int zmq::socks_connecter_t::process_server_response(const socks_auth_response_t &response_)
{
    return response_.response_code == 0 ? 0 : -1;
}

void zmq::socks_connecter_t::error()
{
    rm_fd(_handle);
    close();
    _greeting_encoder.reset();
    _choice_decoder.reset();
    _basic_auth_request_encoder.reset();
    _auth_response_decoder.reset();
    _request_encoder.reset();
    _response_decoder.reset();
    _status = unplugged;
    add_reconnect_timer();
}

int zmq::socks_connecter_t::connect_to_proxy()
{
    zmq_assert(_s == retired_fd);

    //  Resolve the address
    if (_proxy_addr->resolved.tcp_addr != NULL)
    {
        LIBZMQ_DELETE(_proxy_addr->resolved.tcp_addr);
    }

    _proxy_addr->resolved.tcp_addr = new (std::nothrow) tcp_address_t();
    alloc_assert(_proxy_addr->resolved.tcp_addr);
    //  Automatic fallback to ipv4 is disabled here since this was the existing
    //  behaviour, however I don't see a real reason for this. Maybe this can
    //  be changed to true (and then the parameter can be removed entirely).
    _s = tcp_open_socket(_proxy_addr->address.c_str(), options, false, false, _proxy_addr->resolved.tcp_addr);
    if (_s == retired_fd)
    {
        //  TODO we should emit some event in this case!
        LIBZMQ_DELETE(_proxy_addr->resolved.tcp_addr);
        return -1;
    }
    zmq_assert(_proxy_addr->resolved.tcp_addr != NULL);

    // Set the socket to non-blocking mode so that we get async connect().
    unblock_socket(_s);

    const tcp_address_t *const tcp_addr = _proxy_addr->resolved.tcp_addr;

    int rc;

    // Set a source address for conversations
    if (tcp_addr->has_src_addr())
    {
#if defined ZMQ_HAVE_VXWORKS
        rc = ::bind(_s, (sockaddr *)tcp_addr->src_addr(), tcp_addr->src_addrlen());
#else
        rc = ::bind(_s, tcp_addr->src_addr(), tcp_addr->src_addrlen());
#endif
        if (rc == -1)
        {
            close();
            return -1;
        }
    }

    //  Connect to the remote peer.
#if defined ZMQ_HAVE_VXWORKS
    rc = ::connect(_s, (sockaddr *)tcp_addr->addr(), tcp_addr->addrlen());
#else
    rc = ::connect(_s, tcp_addr->addr(), tcp_addr->addrlen());
#endif
    //  Connect was successful immediately.
    if (rc == 0)
        return 0;

        //  Translate error codes indicating asynchronous connect has been
        //  launched to a uniform EINPROGRESS.
#ifdef ZMQ_HAVE_WINDOWS
    const int last_error = WSAGetLastError();
    if (last_error == WSAEINPROGRESS || last_error == WSAEWOULDBLOCK)
        errno = EINPROGRESS;
    else
    {
        errno = wsa_error_to_errno(last_error);
        close();
    }
#else
    if (errno == EINTR)
        errno = EINPROGRESS;
#endif
    return -1;
}

zmq::fd_t zmq::socks_connecter_t::check_proxy_connection()
{
    //  Async connect has finished. Check whether an error occurred
    int err = 0;
#if defined ZMQ_HAVE_HPUX || defined ZMQ_HAVE_VXWORKS
    int len = sizeof err;
#else
    socklen_t len = sizeof err;
#endif

    int rc = getsockopt(_s, SOL_SOCKET, SO_ERROR, reinterpret_cast<char *>(&err), &len);

    //  Assert if the error was caused by 0MQ bug.
    //  Networking problems are OK. No need to assert.
#ifdef ZMQ_HAVE_WINDOWS
    zmq_assert(rc == 0);
    if (err != 0)
    {
        wsa_assert(err == WSAECONNREFUSED || err == WSAETIMEDOUT || err == WSAECONNABORTED || err == WSAEHOSTUNREACH ||
                   err == WSAENETUNREACH || err == WSAENETDOWN || err == WSAEACCES || err == WSAEINVAL ||
                   err == WSAEADDRINUSE);
        return -1;
    }
#else
    //  Following code should handle both Berkeley-derived socket
    //  implementations and Solaris.
    if (rc == -1)
        err = errno;
    if (err != 0)
    {
        errno = err;
        errno_assert(errno == ECONNREFUSED || errno == ECONNRESET || errno == ETIMEDOUT || errno == EHOSTUNREACH ||
                     errno == ENETUNREACH || errno == ENETDOWN || errno == EINVAL);
        return -1;
    }
#endif

    rc = tune_tcp_socket(_s);
    rc = rc | tune_tcp_keepalives(_s, options.tcp_keepalive, options.tcp_keepalive_cnt, options.tcp_keepalive_idle,
                                  options.tcp_keepalive_intvl);
    if (rc != 0)
        return -1;

    return 0;
}

int zmq::socks_connecter_t::parse_address(const std::string &address_, std::string &hostname_, uint16_t &port_)
{
    //  Find the ':' at end that separates address from the port number.
    const size_t idx = address_.rfind(':');
    if (idx == std::string::npos)
    {
        errno = EINVAL;
        return -1;
    }

    //  Extract hostname
    if (idx < 2 || address_[0] != '[' || address_[idx - 1] != ']')
        hostname_ = address_.substr(0, idx);
    else
        hostname_ = address_.substr(1, idx - 2);

    //  Separate the hostname/port.
    const std::string port_str = address_.substr(idx + 1);
    //  Parse the port number (0 is not a valid port).
    port_ = static_cast<uint16_t>(atoi(port_str.c_str()));
    if (port_ == 0)
    {
        errno = EINVAL;
        return -1;
    }
    return 0;
}

//========= end of socks_connecter.cpp ============

//========= begin of stream.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "stream.hpp"
// ans ignore: #include "pipe.hpp"
// ans ignore: #include "wire.hpp"
// ans ignore: #include "random.hpp"
// ans ignore: #include "likely.hpp"
// ans ignore: #include "err.hpp"

zmq::stream_t::stream_t(class ctx_t *parent_, uint32_t tid_, int sid_)
    : routing_socket_base_t(parent_, tid_, sid_), _prefetched(false), _routing_id_sent(false), _current_out(NULL),
      _more_out(false), _next_integral_routing_id(generate_random())
{
    options.type       = ZMQ_STREAM;
    options.raw_socket = true;

    _prefetched_routing_id.init();
    _prefetched_msg.init();
}

zmq::stream_t::~stream_t()
{
    _prefetched_routing_id.close();
    _prefetched_msg.close();
}

void zmq::stream_t::xattach_pipe(pipe_t *pipe_, bool subscribe_to_all_, bool locally_initiated_)
{
    LIBZMQ_UNUSED(subscribe_to_all_);

    zmq_assert(pipe_);

    identify_peer(pipe_, locally_initiated_);
    _fq.attach(pipe_);
}

void zmq::stream_t::xpipe_terminated(pipe_t *pipe_)
{
    erase_out_pipe(pipe_);
    _fq.pipe_terminated(pipe_);
    // TODO router_t calls pipe_->rollback() here; should this be done here as
    // well? then xpipe_terminated could be pulled up to routing_socket_base_t
    if (pipe_ == _current_out)
        _current_out = NULL;
}

void zmq::stream_t::xread_activated(pipe_t *pipe_) { _fq.activated(pipe_); }

int zmq::stream_t::xsend(msg_t *msg_)
{
    //  If this is the first part of the message it's the ID of the
    //  peer to send the message to.
    if (!_more_out)
    {
        zmq_assert(!_current_out);

        //  If we have malformed message (prefix with no subsequent message)
        //  then just silently ignore it.
        //  TODO: The connections should be killed instead.
        if (msg_->flags() & msg_t::more)
        {
            //  Find the pipe associated with the routing id stored in the prefix.
            //  If there's no such pipe return an error

            out_pipe_t *out_pipe =
                lookup_out_pipe(blob_t(static_cast<unsigned char *>(msg_->data()), msg_->size(), reference_tag_t()));

            if (out_pipe)
            {
                _current_out = out_pipe->pipe;
                if (!_current_out->check_write())
                {
                    out_pipe->active = false;
                    _current_out     = NULL;
                    errno            = EAGAIN;
                    return -1;
                }
            }
            else
            {
                errno = EHOSTUNREACH;
                return -1;
            }
        }

        //  Expect one more message frame.
        _more_out = true;

        int rc = msg_->close();
        errno_assert(rc == 0);
        rc = msg_->init();
        errno_assert(rc == 0);
        return 0;
    }

    //  Ignore the MORE flag
    msg_->reset_flags(msg_t::more);

    //  This is the last part of the message.
    _more_out = false;

    //  Push the message into the pipe. If there's no out pipe, just drop it.
    if (_current_out)
    {
        // Close the remote connection if user has asked to do so
        // by sending zero length message.
        // Pending messages in the pipe will be dropped (on receiving term- ack)
        if (msg_->size() == 0)
        {
            _current_out->terminate(false);
            int rc = msg_->close();
            errno_assert(rc == 0);
            rc = msg_->init();
            errno_assert(rc == 0);
            _current_out = NULL;
            return 0;
        }
        bool ok = _current_out->write(msg_);
        if (likely(ok))
            _current_out->flush();
        _current_out = NULL;
    }
    else
    {
        int rc = msg_->close();
        errno_assert(rc == 0);
    }

    //  Detach the message from the data buffer.
    int rc = msg_->init();
    errno_assert(rc == 0);

    return 0;
}

int zmq::stream_t::xsetsockopt(int option_, const void *optval_, size_t optvallen_)
{
    switch (option_)
    {
    case ZMQ_STREAM_NOTIFY:
        return do_setsockopt_int_as_bool_strict(optval_, optvallen_, &options.raw_notify);

    default:
        return routing_socket_base_t::xsetsockopt(option_, optval_, optvallen_);
    }
}

int zmq::stream_t::xrecv(msg_t *msg_)
{
    if (_prefetched)
    {
        if (!_routing_id_sent)
        {
            int rc = msg_->move(_prefetched_routing_id);
            errno_assert(rc == 0);
            _routing_id_sent = true;
        }
        else
        {
            int rc = msg_->move(_prefetched_msg);
            errno_assert(rc == 0);
            _prefetched = false;
        }
        return 0;
    }

    pipe_t *pipe = NULL;
    int     rc   = _fq.recvpipe(&_prefetched_msg, &pipe);
    if (rc != 0)
        return -1;

    zmq_assert(pipe != NULL);
    zmq_assert((_prefetched_msg.flags() & msg_t::more) == 0);

    //  We have received a frame with TCP data.
    //  Rather than sending this frame, we keep it in prefetched
    //  buffer and send a frame with peer's ID.
    const blob_t &routing_id = pipe->get_routing_id();
    rc                       = msg_->close();
    errno_assert(rc == 0);
    rc = msg_->init_size(routing_id.size());
    errno_assert(rc == 0);

    // forward metadata (if any)
    metadata_t *metadata = _prefetched_msg.metadata();
    if (metadata)
        msg_->set_metadata(metadata);

    memcpy(msg_->data(), routing_id.data(), routing_id.size());
    msg_->set_flags(msg_t::more);

    _prefetched      = true;
    _routing_id_sent = true;

    return 0;
}

bool zmq::stream_t::xhas_in()
{
    //  We may already have a message pre-fetched.
    if (_prefetched)
        return true;

    //  Try to read the next message.
    //  The message, if read, is kept in the pre-fetch buffer.
    pipe_t *pipe = NULL;
    int     rc   = _fq.recvpipe(&_prefetched_msg, &pipe);
    if (rc != 0)
        return false;

    zmq_assert(pipe != NULL);
    zmq_assert((_prefetched_msg.flags() & msg_t::more) == 0);

    const blob_t &routing_id = pipe->get_routing_id();
    rc                       = _prefetched_routing_id.init_size(routing_id.size());
    errno_assert(rc == 0);

    // forward metadata (if any)
    metadata_t *metadata = _prefetched_msg.metadata();
    if (metadata)
        _prefetched_routing_id.set_metadata(metadata);

    memcpy(_prefetched_routing_id.data(), routing_id.data(), routing_id.size());
    _prefetched_routing_id.set_flags(msg_t::more);

    _prefetched      = true;
    _routing_id_sent = false;

    return true;
}

bool zmq::stream_t::xhas_out()
{
    //  In theory, STREAM socket is always ready for writing. Whether actual
    //  attempt to write succeeds depends on which pipe the message is going
    //  to be routed to.
    return true;
}

void zmq::stream_t::identify_peer(pipe_t *pipe_, bool locally_initiated_)
{
    //  Always assign routing id for raw-socket
    unsigned char buffer[5];
    buffer[0] = 0;
    blob_t routing_id;
    if (locally_initiated_ && connect_routing_id_is_set())
    {
        const std::string connect_routing_id = extract_connect_routing_id();
        routing_id.set(reinterpret_cast<const unsigned char *>(connect_routing_id.c_str()),
                       connect_routing_id.length());
        //  Not allowed to duplicate an existing rid
        zmq_assert(!has_out_pipe(routing_id));
    }
    else
    {
        put_uint32(buffer + 1, _next_integral_routing_id++);
        routing_id.set(buffer, sizeof buffer);
        memcpy(options.routing_id, routing_id.data(), routing_id.size());
        options.routing_id_size = static_cast<unsigned char>(routing_id.size());
    }
    pipe_->set_router_socket_routing_id(routing_id);
    add_out_pipe(ZMQ_MOVE(routing_id), pipe_);
}

//========= end of stream.cpp ============

//========= begin of stream_connecter_base.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "stream_connecter_base.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "address.hpp"
// ans ignore: #include "random.hpp"

#ifndef ZMQ_HAVE_WINDOWS
#include <unistd.h>
#else
#include <winsock2.h>
#endif

#include <limits>

zmq::stream_connecter_base_t::stream_connecter_base_t(zmq::io_thread_t *io_thread_, zmq::session_base_t *session_,
                                                      const zmq::options_t &options_, zmq::address_t *addr_,
                                                      bool delayed_start_)
    : own_t(io_thread_, options_), io_object_t(io_thread_), _addr(addr_), _s(retired_fd),
      _handle(static_cast<handle_t>(NULL)), _socket(session_->get_socket()), _delayed_start(delayed_start_),
      _reconnect_timer_started(false), _session(session_), _current_reconnect_ivl(options.reconnect_ivl)
{
    zmq_assert(_addr);
    _addr->to_string(_endpoint);
    // TODO the return value is unused! what if it fails? if this is impossible
    // or does not matter, change such that endpoint in initialized using an
    // initializer, and make endpoint const
}

zmq::stream_connecter_base_t::~stream_connecter_base_t()
{
    zmq_assert(!_reconnect_timer_started);
    zmq_assert(!_handle);
    zmq_assert(_s == retired_fd);
}

void zmq::stream_connecter_base_t::process_plug()
{
    if (_delayed_start)
        add_reconnect_timer();
    else
        start_connecting();
}

void zmq::stream_connecter_base_t::process_term(int linger_)
{
    if (_reconnect_timer_started)
    {
        cancel_timer(reconnect_timer_id);
        _reconnect_timer_started = false;
    }

    if (_handle)
    {
        rm_handle();
    }

    if (_s != retired_fd)
        close();

    own_t::process_term(linger_);
}

void zmq::stream_connecter_base_t::add_reconnect_timer()
{
    if (options.reconnect_ivl != -1)
    {
        const int interval = get_new_reconnect_ivl();
        add_timer(interval, reconnect_timer_id);
        _socket->event_connect_retried(make_unconnected_connect_endpoint_pair(_endpoint), interval);
        _reconnect_timer_started = true;
    }
}

int zmq::stream_connecter_base_t::get_new_reconnect_ivl()
{
    //  TODO should the random jitter be really based on the configured initial
    //  reconnect interval options.reconnect_ivl, or better on the
    //  _current_reconnect_ivl?

    //  The new interval is the current interval + random value.
    const int random_jitter = generate_random() % options.reconnect_ivl;
    const int interval      = _current_reconnect_ivl < std::numeric_limits<int>::max() - random_jitter
                             ? _current_reconnect_ivl + random_jitter
                             : std::numeric_limits<int>::max();

    //  Only change the new current reconnect interval if the maximum reconnect
    //  interval was set and if it's larger than the reconnect interval.
    if (options.reconnect_ivl_max > 0 && options.reconnect_ivl_max > options.reconnect_ivl)
    {
        //  Calculate the next interval
        _current_reconnect_ivl = _current_reconnect_ivl < std::numeric_limits<int>::max() / 2
                                     ? std::min(_current_reconnect_ivl * 2, options.reconnect_ivl_max)
                                     : options.reconnect_ivl_max;
    }

    return interval;
}

void zmq::stream_connecter_base_t::rm_handle()
{
    rm_fd(_handle);
    _handle = static_cast<handle_t>(NULL);
}

void zmq::stream_connecter_base_t::close()
{
    // TODO before, this was an assertion for _s != retired_fd, but this does not match usage of close
    if (_s != retired_fd)
    {
#ifdef ZMQ_HAVE_WINDOWS
        const int rc = closesocket(_s);
        wsa_assert(rc != SOCKET_ERROR);
#else
        const int rc = ::close(_s);
        errno_assert(rc == 0);
#endif
        _socket->event_closed(make_unconnected_connect_endpoint_pair(_endpoint), _s);
        _s = retired_fd;
    }
}

void zmq::stream_connecter_base_t::in_event()
{
    //  We are not polling for incoming data, so we are actually called
    //  because of error here. However, we can get error on out event as well
    //  on some platforms, so we'll simply handle both events in the same way.
    out_event();
}

void zmq::stream_connecter_base_t::create_engine(fd_t fd, const std::string &local_address_)
{
    const endpoint_uri_pair_t endpoint_pair(local_address_, _endpoint, endpoint_type_connect);

    //  Create the engine object for this connection.
    stream_engine_t *engine = new (std::nothrow) stream_engine_t(fd, options, endpoint_pair);
    alloc_assert(engine);

    //  Attach the engine to the corresponding session object.
    send_attach(_session, engine);

    //  Shut the connecter down.
    terminate();

    _socket->event_connected(endpoint_pair, fd);
}

void zmq::stream_connecter_base_t::timer_event(int id_)
{
    zmq_assert(id_ == reconnect_timer_id);
    _reconnect_timer_started = false;
    start_connecting();
}

//========= end of stream_connecter_base.cpp ============

//========= begin of stream_engine.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"

#include <limits.h>
#include <string.h>

#ifndef ZMQ_HAVE_WINDOWS
#include <unistd.h>
#endif

#include <new>
#include <sstream>

// ans ignore: #include "stream_engine.hpp"
// ans ignore: #include "io_thread.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "v1_encoder.hpp"
// ans ignore: #include "v1_decoder.hpp"
// ans ignore: #include "v2_encoder.hpp"
// ans ignore: #include "v2_decoder.hpp"
// ans ignore: #include "null_mechanism.hpp"
// ans ignore: #include "plain_client.hpp"
// ans ignore: #include "plain_server.hpp"
// ans ignore: #include "gssapi_client.hpp"
// ans ignore: #include "gssapi_server.hpp"
// ans ignore: #include "curve_client.hpp"
// ans ignore: #include "curve_server.hpp"
// ans ignore: #include "raw_decoder.hpp"
// ans ignore: #include "raw_encoder.hpp"
// ans ignore: #include "config.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "ip.hpp"
// ans ignore: #include "tcp.hpp"
// ans ignore: #include "likely.hpp"
// ans ignore: #include "wire.hpp"

static std::string get_peer_address(zmq::fd_t s_)
{
    std::string peer_address;

    const int family = zmq::get_peer_ip_address(s_, peer_address);
    if (family == 0)
        peer_address.clear();
#if defined ZMQ_HAVE_SO_PEERCRED
    else if (family == PF_UNIX)
    {
        struct ucred cred;
        socklen_t    size = sizeof(cred);
        if (!getsockopt(s_, SOL_SOCKET, SO_PEERCRED, &cred, &size))
        {
            std::ostringstream buf;
            buf << ":" << cred.uid << ":" << cred.gid << ":" << cred.pid;
            peer_address += buf.str();
        }
    }
#elif defined ZMQ_HAVE_LOCAL_PEERCRED
    else if (family == PF_UNIX)
    {
        struct xucred cred;
        socklen_t     size = sizeof(cred);
        if (!getsockopt(_s, 0, LOCAL_PEERCRED, &cred, &size) && cred.cr_version == XUCRED_VERSION)
        {
            std::ostringstream buf;
            buf << ":" << cred.cr_uid << ":";
            if (cred.cr_ngroups > 0)
                buf << cred.cr_groups[0];
            buf << ":";
            _peer_address += buf.str();
        }
    }
#endif

    return peer_address;
}

zmq::stream_engine_t::stream_engine_t(fd_t fd_, const options_t &options_,
                                      const endpoint_uri_pair_t &endpoint_uri_pair_)
    : _s(fd_), _handle(static_cast<handle_t>(NULL)), _inpos(NULL), _insize(0), _decoder(NULL), _outpos(NULL),
      _outsize(0), _encoder(NULL), _metadata(NULL), _handshaking(true), _greeting_size(v2_greeting_size),
      _greeting_bytes_read(0), _session(NULL), _options(options_), _endpoint_uri_pair(endpoint_uri_pair_),
      _plugged(false), _next_msg(&stream_engine_t::routing_id_msg),
      _process_msg(&stream_engine_t::process_routing_id_msg), _io_error(false), _subscription_required(false),
      _mechanism(NULL), _input_stopped(false), _output_stopped(false), _has_handshake_timer(false),
      _has_ttl_timer(false), _has_timeout_timer(false), _has_heartbeat_timer(false), _heartbeat_timeout(0),
      _socket(NULL), _peer_address(get_peer_address(_s))
{
    int rc = _tx_msg.init();
    errno_assert(rc == 0);
    rc = _pong_msg.init();
    errno_assert(rc == 0);

    //  Put the socket into non-blocking mode.
    unblock_socket(_s);

    if (_options.heartbeat_interval > 0)
    {
        _heartbeat_timeout = _options.heartbeat_timeout;
        if (_heartbeat_timeout == -1)
            _heartbeat_timeout = _options.heartbeat_interval;
    }
}

zmq::stream_engine_t::~stream_engine_t()
{
    zmq_assert(!_plugged);

    if (_s != retired_fd)
    {
#ifdef ZMQ_HAVE_WINDOWS
        int rc = closesocket(_s);
        wsa_assert(rc != SOCKET_ERROR);
#else
        int rc = close(_s);
#if defined(__FreeBSD_kernel__) || defined(__FreeBSD__)
        // FreeBSD may return ECONNRESET on close() under load but this is not
        // an error.
        if (rc == -1 && errno == ECONNRESET)
            rc = 0;
#endif
        errno_assert(rc == 0);
#endif
        _s = retired_fd;
    }

    int rc = _tx_msg.close();
    errno_assert(rc == 0);

    //  Drop reference to metadata and destroy it if we are
    //  the only user.
    if (_metadata != NULL)
    {
        if (_metadata->drop_ref())
        {
            LIBZMQ_DELETE(_metadata);
        }
    }

    LIBZMQ_DELETE(_encoder);
    LIBZMQ_DELETE(_decoder);
    LIBZMQ_DELETE(_mechanism);
}

void zmq::stream_engine_t::plug(io_thread_t *io_thread_, session_base_t *session_)
{
    zmq_assert(!_plugged);
    _plugged = true;

    //  Connect to session object.
    zmq_assert(!_session);
    zmq_assert(session_);
    _session = session_;
    _socket  = _session->get_socket();

    //  Connect to I/O threads poller object.
    io_object_t::plug(io_thread_);
    _handle   = add_fd(_s);
    _io_error = false;

    if (_options.raw_socket)
    {
        // no handshaking for raw sock, instantiate raw encoder and decoders
        _encoder = new (std::nothrow) raw_encoder_t(_options.out_batch_size);
        alloc_assert(_encoder);

        _decoder = new (std::nothrow) raw_decoder_t(_options.in_batch_size);
        alloc_assert(_decoder);

        // disable handshaking for raw socket
        _handshaking = false;

        _next_msg    = &stream_engine_t::pull_msg_from_session;
        _process_msg = &stream_engine_t::push_raw_msg_to_session;

        properties_t properties;
        if (init_properties(properties))
        {
            //  Compile metadata.
            zmq_assert(_metadata == NULL);
            _metadata = new (std::nothrow) metadata_t(properties);
            alloc_assert(_metadata);
        }

        if (_options.raw_notify)
        {
            //  For raw sockets, send an initial 0-length message to the
            // application so that it knows a peer has connected.
            msg_t connector;
            connector.init();
            push_raw_msg_to_session(&connector);
            connector.close();
            _session->flush();
        }
    }
    else
    {
        // start optional timer, to prevent handshake hanging on no input
        set_handshake_timer();

        //  Send the 'length' and 'flags' fields of the routing id message.
        //  The 'length' field is encoded in the long format.
        _outpos             = _greeting_send;
        _outpos[_outsize++] = UCHAR_MAX;
        put_uint64(&_outpos[_outsize], _options.routing_id_size + 1);
        _outsize += 8;
        _outpos[_outsize++] = 0x7f;
    }

    set_pollin(_handle);
    set_pollout(_handle);
    //  Flush all the data that may have been already received downstream.
    in_event();
}

void zmq::stream_engine_t::unplug()
{
    zmq_assert(_plugged);
    _plugged = false;

    //  Cancel all timers.
    if (_has_handshake_timer)
    {
        cancel_timer(handshake_timer_id);
        _has_handshake_timer = false;
    }

    if (_has_ttl_timer)
    {
        cancel_timer(heartbeat_ttl_timer_id);
        _has_ttl_timer = false;
    }

    if (_has_timeout_timer)
    {
        cancel_timer(heartbeat_timeout_timer_id);
        _has_timeout_timer = false;
    }

    if (_has_heartbeat_timer)
    {
        cancel_timer(heartbeat_ivl_timer_id);
        _has_heartbeat_timer = false;
    }
    //  Cancel all fd subscriptions.
    if (!_io_error)
        rm_fd(_handle);

    //  Disconnect from I/O threads poller object.
    io_object_t::unplug();

    _session = NULL;
}

void zmq::stream_engine_t::terminate()
{
    unplug();
    delete this;
}

void zmq::stream_engine_t::in_event()
{
    // ignore errors
    const bool res = in_event_internal();
    LIBZMQ_UNUSED(res);
}

bool zmq::stream_engine_t::in_event_internal()
{
    zmq_assert(!_io_error);

    //  If still handshaking, receive and process the greeting message.
    if (unlikely(_handshaking))
        if (!handshake())
            return false;

    zmq_assert(_decoder);

    //  If there has been an I/O error, stop polling.
    if (_input_stopped)
    {
        rm_fd(_handle);
        _io_error = true;
        return true; // TODO or return false in this case too?
    }

    //  If there's no data to process in the buffer...
    if (!_insize)
    {
        //  Retrieve the buffer and read as much data as possible.
        //  Note that buffer can be arbitrarily large. However, we assume
        //  the underlying TCP layer has fixed buffer size and thus the
        //  number of bytes read will be always limited.
        size_t bufsize = 0;
        _decoder->get_buffer(&_inpos, &bufsize);

        const int rc = tcp_read(_s, _inpos, bufsize);

        if (rc == 0)
        {
            // connection closed by peer
            errno = EPIPE;
            error(connection_error);
            return false;
        }
        if (rc == -1)
        {
            if (errno != EAGAIN)
            {
                error(connection_error);
                return false;
            }
            return true;
        }

        //  Adjust input size
        _insize = static_cast<size_t>(rc);
        // Adjust buffer size to received bytes
        _decoder->resize_buffer(_insize);
    }

    int    rc        = 0;
    size_t processed = 0;

    while (_insize > 0)
    {
        rc = _decoder->decode(_inpos, _insize, processed);
        zmq_assert(processed <= _insize);
        _inpos += processed;
        _insize -= processed;
        if (rc == 0 || rc == -1)
            break;
        rc = (this->*_process_msg)(_decoder->msg());
        if (rc == -1)
            break;
    }

    //  Tear down the connection if we have failed to decode input data
    //  or the session has rejected the message.
    if (rc == -1)
    {
        if (errno != EAGAIN)
        {
            error(protocol_error);
            return false;
        }
        _input_stopped = true;
        reset_pollin(_handle);
    }

    _session->flush();
    return true;
}

void zmq::stream_engine_t::out_event()
{
    zmq_assert(!_io_error);

    //  If write buffer is empty, try to read new data from the encoder.
    if (!_outsize)
    {
        //  Even when we stop polling as soon as there is no
        //  data to send, the poller may invoke out_event one
        //  more time due to 'speculative write' optimisation.
        if (unlikely(_encoder == NULL))
        {
            zmq_assert(_handshaking);
            return;
        }

        _outpos  = NULL;
        _outsize = _encoder->encode(&_outpos, 0);

        while (_outsize < static_cast<size_t>(_options.out_batch_size))
        {
            if ((this->*_next_msg)(&_tx_msg) == -1)
                break;
            _encoder->load_msg(&_tx_msg);
            unsigned char *bufptr = _outpos + _outsize;
            size_t         n      = _encoder->encode(&bufptr, _options.out_batch_size - _outsize);
            zmq_assert(n > 0);
            if (_outpos == NULL)
                _outpos = bufptr;
            _outsize += n;
        }

        //  If there is no data to send, stop polling for output.
        if (_outsize == 0)
        {
            _output_stopped = true;
            reset_pollout(_handle);
            return;
        }
    }

    //  If there are any data to write in write buffer, write as much as
    //  possible to the socket. Note that amount of data to write can be
    //  arbitrarily large. However, we assume that underlying TCP layer has
    //  limited transmission buffer and thus the actual number of bytes
    //  written should be reasonably modest.
    const int nbytes = tcp_write(_s, _outpos, _outsize);

    //  IO error has occurred. We stop waiting for output events.
    //  The engine is not terminated until we detect input error;
    //  this is necessary to prevent losing incoming messages.
    if (nbytes == -1)
    {
        reset_pollout(_handle);
        return;
    }

    _outpos += nbytes;
    _outsize -= nbytes;

    //  If we are still handshaking and there are no data
    //  to send, stop polling for output.
    if (unlikely(_handshaking))
        if (_outsize == 0)
            reset_pollout(_handle);
}

void zmq::stream_engine_t::restart_output()
{
    if (unlikely(_io_error))
        return;

    if (likely(_output_stopped))
    {
        set_pollout(_handle);
        _output_stopped = false;
    }

    //  Speculative write: The assumption is that at the moment new message
    //  was sent by the user the socket is probably available for writing.
    //  Thus we try to write the data to socket avoiding polling for POLLOUT.
    //  Consequently, the latency should be better in request/reply scenarios.
    out_event();
}

bool zmq::stream_engine_t::restart_input()
{
    zmq_assert(_input_stopped);
    zmq_assert(_session != NULL);
    zmq_assert(_decoder != NULL);

    int rc = (this->*_process_msg)(_decoder->msg());
    if (rc == -1)
    {
        if (errno == EAGAIN)
            _session->flush();
        else
        {
            error(protocol_error);
            return false;
        }
        return true;
    }

    while (_insize > 0)
    {
        size_t processed = 0;
        rc               = _decoder->decode(_inpos, _insize, processed);
        zmq_assert(processed <= _insize);
        _inpos += processed;
        _insize -= processed;
        if (rc == 0 || rc == -1)
            break;
        rc = (this->*_process_msg)(_decoder->msg());
        if (rc == -1)
            break;
    }

    if (rc == -1 && errno == EAGAIN)
        _session->flush();
    else if (_io_error)
    {
        error(connection_error);
        return false;
    }
    else if (rc == -1)
    {
        error(protocol_error);
        return false;
    }

    else
    {
        _input_stopped = false;
        set_pollin(_handle);
        _session->flush();

        //  Speculative read.
        if (!in_event_internal())
            return false;
    }

    return true;
}

//  Position of the revision field in the greeting.
const size_t revision_pos = 10;

bool zmq::stream_engine_t::handshake()
{
    zmq_assert(_handshaking);
    zmq_assert(_greeting_bytes_read < _greeting_size);
    //  Receive the greeting.
    const int rc = receive_greeting();
    if (rc == -1)
        return false;
    const bool unversioned = rc != 0;

    if (!(this->*select_handshake_fun(unversioned, _greeting_recv[revision_pos]))())
        return false;

    // Start polling for output if necessary.
    if (_outsize == 0)
        set_pollout(_handle);

    //  Handshaking was successful.
    //  Switch into the normal message flow.
    _handshaking = false;

    if (_has_handshake_timer)
    {
        cancel_timer(handshake_timer_id);
        _has_handshake_timer = false;
    }

    return true;
}

int zmq::stream_engine_t::receive_greeting()
{
    bool unversioned = false;
    while (_greeting_bytes_read < _greeting_size)
    {
        const int n = tcp_read(_s, _greeting_recv + _greeting_bytes_read, _greeting_size - _greeting_bytes_read);
        if (n == 0)
        {
            errno = EPIPE;
            error(connection_error);
            return -1;
        }
        if (n == -1)
        {
            if (errno != EAGAIN)
                error(connection_error);
            return -1;
        }

        _greeting_bytes_read += n;

        //  We have received at least one byte from the peer.
        //  If the first byte is not 0xff, we know that the
        //  peer is using unversioned protocol.
        if (_greeting_recv[0] != 0xff)
        {
            unversioned = true;
            break;
        }

        if (_greeting_bytes_read < signature_size)
            continue;

        //  Inspect the right-most bit of the 10th byte (which coincides
        //  with the 'flags' field if a regular message was sent).
        //  Zero indicates this is a header of a routing id message
        //  (i.e. the peer is using the unversioned protocol).
        if (!(_greeting_recv[9] & 0x01))
        {
            unversioned = true;
            break;
        }

        //  The peer is using versioned protocol.
        receive_greeting_versioned();
    }
    return unversioned ? 1 : 0;
}

void zmq::stream_engine_t::receive_greeting_versioned()
{
    //  Send the major version number.
    if (_outpos + _outsize == _greeting_send + signature_size)
    {
        if (_outsize == 0)
            set_pollout(_handle);
        _outpos[_outsize++] = 3; //  Major version number
    }

    if (_greeting_bytes_read > signature_size)
    {
        if (_outpos + _outsize == _greeting_send + signature_size + 1)
        {
            if (_outsize == 0)
                set_pollout(_handle);

            //  Use ZMTP/2.0 to talk to older peers.
            if (_greeting_recv[revision_pos] == ZMTP_1_0 || _greeting_recv[revision_pos] == ZMTP_2_0)
                _outpos[_outsize++] = _options.type;
            else
            {
                _outpos[_outsize++] = 0; //  Minor version number
                memset(_outpos + _outsize, 0, 20);

                zmq_assert(_options.mechanism == ZMQ_NULL || _options.mechanism == ZMQ_PLAIN ||
                           _options.mechanism == ZMQ_CURVE || _options.mechanism == ZMQ_GSSAPI);

                if (_options.mechanism == ZMQ_NULL)
                    memcpy(_outpos + _outsize, "NULL", 4);
                else if (_options.mechanism == ZMQ_PLAIN)
                    memcpy(_outpos + _outsize, "PLAIN", 5);
                else if (_options.mechanism == ZMQ_GSSAPI)
                    memcpy(_outpos + _outsize, "GSSAPI", 6);
                else if (_options.mechanism == ZMQ_CURVE)
                    memcpy(_outpos + _outsize, "CURVE", 5);
                _outsize += 20;
                memset(_outpos + _outsize, 0, 32);
                _outsize += 32;
                _greeting_size = v3_greeting_size;
            }
        }
    }
}

zmq::stream_engine_t::handshake_fun_t zmq::stream_engine_t::select_handshake_fun(bool          unversioned,
                                                                                 unsigned char revision)
{
    //  Is the peer using ZMTP/1.0 with no revision number?
    if (unversioned)
    {
        return &stream_engine_t::handshake_v1_0_unversioned;
    }
    switch (revision)
    {
    case ZMTP_1_0:
        return &stream_engine_t::handshake_v1_0;
    case ZMTP_2_0:
        return &stream_engine_t::handshake_v2_0;
    default:
        return &stream_engine_t::handshake_v3_0;
    }
}

bool zmq::stream_engine_t::handshake_v1_0_unversioned()
{
    //  We send and receive rest of routing id message
    if (_session->zap_enabled())
    {
        // reject ZMTP 1.0 connections if ZAP is enabled
        error(protocol_error);
        return false;
    }

    _encoder = new (std::nothrow) v1_encoder_t(_options.out_batch_size);
    alloc_assert(_encoder);

    _decoder = new (std::nothrow) v1_decoder_t(_options.in_batch_size, _options.maxmsgsize);
    alloc_assert(_decoder);

    //  We have already sent the message header.
    //  Since there is no way to tell the encoder to
    //  skip the message header, we simply throw that
    //  header data away.
    const size_t  header_size = _options.routing_id_size + 1 >= UCHAR_MAX ? 10 : 2;
    unsigned char tmp[10], *bufferp = tmp;

    //  Prepare the routing id message and load it into encoder.
    //  Then consume bytes we have already sent to the peer.
    const int rc = _tx_msg.init_size(_options.routing_id_size);
    zmq_assert(rc == 0);
    memcpy(_tx_msg.data(), _options.routing_id, _options.routing_id_size);
    _encoder->load_msg(&_tx_msg);
    const size_t buffer_size = _encoder->encode(&bufferp, header_size);
    zmq_assert(buffer_size == header_size);

    //  Make sure the decoder sees the data we have already received.
    _inpos  = _greeting_recv;
    _insize = _greeting_bytes_read;

    //  To allow for interoperability with peers that do not forward
    //  their subscriptions, we inject a phantom subscription message
    //  message into the incoming message stream.
    if (_options.type == ZMQ_PUB || _options.type == ZMQ_XPUB)
        _subscription_required = true;

    //  We are sending our routing id now and the next message
    //  will come from the socket.
    _next_msg = &stream_engine_t::pull_msg_from_session;

    //  We are expecting routing id message.
    _process_msg = &stream_engine_t::process_routing_id_msg;

    return true;
}

bool zmq::stream_engine_t::handshake_v1_0()
{
    if (_session->zap_enabled())
    {
        // reject ZMTP 1.0 connections if ZAP is enabled
        error(protocol_error);
        return false;
    }

    _encoder = new (std::nothrow) v1_encoder_t(_options.out_batch_size);
    alloc_assert(_encoder);

    _decoder = new (std::nothrow) v1_decoder_t(_options.in_batch_size, _options.maxmsgsize);
    alloc_assert(_decoder);

    return true;
}

bool zmq::stream_engine_t::handshake_v2_0()
{
    if (_session->zap_enabled())
    {
        // reject ZMTP 2.0 connections if ZAP is enabled
        error(protocol_error);
        return false;
    }

    _encoder = new (std::nothrow) v2_encoder_t(_options.out_batch_size);
    alloc_assert(_encoder);

    _decoder = new (std::nothrow) v2_decoder_t(_options.in_batch_size, _options.maxmsgsize, _options.zero_copy);
    alloc_assert(_decoder);

    return true;
}

bool zmq::stream_engine_t::handshake_v3_0()
{
    _encoder = new (std::nothrow) v2_encoder_t(_options.out_batch_size);
    alloc_assert(_encoder);

    _decoder = new (std::nothrow) v2_decoder_t(_options.in_batch_size, _options.maxmsgsize, _options.zero_copy);
    alloc_assert(_decoder);

    if (_options.mechanism == ZMQ_NULL && memcmp(_greeting_recv + 12, "NULL\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0", 20) == 0)
    {
        _mechanism = new (std::nothrow) null_mechanism_t(_session, _peer_address, _options);
        alloc_assert(_mechanism);
    }
    else if (_options.mechanism == ZMQ_PLAIN &&
             memcmp(_greeting_recv + 12, "PLAIN\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0", 20) == 0)
    {
        if (_options.as_server)
            _mechanism = new (std::nothrow) plain_server_t(_session, _peer_address, _options);
        else
            _mechanism = new (std::nothrow) plain_client_t(_session, _options);
        alloc_assert(_mechanism);
    }
#ifdef ZMQ_HAVE_CURVE
    else if (_options.mechanism == ZMQ_CURVE &&
             memcmp(_greeting_recv + 12, "CURVE\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0", 20) == 0)
    {
        if (_options.as_server)
            _mechanism = new (std::nothrow) curve_server_t(_session, _peer_address, _options);
        else
            _mechanism = new (std::nothrow) curve_client_t(_session, _options);
        alloc_assert(_mechanism);
    }
#endif
#ifdef HAVE_LIBGSSAPI_KRB5
    else if (_options.mechanism == ZMQ_GSSAPI &&
             memcmp(_greeting_recv + 12, "GSSAPI\0\0\0\0\0\0\0\0\0\0\0\0\0\0", 20) == 0)
    {
        if (_options.as_server)
            _mechanism = new (std::nothrow) gssapi_server_t(_session, _peer_address, _options);
        else
            _mechanism = new (std::nothrow) gssapi_client_t(_session, _options);
        alloc_assert(_mechanism);
    }
#endif
    else
    {
        _session->get_socket()->event_handshake_failed_protocol(_session->get_endpoint(),
                                                                ZMQ_PROTOCOL_ERROR_ZMTP_MECHANISM_MISMATCH);
        error(protocol_error);
        return false;
    }
    _next_msg    = &stream_engine_t::next_handshake_command;
    _process_msg = &stream_engine_t::process_handshake_command;

    return true;
}

int zmq::stream_engine_t::routing_id_msg(msg_t *msg_)
{
    int rc = msg_->init_size(_options.routing_id_size);
    errno_assert(rc == 0);
    if (_options.routing_id_size > 0)
        memcpy(msg_->data(), _options.routing_id, _options.routing_id_size);
    _next_msg = &stream_engine_t::pull_msg_from_session;
    return 0;
}

int zmq::stream_engine_t::process_routing_id_msg(msg_t *msg_)
{
    if (_options.recv_routing_id)
    {
        msg_->set_flags(msg_t::routing_id);
        int rc = _session->push_msg(msg_);
        errno_assert(rc == 0);
    }
    else
    {
        int rc = msg_->close();
        errno_assert(rc == 0);
        rc = msg_->init();
        errno_assert(rc == 0);
    }

    if (_subscription_required)
    {
        msg_t subscription;

        //  Inject the subscription message, so that also
        //  ZMQ 2.x peers receive published messages.
        int rc = subscription.init_size(1);
        errno_assert(rc == 0);
        *static_cast<unsigned char *>(subscription.data()) = 1;
        rc                                                 = _session->push_msg(&subscription);
        errno_assert(rc == 0);
    }

    _process_msg = &stream_engine_t::push_msg_to_session;

    return 0;
}

int zmq::stream_engine_t::next_handshake_command(msg_t *msg_)
{
    zmq_assert(_mechanism != NULL);

    if (_mechanism->status() == mechanism_t::ready)
    {
        mechanism_ready();
        return pull_and_encode(msg_);
    }
    if (_mechanism->status() == mechanism_t::error)
    {
        errno = EPROTO;
        return -1;
    }
    else
    {
        const int rc = _mechanism->next_handshake_command(msg_);

        if (rc == 0)
            msg_->set_flags(msg_t::command);

        return rc;
    }
}

int zmq::stream_engine_t::process_handshake_command(msg_t *msg_)
{
    zmq_assert(_mechanism != NULL);
    const int rc = _mechanism->process_handshake_command(msg_);
    if (rc == 0)
    {
        if (_mechanism->status() == mechanism_t::ready)
            mechanism_ready();
        else if (_mechanism->status() == mechanism_t::error)
        {
            errno = EPROTO;
            return -1;
        }
        if (_output_stopped)
            restart_output();
    }

    return rc;
}

void zmq::stream_engine_t::zap_msg_available()
{
    zmq_assert(_mechanism != NULL);

    const int rc = _mechanism->zap_msg_available();
    if (rc == -1)
    {
        error(protocol_error);
        return;
    }
    if (_input_stopped)
        if (!restart_input())
            return;
    if (_output_stopped)
        restart_output();
}

const zmq::endpoint_uri_pair_t &zmq::stream_engine_t::get_endpoint() const { return _endpoint_uri_pair; }

void zmq::stream_engine_t::mechanism_ready()
{
    if (_options.heartbeat_interval > 0)
    {
        add_timer(_options.heartbeat_interval, heartbeat_ivl_timer_id);
        _has_heartbeat_timer = true;
    }

    bool flush_session = false;

    if (_options.recv_routing_id)
    {
        msg_t routing_id;
        _mechanism->peer_routing_id(&routing_id);
        const int rc = _session->push_msg(&routing_id);
        if (rc == -1 && errno == EAGAIN)
        {
            // If the write is failing at this stage with
            // an EAGAIN the pipe must be being shut down,
            // so we can just bail out of the routing id set.
            return;
        }
        errno_assert(rc == 0);
        flush_session = true;
    }

    if (_options.router_notify & ZMQ_NOTIFY_CONNECT)
    {
        msg_t connect_notification;
        connect_notification.init();
        const int rc = _session->push_msg(&connect_notification);
        if (rc == -1 && errno == EAGAIN)
        {
            // If the write is failing at this stage with
            // an EAGAIN the pipe must be being shut down,
            // so we can just bail out of the notification.
            return;
        }
        errno_assert(rc == 0);
        flush_session = true;
    }

    if (flush_session)
        _session->flush();

    _next_msg    = &stream_engine_t::pull_and_encode;
    _process_msg = &stream_engine_t::write_credential;

    //  Compile metadata.
    properties_t properties;
    init_properties(properties);

    //  Add ZAP properties.
    const properties_t &zap_properties = _mechanism->get_zap_properties();
    properties.insert(zap_properties.begin(), zap_properties.end());

    //  Add ZMTP properties.
    const properties_t &zmtp_properties = _mechanism->get_zmtp_properties();
    properties.insert(zmtp_properties.begin(), zmtp_properties.end());

    zmq_assert(_metadata == NULL);
    if (!properties.empty())
    {
        _metadata = new (std::nothrow) metadata_t(properties);
        alloc_assert(_metadata);
    }

    _socket->event_handshake_succeeded(_endpoint_uri_pair, 0);
}

int zmq::stream_engine_t::pull_msg_from_session(msg_t *msg_) { return _session->pull_msg(msg_); }

int zmq::stream_engine_t::push_msg_to_session(msg_t *msg_) { return _session->push_msg(msg_); }

int zmq::stream_engine_t::push_raw_msg_to_session(msg_t *msg_)
{
    if (_metadata && _metadata != msg_->metadata())
        msg_->set_metadata(_metadata);
    return push_msg_to_session(msg_);
}

int zmq::stream_engine_t::write_credential(msg_t *msg_)
{
    zmq_assert(_mechanism != NULL);
    zmq_assert(_session != NULL);

    const blob_t &credential = _mechanism->get_user_id();
    if (credential.size() > 0)
    {
        msg_t msg;
        int   rc = msg.init_size(credential.size());
        zmq_assert(rc == 0);
        memcpy(msg.data(), credential.data(), credential.size());
        msg.set_flags(msg_t::credential);
        rc = _session->push_msg(&msg);
        if (rc == -1)
        {
            rc = msg.close();
            errno_assert(rc == 0);
            return -1;
        }
    }
    _process_msg = &stream_engine_t::decode_and_push;
    return decode_and_push(msg_);
}

int zmq::stream_engine_t::pull_and_encode(msg_t *msg_)
{
    zmq_assert(_mechanism != NULL);

    if (_session->pull_msg(msg_) == -1)
        return -1;
    if (_mechanism->encode(msg_) == -1)
        return -1;
    return 0;
}

int zmq::stream_engine_t::decode_and_push(msg_t *msg_)
{
    zmq_assert(_mechanism != NULL);

    if (_mechanism->decode(msg_) == -1)
        return -1;

    if (_has_timeout_timer)
    {
        _has_timeout_timer = false;
        cancel_timer(heartbeat_timeout_timer_id);
    }

    if (_has_ttl_timer)
    {
        _has_ttl_timer = false;
        cancel_timer(heartbeat_ttl_timer_id);
    }

    if (msg_->flags() & msg_t::command)
    {
        process_command_message(msg_);
    }

    if (_metadata)
        msg_->set_metadata(_metadata);
    if (_session->push_msg(msg_) == -1)
    {
        if (errno == EAGAIN)
            _process_msg = &stream_engine_t::push_one_then_decode_and_push;
        return -1;
    }
    return 0;
}

int zmq::stream_engine_t::push_one_then_decode_and_push(msg_t *msg_)
{
    const int rc = _session->push_msg(msg_);
    if (rc == 0)
        _process_msg = &stream_engine_t::decode_and_push;
    return rc;
}

void zmq::stream_engine_t::error(error_reason_t reason_)
{
    if (_options.raw_socket && _options.raw_notify)
    {
        //  For raw sockets, send a final 0-length message to the application
        //  so that it knows the peer has been disconnected.
        msg_t terminator;
        terminator.init();
        (this->*_process_msg)(&terminator);
        terminator.close();
    }
    zmq_assert(_session);

    if ((_options.router_notify & ZMQ_NOTIFY_DISCONNECT) && !_handshaking)
    {
        // For router sockets with disconnect notification, rollback
        // any incomplete message in the pipe, and push the disconnect
        // notification message.
        _session->rollback();

        msg_t disconnect_notification;
        disconnect_notification.init();
        _session->push_msg(&disconnect_notification);
    }

    // protocol errors have been signaled already at the point where they occurred
    if (reason_ != protocol_error && (_mechanism == NULL || _mechanism->status() == mechanism_t::handshaking))
    {
        int err = errno;
        _socket->event_handshake_failed_no_detail(_endpoint_uri_pair, err);
    }

    _socket->event_disconnected(_endpoint_uri_pair, _s);
    _session->flush();
    _session->engine_error(reason_);
    unplug();
    delete this;
}

void zmq::stream_engine_t::set_handshake_timer()
{
    zmq_assert(!_has_handshake_timer);

    if (!_options.raw_socket && _options.handshake_ivl > 0)
    {
        add_timer(_options.handshake_ivl, handshake_timer_id);
        _has_handshake_timer = true;
    }
}

bool zmq::stream_engine_t::init_properties(properties_t &properties_)
{
    if (_peer_address.empty())
        return false;
    properties_.ZMQ_MAP_INSERT_OR_EMPLACE(std::string(ZMQ_MSG_PROPERTY_PEER_ADDRESS), _peer_address);

    //  Private property to support deprecated SRCFD
    std::ostringstream stream;
    stream << static_cast<int>(_s);
    std::string fd_string = stream.str();
    properties_.ZMQ_MAP_INSERT_OR_EMPLACE(std::string("__fd"), ZMQ_MOVE(fd_string));
    return true;
}

void zmq::stream_engine_t::timer_event(int id_)
{
    if (id_ == handshake_timer_id)
    {
        _has_handshake_timer = false;
        //  handshake timer expired before handshake completed, so engine fail
        error(timeout_error);
    }
    else if (id_ == heartbeat_ivl_timer_id)
    {
        _next_msg = &stream_engine_t::produce_ping_message;
        out_event();
        add_timer(_options.heartbeat_interval, heartbeat_ivl_timer_id);
    }
    else if (id_ == heartbeat_ttl_timer_id)
    {
        _has_ttl_timer = false;
        error(timeout_error);
    }
    else if (id_ == heartbeat_timeout_timer_id)
    {
        _has_timeout_timer = false;
        error(timeout_error);
    }
    else
        // There are no other valid timer ids!
        assert(false);
}

int zmq::stream_engine_t::produce_ping_message(msg_t *msg_)
{
    // 16-bit TTL + \4PING == 7
    const size_t ping_ttl_len = msg_t::ping_cmd_name_size + 2;
    zmq_assert(_mechanism != NULL);

    int rc = msg_->init_size(ping_ttl_len);
    errno_assert(rc == 0);
    msg_->set_flags(msg_t::command);
    // Copy in the command message
    memcpy(msg_->data(), "\4PING", msg_t::ping_cmd_name_size);

    uint16_t ttl_val = htons(_options.heartbeat_ttl);
    memcpy(static_cast<uint8_t *>(msg_->data()) + msg_t::ping_cmd_name_size, &ttl_val, sizeof(ttl_val));

    rc        = _mechanism->encode(msg_);
    _next_msg = &stream_engine_t::pull_and_encode;
    if (!_has_timeout_timer && _heartbeat_timeout > 0)
    {
        add_timer(_heartbeat_timeout, heartbeat_timeout_timer_id);
        _has_timeout_timer = true;
    }
    return rc;
}

int zmq::stream_engine_t::produce_pong_message(msg_t *msg_)
{
    zmq_assert(_mechanism != NULL);

    int rc = msg_->move(_pong_msg);
    errno_assert(rc == 0);

    rc        = _mechanism->encode(msg_);
    _next_msg = &stream_engine_t::pull_and_encode;
    return rc;
}

int zmq::stream_engine_t::process_heartbeat_message(msg_t *msg_)
{
    if (msg_->is_ping())
    {
        // 16-bit TTL + \4PING == 7
        const size_t ping_ttl_len     = msg_t::ping_cmd_name_size + 2;
        const size_t ping_max_ctx_len = 16;
        uint16_t     remote_heartbeat_ttl;

        // Get the remote heartbeat TTL to setup the timer
        memcpy(&remote_heartbeat_ttl, static_cast<uint8_t *>(msg_->data()) + msg_t::ping_cmd_name_size,
               ping_ttl_len - msg_t::ping_cmd_name_size);
        remote_heartbeat_ttl = ntohs(remote_heartbeat_ttl);
        // The remote heartbeat is in 10ths of a second
        // so we multiply it by 100 to get the timer interval in ms.
        remote_heartbeat_ttl *= 100;

        if (!_has_ttl_timer && remote_heartbeat_ttl > 0)
        {
            add_timer(remote_heartbeat_ttl, heartbeat_ttl_timer_id);
            _has_ttl_timer = true;
        }

        //  As per ZMTP 3.1 the PING command might contain an up to 16 bytes
        //  context which needs to be PONGed back, so build the pong message
        //  here and store it. Truncate it if it's too long.
        //  Given the engine goes straight to out_event, sequential PINGs will
        //  not be a problem.
        const size_t context_len = std::min(msg_->size() - ping_ttl_len, ping_max_ctx_len);
        const int    rc          = _pong_msg.init_size(msg_t::ping_cmd_name_size + context_len);
        errno_assert(rc == 0);
        _pong_msg.set_flags(msg_t::command);
        memcpy(_pong_msg.data(), "\4PONG", msg_t::ping_cmd_name_size);
        if (context_len > 0)
            memcpy(static_cast<uint8_t *>(_pong_msg.data()) + msg_t::ping_cmd_name_size,
                   static_cast<uint8_t *>(msg_->data()) + ping_ttl_len, context_len);

        _next_msg = &stream_engine_t::produce_pong_message;
        out_event();
    }

    return 0;
}

int zmq::stream_engine_t::process_command_message(msg_t *msg_)
{
    const uint8_t cmd_name_size    = *(static_cast<const uint8_t *>(msg_->data()));
    const size_t  ping_name_size   = msg_t::ping_cmd_name_size - 1;
    const size_t  sub_name_size    = msg_t::sub_cmd_name_size - 1;
    const size_t  cancel_name_size = msg_t::cancel_cmd_name_size - 1;
    //  Malformed command
    if (unlikely(msg_->size() < cmd_name_size + sizeof(cmd_name_size)))
        return -1;

    uint8_t *cmd_name = (static_cast<uint8_t *>(msg_->data())) + 1;
    if (cmd_name_size == ping_name_size && memcmp(cmd_name, "PING", cmd_name_size) == 0)
        msg_->set_flags(zmq::msg_t::ping);
    if (cmd_name_size == ping_name_size && memcmp(cmd_name, "PONG", cmd_name_size) == 0)
        msg_->set_flags(zmq::msg_t::pong);
    if (cmd_name_size == sub_name_size && memcmp(cmd_name, "SUBSCRIBE", cmd_name_size) == 0)
        msg_->set_flags(zmq::msg_t::subscribe);
    if (cmd_name_size == cancel_name_size && memcmp(cmd_name, "CANCEL", cmd_name_size) == 0)
        msg_->set_flags(zmq::msg_t::cancel);

    if (msg_->is_ping() || msg_->is_pong())
        return process_heartbeat_message(msg_);

    return 0;
}

//========= end of stream_engine.cpp ============

//========= begin of stream_listener_base.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "stream_listener_base.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "socket_base.hpp"
// ans ignore: #include "stream_engine.hpp"

#ifndef ZMQ_HAVE_WINDOWS
#include <unistd.h>
#else
#include <winsock2.h>
#endif

zmq::stream_listener_base_t::stream_listener_base_t(zmq::io_thread_t *io_thread_, zmq::socket_base_t *socket_,
                                                    const zmq::options_t &options_)
    : own_t(io_thread_, options_), io_object_t(io_thread_), _s(retired_fd), _handle(static_cast<handle_t>(NULL)),
      _socket(socket_)
{
}

zmq::stream_listener_base_t::~stream_listener_base_t()
{
    zmq_assert(_s == retired_fd);
    zmq_assert(!_handle);
}

int zmq::stream_listener_base_t::get_local_address(std::string &addr_) const
{
    addr_ = get_socket_name(_s, socket_end_local);
    return addr_.empty() ? -1 : 0;
}

void zmq::stream_listener_base_t::process_plug()
{
    //  Start polling for incoming connections.
    _handle = add_fd(_s);
    set_pollin(_handle);
}

void zmq::stream_listener_base_t::process_term(int linger_)
{
    rm_fd(_handle);
    _handle = static_cast<handle_t>(NULL);
    close();
    own_t::process_term(linger_);
}

int zmq::stream_listener_base_t::close()
{
    // TODO this is identical to stream_connector_base_t::close

    zmq_assert(_s != retired_fd);
#ifdef ZMQ_HAVE_WINDOWS
    const int rc = closesocket(_s);
    wsa_assert(rc != SOCKET_ERROR);
#else
    const int rc = ::close(_s);
    errno_assert(rc == 0);
#endif
    _socket->event_closed(make_unconnected_bind_endpoint_pair(_endpoint), _s);
    _s = retired_fd;

    return 0;
}

void zmq::stream_listener_base_t::create_engine(fd_t fd)
{
    const endpoint_uri_pair_t endpoint_pair(get_socket_name(fd, socket_end_local),
                                            get_socket_name(fd, socket_end_remote), endpoint_type_bind);

    stream_engine_t *engine = new (std::nothrow) stream_engine_t(fd, options, endpoint_pair);
    alloc_assert(engine);

    //  Choose I/O thread to run connecter in. Given that we are already
    //  running in an I/O thread, there must be at least one available.
    io_thread_t *io_thread = choose_io_thread(options.affinity);
    zmq_assert(io_thread);

    //  Create and launch a session object.
    session_base_t *session = session_base_t::create(io_thread, false, _socket, options, NULL);
    errno_assert(session);
    session->inc_seqnum();
    launch_child(session);
    send_attach(session, engine, false);

    _socket->event_accepted(endpoint_pair, fd);
}

//========= end of stream_listener_base.cpp ============

//========= begin of sub.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "sub.hpp"
// ans ignore: #include "msg.hpp"

zmq::sub_t::sub_t(class ctx_t *parent_, uint32_t tid_, int sid_) : xsub_t(parent_, tid_, sid_)
{
    options.type = ZMQ_SUB;

    //  Switch filtering messages on (as opposed to XSUB which where the
    //  filtering is off).
    options.filter = true;
}

zmq::sub_t::~sub_t() {}

int zmq::sub_t::xsetsockopt(int option_, const void *optval_, size_t optvallen_)
{
    if (option_ != ZMQ_SUBSCRIBE && option_ != ZMQ_UNSUBSCRIBE)
    {
        errno = EINVAL;
        return -1;
    }

    //  Create the subscription message.
    msg_t msg;
    int   rc = msg.init_size(optvallen_ + 1);
    errno_assert(rc == 0);
    unsigned char *data = static_cast<unsigned char *>(msg.data());
    *data               = (option_ == ZMQ_SUBSCRIBE);
    //  We explicitly allow a NULL subscription with size zero
    if (optvallen_)
    {
        assert(optval_);
        memcpy(data + 1, optval_, optvallen_);
    }
    //  Pass it further on in the stack.
    rc = xsub_t::xsend(&msg);
    return close_and_return(&msg, rc);
}

int zmq::sub_t::xsend(msg_t *)
{
    //  Override the XSUB's send.
    errno = ENOTSUP;
    return -1;
}

bool zmq::sub_t::xhas_out()
{
    //  Override the XSUB's send.
    return false;
}

//========= end of sub.cpp ============

//========= begin of tcp.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "ip.hpp"
// ans ignore: #include "tcp.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "options.hpp"

#if !defined ZMQ_HAVE_WINDOWS
#include <fcntl.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <unistd.h>
#ifdef ZMQ_HAVE_VXWORKS
#include <sockLib.h>
#endif
#endif

#if defined ZMQ_HAVE_OPENVMS
#include <ioctl.h>
#endif

#ifdef __APPLE__
#include <TargetConditionals.h>
#endif

int zmq::tune_tcp_socket(fd_t s_)
{
    //  Disable Nagle's algorithm. We are doing data batching on 0MQ level,
    //  so using Nagle wouldn't improve throughput in anyway, but it would
    //  hurt latency.
    int nodelay = 1;
    int rc      = setsockopt(s_, IPPROTO_TCP, TCP_NODELAY, reinterpret_cast<char *>(&nodelay), sizeof(int));
    tcp_assert_tuning_error(s_, rc);
    if (rc != 0)
        return rc;

#ifdef ZMQ_HAVE_OPENVMS
    //  Disable delayed acknowledgements as they hurt latency significantly.
    int nodelack = 1;
    rc           = setsockopt(s_, IPPROTO_TCP, TCP_NODELACK, (char *)&nodelack, sizeof(int));
    tcp_assert_tuning_error(s_, rc);
#endif
    return rc;
}

int zmq::set_tcp_send_buffer(fd_t sockfd_, int bufsize_)
{
    const int rc = setsockopt(sockfd_, SOL_SOCKET, SO_SNDBUF, reinterpret_cast<char *>(&bufsize_), sizeof bufsize_);
    tcp_assert_tuning_error(sockfd_, rc);
    return rc;
}

int zmq::set_tcp_receive_buffer(fd_t sockfd_, int bufsize_)
{
    const int rc = setsockopt(sockfd_, SOL_SOCKET, SO_RCVBUF, reinterpret_cast<char *>(&bufsize_), sizeof bufsize_);
    tcp_assert_tuning_error(sockfd_, rc);
    return rc;
}

int zmq::tune_tcp_keepalives(fd_t s_, int keepalive_, int keepalive_cnt_, int keepalive_idle_, int keepalive_intvl_)
{
    // These options are used only under certain #ifdefs below.
    LIBZMQ_UNUSED(keepalive_);
    LIBZMQ_UNUSED(keepalive_cnt_);
    LIBZMQ_UNUSED(keepalive_idle_);
    LIBZMQ_UNUSED(keepalive_intvl_);

    // If none of the #ifdefs apply, then s_ is unused.
    LIBZMQ_UNUSED(s_);

    //  Tuning TCP keep-alives if platform allows it
    //  All values = -1 means skip and leave it for OS
#ifdef ZMQ_HAVE_WINDOWS
    if (keepalive_ != -1)
    {
        tcp_keepalive keepalive_opts;
        keepalive_opts.onoff             = keepalive_;
        keepalive_opts.keepalivetime     = keepalive_idle_ != -1 ? keepalive_idle_ * 1000 : 7200000;
        keepalive_opts.keepaliveinterval = keepalive_intvl_ != -1 ? keepalive_intvl_ * 1000 : 1000;
        DWORD num_bytes_returned;
        int rc = WSAIoctl(s_, SIO_KEEPALIVE_VALS, &keepalive_opts, sizeof(keepalive_opts), NULL, 0, &num_bytes_returned,
                          NULL, NULL);
        tcp_assert_tuning_error(s_, rc);
        if (rc == SOCKET_ERROR)
            return rc;
    }
#else
#ifdef ZMQ_HAVE_SO_KEEPALIVE
    if (keepalive_ != -1)
    {
        int rc = setsockopt(s_, SOL_SOCKET, SO_KEEPALIVE, reinterpret_cast<char *>(&keepalive_), sizeof(int));
        tcp_assert_tuning_error(s_, rc);
        if (rc != 0)
            return rc;

#ifdef ZMQ_HAVE_TCP_KEEPCNT
        if (keepalive_cnt_ != -1)
        {
            int rc = setsockopt(s_, IPPROTO_TCP, TCP_KEEPCNT, &keepalive_cnt_, sizeof(int));
            tcp_assert_tuning_error(s_, rc);
            if (rc != 0)
                return rc;
        }
#endif // ZMQ_HAVE_TCP_KEEPCNT

#ifdef ZMQ_HAVE_TCP_KEEPIDLE
        if (keepalive_idle_ != -1)
        {
            int rc = setsockopt(s_, IPPROTO_TCP, TCP_KEEPIDLE, &keepalive_idle_, sizeof(int));
            tcp_assert_tuning_error(s_, rc);
            if (rc != 0)
                return rc;
        }
#else // ZMQ_HAVE_TCP_KEEPIDLE
#ifdef ZMQ_HAVE_TCP_KEEPALIVE
        if (keepalive_idle_ != -1)
        {
            int rc = setsockopt(s_, IPPROTO_TCP, TCP_KEEPALIVE, &keepalive_idle_, sizeof(int));
            tcp_assert_tuning_error(s_, rc);
            if (rc != 0)
                return rc;
        }
#endif // ZMQ_HAVE_TCP_KEEPALIVE
#endif // ZMQ_HAVE_TCP_KEEPIDLE

#ifdef ZMQ_HAVE_TCP_KEEPINTVL
        if (keepalive_intvl_ != -1)
        {
            int rc = setsockopt(s_, IPPROTO_TCP, TCP_KEEPINTVL, &keepalive_intvl_, sizeof(int));
            tcp_assert_tuning_error(s_, rc);
            if (rc != 0)
                return rc;
        }
#endif // ZMQ_HAVE_TCP_KEEPINTVL
    }
#endif // ZMQ_HAVE_SO_KEEPALIVE
#endif // ZMQ_HAVE_WINDOWS

    return 0;
}

int zmq::tune_tcp_maxrt(fd_t sockfd_, int timeout_)
{
    if (timeout_ <= 0)
        return 0;

    LIBZMQ_UNUSED(sockfd_);

#if defined(ZMQ_HAVE_WINDOWS) && defined(TCP_MAXRT)
    // msdn says it's supported in >= Vista, >= Windows Server 2003
    timeout_ /= 1000; // in seconds
    int rc = setsockopt(sockfd_, IPPROTO_TCP, TCP_MAXRT, reinterpret_cast<char *>(&timeout_), sizeof(timeout_));
    tcp_assert_tuning_error(sockfd_, rc);
    return rc;
// FIXME: should be ZMQ_HAVE_TCP_USER_TIMEOUT
#elif defined(TCP_USER_TIMEOUT)
    int rc = setsockopt(sockfd_, IPPROTO_TCP, TCP_USER_TIMEOUT, &timeout_, sizeof(timeout_));
    tcp_assert_tuning_error(sockfd_, rc);
    return rc;
#else
    return 0;
#endif
}

int zmq::tcp_write(fd_t s_, const void *data_, size_t size_)
{
#ifdef ZMQ_HAVE_WINDOWS

    int nbytes = send(s_, (char *)data_, static_cast<int>(size_), 0);

    //  If not a single byte can be written to the socket in non-blocking mode
    //  we'll get an error (this may happen during the speculative write).
    const int last_error = WSAGetLastError();
    if (nbytes == SOCKET_ERROR && last_error == WSAEWOULDBLOCK)
        return 0;

    //  Signalise peer failure.
    if (nbytes == SOCKET_ERROR &&
        (last_error == WSAENETDOWN || last_error == WSAENETRESET || last_error == WSAEHOSTUNREACH ||
         last_error == WSAECONNABORTED || last_error == WSAETIMEDOUT || last_error == WSAECONNRESET))
        return -1;

    //  Circumvent a Windows bug:
    //  See https://support.microsoft.com/en-us/kb/201213
    //  See https://zeromq.jira.com/browse/LIBZMQ-195
    if (nbytes == SOCKET_ERROR && last_error == WSAENOBUFS)
        return 0;

    wsa_assert(nbytes != SOCKET_ERROR);
    return nbytes;

#else
    ssize_t nbytes = send(s_, static_cast<const char *>(data_), size_, 0);

    //  Several errors are OK. When speculative write is being done we may not
    //  be able to write a single byte from the socket. Also, SIGSTOP issued
    //  by a debugging tool can result in EINTR error.
    if (nbytes == -1 && (errno == EAGAIN || errno == EWOULDBLOCK || errno == EINTR))
        return 0;

    //  Signalise peer failure.
    if (nbytes == -1)
    {
#if !defined(TARGET_OS_IPHONE) || !TARGET_OS_IPHONE
        errno_assert(errno != EACCES && errno != EBADF && errno != EDESTADDRREQ && errno != EFAULT &&
                     errno != EISCONN && errno != EMSGSIZE && errno != ENOMEM && errno != ENOTSOCK &&
                     errno != EOPNOTSUPP);
#else
        errno_assert(errno != EACCES && errno != EDESTADDRREQ && errno != EFAULT && errno != EISCONN &&
                     errno != EMSGSIZE && errno != ENOMEM && errno != ENOTSOCK && errno != EOPNOTSUPP);
#endif
        return -1;
    }

    return static_cast<int>(nbytes);

#endif
}

int zmq::tcp_read(fd_t s_, void *data_, size_t size_)
{
#ifdef ZMQ_HAVE_WINDOWS

    const int rc = recv(s_, static_cast<char *>(data_), static_cast<int>(size_), 0);

    //  If not a single byte can be read from the socket in non-blocking mode
    //  we'll get an error (this may happen during the speculative read).
    if (rc == SOCKET_ERROR)
    {
        const int last_error = WSAGetLastError();
        if (last_error == WSAEWOULDBLOCK)
        {
            errno = EAGAIN;
        }
        else
        {
            wsa_assert(last_error == WSAENETDOWN || last_error == WSAENETRESET || last_error == WSAECONNABORTED ||
                       last_error == WSAETIMEDOUT || last_error == WSAECONNRESET || last_error == WSAECONNREFUSED ||
                       last_error == WSAENOTCONN || last_error == WSAENOBUFS);
            errno = wsa_error_to_errno(last_error);
        }
    }

    return rc == SOCKET_ERROR ? -1 : rc;

#else

    const ssize_t rc = recv(s_, static_cast<char *>(data_), size_, 0);

    //  Several errors are OK. When speculative read is being done we may not
    //  be able to read a single byte from the socket. Also, SIGSTOP issued
    //  by a debugging tool can result in EINTR error.
    if (rc == -1)
    {
#if !defined(TARGET_OS_IPHONE) || !TARGET_OS_IPHONE
        errno_assert(errno != EBADF && errno != EFAULT && errno != ENOMEM && errno != ENOTSOCK);
#else
        errno_assert(errno != EFAULT && errno != ENOMEM && errno != ENOTSOCK);
#endif
        if (errno == EWOULDBLOCK || errno == EINTR)
            errno = EAGAIN;
    }

    return static_cast<int>(rc);

#endif
}

void zmq::tcp_assert_tuning_error(zmq::fd_t s_, int rc_)
{
    if (rc_ == 0)
        return;

    //  Check whether an error occurred
    int err = 0;
#if defined ZMQ_HAVE_HPUX || defined ZMQ_HAVE_VXWORKS
    int len = sizeof err;
#else
    socklen_t len = sizeof err;
#endif

    int rc = getsockopt(s_, SOL_SOCKET, SO_ERROR, reinterpret_cast<char *>(&err), &len);

    //  Assert if the error was caused by 0MQ bug.
    //  Networking problems are OK. No need to assert.
#ifdef ZMQ_HAVE_WINDOWS
    zmq_assert(rc == 0);
    if (err != 0)
    {
        wsa_assert(err == WSAECONNREFUSED || err == WSAECONNRESET || err == WSAECONNABORTED || err == WSAEINTR ||
                   err == WSAETIMEDOUT || err == WSAEHOSTUNREACH || err == WSAENETUNREACH || err == WSAENETDOWN ||
                   err == WSAENETRESET || err == WSAEACCES || err == WSAEINVAL || err == WSAEADDRINUSE);
    }
#else
    //  Following code should handle both Berkeley-derived socket
    //  implementations and Solaris.
    if (rc == -1)
        err = errno;
    if (err != 0)
    {
        errno = err;
        errno_assert(errno == ECONNREFUSED || errno == ECONNRESET || errno == ECONNABORTED || errno == EINTR ||
                     errno == ETIMEDOUT || errno == EHOSTUNREACH || errno == ENETUNREACH || errno == ENETDOWN ||
                     errno == ENETRESET || errno == EINVAL);
    }
#endif
}

void zmq::tcp_tune_loopback_fast_path(const fd_t socket_)
{
#if defined ZMQ_HAVE_WINDOWS && defined SIO_LOOPBACK_FAST_PATH
    int   sio_loopback_fastpath    = 1;
    DWORD number_of_bytes_returned = 0;

    int rc = WSAIoctl(socket_, SIO_LOOPBACK_FAST_PATH, &sio_loopback_fastpath, sizeof sio_loopback_fastpath, NULL, 0,
                      &number_of_bytes_returned, 0, 0);

    if (SOCKET_ERROR == rc)
    {
        DWORD last_error = ::WSAGetLastError();

        if (WSAEOPNOTSUPP == last_error)
        {
            // This system is not Windows 8 or Server 2012, and the call is not supported.
        }
        else
        {
            wsa_assert(false);
        }
    }
#else
    LIBZMQ_UNUSED(socket_);
#endif
}

zmq::fd_t zmq::tcp_open_socket(const char *address_, const zmq::options_t &options_, bool local_,
                               bool fallback_to_ipv4_, zmq::tcp_address_t *out_tcp_addr_)
{
    //  Convert the textual address into address structure.
    int rc = out_tcp_addr_->resolve(address_, local_, options_.ipv6);
    if (rc != 0)
        return retired_fd;

    //  Create the socket.
    fd_t s = open_socket(out_tcp_addr_->family(), SOCK_STREAM, IPPROTO_TCP);

    //  IPv6 address family not supported, try automatic downgrade to IPv4.
    if (s == retired_fd && fallback_to_ipv4_ && out_tcp_addr_->family() == AF_INET6 && errno == EAFNOSUPPORT &&
        options_.ipv6)
    {
        rc = out_tcp_addr_->resolve(address_, local_, false);
        if (rc != 0)
        {
            return retired_fd;
        }
        s = open_socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    }

    if (s == retired_fd)
    {
        return retired_fd;
    }

    //  On some systems, IPv4 mapping in IPv6 sockets is disabled by default.
    //  Switch it on in such cases.
    if (out_tcp_addr_->family() == AF_INET6)
        enable_ipv4_mapping(s);

    // Set the IP Type-Of-Service priority for this socket
    if (options_.tos != 0)
        set_ip_type_of_service(s, options_.tos);

    // Set the socket to loopback fastpath if configured.
    if (options_.loopback_fastpath)
        tcp_tune_loopback_fast_path(s);

    // Bind the socket to a device if applicable
    if (!options_.bound_device.empty())
        if (bind_to_device(s, options_.bound_device) == -1)
            goto setsockopt_error;

    //  Set the socket buffer limits for the underlying socket.
    if (options_.sndbuf >= 0)
        set_tcp_send_buffer(s, options_.sndbuf);
    if (options_.rcvbuf >= 0)
        set_tcp_receive_buffer(s, options_.rcvbuf);

    return s;

setsockopt_error:
#ifdef ZMQ_HAVE_WINDOWS
    rc = closesocket(s);
    wsa_assert(rc != SOCKET_ERROR);
#else
    rc = ::close(s);
    errno_assert(rc == 0);
#endif
    return retired_fd;
}

//========= end of tcp.cpp ============

//========= begin of tcp_address.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <string>

// ans ignore: #include "macros.hpp"
// ans ignore: #include "tcp_address.hpp"
// ans ignore: #include "stdint.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "ip.hpp"

#ifndef ZMQ_HAVE_WINDOWS
#include <arpa/inet.h>
#include <ctype.h>
#include <net/if.h>
#include <netdb.h>
#include <netinet/tcp.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#endif

#include <limits.h>

zmq::tcp_address_t::tcp_address_t() : _has_src_addr(false)
{
    memset(&_address, 0, sizeof(_address));
    memset(&_source_address, 0, sizeof(_source_address));
}

zmq::tcp_address_t::tcp_address_t(const sockaddr *sa_, socklen_t sa_len_) : _has_src_addr(false)
{
    zmq_assert(sa_ && sa_len_ > 0);

    memset(&_address, 0, sizeof(_address));
    memset(&_source_address, 0, sizeof(_source_address));
    if (sa_->sa_family == AF_INET && sa_len_ >= static_cast<socklen_t>(sizeof(_address.ipv4)))
        memcpy(&_address.ipv4, sa_, sizeof(_address.ipv4));
    else if (sa_->sa_family == AF_INET6 && sa_len_ >= static_cast<socklen_t>(sizeof(_address.ipv6)))
        memcpy(&_address.ipv6, sa_, sizeof(_address.ipv6));
}

int zmq::tcp_address_t::resolve(const char *name_, bool local_, bool ipv6_)
{
    // Test the ';' to know if we have a source address in name_
    const char *src_delimiter = strrchr(name_, ';');
    if (src_delimiter)
    {
        std::string src_name(name_, src_delimiter - name_);

        ip_resolver_options_t src_resolver_opts;

        src_resolver_opts
            .bindable(true)
            //  Restrict hostname/service to literals to avoid any DNS
            //  lookups or service-name irregularity due to
            //  indeterminate socktype.
            .allow_dns(false)
            .allow_nic_name(true)
            .ipv6(ipv6_)
            .expect_port(true);

        ip_resolver_t src_resolver(src_resolver_opts);

        const int rc = src_resolver.resolve(&_source_address, src_name.c_str());
        if (rc != 0)
            return -1;
        name_         = src_delimiter + 1;
        _has_src_addr = true;
    }

    ip_resolver_options_t resolver_opts;

    resolver_opts.bindable(local_).allow_dns(!local_).allow_nic_name(local_).ipv6(ipv6_).expect_port(true);

    ip_resolver_t resolver(resolver_opts);

    return resolver.resolve(&_address, name_);
}

template <size_t N1, size_t N2>
static std::string make_address_string(const char *hbuf, uint16_t port, const char (&ipv6_prefix)[N1],
                                       const char (&ipv6_suffix)[N2])
{
    const size_t max_port_str_length = 5;
    char         buf[NI_MAXHOST + sizeof ipv6_prefix + sizeof ipv6_suffix + max_port_str_length];
    char *       pos = buf;
    memcpy(pos, ipv6_prefix, sizeof ipv6_prefix - 1);
    pos += sizeof ipv6_prefix - 1;
    const size_t hbuf_len = strlen(hbuf);
    memcpy(pos, hbuf, hbuf_len);
    pos += hbuf_len;
    memcpy(pos, ipv6_suffix, sizeof ipv6_suffix - 1);
    pos += sizeof ipv6_suffix - 1;
    pos += sprintf(pos, "%d", ntohs(port));
    return std::string(buf, pos - buf);
}

int zmq::tcp_address_t::to_string(std::string &addr_) const
{
    if (_address.family() != AF_INET && _address.family() != AF_INET6)
    {
        addr_.clear();
        return -1;
    }

    //  Not using service resolving because of
    //  https://github.com/zeromq/libzmq/commit/1824574f9b5a8ce786853320e3ea09fe1f822bc4
    char      hbuf[NI_MAXHOST];
    const int rc = getnameinfo(addr(), addrlen(), hbuf, sizeof(hbuf), NULL, 0, NI_NUMERICHOST);
    if (rc != 0)
    {
        addr_.clear();
        return rc;
    }

    const char ipv4_prefix[] = "tcp://";
    const char ipv4_suffix[] = ":";
    const char ipv6_prefix[] = "tcp://[";
    const char ipv6_suffix[] = "]:";
    if (_address.family() == AF_INET6)
    {
        addr_ = make_address_string(hbuf, _address.ipv6.sin6_port, ipv6_prefix, ipv6_suffix);
    }
    else
    {
        addr_ = make_address_string(hbuf, _address.ipv4.sin_port, ipv4_prefix, ipv4_suffix);
    }
    return 0;
}

const sockaddr *zmq::tcp_address_t::addr() const { return _address.as_sockaddr(); }

socklen_t zmq::tcp_address_t::addrlen() const { return _address.sockaddr_len(); }

const sockaddr *zmq::tcp_address_t::src_addr() const { return _source_address.as_sockaddr(); }

socklen_t zmq::tcp_address_t::src_addrlen() const { return _source_address.sockaddr_len(); }

bool zmq::tcp_address_t::has_src_addr() const { return _has_src_addr; }

#if defined    ZMQ_HAVE_WINDOWS
unsigned short zmq::tcp_address_t::family() const
#else
sa_family_t zmq::tcp_address_t::family() const
#endif
{
    return _address.family();
}

zmq::tcp_address_mask_t::tcp_address_mask_t() : _address_mask(-1)
{
    memset(&_network_address, 0, sizeof(_network_address));
}

int zmq::tcp_address_mask_t::mask() const { return _address_mask; }

int zmq::tcp_address_mask_t::resolve(const char *name_, bool ipv6_)
{
    // Find '/' at the end that separates address from the cidr mask number.
    // Allow empty mask clause and treat it like '/32' for ipv4 or '/128' for ipv6.
    std::string addr_str, mask_str;
    const char *delimiter = strrchr(name_, '/');
    if (delimiter != NULL)
    {
        addr_str.assign(name_, delimiter - name_);
        mask_str.assign(delimiter + 1);
        if (mask_str.empty())
        {
            errno = EINVAL;
            return -1;
        }
    }
    else
        addr_str.assign(name_);

    // Parse address part using standard routines.
    ip_resolver_options_t resolver_opts;

    resolver_opts.bindable(false).allow_dns(false).allow_nic_name(false).ipv6(ipv6_).expect_port(false);

    ip_resolver_t resolver(resolver_opts);

    const int rc = resolver.resolve(&_network_address, addr_str.c_str());
    if (rc != 0)
        return rc;

    // Parse the cidr mask number.
    const int full_mask_ipv4 = sizeof(_network_address.ipv4.sin_addr) * CHAR_BIT;
    const int full_mask_ipv6 = sizeof(_network_address.ipv6.sin6_addr) * CHAR_BIT;
    if (mask_str.empty())
    {
        _address_mask = _network_address.family() == AF_INET6 ? full_mask_ipv6 : full_mask_ipv4;
    }
    else if (mask_str == "0")
        _address_mask = 0;
    else
    {
        const long mask = strtol(mask_str.c_str(), NULL, 10);
        if ((mask < 1) || (_network_address.family() == AF_INET6 && mask > full_mask_ipv6) ||
            (_network_address.family() != AF_INET6 && mask > full_mask_ipv4))
        {
            errno = EINVAL;
            return -1;
        }
        _address_mask = static_cast<int>(mask);
    }

    return 0;
}

int zmq::tcp_address_mask_t::to_string(std::string &addr_) const
{
    if (_network_address.family() != AF_INET && _network_address.family() != AF_INET6)
    {
        addr_.clear();
        return -1;
    }
    if (_address_mask == -1)
    {
        addr_.clear();
        return -1;
    }

    char      hbuf[NI_MAXHOST];
    const int rc = getnameinfo(_network_address.as_sockaddr(), _network_address.sockaddr_len(), hbuf, sizeof(hbuf),
                               NULL, 0, NI_NUMERICHOST);
    if (rc != 0)
    {
        addr_.clear();
        return rc;
    }

    const size_t max_mask_len  = 4;
    const char   ipv6_prefix[] = "[";
    const char   ipv6_suffix[] = "]/";
    const char   ipv4_suffix[] = "/";
    char         buf[NI_MAXHOST + sizeof ipv6_prefix + sizeof ipv6_suffix + max_mask_len];
    char *       pos = buf;
    if (_network_address.family() == AF_INET6)
    {
        memcpy(pos, ipv6_prefix, sizeof ipv6_prefix - 1);
        pos += sizeof ipv6_prefix - 1;
    }
    const size_t hbuf_len = strlen(hbuf);
    memcpy(pos, hbuf, hbuf_len);
    pos += hbuf_len;
    if (_network_address.family() == AF_INET6)
    {
        memcpy(pos, ipv6_suffix, sizeof ipv6_suffix - 1);
        pos += sizeof ipv6_suffix - 1;
    }
    else
    {
        memcpy(pos, ipv4_suffix, sizeof ipv4_suffix - 1);
        pos += sizeof ipv4_suffix - 1;
    }
    pos += sprintf(pos, "%d", _address_mask);
    addr_.assign(buf, pos - buf);
    return 0;
}

bool zmq::tcp_address_mask_t::match_address(const struct sockaddr *ss_, const socklen_t ss_len_) const
{
    zmq_assert(_address_mask != -1 && ss_ != NULL && ss_len_ >= static_cast<socklen_t>(sizeof(struct sockaddr)));

    if (ss_->sa_family != _network_address.generic.sa_family)
        return false;

    if (_address_mask > 0)
    {
        int            mask;
        const uint8_t *our_bytes, *their_bytes;
        if (ss_->sa_family == AF_INET6)
        {
            zmq_assert(ss_len_ == sizeof(struct sockaddr_in6));
            their_bytes =
                reinterpret_cast<const uint8_t *>(&((reinterpret_cast<const struct sockaddr_in6 *>(ss_))->sin6_addr));
            our_bytes = reinterpret_cast<const uint8_t *>(&_network_address.ipv6.sin6_addr);
            mask      = sizeof(struct in6_addr) * 8;
        }
        else
        {
            zmq_assert(ss_len_ == sizeof(struct sockaddr_in));
            their_bytes =
                reinterpret_cast<const uint8_t *>(&((reinterpret_cast<const struct sockaddr_in *>(ss_))->sin_addr));
            our_bytes = reinterpret_cast<const uint8_t *>(&_network_address.ipv4.sin_addr);
            mask      = sizeof(struct in_addr) * 8;
        }
        if (_address_mask < mask)
            mask = _address_mask;

        const size_t full_bytes = mask / 8;
        if (memcmp(our_bytes, their_bytes, full_bytes) != 0)
            return false;

        const uint8_t last_byte_bits = 0xffU << (8 - mask % 8);
        if (last_byte_bits)
        {
            if ((their_bytes[full_bytes] & last_byte_bits) != (our_bytes[full_bytes] & last_byte_bits))
                return false;
        }
    }

    return true;
}

//========= end of tcp_address.cpp ============

//========= begin of tcp_connecter.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <new>
#include <string>

// ans ignore: #include "macros.hpp"
// ans ignore: #include "tcp_connecter.hpp"
// ans ignore: #include "stream_engine.hpp"
// ans ignore: #include "io_thread.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "ip.hpp"
// ans ignore: #include "tcp.hpp"
// ans ignore: #include "address.hpp"
// ans ignore: #include "tcp_address.hpp"
// ans ignore: #include "session_base.hpp"

#if !defined ZMQ_HAVE_WINDOWS
#include <arpa/inet.h>
#include <fcntl.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <unistd.h>
#ifdef ZMQ_HAVE_VXWORKS
#include <sockLib.h>
#endif
#ifdef ZMQ_HAVE_OPENVMS
#include <ioctl.h>
#endif
#endif

#ifdef __APPLE__
#include <TargetConditionals.h>
#endif

zmq::tcp_connecter_t::tcp_connecter_t(class io_thread_t *io_thread_, class session_base_t *session_,
                                      const options_t &options_, address_t *addr_, bool delayed_start_)
    : stream_connecter_base_t(io_thread_, session_, options_, addr_, delayed_start_), _connect_timer_started(false)
{
    zmq_assert(_addr->protocol == protocol_name::tcp);
}

zmq::tcp_connecter_t::~tcp_connecter_t() { zmq_assert(!_connect_timer_started); }

void zmq::tcp_connecter_t::process_term(int linger_)
{
    if (_connect_timer_started)
    {
        cancel_timer(connect_timer_id);
        _connect_timer_started = false;
    }

    stream_connecter_base_t::process_term(linger_);
}

void zmq::tcp_connecter_t::out_event()
{
    if (_connect_timer_started)
    {
        cancel_timer(connect_timer_id);
        _connect_timer_started = false;
    }

    //  TODO this is still very similar to (t)ipc_connecter_t, maybe the
    //  differences can be factored out

    rm_handle();

    const fd_t fd = connect();

    //  Handle the error condition by attempt to reconnect.
    if (fd == retired_fd || !tune_socket(fd))
    {
        close();
        add_reconnect_timer();
        return;
    }

    create_engine(fd, get_socket_name<tcp_address_t>(fd, socket_end_local));
}

void zmq::tcp_connecter_t::timer_event(int id_)
{
    if (id_ == connect_timer_id)
    {
        _connect_timer_started = false;
        rm_handle();
        close();
        add_reconnect_timer();
    }
    else
        stream_connecter_base_t::timer_event(id_);
}

void zmq::tcp_connecter_t::start_connecting()
{
    //  Open the connecting socket.
    const int rc = open();

    //  Connect may succeed in synchronous manner.
    if (rc == 0)
    {
        _handle = add_fd(_s);
        out_event();
    }

    //  Connection establishment may be delayed. Poll for its completion.
    else if (rc == -1 && errno == EINPROGRESS)
    {
        _handle = add_fd(_s);
        set_pollout(_handle);
        _socket->event_connect_delayed(make_unconnected_connect_endpoint_pair(_endpoint), zmq_errno());

        //  add userspace connect timeout
        add_connect_timer();
    }

    //  Handle any other error condition by eventual reconnect.
    else
    {
        if (_s != retired_fd)
            close();
        add_reconnect_timer();
    }
}

void zmq::tcp_connecter_t::add_connect_timer()
{
    if (options.connect_timeout > 0)
    {
        add_timer(options.connect_timeout, connect_timer_id);
        _connect_timer_started = true;
    }
}

int zmq::tcp_connecter_t::open()
{
    zmq_assert(_s == retired_fd);

    //  Resolve the address
    if (_addr->resolved.tcp_addr != NULL)
    {
        LIBZMQ_DELETE(_addr->resolved.tcp_addr);
    }

    _addr->resolved.tcp_addr = new (std::nothrow) tcp_address_t();
    alloc_assert(_addr->resolved.tcp_addr);
    _s = tcp_open_socket(_addr->address.c_str(), options, false, true, _addr->resolved.tcp_addr);
    if (_s == retired_fd)
    {
        //  TODO we should emit some event in this case!

        LIBZMQ_DELETE(_addr->resolved.tcp_addr);
        return -1;
    }
    zmq_assert(_addr->resolved.tcp_addr != NULL);

    // Set the socket to non-blocking mode so that we get async connect().
    unblock_socket(_s);

    const tcp_address_t *const tcp_addr = _addr->resolved.tcp_addr;

    int rc;

    // Set a source address for conversations
    if (tcp_addr->has_src_addr())
    {
        //  Allow reusing of the address, to connect to different servers
        //  using the same source port on the client.
        int flag = 1;
#ifdef ZMQ_HAVE_WINDOWS
        rc = setsockopt(_s, SOL_SOCKET, SO_REUSEADDR, reinterpret_cast<const char *>(&flag), sizeof(int));
        wsa_assert(rc != SOCKET_ERROR);
#elif defined ZMQ_HAVE_VXWORKS
        rc = setsockopt(_s, SOL_SOCKET, SO_REUSEADDR, (char *)&flag, sizeof(int));
        errno_assert(rc == 0);
#else
        rc = setsockopt(_s, SOL_SOCKET, SO_REUSEADDR, &flag, sizeof(int));
        errno_assert(rc == 0);
#endif

#if defined ZMQ_HAVE_VXWORKS
        rc = ::bind(_s, (sockaddr *)tcp_addr->src_addr(), tcp_addr->src_addrlen());
#else
        rc = ::bind(_s, tcp_addr->src_addr(), tcp_addr->src_addrlen());
#endif
        if (rc == -1)
            return -1;
    }

    //  Connect to the remote peer.
#if defined ZMQ_HAVE_VXWORKS
    rc = ::connect(_s, (sockaddr *)tcp_addr->addr(), tcp_addr->addrlen());
#else
    rc = ::connect(_s, tcp_addr->addr(), tcp_addr->addrlen());
#endif
    //  Connect was successful immediately.
    if (rc == 0)
    {
        return 0;
    }

    //  Translate error codes indicating asynchronous connect has been
    //  launched to a uniform EINPROGRESS.
#ifdef ZMQ_HAVE_WINDOWS
    const int last_error = WSAGetLastError();
    if (last_error == WSAEINPROGRESS || last_error == WSAEWOULDBLOCK)
        errno = EINPROGRESS;
    else
        errno = wsa_error_to_errno(last_error);
#else
    if (errno == EINTR)
        errno = EINPROGRESS;
#endif
    return -1;
}

zmq::fd_t zmq::tcp_connecter_t::connect()
{
    //  Async connect has finished. Check whether an error occurred
    int err = 0;
#if defined ZMQ_HAVE_HPUX || defined ZMQ_HAVE_VXWORKS
    int len = sizeof err;
#else
    socklen_t len = sizeof err;
#endif

    const int rc = getsockopt(_s, SOL_SOCKET, SO_ERROR, reinterpret_cast<char *>(&err), &len);

    //  Assert if the error was caused by 0MQ bug.
    //  Networking problems are OK. No need to assert.
#ifdef ZMQ_HAVE_WINDOWS
    zmq_assert(rc == 0);
    if (err != 0)
    {
        if (err == WSAEBADF || err == WSAENOPROTOOPT || err == WSAENOTSOCK || err == WSAENOBUFS)
        {
            wsa_assert_no(err);
        }
        return retired_fd;
    }
#else
    //  Following code should handle both Berkeley-derived socket
    //  implementations and Solaris.
    if (rc == -1)
        err = errno;
    if (err != 0)
    {
        errno = err;
#if !defined(TARGET_OS_IPHONE) || !TARGET_OS_IPHONE
        errno_assert(errno != EBADF && errno != ENOPROTOOPT && errno != ENOTSOCK && errno != ENOBUFS);
#else
        errno_assert(errno != ENOPROTOOPT && errno != ENOTSOCK && errno != ENOBUFS);
#endif
        return retired_fd;
    }
#endif

    //  Return the newly connected socket.
    const fd_t result = _s;
    _s                = retired_fd;
    return result;
}

bool zmq::tcp_connecter_t::tune_socket(const fd_t fd_)
{
    const int rc = tune_tcp_socket(fd_) |
                   tune_tcp_keepalives(fd_, options.tcp_keepalive, options.tcp_keepalive_cnt,
                                       options.tcp_keepalive_idle, options.tcp_keepalive_intvl) |
                   tune_tcp_maxrt(fd_, options.tcp_maxrt);
    return rc == 0;
}

//========= end of tcp_connecter.cpp ============

//========= begin of tcp_listener.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <new>

#include <stdio.h>
#include <string>

// ans ignore: #include "tcp_listener.hpp"
// ans ignore: #include "io_thread.hpp"
// ans ignore: #include "config.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "ip.hpp"
// ans ignore: #include "tcp.hpp"
// ans ignore: #include "socket_base.hpp"
// ans ignore: #include "address.hpp"

#ifndef ZMQ_HAVE_WINDOWS
#include <arpa/inet.h>
#include <fcntl.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <sys/socket.h>
#include <unistd.h>
#ifdef ZMQ_HAVE_VXWORKS
#include <sockLib.h>
#endif
#endif

#ifdef ZMQ_HAVE_OPENVMS
#include <ioctl.h>
#endif

zmq::tcp_listener_t::tcp_listener_t(io_thread_t *io_thread_, socket_base_t *socket_, const options_t &options_)
    : stream_listener_base_t(io_thread_, socket_, options_)
{
}

void zmq::tcp_listener_t::in_event()
{
    fd_t fd = accept();

    //  If connection was reset by the peer in the meantime, just ignore it.
    //  TODO: Handle specific errors like ENFILE/EMFILE etc.
    if (fd == retired_fd)
    {
        _socket->event_accept_failed(make_unconnected_bind_endpoint_pair(_endpoint), zmq_errno());
        return;
    }

    int rc = tune_tcp_socket(fd);
    rc     = rc | tune_tcp_keepalives(fd, options.tcp_keepalive, options.tcp_keepalive_cnt, options.tcp_keepalive_idle,
                                  options.tcp_keepalive_intvl);
    rc     = rc | tune_tcp_maxrt(fd, options.tcp_maxrt);
    if (rc != 0)
    {
        _socket->event_accept_failed(make_unconnected_bind_endpoint_pair(_endpoint), zmq_errno());
        return;
    }

    //  Create the engine object for this connection.
    create_engine(fd);
}

std::string zmq::tcp_listener_t::get_socket_name(zmq::fd_t fd_, socket_end_t socket_end_) const
{
    return zmq::get_socket_name<tcp_address_t>(fd_, socket_end_);
}

int zmq::tcp_listener_t::create_socket(const char *addr_)
{
    _s = tcp_open_socket(addr_, options, true, true, &_address);
    if (_s == retired_fd)
    {
        return -1;
    }

    //  TODO why is this only done for the listener?
    make_socket_noninheritable(_s);

    //  Allow reusing of the address.
    int flag = 1;
    int rc;
#ifdef ZMQ_HAVE_WINDOWS
    //  TODO this was changed for Windows from SO_REUSEADDRE to
    //  SE_EXCLUSIVEADDRUSE by 0ab65324195ad70205514d465b03d851a6de051c,
    //  so the comment above is no longer correct; also, now the settings are
    //  different between listener and connecter with a src address.
    //  is this intentional?
    rc = setsockopt(_s, SOL_SOCKET, SO_EXCLUSIVEADDRUSE, reinterpret_cast<const char *>(&flag), sizeof(int));
    wsa_assert(rc != SOCKET_ERROR);
#elif defined ZMQ_HAVE_VXWORKS
    rc = setsockopt(_s, SOL_SOCKET, SO_REUSEADDR, (char *)&flag, sizeof(int));
    errno_assert(rc == 0);
#else
    rc = setsockopt(_s, SOL_SOCKET, SO_REUSEADDR, &flag, sizeof(int));
    errno_assert(rc == 0);
#endif

    //  Bind the socket to the network interface and port.
#if defined ZMQ_HAVE_VXWORKS
    rc = bind(_s, (sockaddr *)_address.addr(), _address.addrlen());
#else
    rc = bind(_s, _address.addr(), _address.addrlen());
#endif
#ifdef ZMQ_HAVE_WINDOWS
    if (rc == SOCKET_ERROR)
    {
        errno = wsa_error_to_errno(WSAGetLastError());
        goto error;
    }
#else
    if (rc != 0)
        goto error;
#endif

    //  Listen for incoming connections.
    rc = listen(_s, options.backlog);
#ifdef ZMQ_HAVE_WINDOWS
    if (rc == SOCKET_ERROR)
    {
        errno = wsa_error_to_errno(WSAGetLastError());
        goto error;
    }
#else
    if (rc != 0)
        goto error;
#endif

    return 0;

error:
    int err = errno;
    close();
    errno = err;
    return -1;
}

int zmq::tcp_listener_t::set_local_address(const char *addr_)
{
    if (options.use_fd != -1)
    {
        //  in this case, the addr_ passed is not used and ignored, since the
        //  socket was already created by the application
        _s = options.use_fd;
    }
    else
    {
        if (create_socket(addr_) == -1)
            return -1;
    }

    _endpoint = get_socket_name(_s, socket_end_local);

    _socket->event_listening(make_unconnected_bind_endpoint_pair(_endpoint), _s);
    return 0;
}

zmq::fd_t zmq::tcp_listener_t::accept()
{
    //  The situation where connection cannot be accepted due to insufficient
    //  resources is considered valid and treated by ignoring the connection.
    //  Accept one connection and deal with different failure modes.
    zmq_assert(_s != retired_fd);

    struct sockaddr_storage ss;
    memset(&ss, 0, sizeof(ss));
#if defined ZMQ_HAVE_HPUX || defined ZMQ_HAVE_VXWORKS
    int ss_len = sizeof(ss);
#else
    socklen_t ss_len = sizeof(ss);
#endif
#if defined ZMQ_HAVE_SOCK_CLOEXEC && defined HAVE_ACCEPT4
    fd_t sock = ::accept4(_s, reinterpret_cast<struct sockaddr *>(&ss), &ss_len, SOCK_CLOEXEC);
#else
    fd_t      sock   = ::accept(_s, reinterpret_cast<struct sockaddr *>(&ss), &ss_len);
#endif

    if (sock == retired_fd)
    {
#if defined ZMQ_HAVE_WINDOWS
        const int last_error = WSAGetLastError();
        wsa_assert(last_error == WSAEWOULDBLOCK || last_error == WSAECONNRESET || last_error == WSAEMFILE ||
                   last_error == WSAENOBUFS);
#elif defined ZMQ_HAVE_ANDROID
        errno_assert(errno == EAGAIN || errno == EWOULDBLOCK || errno == EINTR || errno == ECONNABORTED ||
                     errno == EPROTO || errno == ENOBUFS || errno == ENOMEM || errno == EMFILE || errno == ENFILE ||
                     errno == EINVAL);
#else
        errno_assert(errno == EAGAIN || errno == EWOULDBLOCK || errno == EINTR || errno == ECONNABORTED ||
                     errno == EPROTO || errno == ENOBUFS || errno == ENOMEM || errno == EMFILE || errno == ENFILE);
#endif
        return retired_fd;
    }

    make_socket_noninheritable(sock);

    if (!options.tcp_accept_filters.empty())
    {
        bool matched = false;
        for (options_t::tcp_accept_filters_t::size_type i = 0; i != options.tcp_accept_filters.size(); ++i)
        {
            if (options.tcp_accept_filters[i].match_address(reinterpret_cast<struct sockaddr *>(&ss), ss_len))
            {
                matched = true;
                break;
            }
        }
        if (!matched)
        {
#ifdef ZMQ_HAVE_WINDOWS
            int rc = closesocket(sock);
            wsa_assert(rc != SOCKET_ERROR);
#else
            int rc = ::close(sock);
            errno_assert(rc == 0);
#endif
            return retired_fd;
        }
    }

    if (zmq::set_nosigpipe(sock))
    {
#ifdef ZMQ_HAVE_WINDOWS
        int rc = closesocket(sock);
        wsa_assert(rc != SOCKET_ERROR);
#else
        int rc = ::close(sock);
        errno_assert(rc == 0);
#endif
        return retired_fd;
    }

    // Set the IP Type-Of-Service priority for this client socket
    if (options.tos != 0)
        set_ip_type_of_service(sock, options.tos);

    return sock;
}

//========= end of tcp_listener.cpp ============

//========= begin of thread.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "thread.hpp"
// ans ignore: #include "err.hpp"

bool zmq::thread_t::get_started() const { return _started; }

#ifdef ZMQ_HAVE_WINDOWS

extern "C" {
#if defined  _WIN32_WCE
static DWORD thread_routine(LPVOID arg_)
#else
static unsigned int __stdcall thread_routine(void *arg_)
#endif
{
    zmq::thread_t *self = (zmq::thread_t *)arg_;
    self->applyThreadName();
    self->_tfn(self->_arg);
    return 0;
}
}

void zmq::thread_t::start(thread_fn *tfn_, void *arg_, const char *name_)
{
    _tfn = tfn_;
    _arg = arg_;
    if (name_)
        strncpy(_name, name_, sizeof(_name) - 1);
#if defined _WIN32_WCE
    _descriptor = (HANDLE)CreateThread(NULL, 0, &::thread_routine, this, 0, NULL);
#else
    _descriptor = (HANDLE)_beginthreadex(NULL, 0, &::thread_routine, this, 0, NULL);
#endif
    win_assert(_descriptor != NULL);
    _started = true;
}

bool zmq::thread_t::is_current_thread() const { return GetCurrentThreadId() == GetThreadId(_descriptor); }

void zmq::thread_t::stop()
{
    if (_started)
    {
        DWORD rc = WaitForSingleObject(_descriptor, INFINITE);
        win_assert(rc != WAIT_FAILED);
        BOOL rc2 = CloseHandle(_descriptor);
        win_assert(rc2 != 0);
    }
}

void zmq::thread_t::setSchedulingParameters(int priority_, int scheduling_policy_, const std::set<int> &affinity_cpus_)
{
    // not implemented
    LIBZMQ_UNUSED(priority_);
    LIBZMQ_UNUSED(scheduling_policy_);
    LIBZMQ_UNUSED(affinity_cpus_);
}

void zmq::thread_t::applySchedulingParameters() // to be called in secondary thread context
{
    // not implemented
}

namespace
{
#pragma pack(push, 8)
struct thread_info_t
{
    DWORD  _type;
    LPCSTR _name;
    DWORD  _thread_id;
    DWORD  _flags;
};
#pragma pack(pop)
} // namespace

void zmq::thread_t::applyThreadName() // to be called in secondary thread context
{
    if (!_name[0])
        return;

    thread_info_t thread_info;
    thread_info._type      = 0x1000;
    thread_info._name      = _name;
    thread_info._thread_id = -1;
    thread_info._flags     = 0;

#pragma warning(push)
#pragma warning(disable : 6320 6322)
    __try
    {
        DWORD MS_VC_EXCEPTION = 0x406D1388;
        RaiseException(MS_VC_EXCEPTION, 0, sizeof(thread_info) / sizeof(ULONG_PTR), (ULONG_PTR *)&thread_info);
    }
    __except (EXCEPTION_CONTINUE_EXECUTION)
    {
    }
#pragma warning(pop)
}

#elif defined ZMQ_HAVE_VXWORKS

extern "C" {
static void *thread_routine(void *arg_)
{
    zmq::thread_t *self = (zmq::thread_t *)arg_;
    self->applySchedulingParameters();
    self->_tfn(self->_arg);
    return NULL;
}
}

void zmq::thread_t::start(thread_fn *tfn_, void *arg_, const char *name_)
{
    LIBZMQ_UNUSED(name_);
    _tfn        = tfn_;
    _arg        = arg_;
    _descriptor = taskSpawn(NULL, DEFAULT_PRIORITY, DEFAULT_OPTIONS, DEFAULT_STACK_SIZE, (FUNCPTR)thread_routine,
                            (int)this, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    if (_descriptor != NULL || _descriptor > 0)
        _started = true;
}

void zmq::thread_t::stop()
{
    if (_started)
        while ((_descriptor != NULL || _descriptor > 0) && taskIdVerify(_descriptor) == 0)
        {
        }
}

bool zmq::thread_t::is_current_thread() const { return taskIdSelf() == _descriptor; }

void zmq::thread_t::setSchedulingParameters(int priority_, int schedulingPolicy_, const std::set<int> &affinity_cpus_)
{
    _thread_priority      = priority_;
    _thread_sched_policy  = schedulingPolicy_;
    _thread_affinity_cpus = affinity_cpus_;
}

void zmq::thread_t::applySchedulingParameters() // to be called in secondary thread context
{
    int priority = (_thread_priority >= 0 ? _thread_priority : DEFAULT_PRIORITY);
    priority     = (priority < UCHAR_MAX ? priority : DEFAULT_PRIORITY);
    if (_descriptor != NULL || _descriptor > 0)
    {
        taskPrioritySet(_descriptor, priority);
    }
}

void zmq::thread_t::applyThreadName() // to be called in secondary thread context
{
    // not implemented
}

#else

#include <signal.h>
#include <sys/resource.h>
#include <sys/time.h>
#include <unistd.h>

extern "C" {
static void *thread_routine(void *arg_)
{
#if !defined ZMQ_HAVE_OPENVMS && !defined ZMQ_HAVE_ANDROID
    //  Following code will guarantee more predictable latencies as it'll
    //  disallow any signal handling in the I/O thread.
    sigset_t signal_set;
    int rc = sigfillset(&signal_set);
    errno_assert(rc == 0);
    rc = pthread_sigmask(SIG_BLOCK, &signal_set, NULL);
    posix_assert(rc);
#endif
    zmq::thread_t *self = (zmq::thread_t *)arg_;
    self->applySchedulingParameters();
    self->applyThreadName();
    self->_tfn(self->_arg);
    return NULL;
}
}

void zmq::thread_t::start(thread_fn *tfn_, void *arg_, const char *name_)
{
    _tfn = tfn_;
    _arg = arg_;
    if (name_)
        strncpy(_name, name_, sizeof(_name) - 1);
    int rc = pthread_create(&_descriptor, NULL, thread_routine, this);
    posix_assert(rc);
    _started = true;
}

void zmq::thread_t::stop()
{
    if (_started)
    {
        int rc = pthread_join(_descriptor, NULL);
        posix_assert(rc);
    }
}

bool zmq::thread_t::is_current_thread() const { return bool(pthread_equal(pthread_self(), _descriptor)); }

void zmq::thread_t::setSchedulingParameters(int priority_, int schedulingPolicy_, const std::set<int> &affinity_cpus_)
{
    _thread_priority = priority_;
    _thread_sched_policy = schedulingPolicy_;
    _thread_affinity_cpus = affinity_cpus_;
}

void zmq::thread_t::applySchedulingParameters() // to be called in secondary thread context
{
#if defined _POSIX_THREAD_PRIORITY_SCHEDULING && _POSIX_THREAD_PRIORITY_SCHEDULING >= 0
    int policy = 0;
    struct sched_param param;

#if _POSIX_THREAD_PRIORITY_SCHEDULING == 0 && defined _SC_THREAD_PRIORITY_SCHEDULING
    if (sysconf(_SC_THREAD_PRIORITY_SCHEDULING) < 0)
    {
        return;
    }
#endif
    int rc = pthread_getschedparam(pthread_self(), &policy, &param);
    posix_assert(rc);

    if (_thread_sched_policy != ZMQ_THREAD_SCHED_POLICY_DFLT)
    {
        policy = _thread_sched_policy;
    }

    /* Quoting docs:
       "Linux allows the static priority range 1 to 99 for the SCHED_FIFO and
       SCHED_RR policies, and the priority 0 for the remaining policies."
       Other policies may use the "nice value" in place of the priority:
    */
    bool use_nice_instead_priority = (policy != SCHED_FIFO) && (policy != SCHED_RR);

    if (_thread_priority != ZMQ_THREAD_PRIORITY_DFLT)
    {
        if (use_nice_instead_priority)
            param.sched_priority = 0; // this is the only supported priority for most scheduling policies
        else
            param.sched_priority = _thread_priority; // user should provide a value between 1 and 99
    }

#ifdef __NetBSD__
    if (policy == SCHED_OTHER)
        param.sched_priority = -1;
#endif

    rc = pthread_setschedparam(pthread_self(), policy, &param);

#if defined(__FreeBSD_kernel__) || defined(__FreeBSD__)
    // If this feature is unavailable at run-time, don't abort.
    if (rc == ENOSYS)
        return;
#endif

    posix_assert(rc);

#if !defined ZMQ_HAVE_VXWORKS
    if (use_nice_instead_priority && _thread_priority != ZMQ_THREAD_PRIORITY_DFLT)
    {
        // assume the user wants to decrease the thread's nice value
        // i.e., increase the chance of this thread being scheduled: try setting that to
        // maximum priority.
        rc = nice(-20);

        errno_assert(rc != -1);
        // IMPORTANT: EPERM is typically returned for unprivileged processes: that's because
        //            CAP_SYS_NICE capability is required or RLIMIT_NICE resource limit should be changed to avoid
        //            EPERM!
    }
#endif

#ifdef ZMQ_HAVE_PTHREAD_SET_AFFINITY
    if (!_thread_affinity_cpus.empty())
    {
        cpu_set_t cpuset;
        CPU_ZERO(&cpuset);
        for (std::set<int>::const_iterator it = _thread_affinity_cpus.begin(); it != _thread_affinity_cpus.end(); it++)
        {
            CPU_SET((int)(*it), &cpuset);
        }
        rc = pthread_setaffinity_np(pthread_self(), sizeof(cpu_set_t), &cpuset);
        posix_assert(rc);
    }
#endif
#endif
}

void zmq::thread_t::applyThreadName() // to be called in secondary thread context
{
    /* The thread name is a cosmetic string, added to ease debugging of
     * multi-threaded applications. It is not a big issue if this value
     * can not be set for any reason (such as Permission denied in some
     * cases where the application changes its EUID, etc.) The value of
     * "int rc" is retained where available, to help debuggers stepping
     * through code to see its value - but otherwise it is ignored.
     */
    if (!_name[0])
        return;

        /* Fails with permission denied on Android 5/6 */
#if defined(ZMQ_HAVE_ANDROID)
    return;
#endif

#if defined(ZMQ_HAVE_PTHREAD_SETNAME_1)
    int rc = pthread_setname_np(_name);
    if (rc)
        return;
#elif defined(ZMQ_HAVE_PTHREAD_SETNAME_2)
    int rc = pthread_setname_np(pthread_self(), _name);
    if (rc)
        return;
#elif defined(ZMQ_HAVE_PTHREAD_SETNAME_3)
    int rc = pthread_setname_np(pthread_self(), _name, NULL);
    if (rc)
        return;
#elif defined(ZMQ_HAVE_PTHREAD_SET_NAME)
    pthread_set_name_np(pthread_self(), _name);
#endif
}

#endif

//========= end of thread.cpp ============

//========= begin of timers.cpp ============

/*
Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

This file is part of libzmq, the ZeroMQ core engine in C++.

libzmq is free software; you can redistribute it and/or modify it under
the terms of the GNU Lesser General Public License (LGPL) as published
by the Free Software Foundation; either version 3 of the License, or
(at your option) any later version.

As a special exception, the Contributors give you permission to link
this library with independent modules to produce an executable,
regardless of the license terms of these independent modules, and to
copy and distribute the resulting executable under terms of your choice,
provided that you also meet, for each linked independent module, the
terms and conditions of the license of that module. An independent
module is a module which is not derived from or based on this library.
If you modify this library, you must extend this exception to your
version of the library.

libzmq is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "timers.hpp"
// ans ignore: #include "err.hpp"

#include <algorithm>

zmq::timers_t::timers_t() : _tag(0xCAFEDADA), _next_timer_id(0) {}

zmq::timers_t::~timers_t()
{
    //  Mark the timers as dead
    _tag = 0xdeadbeef;
}

bool zmq::timers_t::check_tag() { return _tag == 0xCAFEDADA; }

int zmq::timers_t::add(size_t interval_, timers_timer_fn handler_, void *arg_)
{
    if (handler_ == NULL)
    {
        errno = EFAULT;
        return -1;
    }

    uint64_t when  = _clock.now_ms() + interval_;
    timer_t  timer = {++_next_timer_id, interval_, handler_, arg_};
    _timers.insert(timersmap_t::value_type(when, timer));

    return timer.timer_id;
}

struct zmq::timers_t::match_by_id
{
    match_by_id(int timer_id_) : _timer_id(timer_id_) {}

    bool operator()(timersmap_t::value_type const &entry_) const { return entry_.second.timer_id == _timer_id; }

  private:
    int _timer_id;
};

int zmq::timers_t::cancel(int timer_id_)
{
    // check first if timer exists at all
    if (_timers.end() == std::find_if(_timers.begin(), _timers.end(), match_by_id(timer_id_)))
    {
        errno = EINVAL;
        return -1;
    }

    // check if timer was already canceled
    if (_cancelled_timers.count(timer_id_))
    {
        errno = EINVAL;
        return -1;
    }

    _cancelled_timers.insert(timer_id_);

    return 0;
}

int zmq::timers_t::set_interval(int timer_id_, size_t interval_)
{
    const timersmap_t::iterator end = _timers.end();
    const timersmap_t::iterator it  = std::find_if(_timers.begin(), end, match_by_id(timer_id_));
    if (it != end)
    {
        timer_t timer  = it->second;
        timer.interval = interval_;
        uint64_t when  = _clock.now_ms() + interval_;
        _timers.erase(it);
        _timers.insert(timersmap_t::value_type(when, timer));

        return 0;
    }

    errno = EINVAL;
    return -1;
}

int zmq::timers_t::reset(int timer_id_)
{
    const timersmap_t::iterator end = _timers.end();
    const timersmap_t::iterator it  = std::find_if(_timers.begin(), end, match_by_id(timer_id_));
    if (it != end)
    {
        timer_t  timer = it->second;
        uint64_t when  = _clock.now_ms() + timer.interval;
        _timers.erase(it);
        _timers.insert(timersmap_t::value_type(when, timer));

        return 0;
    }

    errno = EINVAL;
    return -1;
}

long zmq::timers_t::timeout()
{
    const uint64_t now = _clock.now_ms();
    long           res = -1;

    const timersmap_t::iterator begin = _timers.begin();
    const timersmap_t::iterator end   = _timers.end();
    timersmap_t::iterator       it    = begin;
    for (; it != end; ++it)
    {
        if (0 == _cancelled_timers.erase(it->second.timer_id))
        {
            //  Live timer, lets return the timeout
            res = std::max(static_cast<long>(it->first - now), 0l);
            break;
        }
    }

    //  Remove timed-out timers
    _timers.erase(begin, it);

    return res;
}

int zmq::timers_t::execute()
{
    const uint64_t now = _clock.now_ms();

    const timersmap_t::iterator begin = _timers.begin();
    const timersmap_t::iterator end   = _timers.end();
    timersmap_t::iterator       it    = _timers.begin();
    for (; it != end; ++it)
    {
        if (0 == _cancelled_timers.erase(it->second.timer_id))
        {
            //  Timer is not cancelled

            //  Map is ordered, if we have to wait for current timer we can stop.
            if (it->first > now)
                break;

            const timer_t &timer = it->second;

            timer.handler(timer.timer_id, timer.arg);

            _timers.insert(timersmap_t::value_type(now + timer.interval, timer));
        }
    }
    _timers.erase(begin, it);

    return 0;
}

//========= end of timers.cpp ============

//========= begin of tipc_address.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"

// ans ignore: #include "tipc_address.hpp"

#if defined ZMQ_HAVE_TIPC

// ans ignore: #include "err.hpp"

#include <sstream>
#include <string>

zmq::tipc_address_t::tipc_address_t()
{
    memset(&address, 0, sizeof address);
    _random = false;
}

zmq::tipc_address_t::tipc_address_t(const sockaddr *sa, socklen_t sa_len)
{
    zmq_assert(sa && sa_len > 0);

    memset(&address, 0, sizeof address);
    if (sa->sa_family == AF_TIPC)
        memcpy(&address, sa, sa_len);

    _random = false;
}

void zmq::tipc_address_t::set_random() { _random = true; }
bool zmq::tipc_address_t::is_random() const { return _random; }
bool zmq::tipc_address_t::is_service() const
{
    if (address.addrtype == TIPC_ADDR_ID)
        return false;

    return true;
}
int zmq::tipc_address_t::resolve(const char *name)
{
    unsigned int type  = 0;
    unsigned int lower = 0;
    unsigned int upper = 0;
    unsigned int ref   = 0;
    unsigned int z = 1, c = 0, n = 0;
    char         eof;
    const char * domain;
    int          res;

    if (strncmp(name, "<*>", 3) == 0)
    {
        set_random();
        address.family       = AF_TIPC;
        address.addrtype     = TIPC_ADDR_ID;
        address.addr.id.node = 0;
        address.addr.id.ref  = 0;
        address.scope        = 0;
        return 0;
    }

    res = sscanf(name, "{%u,%u,%u}", &type, &lower, &upper);
    /* Fetch optional domain suffix. */
    if ((domain = strchr(name, '@')))
    {
        if (sscanf(domain, "@%u.%u.%u%c", &z, &c, &n, &eof) != 3)
            return EINVAL;
    }
    if (res == 3)
    {
        if (type < TIPC_RESERVED_TYPES || upper < lower)
            return EINVAL;
        address.family             = AF_TIPC;
        address.addrtype           = TIPC_ADDR_NAMESEQ;
        address.addr.nameseq.type  = type;
        address.addr.nameseq.lower = lower;
        address.addr.nameseq.upper = upper;
        address.scope              = TIPC_ZONE_SCOPE;
        return 0;
    }
    else if (res == 2 && type > TIPC_RESERVED_TYPES)
    {
        address.family                  = AF_TIPC;
        address.addrtype                = TIPC_ADDR_NAME;
        address.addr.name.name.type     = type;
        address.addr.name.name.instance = lower;
        address.addr.name.domain        = tipc_addr(z, c, n);
        address.scope                   = 0;
        return 0;
    }
    else if (res == 0)
    {
        res = sscanf(name, "<%u.%u.%u:%u>", &z, &c, &n, &ref);
        if (res == 4)
        {
            address.family       = AF_TIPC;
            address.addrtype     = TIPC_ADDR_ID;
            address.addr.id.node = tipc_addr(z, c, n);
            address.addr.id.ref  = ref;
            address.scope        = 0;
            return 0;
        }
    }
    return EINVAL;
}

int zmq::tipc_address_t::to_string(std::string &addr_) const
{
    if (address.family != AF_TIPC)
    {
        addr_.clear();
        return -1;
    }
    std::stringstream s;
    if (address.addrtype == TIPC_ADDR_NAMESEQ || address.addrtype == TIPC_ADDR_NAME)
    {
        s << "tipc://"
          << "{" << address.addr.nameseq.type;
        s << ", " << address.addr.nameseq.lower;
        s << ", " << address.addr.nameseq.upper << "}";
        addr_ = s.str();
    }
    else if (address.addrtype == TIPC_ADDR_ID || is_random())
    {
        s << "tipc://"
          << "<" << tipc_zone(address.addr.id.node);
        s << "." << tipc_cluster(address.addr.id.node);
        s << "." << tipc_node(address.addr.id.node);
        s << ":" << address.addr.id.ref << ">";
        addr_ = s.str();
    }
    else
    {
        addr_.clear();
        return -1;
    }
    return 0;
}

const sockaddr *zmq::tipc_address_t::addr() const { return (sockaddr *)&address; }

socklen_t zmq::tipc_address_t::addrlen() const { return (socklen_t)sizeof address; }

#endif

//========= end of tipc_address.cpp ============

//========= begin of tipc_connecter.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"

// ans ignore: #include "tipc_connecter.hpp"

#if defined ZMQ_HAVE_TIPC

#include <new>
#include <string>

// ans ignore: #include "stream_engine.hpp"
// ans ignore: #include "io_thread.hpp"
// ans ignore: #include "platform.hpp"
// ans ignore: #include "random.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "ip.hpp"
// ans ignore: #include "address.hpp"
// ans ignore: #include "tipc_address.hpp"
// ans ignore: #include "session_base.hpp"

#include <sys/socket.h>
#include <sys/types.h>
#include <unistd.h>
#ifdef ZMQ_HAVE_VXWORKS
#include <sockLib.h>
#endif

zmq::tipc_connecter_t::tipc_connecter_t(class io_thread_t *io_thread_, class session_base_t *session_,
                                        const options_t &options_, address_t *addr_, bool delayed_start_)
    : stream_connecter_base_t(io_thread_, session_, options_, addr_, delayed_start_)
{
    zmq_assert(_addr->protocol == "tipc");
}

void zmq::tipc_connecter_t::out_event()
{
    fd_t fd = connect();
    rm_handle();

    //  Handle the error condition by attempt to reconnect.
    if (fd == retired_fd)
    {
        close();
        add_reconnect_timer();
        return;
    }

    create_engine(fd, get_socket_name<tipc_address_t>(fd, socket_end_local));
}

void zmq::tipc_connecter_t::start_connecting()
{
    //  Open the connecting socket.
    int rc = open();

    //  Connect may succeed in synchronous manner.
    if (rc == 0)
    {
        _handle = add_fd(_s);
        out_event();
    }

    //  Connection establishment may be delayed. Poll for its completion.
    else if (rc == -1 && errno == EINPROGRESS)
    {
        _handle = add_fd(_s);
        set_pollout(_handle);
        _socket->event_connect_delayed(make_unconnected_connect_endpoint_pair(_endpoint), zmq_errno());
    }

    //  Handle any other error condition by eventual reconnect.
    else
    {
        if (_s != retired_fd)
            close();
        add_reconnect_timer();
    }
}

int zmq::tipc_connecter_t::open()
{
    zmq_assert(_s == retired_fd);

    // Cannot connect to random tipc addresses
    if (_addr->resolved.tipc_addr->is_random())
    {
        errno = EINVAL;
        return -1;
    }
    //  Create the socket.
    _s = open_socket(AF_TIPC, SOCK_STREAM, 0);
    if (_s == -1)
        return -1;

    //  Set the non-blocking flag.
    unblock_socket(_s);
    //  Connect to the remote peer.
#ifdef ZMQ_HAVE_VXWORKS
    int rc = ::connect(s, (sockaddr *)addr->resolved.tipc_addr->addr(), addr->resolved.tipc_addr->addrlen());
#else
    int rc = ::connect(_s, _addr->resolved.tipc_addr->addr(), _addr->resolved.tipc_addr->addrlen());
#endif
    //  Connect was successful immediately.
    if (rc == 0)
        return 0;

    //  Translate other error codes indicating asynchronous connect has been
    //  launched to a uniform EINPROGRESS.
    if (rc == -1 && errno == EINTR)
    {
        errno = EINPROGRESS;
        return -1;
    }
    //  Forward the error.
    return -1;
}

zmq::fd_t zmq::tipc_connecter_t::connect()
{
    //  Following code should handle both Berkeley-derived socket
    //  implementations and Solaris.
    int err = 0;
#ifdef ZMQ_HAVE_VXWORKS
    int len = sizeof(err);
#else
    socklen_t len = sizeof(err);
#endif
    int rc = getsockopt(_s, SOL_SOCKET, SO_ERROR, (char *)&err, &len);
    if (rc == -1)
        err = errno;
    if (err != 0)
    {
        //  Assert if the error was caused by 0MQ bug.
        //  Networking problems are OK. No need to assert.
        errno = err;
        errno_assert(errno == ECONNREFUSED || errno == ECONNRESET || errno == ETIMEDOUT || errno == EHOSTUNREACH ||
                     errno == ENETUNREACH || errno == ENETDOWN);

        return retired_fd;
    }
    fd_t result = _s;
    _s          = retired_fd;
    return result;
}

#endif

//========= end of tipc_connecter.cpp ============

//========= begin of tipc_listener.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"

// ans ignore: #include "tipc_listener.hpp"

#if defined ZMQ_HAVE_TIPC

#include <new>

#include <string.h>

// ans ignore: #include "tipc_address.hpp"
// ans ignore: #include "io_thread.hpp"
// ans ignore: #include "config.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "ip.hpp"
// ans ignore: #include "socket_base.hpp"
// ans ignore: #include "address.hpp"

#include <fcntl.h>
#include <sys/socket.h>
#include <unistd.h>
#if defined ZMQ_HAVE_VXWORKS
#include <sockLib.h>
#include <tipc/tipc.h>
#else
#include <linux/tipc.h>
#endif

zmq::tipc_listener_t::tipc_listener_t(io_thread_t *io_thread_, socket_base_t *socket_, const options_t &options_)
    : stream_listener_base_t(io_thread_, socket_, options_)
{
}

void zmq::tipc_listener_t::in_event()
{
    fd_t fd = accept();

    //  If connection was reset by the peer in the meantime, just ignore it.
    //  TODO: Handle specific errors like ENFILE/EMFILE etc.
    if (fd == retired_fd)
    {
        _socket->event_accept_failed(make_unconnected_bind_endpoint_pair(_endpoint), zmq_errno());
        return;
    }

    //  Create the engine object for this connection.
    create_engine(fd);
}

std::string zmq::tipc_listener_t::get_socket_name(zmq::fd_t fd_, socket_end_t socket_end_) const
{
    return zmq::get_socket_name<tipc_address_t>(fd_, socket_end_);
}

int zmq::tipc_listener_t::set_local_address(const char *addr_)
{
    // Convert str to address struct
    int rc = _address.resolve(addr_);
    if (rc != 0)
        return -1;

    // Cannot bind non-random Port Identity
    struct sockaddr_tipc *a = (sockaddr_tipc *)_address.addr();
    if (!_address.is_random() && a->addrtype == TIPC_ADDR_ID)
    {
        errno = EINVAL;
        return -1;
    }

    //  Create a listening socket.
    _s = open_socket(AF_TIPC, SOCK_STREAM, 0);
    if (_s == -1)
        return -1;

    // If random Port Identity, update address object to reflect the assigned address
    if (_address.is_random())
    {
        struct sockaddr_storage ss;
        const zmq_socklen_t     sl = get_socket_address(_s, socket_end_local, &ss);
        if (sl == 0)
            goto error;

        _address = tipc_address_t((struct sockaddr *)&ss, sl);
    }

    _address.to_string(_endpoint);

    //  Bind the socket to tipc name
    if (_address.is_service())
    {
#ifdef ZMQ_HAVE_VXWORKS
        rc = bind(_s, (sockaddr *)address.addr(), address.addrlen());
#else
        rc = bind(_s, _address.addr(), _address.addrlen());
#endif
        if (rc != 0)
            goto error;
    }

    //  Listen for incoming connections.
    rc = listen(_s, options.backlog);
    if (rc != 0)
        goto error;

    _socket->event_listening(make_unconnected_bind_endpoint_pair(_endpoint), _s);
    return 0;

error:
    int err = errno;
    close();
    errno = err;
    return -1;
}

zmq::fd_t zmq::tipc_listener_t::accept()
{
    //  Accept one connection and deal with different failure modes.
    //  The situation where connection cannot be accepted due to insufficient
    //  resources is considered valid and treated by ignoring the connection.
    struct sockaddr_storage ss     = {};
    socklen_t               ss_len = sizeof(ss);

    zmq_assert(_s != retired_fd);
#ifdef ZMQ_HAVE_VXWORKS
    fd_t sock = ::accept(_s, (struct sockaddr *)&ss, (int *)&ss_len);
#else
    fd_t sock = ::accept(_s, (struct sockaddr *)&ss, &ss_len);
#endif
    if (sock == -1)
    {
        errno_assert(errno == EAGAIN || errno == EWOULDBLOCK || errno == ENOBUFS || errno == EINTR ||
                     errno == ECONNABORTED || errno == EPROTO || errno == EMFILE || errno == ENFILE);
        return retired_fd;
    }
    /*FIXME Accept filters?*/
    return sock;
}

#endif

//========= end of tipc_listener.cpp ============

//========= begin of trie.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "trie.hpp"

#include <stdlib.h>

#include <algorithm>
#include <new>

zmq::trie_t::trie_t() : _refcnt(0), _min(0), _count(0), _live_nodes(0) {}

zmq::trie_t::~trie_t()
{
    if (_count == 1)
    {
        zmq_assert(_next.node);
        LIBZMQ_DELETE(_next.node);
    }
    else if (_count > 1)
    {
        for (unsigned short i = 0; i != _count; ++i)
        {
            LIBZMQ_DELETE(_next.table[i]);
        }
        free(_next.table);
    }
}

bool zmq::trie_t::add(unsigned char *prefix_, size_t size_)
{
    //  We are at the node corresponding to the prefix. We are done.
    if (!size_)
    {
        ++_refcnt;
        return _refcnt == 1;
    }

    const unsigned char c = *prefix_;
    if (c < _min || c >= _min + _count)
    {
        //  The character is out of range of currently handled
        //  characters. We have to extend the table.
        if (!_count)
        {
            _min       = c;
            _count     = 1;
            _next.node = NULL;
        }
        else if (_count == 1)
        {
            const unsigned char oldc = _min;
            trie_t *            oldp = _next.node;
            _count                   = (_min < c ? c - _min : _min - c) + 1;
            _next.table              = static_cast<trie_t **>(malloc(sizeof(trie_t *) * _count));
            alloc_assert(_next.table);
            for (unsigned short i = 0; i != _count; ++i)
                _next.table[i] = 0;
            _min                     = std::min(_min, c);
            _next.table[oldc - _min] = oldp;
        }
        else if (_min < c)
        {
            //  The new character is above the current character range.
            const unsigned short old_count = _count;
            _count                         = c - _min + 1;
            _next.table                    = static_cast<trie_t **>(realloc(_next.table, sizeof(trie_t *) * _count));
            zmq_assert(_next.table);
            for (unsigned short i = old_count; i != _count; i++)
                _next.table[i] = NULL;
        }
        else
        {
            //  The new character is below the current character range.
            const unsigned short old_count = _count;
            _count                         = (_min + old_count) - c;
            _next.table                    = static_cast<trie_t **>(realloc(_next.table, sizeof(trie_t *) * _count));
            zmq_assert(_next.table);
            memmove(_next.table + _min - c, _next.table, old_count * sizeof(trie_t *));
            for (unsigned short i = 0; i != _min - c; i++)
                _next.table[i] = NULL;
            _min = c;
        }
    }

    //  If next node does not exist, create one.
    if (_count == 1)
    {
        if (!_next.node)
        {
            _next.node = new (std::nothrow) trie_t;
            alloc_assert(_next.node);
            ++_live_nodes;
            zmq_assert(_live_nodes == 1);
        }
        return _next.node->add(prefix_ + 1, size_ - 1);
    }
    if (!_next.table[c - _min])
    {
        _next.table[c - _min] = new (std::nothrow) trie_t;
        alloc_assert(_next.table[c - _min]);
        ++_live_nodes;
        zmq_assert(_live_nodes > 1);
    }
    return _next.table[c - _min]->add(prefix_ + 1, size_ - 1);
}

bool zmq::trie_t::rm(unsigned char *prefix_, size_t size_)
{
    //  TODO: Shouldn't an error be reported if the key does not exist?
    if (!size_)
    {
        if (!_refcnt)
            return false;
        _refcnt--;
        return _refcnt == 0;
    }
    const unsigned char c = *prefix_;
    if (!_count || c < _min || c >= _min + _count)
        return false;

    trie_t *next_node = _count == 1 ? _next.node : _next.table[c - _min];

    if (!next_node)
        return false;

    const bool ret = next_node->rm(prefix_ + 1, size_ - 1);

    //  Prune redundant nodes
    if (next_node->is_redundant())
    {
        LIBZMQ_DELETE(next_node);
        zmq_assert(_count > 0);

        if (_count == 1)
        {
            //  The just pruned node is was the only live node
            _next.node = 0;
            _count     = 0;
            --_live_nodes;
            zmq_assert(_live_nodes == 0);
        }
        else
        {
            _next.table[c - _min] = 0;
            zmq_assert(_live_nodes > 1);
            --_live_nodes;

            //  Compact the table if possible
            if (_live_nodes == 1)
            {
                //  We can switch to using the more compact single-node
                //  representation since the table only contains one live node
                trie_t *node = 0;
                //  Since we always compact the table the pruned node must
                //  either be the left-most or right-most ptr in the node
                //  table
                if (c == _min)
                {
                    //  The pruned node is the left-most node ptr in the
                    //  node table => keep the right-most node
                    node = _next.table[_count - 1];
                    _min += _count - 1;
                }
                else if (c == _min + _count - 1)
                {
                    //  The pruned node is the right-most node ptr in the
                    //  node table => keep the left-most node
                    node = _next.table[0];
                }
                zmq_assert(node);
                free(_next.table);
                _next.node = node;
                _count     = 1;
            }
            else if (c == _min)
            {
                //  We can compact the table "from the left".
                //  Find the left-most non-null node ptr, which we'll use as
                //  our new min
                unsigned char new_min = _min;
                for (unsigned short i = 1; i < _count; ++i)
                {
                    if (_next.table[i])
                    {
                        new_min = i + _min;
                        break;
                    }
                }
                zmq_assert(new_min != _min);

                trie_t **old_table = _next.table;
                zmq_assert(new_min > _min);
                zmq_assert(_count > new_min - _min);

                _count      = _count - (new_min - _min);
                _next.table = static_cast<trie_t **>(malloc(sizeof(trie_t *) * _count));
                alloc_assert(_next.table);

                memmove(_next.table, old_table + (new_min - _min), sizeof(trie_t *) * _count);
                free(old_table);

                _min = new_min;
            }
            else if (c == _min + _count - 1)
            {
                //  We can compact the table "from the right".
                //  Find the right-most non-null node ptr, which we'll use to
                //  determine the new table size
                unsigned short new_count = _count;
                for (unsigned short i = 1; i < _count; ++i)
                {
                    if (_next.table[_count - 1 - i])
                    {
                        new_count = _count - i;
                        break;
                    }
                }
                zmq_assert(new_count != _count);
                _count = new_count;

                trie_t **old_table = _next.table;
                _next.table        = static_cast<trie_t **>(malloc(sizeof(trie_t *) * _count));
                alloc_assert(_next.table);

                memmove(_next.table, old_table, sizeof(trie_t *) * _count);
                free(old_table);
            }
        }
    }
    return ret;
}

bool zmq::trie_t::check(unsigned char *data_, size_t size_)
{
    //  This function is on critical path. It deliberately doesn't use
    //  recursion to get a bit better performance.
    trie_t *current = this;
    while (true)
    {
        //  We've found a corresponding subscription!
        if (current->_refcnt)
            return true;

        //  We've checked all the data and haven't found matching subscription.
        if (!size_)
            return false;

        //  If there's no corresponding slot for the first character
        //  of the prefix, the message does not match.
        const unsigned char c = *data_;
        if (c < current->_min || c >= current->_min + current->_count)
            return false;

        //  Move to the next character.
        if (current->_count == 1)
            current = current->_next.node;
        else
        {
            current = current->_next.table[c - current->_min];
            if (!current)
                return false;
        }
        data_++;
        size_--;
    }
}

void zmq::trie_t::apply(void (*func_)(unsigned char *data_, size_t size_, void *arg_), void *arg_)
{
    unsigned char *buff = NULL;
    apply_helper(&buff, 0, 0, func_, arg_);
    free(buff);
}

void zmq::trie_t::apply_helper(unsigned char **buff_, size_t buffsize_, size_t maxbuffsize_,
                               void (*func_)(unsigned char *data_, size_t size_, void *arg_), void *arg_) const
{
    //  If this node is a subscription, apply the function.
    if (_refcnt)
        func_(*buff_, buffsize_, arg_);

    //  Adjust the buffer.
    if (buffsize_ >= maxbuffsize_)
    {
        maxbuffsize_ = buffsize_ + 256;
        *buff_       = static_cast<unsigned char *>(realloc(*buff_, maxbuffsize_));
        zmq_assert(*buff_);
    }

    //  If there are no subnodes in the trie, return.
    if (_count == 0)
        return;

    //  If there's one subnode (optimisation).
    if (_count == 1)
    {
        (*buff_)[buffsize_] = _min;
        buffsize_++;
        _next.node->apply_helper(buff_, buffsize_, maxbuffsize_, func_, arg_);
        return;
    }

    //  If there are multiple subnodes.
    for (unsigned short c = 0; c != _count; c++)
    {
        (*buff_)[buffsize_] = _min + c;
        if (_next.table[c])
            _next.table[c]->apply_helper(buff_, buffsize_ + 1, maxbuffsize_, func_, arg_);
    }
}

bool zmq::trie_t::is_redundant() const { return _refcnt == 0 && _live_nodes == 0; }

//========= end of trie.cpp ============

//========= begin of tweetnacl.c ============

/*
    Copyright (c) 2016-2017 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
    The precompiled header is not used for c files so this is required here.
*/
// ans ignore: #include "platform.hpp"

#if defined(ZMQ_USE_TWEETNACL)

/*
    Disable warnings for this source only, rather than for the whole
    codebase when building with C99 (gcc >= 4.2) or with Microsoft's compiler
*/
#if defined __GNUC__ && (__GNUC__ > 4 || (__GNUC__ == 4 && __GNUC_MINOR__ >= 2)) && __STDC_VERSION__ < 201112L
#pragma GCC diagnostic ignored "-Wsign-compare"
#elif defined          _MSC_VER
#pragma warning(disable : 4018 4244 4146)
#endif

/* clang-format off */

// ans ignore: #include "tweetnacl.h"

#define FOR(i,n) for (i = 0;i < n;++i)
#define sv static void

static const u8
  _0[16] = {0},
  _9[32] = {9};
static const gf
  gf0 = {0},
  gf1 = {1},
  _121665 = {0xDB41,1},
  D = {0x78a3, 0x1359, 0x4dca, 0x75eb, 0xd8ab, 0x4141, 0x0a4d, 0x0070, 0xe898, 0x7779, 0x4079, 0x8cc7, 0xfe73, 0x2b6f, 0x6cee, 0x5203},
  D2 = {0xf159, 0x26b2, 0x9b94, 0xebd6, 0xb156, 0x8283, 0x149a, 0x00e0, 0xd130, 0xeef3, 0x80f2, 0x198e, 0xfce7, 0x56df, 0xd9dc, 0x2406},
  X = {0xd51a, 0x8f25, 0x2d60, 0xc956, 0xa7b2, 0x9525, 0xc760, 0x692c, 0xdc5c, 0xfdd6, 0xe231, 0xc0a4, 0x53fe, 0xcd6e, 0x36d3, 0x2169},
  Y = {0x6658, 0x6666, 0x6666, 0x6666, 0x6666, 0x6666, 0x6666, 0x6666, 0x6666, 0x6666, 0x6666, 0x6666, 0x6666, 0x6666, 0x6666, 0x6666},
  I = {0xa0b0, 0x4a0e, 0x1b27, 0xc4ee, 0xe478, 0xad2f, 0x1806, 0x2f43, 0xd7a7, 0x3dfb, 0x0099, 0x2b4d, 0xdf0b, 0x4fc1, 0x2480, 0x2b83};

static u32 L32(u32 x,int c) { return (x << c) | ((x&0xffffffff) >> (32 - c)); }

static u32 ld32(const u8 *x)
{
  u32 u = x[3];
  u = (u<<8)|x[2];
  u = (u<<8)|x[1];
  return (u<<8)|x[0];
}

static u64 dl64(const u8 *x)
{
  u64 i,u=0;
  FOR(i,8) u=(u<<8)|x[i];
  return u;
}

sv st32(u8 *x,u32 u)
{
  int i;
  FOR(i,4) { x[i] = u; u >>= 8; }
}

sv ts64(u8 *x,u64 u)
{
  int i;
  for (i = 7;i >= 0;--i) { x[i] = u; u >>= 8; }
}

static int vn(const u8 *x,const u8 *y,int n)
{
  u32 i,d = 0;
  FOR(i,n) d |= x[i]^y[i];
  return (1 & ((d - 1) >> 8)) - 1;
}

int crypto_verify_16(const u8 *x,const u8 *y)
{
  return vn(x,y,16);
}

int crypto_verify_32(const u8 *x,const u8 *y)
{
  return vn(x,y,32);
}

sv core(u8 *out,const u8 *in,const u8 *k,const u8 *c,int h)
{
  u32 w[16],x[16],y[16],t[4];
  int i,j,m;

  FOR(i,4) {
    x[5*i] = ld32(c+4*i);
    x[1+i] = ld32(k+4*i);
    x[6+i] = ld32(in+4*i);
    x[11+i] = ld32(k+16+4*i);
  }

  FOR(i,16) y[i] = x[i];

  FOR(i,20) {
    FOR(j,4) {
      FOR(m,4) t[m] = x[(5*j+4*m)%16];
      t[1] ^= L32(t[0]+t[3], 7);
      t[2] ^= L32(t[1]+t[0], 9);
      t[3] ^= L32(t[2]+t[1],13);
      t[0] ^= L32(t[3]+t[2],18);
      FOR(m,4) w[4*j+(j+m)%4] = t[m];
    }
    FOR(m,16) x[m] = w[m];
  }

  if (h) {
    FOR(i,16) x[i] += y[i];
    FOR(i,4) {
      x[5*i] -= ld32(c+4*i);
      x[6+i] -= ld32(in+4*i);
    }
    FOR(i,4) {
      st32(out+4*i,x[5*i]);
      st32(out+16+4*i,x[6+i]);
    }
  } else
    FOR(i,16) st32(out + 4 * i,x[i] + y[i]);
}

int crypto_core_salsa20(u8 *out,const u8 *in,const u8 *k,const u8 *c)
{
  core(out,in,k,c,0);
  return 0;
}

int crypto_core_hsalsa20(u8 *out,const u8 *in,const u8 *k,const u8 *c)
{
  core(out,in,k,c,1);
  return 0;
}

static const u8 sigma[17] = "expand 32-byte k";

int crypto_stream_salsa20_xor(u8 *c,const u8 *m,u64 b,const u8 *n,const u8 *k)
{
  u8 z[16],x[64];
  u32 u,i;
  if (!b) return 0;
  FOR(i,16) z[i] = 0;
  FOR(i,8) z[i] = n[i];
  while (b >= 64) {
    crypto_core_salsa20(x,z,k,sigma);
    FOR(i,64) c[i] = (m?m[i]:0) ^ x[i];
    u = 1;
    for (i = 8;i < 16;++i) {
      u += (u32) z[i];
      z[i] = u;
      u >>= 8;
    }
    b -= 64;
    c += 64;
    if (m) m += 64;
  }
  if (b) {
    crypto_core_salsa20(x,z,k,sigma);
    FOR(i,b) c[i] = (m?m[i]:0) ^ x[i];
  }
  return 0;
}

int crypto_stream_salsa20(u8 *c,u64 d,const u8 *n,const u8 *k)
{
  return crypto_stream_salsa20_xor(c,0,d,n,k);
}

int crypto_stream(u8 *c,u64 d,const u8 *n,const u8 *k)
{
  u8 s[32];
  crypto_core_hsalsa20(s,n,k,sigma);
  return crypto_stream_salsa20(c,d,n+16,s);
}

int crypto_stream_xor(u8 *c,const u8 *m,u64 d,const u8 *n,const u8 *k)
{
  u8 s[32];
  crypto_core_hsalsa20(s,n,k,sigma);
  return crypto_stream_salsa20_xor(c,m,d,n+16,s);
}

sv add1305(u32 *h,const u32 *c)
{
  u32 j,u = 0;
  FOR(j,17) {
    u += h[j] + c[j];
    h[j] = u & 255;
    u >>= 8;
  }
}

static const u32 minusp[17] = {
  5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 252
} ;

int crypto_onetimeauth(u8 *out,const u8 *m,u64 n,const u8 *k)
{
  u32 s,i,j,u,x[17],r[17],h[17],c[17],g[17];

  FOR(j,17) r[j]=h[j]=0;
  FOR(j,16) r[j]=k[j];
  r[3]&=15;
  r[4]&=252;
  r[7]&=15;
  r[8]&=252;
  r[11]&=15;
  r[12]&=252;
  r[15]&=15;

  while (n > 0) {
    FOR(j,17) c[j] = 0;
    for (j = 0;(j < 16) && (j < n);++j) c[j] = m[j];
    c[j] = 1;
    m += j; n -= j;
    add1305(h,c);
    FOR(i,17) {
      x[i] = 0;
      FOR(j,17) x[i] += h[j] * ((j <= i) ? r[i - j] : 320 * r[i + 17 - j]);
    }
    FOR(i,17) h[i] = x[i];
    u = 0;
    FOR(j,16) {
      u += h[j];
      h[j] = u & 255;
      u >>= 8;
    }
    u += h[16]; h[16] = u & 3;
    u = 5 * (u >> 2);
    FOR(j,16) {
      u += h[j];
      h[j] = u & 255;
      u >>= 8;
    }
    u += h[16]; h[16] = u;
  }

  FOR(j,17) g[j] = h[j];
  add1305(h,minusp);
  s = -(h[16] >> 7);
  FOR(j,17) h[j] ^= s & (g[j] ^ h[j]);

  FOR(j,16) c[j] = k[j + 16];
  c[16] = 0;
  add1305(h,c);
  FOR(j,16) out[j] = h[j];
  return 0;
}

int crypto_onetimeauth_verify(const u8 *h,const u8 *m,u64 n,const u8 *k)
{
  u8 x[16];
  crypto_onetimeauth(x,m,n,k);
  return crypto_verify_16(h,x);
}

int crypto_secretbox(u8 *c,const u8 *m,u64 d,const u8 *n,const u8 *k)
{
  int i;
  if (d < 32) return -1;
  crypto_stream_xor(c,m,d,n,k);
  crypto_onetimeauth(c + 16,c + 32,d - 32,c);
  FOR(i,16) c[i] = 0;
  return 0;
}

int crypto_secretbox_open(u8 *m,const u8 *c,u64 d,const u8 *n,const u8 *k)
{
  int i;
  u8 x[32];
  if (d < 32) return -1;
  crypto_stream(x,32,n,k);
  if (crypto_onetimeauth_verify(c + 16,c + 32,d - 32,x) != 0) return -1;
  crypto_stream_xor(m,c,d,n,k);
  FOR(i,32) m[i] = 0;
  return 0;
}

sv set25519(gf r, const gf a)
{
  int i;
  FOR(i,16) r[i]=a[i];
}

sv car25519(gf o)
{
  int i;
  i64 c;
  FOR(i,16) {
    o[i]+=(1LL<<16);
    c=o[i]>>16;
    o[(i+1)*(i<15)]+=c-1+37*(c-1)*(i==15);
    o[i]-=c<<16;
  }
}

sv sel25519(gf p,gf q,int b)
{
  i64 t,i,c=~(b-1);
  FOR(i,16) {
    t= c&(p[i]^q[i]);
    p[i]^=t;
    q[i]^=t;
  }
}

sv pack25519(u8 *o,const gf n)
{
  int i,j,b;
  gf m,t;
  FOR(i,16) t[i]=n[i];
  car25519(t);
  car25519(t);
  car25519(t);
  FOR(j,2) {
    m[0]=t[0]-0xffed;
    for(i=1;i<15;i++) {
      m[i]=t[i]-0xffff-((m[i-1]>>16)&1);
      m[i-1]&=0xffff;
    }
    m[15]=t[15]-0x7fff-((m[14]>>16)&1);
    b=(m[15]>>16)&1;
    m[14]&=0xffff;
    sel25519(t,m,1-b);
  }
  FOR(i,16) {
    o[2*i]=t[i]&0xff;
    o[2*i+1]=t[i]>>8;
  }
}

static int neq25519(const gf a, const gf b)
{
  u8 c[32],d[32];
  pack25519(c,a);
  pack25519(d,b);
  return crypto_verify_32(c,d);
}

static u8 par25519(const gf a)
{
  u8 d[32];
  pack25519(d,a);
  return d[0]&1;
}

sv unpack25519(gf o, const u8 *n)
{
  int i;
  FOR(i,16) o[i]=n[2*i]+((i64)n[2*i+1]<<8);
  o[15]&=0x7fff;
}

sv A(gf o,const gf a,const gf b)
{
  int i;
  FOR(i,16) o[i]=a[i]+b[i];
}

sv Z(gf o,const gf a,const gf b)
{
  int i;
  FOR(i,16) o[i]=a[i]-b[i];
}

sv M(gf o,const gf a,const gf b)
{
  i64 i,j,t[31];
  FOR(i,31) t[i]=0;
  FOR(i,16) FOR(j,16) t[i+j]+=a[i]*b[j];
  FOR(i,15) t[i]+=38*t[i+16];
  FOR(i,16) o[i]=t[i];
  car25519(o);
  car25519(o);
}

sv S(gf o,const gf a)
{
  M(o,a,a);
}

sv inv25519(gf o,const gf i)
{
  gf c;
  int a;
  FOR(a,16) c[a]=i[a];
  for(a=253;a>=0;a--) {
    S(c,c);
    if(a!=2&&a!=4) M(c,c,i);
  }
  FOR(a,16) o[a]=c[a];
}

sv pow2523(gf o,const gf i)
{
  gf c;
  int a;
  FOR(a,16) c[a]=i[a];
  for(a=250;a>=0;a--) {
    S(c,c);
    if(a!=1) M(c,c,i);
  }
  FOR(a,16) o[a]=c[a];
}

int crypto_scalarmult(u8 *q,const u8 *n,const u8 *p)
{
  u8 z[32];
  i64 x[80],r,i;
  gf a,b,c,d,e,f;
  FOR(i,31) z[i]=n[i];
  z[31]=(n[31]&127)|64;
  z[0]&=248;
  unpack25519(x,p);
  FOR(i,16) {
    b[i]=x[i];
    d[i]=a[i]=c[i]=0;
  }
  a[0]=d[0]=1;
  for(i=254;i>=0;--i) {
    r=(z[i>>3]>>(i&7))&1;
    sel25519(a,b,r);
    sel25519(c,d,r);
    A(e,a,c);
    Z(a,a,c);
    A(c,b,d);
    Z(b,b,d);
    S(d,e);
    S(f,a);
    M(a,c,a);
    M(c,b,e);
    A(e,a,c);
    Z(a,a,c);
    S(b,a);
    Z(c,d,f);
    M(a,c,_121665);
    A(a,a,d);
    M(c,c,a);
    M(a,d,f);
    M(d,b,x);
    S(b,e);
    sel25519(a,b,r);
    sel25519(c,d,r);
  }
  FOR(i,16) {
    x[i+16]=a[i];
    x[i+32]=c[i];
    x[i+48]=b[i];
    x[i+64]=d[i];
  }
  inv25519(x+32,x+32);
  M(x+16,x+16,x+32);
  pack25519(q,x+16);
  return 0;
}

int crypto_scalarmult_base(u8 *q,const u8 *n)
{
  return crypto_scalarmult(q,n,_9);
}

int crypto_box_keypair(u8 *y,u8 *x)
{
  randombytes(x,32);
  return crypto_scalarmult_base(y,x);
}

int crypto_box_beforenm(u8 *k,const u8 *y,const u8 *x)
{
  u8 s[32];
  crypto_scalarmult(s,x,y);
  return crypto_core_hsalsa20(k,_0,s,sigma);
}

int crypto_box_afternm(u8 *c,const u8 *m,u64 d,const u8 *n,const u8 *k)
{
  return crypto_secretbox(c,m,d,n,k);
}

int crypto_box_open_afternm(u8 *m,const u8 *c,u64 d,const u8 *n,const u8 *k)
{
  return crypto_secretbox_open(m,c,d,n,k);
}

int crypto_box(u8 *c,const u8 *m,u64 d,const u8 *n,const u8 *y,const u8 *x)
{
  u8 k[32];
  crypto_box_beforenm(k,y,x);
  return crypto_box_afternm(c,m,d,n,k);
}

int crypto_box_open(u8 *m,const u8 *c,u64 d,const u8 *n,const u8 *y,const u8 *x)
{
  u8 k[32];
  crypto_box_beforenm(k,y,x);
  return crypto_box_open_afternm(m,c,d,n,k);
}

static u64 R(u64 x,int c) { return (x >> c) | (x << (64 - c)); }
static u64 Ch(u64 x,u64 y,u64 z) { return (x & y) ^ (~x & z); }
static u64 Maj(u64 x,u64 y,u64 z) { return (x & y) ^ (x & z) ^ (y & z); }
static u64 Sigma0(u64 x) { return R(x,28) ^ R(x,34) ^ R(x,39); }
static u64 Sigma1(u64 x) { return R(x,14) ^ R(x,18) ^ R(x,41); }
static u64 sigma0(u64 x) { return R(x, 1) ^ R(x, 8) ^ (x >> 7); }
static u64 sigma1(u64 x) { return R(x,19) ^ R(x,61) ^ (x >> 6); }

static const u64 K[80] =
{
  0x428a2f98d728ae22ULL, 0x7137449123ef65cdULL, 0xb5c0fbcfec4d3b2fULL, 0xe9b5dba58189dbbcULL,
  0x3956c25bf348b538ULL, 0x59f111f1b605d019ULL, 0x923f82a4af194f9bULL, 0xab1c5ed5da6d8118ULL,
  0xd807aa98a3030242ULL, 0x12835b0145706fbeULL, 0x243185be4ee4b28cULL, 0x550c7dc3d5ffb4e2ULL,
  0x72be5d74f27b896fULL, 0x80deb1fe3b1696b1ULL, 0x9bdc06a725c71235ULL, 0xc19bf174cf692694ULL,
  0xe49b69c19ef14ad2ULL, 0xefbe4786384f25e3ULL, 0x0fc19dc68b8cd5b5ULL, 0x240ca1cc77ac9c65ULL,
  0x2de92c6f592b0275ULL, 0x4a7484aa6ea6e483ULL, 0x5cb0a9dcbd41fbd4ULL, 0x76f988da831153b5ULL,
  0x983e5152ee66dfabULL, 0xa831c66d2db43210ULL, 0xb00327c898fb213fULL, 0xbf597fc7beef0ee4ULL,
  0xc6e00bf33da88fc2ULL, 0xd5a79147930aa725ULL, 0x06ca6351e003826fULL, 0x142929670a0e6e70ULL,
  0x27b70a8546d22ffcULL, 0x2e1b21385c26c926ULL, 0x4d2c6dfc5ac42aedULL, 0x53380d139d95b3dfULL,
  0x650a73548baf63deULL, 0x766a0abb3c77b2a8ULL, 0x81c2c92e47edaee6ULL, 0x92722c851482353bULL,
  0xa2bfe8a14cf10364ULL, 0xa81a664bbc423001ULL, 0xc24b8b70d0f89791ULL, 0xc76c51a30654be30ULL,
  0xd192e819d6ef5218ULL, 0xd69906245565a910ULL, 0xf40e35855771202aULL, 0x106aa07032bbd1b8ULL,
  0x19a4c116b8d2d0c8ULL, 0x1e376c085141ab53ULL, 0x2748774cdf8eeb99ULL, 0x34b0bcb5e19b48a8ULL,
  0x391c0cb3c5c95a63ULL, 0x4ed8aa4ae3418acbULL, 0x5b9cca4f7763e373ULL, 0x682e6ff3d6b2b8a3ULL,
  0x748f82ee5defb2fcULL, 0x78a5636f43172f60ULL, 0x84c87814a1f0ab72ULL, 0x8cc702081a6439ecULL,
  0x90befffa23631e28ULL, 0xa4506cebde82bde9ULL, 0xbef9a3f7b2c67915ULL, 0xc67178f2e372532bULL,
  0xca273eceea26619cULL, 0xd186b8c721c0c207ULL, 0xeada7dd6cde0eb1eULL, 0xf57d4f7fee6ed178ULL,
  0x06f067aa72176fbaULL, 0x0a637dc5a2c898a6ULL, 0x113f9804bef90daeULL, 0x1b710b35131c471bULL,
  0x28db77f523047d84ULL, 0x32caab7b40c72493ULL, 0x3c9ebe0a15c9bebcULL, 0x431d67c49c100d4cULL,
  0x4cc5d4becb3e42b6ULL, 0x597f299cfc657e2aULL, 0x5fcb6fab3ad6faecULL, 0x6c44198c4a475817ULL
};

int crypto_hashblocks(u8 *x,const u8 *m,u64 n)
{
  u64 z[8],b[8],a[8],w[16],t;
  int i,j;

  FOR(i,8) z[i] = a[i] = dl64(x + 8 * i);

  while (n >= 128) {
    FOR(i,16) w[i] = dl64(m + 8 * i);

    FOR(i,80) {
      FOR(j,8) b[j] = a[j];
      t = a[7] + Sigma1(a[4]) + Ch(a[4],a[5],a[6]) + K[i] + w[i%16];
      b[7] = t + Sigma0(a[0]) + Maj(a[0],a[1],a[2]);
      b[3] += t;
      FOR(j,8) a[(j+1)%8] = b[j];
      if (i%16 == 15)
        FOR(j,16)
          w[j] += w[(j+9)%16] + sigma0(w[(j+1)%16]) + sigma1(w[(j+14)%16]);
    }

    FOR(i,8) { a[i] += z[i]; z[i] = a[i]; }

    m += 128;
    n -= 128;
  }

  FOR(i,8) ts64(x+8*i,z[i]);

  return n;
}

static const u8 iv[64] = {
  0x6a,0x09,0xe6,0x67,0xf3,0xbc,0xc9,0x08,
  0xbb,0x67,0xae,0x85,0x84,0xca,0xa7,0x3b,
  0x3c,0x6e,0xf3,0x72,0xfe,0x94,0xf8,0x2b,
  0xa5,0x4f,0xf5,0x3a,0x5f,0x1d,0x36,0xf1,
  0x51,0x0e,0x52,0x7f,0xad,0xe6,0x82,0xd1,
  0x9b,0x05,0x68,0x8c,0x2b,0x3e,0x6c,0x1f,
  0x1f,0x83,0xd9,0xab,0xfb,0x41,0xbd,0x6b,
  0x5b,0xe0,0xcd,0x19,0x13,0x7e,0x21,0x79
} ;

int crypto_hash(u8 *out,const u8 *m,u64 n)
{
  u8 h[64],x[256];
  u64 i,b = n;

  FOR(i,64) h[i] = iv[i];

  crypto_hashblocks(h,m,n);
  m += n;
  n &= 127;
  m -= n;

  FOR(i,256) x[i] = 0;
  FOR(i,n) x[i] = m[i];
  x[n] = 128;

  n = 256-128*(n<112);
  x[n-9] = b >> 61;
  ts64(x+n-8,b<<3);
  crypto_hashblocks(h,x,n);

  FOR(i,64) out[i] = h[i];

  return 0;
}

sv add(gf p[4],gf q[4])
{
  gf a,b,c,d,t,e,f,g,h;

  Z(a, p[1], p[0]);
  Z(t, q[1], q[0]);
  M(a, a, t);
  A(b, p[0], p[1]);
  A(t, q[0], q[1]);
  M(b, b, t);
  M(c, p[3], q[3]);
  M(c, c, D2);
  M(d, p[2], q[2]);
  A(d, d, d);
  Z(e, b, a);
  Z(f, d, c);
  A(g, d, c);
  A(h, b, a);

  M(p[0], e, f);
  M(p[1], h, g);
  M(p[2], g, f);
  M(p[3], e, h);
}

sv cswap(gf p[4],gf q[4],u8 b)
{
  int i;
  FOR(i,4)
    sel25519(p[i],q[i],b);
}

sv pack(u8 *r,gf p[4])
{
  gf tx, ty, zi;
  inv25519(zi, p[2]);
  M(tx, p[0], zi);
  M(ty, p[1], zi);
  pack25519(r, ty);
  r[31] ^= par25519(tx) << 7;
}

sv scalarmult(gf p[4],gf q[4],const u8 *s)
{
  int i;
  set25519(p[0],gf0);
  set25519(p[1],gf1);
  set25519(p[2],gf1);
  set25519(p[3],gf0);
  for (i = 255;i >= 0;--i) {
    u8 b = (s[i/8]>>(i&7))&1;
    cswap(p,q,b);
    add(q,p);
    add(p,p);
    cswap(p,q,b);
  }
}

sv scalarbase(gf p[4],const u8 *s)
{
  gf q[4];
  set25519(q[0],X);
  set25519(q[1],Y);
  set25519(q[2],gf1);
  M(q[3],X,Y);
  scalarmult(p,q,s);
}

int crypto_sign_keypair(u8 *pk, u8 *sk)
{
  u8 d[64];
  gf p[4];
  int i;

  randombytes(sk, 32);
  crypto_hash(d, sk, 32);
  d[0] &= 248;
  d[31] &= 127;
  d[31] |= 64;

  scalarbase(p,d);
  pack(pk,p);

  FOR(i,32) sk[32 + i] = pk[i];
  return 0;
}

static const u64 L[32] = {0xed, 0xd3, 0xf5, 0x5c, 0x1a, 0x63, 0x12, 0x58, 0xd6, 0x9c, 0xf7, 0xa2, 0xde, 0xf9, 0xde, 0x14, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x10};

sv modL(u8 *r,i64 x[64])
{
  i64 carry,i,j;
  for (i = 63;i >= 32;--i) {
    carry = 0;
    for (j = i - 32;j < i - 12;++j) {
      x[j] += carry - 16 * x[i] * L[j - (i - 32)];
      carry = (x[j] + 128) >> 8;
      x[j] -= carry << 8;
    }
    x[j] += carry;
    x[i] = 0;
  }
  carry = 0;
  FOR(j,32) {
    x[j] += carry - (x[31] >> 4) * L[j];
    carry = x[j] >> 8;
    x[j] &= 255;
  }
  FOR(j,32) x[j] -= carry * L[j];
  FOR(i,32) {
    x[i+1] += x[i] >> 8;
    r[i] = x[i] & 255;
  }
}

sv reduce(u8 *r)
{
  i64 x[64],i;
  FOR(i,64) x[i] = (u64) r[i];
  FOR(i,64) r[i] = 0;
  modL(r,x);
}

int crypto_sign(u8 *sm,u64 *smlen,const u8 *m,u64 n,const u8 *sk)
{
  u8 d[64],h[64],r[64];
  i64 i,j,x[64];
  gf p[4];

  crypto_hash(d, sk, 32);
  d[0] &= 248;
  d[31] &= 127;
  d[31] |= 64;

  *smlen = n+64;
  FOR(i,n) sm[64 + i] = m[i];
  FOR(i,32) sm[32 + i] = d[32 + i];

  crypto_hash(r, sm+32, n+32);
  reduce(r);
  scalarbase(p,r);
  pack(sm,p);

  FOR(i,32) sm[i+32] = sk[i+32];
  crypto_hash(h,sm,n + 64);
  reduce(h);

  FOR(i,64) x[i] = 0;
  FOR(i,32) x[i] = (u64) r[i];
  FOR(i,32) FOR(j,32) x[i+j] += h[i] * (u64) d[j];
  modL(sm + 32,x);

  return 0;
}

static int unpackneg(gf r[4],const u8 p[32])
{
  gf t, chk, num, den, den2, den4, den6;
  set25519(r[2],gf1);
  unpack25519(r[1],p);
  S(num,r[1]);
  M(den,num,D);
  Z(num,num,r[2]);
  A(den,r[2],den);

  S(den2,den);
  S(den4,den2);
  M(den6,den4,den2);
  M(t,den6,num);
  M(t,t,den);

  pow2523(t,t);
  M(t,t,num);
  M(t,t,den);
  M(t,t,den);
  M(r[0],t,den);

  S(chk,r[0]);
  M(chk,chk,den);
  if (neq25519(chk, num)) M(r[0],r[0],I);

  S(chk,r[0]);
  M(chk,chk,den);
  if (neq25519(chk, num)) return -1;

  if (par25519(r[0]) == (p[31]>>7)) Z(r[0],gf0,r[0]);

  M(r[3],r[0],r[1]);
  return 0;
}

int crypto_sign_open(u8 *m,u64 *mlen,const u8 *sm,u64 n,const u8 *pk)
{
  int i;
  u8 t[32],h[64];
  gf p[4],q[4];

  *mlen = -1;
  if (n < 64) return -1;

  if (unpackneg(q,pk)) return -1;

  FOR(i,n) m[i] = sm[i];
  FOR(i,32) m[i+32] = pk[i];
  crypto_hash(h,m,n);
  reduce(h);
  scalarmult(p,q,h);

  scalarbase(q,sm + 32);
  add(p,q);
  pack(t,p);

  n -= 64;
  if (crypto_verify_32(sm, t)) {
    FOR(i,n) m[i] = 0;
    return -1;
  }

  FOR(i,n) m[i] = sm[i + 64];
  *mlen = n;
  return 0;
}


#ifdef ZMQ_HAVE_WINDOWS

#include <windows.h>
#include <wincrypt.h>

#define NCP ((HCRYPTPROV) 0)

HCRYPTPROV hProvider = NCP;

void randombytes(unsigned char *x,unsigned long long xlen)
{
    unsigned i;
    BOOL ret;

    if (hProvider == NCP) {
        for (;;) {
            ret = CryptAcquireContext(&hProvider, NULL, NULL,
                                      PROV_RSA_FULL, CRYPT_VERIFYCONTEXT | CRYPT_SILENT);
            if (ret != FALSE)
                break;
            Sleep (1);
        }
    }
    while (xlen > 0) {
        if (xlen < 1048576)
            i = (unsigned) xlen;
        else
            i = 1048576;

        ret = CryptGenRandom(hProvider, i, x);
        if (ret == FALSE) {
            Sleep(1);
            continue;
        }
        x += i;
        xlen -= i;
    }
}

int randombytes_close(void)
{
    int rc = -1;
    if ((hProvider != NCP) && (CryptReleaseContext(hProvider, 0) != FALSE)) {
        hProvider = NCP;
        rc = 0;
    }
    return rc;
}

int sodium_init (void)
{
    return 0;
}

#else

#include <unistd.h>
#include <assert.h>

#ifdef ZMQ_HAVE_GETRANDOM
#include <sys/random.h>
#else
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

static int fd = -1;
#endif

void randombytes (unsigned char *x,unsigned long long xlen)
{
    int i;
#ifndef ZMQ_HAVE_GETRANDOM
    /*  Require that random_open has already been called, to avoid
        race conditions. */
    assert (fd != -1);
#endif
    while (xlen > 0) {
        if (xlen < 1048576)
            i = xlen;
        else
            i = 1048576;

#ifdef ZMQ_HAVE_GETRANDOM
        i = getrandom (x, i, 0);
#else
        i = read(fd,x,i);
#endif
        if (i < 1) {
            sleep (1);
            continue;
        }
        x += i;
        xlen -= i;
    }
}

/*  Do not call manually! Use random_close from random.hpp */
int randombytes_close (void)
{
    int rc = -1;
#ifndef ZMQ_HAVE_GETRANDOM
    if (fd != -1 && close(fd) == 0) {
        fd = -1;
        rc = 0;
    }
#endif /* ZMQ_HAVE_GETRANDOM */
    return rc;
}

/*  Do not call manually! Use random_open from random.hpp */
int sodium_init (void)
{
#ifndef ZMQ_HAVE_GETRANDOM
    if (fd == -1) {
        for (;;) {
            int flags = O_RDONLY;
#ifdef ZMQ_HAVE_O_CLOEXEC
            flags |= O_CLOEXEC;
#endif
            fd = open ("/dev/urandom", flags);
            if (fd != -1)
                break;
            sleep (1);
        }
#if !defined ZMQ_HAVE_O_CLOEXEC && defined FD_CLOEXEC
        int rc = fcntl (fd, F_SETFD, FD_CLOEXEC);
        assert (rc != -1);
#endif
    }
#endif /* ZMQ_HAVE_GETRANDOM */
    return 0;
}

#endif

#endif
/* clang-format on */

//========= end of tweetnacl.c ============

//========= begin of udp_address.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <string>
#include <sstream>

// ans ignore: #include "macros.hpp"
// ans ignore: #include "udp_address.hpp"
// ans ignore: #include "stdint.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "ip.hpp"

#ifndef ZMQ_HAVE_WINDOWS
#include <sys/types.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <net/if.h>
#include <ctype.h>
#endif

zmq::udp_address_t::udp_address_t() : _bind_interface(-1), _is_multicast(false)
{
    _bind_address   = ip_addr_t::any(AF_INET);
    _target_address = ip_addr_t::any(AF_INET);
}

zmq::udp_address_t::~udp_address_t() {}

int zmq::udp_address_t::resolve(const char *name_, bool bind_, bool ipv6_)
{
    //  No IPv6 support yet
    bool has_interface = false;

    _address = name_;

    //  If we have a semicolon then we should have an interface specifier in the
    //  URL
    const char *src_delimiter = strrchr(name_, ';');
    if (src_delimiter)
    {
        std::string src_name(name_, src_delimiter - name_);

        ip_resolver_options_t src_resolver_opts;

        src_resolver_opts
            .bindable(true)
            //  Restrict hostname/service to literals to avoid any DNS
            //  lookups or service-name irregularity due to
            //  indeterminate socktype.
            .allow_dns(false)
            .allow_nic_name(true)
            .ipv6(ipv6_)
            .expect_port(false);

        ip_resolver_t src_resolver(src_resolver_opts);

        const int rc = src_resolver.resolve(&_bind_address, src_name.c_str());

        if (rc != 0)
        {
            return -1;
        }

        if (_bind_address.is_multicast())
        {
            //  It doesn't make sense to have a multicast address as a source
            errno = EINVAL;
            return -1;
        }

        //  This is a hack because we need the interface index when binding
        //  multicast IPv6, we can't do it by address. Unfortunately for the
        //  time being we don't have a generic platform-independent function to
        //  resolve an interface index from an address, so we only support it
        //  when an actual interface name is provided.
        if (src_name == "*")
        {
            _bind_interface = 0;
        }
        else
        {
            _bind_interface = if_nametoindex(src_name.c_str());
            if (_bind_interface == 0)
            {
                //  Error, probably not an interface name.
                _bind_interface = -1;
            }
        }

        has_interface = true;
        name_         = src_delimiter + 1;
    }

    ip_resolver_options_t resolver_opts;

    resolver_opts.bindable(bind_).allow_dns(!bind_).allow_nic_name(bind_).expect_port(true).ipv6(ipv6_);

    ip_resolver_t resolver(resolver_opts);

    int rc = resolver.resolve(&_target_address, name_);
    if (rc != 0)
    {
        return -1;
    }

    _is_multicast = _target_address.is_multicast();
    uint16_t port = _target_address.port();

    if (has_interface)
    {
        //  If we have an interface specifier then the target address must be a
        //  multicast address
        if (!_is_multicast)
        {
            errno = EINVAL;
            return -1;
        }

        _bind_address.set_port(port);
    }
    else
    {
        //  If we don't have an explicit interface specifier then the URL is
        //  ambiguous: if the target address is multicast then it's the
        //  destination address and the bind address is ANY, if it's unicast
        //  then it's the bind address when 'bind_' is true and the destination
        //  otherwise
        if (_is_multicast || !bind_)
        {
            _bind_address = ip_addr_t::any(_target_address.family());
            _bind_address.set_port(port);
            _bind_interface = 0;
        }
        else
        {
            //  If we were asked for a bind socket and the address
            //  provided was not multicast then it was really meant as
            //  a bind address and the target_address is useless.
            _bind_address = _target_address;
        }
    }

    if (_bind_address.family() != _target_address.family())
    {
        errno = EINVAL;
        return -1;
    }

    //  For IPv6 multicast we *must* have an interface index since we can't
    //  bind by address.
    if (ipv6_ && _is_multicast && _bind_interface < 0)
    {
        errno = ENODEV;
        return -1;
    }

    return 0;
}

int zmq::udp_address_t::family() const { return _bind_address.family(); }

bool zmq::udp_address_t::is_mcast() const { return _is_multicast; }

const zmq::ip_addr_t *zmq::udp_address_t::bind_addr() const { return &_bind_address; }

int zmq::udp_address_t::bind_if() const { return _bind_interface; }

const zmq::ip_addr_t *zmq::udp_address_t::target_addr() const { return &_target_address; }

int zmq::udp_address_t::to_string(std::string &addr_)
{
    // XXX what do (factor TCP code?)
    addr_ = _address;
    return 0;
}

//========= end of udp_address.cpp ============

//========= begin of udp_engine.cpp ============

/*
Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

This file is part of libzmq, the ZeroMQ core engine in C++.

libzmq is free software; you can redistribute it and/or modify it under
the terms of the GNU Lesser General Public License (LGPL) as published
by the Free Software Foundation; either version 3 of the License, or
(at your option) any later version.

As a special exception, the Contributors give you permission to link
this library with independent modules to produce an executable,
regardless of the license terms of these independent modules, and to
copy and distribute the resulting executable under terms of your choice,
provided that you also meet, for each linked independent module, the
terms and conditions of the license of that module. An independent
module is a module which is not derived from or based on this library.
If you modify this library, you must extend this exception to your
version of the library.

libzmq is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"

#if !defined ZMQ_HAVE_WINDOWS
#include <sys/types.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#ifdef ZMQ_HAVE_VXWORKS
#include <sockLib.h>
#endif
#endif

// ans ignore: #include "udp_address.hpp"
// ans ignore: #include "udp_engine.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "ip.hpp"

//  OSX uses a different name for this socket option
#ifndef IPV6_ADD_MEMBERSHIP
#define IPV6_ADD_MEMBERSHIP IPV6_JOIN_GROUP
#endif

#ifdef __APPLE__
#include <TargetConditionals.h>
#endif

zmq::udp_engine_t::udp_engine_t(const options_t &options_)
    : _plugged(false), _fd(-1), _session(NULL), _handle(static_cast<handle_t>(NULL)), _address(NULL),
      _options(options_), _send_enabled(false), _recv_enabled(false)
{
}

zmq::udp_engine_t::~udp_engine_t()
{
    zmq_assert(!_plugged);

    if (_fd != retired_fd)
    {
#ifdef ZMQ_HAVE_WINDOWS
        int rc = closesocket(_fd);
        wsa_assert(rc != SOCKET_ERROR);
#else
        int rc = close(_fd);
        errno_assert(rc == 0);
#endif
        _fd = retired_fd;
    }
}

int zmq::udp_engine_t::init(address_t *address_, bool send_, bool recv_)
{
    zmq_assert(address_);
    zmq_assert(send_ || recv_);
    _send_enabled = send_;
    _recv_enabled = recv_;
    _address      = address_;

    _fd = open_socket(_address->resolved.udp_addr->family(), SOCK_DGRAM, IPPROTO_UDP);
    if (_fd == retired_fd)
        return -1;

    unblock_socket(_fd);

    return 0;
}

void zmq::udp_engine_t::plug(io_thread_t *io_thread_, session_base_t *session_)
{
    zmq_assert(!_plugged);
    _plugged = true;

    zmq_assert(!_session);
    zmq_assert(session_);
    _session = session_;

    //  Connect to I/O threads poller object.
    io_object_t::plug(io_thread_);
    _handle = add_fd(_fd);

    const udp_address_t *const udp_addr = _address->resolved.udp_addr;

    // Bind the socket to a device if applicable
    if (!_options.bound_device.empty())
        bind_to_device(_fd, _options.bound_device);

    if (_send_enabled)
    {
        if (!_options.raw_socket)
        {
            const ip_addr_t *out = udp_addr->target_addr();
            _out_address         = out->as_sockaddr();
            _out_address_len     = out->sockaddr_len();

            if (out->is_multicast())
            {
                int level;
                int optname;

                if (out->family() == AF_INET6)
                {
                    level   = IPPROTO_IPV6;
                    optname = IPV6_MULTICAST_LOOP;
                }
                else
                {
                    level   = IPPROTO_IP;
                    optname = IP_MULTICAST_LOOP;
                }

                int loop = _options.multicast_loop;
                int rc   = setsockopt(_fd, level, optname, reinterpret_cast<char *>(&loop), sizeof(loop));

#ifdef ZMQ_HAVE_WINDOWS
                wsa_assert(rc != SOCKET_ERROR);
#else
                errno_assert(rc == 0);
#endif

                int hops = _options.multicast_hops;

                if (hops > 0)
                {
                    rc = setsockopt(_fd, level, IP_MULTICAST_TTL, reinterpret_cast<char *>(&hops), sizeof(hops));
                }
                else
                {
                    rc = 0;
                }

#ifdef ZMQ_HAVE_WINDOWS
                wsa_assert(rc != SOCKET_ERROR);
#else
                errno_assert(rc == 0);
#endif
                if (out->family() == AF_INET6)
                {
                    int bind_if = udp_addr->bind_if();

                    if (bind_if > 0)
                    {
                        //  If a bind interface is provided we tell the
                        //  kernel to use it to send multicast packets
                        rc = setsockopt(_fd, IPPROTO_IPV6, IPV6_MULTICAST_IF, reinterpret_cast<char *>(&bind_if),
                                        sizeof(bind_if));
                    }
                    else
                    {
                        rc = 0;
                    }
                }
                else
                {
                    struct in_addr bind_addr = udp_addr->bind_addr()->ipv4.sin_addr;

                    if (bind_addr.s_addr != INADDR_ANY)
                    {
                        rc = setsockopt(_fd, IPPROTO_IP, IP_MULTICAST_IF, reinterpret_cast<char *>(&bind_addr),
                                        sizeof(bind_addr));
                    }
                    else
                    {
                        rc = 0;
                    }
                }

#ifdef ZMQ_HAVE_WINDOWS
                wsa_assert(rc != SOCKET_ERROR);
#else
                errno_assert(rc == 0);
#endif
            }
        }
        else
        {
            /// XXX fixme ?
            _out_address     = reinterpret_cast<sockaddr *>(&_raw_address);
            _out_address_len = static_cast<zmq_socklen_t>(sizeof(sockaddr_in));
        }

        set_pollout(_handle);
    }

    if (_recv_enabled)
    {
        int on = 1;
        int rc = setsockopt(_fd, SOL_SOCKET, SO_REUSEADDR, reinterpret_cast<char *>(&on), sizeof(on));
#ifdef ZMQ_HAVE_WINDOWS
        wsa_assert(rc != SOCKET_ERROR);
#else
        errno_assert(rc == 0);
#endif

        const ip_addr_t *bind_addr = udp_addr->bind_addr();
        ip_addr_t        any       = ip_addr_t::any(bind_addr->family());
        const ip_addr_t *real_bind_addr;

        bool multicast = udp_addr->is_mcast();

        if (multicast)
        {
            //  Multicast addresses should be allowed to bind to more than
            //  one port as all ports should receive the message
#ifdef SO_REUSEPORT
            rc = setsockopt(_fd, SOL_SOCKET, SO_REUSEPORT, reinterpret_cast<char *>(&on), sizeof(on));
#ifdef ZMQ_HAVE_WINDOWS
            wsa_assert(rc != SOCKET_ERROR);
#else
            errno_assert(rc == 0);
#endif
#endif

            //  In multicast we should bind ANY and use the mreq struct to
            //  specify the interface
            any.set_port(bind_addr->port());

            real_bind_addr = &any;
        }
        else
        {
            real_bind_addr = bind_addr;
        }

#ifdef ZMQ_HAVE_VXWORKS
        rc = bind(_fd, (sockaddr *)real_bind_addr->as_sockaddr(), real_bind_addr->sockaddr_len());
#else
        rc = bind(_fd, real_bind_addr->as_sockaddr(), real_bind_addr->sockaddr_len());

#endif
#ifdef ZMQ_HAVE_WINDOWS
        wsa_assert(rc != SOCKET_ERROR);
#else
        errno_assert(rc == 0);
#endif

        if (multicast)
        {
            const ip_addr_t *mcast_addr = udp_addr->target_addr();

            if (mcast_addr->family() == AF_INET)
            {
                struct ip_mreq mreq;
                mreq.imr_multiaddr = mcast_addr->ipv4.sin_addr;
                mreq.imr_interface = bind_addr->ipv4.sin_addr;

                rc = setsockopt(_fd, IPPROTO_IP, IP_ADD_MEMBERSHIP, reinterpret_cast<char *>(&mreq), sizeof(mreq));

                errno_assert(rc == 0);
            }
            else if (mcast_addr->family() == AF_INET6)
            {
                struct ipv6_mreq mreq;
                int              iface = _address->resolved.udp_addr->bind_if();

                zmq_assert(iface >= -1);

                mreq.ipv6mr_multiaddr = mcast_addr->ipv6.sin6_addr;
                mreq.ipv6mr_interface = iface;

                rc = setsockopt(_fd, IPPROTO_IPV6, IPV6_ADD_MEMBERSHIP, reinterpret_cast<char *>(&mreq), sizeof(mreq));

                errno_assert(rc == 0);
            }
            else
            {
                //  Shouldn't happen
                abort();
            }

#ifdef ZMQ_HAVE_WINDOWS
            wsa_assert(rc != SOCKET_ERROR);
#else
            errno_assert(rc == 0);
#endif
        }
        set_pollin(_handle);

        //  Call restart output to drop all join/leave commands
        restart_output();
    }
}

void zmq::udp_engine_t::terminate()
{
    zmq_assert(_plugged);
    _plugged = false;

    rm_fd(_handle);

    //  Disconnect from I/O threads poller object.
    io_object_t::unplug();

    delete this;
}

void zmq::udp_engine_t::sockaddr_to_msg(zmq::msg_t *msg_, sockaddr_in *addr_)
{
    const char *const name = inet_ntoa(addr_->sin_addr);

    char      port[6];
    const int port_len = sprintf(port, "%d", static_cast<int>(ntohs(addr_->sin_port)));
    zmq_assert(port_len > 0);

    const size_t name_len = strlen(name);
    const int    size     = static_cast<int>(name_len) + 1 /* colon */
                     + port_len + 1;                       //  terminating NUL
    const int rc = msg_->init_size(size);
    errno_assert(rc == 0);
    msg_->set_flags(msg_t::more);

    //  use memcpy instead of strcpy/strcat, since this is more efficient when
    //  we already know the lengths, which we calculated above
    char *address = static_cast<char *>(msg_->data());
    memcpy(address, name, name_len);
    address += name_len;
    *address++ = ':';
    memcpy(address, port, static_cast<size_t>(port_len));
    address += port_len;
    *address = 0;
}

int zmq::udp_engine_t::resolve_raw_address(char *name_, size_t length_)
{
    memset(&_raw_address, 0, sizeof _raw_address);

    const char *delimiter = NULL;

    // Find delimiter, cannot use memrchr as it is not supported on windows
    if (length_ != 0)
    {
        int   chars_left   = static_cast<int>(length_);
        char *current_char = name_ + length_;
        do
        {
            if (*(--current_char) == ':')
            {
                delimiter = current_char;
                break;
            }
        } while (--chars_left != 0);
    }

    if (!delimiter)
    {
        errno = EINVAL;
        return -1;
    }

    std::string addr_str(name_, delimiter - name_);
    std::string port_str(delimiter + 1, name_ + length_ - delimiter - 1);

    //  Parse the port number (0 is not a valid port).
    uint16_t port = static_cast<uint16_t>(atoi(port_str.c_str()));
    if (port == 0)
    {
        errno = EINVAL;
        return -1;
    }

    _raw_address.sin_family      = AF_INET;
    _raw_address.sin_port        = htons(port);
    _raw_address.sin_addr.s_addr = inet_addr(addr_str.c_str());

    if (_raw_address.sin_addr.s_addr == INADDR_NONE)
    {
        errno = EINVAL;
        return -1;
    }

    return 0;
}

void zmq::udp_engine_t::out_event()
{
    msg_t group_msg;
    int   rc = _session->pull_msg(&group_msg);
    errno_assert(rc == 0 || (rc == -1 && errno == EAGAIN));

    if (rc == 0)
    {
        msg_t body_msg;
        rc = _session->pull_msg(&body_msg);
        //  TODO rc is not checked here. We seem to assume rc == 0. An
        //  assertion should be added.

        const size_t group_size = group_msg.size();
        const size_t body_size  = body_msg.size();
        size_t       size;

        if (_options.raw_socket)
        {
            rc = resolve_raw_address(static_cast<char *>(group_msg.data()), group_size);

            //  We discard the message if address is not valid
            if (rc != 0)
            {
                rc = group_msg.close();
                errno_assert(rc == 0);

                body_msg.close();
                errno_assert(rc == 0);

                return;
            }

            size = body_size;

            memcpy(_out_buffer, body_msg.data(), body_size);
        }
        else
        {
            size = group_size + body_size + 1;

            // TODO: check if larger than maximum size
            _out_buffer[0] = static_cast<unsigned char>(group_size);
            memcpy(_out_buffer + 1, group_msg.data(), group_size);
            memcpy(_out_buffer + 1 + group_size, body_msg.data(), body_size);
        }

        rc = group_msg.close();
        errno_assert(rc == 0);

        body_msg.close();
        errno_assert(rc == 0);

#ifdef ZMQ_HAVE_WINDOWS
        rc = sendto(_fd, _out_buffer, static_cast<int>(size), 0, _out_address, _out_address_len);
        wsa_assert(rc != SOCKET_ERROR);
#elif defined ZMQ_HAVE_VXWORKS
        rc = sendto(_fd, reinterpret_cast<caddr_t>(_out_buffer), size, 0, (sockaddr *)_out_address, _out_address_len);
        errno_assert(rc != -1);
#else
        rc = sendto(_fd, _out_buffer, size, 0, _out_address, _out_address_len);
        errno_assert(rc != -1);
#endif
    }
    else
        reset_pollout(_handle);
}

const zmq::endpoint_uri_pair_t &zmq::udp_engine_t::get_endpoint() const { return _empty_endpoint; }

void zmq::udp_engine_t::restart_output()
{
    //  If we don't support send we just drop all messages
    if (!_send_enabled)
    {
        msg_t msg;
        while (_session->pull_msg(&msg) == 0)
            msg.close();
    }
    else
    {
        set_pollout(_handle);
        out_event();
    }
}

void zmq::udp_engine_t::in_event()
{
    sockaddr_storage in_address;
    zmq_socklen_t    in_addrlen = static_cast<zmq_socklen_t>(sizeof(sockaddr_storage));

    const int nbytes =
        recvfrom(_fd, _in_buffer, MAX_UDP_MSG, 0, reinterpret_cast<sockaddr *>(&in_address), &in_addrlen);
#ifdef ZMQ_HAVE_WINDOWS
    if (nbytes == SOCKET_ERROR)
    {
        const int last_error = WSAGetLastError();
        wsa_assert(last_error == WSAENETDOWN || last_error == WSAENETRESET || last_error == WSAEWOULDBLOCK);
        return;
    }
#else
    if (nbytes == -1)
    {
        errno_assert(errno != EBADF && errno != EFAULT && errno != ENOMEM && errno != ENOTSOCK);
        return;
    }
#endif
    int   rc;
    int   body_size;
    int   body_offset;
    msg_t msg;

    if (_options.raw_socket)
    {
        zmq_assert(in_address.ss_family == AF_INET);
        sockaddr_to_msg(&msg, reinterpret_cast<sockaddr_in *>(&in_address));

        body_size   = nbytes;
        body_offset = 0;
    }
    else
    {
        // TODO in out_event, the group size is an *unsigned* char. what is
        // the maximum value?
        const char *group_buffer = _in_buffer + 1;
        const int   group_size   = _in_buffer[0];

        rc = msg.init_size(group_size);
        errno_assert(rc == 0);
        msg.set_flags(msg_t::more);
        memcpy(msg.data(), group_buffer, group_size);

        //  This doesn't fit, just ingore
        if (nbytes - 1 < group_size)
            return;

        body_size   = nbytes - 1 - group_size;
        body_offset = 1 + group_size;
    }
    // Push group description to session
    rc = _session->push_msg(&msg);
    errno_assert(rc == 0 || (rc == -1 && errno == EAGAIN));

    //  Group description message doesn't fit in the pipe, drop
    if (rc != 0)
    {
        rc = msg.close();
        errno_assert(rc == 0);

        reset_pollin(_handle);
        return;
    }

    rc = msg.close();
    errno_assert(rc == 0);
    rc = msg.init_size(body_size);
    errno_assert(rc == 0);
    memcpy(msg.data(), _in_buffer + body_offset, body_size);

    // Push message body to session
    rc = _session->push_msg(&msg);
    // Message body doesn't fit in the pipe, drop and reset session state
    if (rc != 0)
    {
        rc = msg.close();
        errno_assert(rc == 0);

        _session->reset();
        reset_pollin(_handle);
        return;
    }

    rc = msg.close();
    errno_assert(rc == 0);
    _session->flush();
}

bool zmq::udp_engine_t::restart_input()
{
    if (_recv_enabled)
    {
        set_pollin(_handle);
        in_event();
    }

    return true;
}

//========= end of udp_engine.cpp ============

//========= begin of v1_decoder.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <stdlib.h>
#include <string.h>
#include <limits>
#include <limits.h>

// ans ignore: #include "decoder.hpp"
// ans ignore: #include "v1_decoder.hpp"
// ans ignore: #include "likely.hpp"
// ans ignore: #include "wire.hpp"
// ans ignore: #include "err.hpp"

zmq::v1_decoder_t::v1_decoder_t(size_t bufsize_, int64_t maxmsgsize_)
    : decoder_base_t<v1_decoder_t>(bufsize_), _max_msg_size(maxmsgsize_)
{
    int rc = _in_progress.init();
    errno_assert(rc == 0);

    //  At the beginning, read one byte and go to one_byte_size_ready state.
    next_step(_tmpbuf, 1, &v1_decoder_t::one_byte_size_ready);
}

zmq::v1_decoder_t::~v1_decoder_t()
{
    int rc = _in_progress.close();
    errno_assert(rc == 0);
}

int zmq::v1_decoder_t::one_byte_size_ready(unsigned char const *)
{
    //  First byte of size is read. If it is UCHAR_MAX (0xff) read 8-byte size.
    //  Otherwise allocate the buffer for message data and read the
    //  message data into it.
    if (*_tmpbuf == UCHAR_MAX)
        next_step(_tmpbuf, 8, &v1_decoder_t::eight_byte_size_ready);
    else
    {
        //  There has to be at least one byte (the flags) in the message).
        if (!*_tmpbuf)
        {
            errno = EPROTO;
            return -1;
        }

        if (_max_msg_size >= 0 && static_cast<int64_t>(*_tmpbuf - 1) > _max_msg_size)
        {
            errno = EMSGSIZE;
            return -1;
        }

        int rc = _in_progress.close();
        assert(rc == 0);
        rc = _in_progress.init_size(*_tmpbuf - 1);
        if (rc != 0)
        {
            errno_assert(errno == ENOMEM);
            rc = _in_progress.init();
            errno_assert(rc == 0);
            errno = ENOMEM;
            return -1;
        }

        next_step(_tmpbuf, 1, &v1_decoder_t::flags_ready);
    }
    return 0;
}

int zmq::v1_decoder_t::eight_byte_size_ready(unsigned char const *)
{
    //  8-byte payload length is read. Allocate the buffer
    //  for message body and read the message data into it.
    const uint64_t payload_length = get_uint64(_tmpbuf);

    //  There has to be at least one byte (the flags) in the message).
    if (payload_length == 0)
    {
        errno = EPROTO;
        return -1;
    }

    //  Message size must not exceed the maximum allowed size.
    if (_max_msg_size >= 0 && payload_length - 1 > static_cast<uint64_t>(_max_msg_size))
    {
        errno = EMSGSIZE;
        return -1;
    }

#ifndef __aarch64__
    //  Message size must fit within range of size_t data type.
    if (payload_length - 1 > std::numeric_limits<size_t>::max())
    {
        errno = EMSGSIZE;
        return -1;
    }
#endif

    const size_t msg_size = static_cast<size_t>(payload_length - 1);

    int rc = _in_progress.close();
    assert(rc == 0);
    rc = _in_progress.init_size(msg_size);
    if (rc != 0)
    {
        errno_assert(errno == ENOMEM);
        rc = _in_progress.init();
        errno_assert(rc == 0);
        errno = ENOMEM;
        return -1;
    }

    next_step(_tmpbuf, 1, &v1_decoder_t::flags_ready);
    return 0;
}

int zmq::v1_decoder_t::flags_ready(unsigned char const *)
{
    //  Store the flags from the wire into the message structure.
    _in_progress.set_flags(_tmpbuf[0] & msg_t::more);

    next_step(_in_progress.data(), _in_progress.size(), &v1_decoder_t::message_ready);

    return 0;
}

int zmq::v1_decoder_t::message_ready(unsigned char const *)
{
    //  Message is completely read. Push it further and start reading
    //  new message. (in_progress is a 0-byte message after this point.)
    next_step(_tmpbuf, 1, &v1_decoder_t::one_byte_size_ready);
    return 1;
}

//========= end of v1_decoder.cpp ============

//========= begin of v1_encoder.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "encoder.hpp"
// ans ignore: #include "v1_encoder.hpp"
// ans ignore: #include "msg.hpp"
// ans ignore: #include "wire.hpp"

#include <limits.h>

zmq::v1_encoder_t::v1_encoder_t(size_t bufsize_) : encoder_base_t<v1_encoder_t>(bufsize_)
{
    //  Write 0 bytes to the batch and go to message_ready state.
    next_step(NULL, 0, &v1_encoder_t::message_ready, true);
}

zmq::v1_encoder_t::~v1_encoder_t() {}

void zmq::v1_encoder_t::size_ready()
{
    //  Write message body into the buffer.
    next_step(in_progress()->data(), in_progress()->size(), &v1_encoder_t::message_ready, true);
}

void zmq::v1_encoder_t::message_ready()
{
    //  Get the message size.
    size_t size = in_progress()->size();

    //  Account for the 'flags' byte.
    size++;

    //  For messages less than 255 bytes long, write one byte of message size.
    //  For longer messages write 0xff escape character followed by 8-byte
    //  message size. In both cases 'flags' field follows.
    if (size < UCHAR_MAX)
    {
        _tmpbuf[0] = static_cast<unsigned char>(size);
        _tmpbuf[1] = (in_progress()->flags() & msg_t::more);
        next_step(_tmpbuf, 2, &v1_encoder_t::size_ready, false);
    }
    else
    {
        _tmpbuf[0] = UCHAR_MAX;
        put_uint64(_tmpbuf + 1, size);
        _tmpbuf[9] = (in_progress()->flags() & msg_t::more);
        next_step(_tmpbuf, 10, &v1_encoder_t::size_ready, false);
    }
}

//========= end of v1_encoder.cpp ============

//========= begin of v2_decoder.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <stdlib.h>
#include <string.h>
#include <cmath>

// ans ignore: #include "v2_protocol.hpp"
// ans ignore: #include "v2_decoder.hpp"
// ans ignore: #include "likely.hpp"
// ans ignore: #include "wire.hpp"
// ans ignore: #include "err.hpp"

zmq::v2_decoder_t::v2_decoder_t(size_t bufsize_, int64_t maxmsgsize_, bool zero_copy_)
    : decoder_base_t<v2_decoder_t, shared_message_memory_allocator>(bufsize_), _msg_flags(0), _zero_copy(zero_copy_),
      _max_msg_size(maxmsgsize_)
{
    int rc = _in_progress.init();
    errno_assert(rc == 0);

    //  At the beginning, read one byte and go to flags_ready state.
    next_step(_tmpbuf, 1, &v2_decoder_t::flags_ready);
}

zmq::v2_decoder_t::~v2_decoder_t()
{
    int rc = _in_progress.close();
    errno_assert(rc == 0);
}

int zmq::v2_decoder_t::flags_ready(unsigned char const *)
{
    _msg_flags = 0;
    if (_tmpbuf[0] & v2_protocol_t::more_flag)
        _msg_flags |= msg_t::more;
    if (_tmpbuf[0] & v2_protocol_t::command_flag)
        _msg_flags |= msg_t::command;

    //  The payload length is either one or eight bytes,
    //  depending on whether the 'large' bit is set.
    if (_tmpbuf[0] & v2_protocol_t::large_flag)
        next_step(_tmpbuf, 8, &v2_decoder_t::eight_byte_size_ready);
    else
        next_step(_tmpbuf, 1, &v2_decoder_t::one_byte_size_ready);

    return 0;
}

int zmq::v2_decoder_t::one_byte_size_ready(unsigned char const *read_from_)
{
    return size_ready(_tmpbuf[0], read_from_);
}

int zmq::v2_decoder_t::eight_byte_size_ready(unsigned char const *read_from_)
{
    //  The payload size is encoded as 64-bit unsigned integer.
    //  The most significant byte comes first.
    const uint64_t msg_size = get_uint64(_tmpbuf);

    return size_ready(msg_size, read_from_);
}

int zmq::v2_decoder_t::size_ready(uint64_t msg_size_, unsigned char const *read_pos_)
{
    //  Message size must not exceed the maximum allowed size.
    if (_max_msg_size >= 0)
        if (unlikely(msg_size_ > static_cast<uint64_t>(_max_msg_size)))
        {
            errno = EMSGSIZE;
            return -1;
        }

    //  Message size must fit into size_t data type.
    if (unlikely(msg_size_ != static_cast<size_t>(msg_size_)))
    {
        errno = EMSGSIZE;
        return -1;
    }

    int rc = _in_progress.close();
    assert(rc == 0);

    // the current message can exceed the current buffer. We have to copy the buffer
    // data into a new message and complete it in the next receive.

    shared_message_memory_allocator &allocator = get_allocator();
    if (unlikely(!_zero_copy || msg_size_ > (size_t)(allocator.data() + allocator.size() - read_pos_)))
    {
        // a new message has started, but the size would exceed the pre-allocated arena
        // this happens every time when a message does not fit completely into the buffer
        rc = _in_progress.init_size(static_cast<size_t>(msg_size_));
    }
    else
    {
        // construct message using n bytes from the buffer as storage
        // increase buffer ref count
        // if the message will be a large message, pass a valid refcnt memory location as well
        rc = _in_progress.init(const_cast<unsigned char *>(read_pos_), static_cast<size_t>(msg_size_),
                               shared_message_memory_allocator::call_dec_ref, allocator.buffer(),
                               allocator.provide_content());

        // For small messages, data has been copied and refcount does not have to be increased
        if (_in_progress.is_zcmsg())
        {
            allocator.advance_content();
            allocator.inc_ref();
        }
    }

    if (unlikely(rc))
    {
        errno_assert(errno == ENOMEM);
        rc = _in_progress.init();
        errno_assert(rc == 0);
        errno = ENOMEM;
        return -1;
    }

    _in_progress.set_flags(_msg_flags);
    // this sets read_pos to
    // the message data address if the data needs to be copied
    // for small message / messages exceeding the current buffer
    // or
    // to the current start address in the buffer because the message
    // was constructed to use n bytes from the address passed as argument
    next_step(_in_progress.data(), _in_progress.size(), &v2_decoder_t::message_ready);

    return 0;
}

int zmq::v2_decoder_t::message_ready(unsigned char const *)
{
    //  Message is completely read. Signal this to the caller
    //  and prepare to decode next message.
    next_step(_tmpbuf, 1, &v2_decoder_t::flags_ready);
    return 1;
}

//========= end of v2_decoder.cpp ============

//========= begin of v2_encoder.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
// ans ignore: #include "v2_protocol.hpp"
// ans ignore: #include "v2_encoder.hpp"
// ans ignore: #include "msg.hpp"
// ans ignore: #include "likely.hpp"
// ans ignore: #include "wire.hpp"

#include <limits.h>

zmq::v2_encoder_t::v2_encoder_t(size_t bufsize_) : encoder_base_t<v2_encoder_t>(bufsize_)
{
    //  Write 0 bytes to the batch and go to message_ready state.
    next_step(NULL, 0, &v2_encoder_t::message_ready, true);
}

zmq::v2_encoder_t::~v2_encoder_t() {}

void zmq::v2_encoder_t::message_ready()
{
    //  Encode flags.
    unsigned char &protocol_flags = _tmp_buf[0];
    protocol_flags                = 0;
    if (in_progress()->flags() & msg_t::more)
        protocol_flags |= v2_protocol_t::more_flag;
    if (in_progress()->size() > UCHAR_MAX)
        protocol_flags |= v2_protocol_t::large_flag;
    if (in_progress()->flags() & msg_t::command)
        protocol_flags |= v2_protocol_t::command_flag;

    //  Encode the message length. For messages less then 256 bytes,
    //  the length is encoded as 8-bit unsigned integer. For larger
    //  messages, 64-bit unsigned integer in network byte order is used.
    const size_t size = in_progress()->size();
    if (unlikely(size > UCHAR_MAX))
    {
        put_uint64(_tmp_buf + 1, size);
        next_step(_tmp_buf, 9, &v2_encoder_t::size_ready, false);
    }
    else
    {
        _tmp_buf[1] = static_cast<uint8_t>(size);
        next_step(_tmp_buf, 2, &v2_encoder_t::size_ready, false);
    }
}

void zmq::v2_encoder_t::size_ready()
{
    //  Write message body into the buffer.
    next_step(in_progress()->data(), in_progress()->size(), &v2_encoder_t::message_ready, true);
}

//========= end of v2_encoder.cpp ============

//========= begin of vmci.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
// ans ignore: #include "precompiled.hpp"

// ans ignore: #include "vmci.hpp"

#if defined ZMQ_HAVE_VMCI

#include <cassert>
#include <vmci_sockets.h>

void zmq::tune_vmci_buffer_size(ctx_t *context_, fd_t sockfd_, uint64_t default_size_, uint64_t min_size_,
                                uint64_t max_size_)
{
    int family = context_->get_vmci_socket_family();
    assert(family != -1);

    if (default_size_ != 0)
    {
        int rc = setsockopt(sockfd_, family, SO_VMCI_BUFFER_SIZE, (char *)&default_size_, sizeof default_size_);
#if defined ZMQ_HAVE_WINDOWS
        wsa_assert(rc != SOCKET_ERROR);
#else
        errno_assert(rc == 0);
#endif
    }

    if (min_size_ != 0)
    {
        int rc = setsockopt(sockfd_, family, SO_VMCI_BUFFER_SIZE, (char *)&min_size_, sizeof min_size_);
#if defined ZMQ_HAVE_WINDOWS
        wsa_assert(rc != SOCKET_ERROR);
#else
        errno_assert(rc == 0);
#endif
    }

    if (max_size_ != 0)
    {
        int rc = setsockopt(sockfd_, family, SO_VMCI_BUFFER_SIZE, (char *)&max_size_, sizeof max_size_);
#if defined ZMQ_HAVE_WINDOWS
        wsa_assert(rc != SOCKET_ERROR);
#else
        errno_assert(rc == 0);
#endif
    }
}

#if defined ZMQ_HAVE_WINDOWS
void        zmq::tune_vmci_connect_timeout(ctx_t *context_, fd_t sockfd_, DWORD timeout_)
#else
void zmq::tune_vmci_connect_timeout(ctx_t *context_, fd_t sockfd_, struct timeval timeout_)
#endif
{
    int family = context_->get_vmci_socket_family();
    assert(family != -1);

    int rc = setsockopt(sockfd_, family, SO_VMCI_CONNECT_TIMEOUT, (char *)&timeout_, sizeof timeout_);
#if defined ZMQ_HAVE_WINDOWS
    wsa_assert(rc != SOCKET_ERROR);
#else
    errno_assert(rc == 0);
#endif
}

#endif

//========= end of vmci.cpp ============

//========= begin of vmci_address.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"

// ans ignore: #include "vmci_address.hpp"

#if defined(ZMQ_HAVE_VMCI)

#include <climits>
#include <string>
#include <sstream>

// ans ignore: #include "err.hpp"

zmq::vmci_address_t::vmci_address_t(ctx_t *parent_) : parent(parent_) { memset(&address, 0, sizeof address); }

zmq::vmci_address_t::vmci_address_t(const sockaddr *sa, socklen_t sa_len, ctx_t *parent_) : parent(parent_)
{
    zmq_assert(sa && sa_len > 0);

    memset(&address, 0, sizeof address);
    if (sa->sa_family == parent->get_vmci_socket_family())
        memcpy(&address, sa, sa_len);
}

zmq::vmci_address_t::~vmci_address_t() {}

int zmq::vmci_address_t::resolve(const char *path_)
{
    //  Find the ':' at end that separates address from the port number.
    const char *delimiter = strrchr(path_, ':');
    if (!delimiter)
    {
        errno = EINVAL;
        return -1;
    }

    //  Separate the address/port.
    std::string addr_str(path_, delimiter - path_);
    std::string port_str(delimiter + 1);

    unsigned int cid  = VMADDR_CID_ANY;
    unsigned int port = VMADDR_PORT_ANY;

    if (!addr_str.length())
    {
        errno = EINVAL;
        return -1;
    }
    else if (addr_str == "@")
    {
        cid = VMCISock_GetLocalCID();

        if (cid == VMADDR_CID_ANY)
        {
            errno = ENODEV;
            return -1;
        }
    }
    else if (addr_str != "*" && addr_str != "-1")
    {
        const char *  begin = addr_str.c_str();
        char *        end   = NULL;
        unsigned long l     = strtoul(begin, &end, 10);

        if ((l == 0 && end == begin) || (l == ULONG_MAX && errno == ERANGE) || l > UINT_MAX)
        {
            errno = EINVAL;
            return -1;
        }

        cid = static_cast<unsigned int>(l);
    }

    if (!port_str.length())
    {
        errno = EINVAL;
        return -1;
    }
    else if (port_str != "*" && port_str != "-1")
    {
        const char *  begin = port_str.c_str();
        char *        end   = NULL;
        unsigned long l     = strtoul(begin, &end, 10);

        if ((l == 0 && end == begin) || (l == ULONG_MAX && errno == ERANGE) || l > UINT_MAX)
        {
            errno = EINVAL;
            return -1;
        }

        port = static_cast<unsigned int>(l);
    }

    address.svm_family = static_cast<sa_family_t>(parent->get_vmci_socket_family());
    address.svm_cid    = cid;
    address.svm_port   = port;

    return 0;
}

int zmq::vmci_address_t::to_string(std::string &addr_)
{
    if (address.svm_family != parent->get_vmci_socket_family())
    {
        addr_.clear();
        return -1;
    }

    std::stringstream s;

    s << "vmci://";

    if (address.svm_cid == VMADDR_CID_ANY)
    {
        s << "*";
    }
    else
    {
        s << address.svm_cid;
    }

    s << ":";

    if (address.svm_port == VMADDR_PORT_ANY)
    {
        s << "*";
    }
    else
    {
        s << address.svm_port;
    }

    addr_ = s.str();
    return 0;
}

const sockaddr *zmq::vmci_address_t::addr() const { return reinterpret_cast<const sockaddr *>(&address); }

socklen_t zmq::vmci_address_t::addrlen() const { return static_cast<socklen_t>(sizeof address); }

#endif

//========= end of vmci_address.cpp ============

//========= begin of vmci_connecter.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"

// ans ignore: #include "vmci_connecter.hpp"

#if defined ZMQ_HAVE_VMCI

#include <new>

// ans ignore: #include "stream_engine.hpp"
// ans ignore: #include "io_thread.hpp"
// ans ignore: #include "platform.hpp"
// ans ignore: #include "random.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "ip.hpp"
// ans ignore: #include "address.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "vmci_address.hpp"
// ans ignore: #include "vmci.hpp"

zmq::vmci_connecter_t::vmci_connecter_t(class io_thread_t *io_thread_, class session_base_t *session_,
                                        const options_t &options_, const address_t *addr_, bool delayed_start_)
    : own_t(io_thread_, options_), io_object_t(io_thread_), addr(addr_), s(retired_fd), handle_valid(false),
      delayed_start(delayed_start_), timer_started(false), session(session_),
      current_reconnect_ivl(options.reconnect_ivl)
{
    zmq_assert(addr);
    zmq_assert(addr->protocol == "vmci");
    addr->to_string(endpoint);
    socket = session->get_socket();
}

zmq::vmci_connecter_t::~vmci_connecter_t()
{
    zmq_assert(!timer_started);
    zmq_assert(!handle_valid);
    zmq_assert(s == retired_fd);
}

void zmq::vmci_connecter_t::process_plug()
{
    if (delayed_start)
        add_reconnect_timer();
    else
        start_connecting();
}

void zmq::vmci_connecter_t::process_term(int linger_)
{
    if (timer_started)
    {
        cancel_timer(reconnect_timer_id);
        timer_started = false;
    }

    if (handle_valid)
    {
        rm_fd(handle);
        handle_valid = false;
    }

    if (s != retired_fd)
        close();

    own_t::process_term(linger_);
}

void zmq::vmci_connecter_t::in_event()
{
    //  We are not polling for incoming data, so we are actually called
    //  because of error here. However, we can get error on out event as well
    //  on some platforms, so we'll simply handle both events in the same way.
    out_event();
}

void zmq::vmci_connecter_t::out_event()
{
    fd_t fd = connect();
    rm_fd(handle);
    handle_valid = false;

    //  Handle the error condition by attempt to reconnect.
    if (fd == retired_fd)
    {
        close();
        add_reconnect_timer();
        return;
    }

    tune_vmci_buffer_size(this->get_ctx(), fd, options.vmci_buffer_size, options.vmci_buffer_min_size,
                          options.vmci_buffer_max_size);

    if (options.vmci_connect_timeout > 0)
    {
#if defined ZMQ_HAVE_WINDOWS
        tune_vmci_connect_timeout(this->get_ctx(), fd, options.vmci_connect_timeout);
#else
        struct timeval timeout = {0, options.vmci_connect_timeout * 1000};
        tune_vmci_connect_timeout(this->get_ctx(), fd, timeout);
#endif
    }

    //  Create the engine object for this connection.
    stream_engine_t *engine =
        new (std::nothrow) stream_engine_t(fd, options, make_unconnected_bind_endpoint_pair(endpoint));
    alloc_assert(engine);

    //  Attach the engine to the corresponding session object.
    send_attach(session, engine);

    //  Shut the connecter down.
    terminate();

    socket->event_connected(make_unconnected_bind_endpoint_pair(endpoint), fd);
}

void zmq::vmci_connecter_t::timer_event(int id_)
{
    zmq_assert(id_ == reconnect_timer_id);
    timer_started = false;
    start_connecting();
}

void zmq::vmci_connecter_t::start_connecting()
{
    //  Open the connecting socket.
    int rc = open();

    //  Connect may succeed in synchronous manner.
    if (rc == 0)
    {
        handle       = add_fd(s);
        handle_valid = true;
        out_event();
    }

    //  Handle any other error condition by eventual reconnect.
    else
    {
        if (s != retired_fd)
            close();
        add_reconnect_timer();
    }
}

void zmq::vmci_connecter_t::add_reconnect_timer()
{
    if (options.reconnect_ivl != -1)
    {
        int rc_ivl = get_new_reconnect_ivl();
        add_timer(rc_ivl, reconnect_timer_id);
        socket->event_connect_retried(make_unconnected_bind_endpoint_pair(endpoint), rc_ivl);
        timer_started = true;
    }
}

int zmq::vmci_connecter_t::get_new_reconnect_ivl()
{
    //  The new interval is the current interval + random value.
    int this_interval = current_reconnect_ivl + (generate_random() % options.reconnect_ivl);

    //  Only change the current reconnect interval  if the maximum reconnect
    //  interval was set and if it's larger than the reconnect interval.
    if (options.reconnect_ivl_max > 0 && options.reconnect_ivl_max > options.reconnect_ivl)
    {
        //  Calculate the next interval
        current_reconnect_ivl = current_reconnect_ivl * 2;
        if (current_reconnect_ivl >= options.reconnect_ivl_max)
        {
            current_reconnect_ivl = options.reconnect_ivl_max;
        }
    }
    return this_interval;
}

int zmq::vmci_connecter_t::open()
{
    zmq_assert(s == retired_fd);

    int family = this->get_ctx()->get_vmci_socket_family();
    if (family == -1)
        return -1;

    //  Create the socket.
    s = open_socket(family, SOCK_STREAM, 0);
#ifdef ZMQ_HAVE_WINDOWS
    if (s == INVALID_SOCKET)
    {
        errno = wsa_error_to_errno(WSAGetLastError());
        return -1;
    }
#else
    if (s == -1)
        return -1;
#endif

    //  Connect to the remote peer.
    int rc = ::connect(s, addr->resolved.vmci_addr->addr(), addr->resolved.vmci_addr->addrlen());

    //  Connect was successful immediately.
    if (rc == 0)
        return 0;

    //  Forward the error.
    return -1;
}

void zmq::vmci_connecter_t::close()
{
    zmq_assert(s != retired_fd);
#ifdef ZMQ_HAVE_WINDOWS
    const int rc = closesocket(s);
    wsa_assert(rc != SOCKET_ERROR);
#else
    const int rc = ::close(s);
    errno_assert(rc == 0);
#endif
    socket->event_closed(make_unconnected_bind_endpoint_pair(endpoint), s);
    s = retired_fd;
}

zmq::fd_t zmq::vmci_connecter_t::connect()
{
    //  Following code should handle both Berkeley-derived socket
    //  implementations and Solaris.
    int err = 0;
#if defined ZMQ_HAVE_HPUX
    int len = sizeof(err);
#else
    socklen_t len = sizeof(err);
#endif
    int rc = getsockopt(s, SOL_SOCKET, SO_ERROR, (char *)&err, &len);

    //  Assert if the error was caused by 0MQ bug.
    //  Networking problems are OK. No need to assert.
#ifdef ZMQ_HAVE_WINDOWS
    zmq_assert(rc == 0);
    if (err != 0)
    {
        if (err != WSAECONNREFUSED && err != WSAETIMEDOUT && err != WSAECONNABORTED && err != WSAEHOSTUNREACH &&
            err != WSAENETUNREACH && err != WSAENETDOWN && err != WSAEACCES && err != WSAEINVAL &&
            err != WSAEADDRINUSE && err != WSAECONNRESET)
        {
            wsa_assert_no(err);
        }
        return retired_fd;
    }
#else
    //  Following code should handle both Berkeley-derived socket
    //  implementations and Solaris.
    if (rc == -1)
        err = errno;
    if (err != 0)
    {
        errno = err;
        errno_assert(errno == ECONNREFUSED || errno == ECONNRESET || errno == ETIMEDOUT || errno == EHOSTUNREACH ||
                     errno == ENETUNREACH || errno == ENETDOWN || errno == EINVAL);
        return retired_fd;
    }
#endif

    fd_t result = s;
    s           = retired_fd;
    return result;
}

#endif

//========= end of vmci_connecter.cpp ============

//========= begin of vmci_listener.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"

// ans ignore: #include "vmci_listener.hpp"

#if defined ZMQ_HAVE_VMCI

#include <new>

// ans ignore: #include "stream_engine.hpp"
// ans ignore: #include "vmci_address.hpp"
// ans ignore: #include "io_thread.hpp"
// ans ignore: #include "session_base.hpp"
// ans ignore: #include "config.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "ip.hpp"
// ans ignore: #include "socket_base.hpp"
// ans ignore: #include "vmci.hpp"

#if defined ZMQ_HAVE_WINDOWS
// ans ignore: #include "windows.hpp"
#else
#include <unistd.h>
#include <fcntl.h>
#endif

zmq::vmci_listener_t::vmci_listener_t(io_thread_t *io_thread_, socket_base_t *socket_, const options_t &options_)
    : own_t(io_thread_, options_), io_object_t(io_thread_), s(retired_fd), socket(socket_)
{
}

zmq::vmci_listener_t::~vmci_listener_t() { zmq_assert(s == retired_fd); }

void zmq::vmci_listener_t::process_plug()
{
    //  Start polling for incoming connections.
    handle = add_fd(s);
    set_pollin(handle);
}

void zmq::vmci_listener_t::process_term(int linger_)
{
    rm_fd(handle);
    close();
    own_t::process_term(linger_);
}

void zmq::vmci_listener_t::in_event()
{
    fd_t fd = accept();

    //  If connection was reset by the peer in the meantime, just ignore it.
    if (fd == retired_fd)
    {
        socket->event_accept_failed(make_unconnected_bind_endpoint_pair(endpoint), zmq_errno());
        return;
    }

    tune_vmci_buffer_size(this->get_ctx(), fd, options.vmci_buffer_size, options.vmci_buffer_min_size,
                          options.vmci_buffer_max_size);

    if (options.vmci_connect_timeout > 0)
    {
#if defined ZMQ_HAVE_WINDOWS
        tune_vmci_connect_timeout(this->get_ctx(), fd, options.vmci_connect_timeout);
#else
        struct timeval timeout = {0, options.vmci_connect_timeout * 1000};
        tune_vmci_connect_timeout(this->get_ctx(), fd, timeout);
#endif
    }

    //  Create the engine object for this connection.
    stream_engine_t *engine =
        new (std::nothrow) stream_engine_t(fd, options, make_unconnected_bind_endpoint_pair(endpoint));
    alloc_assert(engine);

    //  Choose I/O thread to run connecter in. Given that we are already
    //  running in an I/O thread, there must be at least one available.
    io_thread_t *io_thread = choose_io_thread(options.affinity);
    zmq_assert(io_thread);

    //  Create and launch a session object.
    session_base_t *session = session_base_t::create(io_thread, false, socket, options, NULL);
    errno_assert(session);
    session->inc_seqnum();
    launch_child(session);
    send_attach(session, engine, false);
    socket->event_accepted(make_unconnected_bind_endpoint_pair(endpoint), fd);
}

int zmq::vmci_listener_t::get_local_address(std::string &addr_)
{
    struct sockaddr_storage ss;
#ifdef ZMQ_HAVE_HPUX
    int sl = sizeof(ss);
#else
    socklen_t sl = sizeof(ss);
#endif
    int rc = getsockname(s, (sockaddr *)&ss, &sl);
    if (rc != 0)
    {
        addr_.clear();
        return rc;
    }

    vmci_address_t addr((struct sockaddr *)&ss, sl, this->get_ctx());
    return addr.to_string(addr_);
}

int zmq::vmci_listener_t::set_local_address(const char *addr_)
{
    //  Create addr on stack for auto-cleanup
    std::string addr(addr_);

    //  Initialise the address structure.
    vmci_address_t address(this->get_ctx());
    int            rc = address.resolve(addr.c_str());
    if (rc != 0)
        return -1;

    //  Create a listening socket.
    s = open_socket(this->get_ctx()->get_vmci_socket_family(), SOCK_STREAM, 0);
#ifdef ZMQ_HAVE_WINDOWS
    if (s == INVALID_SOCKET)
    {
        errno = wsa_error_to_errno(WSAGetLastError());
        return -1;
    }
#if !defined _WIN32_WCE
    //  On Windows, preventing sockets to be inherited by child processes.
    BOOL brc = SetHandleInformation((HANDLE)s, HANDLE_FLAG_INHERIT, 0);
    win_assert(brc);
#endif
#else
    if (s == -1)
        return -1;
#endif

    address.to_string(endpoint);

    //  Bind the socket.
    rc = bind(s, address.addr(), address.addrlen());
#ifdef ZMQ_HAVE_WINDOWS
    if (rc == SOCKET_ERROR)
    {
        errno = wsa_error_to_errno(WSAGetLastError());
        goto error;
    }
#else
    if (rc != 0)
        goto error;
#endif

    //  Listen for incoming connections.
    rc = listen(s, options.backlog);
#ifdef ZMQ_HAVE_WINDOWS
    if (rc == SOCKET_ERROR)
    {
        errno = wsa_error_to_errno(WSAGetLastError());
        goto error;
    }
#else
    if (rc != 0)
        goto error;
#endif

    socket->event_listening(make_unconnected_bind_endpoint_pair(endpoint), s);
    return 0;

error:
    int err = errno;
    close();
    errno = err;
    return -1;
}

void zmq::vmci_listener_t::close()
{
    zmq_assert(s != retired_fd);
#ifdef ZMQ_HAVE_WINDOWS
    int rc = closesocket(s);
    wsa_assert(rc != SOCKET_ERROR);
#else
    int rc = ::close(s);
    errno_assert(rc == 0);
#endif
    socket->event_closed(make_unconnected_bind_endpoint_pair(endpoint), s);
    s = retired_fd;
}

zmq::fd_t zmq::vmci_listener_t::accept()
{
    //  Accept one connection and deal with different failure modes.
    //  The situation where connection cannot be accepted due to insufficient
    //  resources is considered valid and treated by ignoring the connection.
    zmq_assert(s != retired_fd);
    fd_t sock = ::accept(s, NULL, NULL);

#ifdef ZMQ_HAVE_WINDOWS
    if (sock == INVALID_SOCKET)
    {
        wsa_assert(WSAGetLastError() == WSAEWOULDBLOCK || WSAGetLastError() == WSAECONNRESET ||
                   WSAGetLastError() == WSAEMFILE || WSAGetLastError() == WSAENOBUFS);
        return retired_fd;
    }
#if !defined _WIN32_WCE
    //  On Windows, preventing sockets to be inherited by child processes.
    BOOL brc = SetHandleInformation((HANDLE)sock, HANDLE_FLAG_INHERIT, 0);
    win_assert(brc);
#endif
#else
    if (sock == -1)
    {
        errno_assert(errno == EAGAIN || errno == EWOULDBLOCK || errno == EINTR || errno == ECONNABORTED ||
                     errno == EPROTO || errno == ENOBUFS || errno == ENOMEM || errno == EMFILE || errno == ENFILE);
        return retired_fd;
    }
#endif

    //  Race condition can cause socket not to be closed (if fork happens
    //  between accept and this point).
#ifdef FD_CLOEXEC
    int rc = fcntl(sock, F_SETFD, FD_CLOEXEC);
    errno_assert(rc != -1);
#endif

    return sock;
}

#endif

//========= end of vmci_listener.cpp ============

//========= begin of xpub.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <string.h>

// ans ignore: #include "xpub.hpp"
// ans ignore: #include "pipe.hpp"
// ans ignore: #include "err.hpp"
// ans ignore: #include "msg.hpp"
// ans ignore: #include "macros.hpp"
// ans ignore: #include "generic_mtrie_impl.hpp"

zmq::xpub_t::xpub_t(class ctx_t *parent_, uint32_t tid_, int sid_)
    : socket_base_t(parent_, tid_, sid_), _verbose_subs(false), _verbose_unsubs(false), _more(false), _lossy(true),
      _manual(false), _send_last_pipe(false), _pending_pipes(), _welcome_msg()
{
    _last_pipe   = NULL;
    options.type = ZMQ_XPUB;
    _welcome_msg.init();
}

zmq::xpub_t::~xpub_t() { _welcome_msg.close(); }

void zmq::xpub_t::xattach_pipe(pipe_t *pipe_, bool subscribe_to_all_, bool locally_initiated_)
{
    LIBZMQ_UNUSED(locally_initiated_);

    zmq_assert(pipe_);
    _dist.attach(pipe_);

    //  If subscribe_to_all_ is specified, the caller would like to subscribe
    //  to all data on this pipe, implicitly.
    if (subscribe_to_all_)
        _subscriptions.add(NULL, 0, pipe_);

    // if welcome message exists, send a copy of it
    if (_welcome_msg.size() > 0)
    {
        msg_t copy;
        copy.init();
        int rc = copy.copy(_welcome_msg);
        errno_assert(rc == 0);
        bool ok = pipe_->write(&copy);
        zmq_assert(ok);
        pipe_->flush();
    }

    //  The pipe is active when attached. Let's read the subscriptions from
    //  it, if any.
    xread_activated(pipe_);
}

void zmq::xpub_t::xread_activated(pipe_t *pipe_)
{
    //  There are some subscriptions waiting. Let's process them.
    msg_t sub;
    while (pipe_->read(&sub))
    {
        metadata_t *   metadata = sub.metadata();
        unsigned char *msg_data = static_cast<unsigned char *>(sub.data()), *data = NULL;
        size_t         size      = 0;
        bool           subscribe = false;

        //  Apply the subscription to the trie
        if (sub.is_subscribe() || sub.is_cancel())
        {
            data      = static_cast<unsigned char *>(sub.command_body());
            size      = sub.command_body_size();
            subscribe = sub.is_subscribe();
        }
        else if (sub.size() > 0)
        {
            unsigned char first = *msg_data;
            if (first == 0 || first == 1)
            {
                data      = msg_data + 1;
                size      = sub.size() - 1;
                subscribe = first == 1;
            }
        }
        else
        {
            //  Process user message coming upstream from xsub socket
            _pending_data.push_back(blob_t(msg_data, sub.size()));
            if (metadata)
                metadata->add_ref();
            _pending_metadata.push_back(metadata);
            _pending_flags.push_back(sub.flags());
            sub.close();
            continue;
        }

        if (_manual)
        {
            // Store manual subscription to use on termination
            if (!subscribe)
                _manual_subscriptions.rm(data, size, pipe_);
            else
                _manual_subscriptions.add(data, size, pipe_);

            _pending_pipes.push_back(pipe_);

            //  ZMTP 3.1 hack: we need to support sub/cancel commands, but
            //  we can't give them back to userspace as it would be an API
            //  breakage since the payload of the message is completely
            //  different. Manually craft an old-style message instead.
            data = data - 1;
            size = size + 1;
            if (subscribe)
                *data = 1;
            else
                *data = 0;

            _pending_data.push_back(blob_t(data, size));
            if (metadata)
                metadata->add_ref();
            _pending_metadata.push_back(metadata);
            _pending_flags.push_back(0);
        }
        else
        {
            bool notify;
            if (!subscribe)
            {
                mtrie_t::rm_result rm_result = _subscriptions.rm(data, size, pipe_);
                //  TODO reconsider what to do if rm_result == mtrie_t::not_found
                notify = rm_result != mtrie_t::values_remain || _verbose_unsubs;
            }
            else
            {
                bool first_added = _subscriptions.add(data, size, pipe_);
                notify           = first_added || _verbose_subs;
            }

            //  If the request was a new subscription, or the subscription
            //  was removed, or verbose mode is enabled, store it so that
            //  it can be passed to the user on next recv call.
            if (options.type == ZMQ_XPUB && notify)
            {
                data = data - 1;
                size = size + 1;
                if (subscribe)
                    *data = 1;
                else
                    *data = 0;

                _pending_data.push_back(blob_t(data, size));
                if (metadata)
                    metadata->add_ref();
                _pending_metadata.push_back(metadata);
                _pending_flags.push_back(0);
            }
        }
        sub.close();
    }
}

void zmq::xpub_t::xwrite_activated(pipe_t *pipe_) { _dist.activated(pipe_); }

int zmq::xpub_t::xsetsockopt(int option_, const void *optval_, size_t optvallen_)
{
    if (option_ == ZMQ_XPUB_VERBOSE || option_ == ZMQ_XPUB_VERBOSER || option_ == ZMQ_XPUB_MANUAL_LAST_VALUE ||
        option_ == ZMQ_XPUB_NODROP || option_ == ZMQ_XPUB_MANUAL)
    {
        if (optvallen_ != sizeof(int) || *static_cast<const int *>(optval_) < 0)
        {
            errno = EINVAL;
            return -1;
        }
        if (option_ == ZMQ_XPUB_VERBOSE)
        {
            _verbose_subs   = (*static_cast<const int *>(optval_) != 0);
            _verbose_unsubs = false;
        }
        else if (option_ == ZMQ_XPUB_VERBOSER)
        {
            _verbose_subs   = (*static_cast<const int *>(optval_) != 0);
            _verbose_unsubs = _verbose_subs;
        }
        else if (option_ == ZMQ_XPUB_MANUAL_LAST_VALUE)
        {
            _manual         = (*static_cast<const int *>(optval_) != 0);
            _send_last_pipe = _manual;
        }
        else if (option_ == ZMQ_XPUB_NODROP)
            _lossy = (*static_cast<const int *>(optval_) == 0);
        else if (option_ == ZMQ_XPUB_MANUAL)
            _manual = (*static_cast<const int *>(optval_) != 0);
    }
    else if (option_ == ZMQ_SUBSCRIBE && _manual)
    {
        if (_last_pipe != NULL)
            _subscriptions.add((unsigned char *)optval_, optvallen_, _last_pipe);
    }
    else if (option_ == ZMQ_UNSUBSCRIBE && _manual)
    {
        if (_last_pipe != NULL)
            _subscriptions.rm((unsigned char *)optval_, optvallen_, _last_pipe);
    }
    else if (option_ == ZMQ_XPUB_WELCOME_MSG)
    {
        _welcome_msg.close();

        if (optvallen_ > 0)
        {
            int rc = _welcome_msg.init_size(optvallen_);
            errno_assert(rc == 0);

            unsigned char *data = static_cast<unsigned char *>(_welcome_msg.data());
            memcpy(data, optval_, optvallen_);
        }
        else
            _welcome_msg.init();
    }
    else
    {
        errno = EINVAL;
        return -1;
    }
    return 0;
}

static void stub(zmq::mtrie_t::prefix_t data_, size_t size_, void *arg_)
{
    LIBZMQ_UNUSED(data_);
    LIBZMQ_UNUSED(size_);
    LIBZMQ_UNUSED(arg_);
}

void zmq::xpub_t::xpipe_terminated(pipe_t *pipe_)
{
    if (_manual)
    {
        //  Remove the pipe from the trie and send corresponding manual
        //  unsubscriptions upstream.
        _manual_subscriptions.rm(pipe_, send_unsubscription, this, false);
        //  Remove pipe without actually sending the message as it was taken
        //  care of by the manual call above. subscriptions is the real mtrie,
        //  so the pipe must be removed from there or it will be left over.
        _subscriptions.rm(pipe_, stub, (void *)NULL, false);
    }
    else
    {
        //  Remove the pipe from the trie. If there are topics that nobody
        //  is interested in anymore, send corresponding unsubscriptions
        //  upstream.
        _subscriptions.rm(pipe_, send_unsubscription, this, !_verbose_unsubs);
    }

    _dist.pipe_terminated(pipe_);
}

void zmq::xpub_t::mark_as_matching(pipe_t *pipe_, xpub_t *self_) { self_->_dist.match(pipe_); }

void zmq::xpub_t::mark_last_pipe_as_matching(pipe_t *pipe_, xpub_t *self_)
{
    if (self_->_last_pipe == pipe_)
        self_->_dist.match(pipe_);
}

int zmq::xpub_t::xsend(msg_t *msg_)
{
    bool msg_more = (msg_->flags() & msg_t::more) != 0;

    //  For the first part of multi-part message, find the matching pipes.
    if (!_more)
    {
        if (unlikely(_manual && _last_pipe && _send_last_pipe))
        {
            _subscriptions.match(static_cast<unsigned char *>(msg_->data()), msg_->size(), mark_last_pipe_as_matching,
                                 this);
            _last_pipe = NULL;
        }
        else
            _subscriptions.match(static_cast<unsigned char *>(msg_->data()), msg_->size(), mark_as_matching, this);
        // If inverted matching is used, reverse the selection now
        if (options.invert_matching)
        {
            _dist.reverse_match();
        }
    }

    int rc = -1; //  Assume we fail
    if (_lossy || _dist.check_hwm())
    {
        if (_dist.send_to_matching(msg_) == 0)
        {
            //  If we are at the end of multi-part message we can mark
            //  all the pipes as non-matching.
            if (!msg_more)
                _dist.unmatch();
            _more = msg_more;
            rc    = 0; //  Yay, sent successfully
        }
    }
    else
        errno = EAGAIN;
    return rc;
}

bool zmq::xpub_t::xhas_out() { return _dist.has_out(); }

int zmq::xpub_t::xrecv(msg_t *msg_)
{
    //  If there is at least one
    if (_pending_data.empty())
    {
        errno = EAGAIN;
        return -1;
    }

    // User is reading a message, set last_pipe and remove it from the deque
    if (_manual && !_pending_pipes.empty())
    {
        _last_pipe = _pending_pipes.front();
        _pending_pipes.pop_front();
    }

    int rc = msg_->close();
    errno_assert(rc == 0);
    rc = msg_->init_size(_pending_data.front().size());
    errno_assert(rc == 0);
    memcpy(msg_->data(), _pending_data.front().data(), _pending_data.front().size());

    // set metadata only if there is some
    if (metadata_t *metadata = _pending_metadata.front())
    {
        msg_->set_metadata(metadata);
        // Remove ref corresponding to vector placement
        metadata->drop_ref();
    }

    msg_->set_flags(_pending_flags.front());
    _pending_data.pop_front();
    _pending_metadata.pop_front();
    _pending_flags.pop_front();
    return 0;
}

bool zmq::xpub_t::xhas_in() { return !_pending_data.empty(); }

void zmq::xpub_t::send_unsubscription(zmq::mtrie_t::prefix_t data_, size_t size_, xpub_t *self_)
{
    if (self_->options.type != ZMQ_PUB)
    {
        //  Place the unsubscription to the queue of pending (un)subscriptions
        //  to be retrieved by the user later on.
        blob_t unsub(size_ + 1);
        *unsub.data() = 0;
        if (size_ > 0)
            memcpy(unsub.data() + 1, data_, size_);
        self_->_pending_data.ZMQ_PUSH_OR_EMPLACE_BACK(ZMQ_MOVE(unsub));
        self_->_pending_metadata.push_back(NULL);
        self_->_pending_flags.push_back(0);

        if (self_->_manual)
        {
            self_->_last_pipe = NULL;
            self_->_pending_pipes.push_back(NULL);
        }
    }
}

//========= end of xpub.cpp ============

//========= begin of xsub.cpp ============

/*
    Copyright (c) 2007-2016 Contributors as noted in the AUTHORS file

    This file is part of libzmq, the ZeroMQ core engine in C++.

    libzmq is free software; you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, the Contributors give you permission to link
    this library with independent modules to produce an executable,
    regardless of the license terms of these independent modules, and to
    copy and distribute the resulting executable under terms of your choice,
    provided that you also meet, for each linked independent module, the
    terms and conditions of the license of that module. An independent
    module is a module which is not derived from or based on this library.
    If you modify this library, you must extend this exception to your
    version of the library.

    libzmq is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
    License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// ans ignore: #include "precompiled.hpp"
#include <string.h>

// ans ignore: #include "macros.hpp"
// ans ignore: #include "xsub.hpp"
// ans ignore: #include "err.hpp"

zmq::xsub_t::xsub_t(class ctx_t *parent_, uint32_t tid_, int sid_)
    : socket_base_t(parent_, tid_, sid_), _has_message(false), _more(false)
{
    options.type = ZMQ_XSUB;

    //  When socket is being closed down we don't want to wait till pending
    //  subscription commands are sent to the wire.
    options.linger.store(0);

    int rc = _message.init();
    errno_assert(rc == 0);
}

zmq::xsub_t::~xsub_t()
{
    int rc = _message.close();
    errno_assert(rc == 0);
}

void zmq::xsub_t::xattach_pipe(pipe_t *pipe_, bool subscribe_to_all_, bool locally_initiated_)
{
    LIBZMQ_UNUSED(subscribe_to_all_);
    LIBZMQ_UNUSED(locally_initiated_);

    zmq_assert(pipe_);
    _fq.attach(pipe_);
    _dist.attach(pipe_);

    //  Send all the cached subscriptions to the new upstream peer.
    _subscriptions.apply(send_subscription, pipe_);
    pipe_->flush();
}

void zmq::xsub_t::xread_activated(pipe_t *pipe_) { _fq.activated(pipe_); }

void zmq::xsub_t::xwrite_activated(pipe_t *pipe_) { _dist.activated(pipe_); }

void zmq::xsub_t::xpipe_terminated(pipe_t *pipe_)
{
    _fq.pipe_terminated(pipe_);
    _dist.pipe_terminated(pipe_);
}

void zmq::xsub_t::xhiccuped(pipe_t *pipe_)
{
    //  Send all the cached subscriptions to the hiccuped pipe.
    _subscriptions.apply(send_subscription, pipe_);
    pipe_->flush();
}

int zmq::xsub_t::xsend(msg_t *msg_)
{
    size_t         size = msg_->size();
    unsigned char *data = static_cast<unsigned char *>(msg_->data());

    if (msg_->is_subscribe() || (size > 0 && *data == 1))
    {
        //  Process subscribe message
        //  This used to filter out duplicate subscriptions,
        //  however this is alread done on the XPUB side and
        //  doing it here as well breaks ZMQ_XPUB_VERBOSE
        //  when there are forwarding devices involved.
        if (msg_->is_subscribe())
        {
            data = static_cast<unsigned char *>(msg_->command_body());
            size = msg_->command_body_size();
        }
        else
        {
            data = data + 1;
            size = size - 1;
        }
        _subscriptions.add(data, size);
        return _dist.send_to_all(msg_);
    }
    if (msg_->is_cancel() || (size > 0 && *data == 0))
    {
        //  Process unsubscribe message
        if (msg_->is_cancel())
        {
            data = static_cast<unsigned char *>(msg_->command_body());
            size = msg_->command_body_size();
        }
        else
        {
            data = data + 1;
            size = size - 1;
        }
        if (_subscriptions.rm(data, size))
            return _dist.send_to_all(msg_);
    }
    else
        //  User message sent upstream to XPUB socket
        return _dist.send_to_all(msg_);

    int rc = msg_->close();
    errno_assert(rc == 0);
    rc = msg_->init();
    errno_assert(rc == 0);

    return 0;
}

bool zmq::xsub_t::xhas_out()
{
    //  Subscription can be added/removed anytime.
    return true;
}

int zmq::xsub_t::xrecv(msg_t *msg_)
{
    //  If there's already a message prepared by a previous call to zmq_poll,
    //  return it straight ahead.
    if (_has_message)
    {
        int rc = msg_->move(_message);
        errno_assert(rc == 0);
        _has_message = false;
        _more        = (msg_->flags() & msg_t::more) != 0;
        return 0;
    }

    //  TODO: This can result in infinite loop in the case of continuous
    //  stream of non-matching messages which breaks the non-blocking recv
    //  semantics.
    while (true)
    {
        //  Get a message using fair queueing algorithm.
        int rc = _fq.recv(msg_);

        //  If ther