# `Dalle3` Documentation

## Table of Contents

1. [Introduction](#introduction)
2. [Installation](#installation)
3. [Quick Start](#quick-start)
4. [Dalle3 Class](#dalle3-class)
    - [Attributes](#attributes)
    - [Methods](#methods)
5. [Usage Examples](#usage-examples)
6. [Error Handling](#error-handling)
7. [Advanced Usage](#advanced-usage)
8. [References](#references)

---

## Introduction<a name="introduction"></a>

The Dalle3 library is a Python module that provides an easy-to-use interface for generating images from text descriptions using the DALL·E 3 model by OpenAI. DALL·E 3 is a powerful language model capable of converting textual prompts into images. This documentation will guide you through the installation, setup, and usage of the Dalle3 library.

---

## Installation<a name="installation"></a>

To use the Dalle3 model, you must first install swarms:

```bash
pip install swarms
```

---

## Quick Start<a name="quick-start"></a>

Let's get started with a quick example of using the Dalle3 library to generate an image from a text prompt:

```python
from swarm_models.dalle3 import Dalle3

# Create an instance of the Dalle3 class
dalle = Dalle3()

# Define a text prompt
task = "A painting of a dog"

# Generate an image from the text prompt
image_url = dalle3(task)

# Print the generated image URL
print(image_url)
```

This example demonstrates the basic usage of the Dalle3 library to convert a text prompt into an image. The generated image URL will be printed to the console.

---

## Dalle3 Class<a name="dalle3-class"></a>

The Dalle3 library provides a `Dalle3` class that allows you to interact with the DALL·E 3 model. This class has several attributes and methods for generating images from text prompts.

### Attributes<a name="attributes"></a>

- `model` (str): The name of the DALL·E 3 model. Default: "dall-e-3".
- `img` (str): The image URL generated by the Dalle3 API.
- `size` (str): The size of the generated image. Default: "1024x1024".
- `max_retries` (int): The maximum number of API request retries. Default: 3.
- `quality` (str): The quality of the generated image. Default: "standard".
- `n` (int): The number of variations to create. Default: 4.

### Methods<a name="methods"></a>

#### `__call__(self, task: str) -> Dalle3`

This method makes a call to the Dalle3 API and returns the image URL generated from the provided text prompt.

Parameters:
- `task` (str): The text prompt to be converted to an image.

Returns:
- `Dalle3`: An instance of the Dalle3 class with the image URL generated by the Dalle3 API.

#### `create_variations(self, img: str)`

This method creates variations of an image using the Dalle3 API.

Parameters:
- `img` (str): The image to be used for the API request.

Returns:
- `img` (str): The image URL of the generated variations.

---

## Usage Examples<a name="usage-examples"></a>

### Example 1: Basic Image Generation

```python
from swarm_models.dalle3 import Dalle3

# Create an instance of the Dalle3 class
dalle3 = Dalle3()

# Define a text prompt
task = "A painting of a dog"

# Generate an image from the text prompt
image_url = dalle3(task)

# Print the generated image URL
print(image_url)
```

### Example 2: Creating Image Variations

```python
from swarm_models.dalle3 import Dalle3

# Create an instance of the Dalle3 class
dalle3 = Dalle3()

# Define the URL of an existing image
img_url = "https://images.unsplash.com/photo-1694734479898-6ac4633158ac?q=80&w=1287&auto=format&fit=crop&ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D

# Create variations of the image
variations_url = dalle3.create_variations(img_url)

# Print the URLs of the generated variations
print(variations_url)
```

Certainly! Here are additional examples that cover various edge cases and methods of the `Dalle3` class in the Dalle3 library:

### Example 3: Customizing Image Size

You can customize the size of the generated image by specifying the `size` parameter when creating an instance of the `Dalle3` class. Here's how to generate a smaller image:

```python
from swarm_models.dalle3 import Dalle3

# Create an instance of the Dalle3 class with a custom image size
dalle3 = Dalle3(size="512x512")

# Define a text prompt
task = "A small painting of a cat"

# Generate a smaller image from the text prompt
image_url = dalle3(task)

# Print the generated image URL
print(image_url)
```

### Example 4: Adjusting Retry Limit

You can adjust the maximum number of API request retries using the `max_retries` parameter. Here's how to increase the retry limit:

```python
from swarm_models.dalle3 import Dalle3

# Create an instance of the Dalle3 class with a higher retry limit
dalle3 = Dalle3(max_retries=5)

# Define a text prompt
task = "An image of a landscape"

# Generate an image with a higher retry limit
image_url = dalle3(task)

# Print the generated image URL
print(image_url)
```

### Example 5: Generating Image Variations

To create variations of an existing image, you can use the `create_variations` method. Here's an example:

```python
from swarm_models.dalle3 import Dalle3

# Create an instance of the Dalle3 class
dalle3 = Dalle3()

# Define the URL of an existing image
img_url = "https://images.unsplash.com/photo-1677290043066-12eccd944004?q=80&w=1287&auto=format&fit=crop&ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D"

# Create variations of the image
variations_url = dalle3.create_variations(img_url)

# Print the URLs of the generated variations
print(variations_url)
```

### Example 6: Handling API Errors

The Dalle3 library provides error handling for API-related issues. Here's how to handle and display API errors:

```python
from swarm_models.dalle3 import Dalle3

# Create an instance of the Dalle3 class
dalle3 = Dalle3()

# Define a text prompt
task = "Invalid prompt that may cause an API error"

try:
    # Attempt to generate an image with an invalid prompt
    image_url = dalle3(task)
    print(image_url)
except Exception as e:
    print(f"Error occurred: {str(e)}")
```

### Example 7: Customizing Image Quality

You can customize the quality of the generated image by specifying the `quality` parameter. Here's how to generate a high-quality image:

```python
from swarm_models.dalle3 import Dalle3

# Create an instance of the Dalle3 class with high quality
dalle3 = Dalle3(quality="high")

# Define a text prompt
task = "A high-quality image of a sunset"

# Generate a high-quality image from the text prompt
image_url = dalle3(task)

# Print the generated image URL
print(image_url)
```


---

## Error Handling<a name="error-handling"></a>

The Dalle3 library provides error handling for API-related issues. If an error occurs during API communication, the library will handle it and provide detailed error messages. Make sure to handle exceptions appropriately in your code.

---

## Advanced Usage<a name="advanced-usage"></a>

For advanced usage and customization of the Dalle3 library, you can explore the attributes and methods of the `Dalle3` class. Adjusting parameters such as `size`, `max_retries`, and `quality` allows you to fine-tune the image generation process to your specific needs.

---

## References<a name="references"></a>

For more information about the DALL·E 3 model and the Dalle3 library, you can refer to the official OpenAI documentation and resources.

- [OpenAI API Documentation](https://beta.openai.com/docs/)
- [DALL·E 3 Model Information](https://openai.com/research/dall-e-3)
- [Dalle3 GitHub Repository](https://github.com/openai/dall-e-3)

---

This concludes the documentation for the Dalle3 library. You can now use the library to generate images from text prompts and explore its advanced features for various applications.