'use client'
import { unstable_getServerSession } from 'next-auth'
import React, { FormEvent, useState } from 'react'
import useSWR from 'swr'
import { v4 as uuid } from 'uuid'
import { Message } from '../typings'
import fetcher from '../utils/fetchMessages'

type Props = {
	session: Awaited<ReturnType<typeof unstable_getServerSession>>
}

function ChatInput({ session }: Props) {
	const [input, setInput] = useState('')
	const { data: messages, error, mutate } = useSWR('/api/getMessages', fetcher)

	const addMessage = async (e: FormEvent<HTMLFormElement>) => {
		e.preventDefault()

		if (!input || !session) return

		const messageToSend = input

		setInput('')

		const id = uuid()

		const message: Message = {
			id,
			message: messageToSend,
			created_at: Date.now(),
			username: session.user?.name!,
			profilePic: session.user?.image!,
			email: session.user?.email!
		}

		const uploadMessageToUpstash = async () => {
			const data = await fetch('/api/addMessage', {
				method: 'POST',
				headers: {
					'Content-Type': 'application/json'
				},
				body: JSON.stringify({
					message
				})
			}).then((res) => res.json())

			return [data.message, ...messages!]
		}
		await mutate(uploadMessageToUpstash, {
			optimisticData: [message, ...messages!],
			rollbackOnError: true
		})
	}

	return (
		<form
			onSubmit={addMessage}
			className="flex p-10 py-5 space-x-2 border-t border-gray-100 fixed bottom-0 z-50 w-full"
		>
			<input
				type="text"
				className="flex-1 rounded border border-gray-300 focus:outline-none focus:ring-2 focus:rign-blue-600 focus:border-transparent px-5 py-3 disabled:opacity-50 disabled:cursor-not-allowed"
				placeholder="Enter message here..."
				value={input}
				disabled={!session}
				onChange={(e) => setInput(e.target.value)}
			/>
			<button
				type="submit"
				className="bg-blue-500 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded disabled:opacity-50 disabled:cursor-not-allowed"
				disabled={!input}
			>
				Send Message
			</button>
		</form>
	)
}

export default ChatInput
