# SRFI 60 - Integers as bits

Various operations designed to work on integers as strings of bits efficiently.

See the [SRFI document][http://srfi.schemers.org/srfi-60/srfi-60.html] for more information.

## Bitwise operations
[`logand`](#logand)
[`logior`](#logior)
[`logxor`](#logxor)
[`lognot`](#lognot)
[`logtest`](#logtest)
[`bitwise-and`](#bitwise-and)
[`bitwise-ior`](#bitwise-ior)
[`bitwise-xor`](#bitwise-xor)
[`bitwise-not`](#bitwise-not)
[`bitwise-if`](#bitwise-if)
[`bitwise-merge`](#bitwise-merge)
[`any-bits-set?`](#any-bits-set)

## Integer properties
[`logcount`](#logcount)
[`log2-binary-factors`](#log2-binary-factors)
[`bit-count`](#bit-count)
[`integer-length`](#integer-length)
[`first-set-bit`](#first-set-bit)

## Bit within word
[`logbit?`](#logbit)
[`bit-set?`](#bit-set)
[`copy-bit`](#copy-bit)

## Field of bits
[`bit-field`](#bit-field)
[`copy-bit-field`](#copy-bit-field)
[`ash`](#ash)
[`arithmetic-shift`](#ash)
[`rotate-bit-field`](#rotate-bit-field)

## Bits as booleans
[`integer->list`](#integer-list)
[`list->integer`](#list-integer)
[`booleans->integer`](#booleans-integer)

# logand

    (logand n1 ...)

Returns the integer which is the bitwise-AND of the arguments.

Example: ``(number->string (logand #b1100 #b1010) 2) => "1000"``

# bitwise-and

Synonym for ``logand``.

# logior

    (logior n1 ...)

Returns the integer which is the bitwise-OR of the arguments.

Example: ``(number->string (logior #b1100 #b1010) 2) => "1110"``

# bitwise-ior

Synonym for ``logior``.

# logxor

    (logxor n1 ...)

Returns the integer which is the bitwise-XOR of the arguments.

Example: ``(number->string (logxor #b1100 #b1010) 2) => "110"``

# bitwise-xor

Synonym for ``logxor``.

# lognot

    (lognot n)

Returns the integer which is the one's-complement of the argument.

# bitwise-not

Synonym for ``lognot``.

# bitwise-if

    (bitwise-if mask n0 n1)

Returns an integer composed of some bits from ``n0`` and some bits from ``n1``.
A bit of the result is taken from ``n0`` if the corresponding bit of ``mask`` is
1, and from ``n1`` otherwise.

# bitwise-merge

Synonym for ``bitwise-if``.

# logtest

    (logtest j k)

Equivalent to ``(not (zero? (logand j k)))``.

Example: ``(logtest #b0100 #b1011) => #f``

# any-bits-set?

Synonym for ``logtest``.

# logcount

    (logcount n)

Returns the number of bits in ``n``. If ``n`` is positive, the 1-bits in its
binary representation are counted. If negative, the 0-bits in its
two's-complement binary representation are counted. If 0, then 0 is returned.

Example: ``(logcount #b10101010) => 4``

# bit-count

Synonym for ``logcount``.

# integer-length

    (integer-length n)

Returns the number of bits necessary to represent ``n``.

Example: ``(integer-length #b1011) => 4``

# log2-binary-factors

    (log2-binary-factors n)

Returns the bit-index of the least-significant 1-bit in ``n``. If ``n`` contains
no 1-bits, -1 is returned.

Example: ``(log2-binary-factors 4) => 2``

# first-set-bit

Synonym for ``log2-binary-factors``

# logbit?

    (logbit? index n)

Equivalent to ``(logtest (exact (expt 2 index)) n)``.

Example: ``(logbit? 1 #b1101) => #f``

# bit-set?

Synonym for ``logbit?``.

# copy-bit

    (copy-bit index from bit)

Returns ``from``, except in the ``index``th bit, which is 1 if ``bit`` is #t,
and 0 if ``bit`` is #f.

Example: ``(number->string (copy-bit 2 #b1111 #f) 2) => "1011"``

# bit-field

    (bit-field n start end)

Returns the integer composed of the ``start`` (inclusive) through ``end``
(exclusive) bits of ``n``. The ``start``th bit becomes the 0th bit in the
result.

Example: ``(number->string (bit-field #b1101101010 0 4) 2) => "1010"``

# copy-bit-field

    (copy-bit-field to from start end)

Returns ``to``, except possibly in the ``start`` (inclusive) through ``end``
(exclusive) bits, which are the same as those of ``from``. The 0th bit of
``from`` becomes the ``start``th bit of the result.

Example: ``(number->string (copy-bit-field #b1101101010 0 0 4) 2) =>
"1101100000"``

# ash

    (ash n count)

Equivalent to ``(exact (floor (* n (expt 2 count))))``.

Example: ``(number->string (ash #b1010 -1) 2) => "101"``

# arithmetic-shift

Synonym for ``ash``.

# rotate-bit-field

    (rotate-bit-field n count start end)

Returns ``n`` with the bit-field from ``start`` to ``end`` cyclically permuted
by ``count`` bits towards high-order.

Example: ``(number->string (rotate-bit-field #b0100 3 0 4) 2) => "10"``

# reverse-bit-field

    (reverse-bit-field n start end)

Returns ``n`` with the order of bits ``start`` to ``end`` reversed.

Example: ``(number->string (reverse-bit-field #b10100111 0 8) 2) =>
"11100101")``

# integer->list

    (integer->list k)
    
    (integer->list k len)

Returns a list of ``len`` booleans corresponding to each bit of the non-negative
integer ``k``. ``#t`` is coded for each 1; ``#f`` for each 0. ``len`` defaults
to ``(integer-length k)``.

# list->integer

    (list->integer list)

Returns an integer formed from the booleans in ``list``, which must consist only
of booleans. A 1 bit is coded for each ``#t``; a 0 bit for each ``#f``.
``integer->list`` and ``list->integer`` are inverses up to ``equal?``: thus, for
any ``x``, ``(equal x (integer->list (list->integer x))) => #t``.

# booleans->integer

    (booleans->integer b1 ...)

Equivalent to ``(list->integer (list b1 ...))``.
