# SRFI 18 - Multithreading support

The `(srfi 18)` library provides multithreading support. 

See the [Multithreading support SRFI documentation](http://srfi.schemers.org/srfi-18/srfi-18.html) for more information.

- [`thread?`](#thread)
- [`make-thread`](#make-thread)
- [`current-thread`](#current-thread)
- [`current-thread-data`](#current-thread-data)
- [`thread-name`](#thread-name)
- [`thread-specific`](#thread-specific)
- [`thread-specific-set!`](#thread-specific-set)
- [`thread-start!`](#thread-start)
- [`thread-sleep!`](#thread-sleep)
- [`thread-yield!`](#thread-yield)
- [`thread-terminate!`](#thread-terminate)
- [`thread-join!`](#thread-join)
- [`mutex?`](#mutex)
- [`make-mutex`](#make-mutex) 
- [`mutex-lock!`](#mutex-lock)
- [`mutex-unlock!`](#mutex-unlock)
- [`condition-variable?`](#condition-variable)
- [`make-condition-variable`](#make-condition-variable)
- [`condition-variable-wait!`](#condition-variable-wait)
- [`condition-variable-signal!`](#condition-variable-signal)
- [`condition-variable-broadcast!`](#condition-variable-broadcast)
- [`->heap`](#-heap)
- [`Cyc-minor-gc`](#cyc-minor-gc)

# thread?

    (thread? obj) 

Determine if the given object is a thread object.

# make-thread

    (make-thread thunk)

    (make-thread thunk name)

Create a new thread object.

# current-thread

    (current-thread)

Returns the current thread.

# current-thread-data

    (current-thread-data)

Returns the `gc_thread_data` instance for the current thread, wrapped in an opaque object. This is useful when using the FFI as many functions in the Cyclone runtime require the `gc_thread_data` data argument.

# thread-name

    (thread-name t) (vector-ref t 3))

Retrieve the name of the given thread object.

# thread-specific

    (thread-specific t)

Retrieve thread-specific data.

# thread-specific-set!

    (thread-specific-set! t obj)

Set thread-specific data.

# thread-start!

    (thread-start! t)

Makes thread runnable. The thread must be a new thread. thread-start! returns the thread.

# thread-sleep!

    (thread-sleep! timeout)

Block the current thread for `timeout` seconds. Fractional seconds may be provided to sleep for less than one second.

# thread-yield!

    (thread-yield!)

The current thread exits the running state as if its quantum had expired.

# thread-terminate!

    (thread-terminate!)

Immediately abort the current thread.

# thread-join!

    (thread-join! thread)

The current thread waits until `thread` terminates.

# mutex?

    (mutex? obj)

Determine if the given object is a mutex.

# make-mutex 

    (make-mutex)

Create a new mutex object.

NOTE: Creates a new mutex by allocating it on the heap. This is different than other types of objects because by definition a mutex will be used by multiple threads, so no need to risk having the non-creating thread pick up a stack object reference by mistake.

# mutex-lock! 

    (mutex-lock! mutex [timeout])

Lock the given mutex if it is unlocked.  If the mutex is currently locked, the current thread waits until the mutex is unlocked, or until the timeout is reached if `timeout` is supplied.

# mutex-unlock!

    (mutex-unlock! mutex)

    (mutex-unlock! mutex condition-variable)

Unlock the given mutex. If there are threads waiting on the mutex, one of those threads will be unblocked. If condition-variable is supplied, the current thread is blocked and added to the condition-variable before unlocking mutex; the thread can unblock at any time but no later than when an appropriate call to condition-variable-signal! or condition-variable-broadcast! is performed (see below). See SRFI documentation for more information on condition variables.

# condition-variable?

    (condition-variable? obj)

Determine if the given object is a condition variable.

# make-condition-variable

    (make-condition-variable)

Create a new condition variable. Like mutex objects, condition variables are always allocated directly on the heap since they are expected to be shared by more than one thread.

# condition-variable-wait!

    (condition-variable-wait! condition-variable mutex)

Wait on a condition variable. The mutex must be locked by the calling thread, and will be locked upon successful return of this function.

# condition-variable-signal!

    (condition-variable-signal! condition-variable)

If there are threads blocked on the condition-variable, the scheduler selects a thread and unblocks it.

# condition-variable-broadcast!

    (condition-variable-broadcast! condition-variable)

Unblocks all the threads blocked on the condition-variable.

# ->heap

    (->heap obj)

Take a single object and if it is on the stack, return a copy of it that is allocated on the heap. NOTE the original object will still live on the stack, and will eventually be moved itself to the heap if it is referenced during minor GC.

# Cyc-minor-gc

    (Cyc-minor-gc)

Trigger a minor garbage collection.  This is potentially useful to evacuate all objects from a thread's stack to the heap. An object must be moved to the heap before it can be safely used by more than one thread.

