# SRFI 128 - Comparators

The `(srfi 128)` provides comparators, which bundle a type test predicate, an equality predicate, an ordering predicate, and a hash function into a single Scheme object. By packaging these procedures together, they can be treated as a single item for use in the implementation of data structures.

See the [SRFI document](http://srfi.schemers.org/srfi-128/srfi-128.html) for more information.

## Predicates

- [`comparator?                   `](#comparator)
- [`comparator-ordered?           `](#comparator-ordered)
- [`comparator-hashable?          `](#comparator-hashable)

## Constructors

The following comparator constructors all supply appropriate type test predicates, equality predicates, ordering predicates, and hash functions based on the supplied arguments. They are allowed to cache their results: they need not return a newly allocated object, since comparators are pure and functional. In addition, the procedures in a comparator are likewise pure and functional.

- [`make-comparator               `](#make-comparator)
- [`make-pair-comparator          `](#make-pair-comparator)
- [`make-list-comparator          `](#make-list-comparator)
- [`make-vector-comparator        `](#make-vector-comparator)
- [`make-eq-comparator            `](#make-eq-comparator)
- [`make-eqv-comparator           `](#make-eqv-comparator)
- [`make-equal-comparator         `](#make-equal-comparator)

## Standard Hash Functions

These are hash functions for some standard Scheme types, suitable for passing to make-comparator. Users may write their own hash functions with the same signature. However, if programmers wish their hash functions to be backward compatible with the reference implementation of SRFI 69, they are advised to write their hash functions to accept a second argument and ignore it.

- [`boolean-hash                  `](#boolean-hash)
- [`char-hash                     `](#char-hash)
- [`char-ci-hash                  `](#char-ci-hash)
- [`string-hash                   `](#string-hash)
- [`string-ci-hash                `](#string-ci-hash)
- [`symbol-hash                   `](#symbol-hash)
- [`number-hash                   `](#number-hash)

## Default Comparators

- [`make-default-comparator       `](#make-default-comparator)
- [`default-hash                  `](#default-hash)
- [`comparator-register-default!  `](#comparator-register-default)

## Accessors and Invokers

- [`comparator-type-test-predicate`](#comparator-type-test-predicate)
- [`comparator-equality-predicate `](#comparator-equality-predicate)
- [`comparator-ordering-predicate `](#comparator-ordering-predicate)
- [`comparator-hash-function      `](#comparator-hash-function)
- [`comparator-test-type          `](#comparator-test-type)
- [`comparator-check-type         `](#comparator-check-type)
- [`comparator-hash               `](#comparator-hash)

## Bounds and Salt

The following macros allow the callers of hash functions to affect their behavior without interfering with the calling signature of a hash function, which accepts a single argument (the object to be hashed) and returns its hash value. They are provided as macros so that they may be implemented in different ways: as a global variable, a SRFI 39 or R7RS parameter, or an ordinary procedure, whatever is most efficient in a particular implementation.

- [`hash-bound                    `](#hash-bound)
- [`hash-salt                     `](#hash-salt)

## Comparison Predicates

These procedures are analogous to the number, character, and string comparison predicates of Scheme. They allow the convenient use of comparators to handle variable data types.

These procedures apply the equality and ordering predicates of comparator to the objects as follows. If the specified relation returns `#t` for all `objecti` and `objectj` where `n` is the number of objects and `1 <= i < j <= n`, then the procedures return `#t`, but otherwise `#f`. Because the relations are transitive, it suffices to compare each object with its successor. The order in which the values are compared is unspecified.

- [`=?                            `](#)
- [`<?                            `](#-1)
- [`>?                            `](#-2)
- [`<=?                           `](#-3)
- [`>=?                           `](#-4)

## Syntax

- [`comparator-if<=>              `](#comparator-if)

# comparator?

    (comparator? obj)

Returns `#t` if `obj` is a comparator, and `#f` otherwise.

# comparator-ordered?

    (comparator-ordered? comparator)

Returns `#t` if `comparator` has a supplied ordering predicate, and `#f` otherwise.

# comparator-hashable?

    (comparator-hashable? comparator)

Returns `#t` if `comparator` has a supplied hash function, and `#f` otherwise.

# make-comparator

    (make-comparator type-test equality ordering hash)

Returns a comparator which bundles the `type-test`, `equality`, `ordering`, and `hash` procedures provided. However, if `ordering` or `hash` is `#f`, a procedure is provided that signals an error on application. The predicates `comparator-ordered?` and/or `comparator-hashable?`, respectively, will return `#f` in these cases.

Here are calls on `make-comparator` that will return useful comparators for standard Scheme types:

* `(make-comparator boolean? boolean=? (lambda (x y) (and (not x) y)) boolean-hash)` will return a comparator for booleans, expressing the ordering `#f < #t` and the standard hash function for booleans.
    
* `(make-comparator real? = < (lambda (x) (exact (abs x))))` will return a comparator expressing the natural ordering of real numbers and a plausible (but not optimal) hash function.
    
* `(make-comparator string? string=? string<? string-hash)` will return a comparator expressing the implementation's ordering of strings and the standard hash function.
    
* `(make-comparator string? string-ci=? string-ci<? string-ci-hash)` will return a comparator expressing the implementation's case-insensitive ordering of strings and the standard case-insensitive hash function.

# make-pair-comparator

    (make-pair-comparator car-comparator cdr-comparator)

This procedure returns comparators whose functions behave as follows.

* The type test returns `#t` if its argument is a pair, if the car satisfies the type test predicate of car-comparator, and the cdr satisfies the type test predicate of cdr-comparator.

* The equality function returns `#t` if the cars are equal according to car-comparator and the cdrs are equal according to cdr-comparator, and `#f` otherwise.

* The ordering function first compares the cars of its pairs using the equality predicate of car-comparator. If they are not equal, then the ordering predicate of car-comparator is applied to the cars and its value is returned. Otherwise, the predicate compares the cdrs using the equality predicate of cdr-comparator. If they are not equal, then the ordering predicate of cdr-comparator is applied to the cdrs and its value is returned.

* The hash function computes the hash values of the car and the cdr using the hash functions of car-comparator and cdr-comparator respectively and then hashes them together in an implementation-defined way.

# make-list-comparator

    (make-list-comparator element-comparator type-test empty? head tail)

This procedure returns comparators whose functions behave as follows:

* The type test returns `#t` if its argument satisfies type-test and the elements satisfy the type test predicate of element-comparator.

* The total order defined by the equality and ordering functions is as follows (known as lexicographic order):

  * The empty sequence, as determined by calling `empty?`, compares `equal` to itself.
  * The empty sequence compares less than any non-empty sequence.
  * Two non-empty sequences are compared by calling the head procedure on each. If the heads are not equal when compared using element-comparator, the result is the result of that comparison. Otherwise, the results of calling the tail procedure are compared recursively.

* The hash function computes the hash values of the elements using the hash function of element-comparator and then hashes them together in an implementation-defined way.

# make-vector-comparator

    (make-vector-comparator element-comparator type-test length ref)

This procedure returns comparators whose functions behave as follows:

* The type test returns `#t` if its argument satisfies type-test and the elements satisfy the type test predicate of element-comparator.

* The equality predicate returns `#t` if both of the following tests are satisfied in order: the lengths of the vectors are the same in the sense of `=`, and the elements of the vectors are the same in the sense of the equality predicate of element-comparator.

* The ordering predicate returns `#t` if the results of applying length to the first vector is less than the result of applying `length` to the second vector. If the lengths are equal, then the elements are examined pairwise using the ordering predicate of element-comparator. If any pair of elements returns `#t`, then that is the result of the list comparator's ordering predicate; otherwise the result is `#f`.

* The hash function computes the hash values of the elements using the hash function of element-comparator and then hashes them together in an implementation-defined way.

Here is an example, which returns a comparator for byte vectors:

    (make-vector-comparator
      (make-comparator exact-integer? = < number-hash)
      bytevector?
      bytevector-length
      bytevector-u8-ref)

# make-eq-comparator

    (make-eq-comparator)

# make-eqv-comparator

    (make-eqv-comparator)

# make-equal-comparator

    (make-equal-comparator)

These procedures return comparators whose functions behave as follows:

* The type test returns `#t` in all cases.

* The equality functions are `eq?`, `eqv?`, and `equal?` respectively.

* The ordering function is implementation-defined, except that it must conform to the rules for ordering functions. It may signal an error instead.

* The hash function is default-hash.

These comparators accept circular structure and `NaN`s.

# boolean-hash

    (boolean-hash obj)

# char-hash
    (char-hash obj)

# char-ci-hash

    (char-ci-hash obj)

# string-hash
    
    (string-hash obj)

# string-ci-hash
    (string-ci-hash obj)
    
# symbol-hash

    (symbol-hash obj)

# number-hash

    (number-hash obj)


# make-default-comparator

    (make-default-comparator)

Returns a comparator known as a default comparator that accepts Scheme values and orders them in some implementation-defined way, subject to the following conditions:

* Given disjoint types a and b, one of three conditions must hold:

  * All objects of type a compare less than all objects of type b.
  * All objects of type a compare greater than all objects of type b.
  * All objects of both type a and type b compare equal to each other. This is not permitted for any of the Scheme types mentioned below.

* The empty list must be ordered before all pairs.

* When comparing booleans, it must use the total order `#f < #t`.

* When comparing characters, it must use `char=?` and `char<?`.

  Note: In R5RS, this is an implementation-dependent order that is typically the same as Unicode codepoint order; in R6RS and R7RS, it is Unicode codepoint order.

* When comparing pairs, it must behave the same as a comparator returned by make-pair-comparator with default comparators as arguments.

* When comparing symbols, it must use an implementation-dependent total order. One possibility is to use the order obtained by applying `symbol->string` to the symbols and comparing them using the total order implied by `string<?`.

* When comparing bytevectors, it must behave the same as a comparator created by the expression `(make-vector-comparator (make-comparator bytevector? = < number-hash) bytevector? bytevector-length bytevector-u8-ref)`.

* When comparing numbers where either number is complex, since non-real numbers cannot be compared with `<,` the following least-surprising ordering is defined: If the real parts are `<` or `>,` so are the numbers; otherwise, the numbers are ordered by their imaginary parts. This can still produce somewhat surprising results if one real part is exact and the other is inexact.

* When comparing real numbers, it must use `=` and `<.`

* When comparing strings, it must use `string=?` and `string<?`.

  Note: In R5RS, this is lexicographic order on the implementation-dependent order defined by `char<?`; in R6RS it is lexicographic order on Unicode codepoint order; in R7RS it is an implementation-defined order.

* When comparing vectors, it must behave the same as a comparator returned by `(make-vector-comparator (make-default-comparator) vector? vector-length vector-ref)`.

* When comparing members of types registered with `comparator-register-default!`, it must behave in the same way as the comparator registered using that function.

Default comparators use default-hash as their hash function.

# default-hash

    (default-hash obj)

This is the hash function used by default comparators, which accepts a Scheme value and hashes it in some implementation-defined way, subject to the following conditions:

* When applied to a pair, it must return the result of hashing together the values returned by `default-hash` when applied to the car and the cdr.

* When applied to a boolean, character, string, symbol, or number, it must return the same result as `boolean-hash`, `char-hash`, `string-hash`, `symbol-hash`, or `number-hash` respectively.

* When applied to a list or vector, it must return the result of hashing together the values returned by default-hash when applied to each of the elements.

# comparator-register-default!

    (comparator-register-default! comparator)

Registers comparator for use by default comparators, such that if the objects being compared both satisfy the type test predicate of comparator, it will be employed by default comparators to compare them. Returns an unspecified value. It is an error if any value satisfies both the type test predicate of comparator and any of the following type test predicates: `boolean?`, `char?`, `null?`, `pair?`, `symbol?`, `bytevector?`, `number?`, `string?`, `vector?`, or the type test predicate of a comparator that has already been registered.

This procedure is intended only to extend default comparators into territory that would otherwise be undefined, not to override their existing behavior. In general, the ordering of calls to comparator-register-default! should be irrelevant. However, implementations that support inheritance of record types may wish to ensure that default comparators always check subtypes before supertypes.

# comparator-type-test-predicate

    (comparator-type-test-predicate comparator)

# comparator-equality-predicate

    (comparator-equality-predicate comparator)

# comparator-ordering-predicate

    (comparator-ordering-predicate comparator)

# comparator-hash-function

    (comparator-hash-function comparator)

# comparator-test-type

    (comparator-test-type comparator obj)

Invokes the type test predicate of comparator on `obj` and returns what it returns. More convenient than `comparator-type-test-predicate`, but less efficient when the predicate is called repeatedly.

# comparator-check-type

    (comparator-check-type comparator obj)

Invokes the type test predicate of comparator on `obj` and returns true if it returns true, but signals an error otherwise. More convenient than `comparator-type-test-predicate`, but less efficient when the predicate is called repeatedly.

# comparator-hash

    (comparator-hash comparator obj)

Invokes the hash function of comparator on `obj` and returns what it returns. More convenient than `comparator-hash-function`, but less efficient when the function is called repeatedly.

Note: No invokers are required for the equality and ordering predicates, because `=?` and `<?` serve this function.

# hash-bound

*Syntax*

    (hash-bound)

Hash functions should be written so as to return a number between `0` and the largest reasonable number of elements (such as hash buckets) a data structure in the implementation might have. What that value is depends on the implementation. This value provides the current bound as a positive exact integer, typically for use by user-written hash functions. However, they are not required to bound their results in this way.

# hash-salt

*Syntax*

    (hash-salt)

A salt is random data in the form of a non-negative exact integer used as an additional input to a hash function in order to defend against dictionary attacks, or (when used in hash tables) against denial-of-service attacks that overcrowd certain hash buckets, increasing the amortized `O(1)` lookup time to `O(n)`. Salt can also be used to specify which of a family of hash functions should be used for purposes such as cuckoo hashing. This macro provides the current value of the salt, typically for use by user-written hash functions. However, they are not required to make use of the current salt.

The initial value is implementation-dependent, but must be less than the value of `(hash-bound)`, and should be distinct for distinct runs of a program unless otherwise specified by the implementation. Implementations may provide a means to specify the salt value to be used by a particular invocation of a hash function.

# =?

    (=? comparator object1 object2 object3 ...)

# <?

    (<? comparator object1 object2 object3 ...)

# >?

    (>? comparator object1 object2 object3 ...)

# <=?

    (<=? comparator object1 object2 object3 ...)

# >=?

    (>=? comparator object1 object2 object3 ...)

# comparator-if<=>

*Syntax*

    (comparator-if<=> [ <comparator> ] <object1> <object2> <less-than> <equal-to> <greater-than>)

It is an error unless `<comparator>` evaluates to a comparator and `<object1>` and `<object2>` evaluate to objects that the comparator can handle. If the ordering predicate returns true when applied to the values of `<object1>` and `<object2>` in that order, then `<less-than>` is evaluated and its value returned. If the equality predicate returns true when applied in the same way, then `<equal-to>` is evaluated and its value returned. If neither returns true, `<greater-than>` is evaluated and its value returned.

If `<comparator>` is omitted, a default comparator is used.

