# Desktop notification

Castor uses the [JoliNotif](https://github.com/jolicode/jolinotif) library to
display desktop notifications.

## The `notify()` function

You can use the `notify()` function to display a desktop notification:

```php
use Castor\Attribute\AsTask;

use function Castor\notify;

#[AsTask()]
function notify()
{
    notify('Hello world!');
}
```

## Notify with `run()`

You can use the `withNotify` method of the `Context` object to display a
desktop notification when a command has been executed:

```php
use Castor\Attribute\AsTask;

use function Castor\context;
use function Castor\run;

#[AsTask()]
function notify()
{
    run(['echo', 'notify'], context: context()->withNotify(true)); // will display a success notification
    run('command_that_does_not_exist', context: context()->withNotify(true)); // will display a failure notification
}
```

## Choose when to display a notification

You can set the `notify` property in the context to `false` to disable notifications globally:

```php
use Castor\Attribute\AsTask;

use Castor\Context;
use function Castor\context;
use function Castor\notify;
use function Castor\run;
use function Castor\with;

#[AsTask()]
function notify()
{
    with(
        callback: function () {
            notify('Hello world!'); // will not display a notification
            run('ls'); // will not display a notification
            
            run('ls', context: context()->withNotify()); // will display a notification (you override the context value on the fly)
        },
        context: new Context(notify: false),
    );
}
```

You can also set the `notify` property in the context to `null` that mean all user declared notifications will be displayed, but not the ones generated by Castor:

`null` is the default value of the `notify` property.

```php
use Castor\Attribute\AsTask;

use Castor\Context;
use function Castor\context;
use function Castor\notify;

#[AsTask()]
function notify()
{
    with(
        callback: function () {
            run('ls'); // will not display a notification
            run('ls', context: context()->withNotify()); // will display a notification
            notify('Hello world!'); // will display a notification
        },
        context: new Context(notify: null),
    );
}
```

Finally, you can set the `notify` property in the context to `true` to enable notifications globally:

In that case, any call to the `notify()` function will display a desktop notification (user or Castor generated notifications).

```php
use Castor\Attribute\AsTask;

use Castor\Context;
use function Castor\notify;

#[AsTask()]
function notify()
{
    with(
        callback: function () {
            run('ls'); // will display a notification
            notify('Hello world!'); // will display a notification
        },
        context: new Context(notify: true),
    );
}
```

## Customizing the notification title

You can set a custom title for notifications by setting the `notificationTitle` property in the context or
by passing a second argument to the `notify()` function:

> [!NOTE]
> By default the title is "Castor".
> The second argument of the `notify()` function will override the title set in the context.

```php
use Castor\Attribute\AsTask;

use function Castor\notify;
use Castor\Context;

#[AsContext(default: true)]
function my_context(): Context
{
    return new Context(
        notificationTitle: 'My custom title'
    );
}

#[AsTask()]
function notify()
{
    notify('Hello world!'); // will display a notification with the title "My custom title"
    notify('Hello world!', 'Specific title'); // will display a notification with the title "Specific title"
}
```
