<?php

namespace services\common;

use Yii;
use common\enums\CacheEnum;
use common\models\common\Provinces;
use common\components\Service;
use common\helpers\ArrayHelper;
use yii\helpers\Json;

/**
 * Class ProvincesService
 * @package services\common
 * @author jianyan74 <751393839@qq.com>
 */
class ProvincesService extends Service
{
    /**
     * 获取省市区禁用状态
     *
     * @param array $provinceIds
     * @param array $cityIds
     * @param array $areaIds
     * @return mixed
     */
    public function getAreaTree(array $provinceIds, array $cityIds, array $areaIds)
    {
        $address = $this->findAllInCache();

        $allIds = [];
        foreach ($address as &$item) {
            $allIds[$item['pid']][] = $item['id'];
        }

        // 计算选中状态
        foreach ($address as &$item) {
            $item['is_disabled'] = true;
            $data = $allIds[$item['id']] ?? [];

            if ($item['level'] == 1) {
                foreach ($data as $datum) {
                    !in_array($datum, $cityIds) && $item['is_disabled'] = false;
                }
            }

            if ($item['level'] == 2) {
                foreach ($data as $datum) {
                    !in_array($datum, $areaIds) && $item['is_disabled'] = false;
                }
            }

            if ($item['level'] == 3 && !in_array($item['id'], $areaIds)) {
                $item['is_disabled'] = false;
            }

            unset($data);
        }

        // 递归重组省市区
        $address = ArrayHelper::itemsMerge($address, 0);
        // 大区
        $regionalAll = $this->regionalAll();
        $regroupAddress = [];

        foreach ($address as $value) {
            foreach ($regionalAll as $key => $data) {
                foreach ($data as $datum) {
                    $datum == $value['title'] && $regroupAddress[$key][] = $value;
                }
            }
        }

        unset($address, $regionalAll, $allIds);
        return $regroupAddress;
    }

    /**
     * @param int $pid
     * @return int|string
     */
    public function getCountByPid($pid = 0)
    {
        return Provinces::find()
            ->select(['id'])
            ->where(['pid' => $pid])
            ->count();
    }

    /**
     * @param $ids
     * @return array|\yii\db\ActiveRecord[]
     *
     */
    public function findByIds($ids)
    {
        return Provinces::find()
            ->select(['id', 'title', 'pid', 'level'])
            ->orderBy('id asc')
            ->where(['in', 'id', $ids])
            ->asArray()
            ->all();
    }

    /**
     * @param $id
     * @return array|\yii\db\ActiveRecord|null
     */
    public function findById($id)
    {
        return Provinces::find()
            ->where(['id' => $id])
            ->asArray()
            ->one();
    }

    /**
     * @param int $pid
     * @param string $level
     * @return array
     */
    public function getCityMapByPid($pid = 0, $level = '')
    {
        return ArrayHelper::map($this->getCityByPid($pid, $level), 'id', 'title');
    }

    /**
     * 根据父级ID返回信息
     *
     * @param int $pid
     * @return array
     */
    public function getCityByPid($pid = 0, $level = '')
    {
        if ($pid === '') {
            return [];
        }

       return Provinces::find()
            ->where(['pid' => $pid])
            ->orderBy('id asc')
            ->select(['id', 'title', 'pid'])
            ->andFilterWhere(['level' => $level])
            ->cache(600)
            ->asArray()
            ->all();
    }

    /**
     * 根据id获取区域名称
     *
     * @param $id
     * @return mixed
     */
    public function getName($id)
    {
        if ($provinces = Provinces::findOne($id)) {
            return $provinces['title'] ?? '';
        }

        return false;
    }

    /**
     * 根据id数组获取区域名称
     *
     * @param $id
     * @return mixed
     */
    public function getCityListName(array $ids)
    {
        if ($provinces = Provinces::find()->where(['in', 'id', $ids])->orderBy('id asc')->all()) {
            $address = '';

            foreach ($provinces as $province) {
                $address .= $province['title'] . ' ';
            }

            return $address;
        }

        return false;
    }

    /**
     * @return array
     */
    public function getJsonData()
    {
        $data = $this->findAllInCache();
        $items = ArrayHelper::itemsMerge($data, 0, 'id', 'pid', 'child');

        return $items;
    }

    /**
     * @return array|mixed|\yii\db\ActiveRecord[]
     */
    public function findAllInCache()
    {
        $cacheKey = CacheEnum::getPrefix('provinces');

        // 获取缓存
        if (!($data = Yii::$app->cache->get($cacheKey))) {
            $data = Provinces::find()
                ->select(['id', 'title', 'pid', 'level'])
                ->where(['<=', 'level', 4])
                ->orderBy('id asc')
                ->asArray()
                ->all();

            Yii::$app->cache->set($cacheKey, $data, 60 * 60 * 24 * 24);
        }

        return $data;
    }

    /**
     * 获取大区
     *
     * @return array
     */
    public function regionalAll()
    {
        $region = [
            '华东' => [
                '江苏省',
                '上海市',
                '浙江省',
                '安徽省',
                '江西省',
            ],
            '华北' => [
                '天津市',
                '河北省',
                '山西省',
                '内蒙古自治区',
                '山东省',
                '北京市',
            ],
            '华南' => [
                '广东省',
                '广西壮族自治区',
                '海南省',
                '福建省',
            ],
            '华中' => [
                '湖南省',
                '河南省',
                '湖北省',
            ],
            '东北' => [
                '辽宁省',
                '吉林省',
                '黑龙江省',
            ],
            '西北' => [
                '陕西省',
                '甘肃省',
                '青海省',
                '宁夏回族自治区',
                '新疆维吾尔自治区',
            ],
            '西南' => [
                '重庆市',
                '四川省',
                '贵州省',
                '云南省',
                '西藏自治区',
            ],
            '港澳台' => [
                '香港特别行政区',
                '澳门特别行政区',
                '台湾省',
            ],
        ];

        return $region;
    }
}