// Copyright (C) 2024-2025  ilobilo

import drivers.terminal;
import drivers.serial;

import system.cpu;
import boot;
import arch;
import lib;
import std;

#if ILOBILIX_EXTRA_PANIC_MSG
namespace { std::pair<std::string_view, std::string_view> very_important_message(); } // namespace
#endif

namespace lib
{
    [[noreturn]]
    void stop_all()
    {
        arch::halt_others();
        arch::halt(false);
        std::unreachable();
    }

    [[noreturn]]
    void vpanic(std::string_view fmt, std::format_args args, cpu::registers *regs, std::source_location location)
    {
        arch::halt_others();
        log::unsafe::unlock();

        log::println("");
#if ILOBILIX_EXTRA_PANIC_MSG
        auto [unicode, ascii] = very_important_message();
#if !ILOBILIX_MAX_UACPI_POINTS
        if (auto term = term::main(); term)
            term->write(ascii);
#endif
        for (auto chr : unicode)
            serial::printc(chr);
        log::println("");
#endif

        log::fatal("kernel panicked with the following message:");
        log::fatal(fmt, args);
        log::fatal("at {}:{}:{}: {}", location.file_name(), location.line(), location.column(), location.function_name());

        if (regs)
            arch::dump_regs(regs, cpu::extra_regs::read(), log::level::fatal);

        arch::halt(false);
        std::unreachable();
    }
} // namespace lib

extern "C" [[gnu::noreturn]]
void assert_fail(const char *message, const char *file, int line, const char *func)
{
    struct custom_location : std::source_location
    {
        int _line;
        const char *_file;
        const char *_func;

        custom_location(const char *file, int line, const char *func)
            : _line { line }, _file { file }, _func { func } { }

        constexpr std::uint_least32_t line() const noexcept
        { return _line; }

        constexpr const char *file_name() const noexcept
        { return _file; }

        constexpr const char *function_name() const noexcept
        { return _func; }
    };
    custom_location loc { file, line, func };
    lib::vpanic(message, std::make_format_args(), nullptr, loc);
}

#if ILOBILIX_EXTRA_PANIC_MSG
namespace
{
    constexpr std::uint8_t important_message_unicode[]
    {
        0xE2, 0xA1, 0xB4, 0xE2, 0xA0, 0x92, 0xE2, 0xA3, 0x84, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0,
        0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0,
        0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA3, 0xBC, 0xE2, 0xA0, 0x89, 0xE2,
        0xA0, 0xB3, 0xE2, 0xA1, 0x86, 0xE2, 0xA0, 0x80, 0x0A, 0xE2, 0xA3, 0x87, 0xE2, 0xA0, 0xB0, 0xE2,
        0xA0, 0x89, 0xE2, 0xA2, 0x99, 0xE2, 0xA1, 0x84, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA3,
        0xB4, 0xE2, 0xA0, 0x96, 0xE2, 0xA2, 0xA6, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0,
        0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x98, 0xE2, 0xA3, 0x86, 0xE2, 0xA0, 0x81, 0xE2, 0xA0, 0x99, 0xE2, 0xA1,
        0x86, 0x0A, 0xE2, 0xA0, 0x98, 0xE2, 0xA1, 0x87, 0xE2, 0xA2, 0xA0, 0xE2, 0xA0, 0x9E, 0xE2, 0xA0,
        0x89, 0xE2, 0xA0, 0x99, 0xE2, 0xA3, 0xBE, 0xE2, 0xA0, 0x83, 0xE2, 0xA2, 0x80, 0xE2, 0xA1, 0xBC,
        0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA2, 0x80, 0xE2, 0xA3, 0xBC, 0xE2, 0xA1, 0x80, 0xE2, 0xA0,
        0x84, 0xE2, 0xA2, 0xB7, 0xE2, 0xA3, 0x84, 0xE2, 0xA3, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0xB0, 0xE2, 0xA0, 0x92, 0xE2, 0xA0, 0xB2, 0xE2, 0xA1, 0x84, 0xE2, 0xA0, 0x80, 0xE2, 0xA3,
        0x8F, 0xE2, 0xA3, 0x86, 0xE2, 0xA3, 0x80, 0xE2, 0xA1, 0x8D, 0x0A, 0xE2, 0xA0, 0x80, 0xE2, 0xA2,
        0xA0, 0xE2, 0xA1, 0x8F, 0xE2, 0xA0, 0x80, 0xE2, 0xA1, 0xA4, 0xE2, 0xA0, 0x92, 0xE2, 0xA0, 0x83,
        0xE2, 0xA0, 0x80, 0xE2, 0xA1, 0x9C, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA2, 0x80, 0xE2, 0xA3, 0xB4, 0xE2, 0xA0, 0xBE, 0xE2, 0xA0,
        0x9B, 0xE2, 0xA1, 0x81, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA2, 0x80, 0xE2, 0xA3, 0x88,
        0xE2, 0xA1, 0x89, 0xE2, 0xA0, 0x99, 0xE2, 0xA0, 0xB3, 0xE2, 0xA3, 0xA4, 0xE2, 0xA1, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x98, 0xE2, 0xA3, 0x86, 0xE2, 0xA0,
        0x80, 0xE2, 0xA3, 0x87, 0xE2, 0xA1, 0xBC, 0xE2, 0xA2, 0x8B, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80,
        0xE2, 0xA2, 0xB1, 0x0A, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x98, 0xE2, 0xA3, 0x87, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA1, 0x87, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA1, 0xB4, 0xE2, 0xA2,
        0x8B, 0xE2, 0xA1, 0xA3, 0xE2, 0xA0, 0x8A, 0xE2, 0xA1, 0xA9, 0xE2, 0xA0, 0x8B, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0xA3, 0xE2, 0xA1, 0x89, 0xE2, 0xA0, 0xB2, 0xE2,
        0xA3, 0x84, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x99, 0xE2, 0xA2, 0x86, 0xE2, 0xA0, 0x80, 0xE2, 0xA0,
        0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA3, 0xB8, 0xE2, 0xA0, 0x80, 0xE2, 0xA2, 0x89, 0xE2, 0xA0, 0x80,
        0xE2, 0xA2, 0x80, 0xE2, 0xA0, 0xBF, 0xE2, 0xA0, 0x80, 0xE2, 0xA2, 0xB8, 0x0A, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0xB8, 0xE2, 0xA1, 0x84, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x88, 0xE2,
        0xA2, 0xB3, 0xE2, 0xA3, 0x84, 0xE2, 0xA1, 0x87, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA2,
        0x80, 0xE2, 0xA1, 0x9E, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x88, 0xE2, 0xA0, 0x80, 0xE2, 0xA2, 0x80,
        0xE2, 0xA3, 0xB4, 0xE2, 0xA3, 0xBE, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2,
        0xA3, 0xBF, 0xE2, 0xA3, 0xA6, 0xE2, 0xA1, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0,
        0x80, 0xE2, 0xA0, 0x88, 0xE2, 0xA2, 0xA7, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA2, 0xB3,
        0xE2, 0xA3, 0xB0, 0xE2, 0xA0, 0x81, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA3, 0xA0, 0xE2, 0xA0, 0x83, 0x0A, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x98, 0xE2, 0xA2, 0x84, 0xE2, 0xA3, 0x80, 0xE2, 0xA3, 0xB8, 0xE2, 0xA0, 0x83, 0xE2, 0xA0,
        0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA1, 0xB8, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0x80, 0xE2, 0xA2, 0xA0, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2,
        0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3,
        0xBF, 0xE2, 0xA3, 0x86, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x88,
        0xE2, 0xA3, 0x87, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x99, 0xE2, 0xA2, 0x84, 0xE2,
        0xA3, 0x80, 0xE2, 0xA0, 0xA4, 0xE2, 0xA0, 0x9A, 0xE2, 0xA0, 0x81, 0xE2, 0xA0, 0x80, 0x0A, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0,
        0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80,
        0xE2, 0xA1, 0x87, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA2, 0xA0, 0xE2, 0xA3, 0xBF, 0xE2,
        0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3,
        0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA1, 0x84,
        0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA2, 0xB9, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0,
        0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0x0A, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0,
        0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA1, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3,
        0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF,
        0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA1, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA2, 0x98, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0,
        0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80,
        0x0A, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA1, 0x87, 0xE2, 0xA0, 0x80, 0xE2, 0xA2, 0xB0, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3,
        0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA1, 0xBF, 0xE2, 0xA0, 0x9B, 0xE2, 0xA0, 0x81, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0x89, 0xE2, 0xA0, 0x9B, 0xE2, 0xA2, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2,
        0xA3, 0xBF, 0xE2, 0xA3, 0xA7, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA3, 0xBC, 0xE2, 0xA0,
        0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0x0A, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA2, 0xA0, 0xE2, 0xA1,
        0x80, 0xE2, 0xA3, 0xB8, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA0, 0x9F, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA2, 0xBB, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA3, 0xBF, 0xE2, 0xA1,
        0x80, 0xE2, 0xA2, 0x80, 0xE2, 0xA0, 0x87, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0x0A, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0,
        0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x98, 0xE2, 0xA1, 0x87, 0xE2, 0xA0, 0xB9, 0xE2, 0xA0, 0xBF,
        0xE2, 0xA0, 0x8B, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0,
        0x99, 0xE2, 0xA2, 0xBF, 0xE2, 0xA1, 0xBF, 0xE2, 0xA0, 0x81, 0xE2, 0xA1, 0x8F, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0x0A, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0,
        0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0xBB, 0xE2, 0xA3, 0xA4, 0xE2, 0xA3, 0x9E, 0xE2, 0xA0, 0x81, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0,
        0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA2, 0xA2, 0xE2, 0xA3, 0x80,
        0xE2, 0xA3, 0xA0, 0xE2, 0xA0, 0x87, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0,
        0x80, 0xE2, 0xA0, 0x80, 0x0A, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0,
        0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80,
        0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x99, 0xE2, 0xA0, 0xB2, 0xE2, 0xA2, 0xA4, 0xE2, 0xA3, 0x80, 0xE2, 0xA3,
        0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA2, 0x80, 0xE2, 0xA3, 0x80, 0xE2, 0xA3, 0x80, 0xE2, 0xA0, 0xA4,
        0xE2, 0xA0, 0x92, 0xE2, 0xA0, 0x89, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2,
        0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0,
        0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0xE2, 0xA0, 0x80, 0x0A, 0x00
    };

    constexpr std::uint8_t important_message_ascii[]
    {
        0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30,
        0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30,
        0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37,
        0x6D, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30,
        0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20,
        0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x1B,
        0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37,
        0x6D, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30,
        0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30,
        0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37,
        0x6D, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20,
        0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20,
        0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A,
        0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A,
        0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x1B,
        0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30,
        0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x1B,
        0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0x20,
        0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20,
        0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37,
        0x6D, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A,
        0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x1B,
        0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x1B,
        0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30,
        0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20,
        0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37,
        0x6D, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x1B,
        0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x1B,
        0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30,
        0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x1B,
        0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37,
        0x6D, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x1B,
        0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A,
        0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20,
        0x20, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37,
        0x6D, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x1B,
        0x5B, 0x30, 0x6D, 0x0A, 0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96,
        0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0xE2, 0x96, 0x88,
        0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2, 0x96, 0x88, 0xE2,
        0x96, 0x88, 0xE2, 0x96, 0x88, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A,
        0x1B, 0x5B, 0x33, 0x30, 0x3B, 0x31, 0x30, 0x37, 0x6D, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x1B, 0x5B, 0x30, 0x6D, 0x0A, 0x00
    };

    std::pair<std::string_view, std::string_view> very_important_message()
    {
        return {
            reinterpret_cast<const char *>(important_message_unicode),
            reinterpret_cast<const char *>(important_message_ascii)
        };
    }
} // namespace
#endif