// last modified 

#include "Hanz2Piny.h"
#include "./utf8/utf8.h"
#include <iostream>
#include <algorithm>
#include <fstream>
#include <iterator>


using std::vector;
using std::string;
using std::ifstream;
using std::endl;
using std::cerr;
using std::istream_iterator;
using std::pair;



Hanz2Piny::Hanz2Piny ()
{
    ;
}

// 只处理 unicode 在 [0X4E00, 0X9FA5] 间的汉字，检查是否在此区间。
bool
Hanz2Piny::isHanziUnicode (const Unicode unicode) const
{
    return((begin_hanzi_unicode_ <= unicode) && (unicode <= end_hanzi_unicode_));
}

// 基于 unicode 查找对应汉字的拼音，若为多音字则可能返回多个拼音
vector<string>
Hanz2Piny::toPinyinFromUnicode (const Unicode hanzi_unicode, const bool with_tone) const
{
    if (!isHanziUnicode(hanzi_unicode)) {
        return(vector<string>());
    }

    const char* polyphone_with_tone = pinyin_list_with_tone_[hanzi_unicode - begin_hanzi_unicode_];

    // 查找该 unicode 对应汉字的拼音（带声调），若为多音字则多个拼音
    vector<string> polyphone_list_with_tone;
    static const char split_c = ' ';
    const char* p = polyphone_with_tone;
    const char* p_old = p;
    while ('\0' != *p) {
        if (split_c == *p) {
            polyphone_list_with_tone.emplace_back(p_old, p);
            p_old = p + 1;
        }
        ++p;
    }
    polyphone_list_with_tone.emplace_back(p_old, p);
    if (with_tone) {
        return(polyphone_list_with_tone);
    }

    // 不带声调
    vector<string> polyphone_list_without_tone;
    for (const auto& polyphone : polyphone_list_with_tone) {
        const auto& iter = find_if(polyphone.cbegin(), polyphone.cend(), isdigit);
        polyphone_list_without_tone.emplace_back(polyphone.cbegin(), iter);
    }
    return(polyphone_list_without_tone);
}

// 检查字符序列是否为 UTF-8 编码格式
bool
Hanz2Piny::isUtf8 (const string& s) const
{
    return(utf8::is_valid(s.cbegin(), s.cend()));
}

// 基于 UTF-8 查找对应汉字的拼音。
// s 为 UTF-8 编码的字符序列，可以包含多个汉字。处理完每个汉字后，返回
// pair<bool, vector<string>>，first 表示该汉字是否成功转换拼音，second
// 该汉字具体转换的拼音列表（多音字）。
// 非汉字不作任何修改，原样返回。
vector<pair<bool, vector<string>>>
Hanz2Piny::toPinyinFromUtf8 ( const std::string& s,
                              std::map<uint16_t, uint16_t> numMap,
                              const bool with_tone,
                              const bool replace_unknown,
                              const std::string& replace_unknown_with) const
{
    vector<pair<bool, vector<string>>> pinyin_list_list;

    if (!isUtf8(s)) {
        static const auto null_pair = make_pair(false, vector<string>());
        pinyin_list_list.emplace_back(null_pair);
        return(pinyin_list_list);
    }

    for (auto iter = s.cbegin(); iter != s.cend(); NULL) {
        // 将单个字符（可能是汉字，也可能非汉字）的 UTF-8 转换为 unicode
        auto iter_old = iter;
        utf8::next(iter, s.cend());
        Unicode unicode;
        utf8::utf8to16(iter_old, iter, &unicode);

        if(numMap.find((int16_t)unicode)!=numMap.end())
        {
           // std::map<uint16_t, uint16_t> mapTmp;
            unicode = numMap[(uint16_t)unicode];
        }

        // 获取字符对应拼音（如果有的话），存在多音字所以可能返回多个拼音
        vector<string> pinyin_list;
        pinyin_list = toPinyinFromUnicode(unicode, with_tone);
        
        // 对于无法转换出对应拼音的字符，要么返回指定字符串，要么原样返回
        bool ok = !pinyin_list.empty();
        if (!ok) {
            if (replace_unknown) {
                pinyin_list.emplace_back(replace_unknown_with);
            } else {
                pinyin_list.emplace_back(iter_old, iter); 
            }
        }
        
        pinyin_list_list.push_back(make_pair(ok, pinyin_list));
    }

    return(pinyin_list_list);
}

// 检查指定文件是否为 UTF-8 编码
bool
Hanz2Piny::isUtf8File(const string& file_path) const
{
    ifstream ifs(file_path);
    if (!ifs.is_open()) {
        //cerr << "WARNING! fail to open " << file_path << endl;
        return(false);
    }

    return(utf8::is_valid(istream_iterator<char>(ifs), istream_iterator<char>()));
}

// 检查指定字符序列是否以 BOMC 打头
bool
Hanz2Piny::isStartWithBom (const string& s) const
{
    return(utf8::starts_with_bom(s.cbegin(), s.cend()));
}

const Hanz2Piny::Unicode Hanz2Piny::begin_hanzi_unicode_ = 0X4E00, Hanz2Piny::end_hanzi_unicode_ = 0X9FA5;

// 0）基于 unicode 顺序的列表；
// 1）即便某个 unicode 无对应汉字拼音，也需要保留其位置；
// 2）多音字的多个拼音用空格分割，不能有多于的其他空格；
// 3）姓氏异读位于第一个；
const char* Hanz2Piny::pinyin_list_with_tone_[] {
"yi1",
"ding1",
"kao3",
"qi1",
"shang4",
"xia4",
"xx5",
"wan4,mo4",
"zhang4",
"san1",
"shang4,shang3",
"xia4",
"ji1,qi2",
"bu4",
"yu3,yu2,yu4",
"mian3",
"gai4",
"chou3",
"chou3",
"zhuan1",
"qie3",
"pi1",
"shi4",
"shi4",
"qiu1",
"bing3",
"ye4",
"cong2",
"dong1",
"si1",
"cheng2",
"diu1",
"qiu1",
"liang3",
"diu1",
"you3",
"liang3",
"yan2",
"bing4",
"sang4,sang1",
"gun3,shu4",
"jiu1",
"ge4",
"ya1",
"qiang2",
"zhong1,zhong4",
"ji3",
"jie4",
"feng1",
"guan4,kuang4",
"chuan4",
"chan3",
"lin2",
"zhuo2",
"zhu3",
"ba1,xx5",
"wan2",
"dan1",
"wei4,wei2",
"zhu3",
"jing3,dong4",
"li4,li2",
"ju3",
"pie3",
"fu2,pie3",
"yi2",
"yi4",
"nai3",
"wu3",
"jiu3",
"jiu3",
"tuo1,zhe2",
"me5,ma2,ma5",
"yi4",
"xx5",
"zhi1",
"wu1",
"zha4",
"hu1",
"fa2",
"le4,yue4",
"zhong4",
"ping1",
"pang1",
"qiao2",
"hu3",
"guai1",
"cheng2",
"cheng2,sheng4",
"yi3,zhe2",
"ya4",
"zhe2",
"mie1",
"jiu3",
"qi3",
"ye3",
"xi2",
"xiang1",
"gai4",
"jiu3",
"xx5",
"xx5",
"shu1",
"xx5",
"shi3",
"ji1",
"nang1",
"xx5",
"ju4",
"shi2",
"mao3",
"hu1",
"mai3",
"luan4",
"zi1",
"ru3",
"xue2",
"yan3",
"fu3",
"xx5",
"na3",
"gan1",
"xx5",
"yu2",
"xx5",
"zhe3",
"qian2",
"zhi4,luan4",
"gui1",
"gan1",
"luan4",
"lin2",
"yi4",
"jue2",
"le5,liao3,liao4",
"ge4",
"yu3,yu2",
"zheng1",
"shi4",
"shi4",
"er4",
"chu4",
"yu2",
"kui1",
"yu2",
"yun2",
"hu4",
"qi2",
"wu3",
"jing3",
"si4",
"sui4",
"gen4",
"gen4",
"ya4",
"xie1",
"ya4",
"qi2",
"ya4",
"ji2,qi4",
"tou2",
"wang2",
"kang4",
"da4",
"jiao1",
"hai4",
"yi4",
"chan3",
"heng1",
"mu3",
"xx5",
"xiang3",
"jing1",
"ting2",
"liang4",
"xiang3",
"jing1",
"ye4",
"qin1,qing4",
"bo2",
"you4",
"xie4",
"dan3",
"lian2",
"duo3",
"wei3,men2",
"ren2",
"ren2",
"ji2",
"ji2,xx5",
"wang2",
"yi4",
"shen2,shi2",
"ren2",
"le4",
"ding1",
"ze4",
"jin3",
"pu1,pu2",
"chou2,qiu2",
"ba1",
"zhang3",
"jin1",
"jie4",
"bing1",
"reng2",
"cong2",
"fo2",
"san3",
"lun2",
"xx5",
"cang1",
"zai3,zi1,zi3",
"shi4",
"ta1",
"zhang4",
"fu4",
"xian1",
"xian1",
"tuo1",
"hong2",
"tong2",
"ren4",
"qian1",
"gan3",
"ge1,yi4",
"bo2",
"dai4",
"ling4,ling2,ling3",
"yi3",
"chao4",
"chang2",
"sa1",
"chang2",
"yi2",
"mu4",
"men5",
"ren4",
"jia3",
"chao4",
"yang3",
"qian2",
"zhong4",
"pi3",
"wo4",
"wu3",
"jian4",
"jia4,jie5",
"yao3",
"feng1",
"cang1",
"ren4,ren2",
"wang2",
"fen4",
"di1",
"fang3",
"zhong1",
"qi3",
"pei4",
"yu2",
"diao4",
"dun4",
"wu4",
"yi4",
"xin3",
"kang4",
"yi1",
"ji2",
"ai4",
"wu3",
"ji4",
"fu2",
"fa2",
"xiu1",
"jin4",
"pi1",
"dan3",
"fu1",
"tang3",
"zhong4",
"you1",
"huo3",
"hui4,kuai4",
"yu3",
"cui4",
"yun2,chuan2",
"san3",
"wei3",
"chuan2,zhuan4",
"che1",
"ya2",
"qian4",
"shang1",
"chang1",
"lun2",
"cang1",
"xun4",
"xin4",
"wei3",
"zhu4",
"ze5",
"xian2",
"nu3",
"bo2,ba4,bai3",
"gu1,gu4",
"ni3",
"ni4,ni3",
"xie4",
"ban4",
"xu4",
"ling2",
"zhou4",
"shen1",
"qu1",
"ci4,si4",
"beng1",
"shi4,si4",
"jia1",
"pi1",
"yi4",
"si4",
"yi3",
"zheng1",
"dian4,tian2",
"han1",
"mai4",
"dan4",
"zhu4",
"bu4",
"qu1",
"bi3",
"zhao1",
"ci3",
"wei4",
"di1",
"zhu4",
"zuo3",
"you4",
"yang3",
"ti3",
"zhan4",
"he2",
"bi4",
"tuo2",
"she2",
"yu2",
"yi4,die2",
"fu2,fo2",
"zuo4,zuo1",
"gou1,kou4",
"ning4",
"tong2",
"ni3",
"xian1",
"qu2",
"yong1,yong4",
"wa3",
"qian1",
"shi5",
"ka3",
"bao1",
"pei4",
"huai2",
"he4",
"lao3",
"xiang2",
"ge2",
"yang2",
"bai3",
"fa3",
"ming3",
"jia1",
"er4",
"bing4",
"ji2",
"hen3",
"huo2",
"gui3",
"quan2",
"tiao1",
"jiao3",
"ci4",
"yi4",
"shi3",
"xing2",
"shen1",
"tuo1",
"kan3",
"zhi2",
"gai1",
"lai2",
"yi2",
"chi3",
"kua3",
"guang1",
"li4",
"yin1",
"shi4",
"mi3",
"zhu1",
"xu4",
"you4",
"an1",
"lu4",
"mou2",
"er2",
"lun2",
"dong4,tong2",
"cha4",
"chi1",
"xun4",
"gong1,gong4",
"zhou1",
"yi1",
"ru2",
"cun2",
"xia2",
"jia4",
"dai4",
"lu:3",
"ta5",
"jiao3",
"zhen1",
"ce4,zhai1",
"qiao2",
"kuai4",
"chai2",
"ning4",
"nong2",
"jin3",
"wu3",
"hou2",
"jiong3",
"cheng3",
"zhen4",
"zuo4",
"chou3",
"qin1",
"lv3",
"ju2",
"shu4",
"ting3",
"shen4",
"tui4",
"bo2",
"nan2",
"xiao1",
"bian4,pian2",
"tui3",
"yu3",
"xi4",
"cu4",
"e2",
"qiu2",
"xu2",
"guang4",
"ku4",
"wu2",
"jun4,zun4",
"yi4",
"fu3",
"liang2",
"zu3",
"qiao4",
"li4",
"yong3",
"hun4",
"jing4",
"qian4",
"san4",
"pei3",
"su2",
"fu2",
"xi1",
"li3",
"mian3",
"ping1",
"bao3",
"yu2,shu4",
"qi2,si4",
"xia2",
"xin4",
"xiu1",
"yu3",
"di4",
"che1",
"chou2",
"zhi4",
"yan3",
"lia3",
"li4",
"lai2",
"si1",
"jian3",
"xiu1",
"fu3",
"huo4",
"ju4",
"xiao4",
"pai2",
"jian4",
"biao3",
"chu4",
"fei4",
"feng4",
"ya4",
"an3",
"bei4",
"yu4",
"xin1",
"bi3",
"hu3",
"chang1",
"zhi1",
"bing4",
"jiu4",
"yao2",
"cui4",
"lia3",
"wan3",
"lai2",
"cang1",
"zong4",
"ge4",
"guan1",
"bei4",
"tian3",
"shu1",
"shu1",
"men5",
"dao4,dao3",
"tan2",
"jue2",
"chui2",
"xing4",
"peng2",
"tang3,chang2",
"hou4",
"yi3",
"qi1",
"ti4",
"gan4",
"jing4,liang4",
"jie4",
"sui1",
"chang4",
"jie2",
"fang3",
"zhi2",
"kong1,kong3",
"juan4",
"zong1",
"ju4",
"qian4",
"ni2",
"lun2",
"zhuo1",
"wo1",
"luo3",
"song1",
"leng4",
"hun4",
"dong1",
"zi4",
"ben4",
"wu3",
"ju4",
"nai3",
"cai3",
"jian3",
"zhai4",
"ye1",
"zhi2",
"sha2",
"qing1",
"ning4",
"ying1",
"cheng1",
"qian2",
"yan3",
"ruan3",
"zhong4",
"chun3",
"jia3,gei1,jia4",
"ji4,jie2",
"wei3",
"yu3",
"bing4",
"ruo4",
"ti2",
"wei1",
"pian1",
"yan4",
"feng1",
"tang3",
"wo4",
"e4",
"xie2",
"che3",
"sheng3",
"kan3",
"di4",
"zuo4",
"cha1",
"ting2",
"bei4",
"xie4",
"huang2",
"yao3",
"zhan4",
"chou3",
"yan1",
"you2",
"jian4",
"xu3",
"zha1",
"ci1",
"fu4",
"bi1",
"zhi4",
"zong3",
"mian3",
"ji2",
"yi3",
"xie4",
"xun2",
"si1",
"duan1",
"ce4",
"zhen1",
"ou3",
"tou1",
"tou1",
"bei4",
"za2",
"lou2,lu:3",
"jie2",
"wei3",
"fen4",
"chang2",
"gui1,kui3",
"sou3",
"zhi4",
"su4",
"xia1",
"fu4",
"yuan4",
"rong3",
"li4",
"nu4",
"yun4",
"jiang3",
"ma4",
"bang4",
"dian1",
"tang2",
"hao4",
"jie2",
"xi1",
"shan1",
"qian4",
"jue2",
"cang1",
"chu4",
"san3",
"bei4",
"xiao4",
"yong3",
"yao2",
"tan4",
"suo1",
"yang3",
"fa2",
"bing4",
"jia1",
"dai3",
"zai4",
"tang3",
"gu3",
"bin1",
"chu3",
"nuo2",
"can1",
"lei3",
"cui1",
"yong1",
"zao1",
"zong3",
"beng1",
"song3",
"ao4",
"chuan2",
"yu3",
"zhai4",
"zu2",
"shang1",
"chuang3",
"jing4",
"chi4",
"sha3",
"han4",
"zhang1",
"qing1",
"yan4,yan1",
"di4",
"xie4",
"lou2",
"bei4",
"piao4",
"jin3",
"lian4",
"lu4",
"man2",
"qian1",
"xian1",
"tan4",
"ying2",
"xx5",
"zhuan4",
"xiang4",
"shan4",
"qiao2",
"jiong3",
"tui3",
"zun3",
"pu2",
"xi1",
"lao2",
"chang3",
"guang1",
"liao2",
"qi1",
"deng4",
"chan2",
"wei3",
"ji1",
"bo1",
"hui4",
"chuan3",
"jian4",
"dan4",
"jiao3",
"jiu4",
"seng1",
"fen4",
"xian4",
"ju2",
"e4",
"jiao1",
"jian4",
"tong2,zhuang4",
"lin3",
"bo2",
"gu4",
"xian1",
"su4",
"xian4",
"jiang1",
"min3",
"ye4",
"jin4",
"jia4",
"qiao4",
"pi4",
"feng1",
"zhou4",
"ai4",
"sai4",
"yi2",
"jun4",
"nong2",
"chan2",
"yi4",
"dang1",
"jing3",
"xuan1",
"kuai4",
"jian3",
"chu4",
"dan1",
"jiao3",
"sha3",
"zai4",
"can4",
"bin1",
"an2",
"ru2",
"tai2",
"chou2",
"chai2",
"lan2",
"ni3",
"jin3",
"qian4",
"meng2",
"wu3",
"ning2",
"qiong2",
"ni3",
"chang2",
"lie4",
"lei3",
"lv3",
"kuang3",
"bao4",
"yu4",
"biao1",
"zan3",
"zhi4",
"si4",
"you1",
"hao2",
"qing4",
"chen4",
"li4",
"teng2",
"wei3",
"long3",
"chu3",
"chan4",
"rang2",
"shu1",
"hui4",
"li4",
"luo2",
"zan3",
"nuo2",
"tang3",
"yan3",
"lei3",
"nang4",
"er2,ren2,r5",
"wu4",
"yun3",
"zan1",
"yuan2",
"xiong1",
"chong1",
"zhao4",
"xiong1",
"xian1",
"guang1",
"dui4",
"ke4,kei1",
"dui4",
"mian3,wen4",
"tu4",
"chang2",
"er2",
"dui4",
"er2",
"jin1",
"tu4",
"si4",
"yan3",
"yan3",
"shi3",
"shi2 ke4",
"dang3",
"qian1 ke4",
"dou1",
"fen1 ke4",
"hao2 ke4",
"shen1",
"dou1",
"bai3 ke4",
"jing1",
"li2 ke4",
"huang3",
"ru4",
"wang2",
"nei4",
"quan2",
"liang3",
"yu2",
"ba1",
"gong1",
"liu4",
"xi1",
"han5",
"lan2",
"gong4",
"tian1",
"guan1",
"xing4,xing1",
"bing1",
"qi2",
"ju4",
"dian3",
"zi1",
"fen1",
"yang3",
"jian1",
"shou4",
"ji4",
"yi4",
"ji4",
"chan3",
"jiong1",
"mao4",
"ran3",
"nei4",
"yuan2",
"mao3",
"gang1",
"ran3",
"ce4",
"jiong1",
"ce4",
"zai4",
"gua3",
"jiong3",
"mao4",
"zhou4",
"mao4",
"gou4",
"xu3",
"mian3",
"mi4",
"rong3",
"yin2",
"xie3",
"kan3",
"jun1",
"nong2",
"yi2",
"mi2",
"shi4",
"guan1,guan4",
"meng2",
"zhong3",
"ju4",
"yuan1",
"ming2",
"kou4",
"lin2",
"fu4",
"xie3",
"mi4",
"bing1",
"dong1",
"tai4",
"gang1",
"feng2,ping2",
"bing1",
"hu4",
"chong1,chong4",
"jue2",
"hu4",
"kuang4",
"ye3",
"leng3",
"pan4",
"fu2",
"min3",
"dong4",
"xian3",
"lie4",
"qia4",
"jian1",
"jing4",
"sou1",
"mei3",
"tu2",
"qi1",
"gu4",
"zhun3",
"song1",
"jing4",
"liang2,liang4",
"jing4",
"diao1",
"ling2",
"dong4",
"gan4",
"jian3",
"yin1",
"cou4",
"ai2",
"li4",
"cang1",
"ming3",
"zhun3",
"cui1",
"si1",
"duo2",
"jin4",
"lin3",
"lin3",
"ning2",
"xi1",
"du2",
"ji3,ji1",
"fan2",
"fan2",
"fan2",
"feng4",
"ju1",
"chu3",
"zheng1",
"feng1",
"mu4",
"zhi3",
"fu2",
"feng1",
"ping2",
"feng1",
"kai3",
"huang2",
"kai3",
"gan1",
"deng4",
"ping2",
"kan3",
"xiong1",
"kuai4",
"tu1",
"ao1,wa1",
"chu1",
"ji1",
"dang4",
"han2",
"han2",
"zao2",
"dao1",
"diao1",
"dao1",
"ren4",
"ren4",
"chuang1",
"fen1,fen4",
"qie4,qie1",
"yi4",
"ji1",
"kan1",
"qian4",
"cun3",
"chu2",
"wen3",
"ji1",
"dan3",
"xing2",
"hua4,hua2",
"wan2",
"jue2",
"li2",
"yue4",
"lie4",
"liu2",
"ze2",
"gang1",
"chuang4,chuang1",
"fu2",
"chu1",
"qu4",
"diao1",
"shan1",
"min3",
"ling2",
"zhong1",
"pan4",
"bie2",
"jie2",
"jie2",
"pao2,bao4",
"li4",
"shan1",
"bie2,bie4",
"chan3",
"jing3",
"gua1",
"geng1",
"dao4",
"chuang4",
"kui1",
"ku1",
"duo4",
"er4",
"zhi4",
"shua1,shua4",
"quan4",
"sha1,cha4",
"ci4,ci1",
"ke4",
"jie2",
"gui4",
"ci4",
"gui4",
"kai3",
"duo4",
"ji4",
"ti4",
"jing3",
"lou2",
"luo3",
"ze2",
"yuan1",
"cuo4",
"xue1,xiao1",
"ke4",
"la2,la4",
"qian2",
"sha1",
"chuang4",
"gua3",
"jian4",
"cuo4",
"li2",
"ti1",
"fei4",
"pou1",
"chan3",
"qi2",
"chuang4",
"zi4",
"gang1",
"wan1",
"bo1",
"ji1",
"duo1",
"qing2",
"shan4,yan3",
"du1",
"jian4",
"ji4",
"bo1,bao1",
"yan1",
"ju4",
"huo1",
"sheng4",
"jian3",
"duo2",
"duan1",
"wu1",
"gua3",
"fu4",
"sheng4",
"jian4",
"ge1",
"da2",
"kai3",
"chuang4",
"chuan1",
"chan3",
"tuan2",
"lu4",
"li2",
"peng3",
"shan1",
"piao1",
"kou1",
"jiao3,chao1",
"gua1",
"qiao1",
"jue2",
"hua4",
"zha1",
"zhuo2",
"lian2",
"ju4",
"pi1,pi3",
"liu2",
"gui4",
"jiao3",
"gui4",
"jian4",
"jian4",
"tang1",
"huo1",
"ji4",
"jian4",
"yi4",
"jian4",
"zhi4",
"chan2",
"jian3",
"mo2",
"li2",
"zhu2",
"li4",
"ya4",
"quan4",
"ban4",
"gong1",
"jia1",
"wu4",
"mai4",
"lie4",
"jin4",
"keng1",
"xie2,lie4",
"zhi3",
"dong4",
"zhu4",
"nu3",
"jie2",
"qu2",
"shao4",
"yi4",
"zhu1",
"mo4",
"li4",
"jin4,jing4",
"lao2",
"lao2",
"juan4",
"kou3",
"yang2",
"wa1",
"xiao4",
"mou2",
"kuang1",
"jie2",
"lie4",
"he2",
"shi4",
"ke4",
"jin4",
"gao4",
"bo2",
"min3",
"chi4",
"lang2",
"yong3",
"yong3",
"mian3",
"ke4",
"xun1",
"juan4",
"qing2",
"lu4",
"bu4",
"meng3",
"chi4",
"lei1,le4",
"kai4",
"mian3",
"dong4",
"xu4",
"xu4",
"kan1",
"wu4",
"yi4",
"xun1",
"weng3",
"sheng4",
"lao2",
"mu4",
"lu4",
"piao4",
"shi4",
"ji1",
"qin2",
"jiang4",
"chao1",
"quan4",
"xiang4",
"yi4",
"jue2",
"fan1",
"juan1",
"tong2",
"ju4",
"dan1",
"xie2",
"mai4",
"xun1",
"xun1",
"lv4",
"li4",
"che4",
"rang2",
"quan4",
"bao1",
"shao2",
"yun2",
"jiu1",
"bao4",
"gou1,gou4",
"wu4",
"yun2",
"xx5",
"xiong1",
"gai4",
"gai4",
"bao1",
"cong1",
"yi4",
"xiong1",
"peng1",
"ju1",
"tao2",
"ge2",
"pu2",
"e4",
"pao2",
"fu2",
"gong1",
"da2",
"jiu4",
"gong1",
"bi3",
"hua4,hua1",
"bei3",
"nao3",
"chi2",
"fang1",
"jiu4",
"yi2",
"za1",
"jiang4",
"kang4",
"jiang4",
"kuang1",
"hu1",
"xia2",
"qu1",
"fan2",
"gui3",
"qie4",
"zang1",
"kuang1",
"fei3",
"hu1",
"yu3",
"gui3",
"kui4,gui4",
"hui4",
"dan1",
"gui4",
"lian2",
"lian2",
"suan3",
"du2",
"jiu4",
"jue2",
"xi4",
"pi3,pi1",
"qu1,ou1",
"yi1",
"ke1",
"yan3",
"bian3,pian2",
"ni4",
"qu1",
"shi2",
"xun4",
"qian1",
"nian4",
"sa4",
"zu2",
"sheng1",
"wu3",
"hui4",
"ban4",
"shi4",
"xi4",
"wan4",
"hua2,hua4,hua1",
"xie2",
"wan4",
"bei1",
"zu2,cu4",
"zhuo2",
"xie2",
"dan1,shan4",
"mai4",
"nan2",
"dan1",
"ji2",
"bo2",
"shuai4,lu:4",
"bo5,bu3",
"kuang4,guan4",
"bian4",
"bu3",
"zhan4,zhan1",
"ka3,qia3",
"lu2",
"you3",
"lu3",
"xi1",
"gua4",
"wo4",
"xie4",
"jie2",
"jie2,xx5",
"wei4",
"ang2",
"qiong2",
"zhi1",
"mao3",
"yin4",
"wei1",
"shao4",
"ji2",
"que4",
"luan3",
"chi3",
"juan3,juan4",
"xie4",
"xu4",
"jin3",
"que4",
"wu4",
"ji2",
"e4",
"qing1",
"xi1",
"san1",
"chang3,han3",
"wei3",
"e4",
"ting1",
"li4",
"zhe2",
"han3",
"li4",
"ya3",
"ya1,ya4",
"yan4",
"she4",
"zhi3",
"zha3",
"pang2",
"ya2",
"qie4",
"ya2",
"zhi4",
"ce4,si4",
"pang2",
"ti2",
"li2",
"she4",
"hou4",
"ting1",
"zui1",
"cuo4",
"fei4",
"yuan2",
"ce4",
"yuan2",
"xiang1",
"yan3",
"li4",
"jue2",
"sha4,xia4",
"dian1",
"chu2",
"jiu4",
"jin3,qin2",
"ao2",
"gui3",
"yan4",
"si1",
"li4",
"chang3",
"lan2",
"li4",
"yan2",
"yan3",
"yuan2",
"si1,mou3",
"gong1,hong2",
"lin2",
"rou2,qiu2",
"qu4",
"qu4",
"xx5",
"lei3",
"du1",
"xian4",
"zhuan1",
"san1",
"can1,shen1",
"can1",
"san1",
"can1",
"ai4",
"dai4",
"you4",
"cha1,cha2,cha3",
"ji2",
"you3",
"shuang1",
"fan3",
"shou1",
"guai4",
"ba2",
"fa1,fa4",
"ruo4",
"shi4",
"shu1",
"zhuo2",
"qu3",
"shou4",
"bian4",
"xu4",
"jia3",
"pan4",
"sou3",
"ji2",
"wei4",
"sou3",
"die2",
"rui4",
"cong2",
"kou3",
"gu3",
"ju4,gou1",
"ling4",
"gua3",
"dao1,tao1",
"kou4",
"zhi3,zhi1",
"jiao4",
"zhao4,shao4",
"ba1",
"ding1",
"ke3,ke4",
"tai2",
"chi4",
"shi3",
"you4",
"qiu2",
"po3",
"ye4,xie2",
"hao4,hao2",
"si1",
"tan4",
"chi3",
"le4",
"diao1",
"ji1",
"liao3",
"hong1",
"mie1",
"xu1,yu4",
"mang2",
"chi1",
"ge4",
"xuan1,song4",
"yao1",
"zi3",
"he2,ge3",
"ji2",
"diao4",
"cun4",
"tong2,tong4",
"ming2",
"hou4",
"li4",
"tu3,tu4",
"xiang4",
"zha1,zha4",
"xia4,he4",
"ye1",
"lu:3",
"a1",
"ma5,ma3",
"ou3",
"huo1",
"yi1",
"jun1",
"chou3",
"lin4",
"tun1",
"yin2",
"fei4",
"bi3",
"qin4",
"qin4",
"jie4",
"bu4",
"fou3,pi3",
"ba5,ba1,bia1",
"dun1",
"fen1",
"e2",
"han2",
"ting1,yin3,ting4",
"keng1,hang2",
"shun3",
"qi3",
"hong2",
"zhi1,zi1",
"yin3",
"wu2",
"wu2",
"chao3",
"na4",
"xue4",
"xi1",
"chui1",
"dou1",
"wen3",
"hou3",
"hong1,hou3",
"wu2",
"gao4",
"ya5",
"jun4",
"lv3",
"e4",
"ge2",
"mei2",
"dai1",
"qi3",
"cheng2",
"wu2",
"gao4",
"fu1",
"jiao4",
"hong1",
"chi3",
"sheng1",
"na4,na5",
"tun1",
"fu3,m2",
"yi4",
"tai1",
"ou3",
"li4",
"bei5,bai4",
"yuan2",
"guo1,wai1",
"wen5",
"qiang1,qiang4",
"wu1",
"e4",
"shi1",
"juan3",
"pen1",
"wen3",
"ne5,ni2",
"m2,m4",
"ling2",
"ran2",
"you1",
"di3",
"zhou1",
"shi4",
"zhou4",
"tie4,zhan1",
"xi4",
"yi4",
"qi4",
"ping2",
"zi3",
"gu1",
"ci1,zi1",
"wei4",
"xu1",
"he1,a1",
"nao2,nu3",
"xia1",
"pei1",
"yi4",
"xiao1",
"shen1",
"hu1",
"ming4",
"da2",
"qu1",
"ju3",
"gan1",
"za1",
"tuo1",
"duo1",
"pou3",
"pao2",
"bi4",
"fu2",
"yang1",
"he2",
"za3,ze2,zha4",
"he2,he4,hu2,huo2,huo4",
"hai1",
"jiu4",
"yong3",
"fu4",
"da1",
"zhou4",
"wa3",
"ka3",
"gu1",
"ka1",
"zuo3",
"bu4",
"long2",
"dong1",
"ning2",
"ta5",
"si1",
"xian4",
"huo4",
"qi4",
"er4",
"e4",
"guang1",
"zha4",
"xi4,die2",
"yi2",
"lie3,lie1,lie5",
"zi1",
"mie1",
"mi1",
"zhi3",
"yao3",
"ji1",
"zhou4",
"ge1,lo5,luo4",
"shu4",
"zan2,za2",
"xiao4",
"hai1,ke2",
"hui1",
"kua3",
"huai4",
"tao2",
"xian2",
"e4",
"xuan3",
"xiu1",
"guo1",
"yan4,yan1,ye4",
"lao3",
"yi1",
"ai1",
"pin3",
"shen3",
"tong2",
"hong1,hong3,hong4",
"xiong1",
"duo1",
"wa5,wa1",
"ha1,ha3",
"zai1",
"you4",
"di4",
"pai4",
"xiang3",
"ai1",
"gen2,hen3",
"kuang1",
"ya3,ya1",
"da1",
"xiao1",
"bi4",
"hui4,yue3",
"nian2",
"hua1,hua2",
"xing5",
"kuai4",
"duo3",
"fen1",
"ji4",
"nong2",
"mou1",
"yo1,yo5",
"hao4",
"yuan2",
"long4",
"pou3",
"mang2",
"ge1",
"o2,e2,o4,o5",
"chi1",
"shao4",
"li3,li5",
"na3,na5,nei3",
"zu2",
"he2",
"ku1",
"xiao4",
"xian4",
"lao2",
"bo1",
"zhe2",
"zha1",
"liang4",
"ba1",
"mie1",
"lie4",
"sui1",
"fu2",
"bu3,bu1,bu4",
"han1",
"heng1",
"geng3",
"shuo1",
"ge3",
"you4",
"yan4",
"gu1",
"gu1",
"bei5",
"han1",
"suo1",
"chun2",
"yi4",
"ai1,ai4",
"jia2",
"tu1",
"xian2",
"wan3",
"li4,li5",
"xi1",
"tang2",
"zuo4",
"qiu2",
"che1",
"wu2",
"zao4",
"ya3",
"dou1",
"qi3",
"di2",
"qin4",
"mai4",
"mo4",
"hong3",
"dou3",
"qu4",
"lao2,lao4",
"liang3",
"suo3",
"zao4",
"huan4",
"lang5",
"sha1",
"ji1",
"zu3",
"wo1",
"feng3",
"jin4,yin2",
"hu3",
"qi4",
"shou4",
"wei2,wei3",
"shua1",
"chang4",
"er2",
"li4",
"qiang4",
"an3",
"jie4",
"yo1,yo5",
"nian4",
"yu2",
"tian3",
"lai4",
"sha3",
"xi1",
"tuo4",
"hu1",
"ai2",
"zhou1",
"nou3",
"ken3",
"zhuo2",
"zhao4",
"shang1",
"di4",
"heng1",
"lin2",
"a5,a1,a2,a3,a4",
"cai3",
"xiang1",
"tun1",
"wu3",
"wen4",
"cui4",
"sha4",
"gu3",
"qi3",
"qi3",
"tao2",
"dan4",
"dan4",
"ye4",
"zi3",
"bi3",
"cui4",
"chuo4",
"he2",
"ya3",
"qi3",
"zhe2",
"fei1",
"liang3",
"xian2",
"pi2",
"sha2",
"la5,la1",
"ze2",
"ying1",
"gua4",
"pa1",
"zhe3",
"se4",
"zhuan4",
"nie4",
"guo1",
"luo1,luo5",
"yan1",
"di1",
"quan2",
"chan3,tan1",
"bo1,bo5",
"ding4",
"lang1",
"xiao4",
"ju2",
"tang2",
"chi4",
"ti2",
"an2",
"jiu1",
"dan4",
"ka1",
"yong2",
"wei4,wei2",
"nan2",
"shan4",
"yu4",
"zhe2",
"la3,la1",
"jie1",
"hou2",
"han3",
"die2",
"zhou1",
"chai2",
"wai1",
"nuo4,re3",
"yu4",
"yin1",
"za2",
"yao1",
"o1,wo5",
"mian3",
"hu2",
"yun3",
"chuan3",
"hui4",
"huan4",
"huan4",
"xi3",
"he1,he4",
"ji1",
"kui4",
"zhong3",
"wei2",
"sha4",
"xu3",
"huang2",
"du4",
"nie4",
"xuan1",
"liang4",
"yu4",
"sang4",
"chi1",
"qiao2",
"yan4",
"dan1",
"pen4",
"can1",
"li2",
"yo1",
"zha1,cha1",
"wei1",
"miao1",
"ying2",
"pen1,pen4",
"bu3",
"kui2",
"xi3",
"yu4",
"jie2",
"lou2,lou5",
"ku4",
"sao4",
"huo4",
"ti2",
"yao2",
"he4",
"a2,sha4",
"xiu4",
"qiang1",
"se4",
"yong1",
"su4",
"hong3",
"xie2",
"ai4",
"suo1",
"ma5",
"cha1",
"hai4",
"ke4",
"ta4",
"sang3",
"chen1",
"ru4",
"sou1",
"wa1",
"ji1",
"pang3",
"wu1",
"qian4",
"shi4",
"ge2",
"zi1",
"jie1",
"lao4",
"weng1",
"wa4",
"si4",
"chi1",
"hao2",
"suo1",
"",
"hai1",
"suo3",
"qin2",
"nie4",
"he1",
"zhi2",
"sai4",
"en1,en4,en5",
"ge3",
"na2",
"dia3",
"ai3,ai4",
"qiang1",
"tong1",
"bi4",
"ao2",
"ao2",
"lian2",
"zui1",
"zhe4",
"mo4",
"sou4",
"sou3",
"tan3",
"di1",
"qi1",
"jiao4",
"chong1",
"jiao1",
"ge3",
"tan4",
"shan1",
"cao2",
"jia1",
"ai2",
"xiao4",
"piao4,piao1",
"lou2",
"ga2",
"gu3,jia3",
"xiao1",
"hu1",
"hui4",
"guo1",
"ou3",
"xian1",
"ze2",
"chang2",
"xu1",
"po2",
"de1",
"ma5",
"ma4",
"hu2",
"lei5",
"du1",
"ga1",
"tang1",
"ye3",
"beng1",
"ying1",
"sai1",
"jiao4",
"mi4",
"xiao4",
"hua1",
"mai3",
"ran2",
"chuai4,zuo1",
"peng1",
"lao2",
"xiao4",
"ji1",
"zhu3",
"chao2,zhao1",
"kui4",
"zui3",
"xiao1",
"si1",
"hao2",
"fu3",
"liao2",
"qiao2",
"xi1",
"chu4",
"chan3",
"dan4",
"hei1",
"xun4",
"e3",
"zun3",
"fan1",
"chi1",
"hui1",
"zan3",
"chuang2",
"cu4",
"dan4",
"yu4",
"tun1",
"ceng1,cheng1",
"jiao4",
"ye1",
"xi1",
"qi4",
"hao2",
"lian2",
"xu1",
"deng1",
"hui1",
"yin2",
"pu1",
"jue1",
"qin2",
"xun2",
"nie4",
"lu1",
"si1",
"yan3",
"ying4",
"da1",
"zhan1",
"o1",
"zhou4",
"jin4",
"nong2",
"yue3",
"xie4",
"qi4",
"e4",
"zao4",
"yi1",
"shi4",
"jiao4",
"yuan4",
"ai1",
"yong1",
"jue2",
"kuai4",
"yu3",
"pen1",
"dao4",
"ga2",
"hen4",
"dun1",
"dang1",
"xin1",
"sai1",
"pi1",
"pi3",
"yin1",
"zui3",
"ning2",
"di2",
"lan4",
"ta4",
"huo4,o3",
"ru2",
"hao1",
"xia4",
"ye4",
"duo1",
"pi4",
"chou2",
"ji4",
"jin4",
"hao2",
"ti4",
"chang2",
"xun1",
"me1",
"ca1,cha1",
"ti4",
"lu3",
"hui4",
"bo2",
"you1",
"nie4",
"yin2",
"hu4",
"me5",
"hong1",
"zhe2",
"li4",
"liu2",
"hai5",
"nang2",
"xiao1",
"mo2",
"yan4",
"li4",
"lu2",
"long2",
"mo2",
"dan4",
"chen4",
"pin2",
"pi3",
"xiang4",
"huo4",
"mo2",
"xi1",
"duo3",
"ku4",
"yan2",
"chan2",
"ying1",
"rang3",
"dian3",
"la2",
"ta4",
"xiao1",
"jiao2",
"chuo4",
"huan1",
"huo4",
"zhuan4",
"nie4",
"xiao1",
"ca4",
"li2",
"chan3",
"chai4",
"li4",
"yi4",
"luo1",
"nang2",
"za2",
"su1",
"xi3",
"zen5",
"jian1",
"za2",
"zhu3",
"lan2",
"nie4",
"nang1",
"lan3",
"luo5",
"wei2",
"hui2",
"yin1",
"qiu2",
"si4",
"nin2",
"jian3,nan1",
"hui2",
"xin4",
"yin1",
"nan1",
"tuan2",
"tuan2",
"dun4,tun2",
"kang4",
"yuan1",
"jiong3",
"pian1",
"yun2",
"cong1",
"hu2",
"hui2",
"yuan2",
"e2",
"guo2",
"kun4",
"cong1,chuang1",
"wei2",
"tu2",
"wei2",
"lun2",
"guo2",
"qun1",
"ri4",
"ling2",
"gu4",
"guo2",
"tai1",
"guo2",
"tu2",
"you4",
"guo2",
"yin2",
"hun4",
"pu3",
"yu3",
"han2",
"yuan2",
"lun2",
"quan1,juan1,juan4",
"yu3",
"qing1",
"guo2",
"chui2",
"wei2",
"yuan2",
"quan1",
"ku1",
"pu3",
"yuan2",
"yuan2",
"ya4",
"tuan1",
"tu2",
"tu2",
"tuan2",
"lu:e4",
"hui4",
"yi4",
"huan2,yuan2",
"luan2",
"luan2",
"tu3",
"ya4",
"tu3",
"ting3",
"sheng4",
"pu2",
"lu4",
"kuai4",
"ya1",
"zai4",
"wei2,xu1",
"ge1",
"yu4",
"wu1",
"gui1",
"pi3",
"yi2",
"de5,di4",
"qian1,su2",
"qian1",
"zhen4",
"zhuo2",
"dang4",
"qia4",
"xia4",
"shan1",
"kuang4",
"chang3,chang2",
"qi2",
"nie4",
"mo4",
"ji1",
"jia2",
"zhi3",
"zhi3",
"ban3",
"xun1",
"yi4",
"qin3",
"mei2,fen2",
"jun1",
"keng1",
"tun2",
"fang1,fang2",
"ben4",
"ben4",
"tan1",
"kan3",
"huai4",
"zuo4",
"keng1",
"bi4",
"jing3",
"di4",
"jing1",
"ji4",
"kuai4",
"di3",
"jing1",
"jian1",
"tan2",
"li4",
"ba4",
"wu4",
"fen2",
"zhui4",
"po1",
"ban4",
"tang2",
"kun1",
"qu1",
"tan3",
"zhi1",
"tuo2",
"gan1",
"ping2",
"dian4",
"gua4",
"ni2",
"tai2",
"pi1",
"jiong1",
"yang3",
"fo2",
"ao4",
"lu4",
"qiu1",
"mu3",
"ke3",
"gou4",
"xue4",
"ba2",
"chi2,di3",
"che4",
"ling2",
"zhu4",
"fu4",
"hu1",
"zhi4",
"chui2",
"la1",
"long3",
"long3",
"lu2",
"ao4",
"dai4",
"pao2",
"min5",
"xing2",
"dong4",
"ji4",
"he4",
"lv4",
"ci2",
"chi3",
"lei3",
"gai1",
"yin1",
"hou4",
"dui1",
"zhao4",
"fu2",
"guang1",
"yao2",
"duo3,duo4",
"duo3",
"gui3",
"cha2",
"yang2",
"yin2",
"fa2",
"gou4",
"yuan2",
"die2",
"xie2",
"ken3",
"shang3",
"shou3",
"e4",
"bing4",
"dian4",
"hong2",
"ya1",
"kua3",
"da5",
"ka3",
"dang4",
"kai3",
"hang2",
"nao3",
"an3",
"xing1",
"xian4",
"yuan4",
"bang1",
"pou2",
"ba4",
"yi4",
"yin4",
"han4",
"xu4",
"chui2",
"qin2",
"geng3",
"ai1",
"beng3,feng1",
"fang2",
"que4",
"yong3",
"jun4",
"jia1",
"di4",
"mai2,man2",
"lang4",
"juan3",
"cheng2",
"shan1,yan2",
"jin1",
"zhe2",
"lie4",
"lie4",
"pu3,bu4",
"cheng2",
"hua1",
"bu4",
"shi2",
"xun1",
"guo1",
"jiong1",
"ye3",
"nian4",
"di1",
"yu4",
"bu4",
"ya1",
"quan2",
"sui4",
"pi2",
"qing1",
"wan3",
"ju4",
"lun3",
"zheng1",
"kong1",
"shang3,tang3",
"dong1",
"dai4",
"tan4",
"an3",
"cai4",
"chu4",
"beng3",
"kan3",
"zhi2",
"duo3",
"yi4",
"zhi2",
"yi4",
"pei2",
"ji1",
"zhun3",
"qi2",
"sao4",
"ju4",
"ni2",
"ku1",
"ke4",
"tang2",
"kun1",
"ni4",
"jian1",
"dui1",
"jin3",
"gang1",
"yu4",
"e4",
"peng2",
"gu4",
"tu4",
"ling2",
"fang5",
"ya2",
"qian4",
"kun1",
"an4",
"shen1",
"duo4",
"nao3",
"tu1",
"cheng2",
"yin1",
"hun2",
"bi4",
"lian4",
"guo1",
"die2",
"zhuan4",
"hou4",
"bao3,pu4",
"bao3",
"yu2",
"di1",
"mao2",
"jie1",
"ruan2",
"e4",
"geng4",
"kan1",
"zong1",
"yu2",
"huang2",
"e4",
"yao2",
"yan4",
"bao4",
"ji2",
"mei2",
"chang3",
"du3",
"tuo2",
"yin4",
"feng2",
"zhong4",
"jie4",
"jin1",
"heng4",
"gang1",
"chun1",
"jian3",
"ping2",
"lei3",
"jiang3",
"huang1",
"leng2",
"duan4",
"wan1",
"xuan1",
"ji4",
"ji2",
"kuai4",
"ying2",
"ta1",
"cheng2",
"yong3",
"kai3",
"su4",
"su4",
"shi2",
"mi4",
"ta3",
"weng3",
"cheng2",
"tu2",
"tang2",
"que4",
"zhong3",
"li4",
"zhong3",
"bang4",
"sai1,sai4,se4",
"zang4",
"dui1",
"tian2",
"wu4",
"zheng4",
"xun1",
"ge2",
"zhen4",
"ai4",
"gong1",
"yan2",
"kan3",
"tian2",
"yuan2",
"wen1",
"xie4",
"liu4",
"hai3",
"lang3",
"chang2",
"peng2",
"beng4",
"chen2",
"lu4",
"lu3",
"ou1",
"qian4",
"mei2",
"mo4",
"zhuan1",
"shuang3",
"shu2",
"lou3",
"chi2",
"man4",
"biao1",
"jing4",
"ce4",
"shu4",
"zhi4",
"zhang4",
"kan4",
"yong1",
"dian4",
"chen3",
"zhi2",
"xi4",
"guo1",
"qiang3",
"jin3",
"di4",
"shang1",
"mu4",
"cui1",
"yan4",
"ta3",
"zeng1",
"qian2",
"qiang2",
"liang2",
"wei4",
"zhui4",
"qiao1",
"zeng1",
"xu1",
"shan4",
"shan4",
"ba2",
"pu2",
"kuai4",
"dong3",
"fan2",
"que4",
"mo4",
"dun1",
"dun1",
"zun1",
"di4",
"sheng4",
"duo4",
"duo4",
"tan2",
"deng4",
"mu2",
"fen2",
"huang2",
"tan2",
"da5",
"ye4",
"zhu4",
"jian4",
"ao4",
"qiang2",
"ji1",
"qiao1",
"ken3",
"yi4",
"pi2",
"bi4",
"dian4",
"jiang1",
"ye3",
"yong1",
"bo2",
"tan2",
"lan3",
"ju4",
"huai4",
"dang4",
"rang3",
"qian4",
"xun1",
"xian4",
"xi3",
"he4",
"ai4",
"ya1",
"dao3",
"hao2",
"ruan2",
"jin4",
"lei3",
"kuang4",
"lu2",
"yan2",
"tan2",
"wei3",
"huai4",
"long3",
"long3",
"rui4",
"li4",
"lin2",
"rang3",
"chan2",
"xun1",
"yan2",
"lei2",
"ba4",
"wan1",
"shi4",
"ren2",
"xx5",
"zhuang4",
"zhuang4",
"sheng1",
"yi1",
"mai4",
"qiao4",
"zhu4",
"zhuang4",
"hu2",
"hu2",
"kun3",
"yi1",
"hu2",
"xu4",
"kun3",
"shou4",
"mang3",
"zun1",
"shou4",
"yi1",
"zhi3",
"gu3",
"chu4,chu3",
"jiang4,xiang2",
"feng2",
"bei4",
"zhai1",
"bian4",
"sui1",
"qun1",
"ling2",
"fu4",
"cuo4",
"xia4",
"xiong4",
"xie4",
"nao2",
"xia4",
"kui2",
"xi1",
"wai4",
"yuan4",
"mao3",
"su4",
"duo1",
"duo1",
"ye4",
"qing2",
"wai4",
"gou4",
"gou4",
"qi4",
"meng4",
"meng4",
"yin2",
"huo3",
"chen3",
"da4,dai4",
"ze4",
"tian1",
"tai4",
"fu1,fu2",
"guai4",
"yao1",
"yang1",
"hang1,ben4",
"gao3",
"shi1",
"tao1",
"tai4",
"tou2,tou5",
"yan3",
"bi3",
"yi2",
"kua1",
"jia1,jia2,jia4",
"duo2",
"hua4",
"kuang3",
"yun3",
"jia1",
"ba1",
"en1",
"lian2",
"huan4",
"di1",
"yan3,yan1",
"pao4",
"juan4",
"qi2,ji1",
"nai4",
"feng4",
"xie2",
"fen4",
"dian3",
"quan1",
"kui2",
"zou4",
"huan4",
"qi4",
"kai1",
"chi3,she1,zha4",
"ben1,ben4",
"yi4",
"jiang3",
"tao4",
"zang4,zhuang3",
"ben3",
"xi1",
"huang3",
"fei3",
"diao1",
"xun4",
"beng1",
"dian4",
"ao4",
"she1",
"weng3",
"ha3",
"ao4",
"wu4",
"ao4",
"jiang3",
"lian2",
"duo2",
"yun1",
"jiang3",
"shi4",
"fen4",
"huo4",
"bi4",
"luan2",
"duo3",
"nu:3,ru3",
"nu2",
"ding3",
"nai3",
"qian1",
"jian1",
"ta1",
"jiu3",
"nuan2",
"cha4",
"hao3,hao4",
"xian1",
"fan4",
"ji3",
"shuo4",
"ru2",
"fei1",
"wang4",
"hong2",
"zhuang1",
"fu4",
"ma1",
"dan1",
"ren4",
"fu1",
"jing4",
"yan2",
"hai4",
"wen4",
"zhong1",
"pa1",
"du4",
"ji4",
"keng1",
"zhong4",
"yao1",
"jin4",
"yun2",
"miao4",
"pei1",
"chi1",
"yue4",
"zhuang1",
"niu1",
"yan4",
"na4",
"xin1",
"fen2",
"bi3",
"yu2",
"tuo3",
"feng1",
"wan4",
"fang2",
"wu3",
"yu4",
"gui1",
"du4",
"ba2",
"ni1",
"zhou2",
"zhuo2",
"zhao1",
"da2",
"ni3",
"yuan4",
"tou3",
"xian2",
"zhi2",
"e1",
"mei4",
"mo4",
"qi1,qi4",
"bi4",
"shen1",
"qie4",
"e1",
"he2",
"xu3",
"fa2",
"zheng1",
"min2",
"ban4",
"mu3",
"fu1",
"ling2",
"zi3",
"zi3",
"shi3",
"ran3",
"shan1",
"yang1",
"man2",
"jie3",
"gu1",
"si4",
"xing4",
"wei3,wei1",
"zi1",
"ju4",
"shan1",
"pin1",
"ren4",
"yao2",
"dong4",
"jiang1",
"shu1",
"ji2",
"gai1",
"xiang4",
"hua2",
"juan1",
"jiao1",
"gou4",
"lao3,mu3",
"jian1",
"jian1",
"yi2",
"nian4",
"zhi2",
"ji1",
"ji1",
"xian4",
"heng2",
"guang1",
"jun1",
"kua1",
"yan4",
"ming3",
"lie4",
"pei4",
"e4",
"you4",
"yan2",
"cha4",
"xian3",
"yin1",
"shi2",
"gui3",
"quan2",
"zi1",
"song1",
"wei1",
"hong2",
"wa2",
"lou2",
"ya4",
"rao2",
"jiao1",
"luan2",
"ping1",
"xian4",
"shao4",
"li3",
"cheng2",
"xie4",
"mang2",
"fu1",
"suo1",
"mei2",
"wei3",
"ke4",
"chuo4",
"chuo4",
"ting3",
"niang2",
"xing2",
"nan2",
"yu2",
"na4,nuo2",
"pou1",
"nei3",
"juan1",
"shen1",
"zhi4",
"han2",
"di4",
"zhuang1",
"e2",
"pin2",
"tui4",
"xian4",
"mian3,wan3",
"wu2",
"yan2",
"wu3",
"ai1",
"yan2",
"yu2",
"si4",
"yu2",
"wa1",
"li4",
"xian2",
"ju1",
"qu3",
"zhui4",
"qi2",
"xian2",
"zhuo2",
"dong1",
"chang1",
"lu4",
"ai3",
"e1",
"e1",
"lou2",
"mian2",
"cong2",
"pou3",
"ju2",
"po2",
"cai3",
"ling2",
"wan3",
"biao3",
"xiao1",
"shu2",
"qi3",
"hui1",
"fan4",
"wo3",
"rui2",
"tan2",
"fei1",
"fei1",
"jie2",
"tian1",
"ni2",
"quan2",
"jing4",
"hun1",
"jing1",
"qian1",
"dian4",
"xing4",
"hu4",
"wan1",
"lai2",
"bi4",
"yin1",
"zhou1",
"chuo4",
"fu4",
"jing4",
"lun2",
"an4",
"lan2",
"kun1",
"yin2",
"ya4",
"ju1",
"li4",
"dian3",
"xian2",
"hua1",
"hua4",
"ying1",
"chan2",
"shen3",
"ting2",
"dang4",
"yao3",
"wu4",
"nan4",
"chuo4",
"jia3",
"tou1",
"xu4",
"yu4",
"wei2,wei3",
"di4",
"rou2",
"mei3",
"dan1",
"ruan3",
"qin1",
"hui1",
"wo4",
"qian2",
"chun1",
"miao2",
"fu4",
"jie3",
"duan1",
"yi2",
"zhong4",
"mei2",
"huang2",
"mian2",
"an1",
"ying1",
"xuan1",
"jie1",
"wei1",
"mei4",
"yuan4,yuan2",
"zheng1",
"qiu1",
"ti2",
"xie4",
"duo4,tuo2",
"lian4",
"mao4",
"ran3",
"si1",
"pian1",
"wei4",
"wa1",
"cu4",
"hu2",
"ao3",
"jie2",
"bao3",
"xu1",
"tou1",
"gui1",
"chu2",
"yao2",
"pi4",
"xi2",
"yuan2",
"ying4",
"rong2",
"ru4",
"chi1",
"liu2",
"mei3",
"pan2",
"ao3",
"ma1",
"gou4",
"kui4",
"qin2",
"jia4",
"sao3",
"zhen1",
"yuan2",
"jie1",
"rong2",
"ming2",
"ying1",
"ji2",
"su4",
"niao3",
"xian2",
"tao1",
"pang2",
"lang2",
"nao3",
"biao2",
"ai4",
"pi4",
"pin2",
"yi4",
"piao2",
"yu4",
"lei2",
"xuan2",
"man4",
"yi1",
"zhang1",
"kang1",
"yong1",
"ni4",
"li2",
"di2",
"gui1",
"yan1",
"jin3",
"zhuan1",
"chang2",
"ze2",
"han1",
"nen4",
"lao4",
"mo2",
"zhe1",
"hu4",
"hu4",
"ao4",
"nen4",
"qiang2",
"ma5",
"pie4",
"gu1",
"wu3",
"qiao2",
"tuo3",
"zhan3",
"miao2",
"xian2",
"xian2",
"mo4",
"liao2",
"lian2",
"hua4",
"gui1",
"deng1",
"zhi2",
"xu1",
"yi1",
"hua2",
"xi1",
"kui4",
"rao2",
"xi1",
"yan4",
"chan2",
"jiao1",
"mei3",
"fan4",
"fan1",
"xian1",
"yi4",
"hui4",
"jiao4",
"fu4",
"shi4",
"bi4",
"shan4",
"sui4",
"qiang2",
"lian3",
"huan2,qiong2,xuan1",
"xin1",
"niao3",
"dong3",
"yi4",
"can1",
"ai4",
"niang2",
"ning2",
"ma1",
"tiao3",
"chou2",
"jin4",
"ci2",
"yu2",
"pin2",
"rong2",
"xu1",
"nai3",
"yan1",
"tai2",
"ying1",
"qian4",
"niao3",
"yue4",
"ying2",
"mian2",
"bi2",
"mo2",
"shen3",
"xing4",
"ni4",
"du2",
"liu3",
"yuan1",
"lan3",
"yan4",
"shuang1",
"ling2",
"jiao3",
"niang2",
"lan3",
"xian1",
"ying1",
"shuang1",
"hui4",
"quan2",
"mi3",
"li2",
"luan2",
"yan2",
"zhu2",
"lan3",
"zi5,zi3",
"jie2",
"jue2",
"jue2",
"kong3",
"yun4",
"zi1",
"zi4",
"cun2",
"sun1",
"fu2",
"bei4",
"zi1",
"xiao4",
"xin4",
"meng4",
"si4",
"tai1",
"bao1",
"ji4",
"gu1",
"nu2",
"xue2",
"you4",
"zhuan3",
"hai2",
"luan2",
"sun1",
"nao1",
"mie1",
"cong2",
"qian1",
"shu2",
"can4,chan2",
"ya1",
"zi1",
"ni3",
"fu1",
"zi1",
"li2",
"xue2",
"bo4",
"ru2",
"nai2",
"nie4",
"nie4",
"ying1",
"luan2",
"mian2",
"ning2,ning4",
"rong3",
"ta1",
"gui3",
"zhai2",
"qiong2",
"yu3",
"shou3",
"an1",
"jia1,tu2",
"song4",
"wan2",
"rou4",
"yao3",
"hong2",
"yi2",
"jing3",
"zhun1",
"mi4",
"zhu3",
"dang4",
"hong2",
"zong1",
"guan1",
"zhou4",
"ding4",
"wan3",
"yi2",
"bao3",
"shi2",
"shi2",
"chong3",
"shen3",
"ke4",
"xuan1",
"shi4",
"you4",
"huan4",
"yi2",
"tiao3",
"shi3",
"xian4",
"gong1",
"cheng2",
"qun2",
"gong1",
"xiao1",
"zai3",
"zha4",
"bao3",
"hai4",
"yan4",
"xiao1",
"jia1",
"shen3",
"chen2",
"rong2",
"huang3",
"mi4",
"kou4",
"kuan1",
"bin1",
"su4,xiu3,xiu4",
"cai4",
"zan3",
"ji4",
"yuan1",
"ji4",
"yin2",
"mi4",
"kou4",
"qing1",
"he4",
"zhen1",
"jian4",
"fu4",
"ning2",
"bing4",
"huan2",
"mei4",
"qin3",
"han2",
"yu4",
"shi2",
"ning2",
"jin4",
"ning2",
"zhi4",
"yu3",
"bao3",
"kuan1",
"ning2",
"qin3",
"mo4",
"cha2",
"ju4",
"gua3",
"qin3",
"hu1",
"wu4",
"liao2",
"shi2",
"ning2",
"zhai4",
"shen3",
"wei3",
"xie3",
"kuan1",
"hui4",
"liao2",
"jun4",
"huan2",
"yi4",
"yi2",
"bao3",
"qin1",
"chong3",
"bao3",
"feng1",
"cun4",
"dui4",
"si4",
"xun2",
"dao3",
"lv4",
"dui4",
"shou4",
"po3",
"feng1",
"zhuan1",
"fu1",
"she4",
"ke4",
"jiang1,jiang4,qiang1",
"jiang1",
"zhuan1",
"wei4",
"zun1",
"xun2",
"shu4",
"dui4",
"dao3",
"xiao3",
"jie2",
"shao3,shao4",
"er3",
"er3",
"er3",
"ga3",
"jian1",
"shu1",
"chen2",
"shang4",
"shang4",
"mo2",
"ga2",
"chang2",
"liao4",
"xian3",
"xian3",
"kun5",
"wang1",
"wang1",
"you2",
"liao4",
"liao4",
"yao2",
"mang2,pang2",
"wang1",
"wang1",
"wang1",
"ga4",
"yao2",
"duo4",
"kui4",
"zhong3",
"jiu4",
"gan1",
"gu3",
"gan1",
"tui2",
"gan1",
"gan1",
"shi1",
"yin3",
"chi3,che3",
"kao1",
"ni2",
"jin3,jin4",
"wei3,yi3",
"niao4,sui1",
"ju2",
"pi4",
"ceng2",
"xi4",
"bi1",
"ju1,ji1",
"jie4",
"tian2",
"qu1",
"ti4",
"jie4",
"wu1",
"diao3",
"shi1",
"shi3",
"ping2,bing1,bing3",
"ji1",
"xie4",
"zhen3",
"xie4",
"ni2",
"zhan3",
"xi1",
"wei3",
"man3",
"e1",
"lou4",
"ping2",
"ti4",
"fei4",
"shu3,zhu3",
"xie4",
"tu2",
"lu:3",
"lv3",
"xi3",
"ceng2",
"lu:3",
"ju4",
"xie4",
"ju4",
"jue1",
"liao2",
"jue1",
"shu3",
"xi4",
"che4",
"tun2,zhun1",
"ni4",
"shan1",
"wa1",
"xian1",
"li4",
"e4",
"hui4",
"hui4",
"long2",
"yi4",
"qi3",
"ren4",
"wu4",
"han4",
"shen1",
"yu3",
"chu1",
"sui4",
"qi3,kai3",
"ren4",
"yue4",
"ban3",
"yao3",
"ang2",
"ya2",
"wu4",
"jie2",
"e4",
"ji2",
"qian1",
"fen2",
"wan2",
"qi2",
"cen2",
"qian2",
"qi2",
"cha4",
"jie4",
"qu1",
"gang3",
"xian4",
"ao4",
"lan2",
"dao3",
"ba1",
"zuo4",
"zuo4",
"yang3",
"ju4",
"gang1",
"ke3",
"gou3",
"xue2",
"po1",
"li4",
"tiao2",
"qu1",
"yan2",
"fu2",
"xiu4",
"jia3",
"ling3",
"tuo2",
"pi2",
"ao4",
"dai4",
"kuang4",
"yue4",
"qu1",
"hu4",
"po4",
"min2",
"an4",
"tiao2",
"ling2",
"chi2",
"ping2",
"dong1",
"han4",
"kui1",
"xiu4",
"mao3",
"tong2",
"xue2",
"yi4",
"bian4",
"he2",
"ba1,ke1,ke4",
"luo4",
"e4",
"fu4",
"xun2",
"die2",
"lu4",
"en3",
"er2",
"gai1",
"quan1",
"dong4,tong2",
"yi2",
"mu3",
"shi2",
"an1",
"wei2",
"huan2",
"zhi4",
"mi4",
"li3",
"ji4",
"tong2",
"wei2",
"you4",
"xx5",
"xia2",
"li3",
"yao2",
"jiao4",
"zheng1",
"luan2",
"jiao1",
"e2",
"e2",
"yu4",
"xie2",
"bu1",
"qiao4",
"qun1",
"feng1",
"feng1",
"nao2",
"li3",
"you2",
"xian4",
"rong2",
"dao3",
"shen1",
"cheng2",
"tu2",
"geng3",
"jun4",
"gao4",
"xia2",
"yin2",
"wu2",
"lang4",
"kan3",
"lao2",
"lai2",
"xian3",
"que4",
"kong1",
"chong2",
"chong2",
"ta4",
"lin2",
"hua2",
"ju1",
"lai2",
"qi2",
"min2",
"kun1",
"kun1",
"zu2,cui4",
"gu4",
"cui1",
"ya2",
"ya2",
"gang3",
"lun2",
"lun2",
"leng2",
"jue2",
"duo1",
"zheng1",
"guo1",
"yin2",
"dong1",
"han2",
"zheng1",
"wei3",
"xiao2",
"pi2",
"yan1",
"song1",
"jie2",
"beng1",
"zu2",
"ku1",
"dong1",
"zhan3,chan2",
"gu4",
"yin2",
"zi1",
"ze2",
"huang2",
"yu2",
"wai3,wei1",
"yang2",
"feng1",
"qiu2",
"yang2",
"ti2",
"yi3",
"zhi4",
"shi4",
"zai3",
"yao3",
"e4",
"zhu4",
"kan1",
"lu:4",
"yan3",
"mei3",
"han2",
"ji1",
"ji1",
"huan4",
"ting2",
"sheng4",
"mei2",
"qian4,kan3",
"wu4",
"yu2",
"zong1",
"lan2",
"ke3",
"yan2",
"yan2",
"wei3",
"zong1",
"cha2",
"sui4",
"rong2",
"ke1",
"qin1",
"yu2",
"qi2",
"lou3",
"tu2",
"dui1",
"xi1",
"weng3",
"cang1",
"dang4",
"rong2",
"jie2",
"kai3",
"liu2",
"wu4",
"song1",
"qiao1",
"zi1",
"wei2",
"beng1",
"dian1",
"cuo2",
"qian3",
"yong3",
"nie4",
"cuo2",
"ji2",
"shi2",
"ruo4",
"song3",
"zong1",
"jiang4",
"liao2",
"kang1",
"chan3",
"die2",
"cen1",
"ding3",
"tu1",
"lou3",
"zhang4",
"zhan3",
"zhan3",
"ao2",
"cao2",
"qu1",
"qiang1",
"cui1",
"zui3",
"dao3",
"dao3",
"xi2",
"yu4",
"pei4",
"long2",
"xiang4",
"ceng2",
"bo1",
"qin1",
"jiao1",
"yan1",
"lao2",
"zhan4",
"lin2",
"liao2",
"liao2",
"jin1",
"deng4",
"duo4",
"zun1",
"jiao4",
"gui4,jue2",
"yao2",
"jiao1",
"yao2",
"jue2",
"zhan1",
"yi4",
"xue2",
"nao2",
"ye4",
"ye4",
"yi2",
"e4",
"xian3",
"ji2",
"xie4",
"ke3",
"xi1",
"di4",
"ao4",
"zui3",
"wei1",
"yi2",
"rong2",
"dao3",
"ling3",
"jie2",
"yu3",
"yue4",
"yin3",
"ru5",
"jie2",
"li4",
"gui1,xi1",
"long2",
"long2",
"dian1",
"hong1,ying2",
"xi1",
"ju2",
"chan2",
"ying3",
"kui1",
"yan2",
"wei1",
"nao2",
"quan2",
"chao3",
"cuan2",
"luan2",
"dian1",
"dian1",
"nie4",
"yan2",
"yan2",
"yan3",
"kui2",
"yan3",
"chuan1",
"kuai4",
"chuan1",
"zhou1",
"huang1",
"jing1",
"xun2",
"chao2",
"chao2",
"lie4",
"gong1",
"zuo3",
"qiao3",
"ju4",
"gong3",
"ju4",
"wu1",
"pu5",
"pu5",
"cha4,cha1,chai1",
"qiu2",
"qiu2",
"ji3",
"yi3",
"si4",
"ba1",
"zhi1",
"zhao1",
"xiang4",
"yi2",
"jin3",
"xun4",
"juan3",
"ba1",
"xun4",
"jin1",
"fu2",
"za1",
"bi4",
"shi4",
"bu4",
"ding1",
"shuai4",
"fan1",
"nie4",
"shi1",
"fen1",
"pa4",
"zhi3",
"xi1",
"hu4",
"dan4",
"wei2",
"zhang4",
"tang3",
"dai4",
"mo4",
"pei4",
"pa4",
"tie1,tie3,tie4",
"fu2",
"lian2",
"zhi4",
"zhou3",
"bo2",
"zhi4",
"di4",
"mo4",
"yi4",
"yi4",
"ping2",
"qia4",
"juan4",
"ru2",
"shuai4",
"dai4",
"zhen1",
"shui4",
"qiao4",
"zhen1",
"shi1",
"qun2",
"xi2",
"bang1",
"dai4",
"gui1",
"chou2,dao4",
"ping2",
"zhang4",
"san4",
"wan1",
"dai4",
"wei2",
"chang2",
"sha4,qie4",
"qi2",
"ze2",
"guo2",
"mao4",
"du3",
"hou2",
"zheng4",
"xu1",
"mi4",
"wei2",
"wo4",
"fu2",
"yi4",
"bang1",
"ping2",
"die2",
"gong1",
"pan2",
"huang3",
"tao1",
"mi4",
"jia4",
"teng2",
"hui1",
"zhong1",
"shan1",
"man4",
"mu4",
"biao1",
"guo2",
"ze2",
"mu4",
"bang1",
"zhang4",
"jing3",
"chan3",
"fu2",
"zhi4",
"hu1",
"fan1",
"chuang2,zhuang4",
"bi4",
"bi4",
"zhang3",
"mi4",
"qiao1",
"chan1",
"fen2",
"meng2",
"bang1",
"chou2",
"mie4",
"chu2",
"jie2",
"xian3",
"lan2",
"gan4,gan1",
"ping2",
"nian2",
"qian1",
"bing4",
"bing4",
"xing4",
"gan4",
"yao1",
"huan4",
"you4",
"you1",
"ji3",
"guang3,yan3",
"pi3",
"ting1",
"ze4",
"guang3",
"zhuang1",
"mo2",
"qing4",
"bi4",
"qin2",
"dun4",
"chuang2",
"gui3",
"ya3",
"bai4",
"jie4",
"xu4",
"lu2",
"wu3,wu2",
"zhuang1",
"ku4",
"ying1,ying4",
"di3,de5",
"pao2",
"dian4",
"ya1",
"miao4",
"geng1",
"ci4",
"fu3",
"tong2",
"pang2",
"fei4",
"xiang2",
"yi3",
"zhi4",
"tiao1",
"zhi4",
"xiu1",
"du4,duo2",
"zuo4",
"xiao1",
"tu2",
"gui3",
"ku4",
"pang2",
"ting2",
"you3",
"bu1",
"bing4",
"cheng3",
"lai2",
"bi4",
"ji2",
"an1",
"shu4",
"kang1",
"yong1",
"tuo3",
"song1",
"shu4",
"qing3",
"yu4",
"yu3",
"miao4",
"sou1",
"ce4",
"xiang1",
"fei4",
"jiu4",
"e4",
"hui4",
"liu4",
"sha4",
"lian2",
"lang2",
"sou1",
"zhi4",
"bu4",
"qing3",
"jiu4",
"jiu4",
"jin3,qin2",
"ao2",
"kuo4",
"lou2",
"yin4",
"liao4",
"dai4",
"lu4",
"yi4",
"chu2",
"chan2",
"tu2",
"si1",
"xin1",
"miao4",
"chang3",
"wu3",
"fei4",
"guang3",
"ku4",
"kuai4",
"bi4",
"qiang2",
"xie4",
"lin3",
"lin3",
"liao2",
"lu2",
"ji4",
"ying3",
"xian1",
"ting1",
"yong1",
"li2",
"ting1",
"yin3",
"xun2",
"yan2",
"ting2",
"di2",
"pai3",
"jian4",
"hui2",
"nai3",
"hui2",
"gong3",
"nian4",
"kai1",
"bian4",
"yi4",
"qi4",
"nong4,long4",
"fen4",
"ju3",
"yan3",
"yi4",
"zang4",
"bi4",
"yi4",
"yi1",
"er4",
"san1",
"shi4",
"er4",
"shi4",
"shi4",
"gong1",
"diao4",
"yin3",
"hu4",
"fu2",
"hong2",
"wu1",
"tui2",
"chi2",
"jiang4",
"ba4",
"shen3",
"di4,ti4",
"zhang1",
"jue2",
"tao1",
"fu3",
"di3",
"mi2",
"xian2",
"hu2",
"chao1",
"nu3",
"jing4",
"zhen3",
"yi2",
"mi3",
"quan1,juan4",
"wan1",
"shao1",
"ruo4",
"xuan1",
"jing4",
"diao1",
"zhang1",
"jiang4",
"qiang2",
"beng1",
"dan4,tan2",
"qiang2,jiang4,qiang3",
"bi4",
"bi4",
"she4",
"dan4",
"jian3",
"gou4",
"ge1",
"fa1",
"bi4",
"kou1",
"jian3",
"bie4",
"xiao1",
"dan4",
"guo1",
"jiang4",
"hong2",
"mi2",
"guo1",
"wan1",
"jue2",
"ji4",
"ji4",
"gui1",
"dang1,dang4",
"lu4",
"lu4",
"tuan4",
"hui4",
"zhi4",
"hui4",
"hui4",
"yi2",
"yi2",
"yi2",
"yi2",
"huo4",
"yue1",
"shan1",
"xing2",
"wen2",
"tong2",
"yan4",
"yan4",
"yu4",
"chi1",
"cai3",
"biao1",
"diao1",
"bin1",
"peng2",
"yong3",
"piao1",
"zhang1",
"ying3",
"chi1",
"chi4",
"zhuo2",
"tuo3",
"ji2",
"pang2",
"zhong1",
"yi4",
"wang2",
"che4",
"bi3",
"chi2",
"ling2",
"fu2",
"wang3",
"zheng1",
"cu2",
"wang3",
"jing4",
"dai4,dai1",
"xi1",
"xun4",
"hen3",
"yang2",
"huai2",
"lu:4",
"hou4",
"wang3",
"cheng3",
"zhi4",
"xu2",
"jing4",
"tu2",
"cong2",
"zhi5",
"lai2",
"cong2",
"de2,de5,dei3",
"pai2",
"xi3",
"dong1",
"ji4",
"chang2",
"zhi4",
"cong2",
"zhou1",
"lai2",
"yu4",
"xie4",
"jie4",
"jian4",
"shi4",
"jia3",
"bian4",
"huang2",
"fu4",
"xun2",
"wei3",
"pang2",
"yao2",
"wei1",
"xi1",
"zheng1",
"piao4",
"ti2",
"de2",
"zheng1",
"zhi3",
"bie2",
"de2",
"chong1",
"che4",
"jiao3",
"wei4",
"jiao3,jiao4",
"hui1",
"mei2",
"long4",
"xiang1",
"bao4",
"qu2",
"xin1",
"xin1",
"bi4",
"yi4",
"le4",
"ren2",
"dao1",
"ding4",
"gai3",
"ji4",
"ren3",
"ren2",
"chan4",
"tan3",
"te4",
"te4,tei1",
"gan1",
"qi4",
"shi4,tai4",
"cun3",
"zhi4",
"wang4",
"mang2",
"xi1",
"fan1",
"ying4",
"tian3",
"min2,min3,wen3",
"wen3",
"zhong1",
"chong1",
"wu4",
"ji2",
"wu3",
"xi4",
"jia2",
"you1",
"wan4",
"cong1",
"zhong1",
"kuai4",
"yu4,shu1",
"bian4",
"zhi4",
"qi2",
"cui4",
"chen2",
"tai4",
"tun2",
"qian2",
"nian4",
"hun2",
"xiong1",
"niu3",
"kuang2",
"xian1",
"xin1",
"kang1",
"hu1",
"kai4",
"fen4",
"huai2",
"tai4",
"song3",
"wu3",
"ou4",
"chang4",
"chuang4",
"ju4",
"yi4",
"bao3",
"chao1",
"min2",
"pei1",
"zuo4",
"zen3",
"yang4",
"ju4",
"ban4",
"nu4",
"nao2",
"zheng1,zheng4",
"pa4",
"bu4",
"tie1",
"hu4",
"hu4",
"ju4",
"da2",
"lian2",
"si1",
"zhou4",
"di4",
"dai4",
"yi2",
"tu1",
"you2",
"fu1",
"ji2",
"peng1",
"xing4",
"yuan4",
"ni2",
"guai4",
"fu2,fei4",
"xi4",
"bi4",
"you1",
"qie4",
"xuan4",
"cong1",
"bing3",
"huang3",
"xu4",
"chu4",
"bi4",
"shu4",
"xi1",
"tan1",
"yong3",
"zong3",
"dui4,dui3",
"mo5",
"zhi3",
"yi4",
"shi4",
"nen4,nin2",
"xun2",
"shi4",
"xi4",
"lao3",
"heng2",
"kuang1",
"mou2",
"zhi3",
"xie2",
"lian4",
"tiao1",
"huang3",
"die2",
"hao4",
"kong3",
"gui3",
"heng2",
"xi1",
"xiao4",
"shu4",
"si1",
"hu1",
"qiu1",
"yang4",
"hui4",
"hui2",
"chi4",
"jia2",
"yi2",
"xiong1",
"guai4",
"lin4",
"hui1",
"zi4",
"xu4",
"chi3",
"shang4",
"nu:4",
"hen4",
"en1",
"ke4",
"dong4",
"tian2",
"gong1",
"quan1",
"xi1",
"qia4",
"yue4",
"peng1",
"ken3",
"de2",
"hui4",
"e4,e3,wu4",
"xiao5",
"tong4",
"yan1",
"kai3",
"ce4",
"nao3",
"yun4",
"mang2",
"yong3",
"yong3",
"juan1,juan4",
"pi1",
"kun3",
"qiao1,qiao3",
"yue4",
"yu4",
"tu2",
"jie4",
"xi1",
"zhe2",
"lin4",
"ti4",
"han4",
"hao4",
"qie4",
"ti4",
"bu4",
"yi4",
"qian4",
"hui3",
"xi1",
"bei4",
"man2",
"yi1",
"heng1",
"song3",
"quan1",
"cheng3",
"kui1,li3",
"wu4",
"wu4",
"you1",
"li2",
"liang4",
"huan4",
"cong1",
"yi4",
"yue4",
"li4",
"nin2",
"nao3",
"e4",
"que4",
"xuan2",
"qian1",
"wu4",
"min3",
"cong2",
"fei3",
"bei1",
"de2",
"cui4",
"chang4",
"men4",
"li4",
"ji4",
"guan4",
"guan4",
"xing4",
"dao4",
"qi1",
"kong1",
"tian3",
"lun2",
"xi1",
"kan3",
"gun3",
"ni4",
"qing2",
"chou2",
"dun1",
"guo3",
"zhan1",
"jing1",
"wan3",
"yuan1",
"jin1",
"ji4",
"lin2",
"yu4",
"huo4",
"he2",
"quan2",
"tan2",
"ti4",
"ti4",
"nie4",
"wang3",
"chuo4",
"hu1",
"hun1",
"xi1",
"chang3",
"xin1",
"wei2",
"hui4",
"e4",
"suo3",
"zong3",
"jian1",
"yong3",
"dian4",
"ju4",
"can3",
"cheng2",
"de2",
"bei4",
"qie4",
"can2",
"dan4",
"guan4",
"duo4",
"nao3",
"yun4",
"xiang3",
"zhui4",
"die2",
"huang2",
"chun3",
"qiong2",
"re3",
"xing1",
"ce4",
"bian3",
"min3",
"zong1",
"ti2",
"qiao3",
"chou2",
"bei4",
"xuan1",
"wei1",
"ge2",
"qian1",
"wei3",
"yu4",
"yu2",
"bi4",
"xuan1",
"huan4",
"min3",
"bi4",
"yi4",
"mian3",
"yong3",
"kai4",
"dang4",
"yin1",
"e4",
"chen2",
"mao4",
"qia4",
"ke4",
"yu2",
"ai4",
"qie4",
"yan3",
"nuo4",
"gan3",
"yun4",
"zong3",
"sai1",
"leng4",
"fen4",
"ying1",
"kui4",
"kui4",
"que4",
"gong1",
"yun2",
"su4",
"su4",
"qi2",
"yao2",
"song3",
"huang4",
"ji2",
"gu3",
"ju4",
"chuang4",
"ni4",
"xie2",
"kai3",
"zheng3",
"yong3",
"cao3",
"xun4",
"shen4",
"bo2",
"kai4",
"yuan4",
"xi4",
"hun4",
"yong3",
"yang3",
"li4",
"sao1",
"tao1",
"yin1",
"ci2",
"xu4",
"qian4,qie4",
"tai4",
"huang1",
"yun4",
"shen4",
"ming3",
"gong5",
"she4",
"cong2",
"piao1",
"mu4",
"mu4",
"guo2",
"chi4",
"can3",
"can2",
"can2",
"cui1",
"min3",
"te4",
"zhang1",
"tong4",
"ao4",
"shuang3",
"man4",
"guan4",
"que4",
"zao4",
"jiu4",
"hui4",
"kai3",
"lian2",
"ou4",
"song3",
"jin3",
"yin4",
"lv4",
"shang1",
"wei4",
"tuan2",
"man2",
"qian1",
"she4",
"yong1",
"qing4",
"kang1",
"di4",
"zhi2",
"lou2",
"juan4",
"qi1",
"qi1",
"yu4",
"ping2",
"liao2",
"cong4",
"you1",
"chong1",
"zhi4",
"tong4",
"cheng1",
"qi4",
"qu1",
"peng2",
"bei4",
"bie1",
"qiong2",
"jiao1",
"zeng1",
"chi4",
"lian2",
"ping2",
"kui4",
"hui4",
"qiao2",
"cheng2",
"yin4",
"yin4",
"xi3",
"xi3",
"dan4",
"tan2",
"duo4",
"dui4",
"dui4",
"su4",
"jue2",
"ce4",
"xiao1",
"fan1",
"fen4",
"lao2",
"lao4",
"chong1",
"han1",
"qi4",
"xian2",
"min3",
"jing3",
"liao3",
"wu3",
"can3",
"jue2",
"cu4",
"xian4",
"tan3",
"sheng2",
"pi1",
"yi4",
"chu4",
"xian1",
"nao2",
"dan4",
"tan3",
"jing3",
"song1",
"han4",
"jiao3",
"wei4",
"xuan1",
"dong3",
"qin2",
"qin2",
"qu2",
"cao3",
"ken3",
"xie4",
"ying1",
"ao4",
"mao4",
"yi4",
"lin3",
"se4",
"jun4",
"huai2",
"men4",
"lan3",
"ai4",
"lin3",
"yan1",
"kuo4",
"xia4",
"chi4",
"yu3",
"yin4",
"dai1",
"meng3",
"ai4",
"meng2",
"dui4",
"qi2",
"mo3",
"lan2",
"men4",
"chou2",
"zhi4",
"nuo4",
"nuo4",
"yan1",
"yang3",
"bo2",
"zhi2",
"kuang4",
"kuang3",
"you3",
"fu1",
"liu2",
"mie4",
"cheng2",
"hui5",
"chan4",
"meng3",
"lan3",
"huai2",
"xuan2",
"rang4",
"chan4",
"ji4",
"ju4",
"huan1",
"she4",
"yi4",
"lian4",
"nan3",
"mi2",
"tang3",
"jue2",
"gang4",
"gang4,zhuang4",
"zhuang4",
"ge1",
"yue4",
"wu4",
"jian1",
"xu1,qu5",
"shu4",
"rong2",
"xi4",
"cheng2",
"wo3",
"jie4",
"ge1",
"jian1",
"qiang1",
"huo4",
"qiang1",
"zhan4",
"dong4",
"qi1",
"jia2",
"die2",
"zei2",
"jia2",
"ji3",
"zhi2",
"kan1",
"ji2",
"kui2",
"gai4",
"deng3",
"zhan4",
"qiang1",
"ge1",
"jian3",
"jie2",
"yu4",
"jian3",
"yan3",
"lu4",
"hu1",
"zhan4",
"xi4",
"xi4",
"chuo1",
"dai4",
"qu2",
"hu4",
"hu4",
"hu4",
"e4",
"shi4",
"li4",
"mao3",
"hu4",
"li4",
"fang2",
"suo3",
"bian3,pian1",
"dian4",
"jiong1",
"shang3",
"yi2",
"yi3",
"shan4,shan1",
"hu4",
"fei1",
"yan3",
"shou3",
"shou3",
"cai2",
"zha1,zha2,za1",
"qiu2",
"le4",
"pu1",
"ba1,pa2",
"da3,da2",
"reng1",
"fan3,fu2",
"ru4",
"zai4",
"tuo1",
"zhang4",
"diao3",
"kang2,gang1",
"yu1",
"ku1",
"gan3",
"shen1",
"cha1",
"chi3",
"gu3,xi4",
"kou4",
"wu4",
"den4",
"qian1",
"zhi2",
"ren4",
"kuo4",
"men2",
"sao3,sao4",
"yang2",
"niu3",
"ban4",
"che3",
"rao3",
"xi1",
"qian2",
"ban1,pan1",
"jia2",
"yu2",
"fu2",
"ao4",
"xi1",
"pi1",
"zhi3",
"zi4",
"e4",
"den4",
"zhao3",
"cheng2",
"ji4",
"yan3",
"kuang2",
"bian4",
"chao1",
"ju1",
"wen4",
"hu2",
"yue4",
"jue2",
"ba3,ba4",
"qin4",
"zhen3",
"zheng3",
"yun3",
"wan2",
"ne4",
"yi4",
"shu1",
"zhua1",
"pou2",
"tou2",
"dou3",
"kang4",
"zhe2,she2,zhe1",
"pou2",
"fu3",
"pao1",
"ba2",
"ao3",
"ze2",
"tuan2",
"kou1",
"lun1,lun2",
"qiang3,qiang1",
"yun5",
"hu4",
"bao4",
"bing3",
"zhi3",
"peng1",
"nan2",
"bu4",
"pi1",
"tai2",
"yao3",
"zhen3",
"zha1",
"yang1",
"bao4",
"he1",
"ni3",
"ye4",
"di3",
"chi4",
"pi1",
"jia1",
"mo3,ma1,mo4",
"mei4",
"chen1",
"ya1",
"chou1",
"qu1",
"min3",
"chu4",
"jia1",
"fu2,bi4",
"zha3",
"zhu3",
"dan1,dan4",
"chai1",
"mu3",
"nian1",
"la1",
"fu3",
"pao1",
"ban4",
"pai1",
"lin1",
"na2",
"guai3",
"qian2",
"ju4",
"ta4,tuo4",
"ba2",
"tuo1",
"tuo1",
"ao4,niu4",
"ju1",
"zhuo1",
"pan4,pin1",
"zhao1",
"bai4",
"bai4",
"di3",
"ni3",
"ju4",
"kuo4",
"long3",
"jian3",
"qia2",
"yong1",
"lan2",
"ning2,ning3,ning4",
"bo1",
"ze2",
"qian1",
"hen2",
"kuo4",
"shi4",
"jie2",
"zheng3",
"nin3",
"gong3",
"gong3",
"quan2",
"shuan1",
"cun2",
"zan3",
"kao3",
"yi2",
"xie2",
"ce4",
"hui1",
"pin1",
"zhuai1,ye4,zhuai3,zhuai4",
"shi2,she4",
"na2",
"bai1",
"chi2",
"gua4",
"zhi4",
"kuo4",
"duo3",
"duo3",
"zhi3",
"qie4",
"an4",
"nong4",
"zhen4",
"ge2",
"jiao4",
"kua4",
"dong4",
"na2",
"tiao1,tiao3",
"lie4",
"zha1",
"lu:3",
"die2",
"wa1",
"jue2",
"lie3",
"ju3",
"zhi4",
"luan2",
"ya4",
"wo1,zhua1",
"ta4",
"xie2,jia1",
"nao2",
"dang3,dang4",
"jiao3",
"zheng1,zheng4",
"ji3",
"hui1",
"xian2",
"yu3",
"ai1,ai2",
"tuo1",
"nuo2",
"cuo4",
"bo2",
"geng3",
"ti3",
"zhen4",
"cheng2",
"sa1",
"suo1",
"keng1",
"mei3",
"nong4",
"ju2",
"peng2",
"jian3",
"yi4",
"ting3",
"shan1",
"ruo2",
"wan3",
"xie2",
"cha1",
"feng2",
"jiao3",
"wu3",
"jun4",
"jiu4",
"tong3",
"kun3",
"huo4",
"tu2",
"zhuo1",
"pou2",
"lu:3,luo1",
"ba1",
"han4",
"shao1",
"nie1",
"juan1",
"ze4",
"shu4",
"ye2",
"jue2",
"bu3",
"wan2",
"bu4",
"zun4",
"ye4",
"zhai1",
"lv3",
"sou1",
"tuo1",
"lao1",
"sun3",
"bang1",
"jian3",
"huan4",
"dao3",
"wei3",
"wan4",
"qin2",
"peng3",
"she3",
"lie4",
"min2",
"men2",
"fu3",
"bai3,bo4",
"ju4,ju1",
"dao2",
"wo3",
"ai2",
"juan3",
"yue4",
"zong3",
"chen1",
"chui2",
"jie2",
"tu1",
"ben4",
"na4",
"nian3",
"ruo2",
"zuo2",
"wo4",
"qi1",
"xian1",
"cheng2",
"dian1",
"sao3",
"lun1",
"qing4",
"gang1",
"duo1",
"shou4",
"diao4",
"pou2,pou3",
"di3",
"zhang3",
"hun4",
"ji3",
"tao1",
"qia1",
"qi2",
"pai2",
"shu1",
"qian1",
"ling2",
"ye1,ye4",
"ya4",
"jue2",
"zheng1",
"liang3",
"gua4",
"yi4",
"huo4",
"shan4",
"zheng3",
"lu:e4",
"cai3",
"tan4",
"che4",
"bing1",
"jie1",
"ti4",
"kong4",
"tui1",
"yan3",
"cuo4",
"zou1",
"ju1",
"tian4",
"qian2",
"ken4",
"bai1",
"pa2,shou3",
"jie1",
"lu3",
"guai1",
"ming5",
"jie2",
"zhi4",
"dan3",
"meng5",
"chan1,shan3",
"sao1",
"guan4",
"peng4",
"yuan4",
"nuo4",
"jian3",
"zheng1",
"jiu1",
"jian1",
"yu2",
"yan2",
"kui2",
"nan3",
"hong1",
"rou2",
"pi4",
"wei1",
"sai1",
"zou4",
"xuan1",
"miao2",
"ti2",
"nie1",
"cha1",
"shi4",
"zong3",
"zhen4",
"yi1",
"shun3",
"huang2",
"bian1",
"yang2",
"huan4",
"yan3",
"zuan4",
"an3",
"xu1",
"ya4",
"wo4",
"ke2",
"chuai1,chuai3",
"ji2",
"ti4",
"la4",
"la4",
"chen2",
"kai1",
"jiu1",
"jiu1",
"tu2",
"jie1",
"hui1",
"gen4",
"chong4",
"xiao1",
"she2",
"xie1",
"yuan2",
"qian2",
"ye2",
"cha1",
"zha1",
"bei1",
"yao2",
"wei1",
"beng5",
"lan3",
"wen4,wen3",
"qin4",
"chan1",
"ge1,ge2",
"lou3,lou1",
"zong3",
"gen4",
"jiao3",
"gou4",
"qin4",
"rong2",
"que4",
"chou1",
"chuai1",
"zhan3",
"sun3",
"sun1",
"bo2",
"chu4",
"rong2",
"bang4,peng4",
"cuo1",
"sao1",
"ke4",
"yao2",
"dao3",
"zhi1",
"nu4",
"la1",
"jian1",
"sou1",
"qiu3",
"gao3",
"xian3",
"shuo4",
"sang3",
"jin4",
"mie4",
"e4",
"chui2",
"nuo4",
"shan1",
"ta4",
"zha3",
"tang2",
"pan2",
"ban1",
"da1",
"li4",
"tao1",
"hu2",
"zhi4",
"wa1,wa3,wa4",
"hua2",
"qian1",
"wen4",
"qiang3",
"chen1",
"zhen1",
"e4",
"xie2",
"nuo4",
"quan2",
"cha2",
"zha4",
"ge2",
"wu3",
"en4",
"she4",
"kang2",
"she4",
"shu1",
"bai3",
"yao2",
"bin4",
"sou1",
"tan1",
"sa4",
"chan3",
"suo1",
"jiu1",
"chong1",
"chuang1",
"guai1",
"bing4",
"feng2",
"shuai1",
"di4",
"qi4",
"sou1",
"zhai1",
"lian3",
"cheng1",
"chi1",
"guan4",
"lu4",
"luo4",
"lou3",
"zong3",
"gai4",
"hu4",
"zha1",
"chuang3",
"tang4",
"hua4",
"cui1",
"nai2",
"mo2",
"jiang1",
"gui1",
"ying3",
"zhi2",
"ao2",
"zhi4",
"nie4",
"man4",
"shan4",
"kou1",
"shu1",
"she4",
"tuan2",
"jiao3",
"mo1,mo2",
"mo2",
"zhe2",
"can4",
"keng1",
"biao1,biao4",
"jiang4",
"yao2",
"gou4",
"qian1",
"liao4",
"ji1",
"ying1",
"jue1",
"pie1",
"pie1,pie3",
"lao1",
"dun1",
"xian4",
"ruan2",
"gui4",
"zan3",
"yi4",
"xian2",
"cheng1",
"cheng1",
"sa1,sa3",
"nao2",
"hong4",
"si1",
"han4",
"guang4",
"da1",
"zun3",
"nian3",
"lin3",
"zheng3",
"hui1",
"zhuang4",
"jiao3",
"ji3",
"cao1",
"dan3",
"dan3",
"che4",
"bo1",
"che3",
"jue1",
"fu3",
"liao1,liao2",
"ben4",
"fu3",
"qiao4",
"bo1",
"cuo1,zuo3",
"zhuo2",
"zhuan4",
"tuo3",
"pu1",
"qin4",
"dun1",
"nian3",
"hua2",
"xie2",
"lu1",
"jiao3",
"cuan1",
"ta4",
"han4",
"qiao4",
"wo1",
"jian3",
"gan3",
"yong1",
"lei2,lei4",
"nang3",
"lu3",
"shan4",
"zhuo2",
"ze2",
"pu1",
"chuo4",
"ji1",
"dang3",
"se4",
"cao1,cao4",
"qing2",
"qing2",
"huan4",
"jie1",
"qin2",
"kuai3",
"dan1",
"xie2",
"ge3",
"pi3",
"bo4",
"ao4",
"ju4",
"ye4",
"e4",
"meng1",
"sou3",
"mi2",
"ji3",
"tai2",
"zhuo2",
"dao3",
"xing3",
"lan3",
"ca1",
"ju3",
"ye2",
"ru3",
"ye4",
"ye4",
"ni3",
"huo4",
"jie2",
"bin4",
"ning2",
"ge1",
"zhi4",
"zhi4",
"kuo4",
"mo2",
"jian4",
"xie2",
"lie4",
"tan1",
"bai3",
"sou3",
"lu3",
"lve4",
"rao3",
"ti1,zhi4",
"pan1",
"yang3",
"lei2",
"sa4",
"shu1",
"zan3",
"nian3",
"xian3",
"jun4",
"huo1",
"li4",
"la4",
"huan3",
"ying2",
"lu2",
"long3",
"qian1",
"qian1",
"zan3,cuan2",
"qian1",
"lan2",
"xian1",
"ying1",
"mei2",
"rang3",
"chan1",
"weng3",
"cuan1",
"xie2",
"she4",
"luo2",
"jun4",
"mi2",
"chi1",
"zan3",
"luan2",
"tan1",
"zuan4",
"li4",
"dian1",
"wa1",
"dang3",
"jiao3",
"jue2",
"lan3",
"li4",
"nang3",
"zhi1",
"gui4",
"gui3",
"qi1",
"xun2",
"pu1",
"pu1",
"shou1",
"kao3",
"you1",
"gai3",
"yi3",
"gong1",
"han4",
"ban1",
"fang4",
"zheng4",
"po4",
"dian1",
"kou4",
"min3",
"wu4",
"gu4",
"he2",
"chi4",
"xiao4",
"mi3",
"chu4",
"ge2",
"di2",
"xu4",
"jiao4",
"min3",
"chen2",
"jiu4",
"shen1",
"duo2",
"yu3",
"chi4",
"ao2",
"bai4",
"xu4",
"jiao4,jiao1",
"duo2",
"lian3,lian4",
"nie4",
"bi4",
"chang3",
"dian3",
"duo1",
"yi4",
"gan3",
"san4,san3",
"ke3",
"yan4",
"dun1",
"qi3",
"tou3",
"xiao4",
"duo2",
"jiao3",
"jing4",
"yang2",
"xia2",
"min3",
"shu4,shu3,shuo4",
"ai2",
"qiao1",
"ai2",
"zheng3",
"di2",
"zhen4",
"fu1",
"shu4",
"liao2",
"qu1",
"xiong4",
"yi3",
"jiao3",
"shan4",
"jiao3",
"zhuo2",
"yi4",
"lian3",
"bi4",
"tai2",
"xiao4",
"xiao4",
"wen2",
"xue2",
"qi2",
"qi2",
"zhai1",
"bin1",
"jue2",
"zhai1",
"lang2",
"fei3",
"ban1",
"ban1",
"lan2",
"yu3",
"lan2",
"wei3",
"dou4,dou3",
"sheng1",
"liao4",
"jia3",
"hu2",
"xie2",
"jia3",
"yu3",
"zhen1",
"jiao4",
"wo4",
"tiao3,tou3",
"dou4",
"jin1",
"chi4",
"yin2",
"fu3",
"qiang1",
"zhan3",
"qu2",
"zhuo2",
"zhan3",
"duan4",
"zhuo2",
"si1",
"xin1",
"zhuo2",
"zhuo2",
"qin2",
"lin2",
"zhuo2",
"chu4",
"duan4",
"zhu3",
"fang1",
"chan3",
"hang2",
"yu1,wu1",
"shi1",
"pei4",
"you2",
"mie5",
"pang2",
"qi2",
"zhan1",
"mao2,mao4",
"lu:3",
"pei4",
"pi1",
"liu2",
"fu1",
"fang3",
"xuan2,xuan4",
"jing1",
"jing1",
"ni3",
"zu2",
"zhao4",
"yi3",
"liu2",
"shao1",
"jian4",
"yu2",
"yi3",
"qi2",
"zhi4",
"fan1",
"piao1",
"fan1",
"zhan1",
"kuai4",
"sui4",
"yu2",
"wu2",
"ji4",
"ji4",
"ji4",
"huo4",
"ri4",
"dan4",
"jiu4",
"zhi3",
"zao3",
"xie2",
"tiao1",
"xun2",
"xu4",
"ga1",
"la2",
"gan4",
"han4",
"tai2",
"di4",
"xu1",
"chan3",
"shi2",
"kuang4",
"yang2",
"shi2",
"wang4",
"min2",
"min2",
"tun1",
"chun1",
"wu3",
"yun2",
"bei4",
"ang2",
"ze4",
"ban3",
"jie2",
"kun1",
"sheng1",
"hu4",
"fang3",
"hao4",
"gui4",
"chang1",
"xuan1",
"ming2",
"hun1",
"fen1",
"qin3",
"hu1",
"yi4",
"xi1",
"xin1",
"yan2",
"ze4",
"fang3",
"tan2",
"shen4",
"ju4",
"yang2",
"zan3",
"bing3",
"xing1",
"ying4",
"xuan4",
"po4",
"zhen3",
"ling1",
"chun1",
"hao4",
"mei4",
"zuo2",
"mo4",
"bian4",
"xu4",
"hun1",
"zhao1",
"zong4",
"shi4",
"shi4",
"yu4",
"fei4",
"die2",
"mao3",
"ni4",
"chang3",
"wen1",
"dong1",
"ai3",
"bing3",
"ang2",
"zhou4",
"long2",
"xian3",
"kuang4",
"tiao3",
"chao2",
"shi2",
"huang3,huang4",
"huang3",
"xuan1",
"kui2",
"xu1",
"jiao3",
"jin4",
"zhi4",
"jin4",
"shang3",
"tong2",
"hong3",
"yan4",
"gai1",
"xiang3",
"shai4",
"xiao3",
"ye4",
"yun1,yun4",
"hui1",
"han2",
"han4",
"jun4",
"wan3",
"xian4",
"kun1",
"zhou4",
"xi1",
"cheng2,sheng4",
"sheng4",
"bu1",
"zhe2",
"zhe2",
"wu4",
"wan3",
"hui4",
"hao4",
"chen2",
"wan3",
"tian3",
"zhuo2",
"zui4",
"zhou3",
"pu3",
"jing3",
"xi1",
"shan3",
"ni3",
"xi1",
"qing2",
"qi3",
"jing1",
"gui3",
"zheng3",
"yi4",
"zhi4",
"an4",
"wan3",
"lin2",
"liang4",
"cheng1",
"wang3",
"xiao3",
"zan4",
"fei1",
"xuan1",
"geng4",
"yi2",
"xia2",
"yun1",
"hui1",
"xu3",
"min3,min2",
"kui2",
"he4",
"ying4",
"shu3",
"wei3",
"shu3",
"qing2",
"mao4",
"nan2",
"jian3",
"nuan3",
"an4",
"yang2",
"chun1",
"yao2",
"suo3",
"pu3",
"ming2",
"jiao3",
"kai3",
"gao3",
"weng3",
"chang4",
"qi4",
"hao4",
"yan4",
"li4",
"ai4",
"ji4",
"ji4",
"men4",
"zan4",
"xie4",
"hao4",
"mu4",
"mo4",
"cong1",
"ni4",
"zhang1",
"hui4",
"bao4",
"han4",
"xuan2",
"chuan2",
"liao2",
"xian1",
"tan3",
"jing3",
"pie1",
"lin2",
"tun1",
"xi3",
"yi4",
"ji4",
"huang4",
"dai4",
"ye4",
"ye4",
"li4",
"tan2",
"tong2",
"xiao3",
"fei4",
"shen3",
"zhao4",
"hao4",
"yi4",
"xiang3",
"xing1",
"shen1",
"jiao3",
"bao4",
"jing4",
"yan4",
"ai4",
"ye4",
"ru2",
"shu3",
"meng2",
"xun1",
"yao4",
"pu4",
"li4",
"chen2",
"kuang4",
"die2",
"liao3",
"yan4",
"huo4",
"lu2",
"xi1",
"rong2",
"long2",
"nang3",
"luo3",
"luan2",
"shai4",
"tang3",
"yan3",
"zhu2",
"yue1",
"yue1",
"qu1,qu3",
"ye4",
"geng4,geng1",
"ye4",
"hu1",
"he2",
"shu1",
"cao2",
"cao2",
"sheng1",
"man4",
"ceng2",
"ceng2,zeng1",
"ti4",
"zui4",
"can3",
"xu4",
"hui4",
"yin3",
"qie4",
"fen1",
"pi2",
"yue4",
"you3",
"ruan3",
"peng2",
"fen2",
"fu2,fu4",
"ling2",
"fei3",
"qu2",
"ti4",
"nv4",
"tiao4",
"shuo4",
"zhen4",
"lang3",
"lang3",
"juan1",
"ming2",
"huang1",
"wang4",
"tun1",
"chao2,zhao1",
"ji1",
"qi1",
"ying1",
"zong1",
"wang4",
"tong2",
"lang3",
"lao2",
"meng2",
"long2",
"mu4",
"xx5",
"wei4",
"mo4",
"ben3",
"zha2",
"shu4,zhu2",
"shu4,zhu2",
"xx5",
"zhu1",
"ren2",
"ba1",
"pu3,piao2,po4",
"duo3",
"duo3",
"dao1",
"li4",
"gui3",
"ji1",
"jiu1",
"bi3",
"xiu3",
"ting2",
"ci4",
"sha1",
"ru4",
"za2",
"quan2",
"qian1",
"yu2",
"gan1,gan3",
"wu1",
"cha1,cha4",
"shan1",
"xun2",
"fan2",
"wu4",
"zi3",
"li3",
"xing4",
"cai2",
"cun1",
"ren4",
"biao1,shao2",
"tuo1",
"di4",
"zhang4",
"mang2",
"chi4",
"yi4",
"gai4",
"gong1",
"du4",
"yi2",
"qi3",
"shu4",
"gang1,gang4",
"tiao2",
"jiang5",
"mian2",
"wan4",
"lai2",
"jiu3",
"mang2",
"yang2",
"ma4",
"miao3",
"si4",
"yuan2",
"hang2",
"fei4",
"bei1",
"jie2",
"dong1",
"gao3",
"yao3",
"xian1",
"chu3",
"chun1",
"pa2,ba4",
"shu1",
"hua4",
"xin1",
"chou3,niu3",
"zhu4",
"chou3",
"song1",
"ban3,pan4",
"song1",
"ji2",
"wo4",
"jin4",
"gou4",
"ji1",
"mao2",
"pi2",
"bi4",
"wang3",
"ang4",
"fang1",
"fen2",
"yi4",
"fu2",
"nan2",
"xi1",
"hu4",
"ya2",
"dou3",
"xin2",
"zhen3",
"yao1",
"lin2",
"rui4",
"e4",
"mei2",
"zhao4",
"guo3",
"zhi1",
"cong1",
"yun4",
"zui5",
"sheng1",
"shu1",
"zao3",
"di4",
"li4",
"lu2",
"jian3",
"cheng2",
"song1",
"qiang1",
"feng1",
"zhan1",
"xiao1",
"xian1",
"ku1",
"ping2",
"tai2",
"xi3",
"zhi3",
"guai3",
"xiao1",
"jia4",
"jia1",
"gou3,gou1,ju3",
"fu1",
"mo4",
"yi4",
"ye4",
"ye4",
"shi4",
"nie4",
"bi3",
"tuo2",
"yi2",
"ling2",
"bing3",
"ni3",
"la1",
"he2",
"pan2",
"fan2",
"zhong1",
"dai4",
"ci2",
"yang3",
"fu1",
"bai3,bo2,bo4",
"mou3",
"gan1",
"qi1",
"ran3",
"rou2",
"mao4",
"shao2",
"song1",
"zhe4",
"xia2",
"you4",
"shen1",
"gui4,ju3",
"tuo4",
"zuo4",
"nan2",
"ning2",
"yong3",
"di3",
"zhi2",
"zha1",
"cha2,zha1",
"dan4",
"gu1",
"bu4",
"jiu4",
"ao1",
"fu2",
"jian3",
"ba1",
"duo4",
"ke1",
"nai4",
"zhu4",
"bi4",
"liu3",
"chai2",
"shan1",
"si4",
"zhu4",
"pei1",
"shi4",
"guai3",
"cha2",
"yao3",
"cheng1",
"jiu4",
"shi4",
"zhi1",
"liu3",
"mei2",
"li4",
"rong2",
"zha4",
"zao3",
"biao1",
"zhan4",
"zhi4",
"long2",
"dong4",
"lu2",
"sheng1",
"li4",
"lan2",
"yong3",
"shu4",
"xun2",
"shuan1",
"qi4",
"zhen1",
"qi1",
"li4",
"yi2",
"xiang2",
"zhen4",
"li4",
"se4",
"gua1",
"kan1",
"bing1",
"ren3",
"xiao4,jiao4",
"bai3",
"ren3",
"bing4",
"zi1",
"chou2",
"yi4",
"ci4",
"xu3",
"zhu1",
"jian4",
"zui4",
"er2",
"er3",
"you3",
"fa2",
"gong3",
"kao3",
"lao3",
"zhan1",
"li4",
"yin1",
"yang4",
"he2",
"gen1",
"yi4",
"shi4",
"ge2",
"zai1",
"luan2",
"fu2",
"jie2",
"heng2,hang2",
"gui4",
"tao2",
"guang1,guang4",
"wei2",
"kuang4",
"ru2",
"an4",
"an1",
"juan4",
"yi2",
"zhuo1",
"ku1",
"zhi4",
"qiong2",
"tong2",
"sang1",
"sang1",
"huan2",
"ju2,jie2",
"jiu4",
"xue4",
"duo4",
"zhui4",
"yu2",
"zan3",
"",
"ying1",
"jie2",
"liu3",
"zhan4",
"ya1",
"rao2",
"zhen1",
"dang4,dang3",
"qi1",
"qiao2",
"hua4",
"gui4",
"jiang3",
"zhuang1",
"xun2",
"suo1",
"sha1",
"zhen1",
"bei1",
"ting1",
"kuo4",
"jing4",
"bo2",
"ben4",
"fu2",
"rui2",
"tong3",
"jue2",
"xi1",
"lang2",
"liu3",
"feng1",
"qi1",
"wen3",
"jun1",
"gan3",
"su4",
"liang2",
"qiu2",
"ting3",
"you3",
"mei2",
"bang1",
"long4",
"peng1",
"zhuang1",
"di4",
"xuan1",
"tu2",
"zao4",
"ao1",
"gu4",
"bi4",
"di2",
"han2",
"zi3",
"zhi1",
"ren4",
"bei4",
"geng3",
"jian3",
"huan4",
"wan3",
"nuo2",
"jia1",
"tiao2",
"ji4",
"xiao1",
"lu:3",
"kuan3",
"shao1",
"chen2",
"fen2",
"song1",
"meng4",
"wu2",
"li2",
"li2",
"dou4",
"qin3",
"ying3",
"suo1",
"ju1",
"ti1",
"xie4",
"kun3",
"zhuo2",
"shu1",
"chan1",
"fan4",
"wei3",
"jing4",
"li2",
"bing1",
"xia4",
"fo2",
"tao2",
"zhi4",
"lai2",
"lian2",
"jian3",
"zhuo1",
"ling2",
"li2",
"qi4",
"bing3",
"lun2",
"cong1",
"qian4",
"mian2",
"qi2",
"qi2",
"cai4",
"gun4",
"chan2",
"de2",
"fei3",
"pai2",
"bang4",
"bang4",
"hun1",
"zong1",
"cheng2",
"zao3",
"ji2",
"li4",
"peng2",
"yu4",
"yu4",
"gu4",
"jun4",
"dong4",
"tang2",
"gang1",
"wang3",
"di4",
"cuo4",
"fan2",
"cheng1",
"zhan4",
"qi3",
"yuan1",
"yan3",
"yu4",
"quan1",
"yi4",
"sen1",
"ren3",
"chui2",
"leng2",
"qi1",
"zhuo2",
"fu2",
"ke1",
"lai2",
"zou1",
"zou1",
"zhao4,zhuo1",
"guan1",
"fen1",
"fen2",
"shen1",
"qing2",
"ni2",
"wan3",
"guo3",
"lu4",
"hao2",
"jie1",
"yi3",
"chou2",
"ju3",
"ju2",
"cheng2",
"zuo2",
"liang2",
"qiang1",
"zhi2",
"chui2,zhui1",
"ya1",
"ju1",
"bei1",
"jiao1",
"zhuo2",
"zi1",
"bin1",
"peng2",
"ding4",
"chu3",
"chang1",
"men1",
"hua1",
"jian3",
"gui1",
"xi4",
"du2",
"qian4",
"dao4",
"gui4",
"dian3",
"luo2",
"zhi1",
"quan5",
"ming4",
"fu3",
"geng1",
"peng4",
"zhan3",
"yi2",
"tuo3",
"sen1",
"duo3",
"ye1",
"fu4",
"wei3",
"wei1",
"duan4",
"jia3",
"zong1",
"jian1",
"yi2",
"shen4",
"xi2",
"yan4",
"yan3",
"chuan2",
"jian1",
"chun1",
"yu3",
"he2",
"zha1,cha2",
"wo4",
"pian2",
"bi4",
"yao1",
"huo4",
"xu1",
"ruo4",
"yang2",
"la4",
"yan2",
"ben3",
"hun2",
"kui2",
"jie4",
"kui2",
"si1",
"feng1",
"xie1",
"tuo3",
"zhi4",
"jian4",
"mu4",
"mao4",
"chu3",
"hu4,ku3",
"hu2",
"lian4",
"leng2,leng4",
"ting2",
"nan2",
"yu2",
"you2",
"mei2",
"song3",
"xuan4",
"xuan4",
"yang3",
"zhen1",
"pian2",
"die2",
"ji2",
"jie2",
"ye4",
"chu3",
"shun3",
"yu2",
"zou4",
"wei1",
"mei2",
"ti4",
"ji2",
"jie2",
"kai3,jie1",
"qiu1",
"ying2",
"rou3",
"huang2",
"lou2",
"le4,yue4",
"quan2",
"xiang1",
"pin3",
"shi3",
"gai4",
"tan2",
"lan3",
"wen1",
"yu2",
"chen4",
"lu:2",
"ju3",
"shen2",
"chu5",
"bi1",
"xie4",
"jia3",
"yi4",
"zhan3",
"fu2",
"nuo4",
"mi4",
"lang2",
"rong2",
"gu3",
"jian4",
"ju3",
"ta1",
"yao3",
"zhen1",
"bang3",
"sha1",
"yuan2",
"zi3",
"ming2",
"su4",
"jia4",
"yao2",
"jie2",
"huang3",
"gan4",
"fei3",
"zha4",
"qian2",
"ma4",
"sun3",
"yuan2",
"xie4",
"rong2",
"shi2",
"zhi1",
"cui1",
"wen1",
"ting2",
"liu2",
"rong2",
"tang2",
"que4",
"zhai1",
"si1",
"sheng4",
"ta4",
"ke4",
"xi1",
"gu3",
"qi1",
"gao3",
"gao3",
"sun1",
"pan2",
"tao1",
"ge2",
"chun1",
"dian1",
"nou4",
"ji2",
"shuo4",
"gou4",
"chui2",
"qiang1",
"cha2",
"qian3",
"huai2",
"mei2",
"xu4",
"gang4",
"gao1",
"zhuo1",
"tuo2",
"qiao2",
"yang4",
"dian1",
"jia3",
"kan3,jian4",
"zui4",
"dao3",
"long2",
"bing1",
"zhu1",
"sang1",
"xi2",
"ji1",
"lian2",
"hui4",
"yong1",
"qian4",
"guo3",
"gai4",
"gai4",
"tuan2",
"hua4",
"qi4",
"sen1",
"cui1",
"peng2",
"you3",
"hu2",
"jiang3",
"hu4",
"huan4",
"gui4",
"nie4",
"yi4",
"gao1",
"kang1",
"gui1",
"gui1",
"cao2",
"man2",
"jin3",
"di2",
"zhuang1",
"le4",
"lang3",
"chen2",
"cong1",
"li2,chi1",
"xiu1",
"qing2",
"shuang3",
"fan2",
"tong1",
"guan4",
"ze2",
"su4",
"lei3",
"lu3",
"liang2",
"mi4",
"lou2",
"chao2",
"su4",
"ke1",
"chu1",
"tang2,cheng3",
"biao1",
"lu4",
"jiu1",
"zhe4",
"zha1",
"shu1",
"zhang1",
"men2",
"mo2,mu2",
"niao3",
"yang4",
"tiao2",
"peng2",
"zhu4",
"sha1",
"xi1",
"quan2",
"heng2,heng4",
"jian1",
"cong1",
"ji1",
"yan1",
"qiang2",
"xue3",
"ying1",
"er4",
"xun2",
"zhi2",
"qiao2",
"zui1",
"cong2",
"pu3",
"shu4",
"hua4",
"gui4",
"zhen1",
"zun1",
"yue4",
"shan4",
"xi1",
"xun2",
"dian4",
"fa2",
"gan3",
"mo2",
"wu3",
"qiao1",
"rao2",
"lin4",
"liu2",
"qiao2",
"xian4",
"run4",
"fan3",
"zhan3",
"tuo2",
"lao3",
"yun2",
"shun4",
"dun1",
"cheng1",
"tang2",
"meng2",
"ju2",
"cheng2",
"su4",
"jue2",
"jue2",
"dian4",
"hui4",
"ji1",
"nuo3",
"xiang4",
"tuo3",
"ning3",
"rui3",
"zhu1",
"tong2",
"zeng1",
"fen2",
"qiong2",
"ran3",
"heng2",
"qian2",
"gu1",
"liu3",
"lao4",
"gao1",
"chu2",
"xi3",
"sheng4",
"zi3",
"san5",
"ji2",
"dou1",
"jing1",
"lu3",
"jian5",
"chu5",
"yuan2",
"ta4",
"shu1,qiao1",
"jiang1",
"tan2",
"lin3",
"nong2",
"yin3",
"xi2",
"hui4",
"shan1",
"zui4",
"xuan2",
"cheng1",
"gan4",
"ju2",
"zui4",
"yi4",
"qin2",
"pu3",
"yan2",
"lei2",
"feng1",
"hui3",
"dang4",
"ji4",
"sui4",
"bo4",
"ping2",
"cheng2",
"chu3",
"zhua1",
"gui4",
"ji2",
"jie3",
"jia3",
"qing2",
"zhai2",
"jian3",
"qiang2",
"dao4",
"yi3",
"biao3",
"song1",
"she1",
"lin3",
"li4",
"cha2",
"meng2",
"yin2",
"tao2",
"tai2",
"mian2",
"qi2",
"tuan2",
"bin1",
"huo4",
"ji4,qi3",
"qian1",
"ni3",
"ning2",
"yi1",
"gao3",
"kan3",
"yin4",
"nou4",
"qing3",
"yan3",
"qi2",
"mi4",
"zhao4",
"gui4",
"chun1",
"ji1",
"kui2",
"po2",
"deng4",
"chu2",
"ge2",
"mian2",
"you1",
"zhi4",
"huang3",
"qian1",
"lei3",
"lei2",
"sa4",
"lu3",
"li4",
"cuan2",
"lv4",
"mie4",
"hui4",
"ou1",
"lv2",
"zhi4",
"gao1",
"du2",
"yuan2",
"li4",
"fei4",
"zhu4",
"sou3",
"lian2",
"jiang4",
"chu2",
"qing4",
"zhu1",
"lu2",
"yan2",
"li4",
"zhu1",
"chen4",
"jie2",
"e4",
"su1",
"huai2",
"nie4",
"yu4",
"long2",
"lai4",
"jiao5",
"xian3",
"gui1",
"ju3",
"xiao1",
"ling2",
"ying1",
"jian1",
"yin3",
"you2",
"ying2",
"xiang1",
"nong2",
"bo2",
"chan2",
"lan2",
"ju3",
"shuang1",
"she4",
"wei2",
"cong2",
"quan2",
"qu2",
"cang2",
"jiu4",
"yu4",
"luo2",
"li4",
"zan4",
"luan2",
"dang3",
"jue2",
"yan2",
"lan3",
"lan2",
"zhu2",
"lei2",
"li3",
"ba4",
"nang2",
"yu4",
"ling2",
"guang5",
"qian4",
"ci4",
"huan1",
"xin1",
"yu2",
"yi4",
"qian1",
"ou1",
"xu1",
"chao1",
"chu4",
"qi4",
"kai4",
"yi4",
"jue2",
"xi4",
"xu4",
"he1",
"yu4",
"kui4",
"lang2",
"kuan3",
"shuo4",
"xi1",
"ai3",
"yi1",
"qi1",
"chua1,xu1",
"chi3",
"qin1",
"kuan3",
"kan3",
"kuan3",
"kan3",
"chuan3",
"sha4",
"gua1",
"yin1",
"xin1",
"xie1",
"yu2",
"qian4",
"xiao1",
"ye4",
"ge1",
"wu1",
"tan4",
"jin4",
"ou1",
"hu1",
"ti4",
"huan1",
"xu1",
"pen1",
"xi3",
"xiao4",
"xu1",
"she4",
"shan4",
"han1,lian3",
"chu4",
"yi4",
"e4",
"yu2",
"chuo4",
"huan1",
"zhi3",
"zheng4,zheng1",
"ci3",
"bu4",
"wu3",
"qi2",
"bu4",
"bu4",
"wai1,wai3",
"ju4",
"qian2",
"chi2",
"se4",
"chi3",
"se4",
"zhong3",
"sui4",
"sui4",
"li4",
"ze2",
"yu2",
"li4",
"gui1",
"dai3",
"can1,dai3",
"si3",
"jian1",
"zhe2",
"mo4",
"mo4",
"yao1",
"mo4",
"cu2",
"yang1",
"tian3",
"sheng1",
"dai4",
"shang1",
"xu4",
"xun4",
"shu1",
"can2",
"jue2",
"piao3",
"qia4",
"qiu2",
"su4",
"qing2",
"yun3",
"lian4",
"yi4",
"fou3",
"zhi2",
"ye4",
"can2",
"hun1",
"dan1",
"ji2",
"die2",
"zhen1",
"yun3",
"wen1",
"chou4",
"bin4",
"ti4",
"jin4",
"shang1",
"yin2",
"diao1",
"jiu4",
"hui4",
"cuan4",
"yi4",
"dan1",
"du4",
"jiang1",
"lian4",
"bin4",
"du2",
"jian1",
"jian1",
"shu1",
"ou1",
"duan4",
"zhu4",
"yin1,yan1,yin3",
"qing4",
"yi4",
"sha1",
"qiao4",
"ke2",
"xiao2",
"xun4",
"dian4",
"hui3",
"hui3",
"gu3,gu1",
"qiao1",
"ji1",
"yi4",
"ou1",
"hui3",
"duan4",
"yi1",
"xiao1",
"wu2",
"guan4",
"mu3",
"mei3",
"mei3",
"ai3",
"jie3",
"du2",
"yu4",
"bi3,bi1,bi4",
"bi4",
"bi4",
"pi2",
"pi2",
"bi4",
"chan2",
"mao2",
"hao2",
"cai3",
"pi2",
"lie3",
"jia1",
"zhan1",
"sai1",
"mu4",
"tuo4",
"xun2",
"er3",
"rong2",
"xian3",
"ju2",
"mu2",
"hao2",
"qiu2",
"dou4",
"sha1",
"tan3",
"pei2",
"ju2",
"duo1",
"cui4",
"bi1",
"san1",
"san1",
"mao4",
"sai1",
"shu1",
"shu1",
"tuo4",
"he2",
"jian4",
"ta4",
"san1",
"lv2",
"mu2",
"mao2",
"tong2",
"rong3",
"chang3",
"pu3",
"lu3",
"zhan1",
"sao4",
"zhan1",
"meng2",
"lu3",
"qu2",
"die2",
"shi4,zhi1",
"di1,di3",
"min2",
"jue2",
"mang2,meng2",
"qi4",
"pie1",
"nai3",
"qi4",
"dao1",
"xian1",
"chuan1",
"fen1",
"yang2,ri4",
"nei4",
"bin5",
"fu2",
"shen1",
"dong1",
"qing1",
"qi4",
"yin1",
"xi1",
"hai4",
"yang3",
"an1",
"ya4",
"ke4",
"qing1",
"ya4",
"dong1",
"dan4",
"lu:4",
"qing2",
"yang3",
"yun1",
"yun1",
"shui3",
"shui3",
"zheng3",
"bing1",
"yong3",
"dang4",
"shui3",
"le4",
"ni4",
"tun3",
"fan4",
"gui3",
"ting1",
"zhi1",
"qiu2",
"bin1",
"ze4",
"mian3",
"cuan1",
"hui4",
"diao1",
"han4",
"cha4",
"zhuo2",
"chuan4",
"wan2",
"fan4",
"da4",
"xi1",
"tuo1",
"mang2",
"qiu2",
"qi4",
"shan4",
"pin4",
"han4,han2",
"qian1",
"wu1",
"wu1",
"xun4",
"si4",
"ru3",
"gong3,hong4",
"jiang1",
"chi2",
"wu1",
"tu5",
"jiu3",
"tang1,shang1",
"zhi1",
"zhi3",
"qian1",
"mi4",
"gu3",
"wang1",
"jing3",
"jing3",
"rui4",
"jun1",
"hong2",
"tai4",
"quan3",
"ji2",
"bian4",
"bian4",
"gan4",
"wen4",
"zhong1",
"fang1",
"xiong1",
"jue2",
"hu3",
"niu2",
"qi4",
"fen2",
"xu4",
"xu4",
"qin4",
"yi2",
"wo4",
"yun2",
"yuan2",
"hang4",
"yan3",
"shen3,chen2",
"chen2,chen1",
"dan4",
"you2",
"dun4",
"hu4",
"huo4",
"qi1",
"mu4",
"nv4",
"mei2",
"da2,ta4",
"mian3",
"wu4",
"chong1",
"pang1",
"bi3",
"sha1",
"zhi3",
"pei4",
"pan4",
"zhui3",
"za1",
"gou1",
"liu2",
"mei2,mo4",
"ze2",
"feng1",
"ou1,ou4",
"li4",
"lun2",
"cang1",
"feng1",
"gui1",
"hu4",
"mo4",
"mei4",
"shu4",
"ju3",
"za2",
"tuo1",
"tuo2",
"tuo2",
"he2",
"li4",
"mi3",
"yi2",
"fa1",
"fei4",
"you2",
"tian2",
"zhi4",
"zhao3",
"gu1",
"zhan1",
"yan2",
"si1",
"kuang4",
"jiong3",
"ju1",
"xie4",
"qiu2",
"yi4",
"jia1",
"zhong1",
"quan2",
"po1,bo2",
"hui4",
"mi4",
"ben1",
"ze2",
"zhu2",
"le4",
"you3",
"gu1",
"hong2",
"gan1",
"fa3",
"mao3",
"si4",
"hu1",
"ping2",
"ci3",
"fan4",
"zhi1",
"su4",
"ning4",
"cheng1",
"ling2",
"pao4,pao1",
"bo1",
"qi4",
"si4",
"ni2,ni4",
"ju2",
"sa4",
"zhu4",
"sheng1",
"lei4",
"xuan4",
"jue2",
"fu1",
"pan4",
"min3",
"tai4",
"yang1",
"ji3",
"yong3",
"guan4",
"beng4",
"xue2",
"long2,shuang1",
"lu2",
"dan4",
"luo4",
"xie4",
"po1",
"ze2",
"jing1",
"yin2",
"pan2",
"jie2",
"ye4",
"hui1",
"hui2",
"zai4",
"cheng2",
"yin1,yan1",
"wei2",
"hou4",
"jian4",
"yang2",
"lie4",
"si4",
"ji4",
"er2",
"xing2",
"fu2",
"sa3",
"se4",
"zhi3",
"yin4",
"wu2",
"xi3",
"kao3",
"zhu1",
"jiang4",
"luo4",
"luo4",
"an4",
"dong4",
"ti4",
"mou2",
"lei4",
"yi1",
"mi3",
"quan2",
"jin1",
"po4",
"wei3",
"xiao2",
"xie4",
"hong2",
"xu4",
"su4",
"kuang1",
"tao2",
"qie4",
"ju4",
"er3",
"zhou1",
"ru4",
"ping2",
"xun2",
"xiong1",
"zhi4",
"guang1",
"huan2",
"ming2",
"huo2",
"wa1",
"qia4",
"pai4",
"wu1",
"qu1",
"liu2",
"yi4",
"jia1",
"jing4",
"qian3,jian1",
"jiang1,jiang4",
"jiao1",
"zhen1",
"shi1",
"zhuo2",
"ce4",
"fa2",
"kuai4",
"ji4,ji3",
"liu2",
"chan3",
"hun2",
"hu3",
"nong2",
"xun2",
"jin4",
"lie4",
"qiu2",
"wei3",
"zhe4",
"jun4",
"han2",
"bang1,bin1",
"mang2",
"zhuo2",
"you2",
"xi1",
"bo2",
"dou4",
"huan4",
"hong2",
"yi4",
"pu3",
"ying3",
"lan3",
"hao4",
"lang4",
"han3",
"li3",
"geng1",
"fu2",
"wu2",
"li4",
"chun2",
"feng2",
"yi4",
"yu4",
"tong2",
"lao2",
"hai3",
"jin4",
"jia1",
"chong1",
"jiong3",
"mei3",
"sui1",
"cheng1",
"pei4",
"xian4",
"shen4",
"tu2",
"kun4",
"ping1",
"nie4",
"han4",
"jing1",
"xiao1",
"she4",
"nian3",
"tu1",
"yong3,chong1",
"xiao4",
"xian2",
"ting3",
"e2",
"su4",
"tun1",
"juan1",
"cen2",
"ti4",
"li4",
"shui4",
"si4",
"lei4",
"shui4",
"tao1",
"du2",
"lao4",
"lai2",
"lian2",
"wei2",
"wo1,guo1",
"yun2",
"huan4",
"di2",
"heng1",
"run4",
"jian4",
"zhang3,zhang4",
"se4",
"fu2",
"guan4",
"xing4",
"shou4",
"shuan4",
"ya2",
"chuo4",
"zhang4",
"ye4",
"kong1",
"wo4",
"han2",
"tuo4",
"dong1",
"he2",
"wo1",
"ju1",
"she4",
"liang2",
"hun1",
"ta4",
"zhuo1",
"dian4",
"qie4",
"de2",
"juan4",
"zi1",
"xi1",
"xiao2",
"qi2",
"gu3",
"guo3",
"yan1",
"lin2,lin4",
"tang3",
"zhou1",
"peng3",
"hao4",
"chang1",
"shu1",
"qi1",
"fang1",
"zhi2",
"lu4",
"nao4",
"ju2",
"tao2",
"cong2",
"lei4",
"zhe4",
"peng2",
"fei2",
"song1",
"tian3",
"pi4",
"dan4",
"yu4",
"ni2",
"yu1",
"lu4",
"gan4",
"mi4",
"jing4",
"ling2",
"lun2",
"yin2",
"cui4",
"qu2",
"huai2",
"yu4",
"nian3",
"shen1",
"biao1",
"chun2",
"hu1",
"yuan1",
"lai2",
"hun4,hun2",
"qing1",
"yan1",
"qian3",
"tian1",
"miao3",
"zhi3",
"yin3",
"bo2",
"ben4",
"yuan1",
"wen4",
"ruo4",
"fei1",
"qing1",
"yuan1",
"ke3",
"ji4",
"she4",
"yuan1",
"se4",
"lu4",
"zi4",
"du2",
"yi1",
"jian4,jian1",
"sheng2",
"pai4",
"xi1",
"yu2",
"yuan1",
"shen3",
"shen4",
"rou2",
"huan4",
"zhu3",
"jian3",
"nuan3",
"yu2",
"qiu2",
"ting2",
"qu2,ju4",
"du4",
"fan2",
"zha1",
"bo2",
"wo4",
"wo1",
"di4",
"wei1",
"wen1",
"ru2",
"xie4",
"ce4",
"wei4",
"ge1",
"gang3",
"yan3",
"hong2",
"xuan4",
"mi3",
"ke3",
"mao2",
"ying1",
"yan3",
"you2",
"hong1",
"miao3",
"sheng3",
"mei3",
"zai1",
"hun2",
"nai4",
"gui3",
"chi4",
"e4",
"pai4",
"mei2",
"lian4",
"qi4",
"qi4",
"mei2",
"tian2",
"cou4",
"wei2",
"can1",
"tuan1",
"mian3",
"hui4",
"mo4",
"xu3",
"ji2",
"pen2",
"jian1",
"jian3",
"hu2",
"feng4",
"xiang1",
"yi4",
"yin4",
"zhan4",
"shi2",
"jie1",
"zhen1",
"huang2",
"tan4",
"yu2",
"bi4",
"min3",
"shi1",
"tu1",
"sheng1",
"yong3",
"ju2",
"dong4",
"tuan4",
"jiao3",
"jiao3",
"qiu2",
"yan1,yin1",
"tang1",
"long2",
"huo4",
"yuan2",
"nan3",
"ban4,pan2",
"you3",
"quan2",
"zhuang1",
"liang4",
"chan2",
"xian2",
"chun2",
"nie4",
"zi1",
"wan1",
"shi1",
"man3",
"ying2",
"la4",
"kui4",
"feng2",
"jian4",
"xu4",
"lou2,lu:3",
"wei2",
"gai4",
"bo1",
"ying2",
"po1",
"jin4",
"gui4",
"tang2",
"yuan2",
"suo3",
"yuan2",
"lian2",
"yao3",
"meng2",
"zhun3",
"cheng2",
"ke4",
"tai4",
"ta3",
"wa1",
"liu1",
"gou1",
"sao1",
"ming2",
"zha4",
"shi2",
"yi4",
"lun4",
"ma3",
"pu3",
"wei1",
"li4",
"zai1",
"wu4",
"xi1",
"wen1",
"qiang1",
"ze2",
"shi1",
"su4",
"ai2",
"zhen1",
"sou1",
"yun2",
"xiu4",
"yin1",
"rong2",
"hun4",
"su4",
"suo4",
"ni4,niao4",
"ta1",
"shi1",
"ru4",
"wei1",
"pan4",
"chu4",
"chu2",
"pang1",
"weng3",
"cang1",
"mie4",
"ge2",
"dian1",
"hao4",
"huang3",
"xi4,xie1",
"zi1",
"di2",
"zhi4",
"xing2",
"fu3",
"jie2",
"hua2",
"ge1",
"zi3",
"tao1",
"teng2",
"sui1",
"bi4",
"jiao4",
"hui4",
"gun3",
"yin2",
"gao1",
"long2",
"zhi4",
"yan4",
"she4",
"man3",
"ying2",
"chun2",
"lu:4",
"lan4",
"luan2",
"yao2",
"bin1",
"tan1",
"yu4",
"xiu3",
"hu4",
"bi4",
"biao1",
"zhi4",
"jiang4",
"kou4",
"shen4",
"shang1",
"di1",
"mi4",
"ao2",
"lu3",
"hu3",
"hu1",
"you1",
"chan3",
"fan4",
"yong1",
"gun3",
"man3",
"qing3",
"yu2",
"piao4,piao1,piao3",
"ji4",
"ya2",
"chao2",
"qi1",
"xi3",
"ji4",
"lu4",
"lou2",
"long2",
"jin3",
"guo2",
"cong2",
"lou4",
"zhi2",
"gai4",
"qiang2",
"li2",
"yan3",
"cao2",
"jiao4",
"cong1",
"chun2",
"tuan2",
"ou1",
"teng2",
"ye3",
"xi2",
"mi4",
"tang2",
"mo4",
"shang1",
"han4",
"lian2",
"lan3",
"wa1",
"li2",
"gan1",
"feng2",
"xuan2",
"yi1",
"man4",
"zi4",
"mang3",
"kang1",
"ta4",
"peng1",
"shu4",
"zhang3",
"zhang1",
"zhuang4",
"xu4",
"huan4",
"kuo4",
"jian4",
"yan1",
"shuang3",
"liao2",
"cui3",
"ti2",
"yang4",
"jiang1",
"cong2",
"ying3",
"hong2",
"xiu3",
"shu4",
"guan4",
"ying2",
"xiao1",
"zong5",
"kun1",
"xu4",
"lian4",
"zhi4",
"wei2",
"pi4",
"jue2",
"jiao4",
"po1",
"dang4",
"hui4",
"jie2",
"wu3",
"pa2",
"ji2",
"pan1",
"wei2",
"su4",
"qian2",
"qian2",
"xi1",
"lu4",
"xi4",
"xun4",
"dun4",
"huang2",
"min3",
"run4",
"su4",
"lao3",
"zhen1",
"cong2",
"yi4",
"zhe4",
"wan1",
"shan4",
"tan2",
"chao2",
"xun2",
"kui4",
"ye1",
"shao4",
"tu2",
"zhu1",
"sa3,san4",
"hei1",
"bi4",
"shan1",
"chan2",
"chan2",
"shu3",
"tong2",
"pu1",
"lin2",
"wei2",
"se4",
"se4",
"cheng2",
"jiong3",
"cheng2,deng4",
"hua4",
"jiao1",
"lao4",
"che4",
"gan3",
"cun1",
"hong4",
"si1",
"shu4",
"peng2",
"han2",
"yun2",
"liu4",
"hong4",
"fu2",
"hao4",
"he2",
"xian2",
"jian4",
"shan1",
"xi4",
"yu5",
"lu3",
"lan2",
"ning4",
"yu2",
"lin3",
"mian3",
"zao3",
"dang1",
"huan4",
"ze2",
"xie4",
"yu4",
"li3",
"shi4",
"xue2",
"ling2",
"wan4",
"zi1",
"yong1",
"hui4",
"can4",
"lian4",
"dian4",
"ye4",
"ao4",
"huan2",
"zhen1",
"chan2",
"man4",
"dan3",
"dan4,tan2",
"yi4",
"sui4",
"pi4",
"ju4",
"ta4",
"qin2",
"ji1",
"zhuo2",
"lian2",
"nong2",
"guo1",
"jin4",
"fen2",
"se4",
"ji2",
"sui1",
"hui4",
"chu3",
"ta4",
"song1",
"ding3",
"se4",
"zhu3",
"lai4",
"bin1",
"lian2",
"mi3",
"shi1",
"shu4",
"mi4",
"ning4",
"ying2",
"ying2",
"meng2",
"jin4",
"qi2",
"bi4",
"ji4",
"hao2",
"ru2",
"cui4",
"wo4",
"tao1",
"yin3",
"yin3",
"dui4",
"ci2",
"huo4",
"qing4",
"lan4",
"jun4",
"ai3",
"pu2",
"zhuo2,zhao4",
"wei2",
"bin1",
"gu3",
"qian2",
"ying2",
"bin1",
"kuo4",
"fei4",
"cang1",
"me5",
"jian4",
"wei3",
"luo4",
"zan4",
"lv4",
"li4",
"you1",
"yang4",
"lu3",
"si4",
"zhi4",
"ying2",
"du2",
"wang3",
"hui1",
"xie4",
"pan2",
"shen3",
"biao1",
"chan2",
"mo4",
"liu2",
"jian1",
"pu4,bao4",
"se4",
"cheng2",
"gu3",
"bin1",
"huo4",
"xian4",
"lu2",
"qin4",
"han4",
"ying2",
"rong2",
"li4",
"jing4",
"xiao1",
"ying2",
"sui3",
"wei3",
"xie4",
"huai2",
"xue4",
"zhu1",
"long2",
"lai4",
"dui4",
"fan2",
"hu2",
"lai4",
"shu1",
"ling5",
"ying2",
"mi2",
"ji4",
"lian4",
"jian4",
"ying2",
"fen4",
"lin2",
"yi4",
"jian1",
"yue4",
"chan2",
"dai4",
"rang2",
"jian3",
"lan2",
"fan2",
"shuang4",
"yuan1",
"zhuo2",
"feng1",
"she4",
"lei3",
"lan2",
"cong2",
"qu2",
"yong1",
"qian2",
"fa3",
"guan4",
"jue2",
"yan4",
"hao4",
"ying2",
"sa3",
"zan4",
"luan2",
"yan4",
"li2",
"mi3",
"shan4",
"tan1",
"dang3",
"jiao3",
"chan3",
"ying2",
"hao4",
"ba4",
"zhu2",
"lan3,lan4",
"lan2",
"nang3",
"wan1",
"luan2",
"xun2",
"xian3",
"yan4",
"gan4",
"yan4",
"yu4",
"huo3",
"biao1",
"mie4",
"guang1",
"deng1",
"hui1",
"xiao1",
"xiao1",
"hui1",
"hong2",
"ling2",
"zao4",
"zhuan4",
"jiu3",
"zha4",
"xie4",
"chi4",
"zhuo2",
"zai1",
"zai1",
"can4",
"yang2",
"qi4",
"zhong1",
"fen2",
"niu3",
"jiong3,gui4",
"wen2",
"pu1",
"yi4",
"lu2",
"chui1",
"pi1",
"kai4",
"pan4",
"yan2",
"yan2",
"pang4",
"mu4",
"chao3",
"liao4",
"gui4,que1",
"kang4",
"dun4",
"guang1",
"xin1",
"zhi4",
"guang1",
"guang1",
"wei3",
"qiang4",
"bian5",
"da2",
"xia2",
"zheng1",
"zhu2",
"ke3",
"zhao4",
"fu2",
"ba2",
"xie4",
"xie4",
"ling4",
"zhuo1",
"xuan4",
"ju4",
"tan4",
"pao4,bao1,pao2",
"jiong3",
"pao2",
"tai2",
"tai2",
"bing3",
"yang3",
"tong1",
"shan3",
"zhu4",
"zha4,zha2",
"dian3",
"wei4",
"shi2",
"lian4",
"chi4",
"huang3",
"zhou1",
"hu1",
"shuo4",
"lan4",
"ting1",
"jiao3",
"xu4",
"heng2",
"quan3",
"lie4",
"huan4",
"yang2",
"xiao1",
"xiu1",
"xian3",
"yin2",
"wu1",
"zhou1",
"yao2",
"shi4",
"wei1",
"tong2",
"mie4",
"zai1",
"kai4",
"hong1",
"lao4",
"xia2",
"zhu2",
"xuan3",
"zheng1",
"po4",
"yan1",
"hui2",
"guang1",
"che4",
"hui1",
"kao3",
"ju4",
"fan2",
"shao1",
"ye4",
"hui4",
"",
"tang4",
"jin4",
"re4",
"lie4",
"xi1",
"fu2",
"jiong3",
"xie4",
"pu3",
"ting1",
"zhuo2",
"ting3",
"wan2",
"hai3",
"peng1",
"lang3",
"yan4",
"xu4",
"feng1",
"chi4",
"rong2",
"hu2",
"xi1",
"shu1",
"he4",
"xun1",
"ku4",
"juan1",
"xiao1",
"xi1",
"yan1",
"han4",
"zhuang4",
"jun4,qu1",
"di4",
"xie4",
"ji2",
"wu4",
"yan1",
"lv3",
"han2",
"yan4",
"huan4",
"men4",
"ju2",
"dao4",
"bei4",
"fen2",
"lin4",
"kun1",
"hun4",
"tun1",
"xi1",
"cui4",
"wu2",
"hong1,kong4",
"chao3",
"fu3",
"wo4",
"jiao1",
"cong1",
"feng4",
"ping2",
"qiong2",
"ruo4",
"xi1",
"qiong2",
"xin4",
"chao1",
"yan4",
"yan4",
"yi4",
"jue2",
"yu4",
"gang4",
"ran2",
"pi2",
"xiong4",
"gang4",
"sheng1",
"chang4",
"shao1",
"xiong3",
"nian3",
"geng1",
"wei5",
"chen2",
"he4",
"kui3",
"zhong3",
"duan4",
"xia1,ya1",
"hui1",
"feng4",
"lian4",
"xuan1",
"xing1",
"huang2",
"jiao3",
"jian1",
"bi4",
"ying1",
"zhu3",
"wei3",
"tuan1",
"shan3",
"xi1",
"nuan3",
"nuan3",
"chan2",
"yan1",
"jiong3",
"jiong3",
"yu4",
"mei4",
"sha1,sha4",
"wei4",
"ye4",
"jin4",
"qiong2",
"rou2",
"mei2",
"huan4",
"xu4",
"zhao4",
"wei1",
"fan2",
"qiu2",
"sui4",
"yang2",
"lie4",
"zhu3",
"jie1",
"zao4",
"gua1",
"bao1",
"hu2",
"yun4",
"nan3",
"shi4",
"liang5",
"bian1",
"gou4",
"tui4",
"tang2",
"chao3",
"shan1",
"en1",
"bo2",
"huang3",
"xie2",
"xi4",
"wu4",
"xi1",
"yun4",
"he2",
"kao3",
"xi1",
"yun2",
"xiong2",
"nai2",
"shan3",
"qiong2",
"yao4",
"xun1",
"mi4",
"lian2",
"ying2",
"wu3",
"rong2",
"gong1",
"yan4",
"qiang4",
"liu1",
"xi1",
"bi4",
"biao1",
"cong1,zong3",
"lu4",
"jian1",
"shu2",
"yi4",
"lou2",
"peng2",
"sui1",
"yi4",
"teng1",
"jue2",
"zong1",
"yun4,yu4",
"hu4",
"yi2",
"zhi4",
"ao2,ao1",
"wei4",
"liu3",
"han4",
"ou1",
"re4",
"jiong3",
"man4",
"kun1",
"shang1",
"cuan4",
"zeng1",
"jian1",
"xi1",
"xi1",
"xi1",
"yi4",
"xiao4",
"chi4",
"huang2",
"chan3",
"ye4",
"qian2",
"ran2",
"yan4",
"xun2",
"qiao1",
"jun4",
"deng1",
"dun4",
"shen1",
"jiao1",
"fen2",
"si1",
"liao2,liao3",
"yu4",
"lin2",
"tong2",
"shao1",
"fen2",
"fan2",
"yan4,yan1",
"xun2",
"lan4",
"mei3",
"tang4",
"yi4",
"jiong3",
"men4",
"jing5",
"jiao3",
"ying2",
"yu4",
"yi4",
"xue2",
"lan2",
"tai4",
"zao4",
"can4",
"sui4",
"xi1",
"que4",
"zong3",
"lian2",
"hui3",
"zhu2",
"xie4",
"ling2",
"wei1",
"yi4",
"xie2",
"zhao4",
"hui4",
"da2",
"nong2",
"lan2",
"ru2",
"xian3",
"he4",
"xun1",
"jin4",
"chou2",
"dao4",
"yao4",
"he4",
"lan4",
"biao1",
"rong2",
"li4",
"mo4",
"bao4",
"ruo4,re4",
"lv4",
"la4",
"ao1",
"xun1",
"kuang4,huang3,kong4,kuang3",
"shuo4",
"liao2",
"li4",
"lu2",
"jue2",
"liao3",
"yan4",
"xi1",
"xie4",
"long2",
"ye4",
"can1",
"rang3",
"yue4",
"lan4",
"cong2",
"jue2",
"chong2",
"guan4",
"ju5",
"che4",
"mi2",
"tang3",
"lan4",
"zhu2",
"lan3",
"ling2",
"cuan4",
"yu4",
"zhao3,zhua3",
"zhao3",
"pa2",
"zheng1",
"pao2",
"cheng1,chen4",
"yuan2",
"ai4",
"wei4",
"han5",
"jue2",
"jue2",
"fu4",
"ye2",
"ba4",
"die1",
"ye2",
"yao2",
"zu3",
"shuang3",
"er3",
"pan2",
"chuang2",
"ke1",
"zang1",
"die2",
"qiang1",
"yong1",
"qiang2",
"pian4,pian1",
"ban3",
"pan4",
"chao2",
"jian1",
"pai2",
"du2",
"chuang1",
"tou2",
"zha2",
"bian1",
"die2",
"bang3",
"bo2",
"chuang1",
"you3",
"you3",
"du2",
"ya2",
"cheng1",
"niu2",
"niu2",
"pin4",
"jiu1",
"mou2,mu4",
"ta1",
"mu3",
"lao2",
"ren4",
"mang1",
"fang1",
"mao2",
"mu4",
"gang1",
"wu4",
"yan4",
"ge1",
"bei4",
"si4",
"jian4",
"gu3",
"you4",
"ge1",
"sheng1",
"mu3",
"di3",
"qian1",
"quan4",
"quan2",
"zi4",
"te4",
"xi1",
"mang2",
"keng1",
"qian1",
"wu3",
"gu4",
"xi1",
"li2",
"li2",
"pou3",
"ji1",
"gang1",
"te4",
"ben1",
"quan2",
"run2",
"du2",
"ju4",
"jia1",
"jian1",
"feng1",
"pian1",
"ke1",
"ju2",
"kao4",
"chu2",
"xi4",
"bei4",
"luo4",
"jie4",
"ma2",
"san1",
"wei4",
"li2",
"dun1",
"tong2",
"qiao2",
"jiang4",
"xi1",
"li4",
"du2",
"lie4",
"pai2",
"piao1",
"bo2",
"xi1",
"chou1",
"wei2",
"kui2",
"chou1",
"quan3",
"quan3",
"ba2,quan3",
"fan4",
"qiu2",
"ji3",
"chai2",
"zhuo2",
"an4",
"qi4",
"zhuang4",
"guang3",
"ma3",
"you2",
"kang4",
"bo2",
"hou3",
"ya4",
"yin2",
"huan1",
"zhuang4",
"yun3",
"kuang2",
"niu3",
"di2",
"kuang2",
"zhong4",
"mu4",
"bei4",
"pi1",
"ju2",
"yi2",
"sheng1",
"pao2",
"xia2",
"tuo2",
"hu2",
"ling2",
"fei4",
"pi1",
"ni3",
"yao3",
"you4",
"gou3",
"yue4",
"ju1",
"dan4",
"bo2",
"ku3",
"xian3",
"ning2",
"huan2",
"hen3",
"jiao3",
"he2",
"zhao4",
"ji2",
"xun4",
"shan1",
"ta4",
"rong2",
"shou4",
"tong2",
"lao3",
"du2",
"xia2",
"shi1",
"kuai4",
"zheng1",
"yu4",
"sun1",
"yu2",
"bi4",
"mang2",
"xi1",
"juan4",
"li2",
"xia2",
"yin2",
"suan1",
"lang2",
"bei4",
"zhi4",
"yan2",
"sha1",
"li4",
"han4",
"xian3",
"jing1",
"pai2",
"fei1",
"xiao1",
"ba4",
"qi2",
"ni2",
"biao1",
"yin4",
"lai2",
"lie4",
"jian1",
"qiang1",
"kun1",
"yan4",
"guo3",
"zong4",
"mi2",
"chang1",
"yi1",
"zhi4",
"zheng1",
"ya2",
"meng3",
"cai1",
"cu4",
"she1",
"lie4",
"dian3",
"luo2",
"hu2",
"zong1",
"gui4",
"wei3",
"feng1",
"wo1",
"yuan2",
"xing1",
"zhu1",
"mao1",
"wei4",
"yuan2",
"xian4",
"tuan1",
"ya4",
"nao2",
"xie1,he4",
"jia1",
"hou2",
"bian1,pian4",
"you2",
"you2",
"mei2",
"cha2",
"yao2",
"sun1",
"bo2",
"ming2",
"hua2",
"yuan2",
"sou1",
"ma4",
"yuan2",
"dai1",
"yu4",
"shi1",
"hao2",
"qiang1",
"yi4",
"zhen1",
"cang1",
"hao2",
"man4",
"jing4",
"jiang3",
"mo4",
"zhang1",
"chan2",
"ao2",
"ao2",
"hao2",
"cui1",
"ben4",
"jue2",
"bi4",
"bi4",
"huang2",
"pu2",
"lin2",
"yu4",
"tong2,zhuang4",
"yao4",
"liao2",
"shuo4",
"xiao1",
"shou4",
"dun1",
"jiao4",
"ge2",
"juan4",
"du2",
"hui4",
"kuai4",
"xian3",
"xie4",
"ta3",
"xian3",
"xun1",
"ning2",
"bian1",
"huo4",
"nou4",
"meng3",
"lie4",
"nao3",
"guang3",
"shou4",
"lu2",
"ta3",
"xian4",
"mi2",
"rang2",
"huan1",
"nao3",
"luo2",
"xian3",
"qi2",
"jue2",
"xuan2",
"miao4",
"zi1",
"lu:4,shuai4",
"lu2",
"yu4",
"su4",
"wang2,wang4",
"qiu2",
"ga3",
"ding1",
"le4",
"ba1",
"ji1",
"hong2",
"di4",
"chuan4",
"gan1",
"jiu3",
"yu2",
"qi3",
"yu2",
"chang4",
"ma3",
"hong2",
"wu3",
"fu1",
"wen2,min2",
"jie4",
"ya2",
"fen1",
"bian4",
"bang4",
"yue4",
"jue2",
"men2",
"jue2",
"wan2",
"jian1",
"mei2",
"dan3",
"pin2",
"wei3",
"huan2",
"xian4",
"qiang1",
"ling2",
"dai4",
"yi4",
"an2",
"ping2",
"dian4",
"fu2",
"xuan2",
"xi3",
"bo1",
"ci3",
"gou3",
"jia3",
"shao2",
"po4",
"ci2",
"ke1",
"ran3",
"sheng1",
"shen1",
"yi2",
"zu3",
"jia1",
"min2",
"shan1",
"liu3",
"bi4",
"zhen1",
"zhen1",
"jue2",
"fa4",
"long2",
"jin1",
"jiao4",
"jian4",
"li4",
"guang1",
"xian1",
"zhou1",
"gong3",
"yan1",
"xiu4",
"yang2",
"xu3",
"luo4",
"su4",
"zhu1",
"qin2",
"yin2",
"xun2",
"bao3",
"er3",
"xiang4",
"yao2",
"xia2",
"heng2",
"gui1",
"chong1",
"xu4",
"ban1",
"pei4",
"lao3",
"dang1",
"ying1",
"hun2",
"wen2",
"e2",
"cheng2",
"di4",
"wu3",
"wu2",
"cheng2",
"jun4",
"mei2",
"bei4",
"ting3",
"xian4",
"chu4",
"han2",
"xuan2",
"yan2",
"qiu2",
"xuan4",
"lang2",
"li3",
"xiu4",
"fu2",
"liu2",
"ya2",
"xi1",
"ling2",
"li2",
"jin4",
"lian3",
"suo3",
"suo3",
"feng1",
"wan2",
"dian4",
"pin2",
"zhan3",
"se4",
"min2",
"yu4",
"ju1",
"chen1",
"lai2",
"min2",
"sheng4",
"wei2",
"dian3",
"chu4",
"zuo2,zhuo2",
"beng3",
"cheng1",
"hu3",
"qi2",
"e4",
"kun1",
"chang1",
"qi2",
"beng3",
"wan3",
"lu4",
"cong2",
"guan3",
"yan3",
"diao1",
"bei4",
"lin2",
"qin2",
"pi2",
"pa2",
"que4",
"zhuo2",
"qin2",
"fa4",
"jin1",
"qiong2",
"du3",
"jie4",
"hun2",
"yu3",
"mao4",
"mei2",
"chun1",
"xuan1",
"ti2",
"xing1",
"dai4",
"rou2",
"min2",
"jian1",
"wei3",
"ruan3",
"huan4",
"xie2",
"chuan1",
"jian3",
"zhuan4",
"chang4",
"lian4",
"quan2",
"xia2",
"duan4",
"yuan4",
"ya2",
"nao3",
"hu2",
"ying1",
"yu2",
"huang2",
"rui4",
"se4",
"liu2",
"shi1",
"rong2",
"suo3",
"yao2",
"wen1",
"wu3",
"zhen1",
"jin4",
"ying2",
"ma3",
"tao1",
"liu2",
"tang2",
"li4",
"lang2",
"gui1",
"zhen4,tian4",
"qiang1",
"cuo1",
"jue2",
"zhao3",
"yao2",
"ai4",
"bin1",
"shu1",
"chang2",
"kun1",
"zhuan1",
"cong1",
"jin3",
"yi1",
"cui3",
"cong1",
"qi2",
"li2",
"jing3",
"suo3",
"qiu2",
"xuan2",
"ao2",
"lian3",
"men2",
"zhang1",
"yin2",
"ye4",
"ying1",
"wei4",
"lu4",
"wu2",
"deng1",
"xiu4",
"zeng1",
"xun2",
"qu2",
"dang4",
"lin2",
"liao2",
"qiong2,jue2",
"su4",
"huang2",
"gui1",
"pu2",
"jing3",
"fan2",
"jin4",
"liu2",
"ji1",
"hui4",
"jing3",
"ai4",
"bi4",
"can4",
"qu2",
"zao3",
"dang1",
"jiao3",
"gun4",
"tan3",
"hui4",
"huan2",
"se4",
"sui4",
"tian2",
"chu3",
"yu2",
"jin4",
"lu2",
"bin1",
"shu2",
"wen4",
"zui3",
"lan2",
"xi3",
"zi1",
"xuan2",
"ruan3",
"wo4",
"gai4",
"lei2",
"du2",
"li4",
"zhi4",
"rou2",
"li2",
"zan4",
"qiong2",
"ti4",
"gui1",
"sui2",
"la4",
"long2",
"lu2",
"li4",
"zan4",
"lan4",
"ying1",
"mi2",
"xiang1",
"qiong2",
"guan4",
"dao4",
"zan4",
"huan2",
"gua1",
"bo2",
"die2",
"bo2",
"hu4",
"zhi2",
"piao2",
"ban4",
"rang2",
"li4",
"wa3",
"shi2 wa3",
"xiang2",
"qian1 wa3",
"ban3",
"pen2",
"fang3",
"dan3",
"weng4",
"ou1",
"fen1 wa3",
"hao2 wa3",
"wa5",
"hu2",
"ling2",
"yi2",
"ping2",
"ci2",
"bai3 wa3",
"juan1",
"chang2",
"chi1",
"li3 wa3",
"dang4",
"meng3",
"bu4",
"zhui4",
"ping2",
"bian1",
"zhou4",
"zhen1",
"",
"ci2",
"ying1",
"qi4",
"xian2",
"lou3",
"di4",
"ou1",
"meng2",
"zhuan1",
"beng4",
"lin4",
"zeng4",
"wu3",
"pi4",
"dan1",
"weng4",
"ying1",
"yan3",
"gan1",
"dai4",
"shen2,shen4",
"tian2",
"tian2",
"han2",
"chang2",
"sheng1",
"qing2",
"shen1",
"chan3",
"chan3",
"rui2",
"sheng1",
"su1",
"shen1",
"yong4",
"shuai3",
"lu4",
"fu3",
"yong3,tong3",
"beng2,beng4",
"feng4,beng2",
"ning2",
"tian2",
"you2",
"jia3",
"shen1",
"zha2",
"dian4",
"fu2",
"nan2",
"dian4",
"ping1",
"ding1,ting3",
"hua4",
"ting3",
"quan3",
"zai1",
"meng2",
"bi4",
"bi4",
"liu4",
"xun2",
"liu2",
"chang4",
"mu3",
"yun2",
"fan4",
"fu2",
"geng1",
"tian2",
"jie4",
"jie4",
"quan3",
"wei4",
"fu2",
"tian2",
"mu3",
"xx5",
"pan4",
"jiang1",
"wa1",
"da2",
"nan2",
"liu2",
"ben3",
"zhen3",
"chu4,xu4",
"mu3",
"mu3",
"ce4",
"tian2",
"gai1",
"bi4",
"da2",
"zhi4",
"lu:e4",
"qi2",
"lve4",
"pan1",
"yi1",
"fan1,pan1",
"hua4",
"she1",
"yu2",
"mu3",
"jun4",
"yi4",
"liu2",
"she1,yu2",
"die2",
"chou2",
"hua4",
"dang1",
"chuo4",
"ji1",
"wan3",
"jiang1",
"cheng2",
"chang4",
"tuan3",
"lei2",
"ji1",
"cha1",
"liu2",
"die2",
"tuan3",
"lin4",
"jiang1",
"jiang1",
"chou2",
"pi4",
"die2",
"die2",
"shu1,ya3",
"jie2",
"dan4",
"shu1",
"shu1",
"zhi4",
"yi2",
"ne4",
"nai3",
"ding1",
"bi3",
"jie2",
"liao2",
"gang1",
"ge1",
"jiu4",
"zhou3",
"xia4",
"shan4",
"xu1",
"nu:e4,yao4",
"li4",
"yang2",
"chen4",
"you2",
"ba1",
"jie4",
"jue2",
"zhi1",
"xia1",
"cui4",
"bi4",
"yi4",
"li4",
"zong4",
"chuang1",
"feng1",
"zhu4",
"pao4",
"pi2",
"gan1",
"ke1",
"ci1",
"xie4",
"zhi1",
"dan3,da5",
"zhen3",
"fa2",
"zhi3",
"teng2",
"ju1",
"ji2",
"fei4",
"ju1",
"dian4",
"jia1",
"xuan2",
"zha4",
"bing4",
"nie4",
"zheng4,zheng1",
"yong1",
"jing4",
"quan2",
"teng2",
"tong1",
"yi2",
"jie1",
"wei3",
"hui2",
"tan1",
"yang3",
"chi4",
"zhi4",
"hen2",
"ya3",
"mei4",
"dou4",
"jing4",
"xiao1",
"tong4",
"tu1",
"mang2",
"pi3",
"xiao1",
"suan1",
"pu1",
"li4",
"zhi4",
"cuo2",
"duo2",
"wu4",
"sha1",
"lao2",
"shou4",
"huan4",
"xian2",
"yi4",
"peng2",
"zhang4",
"guan3",
"tan2",
"fei4",
"ma2",
"lin2",
"chi1",
"ji4",
"tian3",
"an1",
"chi4",
"bi4",
"bi4",
"min2",
"gu4",
"dui1",
"e1",
"wei3",
"yu1",
"cui4",
"ya3",
"zhu2",
"cu4",
"dan1,dan4",
"shen4",
"zhong3",
"ji4",
"yu4",
"hou2",
"feng1",
"la4",
"yang2",
"chen2",
"tu2",
"yu3",
"guo1",
"wen2",
"huan4",
"ku4",
"jia3",
"yin1",
"yi4",
"lou4",
"sao4",
"jue2",
"chi4",
"xi1",
"guan1",
"yi4",
"wen1",
"ji2",
"chuang1",
"ban1",
"hui4",
"liu2",
"chai4,cuo2",
"shou4",
"nve4",
"dian1",
"da2",
"bie3",
"tan1",
"zhang4",
"biao1",
"shen4",
"cu4",
"luo3",
"yi4",
"zong4",
"chou1",
"zhang4",
"zhai4",
"sou4",
"se4",
"que2",
"diao4",
"lou4",
"lou4",
"mo4",
"qin2",
"yin3",
"ying3",
"huang2",
"fu2",
"liao2",
"long2",
"qiao2",
"liu2",
"lao2",
"xian2",
"fei4",
"dan1",
"yin4",
"he4",
"ai2",
"ban1",
"xian2",
"guan1",
"gui4",
"nong4",
"yu4",
"wei2",
"yi4",
"yong1",
"pi3",
"lei3",
"li4",
"shu3",
"dan4",
"lin3",
"dian4",
"lin3",
"lai4",
"bie3",
"ji4",
"chi1",
"yang3",
"xuan3",
"jie1",
"zheng1",
"me5",
"li4",
"huo4",
"lai4",
"ji1",
"dian1",
"xuan3",
"ying3",
"yin3",
"qu2",
"yong1",
"tan1",
"dian1",
"luo3",
"luan2",
"luan2",
"bo1",
"bo1",
"gui3",
"ba2",
"fa1",
"deng1",
"fa1",
"bai2",
"bai3",
"qie2",
"bi1",
"zao4",
"zao4",
"mao4",
"de5,di1,di2,di4",
"pa1",
"jie1",
"huang2",
"gui1",
"ci3",
"ling2",
"gao1",
"mo4",
"ji2",
"jiao3",
"peng3",
"gao1",
"ai2",
"e2",
"hao4",
"han4",
"bi4",
"wan3",
"chou2",
"qian4",
"xi1",
"ai2",
"xiao3",
"hao4",
"huang3",
"hao4",
"ze2",
"cui3",
"hao4",
"xiao3",
"ye4",
"po2",
"hao4",
"jiao3",
"ai4",
"xing1",
"huang4",
"li4",
"piao3",
"he2",
"jiao4",
"pi2",
"gan3",
"pao4",
"zhou4",
"jun1",
"qiu2",
"cun1",
"que4",
"zha1",
"gu3",
"jun1",
"jun1",
"zhou4",
"zha1,cu3",
"gu3",
"zhan3",
"du2",
"min3",
"qi3",
"ying2",
"yu2",
"bei1",
"zhao1",
"zhong1",
"pen2",
"he2",
"ying2",
"he2",
"yi4",
"bo1",
"wan3",
"he2",
"ang4",
"zhan3",
"yan2",
"jian1,jian4",
"he2",
"yu1",
"kui1",
"fan4",
"gai4,ge3",
"dao4",
"pan2",
"fu3",
"qiu2",
"sheng4,cheng2",
"dao4",
"lu4",
"zhan3",
"meng2",
"li2",
"jin3",
"xu4",
"jian1",
"pan2",
"guan4",
"an1",
"lu2",
"xu3",
"zhou1",
"dang4",
"an1",
"gu3",
"li4",
"mu4",
"ding1",
"gan4",
"xu1",
"mang2",
"wang4",
"zhi2",
"qi4",
"yuan3",
"tian2",
"xiang1,xiang4",
"dun3",
"xin1",
"xi4",
"pan4",
"feng1",
"dun4",
"min2",
"ming2",
"sheng3,xing3",
"shi4",
"yun2",
"mian3,mian4",
"pan1",
"fang3",
"miao3",
"dan1",
"mei2",
"mao4",
"kan4,kan1",
"xian4",
"kou1",
"shi4",
"yang1",
"zheng1",
"yao3",
"shen1",
"huo4",
"da4",
"zhen3",
"kuang4",
"ju1",
"shen4",
"yi2",
"sheng3",
"mei4",
"mo4",
"zhu4",
"zhen1",
"zhen1",
"mian2",
"shi4",
"yuan1",
"die2",
"ni4",
"zi4",
"zi4",
"chao3",
"zha3",
"xuan4",
"bing3",
"mi3",
"long2",
"sui1",
"tong2",
"mi1,mi2",
"die2",
"di4",
"ne4",
"ming2",
"xuan4",
"chi1",
"kuang4",
"juan4",
"mou2",
"zhen4",
"tiao4",
"yang2",
"yan3",
"mo4",
"zhong4",
"mo4",
"zhe5,zhao1,zhao2,zhuo2",
"zheng1",
"mei2",
"suo1",
"shao4",
"han4",
"huan3",
"di4",
"cheng3",
"cuo2",
"juan4",
"e2",
"man3",
"xian4",
"xi1",
"kun4",
"lai4",
"jian3",
"shan3",
"tian3",
"gun4",
"wan3",
"leng4",
"shi4",
"qiong2",
"lie4",
"ya2",
"jing1",
"zheng1",
"li2",
"lai4",
"sui4",
"juan4",
"shui4",
"sui1",
"du1",
"bi4",
"bi4",
"mu4",
"hun1",
"ni4",
"lu4",
"yi4",
"jie2",
"cai3",
"zhou3",
"yu2",
"hun1",
"ma4",
"xia4",
"xing3",
"hui1",
"gun4",
"zai1",
"chun3",
"jian1",
"mei4",
"du3",
"hou2",
"xuan1",
"tian4",
"kui2",
"gao1",
"rui4",
"mao4",
"xu4",
"fa2",
"wo4",
"miao2",
"chou3",
"kui4",
"mi1",
"weng3",
"kou4",
"dang4",
"chen1",
"ke1",
"sou3",
"xia1",
"qiong2",
"mo4",
"ming2",
"man2",
"fen4",
"ze2",
"zhang4",
"yi4",
"diao1",
"kou1",
"mo4",
"shun4",
"cong1",
"lou1",
"chi1",
"man2",
"piao3",
"cheng1",
"gui1",
"meng2",
"wan4",
"run2",
"pie1",
"xi1",
"qiao2",
"pu2",
"zhu3",
"deng4",
"shen3",
"shun4",
"liao4",
"che4",
"xian2",
"kan4",
"ye4",
"xu4",
"tong2",
"mou2",
"lin2",
"gui4",
"jian4",
"ye4",
"ai4",
"hui4",
"zhan1",
"jian3",
"gu3",
"zhao4",
"qu2",
"mei2",
"chou3",
"sao4",
"ning3",
"xun1",
"yao4",
"huo4",
"meng2",
"mian2",
"pin2",
"mian2",
"lei3",
"kuang4",
"jue2",
"xuan1",
"mian2",
"huo4",
"lu2",
"meng2",
"long2",
"guan4",
"man3",
"xi3",
"chu4",
"tang3",
"kan4",
"zhu3",
"mao2",
"jin1",
"jin1",
"yu4",
"shuo4",
"ze2",
"jue2",
"shi3",
"yi3",
"shen3",
"zhi1",
"hou2",
"shen3",
"ying3",
"ju3",
"zhou1",
"jiao3,jiao2",
"cuo2",
"duan3",
"ai3",
"jiao3",
"zeng1",
"yue1",
"ba4",
"shi2,dan4",
"ding4",
"qi4",
"ji1",
"zi3",
"gan1",
"wu4",
"zhe2",
"ku1",
"jiang1",
"xi1",
"fan2",
"kuang4",
"dang4",
"ma3",
"sha1",
"dan1",
"jue2",
"li4",
"fu1",
"min2",
"e4",
"hua1,xu1",
"kang1",
"zhi3",
"qi4",
"kan3",
"jie4",
"bin1",
"e4",
"ya4",
"pi1",
"zhe2",
"yan2",
"sui4",
"zhuan1",
"che1",
"dun4",
"wa3",
"yan4",
"jin1",
"feng1",
"fa3",
"mo4",
"zha3",
"qu1",
"yu4",
"luo3",
"tuo2",
"tuo2",
"di3",
"zhai4",
"zhen1",
"e4",
"fu2,fei4",
"mu3",
"zhu3",
"la2",
"bian1",
"nu3",
"ping1",
"peng1",
"ling2",
"pao4",
"le4",
"po4",
"bo1",
"po4",
"shen1",
"za2",
"ai4",
"li4",
"long2",
"tong2",
"yong4",
"li4",
"kuang4",
"chu3",
"keng1",
"quan2",
"zhu1",
"kuang1",
"gui1",
"e4",
"nao2",
"qia4",
"lu4",
"wei3",
"ai4",
"ge4",
"ken4",
"xing2",
"yan2",
"dong4,tong2",
"peng1",
"xi1",
"lao3",
"hong2",
"shuo4",
"xia2",
"qiao1",
"qing5",
"wei2,ai2,wei4",
"qiao2",
"yi4",
"keng1",
"xiao1",
"que4",
"chan4",
"lang2",
"hong1",
"yu2",
"xiao1",
"xia2",
"mang3",
"luo4",
"yong3",
"che1",
"che4",
"wo4",
"liu2",
"ying4",
"mang2",
"que4",
"yan4",
"sha1",
"kun3",
"yu4",
"chi4",
"hua1",
"lu3",
"chen3",
"jian3",
"nve4",
"song1",
"zhuo2",
"keng1",
"peng2",
"yan1",
"zhui4",
"kong1",
"cheng2",
"qi2",
"zong4",
"qing4",
"lin2",
"jun1",
"bo1",
"ding4",
"min2",
"diao1",
"jian1",
"he4",
"lu4,liu4",
"ai4",
"sui4",
"que4",
"leng2",
"bei1",
"yin2",
"dui4",
"wu3",
"qi2",
"lun3",
"wan3",
"dian3",
"nao2",
"bei4",
"qi4",
"chen3",
"ruan3",
"yan2",
"die2",
"ding4",
"zhou2",
"tuo2",
"jie2",
"ying1",
"bian3",
"ke4",
"bi4",
"wei4",
"shuo4",
"zhen1",
"duan4",
"xia2",
"dang4",
"ti2",
"nao3",
"peng4",
"jian3",
"di4",
"tan4",
"cha2",
"tian2",
"qi4",
"dun4",
"feng1",
"xuan4",
"que4",
"que4",
"ma3",
"gong1",
"nian3",
"su4",
"e2",
"ci2",
"liu2",
"si1",
"tang2",
"bang4",
"hua2",
"pi1",
"wei3",
"sang3",
"lei3",
"cuo1",
"tian2",
"xia2",
"xi1",
"lian2",
"pan2",
"wei2",
"yun3",
"dui1",
"zhe2",
"ke1,ke4",
"la2",
"zhuan1",
"yao2",
"gun3",
"zhuan1",
"chan2",
"qi4",
"ao2",
"peng1",
"liu4",
"lu3",
"kan4",
"chuang3",
"chen3",
"yin3",
"lei3",
"biao1",
"qi4",
"mo2,mo4",
"qi4",
"cui1",
"zong1",
"qing4",
"chuo4",
"lun2",
"ji1",
"shan4",
"lao2",
"qu2",
"zeng1",
"deng4",
"jian4",
"xi4",
"lin2",
"ding4",
"dian4",
"huang2",
"pan2",
"za2",
"qiao1",
"di1",
"li4",
"jian4",
"jiao1",
"xi1",
"zhang3",
"qiao2",
"dun1",
"jian3",
"yu4",
"zhui4",
"he2",
"ke4",
"ze2",
"lei4",
"jie2",
"chu3",
"ye4",
"que4",
"dang4",
"yi3",
"jiang1",
"pi1",
"pi1",
"yu4",
"pin1",
"e4",
"ai4",
"ke1",
"jian1",
"yu4",
"ruan3",
"meng2",
"pao4",
"ci2",
"bo2",
"yang3",
"ma4",
"ca3",
"xian2",
"kuang4",
"lei4",
"lei3",
"zhi4",
"li4",
"li4",
"fan2",
"que4",
"pao4",
"ying1",
"li4",
"long2",
"long2",
"mo4",
"bo2",
"shuang1",
"guan4",
"lan2",
"ca3",
"yan2",
"shi4",
"shi4",
"li3",
"reng2",
"she4",
"yue4",
"si4",
"qi2",
"ta1",
"ma4",
"xie4",
"yao1",
"xian1",
"qi2",
"qi2",
"zhi3",
"beng1",
"dui4",
"zhong4",
"ren4",
"yi1",
"shi2",
"you4",
"zhi4",
"tiao2",
"fu2",
"fu4",
"mi4",
"zu3",
"zhi1",
"suan4",
"mei4",
"zuo4",
"qu1",
"hu4",
"zhu4",
"shen2",
"sui4",
"ci2",
"chai2",
"mi2",
"lv3",
"yu3",
"xiang2",
"wu2",
"tiao1",
"piao4",
"zhu4",
"gui3",
"xia2",
"zhi1",
"ji4,zhai4",
"gao4",
"zhen1",
"gao4",
"shui4",
"jin1",
"shen4",
"gai1",
"kun3",
"di4",
"dao3",
"huo4",
"tao2",
"qi2",
"gu4",
"guan4",
"zui4",
"ling2",
"lu4",
"bing3",
"jin4,jin1",
"dao3",
"zhi2",
"lu4",
"chan2,shan4",
"bi4",
"chu3",
"hui1",
"you3",
"xi4",
"yin1",
"zi1",
"huo4",
"zhen1",
"fu2",
"yuan4",
"wu2",
"xian3",
"yang2",
"ti2",
"yi1",
"mei2",
"si1",
"di4",
"bei4",
"zhuo2",
"zhen1",
"yong3",
"ji4",
"gao4",
"tang2",
"si1",
"ma4",
"ta4",
"fu4",
"xuan1",
"qi2",
"yu4",
"xi3",
"ji1",
"si4",
"chan2",
"tan3",
"gui4",
"sui4",
"li3",
"nong2",
"mi2",
"dao3",
"li4",
"rang2",
"yue4",
"ti2",
"zan4",
"lei4",
"rou2",
"yu3",
"yu2,ou3,yu4",
"li2,chi1",
"xie4",
"qin2",
"he2",
"tu1",
"xiu4",
"si1",
"ren2",
"tu1",
"zi3",
"cha2",
"gan3",
"yi4",
"xian1",
"bing3",
"nian2",
"qiu1",
"qiu1",
"zhong3,zhong4",
"fen4",
"hao4",
"yun2",
"ke1",
"miao3",
"zhi1",
"jing1",
"bi3",
"zhi1",
"yu4",
"mi4,bi4",
"ku4",
"ban4",
"pi1",
"ni2",
"li4",
"you2",
"zu1",
"pi1",
"bo2",
"ling2",
"mo4",
"cheng4,cheng1",
"nian2",
"qin2",
"yang1",
"zuo2",
"zhi4",
"zhi1",
"shu2",
"ju4",
"zi3",
"huo2",
"ji1",
"cheng1,chen4,cheng4",
"tong2",
"zhi4",
"huo2",
"he2",
"yin1",
"zi1",
"zhi2",
"jie1",
"ren3",
"du4",
"yi2",
"zhu1",
"hui4",
"nong2",
"fu4",
"xi1",
"gao3",
"lang2",
"fu1",
"xun4",
"shui4",
"lu:3",
"kun3",
"gan3",
"jing1",
"ti2",
"cheng2",
"tu2",
"shao1,shao4",
"shui4",
"ya4",
"lun3",
"lu4",
"gu4",
"zuo2",
"ren3",
"zhun4",
"bang4",
"bai4",
"ji1",
"zhi2",
"zhi4",
"kun3",
"leng2",
"peng2",
"ke1",
"bing3",
"chou2",
"zui4",
"yu4",
"su1",
"lve4",
"xiang1",
"yi1",
"xi4",
"bian3",
"ji4",
"fu2",
"pi4",
"nuo4",
"jie1",
"zhong3",
"zong1",
"xu3",
"cheng1",
"dao4",
"wen3",
"xian2",
"zi1",
"yu4",
"ji4",
"xu4",
"zhen3",
"zhi4",
"dao4",
"jia4",
"ji1,qi3",
"gao3",
"gao3",
"gu3",
"rong2",
"sui4",
"rong5",
"ji4",
"kang1",
"mu4",
"can3",
"men2",
"zhi4",
"ji4",
"lu4",
"su1",
"ji1",
"ying3",
"wen3",
"qiu1",
"se4",
"he4",
"yi4",
"huang2",
"qie4",
"ji3",
"sui4",
"xiao1",
"pu2",
"jiao1",
"zhuo1",
"tong2",
"zui5",
"lu:3",
"sui4",
"nong2",
"se4",
"hui4",
"rang2",
"nuo4",
"yu4",
"pin1",
"ji4",
"tui2",
"wen3",
"cheng1",
"huo4",
"kuang4",
"lv3",
"biao1",
"se4",
"rang2",
"zhuo1",
"li2",
"cuan2",
"xue2",
"wa1",
"jiu1",
"qiong2",
"xi1",
"qiong2",
"kong1,kong4",
"yu1",
"shen1",
"jing3",
"yao4",
"chuan1",
"zhun1",
"tu1",
"lao2",
"qie4",
"zhai3",
"yao3",
"bian3",
"bao2",
"yao3",
"bing3",
"wa1",
"zhu2",
"jiao4",
"qiao4",
"diao4",
"wu1",
"gui1",
"yao2",
"zhi4",
"chuang1",
"yao3",
"tiao3",
"jiao4",
"chuang1",
"jiong3",
"xiao1",
"cheng2",
"kou4",
"cuan4",
"wo1",
"dan4",
"ku1",
"ke1",
"zhuo2",
"xu1",
"su4",
"guan1",
"kui1",
"dou4",
"zhuo5",
"xun1,yin4",
"wo1",
"wa1",
"ya4",
"yu2",
"ju4",
"qiong2",
"yao2",
"yao2",
"tiao3",
"chao2",
"yu3",
"tian2",
"diao4",
"ju4",
"liao4",
"xi1",
"wu4",
"kui1",
"chuang1",
"zhao1",
"kuan3",
"kuan3,cuan4",
"long2",
"cheng1",
"cui4",
"liao2",
"zao4",
"cuan4",
"qiao4",
"qiong2",
"dou4",
"zao4",
"long3",
"qie4",
"li4",
"chu4",
"shi2",
"fu4",
"qian1",
"chu4",
"hong2",
"qi2",
"hao2",
"sheng1",
"fen1",
"shu4",
"miao4",
"qu3",
"zhan4",
"zhu4",
"ling2",
"long2",
"bing4",
"jing4",
"jing4",
"zhang1",
"bai3",
"si4",
"jun4",
"hong2",
"tong2",
"song3",
"jing4",
"diao4",
"yi4",
"shu4",
"jing4",
"qu3",
"jie2",
"ping1",
"duan1",
"li2",
"zhuan3",
"ceng2",
"deng1",
"cun1",
"wai1",
"jing4",
"kan3",
"jing4",
"zhu2",
"zhu2,du3",
"le4",
"peng2",
"yu2",
"chi2",
"gan1",
"mang2",
"zhu2",
"wan2",
"du3",
"ji1",
"jiao3",
"ba1",
"suan4",
"ji2",
"qin3",
"zhao4",
"sun3",
"ya2",
"zhui4",
"yuan2",
"hu4",
"gang1",
"xiao4",
"cen2",
"pi2",
"bi3",
"jian3",
"yi3",
"dong1",
"shan1",
"sheng1",
"da1",
"di2",
"zhu2",
"na4",
"chi1",
"gu1",
"li4",
"qie4",
"min3",
"bao1",
"tiao2",
"si4",
"fu2",
"ce4",
"ben4",
"fa2",
"da2",
"zi3",
"di4",
"ling2",
"ze2",
"nu2",
"fu2",
"gou3",
"fan2",
"jia1",
"gan3",
"fan4",
"shi3",
"mao3",
"po3",
"ti5",
"jian1",
"qiong2",
"long2,long3",
"min3",
"bian1",
"luo4",
"gui4",
"qu1",
"chi2",
"yin1",
"yao4",
"xian3",
"bi3",
"qiong2",
"gua1",
"deng3",
"jiao3,jiao4",
"jin1",
"quan2",
"sun3",
"ru2",
"fa2",
"kuang1",
"zhu4",
"tong3",
"ji1",
"da2,da1",
"hang2",
"ce4",
"zhong4",
"kou4",
"lai2",
"bi4",
"shai1",
"dang1",
"zheng1",
"ce4",
"fu1",
"yun2",
"tu2",
"pa2",
"li2",
"lang2",
"ju3",
"guan3",
"jian3",
"han2",
"tong2",
"xia2",
"zhi4",
"cheng2",
"suan4",
"shi4",
"zhu4",
"zuo2",
"xiao3",
"shao1",
"ting2",
"ce4",
"yan2",
"gao4",
"kuai4",
"gan1",
"chou2",
"kuang1",
"gang4",
"yun2",
"ou1",
"qian1",
"xiao3",
"jian3",
"pou2",
"lai2",
"zou1",
"bi3",
"bi4",
"bi4",
"ge4",
"tai2",
"guai3",
"yu1",
"jian1",
"dao4",
"gu1",
"chi2",
"zheng1",
"qing4",
"sha4",
"zhou3",
"lu4",
"bo2",
"ji1",
"lin2",
"suan4",
"jun4",
"fu2",
"zha2",
"gu1",
"kong1",
"qian2",
"qian1",
"jun4",
"chui2",
"guan3",
"yuan1",
"ce4",
"zu2",
"bo3",
"ze2",
"qie4",
"tuo4",
"luo2",
"dan1",
"xiao1",
"ruo4",
"jian4",
"xuan1",
"bian1",
"sun3",
"xiang1",
"xian3",
"ping2",
"zhen1",
"xing1",
"hu2",
"yi2",
"zhu4",
"yue1",
"chun1",
"lv4",
"wu1",
"dong3",
"shuo4",
"ji2",
"jie2",
"huang2",
"xing1",
"mei4",
"fan4",
"chuan2",
"zhuan4",
"pian1",
"feng1",
"zhu4",
"huang2",
"qie4",
"hou2",
"qiu1",
"miao3",
"qian4",
"gu1",
"kui4",
"shi5",
"lou3",
"yun2",
"he2",
"tang2",
"yue4",
"chou1",
"gao1",
"fei3",
"ruo4",
"zheng1",
"gou1",
"nie4",
"qian4",
"xiao3",
"cuan4",
"long3",
"peng2",
"du3",
"li4",
"bi4",
"zhuo2",
"chu2",
"shai1",
"chi2",
"zhu4",
"qiang1",
"long2",
"lan2",
"jian1",
"bu4",
"li2",
"hui4",
"bi4",
"di2",
"cong1",
"yan1",
"peng2",
"can3",
"zhuan4,suan3",
"pi2",
"piao3",
"dou1",
"yu4",
"mie4",
"tuan2",
"ze2",
"xi3",
"gui4",
"yi2",
"hu4",
"chan3",
"kou4",
"cu4",
"ping2",
"zao4",
"ji1",
"gui3",
"su4",
"lou3",
"ce4",
"lu4",
"nian3",
"suo1",
"cuan4",
"diao1",
"suo1",
"le4",
"duan4",
"zhu4",
"xiao1",
"bo2",
"mi4",
"shai1",
"dang4",
"liao2",
"dan1",
"dian4",
"fu3",
"jian3",
"min3",
"kui4",
"dai4",
"jiao1",
"deng1",
"huang2",
"sun3",
"lao2",
"zan1",
"xiao1",
"lu4",
"shi4",
"zan1",
"qi5",
"pai2",
"qi2",
"pai2",
"gan3",
"ju4",
"lu4",
"lu4",
"yan2",
"bo3,bo4",
"dang1",
"sai4",
"ke1",
"gou1",
"qian1",
"lian2",
"bu4",
"zhou4",
"lai4",
"shi5",
"lan2",
"kui4",
"yu2",
"yue4",
"hao2",
"zhen1",
"tai2",
"ti4",
"nie4",
"chou2",
"ji2",
"yi2",
"qi2",
"teng2",
"zhuan4",
"zhou4",
"fan1",
"sou3",
"zhou4",
"qian5",
"zhuo2",
"teng2",
"lu4",
"lu2",
"jian3",
"tuo4",
"ying2",
"yu4",
"lai4",
"long2",
"qie4",
"lian2",
"lan2",
"qian1",
"yue4",
"zhong1",
"qu2",
"lian2",
"bian1",
"duan4",
"zuan3",
"li2",
"shai1",
"luo2",
"ying2",
"yue4",
"zhuo2",
"yu4",
"mi3",
"di2",
"fan2",
"shen1",
"zhe2",
"shen1",
"nu:3",
"he2",
"lei4",
"xian1",
"zi3",
"ni2",
"cun4",
"zhang4",
"qian1",
"zhai1",
"bi3",
"ban3",
"wu4",
"sha1",
"kang1",
"rou2",
"fen3",
"bi4",
"cui4",
"yin5",
"zhe2",
"mi3",
"tai5",
"hu4",
"ba1",
"li4",
"gan1",
"ju4",
"po4",
"mo4",
"cu1",
"zhan1,nian2",
"zhou4",
"chi1",
"su4",
"tiao4",
"li4",
"xi1",
"su4",
"hong2",
"tong2",
"zi1",
"ce4",
"yue4",
"zhou1,yu4",
"lin2",
"zhuang1",
"bai3 mi3",
"lao1",
"fen4",
"er2",
"qu1",
"he2",
"liang2",
"xian4",
"fu1",
"liang2",
"can4",
"jing1",
"li3",
"yue4",
"lu4",
"ju2",
"qi2",
"cui4",
"bai4",
"zhang1",
"lin2",
"zong4",
"jing1",
"guo3",
"hua1",
"san3",
"san3",
"tang2",
"bian3",
"rou2",
"mian4",
"hou2",
"xu3",
"zong4",
"hu2,hu4",
"jian4",
"zan1",
"ci2",
"li2",
"xie4",
"fu1",
"nuo4",
"bei4",
"gu3",
"xiu3",
"gao1",
"tang2",
"qiu3",
"jia1",
"cao1",
"zhuang1",
"tang2",
"mi2,mei2",
"san3",
"fen4",
"zao1",
"kang1",
"jiang4",
"mo2",
"san3",
"san3",
"nuo4",
"xi1",
"liang2",
"jiang4",
"kuai4",
"bo4",
"huan2",
"shu3",
"zong4",
"xian4",
"nuo4",
"tuan2",
"nie4",
"li4",
"zuo4",
"di2",
"nie4",
"tiao4",
"lan4",
"mi4",
"si1",
"jiu3",
"xi4,ji4",
"gong1",
"zheng3",
"jiu1",
"you4",
"ji4",
"cha4",
"zhou4",
"xun2",
"yue1",
"hong2",
"yu1",
"he2",
"wan2",
"ren4",
"wen3",
"wen2",
"qiu2",
"na4",
"zi1",
"tou3",
"niu3",
"fou2",
"ji4",
"shu1",
"chun2",
"pi1",
"zhen4",
"sha1",
"hong2",
"zhi3",
"ji2",
"fen1",
"yun2",
"ren4",
"dan3",
"jin1",
"su4",
"fang3",
"suo3",
"cui4",
"jiu3",
"za1",
"ba5",
"jin3",
"fu1",
"zhi4",
"qi1",
"zi3",
"chou2",
"hong2",
"za1",
"lei4,lei3,lei2",
"xi4",
"fu2",
"xie4",
"shen1",
"bo1",
"zhu4",
"qu1",
"ling2",
"zhu4",
"shao4",
"gan4",
"yang3",
"fu2",
"tuo2",
"zhen3",
"dai4",
"chu4",
"shi1",
"zhong1",
"xian2",
"zu3",
"jiong1",
"ban4",
"qu2",
"mo4",
"shu4",
"zui4",
"kuang4",
"jing1",
"ren4",
"hang2",
"xie4",
"jie2",
"zhu1",
"chou2",
"gua4,kua1",
"bai3",
"jue2",
"kuang4",
"hu2",
"ci4",
"huan2",
"geng1",
"tao1",
"jie2,xie2",
"ku4",
"jiao3",
"quan2",
"gai3",
"luo4",
"xuan4",
"bing3",
"xian4",
"fu2",
"gei3",
"dong4",
"rong2",
"tiao4",
"yin1",
"lei3",
"xie4",
"juan4",
"xu4",
"gai1",
"die2",
"tong3",
"si1",
"jiang4",
"xiang2",
"hui4",
"jue2",
"zhi2",
"jian3",
"juan4",
"chi1",
"mian3,wen4",
"zhen4",
"lv3",
"cheng2",
"qiu2",
"shu1",
"bang3",
"tong3",
"xiao1",
"huan2",
"qin1",
"geng3",
"xiu3",
"ti2",
"tou4",
"xie2",
"hong2",
"xi4",
"fu2",
"ting1",
"sui1",
"dui4",
"kun3",
"fu1",
"jing1",
"hu4",
"zhi1",
"yan2",
"jiong3",
"feng2",
"ji4",
"xu4",
"ren3",
"zong1",
"chen1",
"duo3",
"li4",
"lv4",
"liang2",
"chou2",
"quan3",
"shao4",
"qi2",
"qi2",
"zhun3",
"qi2",
"wan3",
"qian4",
"xian4",
"shou4",
"wei2",
"qi3",
"tao2",
"wan3",
"gang1",
"wang3",
"beng1",
"zhui4",
"cai3",
"guo3",
"cui4",
"lun2",
"liu3",
"qi3",
"zhan4",
"bi4",
"chuo4",
"ling2",
"mian2",
"qi1",
"jie2",
"tian2",
"zong1",
"gun3",
"zou1",
"xi1",
"zi1",
"xing4",
"liang3",
"jin3",
"fei1",
"rui2",
"min2",
"yu4",
"zong3",
"fan2",
"lu:4",
"xu4",
"ying1",
"shang4",
"qi5",
"xu4",
"xiang1",
"jian1",
"ke4",
"xian4",
"ruan3",
"mian2",
"ji1",
"duan4",
"chong2",
"di4",
"min2",
"miao2",
"yuan2",
"xie4",
"bao3",
"si1",
"qiu1",
"bian1",
"huan3",
"geng1",
"zong3",
"mian3",
"wei4",
"fu4",
"wei3",
"tou2",
"gou1",
"miao3",
"xie2",
"lian4",
"zong1",
"bian4",
"yun4",
"yin1",
"ti2",
"gua1",
"zhi4",
"yun4",
"cheng1",
"chan2",
"dai4",
"xia2",
"yuan2",
"zong3",
"xu1",
"sheng2",
"wei1",
"geng1",
"xuan1",
"ying2",
"jin4",
"yi4",
"zhui4",
"ni4",
"bang1",
"gu3",
"pan2",
"zhou4",
"jian1",
"ci1",
"quan2",
"shuang3",
"yun4",
"xia2",
"cui1",
"xi1",
"rong2",
"tao1",
"fu4",
"yun2",
"chen1",
"gao3",
"ru4",
"hu2",
"zai4",
"teng2",
"xian4",
"su4",
"zhen3",
"zong4",
"tao1",
"huang3",
"cai4",
"bi4",
"feng4",
"cu4",
"li2",
"suo1",
"yan3",
"xi3",
"zong4",
"lei2",
"juan4",
"qian4",
"man4",
"zhi2",
"lv3",
"mu4",
"piao3",
"lian2",
"mi2",
"xuan4",
"zong3",
"ji1",
"shan1",
"sui4",
"fan2",
"lv4",
"beng3",
"yi1",
"sao1",
"mou2",
"yao2,you2,zhou4",
"qiang3",
"hun2",
"xian1",
"xi4",
"sha5",
"xiu4",
"ran2",
"xuan4",
"sui4",
"qiao1",
"zeng1",
"zuo3",
"zhi1",
"shan4",
"san3",
"lin2",
"yu4",
"fan1",
"liao2",
"chuo4",
"zun1",
"jian4",
"rao4",
"chan3",
"rui3",
"xiu4",
"hui4",
"hua4",
"zuan3",
"xi1",
"qiang3",
"yun5",
"da2",
"sheng2",
"hui4",
"xi4",
"se4",
"jian3",
"jiang1",
"huan2",
"zao3",
"cong1",
"xie4",
"jiao3",
"bi4",
"dan4",
"yi4",
"nong3",
"sui4",
"yi4",
"shai3",
"xu1",
"ji4",
"bin1",
"qian3",
"lan2",
"pu2",
"xun1",
"zuan3",
"qi2",
"peng2",
"yao4",
"mo4",
"lei4",
"xie2",
"zuan3",
"kuang4",
"you1",
"xu4",
"lei2",
"xian1",
"chan2",
"jiao3",
"lu2",
"chan2",
"ying1",
"cai2",
"rang3",
"xian1",
"zui1",
"zuan3",
"luo4",
"li2,xi3",
"dao4",
"lan3",
"lei2",
"lian4",
"si1",
"jiu1",
"yu1",
"hong2",
"zhou4",
"xian1,qian4",
"ge1,he2",
"yue1,yao1",
"ji2",
"wan2",
"kuang4",
"ji4,ji3",
"ren4",
"wei3",
"yun2",
"hong2",
"chun2",
"pi1",
"sha1",
"gang1",
"na4",
"ren4",
"zong4",
"lun2",
"fen1",
"zhi3",
"wen2",
"fang3",
"zhu4",
"zhen4",
"niu3",
"shu1",
"xian4",
"gan4",
"xie4",
"fu2",
"lian4",
"zu3",
"shen1",
"xi4",
"zhi1",
"zhong1",
"zhou4",
"ban4",
"fu2",
"chu4",
"shao4",
"yi4",
"jing1",
"dai4",
"bang3",
"rong2",
"jie2,jie1",
"ku4",
"rao4",
"die2",
"hang2",
"hui4",
"gei3,ji3",
"xuan4",
"jiang4",
"luo4,lao4",
"jue2",
"jiao3",
"tong3",
"geng3",
"xiao1",
"juan4",
"xiu4",
"xi4",
"sui2",
"tao1,di2",
"ji4",
"ti2",
"ji4",
"xu4",
"ling2",
"ying1",
"xu4",
"qi3",
"fei1",
"chuo4,chao1",
"shang4",
"gun3",
"sheng2",
"wei2",
"mian2",
"shou4",
"beng1,beng3",
"chou2",
"tao2",
"liu3",
"quan3",
"zong1,zeng4",
"zhan4",
"wan3",
"lu:4",
"zhui4,chuo4",
"zi1",
"ke4",
"xiang1",
"jian1",
"mian3",
"lan3",
"ti2",
"miao3",
"ji1,qi1",
"yun1,yun4",
"hui4",
"si1",
"duo3",
"duan4",
"bian4",
"xian4",
"gou1",
"zhui4",
"huan3",
"di4",
"lu:3",
"bian1",
"min2",
"yuan2",
"jin4",
"fu4",
"ru4",
"zhen3",
"feng4,feng2",
"cui1",
"gao3",
"chan2",
"li2",
"yi4",
"jian1",
"bin1",
"piao3",
"man4",
"lei2",
"ying1",
"suo1",
"mou2,miao4,liao3,miu4,mu4",
"sao1",
"xie2",
"liao2",
"shan4",
"zeng1,zeng4",
"jiang1",
"qian3",
"qiao1,sao1",
"huan2",
"jiao3",
"zuan3",
"fou3",
"xie4",
"gang1",
"fou3",
"que1",
"fou3",
"qi5",
"bo1",
"ping2",
"xiang4",
"zhao5",
"gang1",
"ying1",
"ying1",
"qing4",
"xia4",
"guan4",
"zun1",
"tan2",
"cheng1",
"qi4",
"weng4",
"ying1",
"lei2",
"tan2",
"lu2",
"guan4",
"wang3",
"wang3",
"wang3",
"wang3",
"han3",
"xx5",
"luo2",
"fu2",
"shen1",
"fa2",
"gu1",
"zhu3",
"ju1",
"mao2",
"gu3",
"min2",
"gang1",
"ba4,ba5",
"gua4",
"ti2",
"juan4",
"fu2",
"shen4",
"yan3",
"zhao4",
"zui4",
"gua4",
"zhuo2",
"yu4",
"zhi4",
"an3",
"fa2",
"lan3",
"shu3",
"si1",
"pi2",
"ma4",
"liu3",
"ba4",
"fa2",
"li2",
"chao2",
"wei4",
"bi4",
"ji4",
"zeng1",
"tong2",
"liu3",
"ji1",
"juan4",
"mi4",
"zhao4",
"luo2",
"pi2",
"ji1",
"ji1",
"luan2",
"yang2",
"mi3",
"qiang1",
"da2",
"mei3",
"yang2",
"you3",
"you3",
"fen2",
"ba1",
"gao1",
"yang4",
"gu3",
"qiang1",
"zang1",
"gao1",
"ling2",
"yi4",
"zhu4",
"di1",
"xiu1",
"qiang3",
"yi2",
"xian4",
"rong2",
"qun2",
"qun2",
"qiang3",
"huan2",
"suo1",
"xian4",
"yi4",
"yang5",
"qiang1",
"qian2",
"yu2",
"geng1",
"jie2",
"tang1",
"yuan2",
"xi1",
"fan2",
"shan1",
"fen3",
"shan1",
"lian3",
"lei2",
"geng1",
"nou2",
"qiang4",
"chan4",
"yu3",
"gong4",
"yi4",
"chong1",
"weng1",
"fen1",
"hong2",
"chi4",
"chi4",
"cui4",
"fu2",
"xia2",
"ben3",
"yi4",
"la1",
"yi4",
"pi1",
"ling2",
"liu4",
"zhi4",
"qu2",
"xi2",
"xie2",
"xiang2",
"xi1",
"xi1",
"ke2",
"qiao4,qiao2",
"hui4",
"hui1",
"shu4",
"sha4",
"hong2",
"jiang1",
"di2,zhai2",
"cui4",
"fei3",
"dao4",
"sha4",
"chi4",
"zhu4",
"jian3",
"xuan1",
"chi4",
"pian1",
"zong1",
"wan2",
"hui1",
"hou2",
"he2",
"he4",
"han4",
"ao2",
"piao1",
"yi4",
"lian2",
"hou2",
"ao2",
"lin2",
"pen3",
"qiao4",
"ao2",
"fan1",
"yi4",
"hui4",
"xuan1",
"dao4",
"yao4",
"lao3",
"lao3",
"kao3",
"mao4",
"zhe3",
"qi2",
"gou3",
"gou3",
"gou3",
"die2",
"die2",
"er2",
"shua3",
"ruan3",
"er2",
"nai4",
"duan1",
"lei3",
"ting1",
"zi3",
"geng1",
"chao4",
"hao4",
"yun2",
"ba4,pa2",
"pi1",
"yi2",
"si4",
"qu4",
"jia1",
"ju4",
"huo1",
"chu2",
"lao4",
"lun3",
"ji2",
"tang1",
"ou3",
"lou2",
"nou4",
"jiang3",
"pang3",
"zha2",
"lou2",
"ji1",
"lao4",
"huo4",
"you1",
"mo4",
"huai2",
"er3",
"yi4",
"ding1",
"ye2,ye1,ye5",
"da1",
"song3",
"qin2",
"yun2",
"chi3",
"dan1",
"dan1",
"hong2",
"geng3",
"zhi2",
"pan4",
"nie4",
"dan1",
"zhen3",
"che4",
"ling2",
"zheng1",
"you3",
"wa4",
"liao2",
"long2",
"zhi2",
"ning2",
"tiao1",
"er2",
"ya4",
"tie1",
"guo1",
"xu4",
"lian2",
"hao4",
"sheng4",
"lie4",
"pin4",
"jing1",
"ju4",
"bi3",
"di3",
"guo2",
"wen2",
"xu4",
"ping1",
"cong1",
"ding4",
"ni2",
"ting2",
"ju3",
"cong1",
"kui1",
"lian2",
"kui4",
"cong1",
"lian2",
"weng3",
"kui4",
"lian2",
"lian2",
"cong1",
"ao2",
"sheng1",
"song3",
"ting1",
"kui4",
"nie4",
"zhi2",
"dan1",
"ning2",
"qie2",
"ni3",
"ting1",
"ting1",
"long2",
"yu4",
"yu4",
"zhao4",
"si4",
"su4",
"yi4",
"su4",
"si4",
"zhao4",
"zhao4",
"rou4",
"yi4",
"lei4",
"ji1",
"qiu2",
"ken3",
"cao4",
"ge1",
"bo2",
"huan4",
"huang1",
"chi3",
"ren4",
"xiao4,xiao1",
"ru3",
"zhou3",
"yuan1",
"du4,du3",
"gang1",
"rong2",
"gan1",
"cha1",
"wo4",
"chang2",
"gu3",
"zhi1",
"han2",
"fu1",
"fei2",
"fen2",
"pei1",
"pang4",
"jian1",
"fang2",
"zhun1",
"you2",
"na4",
"ang1",
"ken3",
"ran2",
"gong1",
"yu4",
"wen3",
"yao2",
"jin4",
"pi2",
"qian3",
"xi4",
"xi1",
"fei4",
"ken3",
"jing3",
"tai4",
"shen4",
"zhong3",
"zhang4",
"xie2",
"shen4",
"wei4",
"zhou4",
"die2",
"dan3",
"fei4",
"ba2",
"bo2",
"qu2",
"tian2",
"bei4,bei1",
"gua1",
"tai1",
"zi3",
"fei3",
"zhi1",
"ni4",
"ping2",
"zi4",
"fu3",
"pang4,pan2",
"zhen1",
"xian2",
"zuo4",
"pei1",
"jia3",
"sheng4",
"zhi1",
"bao1",
"mu3",
"qu1",
"hu2",
"ke1",
"chi3",
"yin4",
"xu1",
"yang1",
"long2",
"dong4",
"ka3",
"lu2",
"jing4",
"nu3",
"yan1",
"pang1",
"kua4",
"yi2",
"guang1",
"hai3",
"ge1",
"dong4",
"chi1",
"jiao1",
"xiong1",
"xiong1",
"er2",
"an4",
"heng2",
"pian2",
"neng2",
"zi4",
"gui1",
"cheng2",
"tiao3",
"zhi1",
"cui4",
"mei2",
"xie2",
"cui4",
"xie2",
"mai4",
"mai4,mo4",
"ji3",
"xie2",
"nin5",
"kuai4",
"sa4",
"zang4,zang1",
"qi2",
"nao3",
"mi3",
"nong2",
"luan2",
"wan4",
"bo2",
"wen3",
"wan3",
"xiu1",
"jiao3,jue2",
"jing4",
"you3",
"heng1",
"cuo3",
"lie4",
"shan1",
"ting3",
"mei2",
"chun2",
"shen4",
"qian3",
"de5",
"juan1",
"ji2",
"xiu1",
"xin4",
"tuo1",
"pao1",
"cheng2",
"nei3",
"pu2,fu3",
"dou4",
"tuo1",
"niao4",
"nao3",
"pi3",
"gu3",
"luo2",
"li4",
"lian3",
"zhang4",
"cui4",
"jie1",
"liang3",
"shui2",
"pi2",
"biao1",
"lun2",
"pian2",
"lei3",
"kui4",
"chui2",
"dan4",
"tian3",
"nei3",
"jing1",
"nai2",
"la4,xi1",
"ye4",
"yan1",
"ren3",
"shen4",
"chuo4",
"fu3",
"fu3",
"ju1",
"fei2",
"qiang1",
"wan4",
"dong4",
"pi2",
"guo2",
"zong1",
"ding4",
"wo4",
"mei2",
"ni2",
"zhuan4",
"chi4",
"cou4",
"luo2",
"ou3",
"di4",
"an1",
"xing1",
"nao3",
"shu4",
"shuan4",
"nan3",
"yun4",
"zhong3",
"rou2",
"e4",
"sai1",
"tu2",
"yao1",
"jian4",
"wei3",
"jiao3",
"yu2",
"jia1",
"duan4",
"bi4",
"chang2",
"fu4",
"xian4",
"ni4",
"mian3",
"wa4",
"teng2",
"tui3",
"bang3,bang4,pang1,pang2",
"qian3",
"lu:3",
"wa4",
"shou4",
"tang2",
"su4",
"zhui4",
"ge2",
"yi4",
"bo2",
"liao2",
"ji2",
"pi2",
"xie2",
"gao1,gao4",
"lv3",
"bin4",
"ou1",
"chang2",
"lu4,biao1",
"guo2",
"pang1",
"chuai2",
"biao1",
"jiang3",
"fu1",
"tang2",
"mo2",
"xi1",
"zhuan1",
"lu:4",
"jiao1",
"ying4",
"lv2",
"zhi4",
"xue3",
"cun1",
"lin4",
"tong2",
"peng2",
"ni4",
"chuai4",
"liao2",
"cui4",
"gui1",
"xiao1",
"teng1",
"fan2",
"zhi2",
"jiao1",
"shan4",
"hu1",
"cui4",
"run4",
"xiang1",
"sui3",
"fen4",
"ying1",
"shan1",
"zhua1",
"dan3",
"kuai4",
"nong2",
"tun2",
"lian2",
"bi4",
"yong1",
"jue2",
"chu4",
"yi4",
"juan3",
"la4",
"lian3",
"sao1,sao4",
"tun2",
"gu3",
"qi2",
"cui4",
"bin4",
"xun1",
"nao4",
"wo4",
"zang4",
"xian4",
"biao1",
"xing4",
"kuan1",
"la4",
"yan1",
"lu2",
"huo4",
"za1",
"luo3",
"qu2",
"zang4",
"luan2",
"ni2",
"za1",
"chen2",
"qian1",
"wo4",
"guang4",
"zang1",
"lin2",
"guang3",
"zi4",
"jiao3",
"nie4",
"chou4,xiu4",
"ji4",
"gao1",
"chou4",
"mian2",
"nie4",
"zhi4",
"zhi4",
"ge2",
"jian4",
"die2",
"zhi1",
"xiu1",
"tai2",
"zhen1",
"jiu4",
"xian4",
"yu2",
"cha1",
"yao3",
"yu2",
"chong1",
"xi4",
"xi4",
"jiu4",
"yu2",
"yu3",
"xing4",
"ju3",
"jiu4",
"xin4",
"she2",
"she3,she4",
"she4",
"jiu3",
"shi4",
"tan1",
"shu1",
"shi4",
"tian3",
"tan4",
"pu4",
"pu4",
"guan3",
"hua4",
"tian4",
"chuan3",
"shun4",
"xia2",
"wu3",
"zhou1",
"dao1",
"gang1",
"shan1",
"yi3",
"fan2",
"pa1",
"tai4",
"fan2",
"ban3",
"chuan2",
"hang2",
"fang3",
"ban1,pan2",
"bi3",
"lu2",
"zhong1",
"jian4",
"cang1",
"ling2",
"zhu2",
"ze2",
"duo4",
"bo2",
"xian2",
"ge3",
"chuan2",
"xia2",
"lu2",
"qiong2",
"pang2",
"xi1",
"kua1",
"fu2",
"zao4",
"feng2",
"li2",
"shao1",
"yu2",
"lang2",
"ting3",
"yu4",
"wei3",
"bo2",
"meng3",
"nian4",
"ju1",
"huang2",
"shou3",
"ke4",
"bian4",
"mu4",
"die2",
"dao4",
"bang4",
"cha1",
"yi4",
"sou1",
"cang1",
"cao2",
"lou2",
"dai4",
"xue3",
"yao4",
"chong1",
"deng1",
"dang1",
"qiang2",
"lu3",
"yi3",
"ji2",
"jian4",
"huo4",
"meng2",
"qi2",
"lu3",
"lu2",
"chan2",
"shuang1",
"gen3,gen4",
"liang2",
"jian1",
"jian1",
"se4,shai3",
"yan4",
"fu2",
"ping1",
"yan4",
"yan4",
"cao3",
"cao3",
"yi4",
"le4",
"ting1",
"jiao1",
"ai4,yi4",
"nai3",
"tiao2",
"jiao1",
"jie2,jie1",
"peng2",
"wan2",
"yi4",
"chai1",
"mian2",
"mi3",
"gan1",
"qian1",
"yu4",
"yu4",
"shao2,que4",
"xiong1",
"du4",
"hu4",
"qi3",
"mang2",
"zi4",
"hui4",
"sui1",
"zhi4",
"xiang1",
"pi2",
"fu2",
"tun2",
"wei3",
"wu2",
"zhi1",
"qi4",
"shan1",
"wen2",
"qian4",
"ren2",
"fu2",
"kou1",
"jie4,gai4",
"lu2",
"xu4",
"ji2",
"qin2",
"qi2",
"yuan2",
"fen1",
"ba1",
"rui4",
"xin1,xin4",
"ji4",
"hua1",
"lun2",
"fang1",
"wu4,hu1",
"jue2",
"gou3",
"zhi3",
"yun2,yi4",
"qin2",
"ao3",
"chu2",
"mao4",
"ya2",
"fei4,fu2",
"reng4",
"hang2",
"cong1",
"yin2",
"you3",
"bian4",
"yi4",
"qie1",
"wei3",
"li4",
"pi3",
"e4",
"xian4",
"chang2",
"cang1",
"zhu4",
"su1",
"ti2",
"yuan4",
"ran3",
"ling2",
"tai2,tai1",
"shao2,tiao2",
"di2",
"miao2",
"qing3",
"li4",
"yong4",
"ke1",
"mu4",
"bei4",
"bao1",
"gou3",
"min2",
"yi3",
"yi3",
"ju4",
"pie3",
"ruo4",
"ku3",
"ning2",
"ni3",
"pa1",
"bing3",
"shan1",
"xiu2",
"yao3",
"xian1",
"ben3",
"hong2",
"ying1",
"zha3",
"dong1",
"ju1",
"die2",
"nie2",
"gan1",
"hu1",
"ping2",
"mei2",
"fu2",
"sheng1",
"gu1",
"bi4",
"wei4",
"fu2",
"zhuo2",
"mao4",
"fan4",
"jia1,qie2",
"mao2",
"mao2,mao3",
"ba2",
"zi3",
"mo4",
"zi1",
"zhi3",
"chi2",
"ji2",
"jing1",
"long2",
"cong1",
"niao3",
"yuan2",
"xue2",
"ying2",
"qiong2",
"ge2",
"ming2",
"li4",
"rong2",
"yin4",
"gen4",
"qian4,xi1",
"chai3",
"chen2",
"yu4",
"xiu1",
"zi4",
"lie4",
"wu2",
"ji4",
"gui1",
"ce4",
"jian3",
"ci2",
"gou4",
"guang1",
"mang2",
"cha2",
"jiao1",
"jiao1",
"fu2",
"yu2",
"zhu1",
"zi1",
"jiang1",
"hui2",
"yin1",
"cha2",
"fa2",
"rong2",
"ru2",
"chong1",
"mang3",
"tong2",
"zhong4",
"qian1",
"zhu2",
"xun2",
"huan2",
"fu1",
"quan2",
"gai1",
"da2",
"jing1",
"xing4",
"chuan3",
"cao3,cao4",
"jing1",
"er2",
"an4",
"qiao2",
"chi2",
"ren3",
"jian4",
"ti2,yi2",
"huang1",
"ping2",
"li4",
"jin1",
"lao3",
"shu4",
"zhuang1",
"da2",
"jia2",
"rao2",
"bi4",
"ce4",
"qiao2",
"hui4",
"ji4,qi2",
"dang4,tang4",
"zi4",
"rong2",
"hun1",
"xing2",
"luo4",
"ying2",
"xun2",
"jin4",
"sun1",
"yin4",
"mai3",
"hong2",
"zhou4",
"yao4",
"du4",
"wei3",
"li2",
"dou4",
"fu1",
"ren3",
"yin2",
"he2,he4",
"bi2",
"bu4",
"yun3",
"di2",
"tu2",
"sui1",
"sui1",
"cheng2",
"chen2",
"wu2",
"bie2",
"xi1",
"geng3",
"li4",
"pu2",
"zhu4",
"mo4",
"li4",
"zhuang1",
"zuo2",
"tuo1",
"qiu2",
"sha1,suo1",
"suo1",
"chen2",
"peng2",
"ju3",
"mei2",
"meng2",
"xing4",
"jing1",
"che1",
"shen1,xin1",
"jun1",
"yan2",
"ting2",
"you2",
"cuo4",
"guan3,guan1,wan3",
"han4",
"you3",
"cuo4",
"jia2",
"wang2",
"su4",
"niu3",
"shao1",
"xian4",
"lang4",
"fu2,piao3",
"e2",
"mo4",
"wen4",
"jie2",
"nan2",
"mu4",
"kan3",
"lai2",
"lian2",
"shi2,shi4",
"wo1",
"tu4",
"xian1",
"huo4",
"you2",
"ying2",
"ying1",
"gong4",
"chun2",
"mang3",
"mang3",
"ci4",
"wan3",
"jing1",
"di4",
"qu2",
"dong1",
"jian1",
"zou1",
"gu1",
"la1",
"lu4",
"ju2",
"wei4",
"jun1,jun4",
"nie4",
"kun1",
"he2",
"pu2",
"zi1",
"gao3",
"guo3",
"fu2",
"lun2",
"chang1",
"chou2",
"song1",
"chui2",
"zhan4",
"men2",
"cai4",
"ba2",
"li2",
"tu4",
"bo1",
"han4",
"bao4",
"qin4",
"juan3",
"xi1",
"qin2",
"di3",
"jie1",
"pu2",
"dang4",
"jin3",
"qiao2",
"tai2",
"geng1",
"hua2",
"gu1",
"ling2",
"fei1,fei3",
"qin2",
"an1",
"wang3",
"beng3",
"zhou3",
"yan1,yu1",
"zu1",
"jian1",
"lin3",
"tan3",
"shu1",
"tian2",
"dao4",
"hu3",
"qi2",
"he2",
"cui4",
"tao2",
"chun1",
"bi4",
"chang2",
"huan2",
"fei4",
"lai2",
"qi1",
"meng2",
"ping2",
"wei3",
"dan4",
"sha4",
"huan2",
"yan3",
"yi2",
"tiao2",
"ji4",
"wan3",
"ce4",
"nai4",
"zhen3",
"tuo4",
"jiu1",
"tie1",
"luo2",
"bi4",
"yi4",
"pan1",
"bo5",
"pao1",
"ding4",
"ying2",
"ying2",
"ying2",
"xiao1",
"sa4",
"qiu1",
"ke1",
"xiang4",
"wan4",
"yu3",
"yu2",
"fu4",
"lian4",
"xuan1",
"xuan1",
"nan3",
"ce4",
"wo1",
"chun3",
"xiao1",
"yu2",
"bian3,bian1",
"mao4",
"an1",
"e4",
"luo4,la4,lao4",
"ying2",
"kuo4",
"kuo4",
"jiang1",
"mian3",
"zuo4",
"zuo4",
"zu1",
"bao3",
"rou2",
"xi3",
"ye4",
"an1",
"qu2",
"jian1",
"fu2",
"lu:4",
"jing1",
"pen2",
"feng1",
"hong2",
"hong2",
"hou2",
"yan4",
"tu2",
"zhu4",
"zi1",
"xiang1",
"shen4",
"ge2,ge3",
"qia1",
"qing2",
"mi3",
"huang2",
"shen1",
"pu2",
"gai4",
"dong3",
"zhou4",
"qian2",
"wei3",
"bo2",
"wei1",
"pa1",
"ji4",
"hu2",
"zang4",
"jia1",
"duan4",
"yao4",
"sui1",
"cong1",
"quan2",
"wei1",
"zhen1",
"kui2",
"ting2",
"hun1",
"xi3",
"shi1",
"qi4",
"lan2",
"zong1",
"yao1",
"yuan1",
"mei2",
"yun1",
"shu4",
"di4",
"zhuan4",
"guan1",
"ran3",
"xue1",
"chan3",
"kai3",
"kui4",
"hua1",
"jiang3",
"lou2",
"wei3",
"pai4",
"you5",
"sou1",
"yin1",
"shi1",
"chun2",
"shi2",
"yun1",
"zhen1",
"lang4",
"ru2",
"meng2,meng1,meng3",
"li4",
"que1",
"suan4",
"yuan2",
"li4",
"ju3",
"xi1",
"bang4",
"chu2",
"xu2",
"tu2",
"liu2",
"huo4",
"dian3",
"qian4",
"zu1",
"po4",
"cuo2",
"yuan1",
"chu2",
"yu4",
"kuai3",
"pan2",
"pu2",
"pu2",
"na4",
"shuo4",
"xi2",
"fen2",
"yun2",
"zheng1",
"jian1",
"ji2",
"ruo4",
"cang1",
"en1",
"mi2",
"hao1",
"sun1",
"zhen1",
"ming2",
"sou1",
"xu4",
"liu2",
"xi2",
"gu3",
"lang2",
"rong2",
"weng3",
"gai4",
"cuo4",
"shi1",
"tang2",
"luo3",
"ru4",
"suo1",
"xuan1",
"bei4",
"yao3,zhuo2",
"gui4",
"bi4",
"zong3",
"gun3",
"zuo4",
"tiao2",
"ce4",
"pei4",
"lan2",
"dan4",
"ji4",
"li2",
"shen1",
"lang3",
"yu4",
"ling2",
"ying2",
"mo4",
"diao4",
"tiao2",
"mao3",
"tong1",
"zhu2",
"peng2",
"an1",
"lian2",
"cong1",
"xi3",
"ping2",
"qiu1",
"jin3",
"chun2",
"jie2",
"wei2",
"tui1",
"cao2",
"yu4",
"yi4",
"zi2",
"liao3,lu4",
"bi4",
"lu3",
"xu4",
"bu4",
"zhang1",
"lei2",
"qiang2",
"man4",
"yan2",
"ling2",
"ji4",
"biao1",
"gun3",
"han3",
"di2",
"su4",
"lu4",
"she4",
"shang1",
"di2",
"mie4",
"xun1",
"man4,man2",
"bo2",
"di4",
"cuo2",
"zhe4",
"shen1",
"xuan4",
"wei4,yu4",
"hu2",
"ao2",
"mi3",
"lou2",
"cu4",
"zhong1",
"cai4",
"po2",
"jiang3",
"mi4",
"cong1",
"niao3",
"hui4",
"juan4",
"yin2",
"jian4",
"nian1",
"shu1",
"yin1",
"guo2",
"chen2",
"hu4",
"sha1",
"kou4",
"qian4",
"ma2",
"cang2",
"ze2",
"qiang2",
"dou1",
"lian3",
"lin4",
"kou4",
"ai3",
"bi4",
"li2",
"wei3",
"ji2",
"qian2",
"sheng4",
"fan1,bo1,fan2",
"meng2",
"ou3",
"chan3",
"dian3",
"xun4",
"jiao1,qiao2",
"rui3",
"rui3",
"lei3",
"yu2",
"qiao2",
"chu2",
"hua2",
"jian1",
"mai3",
"yun2",
"bao1",
"you2",
"qu2",
"lu4",
"rao2",
"hui4",
"e4",
"ti2",
"fei3",
"jue2",
"zui4",
"fa4",
"ru2",
"fen2",
"kui4",
"shun4",
"rui2",
"ya3",
"xu1",
"fu4",
"jue2",
"dang4",
"wu2",
"dong3",
"si1",
"xiao1",
"xi4",
"long2",
"yun4",
"shao1",
"qi2",
"jian1",
"yun4",
"sun1",
"ling2",
"yu4",
"xia2",
"weng4",
"ji2",
"hong4",
"si4",
"nong2",
"lei3",
"xuan1",
"yun4",
"yu4",
"xi2",
"hao4",
"bao2,bo2,bo4",
"hao1",
"ai4",
"wei1",
"hui4",
"wei4",
"ji4",
"ci2",
"xiang1",
"wan4",
"mie4",
"yi4",
"leng2",
"jiang1",
"can4",
"shen1",
"qiang2",
"lian2",
"ke1",
"yuan2",
"da2",
"ti4",
"tang1",
"xue1",
"bi4",
"zhan1",
"sun1",
"xian1",
"fan2",
"ding3",
"xiao4",
"gu3",
"xie4",
"shu3",
"jian4",
"kao3",
"hong1",
"sa4",
"xin1",
"xun1",
"yao4",
"bai4",
"sou3",
"shu3",
"xun1",
"dui4",
"pin2",
"wei3",
"ning2",
"chou2",
"mai2,wo1",
"ru2",
"piao1",
"tai2",
"ji4",
"zao3",
"chen2",
"zhen1",
"er3",
"ni3",
"ying2",
"gao3",
"cong2",
"xiao1",
"qi2",
"fa2",
"jian3",
"xu4",
"kui2",
"ji2,jie4",
"bian3",
"diao4",
"mi4",
"lan2",
"jin4",
"cang2,zang4",
"miao3",
"qiong2",
"qie4",
"xian3",
"liao2",
"ou3",
"xian2",
"su4",
"lu:2",
"yi4",
"xu4",
"xie3",
"li2",
"yi4",
"la3",
"lei3",
"jiao4",
"di2",
"zhi3",
"bei1",
"teng2",
"yao4",
"mo2",
"huan4",
"biao1",
"fan1",
"sou3",
"tan2",
"tui1",
"qiong2",
"qiao2",
"wei4",
"liu2",
"hui4",
"ou1",
"gao3",
"yun4",
"bao3",
"li4",
"shu3",
"chu2",
"ai3",
"lin4",
"zao3",
"xuan1",
"qin4",
"lai4",
"huo4",
"tuo4",
"wu4",
"rui3",
"rui3",
"qi2",
"heng2",
"lu2",
"su1",
"tui2",
"meng2",
"yun4",
"ping2",
"yu3",
"xun1",
"ji4",
"jiong1",
"xuan1",
"mo2",
"qiu1",
"su1",
"jiong1",
"peng2",
"nie4",
"nie4",
"rang2",
"yi4",
"xian3",
"yu2",
"ju2",
"lian3",
"lian3",
"yin3",
"qiang2",
"ying1",
"long2",
"tou3",
"hua1",
"yue4",
"ling2",
"qu2",
"yao2",
"fan2",
"mei2",
"han4",
"kui1",
"lan2",
"ji4",
"dang4",
"man4",
"lei4",
"lei2",
"hui1",
"feng1",
"zhi2",
"wei4",
"kui2",
"zhan4",
"huai2",
"li2",
"ji4",
"mi2",
"lei3",
"huai4",
"luo2",
"ji1",
"kui2",
"lu4",
"jian1",
"sa4",
"teng2",
"lei2",
"quan3",
"xiao1",
"yi4",
"luan2",
"men2",
"bie1",
"hu1",
"hu3",
"lu3",
"nu:e4",
"lu:4",
"si1",
"xiao1",
"qian2",
"chu4",
"hu1",
"xu1",
"cuo2",
"fu2",
"xu1",
"xu1",
"lu3",
"hu3",
"yu2",
"hao4",
"jiao1",
"ju4",
"guo2",
"bao4",
"yan2",
"zhan4",
"zhan4",
"kui1",
"bin1",
"xi4",
"shu4",
"chong2",
"qiu2",
"diao1",
"ji3",
"qiu2",
"ding1",
"shi1",
"xia1",
"jue2",
"zhe2",
"she2",
"yu1",
"gan1",
"zi3",
"hong2",
"hui1,hui3",
"meng2",
"ge4",
"sui1",
"xia1,ha2",
"chai4",
"shi2",
"yi3",
"ma3,ma1,ma4",
"xiang4",
"fang1",
"e4",
"ba1",
"chi3",
"qian1",
"wen2",
"wen2",
"rui4",
"bang4,beng4",
"pi2",
"yue4",
"yue4",
"jun1",
"qi2",
"tong2",
"yin3",
"qi2",
"can2",
"yuan2",
"jue2",
"hui2",
"qin2",
"qi2",
"zhong4",
"ya2",
"hao2",
"mu4",
"wang2",
"fen2",
"fen2",
"hang2",
"gong1",
"zao3",
"fu4",
"ran2",
"jie4",
"fu2",
"chi1",
"dou3",
"bao4",
"xian3",
"ni2",
"dai4",
"qiu1",
"you2",
"zha4",
"ping2",
"chi2",
"you4",
"e2",
"han1",
"ju4",
"li4",
"fu4",
"ran2",
"zha2",
"gou3",
"pi2",
"pi2",
"xian2",
"zhu4",
"diao1",
"bie2",
"bing3",
"gu1",
"ran2",
"qu1",
"she2",
"tie3",
"ling2",
"gu3",
"dan4",
"gu3",
"ying2",
"li4",
"cheng1",
"qu1",
"mou2",
"ge2",
"ci4",
"hui2",
"hui2",
"mang2",
"fu4",
"yang2",
"wa1",
"lie4",
"zhu1",
"yi1",
"xian2",
"kuo4",
"jiao1",
"li4",
"yi4",
"ping2",
"jie2",
"ha2,ge2",
"she2",
"yi2",
"wang3",
"mo4",
"qiong2",
"qie4",
"gui3",
"qiong2",
"zhi4",
"man2",
"lao3",
"zhe2",
"jia2",
"nao2",
"si1",
"qi2",
"xing2",
"jie4",
"qiu2",
"shao1,xiao1",
"yong3",
"jia2",
"tui4",
"che1",
"bei4",
"e2",
"han4",
"shu3",
"xuan2",
"feng1",
"shen4",
"shen4",
"fu3",
"xian4",
"zhe1,zhe2",
"wu2",
"fu2",
"li2",
"lang2",
"bi4",
"chu2",
"yuan1",
"you3",
"jie2",
"dan4",
"yan2",
"ting2",
"dian4",
"tui4",
"hui2",
"wo1",
"zhi1",
"song1",
"fei3",
"ju1",
"mi4",
"qi2",
"qi2",
"yu4",
"jun4",
"la4",
"meng3",
"qiang1",
"si1",
"xi1",
"lun2",
"li4",
"die2",
"tiao2",
"tao2",
"kun1",
"han2",
"han4",
"yu4",
"bang4",
"fei2",
"pi2",
"wei1",
"dun1",
"yi4",
"yuan1",
"suo4",
"quan2",
"qian3",
"rui4",
"ni2",
"qing1",
"wei4",
"liang3",
"guo3",
"wan1",
"dong1",
"e4",
"ban3",
"di4,zhuo1",
"wang3",
"can2",
"yang3",
"ying2",
"guo1",
"chan2",
"ding4",
"la4",
"ke1",
"ji2",
"xie1",
"ting2",
"mao4",
"xu1",
"mian2",
"yu2",
"jie1",
"shi2",
"xuan1",
"huang2",
"yan3",
"bian1",
"rou2",
"wei1",
"fu4",
"yuan2",
"mei4",
"wei4",
"fu2",
"ru2",
"xie2",
"you2",
"qiu2",
"mao2",
"xia1",
"ying1",
"shi1",
"chong2",
"tang1",
"zhu1",
"zong1",
"ti2",
"fu4",
"yuan2",
"kui2",
"meng2",
"la4",
"dai4",
"hu2",
"qiu1",
"die2",
"li4",
"wo1",
"yun1",
"qu3",
"nan3",
"lou2",
"chun1",
"rong2",
"ying2",
"jiang1",
"ban5",
"lang2",
"pang2",
"si1",
"xi1",
"ci4",
"xi1",
"yuan2",
"weng1",
"lian2",
"sou1",
"ban1",
"rong2",
"rong2",
"ji2",
"wu1",
"xiu4",
"han4",
"qin2",
"yi2",
"bi1",
"hua2",
"tang2",
"yi3",
"du4",
"nai4",
"he2",
"hu2",
"gui1",
"ma3",
"ming2",
"yi4",
"wen2",
"ying2",
"teng2",
"zhong1",
"cang1",
"sao1",
"qi2",
"man3",
"tiao5",
"shang1",
"shi4",
"cao2",
"chi1",
"di4",
"ao2",
"lu4",
"wei4",
"zhi4",
"tang2",
"chen2",
"piao1",
"qu2",
"pi2",
"yu2",
"jian4",
"luo2",
"lou2",
"qin3",
"zhong1",
"yin3",
"jiang1",
"shuai4",
"wen2",
"xiao1",
"wan4",
"zhe2",
"zhe4",
"ma2",
"ma2",
"guo1",
"liu2",
"mao2",
"xi1",
"cong1",
"li2",
"man3",
"xiao1",
"chang5",
"zhang1",
"mang3",
"xiang4",
"mo4",
"zui1",
"si1",
"qiu1",
"te4",
"zhi2",
"peng2",
"peng2",
"jiao3",
"qu2",
"bie1",
"liao2",
"pan2",
"gui3",
"xi3",
"ji3",
"zhuan1",
"huang2",
"fei2",
"lao2",
"jue2",
"jue2",
"hui4",
"yin2",
"chan2",
"jiao1",
"shan4",
"nao2",
"xiao1",
"wu2",
"chong2",
"xun2",
"si1",
"chu2",
"cheng1",
"dang1",
"li3",
"xie4",
"shan4",
"yi3",
"jing3",
"da2",
"chan2",
"qi4",
"ci1",
"xiang3",
"she4",
"luo3",
"qin2",
"ying2",
"chai4",
"li4",
"zei2",
"xuan1",
"lian2",
"zhu2",
"ze2",
"xie1",
"mang3",
"xie4",
"qi2",
"rong2",
"jian3",
"meng3",
"hao2",
"ru2",
"huo4",
"zhuo2",
"jie2",
"pin2",
"he1",
"mie4",
"fan2",
"lei3",
"jie2",
"la4",
"min3",
"li2,li3",
"chun3",
"li4",
"qiu1",
"nie4",
"lu2",
"du4",
"xiao1",
"zhu1",
"long2",
"li2",
"long2",
"feng1",
"ye1",
"pi2",
"shang4",
"gu3",
"juan1",
"ying1",
"shu3",
"xi1",
"can2",
"qu2",
"quan2",
"du4",
"can2",
"man2",
"qu2",
"jie2",
"zhu2",
"zhuo1",
"xue4",
"huang1",
"nv4",
"pei1",
"nu:4",
"xin4",
"zhong4",
"mai4",
"er4",
"ke4",
"mie4",
"xi4",
"xing2,hang2",
"yan3",
"kan4",
"yuan4",
"qu2",
"ling2",
"xuan4",
"shu4",
"xian2",
"tong4",
"xiang4",
"jie1",
"xian2",
"ya2",
"hu2",
"wei4",
"dao4",
"chong1",
"wei4",
"dao4",
"zhun1",
"heng2",
"qu2",
"yi1,yi4",
"yi1",
"bu3",
"gan3",
"yu2",
"biao3",
"cha3,cha4",
"yi2",
"shan1",
"chen4",
"fu1",
"gun3",
"fen1",
"shuai1,cui1",
"jie2",
"na4",
"zhong1",
"dan3",
"ni4",
"zhong4",
"zhong1",
"jie4",
"qi2",
"xie2",
"ran2",
"zhi1",
"ren4",
"qin1",
"jin1",
"jun1",
"yuan2",
"mei4",
"chai4",
"ao3",
"niao3",
"hui1",
"ran2",
"jia1",
"tuo2",
"ling3",
"dai4",
"bao4",
"pao2",
"yao4",
"zuo4",
"bi4",
"shao4",
"tan3",
"ju4",
"he4",
"xue2",
"xiu4",
"zhen3",
"yi2",
"pa4",
"fu2",
"di1",
"wa4",
"fu4",
"gun3",
"zhi4",
"zhi4",
"ran2",
"pan4",
"yi4",
"mao4",
"tuo1",
"na4",
"gou1",
"xuan4",
"zhe2",
"qu1",
"bei4",
"yu4",
"xi2",
"ni3,mi5",
"bo2",
"bo1",
"fu2",
"yi2",
"chi3",
"ku4",
"ren4",
"jiang4",
"jia2",
"jian4",
"bo2",
"jie2",
"er2",
"ge1",
"ru2",
"zhu1",
"gua4",
"yin1",
"cai2",
"lie4",
"ka3",
"xing5",
"zhuang1",
"dang1",
"xu1",
"kun1",
"ken4",
"niao3",
"shu4",
"jia2",
"kun3",
"cheng2,cheng3",
"li3",
"juan1",
"shen1",
"pou2",
"ge2",
"yi4",
"yu4",
"zhen3",
"liu2",
"qiu2",
"qun2",
"ji4",
"yi4",
"bu3",
"zhuang1",
"shui4",
"sha1",
"qun2",
"li3",
"lian2",
"lian3",
"ku4",
"jian3",
"fou2",
"chan1",
"bi4,pi2",
"kun1",
"tao2",
"yuan4",
"ling2",
"chi3",
"chang1",
"chou2",
"duo1",
"biao3",
"liang3",
"chang2",
"pei2",
"pei2",
"fei1",
"yuan1",
"luo3",
"guo3",
"yan3",
"du2",
"ti4,xi1",
"zhi4",
"ju1",
"yi3",
"qi2",
"guo3",
"gua4",
"ken4",
"qi1",
"ti4",
"ti2",
"fu4",
"chong2",
"xie1",
"bian3",
"die2",
"kun1",
"duan1",
"xiu4",
"xiu4",
"he4",
"yuan4",
"bao1",
"bao3",
"fu4",
"yu2",
"tuan4",
"yan3",
"hui1",
"bei4",
"chu3,zhu3",
"lu:3",
"pao2",
"dan1",
"yun3",
"ta1",
"gou1",
"da1",
"huai2",
"rong2",
"yuan4",
"ru4",
"nai4",
"jiong3",
"suo3",
"ban1",
"tui4,tun4",
"chi3",
"sang3",
"niao3",
"ying1",
"jie4",
"qian1",
"huai2",
"ku4",
"lian2",
"lan2",
"li2",
"zhe3,xi2",
"shi1",
"lv3",
"yi4",
"die1",
"xie4",
"xian1",
"wei4",
"biao3",
"cao2",
"ji1",
"qiang3",
"sen1",
"bao1",
"xiang1",
"bi4",
"fu2",
"jian3",
"zhuan4",
"jian3",
"cui4",
"ji2",
"dan1",
"za2",
"fan2",
"bo2",
"xiang4",
"xin2",
"bie2",
"rao2",
"man3",
"lan2",
"ao3",
"ze2",
"gui4",
"cao4",
"sui4",
"nong2",
"chan1",
"lian3",
"bi4",
"jin1",
"dang1",
"shu2",
"zhan4",
"bi4",
"lan2",
"fu2",
"ru2",
"zhi3",
"dui4",
"shu3",
"wa4",
"shi4",
"bai3",
"xie2",
"bo2",
"chen4",
"lai4",
"long2",
"xi2",
"xian1",
"lan2",
"zhe3",
"dai4",
"ju3",
"zan4",
"shi1",
"jian3",
"pan4",
"yi4",
"lan2",
"ya4",
"xi1",
"xi1",
"yao4,yao1",
"feng3",
"tan2,qin2",
"fu4",
"fiao4",
"fu4",
"ba4",
"he2",
"ji1",
"ji1",
"jian4",
"guan1",
"bian4",
"yan4",
"gui1",
"jue2,jiao4",
"pian3",
"mao4",
"mi4",
"mi4",
"mie4",
"shi4",
"si1",
"chan1",
"luo2",
"jue2",
"mi4",
"tiao4",
"lian2",
"yao4",
"zhi4",
"jun1",
"xi2",
"shan3",
"wei1",
"xi4",
"tian3",
"yu2",
"lan3",
"e4",
"du3",
"qin1",
"pang3",
"ji4",
"ming2",
"ying2",
"gou4",
"qu4",
"zhan4",
"jin4",
"guan1",
"deng1",
"jian4",
"luo2",
"qu4",
"jian1",
"wei2",
"jue2",
"qu1",
"luo2",
"lan3",
"shen3",
"di2",
"guan1",
"jian4,xian4",
"guan1,guan4",
"yan4",
"gui1",
"mi4",
"shi4",
"chan1",
"lan3",
"jue2,jiao4",
"ji4",
"xi2",
"di2",
"tian3",
"yu2",
"gou4",
"jin4",
"qu4",
"jiao3,jue2",
"qiu2",
"jin1",
"cu1",
"jue2",
"zhi4",
"chao4",
"ji2",
"gu1",
"dan4",
"zi1,zui3",
"di3",
"shang1",
"hua4",
"quan2",
"ge2",
"shi4",
"jie3,xie4,jie4",
"gui3",
"gong1",
"chu4",
"jie3,jie4,xie4",
"hun4",
"qiu2",
"xing1",
"su4",
"ni2",
"ji1",
"lu4",
"zhi4",
"zha1",
"bi4",
"xing1",
"hu2",
"shang1",
"gong1",
"zhi4",
"xue2",
"chu4",
"xi1",
"yi2",
"li4",
"jue2",
"xi1",
"yan4",
"xi1",
"yan2",
"yan2",
"ding4",
"fu4",
"qiu2",
"qiu2",
"jiao4",
"hong1",
"ji4",
"fan4",
"xun4",
"diao4",
"hong4",
"chai4",
"tao3",
"xu1",
"jie2",
"yi2",
"ren4",
"xun4",
"yin2",
"shan4",
"qi4",
"tuo1",
"ji4",
"xun4",
"yin2",
"e2",
"fen1",
"ya4",
"yao1",
"song4",
"shen3",
"yin2",
"xin1",
"jue2",
"xiao2",
"ne4",
"chen2",
"you2",
"zhi3",
"xiong1",
"fang3",
"xin4",
"chao1",
"she4",
"yan2",
"sa3",
"zhun4",
"xu3",
"yi4",
"yi4",
"su4",
"chi1",
"he1",
"shen1",
"he2",
"xu4",
"zhen3",
"zhu4",
"zheng4",
"gou4",
"zi1,zi3",
"zi3",
"zhan1",
"gu3",
"fu4",
"jian3",
"die2",
"ling2",
"di3",
"yang4",
"li4",
"nao2",
"pan4",
"zhou4",
"gan4",
"yi4",
"ju4",
"yao4",
"zha4",
"tuo2",
"yi2",
"qu3",
"zhao4",
"ping2",
"bi4",
"xiong4",
"qu1",
"ba2",
"da2",
"zu3",
"tao1",
"zhu3",
"ci2",
"zhe2",
"yong3",
"xu3",
"xun2",
"yi4",
"huang3",
"he2",
"shi4",
"cha2",
"jiao1",
"shi1",
"hen3",
"cha4",
"gou4",
"gui3",
"quan2",
"hui4",
"jie2",
"hua4",
"gai1",
"xiang2",
"wei1",
"shen1",
"zhou4",
"tong2",
"mi2",
"zhan1",
"ming4",
"e4",
"hui1",
"yan2",
"xiong1",
"gua4",
"er4",
"bing4",
"tiao3",
"chi3",
"lei3",
"zhu1",
"kuang1",
"kua1",
"wu1",
"yu4",
"teng2",
"ji4",
"zhi4",
"ren4",
"cu4",
"lang3",
"e2",
"kuang2",
"ei2",
"shi4",
"ting3",
"dan4",
"bei4",
"chan2",
"you4",
"keng1",
"qiao4",
"qin1",
"shua4",
"an1",
"yu3",
"xiao4",
"cheng2",
"jie4",
"xian4",
"wu1",
"wu4",
"gao4",
"song4",
"pu3",
"hui4",
"jing4",
"shuo1",
"zhen4",
"shuo1",
"du2",
"hua1",
"chang4",
"shui2",
"jie2",
"ke4",
"qu1",
"cong2",
"xiao2",
"sui4",
"wang3",
"xian2",
"fei3",
"chi1",
"ta4",
"yi4",
"ni4",
"yin2",
"diao4",
"pi3",
"zhuo2",
"chan3",
"chen1",
"zhun1",
"ji4",
"qi1",
"tan2",
"zhui4",
"wei3",
"ju1",
"qing3",
"jian4",
"zheng4",
"ze2",
"zou1",
"qian1",
"zhuo2",
"liang4",
"jian4",
"chu4",
"hao2",
"lun4",
"shen3",
"biao3",
"hua4",
"pian2",
"yu2",
"die2",
"xu1",
"pian3",
"shi4",
"xuan1",
"shi4",
"hun4",
"hua4",
"e4",
"zhong4",
"di4",
"xie2",
"fu2",
"pu3",
"ting2",
"jian4",
"qi3",
"yu4",
"zi1",
"zhuan1",
"xi3",
"hui4",
"yin1",
"an1",
"xian2",
"nan2",
"chen2",
"feng3",
"zhu1",
"yang2",
"yan4",
"huang2",
"xuan1",
"ge2",
"nuo4",
"qi1",
"mou2",
"ye4",
"wei4",
"xing1",
"teng2",
"zhou1",
"shan4",
"jian3",
"po2",
"kui4",
"huang3",
"huo4",
"ge1",
"ying2",
"mi2",
"xiao3",
"mi4",
"xi4",
"qiang1",
"chen1",
"xue4",
"ti2",
"su4",
"bang4",
"chi2",
"qian1",
"shi4",
"jiang3",
"yuan2",
"xie4",
"he4",
"tao1",
"yao2",
"yao2",
"lu1",
"yu2",
"biao1",
"cong4",
"qi4",
"li2",
"mo2",
"mo2",
"shang1",
"zhe2",
"miu4",
"jian3",
"ze2",
"jie1",
"lian2",
"lou2",
"can4",
"ou1",
"gun4",
"xi2",
"zhuo2",
"ao2",
"ao2",
"jin3",
"zhe2",
"yi2",
"hu1",
"jiang4",
"man2",
"chao2",
"han4",
"hua2",
"chan3",
"xu1",
"zeng1",
"se4",
"xi1",
"zha1",
"dui4",
"zheng4",
"nao2",
"lan2",
"e2",
"ying1",
"jue2",
"ji1",
"zun3",
"jiao3",
"bo4",
"hui4",
"zhuan4",
"wu2",
"zen4",
"zha2",
"shi2",
"qiao4",
"tan2",
"zen4",
"pu3",
"sheng2",
"xuan1",
"zao4",
"tan2",
"dang3",
"sui4",
"xian3",
"ji1",
"jiao4",
"jing3",
"zhan4",
"nang2",
"yi1",
"ai3",
"zhan1",
"pi4",
"hui3",
"hua4",
"yi4",
"yi4",
"shan4",
"rang4",
"nou4",
"qian3",
"dui4",
"ta4",
"hu4",
"zhou1",
"hao2",
"ai4",
"ying1",
"jian4",
"yu4",
"jian3",
"hui4",
"du2",
"zhe2",
"xuan4",
"zan4",
"lei3",
"shen3",
"wei4",
"chan3",
"li4",
"yi2",
"bian4",
"zhe2",
"yan4",
"e4",
"chou2",
"wei4",
"chou2",
"yao4",
"chan2",
"rang4",
"yin3",
"lan2",
"chen4",
"xie2",
"nie4",
"huan1",
"zan4",
"yi4",
"dang3",
"zhan2",
"yan4",
"du2",
"yan2",
"ji4",
"ding4",
"fu4",
"ren4",
"ji1",
"jie2",
"hong4",
"tao3",
"rang4",
"shan4",
"qi4",
"tuo1",
"xun4",
"yi4",
"xun4",
"ji4",
"ren4",
"jiang3",
"hui4",
"ou1",
"ju4",
"ya4",
"ne4",
"xu3",
"e2",
"lun4,lun2",
"xiong1",
"song4",
"feng3",
"she4",
"fang3",
"jue2",
"zheng4",
"gu3",
"he1",
"ping2",
"zu3",
"shi2,zhi4",
"xiong4",
"zha4",
"su4",
"zhen3",
"di3",
"zhou1",
"ci2",
"qu1",
"zhao4",
"bi4",
"yi4",
"yi2",
"kuang1",
"lei3",
"shi4",
"gua4",
"shi1",
"jie2",
"hui1",
"cheng2",
"zhu1",
"shen1",
"hua4",
"dan4",
"gou4",
"quan2",
"gui3",
"xun2",
"yi4",
"zheng4",
"gai1",
"xiang2",
"cha4",
"hun4",
"xu3",
"zhou1",
"jie4",
"wu1",
"yu3,yu4",
"qiao4",
"wu4",
"gao4",
"you4",
"hui4",
"kuang2",
"shuo1,shui4",
"song4",
"ei2,ei1,ei3,ei4,xi1",
"qing3",
"zhu1",
"zou1",
"nuo4",
"du2,dou4",
"zhuo2",
"fei3",
"ke4",
"wei3",
"yu2",
"shei2",
"shen3",
"diao4,tiao2",
"chan3",
"liang4",
"zhun1",
"sui4",
"tan2",
"shen3",
"yi4",
"mou2",
"chen2",
"die2",
"huang3",
"jian4",
"xie2",
"xue4",
"ye4",
"wei4",
"e4",
"yu4",
"xuan1",
"chan2",
"zi1",
"an1",
"yan4",
"di4",
"mi2,mei4",
"pian3",
"xu1",
"mo2",
"dang3",
"su4",
"xie4",
"yao2",
"bang4",
"shi4,yi4",
"qian1",
"mi4",
"jin3",
"man2,man4",
"zhe2",
"jian3",
"miu4",
"tan2",
"zen4",
"qiao2,qiao4",
"lan2",
"pu3",
"jue2",
"yan4",
"qian3",
"zhan1",
"chen4",
"gu3",
"qian1",
"hong2",
"xia1",
"ji2",
"hong2",
"han1",
"hong1",
"xi1",
"xi1",
"huo1,hua2,huo4",
"liao2",
"han3",
"du2",
"long2",
"dou4",
"jiang1",
"qi3",
"chi3",
"li3,feng1",
"deng1",
"wan1",
"bi1",
"shu4",
"xian4",
"feng1",
"zhi4",
"zhi4",
"yan4",
"yan4",
"shi3",
"chu4",
"hui1",
"tun2",
"yi4",
"tun2",
"yi4",
"jian1",
"ba1",
"hou4",
"e4",
"chu2",
"xiang4",
"huan4",
"jian1",
"ken3",
"gai1",
"qu2",
"fu1",
"xi1",
"bin1",
"hao2",
"yu4",
"zhu1",
"jia1",
"fen2",
"xi1",
"bo2",
"wen1",
"huan2",
"bin1",
"di2",
"zong1",
"fen2",
"yi4",
"zhi4",
"bao4",
"chai2",
"an4,han4",
"pi2",
"na4",
"pi1",
"gou3",
"na4",
"you4",
"diao1",
"mo4",
"si4",
"xiu1",
"huan2",
"kun1",
"he2",
"he2,mo4",
"mo4",
"an4",
"mao4",
"li2",
"ni2",
"bi3",
"yu3",
"jia1",
"tuan1",
"mao1",
"pi2",
"xi1",
"yi4",
"ju4",
"mo4",
"chu1",
"tan2",
"huan1",
"jue2",
"bei4",
"zhen1",
"yuan2",
"fu4",
"cai2",
"gong4",
"te4",
"yi2",
"hang2",
"wan2",
"pin2",
"huo4",
"fan4",
"tan1",
"guan4",
"ze2",
"zhi4",
"er4",
"zhu4",
"shi4",
"bi4",
"zi1",
"er4",
"gui4",
"pian3",
"bian3",
"mai3",
"dai4",
"sheng4",
"kuang4",
"fei4",
"tie1",
"yi2",
"chi2",
"mao4",
"he4",
"bi4",
"lu4",
"lin4",
"hui4",
"gai1",
"pian2",
"zi1",
"jia3",
"xu4",
"zei2",
"jiao3",
"gai1",
"zang1",
"jian4",
"ying4",
"xun4",
"zhen4",
"she1",
"bin1",
"bin1",
"qiu2",
"she1",
"chuan4",
"zang1",
"zhou1",
"lai4",
"zan4",
"ci4",
"chen1",
"shang3",
"tian3",
"pei2",
"geng1",
"xian2",
"mai4",
"jian4",
"sui4",
"fu4",
"tan4",
"cong2",
"cong2",
"zhi4",
"ji1",
"zhang4",
"du3",
"jin4",
"xiong1",
"chun3",
"yun3",
"bao3",
"zai1",
"lai4",
"feng4",
"cang4",
"ji1",
"sheng4",
"yi4",
"zhuan4",
"fu4",
"gou4",
"sai4",
"ze2",
"liao2",
"yi4",
"bai4",
"chen3",
"wan4",
"zhi4",
"zhui4",
"biao1",
"yun1",
"zeng4",
"tan3",
"zan4",
"yan4",
"pu2",
"shan4",
"wan4",
"ying2",
"jin4",
"gan4",
"xian2",
"zang1",
"bi4",
"du2",
"shu2",
"yan4",
"shang3",
"xuan4",
"long4",
"gan4",
"zang1",
"bei4",
"zhen1",
"fu4",
"yuan2",
"gong4",
"cai2",
"ze2",
"xian2",
"bai4",
"zhang4",
"huo4",
"zhi4",
"fan4",
"tan1",
"pin2",
"bian3",
"gou4",
"zhu4",
"guan4",
"er4",
"jian4",
"ben1,bi4",
"shi4",
"tie1",
"gui4",
"kuang4",
"dai4",
"mao4",
"fei4",
"he4",
"yi2",
"zei2",
"zhi4",
"jia3,gu3",
"hui4",
"zi1",
"lin4",
"lu4",
"zang1",
"zi1",
"gai1",
"jin4",
"qiu2",
"zhen4",
"lai4",
"she1",
"fu4",
"du3",
"ji1",
"shu2",
"shang3",
"ci4",
"bi4",
"zhou1",
"geng1",
"pei2",
"dan3",
"lai4",
"feng4",
"zhui4",
"fu4",
"zhuan4,zuan4",
"sai4",
"ze2",
"yan4",
"zan4",
"yun1",
"zeng4",
"shan4",
"ying2",
"gan4",
"chi4",
"xi1",
"she4",
"nan3",
"tong2",
"xi4",
"cheng1",
"he4",
"cheng1",
"zhe3",
"xia2",
"tang2",
"zou3",
"zou3",
"li4",
"jiu1",
"fu4",
"zhao4",
"gan3",
"qi3",
"shan4",
"qiong2",
"yin3",
"xian3",
"zi1",
"jue2",
"qin3",
"chi2",
"ci1",
"chen4",
"chen4",
"die2",
"ju1,qie4",
"chao1",
"di1",
"xi4",
"zhan1",
"jue2",
"yue4",
"qu1",
"ji2",
"chi2",
"chu2",
"gua1",
"xue4",
"zi1",
"tiao2",
"duo3",
"lie4",
"gan3",
"suo1",
"cu4",
"xi2",
"zhao4",
"su4",
"yin3",
"ju2",
"jian4",
"que4",
"tang4,tang1",
"chuo4",
"cui3",
"lu4",
"qu4",
"dang4",
"qiu1",
"zi1",
"ti2",
"qu1",
"chi4",
"huang2",
"qiao2",
"qiao1",
"jiao4",
"zao4",
"ti4,yue4",
"er3",
"zan3",
"zan3",
"zu2,ju4",
"pa1",
"bao4",
"ku4",
"ke1",
"dun3",
"jue2",
"fu1",
"chen3",
"jian3",
"fang4",
"zhi3",
"ta1",
"yue4",
"pa2",
"qi2,ji1,qi3,qi4,zhi1",
"yue4",
"qiang1,qiang4",
"tuo4",
"tai2",
"yi4",
"nian3",
"ling2",
"mei4",
"ba2",
"die1",
"ku1",
"tuo2",
"jia1",
"ci3",
"pao3,pao2",
"qia3",
"zhu4",
"ju1",
"die2",
"zhi2",
"fu1",
"pan2",
"ju3",
"shan1",
"bo3",
"ni2",
"ju4",
"li4",
"gen1",
"yi2",
"ji1",
"duo4",
"xian3",
"jiao1",
"duo4",
"zhu1",
"quan2",
"kua4",
"zhuai3",
"gui4",
"qiong2",
"kui3",
"xiang2",
"chi4",
"lu4",
"pian2",
"zhi4",
"jia2",
"tiao4",
"cai3",
"jian4",
"da1",
"qiao1",
"bi4",
"xian1",
"duo4",
"ji1",
"ju2",
"ji4",
"shu1",
"tu2",
"chu4",
"jing4",
"nie4",
"xiao1",
"bu4",
"xue2,chi4",
"qun1",
"mu3",
"shu1",
"liang2,liang4",
"yong3",
"jiao3",
"chou2",
"qiao1",
"mou2",
"ta4,ta1",
"jian4",
"qi2",
"wo1",
"wei3",
"chuo1",
"jie2",
"ji2",
"nie4",
"ju1",
"nie4",
"lun2",
"lu4",
"leng4",
"huai2",
"ju4",
"chi2",
"wan3",
"quan2",
"ti1",
"bo2",
"cu4",
"qie4",
"ji3",
"cu4",
"zong1",
"cai3",
"zong1",
"peng4",
"zhi4",
"zheng1",
"dian3",
"zhi2",
"yu2",
"duo2",
"dun4",
"chuan3",
"yong3",
"zhong3",
"di4",
"zha3",
"chen3",
"chuai4",
"jian4",
"gua1",
"tang2",
"ju3",
"fu2",
"zu2",
"die2",
"pian2",
"rou2",
"nuo4",
"ti2",
"cha3",
"tui3",
"jian3",
"dao3",
"cuo1",
"xi1",
"ta4",
"qiang1",
"nian3",
"dian1",
"ti2",
"ji2",
"nie4",
"pan2",
"liu4",
"zan4",
"bi4",
"chong1",
"lu4",
"liao2",
"cu4",
"tang1",
"dai4",
"suo1",
"xi3",
"kui3",
"ji1",
"zhi2",
"qiang1,qiang4",
"di2",
"pan2",
"zong1",
"lian2",
"beng4",
"zao1",
"nian3",
"bie2",
"tui2",
"ju2",
"deng1",
"ceng4",
"xian1",
"fan2",
"chu2",
"zhong1",
"dun1",
"bo1",
"cu4",
"cu4",
"jue2,jue3",
"jue2",
"lin4",
"ta2",
"qiao1",
"jue1",
"pu3",
"liao1",
"dun1",
"cuan1",
"guan4",
"zao4",
"da2",
"bi4",
"bi4",
"zhu2",
"ju4",
"chu2",
"qiao4",
"dun3",
"chou2",
"ji1",
"wu3",
"yue4",
"nian3",
"lin4",
"lie4",
"zhi2",
"li4",
"zhi4",
"chan2",
"chu2",
"duan4",
"wei4",
"long2",
"lin4",
"xian1",
"wei4",
"zuan1",
"lan2",
"xie4",
"rang2",
"xie4",
"nie4",
"ta4",
"qu2",
"ji2",
"cuan1",
"cuo2",
"xi3",
"kui2",
"jue2",
"lin4",
"shen1",
"gong1",
"dan1",
"fen1",
"qu1",
"ti3",
"duo3",
"duo3",
"gong1",
"lang2",
"ren3",
"luo3",
"ai3",
"ji1",
"ju1",
"tang3",
"kong1",
"lao4",
"yan3",
"mei3",
"kang1",
"qu1",
"lou2",
"lao4",
"duo3",
"zhi2",
"yan4",
"ti3",
"dao4",
"ying1",
"yu4",
"che1",
"ya4",
"gui3",
"jun1",
"wei4",
"yue4",
"xin4",
"dai4",
"xuan1",
"fan4",
"ren4",
"shan1",
"kuang2",
"shu1",
"tun2",
"chen2",
"dai4",
"e4",
"na4",
"qi2",
"mao2",
"ruan3",
"kuang2",
"qian2",
"zhuan3",
"hong1",
"hu1",
"qu2",
"kuang4",
"di3",
"ling2",
"dai4",
"ao1",
"zhen3",
"fan4",
"kuang1",
"yang3",
"peng1",
"bei4",
"gu1",
"gu1",
"pao2",
"zhu4",
"rong3",
"e4",
"ba2",
"zhou2",
"zhi3",
"yao2",
"ke1",
"yi4",
"qing1",
"shi4",
"ping2",
"er2",
"gong3",
"ju2",
"jiao4",
"guang1",
"he2",
"kai3",
"quan2",
"zhou1",
"zai4",
"zhi4",
"she1",
"liang4",
"yu4",
"shao1",
"you2",
"wan4",
"yin3",
"zhe2",
"wan3",
"fu3",
"qing1",
"zhou1",
"ni2",
"leng2",
"zhe2",
"zhan4",
"liang4",
"zi1",
"hui1",
"wang3",
"chuo4",
"guo3",
"kan3",
"yi3",
"peng2",
"qian4",
"gun3",
"nian3",
"ping2",
"guan3",
"bei4",
"lun2",
"pai2",
"liang2",
"ruan3",
"rou2",
"ji2",
"yang2",
"xian2",
"chuan2",
"cou4",
"chun1",
"ge2",
"you2",
"hong1",
"shu1",
"fu4",
"zi1",
"fu2",
"wen1",
"ben4",
"zhan3",
"yu2",
"wen1",
"tao1",
"gu3",
"zhen1",
"xia2",
"yuan2",
"lu4",
"jiu1",
"chao2",
"zhuan3",
"wei4",
"hun2",
"xue3",
"zhe2",
"jiao4",
"zhan4",
"bu2",
"lao3",
"fen2",
"fan1",
"lin2",
"ge2",
"se4",
"kan3",
"huan4",
"yi3",
"ji2",
"zhui4",
"er2",
"yu2",
"xian4",
"hong1",
"lei2",
"pei4",
"li4",
"li4",
"lu2",
"lin4",
"che1,ju1",
"ya4,ga2,zha2",
"gui3",
"xuan1",
"dai4",
"ren4",
"zhuan3,zhuai3,zhuan4",
"e4",
"lun2",
"ruan3",
"hong1",
"gu1",
"ke1,ke3",
"lu2",
"zhou2,zhou4",
"zhi3",
"yi4",
"hu1",
"zhen3",
"li4",
"yao2",
"qing1",
"shi4",
"zai4,zai3",
"zhi4",
"jiao4",
"zhou1",
"quan2",
"lu4",
"jiao4",
"zhe2",
"fu3",
"liang4",
"nian3",
"bei4",
"hui1",
"gun3",
"wang3",
"liang2",
"chuo4",
"zi1",
"cou4",
"fu2",
"ji2",
"wen1",
"shu1",
"pei4",
"yuan2",
"xia2",
"zhan3",
"lu4",
"zhe2,che4",
"lin2",
"xin1",
"gu1",
"ci2",
"ci2",
"pi4,bi4",
"zui4",
"bian4",
"la4",
"la4",
"ci2",
"xue1",
"ban4",
"bian4",
"bian4",
"bian4",
"xue1",
"bian4",
"ban1",
"ci2",
"bian4",
"bian4",
"chen2",
"ru3",
"nong2",
"nong2",
"chan3",
"chuo4",
"chuo4",
"yi1",
"reng2",
"bian1,bian5",
"bian1",
"shi2",
"xx5",
"liao2",
"da2",
"chan1",
"gan1",
"qian1",
"yu1",
"yu1",
"qi4",
"xun4",
"yi2,yi3",
"guo4,guo5",
"mai4",
"qi1",
"za1",
"wang4",
"tu4",
"zhun1",
"ying2",
"da2",
"yun4",
"jin4",
"hang2",
"ya4",
"fan3",
"wu3",
"da2",
"e2",
"hai2,huan2",
"zhe4",
"da2",
"jin4",
"yuan3,yuan4",
"wei2",
"lian2",
"chi2",
"che4",
"ni4",
"tiao2",
"zhi4",
"yi2,yi3",
"jiong3",
"jia1",
"chen2",
"dai4",
"er3",
"di2",
"po4",
"zhu4",
"die2",
"ze2",
"tao2",
"shu4",
"tuo2",
"qu5",
"jing4",
"hui2",
"dong4",
"you4",
"mi2",
"beng4",
"ji4",
"nai3",
"yi2",
"jie2",
"zhui1,dui1",
"lie4",
"xun4",
"tui4",
"song4",
"shi4,kuo4",
"tao2",
"pang2",
"hou4",
"ni4",
"dun4",
"jiong3",
"xuan3",
"xun4",
"bu1",
"you2",
"xiao1",
"qiu2",
"tou4",
"zhu2",
"qiu2",
"di4",
"di4",
"tu2",
"jing4",
"ti4",
"dou4",
"yi3",
"zhe4",
"tong1,tong4",
"guang4",
"wu4",
"shi4",
"cheng3",
"su4",
"zao4",
"qun1",
"feng2",
"lian2",
"suo4",
"hui2",
"li3",
"gu3",
"lai2",
"ben4",
"cuo4",
"jue2",
"beng4",
"huan4",
"dai3,dai4",
"lu4",
"you2",
"zhou1",
"jin4",
"yu4",
"chuo4",
"kui2",
"wei1",
"ti4",
"yi4",
"da2",
"yuan3",
"luo2",
"bi1",
"nuo4",
"yu2",
"dang4",
"sui2",
"dun4",
"sui4",
"yan3",
"chuan2",
"chi2",
"ti2",
"yu4",
"shi2",
"zhen1",
"you2",
"yun4",
"e4",
"bian4",
"guo4",
"e4",
"xia2",
"huang2",
"qiu2",
"dao4",
"da2",
"wei2",
"nan2",
"yi2",
"gou4",
"yao2",
"chou4",
"liu4",
"xun4",
"ta4",
"di4",
"chi2",
"yuan3",
"su4",
"ta4",
"qian3",
"ma3",
"yao2",
"guan4",
"zhang1",
"ao2",
"shi4",
"ca4",
"su4",
"su4",
"zao1",
"zhe1",
"dun4",
"di4",
"lou2",
"chi2",
"cuo1",
"lin2",
"zun1",
"rao4",
"qian1",
"xuan3",
"yu4",
"yi2",
"wu4",
"liao2",
"ju4",
"shi4",
"bi4",
"yao1",
"mai4",
"xie4",
"sui4",
"hai2",
"zhan1",
"teng2",
"er3",
"miao3",
"bian1",
"bian1",
"la1",
"li2",
"yuan2",
"yao2",
"luo2",
"li3",
"yi4",
"ting2",
"deng4",
"qi3",
"yong1",
"shan1",
"han2",
"yu2",
"mang2",
"ru2",
"qiong2",
"xi1",
"kuang4",
"fu1",
"kang4",
"bin1",
"fang1",
"xing2",
"na4,na1,nuo2,na3",
"xin1",
"shen3",
"bang1",
"yuan2",
"cun1",
"huo3",
"xie2",
"bang1",
"wu1",
"ju4",
"you2",
"han2",
"tai2",
"qiu1",
"bi4",
"pi1",
"bing3",
"shao4",
"bei4",
"wa3",
"di3",
"zou1",
"ye4",
"lin2",
"kuang1",
"gui1",
"zhu1",
"shi1",
"ku1",
"yu4",
"gai1",
"he2",
"qie4,xi4",
"zhi4",
"ji2",
"xun2",
"hou4",
"xing2",
"jiao1",
"xi2",
"gui1",
"nuo2",
"lang2",
"jia2",
"kuai4",
"zheng4",
"lang2",
"yun4",
"yan2",
"cheng2",
"dou4",
"chi1",
"lv3",
"fu3",
"wu2",
"fu2",
"gao4",
"hao3",
"lang2",
"jia2",
"geng3",
"jun4",
"ying3",
"bo2",
"xi4",
"bei4",
"li4",
"yun2",
"bu4",
"xiao2",
"qi1",
"pi2",
"qing1",
"guo1",
"zhou1",
"tan2",
"zou1",
"ping2",
"lai2",
"ni2",
"chen1",
"you2",
"bu4",
"xiang1",
"dan1",
"ju2",
"yong1",
"qiao1",
"yi1",
"dou1,du1",
"yan3",
"mei2",
"ruo4",
"bei4",
"e4",
"shu1",
"juan4",
"yu3",
"yun4",
"hou4",
"kui2",
"xiang1",
"xiang1",
"sou1",
"tang2",
"ming2",
"xi4",
"ru4",
"chu4",
"zi1",
"zou1",
"ye4",
"wu1",
"xiang1",
"yun2",
"hao4",
"yong1",
"bi3",
"mo4",
"chao2",
"fu1",
"liao3",
"yin2",
"zhuan1",
"hu4",
"qiao1",
"yan1",
"zhang1",
"man4",
"qiao1",
"xu3",
"deng4",
"bi4",
"xun2",
"bi4",
"zeng2",
"wei2",
"zheng4",
"mao4",
"shan4",
"lin2",
"po2",
"dan1",
"meng2",
"ye4",
"cao4",
"kuai4",
"feng1",
"meng2",
"zou1",
"kuang4",
"lian3",
"zan4",
"chan2",
"you1",
"qi2",
"yan4",
"chan2",
"zan4",
"ling2",
"huan1",
"xi1",
"feng1",
"zan4",
"li4",
"you3",
"ding1,ding3",
"qiu2",
"zhuo2",
"pei4",
"zhou4",
"yi2",
"gan1",
"yu2",
"jiu3",
"yan3",
"zui4",
"mao2",
"zhen4,dan1",
"xu4",
"dou4",
"zhen1",
"fen1",
"yuan2",
"fu5",
"yun4",
"tai4",
"tian1",
"qia3",
"tuo2",
"cu4,zuo4",
"han1",
"gu1",
"su1",
"fa1,po1",
"chou2",
"zai4",
"ming3",
"lao4",
"chuo4",
"chou2",
"you4",
"tong2",
"zhi3",
"xian1",
"jiang4",
"cheng2",
"yin4",
"tu2",
"jiao4",
"mei2",
"ku4",
"suan1",
"lei4",
"pu2",
"zui4",
"hai3",
"yan4",
"shi1",
"niang4",
"wei2",
"lu4",
"lan3",
"yan1",
"tao2",
"pei1",
"zhan3",
"chun2",
"tan2",
"zui4",
"chuo4",
"cu4",
"kun1",
"ti2",
"xian2",
"du1",
"hu2",
"xu3",
"xing3",
"tan3",
"qiu2",
"chun2",
"yun4",
"po4",
"ke1",
"sou1",
"mi2",
"quan2",
"chou3",
"cuo2",
"yun4",
"yong4",
"ang4",
"zha4",
"hai3",
"tang2",
"jiang4",
"piao3",
"chen3",
"yu4",
"li2",
"zao1",
"lao2",
"yi1",
"jiang4",
"bu2",
"jiao4",
"xi1",
"tan2",
"fa1",
"nong2",
"yi4",
"li3",
"ju4",
"yan4",
"yi4",
"niang4",
"ru2",
"xun1",
"chou2",
"yan4",
"ling2",
"mi2",
"mi2",
"niang4",
"xin4",
"jiao4",
"shai1",
"mi2",
"yan4",
"bian4",
"cai3,cai4",
"shi4",
"you4",
"shi4",
"shi4",
"li3",
"zhong4,chong2",
"ye3",
"liang4,liang2",
"li2",
"jin1",
"jin1",
"qiu2",
"yi3",
"liao3",
"dao1",
"zhao1",
"ding1",
"po4",
"qiu2",
"ba1",
"fu3",
"zhen1",
"zhi2",
"ba1",
"luan4",
"fu3",
"nai3",
"diao4",
"shan4",
"qiao3",
"kou4",
"chuan4",
"zi3",
"fan3",
"yu2",
"hua2",
"han4",
"gang1",
"qi2",
"mang2",
"ri4",
"di4",
"si4",
"xi4",
"yi4",
"chai1",
"shi1",
"tu3",
"xi1",
"nv3",
"qian1",
"qiu2",
"jian4",
"pi4",
"ye2",
"jin1",
"ba3",
"fang1",
"chen2",
"xing2",
"dou3",
"yue4",
"yan2",
"fu1",
"pi1",
"na4",
"xin1",
"e2",
"jue2",
"dun4",
"gou1",
"yin3",
"qian2",
"ban3",
"sa4",
"ren2",
"chao1",
"niu3",
"fen1",
"yun3",
"yi3",
"qin2",
"pi1",
"guo1",
"hong2",
"yin2",
"jun1",
"diao4",
"yi4",
"zhong1",
"xi3",
"gai4",
"ri4",
"huo3",
"tai4",
"kang4",
"yuan2",
"lu2",
"e4",
"qin2",
"duo2",
"zi1",
"ni3",
"tu2",
"shi4",
"min2",
"gu1",
"ke1",
"ling2",
"bing3",
"si4",
"gu3",
"bo2",
"pi1",
"yu4",
"si4",
"zuo2",
"bu1",
"you2",
"tian2",
"jia3",
"zhen1",
"shi3",
"shi4",
"tie3,zhi4",
"ju4",
"zhen1",
"shi1",
"shi1",
"xuan4",
"zhao1",
"bao4",
"he2",
"bi4",
"sheng1",
"chu2",
"shi2",
"bo2",
"zhu4",
"chi4",
"za1",
"po1",
"tong2",
"qian2",
"fu2",
"zhai3",
"liu3",
"qian1",
"fu2",
"li4",
"yue4",
"pi1",
"yang1",
"ban4",
"bo1",
"jie2",
"gou1",
"shu4",
"zheng1",
"mu3",
"xi3,ni3",
"xi3",
"di4",
"jia1",
"mu4",
"tan3",
"huan2",
"yi3",
"si1",
"kuang4",
"ka3",
"bei3",
"jian4",
"tong2",
"xing2",
"hong2",
"jiao3",
"chi3",
"er4",
"luo4",
"bing3",
"shi4",
"mou2",
"jia1",
"yin2",
"jun1",
"zhou1",
"chong4",
"xiang3",
"tong2",
"mo4",
"lei4",
"ji1",
"yu4",
"xu4",
"ren2",
"zun4",
"zhi4",
"qiong1",
"shan4",
"chi4",
"xian3",
"xing2",
"quan2",
"pi1",
"tie3",
"zhu1",
"xiang4",
"ming2",
"kua3",
"yao2",
"xian1",
"xian2",
"xiu1",
"jun1",
"cha1",
"lao3",
"ji2",
"pi3",
"ru2",
"mi3",
"yi1",
"yin1",
"guang1",
"an3",
"diu1",
"you3",
"se4",
"kao4",
"qian2",
"luan2",
"si1",
"ai1",
"diao4",
"han4",
"rui4",
"shi4",
"keng1",
"qiu2",
"xiao1",
"zhe2",
"xiu4",
"zang4",
"ti2",
"cuo4",
"gua1",
"hong4",
"zhong1",
"tou1",
"lv3",
"mei2",
"lang2",
"wan4",
"xin1",
"yun2",
"bei4",
"wu4",
"su4",
"yu4",
"chan2",
"ding4",
"bo2",
"han4",
"jia2",
"hong2",
"cuan1",
"feng1",
"chan1",
"wan3",
"zhi4",
"si1",
"xuan1",
"wu2",
"yu3",
"tiao2",
"kuang4",
"zhuo2",
"lve4",
"xing2",
"qin3",
"shen4",
"han2",
"lve4",
"ye2",
"chu2",
"zeng4",
"ju1",
"xian4",
"tie3",
"mang2",
"pu4",
"li2",
"pan4",
"rui4",
"cheng2",
"gao4",
"li3",
"te4",
"bing1",
"zhu4",
"zhen4",
"tu1",
"liu3",
"zui4",
"ju4",
"chang3",
"wan3",
"jian4",
"gang1",
"diao4",
"tao2",
"chang2",
"lun2",
"guo3",
"ling2",
"pi1",
"lu4",
"li2",
"qiang1",
"pou2",
"juan3",
"min2",
"zui4",
"peng2",
"an4",
"pi1",
"xian4",
"ya1",
"zhui1",
"lei4",
"ke1",
"kong1",
"ta4",
"kun1",
"du2",
"nei4",
"chui2",
"zi1",
"zheng1",
"ben1",
"nie4",
"zong4",
"chun2",
"tan2",
"ding4",
"qi2",
"qian2",
"zhui4",
"ji1",
"yu4",
"jin3",
"guan3",
"mao2",
"chang1",
"tian3",
"xi1",
"lian4",
"tao2",
"gu4",
"cuo4",
"shu4",
"zhen1",
"lu4",
"meng3",
"lu4",
"hua1",
"biao3",
"ga2",
"lai2",
"ken3",
"fang1",
"wu5",
"nai4",
"wan4",
"zan4",
"hu3",
"de2",
"xian1",
"pian1",
"huo1",
"liang4",
"fa3",
"men2",
"kai3",
"ying1",
"di1",
"lian4",
"guo1",
"xian3",
"du4",
"tu2",
"wei2",
"cong1",
"fu4",
"rou2",
"ji2",
"e4",
"jun1",
"chen3",
"ti2",
"zha2",
"hu4",
"yang2",
"duan4",
"xia2",
"yu2",
"keng1",
"sheng1",
"huang2",
"wei3",
"fu4",
"zhao1",
"cha1",
"qie4",
"shi1",
"hong1",
"kui2",
"tian3",
"mou2",
"qiao1",
"qiao1",
"hou2",
"tou1",
"cong1",
"huan2",
"ye4",
"min2",
"jian4",
"duan1",
"jian4",
"song1",
"kui2",
"hu2",
"xuan1",
"duo3",
"jie2",
"zhen1",
"bian1",
"zhong1",
"zi1",
"xiu1",
"ye2",
"mei3",
"pai4",
"ai1",
"jie4",
"qian5",
"mei2",
"suo3",
"ta4",
"bang4",
"xia2",
"lian2",
"suo3",
"kai4",
"liu2",
"yao2",
"ye4",
"nou4",
"weng1",
"rong2",
"tang2",
"suo3",
"qiang1",
"li4",
"shuo4",
"chui2",
"bo2",
"pan2",
"da1",
"bi4",
"sang3",
"gang1",
"zi1",
"wu1",
"ying2",
"huang3",
"tiao2",
"liu2",
"kai3",
"sun3",
"sha1",
"sou1",
"wan4",
"hao4",
"zhen4",
"zhen4",
"lang2",
"yi4",
"yuan2",
"tang3",
"nie4",
"xi2",
"jia1",
"ge1",
"ma3",
"juan1",
"xx5",
"zu3",
"suo3",
"xia4",
"feng1",
"wen1",
"na2",
"lu3",
"suo3",
"ou1",
"zu2",
"tuan2",
"xiu1",
"guan4",
"xuan4",
"lian4",
"shou4",
"ao4",
"man3",
"mo4",
"luo2",
"bi4",
"wei4",
"liu2",
"di2",
"san3",
"zong3",
"yi2",
"lu4",
"ao2",
"keng1",
"qiang1",
"cui1",
"qi1",
"chang2",
"tang1",
"man4",
"yong1",
"chan3",
"feng1",
"jing4",
"biao1",
"shu4",
"lou4",
"xiu4",
"cong1",
"long2",
"zan4",
"jian4",
"cao2",
"li2",
"xia4",
"xi1",
"kang1",
"shuang3",
"beng4",
"zhang5",
"qian5",
"cheng1",
"lu4",
"hua2",
"ji2",
"pu2",
"hui4",
"qiang3",
"po1",
"lin2",
"se4",
"xiu4",
"san3",
"cheng1",
"kui4",
"si1",
"liu2",
"nao2",
"huang2",
"pie3",
"sui4",
"fan2",
"qiao2",
"quan1",
"yang2",
"tang1",
"xiang4",
"jue2",
"jiao1",
"zun1",
"liao2",
"qie4",
"lao2",
"dui4",
"xin2",
"zan1",
"ji1",
"jian3",
"zhong1",
"deng4",
"ya1",
"ying3",
"dui1",
"jue2",
"nou4",
"zan1",
"pu3",
"tie3",
"fan2",
"cheng1",
"ding3",
"shan4",
"kai1",
"jian1",
"fei4",
"sui4",
"lu3",
"juan1",
"hui4",
"yu4",
"lian2",
"zhuo2",
"qiao1",
"jian4",
"zhuo2",
"lei2",
"bi4",
"tie3",
"huan2",
"ye4",
"duo2",
"guo3",
"dang1",
"ju4",
"fen2",
"da2",
"bei4",
"yi4",
"ai4",
"zong1",
"xun4",
"diao4",
"zhu4",
"heng2",
"zhui4",
"ji1",
"nie4",
"he2",
"huo4",
"qing1",
"bin1",
"ying1",
"kui4",
"ning2",
"xu1",
"jian4",
"jian4",
"qian3",
"cha3",
"zhi4",
"mie4",
"li2",
"lei2",
"ji1",
"zuan4",
"kuang4",
"shang3",
"peng2",
"la4",
"du2",
"shuo4",
"chuo4",
"lu:4",
"biao1",
"bao4",
"lu3",
"xian5",
"kuan1",
"long2",
"e4",
"lu2",
"xin1",
"jian4",
"lan4",
"bo2",
"jian1",
"yao4",
"chan2",
"xiang1",
"jian4",
"xi4",
"guan4",
"cang2",
"nie4",
"lei3",
"cuan1",
"qu2",
"pan4",
"luo2",
"zuan1",
"luan2",
"zao2",
"nie4",
"jue2",
"tang3",
"shu3",
"lan2",
"jin1",
"ga2",
"yi3",
"zhen1",
"ding1,ding4",
"zhao1",
"po1",
"liao3,liao4",
"tu3",
"qian1",
"chuan4",
"shan1",
"ji2",
"fan2",
"diao4",
"men2",
"nu:3",
"yang2",
"chai1",
"xing2",
"gai4",
"bu4",
"tai4",
"ju4",
"dun4",
"chao1",
"zhong1",
"na4",
"bei4",
"gang1",
"ban3",
"qian2",
"yue4",
"qin1",
"jun1",
"wu1",
"gou1",
"kang4",
"fang1",
"huo3",
"tou3",
"niu3",
"ba3,pa2",
"yu4",
"qian2",
"zheng1",
"qian2",
"gu3",
"bo1",
"ke1",
"po3",
"bu1,bu4",
"bo2",
"yue4",
"zuan1,zuan4",
"mu4",
"tan3",
"jia3",
"dian4,tian2",
"you2",
"tie3",
"bo2",
"ling2",
"shuo4",
"qian1",
"mao3",
"bao4",
"shi4",
"xuan4",
"ta1",
"bi4",
"ni2",
"pi2",
"duo2",
"xing2",
"kao4",
"lao3",
"er3",
"mang2",
"ya4",
"you3",
"cheng2",
"jia2",
"ye2",
"nao2",
"zhi4",
"dang1,cheng1",
"tong2",
"lu:3",
"diao4",
"yin1",
"kai3",
"zha2",
"zhu1",
"xi3,xian3",
"ding4,ting3",
"diu1",
"xian1",
"hua2",
"quan2",
"sha1",
"ha1",
"diao4,yao2",
"ge4",
"ming2",
"zheng1",
"se4",
"jiao3",
"yi1",
"chan3",
"chong4",
"tang1",
"an3",
"yin2",
"ru2",
"zhu4",
"lao2",
"pu4,pu1",
"wu2",
"lai2",
"te4",
"lian4",
"keng1",
"xiao1",
"suo3",
"li3",
"zeng4",
"chu2",
"guo1",
"gao4",
"e2",
"xiu4",
"cuo4",
"lu:e4",
"feng1",
"xin1",
"liu3",
"kai1",
"jian3",
"rui4",
"ti1",
"lang2",
"qin2",
"ju1,ju2",
"a1",
"qiang1",
"zhe3",
"nuo4",
"cuo4",
"mao2",
"ben1",
"qi2",
"de2",
"ke4",
"kun1",
"chang1",
"xi1",
"gu4",
"luo2",
"chui2",
"zhui1",
"jin3",
"zhi4",
"xian1",
"juan4",
"huo1",
"pei2",
"tan2",
"ding4",
"jian4",
"ju4,ju1",
"meng3",
"zi1",
"qie4",
"ying1",
"kai3",
"qiang1",
"si1",
"e4",
"cha1",
"qiao1",
"zhong1",
"duan4",
"sou1",
"huang2",
"huan2",
"ai1",
"du4",
"mei3",
"lou4",
"zi1",
"fei4",
"mei2",
"mo4",
"zhen4",
"bo2",
"ge2",
"nie4",
"tang3",
"juan1",
"nie4",
"na2",
"liu2",
"gao3,hao4",
"bang4",
"yi4",
"jia1",
"bin1",
"rong2",
"biao1",
"tang1",
"man4",
"luo2",
"beng4",
"yong1",
"jing4",
"di1,di2",
"zu2",
"xuan4",
"liu2",
"tan2,xin2",
"jue2",
"liao4",
"pu2",
"lu3",
"dun1",
"lan2",
"pu3",
"cuan1",
"qiang1,qiang3",
"deng4",
"huo4",
"lei2",
"huan2",
"zhuo2",
"lian2",
"yi4",
"cha3",
"biao1",
"la4",
"chan2",
"xiang1",
"zhang3",
"chang2",
"jiu3",
"ao3",
"die2",
"qu1",
"liao3",
"mi2",
"zhang3,chang2",
"men2",
"ma4",
"shuan1",
"shan3",
"huo4",
"men2",
"yan2",
"bi4",
"han4",
"bi4",
"shan1",
"kai1",
"kang4",
"beng1",
"hong2",
"run4",
"san4",
"xian2",
"jian1,jian4",
"jian1",
"min3",
"xia1",
"shui5",
"dou4",
"zha2",
"nao4",
"zhan1",
"peng1",
"xia3",
"ling2",
"bian4",
"bi4",
"run4",
"ai4",
"guan1",
"ge2",
"ge2,he2",
"fa2",
"chu4",
"hong4",
"gui1",
"min3",
"se1",
"kun3",
"lang4",
"lv2",
"ting2",
"sha4",
"ju2",
"yue4",
"yue4",
"chan3",
"qu4",
"lin4",
"chang1",
"shai4",
"kun3",
"yan1",
"wen2",
"yan2",
"e4",
"hun1",
"yu4",
"wen2",
"hong4",
"bao1",
"hong4",
"qu4",
"yao3",
"wen2",
"ban3",
"an4",
"wei2",
"yin1",
"kuo4",
"que4",
"lan2",
"du1",
"quan2",
"feng1",
"tian2",
"nie4",
"ta4",
"kai3",
"he2",
"que4",
"chuang3",
"guan1",
"dou4",
"qi3",
"kui1",
"tang2",
"guan1",
"piao2",
"kan4",
"xi1",
"hui4",
"chan3",
"pi4",
"dang4",
"huan2",
"ta4",
"wen2",
"ta1",
"men2",
"shuan1",
"shan3",
"yan2",
"han4",
"bi4",
"wen4",
"chuang3",
"run4",
"wei2",
"xian2",
"hong2",
"jian1,jian4",
"min3",
"kang1",
"men4,men1",
"zha2",
"nao4",
"gui1",
"wen2",
"ta4",
"min3",
"lu:2",
"kai3",
"fa2",
"ge2",
"he2",
"kun3",
"jiu1",
"yue4",
"lang2,lang4",
"du1,she2",
"yu4",
"yan1",
"chang1",
"xi4",
"wen2",
"hun1",
"yan2",
"e4,yan1",
"chan3",
"lan2",
"qu4",
"hui4",
"kuo4",
"que4",
"he2",
"tian2",
"ta4",
"que1,que4",
"kan4",
"huan2",
"fu4",
"fu4",
"le4",
"dui4",
"xin4",
"qian1",
"wu4",
"gai4",
"tuo2",
"yin1",
"yang2",
"dou3",
"e4",
"sheng1",
"ban3",
"pei2",
"keng1",
"yun3",
"ruan3",
"zhi3",
"pi2",
"jing3",
"fang2",
"yang2",
"yin1",
"zhen4",
"jie1",
"cheng1",
"e4",
"qu1",
"di3",
"zu3",
"zuo4",
"dian4",
"ling3",
"a1,e1",
"tuo2",
"tuo2",
"bei1,po1",
"bing3",
"fu4",
"ji4",
"lu4,liu4",
"long3",
"chen2",
"xing2",
"duo4",
"lou4",
"mo4",
"jiang4,xiang2",
"shu1",
"sui2",
"xian4",
"er2",
"gui3",
"yu1",
"gai1",
"shan3",
"jun4",
"qiao4",
"xing2",
"chun2",
"fu4",
"bi4",
"xia2",
"shan3",
"sheng1",
"zhi4",
"pu1",
"dou3",
"yuan4",
"zhen4",
"chu2",
"xian4",
"dao3",
"nie4",
"yun3",
"xian3",
"pei2",
"fei4",
"zou1",
"yi4",
"dui4",
"lun2",
"yin1",
"ju1",
"chui2",
"chen2",
"pi2",
"ling2",
"tao2",
"xian4",
"lu4",
"sheng1",
"xian3",
"yin1",
"zhu3",
"yang2",
"reng2",
"xia2",
"chong2",
"yan4",
"yin1",
"yu2",
"di1",
"yu2",
"long2,long1",
"wei1",
"wei1",
"nie4",
"dui4",
"sui2",
"an3",
"huang2",
"jie1",
"sui2",
"yin3,yin4",
"gai4",
"yan3",
"hui1",
"ge2",
"yun3",
"wu4",
"kui2,wei3",
"ai4",
"xi4",
"tang2",
"ji4",
"zhang4",
"dao3",
"ao2",
"xi4",
"yin3",
"sa4",
"rao3",
"lin2",
"tui2",
"deng4",
"jiao3",
"sui4",
"sui2",
"yu4",
"xian3",
"fen2",
"ni3",
"er2",
"ji1",
"dao3",
"xi2",
"yin3",
"zhi4",
"hui1",
"long3",
"xi1",
"li4",
"li4",
"li4",
"zhui1",
"hu2",
"zhi1",
"sun3",
"juan4,jun4",
"nan2,nan4",
"yi4",
"que4,qiao1",
"yan4",
"qin2",
"qian1",
"xiong2",
"ya3",
"ji2",
"gu4",
"huan2",
"zhi4",
"gou4",
"juan4",
"ci2",
"yong1",
"ju1",
"chu2",
"hu1",
"za2",
"luo4",
"yu2",
"chou2",
"diao1",
"sui1",
"han4",
"huo4",
"shuang1",
"guan4,huan2",
"chu2",
"za2",
"yong1",
"ji1",
"sui2",
"chou2",
"liu4",
"li2",
"nan2",
"xue2",
"za2",
"ji2",
"ji2",
"yu3,yu4",
"yu2",
"xue3",
"na3",
"fou3",
"se4",
"mu4",
"wen2",
"fen1",
"pang2",
"yun2",
"li4",
"chi4",
"yang1",
"ling2",
"lei2",
"an2",
"bao2",
"wu4",
"dian4",
"dang4",
"hu4",
"wu4",
"diao4",
"xu1",
"ji4",
"mu4",
"chen2",
"xiao1",
"zha2",
"ting2",
"zhen4",
"pei4",
"mei2",
"ling2",
"qi1",
"zhou1",
"huo4",
"sha4",
"fei1",
"hong2",
"zhan1",
"yin1",
"ni2",
"zhu4",
"tun2",
"lin2",
"ling2",
"dong4",
"ying1",
"wu4",
"ling2",
"shuang1",
"ling2",
"xia2",
"hong2",
"yin1",
"mai4",
"mai4",
"yun3",
"liu4",
"meng4",
"bin1",
"wu4",
"wei4",
"kuo4",
"yin2",
"xi2",
"yi4",
"ai3",
"dan4",
"teng4",
"xian4",
"yu4",
"lu4,lou4",
"long2",
"dai4",
"ji2",
"pang1",
"yang2",
"ba4",
"pi1",
"wei1",
"feng1",
"xi4",
"ji4",
"mai2",
"meng2",
"meng2",
"lei2",
"li4",
"huo4",
"ai3",
"fei4",
"dai4",
"long2",
"ling2",
"ai4",
"feng1",
"li4",
"bao3",
"he4",
"he4",
"he4",
"bing4",
"qing1",
"qing1",
"jing4,liang4",
"tian1",
"zhen1",
"jing4",
"cheng1",
"qing4",
"jing4",
"jing4",
"dian4",
"jing4",
"tian1",
"fei1",
"fei1",
"kao4",
"mi2,mi3",
"mian4",
"mian4",
"bao4",
"ye4",
"tian3",
"hui4",
"ye4",
"ge2",
"ding1",
"cha2",
"jian1",
"ren4",
"di2",
"du4",
"wu4",
"ren4",
"qin2",
"jin4",
"xue1",
"niu3",
"ba3",
"yin3",
"sa3",
"na4",
"mo4",
"zu3",
"da2",
"ban4",
"yi4",
"yao4",
"tao2",
"bei4",
"jie1",
"hong2",
"pao2",
"yang1",
"bing3",
"yin1",
"ge2",
"tao2",
"jie2",
"xie2",
"an1",
"an1",
"hen2",
"gong3",
"qia3",
"da2",
"qiao2",
"ting1",
"man2",
"ying4",
"sui1",
"tiao2",
"qiao4",
"xuan4",
"kong4",
"beng3",
"ta4",
"shang4,zhang3",
"bing3",
"kuo4",
"ju1",
"la5",
"xie4",
"rou2",
"bang1",
"eng1",
"qiu1",
"qiu1",
"he2",
"qiao4",
"mu4",
"ju1",
"jian1",
"bian1",
"di1",
"jian1",
"wen1",
"tao1",
"gou1",
"ta4",
"bei4",
"xie2",
"pan2",
"ge2",
"bi4",
"kuo4",
"tang1",
"lou2",
"gui4",
"qiao2",
"xue1",
"ji1",
"jian1",
"jiang1",
"chan4",
"da2",
"hu4",
"xian3",
"qian1",
"du2",
"wa4",
"jian1",
"lan2",
"wei2",
"ren4",
"fu2",
"mei4",
"quan4",
"ge2",
"wei3",
"qiao4",
"han2",
"chang4",
"kuo4",
"rou2",
"yun4",
"she4",
"wei3",
"ge2",
"bai4",
"tao1",
"gou1",
"yun4",
"gao1",
"bi4",
"wei3",
"sui4",
"du2",
"wa4",
"du2",
"wei2",
"ren4",
"fu2",
"han2",
"wei3",
"yun4",
"tao1",
"jiu3",
"jiu3",
"xian1",
"xie4",
"xian1",
"ji1",
"yin1",
"za2",
"yun4",
"shao2",
"le4",
"peng2",
"huang2",
"ying1",
"yun4",
"peng2",
"an1",
"yin1",
"xiang3",
"hu4",
"ye4",
"ding3",
"qing3",
"qiu2",
"xiang4",
"shun4",
"han1",
"xu1",
"yi2",
"xu1",
"e3",
"song4",
"kui3",
"qi2",
"hang2",
"yu4",
"wan2",
"ban1",
"dun4",
"di2",
"dan1",
"pan4",
"po1",
"ling3",
"che4",
"jing3",
"lei4",
"he2",
"qiao1",
"e4",
"e2",
"wei3",
"xie2",
"kuo4",
"shen3",
"yi2",
"yi2",
"hai2",
"dui3",
"yu3",
"ping1",
"lei4",
"fu3",
"jia2",
"tou2",
"hui4",
"kui2",
"jia2",
"luo1",
"ting3",
"cheng1",
"ying3",
"yun1",
"hu2",
"han4",
"jing3",
"tui2",
"tui2",
"pin2",
"lai4",
"tui2",
"zi1",
"zi1",
"chui2",
"ding4",
"lai4",
"tan2",
"han4",
"qian1",
"ke1",
"cui4",
"xuan3",
"qin1",
"yi2",
"sai1",
"ti2",
"e2",
"e4",
"yan2",
"wen4",
"kan3",
"yong2",
"zhuan1",
"yan2",
"xian3",
"xin4",
"yi3",
"yuan4",
"sang3",
"dian1",
"dian1",
"jiang3",
"kui1",
"lei4",
"lao2",
"piao3",
"wai4",
"man2",
"qi1",
"yao2",
"hao4",
"qiao2",
"gu4",
"xun4",
"yan3",
"hui4",
"chan4",
"ru2",
"meng2",
"bin1",
"xian3",
"pin2",
"lu2",
"lan3",
"nie4",
"quan2",
"ye4,xie2",
"ding3",
"qing3,qing1",
"han1",
"xiang4",
"shun4",
"xu1",
"xu4",
"wan2",
"gu4",
"dun4",
"qi2",
"ban1",
"song4",
"hang2",
"yu4",
"lu2",
"ling3",
"po1",
"jing3",
"jie2,xie2",
"jia2",
"ting3",
"he2",
"ying3",
"jiong3",
"ke1,ke2",
"yi2",
"pin2",
"hui4",
"tui2",
"han4",
"ying3",
"ying3",
"ke1",
"ti2",
"yong2",
"e4",
"zhuan1",
"yan2",
"e2",
"nie4",
"man1",
"dian1",
"sang3",
"hao4",
"lei4",
"chan4",
"ru2",
"pin2",
"quan2",
"feng1",
"biao1",
"gua1",
"fu2",
"xia1",
"zhan3",
"biao1",
"sa4",
"ba2",
"tai2",
"lie4",
"gua1",
"xuan4",
"shao1",
"ju4",
"biao1",
"si1",
"wei3",
"yang2",
"yao2",
"sou1",
"kai3",
"sou1",
"fan1",
"liu2",
"xi2",
"liao2",
"piao1",
"piao1",
"liu2",
"biao1",
"biao1",
"biao1",
"liao2",
"biao1",
"se4",
"feng1",
"xiu1",
"feng1",
"yang2",
"zhan3",
"biao1",
"sa4",
"ju4",
"si1",
"sou1",
"yao2",
"liu2",
"piao1",
"biao1",
"biao1",
"fei1",
"fan1",
"fei1",
"fei1",
"shi2,si4",
"shi2",
"can1",
"ji1",
"ding4",
"si4",
"tuo1",
"zhan1",
"sun1",
"xiang3",
"tun2",
"ren4",
"yu4",
"juan4,yang3",
"chi4",
"yin3",
"fan4",
"fan4",
"sun1",
"yin3",
"tou3",
"yi2",
"zuo4",
"bi4",
"jie3",
"tao1",
"bao3",
"ci2",
"tie4",
"si4",
"bao3",
"shi4",
"duo4",
"hai4",
"ren4",
"tian3",
"jiao3",
"jia2",
"bing3",
"yao2",
"tong2",
"ci2",
"xiang3",
"yang3",
"juan4",
"er3",
"yan4",
"le5",
"xi1",
"can1",
"bo1",
"nei3",
"e4",
"bu4",
"jun4",
"dou4",
"su4",
"yu2",
"shi4",
"yao2",
"hun2",
"guo3",
"shi4",
"jian4",
"zhui4",
"bing3",
"xian4",
"bu4",
"ye4",
"tan2",
"fei1",
"zhang1",
"wei4",
"guan3",
"e4",
"nuan3",
"hun2",
"hu2",
"huang2",
"tie4",
"hui4",
"jian1",
"hou2",
"he2",
"tang2",
"fen1",
"wei4",
"gu3",
"cha1",
"song4",
"tang2",
"bo2",
"gao1",
"xi4",
"kui4",
"liu4",
"sou1",
"tao2",
"ye4",
"wen1",
"mo2",
"tang2",
"man2",
"bi4",
"yu4",
"xiu1",
"jin3",
"san3",
"kui4",
"zhuan4",
"shan4",
"chi4",
"dan4",
"yi4",
"ji1",
"rao2",
"cheng1",
"yong1",
"tao1",
"wei4",
"xiang3",
"zhan1",
"fen1",
"hai4",
"meng2",
"yan4",
"mo2",
"chan2",
"xiang3",
"luo2",
"zan4",
"nang2",
"shi2",
"ding4",
"ji1",
"tuo1",
"xing2",
"tun2",
"xi4",
"ren4",
"yu4",
"chi4",
"fan4",
"yin3,yin4",
"jian4",
"shi4",
"bao3",
"si4",
"duo4",
"yi2",
"er3",
"rao2",
"xiang3",
"he2",
"le5",
"jiao3",
"xi1",
"bing3",
"bo1",
"dou4",
"e4",
"yu2",
"nei3",
"jun4",
"guo3",
"hun2",
"xian4",
"guan3",
"cha1",
"kui4",
"gu3",
"sou1",
"chan2",
"ye4",
"mo2",
"bo2",
"liu2,liu4",
"xiu1",
"jin3",
"man2",
"san3",
"zhuan4",
"nang2,nang3",
"shou3",
"kui2",
"guo2",
"xiang1",
"fen2",
"bo2",
"ni3",
"bi4",
"bo2",
"tu2",
"han1",
"fei1",
"jian1",
"yan3",
"ai4",
"fu4",
"xian1",
"yun1",
"xin1",
"fen2",
"pin1",
"xin1",
"ma3",
"yu4",
"feng2",
"han4",
"di2",
"tuo2",
"tuo1",
"chi2",
"xun2",
"zhu4",
"zhi1",
"pei4",
"xin4",
"ri4",
"sa4",
"yun3",
"wen2",
"zhi2",
"dan4",
"lv2",
"you2",
"bo2",
"bao3",
"jue2",
"tuo2",
"yi4",
"qu1",
"wen2",
"qu1",
"jiong1",
"po3",
"zhao1",
"yuan1",
"pei2",
"zhou4",
"ju4",
"zhu4",
"nu2",
"ju1",
"pi1",
"zang3",
"jia4",
"ling2",
"zhen1",
"tai2",
"fu4",
"yang3",
"shi3",
"bi4",
"tuo2",
"tuo2",
"si4",
"liu2",
"ma4",
"pian2",
"tao2",
"zhi4",
"rong2",
"teng2",
"dong4",
"xun1",
"quan1",
"shen1",
"jiong1",
"er3",
"hai4",
"bo2",
"zhu1",
"yin1",
"luo4",
"zhou1",
"dan4",
"xie4",
"liu2",
"ju2",
"song3",
"qin1",
"mang2",
"lang2",
"han4",
"tu2",
"xuan4",
"tui4",
"jun4",
"e3",
"cheng3",
"xing1",
"ai2",
"lu4",
"zhui1",
"zhou1",
"she4",
"pian2",
"kun1",
"tao2",
"lai2",
"zong1",
"ke4",
"qi2",
"qi2",
"yan4",
"fei1",
"sao1",
"yan4",
"ge2",
"yao3",
"wu4",
"pian4",
"cong1",
"pian4",
"qian2",
"fei1",
"huang2",
"qian2",
"huo1",
"yu2",
"ti2",
"quan2",
"xia2",
"zong1",
"kui2",
"rou2",
"si1",
"gua1",
"tuo2",
"gui1",
"sou1",
"qian1",
"cheng2",
"zhi4",
"liu2",
"peng2",
"teng2",
"xi2",
"cao3",
"du2",
"yan4",
"yuan2",
"zou1",
"sao1",
"shan4",
"qi2",
"zhi4",
"shuang1",
"lu4",
"xi2",
"luo2",
"zhang1",
"mo4",
"ao4",
"can1",
"biao1",
"cong1",
"qu1",
"bi4",
"zhi4",
"yu4",
"xu1",
"hua2",
"bo1",
"su4",
"xiao1",
"lin2",
"zhan4",
"dun1",
"liu2",
"tuo2",
"ceng2",
"tan2",
"jiao1",
"tie3",
"yan4",
"luo2",
"zhan1",
"jing1",
"yi4",
"ye4",
"tuo1",
"pin1",
"zhou4",
"yan4",
"long2",
"lv2",
"teng2",
"xiang1",
"ji4",
"shuang1",
"ju2",
"xi2",
"huan1",
"li2",
"biao1",
"ma3",
"yu4",
"tuo2,duo4",
"xun4",
"chi2",
"qu1",
"ri4",
"bo2",
"lu:2",
"zang3",
"shi3",
"si4",
"fu4",
"ju1",
"zou1",
"zhu4",
"tuo2",
"nu2",
"jia4",
"yi4",
"tai2",
"xiao1",
"ma4",
"yin1",
"jiao1",
"hua2",
"luo4",
"hai4",
"pian2",
"biao1",
"li2",
"cheng3",
"yan4",
"xing1",
"qin1",
"jun4",
"qi2",
"qi2,ji4",
"ke4",
"zhui1",
"zong1",
"su4",
"can1",
"pian4",
"zhi4",
"kui2",
"sao1",
"wu4",
"ao4",
"liu2",
"qian1",
"shan4",
"piao4",
"luo2",
"cong1",
"chan3",
"zhou4",
"ji4",
"shuang1",
"xiang1",
"gu3",
"wei3",
"wei3",
"wei3",
"yu2",
"gan4",
"yi4",
"ang1",
"tou2",
"xie4",
"bao4",
"bei4",
"ci1",
"ti3",
"di3",
"ku1",
"hai2",
"qiao1",
"hou2",
"kua4",
"ge2",
"tui3",
"geng3",
"pian2",
"bi4",
"ke1",
"qia4",
"yu2",
"sui3",
"lou2",
"bo2",
"xiao1",
"bang3",
"bo2",
"ci1",
"kuan1",
"bin4",
"mo2",
"liao2",
"lou2",
"xiao1",
"du2",
"zang1",
"sui3",
"ti3",
"bin4",
"kuan1",
"lu2",
"gao1",
"gao1",
"qiao4",
"kao1",
"qiao3",
"lao2",
"sao4",
"biao1",
"kun1",
"kun1",
"di2",
"fang3",
"xiu1",
"ran2",
"mao2",
"dan4",
"kun1",
"bin4",
"fa4",
"tiao2",
"pi1",
"zi1",
"fa4",
"ran2",
"ti4",
"bao4",
"pi4",
"mao2",
"fu2",
"er2",
"rong2",
"qu1",
"gong1",
"xiu1",
"kuo4",
"ji4",
"peng2",
"zhua1",
"shao1",
"suo1",
"ti4",
"li4",
"bin4",
"zong1",
"di2,ti4",
"peng2",
"song1",
"zheng1",
"quan2",
"zong1",
"shun4",
"jian1",
"tuo3",
"hu2",
"la4",
"jiu1",
"qi2",
"lian2",
"zhen3",
"bin4",
"peng2",
"ma4",
"san1",
"man4",
"man2",
"seng1",
"xu1",
"lie4",
"qian1",
"qian1",
"nang2",
"huan2",
"kuai4",
"ning2",
"bin4",
"lie4",
"rang2",
"dou4",
"dou4",
"nao4",
"hong4",
"xi4",
"dou4",
"kan4",
"dou4",
"dou4",
"jiu1",
"chang4",
"yu4",
"yu4",
"ge2,li4",
"yan4",
"fu3",
"qian2",
"gui1",
"zong1",
"liu4",
"gui1",
"shang1",
"yu4",
"gui3",
"mei4",
"ji4",
"qi2",
"ga4",
"kui2",
"hun2",
"ba2",
"po4",
"mei4",
"xu4",
"yan3",
"xiao1",
"liang3",
"yu4",
"tui2",
"qi1",
"wang3",
"liang3",
"wei4",
"gan1",
"chi1",
"piao1",
"bi4",
"mo2",
"ji3",
"xu1",
"chou3",
"yan3",
"zhan1",
"yu2",
"dao1",
"ren2",
"jie2",
"ba1",
"hong2",
"tuo1",
"diao4",
"ji3",
"yu2",
"e2",
"e4",
"sha1",
"hang2",
"tun2",
"mo4",
"jie4",
"shen3",
"fan3",
"yuan2",
"pi2",
"lu3",
"wen2",
"hu2",
"lu2",
"za1",
"fang2",
"fen2",
"na4",
"you2",
"pian4",
"mo2",
"he2",
"xia2",
"qu1",
"han2",
"pi1",
"ling2",
"tuo2",
"bo1",
"qiu2",
"ping2",
"fu2",
"bi4",
"ci3",
"wei4",
"ju1",
"diao1",
"ba4",
"you2",
"gun3",
"pi1",
"nian2",
"xing1",
"tai2",
"bao4",
"fu4",
"zha3",
"ju4",
"gu1",
"shi2",
"dong1",
"dai5",
"ta4",
"jie2",
"shu1",
"hou4",
"xiang3",
"er2",
"an4",
"wei2",
"zhao4",
"zhu1",
"yin4",
"lie4",
"luo4",
"tong2",
"ti3",
"qi2",
"bing4",
"wei3",
"jiao1",
"ku1",
"gui1",
"xian1",
"ge2",
"hui2",
"lao3",
"fu2",
"kao4",
"xiu1",
"duo2",
"jun1",
"ti2",
"mian3",
"shao1",
"zha3",
"suo1",
"qin1",
"yu2",
"nei3",
"zhe2",
"gun3",
"geng3",
"su1",
"wu2",
"qiu2",
"shan1",
"fu3",
"huan4",
"chou2",
"li3",
"sha1",
"sha1",
"kao4",
"meng2",
"cheng2",
"li2",
"zou3",
"xi1",
"yong3",
"shen1",
"zi1",
"qi2",
"zheng1",
"xiang3",
"nei3",
"chun2",
"ji4",
"diao1",
"qie4",
"gu4",
"zhou3",
"dong1",
"lai2",
"fei4",
"ni2",
"yi4",
"kun1",
"lu4",
"jiu4",
"chang1",
"jing1",
"lun2",
"ling2",
"zou1",
"li2",
"meng3",
"zong1",
"zhi4",
"nian2",
"hu3",
"yu2",
"di3",
"shi1",
"shen1",
"huan4",
"ti2",
"hou2",
"xing1",
"zhu1",
"la4",
"zong1",
"zei2",
"bian1",
"bian1",
"huan4",
"quan2",
"zei2",
"wei1",
"wei1",
"yu2",
"chun1",
"rou2",
"die2",
"huang2",
"lian4",
"yan3",
"qiu1",
"qiu1",
"jian3",
"bi1",
"e4",
"yang2",
"fu4",
"sai1",
"gan3",
"xia1",
"tuo3",
"hu2",
"shi4",
"ruo4",
"xuan1",
"wen1",
"qian4",
"hao4",
"wu1",
"fang2",
"sao1",
"liu2",
"ma3",
"shi2",
"shi1",
"guan1",
"zi1",
"teng2",
"ta3",
"yao2",
"e2",
"yong2",
"qian2",
"qi2",
"wen1",
"ruo4",
"shen2",
"lian2",
"ao2",
"le4",
"hui1",
"min3",
"ji4",
"tiao2",
"qu1",
"jian1",
"shen1",
"man2",
"xi2",
"qiu2",
"biao4",
"ji4",
"ji4",
"zhu2",
"jiang1",
"xiu1",
"zhuan1",
"yong1",
"zhang1",
"kang1",
"xue3",
"bie1",
"yu4",
"qu1",
"xiang4",
"bo1",
"jiao3",
"xun2",
"su4",
"huang2",
"zun1",
"shan4",
"shan4",
"fan1",
"gui4",
"lin2",
"xun2",
"miao2",
"xi3",
"zeng1",
"xiang1",
"fen4",
"guan1",
"hou4",
"kuai4",
"zei2",
"sao1",
"zhan1",
"gan3",
"gui4",
"ying4",
"li3",
"chang2",
"lei2",
"shu3",
"ai4",
"ru2",
"ji4",
"xu4",
"hu4",
"shu3",
"li4",
"lie4",
"li4",
"mie4",
"zhen1",
"xiang3",
"e4",
"lu2",
"guan4",
"li2",
"xian1",
"yu2",
"dao1",
"ji3",
"you2",
"tun2",
"lu3",
"fang2",
"ba1",
"he2",
"ba4",
"ping2",
"nian2",
"lu2",
"you2",
"zha3",
"fu4",
"ba4",
"bao4",
"hou4",
"pi2",
"tai2",
"gui1",
"jie2",
"kao4",
"wei3",
"er2",
"tong2",
"zei2",
"hou4",
"kuai4",
"ji4",
"jiao1",
"xian1,xian3",
"zha3",
"xiang3",
"xun2",
"geng3",
"li2",
"lian2",
"jian1",
"li3",
"shi2",
"tiao2",
"gun3",
"sha1",
"huan4",
"jun1",
"ji4",
"yong3",
"qing1",
"ling2",
"qi2",
"zou1",
"fei1",
"kun1",
"chang1",
"gu4",
"ni2",
"nian2",
"diao1",
"jing1",
"shen1",
"shi1",
"zi1",
"fen4",
"die2",
"bi1",
"chang2",
"ti2",
"wen1",
"wei1",
"sai1",
"e4",
"qiu1",
"fu4",
"huang2",
"quan2",
"jiang1",
"bian1",
"sao1",
"ao2",
"qi2",
"ta3",
"guan1",
"yao2",
"pang2",
"jian1",
"le4",
"biao4",
"xue3",
"bie1",
"man2",
"min3",
"yong1",
"wei4",
"xi2",
"gui4",
"shan4",
"lin2",
"zun1",
"hu4",
"gan3",
"li3",
"zhan1,shan4",
"guan3",
"niao3",
"yi3",
"fu2",
"li4",
"jiu1",
"bu2",
"yan4",
"fu3",
"diao1",
"ji1",
"feng4",
"ru4",
"gan1",
"shi1",
"feng4",
"ming2",
"bao3",
"yuan1",
"zhi1",
"hu4",
"qin2",
"fu1",
"fen1",
"wen2",
"yan2",
"shi1",
"yu4",
"fou3",
"yao1",
"jue2",
"jue2",
"pi1",
"huan1",
"zhen4",
"bao3",
"yan4",
"ya1",
"zheng4",
"fang1",
"feng4",
"wen2",
"ou1",
"dai4",
"ge1",
"ru2",
"ling2",
"mie4",
"fu2",
"tuo2",
"min2",
"li4",
"bian3",
"zhi4",
"ge1",
"yuan1",
"ci2",
"qu2",
"xiao1",
"chi1",
"dan4",
"ju1",
"you4",
"gu1",
"zhong1",
"yu4",
"yang1",
"yu4",
"ya1",
"tie3",
"yu4",
"tian2",
"ying1",
"dui1",
"wu1",
"er2",
"gua1",
"ai4",
"zhi1",
"yan4",
"heng2",
"xiao1",
"jia2",
"lie4",
"zhu1",
"yang2",
"ti2",
"hong2",
"luo4",
"ru2",
"mou2",
"ge1",
"ren4",
"jiao1",
"xiu1",
"zhou1",
"chi1",
"luo4",
"heng2",
"nian2",
"e3",
"luan2",
"jia2",
"ji4",
"tu2",
"huan1",
"tuo3",
"bu3",
"wu2",
"juan1",
"yu4",
"bo2",
"jun4",
"jun4",
"bi1",
"xi1",
"jun4",
"ju2",
"tu1",
"jing1",
"ti2",
"e2",
"e2",
"kuang2",
"hu2",
"wu3",
"shen1",
"lai4",
"jiao5",
"pan4",
"lu4",
"pi2",
"shu1",
"fu2",
"an1",
"zhao4",
"peng2",
"qin2",
"qian1",
"bei1",
"diao1",
"lu4",
"que4",
"jian1",
"ju2",
"tu4",
"ya1",
"yuan1",
"qi2",
"li2",
"ye4",
"zhui1",
"kong1",
"duo4",
"kun1",
"sheng1",
"qi2",
"jing1",
"yi4",
"yi4",
"jing1",
"zi1",
"lai2",
"dong1",
"qi1",
"chun2",
"geng1",
"ju1",
"jue2",
"yi4",
"zun1",
"ji1",
"shu4",
"ying1",
"chi4",
"miao2",
"rou2",
"an1",
"qiu1",
"ti2",
"hu2",
"ti2",
"e4",
"jie1",
"mao2",
"fu2",
"chun1",
"tu2",
"yan3",
"he2",
"yuan2",
"pian1",
"kun1",
"mei2",
"hu2",
"ying1",
"chuan4",
"wu4",
"ju2",
"dong1",
"cang1",
"fang3",
"he4",
"ying1",
"yuan2",
"xian1",
"weng1",
"shi1",
"he4",
"chu2",
"tang2",
"xia2",
"ruo4",
"liu2",
"ji2",
"gu2",
"jian1",
"sun3",
"han4",
"ci2",
"ci2",
"yi4",
"yao4",
"yan4",
"ji1",
"li4",
"tian2",
"kou4",
"ti1",
"ti1",
"yi4",
"tu2",
"ma3",
"xiao1",
"gao1",
"zhen1",
"chen2",
"ji2",
"tuan2",
"zhe4",
"ao2",
"yao3",
"yi1",
"ou1",
"chi4",
"zhi4",
"liu4",
"yong1",
"lv2",
"bi4",
"shuang1",
"zhuo2",
"yu2",
"wu2",
"jue2",
"yin2",
"ti2",
"si1",
"jiao1",
"yi4",
"hua2",
"bi4",
"ying1",
"su4",
"huang2",
"fan2",
"jiao1",
"liao2",
"yan4",
"gao1",
"jiu4",
"xian2",
"xian2",
"tu2",
"mai3",
"zun1",
"yu4",
"ying1",
"lu4",
"tuan2",
"xian2",
"xue2",
"yi4",
"pi4",
"chu3",
"luo2",
"xi1",
"yi2",
"ji1",
"ze2",
"yu2",
"zhan1",
"ye4",
"yang2",
"pi4",
"ning2",
"hu4",
"mi2",
"ying1",
"meng2",
"di2",
"yue4",
"yu2",
"lei3",
"bu3",
"lu2",
"he4",
"long2",
"shuang1",
"yue4",
"ying1",
"guan4",
"qu2",
"li2",
"luan2",
"niao3,diao3",
"jiu1",
"ji1",
"yuan1",
"ming2",
"shi1",
"ou1",
"ya1",
"cang1",
"bao3",
"zhen4",
"gu1",
"dong1",
"lu2",
"ya1",
"xiao1",
"yang1",
"ling2",
"chi1",
"qu2",
"yuan1",
"xue2",
"tuo2",
"si1",
"zhi4",
"er2",
"gua1",
"xiu1",
"heng2",
"zhou1",
"ge1",
"luan2",
"hong2",
"wu2",
"bo2",
"li2",
"juan1",
"gu3,hu2",
"e2",
"yu4",
"xian2",
"ti2",
"wu3",
"que4",
"miao2",
"an1",
"kun1",
"bei1",
"peng2",
"qian1",
"chun2",
"geng1",
"yuan1",
"su4",
"hu2",
"he2",
"e4",
"hu2",
"qiu1",
"ci2",
"mei2",
"wu4",
"yi4",
"yao4",
"weng1",
"liu2",
"ji2",
"yi4",
"jian1",
"he4",
"yi1",
"ying1",
"zhe4",
"liu4",
"liao2",
"jiao1",
"jiu4",
"yu4",
"lu4",
"huan2",
"zhan1",
"ying1",
"hu4",
"meng2",
"guan4",
"shuang1",
"lu3",
"jin1",
"ling2",
"jian3",
"xian2",
"cuo2",
"jian3",
"jian3",
"yan2",
"cuo2",
"lu4",
"you1",
"cu1",
"ji3",
"pao2,biao1",
"cu1",
"pao2",
"zhu4",
"jun1",
"zhu3",
"jian1",
"mi2",
"mi2",
"wu2",
"liu2",
"chen2",
"jun1",
"lin2",
"ni2",
"qi2",
"lu4",
"jiu4",
"jun1",
"jing1",
"li4",
"xiang1",
"xian2",
"jia1",
"mi2",
"li4",
"she4",
"zhang1",
"lin2",
"jing1",
"qi2",
"ling2",
"yan2",
"cu1",
"mai4",
"mai4",
"he2",
"chao3",
"fu1",
"mian4",
"mian4",
"fu1",
"pao4",
"qu4",
"qu1",
"mou2",
"fu1",
"xian4",
"lai2",
"qu1",
"mian4",
"chi5",
"feng1",
"fu1",
"qu1",
"mian4",
"ma2",
"me5",
"mo2",
"hui1",
"mo3",
"zou1",
"nun2",
"fen2",
"huang2",
"huang2",
"jin1",
"guang1",
"tian1",
"tou3",
"hong2",
"hua4",
"kuang4",
"hong2",
"shu3",
"li2",
"nian2",
"chi1",
"hei1",
"hei1",
"yi4",
"qian2",
"zhen3",
"xi4",
"tun1",
"mo4",
"mo4",
"qian2",
"dai4",
"chu4",
"you3",
"dian3",
"yi1",
"xia2",
"yan3",
"qu1",
"mei3",
"yan3",
"qing2",
"yu4",
"li2",
"dang3",
"du2",
"can3",
"yan1",
"yan2",
"yan3",
"tan2",
"an4",
"zhen3",
"dai4",
"can3",
"yi1",
"mei2",
"zhan3",
"yan3",
"du2",
"lu2",
"zhi3",
"fen3",
"fu2",
"fu3",
"mian3",
"min3",
"yuan2",
"cu4",
"qu4",
"chao2",
"wa1",
"zhu1",
"zhi1",
"meng2",
"ao2",
"bie1",
"tuo2",
"bi4",
"yuan2",
"chao2",
"tuo2",
"ding3",
"mi4",
"nai4",
"ding3",
"zi1",
"gu3",
"gu3",
"dong1",
"fen2",
"tao2",
"yuan1",
"pi2",
"chang1",
"gao1",
"qi4",
"yuan1",
"tang1",
"teng1",
"shu3",
"shu3",
"fen2",
"fei4",
"wen2",
"ba2",
"diao1",
"tuo2",
"zhong1",
"qu2",
"sheng1",
"shi2",
"you4",
"shi2",
"ting2",
"wu2",
"ju2",
"jing1",
"hun2",
"ju2",
"yan3",
"tu1",
"si1",
"xi1",
"xian4",
"yan3",
"lei2",
"bi2",
"yao4",
"qiu2",
"han1",
"wu4",
"wu4",
"hou1",
"xie4",
"e4",
"zha1",
"xiu4",
"weng4",
"zha1",
"nong2",
"nang4",
"qi2",
"zhai1",
"ji4",
"zi1",
"ji1",
"ji1",
"qi2",
"ji1",
"chi3",
"chen4",
"chen4",
"he2",
"ya2",
"yin2",
"xie4",
"bao1",
"ze2",
"xie4",
"chai2",
"chi1",
"nian4",
"ju3",
"tiao2",
"ling2",
"ling2",
"chu1",
"quan2",
"xie4",
"ken3",
"nie4",
"jiu4",
"yao3",
"chuo4",
"yun3",
"yu3",
"chu3",
"yi3",
"ni2",
"ze2",
"zou1",
"qu3",
"yun3",
"yan3",
"yu2",
"e4",
"wo4",
"yi4",
"ci1",
"zou1",
"dian1",
"chu3",
"jin4",
"ya4",
"chi3",
"chen4",
"he2",
"yin2",
"ju3",
"ling2",
"bao1",
"tiao2",
"zi1",
"ken3,yin2",
"yu3",
"chuo4",
"qu3",
"wo4",
"long2",
"pang2",
"gong1",
"pang2",
"yan3",
"long2",
"long3",
"gong1",
"kan1",
"da2",
"ling2",
"da2",
"long2",
"gong1",
"kan1",
"gui1",
"qiu1",
"bie1",
"gui1",
"yue4",
"chui4",
"he2",
"jue2",
"xie2",
"yu4"
};

