use crate::abs::{AbsAxis, AbsInfo};
use crate::convert::Convert;
use crate::glue;
use crate::interceptor::Interceptor;
use crate::key::Key;
use crate::rel::RelAxis;

pub struct RelCaps<'a> {
    current: u16,
    interceptor: &'a Interceptor,
}

impl<'a> RelCaps<'a> {
    pub(super) fn new(interceptor: &'a Interceptor) -> Self {
        let has =
            unsafe { glue::libevdev_has_event_type(interceptor.evdev.as_ptr(), glue::EV_REL) == 1 };

        Self {
            current: if has { 0 } else { glue::REL_MAX as _ },
            interceptor,
        }
    }
}

impl Iterator for RelCaps<'_> {
    type Item = RelAxis;

    fn next(&mut self) -> Option<Self::Item> {
        while self.current < glue::REL_MAX as _ {
            let has = unsafe {
                glue::libevdev_has_event_code(
                    self.interceptor.evdev.as_ptr(),
                    glue::EV_REL,
                    self.current as _,
                ) == 1
            };

            self.current += 1;

            if !has {
                continue;
            }

            if let Some(axis) = RelAxis::from_raw(self.current - 1) {
                return Some(axis);
            }
        }

        None
    }
}

pub struct AbsCaps<'a> {
    current: u16,
    interceptor: &'a Interceptor,
}

impl<'a> AbsCaps<'a> {
    pub(super) fn new(interceptor: &'a Interceptor) -> Self {
        let has =
            unsafe { glue::libevdev_has_event_type(interceptor.evdev.as_ptr(), glue::EV_ABS) == 1 };

        Self {
            current: if has { 0 } else { glue::ABS_MAX as _ },
            interceptor,
        }
    }
}

impl Iterator for AbsCaps<'_> {
    type Item = (AbsAxis, AbsInfo);

    fn next(&mut self) -> Option<Self::Item> {
        while self.current < glue::ABS_MAX as _ {
            let has = unsafe {
                glue::libevdev_has_event_code(
                    self.interceptor.evdev.as_ptr(),
                    glue::EV_ABS,
                    self.current as _,
                ) == 1
            };

            self.current += 1;

            if !has {
                continue;
            }

            if let Some(axis) = AbsAxis::from_raw(self.current - 1) {
                let info = unsafe {
                    glue::libevdev_get_abs_info(
                        self.interceptor.evdev.as_ptr(),
                        (self.current - 1) as _,
                    )
                };

                let info = unsafe { info.as_ref().unwrap() };
                let info = AbsInfo {
                    min: info.minimum,
                    max: info.maximum,
                    fuzz: info.fuzz,
                    flat: info.flat,
                    resolution: info.resolution,
                };

                return Some((axis, info));
            }
        }

        None
    }
}

pub struct KeyCaps<'a> {
    current: u16,
    interceptor: &'a Interceptor,
}

impl<'a> KeyCaps<'a> {
    pub(super) fn new(interceptor: &'a Interceptor) -> Self {
        let has =
            unsafe { glue::libevdev_has_event_type(interceptor.evdev.as_ptr(), glue::EV_KEY) == 1 };

        Self {
            current: if has { 0 } else { glue::KEY_MAX as _ },
            interceptor,
        }
    }
}

impl Iterator for KeyCaps<'_> {
    type Item = Key;

    fn next(&mut self) -> Option<Self::Item> {
        while self.current < glue::KEY_MAX as _ {
            let has = unsafe {
                glue::libevdev_has_event_code(
                    self.interceptor.evdev.as_ptr(),
                    glue::EV_KEY,
                    self.current as _,
                ) == 1
            };

            self.current += 1;

            if !has {
                continue;
            }

            if let Some(stroke) = Key::from_raw(self.current - 1) {
                return Some(stroke);
            }
        }

        None
    }
}

pub struct Repeat {
    pub delay: Option<i32>,
    pub period: Option<i32>,
}

impl Repeat {
    pub(super) fn new(interceptor: &Interceptor) -> Self {
        let has = unsafe {
            glue::libevdev_has_event_code(interceptor.evdev.as_ptr(), glue::EV_REP, glue::REP_DELAY)
                == 1
        };

        let delay = if has {
            Some(unsafe {
                glue::libevdev_get_event_value(
                    interceptor.evdev.as_ptr(),
                    glue::EV_REP,
                    glue::REP_DELAY,
                )
            })
        } else {
            None
        };

        let has = unsafe {
            glue::libevdev_has_event_code(
                interceptor.evdev.as_ptr(),
                glue::EV_REP,
                glue::REP_PERIOD,
            ) == 1
        };

        let period = if has {
            Some(unsafe {
                glue::libevdev_get_event_value(
                    interceptor.evdev.as_ptr(),
                    glue::EV_REP,
                    glue::REP_PERIOD,
                )
            })
        } else {
            None
        };

        Self { delay, period }
    }
}
