# Disk Space Manager API

This document includes descriptions about Disk Space Manager,
especially dealing with `file.h` and `file.cpp` files.

## Main concepts

Every read and write operations associated with page are done by following methods;

* `file::file_read_page`
* `file::file_write_page`
* `file::page_read_value` (private)
* `file::page_write_value` (private)

In-memory structure of page consists of array of bool. Conversions between bool and
other types are performed with `memcpy` method internally.
While dealing with Disk IO, seeks, read and write methods are observed. If some return
results are not complied with expected value, then raise exception right away.

## APIs

### file::file_open_database_file

`int file_open_database_file(const char* pathname)`

Open database file with given pathname. if file is not exists, then create new one.
Database file's mode will be changed as 660(RW-RW---) and be owned by the current user.
If trying to access file already opened, there are no actions.
In the case of creating new file, `file::file_init` and `file::file_scale` will be executed.

#### Parameters

| parameter | description |
| --- | --- |
| pathname | file name to associate the database file |

#### Return value

If successful, return file descriptor of database file. This value will be used in other operations in file disk manager.

#### Example

```cpp
int fd = file_open_database_file("database.db");
```

#### See also

* `file::file_init`
* `file::file_scale`


### file::file_is_initialized

`bool file_is_initialized(int fd)`

Check whether given file discriptor is initialized as database file.
Return true if database file fulfills following conditions;

1. size is more than 0
1. total number of pages in header page is greater than 0

This method should be called with fd generated by `file::file_open_database_file`.

#### Parameters

| parameter | description |
| --- | --- |
| fd | file descriptor returned from file_open_database_file |

#### Return value

True if given file descriptor indicates a file organized as database file.

#### Example

```cpp
bool isInit = file_is_initialized(fd);
```


### file::file_init

`void file_init(int fd)`

Initialize header page of database file.
If file descriptor points a file containing any contents, these will be ereased at first.
Then set empty value(0 for integer value) into free page number and total number of pages.
After executed this method, the file size will be 2 times of page number size.
This method should be called with fd generated by `file::file_open_database_file`.

#### Parameters

| parameter | description |
| --- | --- |
| fd | file descriptor returned from file_open_database_file |

#### Example

```cpp
file_init(fd);
```


### file::file_scale

`void file_scale(int fd)`

Scale up database file per default size.
Default database file size is derived from `file::INITIAL_DB_FILE_SIZE`.
The amount of increases is decided by the maximun of `unsigned int 64` which is used as a type of page number.
Through currents and capacity of total page number, If the database file can not be increased, exception will occur.
This method should be called with fd generated by `file::file_open_database_file`.

#### Parameters

| parameter | description |
| --- | --- |
| fd | file descriptor returned from file_open_database_file |

#### Example

```cpp
file_init(fd);
```


### file::file_alloc_page

`pagenum_t file_alloc_page(int fd)`

Allocate an on-disk page from the free page list.
Check the available free pages in database file, then return one of them.
It accompanies with an action changing next free page number in header page.
If there is no available free pages, new page will be allocated after scale up the file.
This method should be called with fd generated by `file::file_open_database_file`.

#### Parameters

| parameter | description |
| --- | --- |
| fd | file descriptor returned from file_open_database_file |

#### Return value

If successful, return the number of page which is allocated.

#### Example

```cpp
pagenum_t pageNum = file_alloc_page(fd);
```

#### See also

* `file::file_scale`


### file::file_free_page

`void file_free_page(int fd, pagenum_t pagenum)`

Free an on-disk page to the free page list.
The page according to file descriptor and page number will be converted to free page format
which contains next free page number and is filled with empty value for remnant.
Chain of free pages of the file where the page exist will include the page.
This method should be called with fd generated by `file::file_open_database_file`.
The origin of `pagenum` should be same as `fd` in parameters.

#### Parameters

| parameter | description |
| --- | --- |
| fd | file descriptor returned from file_open_database_file |
| pagenum | page number allocated from file_alloc_page |

#### Example

```cpp
file_free_page(fd, pagenum);
```

### file::file_read_page

`void file_read_page(int fd, pagenum_t pagenum, page_t* dest)`

Read an on-disk page into the in-memory page structure(dest).
dest parameter will contain the value of the allocated page indicated by file descriptor and page number.
This method should be called with fd generated by `file::file_open_database_file`.
The origin of `pagenum` should be same as `fd` in parameters.

#### Parameters

| parameter | description |
| --- | --- |
| fd | file descriptor returned from file_open_database_file |
| pagenum | page number allocated from file_alloc_page |
| dest | page strcture which is not null |

#### Example
```cpp
struct page_t page;
file_read_page(fd, pagenum, &page);
```

### file::file_write_page

`void file_write_page(int fd, pagenum_t pagenum, const page_t* src)`

Write an in-memory page(src) to the on-disk page.
All data contained in src will be pasted into page indicated by both file descriptor and page number.
This method should be called with fd generated by `file::file_open_database_file`.
The origin of `pagenum` should be same as `fd` in parameters.

#### Parameters

| parameter | description |
| --- | --- |
| fd | file descriptor returned from file_open_database_file |
| pagenum | page number allocated from file_alloc_page |
| dest | page strcture which is not null |

#### Example
```cpp
char charArr[4096];
struct page_t page;
page.data = charArr;
file_write_page(fd, pagenum, &page);
```

### file::file_close_database_file

`void file_close_database_file()`

Stop referencing the database file.
All files opened by `file::file_open_database_file` will be closed and
file descriptors opened before will be meaningless after the method is executed.

#### Example
```cpp
file_close_database_file();
```

## UnitTests

### FileinitializerTest::initialization

Mainly about `file::file_open_database_file` method.
With pathname not existing one, new database file is created.
The return value, file descriptor is bigger than zero.
The number of total pages is same as initial database file size(10MiB) divided by page size(4KiB).
The file size estimated by System call is equal to initial database file size.
Finally, deleting database file is done gracefully without any exceptions.

### FileManagerTest::scaleFileSize

Mainly about private `file::file_scale` method.
Some tricks are performed to trigger to execute this private method.
Change the value of free page number in header page to zero then call `file::file_alloc_page`.
Then the number of total pages and physical file size are double compared to initial status.

### FileManagerTest::chainingFreePages

Mainly the links among free pages at initial state.
Starting with the first free page number in header page,
sequential accesses are performed following the values indicated by others.
At the end, the number of pages reached as a free page is equal to
the number of total pages in header page subtracted 1 for header page.

### FileManagerTest::pageManagement

Mainly about `file::file_alloc_page` and `file::file_free_page` methods.
To begin with, allocate two pages and Free one of them.
Count all free pages by following the links among them and compare to
the number of total pages in header page subtracted 2 for header page and allocated one.

### FileManagerTest::pageIO

Mainly about `file::file_write_page` and `file::file_read_page` methods.
To begin with, creating dummy page which has sequential number per bytes.
Allocate a page and write the dummy on it, then read into another variable.
Compare all contents between dummy and new variable to assert that all bytes are same.