/*
 * Copyright 2016-2017 Axioma srl.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.holonplatform.core.internal.utils;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;

/**
 * Utility class for hashing operations.
 * 
 * @since 5.0.0
 */
public final class Hash {

	private static final SecureRandom secureRandom = new SecureRandom();

	/**
	 * Default bytes count for random salt generation
	 */
	public static final int DEFAULT_SALT_BYTES_SIZE = 16;

	/**
	 * Default hash iterations
	 */
	public static final int DEFAULT_HASH_ITERATIONS = 1;

	/*
	 * Empty private constructor: this class is intended only to provide constants ad utility methods.
	 */
	private Hash() {
	}

	/**
	 * Generate secure random salt data to use with hash algorithm
	 * @return Secure random salt data
	 */
	public static byte[] generateSalt() {
		return generateSalt(DEFAULT_SALT_BYTES_SIZE);
	}

	/**
	 * Generate secure random salt data to use with hash algorithm
	 * @param numBytes Size of generated bytes
	 * @return Secure random salt data
	 */
	public static byte[] generateSalt(int numBytes) {
		byte[] bytes = new byte[numBytes];
		secureRandom.nextBytes(bytes);
		return bytes;
	}

	/**
	 * Perform hashing on given String <code>source</code>. UTF-8 encoding is used for source to bytes conversion.
	 * @param algorithmName Hash algorithm to use
	 * @param source Source string
	 * @param salt Optional salt
	 * @param iterations Hash iterations to perform
	 * @return Hashed source as bytes array
	 * @throws NoSuchAlgorithmException Hash algorithm is not available in the environment
	 */
	public static byte[] hash(String algorithmName, String source, byte[] salt, int iterations)
			throws NoSuchAlgorithmException {
		return hash(algorithmName, ConversionUtils.toBytes(source), salt, iterations);
	}

	/**
	 * Perform hashing on given <code>source</code>
	 * @param algorithmName Hash algorithm to use
	 * @param source Source bytes
	 * @param salt Optional salt
	 * @param iterations Hash iterations to perform
	 * @return Hashed source as bytes array
	 * @throws NoSuchAlgorithmException Hash algorithm is not available in the environment
	 */
	public static byte[] hash(String algorithmName, byte[] source, byte[] salt, int iterations)
			throws NoSuchAlgorithmException {
		if (algorithmName == null) {
			throw new IllegalArgumentException("Null hash algorithm name");
		}
		if (source == null || source.length == 0) {
			throw new IllegalArgumentException("Null hash bytes source");
		}

		MessageDigest digest = MessageDigest.getInstance(algorithmName);

		// salt
		if (salt != null && salt.length > 0) {
			digest.reset();
			digest.update(salt);
		}

		byte[] hashed = digest.digest(source);

		// iterations
		int hashIterations = iterations - DEFAULT_HASH_ITERATIONS; // first iteration already done
		for (int i = 0; i < hashIterations; i++) {
			digest.reset();
			hashed = digest.digest(hashed);
		}

		return hashed;
	}

}
