import axios from 'axios';
import Gon from 'gon';
import { camelizeKeys } from 'humps';
import compact from 'lodash/compact';
import groupBy from 'lodash/groupBy';

import { PanelModeCodes } from '@/pages/tournament/ControlPanel';

import { actions } from '../slices';

import Channel from './Channel';

const tournamentId = Gon.getAsset('tournament_id');
const channel = new Channel();

export const setTournamentChannel = (newTournamentId = tournamentId) => {
  const newChannelName = `tournament_admin:${newTournamentId}`;
  channel.setupChannel(newChannelName);
  return channel;
};

const initTournamentChannel = dispatch => {
  const onJoinFailure = err => {
    console.error(err);
    // window.location.reload();
  };

  const onJoinSuccess = response => {
    // dispatch(
    //   actions.setTournamentData({
    //     ...response.tournament,
    //     topPlayerIds: response.topPlayerIds || [],
    //     matches: {},
    //     ranking: response.ranking || { entries: [] },
    //     clans: response.clans || {},
    //     players: {},
    //     showBots: response.tournament.type !== TournamentTypes.show,
    //     channel: { online: true },
    //     playersPageNumber: 1,
    //     playersPageSize: 20,
    //   }),
    // );

    dispatch(actions.updateTournamentPlayers(compact(response.players)));
    dispatch(actions.updateTournamentMatches(compact(response.matches)));
    dispatch(actions.setTournamentTaskList(compact(response.tasksInfo)));
  };

  channel.join().receive('ok', onJoinSuccess).receive('error', onJoinFailure);

  channel.onError(() => {
    dispatch(actions.updateTournamentChannelState(false));
  });
};

// export const soundNotification = notification();

export const connectToTournament = (_machine, newTournamentId) => dispatch => {
  setTournamentChannel(newTournamentId);
  initTournamentChannel(dispatch);

  const handleUpdate = response => {
    dispatch(actions.updateTournamentData(response.tournament));
    dispatch(actions.updateTournamentPlayers(compact(response.players || [])));
    dispatch(actions.updateTournamentMatches(compact(response.matches || [])));
    if (response.tasksInfo) {
      dispatch(actions.setTournamentTaskList(compact(response.tasksInfo)));
    }
  };

  const handleMatchesUpdate = response => {
    dispatch(actions.updateTournamentMatches(compact(response.matches)));
  };

  const handlePlayersUpdate = response => {
    dispatch(actions.updateTournamentPlayers(compact(response.players)));
  };

  const handleTournamentRoundCreated = response => {
    dispatch(actions.updateTournamentData(response.tournament));
  };

  const handleRoundFinished = response => {
    dispatch(
      actions.updateTournamentData({
        ...response.tournament,
        topPlayerIds: response.topPlayerIds,
        playersPageNumber: 1,
        playersPageSize: 20,
      }),
    );

    dispatch(actions.updateTournamentPlayers(compact(response.players)));
    dispatch(actions.updateTopPlayers(compact(response.players)));
  };

  const handleTournamentRestarted = response => {
    dispatch(
      actions.setTournamentData({
        ...response.tournament,
        channel: { online: true },
        playersPageNumber: 1,
        playersPageSize: 20,
        matches: {},
        players: {},
      }),
    );
  };

  const handlePlayerJoined = response => {
    dispatch(actions.addTournamentPlayer(response));
    dispatch(actions.updateTournamentData(response.tournament));
  };

  const handlePlayerLeft = response => {
    dispatch(actions.removeTournamentPlayer(response));
    dispatch(actions.updateTournamentData(response.tournament));
  };

  const handleMatchUpserted = response => {
    dispatch(actions.updateTournamentMatches(compact([response.match])));
    dispatch(actions.updateTournamentPlayers(compact(response.players)));
  };

  const handleTournamentFinished = response => {
    dispatch(actions.updateTournamentData(response.tournament));
  };

  return channel
    .addListener('tournament:update', handleUpdate)
    .addListener('tournament:matches:update', handleMatchesUpdate)
    .addListener('tournament:players:update', handlePlayersUpdate)
    .addListener('tournament:round_created', handleTournamentRoundCreated)
    .addListener('tournament:round_finished', handleRoundFinished)
    .addListener('tournament:player:joined', handlePlayerJoined)
    .addListener('tournament:player:left', handlePlayerLeft)
    .addListener('tournament:match:upserted', handleMatchUpserted)
    .addListener('tournament:restarted', handleTournamentRestarted)
    .addListener('tournament:finished', handleTournamentFinished);
};

// TODO (tournaments): request matches by searched player id
export const uploadPlayers = playerIds => (dispatch, getState) => {
  const state = getState();

  const { isLive, id } = state.tournament;

  if (isLive) {
    channel
      .push('tournament:players:request', { playerIds })
      .receive('ok', response => {
        dispatch(actions.updateTournamentPlayers(response.players));
      })
      .receive('error', error => console.error(error));
  } else {
    const playerIdsStr = playerIds.join(',');

    axios
      .get(`/api/v1/tournaments/${id}/players?player_ids=${playerIdsStr}`, {
        headers: {
          'Content-Type': 'application/json',
          'x-csrf-token': window.csrf_token,
        },
      })
      .then(response => {
        dispatch(actions.updateTournamentPlayers(response.players));
      })
      .catch(error => console.error(error));
  }
};

export const requestMatchesByPlayerId = userId => dispatch => {
  channel
    .push('tournament:matches:request', { playerId: userId })
    .receive('ok', data => {
      dispatch(actions.updateTournamentMatches(data.matches));
      dispatch(actions.updateTournamentPlayers(data.players));
    })
    .receive('error', error => console.error(error));
};

export const uploadPlayersMatches = playerId => (dispatch, getState) => {
  const state = getState();

  const { isLive, id } = state.tournament;

  if (isLive) {
    requestMatchesByPlayerId(playerId)(dispatch);
  } else {
    axios
      .get(`/api/v1/tournaments/${id}/matches?player_id=${playerId}`, {
        headers: {
          'Content-Type': 'application/json',
          'x-csrf-token': window.csrf_token,
        },
      })
      .then(response => {
        dispatch(actions.updateTournamentMatches(response.matches));
      })
      .catch(error => console.error(error));
  }
};

export const createCustomRound = params => {
  channel
    .push('tournament:start_round', params)
    .receive('error', error => console.error(error));
};

export const startTournament = () => {
  channel
    .push('tournament:start', {})
    .receive('error', error => console.error(error));
};

export const cancelTournament = () => dispatch => {
  channel
    .push('tournament:cancel', {})
    .receive('ok', response => {
      dispatch(actions.updateTournamentData(response.tournament));
    })
    .receive('error', error => console.error(error));
};

export const restartTournament = () => {
  channel
    .push('tournament:restart', {})
    .receive('error', error => console.error(error));
};

export const startRoundTournament = () => {
  channel
    .push('tournament:start_round', {})
    .receive('error', error => console.error(error));
};

export const finishRoundTournament = () => {
  channel
    .push('tournament:finish_round', {})
    .receive('error', error => console.error(error));
};

export const toggleVisibleGameResult = gameId => {
  channel
    .push('tournament:toggle_match_visible', { gameId })
    .receive('error', error => console.error(error));
};

export const openUpTournament = () => {
  channel
    .push('tournament:open_up', {})
    .receive('error', error => console.error(error));
};

export const showTournamentResults = () => {
  channel
    .push('tournament:toggle_show_results', {})
    .receive('error', error => console.error(error));
};

export const sendMatchGameOver = matchId => {
  channel
    .push('tournament:match:game_over', { matchId })
    .receive('error', error => console.error(error));
};

export const toggleBanUser = (userId, isBanned) => dispatch => {
  channel
    .push('tournament:ban:player', { userId })
    .receive('ok', () => dispatch(actions.updateTournamentPlayers([{ id: userId, isBanned }])))
    .receive('error', error => console.error(error));
};

export const getResults = (type, params, onSuccess) => () => {
  channel
    .push('tournament:get_results', { params: { type, ...params } })
    .receive('ok', payload => {
      const data = camelizeKeys(payload);

      if (type === PanelModeCodes.topUserByClansMode) {
        const result = Object.values(groupBy(data.results, item => item.clanRank));
        onSuccess(result);
      } else {
        onSuccess(data.results);
      }
    });
};

export const getTask = (taskId, onSuccess) => () => {
  channel
    .push('tournament:get_task', { taskId })
    .receive('ok', payload => {
      const data = camelizeKeys(payload);

      onSuccess(data.descriptionRu);
    });
};
