/*
 * Copyright (C) 2017-2021 HERE Europe B.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.here.msdkuiapp.espresso.impl.views.route.screens

import androidx.test.espresso.Espresso.onView
import androidx.test.espresso.ViewInteraction
import androidx.test.espresso.matcher.ViewMatchers
import androidx.test.espresso.matcher.ViewMatchers.withId
import androidx.test.espresso.matcher.ViewMatchers.withParent
import android.view.View
import com.here.msdkuiapp.R
import com.here.msdkuiapp.espresso.impl.core.CoreMatchers
import com.here.msdkuiapp.espresso.impl.core.CoreMatchers.withIndex
import com.here.msdkuiapp.espresso.impl.testdata.Constants.ROUTE_RESULT_1
import org.hamcrest.Matcher
import org.hamcrest.Matchers
import org.hamcrest.Matchers.allOf

/**
 * Route ViewInteractions selectors
 */
object RouteView {

    /**
     * The [Matcher]<[View]> on route description view
     */
    val onRouteDescriptionView: Matcher<View>
        get() = withId(R.id.route_description_list)

    /**
     * The [ViewInteraction] route description list on route
     */
    val onRouteDescriptionList: ViewInteraction
        get() = onView(withId(R.id.route_description_list))

    /**
     * The [ViewInteraction] maneuver description list on route
     */
    val onRouteManeuversList: ViewInteraction
        get() = onView(withId(R.id.guidance_maneuver_list))

    /**
     * The [ViewInteraction] duration time of the first item in routes list
     */
    val onRouteListItemDuration: ViewInteraction
        get() = onRouteListItemDuration(ROUTE_RESULT_1)

    /**
     * The [ViewInteraction] delay information of the first item in routes list
     */
    val onRouteListItemDelayInformation: ViewInteraction
        get() = onRouteListItemDelayInformation(ROUTE_RESULT_1)

    /**
     * The [ViewInteraction] address & distance details of the first item in routes list
     */
    val onRouteListItemDetails: ViewInteraction
        get() = onRouteListItemDetails(ROUTE_RESULT_1)

    /**
     * The [ViewInteraction] arrival time of the first item in routes list
     */
    val onRouteListItemArrival: ViewInteraction
        get() = onRouteListItemArrival(ROUTE_RESULT_1)

    /**
     * The [ViewInteraction] duration time on route overview
     */
    val onRouteOverviewDuration: ViewInteraction
        get() = onView(allOf(ViewMatchers.withId(R.id.desc_time), withParent(withParent(withId(R.id.description)))))

    /**
     * The [ViewInteraction] delay information time on route overview
     */
    val onRouteOverviewDelayInformation: ViewInteraction
        get() = onView(allOf(withId(R.id.desc_traffic_warning), withParent(withParent(withId(R.id.description)))))

    /**
     * The [ViewInteraction] address & distance details on route overview
     */
    val onRouteOverviewDetails: ViewInteraction
        get() = onView(Matchers.allOf(ViewMatchers.withId(R.id.desc_details), ViewMatchers.withParent(ViewMatchers.withParent(ViewMatchers.withId(R.id.description)))))

    /**
     * The [ViewInteraction] arrival time on route overview
     */
    val onRouteOverviewArrival: ViewInteraction
        get() = onView(Matchers.allOf(ViewMatchers.withId(R.id.desc_arrival), ViewMatchers.withParent(ViewMatchers.withParent(ViewMatchers.withId(R.id.description)))))

    /**
     * The [String] instruction text on maneuver description list
     */
    val onManeuverInstructionText: String
        get() = CoreMatchers.getTextById(R.string.msdkui_arrive)

    /**
     * The [ViewInteraction] line progress bar information on route overview
     */
    val onRouteDescLineBar: ViewInteraction
        get() = onRouteDescLineBar(ROUTE_RESULT_1)

    /**
     * The [ViewInteraction] route item displayed on route overview
     */
    val onRouteOverviewItem: ViewInteraction
        get() = onView(withId(R.id.description))

    /**
     * The [ViewInteraction] destination on route overview
     */
    val onDestinationText: ViewInteraction
        get() = onView(withId(R.id.destination))

    /**
     * The [ViewInteraction] distance on maneuver description list
     */
    fun onManeuverDistance(item: Int): ViewInteraction = onView(withIndex(withId(R.id.maneuver_distance_view), item))

    /**
     * The [ViewInteraction] icon type on maneuver description list
     */
    fun onManeuverIconType(item: Int): ViewInteraction = onView(withIndex(withId(R.id.maneuver_icon_view), item))

    /**
     * The [ViewInteraction] instruction on maneuver description list
     */
    fun onManeuverInstruction(item: Int): ViewInteraction = onView(withIndex(withId(R.id.maneuver_instruction_view), item))

    /**
     * The [ViewInteraction] location address on maneuver description list
     */
    fun onManeuverAddress(item: Int): ViewInteraction = onView(withIndex(withId(R.id.maneuver_address_view), item))

    /**
     * The [ViewInteraction] duration time on route description list
     */
    fun onRouteListItemDuration(item: Int): ViewInteraction = onView(withIndex(withId(R.id.desc_time), item))

    /**
     * The [ViewInteraction] icon type on route description list
     */
    fun onRouteListItemIconType(item: Int): ViewInteraction = onView(withIndex(withId(R.id.desc_type_icon), item))

    /**
     * The [ViewInteraction] delay information on route description list
     */
    fun onRouteListItemDelayInformation(item: Int): ViewInteraction = onView(withIndex(withId(R.id.desc_traffic_warning), item))

    /**
     * The [ViewInteraction] address & distance details on route description list
     */
    fun onRouteListItemDetails(item: Int): ViewInteraction = onView(withIndex(withId(R.id.desc_details), item))

    /**
     * The [ViewInteraction] arrival on route description list
     */
    fun onRouteListItemArrival(item: Int): ViewInteraction = onView(withIndex(withId(R.id.desc_arrival), item))

    /**
     * The [ViewInteraction] line progress bar information on route description list
     */
    fun onRouteDescLineBar(item: Int): ViewInteraction = onView(withIndex(withId(R.id.desc_bar), item))

    /**
     * The [ViewInteraction] for see steps button on route maneuver list.
     */
    val onSeeManeuverSteps: ViewInteraction get() = onView(withId(R.id.see_steps))
}