import React, { useEffect } from 'react'
import isPlainObject from 'lodash.isplainobject'
import isString from 'lodash.isstring'
import isNil from 'lodash.isnil'
import { ITEM_TYPES, VARIANTS } from './DropList.constants'
import { SelectTag } from './DropList.togglers'
import { ListItemUI, EmptyListUI } from './DropList.css'
import Combobox from './DropList.Combobox'
import Select from './DropList.Select'

export function getDropListVariant({
  variant,
  autoSetComboboxAt,
  numberOfItems,
}) {
  return variant.toLowerCase() === VARIANTS.COMBOBOX ||
    (autoSetComboboxAt > 0 && numberOfItems >= autoSetComboboxAt)
    ? Combobox
    : Select
}

// No need to test this helper
/* istanbul ignore next */
export function displayWarnings({
  toggler,
  withMultipleSelection,
  menuCSS,
  tippyOptions,
}) {
  if (
    process.env.NODE_ENV !== 'production' &&
    process.env.NODE_ENV !== 'test'
  ) {
    if (!React.isValidElement(toggler)) {
      console.info(
        'Pass one of the provided togglers or a custom one to the `toggler` prop'
      )
    }
    if (isTogglerOfType(toggler, SelectTag) && withMultipleSelection) {
      console.info(
        'The Select toggler option should not have withMultipleSelection enabled, it has been disabled for you'
      )
    }
    if (!isNil(menuCSS) && tippyOptions.appendTo === undefined) {
      console.error(
        'menuCSS is only needed when using tippyOptions.appendTo to portal the DropList, please use regular styled components if you need custom styles'
      )
    }
  }
}

export function useWarnings(props) {
  useEffect(() => {
    displayWarnings(props)
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [])
}

export function isTogglerOfType(toggler, type) {
  return React.isValidElement(toggler) && toggler.type === type
}

export function itemToString(item) {
  if (isNil(item) || checkIfGroupOrDividerItem(item)) return ''
  // Items can be simple strings
  if (isString(item)) return item

  if (isPlainObject(item)) {
    // Object items should have 'label' or 'value', obtain which one is used per item
    const itemContentKeyName = getItemContentKeyName(item)

    return itemContentKeyName ? item[itemContentKeyName] : ''
  }

  return ''
}

export function parseSelectionFromProps({ withMultipleSelection, selection }) {
  if (withMultipleSelection) {
    return !isNil(selection) ? [].concat(selection) : []
  }

  return !isNil(selection) ? selection : null
}

export function isItemSelected({ item, selectedItem, selectedItems }) {
  if (isNil(selectedItem) && selectedItems.length === 0) return false

  if (isPlainObject(item)) {
    const itemContentKey = getItemContentKeyName(item)
    const itemContent = item[itemContentKey]

    if (!isNil(selectedItem) && selectedItems.length === 0) {
      const selectedItemContentKey = getItemContentKeyName(selectedItem)
      const selectedItemContent = selectedItem[selectedItemContentKey]

      return selectedItemContent === itemContent
    }

    return Boolean(
      selectedItems.find(item => item[itemContentKey] === itemContent)
    )
  }

  return selectedItem === item || selectedItems.includes(item)
}

export function getItemContentKeyName(item) {
  if (objectHasKey(item, 'label')) return 'label'
  if (objectHasKey(item, 'value')) return 'value'

  return undefined
}

export function objectHasKey(obj, key) {
  return isPlainObject(obj) && !isNil(obj[key])
}

export function findItemInArray({ item, arr, key = 'label' }) {
  if (isNil(item)) return undefined

  return arr.find(i => {
    if (isPlainObject(i)) {
      return i[key] === item[key]
    }
    if (isPlainObject(item)) {
      return i === item[key]
    }
    return i === item
  })
}

export function removeItemFromArray({ item, arr, key = 'label' }) {
  return arr.filter(i => {
    if (isPlainObject(i)) {
      return i[key] !== item[key]
    }
    if (isPlainObject(item)) {
      return i !== item[key]
    }
    return i !== item
  })
}

export function isItemADivider(item) {
  return objectHasKey(item, 'type') && item.type === ITEM_TYPES.DIVIDER
}

export function isItemAGroup(item) {
  return objectHasKey(item, 'type') && item.type === ITEM_TYPES.GROUP
}

export function isItemAGroupLabel(item) {
  return objectHasKey(item, 'type') && item.type === ITEM_TYPES.GROUP_LABEL
}

export function isItemInert(item) {
  return objectHasKey(item, 'type') && item.type === ITEM_TYPES.INERT
}

export function isItemAction(item) {
  return objectHasKey(item, 'type') && item.type === ITEM_TYPES.ACTION
}

export function isItemRegular(item) {
  return (
    !isItemADivider(item) && !isItemAGroup(item) && !isItemAGroupLabel(item)
  )
}

export function flattenListItems(listItems) {
  return listItems.reduce((accumulator, listItem) => {
    const contentKey = getItemContentKeyName(listItem)

    if (isItemAGroup(listItem)) {
      const itemsInGroup = listItem.items.map(item => ({
        ...item,
        group: listItem[contentKey],
      }))

      return itemsInGroup.length > 0
        ? accumulator
            .concat({
              type: ITEM_TYPES.GROUP_LABEL,
              [contentKey]: listItem[contentKey],
            })
            .concat(itemsInGroup)
        : accumulator
    }

    return accumulator.concat(listItem)
  }, [])
}

export function renderListContents({
  customEmptyList,
  inputValue,
  items,
  renderListItem,
}) {
  const isEmptyList = items.length === 0

  if (!isEmptyList) {
    return items.map(renderListItem)
  }

  if (isEmptyList && customEmptyList) {
    return React.isValidElement(customEmptyList) ? (
      React.cloneElement(customEmptyList)
    ) : (
      <EmptyListUI>No items</EmptyListUI>
    )
  }

  if (isEmptyList && !customEmptyList && isNil(inputValue)) {
    return <EmptyListUI>No items</EmptyListUI>
  }

  return <ListItemUI>No results for {inputValue}</ListItemUI>
}

// No need to test this helper
/* istanbul ignore next */
export function requiredItemPropsCheck(props, propName, componentName) {
  if (!props.label && !props.value) {
    return new Error(
      `One of 'label' or 'value' is required by '${componentName}' component.`
    )
  }
}

export function checkIfGroupOrDividerItem(item) {
  return isItemADivider(item) || isItemAGroup(item) || isItemAGroupLabel(item)
}

export function isItemHighlightable(item) {
  return (
    !checkIfGroupOrDividerItem(item) && !item.isDisabled && !isItemInert(item)
  )
}

export function getEnabledItemIndex({
  currentHighlightedIndex,
  nextHighlightedIndex,
  items,
  arrowKey,
}) {
  // When nextHighlightedIndex === -1 it means there are no items to be highlighted
  // like in the case of a combobox being filtered to "no results"
  if (nextHighlightedIndex === -1) return -1

  const isNextIndexItemHighlightable = isItemHighlightable(
    items[nextHighlightedIndex]
  )

  if (
    isNextIndexItemHighlightable &&
    currentHighlightedIndex !== nextHighlightedIndex
  ) {
    return nextHighlightedIndex
  }

  const highlightableItems = items.filter(item => {
    return (
      !checkIfGroupOrDividerItem(item) && !item.isDisabled && !isItemInert(item)
    )
  })

  if (highlightableItems.length === 1) {
    return items.findIndex(isItemHighlightable)
  }

  let newNextHighlightedIndex
  const firstIndex = 0
  const lastIndex = items.length - 1

  if (arrowKey === 'UP') {
    const isNextIndexAfterFirst = nextHighlightedIndex - 1 >= firstIndex

    newNextHighlightedIndex = isNextIndexAfterFirst
      ? nextHighlightedIndex - 1
      : lastIndex
  } else {
    const isNextIndexBeforeLast = nextHighlightedIndex + 1 <= lastIndex

    newNextHighlightedIndex = isNextIndexBeforeLast
      ? nextHighlightedIndex + 1
      : firstIndex
  }

  return getEnabledItemIndex({
    currentHighlightedIndex,
    nextHighlightedIndex: newNextHighlightedIndex,
    items,
    arrowKey,
  })
}

export function getMenuWidth(variant, menuWidth) {
  if (!isNil(menuWidth)) return menuWidth

  return variant.toLowerCase() === 'combobox' ? '220px' : '200px'
}

/**
 * Emphasize the first matched section of a string that respects uppercase
 * @param {string} str The string to process
 * @param {string} subsection The section of the string to emphasize
 * @returns string
 */
export function emphasizeSubstring(str, subsection, htmlTag = 'strong') {
  const strUppercased = str.toUpperCase()
  const subsectionUppercased = subsection.toUpperCase()
  const idx = strUppercased.indexOf(subsectionUppercased)

  if (!subsectionUppercased || idx === -1) {
    return str
  }

  const l = subsectionUppercased.length

  return (
    str.substring(0, idx) +
    `<${htmlTag}>` +
    str.substring(idx, l + idx) +
    `</${htmlTag}>` +
    str.substring(idx + l)
  )
}
