﻿// Georgy Treshchev 2024.

#include "SpeechRecognizerThread.h"

#include "Misc/EngineVersionComparison.h"

#include "SpeechRecognizerDefines.h"
#include "SpeechRecognizerTypes.h"
#include "Containers/StringConv.h"

#include "AudioResampler.h"
#include "HAL/RunnableThread.h"
#include "SampleBuffer.h"
#include "SpeechRecognizerModel.h"

#include "Async/Async.h"
#include "AudioThread.h"
#include "SpeechRecognizerSettings.h"

#include "SpeechRecognizerPrivate.h"
#include "Engine/AssetManager.h"

namespace RSR_StringUtils
{
	char* Strdup(const char* StringSource, SIZE_T StringLength)
	{
		if (!StringSource || StringLength == 0)
		{
			return nullptr;
		}

		char* StringDestination = static_cast<char*>(FMemory::Malloc((StringLength + 1) * sizeof(char)));
		if (StringDestination == nullptr)
		{
			return nullptr;
		}

		TCString<char>::Strcpy(StringDestination, StringLength + 1, StringSource);
		return StringDestination;
	}

	void Free(char* String)
	{
		if (String)
		{
			FMemory::Free(String);
		}
	}
}

/**
 * Called when a new text segment is generated by the whisper
 *
 * @param WhisperContext The whisper context associated with the text segment
 * @param WhisperState The whisper state associated with the text segment
 * @param NewSegmentCount The number of new text segments that have been generated
 * @param UserData User-defined data pointer, which should be a pointer to a WhisperSpeechRecognizerUserData struct
 */
void WhisperNewTextSegmentCallback(whisper_context* WhisperContext, whisper_state* WhisperState, int NewSegmentCount, void* UserData)
{
	if (!WhisperContext || !WhisperState || !UserData)
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Warning, TEXT("Invalid parameters passed to the whisper new text segment callback"));
		return;
	}

	TSharedPtr<FSpeechRecognizerThread> SpeechRecognizerSharedPtr = static_cast<FWhisperSpeechRecognizerUserData*>(UserData)->SpeechRecognizerWeakPtr.Pin();
	if (!SpeechRecognizerSharedPtr.IsValid() || SpeechRecognizerSharedPtr->GetIsStopped() || SpeechRecognizerSharedPtr->GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Warning, TEXT("Aborting whisper new text segment callback due to stop request"));
		return;
	}

	// We should still process the text segment even if the speech recognizer is stopped
	/*if (SpeechRecognizerSharedPtr->GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Warning, TEXT("Speech recognizer is stopped, text segment will not be processed"));
		return;
	}*/

	const int32 TotalSegmentCount = whisper_full_n_segments(WhisperContext);
	const int32 StartIndex = TotalSegmentCount - NewSegmentCount;

	for (int32 Index = StartIndex; Index < TotalSegmentCount; ++Index)
	{
		const char* TextPerSegment = whisper_full_get_segment_text(WhisperContext, static_cast<int>(Index));
		// StringCast from UTF8CHAR to TCHAR is not supported in UE 5.0 and older
		FString TextPerSegment_String =
#if UE_VERSION_OLDER_THAN(5, 1, 0)
		UTF8_TO_TCHAR(TextPerSegment);
#else
			[&TextPerSegment]()
			{
				auto TextPerSegment_TCHAR = StringCast<TCHAR>((const UTF8CHAR*)TextPerSegment);
				return FString(TextPerSegment_TCHAR.Get());
			}();
#endif

		AsyncTask(ENamedThreads::GameThread, [SpeechRecognizerSharedPtr, TextPerSegment_String = MoveTemp(TextPerSegment_String)]() mutable
		{
			if (SpeechRecognizerSharedPtr.IsValid())
			{
				UE_LOG(LogRuntimeSpeechRecognizer, Log, TEXT("Recognized text segment: \"%s\""), *TextPerSegment_String);
				SpeechRecognizerSharedPtr->OnRecognizedTextSegment.Broadcast(TextPerSegment_String);
			}
		});
	}
}

/**
 * Called every time before the encoder starts. Implements the aborting mechanism for the whisper encoder
 *
 * @param WhisperContext The whisper context associated with the user data
 * @param WhisperState The whisper state associated with the user data
 * @param UserData User-defined data pointer, which should be a pointer to a WhisperSpeechRecognizerUserData struct
 */
bool WhisperEncoderBeginCallback(whisper_context* WhisperContext, whisper_state* WhisperState, void* UserData)
{
	if (!WhisperContext || !WhisperState || !UserData)
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Warning, TEXT("Aborting whisper recognition due to invalid context or user data"));
		return false;
	}

	TSharedPtr<FSpeechRecognizerThread> SpeechRecognizerSharedPtr = static_cast<FWhisperSpeechRecognizerUserData*>(UserData)->SpeechRecognizerWeakPtr.Pin();
	if (!SpeechRecognizerSharedPtr.IsValid() || SpeechRecognizerSharedPtr->GetIsStopped() || SpeechRecognizerSharedPtr->GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Warning, TEXT("Aborting whisper recognition due to stop request (encoder begin callback)"));
		return false;
	}

	return true;
}

/**
 * Called every time before ggml computation starts. Implements the aborting mechanism for the whisper encoder
 *
 * @param UserData User-defined data pointer, which should be a pointer to a WhisperSpeechRecognizerUserData struct
 */
bool WhisperEncoderAbortCallback(void* UserData)
{
	if (!UserData)
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Warning, TEXT("Aborting whisper recognition due to invalid context or user data"));
		return true;
	}

	TSharedPtr<FSpeechRecognizerThread> SpeechRecognizerSharedPtr = static_cast<FWhisperSpeechRecognizerUserData*>(UserData)->SpeechRecognizerWeakPtr.Pin();
	if (!SpeechRecognizerSharedPtr.IsValid() || SpeechRecognizerSharedPtr->GetIsStopped() || SpeechRecognizerSharedPtr->GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Warning, TEXT("Aborting whisper recognition due to stop request (encoder abort callback)"));
		return true;
	}

	return false;
}

void WhisperProgressCallback(whisper_context* WhisperContext, whisper_state* WhisperState, int Progress, void* UserData)
{
	if (!UserData)
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Warning, TEXT("Aborting whisper recognition due to invalid context or user data"));
		return;
	}

	TSharedPtr<FSpeechRecognizerThread> SpeechRecognizerSharedPtr = static_cast<FWhisperSpeechRecognizerUserData*>(UserData)->SpeechRecognizerWeakPtr.Pin();
	if (!SpeechRecognizerSharedPtr.IsValid() || SpeechRecognizerSharedPtr->GetIsStopped() || SpeechRecognizerSharedPtr->GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Warning, TEXT("Aborting whisper progress callback due to stop request"));
		return;
	}

	// There is a bug in the whisper library where the progress callback is sometimes called with a value larger than 100
	Progress = FMath::Clamp(Progress, 0, 100);
	AsyncTask(ENamedThreads::GameThread, [SpeechRecognizerSharedPtr, Progress]() mutable
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Log, TEXT("Speech recognition progress: %d"), Progress);
		SpeechRecognizerSharedPtr->LastProgress = Progress;
		SpeechRecognizerSharedPtr->OnRecognitionProgress.Broadcast(Progress);
	});
}

FWhisperSpeechRecognizerState::FWhisperSpeechRecognizerState()
	: WhisperContext(nullptr)
, WhisperParameters(nullptr)
{}

bool FWhisperSpeechRecognizerState::Init(uint8* BulkDataPtr, int64 BulkDataSize, TSharedPtr<FSpeechRecognizerThread> SpeechRecognizerPtr)
{
	WhisperContext = whisper_init_from_buffer_with_params(BulkDataPtr, BulkDataSize, whisper_context_default_params());
	if (!WhisperContext)
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Failed to create whisper context from buffer"));
		return false;
	}

	WhisperParameters = new whisper_full_params(whisper_full_default_params(whisper_sampling_strategy::WHISPER_SAMPLING_GREEDY));
	if (!WhisperParameters)
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Failed to create whisper parameters"));
		return false;
	}

	WhisperParameters->initial_prompt = nullptr;
	WhisperUserData = FWhisperSpeechRecognizerUserData{SpeechRecognizerPtr};
	return true;
}

void FWhisperSpeechRecognizerState::Release()
{
	FScopeLock Lock(&ReleaseGuard);
	if (WhisperContext)
	{
		whisper_free(WhisperContext);
		WhisperContext = nullptr;
	}

	ClearInitialPrompt();

	if (WhisperParameters)
	{
		delete WhisperParameters;
		WhisperParameters = nullptr;
	}

	WhisperUserData = FWhisperSpeechRecognizerUserData();
}

void FWhisperSpeechRecognizerState::ClearInitialPrompt()
{
	if (WhisperParameters && WhisperParameters->initial_prompt)
	{
		RSR_StringUtils::Free((char*)WhisperParameters->initial_prompt);
		WhisperParameters->initial_prompt = nullptr;
	}
}

FSpeechRecognitionParameters FSpeechRecognitionParameters::GetNonStreamingDefaults()
{
	// These are the default values for the whisper.cpp library
	FSpeechRecognitionParameters Parameters;
	Parameters.bNoContext = false;
	Parameters.bSingleSegment = false;
	Parameters.MaxTokens = 0;
	Parameters.AudioContextSize = 0;
	Parameters.TemperatureToIncrease = 0.4f;
	return Parameters;
}

FSpeechRecognitionParameters FSpeechRecognitionParameters::GetStreamingDefaults()
{
	// Taken from https://github.com/ggerganov/whisper.cpp/blob/master/examples/stream.wasm/emscripten.cpp
	FSpeechRecognitionParameters Parameters;
	Parameters.bNoContext = true;
	Parameters.bSingleSegment = true;
	Parameters.MaxTokens = 32;
	Parameters.AudioContextSize = 768;
	Parameters.TemperatureToIncrease = -1.0f;
	return Parameters;
}

void FSpeechRecognitionParameters::SetNonStreamingDefaults()
{
	*this = GetNonStreamingDefaults();
}

void FSpeechRecognitionParameters::SetStreamingDefaults()
{
	*this = GetStreamingDefaults();
}

void FSpeechRecognitionParameters::FillWhisperStateParameters(FWhisperSpeechRecognizerState& WhisperState) const
{
	// Disable all prints
	{
		WhisperState.WhisperParameters->print_realtime = false;
		WhisperState.WhisperParameters->print_progress = false;
		WhisperState.WhisperParameters->print_timestamps = false;
		WhisperState.WhisperParameters->print_special = false;
	}

	WhisperState.WhisperParameters->translate = bTranslateToEnglish;

	WhisperState.WhisperParameters->no_context = bNoContext;
	WhisperState.WhisperParameters->single_segment = bSingleSegment;
	WhisperState.WhisperParameters->max_tokens = MaxTokens;
	WhisperState.WhisperParameters->audio_ctx = AudioContextSize;
	WhisperState.WhisperParameters->temperature_inc = TemperatureToIncrease;
	WhisperState.WhisperParameters->entropy_thold = EntropyThreshold;

	WhisperState.WhisperParameters->language = EnumToString(Language);
	WhisperState.WhisperParameters->n_threads = NumOfThreads > 0 ? NumOfThreads : (FPlatformProcess::SupportsMultithreading() ? FMath::Min(6, FPlatformMisc::NumberOfCoresIncludingHyperthreads()) : 1);

	WhisperState.WhisperParameters->suppress_blank = bSuppressBlank;
	WhisperState.WhisperParameters->suppress_non_speech_tokens = bSuppressNonSpeechTokens;

	WhisperState.WhisperParameters->beam_search.beam_size = BeamSize;

	WhisperState.ClearInitialPrompt();

	if (InitialPrompt.Len() > 0)
	{
		// StringCast to UTF8CHAR is not supported in UE 4.27 and older
		// We should use strdup to extend the lifetime of the string since TCHAR_TO_UTF8 and StringCast return temporary pointers
		// (The memory is freed on FWhisperSpeechRecognizerState::Release)
#if UE_VERSION_OLDER_THAN(5, 0, 0)
		WhisperState.WhisperParameters->initial_prompt = RSR_StringUtils::Strdup(TCHAR_TO_UTF8(*InitialPrompt), InitialPrompt.Len());
#else
		const auto InitialPrompt_UTF8 = StringCast<UTF8CHAR>(*InitialPrompt);
		WhisperState.WhisperParameters->initial_prompt = RSR_StringUtils::Strdup((const char*)InitialPrompt_UTF8.Get(), InitialPrompt_UTF8.Length());
#endif
	}

	// Setting up the new segment callback, which is called on every new recognized text segment
	{
		WhisperState.WhisperParameters->new_segment_callback = WhisperNewTextSegmentCallback;
		WhisperState.WhisperParameters->new_segment_callback_user_data = &WhisperState.WhisperUserData;
	}

	// Setting up the abort mechanism callback, which is called every time before the encoder starts
	{
		WhisperState.WhisperParameters->encoder_begin_callback = WhisperEncoderBeginCallback;
		WhisperState.WhisperParameters->encoder_begin_callback_user_data = &WhisperState.WhisperUserData;
	}

	// Setting up the abort mechanism callback, which is called every time before ggml computation starts
	{
		WhisperState.WhisperParameters->abort_callback = WhisperEncoderAbortCallback;
		WhisperState.WhisperParameters->abort_callback_user_data = &WhisperState.WhisperUserData;
	}

	// Setting up the progress callback, which is called every time the progress changes
	{
		WhisperState.WhisperParameters->progress_callback = WhisperProgressCallback;
		WhisperState.WhisperParameters->progress_callback_user_data = &WhisperState.WhisperUserData;
	}
}

bool FSpeechRecognizerThread::FPendingAudioData::AddAudio(Audio::FAlignedFloatBuffer&& AudioData, float SampleRate, uint32 NumOfChannels)
{
	if (SampleRate <= 0.0f || NumOfChannels <= 0)
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Invalid sample rate (%f) or number of channels (%d). Both must be greater than 0"), SampleRate, NumOfChannels);
		return false;
	}
	FScopeLock Lock(&DataGuard);
	AudioDataMap.FindOrAdd(TTuple<float, uint32>{SampleRate, NumOfChannels}).Append(MoveTemp(AudioData));
	RecalculateTotalMixedAndResampledSize();
	return true;
}

int64 FSpeechRecognizerThread::FPendingAudioData::GetTotalMixedAndResampledSize() const
{
	FScopeLock Lock(&DataGuard);
	return TotalMixedAndResampledSize;
}

bool FSpeechRecognizerThread::FPendingAudioData::GetMixedAndResampledAudio(Audio::FAlignedFloatBuffer& OutPCMData)
{
	FScopeLock Lock(&DataGuard);
	for (auto AudioDataPair : AudioDataMap)
	{
		const int64 OriginalNumSamples = AudioDataPair.Get<1>().Num();
		const float OriginalSampleRate = AudioDataPair.Get<0>().Get<0>();
		const uint32 OriginalNumOfChannels = AudioDataPair.Get<0>().Get<1>();
		Audio::FAlignedFloatBuffer& PCMData = AudioDataPair.Get<1>();

		// Resampling to WHISPER_SAMPLE_RATE (16kHz by default) if needed
		if (static_cast<uint32>(OriginalSampleRate) != WHISPER_SAMPLE_RATE)
		{
			Audio::FAlignedFloatBuffer ResampledPCMData;

			const Audio::FResamplingParameters ResampleParameters = {
				Audio::EResamplingMethod::Linear,
				static_cast<int32>(OriginalNumOfChannels),
				static_cast<float>(OriginalSampleRate),
				static_cast<float>(WHISPER_SAMPLE_RATE),
				PCMData
			};

			ResampledPCMData.AddUninitialized(Audio::GetOutputBufferSize(ResampleParameters));
			Audio::FResamplerResults ResampleResults;
			ResampleResults.OutBuffer = &ResampledPCMData;

			if (!Audio::Resample(ResampleParameters, ResampleResults))
			{
				UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Failed to resample audio data from %f to %f"), OriginalSampleRate, static_cast<float>(WHISPER_SAMPLE_RATE));
				return false;
			}

			PCMData = MoveTemp(ResampledPCMData);
		}

		// Reducing the number of channels to 1 if needed
		if (OriginalNumOfChannels != 1)
		{
			Audio::TSampleBuffer<float> PCMSampleBuffer(PCMData, OriginalNumOfChannels, OriginalSampleRate);
			{
				PCMSampleBuffer.MixBufferToChannels(1);
			}
			Audio::FAlignedFloatBuffer WaveDataTemp = Audio::FAlignedFloatBuffer(PCMSampleBuffer.GetData(), PCMSampleBuffer.GetNumSamples());
			PCMData = MoveTemp(WaveDataTemp);
		}
		
		OutPCMData.Append(MoveTemp(PCMData));
	}
	AudioDataMap.Empty();
	TotalMixedAndResampledSize = 0;
	return true;
}

void FSpeechRecognizerThread::FPendingAudioData::RecalculateTotalMixedAndResampledSize()
{
	constexpr float RequiredSampleRate = WHISPER_SAMPLE_RATE;
	constexpr uint32 RequiredNumOfChannels = 1;

	int64 TotalSize = 0;
	for (auto AudioDataPair : AudioDataMap)
	{
		const int64 OriginalNumSamples = AudioDataPair.Get<1>().Num();
		const float OriginalSampleRate = AudioDataPair.Get<0>().Get<0>();
		const uint32 OriginalNumOfChannels = AudioDataPair.Get<0>().Get<1>();

		// Calculate the ratio between the original and required sample rates
		const float SampleRateRatio = OriginalSampleRate / RequiredSampleRate;

		// Calculate the ratio between the original and required number of channels
		const float NumChannelsRatio = static_cast<float>(OriginalNumOfChannels) / RequiredNumOfChannels;

		// Calculate the estimated number of samples after resampling/remixing
		TotalSize += static_cast<int64>(OriginalNumSamples / SampleRateRatio / NumChannelsRatio);
	}
	TotalMixedAndResampledSize = TotalSize;
}

FSpeechRecognizerThread::FSpeechRecognizerThread()
	: bIsStopped(true)
, bIsFinished(true)
, bIsStopping(false)
{
	whisper_log_set([](enum ggml_log_level Level, const char* Text, void* UserData)
	{
		if (Level == GGML_LOG_LEVEL_ERROR)
		{
			UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("%s"), *FString(Text));
		}
		else if (Level == GGML_LOG_LEVEL_WARN)
		{
			UE_LOG(LogRuntimeSpeechRecognizer, Warning, TEXT("%s"), *FString(Text));
		}
		else
		{
			UE_LOG(LogRuntimeSpeechRecognizer, Log, TEXT("%s"), *FString(Text));
		}
	}, nullptr);
}

FSpeechRecognizerThread::~FSpeechRecognizerThread()
{
	ReleaseMemory();
}

TFuture<bool> FSpeechRecognizerThread::StartThread()
{
	if (!GetIsStopped())
	{
		const FString ShortErrorMessage = TEXT("Thread start failed");
		const FString LongErrorMessage = TEXT("Unable to start an already running thread");
		ReportError(ShortErrorMessage, LongErrorMessage);
		return MakeFulfilledPromise<bool>(false).GetFuture();
	}

	if (GetIsStopping())
	{
		const FString ShortErrorMessage = TEXT("Thread start failed");
		const FString LongErrorMessage = TEXT("Unable to start a thread that is stopping");
		ReportError(ShortErrorMessage, LongErrorMessage);
		return MakeFulfilledPromise<bool>(false).GetFuture();
	}

	if (!DoesSharedInstanceExist())
	{
		const FString ShortErrorMessage = TEXT("Recognizer initialization failed");
		const FString LongErrorMessage = TEXT("Speech recognizer can only be initialized as being created as shared instance");
		ReportError(ShortErrorMessage, LongErrorMessage);
		return MakeFulfilledPromise<bool>(false).GetFuture();
	}

	TSharedPtr<FSpeechRecognizerThread> ThisShared = AsShared();
	if (!ThisShared.IsValid())
	{
		const FString ShortErrorMessage = TEXT("Failed to get shared instance");
		const FString LongErrorMessage = TEXT("Failed to get shared instance of speech recognizer");
		ReportError(ShortErrorMessage, LongErrorMessage);
		return MakeFulfilledPromise<bool>(false).GetFuture();
	}

	// Check if the language is valid
	{
		const char* LanguageString = EnumToString(RecognitionParameters.Language);
		if (RecognitionParameters.Language != ESpeechRecognizerLanguage::Auto && whisper_lang_id(LanguageString) == -1)
		{
			const FString ShortErrorMessage = TEXT("Language identification failed");
			const FString LongErrorMessage = FString::Printf(TEXT("Failed to identify the language '%hs'"), LanguageString);
			ReportError(ShortErrorMessage, LongErrorMessage);
			return MakeFulfilledPromise<bool>(false).GetFuture();
		}
	}

	if (StartThreadPromise.IsValid())
	{
		const FString ShortErrorMessage = TEXT("Thread start failed");
		const FString LongErrorMessage = TEXT("Unable to start a thread that is already starting");
		ReportError(ShortErrorMessage, LongErrorMessage);
		return MakeFulfilledPromise<bool>(false).GetFuture();
	}

	StartThreadPromise = MakeUnique<TPromise<bool>>();

	auto SetStartThreadPromiseValue = [](TSharedRef<FSpeechRecognizerThread> ThisShared, bool bSuccess)
	{
		ThisShared->StartThreadPromise->SetValue(bSuccess);
		ThisShared->StartThreadPromise.Reset();
	};

	auto OnLanguageModelLoaded = [ThisShared, SetStartThreadPromiseValue](bool bSuccess, uint8* ModelBulkDataPtr, int64 ModelBulkDataSize)
	{
		if (!ThisShared.IsValid())
		{
			UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Failed to get shared instance"));
			return;
		}

		if (!bSuccess)
		{
			SetStartThreadPromiseValue(ThisShared.ToSharedRef(), false);
			return;
		}

		if (!ThisShared->WhisperState.Init(ModelBulkDataPtr, ModelBulkDataSize, ThisShared))
		{
			const FString ShortErrorMessage = TEXT("Recognizer initialization failed");
			const FString LongErrorMessage = TEXT("Failed to initialize whisper from the language model");
			ThisShared->ReportError(ShortErrorMessage, LongErrorMessage);
			SetStartThreadPromiseValue(ThisShared.ToSharedRef(), false);
			FMemory::Free(ModelBulkDataPtr);
			return;
		}

		FMemory::Free(ModelBulkDataPtr);

		// Ensure that the language model supports the requested features
		{
			const bool bMultilingual = whisper_is_multilingual(ThisShared->WhisperState.WhisperContext) != 0;
			if (ThisShared->RecognitionParameters.Language == ESpeechRecognizerLanguage::Auto && !bMultilingual)
			{
				const FString ShortErrorMessage = TEXT("Automatic language detection failed");
				const FString LongErrorMessage = TEXT("The selected language model does not support multilingual recognition therefore automatic language detection is not possible");
				ThisShared->ReportError(ShortErrorMessage, LongErrorMessage);
				SetStartThreadPromiseValue(ThisShared.ToSharedRef(), false);
				return;
			}

			if (ThisShared->RecognitionParameters.bTranslateToEnglish && !bMultilingual)
			{
				const FString ShortErrorMessage = TEXT("Translation failed");
				const FString LongErrorMessage = TEXT("The selected language model does not support multilingual recognition therefore translation is not possible");
				ThisShared->ReportError(ShortErrorMessage, LongErrorMessage);
				SetStartThreadPromiseValue(ThisShared.ToSharedRef(), false);
				return;
			}
		}

		ThisShared->Thread.Reset();

		ThisShared->bIsStopping.AtomicSet(false);
		ThisShared->RecognitionParameters.FillWhisperStateParameters(ThisShared->WhisperState);
		ThisShared->bIsStopped.AtomicSet(false);
		ThisShared->bIsFinished.AtomicSet(true);

		FRunnableThread* ThreadPtr = FRunnableThread::Create(ThisShared.Get(), TEXT("SpeechRecognizerThread"), 0, TPri_Highest, FPlatformAffinity::GetTaskGraphHighPriorityTaskMask());
		if (!ThreadPtr)
		{
			const FString ShortErrorMessage = TEXT("Thread creation failed");
			const FString LongErrorMessage = TEXT("Failed to create the thread for the speech recognizer");
			ThisShared->ReportError(ShortErrorMessage, LongErrorMessage);
			SetStartThreadPromiseValue(ThisShared.ToSharedRef(), false);
			return;
		}

		ThisShared->Thread.Reset(ThreadPtr);
		SetStartThreadPromiseValue(ThisShared.ToSharedRef(), true);
	};

	LoadLanguageModel(OnLanguageModelLoaded);
	return StartThreadPromise->GetFuture();
}

void FSpeechRecognizerThread::StopThread()
{
	if (DoesSharedInstanceExist())
	{
		TSharedPtr<FSpeechRecognizerThread> ThisShared = AsShared();
		AsyncTask(ENamedThreads::AnyBackgroundHiPriTask, [ThisShared]() mutable
		{
			if (ThisShared)
			{
				ThisShared->ReleaseMemory();
			}
		});
	}
}

void FSpeechRecognizerThread::ProcessPCMData(Audio::FAlignedFloatBuffer PCMData, float SampleRate, uint32 NumOfChannels, bool bLast)
{
	if (GetIsStopped())
	{
		const FString ShortErrorMessage = TEXT("Audio processing failed");
		const FString LongErrorMessage = TEXT("The audio data could not be processed to the recognizer since the thread is stopped");
		ReportError(ShortErrorMessage, LongErrorMessage);
		return;
	}

	if (GetIsStopping())
	{
		const FString ShortErrorMessage = TEXT("Audio processing failed");
		const FString LongErrorMessage = TEXT("The audio data could not be processed to the recognizer since the thread is stopping");
		ReportError(ShortErrorMessage, LongErrorMessage);
		return;
	}

	if (SampleRate <= 0.0f || NumOfChannels <= 0)
	{
		const FString ShortErrorMessage = TEXT("Audio processing failed");
		const FString LongErrorMessage = TEXT("Invalid sample rate or number of channels");
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Invalid sample rate (%f) or number of channels (%d). Both must be greater than 0"), SampleRate, NumOfChannels);
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("P.S. Please make sure that the SampleRate pin of the ProcessPCMData function is properly connected, as this is a common mistake"));
		return;
	}

	// Make sure to process the data in background thread
	if (IsInGameThread())
	{
		AsyncTask(ENamedThreads::AnyBackgroundHiPriTask, [SpeechRecognizerSharedPtr = WhisperState.WhisperUserData.SpeechRecognizerWeakPtr.Pin(), PCMData = MoveTemp(PCMData), SampleRate, NumOfChannels, bLast]() mutable
		{
			if (!SpeechRecognizerSharedPtr.IsValid())
			{
				UE_LOG(LogRuntimeSpeechRecognizer, Warning, TEXT("Failed to process the audio data since the thread worker is invalid"));
				return;
			}
			SpeechRecognizerSharedPtr->ProcessPCMData(MoveTemp(PCMData), SampleRate, NumOfChannels, bLast);
		});
		return;
	}

	if (bLast)
	{
		if (!PendingAudio.AddAudio(MoveTemp(PCMData), SampleRate, NumOfChannels))
		{
			const FString ShortErrorMessage = TEXT("Audio processing failed");
			const FString LongErrorMessage = TEXT("Failed to add the audio data to the pending audio");
			ReportError(ShortErrorMessage, LongErrorMessage);
			return;
		}
		const int32 NumOfQueuedSamples = PendingAudio.GetTotalMixedAndResampledSize();
		Audio::FAlignedFloatBuffer PendingAudioData;
		if (!PendingAudio.GetMixedAndResampledAudio(PendingAudioData))
		{
			const FString ShortErrorMessage = TEXT("Audio processing failed");
			const FString LongErrorMessage = TEXT("The audio data could not be processed to the recognizer since the pending audio data could not be mixed and resampled");
			ReportError(ShortErrorMessage, LongErrorMessage);
			return;
		}
		AudioQueue.Enqueue(MoveTemp(PendingAudioData));
		UE_LOG(LogRuntimeSpeechRecognizer, Log, TEXT("Enqueued audio data from the pending audio to the queue of the speech recognizer as the last data (num of samples: %d)"), NumOfQueuedSamples);
	}
	else if (RecognitionParameters.StepSizeMs > 0)
	{
		// Calculate the number of samples per step
		const int32 NumOfSamplesPerStep = (1e-3 * RecognitionParameters.StepSizeMs) * WHISPER_SAMPLE_RATE;

		// If pending audio is insufficient to fill the step size, append new data until sufficient
		if (PCMData.Num() + PendingAudio.GetTotalMixedAndResampledSize() < NumOfSamplesPerStep)
		{
			if (!PendingAudio.AddAudio(MoveTemp(PCMData), SampleRate, NumOfChannels))
			{
				const FString ShortErrorMessage = TEXT("Audio processing failed");
				const FString LongErrorMessage = TEXT("Failed to add the audio data to the pending audio");
				ReportError(ShortErrorMessage, LongErrorMessage);
				return;
			}
			UE_LOG(LogRuntimeSpeechRecognizer, Log, TEXT("Pending audio data instead of enqueuing it since it is not enough to fill the step size (pending: %lld, num of samples per step: %d)"), PendingAudio.GetTotalMixedAndResampledSize(), NumOfSamplesPerStep);
		}
		else
		{
			if (!PendingAudio.AddAudio(MoveTemp(PCMData), SampleRate, NumOfChannels))
			{
				const FString ShortErrorMessage = TEXT("Audio processing failed");
				const FString LongErrorMessage = TEXT("Failed to add the audio data to the pending audio");
				ReportError(ShortErrorMessage, LongErrorMessage);
				return;
			}
			const int32 NumOfQueuedSamples = PendingAudio.GetTotalMixedAndResampledSize();
			Audio::FAlignedFloatBuffer PendingAudioData;
			if (!PendingAudio.GetMixedAndResampledAudio(PendingAudioData))
			{
				const FString ShortErrorMessage = TEXT("Audio processing failed");
				const FString LongErrorMessage = TEXT("The audio data could not be processed to the recognizer since the pending audio data could not be mixed and resampled");
				ReportError(ShortErrorMessage, LongErrorMessage);
				return;
			}
			AudioQueue.Enqueue(MoveTemp(PendingAudioData));
			UE_LOG(LogRuntimeSpeechRecognizer, Log, TEXT("Enqueued audio data from the pending audio to the queue of the speech recognizer (num of samples: %d)"), NumOfQueuedSamples);
		}
	}
	else
	{
		const int32 NumOfQueuedSamples = PCMData.Num();
		AudioQueue.Enqueue(MoveTemp(PCMData));
		UE_LOG(LogRuntimeSpeechRecognizer, Log, TEXT("Enqueued audio data from the pending audio to the queue of the speech recognizer as the last data (num of samples: %d)"), NumOfQueuedSamples);
	}
}

void FSpeechRecognizerThread::ForceProcessPendingAudioData()
{
	if (GetIsStopped())
	{
		const FString ShortErrorMessage = TEXT("Audio processing failed");
		const FString LongErrorMessage = TEXT("The audio data could not be processed to the recognizer since the thread is stopped");
		ReportError(ShortErrorMessage, LongErrorMessage);
		return;
	}

	if (GetIsStopping())
	{
		const FString ShortErrorMessage = TEXT("Audio processing failed");
		const FString LongErrorMessage = TEXT("The audio data could not be processed to the recognizer since the thread is stopping");
		ReportError(ShortErrorMessage, LongErrorMessage);
		return;
	}

	// Make sure to process the data in background thread
	if (IsInGameThread())
	{
		AsyncTask(ENamedThreads::AnyBackgroundHiPriTask, [SpeechRecognizerSharedPtr = WhisperState.WhisperUserData.SpeechRecognizerWeakPtr.Pin()]() mutable
		{
			if (!SpeechRecognizerSharedPtr.IsValid())
			{
				UE_LOG(LogRuntimeSpeechRecognizer, Warning, TEXT("Failed to force process the audio data since the thread worker is invalid"));
				return;
			}
			SpeechRecognizerSharedPtr->ForceProcessPendingAudioData();
		});
		return;
	}

	const int32 NumOfQueuedSamples = PendingAudio.GetTotalMixedAndResampledSize();
	Audio::FAlignedFloatBuffer PendingAudioData;
	if (!PendingAudio.GetMixedAndResampledAudio(PendingAudioData))
	{
		const FString ShortErrorMessage = TEXT("Audio processing failed");
		const FString LongErrorMessage = TEXT("The audio data could not be processed to the recognizer since the pending audio data could not be mixed and resampled");
		ReportError(ShortErrorMessage, LongErrorMessage);
		return;
	}

	AudioQueue.Enqueue(MoveTemp(PendingAudioData));
	UE_LOG(LogRuntimeSpeechRecognizer, Log, TEXT("Enqueued audio data from the pending audio to the queue of the speech recognizer as the last data (num of samples: %d)"), NumOfQueuedSamples);
}

void FSpeechRecognizerThread::ClearAudioData(bool bClearPendingAudioData, bool bClearAudioQueue)
{
	if (bClearPendingAudioData)
	{
		PendingAudio = FPendingAudioData();
	}
	if (bClearAudioQueue)
	{
		AudioQueue.Empty();
	}
}

bool FSpeechRecognizerThread::Init()
{
	if (GetIsStopped())
	{
		const FString ShortErrorMessage = TEXT("Thread initialization failed");
		const FString LongErrorMessage = TEXT("Unable to initialize a stopped thread");
		ReportError(ShortErrorMessage, LongErrorMessage);
		return false;
	}

	if (GetIsStopping())
	{
		const FString ShortErrorMessage = TEXT("Thread initialization failed");
		const FString LongErrorMessage = TEXT("Unable to initialize a thread that is stopping");
		ReportError(ShortErrorMessage, LongErrorMessage);
		return false;
	}

	if (!WhisperState.WhisperUserData.SpeechRecognizerWeakPtr.IsValid())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("The speech recognizer is not valid"));
		return false;
	}

	return FRunnable::Init();
}

uint32 FSpeechRecognizerThread::Run()
{
	while (!GetIsStopped() && !GetIsStopping())
	{
		Audio::FAlignedFloatBuffer NewQueuedBuffer;
		while (AudioQueue.Dequeue(NewQueuedBuffer))
		{
			bIsFinished.AtomicSet(false);

			// Resize the buffer to the minimum required size (1 second, plus 10% more due to a minor bug in checking the buffer size)
			// see https://github.com/ggerganov/whisper.cpp/issues/39
			constexpr float MinBufferDurationSec = 1.1;
			if (NewQueuedBuffer.Num() < WHISPER_SAMPLE_RATE * MinBufferDurationSec)
			{
				NewQueuedBuffer.AddZeroed(WHISPER_SAMPLE_RATE * MinBufferDurationSec - NewQueuedBuffer.Num());
			}
			if (whisper_full_parallel(WhisperState.WhisperContext, *WhisperState.WhisperParameters, NewQueuedBuffer.GetData(), NewQueuedBuffer.Num(), 1) != 0)
			{
				UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Failed to process audio data with the size of %d samples to the whisper recognizer"), NewQueuedBuffer.Num());
			}
			else
			{
				UE_LOG(LogRuntimeSpeechRecognizer, Log, TEXT("Processed audio data with the size of %d samples to the whisper recognizer"), NewQueuedBuffer.Num());
			}
		}

		if (DoesSharedInstanceExist() && PendingAudio.GetTotalMixedAndResampledSize() == 0 && !GetIsFinished())
		{
			bIsFinished.AtomicSet(true);
			TSharedPtr<FSpeechRecognizerThread> ThisShared = AsShared();
			AsyncTask(ENamedThreads::GameThread, [ThisShared]()
			{
				if (!ThisShared.IsValid())
				{
					UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Failed to get shared instance"));
					return;
				}
				if (ThisShared->LastProgress < 100)
				{
					UE_LOG(LogRuntimeSpeechRecognizer, Log, TEXT("Speech recognition progress: %d"), 100);
					ThisShared->LastProgress = 100;
					ThisShared->OnRecognitionProgress.Broadcast(100);
				}
				ThisShared->OnRecognitionFinished.Broadcast();
				UE_LOG(LogRuntimeSpeechRecognizer, Log, TEXT("Speech recognition finished"));
			});
		}
	}

	if (GetIsStopping())
	{
		if (DoesSharedInstanceExist())
		{
			TSharedPtr<FSpeechRecognizerThread> ThisShared = AsShared();
			AsyncTask(ENamedThreads::GameThread, [ThisShared]()
			{
				if (!ThisShared.IsValid())
				{
					UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Failed to get shared instance"));
					return;
				}
				ThisShared->OnRecognitionStopped.Broadcast();
			});
		}
	}

	return 0;
}

void FSpeechRecognizerThread::Stop()
{
	bIsStopping.AtomicSet(true);
	bIsFinished.AtomicSet(true);
	WhisperState.WhisperUserData = FWhisperSpeechRecognizerUserData();
	FRunnable::Stop();
	UE_LOG(LogRuntimeSpeechRecognizer, Log, TEXT("Stopping the speech recognizer thread"));
}

void FSpeechRecognizerThread::Exit()
{
	bIsStopping.AtomicSet(false);
	bIsStopped.AtomicSet(true);
	bIsFinished.AtomicSet(true);
	ReleaseMemory();
	FRunnable::Exit();
}

bool FSpeechRecognizerThread::SetRecognitionParameters(const FSpeechRecognitionParameters& Parameters)
{
	if (!GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set recognition parameters while the thread is running"));
		return false;
	}

	if (GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set recognition parameters while the thread is stopping"));
		return false;
	}

	RecognitionParameters = Parameters;
	return true;
}

FSpeechRecognitionParameters FSpeechRecognizerThread::GetNonStreamingDefaults()
{
	return FSpeechRecognitionParameters::GetNonStreamingDefaults();
}

FSpeechRecognitionParameters FSpeechRecognizerThread::GetStreamingDefaults()
{
	return FSpeechRecognitionParameters::GetStreamingDefaults();
}

FSpeechRecognitionParameters FSpeechRecognizerThread::GetRecognitionParameters() const
{
	return RecognitionParameters;
}

bool FSpeechRecognizerThread::SetNonStreamingDefaults()
{
	if (!GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set non-streaming defaults while the thread is running"));
		return false;
	}

	if (GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set non-streaming defaults while the thread is stopping"));
		return false;
	}

	RecognitionParameters.SetNonStreamingDefaults();
	return true;
}

bool FSpeechRecognizerThread::SetStreamingDefaults()
{
	if (!GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set streaming defaults while the thread is running"));
		return false;
	}

	if (GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set streaming defaults while the thread is stopping"));
		return false;
	}

	RecognitionParameters.SetStreamingDefaults();
	return true;
}

bool FSpeechRecognizerThread::SetNumOfThreads(int32 Value)
{
	if (!GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set the number of threads while the thread is running"));
		return false;
	}

	if (GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set the number of threads while the thread is stopping"));
		return false;
	}

	RecognitionParameters.NumOfThreads = Value;
	return true;
}

bool FSpeechRecognizerThread::SetLanguage(ESpeechRecognizerLanguage Language)
{
	if (!GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set language while the thread is running"));
		return false;
	}

	if (GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set language while the thread is stopping"));
		return false;
	}

	const USpeechRecognizerSettings* SpeechRecognizerSettings = GetDefault<USpeechRecognizerSettings>();
	if (RecognitionParameters.Language == ESpeechRecognizerLanguage::Auto && SpeechRecognizerSettings->ModelLanguage == ESpeechRecognizerModelLanguage::EnglishOnly)
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set language to 'auto' when the model is English only"));
		return false;
	}

	RecognitionParameters.Language = Language;
	return true;
}

bool FSpeechRecognizerThread::SetTranslateToEnglish(bool bTranslate)
{
	if (!GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set translation while the thread is running"));
		return false;
	}

	if (GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set translation while the thread is stopping"));
		return false;
	}

	RecognitionParameters.bTranslateToEnglish = bTranslate;
	return true;
}

bool FSpeechRecognizerThread::SetStepSize(int32 Value)
{
	if (!GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set step size while the thread is running"));
		return false;
	}

	if (GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set step size while the thread is stopping"));
		return false;
	}

	RecognitionParameters.StepSizeMs = Value;
	return true;
}

bool FSpeechRecognizerThread::SetNoContext(bool bNoContext)
{
	if (!GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set no context while the thread is running"));
		return false;
	}

	if (GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set no context while the thread is stopping"));
		return false;
	}

	RecognitionParameters.bNoContext = bNoContext;
	return true;
}

bool FSpeechRecognizerThread::SetSingleSegment(bool bSingleSegment)
{
	if (!GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set single segment while the thread is running"));
		return false;
	}

	if (GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set single segment while the thread is stopping"));
		return false;
	}

	RecognitionParameters.bSingleSegment = bSingleSegment;
	return true;
}

bool FSpeechRecognizerThread::SetMaxTokens(int32 Value)
{
	if (!GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set max tokens while the thread is running"));
		return false;
	}

	if (GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set max tokens while the thread is stopping"));
		return false;
	}

	RecognitionParameters.MaxTokens = Value;
	return true;
}

bool FSpeechRecognizerThread::SetSpeedUp(bool bSpeedUp)
{
	if (!GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set speed up while the thread is running"));
		return false;
	}

	if (GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set speed up while the thread is stopping"));
		return false;
	}

	RecognitionParameters.bSpeedUp = bSpeedUp;
	return true;
}

bool FSpeechRecognizerThread::SetAudioContextSize(int32 Value)
{
	if (!GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set audio context size while the thread is running"));
		return false;
	}

	if (GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set audio context size while the thread is stopping"));
		return false;
	}

	RecognitionParameters.AudioContextSize = Value;
	return true;
}

bool FSpeechRecognizerThread::SetTemperatureToIncrease(float Value)
{
	if (!GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set temperature to increase while the thread is running"));
		return false;
	}

	if (GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set temperature to increase while the thread is stopping"));
		return false;
	}

	RecognitionParameters.TemperatureToIncrease = Value;
	return true;
}

bool FSpeechRecognizerThread::SetEntropyThreshold(float Value)
{
	if (!GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set entropy threshold while the thread is running"));
		return false;
	}

	if (GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set entropy threshold while the thread is stopping"));
		return false;
	}

	RecognitionParameters.EntropyThreshold = Value;
	return true;
}

bool FSpeechRecognizerThread::SetSuppressBlank(bool Value)
{
	if (!GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set suppress blanks in output while the thread is running"));
		return false;
	}

	if (GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set suppress blanks in output while the thread is stopping"));
		return false;
	}

	RecognitionParameters.bSuppressBlank = Value;
	return true;
}

bool FSpeechRecognizerThread::SetSuppressNonSpeechTokens(bool Value)
{
	if (!GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set suppress non speech tokens in output while the thread is running"));
		return false;
	}

	if (GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set suppress non speech tokens in output while the thread is stopping"));
		return false;
	}

	RecognitionParameters.bSuppressNonSpeechTokens = Value;
	return true;
}

bool FSpeechRecognizerThread::SetBeamSize(int32 Value)
{
	if (!GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set beam size while the thread is running"));
		return false;
	}

	if (GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set beam size while the thread is stopping"));
		return false;
	}

	RecognitionParameters.BeamSize = Value;
	return true;
}

bool FSpeechRecognizerThread::SetInitialPrompt(const FString& Value)
{
	if (!GetIsStopped())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set beam size while the thread is running"));
		return false;
	}

	if (GetIsStopping())
	{
		UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Unable to set beam size while the thread is stopping"));
		return false;
	}

	RecognitionParameters.InitialPrompt = Value;
	return true;
}

bool FSpeechRecognizerThread::GetIsStopped() const
{
	return bIsStopped;
}

bool FSpeechRecognizerThread::GetIsStopping() const
{
	return bIsStopping;
}

bool FSpeechRecognizerThread::GetIsFinished() const
{
	return bIsFinished;
}

void FSpeechRecognizerThread::LoadLanguageModel(FOnLanguageModelLoaded&& OnLoadLanguageModel)
{
	const USpeechRecognizerSettings* SpeechRecognizerSettings = GetDefault<USpeechRecognizerSettings>();
	if (!SpeechRecognizerSettings)
	{
		const FString ShortErrorMessage = TEXT("Language model loading failed");
		const FString LongErrorMessage = TEXT("Failed to load the default speech recognizer settings");
		ReportError(ShortErrorMessage, LongErrorMessage);
		OnLoadLanguageModel(false, nullptr, 0);
		return;
	}

	TSharedPtr<FSpeechRecognizerThread> ThisShared = AsShared();
	if (!ThisShared.IsValid())
	{
		const FString ShortErrorMessage = TEXT("Failed to get shared instance");
		const FString LongErrorMessage = TEXT("Failed to get shared instance of speech recognizer");
		ReportError(ShortErrorMessage, LongErrorMessage);
		OnLoadLanguageModel(false, nullptr, 0);
		return;
	}

	const FString AssetPath = SpeechRecognizerSettings->GetLanguageModelAssetPath();

	TSoftObjectPtr<USpeechRecognizerModel> LazySpeechRecognizerModel = TSoftObjectPtr<USpeechRecognizerModel>(FSoftObjectPath(AssetPath));
	UAssetManager::GetStreamableManager().RequestAsyncLoad(LazySpeechRecognizerModel.ToSoftObjectPath(),
		[ThisShared, OnLoadLanguageModel = MoveTemp(OnLoadLanguageModel), LazySpeechRecognizerModel, AssetPath]() mutable
		{
			if (!ThisShared.IsValid())
			{
				UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Failed to get shared instance"));
				return;
			}

			AsyncTask(ENamedThreads::AnyBackgroundHiPriTask, [ThisShared, OnLoadLanguageModel = MoveTemp(OnLoadLanguageModel), LazySpeechRecognizerModel, AssetPath]() mutable
			{
				if (!ThisShared.IsValid())
				{
					UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Failed to get shared instance"));
					return;
				}

				auto ExecuteResultFromBGThread = [](FOnLanguageModelLoaded&& OnLoadLanguageModel, bool bSuccess, uint8* ModelBulkDataPtr, int64 ModelBulkDataSize) mutable
				{
					AsyncTask(ENamedThreads::GameThread, [OnLoadLanguageModel = MoveTemp(OnLoadLanguageModel), bSuccess, ModelBulkDataPtr, ModelBulkDataSize]() mutable
					{
						OnLoadLanguageModel(bSuccess, ModelBulkDataPtr, ModelBulkDataSize);
					});
				};

				if (!LazySpeechRecognizerModel.Get())
				{
					const FString ShortErrorMessage = TEXT("Language model loading failed");
					const FString LongErrorMessage = FString::Printf(TEXT("Failed to load the language model asset '%s'"), *AssetPath);
					ThisShared->ReportError(ShortErrorMessage, LongErrorMessage);
					ExecuteResultFromBGThread(MoveTemp(OnLoadLanguageModel), false, nullptr, 0);
					return;
				}

				USpeechRecognizerModel* SpeechRecognizerModel = LazySpeechRecognizerModel.Get();

				uint8* ModelBulkDataPtr = nullptr;
				SpeechRecognizerModel->LanguageModelBulkData.GetCopy(reinterpret_cast<void**>(&ModelBulkDataPtr), true);
				const int64 ModelBulkDataSize = SpeechRecognizerModel->LanguageModelBulkData.GetBulkDataSize();

				if (!ModelBulkDataPtr)
				{
					const FString ShortErrorMessage = TEXT("Language model buffer retrieval failed");
					const FString LongErrorMessage = FString::Printf(TEXT("Failed to retrieve the buffer data of the language model from the asset '%s'"), *AssetPath);
					ThisShared->ReportError(ShortErrorMessage, LongErrorMessage);
					ExecuteResultFromBGThread(MoveTemp(OnLoadLanguageModel), false, nullptr, 0);
					return;
				}

				AsyncTask(ENamedThreads::GameThread, [ThisShared, SpeechRecognizerModel, ModelBulkDataPtr, ModelBulkDataSize, OnLoadLanguageModel = MoveTemp(OnLoadLanguageModel)]() mutable
				{
					if (!ThisShared.IsValid())
					{
						UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("Failed to get shared instance"));
						FMemory::Free(ModelBulkDataPtr);
						return;
					}
					OnLoadLanguageModel(true, ModelBulkDataPtr, ModelBulkDataSize);
				});
			});
		}, FStreamableManager::AsyncLoadHighPriority);
}

void FSpeechRecognizerThread::ReleaseMemory()
{
	Thread.Reset();
	WhisperState.Release();
}

void FSpeechRecognizerThread::ReportError(const FString& ShortErrorMessage, const FString& LongErrorMessage)
{
	if (DoesSharedInstanceExist())
	{
		TSharedPtr<FSpeechRecognizerThread> ThisShared = AsShared();
		AsyncTask(ENamedThreads::GameThread, [ThisShared, ShortErrorMessage, LongErrorMessage]() mutable
		{
			if (ThisShared)
			{
				UE_LOG(LogRuntimeSpeechRecognizer, Error, TEXT("%s: %s"), *ShortErrorMessage, *LongErrorMessage);
				ThisShared->OnRecognitionError.Broadcast(ShortErrorMessage, LongErrorMessage);
			}
		});
	}
}
