/*
 * Copyright (C) 2016 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * This code was translated from the JSyn Java code.
 * JSyn is Copyright 2009 Phil Burk, Mobileer Inc
 * JSyn is licensed under the Apache License, Version 2.0
 */

#ifndef SYNTHMARK_SQUARE_OSCILLATOR_DPW_H
#define SYNTHMARK_SQUARE_OSCILLATOR_DPW_H

#include <cstdint>
#include <math.h>
#include "SynthTools.h"
#include "DifferentiatedParabola.h"

namespace marksynth {
/**
 * Square waves contains the odd partials of a fundamental.
 * The square wave is generated by combining two sawtooth waveforms
 * that are 180 degrees out of phase. This causes the even partials
 * to be cancelled out.
 */
class SquareOscillatorDPW : public SawtoothOscillator
{
public:
    SquareOscillatorDPW()
    : SawtoothOscillator()
    , dpw1()
    , dpw2() {}

    virtual ~SquareOscillatorDPW() = default;

    virtual inline synth_float_t translatePhase(synth_float_t phase1,
            synth_float_t phaseIncrement) {
        synth_float_t val1 = dpw1.next(phase1, phaseIncrement);

        /* Generate second sawtooth so we can add them together. */
        synth_float_t phase2 = phase1 + 1.0; /* 180 degrees out of phase. */
        if (phase2 >= 1.0)
            phase2 -= 2.0;
        synth_float_t val2 = dpw1.next(phase2, phaseIncrement);

        /*
         * Need to adjust amplitude based on positive phaseInc. little less than half at
         * Nyquist/2.0!
         */
        const synth_float_t STARTAMP = 0.92; // derived empirically
        synth_float_t positivePhaseIncrement = (phaseIncrement < 0.0)
                ? phaseIncrement
                : 0.0 - phaseIncrement;
        synth_float_t scale = STARTAMP - positivePhaseIncrement;
        return scale * (val1 - val2);
    }

private:
    DifferentiatedParabola dpw1;
    DifferentiatedParabola dpw2;
};
};
#endif // SYNTHMARK_SQUARE_OSCILLATOR_DPW_H
