# Use Flax NNX and Linen together

This guide is for existing Flax users who want to make their codebase a mixture of Flax Linen and Flax NNX `Module`s, which is made possible thanks to the `flax.nnx.bridge` API.

This will be helpful if you:

* Want to migrate your codebase to NNX gradually, one module at a time;
* Have external dependency that already moved to NNX but you haven't, or is still in Linen while you've moved to NNX.

We hope this allows you to move and try out NNX at your own pace, and leverage the best of both worlds. We will also talk about how to resolve the caveats of interoperating the two APIs, on a few aspects that they are fundamentally different.

**Note**:

This guide is about glueing Linen and NNX modules. To migrate an existing Linen module to NNX, check out the [Migrate from Flax Linen to Flax NNX](https://flax.readthedocs.io/en/latest/guides/linen_to_nnx.html) guide.

And all built-in Linen layers should have equivalent NNX versions! Check out the list of [Built-in NNX layers](https://flax.readthedocs.io/en/latest/api_reference/flax.nnx/nn/index.html).


```python
import os
os.environ["XLA_FLAGS"] = '--xla_force_host_platform_device_count=8'

from flax import nnx
from flax import linen as nn
from flax.nnx import bridge
import jax
from jax import numpy as jnp
from jax.experimental import mesh_utils
from typing import *
```

## Submodule is all you need

A Flax model is always a tree of modules - either old Linen modules (`flax.linen.Module`, usually written as `nn.Module`) or NNX modules (`nnx.Module`).

An `nnx.bridge` wrapper glues the two types together, in both ways:

* `nnx.bridge.ToNNX`: Convert a Linen module to NNX, so that it can be a submodule of another NNX module, or stand alone to be trained in NNX-style training loops.
* `nnx.bridge.ToLinen`: Vice versa, convert a NNX module to Linen.

This means you can move in either top-down or bottom-up behavior: convert the whole Linen module to NNX, then gradually move down, or convert all the lower level modules to NNX then move up.


## The Basics

There are two fundamental difference between Linen and NNX modules:

* **Stateless vs. stateful**: Linen module instances are stateless: variables are returned from a purely functional `.init()` call and managed separately. NNX modules, however, owns its variables as instance attributes.

* **Lazy vs. eager**: Linen modules only allocate space to create variables when they actually see their input. Whereas NNX module instances create their variables the moment they are instantiated, without seeing a sample input.

With that in mind, let's look at how the `nnx.bridge` wrappers tackle the differences.

### Linen -> NNX

Since Linen modules may require an input to create variables, we semi-formally supported lazy initialization in the NNX modules converted from Linen. The Linen variables are created when you give it a sample input.

For you, it's calling `nnx.bridge.lazy_init()` where you call `module.init()` in Linen code.

(Note: you can call `nnx.display` upon any NNX module to inspect all its variables and state.)


```python
class LinenDot(nn.Module):
  out_dim: int
  w_init: Callable[..., Any] = nn.initializers.lecun_normal()
  @nn.compact
  def __call__(self, x):
    # Linen might need the input shape to create the weight!
    w = self.param('w', self.w_init, (x.shape[-1], self.out_dim))
    return x @ w

x = jax.random.normal(jax.random.key(42), (4, 32))
model = bridge.ToNNX(LinenDot(64),
                     rngs=nnx.Rngs(0))  # => `model = LinenDot(64)` in Linen
bridge.lazy_init(model, x)              # => `var = model.init(key, x)` in Linen
y = model(x)                            # => `y = model.apply(var, x)` in Linen

nnx.display(model)

# In-place swap your weight array and the model still works!
model.w.value = jax.random.normal(jax.random.key(1), (32, 64))
assert not jnp.allclose(y, model(x))
```


<script> (()=>{ if (customElements.get('treescope-container') === undefined) { class TreescopeContainer extends HTMLElement { constructor() { super(); this.attachShadow({mode: "open"}); this.defns = {}; this.state = {}; } } customElements.define("treescope-container", TreescopeContainer); } if (customElements.get('treescope-run-here') === undefined) { class RunHere extends HTMLElement { constructor() { super() } connectedCallback() { const run = child => { const fn = new Function(child.textContent); child.textContent = ""; fn.call(this); this.remove(); }; const child = this.querySelector("script"); if (child) { run(child); } else { new MutationObserver(()=>{ run(this.querySelector("script")); }).observe(this, {childList: true}); } } } customElements.define("treescope-run-here", RunHere); } })(); </script> <treescope-container class="treescope_out_e6ae1e1419c34e9b9e4dbf513b5e956f" ></treescope-container> <treescope-run-here><script type="application/octet-stream"> const root = ( Array.from(document.getElementsByClassName( "treescope_out_e6ae1e1419c34e9b9e4dbf513b5e956f")) .filter((elt) => !elt.dataset.setup) )[0]; root.dataset.setup = 1; const msg = document.createElement("span"); msg.style = "color: #cccccc; font-family: monospace;"; msg.textContent = "(Loading...)"; root.state.loadingMsg = msg; root.shadowRoot.appendChild(msg); root.state.chain = new Promise((resolve, reject) => { const observer = new IntersectionObserver((entries) => { for (const entry of entries) { if (entry.isIntersecting) { resolve(); observer.disconnect(); return; } } }, {rootMargin: "1000px"}); window.setTimeout(() => { observer.observe(root); }, 0); }); root.state.deferring = false; const _insertNode = (node) => { for (let oldScript of node.querySelectorAll("script")) { let newScript = document.createElement("script"); newScript.type = oldScript.type; newScript.textContent = oldScript.textContent; oldScript.parentNode.replaceChild(newScript, oldScript); } if (root.state.loadingMsg) { root.state.loadingMsg.remove(); root.state.loadingMsg = null; } root.shadowRoot.appendChild(node); }; root.defns.insertContent = ((contentNode, compressed) => { if (compressed) { root.state.deferring = true; } if (root.state.deferring) { root.state.chain = (async () => { await root.state.chain; if (compressed) { const encoded = contentNode.textContent; const blob = new Blob([ Uint8Array.from(atob(encoded), (m) => m.codePointAt(0)) ]); const reader = blob.stream().pipeThrough( new DecompressionStream("deflate") ).pipeThrough( new TextDecoderStream("utf-8") ).getReader(); const parts = []; while (true) { const step = await reader.read(); if (step.done) { break; } parts.push(step.value); } const tpl = document.createElement('template'); tpl.innerHTML = parts.join(""); _insertNode(tpl.content); } else { _insertNode(contentNode.content); } })(); } else { _insertNode(contentNode.content); } }); </script></treescope-run-here><div style="display:none"> <script type="application/octet-stream" >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</script> <treescope-run-here><script type="application/octet-stream"> const root = ( Array.from(document.getElementsByClassName( "treescope_out_e6ae1e1419c34e9b9e4dbf513b5e956f")) .filter((elt) => !elt.dataset['step0']) )[0]; root.dataset['step0'] = 1; root.defns.insertContent( this.parentNode.querySelector('script[type="application/octet-stream"]'), true ); this.parentNode.remove(); </script></treescope-run-here> </div>



<div style="display:none"> <script type="application/octet-stream" >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</script> <treescope-run-here><script type="application/octet-stream"> const root = ( Array.from(document.getElementsByClassName( "treescope_out_e6ae1e1419c34e9b9e4dbf513b5e956f")) .filter((elt) => !elt.dataset['step1']) )[0]; root.dataset['step1'] = 1; root.defns.insertContent( this.parentNode.querySelector('script[type="application/octet-stream"]'), true ); this.parentNode.remove(); </script></treescope-run-here> </div>


`nnx.bridge.lazy_init` also works even if the top-level module is a pure-NNX one, so you can do sub-moduling as you wish:


```python
class NNXOuter(nnx.Module):
  def __init__(self, out_dim: int, rngs: nnx.Rngs):
    self.dot = nnx.bridge.ToNNX(LinenDot(out_dim), rngs=rngs)
    self.b = nnx.Param(jax.random.uniform(rngs.params(), (1, out_dim,)))

  def __call__(self, x):
    return self.dot(x) + self.b

x = jax.random.normal(jax.random.key(42), (4, 32))
model = bridge.lazy_init(NNXOuter(64, rngs=nnx.Rngs(0)), x)  # Can fit into one line
nnx.display(model)
```


<script> (()=>{ if (customElements.get('treescope-container') === undefined) { class TreescopeContainer extends HTMLElement { constructor() { super(); this.attachShadow({mode: "open"}); this.defns = {}; this.state = {}; } } customElements.define("treescope-container", TreescopeContainer); } if (customElements.get('treescope-run-here') === undefined) { class RunHere extends HTMLElement { constructor() { super() } connectedCallback() { const run = child => { const fn = new Function(child.textContent); child.textContent = ""; fn.call(this); this.remove(); }; const child = this.querySelector("script"); if (child) { run(child); } else { new MutationObserver(()=>{ run(this.querySelector("script")); }).observe(this, {childList: true}); } } } customElements.define("treescope-run-here", RunHere); } })(); </script> <treescope-container class="treescope_out_ac9736b84aee42a9b67a8d8e0ddbf454" ></treescope-container> <treescope-run-here><script type="application/octet-stream"> const root = ( Array.from(document.getElementsByClassName( "treescope_out_ac9736b84aee42a9b67a8d8e0ddbf454")) .filter((elt) => !elt.dataset.setup) )[0]; root.dataset.setup = 1; const msg = document.createElement("span"); msg.style = "color: #cccccc; font-family: monospace;"; msg.textContent = "(Loading...)"; root.state.loadingMsg = msg; root.shadowRoot.appendChild(msg); root.state.chain = new Promise((resolve, reject) => { const observer = new IntersectionObserver((entries) => { for (const entry of entries) { if (entry.isIntersecting) { resolve(); observer.disconnect(); return; } } }, {rootMargin: "1000px"}); window.setTimeout(() => { observer.observe(root); }, 0); }); root.state.deferring = false; const _insertNode = (node) => { for (let oldScript of node.querySelectorAll("script")) { let newScript = document.createElement("script"); newScript.type = oldScript.type; newScript.textContent = oldScript.textContent; oldScript.parentNode.replaceChild(newScript, oldScript); } if (root.state.loadingMsg) { root.state.loadingMsg.remove(); root.state.loadingMsg = null; } root.shadowRoot.appendChild(node); }; root.defns.insertContent = ((contentNode, compressed) => { if (compressed) { root.state.deferring = true; } if (root.state.deferring) { root.state.chain = (async () => { await root.state.chain; if (compressed) { const encoded = contentNode.textContent; const blob = new Blob([ Uint8Array.from(atob(encoded), (m) => m.codePointAt(0)) ]); const reader = blob.stream().pipeThrough( new DecompressionStream("deflate") ).pipeThrough( new TextDecoderStream("utf-8") ).getReader(); const parts = []; while (true) { const step = await reader.read(); if (step.done) { break; } parts.push(step.value); } const tpl = document.createElement('template'); tpl.innerHTML = parts.join(""); _insertNode(tpl.content); } else { _insertNode(contentNode.content); } })(); } else { _insertNode(contentNode.content); } }); </script></treescope-run-here><div style="display:none"> <script type="application/octet-stream" >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</script> <treescope-run-here><script type="application/octet-stream"> const root = ( Array.from(document.getElementsByClassName( "treescope_out_ac9736b84aee42a9b67a8d8e0ddbf454")) .filter((elt) => !elt.dataset['step0']) )[0]; root.dataset['step0'] = 1; root.defns.insertContent( this.parentNode.querySelector('script[type="application/octet-stream"]'), true ); this.parentNode.remove(); </script></treescope-run-here> </div>



<div style="display:none"> <script type="application/octet-stream" >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</script> <treescope-run-here><script type="application/octet-stream"> const root = ( Array.from(document.getElementsByClassName( "treescope_out_ac9736b84aee42a9b67a8d8e0ddbf454")) .filter((elt) => !elt.dataset['step1']) )[0]; root.dataset['step1'] = 1; root.defns.insertContent( this.parentNode.querySelector('script[type="application/octet-stream"]'), true ); this.parentNode.remove(); </script></treescope-run-here> </div>


The Linen weight is already converted to a typical NNX variable, which is a thin wrapper of the actual JAX array value within. Here, `w` is an `nnx.Param` because it belongs to the `params` collection of `LinenDot` module.

We will talk more about different collections and types in the [NNX Variable <-> Linen Collections](#variable-types-vs-collections) section. Right now, just know that they are converted to NNX variables like native ones.


```python
assert isinstance(model.dot.w, nnx.Param)
assert isinstance(model.dot.w.value, jax.Array)
```

If you create this model witout using `nnx.bridge.lazy_init`, the NNX variables defined outside will be initialized as usual, but the Linen part (wrapped inside `ToNNX`) will not.


```python
partial_model = NNXOuter(64, rngs=nnx.Rngs(0))
nnx.display(partial_model)
```


<script> (()=>{ if (customElements.get('treescope-container') === undefined) { class TreescopeContainer extends HTMLElement { constructor() { super(); this.attachShadow({mode: "open"}); this.defns = {}; this.state = {}; } } customElements.define("treescope-container", TreescopeContainer); } if (customElements.get('treescope-run-here') === undefined) { class RunHere extends HTMLElement { constructor() { super() } connectedCallback() { const run = child => { const fn = new Function(child.textContent); child.textContent = ""; fn.call(this); this.remove(); }; const child = this.querySelector("script"); if (child) { run(child); } else { new MutationObserver(()=>{ run(this.querySelector("script")); }).observe(this, {childList: true}); } } } customElements.define("treescope-run-here", RunHere); } })(); </script> <treescope-container class="treescope_out_f88cc882b4c04e40a80bb08349fa3708" ></treescope-container> <treescope-run-here><script type="application/octet-stream"> const root = ( Array.from(document.getElementsByClassName( "treescope_out_f88cc882b4c04e40a80bb08349fa3708")) .filter((elt) => !elt.dataset.setup) )[0]; root.dataset.setup = 1; const msg = document.createElement("span"); msg.style = "color: #cccccc; font-family: monospace;"; msg.textContent = "(Loading...)"; root.state.loadingMsg = msg; root.shadowRoot.appendChild(msg); root.state.chain = new Promise((resolve, reject) => { const observer = new IntersectionObserver((entries) => { for (const entry of entries) { if (entry.isIntersecting) { resolve(); observer.disconnect(); return; } } }, {rootMargin: "1000px"}); window.setTimeout(() => { observer.observe(root); }, 0); }); root.state.deferring = false; const _insertNode = (node) => { for (let oldScript of node.querySelectorAll("script")) { let newScript = document.createElement("script"); newScript.type = oldScript.type; newScript.textContent = oldScript.textContent; oldScript.parentNode.replaceChild(newScript, oldScript); } if (root.state.loadingMsg) { root.state.loadingMsg.remove(); root.state.loadingMsg = null; } root.shadowRoot.appendChild(node); }; root.defns.insertContent = ((contentNode, compressed) => { if (compressed) { root.state.deferring = true; } if (root.state.deferring) { root.state.chain = (async () => { await root.state.chain; if (compressed) { const encoded = contentNode.textContent; const blob = new Blob([ Uint8Array.from(atob(encoded), (m) => m.codePointAt(0)) ]); const reader = blob.stream().pipeThrough( new DecompressionStream("deflate") ).pipeThrough( new TextDecoderStream("utf-8") ).getReader(); const parts = []; while (true) { const step = await reader.read(); if (step.done) { break; } parts.push(step.value); } const tpl = document.createElement('template'); tpl.innerHTML = parts.join(""); _insertNode(tpl.content); } else { _insertNode(contentNode.content); } })(); } else { _insertNode(contentNode.content); } }); </script></treescope-run-here><div style="display:none"> <script type="application/octet-stream" >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</script> <treescope-run-here><script type="application/octet-stream"> const root = ( Array.from(document.getElementsByClassName( "treescope_out_f88cc882b4c04e40a80bb08349fa3708")) .filter((elt) => !elt.dataset['step0']) )[0]; root.dataset['step0'] = 1; root.defns.insertContent( this.parentNode.querySelector('script[type="application/octet-stream"]'), true ); this.parentNode.remove(); </script></treescope-run-here> </div>



<div style="display:none"> <script type="application/octet-stream" >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</script> <treescope-run-here><script type="application/octet-stream"> const root = ( Array.from(document.getElementsByClassName( "treescope_out_f88cc882b4c04e40a80bb08349fa3708")) .filter((elt) => !elt.dataset['step1']) )[0]; root.dataset['step1'] = 1; root.defns.insertContent( this.parentNode.querySelector('script[type="application/octet-stream"]'), true ); this.parentNode.remove(); </script></treescope-run-here> </div>



```python
full_model = bridge.lazy_init(partial_model, x)
nnx.display(full_model)
```


<script> (()=>{ if (customElements.get('treescope-container') === undefined) { class TreescopeContainer extends HTMLElement { constructor() { super(); this.attachShadow({mode: "open"}); this.defns = {}; this.state = {}; } } customElements.define("treescope-container", TreescopeContainer); } if (customElements.get('treescope-run-here') === undefined) { class RunHere extends HTMLElement { constructor() { super() } connectedCallback() { const run = child => { const fn = new Function(child.textContent); child.textContent = ""; fn.call(this); this.remove(); }; const child = this.querySelector("script"); if (child) { run(child); } else { new MutationObserver(()=>{ run(this.querySelector("script")); }).observe(this, {childList: true}); } } } customElements.define("treescope-run-here", RunHere); } })(); </script> <treescope-container class="treescope_out_b7009cb139f8490c85277bee2902b039" ></treescope-container> <treescope-run-here><script type="application/octet-stream"> const root = ( Array.from(document.getElementsByClassName( "treescope_out_b7009cb139f8490c85277bee2902b039")) .filter((elt) => !elt.dataset.setup) )[0]; root.dataset.setup = 1; const msg = document.createElement("span"); msg.style = "color: #cccccc; font-family: monospace;"; msg.textContent = "(Loading...)"; root.state.loadingMsg = msg; root.shadowRoot.appendChild(msg); root.state.chain = new Promise((resolve, reject) => { const observer = new IntersectionObserver((entries) => { for (const entry of entries) { if (entry.isIntersecting) { resolve(); observer.disconnect(); return; } } }, {rootMargin: "1000px"}); window.setTimeout(() => { observer.observe(root); }, 0); }); root.state.deferring = false; const _insertNode = (node) => { for (let oldScript of node.querySelectorAll("script")) { let newScript = document.createElement("script"); newScript.type = oldScript.type; newScript.textContent = oldScript.textContent; oldScript.parentNode.replaceChild(newScript, oldScript); } if (root.state.loadingMsg) { root.state.loadingMsg.remove(); root.state.loadingMsg = null; } root.shadowRoot.appendChild(node); }; root.defns.insertContent = ((contentNode, compressed) => { if (compressed) { root.state.deferring = true; } if (root.state.deferring) { root.state.chain = (async () => { await root.state.chain; if (compressed) { const encoded = contentNode.textContent; const blob = new Blob([ Uint8Array.from(atob(encoded), (m) => m.codePointAt(0)) ]); const reader = blob.stream().pipeThrough( new DecompressionStream("deflate") ).pipeThrough( new TextDecoderStream("utf-8") ).getReader(); const parts = []; while (true) { const step = await reader.read(); if (step.done) { break; } parts.push(step.value); } const tpl = document.createElement('template'); tpl.innerHTML = parts.join(""); _insertNode(tpl.content); } else { _insertNode(contentNode.content); } })(); } else { _insertNode(contentNode.content); } }); </script></treescope-run-here><div style="display:none"> <script type="application/octet-stream" >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</script> <treescope-run-here><script type="application/octet-stream"> const root = ( Array.from(document.getElementsByClassName( "treescope_out_b7009cb139f8490c85277bee2902b039")) .filter((elt) => !elt.dataset['step0']) )[0]; root.dataset['step0'] = 1; root.defns.insertContent( this.parentNode.querySelector('script[type="application/octet-stream"]'), true ); this.parentNode.remove(); </script></treescope-run-here> </div>



<div style="display:none"> <script type="application/octet-stream" >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</script> <treescope-run-here><script type="application/octet-stream"> const root = ( Array.from(document.getElementsByClassName( "treescope_out_b7009cb139f8490c85277bee2902b039")) .filter((elt) => !elt.dataset['step1']) )[0]; root.dataset['step1'] = 1; root.defns.insertContent( this.parentNode.querySelector('script[type="application/octet-stream"]'), true ); this.parentNode.remove(); </script></treescope-run-here> </div>


### NNX -> Linen

To convert an NNX module to Linen, you should forward your creation arguments to `bridge.ToLinen` and let it handle the actual creation process.

This is because NNX module instance initializes all its variables eagerly when it is created, which consumes memory and compute. On the other hand, Linen modules are stateless, and the typical `init` and `apply` process involves multiple creation of them. So `bridge.to_linen` will handle the actual module creation and make sure no memory is allocated twice.


```python
class NNXDot(nnx.Module):
  def __init__(self, in_dim: int, out_dim: int, rngs: nnx.Rngs):
    self.w = nnx.Param(nnx.initializers.lecun_normal()(
      rngs.params(), (in_dim, out_dim)))
  def __call__(self, x: jax.Array):
    return x @ self.w

x = jax.random.normal(jax.random.key(42), (4, 32))
# Pass in the arguments, not an actual module
model = bridge.to_linen(NNXDot, 32, out_dim=64)
variables = model.init(jax.random.key(0), x)
y = model.apply(variables, x)

print(list(variables.keys()))
print(variables['params']['w'].shape)  # => (32, 64)
print(y.shape)                         # => (4, 64)

```

    ['nnx', 'params']
    (32, 64)
    (4, 64)


Note that `ToLinen` modules need to track an extra variable collection - `nnx` - for the static metadata of the underlying NNX module.


```python
# This new field stores the static data that defines the underlying `NNXDot`
print(type(variables['nnx']['graphdef']))    # => `nnx.graph.NodeDef`
```

    <class 'flax.nnx.graph.NodeDef'>


`bridge.to_linen` is actually a convenience wrapper around the Linen module `bridge.ToLinen`. Most likely you won't need to use `ToLinen` directly at all, unless you are using one of the built-in arguments of `ToLinen`. For example, if your NNX module doesn't want to be initialized with RNG handling:


```python
class NNXAddConstant(nnx.Module):
  def __init__(self):
    self.constant = nnx.Variable(jnp.array(1))
  def __call__(self, x):
    return x + self.constant

# You have to use `skip_rng=True` because this module's `__init__` don't
# take `rng` as argument
model = bridge.ToLinen(NNXAddConstant, skip_rng=True)
y, var = model.init_with_output(jax.random.key(0), x)
```

Similar to `ToNNX`, you can use `ToLinen` to create a submodule of another Linen module.


```python
class LinenOuter(nn.Module):
  out_dim: int
  @nn.compact
  def __call__(self, x):
    dot = bridge.to_linen(NNXDot, x.shape[-1], self.out_dim)
    b = self.param('b', nn.initializers.lecun_normal(), (1, self.out_dim))
    return dot(x) + b

x = jax.random.normal(jax.random.key(42), (4, 32))
model = LinenOuter(out_dim=64)
y, variables = model.init_with_output(jax.random.key(0), x)
w, b = variables['params']['ToLinen_0']['w'], variables['params']['b']
print(w.shape, b.shape, y.shape)
```

    (32, 64) (1, 64) (4, 64)


## Handling RNG keys

All Flax modules, Linen or NNX, automatically handle the RNG keys for variable creation and random layers like dropouts. However, the specific logics of RNG key splitting are different, so you cannot generate the same params between Linen and NNX modules, even if you pass in same keys.

Another difference is that NNX modules are stateful, so they can track and update the RNG keys within themselves.

### Linen to NNX

If you convert a Linen module to NNX, you enjoy the stateful benefit and don't need to pass in extra RNG keys on every module call. You can use always `nnx.reseed` to reset the RNG state within.


```python
x = jax.random.normal(jax.random.key(42), (4, 32))
model = bridge.ToNNX(nn.Dropout(rate=0.5, deterministic=False), rngs=nnx.Rngs(dropout=0))
# We don't really need to call lazy_init because no extra params were created here,
# but it's a good practice to always add this line.
bridge.lazy_init(model, x)
y1, y2 = model(x), model(x)
assert not jnp.allclose(y1, y2)  # Two runs yield different outputs!

# Reset the dropout RNG seed, so that next model run will be the same as the first.
nnx.reseed(model, dropout=0)
assert jnp.allclose(y1, model(x))
```

### NNX to Linen

If you convert an NNX module to Linen, the underlying NNX module's RNG states will still be part of the top-level `variables`. On the other hand, Linen `apply()` call accepts different RNG keys on each call, which resets the internal Linen environment and allow different random data to be generated.

Now, it really depends on whether your underlying NNX module generates new random data from its RNG state, or from the passed-in argument. Fortunately, `nnx.Dropout` supports both - using passed-in keys if there is any, and use its own RNG state if not.

And this leaves you with two style options of handling the RNG keys:

* The NNX style (recommended): Let the underlying NNX state manage the RNG keys, no need to pass in extra keys in `apply()`. This means a few more lines to mutate the `variables` for every apply call, but things will look easier once your whole model no longer needs `ToLinen`.

* The Linen style: Just pass different RNG keys for every `apply()` call.


```python
x = jax.random.normal(jax.random.key(42), (4, 32))
model = bridge.to_linen(nnx.Dropout, rate=0.5)
variables = model.init({'dropout': jax.random.key(0)}, x)

# The NNX RNG state was stored inside `variables`
print('The RNG key in state:', variables['RngKey']['rngs']['dropout']['key'].value)
print('Number of key splits:', variables['RngCount']['rngs']['dropout']['count'].value)

# NNX style: Must set `RngCount` as mutable and update the variables after every `apply`
y1, updates = model.apply(variables, x, mutable=['RngCount'])
variables |= updates
y2, updates = model.apply(variables, x, mutable=['RngCount'])
variables |= updates
print('Number of key splits after y2:', variables['RngCount']['rngs']['dropout']['count'].value)
assert not jnp.allclose(y1, y2)  # Every call yields different output!

# Linen style: Just pass different RNG keys for every `apply()` call.
y3 = model.apply(variables, x, rngs={'dropout': jax.random.key(1)})
y4 = model.apply(variables, x, rngs={'dropout': jax.random.key(2)})
assert not jnp.allclose(y3, y4)  # Every call yields different output!
y5 = model.apply(variables, x, rngs={'dropout': jax.random.key(1)})
assert jnp.allclose(y3, y5)      # When you use same top-level RNG, outputs are same
```

    The RNG key in state: Array((), dtype=key<fry>) overlaying:
    [1428664606 3351135085]
    Number of key splits: 0
    Number of key splits after y2: 2


## NNX variable types vs. Linen collections

When you want to group some variables as one category, in Linen you use different collections; in NNX, since all variables shall be top-level Python attributes, you use different variable types.

Therefore, when mixing Linen and NNX modules, Flax must know the 1-to-1 mapping between Linen collections and NNX variable types, so that `ToNNX` and `ToLinen` can do the conversion automatically.

Flax keeps a registry for this, and it already covers all Flax's built-in Linen collections. You can register extra mapping of NNX variable type and Linen collection names using `nnx.register_variable_name_type_pair`.

### Linen to NNX

For any collection of your Linen module, `ToNNX` will convert all its endpoint arrays (aka. leaves) to a subtype of `nnx.Variable`, either from registry or automatically created on-the-fly.

(However, we still keep the whole collection as one class attribute, because Linen modules may have duplicated names over different collections.)


```python
class LinenMultiCollections(nn.Module):
  out_dim: int
  def setup(self):
    self.w = self.param('w', nn.initializers.lecun_normal(), (x.shape[-1], self.out_dim))
    self.b = self.param('b', nn.zeros_init(), (self.out_dim,))
    self.count = self.variable('counter', 'count', lambda: jnp.zeros((), jnp.int32))

  def __call__(self, x):
    if not self.is_initializing():
      self.count.value += 1
    y = x @ self.w + self.b
    self.sow('intermediates', 'dot_sum', jnp.sum(y))
    return y

x = jax.random.normal(jax.random.key(42), (2, 4))
model = bridge.lazy_init(bridge.ToNNX(LinenMultiCollections(3), rngs=nnx.Rngs(0)), x)
print(model.w)        # Of type `nnx.Param` - note this is still under attribute `params`
print(model.b)        # Of type `nnx.Param`
print(model.count)    # Of type `counter` - auto-created type from the collection name
print(type(model.count))

y = model(x, mutable=True)  # Linen's `sow()` needs `mutable=True` to trigger
print(model.dot_sum)        # Of type `nnx.Intermediates`
```

    Param(
      value=Array([[ 0.35401407,  0.38010964, -0.20674096],
             [-0.7356256 ,  0.35613298, -0.5099556 ],
             [-0.4783049 ,  0.4310735 ,  0.30137998],
             [-0.6102254 , -0.2668519 , -1.053598  ]], dtype=float32)
    )
    Param(
      value=Array([0., 0., 0.], dtype=float32)
    )
    counter(
      value=Array(0, dtype=int32)
    )
    <class 'abc.counter'>
    (Intermediate(
      value=Array(6.932987, dtype=float32)
    ),)


You can quickly separate different types of NNX variables apart using `nnx.split`.

This can be handy when you only want to set some variables as trainable.


```python
# Separate variables of different types with nnx.split
CountType = type(model.count)
static, params, counter, the_rest = nnx.split(model, nnx.Param, CountType, ...)
print('All Params:', list(params.keys()))
print('All Counters:', list(counter.keys()))
print('All the rest (intermediates and RNG keys):', list(the_rest.keys()))

model = nnx.merge(static, params, counter, the_rest)  # You can merge them back at any time
y = model(x, mutable=True)  # still works!
```

    All Params: ['b', 'w']
    All Counters: ['count']
    All the rest (intermediates and RNG keys): ['dot_sum', 'rngs']


### NNX to Linen

If you define custom NNX variable types, you should register their names with `nnx.register_variable_name_type_pair` so that they go to the desired collections.


```python
class Count(nnx.Variable): pass
nnx.register_variable_name_type_pair('counts', Count, overwrite=True)

class NNXMultiCollections(nnx.Module):
  def __init__(self, din, dout, rngs):
    self.w = nnx.Param(nnx.initializers.lecun_normal()(rngs.params(), (din, dout)))
    self.lora = nnx.LoRA(din, 3, dout, rngs=rngs)
    self.count = Count(jnp.array(0))

  def __call__(self, x):
    self.count += 1
    return (x @ self.w.value) + self.lora(x)

xkey, pkey, dkey = jax.random.split(jax.random.key(0), 3)
x = jax.random.normal(xkey, (2, 4))
model = bridge.to_linen(NNXMultiCollections, 4, 3)
var = model.init({'params': pkey, 'dropout': dkey}, x)
print('All Linen collections:', list(var.keys()))
print(var['params'])
```

    All Linen collections: ['nnx', 'LoRAParam', 'params', 'counts']
    {'w': Array([[ 0.2916921 ,  0.22780475,  0.06553137],
           [ 0.17487915, -0.34043145,  0.24764155],
           [ 0.6420431 ,  0.6220095 , -0.44769976],
           [ 0.11161668,  0.83873135, -0.7446058 ]], dtype=float32)}


## Partition metadata

Flax uses a metadata wrapper box over the raw JAX array to annotate how a variable should be sharded.

In Linen, this is an optional feature that triggered by using `nn.with_partitioning` on initializers (see more on [Linen partition metadata guide](https://flax.readthedocs.io/en/latest/guides/parallel_training/flax_on_pjit.html)). In NNX, since all NNX variables are wrapped by `nnx.Variable` class anyway, that class will hold the sharding annotations too.

The `bridge.ToNNX` and `bridge.ToLinen` API will automatically convert the sharding annotations, if you use the built-in annotation methods (aka. `nn.with_partitioning` for Linen and `nnx.with_partitioning` for NNX).

### Linen to NNX

Even if you are not using any partition metadata in your Linen module, the variable JAX arrays will be converted to `nnx.Variable`s that wraps the true JAX array within.

If you use `nn.with_partitioning` to annotate your Linen module's variables, the annotation will be converted to a `.sharding` field in the corresponding `nnx.Variable`.

You can then use `nnx.with_sharding_constraint` to explicitly put the arrays into the annotated partitions within a `jax.jit`-compiled function, to initialize the whole model with every array at the right sharding.


```python
class LinenDotWithPartitioning(nn.Module):
  out_dim: int
  @nn.compact
  def __call__(self, x):
    w = self.param('w', nn.with_partitioning(nn.initializers.lecun_normal(),
                                             ('in', 'out')),
                   (x.shape[-1], self.out_dim))
    return x @ w

@nnx.jit
def create_sharded_nnx_module(x):
  model = bridge.lazy_init(
    bridge.ToNNX(LinenDotWithPartitioning(64), rngs=nnx.Rngs(0)), x)
  state = nnx.state(model)
  sharded_state = nnx.with_sharding_constraint(state, nnx.get_partition_spec(state))
  nnx.update(model, sharded_state)
  return model


print(f'We have {len(jax.devices())} fake JAX devices now to partition this model...')
mesh = jax.sharding.Mesh(devices=mesh_utils.create_device_mesh((2, 4)),
                         axis_names=('in', 'out'))
x = jax.random.normal(jax.random.key(42), (4, 32))
with mesh:
  model = create_sharded_nnx_module(x)

print(type(model.w))           # `nnx.Param`
print(model.w.sharding)        # The partition annotation attached with `w`
print(model.w.value.sharding)  # The underlying JAX array is sharded across the 2x4 mesh
```

    We have 8 fake JAX devices now to partition this model...
    <class 'flax.nnx.variables.Param'>
    ('in', 'out')
    GSPMDSharding({devices=[2,4]<=[8]})


### NNX to Linen

If you are not using any metadata feature of the `nnx.Variable` (i.e., no sharding annotation, no registered hooks), the converted Linen module will not add a metadata wrapper to your NNX variable, and you don't need to worry about it.

But if you did add sharding annotations to your NNX variables, `ToLinen` will convert them to a default Linen partition metadata class called `bridge.NNXMeta`, retaining all the metadata you put into the NNX variable.

Like with any Linen metadata wrappers, you can use `linen.unbox()` to get the raw JAX array tree.


```python
class NNXDotWithParititioning(nnx.Module):
  def __init__(self, in_dim: int, out_dim: int, rngs: nnx.Rngs):
    init_fn = nnx.with_partitioning(nnx.initializers.lecun_normal(), ('in', 'out'))
    self.w = nnx.Param(init_fn(rngs.params(), (in_dim, out_dim)))
  def __call__(self, x: jax.Array):
    return x @ self.w

x = jax.random.normal(jax.random.key(42), (4, 32))

@jax.jit
def create_sharded_variables(key, x):
  model = bridge.to_linen(NNXDotWithParititioning, 32, 64)
  variables = model.init(key, x)
  # A `NNXMeta` wrapper of the underlying `nnx.Param`
  assert type(variables['params']['w']) == bridge.NNXMeta
  # The annotation coming from the `nnx.Param` => (in, out)
  assert variables['params']['w'].metadata['sharding'] == ('in', 'out')

  unboxed_variables = nn.unbox(variables)
  variable_pspecs = nn.get_partition_spec(variables)
  assert isinstance(unboxed_variables['params']['w'], jax.Array)
  assert variable_pspecs['params']['w'] == jax.sharding.PartitionSpec('in', 'out')

  sharded_vars = jax.tree.map(jax.lax.with_sharding_constraint,
                              nn.unbox(variables),
                              nn.get_partition_spec(variables))
  return sharded_vars

with mesh:
  variables = create_sharded_variables(jax.random.key(0), x)

# The underlying JAX array is sharded across the 2x4 mesh
print(variables['params']['w'].sharding)
```

    GSPMDSharding({devices=[2,4]<=[8]})


## Lifted transforms

In general, if you want to apply Linen/NNX-style lifted transforms upon an `nnx.bridge`-converted module, just go ahead and do it in the usual Linen/NNX syntax.

For Linen-style transforms, note that `bridge.ToLinen` is the top level module class, so you may want to just use it as the first argument of your transforms (which needs to be a `linen.Module` class in most cases)

### Linen to NNX

NNX style lifted transforms are similar to JAX transforms, and they work on functions.


```python
class NNXVmapped(nnx.Module):
  def __init__(self, out_dim: int, vmap_axis_size: int, rngs: nnx.Rngs):
    self.linen_dot = nnx.bridge.ToNNX(nn.Dense(out_dim, use_bias=False), rngs=rngs)
    self.vmap_axis_size = vmap_axis_size

  def __call__(self, x):

    @nnx.split_rngs(splits=self.vmap_axis_size)
    @nnx.vmap(in_axes=(0, 0), axis_size=self.vmap_axis_size)
    def vmap_fn(submodule, x):
      return submodule(x)

    return vmap_fn(self.linen_dot, x)

x = jax.random.normal(jax.random.key(0), (4, 32))
model = bridge.lazy_init(NNXVmapped(64, 4, rngs=nnx.Rngs(0)), x)

print(model.linen_dot.kernel.shape) # (4, 32, 64) - first axis with dim 4 got vmapped
y = model(x)
print(y.shape)
```

    (4, 32, 64)
    (4, 64)


### NNX to Linen

Note that `bridge.ToLinen` is the top level module class, so you may want to just use it as the first argument of your transforms (which needs to be a `linen.Module` class in most cases).

Also, since `bridge.ToLinen` introduced this extra `nnx` collection, you need to mark it when using the axis-changing transforms (`linen.vmap`, `linen.scan`, etc) to make sure they are passed inside.


```python
class LinenVmapped(nn.Module):
  dout: int
  @nn.compact
  def __call__(self, x):
    inner = nn.vmap(bridge.ToLinen, variable_axes={'params': 0, 'nnx': None}, split_rngs={'params': True}
                    )(nnx.Linear, args=(x.shape[-1], self.dout))
    return inner(x)

x = jax.random.normal(jax.random.key(42), (4, 32))
model = LinenVmapped(64)
var = model.init(jax.random.key(0), x)
print(var['params']['VmapToLinen_0']['kernel'].shape)  # (4, 32, 64) - leading dim 4 got vmapped
y = model.apply(var, x)
print(y.shape)
```

    (4, 32, 64)
    (4, 64)

