/*
 * Copyright 2015 Google, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.classyshark.silverghost.contentreader;

import com.google.classyshark.silverghost.contentreader.aar.AarReader;
import com.google.classyshark.silverghost.contentreader.apk.ApkReader;
import com.google.classyshark.silverghost.contentreader.clazz.ClazzReader;
import com.google.classyshark.silverghost.contentreader.dex.DexReader;
import com.google.classyshark.silverghost.contentreader.jar.JarReader;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Is a function : (binary file) --> {contents: classnames, components}
 */
public class ContentReader {

    /**
     * components that are part of jar & apk
     */
    public enum ARCHIVE_COMPONENT {
        ANDROID_MANIFEST, NATIVE_LIBRARY
    }

    public static class Component {

        public Component(String name, ARCHIVE_COMPONENT component) {
            this.name = name;
            this.component = component;
        }

        public String name;
        public ARCHIVE_COMPONENT component;
    }

    private BinaryContentReader formatReader;
    private List<String> allClassNames;

    public ContentReader(File binaryArchive) {
        allClassNames = new ArrayList<>();


        final String archiveName = binaryArchive.getName().toLowerCase();

        if (archiveName.endsWith(".jar")) {
                formatReader = new JarReader(binaryArchive);
        } else if (archiveName.endsWith(".dex")) {
            formatReader = new DexReader(binaryArchive);
        } else if (archiveName.endsWith(".apk")) {
            formatReader = new ApkReader(binaryArchive);
        } else if (archiveName.endsWith(".aar")) {
            formatReader = new AarReader(binaryArchive);
        } else {
            formatReader = new ClazzReader(binaryArchive);
        }
    }

    public void load() {
        try {
            if (allClassNames.isEmpty()) {
                formatReader.read();
                allClassNames = formatReader.getClassNames();
            }
        } catch (Exception e) {
            allClassNames = new ArrayList<>();
        }
    }

    public List<String> getAllClassNames() {
        // TODO add wrong state exception if read wasn't called
        return Collections.unmodifiableList(allClassNames);
    }

    public List<Component> getAllComponents() {
        return formatReader.getComponents();
    }
}
