<?php

/**
 * ---------------------------------------------------------------------
 *
 * GLPI - Gestionnaire Libre de Parc Informatique
 *
 * http://glpi-project.org
 *
 * @copyright 2015-2024 Teclib' and contributors.
 * @copyright 2003-2014 by the INDEPNET Development Team.
 * @licence   https://www.gnu.org/licenses/gpl-3.0.html
 *
 * ---------------------------------------------------------------------
 *
 * LICENSE
 *
 * This file is part of GLPI.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * ---------------------------------------------------------------------
 */

namespace tests\units\Glpi\Asset\Capacity;

use DbTestCase;
use DeviceHardDrive;
use DisplayPreference;
use Entity;
use Item_DeviceHardDrive;
use Item_Devices;
use Log;

class HasDevicesCapacity extends DbTestCase
{
    protected function getTargetCapacity(): string
    {
        return \Glpi\Asset\Capacity\HasDevicesCapacity::class;
    }

    public function testCapacityActivation(): void
    {
        global $CFG_GLPI;

        $root_entity_id = getItemByTypeName(Entity::class, '_test_root_entity', true);

        $definition_1 = $this->initAssetDefinition(
            capacities: [
                \Glpi\Asset\Capacity\HasDevicesCapacity::class,
                \Glpi\Asset\Capacity\HasNotepadCapacity::class,
            ]
        );
        $classname_1  = $definition_1->getAssetClassName();
        $definition_2 = $this->initAssetDefinition(
            capacities: [
                \Glpi\Asset\Capacity\HasHistoryCapacity::class,
            ]
        );
        $classname_2  = $definition_2->getAssetClassName();
        $definition_3 = $this->initAssetDefinition(
            capacities: [
                \Glpi\Asset\Capacity\HasDevicesCapacity::class,
                \Glpi\Asset\Capacity\HasHistoryCapacity::class,
            ]
        );
        $classname_3  = $definition_3->getAssetClassName();

        $has_capacity_mapping = [
            $classname_1 => true,
            $classname_2 => false,
            $classname_3 => true,
        ];

        foreach ($has_capacity_mapping as $classname => $has_capacity) {
            // Check that the class is globally registered
            if ($has_capacity) {
                $this->array($CFG_GLPI['itemdevices_types'])->contains($classname);
                $this->array($CFG_GLPI['itemdevices_itemaffinity'])->contains($classname);
            } else {
                $this->array($CFG_GLPI['itemdevices_types'])->notContains($classname);
                $this->array($CFG_GLPI['itemdevices_itemaffinity'])->notContains($classname);
            }
            foreach (array_keys($CFG_GLPI) as $config_key) {
                if (preg_match('/^itemdevice[a-z+]_types$/', $config_key)) {
                    if ($has_capacity) {
                        $this->array($CFG_GLPI[$config_key])->contains($classname);
                    } else {
                        $this->array($CFG_GLPI[$config_key])->notContains($classname);
                    }
                }
            }
            if ($has_capacity) {
                $this->array(Item_Devices::getItemAffinities($classname))->isNotEmpty();
            } else {
                $this->array(Item_Devices::getItemAffinities($classname))->isEmpty();
            }

            // Check that the corresponding tab is present on items
            $item = $this->createItem($classname, ['name' => __FUNCTION__, 'entities_id' => $root_entity_id]);
            $this->login(); // must be logged in to get tabs list
            if ($has_capacity) {
                $this->array($item->defineAllTabs())->hasKey('Item_Devices$1');
            } else {
                $this->array($item->defineAllTabs())->notHasKey('Item_Devices$1');
            }

            // Check that the related search options are available
            $so_keys = [
                13,   // Graphic card / Designation
                14,   // Motherboard / Designation
                17,   // Processor / Designation
                18,   // Processor / Number of cores
                34,   // Processor / Number of threads
                35,   // Processor / Number of processors
                35,   // Processor / Frequency
                39,   // Power supply / Designation
                95,   // PCI / Designation
                110,  // Memory / Designation
                111,  // Memory / Size
                112,  // Network card / Designation
                113,  // Network card / MAC
                114,  // Hard drive / Designation
                115,  // Hard drive / Capacity
                116,  // Hard drive / Type
                1313, // Firmware / Designation
                1314, // Firmware / Version
                1315, // Firmware / Type
                1316, // Firmware / Model
                1317, // Firmware / Manufacturer
                1318, // Firmware / Serial
                1319, // Firmware / Other serial
            ];
            if ($has_capacity) {
                $this->array($item->getOptions())->hasKeys($so_keys);
            } else {
                $this->array($item->getOptions())->notHasKeys($so_keys);
            }
        }
    }

    public function testCapacityDeactivation(): void
    {
        /** @var array $CFG_GLPI */
        global $CFG_GLPI;

        $root_entity_id = getItemByTypeName(Entity::class, '_test_root_entity', true);

        $definition_1 = $this->initAssetDefinition(
            capacities: [
                \Glpi\Asset\Capacity\HasDevicesCapacity::class,
                \Glpi\Asset\Capacity\HasHistoryCapacity::class,
            ]
        );
        $classname_1  = $definition_1->getAssetClassName();
        $definition_2 = $this->initAssetDefinition(
            capacities: [
                \Glpi\Asset\Capacity\HasDevicesCapacity::class,
                \Glpi\Asset\Capacity\HasHistoryCapacity::class,
            ]
        );
        $classname_2  = $definition_2->getAssetClassName();

        $item_1 = $this->createItem(
            $classname_1,
            [
                'name' => __FUNCTION__,
                'entities_id' => $root_entity_id,
            ]
        );
        $item_2 = $this->createItem(
            $classname_2,
            [
                'name' => __FUNCTION__,
                'entities_id' => $root_entity_id,
            ]
        );

        $device = $this->createItem(
            DeviceHardDrive::class,
            [
                'designation' => 'HDD',
            ]
        );

        $device_item_1 = $this->createItem(
            Item_DeviceHardDrive::class,
            [
                'deviceharddrives_id' => $device->getID(),
                'itemtype'            => $item_1->getType(),
                'items_id'            => $item_1->getID(),
            ]
        );
        $device_item_2 = $this->createItem(
            Item_DeviceHardDrive::class,
            [
                'deviceharddrives_id' => $device->getID(),
                'itemtype'            => $item_2->getType(),
                'items_id'            => $item_2->getID(),
            ]
        );
        $displaypref_1   = $this->createItem(
            DisplayPreference::class,
            [
                'itemtype' => $classname_1,
                'num'      => 115, // capacity
                'users_id' => 0,
            ]
        );
        $displaypref_2   = $this->createItem(
            DisplayPreference::class,
            [
                'itemtype' => $classname_2,
                'num'      => 115, // capacity
                'users_id' => 0,
            ]
        );

        $item_1_logs_criteria = [
            'itemtype'      => $classname_1,
            'itemtype_link' => DeviceHardDrive::class,
        ];
        $item_2_logs_criteria = [
            'itemtype'      => $classname_2,
            'itemtype_link' => DeviceHardDrive::class,
        ];

        // Ensure relation, display preferences and logs exists, and class is registered to global config
        $this->object(Item_DeviceHardDrive::getById($device_item_1->getID()))->isInstanceOf(Item_DeviceHardDrive::class);
        $this->object(DisplayPreference::getById($displaypref_1->getID()))->isInstanceOf(DisplayPreference::class);
        $this->integer(countElementsInTable(Log::getTable(), $item_1_logs_criteria))->isEqualTo(1); // link
        $this->object(Item_DeviceHardDrive::getById($device_item_2->getID()))->isInstanceOf(Item_DeviceHardDrive::class);
        $this->object(DisplayPreference::getById($displaypref_2->getID()))->isInstanceOf(DisplayPreference::class);
        $this->integer(countElementsInTable(Log::getTable(), $item_2_logs_criteria))->isEqualTo(1); // link
        $this->array($CFG_GLPI['itemdevices_types'])->contains($classname_1);
        $this->array($CFG_GLPI['itemdevices_types'])->contains($classname_2);

        // Disable capacity and check that relations have been cleaned, and class is unregistered from global config
        $this->boolean($definition_1->update(['id' => $definition_1->getID(), 'capacities' => []]))->isTrue();
        $this->boolean(Item_DeviceHardDrive::getById($device_item_1->getID()))->isFalse();
        $this->boolean(DisplayPreference::getById($displaypref_1->getID()))->isFalse();
        $this->integer(countElementsInTable(Log::getTable(), $item_1_logs_criteria))->isEqualTo(0);
        $this->array($CFG_GLPI['itemdevices_types'])->notContains($classname_1);

        // Ensure relations, logs and global registration are preserved for other definition
        $this->object(Item_DeviceHardDrive::getById($device_item_2->getID()))->isInstanceOf(Item_DeviceHardDrive::class);
        $this->object(DisplayPreference::getById($displaypref_2->getID()))->isInstanceOf(DisplayPreference::class);
        $this->integer(countElementsInTable(Log::getTable(), $item_2_logs_criteria))->isEqualTo(1);
        $this->array($CFG_GLPI['itemdevices_types'])->contains($classname_2);
    }

    public function testCloneAsset()
    {
        $definition = $this->initAssetDefinition(
            capacities: [\Glpi\Asset\Capacity\HasDevicesCapacity::class]
        );
        $class = $definition->getAssetClassName();
        $entity = $this->getTestRootEntity(true);

        /** @var \Glpi\Asset\Asset $asset */
        $asset = $this->createItem(
            $class,
            [
                'name'        => 'Test asset',
                'entities_id' => $entity,
            ]
        );

        $device = $this->createItem(
            DeviceHardDrive::class,
            [
                'designation' => 'HDD',
            ]
        );

        $this->createItem(
            Item_DeviceHardDrive::class,
            [
                'deviceharddrives_id' => $device->getID(),
                'itemtype'            => $class,
                'items_id'            => $asset->getID(),
            ]
        );

        $this->integer($clone_id = $asset->clone())->isGreaterThan(0);
        $this->array(getAllDataFromTable(Item_DeviceHardDrive::getTable(), [
            'deviceharddrives_id' => $device->getID(),
            'itemtype'            => $class,
            'items_id'            => $clone_id,
        ]))->hasSize(1);
    }

    public function testIsUsed(): void
    {
        $entity_id = $this->getTestRootEntity(true);

        $definition = $this->initAssetDefinition(
            capacities: [\Glpi\Asset\Capacity\HasDevicesCapacity::class]
        );

        $asset = $this->createItem($definition->getAssetClassName(), [
            'name' => 'Test asset',
            'entities_id' => $entity_id,
        ]);

        // Check that the capacity is not yet considered as used
        $capacity = new \Glpi\Asset\Capacity\HasDevicesCapacity();
        $this->boolean($capacity->isUsed($definition->getAssetClassName()))->isFalse();

        // Create a relation with a device
        $device = $this->createItem(
            DeviceHardDrive::class,
            [
                'designation' => 'HDD',
            ]
        );

        $this->createItem(
            Item_DeviceHardDrive::class,
            [
                'deviceharddrives_id' => $device->getID(),
                'itemtype'            => $definition->getAssetClassName(),
                'items_id'            => $asset->getID(),
            ]
        );

        // Check that the capacity is considered as used
        $this->boolean($capacity->isUsed($definition->getAssetClassName()))->isTrue();
    }

    public function testGetCapacityUsageDescription(): void
    {
        $entity_id = $this->getTestRootEntity(true);

        $definition = $this->initAssetDefinition(
            system_name: 'TestAsset',
            capacities: [\Glpi\Asset\Capacity\HasDevicesCapacity::class]
        );
        $capacity = new \Glpi\Asset\Capacity\HasDevicesCapacity();

        // Check that the capacity usage description is correct
        $this->string($capacity->getCapacityUsageDescription($definition->getAssetClassName()))
            ->isEqualTo('0 component(s) attached to 0 asset(s)');

        // Create assets
        $asset_1 = $this->createItem($definition->getAssetClassName(), [
            'name' => 'Test asset',
            'entities_id' => $this->getTestRootEntity(true),
        ]);
        $asset_2 = $this->createItem($definition->getAssetClassName(), [
            'name' => 'Test asset 2',
            'entities_id' => $this->getTestRootEntity(true),
        ]);

        // Check that the capacity usage description is correct
        $this->string($capacity->getCapacityUsageDescription($definition->getAssetClassName()))
            ->isEqualTo('0 component(s) attached to 0 asset(s)');

        $device_1 = $this->createItem(
            DeviceHardDrive::class,
            [
                'designation' => 'HDD abcd',
            ]
        );
        $device_2 = $this->createItem(
            DeviceHardDrive::class,
            [
                'designation' => 'HDD efgh',
            ]
        );

        // Attach the first component to the first asset
        for ($i = 0; $i < 3; $i++) {
            // Each relation should be counted once, as it corresponds to a unique component
            $this->createItem(
                Item_DeviceHardDrive::class,
                [
                    'deviceharddrives_id' => $device_1->getID(),
                    'itemtype'            => $definition->getAssetClassName(),
                    'items_id'            => $asset_1->getID(),
                ]
            );
        }

        // Check that the capacity usage description is correct
        $this->string($capacity->getCapacityUsageDescription($definition->getAssetClassName()))
            ->isEqualTo('1 component(s) attached to 1 asset(s)');

        // Attach the first component to the second asset
        for ($i = 0; $i < 3; $i++) {
            // Each relation should be counted once, as it corresponds to a unique component
            $this->createItem(
                Item_DeviceHardDrive::class,
                [
                    'deviceharddrives_id' => $device_1->getID(),
                    'itemtype'            => $definition->getAssetClassName(),
                    'items_id'            => $asset_2->getID(),
                ]
            );
        }

        // Check that the capacity usage description is correct
        $this->string($capacity->getCapacityUsageDescription($definition->getAssetClassName()))
            ->isEqualTo('1 component(s) attached to 2 asset(s)');

        // Attach the second component to the first asset
        for ($i = 0; $i < 3; $i++) {
            // Each relation should be counted once, as it corresponds to a unique component
            $this->createItem(
                Item_DeviceHardDrive::class,
                [
                    'deviceharddrives_id' => $device_2->getID(),
                    'itemtype'            => $definition->getAssetClassName(),
                    'items_id'            => $asset_1->getID(),
                ]
            );
        }

        // Check that the capacity usage description is correct
        $this->string($capacity->getCapacityUsageDescription($definition->getAssetClassName()))
            ->isEqualTo('2 component(s) attached to 2 asset(s)');
    }
}
